use std::io::Write;

pub fn try_format(writer: &super::Writer, tokens: &str) -> String {
    let preamble = if writer.no_bindgen_comment {
        String::new()
    } else {
        let name = std::env!("CARGO_PKG_NAME");
        let version = std::env!("CARGO_PKG_VERSION");

        format!(
            r#"// Bindings generated by `{name}` {version}


"#
        )
    };

    // Packaging - e.g. windows/windows-sys crates - assumes the crate will allow whatever warnings it deems fit.
    let allow = if writer.package || writer.no_inner_attributes {
        ""
    } else {
        "#![allow(non_snake_case, non_upper_case_globals, non_camel_case_types, dead_code, clippy::all)]\n"
    };
    let tokens = format!("{preamble}{allow}{tokens}");

    let mut cmd = std::process::Command::new("rustfmt");
    cmd.stdin(std::process::Stdio::piped());
    cmd.stdout(std::process::Stdio::piped());
    cmd.stderr(std::process::Stdio::null());

    if !writer.rustfmt_config.is_empty() {
        cmd.arg("--config");
        cmd.arg(writer.rustfmt_config.as_str());
    }

    let Ok(mut child) = cmd.spawn() else {
        return tokens;
    };

    let Some(mut stdin) = child.stdin.take() else {
        return tokens;
    };

    if stdin.write_all(tokens.as_bytes()).is_err() {
        return tokens;
    }

    drop(stdin);

    let Ok(output) = child.wait_with_output() else {
        return tokens;
    };

    if !output.status.success() {
        return tokens;
    }

    if let Ok(result) = String::from_utf8(output.stdout) {
        result
    } else {
        tokens
    }
}
