/*
 * Copyright (C) 2020 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "EncodingTables.h"

#include "TextCodecICU.h"
#include <mutex>

namespace PAL {

#if ASSERT_ENABLED
// From https://encoding.spec.whatwg.org/index-jis0208.txt
constexpr std::array<std::pair<uint16_t, UChar>, 7724> jis0208Reference { {
    { 0, 0x3000 }, { 1, 0x3001 }, { 2, 0x3002 }, { 3, 0xFF0C }, { 4, 0xFF0E }, { 5, 0x30FB }, { 6, 0xFF1A }, { 7, 0xFF1B },
    { 8, 0xFF1F }, { 9, 0xFF01 }, { 10, 0x309B }, { 11, 0x309C }, { 12, 0x00B4 }, { 13, 0xFF40 }, { 14, 0x00A8 }, { 15, 0xFF3E },
    { 16, 0xFFE3 }, { 17, 0xFF3F }, { 18, 0x30FD }, { 19, 0x30FE }, { 20, 0x309D }, { 21, 0x309E }, { 22, 0x3003 }, { 23, 0x4EDD },
    { 24, 0x3005 }, { 25, 0x3006 }, { 26, 0x3007 }, { 27, 0x30FC }, { 28, 0x2015 }, { 29, 0x2010 }, { 30, 0xFF0F }, { 31, 0xFF3C },
    { 32, 0xFF5E }, { 33, 0x2225 }, { 34, 0xFF5C }, { 35, 0x2026 }, { 36, 0x2025 }, { 37, 0x2018 }, { 38, 0x2019 }, { 39, 0x201C },
    { 40, 0x201D }, { 41, 0xFF08 }, { 42, 0xFF09 }, { 43, 0x3014 }, { 44, 0x3015 }, { 45, 0xFF3B }, { 46, 0xFF3D }, { 47, 0xFF5B },
    { 48, 0xFF5D }, { 49, 0x3008 }, { 50, 0x3009 }, { 51, 0x300A }, { 52, 0x300B }, { 53, 0x300C }, { 54, 0x300D }, { 55, 0x300E },
    { 56, 0x300F }, { 57, 0x3010 }, { 58, 0x3011 }, { 59, 0xFF0B }, { 60, 0xFF0D }, { 61, 0x00B1 }, { 62, 0x00D7 }, { 63, 0x00F7 },
    { 64, 0xFF1D }, { 65, 0x2260 }, { 66, 0xFF1C }, { 67, 0xFF1E }, { 68, 0x2266 }, { 69, 0x2267 }, { 70, 0x221E }, { 71, 0x2234 },
    { 72, 0x2642 }, { 73, 0x2640 }, { 74, 0x00B0 }, { 75, 0x2032 }, { 76, 0x2033 }, { 77, 0x2103 }, { 78, 0xFFE5 }, { 79, 0xFF04 },
    { 80, 0xFFE0 }, { 81, 0xFFE1 }, { 82, 0xFF05 }, { 83, 0xFF03 }, { 84, 0xFF06 }, { 85, 0xFF0A }, { 86, 0xFF20 }, { 87, 0x00A7 },
    { 88, 0x2606 }, { 89, 0x2605 }, { 90, 0x25CB }, { 91, 0x25CF }, { 92, 0x25CE }, { 93, 0x25C7 }, { 94, 0x25C6 }, { 95, 0x25A1 },
    { 96, 0x25A0 }, { 97, 0x25B3 }, { 98, 0x25B2 }, { 99, 0x25BD }, { 100, 0x25BC }, { 101, 0x203B }, { 102, 0x3012 }, { 103, 0x2192 },
    { 104, 0x2190 }, { 105, 0x2191 }, { 106, 0x2193 }, { 107, 0x3013 }, { 119, 0x2208 }, { 120, 0x220B }, { 121, 0x2286 }, { 122, 0x2287 },
    { 123, 0x2282 }, { 124, 0x2283 }, { 125, 0x222A }, { 126, 0x2229 }, { 135, 0x2227 }, { 136, 0x2228 }, { 137, 0xFFE2 }, { 138, 0x21D2 },
    { 139, 0x21D4 }, { 140, 0x2200 }, { 141, 0x2203 }, { 153, 0x2220 }, { 154, 0x22A5 }, { 155, 0x2312 }, { 156, 0x2202 }, { 157, 0x2207 },
    { 158, 0x2261 }, { 159, 0x2252 }, { 160, 0x226A }, { 161, 0x226B }, { 162, 0x221A }, { 163, 0x223D }, { 164, 0x221D }, { 165, 0x2235 },
    { 166, 0x222B }, { 167, 0x222C }, { 175, 0x212B }, { 176, 0x2030 }, { 177, 0x266F }, { 178, 0x266D }, { 179, 0x266A }, { 180, 0x2020 },
    { 181, 0x2021 }, { 182, 0x00B6 }, { 187, 0x25EF }, { 203, 0xFF10 }, { 204, 0xFF11 }, { 205, 0xFF12 }, { 206, 0xFF13 }, { 207, 0xFF14 },
    { 208, 0xFF15 }, { 209, 0xFF16 }, { 210, 0xFF17 }, { 211, 0xFF18 }, { 212, 0xFF19 }, { 220, 0xFF21 }, { 221, 0xFF22 }, { 222, 0xFF23 },
    { 223, 0xFF24 }, { 224, 0xFF25 }, { 225, 0xFF26 }, { 226, 0xFF27 }, { 227, 0xFF28 }, { 228, 0xFF29 }, { 229, 0xFF2A }, { 230, 0xFF2B },
    { 231, 0xFF2C }, { 232, 0xFF2D }, { 233, 0xFF2E }, { 234, 0xFF2F }, { 235, 0xFF30 }, { 236, 0xFF31 }, { 237, 0xFF32 }, { 238, 0xFF33 },
    { 239, 0xFF34 }, { 240, 0xFF35 }, { 241, 0xFF36 }, { 242, 0xFF37 }, { 243, 0xFF38 }, { 244, 0xFF39 }, { 245, 0xFF3A }, { 252, 0xFF41 },
    { 253, 0xFF42 }, { 254, 0xFF43 }, { 255, 0xFF44 }, { 256, 0xFF45 }, { 257, 0xFF46 }, { 258, 0xFF47 }, { 259, 0xFF48 }, { 260, 0xFF49 },
    { 261, 0xFF4A }, { 262, 0xFF4B }, { 263, 0xFF4C }, { 264, 0xFF4D }, { 265, 0xFF4E }, { 266, 0xFF4F }, { 267, 0xFF50 }, { 268, 0xFF51 },
    { 269, 0xFF52 }, { 270, 0xFF53 }, { 271, 0xFF54 }, { 272, 0xFF55 }, { 273, 0xFF56 }, { 274, 0xFF57 }, { 275, 0xFF58 }, { 276, 0xFF59 },
    { 277, 0xFF5A }, { 282, 0x3041 }, { 283, 0x3042 }, { 284, 0x3043 }, { 285, 0x3044 }, { 286, 0x3045 }, { 287, 0x3046 }, { 288, 0x3047 },
    { 289, 0x3048 }, { 290, 0x3049 }, { 291, 0x304A }, { 292, 0x304B }, { 293, 0x304C }, { 294, 0x304D }, { 295, 0x304E }, { 296, 0x304F },
    { 297, 0x3050 }, { 298, 0x3051 }, { 299, 0x3052 }, { 300, 0x3053 }, { 301, 0x3054 }, { 302, 0x3055 }, { 303, 0x3056 }, { 304, 0x3057 },
    { 305, 0x3058 }, { 306, 0x3059 }, { 307, 0x305A }, { 308, 0x305B }, { 309, 0x305C }, { 310, 0x305D }, { 311, 0x305E }, { 312, 0x305F },
    { 313, 0x3060 }, { 314, 0x3061 }, { 315, 0x3062 }, { 316, 0x3063 }, { 317, 0x3064 }, { 318, 0x3065 }, { 319, 0x3066 }, { 320, 0x3067 },
    { 321, 0x3068 }, { 322, 0x3069 }, { 323, 0x306A }, { 324, 0x306B }, { 325, 0x306C }, { 326, 0x306D }, { 327, 0x306E }, { 328, 0x306F },
    { 329, 0x3070 }, { 330, 0x3071 }, { 331, 0x3072 }, { 332, 0x3073 }, { 333, 0x3074 }, { 334, 0x3075 }, { 335, 0x3076 }, { 336, 0x3077 },
    { 337, 0x3078 }, { 338, 0x3079 }, { 339, 0x307A }, { 340, 0x307B }, { 341, 0x307C }, { 342, 0x307D }, { 343, 0x307E }, { 344, 0x307F },
    { 345, 0x3080 }, { 346, 0x3081 }, { 347, 0x3082 }, { 348, 0x3083 }, { 349, 0x3084 }, { 350, 0x3085 }, { 351, 0x3086 }, { 352, 0x3087 },
    { 353, 0x3088 }, { 354, 0x3089 }, { 355, 0x308A }, { 356, 0x308B }, { 357, 0x308C }, { 358, 0x308D }, { 359, 0x308E }, { 360, 0x308F },
    { 361, 0x3090 }, { 362, 0x3091 }, { 363, 0x3092 }, { 364, 0x3093 }, { 376, 0x30A1 }, { 377, 0x30A2 }, { 378, 0x30A3 }, { 379, 0x30A4 },
    { 380, 0x30A5 }, { 381, 0x30A6 }, { 382, 0x30A7 }, { 383, 0x30A8 }, { 384, 0x30A9 }, { 385, 0x30AA }, { 386, 0x30AB }, { 387, 0x30AC },
    { 388, 0x30AD }, { 389, 0x30AE }, { 390, 0x30AF }, { 391, 0x30B0 }, { 392, 0x30B1 }, { 393, 0x30B2 }, { 394, 0x30B3 }, { 395, 0x30B4 },
    { 396, 0x30B5 }, { 397, 0x30B6 }, { 398, 0x30B7 }, { 399, 0x30B8 }, { 400, 0x30B9 }, { 401, 0x30BA }, { 402, 0x30BB }, { 403, 0x30BC },
    { 404, 0x30BD }, { 405, 0x30BE }, { 406, 0x30BF }, { 407, 0x30C0 }, { 408, 0x30C1 }, { 409, 0x30C2 }, { 410, 0x30C3 }, { 411, 0x30C4 },
    { 412, 0x30C5 }, { 413, 0x30C6 }, { 414, 0x30C7 }, { 415, 0x30C8 }, { 416, 0x30C9 }, { 417, 0x30CA }, { 418, 0x30CB }, { 419, 0x30CC },
    { 420, 0x30CD }, { 421, 0x30CE }, { 422, 0x30CF }, { 423, 0x30D0 }, { 424, 0x30D1 }, { 425, 0x30D2 }, { 426, 0x30D3 }, { 427, 0x30D4 },
    { 428, 0x30D5 }, { 429, 0x30D6 }, { 430, 0x30D7 }, { 431, 0x30D8 }, { 432, 0x30D9 }, { 433, 0x30DA }, { 434, 0x30DB }, { 435, 0x30DC },
    { 436, 0x30DD }, { 437, 0x30DE }, { 438, 0x30DF }, { 439, 0x30E0 }, { 440, 0x30E1 }, { 441, 0x30E2 }, { 442, 0x30E3 }, { 443, 0x30E4 },
    { 444, 0x30E5 }, { 445, 0x30E6 }, { 446, 0x30E7 }, { 447, 0x30E8 }, { 448, 0x30E9 }, { 449, 0x30EA }, { 450, 0x30EB }, { 451, 0x30EC },
    { 452, 0x30ED }, { 453, 0x30EE }, { 454, 0x30EF }, { 455, 0x30F0 }, { 456, 0x30F1 }, { 457, 0x30F2 }, { 458, 0x30F3 }, { 459, 0x30F4 },
    { 460, 0x30F5 }, { 461, 0x30F6 }, { 470, 0x0391 }, { 471, 0x0392 }, { 472, 0x0393 }, { 473, 0x0394 }, { 474, 0x0395 }, { 475, 0x0396 },
    { 476, 0x0397 }, { 477, 0x0398 }, { 478, 0x0399 }, { 479, 0x039A }, { 480, 0x039B }, { 481, 0x039C }, { 482, 0x039D }, { 483, 0x039E },
    { 484, 0x039F }, { 485, 0x03A0 }, { 486, 0x03A1 }, { 487, 0x03A3 }, { 488, 0x03A4 }, { 489, 0x03A5 }, { 490, 0x03A6 }, { 491, 0x03A7 },
    { 492, 0x03A8 }, { 493, 0x03A9 }, { 502, 0x03B1 }, { 503, 0x03B2 }, { 504, 0x03B3 }, { 505, 0x03B4 }, { 506, 0x03B5 }, { 507, 0x03B6 },
    { 508, 0x03B7 }, { 509, 0x03B8 }, { 510, 0x03B9 }, { 511, 0x03BA }, { 512, 0x03BB }, { 513, 0x03BC }, { 514, 0x03BD }, { 515, 0x03BE },
    { 516, 0x03BF }, { 517, 0x03C0 }, { 518, 0x03C1 }, { 519, 0x03C3 }, { 520, 0x03C4 }, { 521, 0x03C5 }, { 522, 0x03C6 }, { 523, 0x03C7 },
    { 524, 0x03C8 }, { 525, 0x03C9 }, { 564, 0x0410 }, { 565, 0x0411 }, { 566, 0x0412 }, { 567, 0x0413 }, { 568, 0x0414 }, { 569, 0x0415 },
    { 570, 0x0401 }, { 571, 0x0416 }, { 572, 0x0417 }, { 573, 0x0418 }, { 574, 0x0419 }, { 575, 0x041A }, { 576, 0x041B }, { 577, 0x041C },
    { 578, 0x041D }, { 579, 0x041E }, { 580, 0x041F }, { 581, 0x0420 }, { 582, 0x0421 }, { 583, 0x0422 }, { 584, 0x0423 }, { 585, 0x0424 },
    { 586, 0x0425 }, { 587, 0x0426 }, { 588, 0x0427 }, { 589, 0x0428 }, { 590, 0x0429 }, { 591, 0x042A }, { 592, 0x042B }, { 593, 0x042C },
    { 594, 0x042D }, { 595, 0x042E }, { 596, 0x042F }, { 612, 0x0430 }, { 613, 0x0431 }, { 614, 0x0432 }, { 615, 0x0433 }, { 616, 0x0434 },
    { 617, 0x0435 }, { 618, 0x0451 }, { 619, 0x0436 }, { 620, 0x0437 }, { 621, 0x0438 }, { 622, 0x0439 }, { 623, 0x043A }, { 624, 0x043B },
    { 625, 0x043C }, { 626, 0x043D }, { 627, 0x043E }, { 628, 0x043F }, { 629, 0x0440 }, { 630, 0x0441 }, { 631, 0x0442 }, { 632, 0x0443 },
    { 633, 0x0444 }, { 634, 0x0445 }, { 635, 0x0446 }, { 636, 0x0447 }, { 637, 0x0448 }, { 638, 0x0449 }, { 639, 0x044A }, { 640, 0x044B },
    { 641, 0x044C }, { 642, 0x044D }, { 643, 0x044E }, { 644, 0x044F }, { 658, 0x2500 }, { 659, 0x2502 }, { 660, 0x250C }, { 661, 0x2510 },
    { 662, 0x2518 }, { 663, 0x2514 }, { 664, 0x251C }, { 665, 0x252C }, { 666, 0x2524 }, { 667, 0x2534 }, { 668, 0x253C }, { 669, 0x2501 },
    { 670, 0x2503 }, { 671, 0x250F }, { 672, 0x2513 }, { 673, 0x251B }, { 674, 0x2517 }, { 675, 0x2523 }, { 676, 0x2533 }, { 677, 0x252B },
    { 678, 0x253B }, { 679, 0x254B }, { 680, 0x2520 }, { 681, 0x252F }, { 682, 0x2528 }, { 683, 0x2537 }, { 684, 0x253F }, { 685, 0x251D },
    { 686, 0x2530 }, { 687, 0x2525 }, { 688, 0x2538 }, { 689, 0x2542 }, { 1128, 0x2460 }, { 1129, 0x2461 }, { 1130, 0x2462 }, { 1131, 0x2463 },
    { 1132, 0x2464 }, { 1133, 0x2465 }, { 1134, 0x2466 }, { 1135, 0x2467 }, { 1136, 0x2468 }, { 1137, 0x2469 }, { 1138, 0x246A }, { 1139, 0x246B },
    { 1140, 0x246C }, { 1141, 0x246D }, { 1142, 0x246E }, { 1143, 0x246F }, { 1144, 0x2470 }, { 1145, 0x2471 }, { 1146, 0x2472 }, { 1147, 0x2473 },
    { 1148, 0x2160 }, { 1149, 0x2161 }, { 1150, 0x2162 }, { 1151, 0x2163 }, { 1152, 0x2164 }, { 1153, 0x2165 }, { 1154, 0x2166 }, { 1155, 0x2167 },
    { 1156, 0x2168 }, { 1157, 0x2169 }, { 1159, 0x3349 }, { 1160, 0x3314 }, { 1161, 0x3322 }, { 1162, 0x334D }, { 1163, 0x3318 }, { 1164, 0x3327 },
    { 1165, 0x3303 }, { 1166, 0x3336 }, { 1167, 0x3351 }, { 1168, 0x3357 }, { 1169, 0x330D }, { 1170, 0x3326 }, { 1171, 0x3323 }, { 1172, 0x332B },
    { 1173, 0x334A }, { 1174, 0x333B }, { 1175, 0x339C }, { 1176, 0x339D }, { 1177, 0x339E }, { 1178, 0x338E }, { 1179, 0x338F }, { 1180, 0x33C4 },
    { 1181, 0x33A1 }, { 1190, 0x337B }, { 1191, 0x301D }, { 1192, 0x301F }, { 1193, 0x2116 }, { 1194, 0x33CD }, { 1195, 0x2121 }, { 1196, 0x32A4 },
    { 1197, 0x32A5 }, { 1198, 0x32A6 }, { 1199, 0x32A7 }, { 1200, 0x32A8 }, { 1201, 0x3231 }, { 1202, 0x3232 }, { 1203, 0x3239 }, { 1204, 0x337E },
    { 1205, 0x337D }, { 1206, 0x337C }, { 1207, 0x2252 }, { 1208, 0x2261 }, { 1209, 0x222B }, { 1210, 0x222E }, { 1211, 0x2211 }, { 1212, 0x221A },
    { 1213, 0x22A5 }, { 1214, 0x2220 }, { 1215, 0x221F }, { 1216, 0x22BF }, { 1217, 0x2235 }, { 1218, 0x2229 }, { 1219, 0x222A }, { 1410, 0x4E9C },
    { 1411, 0x5516 }, { 1412, 0x5A03 }, { 1413, 0x963F }, { 1414, 0x54C0 }, { 1415, 0x611B }, { 1416, 0x6328 }, { 1417, 0x59F6 }, { 1418, 0x9022 },
    { 1419, 0x8475 }, { 1420, 0x831C }, { 1421, 0x7A50 }, { 1422, 0x60AA }, { 1423, 0x63E1 }, { 1424, 0x6E25 }, { 1425, 0x65ED }, { 1426, 0x8466 },
    { 1427, 0x82A6 }, { 1428, 0x9BF5 }, { 1429, 0x6893 }, { 1430, 0x5727 }, { 1431, 0x65A1 }, { 1432, 0x6271 }, { 1433, 0x5B9B }, { 1434, 0x59D0 },
    { 1435, 0x867B }, { 1436, 0x98F4 }, { 1437, 0x7D62 }, { 1438, 0x7DBE }, { 1439, 0x9B8E }, { 1440, 0x6216 }, { 1441, 0x7C9F }, { 1442, 0x88B7 },
    { 1443, 0x5B89 }, { 1444, 0x5EB5 }, { 1445, 0x6309 }, { 1446, 0x6697 }, { 1447, 0x6848 }, { 1448, 0x95C7 }, { 1449, 0x978D }, { 1450, 0x674F },
    { 1451, 0x4EE5 }, { 1452, 0x4F0A }, { 1453, 0x4F4D }, { 1454, 0x4F9D }, { 1455, 0x5049 }, { 1456, 0x56F2 }, { 1457, 0x5937 }, { 1458, 0x59D4 },
    { 1459, 0x5A01 }, { 1460, 0x5C09 }, { 1461, 0x60DF }, { 1462, 0x610F }, { 1463, 0x6170 }, { 1464, 0x6613 }, { 1465, 0x6905 }, { 1466, 0x70BA },
    { 1467, 0x754F }, { 1468, 0x7570 }, { 1469, 0x79FB }, { 1470, 0x7DAD }, { 1471, 0x7DEF }, { 1472, 0x80C3 }, { 1473, 0x840E }, { 1474, 0x8863 },
    { 1475, 0x8B02 }, { 1476, 0x9055 }, { 1477, 0x907A }, { 1478, 0x533B }, { 1479, 0x4E95 }, { 1480, 0x4EA5 }, { 1481, 0x57DF }, { 1482, 0x80B2 },
    { 1483, 0x90C1 }, { 1484, 0x78EF }, { 1485, 0x4E00 }, { 1486, 0x58F1 }, { 1487, 0x6EA2 }, { 1488, 0x9038 }, { 1489, 0x7A32 }, { 1490, 0x8328 },
    { 1491, 0x828B }, { 1492, 0x9C2F }, { 1493, 0x5141 }, { 1494, 0x5370 }, { 1495, 0x54BD }, { 1496, 0x54E1 }, { 1497, 0x56E0 }, { 1498, 0x59FB },
    { 1499, 0x5F15 }, { 1500, 0x98F2 }, { 1501, 0x6DEB }, { 1502, 0x80E4 }, { 1503, 0x852D }, { 1504, 0x9662 }, { 1505, 0x9670 }, { 1506, 0x96A0 },
    { 1507, 0x97FB }, { 1508, 0x540B }, { 1509, 0x53F3 }, { 1510, 0x5B87 }, { 1511, 0x70CF }, { 1512, 0x7FBD }, { 1513, 0x8FC2 }, { 1514, 0x96E8 },
    { 1515, 0x536F }, { 1516, 0x9D5C }, { 1517, 0x7ABA }, { 1518, 0x4E11 }, { 1519, 0x7893 }, { 1520, 0x81FC }, { 1521, 0x6E26 }, { 1522, 0x5618 },
    { 1523, 0x5504 }, { 1524, 0x6B1D }, { 1525, 0x851A }, { 1526, 0x9C3B }, { 1527, 0x59E5 }, { 1528, 0x53A9 }, { 1529, 0x6D66 }, { 1530, 0x74DC },
    { 1531, 0x958F }, { 1532, 0x5642 }, { 1533, 0x4E91 }, { 1534, 0x904B }, { 1535, 0x96F2 }, { 1536, 0x834F }, { 1537, 0x990C }, { 1538, 0x53E1 },
    { 1539, 0x55B6 }, { 1540, 0x5B30 }, { 1541, 0x5F71 }, { 1542, 0x6620 }, { 1543, 0x66F3 }, { 1544, 0x6804 }, { 1545, 0x6C38 }, { 1546, 0x6CF3 },
    { 1547, 0x6D29 }, { 1548, 0x745B }, { 1549, 0x76C8 }, { 1550, 0x7A4E }, { 1551, 0x9834 }, { 1552, 0x82F1 }, { 1553, 0x885B }, { 1554, 0x8A60 },
    { 1555, 0x92ED }, { 1556, 0x6DB2 }, { 1557, 0x75AB }, { 1558, 0x76CA }, { 1559, 0x99C5 }, { 1560, 0x60A6 }, { 1561, 0x8B01 }, { 1562, 0x8D8A },
    { 1563, 0x95B2 }, { 1564, 0x698E }, { 1565, 0x53AD }, { 1566, 0x5186 }, { 1567, 0x5712 }, { 1568, 0x5830 }, { 1569, 0x5944 }, { 1570, 0x5BB4 },
    { 1571, 0x5EF6 }, { 1572, 0x6028 }, { 1573, 0x63A9 }, { 1574, 0x63F4 }, { 1575, 0x6CBF }, { 1576, 0x6F14 }, { 1577, 0x708E }, { 1578, 0x7114 },
    { 1579, 0x7159 }, { 1580, 0x71D5 }, { 1581, 0x733F }, { 1582, 0x7E01 }, { 1583, 0x8276 }, { 1584, 0x82D1 }, { 1585, 0x8597 }, { 1586, 0x9060 },
    { 1587, 0x925B }, { 1588, 0x9D1B }, { 1589, 0x5869 }, { 1590, 0x65BC }, { 1591, 0x6C5A }, { 1592, 0x7525 }, { 1593, 0x51F9 }, { 1594, 0x592E },
    { 1595, 0x5965 }, { 1596, 0x5F80 }, { 1597, 0x5FDC }, { 1598, 0x62BC }, { 1599, 0x65FA }, { 1600, 0x6A2A }, { 1601, 0x6B27 }, { 1602, 0x6BB4 },
    { 1603, 0x738B }, { 1604, 0x7FC1 }, { 1605, 0x8956 }, { 1606, 0x9D2C }, { 1607, 0x9D0E }, { 1608, 0x9EC4 }, { 1609, 0x5CA1 }, { 1610, 0x6C96 },
    { 1611, 0x837B }, { 1612, 0x5104 }, { 1613, 0x5C4B }, { 1614, 0x61B6 }, { 1615, 0x81C6 }, { 1616, 0x6876 }, { 1617, 0x7261 }, { 1618, 0x4E59 },
    { 1619, 0x4FFA }, { 1620, 0x5378 }, { 1621, 0x6069 }, { 1622, 0x6E29 }, { 1623, 0x7A4F }, { 1624, 0x97F3 }, { 1625, 0x4E0B }, { 1626, 0x5316 },
    { 1627, 0x4EEE }, { 1628, 0x4F55 }, { 1629, 0x4F3D }, { 1630, 0x4FA1 }, { 1631, 0x4F73 }, { 1632, 0x52A0 }, { 1633, 0x53EF }, { 1634, 0x5609 },
    { 1635, 0x590F }, { 1636, 0x5AC1 }, { 1637, 0x5BB6 }, { 1638, 0x5BE1 }, { 1639, 0x79D1 }, { 1640, 0x6687 }, { 1641, 0x679C }, { 1642, 0x67B6 },
    { 1643, 0x6B4C }, { 1644, 0x6CB3 }, { 1645, 0x706B }, { 1646, 0x73C2 }, { 1647, 0x798D }, { 1648, 0x79BE }, { 1649, 0x7A3C }, { 1650, 0x7B87 },
    { 1651, 0x82B1 }, { 1652, 0x82DB }, { 1653, 0x8304 }, { 1654, 0x8377 }, { 1655, 0x83EF }, { 1656, 0x83D3 }, { 1657, 0x8766 }, { 1658, 0x8AB2 },
    { 1659, 0x5629 }, { 1660, 0x8CA8 }, { 1661, 0x8FE6 }, { 1662, 0x904E }, { 1663, 0x971E }, { 1664, 0x868A }, { 1665, 0x4FC4 }, { 1666, 0x5CE8 },
    { 1667, 0x6211 }, { 1668, 0x7259 }, { 1669, 0x753B }, { 1670, 0x81E5 }, { 1671, 0x82BD }, { 1672, 0x86FE }, { 1673, 0x8CC0 }, { 1674, 0x96C5 },
    { 1675, 0x9913 }, { 1676, 0x99D5 }, { 1677, 0x4ECB }, { 1678, 0x4F1A }, { 1679, 0x89E3 }, { 1680, 0x56DE }, { 1681, 0x584A }, { 1682, 0x58CA },
    { 1683, 0x5EFB }, { 1684, 0x5FEB }, { 1685, 0x602A }, { 1686, 0x6094 }, { 1687, 0x6062 }, { 1688, 0x61D0 }, { 1689, 0x6212 }, { 1690, 0x62D0 },
    { 1691, 0x6539 }, { 1692, 0x9B41 }, { 1693, 0x6666 }, { 1694, 0x68B0 }, { 1695, 0x6D77 }, { 1696, 0x7070 }, { 1697, 0x754C }, { 1698, 0x7686 },
    { 1699, 0x7D75 }, { 1700, 0x82A5 }, { 1701, 0x87F9 }, { 1702, 0x958B }, { 1703, 0x968E }, { 1704, 0x8C9D }, { 1705, 0x51F1 }, { 1706, 0x52BE },
    { 1707, 0x5916 }, { 1708, 0x54B3 }, { 1709, 0x5BB3 }, { 1710, 0x5D16 }, { 1711, 0x6168 }, { 1712, 0x6982 }, { 1713, 0x6DAF }, { 1714, 0x788D },
    { 1715, 0x84CB }, { 1716, 0x8857 }, { 1717, 0x8A72 }, { 1718, 0x93A7 }, { 1719, 0x9AB8 }, { 1720, 0x6D6C }, { 1721, 0x99A8 }, { 1722, 0x86D9 },
    { 1723, 0x57A3 }, { 1724, 0x67FF }, { 1725, 0x86CE }, { 1726, 0x920E }, { 1727, 0x5283 }, { 1728, 0x5687 }, { 1729, 0x5404 }, { 1730, 0x5ED3 },
    { 1731, 0x62E1 }, { 1732, 0x64B9 }, { 1733, 0x683C }, { 1734, 0x6838 }, { 1735, 0x6BBB }, { 1736, 0x7372 }, { 1737, 0x78BA }, { 1738, 0x7A6B },
    { 1739, 0x899A }, { 1740, 0x89D2 }, { 1741, 0x8D6B }, { 1742, 0x8F03 }, { 1743, 0x90ED }, { 1744, 0x95A3 }, { 1745, 0x9694 }, { 1746, 0x9769 },
    { 1747, 0x5B66 }, { 1748, 0x5CB3 }, { 1749, 0x697D }, { 1750, 0x984D }, { 1751, 0x984E }, { 1752, 0x639B }, { 1753, 0x7B20 }, { 1754, 0x6A2B },
    { 1755, 0x6A7F }, { 1756, 0x68B6 }, { 1757, 0x9C0D }, { 1758, 0x6F5F }, { 1759, 0x5272 }, { 1760, 0x559D }, { 1761, 0x6070 }, { 1762, 0x62EC },
    { 1763, 0x6D3B }, { 1764, 0x6E07 }, { 1765, 0x6ED1 }, { 1766, 0x845B }, { 1767, 0x8910 }, { 1768, 0x8F44 }, { 1769, 0x4E14 }, { 1770, 0x9C39 },
    { 1771, 0x53F6 }, { 1772, 0x691B }, { 1773, 0x6A3A }, { 1774, 0x9784 }, { 1775, 0x682A }, { 1776, 0x515C }, { 1777, 0x7AC3 }, { 1778, 0x84B2 },
    { 1779, 0x91DC }, { 1780, 0x938C }, { 1781, 0x565B }, { 1782, 0x9D28 }, { 1783, 0x6822 }, { 1784, 0x8305 }, { 1785, 0x8431 }, { 1786, 0x7CA5 },
    { 1787, 0x5208 }, { 1788, 0x82C5 }, { 1789, 0x74E6 }, { 1790, 0x4E7E }, { 1791, 0x4F83 }, { 1792, 0x51A0 }, { 1793, 0x5BD2 }, { 1794, 0x520A },
    { 1795, 0x52D8 }, { 1796, 0x52E7 }, { 1797, 0x5DFB }, { 1798, 0x559A }, { 1799, 0x582A }, { 1800, 0x59E6 }, { 1801, 0x5B8C }, { 1802, 0x5B98 },
    { 1803, 0x5BDB }, { 1804, 0x5E72 }, { 1805, 0x5E79 }, { 1806, 0x60A3 }, { 1807, 0x611F }, { 1808, 0x6163 }, { 1809, 0x61BE }, { 1810, 0x63DB },
    { 1811, 0x6562 }, { 1812, 0x67D1 }, { 1813, 0x6853 }, { 1814, 0x68FA }, { 1815, 0x6B3E }, { 1816, 0x6B53 }, { 1817, 0x6C57 }, { 1818, 0x6F22 },
    { 1819, 0x6F97 }, { 1820, 0x6F45 }, { 1821, 0x74B0 }, { 1822, 0x7518 }, { 1823, 0x76E3 }, { 1824, 0x770B }, { 1825, 0x7AFF }, { 1826, 0x7BA1 },
    { 1827, 0x7C21 }, { 1828, 0x7DE9 }, { 1829, 0x7F36 }, { 1830, 0x7FF0 }, { 1831, 0x809D }, { 1832, 0x8266 }, { 1833, 0x839E }, { 1834, 0x89B3 },
    { 1835, 0x8ACC }, { 1836, 0x8CAB }, { 1837, 0x9084 }, { 1838, 0x9451 }, { 1839, 0x9593 }, { 1840, 0x9591 }, { 1841, 0x95A2 }, { 1842, 0x9665 },
    { 1843, 0x97D3 }, { 1844, 0x9928 }, { 1845, 0x8218 }, { 1846, 0x4E38 }, { 1847, 0x542B }, { 1848, 0x5CB8 }, { 1849, 0x5DCC }, { 1850, 0x73A9 },
    { 1851, 0x764C }, { 1852, 0x773C }, { 1853, 0x5CA9 }, { 1854, 0x7FEB }, { 1855, 0x8D0B }, { 1856, 0x96C1 }, { 1857, 0x9811 }, { 1858, 0x9854 },
    { 1859, 0x9858 }, { 1860, 0x4F01 }, { 1861, 0x4F0E }, { 1862, 0x5371 }, { 1863, 0x559C }, { 1864, 0x5668 }, { 1865, 0x57FA }, { 1866, 0x5947 },
    { 1867, 0x5B09 }, { 1868, 0x5BC4 }, { 1869, 0x5C90 }, { 1870, 0x5E0C }, { 1871, 0x5E7E }, { 1872, 0x5FCC }, { 1873, 0x63EE }, { 1874, 0x673A },
    { 1875, 0x65D7 }, { 1876, 0x65E2 }, { 1877, 0x671F }, { 1878, 0x68CB }, { 1879, 0x68C4 }, { 1880, 0x6A5F }, { 1881, 0x5E30 }, { 1882, 0x6BC5 },
    { 1883, 0x6C17 }, { 1884, 0x6C7D }, { 1885, 0x757F }, { 1886, 0x7948 }, { 1887, 0x5B63 }, { 1888, 0x7A00 }, { 1889, 0x7D00 }, { 1890, 0x5FBD },
    { 1891, 0x898F }, { 1892, 0x8A18 }, { 1893, 0x8CB4 }, { 1894, 0x8D77 }, { 1895, 0x8ECC }, { 1896, 0x8F1D }, { 1897, 0x98E2 }, { 1898, 0x9A0E },
    { 1899, 0x9B3C }, { 1900, 0x4E80 }, { 1901, 0x507D }, { 1902, 0x5100 }, { 1903, 0x5993 }, { 1904, 0x5B9C }, { 1905, 0x622F }, { 1906, 0x6280 },
    { 1907, 0x64EC }, { 1908, 0x6B3A }, { 1909, 0x72A0 }, { 1910, 0x7591 }, { 1911, 0x7947 }, { 1912, 0x7FA9 }, { 1913, 0x87FB }, { 1914, 0x8ABC },
    { 1915, 0x8B70 }, { 1916, 0x63AC }, { 1917, 0x83CA }, { 1918, 0x97A0 }, { 1919, 0x5409 }, { 1920, 0x5403 }, { 1921, 0x55AB }, { 1922, 0x6854 },
    { 1923, 0x6A58 }, { 1924, 0x8A70 }, { 1925, 0x7827 }, { 1926, 0x6775 }, { 1927, 0x9ECD }, { 1928, 0x5374 }, { 1929, 0x5BA2 }, { 1930, 0x811A },
    { 1931, 0x8650 }, { 1932, 0x9006 }, { 1933, 0x4E18 }, { 1934, 0x4E45 }, { 1935, 0x4EC7 }, { 1936, 0x4F11 }, { 1937, 0x53CA }, { 1938, 0x5438 },
    { 1939, 0x5BAE }, { 1940, 0x5F13 }, { 1941, 0x6025 }, { 1942, 0x6551 }, { 1943, 0x673D }, { 1944, 0x6C42 }, { 1945, 0x6C72 }, { 1946, 0x6CE3 },
    { 1947, 0x7078 }, { 1948, 0x7403 }, { 1949, 0x7A76 }, { 1950, 0x7AAE }, { 1951, 0x7B08 }, { 1952, 0x7D1A }, { 1953, 0x7CFE }, { 1954, 0x7D66 },
    { 1955, 0x65E7 }, { 1956, 0x725B }, { 1957, 0x53BB }, { 1958, 0x5C45 }, { 1959, 0x5DE8 }, { 1960, 0x62D2 }, { 1961, 0x62E0 }, { 1962, 0x6319 },
    { 1963, 0x6E20 }, { 1964, 0x865A }, { 1965, 0x8A31 }, { 1966, 0x8DDD }, { 1967, 0x92F8 }, { 1968, 0x6F01 }, { 1969, 0x79A6 }, { 1970, 0x9B5A },
    { 1971, 0x4EA8 }, { 1972, 0x4EAB }, { 1973, 0x4EAC }, { 1974, 0x4F9B }, { 1975, 0x4FA0 }, { 1976, 0x50D1 }, { 1977, 0x5147 }, { 1978, 0x7AF6 },
    { 1979, 0x5171 }, { 1980, 0x51F6 }, { 1981, 0x5354 }, { 1982, 0x5321 }, { 1983, 0x537F }, { 1984, 0x53EB }, { 1985, 0x55AC }, { 1986, 0x5883 },
    { 1987, 0x5CE1 }, { 1988, 0x5F37 }, { 1989, 0x5F4A }, { 1990, 0x602F }, { 1991, 0x6050 }, { 1992, 0x606D }, { 1993, 0x631F }, { 1994, 0x6559 },
    { 1995, 0x6A4B }, { 1996, 0x6CC1 }, { 1997, 0x72C2 }, { 1998, 0x72ED }, { 1999, 0x77EF }, { 2000, 0x80F8 }, { 2001, 0x8105 }, { 2002, 0x8208 },
    { 2003, 0x854E }, { 2004, 0x90F7 }, { 2005, 0x93E1 }, { 2006, 0x97FF }, { 2007, 0x9957 }, { 2008, 0x9A5A }, { 2009, 0x4EF0 }, { 2010, 0x51DD },
    { 2011, 0x5C2D }, { 2012, 0x6681 }, { 2013, 0x696D }, { 2014, 0x5C40 }, { 2015, 0x66F2 }, { 2016, 0x6975 }, { 2017, 0x7389 }, { 2018, 0x6850 },
    { 2019, 0x7C81 }, { 2020, 0x50C5 }, { 2021, 0x52E4 }, { 2022, 0x5747 }, { 2023, 0x5DFE }, { 2024, 0x9326 }, { 2025, 0x65A4 }, { 2026, 0x6B23 },
    { 2027, 0x6B3D }, { 2028, 0x7434 }, { 2029, 0x7981 }, { 2030, 0x79BD }, { 2031, 0x7B4B }, { 2032, 0x7DCA }, { 2033, 0x82B9 }, { 2034, 0x83CC },
    { 2035, 0x887F }, { 2036, 0x895F }, { 2037, 0x8B39 }, { 2038, 0x8FD1 }, { 2039, 0x91D1 }, { 2040, 0x541F }, { 2041, 0x9280 }, { 2042, 0x4E5D },
    { 2043, 0x5036 }, { 2044, 0x53E5 }, { 2045, 0x533A }, { 2046, 0x72D7 }, { 2047, 0x7396 }, { 2048, 0x77E9 }, { 2049, 0x82E6 }, { 2050, 0x8EAF },
    { 2051, 0x99C6 }, { 2052, 0x99C8 }, { 2053, 0x99D2 }, { 2054, 0x5177 }, { 2055, 0x611A }, { 2056, 0x865E }, { 2057, 0x55B0 }, { 2058, 0x7A7A },
    { 2059, 0x5076 }, { 2060, 0x5BD3 }, { 2061, 0x9047 }, { 2062, 0x9685 }, { 2063, 0x4E32 }, { 2064, 0x6ADB }, { 2065, 0x91E7 }, { 2066, 0x5C51 },
    { 2067, 0x5C48 }, { 2068, 0x6398 }, { 2069, 0x7A9F }, { 2070, 0x6C93 }, { 2071, 0x9774 }, { 2072, 0x8F61 }, { 2073, 0x7AAA }, { 2074, 0x718A },
    { 2075, 0x9688 }, { 2076, 0x7C82 }, { 2077, 0x6817 }, { 2078, 0x7E70 }, { 2079, 0x6851 }, { 2080, 0x936C }, { 2081, 0x52F2 }, { 2082, 0x541B },
    { 2083, 0x85AB }, { 2084, 0x8A13 }, { 2085, 0x7FA4 }, { 2086, 0x8ECD }, { 2087, 0x90E1 }, { 2088, 0x5366 }, { 2089, 0x8888 }, { 2090, 0x7941 },
    { 2091, 0x4FC2 }, { 2092, 0x50BE }, { 2093, 0x5211 }, { 2094, 0x5144 }, { 2095, 0x5553 }, { 2096, 0x572D }, { 2097, 0x73EA }, { 2098, 0x578B },
    { 2099, 0x5951 }, { 2100, 0x5F62 }, { 2101, 0x5F84 }, { 2102, 0x6075 }, { 2103, 0x6176 }, { 2104, 0x6167 }, { 2105, 0x61A9 }, { 2106, 0x63B2 },
    { 2107, 0x643A }, { 2108, 0x656C }, { 2109, 0x666F }, { 2110, 0x6842 }, { 2111, 0x6E13 }, { 2112, 0x7566 }, { 2113, 0x7A3D }, { 2114, 0x7CFB },
    { 2115, 0x7D4C }, { 2116, 0x7D99 }, { 2117, 0x7E4B }, { 2118, 0x7F6B }, { 2119, 0x830E }, { 2120, 0x834A }, { 2121, 0x86CD }, { 2122, 0x8A08 },
    { 2123, 0x8A63 }, { 2124, 0x8B66 }, { 2125, 0x8EFD }, { 2126, 0x981A }, { 2127, 0x9D8F }, { 2128, 0x82B8 }, { 2129, 0x8FCE }, { 2130, 0x9BE8 },
    { 2131, 0x5287 }, { 2132, 0x621F }, { 2133, 0x6483 }, { 2134, 0x6FC0 }, { 2135, 0x9699 }, { 2136, 0x6841 }, { 2137, 0x5091 }, { 2138, 0x6B20 },
    { 2139, 0x6C7A }, { 2140, 0x6F54 }, { 2141, 0x7A74 }, { 2142, 0x7D50 }, { 2143, 0x8840 }, { 2144, 0x8A23 }, { 2145, 0x6708 }, { 2146, 0x4EF6 },
    { 2147, 0x5039 }, { 2148, 0x5026 }, { 2149, 0x5065 }, { 2150, 0x517C }, { 2151, 0x5238 }, { 2152, 0x5263 }, { 2153, 0x55A7 }, { 2154, 0x570F },
    { 2155, 0x5805 }, { 2156, 0x5ACC }, { 2157, 0x5EFA }, { 2158, 0x61B2 }, { 2159, 0x61F8 }, { 2160, 0x62F3 }, { 2161, 0x6372 }, { 2162, 0x691C },
    { 2163, 0x6A29 }, { 2164, 0x727D }, { 2165, 0x72AC }, { 2166, 0x732E }, { 2167, 0x7814 }, { 2168, 0x786F }, { 2169, 0x7D79 }, { 2170, 0x770C },
    { 2171, 0x80A9 }, { 2172, 0x898B }, { 2173, 0x8B19 }, { 2174, 0x8CE2 }, { 2175, 0x8ED2 }, { 2176, 0x9063 }, { 2177, 0x9375 }, { 2178, 0x967A },
    { 2179, 0x9855 }, { 2180, 0x9A13 }, { 2181, 0x9E78 }, { 2182, 0x5143 }, { 2183, 0x539F }, { 2184, 0x53B3 }, { 2185, 0x5E7B }, { 2186, 0x5F26 },
    { 2187, 0x6E1B }, { 2188, 0x6E90 }, { 2189, 0x7384 }, { 2190, 0x73FE }, { 2191, 0x7D43 }, { 2192, 0x8237 }, { 2193, 0x8A00 }, { 2194, 0x8AFA },
    { 2195, 0x9650 }, { 2196, 0x4E4E }, { 2197, 0x500B }, { 2198, 0x53E4 }, { 2199, 0x547C }, { 2200, 0x56FA }, { 2201, 0x59D1 }, { 2202, 0x5B64 },
    { 2203, 0x5DF1 }, { 2204, 0x5EAB }, { 2205, 0x5F27 }, { 2206, 0x6238 }, { 2207, 0x6545 }, { 2208, 0x67AF }, { 2209, 0x6E56 }, { 2210, 0x72D0 },
    { 2211, 0x7CCA }, { 2212, 0x88B4 }, { 2213, 0x80A1 }, { 2214, 0x80E1 }, { 2215, 0x83F0 }, { 2216, 0x864E }, { 2217, 0x8A87 }, { 2218, 0x8DE8 },
    { 2219, 0x9237 }, { 2220, 0x96C7 }, { 2221, 0x9867 }, { 2222, 0x9F13 }, { 2223, 0x4E94 }, { 2224, 0x4E92 }, { 2225, 0x4F0D }, { 2226, 0x5348 },
    { 2227, 0x5449 }, { 2228, 0x543E }, { 2229, 0x5A2F }, { 2230, 0x5F8C }, { 2231, 0x5FA1 }, { 2232, 0x609F }, { 2233, 0x68A7 }, { 2234, 0x6A8E },
    { 2235, 0x745A }, { 2236, 0x7881 }, { 2237, 0x8A9E }, { 2238, 0x8AA4 }, { 2239, 0x8B77 }, { 2240, 0x9190 }, { 2241, 0x4E5E }, { 2242, 0x9BC9 },
    { 2243, 0x4EA4 }, { 2244, 0x4F7C }, { 2245, 0x4FAF }, { 2246, 0x5019 }, { 2247, 0x5016 }, { 2248, 0x5149 }, { 2249, 0x516C }, { 2250, 0x529F },
    { 2251, 0x52B9 }, { 2252, 0x52FE }, { 2253, 0x539A }, { 2254, 0x53E3 }, { 2255, 0x5411 }, { 2256, 0x540E }, { 2257, 0x5589 }, { 2258, 0x5751 },
    { 2259, 0x57A2 }, { 2260, 0x597D }, { 2261, 0x5B54 }, { 2262, 0x5B5D }, { 2263, 0x5B8F }, { 2264, 0x5DE5 }, { 2265, 0x5DE7 }, { 2266, 0x5DF7 },
    { 2267, 0x5E78 }, { 2268, 0x5E83 }, { 2269, 0x5E9A }, { 2270, 0x5EB7 }, { 2271, 0x5F18 }, { 2272, 0x6052 }, { 2273, 0x614C }, { 2274, 0x6297 },
    { 2275, 0x62D8 }, { 2276, 0x63A7 }, { 2277, 0x653B }, { 2278, 0x6602 }, { 2279, 0x6643 }, { 2280, 0x66F4 }, { 2281, 0x676D }, { 2282, 0x6821 },
    { 2283, 0x6897 }, { 2284, 0x69CB }, { 2285, 0x6C5F }, { 2286, 0x6D2A }, { 2287, 0x6D69 }, { 2288, 0x6E2F }, { 2289, 0x6E9D }, { 2290, 0x7532 },
    { 2291, 0x7687 }, { 2292, 0x786C }, { 2293, 0x7A3F }, { 2294, 0x7CE0 }, { 2295, 0x7D05 }, { 2296, 0x7D18 }, { 2297, 0x7D5E }, { 2298, 0x7DB1 },
    { 2299, 0x8015 }, { 2300, 0x8003 }, { 2301, 0x80AF }, { 2302, 0x80B1 }, { 2303, 0x8154 }, { 2304, 0x818F }, { 2305, 0x822A }, { 2306, 0x8352 },
    { 2307, 0x884C }, { 2308, 0x8861 }, { 2309, 0x8B1B }, { 2310, 0x8CA2 }, { 2311, 0x8CFC }, { 2312, 0x90CA }, { 2313, 0x9175 }, { 2314, 0x9271 },
    { 2315, 0x783F }, { 2316, 0x92FC }, { 2317, 0x95A4 }, { 2318, 0x964D }, { 2319, 0x9805 }, { 2320, 0x9999 }, { 2321, 0x9AD8 }, { 2322, 0x9D3B },
    { 2323, 0x525B }, { 2324, 0x52AB }, { 2325, 0x53F7 }, { 2326, 0x5408 }, { 2327, 0x58D5 }, { 2328, 0x62F7 }, { 2329, 0x6FE0 }, { 2330, 0x8C6A },
    { 2331, 0x8F5F }, { 2332, 0x9EB9 }, { 2333, 0x514B }, { 2334, 0x523B }, { 2335, 0x544A }, { 2336, 0x56FD }, { 2337, 0x7A40 }, { 2338, 0x9177 },
    { 2339, 0x9D60 }, { 2340, 0x9ED2 }, { 2341, 0x7344 }, { 2342, 0x6F09 }, { 2343, 0x8170 }, { 2344, 0x7511 }, { 2345, 0x5FFD }, { 2346, 0x60DA },
    { 2347, 0x9AA8 }, { 2348, 0x72DB }, { 2349, 0x8FBC }, { 2350, 0x6B64 }, { 2351, 0x9803 }, { 2352, 0x4ECA }, { 2353, 0x56F0 }, { 2354, 0x5764 },
    { 2355, 0x58BE }, { 2356, 0x5A5A }, { 2357, 0x6068 }, { 2358, 0x61C7 }, { 2359, 0x660F }, { 2360, 0x6606 }, { 2361, 0x6839 }, { 2362, 0x68B1 },
    { 2363, 0x6DF7 }, { 2364, 0x75D5 }, { 2365, 0x7D3A }, { 2366, 0x826E }, { 2367, 0x9B42 }, { 2368, 0x4E9B }, { 2369, 0x4F50 }, { 2370, 0x53C9 },
    { 2371, 0x5506 }, { 2372, 0x5D6F }, { 2373, 0x5DE6 }, { 2374, 0x5DEE }, { 2375, 0x67FB }, { 2376, 0x6C99 }, { 2377, 0x7473 }, { 2378, 0x7802 },
    { 2379, 0x8A50 }, { 2380, 0x9396 }, { 2381, 0x88DF }, { 2382, 0x5750 }, { 2383, 0x5EA7 }, { 2384, 0x632B }, { 2385, 0x50B5 }, { 2386, 0x50AC },
    { 2387, 0x518D }, { 2388, 0x6700 }, { 2389, 0x54C9 }, { 2390, 0x585E }, { 2391, 0x59BB }, { 2392, 0x5BB0 }, { 2393, 0x5F69 }, { 2394, 0x624D },
    { 2395, 0x63A1 }, { 2396, 0x683D }, { 2397, 0x6B73 }, { 2398, 0x6E08 }, { 2399, 0x707D }, { 2400, 0x91C7 }, { 2401, 0x7280 }, { 2402, 0x7815 },
    { 2403, 0x7826 }, { 2404, 0x796D }, { 2405, 0x658E }, { 2406, 0x7D30 }, { 2407, 0x83DC }, { 2408, 0x88C1 }, { 2409, 0x8F09 }, { 2410, 0x969B },
    { 2411, 0x5264 }, { 2412, 0x5728 }, { 2413, 0x6750 }, { 2414, 0x7F6A }, { 2415, 0x8CA1 }, { 2416, 0x51B4 }, { 2417, 0x5742 }, { 2418, 0x962A },
    { 2419, 0x583A }, { 2420, 0x698A }, { 2421, 0x80B4 }, { 2422, 0x54B2 }, { 2423, 0x5D0E }, { 2424, 0x57FC }, { 2425, 0x7895 }, { 2426, 0x9DFA },
    { 2427, 0x4F5C }, { 2428, 0x524A }, { 2429, 0x548B }, { 2430, 0x643E }, { 2431, 0x6628 }, { 2432, 0x6714 }, { 2433, 0x67F5 }, { 2434, 0x7A84 },
    { 2435, 0x7B56 }, { 2436, 0x7D22 }, { 2437, 0x932F }, { 2438, 0x685C }, { 2439, 0x9BAD }, { 2440, 0x7B39 }, { 2441, 0x5319 }, { 2442, 0x518A },
    { 2443, 0x5237 }, { 2444, 0x5BDF }, { 2445, 0x62F6 }, { 2446, 0x64AE }, { 2447, 0x64E6 }, { 2448, 0x672D }, { 2449, 0x6BBA }, { 2450, 0x85A9 },
    { 2451, 0x96D1 }, { 2452, 0x7690 }, { 2453, 0x9BD6 }, { 2454, 0x634C }, { 2455, 0x9306 }, { 2456, 0x9BAB }, { 2457, 0x76BF }, { 2458, 0x6652 },
    { 2459, 0x4E09 }, { 2460, 0x5098 }, { 2461, 0x53C2 }, { 2462, 0x5C71 }, { 2463, 0x60E8 }, { 2464, 0x6492 }, { 2465, 0x6563 }, { 2466, 0x685F },
    { 2467, 0x71E6 }, { 2468, 0x73CA }, { 2469, 0x7523 }, { 2470, 0x7B97 }, { 2471, 0x7E82 }, { 2472, 0x8695 }, { 2473, 0x8B83 }, { 2474, 0x8CDB },
    { 2475, 0x9178 }, { 2476, 0x9910 }, { 2477, 0x65AC }, { 2478, 0x66AB }, { 2479, 0x6B8B }, { 2480, 0x4ED5 }, { 2481, 0x4ED4 }, { 2482, 0x4F3A },
    { 2483, 0x4F7F }, { 2484, 0x523A }, { 2485, 0x53F8 }, { 2486, 0x53F2 }, { 2487, 0x55E3 }, { 2488, 0x56DB }, { 2489, 0x58EB }, { 2490, 0x59CB },
    { 2491, 0x59C9 }, { 2492, 0x59FF }, { 2493, 0x5B50 }, { 2494, 0x5C4D }, { 2495, 0x5E02 }, { 2496, 0x5E2B }, { 2497, 0x5FD7 }, { 2498, 0x601D },
    { 2499, 0x6307 }, { 2500, 0x652F }, { 2501, 0x5B5C }, { 2502, 0x65AF }, { 2503, 0x65BD }, { 2504, 0x65E8 }, { 2505, 0x679D }, { 2506, 0x6B62 },
    { 2507, 0x6B7B }, { 2508, 0x6C0F }, { 2509, 0x7345 }, { 2510, 0x7949 }, { 2511, 0x79C1 }, { 2512, 0x7CF8 }, { 2513, 0x7D19 }, { 2514, 0x7D2B },
    { 2515, 0x80A2 }, { 2516, 0x8102 }, { 2517, 0x81F3 }, { 2518, 0x8996 }, { 2519, 0x8A5E }, { 2520, 0x8A69 }, { 2521, 0x8A66 }, { 2522, 0x8A8C },
    { 2523, 0x8AEE }, { 2524, 0x8CC7 }, { 2525, 0x8CDC }, { 2526, 0x96CC }, { 2527, 0x98FC }, { 2528, 0x6B6F }, { 2529, 0x4E8B }, { 2530, 0x4F3C },
    { 2531, 0x4F8D }, { 2532, 0x5150 }, { 2533, 0x5B57 }, { 2534, 0x5BFA }, { 2535, 0x6148 }, { 2536, 0x6301 }, { 2537, 0x6642 }, { 2538, 0x6B21 },
    { 2539, 0x6ECB }, { 2540, 0x6CBB }, { 2541, 0x723E }, { 2542, 0x74BD }, { 2543, 0x75D4 }, { 2544, 0x78C1 }, { 2545, 0x793A }, { 2546, 0x800C },
    { 2547, 0x8033 }, { 2548, 0x81EA }, { 2549, 0x8494 }, { 2550, 0x8F9E }, { 2551, 0x6C50 }, { 2552, 0x9E7F }, { 2553, 0x5F0F }, { 2554, 0x8B58 },
    { 2555, 0x9D2B }, { 2556, 0x7AFA }, { 2557, 0x8EF8 }, { 2558, 0x5B8D }, { 2559, 0x96EB }, { 2560, 0x4E03 }, { 2561, 0x53F1 }, { 2562, 0x57F7 },
    { 2563, 0x5931 }, { 2564, 0x5AC9 }, { 2565, 0x5BA4 }, { 2566, 0x6089 }, { 2567, 0x6E7F }, { 2568, 0x6F06 }, { 2569, 0x75BE }, { 2570, 0x8CEA },
    { 2571, 0x5B9F }, { 2572, 0x8500 }, { 2573, 0x7BE0 }, { 2574, 0x5072 }, { 2575, 0x67F4 }, { 2576, 0x829D }, { 2577, 0x5C61 }, { 2578, 0x854A },
    { 2579, 0x7E1E }, { 2580, 0x820E }, { 2581, 0x5199 }, { 2582, 0x5C04 }, { 2583, 0x6368 }, { 2584, 0x8D66 }, { 2585, 0x659C }, { 2586, 0x716E },
    { 2587, 0x793E }, { 2588, 0x7D17 }, { 2589, 0x8005 }, { 2590, 0x8B1D }, { 2591, 0x8ECA }, { 2592, 0x906E }, { 2593, 0x86C7 }, { 2594, 0x90AA },
    { 2595, 0x501F }, { 2596, 0x52FA }, { 2597, 0x5C3A }, { 2598, 0x6753 }, { 2599, 0x707C }, { 2600, 0x7235 }, { 2601, 0x914C }, { 2602, 0x91C8 },
    { 2603, 0x932B }, { 2604, 0x82E5 }, { 2605, 0x5BC2 }, { 2606, 0x5F31 }, { 2607, 0x60F9 }, { 2608, 0x4E3B }, { 2609, 0x53D6 }, { 2610, 0x5B88 },
    { 2611, 0x624B }, { 2612, 0x6731 }, { 2613, 0x6B8A }, { 2614, 0x72E9 }, { 2615, 0x73E0 }, { 2616, 0x7A2E }, { 2617, 0x816B }, { 2618, 0x8DA3 },
    { 2619, 0x9152 }, { 2620, 0x9996 }, { 2621, 0x5112 }, { 2622, 0x53D7 }, { 2623, 0x546A }, { 2624, 0x5BFF }, { 2625, 0x6388 }, { 2626, 0x6A39 },
    { 2627, 0x7DAC }, { 2628, 0x9700 }, { 2629, 0x56DA }, { 2630, 0x53CE }, { 2631, 0x5468 }, { 2632, 0x5B97 }, { 2633, 0x5C31 }, { 2634, 0x5DDE },
    { 2635, 0x4FEE }, { 2636, 0x6101 }, { 2637, 0x62FE }, { 2638, 0x6D32 }, { 2639, 0x79C0 }, { 2640, 0x79CB }, { 2641, 0x7D42 }, { 2642, 0x7E4D },
    { 2643, 0x7FD2 }, { 2644, 0x81ED }, { 2645, 0x821F }, { 2646, 0x8490 }, { 2647, 0x8846 }, { 2648, 0x8972 }, { 2649, 0x8B90 }, { 2650, 0x8E74 },
    { 2651, 0x8F2F }, { 2652, 0x9031 }, { 2653, 0x914B }, { 2654, 0x916C }, { 2655, 0x96C6 }, { 2656, 0x919C }, { 2657, 0x4EC0 }, { 2658, 0x4F4F },
    { 2659, 0x5145 }, { 2660, 0x5341 }, { 2661, 0x5F93 }, { 2662, 0x620E }, { 2663, 0x67D4 }, { 2664, 0x6C41 }, { 2665, 0x6E0B }, { 2666, 0x7363 },
    { 2667, 0x7E26 }, { 2668, 0x91CD }, { 2669, 0x9283 }, { 2670, 0x53D4 }, { 2671, 0x5919 }, { 2672, 0x5BBF }, { 2673, 0x6DD1 }, { 2674, 0x795D },
    { 2675, 0x7E2E }, { 2676, 0x7C9B }, { 2677, 0x587E }, { 2678, 0x719F }, { 2679, 0x51FA }, { 2680, 0x8853 }, { 2681, 0x8FF0 }, { 2682, 0x4FCA },
    { 2683, 0x5CFB }, { 2684, 0x6625 }, { 2685, 0x77AC }, { 2686, 0x7AE3 }, { 2687, 0x821C }, { 2688, 0x99FF }, { 2689, 0x51C6 }, { 2690, 0x5FAA },
    { 2691, 0x65EC }, { 2692, 0x696F }, { 2693, 0x6B89 }, { 2694, 0x6DF3 }, { 2695, 0x6E96 }, { 2696, 0x6F64 }, { 2697, 0x76FE }, { 2698, 0x7D14 },
    { 2699, 0x5DE1 }, { 2700, 0x9075 }, { 2701, 0x9187 }, { 2702, 0x9806 }, { 2703, 0x51E6 }, { 2704, 0x521D }, { 2705, 0x6240 }, { 2706, 0x6691 },
    { 2707, 0x66D9 }, { 2708, 0x6E1A }, { 2709, 0x5EB6 }, { 2710, 0x7DD2 }, { 2711, 0x7F72 }, { 2712, 0x66F8 }, { 2713, 0x85AF }, { 2714, 0x85F7 },
    { 2715, 0x8AF8 }, { 2716, 0x52A9 }, { 2717, 0x53D9 }, { 2718, 0x5973 }, { 2719, 0x5E8F }, { 2720, 0x5F90 }, { 2721, 0x6055 }, { 2722, 0x92E4 },
    { 2723, 0x9664 }, { 2724, 0x50B7 }, { 2725, 0x511F }, { 2726, 0x52DD }, { 2727, 0x5320 }, { 2728, 0x5347 }, { 2729, 0x53EC }, { 2730, 0x54E8 },
    { 2731, 0x5546 }, { 2732, 0x5531 }, { 2733, 0x5617 }, { 2734, 0x5968 }, { 2735, 0x59BE }, { 2736, 0x5A3C }, { 2737, 0x5BB5 }, { 2738, 0x5C06 },
    { 2739, 0x5C0F }, { 2740, 0x5C11 }, { 2741, 0x5C1A }, { 2742, 0x5E84 }, { 2743, 0x5E8A }, { 2744, 0x5EE0 }, { 2745, 0x5F70 }, { 2746, 0x627F },
    { 2747, 0x6284 }, { 2748, 0x62DB }, { 2749, 0x638C }, { 2750, 0x6377 }, { 2751, 0x6607 }, { 2752, 0x660C }, { 2753, 0x662D }, { 2754, 0x6676 },
    { 2755, 0x677E }, { 2756, 0x68A2 }, { 2757, 0x6A1F }, { 2758, 0x6A35 }, { 2759, 0x6CBC }, { 2760, 0x6D88 }, { 2761, 0x6E09 }, { 2762, 0x6E58 },
    { 2763, 0x713C }, { 2764, 0x7126 }, { 2765, 0x7167 }, { 2766, 0x75C7 }, { 2767, 0x7701 }, { 2768, 0x785D }, { 2769, 0x7901 }, { 2770, 0x7965 },
    { 2771, 0x79F0 }, { 2772, 0x7AE0 }, { 2773, 0x7B11 }, { 2774, 0x7CA7 }, { 2775, 0x7D39 }, { 2776, 0x8096 }, { 2777, 0x83D6 }, { 2778, 0x848B },
    { 2779, 0x8549 }, { 2780, 0x885D }, { 2781, 0x88F3 }, { 2782, 0x8A1F }, { 2783, 0x8A3C }, { 2784, 0x8A54 }, { 2785, 0x8A73 }, { 2786, 0x8C61 },
    { 2787, 0x8CDE }, { 2788, 0x91A4 }, { 2789, 0x9266 }, { 2790, 0x937E }, { 2791, 0x9418 }, { 2792, 0x969C }, { 2793, 0x9798 }, { 2794, 0x4E0A },
    { 2795, 0x4E08 }, { 2796, 0x4E1E }, { 2797, 0x4E57 }, { 2798, 0x5197 }, { 2799, 0x5270 }, { 2800, 0x57CE }, { 2801, 0x5834 }, { 2802, 0x58CC },
    { 2803, 0x5B22 }, { 2804, 0x5E38 }, { 2805, 0x60C5 }, { 2806, 0x64FE }, { 2807, 0x6761 }, { 2808, 0x6756 }, { 2809, 0x6D44 }, { 2810, 0x72B6 },
    { 2811, 0x7573 }, { 2812, 0x7A63 }, { 2813, 0x84B8 }, { 2814, 0x8B72 }, { 2815, 0x91B8 }, { 2816, 0x9320 }, { 2817, 0x5631 }, { 2818, 0x57F4 },
    { 2819, 0x98FE }, { 2820, 0x62ED }, { 2821, 0x690D }, { 2822, 0x6B96 }, { 2823, 0x71ED }, { 2824, 0x7E54 }, { 2825, 0x8077 }, { 2826, 0x8272 },
    { 2827, 0x89E6 }, { 2828, 0x98DF }, { 2829, 0x8755 }, { 2830, 0x8FB1 }, { 2831, 0x5C3B }, { 2832, 0x4F38 }, { 2833, 0x4FE1 }, { 2834, 0x4FB5 },
    { 2835, 0x5507 }, { 2836, 0x5A20 }, { 2837, 0x5BDD }, { 2838, 0x5BE9 }, { 2839, 0x5FC3 }, { 2840, 0x614E }, { 2841, 0x632F }, { 2842, 0x65B0 },
    { 2843, 0x664B }, { 2844, 0x68EE }, { 2845, 0x699B }, { 2846, 0x6D78 }, { 2847, 0x6DF1 }, { 2848, 0x7533 }, { 2849, 0x75B9 }, { 2850, 0x771F },
    { 2851, 0x795E }, { 2852, 0x79E6 }, { 2853, 0x7D33 }, { 2854, 0x81E3 }, { 2855, 0x82AF }, { 2856, 0x85AA }, { 2857, 0x89AA }, { 2858, 0x8A3A },
    { 2859, 0x8EAB }, { 2860, 0x8F9B }, { 2861, 0x9032 }, { 2862, 0x91DD }, { 2863, 0x9707 }, { 2864, 0x4EBA }, { 2865, 0x4EC1 }, { 2866, 0x5203 },
    { 2867, 0x5875 }, { 2868, 0x58EC }, { 2869, 0x5C0B }, { 2870, 0x751A }, { 2871, 0x5C3D }, { 2872, 0x814E }, { 2873, 0x8A0A }, { 2874, 0x8FC5 },
    { 2875, 0x9663 }, { 2876, 0x976D }, { 2877, 0x7B25 }, { 2878, 0x8ACF }, { 2879, 0x9808 }, { 2880, 0x9162 }, { 2881, 0x56F3 }, { 2882, 0x53A8 },
    { 2883, 0x9017 }, { 2884, 0x5439 }, { 2885, 0x5782 }, { 2886, 0x5E25 }, { 2887, 0x63A8 }, { 2888, 0x6C34 }, { 2889, 0x708A }, { 2890, 0x7761 },
    { 2891, 0x7C8B }, { 2892, 0x7FE0 }, { 2893, 0x8870 }, { 2894, 0x9042 }, { 2895, 0x9154 }, { 2896, 0x9310 }, { 2897, 0x9318 }, { 2898, 0x968F },
    { 2899, 0x745E }, { 2900, 0x9AC4 }, { 2901, 0x5D07 }, { 2902, 0x5D69 }, { 2903, 0x6570 }, { 2904, 0x67A2 }, { 2905, 0x8DA8 }, { 2906, 0x96DB },
    { 2907, 0x636E }, { 2908, 0x6749 }, { 2909, 0x6919 }, { 2910, 0x83C5 }, { 2911, 0x9817 }, { 2912, 0x96C0 }, { 2913, 0x88FE }, { 2914, 0x6F84 },
    { 2915, 0x647A }, { 2916, 0x5BF8 }, { 2917, 0x4E16 }, { 2918, 0x702C }, { 2919, 0x755D }, { 2920, 0x662F }, { 2921, 0x51C4 }, { 2922, 0x5236 },
    { 2923, 0x52E2 }, { 2924, 0x59D3 }, { 2925, 0x5F81 }, { 2926, 0x6027 }, { 2927, 0x6210 }, { 2928, 0x653F }, { 2929, 0x6574 }, { 2930, 0x661F },
    { 2931, 0x6674 }, { 2932, 0x68F2 }, { 2933, 0x6816 }, { 2934, 0x6B63 }, { 2935, 0x6E05 }, { 2936, 0x7272 }, { 2937, 0x751F }, { 2938, 0x76DB },
    { 2939, 0x7CBE }, { 2940, 0x8056 }, { 2941, 0x58F0 }, { 2942, 0x88FD }, { 2943, 0x897F }, { 2944, 0x8AA0 }, { 2945, 0x8A93 }, { 2946, 0x8ACB },
    { 2947, 0x901D }, { 2948, 0x9192 }, { 2949, 0x9752 }, { 2950, 0x9759 }, { 2951, 0x6589 }, { 2952, 0x7A0E }, { 2953, 0x8106 }, { 2954, 0x96BB },
    { 2955, 0x5E2D }, { 2956, 0x60DC }, { 2957, 0x621A }, { 2958, 0x65A5 }, { 2959, 0x6614 }, { 2960, 0x6790 }, { 2961, 0x77F3 }, { 2962, 0x7A4D },
    { 2963, 0x7C4D }, { 2964, 0x7E3E }, { 2965, 0x810A }, { 2966, 0x8CAC }, { 2967, 0x8D64 }, { 2968, 0x8DE1 }, { 2969, 0x8E5F }, { 2970, 0x78A9 },
    { 2971, 0x5207 }, { 2972, 0x62D9 }, { 2973, 0x63A5 }, { 2974, 0x6442 }, { 2975, 0x6298 }, { 2976, 0x8A2D }, { 2977, 0x7A83 }, { 2978, 0x7BC0 },
    { 2979, 0x8AAC }, { 2980, 0x96EA }, { 2981, 0x7D76 }, { 2982, 0x820C }, { 2983, 0x8749 }, { 2984, 0x4ED9 }, { 2985, 0x5148 }, { 2986, 0x5343 },
    { 2987, 0x5360 }, { 2988, 0x5BA3 }, { 2989, 0x5C02 }, { 2990, 0x5C16 }, { 2991, 0x5DDD }, { 2992, 0x6226 }, { 2993, 0x6247 }, { 2994, 0x64B0 },
    { 2995, 0x6813 }, { 2996, 0x6834 }, { 2997, 0x6CC9 }, { 2998, 0x6D45 }, { 2999, 0x6D17 }, { 3000, 0x67D3 }, { 3001, 0x6F5C }, { 3002, 0x714E },
    { 3003, 0x717D }, { 3004, 0x65CB }, { 3005, 0x7A7F }, { 3006, 0x7BAD }, { 3007, 0x7DDA }, { 3008, 0x7E4A }, { 3009, 0x7FA8 }, { 3010, 0x817A },
    { 3011, 0x821B }, { 3012, 0x8239 }, { 3013, 0x85A6 }, { 3014, 0x8A6E }, { 3015, 0x8CCE }, { 3016, 0x8DF5 }, { 3017, 0x9078 }, { 3018, 0x9077 },
    { 3019, 0x92AD }, { 3020, 0x9291 }, { 3021, 0x9583 }, { 3022, 0x9BAE }, { 3023, 0x524D }, { 3024, 0x5584 }, { 3025, 0x6F38 }, { 3026, 0x7136 },
    { 3027, 0x5168 }, { 3028, 0x7985 }, { 3029, 0x7E55 }, { 3030, 0x81B3 }, { 3031, 0x7CCE }, { 3032, 0x564C }, { 3033, 0x5851 }, { 3034, 0x5CA8 },
    { 3035, 0x63AA }, { 3036, 0x66FE }, { 3037, 0x66FD }, { 3038, 0x695A }, { 3039, 0x72D9 }, { 3040, 0x758F }, { 3041, 0x758E }, { 3042, 0x790E },
    { 3043, 0x7956 }, { 3044, 0x79DF }, { 3045, 0x7C97 }, { 3046, 0x7D20 }, { 3047, 0x7D44 }, { 3048, 0x8607 }, { 3049, 0x8A34 }, { 3050, 0x963B },
    { 3051, 0x9061 }, { 3052, 0x9F20 }, { 3053, 0x50E7 }, { 3054, 0x5275 }, { 3055, 0x53CC }, { 3056, 0x53E2 }, { 3057, 0x5009 }, { 3058, 0x55AA },
    { 3059, 0x58EE }, { 3060, 0x594F }, { 3061, 0x723D }, { 3062, 0x5B8B }, { 3063, 0x5C64 }, { 3064, 0x531D }, { 3065, 0x60E3 }, { 3066, 0x60F3 },
    { 3067, 0x635C }, { 3068, 0x6383 }, { 3069, 0x633F }, { 3070, 0x63BB }, { 3071, 0x64CD }, { 3072, 0x65E9 }, { 3073, 0x66F9 }, { 3074, 0x5DE3 },
    { 3075, 0x69CD }, { 3076, 0x69FD }, { 3077, 0x6F15 }, { 3078, 0x71E5 }, { 3079, 0x4E89 }, { 3080, 0x75E9 }, { 3081, 0x76F8 }, { 3082, 0x7A93 },
    { 3083, 0x7CDF }, { 3084, 0x7DCF }, { 3085, 0x7D9C }, { 3086, 0x8061 }, { 3087, 0x8349 }, { 3088, 0x8358 }, { 3089, 0x846C }, { 3090, 0x84BC },
    { 3091, 0x85FB }, { 3092, 0x88C5 }, { 3093, 0x8D70 }, { 3094, 0x9001 }, { 3095, 0x906D }, { 3096, 0x9397 }, { 3097, 0x971C }, { 3098, 0x9A12 },
    { 3099, 0x50CF }, { 3100, 0x5897 }, { 3101, 0x618E }, { 3102, 0x81D3 }, { 3103, 0x8535 }, { 3104, 0x8D08 }, { 3105, 0x9020 }, { 3106, 0x4FC3 },
    { 3107, 0x5074 }, { 3108, 0x5247 }, { 3109, 0x5373 }, { 3110, 0x606F }, { 3111, 0x6349 }, { 3112, 0x675F }, { 3113, 0x6E2C }, { 3114, 0x8DB3 },
    { 3115, 0x901F }, { 3116, 0x4FD7 }, { 3117, 0x5C5E }, { 3118, 0x8CCA }, { 3119, 0x65CF }, { 3120, 0x7D9A }, { 3121, 0x5352 }, { 3122, 0x8896 },
    { 3123, 0x5176 }, { 3124, 0x63C3 }, { 3125, 0x5B58 }, { 3126, 0x5B6B }, { 3127, 0x5C0A }, { 3128, 0x640D }, { 3129, 0x6751 }, { 3130, 0x905C },
    { 3131, 0x4ED6 }, { 3132, 0x591A }, { 3133, 0x592A }, { 3134, 0x6C70 }, { 3135, 0x8A51 }, { 3136, 0x553E }, { 3137, 0x5815 }, { 3138, 0x59A5 },
    { 3139, 0x60F0 }, { 3140, 0x6253 }, { 3141, 0x67C1 }, { 3142, 0x8235 }, { 3143, 0x6955 }, { 3144, 0x9640 }, { 3145, 0x99C4 }, { 3146, 0x9A28 },
    { 3147, 0x4F53 }, { 3148, 0x5806 }, { 3149, 0x5BFE }, { 3150, 0x8010 }, { 3151, 0x5CB1 }, { 3152, 0x5E2F }, { 3153, 0x5F85 }, { 3154, 0x6020 },
    { 3155, 0x614B }, { 3156, 0x6234 }, { 3157, 0x66FF }, { 3158, 0x6CF0 }, { 3159, 0x6EDE }, { 3160, 0x80CE }, { 3161, 0x817F }, { 3162, 0x82D4 },
    { 3163, 0x888B }, { 3164, 0x8CB8 }, { 3165, 0x9000 }, { 3166, 0x902E }, { 3167, 0x968A }, { 3168, 0x9EDB }, { 3169, 0x9BDB }, { 3170, 0x4EE3 },
    { 3171, 0x53F0 }, { 3172, 0x5927 }, { 3173, 0x7B2C }, { 3174, 0x918D }, { 3175, 0x984C }, { 3176, 0x9DF9 }, { 3177, 0x6EDD }, { 3178, 0x7027 },
    { 3179, 0x5353 }, { 3180, 0x5544 }, { 3181, 0x5B85 }, { 3182, 0x6258 }, { 3183, 0x629E }, { 3184, 0x62D3 }, { 3185, 0x6CA2 }, { 3186, 0x6FEF },
    { 3187, 0x7422 }, { 3188, 0x8A17 }, { 3189, 0x9438 }, { 3190, 0x6FC1 }, { 3191, 0x8AFE }, { 3192, 0x8338 }, { 3193, 0x51E7 }, { 3194, 0x86F8 },
    { 3195, 0x53EA }, { 3196, 0x53E9 }, { 3197, 0x4F46 }, { 3198, 0x9054 }, { 3199, 0x8FB0 }, { 3200, 0x596A }, { 3201, 0x8131 }, { 3202, 0x5DFD },
    { 3203, 0x7AEA }, { 3204, 0x8FBF }, { 3205, 0x68DA }, { 3206, 0x8C37 }, { 3207, 0x72F8 }, { 3208, 0x9C48 }, { 3209, 0x6A3D }, { 3210, 0x8AB0 },
    { 3211, 0x4E39 }, { 3212, 0x5358 }, { 3213, 0x5606 }, { 3214, 0x5766 }, { 3215, 0x62C5 }, { 3216, 0x63A2 }, { 3217, 0x65E6 }, { 3218, 0x6B4E },
    { 3219, 0x6DE1 }, { 3220, 0x6E5B }, { 3221, 0x70AD }, { 3222, 0x77ED }, { 3223, 0x7AEF }, { 3224, 0x7BAA }, { 3225, 0x7DBB }, { 3226, 0x803D },
    { 3227, 0x80C6 }, { 3228, 0x86CB }, { 3229, 0x8A95 }, { 3230, 0x935B }, { 3231, 0x56E3 }, { 3232, 0x58C7 }, { 3233, 0x5F3E }, { 3234, 0x65AD },
    { 3235, 0x6696 }, { 3236, 0x6A80 }, { 3237, 0x6BB5 }, { 3238, 0x7537 }, { 3239, 0x8AC7 }, { 3240, 0x5024 }, { 3241, 0x77E5 }, { 3242, 0x5730 },
    { 3243, 0x5F1B }, { 3244, 0x6065 }, { 3245, 0x667A }, { 3246, 0x6C60 }, { 3247, 0x75F4 }, { 3248, 0x7A1A }, { 3249, 0x7F6E }, { 3250, 0x81F4 },
    { 3251, 0x8718 }, { 3252, 0x9045 }, { 3253, 0x99B3 }, { 3254, 0x7BC9 }, { 3255, 0x755C }, { 3256, 0x7AF9 }, { 3257, 0x7B51 }, { 3258, 0x84C4 },
    { 3259, 0x9010 }, { 3260, 0x79E9 }, { 3261, 0x7A92 }, { 3262, 0x8336 }, { 3263, 0x5AE1 }, { 3264, 0x7740 }, { 3265, 0x4E2D }, { 3266, 0x4EF2 },
    { 3267, 0x5B99 }, { 3268, 0x5FE0 }, { 3269, 0x62BD }, { 3270, 0x663C }, { 3271, 0x67F1 }, { 3272, 0x6CE8 }, { 3273, 0x866B }, { 3274, 0x8877 },
    { 3275, 0x8A3B }, { 3276, 0x914E }, { 3277, 0x92F3 }, { 3278, 0x99D0 }, { 3279, 0x6A17 }, { 3280, 0x7026 }, { 3281, 0x732A }, { 3282, 0x82E7 },
    { 3283, 0x8457 }, { 3284, 0x8CAF }, { 3285, 0x4E01 }, { 3286, 0x5146 }, { 3287, 0x51CB }, { 3288, 0x558B }, { 3289, 0x5BF5 }, { 3290, 0x5E16 },
    { 3291, 0x5E33 }, { 3292, 0x5E81 }, { 3293, 0x5F14 }, { 3294, 0x5F35 }, { 3295, 0x5F6B }, { 3296, 0x5FB4 }, { 3297, 0x61F2 }, { 3298, 0x6311 },
    { 3299, 0x66A2 }, { 3300, 0x671D }, { 3301, 0x6F6E }, { 3302, 0x7252 }, { 3303, 0x753A }, { 3304, 0x773A }, { 3305, 0x8074 }, { 3306, 0x8139 },
    { 3307, 0x8178 }, { 3308, 0x8776 }, { 3309, 0x8ABF }, { 3310, 0x8ADC }, { 3311, 0x8D85 }, { 3312, 0x8DF3 }, { 3313, 0x929A }, { 3314, 0x9577 },
    { 3315, 0x9802 }, { 3316, 0x9CE5 }, { 3317, 0x52C5 }, { 3318, 0x6357 }, { 3319, 0x76F4 }, { 3320, 0x6715 }, { 3321, 0x6C88 }, { 3322, 0x73CD },
    { 3323, 0x8CC3 }, { 3324, 0x93AE }, { 3325, 0x9673 }, { 3326, 0x6D25 }, { 3327, 0x589C }, { 3328, 0x690E }, { 3329, 0x69CC }, { 3330, 0x8FFD },
    { 3331, 0x939A }, { 3332, 0x75DB }, { 3333, 0x901A }, { 3334, 0x585A }, { 3335, 0x6802 }, { 3336, 0x63B4 }, { 3337, 0x69FB }, { 3338, 0x4F43 },
    { 3339, 0x6F2C }, { 3340, 0x67D8 }, { 3341, 0x8FBB }, { 3342, 0x8526 }, { 3343, 0x7DB4 }, { 3344, 0x9354 }, { 3345, 0x693F }, { 3346, 0x6F70 },
    { 3347, 0x576A }, { 3348, 0x58F7 }, { 3349, 0x5B2C }, { 3350, 0x7D2C }, { 3351, 0x722A }, { 3352, 0x540A }, { 3353, 0x91E3 }, { 3354, 0x9DB4 },
    { 3355, 0x4EAD }, { 3356, 0x4F4E }, { 3357, 0x505C }, { 3358, 0x5075 }, { 3359, 0x5243 }, { 3360, 0x8C9E }, { 3361, 0x5448 }, { 3362, 0x5824 },
    { 3363, 0x5B9A }, { 3364, 0x5E1D }, { 3365, 0x5E95 }, { 3366, 0x5EAD }, { 3367, 0x5EF7 }, { 3368, 0x5F1F }, { 3369, 0x608C }, { 3370, 0x62B5 },
    { 3371, 0x633A }, { 3372, 0x63D0 }, { 3373, 0x68AF }, { 3374, 0x6C40 }, { 3375, 0x7887 }, { 3376, 0x798E }, { 3377, 0x7A0B }, { 3378, 0x7DE0 },
    { 3379, 0x8247 }, { 3380, 0x8A02 }, { 3381, 0x8AE6 }, { 3382, 0x8E44 }, { 3383, 0x9013 }, { 3384, 0x90B8 }, { 3385, 0x912D }, { 3386, 0x91D8 },
    { 3387, 0x9F0E }, { 3388, 0x6CE5 }, { 3389, 0x6458 }, { 3390, 0x64E2 }, { 3391, 0x6575 }, { 3392, 0x6EF4 }, { 3393, 0x7684 }, { 3394, 0x7B1B },
    { 3395, 0x9069 }, { 3396, 0x93D1 }, { 3397, 0x6EBA }, { 3398, 0x54F2 }, { 3399, 0x5FB9 }, { 3400, 0x64A4 }, { 3401, 0x8F4D }, { 3402, 0x8FED },
    { 3403, 0x9244 }, { 3404, 0x5178 }, { 3405, 0x586B }, { 3406, 0x5929 }, { 3407, 0x5C55 }, { 3408, 0x5E97 }, { 3409, 0x6DFB }, { 3410, 0x7E8F },
    { 3411, 0x751C }, { 3412, 0x8CBC }, { 3413, 0x8EE2 }, { 3414, 0x985B }, { 3415, 0x70B9 }, { 3416, 0x4F1D }, { 3417, 0x6BBF }, { 3418, 0x6FB1 },
    { 3419, 0x7530 }, { 3420, 0x96FB }, { 3421, 0x514E }, { 3422, 0x5410 }, { 3423, 0x5835 }, { 3424, 0x5857 }, { 3425, 0x59AC }, { 3426, 0x5C60 },
    { 3427, 0x5F92 }, { 3428, 0x6597 }, { 3429, 0x675C }, { 3430, 0x6E21 }, { 3431, 0x767B }, { 3432, 0x83DF }, { 3433, 0x8CED }, { 3434, 0x9014 },
    { 3435, 0x90FD }, { 3436, 0x934D }, { 3437, 0x7825 }, { 3438, 0x783A }, { 3439, 0x52AA }, { 3440, 0x5EA6 }, { 3441, 0x571F }, { 3442, 0x5974 },
    { 3443, 0x6012 }, { 3444, 0x5012 }, { 3445, 0x515A }, { 3446, 0x51AC }, { 3447, 0x51CD }, { 3448, 0x5200 }, { 3449, 0x5510 }, { 3450, 0x5854 },
    { 3451, 0x5858 }, { 3452, 0x5957 }, { 3453, 0x5B95 }, { 3454, 0x5CF6 }, { 3455, 0x5D8B }, { 3456, 0x60BC }, { 3457, 0x6295 }, { 3458, 0x642D },
    { 3459, 0x6771 }, { 3460, 0x6843 }, { 3461, 0x68BC }, { 3462, 0x68DF }, { 3463, 0x76D7 }, { 3464, 0x6DD8 }, { 3465, 0x6E6F }, { 3466, 0x6D9B },
    { 3467, 0x706F }, { 3468, 0x71C8 }, { 3469, 0x5F53 }, { 3470, 0x75D8 }, { 3471, 0x7977 }, { 3472, 0x7B49 }, { 3473, 0x7B54 }, { 3474, 0x7B52 },
    { 3475, 0x7CD6 }, { 3476, 0x7D71 }, { 3477, 0x5230 }, { 3478, 0x8463 }, { 3479, 0x8569 }, { 3480, 0x85E4 }, { 3481, 0x8A0E }, { 3482, 0x8B04 },
    { 3483, 0x8C46 }, { 3484, 0x8E0F }, { 3485, 0x9003 }, { 3486, 0x900F }, { 3487, 0x9419 }, { 3488, 0x9676 }, { 3489, 0x982D }, { 3490, 0x9A30 },
    { 3491, 0x95D8 }, { 3492, 0x50CD }, { 3493, 0x52D5 }, { 3494, 0x540C }, { 3495, 0x5802 }, { 3496, 0x5C0E }, { 3497, 0x61A7 }, { 3498, 0x649E },
    { 3499, 0x6D1E }, { 3500, 0x77B3 }, { 3501, 0x7AE5 }, { 3502, 0x80F4 }, { 3503, 0x8404 }, { 3504, 0x9053 }, { 3505, 0x9285 }, { 3506, 0x5CE0 },
    { 3507, 0x9D07 }, { 3508, 0x533F }, { 3509, 0x5F97 }, { 3510, 0x5FB3 }, { 3511, 0x6D9C }, { 3512, 0x7279 }, { 3513, 0x7763 }, { 3514, 0x79BF },
    { 3515, 0x7BE4 }, { 3516, 0x6BD2 }, { 3517, 0x72EC }, { 3518, 0x8AAD }, { 3519, 0x6803 }, { 3520, 0x6A61 }, { 3521, 0x51F8 }, { 3522, 0x7A81 },
    { 3523, 0x6934 }, { 3524, 0x5C4A }, { 3525, 0x9CF6 }, { 3526, 0x82EB }, { 3527, 0x5BC5 }, { 3528, 0x9149 }, { 3529, 0x701E }, { 3530, 0x5678 },
    { 3531, 0x5C6F }, { 3532, 0x60C7 }, { 3533, 0x6566 }, { 3534, 0x6C8C }, { 3535, 0x8C5A }, { 3536, 0x9041 }, { 3537, 0x9813 }, { 3538, 0x5451 },
    { 3539, 0x66C7 }, { 3540, 0x920D }, { 3541, 0x5948 }, { 3542, 0x90A3 }, { 3543, 0x5185 }, { 3544, 0x4E4D }, { 3545, 0x51EA }, { 3546, 0x8599 },
    { 3547, 0x8B0E }, { 3548, 0x7058 }, { 3549, 0x637A }, { 3550, 0x934B }, { 3551, 0x6962 }, { 3552, 0x99B4 }, { 3553, 0x7E04 }, { 3554, 0x7577 },
    { 3555, 0x5357 }, { 3556, 0x6960 }, { 3557, 0x8EDF }, { 3558, 0x96E3 }, { 3559, 0x6C5D }, { 3560, 0x4E8C }, { 3561, 0x5C3C }, { 3562, 0x5F10 },
    { 3563, 0x8FE9 }, { 3564, 0x5302 }, { 3565, 0x8CD1 }, { 3566, 0x8089 }, { 3567, 0x8679 }, { 3568, 0x5EFF }, { 3569, 0x65E5 }, { 3570, 0x4E73 },
    { 3571, 0x5165 }, { 3572, 0x5982 }, { 3573, 0x5C3F }, { 3574, 0x97EE }, { 3575, 0x4EFB }, { 3576, 0x598A }, { 3577, 0x5FCD }, { 3578, 0x8A8D },
    { 3579, 0x6FE1 }, { 3580, 0x79B0 }, { 3581, 0x7962 }, { 3582, 0x5BE7 }, { 3583, 0x8471 }, { 3584, 0x732B }, { 3585, 0x71B1 }, { 3586, 0x5E74 },
    { 3587, 0x5FF5 }, { 3588, 0x637B }, { 3589, 0x649A }, { 3590, 0x71C3 }, { 3591, 0x7C98 }, { 3592, 0x4E43 }, { 3593, 0x5EFC }, { 3594, 0x4E4B },
    { 3595, 0x57DC }, { 3596, 0x56A2 }, { 3597, 0x60A9 }, { 3598, 0x6FC3 }, { 3599, 0x7D0D }, { 3600, 0x80FD }, { 3601, 0x8133 }, { 3602, 0x81BF },
    { 3603, 0x8FB2 }, { 3604, 0x8997 }, { 3605, 0x86A4 }, { 3606, 0x5DF4 }, { 3607, 0x628A }, { 3608, 0x64AD }, { 3609, 0x8987 }, { 3610, 0x6777 },
    { 3611, 0x6CE2 }, { 3612, 0x6D3E }, { 3613, 0x7436 }, { 3614, 0x7834 }, { 3615, 0x5A46 }, { 3616, 0x7F75 }, { 3617, 0x82AD }, { 3618, 0x99AC },
    { 3619, 0x4FF3 }, { 3620, 0x5EC3 }, { 3621, 0x62DD }, { 3622, 0x6392 }, { 3623, 0x6557 }, { 3624, 0x676F }, { 3625, 0x76C3 }, { 3626, 0x724C },
    { 3627, 0x80CC }, { 3628, 0x80BA }, { 3629, 0x8F29 }, { 3630, 0x914D }, { 3631, 0x500D }, { 3632, 0x57F9 }, { 3633, 0x5A92 }, { 3634, 0x6885 },
    { 3635, 0x6973 }, { 3636, 0x7164 }, { 3637, 0x72FD }, { 3638, 0x8CB7 }, { 3639, 0x58F2 }, { 3640, 0x8CE0 }, { 3641, 0x966A }, { 3642, 0x9019 },
    { 3643, 0x877F }, { 3644, 0x79E4 }, { 3645, 0x77E7 }, { 3646, 0x8429 }, { 3647, 0x4F2F }, { 3648, 0x5265 }, { 3649, 0x535A }, { 3650, 0x62CD },
    { 3651, 0x67CF }, { 3652, 0x6CCA }, { 3653, 0x767D }, { 3654, 0x7B94 }, { 3655, 0x7C95 }, { 3656, 0x8236 }, { 3657, 0x8584 }, { 3658, 0x8FEB },
    { 3659, 0x66DD }, { 3660, 0x6F20 }, { 3661, 0x7206 }, { 3662, 0x7E1B }, { 3663, 0x83AB }, { 3664, 0x99C1 }, { 3665, 0x9EA6 }, { 3666, 0x51FD },
    { 3667, 0x7BB1 }, { 3668, 0x7872 }, { 3669, 0x7BB8 }, { 3670, 0x8087 }, { 3671, 0x7B48 }, { 3672, 0x6AE8 }, { 3673, 0x5E61 }, { 3674, 0x808C },
    { 3675, 0x7551 }, { 3676, 0x7560 }, { 3677, 0x516B }, { 3678, 0x9262 }, { 3679, 0x6E8C }, { 3680, 0x767A }, { 3681, 0x9197 }, { 3682, 0x9AEA },
    { 3683, 0x4F10 }, { 3684, 0x7F70 }, { 3685, 0x629C }, { 3686, 0x7B4F }, { 3687, 0x95A5 }, { 3688, 0x9CE9 }, { 3689, 0x567A }, { 3690, 0x5859 },
    { 3691, 0x86E4 }, { 3692, 0x96BC }, { 3693, 0x4F34 }, { 3694, 0x5224 }, { 3695, 0x534A }, { 3696, 0x53CD }, { 3697, 0x53DB }, { 3698, 0x5E06 },
    { 3699, 0x642C }, { 3700, 0x6591 }, { 3701, 0x677F }, { 3702, 0x6C3E }, { 3703, 0x6C4E }, { 3704, 0x7248 }, { 3705, 0x72AF }, { 3706, 0x73ED },
    { 3707, 0x7554 }, { 3708, 0x7E41 }, { 3709, 0x822C }, { 3710, 0x85E9 }, { 3711, 0x8CA9 }, { 3712, 0x7BC4 }, { 3713, 0x91C6 }, { 3714, 0x7169 },
    { 3715, 0x9812 }, { 3716, 0x98EF }, { 3717, 0x633D }, { 3718, 0x6669 }, { 3719, 0x756A }, { 3720, 0x76E4 }, { 3721, 0x78D0 }, { 3722, 0x8543 },
    { 3723, 0x86EE }, { 3724, 0x532A }, { 3725, 0x5351 }, { 3726, 0x5426 }, { 3727, 0x5983 }, { 3728, 0x5E87 }, { 3729, 0x5F7C }, { 3730, 0x60B2 },
    { 3731, 0x6249 }, { 3732, 0x6279 }, { 3733, 0x62AB }, { 3734, 0x6590 }, { 3735, 0x6BD4 }, { 3736, 0x6CCC }, { 3737, 0x75B2 }, { 3738, 0x76AE },
    { 3739, 0x7891 }, { 3740, 0x79D8 }, { 3741, 0x7DCB }, { 3742, 0x7F77 }, { 3743, 0x80A5 }, { 3744, 0x88AB }, { 3745, 0x8AB9 }, { 3746, 0x8CBB },
    { 3747, 0x907F }, { 3748, 0x975E }, { 3749, 0x98DB }, { 3750, 0x6A0B }, { 3751, 0x7C38 }, { 3752, 0x5099 }, { 3753, 0x5C3E }, { 3754, 0x5FAE },
    { 3755, 0x6787 }, { 3756, 0x6BD8 }, { 3757, 0x7435 }, { 3758, 0x7709 }, { 3759, 0x7F8E }, { 3760, 0x9F3B }, { 3761, 0x67CA }, { 3762, 0x7A17 },
    { 3763, 0x5339 }, { 3764, 0x758B }, { 3765, 0x9AED }, { 3766, 0x5F66 }, { 3767, 0x819D }, { 3768, 0x83F1 }, { 3769, 0x8098 }, { 3770, 0x5F3C },
    { 3771, 0x5FC5 }, { 3772, 0x7562 }, { 3773, 0x7B46 }, { 3774, 0x903C }, { 3775, 0x6867 }, { 3776, 0x59EB }, { 3777, 0x5A9B }, { 3778, 0x7D10 },
    { 3779, 0x767E }, { 3780, 0x8B2C }, { 3781, 0x4FF5 }, { 3782, 0x5F6A }, { 3783, 0x6A19 }, { 3784, 0x6C37 }, { 3785, 0x6F02 }, { 3786, 0x74E2 },
    { 3787, 0x7968 }, { 3788, 0x8868 }, { 3789, 0x8A55 }, { 3790, 0x8C79 }, { 3791, 0x5EDF }, { 3792, 0x63CF }, { 3793, 0x75C5 }, { 3794, 0x79D2 },
    { 3795, 0x82D7 }, { 3796, 0x9328 }, { 3797, 0x92F2 }, { 3798, 0x849C }, { 3799, 0x86ED }, { 3800, 0x9C2D }, { 3801, 0x54C1 }, { 3802, 0x5F6C },
    { 3803, 0x658C }, { 3804, 0x6D5C }, { 3805, 0x7015 }, { 3806, 0x8CA7 }, { 3807, 0x8CD3 }, { 3808, 0x983B }, { 3809, 0x654F }, { 3810, 0x74F6 },
    { 3811, 0x4E0D }, { 3812, 0x4ED8 }, { 3813, 0x57E0 }, { 3814, 0x592B }, { 3815, 0x5A66 }, { 3816, 0x5BCC }, { 3817, 0x51A8 }, { 3818, 0x5E03 },
    { 3819, 0x5E9C }, { 3820, 0x6016 }, { 3821, 0x6276 }, { 3822, 0x6577 }, { 3823, 0x65A7 }, { 3824, 0x666E }, { 3825, 0x6D6E }, { 3826, 0x7236 },
    { 3827, 0x7B26 }, { 3828, 0x8150 }, { 3829, 0x819A }, { 3830, 0x8299 }, { 3831, 0x8B5C }, { 3832, 0x8CA0 }, { 3833, 0x8CE6 }, { 3834, 0x8D74 },
    { 3835, 0x961C }, { 3836, 0x9644 }, { 3837, 0x4FAE }, { 3838, 0x64AB }, { 3839, 0x6B66 }, { 3840, 0x821E }, { 3841, 0x8461 }, { 3842, 0x856A },
    { 3843, 0x90E8 }, { 3844, 0x5C01 }, { 3845, 0x6953 }, { 3846, 0x98A8 }, { 3847, 0x847A }, { 3848, 0x8557 }, { 3849, 0x4F0F }, { 3850, 0x526F },
    { 3851, 0x5FA9 }, { 3852, 0x5E45 }, { 3853, 0x670D }, { 3854, 0x798F }, { 3855, 0x8179 }, { 3856, 0x8907 }, { 3857, 0x8986 }, { 3858, 0x6DF5 },
    { 3859, 0x5F17 }, { 3860, 0x6255 }, { 3861, 0x6CB8 }, { 3862, 0x4ECF }, { 3863, 0x7269 }, { 3864, 0x9B92 }, { 3865, 0x5206 }, { 3866, 0x543B },
    { 3867, 0x5674 }, { 3868, 0x58B3 }, { 3869, 0x61A4 }, { 3870, 0x626E }, { 3871, 0x711A }, { 3872, 0x596E }, { 3873, 0x7C89 }, { 3874, 0x7CDE },
    { 3875, 0x7D1B }, { 3876, 0x96F0 }, { 3877, 0x6587 }, { 3878, 0x805E }, { 3879, 0x4E19 }, { 3880, 0x4F75 }, { 3881, 0x5175 }, { 3882, 0x5840 },
    { 3883, 0x5E63 }, { 3884, 0x5E73 }, { 3885, 0x5F0A }, { 3886, 0x67C4 }, { 3887, 0x4E26 }, { 3888, 0x853D }, { 3889, 0x9589 }, { 3890, 0x965B },
    { 3891, 0x7C73 }, { 3892, 0x9801 }, { 3893, 0x50FB }, { 3894, 0x58C1 }, { 3895, 0x7656 }, { 3896, 0x78A7 }, { 3897, 0x5225 }, { 3898, 0x77A5 },
    { 3899, 0x8511 }, { 3900, 0x7B86 }, { 3901, 0x504F }, { 3902, 0x5909 }, { 3903, 0x7247 }, { 3904, 0x7BC7 }, { 3905, 0x7DE8 }, { 3906, 0x8FBA },
    { 3907, 0x8FD4 }, { 3908, 0x904D }, { 3909, 0x4FBF }, { 3910, 0x52C9 }, { 3911, 0x5A29 }, { 3912, 0x5F01 }, { 3913, 0x97AD }, { 3914, 0x4FDD },
    { 3915, 0x8217 }, { 3916, 0x92EA }, { 3917, 0x5703 }, { 3918, 0x6355 }, { 3919, 0x6B69 }, { 3920, 0x752B }, { 3921, 0x88DC }, { 3922, 0x8F14 },
    { 3923, 0x7A42 }, { 3924, 0x52DF }, { 3925, 0x5893 }, { 3926, 0x6155 }, { 3927, 0x620A }, { 3928, 0x66AE }, { 3929, 0x6BCD }, { 3930, 0x7C3F },
    { 3931, 0x83E9 }, { 3932, 0x5023 }, { 3933, 0x4FF8 }, { 3934, 0x5305 }, { 3935, 0x5446 }, { 3936, 0x5831 }, { 3937, 0x5949 }, { 3938, 0x5B9D },
    { 3939, 0x5CF0 }, { 3940, 0x5CEF }, { 3941, 0x5D29 }, { 3942, 0x5E96 }, { 3943, 0x62B1 }, { 3944, 0x6367 }, { 3945, 0x653E }, { 3946, 0x65B9 },
    { 3947, 0x670B }, { 3948, 0x6CD5 }, { 3949, 0x6CE1 }, { 3950, 0x70F9 }, { 3951, 0x7832 }, { 3952, 0x7E2B }, { 3953, 0x80DE }, { 3954, 0x82B3 },
    { 3955, 0x840C }, { 3956, 0x84EC }, { 3957, 0x8702 }, { 3958, 0x8912 }, { 3959, 0x8A2A }, { 3960, 0x8C4A }, { 3961, 0x90A6 }, { 3962, 0x92D2 },
    { 3963, 0x98FD }, { 3964, 0x9CF3 }, { 3965, 0x9D6C }, { 3966, 0x4E4F }, { 3967, 0x4EA1 }, { 3968, 0x508D }, { 3969, 0x5256 }, { 3970, 0x574A },
    { 3971, 0x59A8 }, { 3972, 0x5E3D }, { 3973, 0x5FD8 }, { 3974, 0x5FD9 }, { 3975, 0x623F }, { 3976, 0x66B4 }, { 3977, 0x671B }, { 3978, 0x67D0 },
    { 3979, 0x68D2 }, { 3980, 0x5192 }, { 3981, 0x7D21 }, { 3982, 0x80AA }, { 3983, 0x81A8 }, { 3984, 0x8B00 }, { 3985, 0x8C8C }, { 3986, 0x8CBF },
    { 3987, 0x927E }, { 3988, 0x9632 }, { 3989, 0x5420 }, { 3990, 0x982C }, { 3991, 0x5317 }, { 3992, 0x50D5 }, { 3993, 0x535C }, { 3994, 0x58A8 },
    { 3995, 0x64B2 }, { 3996, 0x6734 }, { 3997, 0x7267 }, { 3998, 0x7766 }, { 3999, 0x7A46 }, { 4000, 0x91E6 }, { 4001, 0x52C3 }, { 4002, 0x6CA1 },
    { 4003, 0x6B86 }, { 4004, 0x5800 }, { 4005, 0x5E4C }, { 4006, 0x5954 }, { 4007, 0x672C }, { 4008, 0x7FFB }, { 4009, 0x51E1 }, { 4010, 0x76C6 },
    { 4011, 0x6469 }, { 4012, 0x78E8 }, { 4013, 0x9B54 }, { 4014, 0x9EBB }, { 4015, 0x57CB }, { 4016, 0x59B9 }, { 4017, 0x6627 }, { 4018, 0x679A },
    { 4019, 0x6BCE }, { 4020, 0x54E9 }, { 4021, 0x69D9 }, { 4022, 0x5E55 }, { 4023, 0x819C }, { 4024, 0x6795 }, { 4025, 0x9BAA }, { 4026, 0x67FE },
    { 4027, 0x9C52 }, { 4028, 0x685D }, { 4029, 0x4EA6 }, { 4030, 0x4FE3 }, { 4031, 0x53C8 }, { 4032, 0x62B9 }, { 4033, 0x672B }, { 4034, 0x6CAB },
    { 4035, 0x8FC4 }, { 4036, 0x4FAD }, { 4037, 0x7E6D }, { 4038, 0x9EBF }, { 4039, 0x4E07 }, { 4040, 0x6162 }, { 4041, 0x6E80 }, { 4042, 0x6F2B },
    { 4043, 0x8513 }, { 4044, 0x5473 }, { 4045, 0x672A }, { 4046, 0x9B45 }, { 4047, 0x5DF3 }, { 4048, 0x7B95 }, { 4049, 0x5CAC }, { 4050, 0x5BC6 },
    { 4051, 0x871C }, { 4052, 0x6E4A }, { 4053, 0x84D1 }, { 4054, 0x7A14 }, { 4055, 0x8108 }, { 4056, 0x5999 }, { 4057, 0x7C8D }, { 4058, 0x6C11 },
    { 4059, 0x7720 }, { 4060, 0x52D9 }, { 4061, 0x5922 }, { 4062, 0x7121 }, { 4063, 0x725F }, { 4064, 0x77DB }, { 4065, 0x9727 }, { 4066, 0x9D61 },
    { 4067, 0x690B }, { 4068, 0x5A7F }, { 4069, 0x5A18 }, { 4070, 0x51A5 }, { 4071, 0x540D }, { 4072, 0x547D }, { 4073, 0x660E }, { 4074, 0x76DF },
    { 4075, 0x8FF7 }, { 4076, 0x9298 }, { 4077, 0x9CF4 }, { 4078, 0x59EA }, { 4079, 0x725D }, { 4080, 0x6EC5 }, { 4081, 0x514D }, { 4082, 0x68C9 },
    { 4083, 0x7DBF }, { 4084, 0x7DEC }, { 4085, 0x9762 }, { 4086, 0x9EBA }, { 4087, 0x6478 }, { 4088, 0x6A21 }, { 4089, 0x8302 }, { 4090, 0x5984 },
    { 4091, 0x5B5F }, { 4092, 0x6BDB }, { 4093, 0x731B }, { 4094, 0x76F2 }, { 4095, 0x7DB2 }, { 4096, 0x8017 }, { 4097, 0x8499 }, { 4098, 0x5132 },
    { 4099, 0x6728 }, { 4100, 0x9ED9 }, { 4101, 0x76EE }, { 4102, 0x6762 }, { 4103, 0x52FF }, { 4104, 0x9905 }, { 4105, 0x5C24 }, { 4106, 0x623B },
    { 4107, 0x7C7E }, { 4108, 0x8CB0 }, { 4109, 0x554F }, { 4110, 0x60B6 }, { 4111, 0x7D0B }, { 4112, 0x9580 }, { 4113, 0x5301 }, { 4114, 0x4E5F },
    { 4115, 0x51B6 }, { 4116, 0x591C }, { 4117, 0x723A }, { 4118, 0x8036 }, { 4119, 0x91CE }, { 4120, 0x5F25 }, { 4121, 0x77E2 }, { 4122, 0x5384 },
    { 4123, 0x5F79 }, { 4124, 0x7D04 }, { 4125, 0x85AC }, { 4126, 0x8A33 }, { 4127, 0x8E8D }, { 4128, 0x9756 }, { 4129, 0x67F3 }, { 4130, 0x85AE },
    { 4131, 0x9453 }, { 4132, 0x6109 }, { 4133, 0x6108 }, { 4134, 0x6CB9 }, { 4135, 0x7652 }, { 4136, 0x8AED }, { 4137, 0x8F38 }, { 4138, 0x552F },
    { 4139, 0x4F51 }, { 4140, 0x512A }, { 4141, 0x52C7 }, { 4142, 0x53CB }, { 4143, 0x5BA5 }, { 4144, 0x5E7D }, { 4145, 0x60A0 }, { 4146, 0x6182 },
    { 4147, 0x63D6 }, { 4148, 0x6709 }, { 4149, 0x67DA }, { 4150, 0x6E67 }, { 4151, 0x6D8C }, { 4152, 0x7336 }, { 4153, 0x7337 }, { 4154, 0x7531 },
    { 4155, 0x7950 }, { 4156, 0x88D5 }, { 4157, 0x8A98 }, { 4158, 0x904A }, { 4159, 0x9091 }, { 4160, 0x90F5 }, { 4161, 0x96C4 }, { 4162, 0x878D },
    { 4163, 0x5915 }, { 4164, 0x4E88 }, { 4165, 0x4F59 }, { 4166, 0x4E0E }, { 4167, 0x8A89 }, { 4168, 0x8F3F }, { 4169, 0x9810 }, { 4170, 0x50AD },
    { 4171, 0x5E7C }, { 4172, 0x5996 }, { 4173, 0x5BB9 }, { 4174, 0x5EB8 }, { 4175, 0x63DA }, { 4176, 0x63FA }, { 4177, 0x64C1 }, { 4178, 0x66DC },
    { 4179, 0x694A }, { 4180, 0x69D8 }, { 4181, 0x6D0B }, { 4182, 0x6EB6 }, { 4183, 0x7194 }, { 4184, 0x7528 }, { 4185, 0x7AAF }, { 4186, 0x7F8A },
    { 4187, 0x8000 }, { 4188, 0x8449 }, { 4189, 0x84C9 }, { 4190, 0x8981 }, { 4191, 0x8B21 }, { 4192, 0x8E0A }, { 4193, 0x9065 }, { 4194, 0x967D },
    { 4195, 0x990A }, { 4196, 0x617E }, { 4197, 0x6291 }, { 4198, 0x6B32 }, { 4199, 0x6C83 }, { 4200, 0x6D74 }, { 4201, 0x7FCC }, { 4202, 0x7FFC },
    { 4203, 0x6DC0 }, { 4204, 0x7F85 }, { 4205, 0x87BA }, { 4206, 0x88F8 }, { 4207, 0x6765 }, { 4208, 0x83B1 }, { 4209, 0x983C }, { 4210, 0x96F7 },
    { 4211, 0x6D1B }, { 4212, 0x7D61 }, { 4213, 0x843D }, { 4214, 0x916A }, { 4215, 0x4E71 }, { 4216, 0x5375 }, { 4217, 0x5D50 }, { 4218, 0x6B04 },
    { 4219, 0x6FEB }, { 4220, 0x85CD }, { 4221, 0x862D }, { 4222, 0x89A7 }, { 4223, 0x5229 }, { 4224, 0x540F }, { 4225, 0x5C65 }, { 4226, 0x674E },
    { 4227, 0x68A8 }, { 4228, 0x7406 }, { 4229, 0x7483 }, { 4230, 0x75E2 }, { 4231, 0x88CF }, { 4232, 0x88E1 }, { 4233, 0x91CC }, { 4234, 0x96E2 },
    { 4235, 0x9678 }, { 4236, 0x5F8B }, { 4237, 0x7387 }, { 4238, 0x7ACB }, { 4239, 0x844E }, { 4240, 0x63A0 }, { 4241, 0x7565 }, { 4242, 0x5289 },
    { 4243, 0x6D41 }, { 4244, 0x6E9C }, { 4245, 0x7409 }, { 4246, 0x7559 }, { 4247, 0x786B }, { 4248, 0x7C92 }, { 4249, 0x9686 }, { 4250, 0x7ADC },
    { 4251, 0x9F8D }, { 4252, 0x4FB6 }, { 4253, 0x616E }, { 4254, 0x65C5 }, { 4255, 0x865C }, { 4256, 0x4E86 }, { 4257, 0x4EAE }, { 4258, 0x50DA },
    { 4259, 0x4E21 }, { 4260, 0x51CC }, { 4261, 0x5BEE }, { 4262, 0x6599 }, { 4263, 0x6881 }, { 4264, 0x6DBC }, { 4265, 0x731F }, { 4266, 0x7642 },
    { 4267, 0x77AD }, { 4268, 0x7A1C }, { 4269, 0x7CE7 }, { 4270, 0x826F }, { 4271, 0x8AD2 }, { 4272, 0x907C }, { 4273, 0x91CF }, { 4274, 0x9675 },
    { 4275, 0x9818 }, { 4276, 0x529B }, { 4277, 0x7DD1 }, { 4278, 0x502B }, { 4279, 0x5398 }, { 4280, 0x6797 }, { 4281, 0x6DCB }, { 4282, 0x71D0 },
    { 4283, 0x7433 }, { 4284, 0x81E8 }, { 4285, 0x8F2A }, { 4286, 0x96A3 }, { 4287, 0x9C57 }, { 4288, 0x9E9F }, { 4289, 0x7460 }, { 4290, 0x5841 },
    { 4291, 0x6D99 }, { 4292, 0x7D2F }, { 4293, 0x985E }, { 4294, 0x4EE4 }, { 4295, 0x4F36 }, { 4296, 0x4F8B }, { 4297, 0x51B7 }, { 4298, 0x52B1 },
    { 4299, 0x5DBA }, { 4300, 0x601C }, { 4301, 0x73B2 }, { 4302, 0x793C }, { 4303, 0x82D3 }, { 4304, 0x9234 }, { 4305, 0x96B7 }, { 4306, 0x96F6 },
    { 4307, 0x970A }, { 4308, 0x9E97 }, { 4309, 0x9F62 }, { 4310, 0x66A6 }, { 4311, 0x6B74 }, { 4312, 0x5217 }, { 4313, 0x52A3 }, { 4314, 0x70C8 },
    { 4315, 0x88C2 }, { 4316, 0x5EC9 }, { 4317, 0x604B }, { 4318, 0x6190 }, { 4319, 0x6F23 }, { 4320, 0x7149 }, { 4321, 0x7C3E }, { 4322, 0x7DF4 },
    { 4323, 0x806F }, { 4324, 0x84EE }, { 4325, 0x9023 }, { 4326, 0x932C }, { 4327, 0x5442 }, { 4328, 0x9B6F }, { 4329, 0x6AD3 }, { 4330, 0x7089 },
    { 4331, 0x8CC2 }, { 4332, 0x8DEF }, { 4333, 0x9732 }, { 4334, 0x52B4 }, { 4335, 0x5A41 }, { 4336, 0x5ECA }, { 4337, 0x5F04 }, { 4338, 0x6717 },
    { 4339, 0x697C }, { 4340, 0x6994 }, { 4341, 0x6D6A }, { 4342, 0x6F0F }, { 4343, 0x7262 }, { 4344, 0x72FC }, { 4345, 0x7BED }, { 4346, 0x8001 },
    { 4347, 0x807E }, { 4348, 0x874B }, { 4349, 0x90CE }, { 4350, 0x516D }, { 4351, 0x9E93 }, { 4352, 0x7984 }, { 4353, 0x808B }, { 4354, 0x9332 },
    { 4355, 0x8AD6 }, { 4356, 0x502D }, { 4357, 0x548C }, { 4358, 0x8A71 }, { 4359, 0x6B6A }, { 4360, 0x8CC4 }, { 4361, 0x8107 }, { 4362, 0x60D1 },
    { 4363, 0x67A0 }, { 4364, 0x9DF2 }, { 4365, 0x4E99 }, { 4366, 0x4E98 }, { 4367, 0x9C10 }, { 4368, 0x8A6B }, { 4369, 0x85C1 }, { 4370, 0x8568 },
    { 4371, 0x6900 }, { 4372, 0x6E7E }, { 4373, 0x7897 }, { 4374, 0x8155 }, { 4418, 0x5F0C }, { 4419, 0x4E10 }, { 4420, 0x4E15 }, { 4421, 0x4E2A },
    { 4422, 0x4E31 }, { 4423, 0x4E36 }, { 4424, 0x4E3C }, { 4425, 0x4E3F }, { 4426, 0x4E42 }, { 4427, 0x4E56 }, { 4428, 0x4E58 }, { 4429, 0x4E82 },
    { 4430, 0x4E85 }, { 4431, 0x8C6B }, { 4432, 0x4E8A }, { 4433, 0x8212 }, { 4434, 0x5F0D }, { 4435, 0x4E8E }, { 4436, 0x4E9E }, { 4437, 0x4E9F },
    { 4438, 0x4EA0 }, { 4439, 0x4EA2 }, { 4440, 0x4EB0 }, { 4441, 0x4EB3 }, { 4442, 0x4EB6 }, { 4443, 0x4ECE }, { 4444, 0x4ECD }, { 4445, 0x4EC4 },
    { 4446, 0x4EC6 }, { 4447, 0x4EC2 }, { 4448, 0x4ED7 }, { 4449, 0x4EDE }, { 4450, 0x4EED }, { 4451, 0x4EDF }, { 4452, 0x4EF7 }, { 4453, 0x4F09 },
    { 4454, 0x4F5A }, { 4455, 0x4F30 }, { 4456, 0x4F5B }, { 4457, 0x4F5D }, { 4458, 0x4F57 }, { 4459, 0x4F47 }, { 4460, 0x4F76 }, { 4461, 0x4F88 },
    { 4462, 0x4F8F }, { 4463, 0x4F98 }, { 4464, 0x4F7B }, { 4465, 0x4F69 }, { 4466, 0x4F70 }, { 4467, 0x4F91 }, { 4468, 0x4F6F }, { 4469, 0x4F86 },
    { 4470, 0x4F96 }, { 4471, 0x5118 }, { 4472, 0x4FD4 }, { 4473, 0x4FDF }, { 4474, 0x4FCE }, { 4475, 0x4FD8 }, { 4476, 0x4FDB }, { 4477, 0x4FD1 },
    { 4478, 0x4FDA }, { 4479, 0x4FD0 }, { 4480, 0x4FE4 }, { 4481, 0x4FE5 }, { 4482, 0x501A }, { 4483, 0x5028 }, { 4484, 0x5014 }, { 4485, 0x502A },
    { 4486, 0x5025 }, { 4487, 0x5005 }, { 4488, 0x4F1C }, { 4489, 0x4FF6 }, { 4490, 0x5021 }, { 4491, 0x5029 }, { 4492, 0x502C }, { 4493, 0x4FFE },
    { 4494, 0x4FEF }, { 4495, 0x5011 }, { 4496, 0x5006 }, { 4497, 0x5043 }, { 4498, 0x5047 }, { 4499, 0x6703 }, { 4500, 0x5055 }, { 4501, 0x5050 },
    { 4502, 0x5048 }, { 4503, 0x505A }, { 4504, 0x5056 }, { 4505, 0x506C }, { 4506, 0x5078 }, { 4507, 0x5080 }, { 4508, 0x509A }, { 4509, 0x5085 },
    { 4510, 0x50B4 }, { 4511, 0x50B2 }, { 4512, 0x50C9 }, { 4513, 0x50CA }, { 4514, 0x50B3 }, { 4515, 0x50C2 }, { 4516, 0x50D6 }, { 4517, 0x50DE },
    { 4518, 0x50E5 }, { 4519, 0x50ED }, { 4520, 0x50E3 }, { 4521, 0x50EE }, { 4522, 0x50F9 }, { 4523, 0x50F5 }, { 4524, 0x5109 }, { 4525, 0x5101 },
    { 4526, 0x5102 }, { 4527, 0x5116 }, { 4528, 0x5115 }, { 4529, 0x5114 }, { 4530, 0x511A }, { 4531, 0x5121 }, { 4532, 0x513A }, { 4533, 0x5137 },
    { 4534, 0x513C }, { 4535, 0x513B }, { 4536, 0x513F }, { 4537, 0x5140 }, { 4538, 0x5152 }, { 4539, 0x514C }, { 4540, 0x5154 }, { 4541, 0x5162 },
    { 4542, 0x7AF8 }, { 4543, 0x5169 }, { 4544, 0x516A }, { 4545, 0x516E }, { 4546, 0x5180 }, { 4547, 0x5182 }, { 4548, 0x56D8 }, { 4549, 0x518C },
    { 4550, 0x5189 }, { 4551, 0x518F }, { 4552, 0x5191 }, { 4553, 0x5193 }, { 4554, 0x5195 }, { 4555, 0x5196 }, { 4556, 0x51A4 }, { 4557, 0x51A6 },
    { 4558, 0x51A2 }, { 4559, 0x51A9 }, { 4560, 0x51AA }, { 4561, 0x51AB }, { 4562, 0x51B3 }, { 4563, 0x51B1 }, { 4564, 0x51B2 }, { 4565, 0x51B0 },
    { 4566, 0x51B5 }, { 4567, 0x51BD }, { 4568, 0x51C5 }, { 4569, 0x51C9 }, { 4570, 0x51DB }, { 4571, 0x51E0 }, { 4572, 0x8655 }, { 4573, 0x51E9 },
    { 4574, 0x51ED }, { 4575, 0x51F0 }, { 4576, 0x51F5 }, { 4577, 0x51FE }, { 4578, 0x5204 }, { 4579, 0x520B }, { 4580, 0x5214 }, { 4581, 0x520E },
    { 4582, 0x5227 }, { 4583, 0x522A }, { 4584, 0x522E }, { 4585, 0x5233 }, { 4586, 0x5239 }, { 4587, 0x524F }, { 4588, 0x5244 }, { 4589, 0x524B },
    { 4590, 0x524C }, { 4591, 0x525E }, { 4592, 0x5254 }, { 4593, 0x526A }, { 4594, 0x5274 }, { 4595, 0x5269 }, { 4596, 0x5273 }, { 4597, 0x527F },
    { 4598, 0x527D }, { 4599, 0x528D }, { 4600, 0x5294 }, { 4601, 0x5292 }, { 4602, 0x5271 }, { 4603, 0x5288 }, { 4604, 0x5291 }, { 4605, 0x8FA8 },
    { 4606, 0x8FA7 }, { 4607, 0x52AC }, { 4608, 0x52AD }, { 4609, 0x52BC }, { 4610, 0x52B5 }, { 4611, 0x52C1 }, { 4612, 0x52CD }, { 4613, 0x52D7 },
    { 4614, 0x52DE }, { 4615, 0x52E3 }, { 4616, 0x52E6 }, { 4617, 0x98ED }, { 4618, 0x52E0 }, { 4619, 0x52F3 }, { 4620, 0x52F5 }, { 4621, 0x52F8 },
    { 4622, 0x52F9 }, { 4623, 0x5306 }, { 4624, 0x5308 }, { 4625, 0x7538 }, { 4626, 0x530D }, { 4627, 0x5310 }, { 4628, 0x530F }, { 4629, 0x5315 },
    { 4630, 0x531A }, { 4631, 0x5323 }, { 4632, 0x532F }, { 4633, 0x5331 }, { 4634, 0x5333 }, { 4635, 0x5338 }, { 4636, 0x5340 }, { 4637, 0x5346 },
    { 4638, 0x5345 }, { 4639, 0x4E17 }, { 4640, 0x5349 }, { 4641, 0x534D }, { 4642, 0x51D6 }, { 4643, 0x535E }, { 4644, 0x5369 }, { 4645, 0x536E },
    { 4646, 0x5918 }, { 4647, 0x537B }, { 4648, 0x5377 }, { 4649, 0x5382 }, { 4650, 0x5396 }, { 4651, 0x53A0 }, { 4652, 0x53A6 }, { 4653, 0x53A5 },
    { 4654, 0x53AE }, { 4655, 0x53B0 }, { 4656, 0x53B6 }, { 4657, 0x53C3 }, { 4658, 0x7C12 }, { 4659, 0x96D9 }, { 4660, 0x53DF }, { 4661, 0x66FC },
    { 4662, 0x71EE }, { 4663, 0x53EE }, { 4664, 0x53E8 }, { 4665, 0x53ED }, { 4666, 0x53FA }, { 4667, 0x5401 }, { 4668, 0x543D }, { 4669, 0x5440 },
    { 4670, 0x542C }, { 4671, 0x542D }, { 4672, 0x543C }, { 4673, 0x542E }, { 4674, 0x5436 }, { 4675, 0x5429 }, { 4676, 0x541D }, { 4677, 0x544E },
    { 4678, 0x548F }, { 4679, 0x5475 }, { 4680, 0x548E }, { 4681, 0x545F }, { 4682, 0x5471 }, { 4683, 0x5477 }, { 4684, 0x5470 }, { 4685, 0x5492 },
    { 4686, 0x547B }, { 4687, 0x5480 }, { 4688, 0x5476 }, { 4689, 0x5484 }, { 4690, 0x5490 }, { 4691, 0x5486 }, { 4692, 0x54C7 }, { 4693, 0x54A2 },
    { 4694, 0x54B8 }, { 4695, 0x54A5 }, { 4696, 0x54AC }, { 4697, 0x54C4 }, { 4698, 0x54C8 }, { 4699, 0x54A8 }, { 4700, 0x54AB }, { 4701, 0x54C2 },
    { 4702, 0x54A4 }, { 4703, 0x54BE }, { 4704, 0x54BC }, { 4705, 0x54D8 }, { 4706, 0x54E5 }, { 4707, 0x54E6 }, { 4708, 0x550F }, { 4709, 0x5514 },
    { 4710, 0x54FD }, { 4711, 0x54EE }, { 4712, 0x54ED }, { 4713, 0x54FA }, { 4714, 0x54E2 }, { 4715, 0x5539 }, { 4716, 0x5540 }, { 4717, 0x5563 },
    { 4718, 0x554C }, { 4719, 0x552E }, { 4720, 0x555C }, { 4721, 0x5545 }, { 4722, 0x5556 }, { 4723, 0x5557 }, { 4724, 0x5538 }, { 4725, 0x5533 },
    { 4726, 0x555D }, { 4727, 0x5599 }, { 4728, 0x5580 }, { 4729, 0x54AF }, { 4730, 0x558A }, { 4731, 0x559F }, { 4732, 0x557B }, { 4733, 0x557E },
    { 4734, 0x5598 }, { 4735, 0x559E }, { 4736, 0x55AE }, { 4737, 0x557C }, { 4738, 0x5583 }, { 4739, 0x55A9 }, { 4740, 0x5587 }, { 4741, 0x55A8 },
    { 4742, 0x55DA }, { 4743, 0x55C5 }, { 4744, 0x55DF }, { 4745, 0x55C4 }, { 4746, 0x55DC }, { 4747, 0x55E4 }, { 4748, 0x55D4 }, { 4749, 0x5614 },
    { 4750, 0x55F7 }, { 4751, 0x5616 }, { 4752, 0x55FE }, { 4753, 0x55FD }, { 4754, 0x561B }, { 4755, 0x55F9 }, { 4756, 0x564E }, { 4757, 0x5650 },
    { 4758, 0x71DF }, { 4759, 0x5634 }, { 4760, 0x5636 }, { 4761, 0x5632 }, { 4762, 0x5638 }, { 4763, 0x566B }, { 4764, 0x5664 }, { 4765, 0x562F },
    { 4766, 0x566C }, { 4767, 0x566A }, { 4768, 0x5686 }, { 4769, 0x5680 }, { 4770, 0x568A }, { 4771, 0x56A0 }, { 4772, 0x5694 }, { 4773, 0x568F },
    { 4774, 0x56A5 }, { 4775, 0x56AE }, { 4776, 0x56B6 }, { 4777, 0x56B4 }, { 4778, 0x56C2 }, { 4779, 0x56BC }, { 4780, 0x56C1 }, { 4781, 0x56C3 },
    { 4782, 0x56C0 }, { 4783, 0x56C8 }, { 4784, 0x56CE }, { 4785, 0x56D1 }, { 4786, 0x56D3 }, { 4787, 0x56D7 }, { 4788, 0x56EE }, { 4789, 0x56F9 },
    { 4790, 0x5700 }, { 4791, 0x56FF }, { 4792, 0x5704 }, { 4793, 0x5709 }, { 4794, 0x5708 }, { 4795, 0x570B }, { 4796, 0x570D }, { 4797, 0x5713 },
    { 4798, 0x5718 }, { 4799, 0x5716 }, { 4800, 0x55C7 }, { 4801, 0x571C }, { 4802, 0x5726 }, { 4803, 0x5737 }, { 4804, 0x5738 }, { 4805, 0x574E },
    { 4806, 0x573B }, { 4807, 0x5740 }, { 4808, 0x574F }, { 4809, 0x5769 }, { 4810, 0x57C0 }, { 4811, 0x5788 }, { 4812, 0x5761 }, { 4813, 0x577F },
    { 4814, 0x5789 }, { 4815, 0x5793 }, { 4816, 0x57A0 }, { 4817, 0x57B3 }, { 4818, 0x57A4 }, { 4819, 0x57AA }, { 4820, 0x57B0 }, { 4821, 0x57C3 },
    { 4822, 0x57C6 }, { 4823, 0x57D4 }, { 4824, 0x57D2 }, { 4825, 0x57D3 }, { 4826, 0x580A }, { 4827, 0x57D6 }, { 4828, 0x57E3 }, { 4829, 0x580B },
    { 4830, 0x5819 }, { 4831, 0x581D }, { 4832, 0x5872 }, { 4833, 0x5821 }, { 4834, 0x5862 }, { 4835, 0x584B }, { 4836, 0x5870 }, { 4837, 0x6BC0 },
    { 4838, 0x5852 }, { 4839, 0x583D }, { 4840, 0x5879 }, { 4841, 0x5885 }, { 4842, 0x58B9 }, { 4843, 0x589F }, { 4844, 0x58AB }, { 4845, 0x58BA },
    { 4846, 0x58DE }, { 4847, 0x58BB }, { 4848, 0x58B8 }, { 4849, 0x58AE }, { 4850, 0x58C5 }, { 4851, 0x58D3 }, { 4852, 0x58D1 }, { 4853, 0x58D7 },
    { 4854, 0x58D9 }, { 4855, 0x58D8 }, { 4856, 0x58E5 }, { 4857, 0x58DC }, { 4858, 0x58E4 }, { 4859, 0x58DF }, { 4860, 0x58EF }, { 4861, 0x58FA },
    { 4862, 0x58F9 }, { 4863, 0x58FB }, { 4864, 0x58FC }, { 4865, 0x58FD }, { 4866, 0x5902 }, { 4867, 0x590A }, { 4868, 0x5910 }, { 4869, 0x591B },
    { 4870, 0x68A6 }, { 4871, 0x5925 }, { 4872, 0x592C }, { 4873, 0x592D }, { 4874, 0x5932 }, { 4875, 0x5938 }, { 4876, 0x593E }, { 4877, 0x7AD2 },
    { 4878, 0x5955 }, { 4879, 0x5950 }, { 4880, 0x594E }, { 4881, 0x595A }, { 4882, 0x5958 }, { 4883, 0x5962 }, { 4884, 0x5960 }, { 4885, 0x5967 },
    { 4886, 0x596C }, { 4887, 0x5969 }, { 4888, 0x5978 }, { 4889, 0x5981 }, { 4890, 0x599D }, { 4891, 0x4F5E }, { 4892, 0x4FAB }, { 4893, 0x59A3 },
    { 4894, 0x59B2 }, { 4895, 0x59C6 }, { 4896, 0x59E8 }, { 4897, 0x59DC }, { 4898, 0x598D }, { 4899, 0x59D9 }, { 4900, 0x59DA }, { 4901, 0x5A25 },
    { 4902, 0x5A1F }, { 4903, 0x5A11 }, { 4904, 0x5A1C }, { 4905, 0x5A09 }, { 4906, 0x5A1A }, { 4907, 0x5A40 }, { 4908, 0x5A6C }, { 4909, 0x5A49 },
    { 4910, 0x5A35 }, { 4911, 0x5A36 }, { 4912, 0x5A62 }, { 4913, 0x5A6A }, { 4914, 0x5A9A }, { 4915, 0x5ABC }, { 4916, 0x5ABE }, { 4917, 0x5ACB },
    { 4918, 0x5AC2 }, { 4919, 0x5ABD }, { 4920, 0x5AE3 }, { 4921, 0x5AD7 }, { 4922, 0x5AE6 }, { 4923, 0x5AE9 }, { 4924, 0x5AD6 }, { 4925, 0x5AFA },
    { 4926, 0x5AFB }, { 4927, 0x5B0C }, { 4928, 0x5B0B }, { 4929, 0x5B16 }, { 4930, 0x5B32 }, { 4931, 0x5AD0 }, { 4932, 0x5B2A }, { 4933, 0x5B36 },
    { 4934, 0x5B3E }, { 4935, 0x5B43 }, { 4936, 0x5B45 }, { 4937, 0x5B40 }, { 4938, 0x5B51 }, { 4939, 0x5B55 }, { 4940, 0x5B5A }, { 4941, 0x5B5B },
    { 4942, 0x5B65 }, { 4943, 0x5B69 }, { 4944, 0x5B70 }, { 4945, 0x5B73 }, { 4946, 0x5B75 }, { 4947, 0x5B78 }, { 4948, 0x6588 }, { 4949, 0x5B7A },
    { 4950, 0x5B80 }, { 4951, 0x5B83 }, { 4952, 0x5BA6 }, { 4953, 0x5BB8 }, { 4954, 0x5BC3 }, { 4955, 0x5BC7 }, { 4956, 0x5BC9 }, { 4957, 0x5BD4 },
    { 4958, 0x5BD0 }, { 4959, 0x5BE4 }, { 4960, 0x5BE6 }, { 4961, 0x5BE2 }, { 4962, 0x5BDE }, { 4963, 0x5BE5 }, { 4964, 0x5BEB }, { 4965, 0x5BF0 },
    { 4966, 0x5BF6 }, { 4967, 0x5BF3 }, { 4968, 0x5C05 }, { 4969, 0x5C07 }, { 4970, 0x5C08 }, { 4971, 0x5C0D }, { 4972, 0x5C13 }, { 4973, 0x5C20 },
    { 4974, 0x5C22 }, { 4975, 0x5C28 }, { 4976, 0x5C38 }, { 4977, 0x5C39 }, { 4978, 0x5C41 }, { 4979, 0x5C46 }, { 4980, 0x5C4E }, { 4981, 0x5C53 },
    { 4982, 0x5C50 }, { 4983, 0x5C4F }, { 4984, 0x5B71 }, { 4985, 0x5C6C }, { 4986, 0x5C6E }, { 4987, 0x4E62 }, { 4988, 0x5C76 }, { 4989, 0x5C79 },
    { 4990, 0x5C8C }, { 4991, 0x5C91 }, { 4992, 0x5C94 }, { 4993, 0x599B }, { 4994, 0x5CAB }, { 4995, 0x5CBB }, { 4996, 0x5CB6 }, { 4997, 0x5CBC },
    { 4998, 0x5CB7 }, { 4999, 0x5CC5 }, { 5000, 0x5CBE }, { 5001, 0x5CC7 }, { 5002, 0x5CD9 }, { 5003, 0x5CE9 }, { 5004, 0x5CFD }, { 5005, 0x5CFA },
    { 5006, 0x5CED }, { 5007, 0x5D8C }, { 5008, 0x5CEA }, { 5009, 0x5D0B }, { 5010, 0x5D15 }, { 5011, 0x5D17 }, { 5012, 0x5D5C }, { 5013, 0x5D1F },
    { 5014, 0x5D1B }, { 5015, 0x5D11 }, { 5016, 0x5D14 }, { 5017, 0x5D22 }, { 5018, 0x5D1A }, { 5019, 0x5D19 }, { 5020, 0x5D18 }, { 5021, 0x5D4C },
    { 5022, 0x5D52 }, { 5023, 0x5D4E }, { 5024, 0x5D4B }, { 5025, 0x5D6C }, { 5026, 0x5D73 }, { 5027, 0x5D76 }, { 5028, 0x5D87 }, { 5029, 0x5D84 },
    { 5030, 0x5D82 }, { 5031, 0x5DA2 }, { 5032, 0x5D9D }, { 5033, 0x5DAC }, { 5034, 0x5DAE }, { 5035, 0x5DBD }, { 5036, 0x5D90 }, { 5037, 0x5DB7 },
    { 5038, 0x5DBC }, { 5039, 0x5DC9 }, { 5040, 0x5DCD }, { 5041, 0x5DD3 }, { 5042, 0x5DD2 }, { 5043, 0x5DD6 }, { 5044, 0x5DDB }, { 5045, 0x5DEB },
    { 5046, 0x5DF2 }, { 5047, 0x5DF5 }, { 5048, 0x5E0B }, { 5049, 0x5E1A }, { 5050, 0x5E19 }, { 5051, 0x5E11 }, { 5052, 0x5E1B }, { 5053, 0x5E36 },
    { 5054, 0x5E37 }, { 5055, 0x5E44 }, { 5056, 0x5E43 }, { 5057, 0x5E40 }, { 5058, 0x5E4E }, { 5059, 0x5E57 }, { 5060, 0x5E54 }, { 5061, 0x5E5F },
    { 5062, 0x5E62 }, { 5063, 0x5E64 }, { 5064, 0x5E47 }, { 5065, 0x5E75 }, { 5066, 0x5E76 }, { 5067, 0x5E7A }, { 5068, 0x9EBC }, { 5069, 0x5E7F },
    { 5070, 0x5EA0 }, { 5071, 0x5EC1 }, { 5072, 0x5EC2 }, { 5073, 0x5EC8 }, { 5074, 0x5ED0 }, { 5075, 0x5ECF }, { 5076, 0x5ED6 }, { 5077, 0x5EE3 },
    { 5078, 0x5EDD }, { 5079, 0x5EDA }, { 5080, 0x5EDB }, { 5081, 0x5EE2 }, { 5082, 0x5EE1 }, { 5083, 0x5EE8 }, { 5084, 0x5EE9 }, { 5085, 0x5EEC },
    { 5086, 0x5EF1 }, { 5087, 0x5EF3 }, { 5088, 0x5EF0 }, { 5089, 0x5EF4 }, { 5090, 0x5EF8 }, { 5091, 0x5EFE }, { 5092, 0x5F03 }, { 5093, 0x5F09 },
    { 5094, 0x5F5D }, { 5095, 0x5F5C }, { 5096, 0x5F0B }, { 5097, 0x5F11 }, { 5098, 0x5F16 }, { 5099, 0x5F29 }, { 5100, 0x5F2D }, { 5101, 0x5F38 },
    { 5102, 0x5F41 }, { 5103, 0x5F48 }, { 5104, 0x5F4C }, { 5105, 0x5F4E }, { 5106, 0x5F2F }, { 5107, 0x5F51 }, { 5108, 0x5F56 }, { 5109, 0x5F57 },
    { 5110, 0x5F59 }, { 5111, 0x5F61 }, { 5112, 0x5F6D }, { 5113, 0x5F73 }, { 5114, 0x5F77 }, { 5115, 0x5F83 }, { 5116, 0x5F82 }, { 5117, 0x5F7F },
    { 5118, 0x5F8A }, { 5119, 0x5F88 }, { 5120, 0x5F91 }, { 5121, 0x5F87 }, { 5122, 0x5F9E }, { 5123, 0x5F99 }, { 5124, 0x5F98 }, { 5125, 0x5FA0 },
    { 5126, 0x5FA8 }, { 5127, 0x5FAD }, { 5128, 0x5FBC }, { 5129, 0x5FD6 }, { 5130, 0x5FFB }, { 5131, 0x5FE4 }, { 5132, 0x5FF8 }, { 5133, 0x5FF1 },
    { 5134, 0x5FDD }, { 5135, 0x60B3 }, { 5136, 0x5FFF }, { 5137, 0x6021 }, { 5138, 0x6060 }, { 5139, 0x6019 }, { 5140, 0x6010 }, { 5141, 0x6029 },
    { 5142, 0x600E }, { 5143, 0x6031 }, { 5144, 0x601B }, { 5145, 0x6015 }, { 5146, 0x602B }, { 5147, 0x6026 }, { 5148, 0x600F }, { 5149, 0x603A },
    { 5150, 0x605A }, { 5151, 0x6041 }, { 5152, 0x606A }, { 5153, 0x6077 }, { 5154, 0x605F }, { 5155, 0x604A }, { 5156, 0x6046 }, { 5157, 0x604D },
    { 5158, 0x6063 }, { 5159, 0x6043 }, { 5160, 0x6064 }, { 5161, 0x6042 }, { 5162, 0x606C }, { 5163, 0x606B }, { 5164, 0x6059 }, { 5165, 0x6081 },
    { 5166, 0x608D }, { 5167, 0x60E7 }, { 5168, 0x6083 }, { 5169, 0x609A }, { 5170, 0x6084 }, { 5171, 0x609B }, { 5172, 0x6096 }, { 5173, 0x6097 },
    { 5174, 0x6092 }, { 5175, 0x60A7 }, { 5176, 0x608B }, { 5177, 0x60E1 }, { 5178, 0x60B8 }, { 5179, 0x60E0 }, { 5180, 0x60D3 }, { 5181, 0x60B4 },
    { 5182, 0x5FF0 }, { 5183, 0x60BD }, { 5184, 0x60C6 }, { 5185, 0x60B5 }, { 5186, 0x60D8 }, { 5187, 0x614D }, { 5188, 0x6115 }, { 5189, 0x6106 },
    { 5190, 0x60F6 }, { 5191, 0x60F7 }, { 5192, 0x6100 }, { 5193, 0x60F4 }, { 5194, 0x60FA }, { 5195, 0x6103 }, { 5196, 0x6121 }, { 5197, 0x60FB },
    { 5198, 0x60F1 }, { 5199, 0x610D }, { 5200, 0x610E }, { 5201, 0x6147 }, { 5202, 0x613E }, { 5203, 0x6128 }, { 5204, 0x6127 }, { 5205, 0x614A },
    { 5206, 0x613F }, { 5207, 0x613C }, { 5208, 0x612C }, { 5209, 0x6134 }, { 5210, 0x613D }, { 5211, 0x6142 }, { 5212, 0x6144 }, { 5213, 0x6173 },
    { 5214, 0x6177 }, { 5215, 0x6158 }, { 5216, 0x6159 }, { 5217, 0x615A }, { 5218, 0x616B }, { 5219, 0x6174 }, { 5220, 0x616F }, { 5221, 0x6165 },
    { 5222, 0x6171 }, { 5223, 0x615F }, { 5224, 0x615D }, { 5225, 0x6153 }, { 5226, 0x6175 }, { 5227, 0x6199 }, { 5228, 0x6196 }, { 5229, 0x6187 },
    { 5230, 0x61AC }, { 5231, 0x6194 }, { 5232, 0x619A }, { 5233, 0x618A }, { 5234, 0x6191 }, { 5235, 0x61AB }, { 5236, 0x61AE }, { 5237, 0x61CC },
    { 5238, 0x61CA }, { 5239, 0x61C9 }, { 5240, 0x61F7 }, { 5241, 0x61C8 }, { 5242, 0x61C3 }, { 5243, 0x61C6 }, { 5244, 0x61BA }, { 5245, 0x61CB },
    { 5246, 0x7F79 }, { 5247, 0x61CD }, { 5248, 0x61E6 }, { 5249, 0x61E3 }, { 5250, 0x61F6 }, { 5251, 0x61FA }, { 5252, 0x61F4 }, { 5253, 0x61FF },
    { 5254, 0x61FD }, { 5255, 0x61FC }, { 5256, 0x61FE }, { 5257, 0x6200 }, { 5258, 0x6208 }, { 5259, 0x6209 }, { 5260, 0x620D }, { 5261, 0x620C },
    { 5262, 0x6214 }, { 5263, 0x621B }, { 5264, 0x621E }, { 5265, 0x6221 }, { 5266, 0x622A }, { 5267, 0x622E }, { 5268, 0x6230 }, { 5269, 0x6232 },
    { 5270, 0x6233 }, { 5271, 0x6241 }, { 5272, 0x624E }, { 5273, 0x625E }, { 5274, 0x6263 }, { 5275, 0x625B }, { 5276, 0x6260 }, { 5277, 0x6268 },
    { 5278, 0x627C }, { 5279, 0x6282 }, { 5280, 0x6289 }, { 5281, 0x627E }, { 5282, 0x6292 }, { 5283, 0x6293 }, { 5284, 0x6296 }, { 5285, 0x62D4 },
    { 5286, 0x6283 }, { 5287, 0x6294 }, { 5288, 0x62D7 }, { 5289, 0x62D1 }, { 5290, 0x62BB }, { 5291, 0x62CF }, { 5292, 0x62FF }, { 5293, 0x62C6 },
    { 5294, 0x64D4 }, { 5295, 0x62C8 }, { 5296, 0x62DC }, { 5297, 0x62CC }, { 5298, 0x62CA }, { 5299, 0x62C2 }, { 5300, 0x62C7 }, { 5301, 0x629B },
    { 5302, 0x62C9 }, { 5303, 0x630C }, { 5304, 0x62EE }, { 5305, 0x62F1 }, { 5306, 0x6327 }, { 5307, 0x6302 }, { 5308, 0x6308 }, { 5309, 0x62EF },
    { 5310, 0x62F5 }, { 5311, 0x6350 }, { 5312, 0x633E }, { 5313, 0x634D }, { 5314, 0x641C }, { 5315, 0x634F }, { 5316, 0x6396 }, { 5317, 0x638E },
    { 5318, 0x6380 }, { 5319, 0x63AB }, { 5320, 0x6376 }, { 5321, 0x63A3 }, { 5322, 0x638F }, { 5323, 0x6389 }, { 5324, 0x639F }, { 5325, 0x63B5 },
    { 5326, 0x636B }, { 5327, 0x6369 }, { 5328, 0x63BE }, { 5329, 0x63E9 }, { 5330, 0x63C0 }, { 5331, 0x63C6 }, { 5332, 0x63E3 }, { 5333, 0x63C9 },
    { 5334, 0x63D2 }, { 5335, 0x63F6 }, { 5336, 0x63C4 }, { 5337, 0x6416 }, { 5338, 0x6434 }, { 5339, 0x6406 }, { 5340, 0x6413 }, { 5341, 0x6426 },
    { 5342, 0x6436 }, { 5343, 0x651D }, { 5344, 0x6417 }, { 5345, 0x6428 }, { 5346, 0x640F }, { 5347, 0x6467 }, { 5348, 0x646F }, { 5349, 0x6476 },
    { 5350, 0x644E }, { 5351, 0x652A }, { 5352, 0x6495 }, { 5353, 0x6493 }, { 5354, 0x64A5 }, { 5355, 0x64A9 }, { 5356, 0x6488 }, { 5357, 0x64BC },
    { 5358, 0x64DA }, { 5359, 0x64D2 }, { 5360, 0x64C5 }, { 5361, 0x64C7 }, { 5362, 0x64BB }, { 5363, 0x64D8 }, { 5364, 0x64C2 }, { 5365, 0x64F1 },
    { 5366, 0x64E7 }, { 5367, 0x8209 }, { 5368, 0x64E0 }, { 5369, 0x64E1 }, { 5370, 0x62AC }, { 5371, 0x64E3 }, { 5372, 0x64EF }, { 5373, 0x652C },
    { 5374, 0x64F6 }, { 5375, 0x64F4 }, { 5376, 0x64F2 }, { 5377, 0x64FA }, { 5378, 0x6500 }, { 5379, 0x64FD }, { 5380, 0x6518 }, { 5381, 0x651C },
    { 5382, 0x6505 }, { 5383, 0x6524 }, { 5384, 0x6523 }, { 5385, 0x652B }, { 5386, 0x6534 }, { 5387, 0x6535 }, { 5388, 0x6537 }, { 5389, 0x6536 },
    { 5390, 0x6538 }, { 5391, 0x754B }, { 5392, 0x6548 }, { 5393, 0x6556 }, { 5394, 0x6555 }, { 5395, 0x654D }, { 5396, 0x6558 }, { 5397, 0x655E },
    { 5398, 0x655D }, { 5399, 0x6572 }, { 5400, 0x6578 }, { 5401, 0x6582 }, { 5402, 0x6583 }, { 5403, 0x8B8A }, { 5404, 0x659B }, { 5405, 0x659F },
    { 5406, 0x65AB }, { 5407, 0x65B7 }, { 5408, 0x65C3 }, { 5409, 0x65C6 }, { 5410, 0x65C1 }, { 5411, 0x65C4 }, { 5412, 0x65CC }, { 5413, 0x65D2 },
    { 5414, 0x65DB }, { 5415, 0x65D9 }, { 5416, 0x65E0 }, { 5417, 0x65E1 }, { 5418, 0x65F1 }, { 5419, 0x6772 }, { 5420, 0x660A }, { 5421, 0x6603 },
    { 5422, 0x65FB }, { 5423, 0x6773 }, { 5424, 0x6635 }, { 5425, 0x6636 }, { 5426, 0x6634 }, { 5427, 0x661C }, { 5428, 0x664F }, { 5429, 0x6644 },
    { 5430, 0x6649 }, { 5431, 0x6641 }, { 5432, 0x665E }, { 5433, 0x665D }, { 5434, 0x6664 }, { 5435, 0x6667 }, { 5436, 0x6668 }, { 5437, 0x665F },
    { 5438, 0x6662 }, { 5439, 0x6670 }, { 5440, 0x6683 }, { 5441, 0x6688 }, { 5442, 0x668E }, { 5443, 0x6689 }, { 5444, 0x6684 }, { 5445, 0x6698 },
    { 5446, 0x669D }, { 5447, 0x66C1 }, { 5448, 0x66B9 }, { 5449, 0x66C9 }, { 5450, 0x66BE }, { 5451, 0x66BC }, { 5452, 0x66C4 }, { 5453, 0x66B8 },
    { 5454, 0x66D6 }, { 5455, 0x66DA }, { 5456, 0x66E0 }, { 5457, 0x663F }, { 5458, 0x66E6 }, { 5459, 0x66E9 }, { 5460, 0x66F0 }, { 5461, 0x66F5 },
    { 5462, 0x66F7 }, { 5463, 0x670F }, { 5464, 0x6716 }, { 5465, 0x671E }, { 5466, 0x6726 }, { 5467, 0x6727 }, { 5468, 0x9738 }, { 5469, 0x672E },
    { 5470, 0x673F }, { 5471, 0x6736 }, { 5472, 0x6741 }, { 5473, 0x6738 }, { 5474, 0x6737 }, { 5475, 0x6746 }, { 5476, 0x675E }, { 5477, 0x6760 },
    { 5478, 0x6759 }, { 5479, 0x6763 }, { 5480, 0x6764 }, { 5481, 0x6789 }, { 5482, 0x6770 }, { 5483, 0x67A9 }, { 5484, 0x677C }, { 5485, 0x676A },
    { 5486, 0x678C }, { 5487, 0x678B }, { 5488, 0x67A6 }, { 5489, 0x67A1 }, { 5490, 0x6785 }, { 5491, 0x67B7 }, { 5492, 0x67EF }, { 5493, 0x67B4 },
    { 5494, 0x67EC }, { 5495, 0x67B3 }, { 5496, 0x67E9 }, { 5497, 0x67B8 }, { 5498, 0x67E4 }, { 5499, 0x67DE }, { 5500, 0x67DD }, { 5501, 0x67E2 },
    { 5502, 0x67EE }, { 5503, 0x67B9 }, { 5504, 0x67CE }, { 5505, 0x67C6 }, { 5506, 0x67E7 }, { 5507, 0x6A9C }, { 5508, 0x681E }, { 5509, 0x6846 },
    { 5510, 0x6829 }, { 5511, 0x6840 }, { 5512, 0x684D }, { 5513, 0x6832 }, { 5514, 0x684E }, { 5515, 0x68B3 }, { 5516, 0x682B }, { 5517, 0x6859 },
    { 5518, 0x6863 }, { 5519, 0x6877 }, { 5520, 0x687F }, { 5521, 0x689F }, { 5522, 0x688F }, { 5523, 0x68AD }, { 5524, 0x6894 }, { 5525, 0x689D },
    { 5526, 0x689B }, { 5527, 0x6883 }, { 5528, 0x6AAE }, { 5529, 0x68B9 }, { 5530, 0x6874 }, { 5531, 0x68B5 }, { 5532, 0x68A0 }, { 5533, 0x68BA },
    { 5534, 0x690F }, { 5535, 0x688D }, { 5536, 0x687E }, { 5537, 0x6901 }, { 5538, 0x68CA }, { 5539, 0x6908 }, { 5540, 0x68D8 }, { 5541, 0x6922 },
    { 5542, 0x6926 }, { 5543, 0x68E1 }, { 5544, 0x690C }, { 5545, 0x68CD }, { 5546, 0x68D4 }, { 5547, 0x68E7 }, { 5548, 0x68D5 }, { 5549, 0x6936 },
    { 5550, 0x6912 }, { 5551, 0x6904 }, { 5552, 0x68D7 }, { 5553, 0x68E3 }, { 5554, 0x6925 }, { 5555, 0x68F9 }, { 5556, 0x68E0 }, { 5557, 0x68EF },
    { 5558, 0x6928 }, { 5559, 0x692A }, { 5560, 0x691A }, { 5561, 0x6923 }, { 5562, 0x6921 }, { 5563, 0x68C6 }, { 5564, 0x6979 }, { 5565, 0x6977 },
    { 5566, 0x695C }, { 5567, 0x6978 }, { 5568, 0x696B }, { 5569, 0x6954 }, { 5570, 0x697E }, { 5571, 0x696E }, { 5572, 0x6939 }, { 5573, 0x6974 },
    { 5574, 0x693D }, { 5575, 0x6959 }, { 5576, 0x6930 }, { 5577, 0x6961 }, { 5578, 0x695E }, { 5579, 0x695D }, { 5580, 0x6981 }, { 5581, 0x696A },
    { 5582, 0x69B2 }, { 5583, 0x69AE }, { 5584, 0x69D0 }, { 5585, 0x69BF }, { 5586, 0x69C1 }, { 5587, 0x69D3 }, { 5588, 0x69BE }, { 5589, 0x69CE },
    { 5590, 0x5BE8 }, { 5591, 0x69CA }, { 5592, 0x69DD }, { 5593, 0x69BB }, { 5594, 0x69C3 }, { 5595, 0x69A7 }, { 5596, 0x6A2E }, { 5597, 0x6991 },
    { 5598, 0x69A0 }, { 5599, 0x699C }, { 5600, 0x6995 }, { 5601, 0x69B4 }, { 5602, 0x69DE }, { 5603, 0x69E8 }, { 5604, 0x6A02 }, { 5605, 0x6A1B },
    { 5606, 0x69FF }, { 5607, 0x6B0A }, { 5608, 0x69F9 }, { 5609, 0x69F2 }, { 5610, 0x69E7 }, { 5611, 0x6A05 }, { 5612, 0x69B1 }, { 5613, 0x6A1E },
    { 5614, 0x69ED }, { 5615, 0x6A14 }, { 5616, 0x69EB }, { 5617, 0x6A0A }, { 5618, 0x6A12 }, { 5619, 0x6AC1 }, { 5620, 0x6A23 }, { 5621, 0x6A13 },
    { 5622, 0x6A44 }, { 5623, 0x6A0C }, { 5624, 0x6A72 }, { 5625, 0x6A36 }, { 5626, 0x6A78 }, { 5627, 0x6A47 }, { 5628, 0x6A62 }, { 5629, 0x6A59 },
    { 5630, 0x6A66 }, { 5631, 0x6A48 }, { 5632, 0x6A38 }, { 5633, 0x6A22 }, { 5634, 0x6A90 }, { 5635, 0x6A8D }, { 5636, 0x6AA0 }, { 5637, 0x6A84 },
    { 5638, 0x6AA2 }, { 5639, 0x6AA3 }, { 5640, 0x6A97 }, { 5641, 0x8617 }, { 5642, 0x6ABB }, { 5643, 0x6AC3 }, { 5644, 0x6AC2 }, { 5645, 0x6AB8 },
    { 5646, 0x6AB3 }, { 5647, 0x6AAC }, { 5648, 0x6ADE }, { 5649, 0x6AD1 }, { 5650, 0x6ADF }, { 5651, 0x6AAA }, { 5652, 0x6ADA }, { 5653, 0x6AEA },
    { 5654, 0x6AFB }, { 5655, 0x6B05 }, { 5656, 0x8616 }, { 5657, 0x6AFA }, { 5658, 0x6B12 }, { 5659, 0x6B16 }, { 5660, 0x9B31 }, { 5661, 0x6B1F },
    { 5662, 0x6B38 }, { 5663, 0x6B37 }, { 5664, 0x76DC }, { 5665, 0x6B39 }, { 5666, 0x98EE }, { 5667, 0x6B47 }, { 5668, 0x6B43 }, { 5669, 0x6B49 },
    { 5670, 0x6B50 }, { 5671, 0x6B59 }, { 5672, 0x6B54 }, { 5673, 0x6B5B }, { 5674, 0x6B5F }, { 5675, 0x6B61 }, { 5676, 0x6B78 }, { 5677, 0x6B79 },
    { 5678, 0x6B7F }, { 5679, 0x6B80 }, { 5680, 0x6B84 }, { 5681, 0x6B83 }, { 5682, 0x6B8D }, { 5683, 0x6B98 }, { 5684, 0x6B95 }, { 5685, 0x6B9E },
    { 5686, 0x6BA4 }, { 5687, 0x6BAA }, { 5688, 0x6BAB }, { 5689, 0x6BAF }, { 5690, 0x6BB2 }, { 5691, 0x6BB1 }, { 5692, 0x6BB3 }, { 5693, 0x6BB7 },
    { 5694, 0x6BBC }, { 5695, 0x6BC6 }, { 5696, 0x6BCB }, { 5697, 0x6BD3 }, { 5698, 0x6BDF }, { 5699, 0x6BEC }, { 5700, 0x6BEB }, { 5701, 0x6BF3 },
    { 5702, 0x6BEF }, { 5703, 0x9EBE }, { 5704, 0x6C08 }, { 5705, 0x6C13 }, { 5706, 0x6C14 }, { 5707, 0x6C1B }, { 5708, 0x6C24 }, { 5709, 0x6C23 },
    { 5710, 0x6C5E }, { 5711, 0x6C55 }, { 5712, 0x6C62 }, { 5713, 0x6C6A }, { 5714, 0x6C82 }, { 5715, 0x6C8D }, { 5716, 0x6C9A }, { 5717, 0x6C81 },
    { 5718, 0x6C9B }, { 5719, 0x6C7E }, { 5720, 0x6C68 }, { 5721, 0x6C73 }, { 5722, 0x6C92 }, { 5723, 0x6C90 }, { 5724, 0x6CC4 }, { 5725, 0x6CF1 },
    { 5726, 0x6CD3 }, { 5727, 0x6CBD }, { 5728, 0x6CD7 }, { 5729, 0x6CC5 }, { 5730, 0x6CDD }, { 5731, 0x6CAE }, { 5732, 0x6CB1 }, { 5733, 0x6CBE },
    { 5734, 0x6CBA }, { 5735, 0x6CDB }, { 5736, 0x6CEF }, { 5737, 0x6CD9 }, { 5738, 0x6CEA }, { 5739, 0x6D1F }, { 5740, 0x884D }, { 5741, 0x6D36 },
    { 5742, 0x6D2B }, { 5743, 0x6D3D }, { 5744, 0x6D38 }, { 5745, 0x6D19 }, { 5746, 0x6D35 }, { 5747, 0x6D33 }, { 5748, 0x6D12 }, { 5749, 0x6D0C },
    { 5750, 0x6D63 }, { 5751, 0x6D93 }, { 5752, 0x6D64 }, { 5753, 0x6D5A }, { 5754, 0x6D79 }, { 5755, 0x6D59 }, { 5756, 0x6D8E }, { 5757, 0x6D95 },
    { 5758, 0x6FE4 }, { 5759, 0x6D85 }, { 5760, 0x6DF9 }, { 5761, 0x6E15 }, { 5762, 0x6E0A }, { 5763, 0x6DB5 }, { 5764, 0x6DC7 }, { 5765, 0x6DE6 },
    { 5766, 0x6DB8 }, { 5767, 0x6DC6 }, { 5768, 0x6DEC }, { 5769, 0x6DDE }, { 5770, 0x6DCC }, { 5771, 0x6DE8 }, { 5772, 0x6DD2 }, { 5773, 0x6DC5 },
    { 5774, 0x6DFA }, { 5775, 0x6DD9 }, { 5776, 0x6DE4 }, { 5777, 0x6DD5 }, { 5778, 0x6DEA }, { 5779, 0x6DEE }, { 5780, 0x6E2D }, { 5781, 0x6E6E },
    { 5782, 0x6E2E }, { 5783, 0x6E19 }, { 5784, 0x6E72 }, { 5785, 0x6E5F }, { 5786, 0x6E3E }, { 5787, 0x6E23 }, { 5788, 0x6E6B }, { 5789, 0x6E2B },
    { 5790, 0x6E76 }, { 5791, 0x6E4D }, { 5792, 0x6E1F }, { 5793, 0x6E43 }, { 5794, 0x6E3A }, { 5795, 0x6E4E }, { 5796, 0x6E24 }, { 5797, 0x6EFF },
    { 5798, 0x6E1D }, { 5799, 0x6E38 }, { 5800, 0x6E82 }, { 5801, 0x6EAA }, { 5802, 0x6E98 }, { 5803, 0x6EC9 }, { 5804, 0x6EB7 }, { 5805, 0x6ED3 },
    { 5806, 0x6EBD }, { 5807, 0x6EAF }, { 5808, 0x6EC4 }, { 5809, 0x6EB2 }, { 5810, 0x6ED4 }, { 5811, 0x6ED5 }, { 5812, 0x6E8F }, { 5813, 0x6EA5 },
    { 5814, 0x6EC2 }, { 5815, 0x6E9F }, { 5816, 0x6F41 }, { 5817, 0x6F11 }, { 5818, 0x704C }, { 5819, 0x6EEC }, { 5820, 0x6EF8 }, { 5821, 0x6EFE },
    { 5822, 0x6F3F }, { 5823, 0x6EF2 }, { 5824, 0x6F31 }, { 5825, 0x6EEF }, { 5826, 0x6F32 }, { 5827, 0x6ECC }, { 5828, 0x6F3E }, { 5829, 0x6F13 },
    { 5830, 0x6EF7 }, { 5831, 0x6F86 }, { 5832, 0x6F7A }, { 5833, 0x6F78 }, { 5834, 0x6F81 }, { 5835, 0x6F80 }, { 5836, 0x6F6F }, { 5837, 0x6F5B },
    { 5838, 0x6FF3 }, { 5839, 0x6F6D }, { 5840, 0x6F82 }, { 5841, 0x6F7C }, { 5842, 0x6F58 }, { 5843, 0x6F8E }, { 5844, 0x6F91 }, { 5845, 0x6FC2 },
    { 5846, 0x6F66 }, { 5847, 0x6FB3 }, { 5848, 0x6FA3 }, { 5849, 0x6FA1 }, { 5850, 0x6FA4 }, { 5851, 0x6FB9 }, { 5852, 0x6FC6 }, { 5853, 0x6FAA },
    { 5854, 0x6FDF }, { 5855, 0x6FD5 }, { 5856, 0x6FEC }, { 5857, 0x6FD4 }, { 5858, 0x6FD8 }, { 5859, 0x6FF1 }, { 5860, 0x6FEE }, { 5861, 0x6FDB },
    { 5862, 0x7009 }, { 5863, 0x700B }, { 5864, 0x6FFA }, { 5865, 0x7011 }, { 5866, 0x7001 }, { 5867, 0x700F }, { 5868, 0x6FFE }, { 5869, 0x701B },
    { 5870, 0x701A }, { 5871, 0x6F74 }, { 5872, 0x701D }, { 5873, 0x7018 }, { 5874, 0x701F }, { 5875, 0x7030 }, { 5876, 0x703E }, { 5877, 0x7032 },
    { 5878, 0x7051 }, { 5879, 0x7063 }, { 5880, 0x7099 }, { 5881, 0x7092 }, { 5882, 0x70AF }, { 5883, 0x70F1 }, { 5884, 0x70AC }, { 5885, 0x70B8 },
    { 5886, 0x70B3 }, { 5887, 0x70AE }, { 5888, 0x70DF }, { 5889, 0x70CB }, { 5890, 0x70DD }, { 5891, 0x70D9 }, { 5892, 0x7109 }, { 5893, 0x70FD },
    { 5894, 0x711C }, { 5895, 0x7119 }, { 5896, 0x7165 }, { 5897, 0x7155 }, { 5898, 0x7188 }, { 5899, 0x7166 }, { 5900, 0x7162 }, { 5901, 0x714C },
    { 5902, 0x7156 }, { 5903, 0x716C }, { 5904, 0x718F }, { 5905, 0x71FB }, { 5906, 0x7184 }, { 5907, 0x7195 }, { 5908, 0x71A8 }, { 5909, 0x71AC },
    { 5910, 0x71D7 }, { 5911, 0x71B9 }, { 5912, 0x71BE }, { 5913, 0x71D2 }, { 5914, 0x71C9 }, { 5915, 0x71D4 }, { 5916, 0x71CE }, { 5917, 0x71E0 },
    { 5918, 0x71EC }, { 5919, 0x71E7 }, { 5920, 0x71F5 }, { 5921, 0x71FC }, { 5922, 0x71F9 }, { 5923, 0x71FF }, { 5924, 0x720D }, { 5925, 0x7210 },
    { 5926, 0x721B }, { 5927, 0x7228 }, { 5928, 0x722D }, { 5929, 0x722C }, { 5930, 0x7230 }, { 5931, 0x7232 }, { 5932, 0x723B }, { 5933, 0x723C },
    { 5934, 0x723F }, { 5935, 0x7240 }, { 5936, 0x7246 }, { 5937, 0x724B }, { 5938, 0x7258 }, { 5939, 0x7274 }, { 5940, 0x727E }, { 5941, 0x7282 },
    { 5942, 0x7281 }, { 5943, 0x7287 }, { 5944, 0x7292 }, { 5945, 0x7296 }, { 5946, 0x72A2 }, { 5947, 0x72A7 }, { 5948, 0x72B9 }, { 5949, 0x72B2 },
    { 5950, 0x72C3 }, { 5951, 0x72C6 }, { 5952, 0x72C4 }, { 5953, 0x72CE }, { 5954, 0x72D2 }, { 5955, 0x72E2 }, { 5956, 0x72E0 }, { 5957, 0x72E1 },
    { 5958, 0x72F9 }, { 5959, 0x72F7 }, { 5960, 0x500F }, { 5961, 0x7317 }, { 5962, 0x730A }, { 5963, 0x731C }, { 5964, 0x7316 }, { 5965, 0x731D },
    { 5966, 0x7334 }, { 5967, 0x732F }, { 5968, 0x7329 }, { 5969, 0x7325 }, { 5970, 0x733E }, { 5971, 0x734E }, { 5972, 0x734F }, { 5973, 0x9ED8 },
    { 5974, 0x7357 }, { 5975, 0x736A }, { 5976, 0x7368 }, { 5977, 0x7370 }, { 5978, 0x7378 }, { 5979, 0x7375 }, { 5980, 0x737B }, { 5981, 0x737A },
    { 5982, 0x73C8 }, { 5983, 0x73B3 }, { 5984, 0x73CE }, { 5985, 0x73BB }, { 5986, 0x73C0 }, { 5987, 0x73E5 }, { 5988, 0x73EE }, { 5989, 0x73DE },
    { 5990, 0x74A2 }, { 5991, 0x7405 }, { 5992, 0x746F }, { 5993, 0x7425 }, { 5994, 0x73F8 }, { 5995, 0x7432 }, { 5996, 0x743A }, { 5997, 0x7455 },
    { 5998, 0x743F }, { 5999, 0x745F }, { 6000, 0x7459 }, { 6001, 0x7441 }, { 6002, 0x745C }, { 6003, 0x7469 }, { 6004, 0x7470 }, { 6005, 0x7463 },
    { 6006, 0x746A }, { 6007, 0x7476 }, { 6008, 0x747E }, { 6009, 0x748B }, { 6010, 0x749E }, { 6011, 0x74A7 }, { 6012, 0x74CA }, { 6013, 0x74CF },
    { 6014, 0x74D4 }, { 6015, 0x73F1 }, { 6016, 0x74E0 }, { 6017, 0x74E3 }, { 6018, 0x74E7 }, { 6019, 0x74E9 }, { 6020, 0x74EE }, { 6021, 0x74F2 },
    { 6022, 0x74F0 }, { 6023, 0x74F1 }, { 6024, 0x74F8 }, { 6025, 0x74F7 }, { 6026, 0x7504 }, { 6027, 0x7503 }, { 6028, 0x7505 }, { 6029, 0x750C },
    { 6030, 0x750E }, { 6031, 0x750D }, { 6032, 0x7515 }, { 6033, 0x7513 }, { 6034, 0x751E }, { 6035, 0x7526 }, { 6036, 0x752C }, { 6037, 0x753C },
    { 6038, 0x7544 }, { 6039, 0x754D }, { 6040, 0x754A }, { 6041, 0x7549 }, { 6042, 0x755B }, { 6043, 0x7546 }, { 6044, 0x755A }, { 6045, 0x7569 },
    { 6046, 0x7564 }, { 6047, 0x7567 }, { 6048, 0x756B }, { 6049, 0x756D }, { 6050, 0x7578 }, { 6051, 0x7576 }, { 6052, 0x7586 }, { 6053, 0x7587 },
    { 6054, 0x7574 }, { 6055, 0x758A }, { 6056, 0x7589 }, { 6057, 0x7582 }, { 6058, 0x7594 }, { 6059, 0x759A }, { 6060, 0x759D }, { 6061, 0x75A5 },
    { 6062, 0x75A3 }, { 6063, 0x75C2 }, { 6064, 0x75B3 }, { 6065, 0x75C3 }, { 6066, 0x75B5 }, { 6067, 0x75BD }, { 6068, 0x75B8 }, { 6069, 0x75BC },
    { 6070, 0x75B1 }, { 6071, 0x75CD }, { 6072, 0x75CA }, { 6073, 0x75D2 }, { 6074, 0x75D9 }, { 6075, 0x75E3 }, { 6076, 0x75DE }, { 6077, 0x75FE },
    { 6078, 0x75FF }, { 6079, 0x75FC }, { 6080, 0x7601 }, { 6081, 0x75F0 }, { 6082, 0x75FA }, { 6083, 0x75F2 }, { 6084, 0x75F3 }, { 6085, 0x760B },
    { 6086, 0x760D }, { 6087, 0x7609 }, { 6088, 0x761F }, { 6089, 0x7627 }, { 6090, 0x7620 }, { 6091, 0x7621 }, { 6092, 0x7622 }, { 6093, 0x7624 },
    { 6094, 0x7634 }, { 6095, 0x7630 }, { 6096, 0x763B }, { 6097, 0x7647 }, { 6098, 0x7648 }, { 6099, 0x7646 }, { 6100, 0x765C }, { 6101, 0x7658 },
    { 6102, 0x7661 }, { 6103, 0x7662 }, { 6104, 0x7668 }, { 6105, 0x7669 }, { 6106, 0x766A }, { 6107, 0x7667 }, { 6108, 0x766C }, { 6109, 0x7670 },
    { 6110, 0x7672 }, { 6111, 0x7676 }, { 6112, 0x7678 }, { 6113, 0x767C }, { 6114, 0x7680 }, { 6115, 0x7683 }, { 6116, 0x7688 }, { 6117, 0x768B },
    { 6118, 0x768E }, { 6119, 0x7696 }, { 6120, 0x7693 }, { 6121, 0x7699 }, { 6122, 0x769A }, { 6123, 0x76B0 }, { 6124, 0x76B4 }, { 6125, 0x76B8 },
    { 6126, 0x76B9 }, { 6127, 0x76BA }, { 6128, 0x76C2 }, { 6129, 0x76CD }, { 6130, 0x76D6 }, { 6131, 0x76D2 }, { 6132, 0x76DE }, { 6133, 0x76E1 },
    { 6134, 0x76E5 }, { 6135, 0x76E7 }, { 6136, 0x76EA }, { 6137, 0x862F }, { 6138, 0x76FB }, { 6139, 0x7708 }, { 6140, 0x7707 }, { 6141, 0x7704 },
    { 6142, 0x7729 }, { 6143, 0x7724 }, { 6144, 0x771E }, { 6145, 0x7725 }, { 6146, 0x7726 }, { 6147, 0x771B }, { 6148, 0x7737 }, { 6149, 0x7738 },
    { 6150, 0x7747 }, { 6151, 0x775A }, { 6152, 0x7768 }, { 6153, 0x776B }, { 6154, 0x775B }, { 6155, 0x7765 }, { 6156, 0x777F }, { 6157, 0x777E },
    { 6158, 0x7779 }, { 6159, 0x778E }, { 6160, 0x778B }, { 6161, 0x7791 }, { 6162, 0x77A0 }, { 6163, 0x779E }, { 6164, 0x77B0 }, { 6165, 0x77B6 },
    { 6166, 0x77B9 }, { 6167, 0x77BF }, { 6168, 0x77BC }, { 6169, 0x77BD }, { 6170, 0x77BB }, { 6171, 0x77C7 }, { 6172, 0x77CD }, { 6173, 0x77D7 },
    { 6174, 0x77DA }, { 6175, 0x77DC }, { 6176, 0x77E3 }, { 6177, 0x77EE }, { 6178, 0x77FC }, { 6179, 0x780C }, { 6180, 0x7812 }, { 6181, 0x7926 },
    { 6182, 0x7820 }, { 6183, 0x792A }, { 6184, 0x7845 }, { 6185, 0x788E }, { 6186, 0x7874 }, { 6187, 0x7886 }, { 6188, 0x787C }, { 6189, 0x789A },
    { 6190, 0x788C }, { 6191, 0x78A3 }, { 6192, 0x78B5 }, { 6193, 0x78AA }, { 6194, 0x78AF }, { 6195, 0x78D1 }, { 6196, 0x78C6 }, { 6197, 0x78CB },
    { 6198, 0x78D4 }, { 6199, 0x78BE }, { 6200, 0x78BC }, { 6201, 0x78C5 }, { 6202, 0x78CA }, { 6203, 0x78EC }, { 6204, 0x78E7 }, { 6205, 0x78DA },
    { 6206, 0x78FD }, { 6207, 0x78F4 }, { 6208, 0x7907 }, { 6209, 0x7912 }, { 6210, 0x7911 }, { 6211, 0x7919 }, { 6212, 0x792C }, { 6213, 0x792B },
    { 6214, 0x7940 }, { 6215, 0x7960 }, { 6216, 0x7957 }, { 6217, 0x795F }, { 6218, 0x795A }, { 6219, 0x7955 }, { 6220, 0x7953 }, { 6221, 0x797A },
    { 6222, 0x797F }, { 6223, 0x798A }, { 6224, 0x799D }, { 6225, 0x79A7 }, { 6226, 0x9F4B }, { 6227, 0x79AA }, { 6228, 0x79AE }, { 6229, 0x79B3 },
    { 6230, 0x79B9 }, { 6231, 0x79BA }, { 6232, 0x79C9 }, { 6233, 0x79D5 }, { 6234, 0x79E7 }, { 6235, 0x79EC }, { 6236, 0x79E1 }, { 6237, 0x79E3 },
    { 6238, 0x7A08 }, { 6239, 0x7A0D }, { 6240, 0x7A18 }, { 6241, 0x7A19 }, { 6242, 0x7A20 }, { 6243, 0x7A1F }, { 6244, 0x7980 }, { 6245, 0x7A31 },
    { 6246, 0x7A3B }, { 6247, 0x7A3E }, { 6248, 0x7A37 }, { 6249, 0x7A43 }, { 6250, 0x7A57 }, { 6251, 0x7A49 }, { 6252, 0x7A61 }, { 6253, 0x7A62 },
    { 6254, 0x7A69 }, { 6255, 0x9F9D }, { 6256, 0x7A70 }, { 6257, 0x7A79 }, { 6258, 0x7A7D }, { 6259, 0x7A88 }, { 6260, 0x7A97 }, { 6261, 0x7A95 },
    { 6262, 0x7A98 }, { 6263, 0x7A96 }, { 6264, 0x7AA9 }, { 6265, 0x7AC8 }, { 6266, 0x7AB0 }, { 6267, 0x7AB6 }, { 6268, 0x7AC5 }, { 6269, 0x7AC4 },
    { 6270, 0x7ABF }, { 6271, 0x9083 }, { 6272, 0x7AC7 }, { 6273, 0x7ACA }, { 6274, 0x7ACD }, { 6275, 0x7ACF }, { 6276, 0x7AD5 }, { 6277, 0x7AD3 },
    { 6278, 0x7AD9 }, { 6279, 0x7ADA }, { 6280, 0x7ADD }, { 6281, 0x7AE1 }, { 6282, 0x7AE2 }, { 6283, 0x7AE6 }, { 6284, 0x7AED }, { 6285, 0x7AF0 },
    { 6286, 0x7B02 }, { 6287, 0x7B0F }, { 6288, 0x7B0A }, { 6289, 0x7B06 }, { 6290, 0x7B33 }, { 6291, 0x7B18 }, { 6292, 0x7B19 }, { 6293, 0x7B1E },
    { 6294, 0x7B35 }, { 6295, 0x7B28 }, { 6296, 0x7B36 }, { 6297, 0x7B50 }, { 6298, 0x7B7A }, { 6299, 0x7B04 }, { 6300, 0x7B4D }, { 6301, 0x7B0B },
    { 6302, 0x7B4C }, { 6303, 0x7B45 }, { 6304, 0x7B75 }, { 6305, 0x7B65 }, { 6306, 0x7B74 }, { 6307, 0x7B67 }, { 6308, 0x7B70 }, { 6309, 0x7B71 },
    { 6310, 0x7B6C }, { 6311, 0x7B6E }, { 6312, 0x7B9D }, { 6313, 0x7B98 }, { 6314, 0x7B9F }, { 6315, 0x7B8D }, { 6316, 0x7B9C }, { 6317, 0x7B9A },
    { 6318, 0x7B8B }, { 6319, 0x7B92 }, { 6320, 0x7B8F }, { 6321, 0x7B5D }, { 6322, 0x7B99 }, { 6323, 0x7BCB }, { 6324, 0x7BC1 }, { 6325, 0x7BCC },
    { 6326, 0x7BCF }, { 6327, 0x7BB4 }, { 6328, 0x7BC6 }, { 6329, 0x7BDD }, { 6330, 0x7BE9 }, { 6331, 0x7C11 }, { 6332, 0x7C14 }, { 6333, 0x7BE6 },
    { 6334, 0x7BE5 }, { 6335, 0x7C60 }, { 6336, 0x7C00 }, { 6337, 0x7C07 }, { 6338, 0x7C13 }, { 6339, 0x7BF3 }, { 6340, 0x7BF7 }, { 6341, 0x7C17 },
    { 6342, 0x7C0D }, { 6343, 0x7BF6 }, { 6344, 0x7C23 }, { 6345, 0x7C27 }, { 6346, 0x7C2A }, { 6347, 0x7C1F }, { 6348, 0x7C37 }, { 6349, 0x7C2B },
    { 6350, 0x7C3D }, { 6351, 0x7C4C }, { 6352, 0x7C43 }, { 6353, 0x7C54 }, { 6354, 0x7C4F }, { 6355, 0x7C40 }, { 6356, 0x7C50 }, { 6357, 0x7C58 },
    { 6358, 0x7C5F }, { 6359, 0x7C64 }, { 6360, 0x7C56 }, { 6361, 0x7C65 }, { 6362, 0x7C6C }, { 6363, 0x7C75 }, { 6364, 0x7C83 }, { 6365, 0x7C90 },
    { 6366, 0x7CA4 }, { 6367, 0x7CAD }, { 6368, 0x7CA2 }, { 6369, 0x7CAB }, { 6370, 0x7CA1 }, { 6371, 0x7CA8 }, { 6372, 0x7CB3 }, { 6373, 0x7CB2 },
    { 6374, 0x7CB1 }, { 6375, 0x7CAE }, { 6376, 0x7CB9 }, { 6377, 0x7CBD }, { 6378, 0x7CC0 }, { 6379, 0x7CC5 }, { 6380, 0x7CC2 }, { 6381, 0x7CD8 },
    { 6382, 0x7CD2 }, { 6383, 0x7CDC }, { 6384, 0x7CE2 }, { 6385, 0x9B3B }, { 6386, 0x7CEF }, { 6387, 0x7CF2 }, { 6388, 0x7CF4 }, { 6389, 0x7CF6 },
    { 6390, 0x7CFA }, { 6391, 0x7D06 }, { 6392, 0x7D02 }, { 6393, 0x7D1C }, { 6394, 0x7D15 }, { 6395, 0x7D0A }, { 6396, 0x7D45 }, { 6397, 0x7D4B },
    { 6398, 0x7D2E }, { 6399, 0x7D32 }, { 6400, 0x7D3F }, { 6401, 0x7D35 }, { 6402, 0x7D46 }, { 6403, 0x7D73 }, { 6404, 0x7D56 }, { 6405, 0x7D4E },
    { 6406, 0x7D72 }, { 6407, 0x7D68 }, { 6408, 0x7D6E }, { 6409, 0x7D4F }, { 6410, 0x7D63 }, { 6411, 0x7D93 }, { 6412, 0x7D89 }, { 6413, 0x7D5B },
    { 6414, 0x7D8F }, { 6415, 0x7D7D }, { 6416, 0x7D9B }, { 6417, 0x7DBA }, { 6418, 0x7DAE }, { 6419, 0x7DA3 }, { 6420, 0x7DB5 }, { 6421, 0x7DC7 },
    { 6422, 0x7DBD }, { 6423, 0x7DAB }, { 6424, 0x7E3D }, { 6425, 0x7DA2 }, { 6426, 0x7DAF }, { 6427, 0x7DDC }, { 6428, 0x7DB8 }, { 6429, 0x7D9F },
    { 6430, 0x7DB0 }, { 6431, 0x7DD8 }, { 6432, 0x7DDD }, { 6433, 0x7DE4 }, { 6434, 0x7DDE }, { 6435, 0x7DFB }, { 6436, 0x7DF2 }, { 6437, 0x7DE1 },
    { 6438, 0x7E05 }, { 6439, 0x7E0A }, { 6440, 0x7E23 }, { 6441, 0x7E21 }, { 6442, 0x7E12 }, { 6443, 0x7E31 }, { 6444, 0x7E1F }, { 6445, 0x7E09 },
    { 6446, 0x7E0B }, { 6447, 0x7E22 }, { 6448, 0x7E46 }, { 6449, 0x7E66 }, { 6450, 0x7E3B }, { 6451, 0x7E35 }, { 6452, 0x7E39 }, { 6453, 0x7E43 },
    { 6454, 0x7E37 }, { 6455, 0x7E32 }, { 6456, 0x7E3A }, { 6457, 0x7E67 }, { 6458, 0x7E5D }, { 6459, 0x7E56 }, { 6460, 0x7E5E }, { 6461, 0x7E59 },
    { 6462, 0x7E5A }, { 6463, 0x7E79 }, { 6464, 0x7E6A }, { 6465, 0x7E69 }, { 6466, 0x7E7C }, { 6467, 0x7E7B }, { 6468, 0x7E83 }, { 6469, 0x7DD5 },
    { 6470, 0x7E7D }, { 6471, 0x8FAE }, { 6472, 0x7E7F }, { 6473, 0x7E88 }, { 6474, 0x7E89 }, { 6475, 0x7E8C }, { 6476, 0x7E92 }, { 6477, 0x7E90 },
    { 6478, 0x7E93 }, { 6479, 0x7E94 }, { 6480, 0x7E96 }, { 6481, 0x7E8E }, { 6482, 0x7E9B }, { 6483, 0x7E9C }, { 6484, 0x7F38 }, { 6485, 0x7F3A },
    { 6486, 0x7F45 }, { 6487, 0x7F4C }, { 6488, 0x7F4D }, { 6489, 0x7F4E }, { 6490, 0x7F50 }, { 6491, 0x7F51 }, { 6492, 0x7F55 }, { 6493, 0x7F54 },
    { 6494, 0x7F58 }, { 6495, 0x7F5F }, { 6496, 0x7F60 }, { 6497, 0x7F68 }, { 6498, 0x7F69 }, { 6499, 0x7F67 }, { 6500, 0x7F78 }, { 6501, 0x7F82 },
    { 6502, 0x7F86 }, { 6503, 0x7F83 }, { 6504, 0x7F88 }, { 6505, 0x7F87 }, { 6506, 0x7F8C }, { 6507, 0x7F94 }, { 6508, 0x7F9E }, { 6509, 0x7F9D },
    { 6510, 0x7F9A }, { 6511, 0x7FA3 }, { 6512, 0x7FAF }, { 6513, 0x7FB2 }, { 6514, 0x7FB9 }, { 6515, 0x7FAE }, { 6516, 0x7FB6 }, { 6517, 0x7FB8 },
    { 6518, 0x8B71 }, { 6519, 0x7FC5 }, { 6520, 0x7FC6 }, { 6521, 0x7FCA }, { 6522, 0x7FD5 }, { 6523, 0x7FD4 }, { 6524, 0x7FE1 }, { 6525, 0x7FE6 },
    { 6526, 0x7FE9 }, { 6527, 0x7FF3 }, { 6528, 0x7FF9 }, { 6529, 0x98DC }, { 6530, 0x8006 }, { 6531, 0x8004 }, { 6532, 0x800B }, { 6533, 0x8012 },
    { 6534, 0x8018 }, { 6535, 0x8019 }, { 6536, 0x801C }, { 6537, 0x8021 }, { 6538, 0x8028 }, { 6539, 0x803F }, { 6540, 0x803B }, { 6541, 0x804A },
    { 6542, 0x8046 }, { 6543, 0x8052 }, { 6544, 0x8058 }, { 6545, 0x805A }, { 6546, 0x805F }, { 6547, 0x8062 }, { 6548, 0x8068 }, { 6549, 0x8073 },
    { 6550, 0x8072 }, { 6551, 0x8070 }, { 6552, 0x8076 }, { 6553, 0x8079 }, { 6554, 0x807D }, { 6555, 0x807F }, { 6556, 0x8084 }, { 6557, 0x8086 },
    { 6558, 0x8085 }, { 6559, 0x809B }, { 6560, 0x8093 }, { 6561, 0x809A }, { 6562, 0x80AD }, { 6563, 0x5190 }, { 6564, 0x80AC }, { 6565, 0x80DB },
    { 6566, 0x80E5 }, { 6567, 0x80D9 }, { 6568, 0x80DD }, { 6569, 0x80C4 }, { 6570, 0x80DA }, { 6571, 0x80D6 }, { 6572, 0x8109 }, { 6573, 0x80EF },
    { 6574, 0x80F1 }, { 6575, 0x811B }, { 6576, 0x8129 }, { 6577, 0x8123 }, { 6578, 0x812F }, { 6579, 0x814B }, { 6580, 0x968B }, { 6581, 0x8146 },
    { 6582, 0x813E }, { 6583, 0x8153 }, { 6584, 0x8151 }, { 6585, 0x80FC }, { 6586, 0x8171 }, { 6587, 0x816E }, { 6588, 0x8165 }, { 6589, 0x8166 },
    { 6590, 0x8174 }, { 6591, 0x8183 }, { 6592, 0x8188 }, { 6593, 0x818A }, { 6594, 0x8180 }, { 6595, 0x8182 }, { 6596, 0x81A0 }, { 6597, 0x8195 },
    { 6598, 0x81A4 }, { 6599, 0x81A3 }, { 6600, 0x815F }, { 6601, 0x8193 }, { 6602, 0x81A9 }, { 6603, 0x81B0 }, { 6604, 0x81B5 }, { 6605, 0x81BE },
    { 6606, 0x81B8 }, { 6607, 0x81BD }, { 6608, 0x81C0 }, { 6609, 0x81C2 }, { 6610, 0x81BA }, { 6611, 0x81C9 }, { 6612, 0x81CD }, { 6613, 0x81D1 },
    { 6614, 0x81D9 }, { 6615, 0x81D8 }, { 6616, 0x81C8 }, { 6617, 0x81DA }, { 6618, 0x81DF }, { 6619, 0x81E0 }, { 6620, 0x81E7 }, { 6621, 0x81FA },
    { 6622, 0x81FB }, { 6623, 0x81FE }, { 6624, 0x8201 }, { 6625, 0x8202 }, { 6626, 0x8205 }, { 6627, 0x8207 }, { 6628, 0x820A }, { 6629, 0x820D },
    { 6630, 0x8210 }, { 6631, 0x8216 }, { 6632, 0x8229 }, { 6633, 0x822B }, { 6634, 0x8238 }, { 6635, 0x8233 }, { 6636, 0x8240 }, { 6637, 0x8259 },
    { 6638, 0x8258 }, { 6639, 0x825D }, { 6640, 0x825A }, { 6641, 0x825F }, { 6642, 0x8264 }, { 6643, 0x8262 }, { 6644, 0x8268 }, { 6645, 0x826A },
    { 6646, 0x826B }, { 6647, 0x822E }, { 6648, 0x8271 }, { 6649, 0x8277 }, { 6650, 0x8278 }, { 6651, 0x827E }, { 6652, 0x828D }, { 6653, 0x8292 },
    { 6654, 0x82AB }, { 6655, 0x829F }, { 6656, 0x82BB }, { 6657, 0x82AC }, { 6658, 0x82E1 }, { 6659, 0x82E3 }, { 6660, 0x82DF }, { 6661, 0x82D2 },
    { 6662, 0x82F4 }, { 6663, 0x82F3 }, { 6664, 0x82FA }, { 6665, 0x8393 }, { 6666, 0x8303 }, { 6667, 0x82FB }, { 6668, 0x82F9 }, { 6669, 0x82DE },
    { 6670, 0x8306 }, { 6671, 0x82DC }, { 6672, 0x8309 }, { 6673, 0x82D9 }, { 6674, 0x8335 }, { 6675, 0x8334 }, { 6676, 0x8316 }, { 6677, 0x8332 },
    { 6678, 0x8331 }, { 6679, 0x8340 }, { 6680, 0x8339 }, { 6681, 0x8350 }, { 6682, 0x8345 }, { 6683, 0x832F }, { 6684, 0x832B }, { 6685, 0x8317 },
    { 6686, 0x8318 }, { 6687, 0x8385 }, { 6688, 0x839A }, { 6689, 0x83AA }, { 6690, 0x839F }, { 6691, 0x83A2 }, { 6692, 0x8396 }, { 6693, 0x8323 },
    { 6694, 0x838E }, { 6695, 0x8387 }, { 6696, 0x838A }, { 6697, 0x837C }, { 6698, 0x83B5 }, { 6699, 0x8373 }, { 6700, 0x8375 }, { 6701, 0x83A0 },
    { 6702, 0x8389 }, { 6703, 0x83A8 }, { 6704, 0x83F4 }, { 6705, 0x8413 }, { 6706, 0x83EB }, { 6707, 0x83CE }, { 6708, 0x83FD }, { 6709, 0x8403 },
    { 6710, 0x83D8 }, { 6711, 0x840B }, { 6712, 0x83C1 }, { 6713, 0x83F7 }, { 6714, 0x8407 }, { 6715, 0x83E0 }, { 6716, 0x83F2 }, { 6717, 0x840D },
    { 6718, 0x8422 }, { 6719, 0x8420 }, { 6720, 0x83BD }, { 6721, 0x8438 }, { 6722, 0x8506 }, { 6723, 0x83FB }, { 6724, 0x846D }, { 6725, 0x842A },
    { 6726, 0x843C }, { 6727, 0x855A }, { 6728, 0x8484 }, { 6729, 0x8477 }, { 6730, 0x846B }, { 6731, 0x84AD }, { 6732, 0x846E }, { 6733, 0x8482 },
    { 6734, 0x8469 }, { 6735, 0x8446 }, { 6736, 0x842C }, { 6737, 0x846F }, { 6738, 0x8479 }, { 6739, 0x8435 }, { 6740, 0x84CA }, { 6741, 0x8462 },
    { 6742, 0x84B9 }, { 6743, 0x84BF }, { 6744, 0x849F }, { 6745, 0x84D9 }, { 6746, 0x84CD }, { 6747, 0x84BB }, { 6748, 0x84DA }, { 6749, 0x84D0 },
    { 6750, 0x84C1 }, { 6751, 0x84C6 }, { 6752, 0x84D6 }, { 6753, 0x84A1 }, { 6754, 0x8521 }, { 6755, 0x84FF }, { 6756, 0x84F4 }, { 6757, 0x8517 },
    { 6758, 0x8518 }, { 6759, 0x852C }, { 6760, 0x851F }, { 6761, 0x8515 }, { 6762, 0x8514 }, { 6763, 0x84FC }, { 6764, 0x8540 }, { 6765, 0x8563 },
    { 6766, 0x8558 }, { 6767, 0x8548 }, { 6768, 0x8541 }, { 6769, 0x8602 }, { 6770, 0x854B }, { 6771, 0x8555 }, { 6772, 0x8580 }, { 6773, 0x85A4 },
    { 6774, 0x8588 }, { 6775, 0x8591 }, { 6776, 0x858A }, { 6777, 0x85A8 }, { 6778, 0x856D }, { 6779, 0x8594 }, { 6780, 0x859B }, { 6781, 0x85EA },
    { 6782, 0x8587 }, { 6783, 0x859C }, { 6784, 0x8577 }, { 6785, 0x857E }, { 6786, 0x8590 }, { 6787, 0x85C9 }, { 6788, 0x85BA }, { 6789, 0x85CF },
    { 6790, 0x85B9 }, { 6791, 0x85D0 }, { 6792, 0x85D5 }, { 6793, 0x85DD }, { 6794, 0x85E5 }, { 6795, 0x85DC }, { 6796, 0x85F9 }, { 6797, 0x860A },
    { 6798, 0x8613 }, { 6799, 0x860B }, { 6800, 0x85FE }, { 6801, 0x85FA }, { 6802, 0x8606 }, { 6803, 0x8622 }, { 6804, 0x861A }, { 6805, 0x8630 },
    { 6806, 0x863F }, { 6807, 0x864D }, { 6808, 0x4E55 }, { 6809, 0x8654 }, { 6810, 0x865F }, { 6811, 0x8667 }, { 6812, 0x8671 }, { 6813, 0x8693 },
    { 6814, 0x86A3 }, { 6815, 0x86A9 }, { 6816, 0x86AA }, { 6817, 0x868B }, { 6818, 0x868C }, { 6819, 0x86B6 }, { 6820, 0x86AF }, { 6821, 0x86C4 },
    { 6822, 0x86C6 }, { 6823, 0x86B0 }, { 6824, 0x86C9 }, { 6825, 0x8823 }, { 6826, 0x86AB }, { 6827, 0x86D4 }, { 6828, 0x86DE }, { 6829, 0x86E9 },
    { 6830, 0x86EC }, { 6831, 0x86DF }, { 6832, 0x86DB }, { 6833, 0x86EF }, { 6834, 0x8712 }, { 6835, 0x8706 }, { 6836, 0x8708 }, { 6837, 0x8700 },
    { 6838, 0x8703 }, { 6839, 0x86FB }, { 6840, 0x8711 }, { 6841, 0x8709 }, { 6842, 0x870D }, { 6843, 0x86F9 }, { 6844, 0x870A }, { 6845, 0x8734 },
    { 6846, 0x873F }, { 6847, 0x8737 }, { 6848, 0x873B }, { 6849, 0x8725 }, { 6850, 0x8729 }, { 6851, 0x871A }, { 6852, 0x8760 }, { 6853, 0x875F },
    { 6854, 0x8778 }, { 6855, 0x874C }, { 6856, 0x874E }, { 6857, 0x8774 }, { 6858, 0x8757 }, { 6859, 0x8768 }, { 6860, 0x876E }, { 6861, 0x8759 },
    { 6862, 0x8753 }, { 6863, 0x8763 }, { 6864, 0x876A }, { 6865, 0x8805 }, { 6866, 0x87A2 }, { 6867, 0x879F }, { 6868, 0x8782 }, { 6869, 0x87AF },
    { 6870, 0x87CB }, { 6871, 0x87BD }, { 6872, 0x87C0 }, { 6873, 0x87D0 }, { 6874, 0x96D6 }, { 6875, 0x87AB }, { 6876, 0x87C4 }, { 6877, 0x87B3 },
    { 6878, 0x87C7 }, { 6879, 0x87C6 }, { 6880, 0x87BB }, { 6881, 0x87EF }, { 6882, 0x87F2 }, { 6883, 0x87E0 }, { 6884, 0x880F }, { 6885, 0x880D },
    { 6886, 0x87FE }, { 6887, 0x87F6 }, { 6888, 0x87F7 }, { 6889, 0x880E }, { 6890, 0x87D2 }, { 6891, 0x8811 }, { 6892, 0x8816 }, { 6893, 0x8815 },
    { 6894, 0x8822 }, { 6895, 0x8821 }, { 6896, 0x8831 }, { 6897, 0x8836 }, { 6898, 0x8839 }, { 6899, 0x8827 }, { 6900, 0x883B }, { 6901, 0x8844 },
    { 6902, 0x8842 }, { 6903, 0x8852 }, { 6904, 0x8859 }, { 6905, 0x885E }, { 6906, 0x8862 }, { 6907, 0x886B }, { 6908, 0x8881 }, { 6909, 0x887E },
    { 6910, 0x889E }, { 6911, 0x8875 }, { 6912, 0x887D }, { 6913, 0x88B5 }, { 6914, 0x8872 }, { 6915, 0x8882 }, { 6916, 0x8897 }, { 6917, 0x8892 },
    { 6918, 0x88AE }, { 6919, 0x8899 }, { 6920, 0x88A2 }, { 6921, 0x888D }, { 6922, 0x88A4 }, { 6923, 0x88B0 }, { 6924, 0x88BF }, { 6925, 0x88B1 },
    { 6926, 0x88C3 }, { 6927, 0x88C4 }, { 6928, 0x88D4 }, { 6929, 0x88D8 }, { 6930, 0x88D9 }, { 6931, 0x88DD }, { 6932, 0x88F9 }, { 6933, 0x8902 },
    { 6934, 0x88FC }, { 6935, 0x88F4 }, { 6936, 0x88E8 }, { 6937, 0x88F2 }, { 6938, 0x8904 }, { 6939, 0x890C }, { 6940, 0x890A }, { 6941, 0x8913 },
    { 6942, 0x8943 }, { 6943, 0x891E }, { 6944, 0x8925 }, { 6945, 0x892A }, { 6946, 0x892B }, { 6947, 0x8941 }, { 6948, 0x8944 }, { 6949, 0x893B },
    { 6950, 0x8936 }, { 6951, 0x8938 }, { 6952, 0x894C }, { 6953, 0x891D }, { 6954, 0x8960 }, { 6955, 0x895E }, { 6956, 0x8966 }, { 6957, 0x8964 },
    { 6958, 0x896D }, { 6959, 0x896A }, { 6960, 0x896F }, { 6961, 0x8974 }, { 6962, 0x8977 }, { 6963, 0x897E }, { 6964, 0x8983 }, { 6965, 0x8988 },
    { 6966, 0x898A }, { 6967, 0x8993 }, { 6968, 0x8998 }, { 6969, 0x89A1 }, { 6970, 0x89A9 }, { 6971, 0x89A6 }, { 6972, 0x89AC }, { 6973, 0x89AF },
    { 6974, 0x89B2 }, { 6975, 0x89BA }, { 6976, 0x89BD }, { 6977, 0x89BF }, { 6978, 0x89C0 }, { 6979, 0x89DA }, { 6980, 0x89DC }, { 6981, 0x89DD },
    { 6982, 0x89E7 }, { 6983, 0x89F4 }, { 6984, 0x89F8 }, { 6985, 0x8A03 }, { 6986, 0x8A16 }, { 6987, 0x8A10 }, { 6988, 0x8A0C }, { 6989, 0x8A1B },
    { 6990, 0x8A1D }, { 6991, 0x8A25 }, { 6992, 0x8A36 }, { 6993, 0x8A41 }, { 6994, 0x8A5B }, { 6995, 0x8A52 }, { 6996, 0x8A46 }, { 6997, 0x8A48 },
    { 6998, 0x8A7C }, { 6999, 0x8A6D }, { 7000, 0x8A6C }, { 7001, 0x8A62 }, { 7002, 0x8A85 }, { 7003, 0x8A82 }, { 7004, 0x8A84 }, { 7005, 0x8AA8 },
    { 7006, 0x8AA1 }, { 7007, 0x8A91 }, { 7008, 0x8AA5 }, { 7009, 0x8AA6 }, { 7010, 0x8A9A }, { 7011, 0x8AA3 }, { 7012, 0x8AC4 }, { 7013, 0x8ACD },
    { 7014, 0x8AC2 }, { 7015, 0x8ADA }, { 7016, 0x8AEB }, { 7017, 0x8AF3 }, { 7018, 0x8AE7 }, { 7019, 0x8AE4 }, { 7020, 0x8AF1 }, { 7021, 0x8B14 },
    { 7022, 0x8AE0 }, { 7023, 0x8AE2 }, { 7024, 0x8AF7 }, { 7025, 0x8ADE }, { 7026, 0x8ADB }, { 7027, 0x8B0C }, { 7028, 0x8B07 }, { 7029, 0x8B1A },
    { 7030, 0x8AE1 }, { 7031, 0x8B16 }, { 7032, 0x8B10 }, { 7033, 0x8B17 }, { 7034, 0x8B20 }, { 7035, 0x8B33 }, { 7036, 0x97AB }, { 7037, 0x8B26 },
    { 7038, 0x8B2B }, { 7039, 0x8B3E }, { 7040, 0x8B28 }, { 7041, 0x8B41 }, { 7042, 0x8B4C }, { 7043, 0x8B4F }, { 7044, 0x8B4E }, { 7045, 0x8B49 },
    { 7046, 0x8B56 }, { 7047, 0x8B5B }, { 7048, 0x8B5A }, { 7049, 0x8B6B }, { 7050, 0x8B5F }, { 7051, 0x8B6C }, { 7052, 0x8B6F }, { 7053, 0x8B74 },
    { 7054, 0x8B7D }, { 7055, 0x8B80 }, { 7056, 0x8B8C }, { 7057, 0x8B8E }, { 7058, 0x8B92 }, { 7059, 0x8B93 }, { 7060, 0x8B96 }, { 7061, 0x8B99 },
    { 7062, 0x8B9A }, { 7063, 0x8C3A }, { 7064, 0x8C41 }, { 7065, 0x8C3F }, { 7066, 0x8C48 }, { 7067, 0x8C4C }, { 7068, 0x8C4E }, { 7069, 0x8C50 },
    { 7070, 0x8C55 }, { 7071, 0x8C62 }, { 7072, 0x8C6C }, { 7073, 0x8C78 }, { 7074, 0x8C7A }, { 7075, 0x8C82 }, { 7076, 0x8C89 }, { 7077, 0x8C85 },
    { 7078, 0x8C8A }, { 7079, 0x8C8D }, { 7080, 0x8C8E }, { 7081, 0x8C94 }, { 7082, 0x8C7C }, { 7083, 0x8C98 }, { 7084, 0x621D }, { 7085, 0x8CAD },
    { 7086, 0x8CAA }, { 7087, 0x8CBD }, { 7088, 0x8CB2 }, { 7089, 0x8CB3 }, { 7090, 0x8CAE }, { 7091, 0x8CB6 }, { 7092, 0x8CC8 }, { 7093, 0x8CC1 },
    { 7094, 0x8CE4 }, { 7095, 0x8CE3 }, { 7096, 0x8CDA }, { 7097, 0x8CFD }, { 7098, 0x8CFA }, { 7099, 0x8CFB }, { 7100, 0x8D04 }, { 7101, 0x8D05 },
    { 7102, 0x8D0A }, { 7103, 0x8D07 }, { 7104, 0x8D0F }, { 7105, 0x8D0D }, { 7106, 0x8D10 }, { 7107, 0x9F4E }, { 7108, 0x8D13 }, { 7109, 0x8CCD },
    { 7110, 0x8D14 }, { 7111, 0x8D16 }, { 7112, 0x8D67 }, { 7113, 0x8D6D }, { 7114, 0x8D71 }, { 7115, 0x8D73 }, { 7116, 0x8D81 }, { 7117, 0x8D99 },
    { 7118, 0x8DC2 }, { 7119, 0x8DBE }, { 7120, 0x8DBA }, { 7121, 0x8DCF }, { 7122, 0x8DDA }, { 7123, 0x8DD6 }, { 7124, 0x8DCC }, { 7125, 0x8DDB },
    { 7126, 0x8DCB }, { 7127, 0x8DEA }, { 7128, 0x8DEB }, { 7129, 0x8DDF }, { 7130, 0x8DE3 }, { 7131, 0x8DFC }, { 7132, 0x8E08 }, { 7133, 0x8E09 },
    { 7134, 0x8DFF }, { 7135, 0x8E1D }, { 7136, 0x8E1E }, { 7137, 0x8E10 }, { 7138, 0x8E1F }, { 7139, 0x8E42 }, { 7140, 0x8E35 }, { 7141, 0x8E30 },
    { 7142, 0x8E34 }, { 7143, 0x8E4A }, { 7144, 0x8E47 }, { 7145, 0x8E49 }, { 7146, 0x8E4C }, { 7147, 0x8E50 }, { 7148, 0x8E48 }, { 7149, 0x8E59 },
    { 7150, 0x8E64 }, { 7151, 0x8E60 }, { 7152, 0x8E2A }, { 7153, 0x8E63 }, { 7154, 0x8E55 }, { 7155, 0x8E76 }, { 7156, 0x8E72 }, { 7157, 0x8E7C },
    { 7158, 0x8E81 }, { 7159, 0x8E87 }, { 7160, 0x8E85 }, { 7161, 0x8E84 }, { 7162, 0x8E8B }, { 7163, 0x8E8A }, { 7164, 0x8E93 }, { 7165, 0x8E91 },
    { 7166, 0x8E94 }, { 7167, 0x8E99 }, { 7168, 0x8EAA }, { 7169, 0x8EA1 }, { 7170, 0x8EAC }, { 7171, 0x8EB0 }, { 7172, 0x8EC6 }, { 7173, 0x8EB1 },
    { 7174, 0x8EBE }, { 7175, 0x8EC5 }, { 7176, 0x8EC8 }, { 7177, 0x8ECB }, { 7178, 0x8EDB }, { 7179, 0x8EE3 }, { 7180, 0x8EFC }, { 7181, 0x8EFB },
    { 7182, 0x8EEB }, { 7183, 0x8EFE }, { 7184, 0x8F0A }, { 7185, 0x8F05 }, { 7186, 0x8F15 }, { 7187, 0x8F12 }, { 7188, 0x8F19 }, { 7189, 0x8F13 },
    { 7190, 0x8F1C }, { 7191, 0x8F1F }, { 7192, 0x8F1B }, { 7193, 0x8F0C }, { 7194, 0x8F26 }, { 7195, 0x8F33 }, { 7196, 0x8F3B }, { 7197, 0x8F39 },
    { 7198, 0x8F45 }, { 7199, 0x8F42 }, { 7200, 0x8F3E }, { 7201, 0x8F4C }, { 7202, 0x8F49 }, { 7203, 0x8F46 }, { 7204, 0x8F4E }, { 7205, 0x8F57 },
    { 7206, 0x8F5C }, { 7207, 0x8F62 }, { 7208, 0x8F63 }, { 7209, 0x8F64 }, { 7210, 0x8F9C }, { 7211, 0x8F9F }, { 7212, 0x8FA3 }, { 7213, 0x8FAD },
    { 7214, 0x8FAF }, { 7215, 0x8FB7 }, { 7216, 0x8FDA }, { 7217, 0x8FE5 }, { 7218, 0x8FE2 }, { 7219, 0x8FEA }, { 7220, 0x8FEF }, { 7221, 0x9087 },
    { 7222, 0x8FF4 }, { 7223, 0x9005 }, { 7224, 0x8FF9 }, { 7225, 0x8FFA }, { 7226, 0x9011 }, { 7227, 0x9015 }, { 7228, 0x9021 }, { 7229, 0x900D },
    { 7230, 0x901E }, { 7231, 0x9016 }, { 7232, 0x900B }, { 7233, 0x9027 }, { 7234, 0x9036 }, { 7235, 0x9035 }, { 7236, 0x9039 }, { 7237, 0x8FF8 },
    { 7238, 0x904F }, { 7239, 0x9050 }, { 7240, 0x9051 }, { 7241, 0x9052 }, { 7242, 0x900E }, { 7243, 0x9049 }, { 7244, 0x903E }, { 7245, 0x9056 },
    { 7246, 0x9058 }, { 7247, 0x905E }, { 7248, 0x9068 }, { 7249, 0x906F }, { 7250, 0x9076 }, { 7251, 0x96A8 }, { 7252, 0x9072 }, { 7253, 0x9082 },
    { 7254, 0x907D }, { 7255, 0x9081 }, { 7256, 0x9080 }, { 7257, 0x908A }, { 7258, 0x9089 }, { 7259, 0x908F }, { 7260, 0x90A8 }, { 7261, 0x90AF },
    { 7262, 0x90B1 }, { 7263, 0x90B5 }, { 7264, 0x90E2 }, { 7265, 0x90E4 }, { 7266, 0x6248 }, { 7267, 0x90DB }, { 7268, 0x9102 }, { 7269, 0x9112 },
    { 7270, 0x9119 }, { 7271, 0x9132 }, { 7272, 0x9130 }, { 7273, 0x914A }, { 7274, 0x9156 }, { 7275, 0x9158 }, { 7276, 0x9163 }, { 7277, 0x9165 },
    { 7278, 0x9169 }, { 7279, 0x9173 }, { 7280, 0x9172 }, { 7281, 0x918B }, { 7282, 0x9189 }, { 7283, 0x9182 }, { 7284, 0x91A2 }, { 7285, 0x91AB },
    { 7286, 0x91AF }, { 7287, 0x91AA }, { 7288, 0x91B5 }, { 7289, 0x91B4 }, { 7290, 0x91BA }, { 7291, 0x91C0 }, { 7292, 0x91C1 }, { 7293, 0x91C9 },
    { 7294, 0x91CB }, { 7295, 0x91D0 }, { 7296, 0x91D6 }, { 7297, 0x91DF }, { 7298, 0x91E1 }, { 7299, 0x91DB }, { 7300, 0x91FC }, { 7301, 0x91F5 },
    { 7302, 0x91F6 }, { 7303, 0x921E }, { 7304, 0x91FF }, { 7305, 0x9214 }, { 7306, 0x922C }, { 7307, 0x9215 }, { 7308, 0x9211 }, { 7309, 0x925E },
    { 7310, 0x9257 }, { 7311, 0x9245 }, { 7312, 0x9249 }, { 7313, 0x9264 }, { 7314, 0x9248 }, { 7315, 0x9295 }, { 7316, 0x923F }, { 7317, 0x924B },
    { 7318, 0x9250 }, { 7319, 0x929C }, { 7320, 0x9296 }, { 7321, 0x9293 }, { 7322, 0x929B }, { 7323, 0x925A }, { 7324, 0x92CF }, { 7325, 0x92B9 },
    { 7326, 0x92B7 }, { 7327, 0x92E9 }, { 7328, 0x930F }, { 7329, 0x92FA }, { 7330, 0x9344 }, { 7331, 0x932E }, { 7332, 0x9319 }, { 7333, 0x9322 },
    { 7334, 0x931A }, { 7335, 0x9323 }, { 7336, 0x933A }, { 7337, 0x9335 }, { 7338, 0x933B }, { 7339, 0x935C }, { 7340, 0x9360 }, { 7341, 0x937C },
    { 7342, 0x936E }, { 7343, 0x9356 }, { 7344, 0x93B0 }, { 7345, 0x93AC }, { 7346, 0x93AD }, { 7347, 0x9394 }, { 7348, 0x93B9 }, { 7349, 0x93D6 },
    { 7350, 0x93D7 }, { 7351, 0x93E8 }, { 7352, 0x93E5 }, { 7353, 0x93D8 }, { 7354, 0x93C3 }, { 7355, 0x93DD }, { 7356, 0x93D0 }, { 7357, 0x93C8 },
    { 7358, 0x93E4 }, { 7359, 0x941A }, { 7360, 0x9414 }, { 7361, 0x9413 }, { 7362, 0x9403 }, { 7363, 0x9407 }, { 7364, 0x9410 }, { 7365, 0x9436 },
    { 7366, 0x942B }, { 7367, 0x9435 }, { 7368, 0x9421 }, { 7369, 0x943A }, { 7370, 0x9441 }, { 7371, 0x9452 }, { 7372, 0x9444 }, { 7373, 0x945B },
    { 7374, 0x9460 }, { 7375, 0x9462 }, { 7376, 0x945E }, { 7377, 0x946A }, { 7378, 0x9229 }, { 7379, 0x9470 }, { 7380, 0x9475 }, { 7381, 0x9477 },
    { 7382, 0x947D }, { 7383, 0x945A }, { 7384, 0x947C }, { 7385, 0x947E }, { 7386, 0x9481 }, { 7387, 0x947F }, { 7388, 0x9582 }, { 7389, 0x9587 },
    { 7390, 0x958A }, { 7391, 0x9594 }, { 7392, 0x9596 }, { 7393, 0x9598 }, { 7394, 0x9599 }, { 7395, 0x95A0 }, { 7396, 0x95A8 }, { 7397, 0x95A7 },
    { 7398, 0x95AD }, { 7399, 0x95BC }, { 7400, 0x95BB }, { 7401, 0x95B9 }, { 7402, 0x95BE }, { 7403, 0x95CA }, { 7404, 0x6FF6 }, { 7405, 0x95C3 },
    { 7406, 0x95CD }, { 7407, 0x95CC }, { 7408, 0x95D5 }, { 7409, 0x95D4 }, { 7410, 0x95D6 }, { 7411, 0x95DC }, { 7412, 0x95E1 }, { 7413, 0x95E5 },
    { 7414, 0x95E2 }, { 7415, 0x9621 }, { 7416, 0x9628 }, { 7417, 0x962E }, { 7418, 0x962F }, { 7419, 0x9642 }, { 7420, 0x964C }, { 7421, 0x964F },
    { 7422, 0x964B }, { 7423, 0x9677 }, { 7424, 0x965C }, { 7425, 0x965E }, { 7426, 0x965D }, { 7427, 0x965F }, { 7428, 0x9666 }, { 7429, 0x9672 },
    { 7430, 0x966C }, { 7431, 0x968D }, { 7432, 0x9698 }, { 7433, 0x9695 }, { 7434, 0x9697 }, { 7435, 0x96AA }, { 7436, 0x96A7 }, { 7437, 0x96B1 },
    { 7438, 0x96B2 }, { 7439, 0x96B0 }, { 7440, 0x96B4 }, { 7441, 0x96B6 }, { 7442, 0x96B8 }, { 7443, 0x96B9 }, { 7444, 0x96CE }, { 7445, 0x96CB },
    { 7446, 0x96C9 }, { 7447, 0x96CD }, { 7448, 0x894D }, { 7449, 0x96DC }, { 7450, 0x970D }, { 7451, 0x96D5 }, { 7452, 0x96F9 }, { 7453, 0x9704 },
    { 7454, 0x9706 }, { 7455, 0x9708 }, { 7456, 0x9713 }, { 7457, 0x970E }, { 7458, 0x9711 }, { 7459, 0x970F }, { 7460, 0x9716 }, { 7461, 0x9719 },
    { 7462, 0x9724 }, { 7463, 0x972A }, { 7464, 0x9730 }, { 7465, 0x9739 }, { 7466, 0x973D }, { 7467, 0x973E }, { 7468, 0x9744 }, { 7469, 0x9746 },
    { 7470, 0x9748 }, { 7471, 0x9742 }, { 7472, 0x9749 }, { 7473, 0x975C }, { 7474, 0x9760 }, { 7475, 0x9764 }, { 7476, 0x9766 }, { 7477, 0x9768 },
    { 7478, 0x52D2 }, { 7479, 0x976B }, { 7480, 0x9771 }, { 7481, 0x9779 }, { 7482, 0x9785 }, { 7483, 0x977C }, { 7484, 0x9781 }, { 7485, 0x977A },
    { 7486, 0x9786 }, { 7487, 0x978B }, { 7488, 0x978F }, { 7489, 0x9790 }, { 7490, 0x979C }, { 7491, 0x97A8 }, { 7492, 0x97A6 }, { 7493, 0x97A3 },
    { 7494, 0x97B3 }, { 7495, 0x97B4 }, { 7496, 0x97C3 }, { 7497, 0x97C6 }, { 7498, 0x97C8 }, { 7499, 0x97CB }, { 7500, 0x97DC }, { 7501, 0x97ED },
    { 7502, 0x9F4F }, { 7503, 0x97F2 }, { 7504, 0x7ADF }, { 7505, 0x97F6 }, { 7506, 0x97F5 }, { 7507, 0x980F }, { 7508, 0x980C }, { 7509, 0x9838 },
    { 7510, 0x9824 }, { 7511, 0x9821 }, { 7512, 0x9837 }, { 7513, 0x983D }, { 7514, 0x9846 }, { 7515, 0x984F }, { 7516, 0x984B }, { 7517, 0x986B },
    { 7518, 0x986F }, { 7519, 0x9870 }, { 7520, 0x9871 }, { 7521, 0x9874 }, { 7522, 0x9873 }, { 7523, 0x98AA }, { 7524, 0x98AF }, { 7525, 0x98B1 },
    { 7526, 0x98B6 }, { 7527, 0x98C4 }, { 7528, 0x98C3 }, { 7529, 0x98C6 }, { 7530, 0x98E9 }, { 7531, 0x98EB }, { 7532, 0x9903 }, { 7533, 0x9909 },
    { 7534, 0x9912 }, { 7535, 0x9914 }, { 7536, 0x9918 }, { 7537, 0x9921 }, { 7538, 0x991D }, { 7539, 0x991E }, { 7540, 0x9924 }, { 7541, 0x9920 },
    { 7542, 0x992C }, { 7543, 0x992E }, { 7544, 0x993D }, { 7545, 0x993E }, { 7546, 0x9942 }, { 7547, 0x9949 }, { 7548, 0x9945 }, { 7549, 0x9950 },
    { 7550, 0x994B }, { 7551, 0x9951 }, { 7552, 0x9952 }, { 7553, 0x994C }, { 7554, 0x9955 }, { 7555, 0x9997 }, { 7556, 0x9998 }, { 7557, 0x99A5 },
    { 7558, 0x99AD }, { 7559, 0x99AE }, { 7560, 0x99BC }, { 7561, 0x99DF }, { 7562, 0x99DB }, { 7563, 0x99DD }, { 7564, 0x99D8 }, { 7565, 0x99D1 },
    { 7566, 0x99ED }, { 7567, 0x99EE }, { 7568, 0x99F1 }, { 7569, 0x99F2 }, { 7570, 0x99FB }, { 7571, 0x99F8 }, { 7572, 0x9A01 }, { 7573, 0x9A0F },
    { 7574, 0x9A05 }, { 7575, 0x99E2 }, { 7576, 0x9A19 }, { 7577, 0x9A2B }, { 7578, 0x9A37 }, { 7579, 0x9A45 }, { 7580, 0x9A42 }, { 7581, 0x9A40 },
    { 7582, 0x9A43 }, { 7583, 0x9A3E }, { 7584, 0x9A55 }, { 7585, 0x9A4D }, { 7586, 0x9A5B }, { 7587, 0x9A57 }, { 7588, 0x9A5F }, { 7589, 0x9A62 },
    { 7590, 0x9A65 }, { 7591, 0x9A64 }, { 7592, 0x9A69 }, { 7593, 0x9A6B }, { 7594, 0x9A6A }, { 7595, 0x9AAD }, { 7596, 0x9AB0 }, { 7597, 0x9ABC },
    { 7598, 0x9AC0 }, { 7599, 0x9ACF }, { 7600, 0x9AD1 }, { 7601, 0x9AD3 }, { 7602, 0x9AD4 }, { 7603, 0x9ADE }, { 7604, 0x9ADF }, { 7605, 0x9AE2 },
    { 7606, 0x9AE3 }, { 7607, 0x9AE6 }, { 7608, 0x9AEF }, { 7609, 0x9AEB }, { 7610, 0x9AEE }, { 7611, 0x9AF4 }, { 7612, 0x9AF1 }, { 7613, 0x9AF7 },
    { 7614, 0x9AFB }, { 7615, 0x9B06 }, { 7616, 0x9B18 }, { 7617, 0x9B1A }, { 7618, 0x9B1F }, { 7619, 0x9B22 }, { 7620, 0x9B23 }, { 7621, 0x9B25 },
    { 7622, 0x9B27 }, { 7623, 0x9B28 }, { 7624, 0x9B29 }, { 7625, 0x9B2A }, { 7626, 0x9B2E }, { 7627, 0x9B2F }, { 7628, 0x9B32 }, { 7629, 0x9B44 },
    { 7630, 0x9B43 }, { 7631, 0x9B4F }, { 7632, 0x9B4D }, { 7633, 0x9B4E }, { 7634, 0x9B51 }, { 7635, 0x9B58 }, { 7636, 0x9B74 }, { 7637, 0x9B93 },
    { 7638, 0x9B83 }, { 7639, 0x9B91 }, { 7640, 0x9B96 }, { 7641, 0x9B97 }, { 7642, 0x9B9F }, { 7643, 0x9BA0 }, { 7644, 0x9BA8 }, { 7645, 0x9BB4 },
    { 7646, 0x9BC0 }, { 7647, 0x9BCA }, { 7648, 0x9BB9 }, { 7649, 0x9BC6 }, { 7650, 0x9BCF }, { 7651, 0x9BD1 }, { 7652, 0x9BD2 }, { 7653, 0x9BE3 },
    { 7654, 0x9BE2 }, { 7655, 0x9BE4 }, { 7656, 0x9BD4 }, { 7657, 0x9BE1 }, { 7658, 0x9C3A }, { 7659, 0x9BF2 }, { 7660, 0x9BF1 }, { 7661, 0x9BF0 },
    { 7662, 0x9C15 }, { 7663, 0x9C14 }, { 7664, 0x9C09 }, { 7665, 0x9C13 }, { 7666, 0x9C0C }, { 7667, 0x9C06 }, { 7668, 0x9C08 }, { 7669, 0x9C12 },
    { 7670, 0x9C0A }, { 7671, 0x9C04 }, { 7672, 0x9C2E }, { 7673, 0x9C1B }, { 7674, 0x9C25 }, { 7675, 0x9C24 }, { 7676, 0x9C21 }, { 7677, 0x9C30 },
    { 7678, 0x9C47 }, { 7679, 0x9C32 }, { 7680, 0x9C46 }, { 7681, 0x9C3E }, { 7682, 0x9C5A }, { 7683, 0x9C60 }, { 7684, 0x9C67 }, { 7685, 0x9C76 },
    { 7686, 0x9C78 }, { 7687, 0x9CE7 }, { 7688, 0x9CEC }, { 7689, 0x9CF0 }, { 7690, 0x9D09 }, { 7691, 0x9D08 }, { 7692, 0x9CEB }, { 7693, 0x9D03 },
    { 7694, 0x9D06 }, { 7695, 0x9D2A }, { 7696, 0x9D26 }, { 7697, 0x9DAF }, { 7698, 0x9D23 }, { 7699, 0x9D1F }, { 7700, 0x9D44 }, { 7701, 0x9D15 },
    { 7702, 0x9D12 }, { 7703, 0x9D41 }, { 7704, 0x9D3F }, { 7705, 0x9D3E }, { 7706, 0x9D46 }, { 7707, 0x9D48 }, { 7708, 0x9D5D }, { 7709, 0x9D5E },
    { 7710, 0x9D64 }, { 7711, 0x9D51 }, { 7712, 0x9D50 }, { 7713, 0x9D59 }, { 7714, 0x9D72 }, { 7715, 0x9D89 }, { 7716, 0x9D87 }, { 7717, 0x9DAB },
    { 7718, 0x9D6F }, { 7719, 0x9D7A }, { 7720, 0x9D9A }, { 7721, 0x9DA4 }, { 7722, 0x9DA9 }, { 7723, 0x9DB2 }, { 7724, 0x9DC4 }, { 7725, 0x9DC1 },
    { 7726, 0x9DBB }, { 7727, 0x9DB8 }, { 7728, 0x9DBA }, { 7729, 0x9DC6 }, { 7730, 0x9DCF }, { 7731, 0x9DC2 }, { 7732, 0x9DD9 }, { 7733, 0x9DD3 },
    { 7734, 0x9DF8 }, { 7735, 0x9DE6 }, { 7736, 0x9DED }, { 7737, 0x9DEF }, { 7738, 0x9DFD }, { 7739, 0x9E1A }, { 7740, 0x9E1B }, { 7741, 0x9E1E },
    { 7742, 0x9E75 }, { 7743, 0x9E79 }, { 7744, 0x9E7D }, { 7745, 0x9E81 }, { 7746, 0x9E88 }, { 7747, 0x9E8B }, { 7748, 0x9E8C }, { 7749, 0x9E92 },
    { 7750, 0x9E95 }, { 7751, 0x9E91 }, { 7752, 0x9E9D }, { 7753, 0x9EA5 }, { 7754, 0x9EA9 }, { 7755, 0x9EB8 }, { 7756, 0x9EAA }, { 7757, 0x9EAD },
    { 7758, 0x9761 }, { 7759, 0x9ECC }, { 7760, 0x9ECE }, { 7761, 0x9ECF }, { 7762, 0x9ED0 }, { 7763, 0x9ED4 }, { 7764, 0x9EDC }, { 7765, 0x9EDE },
    { 7766, 0x9EDD }, { 7767, 0x9EE0 }, { 7768, 0x9EE5 }, { 7769, 0x9EE8 }, { 7770, 0x9EEF }, { 7771, 0x9EF4 }, { 7772, 0x9EF6 }, { 7773, 0x9EF7 },
    { 7774, 0x9EF9 }, { 7775, 0x9EFB }, { 7776, 0x9EFC }, { 7777, 0x9EFD }, { 7778, 0x9F07 }, { 7779, 0x9F08 }, { 7780, 0x76B7 }, { 7781, 0x9F15 },
    { 7782, 0x9F21 }, { 7783, 0x9F2C }, { 7784, 0x9F3E }, { 7785, 0x9F4A }, { 7786, 0x9F52 }, { 7787, 0x9F54 }, { 7788, 0x9F63 }, { 7789, 0x9F5F },
    { 7790, 0x9F60 }, { 7791, 0x9F61 }, { 7792, 0x9F66 }, { 7793, 0x9F67 }, { 7794, 0x9F6C }, { 7795, 0x9F6A }, { 7796, 0x9F77 }, { 7797, 0x9F72 },
    { 7798, 0x9F76 }, { 7799, 0x9F95 }, { 7800, 0x9F9C }, { 7801, 0x9FA0 }, { 7802, 0x582F }, { 7803, 0x69C7 }, { 7804, 0x9059 }, { 7805, 0x7464 },
    { 7806, 0x51DC }, { 7807, 0x7199 }, { 8272, 0x7E8A }, { 8273, 0x891C }, { 8274, 0x9348 }, { 8275, 0x9288 }, { 8276, 0x84DC }, { 8277, 0x4FC9 },
    { 8278, 0x70BB }, { 8279, 0x6631 }, { 8280, 0x68C8 }, { 8281, 0x92F9 }, { 8282, 0x66FB }, { 8283, 0x5F45 }, { 8284, 0x4E28 }, { 8285, 0x4EE1 },
    { 8286, 0x4EFC }, { 8287, 0x4F00 }, { 8288, 0x4F03 }, { 8289, 0x4F39 }, { 8290, 0x4F56 }, { 8291, 0x4F92 }, { 8292, 0x4F8A }, { 8293, 0x4F9A },
    { 8294, 0x4F94 }, { 8295, 0x4FCD }, { 8296, 0x5040 }, { 8297, 0x5022 }, { 8298, 0x4FFF }, { 8299, 0x501E }, { 8300, 0x5046 }, { 8301, 0x5070 },
    { 8302, 0x5042 }, { 8303, 0x5094 }, { 8304, 0x50F4 }, { 8305, 0x50D8 }, { 8306, 0x514A }, { 8307, 0x5164 }, { 8308, 0x519D }, { 8309, 0x51BE },
    { 8310, 0x51EC }, { 8311, 0x5215 }, { 8312, 0x529C }, { 8313, 0x52A6 }, { 8314, 0x52C0 }, { 8315, 0x52DB }, { 8316, 0x5300 }, { 8317, 0x5307 },
    { 8318, 0x5324 }, { 8319, 0x5372 }, { 8320, 0x5393 }, { 8321, 0x53B2 }, { 8322, 0x53DD }, { 8323, 0xFA0E }, { 8324, 0x549C }, { 8325, 0x548A },
    { 8326, 0x54A9 }, { 8327, 0x54FF }, { 8328, 0x5586 }, { 8329, 0x5759 }, { 8330, 0x5765 }, { 8331, 0x57AC }, { 8332, 0x57C8 }, { 8333, 0x57C7 },
    { 8334, 0xFA0F }, { 8335, 0xFA10 }, { 8336, 0x589E }, { 8337, 0x58B2 }, { 8338, 0x590B }, { 8339, 0x5953 }, { 8340, 0x595B }, { 8341, 0x595D },
    { 8342, 0x5963 }, { 8343, 0x59A4 }, { 8344, 0x59BA }, { 8345, 0x5B56 }, { 8346, 0x5BC0 }, { 8347, 0x752F }, { 8348, 0x5BD8 }, { 8349, 0x5BEC },
    { 8350, 0x5C1E }, { 8351, 0x5CA6 }, { 8352, 0x5CBA }, { 8353, 0x5CF5 }, { 8354, 0x5D27 }, { 8355, 0x5D53 }, { 8356, 0xFA11 }, { 8357, 0x5D42 },
    { 8358, 0x5D6D }, { 8359, 0x5DB8 }, { 8360, 0x5DB9 }, { 8361, 0x5DD0 }, { 8362, 0x5F21 }, { 8363, 0x5F34 }, { 8364, 0x5F67 }, { 8365, 0x5FB7 },
    { 8366, 0x5FDE }, { 8367, 0x605D }, { 8368, 0x6085 }, { 8369, 0x608A }, { 8370, 0x60DE }, { 8371, 0x60D5 }, { 8372, 0x6120 }, { 8373, 0x60F2 },
    { 8374, 0x6111 }, { 8375, 0x6137 }, { 8376, 0x6130 }, { 8377, 0x6198 }, { 8378, 0x6213 }, { 8379, 0x62A6 }, { 8380, 0x63F5 }, { 8381, 0x6460 },
    { 8382, 0x649D }, { 8383, 0x64CE }, { 8384, 0x654E }, { 8385, 0x6600 }, { 8386, 0x6615 }, { 8387, 0x663B }, { 8388, 0x6609 }, { 8389, 0x662E },
    { 8390, 0x661E }, { 8391, 0x6624 }, { 8392, 0x6665 }, { 8393, 0x6657 }, { 8394, 0x6659 }, { 8395, 0xFA12 }, { 8396, 0x6673 }, { 8397, 0x6699 },
    { 8398, 0x66A0 }, { 8399, 0x66B2 }, { 8400, 0x66BF }, { 8401, 0x66FA }, { 8402, 0x670E }, { 8403, 0xF929 }, { 8404, 0x6766 }, { 8405, 0x67BB },
    { 8406, 0x6852 }, { 8407, 0x67C0 }, { 8408, 0x6801 }, { 8409, 0x6844 }, { 8410, 0x68CF }, { 8411, 0xFA13 }, { 8412, 0x6968 }, { 8413, 0xFA14 },
    { 8414, 0x6998 }, { 8415, 0x69E2 }, { 8416, 0x6A30 }, { 8417, 0x6A6B }, { 8418, 0x6A46 }, { 8419, 0x6A73 }, { 8420, 0x6A7E }, { 8421, 0x6AE2 },
    { 8422, 0x6AE4 }, { 8423, 0x6BD6 }, { 8424, 0x6C3F }, { 8425, 0x6C5C }, { 8426, 0x6C86 }, { 8427, 0x6C6F }, { 8428, 0x6CDA }, { 8429, 0x6D04 },
    { 8430, 0x6D87 }, { 8431, 0x6D6F }, { 8432, 0x6D96 }, { 8433, 0x6DAC }, { 8434, 0x6DCF }, { 8435, 0x6DF8 }, { 8436, 0x6DF2 }, { 8437, 0x6DFC },
    { 8438, 0x6E39 }, { 8439, 0x6E5C }, { 8440, 0x6E27 }, { 8441, 0x6E3C }, { 8442, 0x6EBF }, { 8443, 0x6F88 }, { 8444, 0x6FB5 }, { 8445, 0x6FF5 },
    { 8446, 0x7005 }, { 8447, 0x7007 }, { 8448, 0x7028 }, { 8449, 0x7085 }, { 8450, 0x70AB }, { 8451, 0x710F }, { 8452, 0x7104 }, { 8453, 0x715C },
    { 8454, 0x7146 }, { 8455, 0x7147 }, { 8456, 0xFA15 }, { 8457, 0x71C1 }, { 8458, 0x71FE }, { 8459, 0x72B1 }, { 8460, 0x72BE }, { 8461, 0x7324 },
    { 8462, 0xFA16 }, { 8463, 0x7377 }, { 8464, 0x73BD }, { 8465, 0x73C9 }, { 8466, 0x73D6 }, { 8467, 0x73E3 }, { 8468, 0x73D2 }, { 8469, 0x7407 },
    { 8470, 0x73F5 }, { 8471, 0x7426 }, { 8472, 0x742A }, { 8473, 0x7429 }, { 8474, 0x742E }, { 8475, 0x7462 }, { 8476, 0x7489 }, { 8477, 0x749F },
    { 8478, 0x7501 }, { 8479, 0x756F }, { 8480, 0x7682 }, { 8481, 0x769C }, { 8482, 0x769E }, { 8483, 0x769B }, { 8484, 0x76A6 }, { 8485, 0xFA17 },
    { 8486, 0x7746 }, { 8487, 0x52AF }, { 8488, 0x7821 }, { 8489, 0x784E }, { 8490, 0x7864 }, { 8491, 0x787A }, { 8492, 0x7930 }, { 8493, 0xFA18 },
    { 8494, 0xFA19 }, { 8495, 0xFA1A }, { 8496, 0x7994 }, { 8497, 0xFA1B }, { 8498, 0x799B }, { 8499, 0x7AD1 }, { 8500, 0x7AE7 }, { 8501, 0xFA1C },
    { 8502, 0x7AEB }, { 8503, 0x7B9E }, { 8504, 0xFA1D }, { 8505, 0x7D48 }, { 8506, 0x7D5C }, { 8507, 0x7DB7 }, { 8508, 0x7DA0 }, { 8509, 0x7DD6 },
    { 8510, 0x7E52 }, { 8511, 0x7F47 }, { 8512, 0x7FA1 }, { 8513, 0xFA1E }, { 8514, 0x8301 }, { 8515, 0x8362 }, { 8516, 0x837F }, { 8517, 0x83C7 },
    { 8518, 0x83F6 }, { 8519, 0x8448 }, { 8520, 0x84B4 }, { 8521, 0x8553 }, { 8522, 0x8559 }, { 8523, 0x856B }, { 8524, 0xFA1F }, { 8525, 0x85B0 },
    { 8526, 0xFA20 }, { 8527, 0xFA21 }, { 8528, 0x8807 }, { 8529, 0x88F5 }, { 8530, 0x8A12 }, { 8531, 0x8A37 }, { 8532, 0x8A79 }, { 8533, 0x8AA7 },
    { 8534, 0x8ABE }, { 8535, 0x8ADF }, { 8536, 0xFA22 }, { 8537, 0x8AF6 }, { 8538, 0x8B53 }, { 8539, 0x8B7F }, { 8540, 0x8CF0 }, { 8541, 0x8CF4 },
    { 8542, 0x8D12 }, { 8543, 0x8D76 }, { 8544, 0xFA23 }, { 8545, 0x8ECF }, { 8546, 0xFA24 }, { 8547, 0xFA25 }, { 8548, 0x9067 }, { 8549, 0x90DE },
    { 8550, 0xFA26 }, { 8551, 0x9115 }, { 8552, 0x9127 }, { 8553, 0x91DA }, { 8554, 0x91D7 }, { 8555, 0x91DE }, { 8556, 0x91ED }, { 8557, 0x91EE },
    { 8558, 0x91E4 }, { 8559, 0x91E5 }, { 8560, 0x9206 }, { 8561, 0x9210 }, { 8562, 0x920A }, { 8563, 0x923A }, { 8564, 0x9240 }, { 8565, 0x923C },
    { 8566, 0x924E }, { 8567, 0x9259 }, { 8568, 0x9251 }, { 8569, 0x9239 }, { 8570, 0x9267 }, { 8571, 0x92A7 }, { 8572, 0x9277 }, { 8573, 0x9278 },
    { 8574, 0x92E7 }, { 8575, 0x92D7 }, { 8576, 0x92D9 }, { 8577, 0x92D0 }, { 8578, 0xFA27 }, { 8579, 0x92D5 }, { 8580, 0x92E0 }, { 8581, 0x92D3 },
    { 8582, 0x9325 }, { 8583, 0x9321 }, { 8584, 0x92FB }, { 8585, 0xFA28 }, { 8586, 0x931E }, { 8587, 0x92FF }, { 8588, 0x931D }, { 8589, 0x9302 },
    { 8590, 0x9370 }, { 8591, 0x9357 }, { 8592, 0x93A4 }, { 8593, 0x93C6 }, { 8594, 0x93DE }, { 8595, 0x93F8 }, { 8596, 0x9431 }, { 8597, 0x9445 },
    { 8598, 0x9448 }, { 8599, 0x9592 }, { 8600, 0xF9DC }, { 8601, 0xFA29 }, { 8602, 0x969D }, { 8603, 0x96AF }, { 8604, 0x9733 }, { 8605, 0x973B },
    { 8606, 0x9743 }, { 8607, 0x974D }, { 8608, 0x974F }, { 8609, 0x9751 }, { 8610, 0x9755 }, { 8611, 0x9857 }, { 8612, 0x9865 }, { 8613, 0xFA2A },
    { 8614, 0xFA2B }, { 8615, 0x9927 }, { 8616, 0xFA2C }, { 8617, 0x999E }, { 8618, 0x9A4E }, { 8619, 0x9AD9 }, { 8620, 0x9ADC }, { 8621, 0x9B75 },
    { 8622, 0x9B72 }, { 8623, 0x9B8F }, { 8624, 0x9BB1 }, { 8625, 0x9BBB }, { 8626, 0x9C00 }, { 8627, 0x9D70 }, { 8628, 0x9D6B }, { 8629, 0xFA2D },
    { 8630, 0x9E19 }, { 8631, 0x9ED1 }, { 8634, 0x2170 }, { 8635, 0x2171 }, { 8636, 0x2172 }, { 8637, 0x2173 }, { 8638, 0x2174 }, { 8639, 0x2175 },
    { 8640, 0x2176 }, { 8641, 0x2177 }, { 8642, 0x2178 }, { 8643, 0x2179 }, { 8644, 0xFFE2 }, { 8645, 0xFFE4 }, { 8646, 0xFF07 }, { 8647, 0xFF02 },
    { 10716, 0x2170 }, { 10717, 0x2171 }, { 10718, 0x2172 }, { 10719, 0x2173 }, { 10720, 0x2174 }, { 10721, 0x2175 }, { 10722, 0x2176 }, { 10723, 0x2177 },
    { 10724, 0x2178 }, { 10725, 0x2179 }, { 10726, 0x2160 }, { 10727, 0x2161 }, { 10728, 0x2162 }, { 10729, 0x2163 }, { 10730, 0x2164 }, { 10731, 0x2165 },
    { 10732, 0x2166 }, { 10733, 0x2167 }, { 10734, 0x2168 }, { 10735, 0x2169 }, { 10736, 0xFFE2 }, { 10737, 0xFFE4 }, { 10738, 0xFF07 }, { 10739, 0xFF02 },
    { 10740, 0x3231 }, { 10741, 0x2116 }, { 10742, 0x2121 }, { 10743, 0x2235 }, { 10744, 0x7E8A }, { 10745, 0x891C }, { 10746, 0x9348 }, { 10747, 0x9288 },
    { 10748, 0x84DC }, { 10749, 0x4FC9 }, { 10750, 0x70BB }, { 10751, 0x6631 }, { 10752, 0x68C8 }, { 10753, 0x92F9 }, { 10754, 0x66FB }, { 10755, 0x5F45 },
    { 10756, 0x4E28 }, { 10757, 0x4EE1 }, { 10758, 0x4EFC }, { 10759, 0x4F00 }, { 10760, 0x4F03 }, { 10761, 0x4F39 }, { 10762, 0x4F56 }, { 10763, 0x4F92 },
    { 10764, 0x4F8A }, { 10765, 0x4F9A }, { 10766, 0x4F94 }, { 10767, 0x4FCD }, { 10768, 0x5040 }, { 10769, 0x5022 }, { 10770, 0x4FFF }, { 10771, 0x501E },
    { 10772, 0x5046 }, { 10773, 0x5070 }, { 10774, 0x5042 }, { 10775, 0x5094 }, { 10776, 0x50F4 }, { 10777, 0x50D8 }, { 10778, 0x514A }, { 10779, 0x5164 },
    { 10780, 0x519D }, { 10781, 0x51BE }, { 10782, 0x51EC }, { 10783, 0x5215 }, { 10784, 0x529C }, { 10785, 0x52A6 }, { 10786, 0x52C0 }, { 10787, 0x52DB },
    { 10788, 0x5300 }, { 10789, 0x5307 }, { 10790, 0x5324 }, { 10791, 0x5372 }, { 10792, 0x5393 }, { 10793, 0x53B2 }, { 10794, 0x53DD }, { 10795, 0xFA0E },
    { 10796, 0x549C }, { 10797, 0x548A }, { 10798, 0x54A9 }, { 10799, 0x54FF }, { 10800, 0x5586 }, { 10801, 0x5759 }, { 10802, 0x5765 }, { 10803, 0x57AC },
    { 10804, 0x57C8 }, { 10805, 0x57C7 }, { 10806, 0xFA0F }, { 10807, 0xFA10 }, { 10808, 0x589E }, { 10809, 0x58B2 }, { 10810, 0x590B }, { 10811, 0x5953 },
    { 10812, 0x595B }, { 10813, 0x595D }, { 10814, 0x5963 }, { 10815, 0x59A4 }, { 10816, 0x59BA }, { 10817, 0x5B56 }, { 10818, 0x5BC0 }, { 10819, 0x752F },
    { 10820, 0x5BD8 }, { 10821, 0x5BEC }, { 10822, 0x5C1E }, { 10823, 0x5CA6 }, { 10824, 0x5CBA }, { 10825, 0x5CF5 }, { 10826, 0x5D27 }, { 10827, 0x5D53 },
    { 10828, 0xFA11 }, { 10829, 0x5D42 }, { 10830, 0x5D6D }, { 10831, 0x5DB8 }, { 10832, 0x5DB9 }, { 10833, 0x5DD0 }, { 10834, 0x5F21 }, { 10835, 0x5F34 },
    { 10836, 0x5F67 }, { 10837, 0x5FB7 }, { 10838, 0x5FDE }, { 10839, 0x605D }, { 10840, 0x6085 }, { 10841, 0x608A }, { 10842, 0x60DE }, { 10843, 0x60D5 },
    { 10844, 0x6120 }, { 10845, 0x60F2 }, { 10846, 0x6111 }, { 10847, 0x6137 }, { 10848, 0x6130 }, { 10849, 0x6198 }, { 10850, 0x6213 }, { 10851, 0x62A6 },
    { 10852, 0x63F5 }, { 10853, 0x6460 }, { 10854, 0x649D }, { 10855, 0x64CE }, { 10856, 0x654E }, { 10857, 0x6600 }, { 10858, 0x6615 }, { 10859, 0x663B },
    { 10860, 0x6609 }, { 10861, 0x662E }, { 10862, 0x661E }, { 10863, 0x6624 }, { 10864, 0x6665 }, { 10865, 0x6657 }, { 10866, 0x6659 }, { 10867, 0xFA12 },
    { 10868, 0x6673 }, { 10869, 0x6699 }, { 10870, 0x66A0 }, { 10871, 0x66B2 }, { 10872, 0x66BF }, { 10873, 0x66FA }, { 10874, 0x670E }, { 10875, 0xF929 },
    { 10876, 0x6766 }, { 10877, 0x67BB }, { 10878, 0x6852 }, { 10879, 0x67C0 }, { 10880, 0x6801 }, { 10881, 0x6844 }, { 10882, 0x68CF }, { 10883, 0xFA13 },
    { 10884, 0x6968 }, { 10885, 0xFA14 }, { 10886, 0x6998 }, { 10887, 0x69E2 }, { 10888, 0x6A30 }, { 10889, 0x6A6B }, { 10890, 0x6A46 }, { 10891, 0x6A73 },
    { 10892, 0x6A7E }, { 10893, 0x6AE2 }, { 10894, 0x6AE4 }, { 10895, 0x6BD6 }, { 10896, 0x6C3F }, { 10897, 0x6C5C }, { 10898, 0x6C86 }, { 10899, 0x6C6F },
    { 10900, 0x6CDA }, { 10901, 0x6D04 }, { 10902, 0x6D87 }, { 10903, 0x6D6F }, { 10904, 0x6D96 }, { 10905, 0x6DAC }, { 10906, 0x6DCF }, { 10907, 0x6DF8 },
    { 10908, 0x6DF2 }, { 10909, 0x6DFC }, { 10910, 0x6E39 }, { 10911, 0x6E5C }, { 10912, 0x6E27 }, { 10913, 0x6E3C }, { 10914, 0x6EBF }, { 10915, 0x6F88 },
    { 10916, 0x6FB5 }, { 10917, 0x6FF5 }, { 10918, 0x7005 }, { 10919, 0x7007 }, { 10920, 0x7028 }, { 10921, 0x7085 }, { 10922, 0x70AB }, { 10923, 0x710F },
    { 10924, 0x7104 }, { 10925, 0x715C }, { 10926, 0x7146 }, { 10927, 0x7147 }, { 10928, 0xFA15 }, { 10929, 0x71C1 }, { 10930, 0x71FE }, { 10931, 0x72B1 },
    { 10932, 0x72BE }, { 10933, 0x7324 }, { 10934, 0xFA16 }, { 10935, 0x7377 }, { 10936, 0x73BD }, { 10937, 0x73C9 }, { 10938, 0x73D6 }, { 10939, 0x73E3 },
    { 10940, 0x73D2 }, { 10941, 0x7407 }, { 10942, 0x73F5 }, { 10943, 0x7426 }, { 10944, 0x742A }, { 10945, 0x7429 }, { 10946, 0x742E }, { 10947, 0x7462 },
    { 10948, 0x7489 }, { 10949, 0x749F }, { 10950, 0x7501 }, { 10951, 0x756F }, { 10952, 0x7682 }, { 10953, 0x769C }, { 10954, 0x769E }, { 10955, 0x769B },
    { 10956, 0x76A6 }, { 10957, 0xFA17 }, { 10958, 0x7746 }, { 10959, 0x52AF }, { 10960, 0x7821 }, { 10961, 0x784E }, { 10962, 0x7864 }, { 10963, 0x787A },
    { 10964, 0x7930 }, { 10965, 0xFA18 }, { 10966, 0xFA19 }, { 10967, 0xFA1A }, { 10968, 0x7994 }, { 10969, 0xFA1B }, { 10970, 0x799B }, { 10971, 0x7AD1 },
    { 10972, 0x7AE7 }, { 10973, 0xFA1C }, { 10974, 0x7AEB }, { 10975, 0x7B9E }, { 10976, 0xFA1D }, { 10977, 0x7D48 }, { 10978, 0x7D5C }, { 10979, 0x7DB7 },
    { 10980, 0x7DA0 }, { 10981, 0x7DD6 }, { 10982, 0x7E52 }, { 10983, 0x7F47 }, { 10984, 0x7FA1 }, { 10985, 0xFA1E }, { 10986, 0x8301 }, { 10987, 0x8362 },
    { 10988, 0x837F }, { 10989, 0x83C7 }, { 10990, 0x83F6 }, { 10991, 0x8448 }, { 10992, 0x84B4 }, { 10993, 0x8553 }, { 10994, 0x8559 }, { 10995, 0x856B },
    { 10996, 0xFA1F }, { 10997, 0x85B0 }, { 10998, 0xFA20 }, { 10999, 0xFA21 }, { 11000, 0x8807 }, { 11001, 0x88F5 }, { 11002, 0x8A12 }, { 11003, 0x8A37 },
    { 11004, 0x8A79 }, { 11005, 0x8AA7 }, { 11006, 0x8ABE }, { 11007, 0x8ADF }, { 11008, 0xFA22 }, { 11009, 0x8AF6 }, { 11010, 0x8B53 }, { 11011, 0x8B7F },
    { 11012, 0x8CF0 }, { 11013, 0x8CF4 }, { 11014, 0x8D12 }, { 11015, 0x8D76 }, { 11016, 0xFA23 }, { 11017, 0x8ECF }, { 11018, 0xFA24 }, { 11019, 0xFA25 },
    { 11020, 0x9067 }, { 11021, 0x90DE }, { 11022, 0xFA26 }, { 11023, 0x9115 }, { 11024, 0x9127 }, { 11025, 0x91DA }, { 11026, 0x91D7 }, { 11027, 0x91DE },
    { 11028, 0x91ED }, { 11029, 0x91EE }, { 11030, 0x91E4 }, { 11031, 0x91E5 }, { 11032, 0x9206 }, { 11033, 0x9210 }, { 11034, 0x920A }, { 11035, 0x923A },
    { 11036, 0x9240 }, { 11037, 0x923C }, { 11038, 0x924E }, { 11039, 0x9259 }, { 11040, 0x9251 }, { 11041, 0x9239 }, { 11042, 0x9267 }, { 11043, 0x92A7 },
    { 11044, 0x9277 }, { 11045, 0x9278 }, { 11046, 0x92E7 }, { 11047, 0x92D7 }, { 11048, 0x92D9 }, { 11049, 0x92D0 }, { 11050, 0xFA27 }, { 11051, 0x92D5 },
    { 11052, 0x92E0 }, { 11053, 0x92D3 }, { 11054, 0x9325 }, { 11055, 0x9321 }, { 11056, 0x92FB }, { 11057, 0xFA28 }, { 11058, 0x931E }, { 11059, 0x92FF },
    { 11060, 0x931D }, { 11061, 0x9302 }, { 11062, 0x9370 }, { 11063, 0x9357 }, { 11064, 0x93A4 }, { 11065, 0x93C6 }, { 11066, 0x93DE }, { 11067, 0x93F8 },
    { 11068, 0x9431 }, { 11069, 0x9445 }, { 11070, 0x9448 }, { 11071, 0x9592 }, { 11072, 0xF9DC }, { 11073, 0xFA29 }, { 11074, 0x969D }, { 11075, 0x96AF },
    { 11076, 0x9733 }, { 11077, 0x973B }, { 11078, 0x9743 }, { 11079, 0x974D }, { 11080, 0x974F }, { 11081, 0x9751 }, { 11082, 0x9755 }, { 11083, 0x9857 },
    { 11084, 0x9865 }, { 11085, 0xFA2A }, { 11086, 0xFA2B }, { 11087, 0x9927 }, { 11088, 0xFA2C }, { 11089, 0x999E }, { 11090, 0x9A4E }, { 11091, 0x9AD9 },
    { 11092, 0x9ADC }, { 11093, 0x9B75 }, { 11094, 0x9B72 }, { 11095, 0x9B8F }, { 11096, 0x9BB1 }, { 11097, 0x9BBB }, { 11098, 0x9C00 }, { 11099, 0x9D70 },
    { 11100, 0x9D6B }, { 11101, 0xFA2D }, { 11102, 0x9E19 }, { 11103, 0x9ED1 }
} };
#endif // ASSERT_ENABLED

// These are values from https://encoding.spec.whatwg.org/index-jis0208.txt that are not in ICU.
constexpr std::array<std::pair<uint16_t, UChar>, 388> jis0208Extras { {
    { 10716, 0x2170 }, { 10717, 0x2171 }, { 10718, 0x2172 }, { 10719, 0x2173 }, { 10720, 0x2174 }, { 10721, 0x2175 }, { 10722, 0x2176 }, { 10723, 0x2177 },
    { 10724, 0x2178 }, { 10725, 0x2179 }, { 10726, 0x2160 }, { 10727, 0x2161 }, { 10728, 0x2162 }, { 10729, 0x2163 }, { 10730, 0x2164 }, { 10731, 0x2165 },
    { 10732, 0x2166 }, { 10733, 0x2167 }, { 10734, 0x2168 }, { 10735, 0x2169 }, { 10736, 0xffe2 }, { 10737, 0xffe4 }, { 10738, 0xff07 }, { 10739, 0xff02 },
    { 10740, 0x3231 }, { 10741, 0x2116 }, { 10742, 0x2121 }, { 10743, 0x2235 }, { 10744, 0x7e8a }, { 10745, 0x891c }, { 10746, 0x9348 }, { 10747, 0x9288 },
    { 10748, 0x84dc }, { 10749, 0x4fc9 }, { 10750, 0x70bb }, { 10751, 0x6631 }, { 10752, 0x68c8 }, { 10753, 0x92f9 }, { 10754, 0x66fb }, { 10755, 0x5f45 },
    { 10756, 0x4e28 }, { 10757, 0x4ee1 }, { 10758, 0x4efc }, { 10759, 0x4f00 }, { 10760, 0x4f03 }, { 10761, 0x4f39 }, { 10762, 0x4f56 }, { 10763, 0x4f92 },
    { 10764, 0x4f8a }, { 10765, 0x4f9a }, { 10766, 0x4f94 }, { 10767, 0x4fcd }, { 10768, 0x5040 }, { 10769, 0x5022 }, { 10770, 0x4fff }, { 10771, 0x501e },
    { 10772, 0x5046 }, { 10773, 0x5070 }, { 10774, 0x5042 }, { 10775, 0x5094 }, { 10776, 0x50f4 }, { 10777, 0x50d8 }, { 10778, 0x514a }, { 10779, 0x5164 },
    { 10780, 0x519d }, { 10781, 0x51be }, { 10782, 0x51ec }, { 10783, 0x5215 }, { 10784, 0x529c }, { 10785, 0x52a6 }, { 10786, 0x52c0 }, { 10787, 0x52db },
    { 10788, 0x5300 }, { 10789, 0x5307 }, { 10790, 0x5324 }, { 10791, 0x5372 }, { 10792, 0x5393 }, { 10793, 0x53b2 }, { 10794, 0x53dd }, { 10795, 0xfa0e },
    { 10796, 0x549c }, { 10797, 0x548a }, { 10798, 0x54a9 }, { 10799, 0x54ff }, { 10800, 0x5586 }, { 10801, 0x5759 }, { 10802, 0x5765 }, { 10803, 0x57ac },
    { 10804, 0x57c8 }, { 10805, 0x57c7 }, { 10806, 0xfa0f }, { 10807, 0xfa10 }, { 10808, 0x589e }, { 10809, 0x58b2 }, { 10810, 0x590b }, { 10811, 0x5953 },
    { 10812, 0x595b }, { 10813, 0x595d }, { 10814, 0x5963 }, { 10815, 0x59a4 }, { 10816, 0x59ba }, { 10817, 0x5b56 }, { 10818, 0x5bc0 }, { 10819, 0x752f },
    { 10820, 0x5bd8 }, { 10821, 0x5bec }, { 10822, 0x5c1e }, { 10823, 0x5ca6 }, { 10824, 0x5cba }, { 10825, 0x5cf5 }, { 10826, 0x5d27 }, { 10827, 0x5d53 },
    { 10828, 0xfa11 }, { 10829, 0x5d42 }, { 10830, 0x5d6d }, { 10831, 0x5db8 }, { 10832, 0x5db9 }, { 10833, 0x5dd0 }, { 10834, 0x5f21 }, { 10835, 0x5f34 },
    { 10836, 0x5f67 }, { 10837, 0x5fb7 }, { 10838, 0x5fde }, { 10839, 0x605d }, { 10840, 0x6085 }, { 10841, 0x608a }, { 10842, 0x60de }, { 10843, 0x60d5 },
    { 10844, 0x6120 }, { 10845, 0x60f2 }, { 10846, 0x6111 }, { 10847, 0x6137 }, { 10848, 0x6130 }, { 10849, 0x6198 }, { 10850, 0x6213 }, { 10851, 0x62a6 },
    { 10852, 0x63f5 }, { 10853, 0x6460 }, { 10854, 0x649d }, { 10855, 0x64ce }, { 10856, 0x654e }, { 10857, 0x6600 }, { 10858, 0x6615 }, { 10859, 0x663b },
    { 10860, 0x6609 }, { 10861, 0x662e }, { 10862, 0x661e }, { 10863, 0x6624 }, { 10864, 0x6665 }, { 10865, 0x6657 }, { 10866, 0x6659 }, { 10867, 0xfa12 },
    { 10868, 0x6673 }, { 10869, 0x6699 }, { 10870, 0x66a0 }, { 10871, 0x66b2 }, { 10872, 0x66bf }, { 10873, 0x66fa }, { 10874, 0x670e }, { 10875, 0xf929 },
    { 10876, 0x6766 }, { 10877, 0x67bb }, { 10878, 0x6852 }, { 10879, 0x67c0 }, { 10880, 0x6801 }, { 10881, 0x6844 }, { 10882, 0x68cf }, { 10883, 0xfa13 },
    { 10884, 0x6968 }, { 10885, 0xfa14 }, { 10886, 0x6998 }, { 10887, 0x69e2 }, { 10888, 0x6a30 }, { 10889, 0x6a6b }, { 10890, 0x6a46 }, { 10891, 0x6a73 },
    { 10892, 0x6a7e }, { 10893, 0x6ae2 }, { 10894, 0x6ae4 }, { 10895, 0x6bd6 }, { 10896, 0x6c3f }, { 10897, 0x6c5c }, { 10898, 0x6c86 }, { 10899, 0x6c6f },
    { 10900, 0x6cda }, { 10901, 0x6d04 }, { 10902, 0x6d87 }, { 10903, 0x6d6f }, { 10904, 0x6d96 }, { 10905, 0x6dac }, { 10906, 0x6dcf }, { 10907, 0x6df8 },
    { 10908, 0x6df2 }, { 10909, 0x6dfc }, { 10910, 0x6e39 }, { 10911, 0x6e5c }, { 10912, 0x6e27 }, { 10913, 0x6e3c }, { 10914, 0x6ebf }, { 10915, 0x6f88 },
    { 10916, 0x6fb5 }, { 10917, 0x6ff5 }, { 10918, 0x7005 }, { 10919, 0x7007 }, { 10920, 0x7028 }, { 10921, 0x7085 }, { 10922, 0x70ab }, { 10923, 0x710f },
    { 10924, 0x7104 }, { 10925, 0x715c }, { 10926, 0x7146 }, { 10927, 0x7147 }, { 10928, 0xfa15 }, { 10929, 0x71c1 }, { 10930, 0x71fe }, { 10931, 0x72b1 },
    { 10932, 0x72be }, { 10933, 0x7324 }, { 10934, 0xfa16 }, { 10935, 0x7377 }, { 10936, 0x73bd }, { 10937, 0x73c9 }, { 10938, 0x73d6 }, { 10939, 0x73e3 },
    { 10940, 0x73d2 }, { 10941, 0x7407 }, { 10942, 0x73f5 }, { 10943, 0x7426 }, { 10944, 0x742a }, { 10945, 0x7429 }, { 10946, 0x742e }, { 10947, 0x7462 },
    { 10948, 0x7489 }, { 10949, 0x749f }, { 10950, 0x7501 }, { 10951, 0x756f }, { 10952, 0x7682 }, { 10953, 0x769c }, { 10954, 0x769e }, { 10955, 0x769b },
    { 10956, 0x76a6 }, { 10957, 0xfa17 }, { 10958, 0x7746 }, { 10959, 0x52af }, { 10960, 0x7821 }, { 10961, 0x784e }, { 10962, 0x7864 }, { 10963, 0x787a },
    { 10964, 0x7930 }, { 10965, 0xfa18 }, { 10966, 0xfa19 }, { 10967, 0xfa1a }, { 10968, 0x7994 }, { 10969, 0xfa1b }, { 10970, 0x799b }, { 10971, 0x7ad1 },
    { 10972, 0x7ae7 }, { 10973, 0xfa1c }, { 10974, 0x7aeb }, { 10975, 0x7b9e }, { 10976, 0xfa1d }, { 10977, 0x7d48 }, { 10978, 0x7d5c }, { 10979, 0x7db7 },
    { 10980, 0x7da0 }, { 10981, 0x7dd6 }, { 10982, 0x7e52 }, { 10983, 0x7f47 }, { 10984, 0x7fa1 }, { 10985, 0xfa1e }, { 10986, 0x8301 }, { 10987, 0x8362 },
    { 10988, 0x837f }, { 10989, 0x83c7 }, { 10990, 0x83f6 }, { 10991, 0x8448 }, { 10992, 0x84b4 }, { 10993, 0x8553 }, { 10994, 0x8559 }, { 10995, 0x856b },
    { 10996, 0xfa1f }, { 10997, 0x85b0 }, { 10998, 0xfa20 }, { 10999, 0xfa21 }, { 11000, 0x8807 }, { 11001, 0x88f5 }, { 11002, 0x8a12 }, { 11003, 0x8a37 },
    { 11004, 0x8a79 }, { 11005, 0x8aa7 }, { 11006, 0x8abe }, { 11007, 0x8adf }, { 11008, 0xfa22 }, { 11009, 0x8af6 }, { 11010, 0x8b53 }, { 11011, 0x8b7f },
    { 11012, 0x8cf0 }, { 11013, 0x8cf4 }, { 11014, 0x8d12 }, { 11015, 0x8d76 }, { 11016, 0xfa23 }, { 11017, 0x8ecf }, { 11018, 0xfa24 }, { 11019, 0xfa25 },
    { 11020, 0x9067 }, { 11021, 0x90de }, { 11022, 0xfa26 }, { 11023, 0x9115 }, { 11024, 0x9127 }, { 11025, 0x91da }, { 11026, 0x91d7 }, { 11027, 0x91de },
    { 11028, 0x91ed }, { 11029, 0x91ee }, { 11030, 0x91e4 }, { 11031, 0x91e5 }, { 11032, 0x9206 }, { 11033, 0x9210 }, { 11034, 0x920a }, { 11035, 0x923a },
    { 11036, 0x9240 }, { 11037, 0x923c }, { 11038, 0x924e }, { 11039, 0x9259 }, { 11040, 0x9251 }, { 11041, 0x9239 }, { 11042, 0x9267 }, { 11043, 0x92a7 },
    { 11044, 0x9277 }, { 11045, 0x9278 }, { 11046, 0x92e7 }, { 11047, 0x92d7 }, { 11048, 0x92d9 }, { 11049, 0x92d0 }, { 11050, 0xfa27 }, { 11051, 0x92d5 },
    { 11052, 0x92e0 }, { 11053, 0x92d3 }, { 11054, 0x9325 }, { 11055, 0x9321 }, { 11056, 0x92fb }, { 11057, 0xfa28 }, { 11058, 0x931e }, { 11059, 0x92ff },
    { 11060, 0x931d }, { 11061, 0x9302 }, { 11062, 0x9370 }, { 11063, 0x9357 }, { 11064, 0x93a4 }, { 11065, 0x93c6 }, { 11066, 0x93de }, { 11067, 0x93f8 },
    { 11068, 0x9431 }, { 11069, 0x9445 }, { 11070, 0x9448 }, { 11071, 0x9592 }, { 11072, 0xf9dc }, { 11073, 0xfa29 }, { 11074, 0x969d }, { 11075, 0x96af },
    { 11076, 0x9733 }, { 11077, 0x973b }, { 11078, 0x9743 }, { 11079, 0x974d }, { 11080, 0x974f }, { 11081, 0x9751 }, { 11082, 0x9755 }, { 11083, 0x9857 },
    { 11084, 0x9865 }, { 11085, 0xfa2a }, { 11086, 0xfa2b }, { 11087, 0x9927 }, { 11088, 0xfa2c }, { 11089, 0x999e }, { 11090, 0x9a4e }, { 11091, 0x9ad9 },
    { 11092, 0x9adc }, { 11093, 0x9b75 }, { 11094, 0x9b72 }, { 11095, 0x9b8f }, { 11096, 0x9bb1 }, { 11097, 0x9bbb }, { 11098, 0x9c00 }, { 11099, 0x9d70 },
    { 11100, 0x9d6b }, { 11101, 0xfa2d }, { 11102, 0x9e19 }, { 11103, 0x9ed1 }
} };

const std::array<std::pair<uint16_t, UChar>, 7724>& jis0208()
{
    // Allocate this at runtime because building it at compile time would make the binary much larger and this is often not used.
    static std::array<std::pair<uint16_t, UChar>, 7724>* array;
    static std::once_flag flag;
    std::call_once(flag, [] {
        array = new std::array<std::pair<uint16_t, UChar>, 7724>;
        size_t arrayIndex = 0;
        
        UErrorCode error = U_ZERO_ERROR;
        auto icuConverter = ICUConverterPtr { ucnv_open("EUC-JP", &error) };
        ASSERT(!error);

        constexpr size_t range = 94;
        uint8_t icuInput[2];
        UChar icuOutput;
        for (size_t i = 0; i < range; i++) {
            for (size_t j = 0; j < range; j++) {
                icuInput[0] = 0xA1 + i;
                icuInput[1] = 0xA1 + j;

                UChar* output = &icuOutput;
                const char* input = reinterpret_cast<const char*>(icuInput);
                ucnv_toUnicode(icuConverter.get(), &output, output + 1, &input, input + sizeof(icuInput), nullptr, true, &error);
                ASSERT(!error);
                if (icuOutput != 0xFFFD) {
                    uint16_t pointer = i * range + j;
                    (*array)[arrayIndex++] = { pointer, icuOutput };
                }
            }
        }
        
        for (auto& extra : jis0208Extras)
            (*array)[arrayIndex++] = extra;
        RELEASE_ASSERT_WITH_SECURITY_IMPLICATION(arrayIndex == 7724);
        ASSERT(*array == jis0208Reference);
    });
    return *array;
}

#if ASSERT_ENABLED
// From https://encoding.spec.whatwg.org/index-jis0212.txt
const std::array<std::pair<uint16_t, UChar>, 6067> jis0212Reference { {
    { 108, 0x02D8 }, { 109, 0x02C7 }, { 110, 0x00B8 }, { 111, 0x02D9 }, { 112, 0x02DD }, { 113, 0x00AF }, { 114, 0x02DB }, { 115, 0x02DA },
    { 116, 0xFF5E }, { 117, 0x0384 }, { 118, 0x0385 }, { 127, 0x00A1 }, { 128, 0x00A6 }, { 129, 0x00BF }, { 168, 0x00BA }, { 169, 0x00AA },
    { 170, 0x00A9 }, { 171, 0x00AE }, { 172, 0x2122 }, { 173, 0x00A4 }, { 174, 0x2116 }, { 534, 0x0386 }, { 535, 0x0388 }, { 536, 0x0389 },
    { 537, 0x038A }, { 538, 0x03AA }, { 540, 0x038C }, { 542, 0x038E }, { 543, 0x03AB }, { 545, 0x038F }, { 550, 0x03AC }, { 551, 0x03AD },
    { 552, 0x03AE }, { 553, 0x03AF }, { 554, 0x03CA }, { 555, 0x0390 }, { 556, 0x03CC }, { 557, 0x03C2 }, { 558, 0x03CD }, { 559, 0x03CB },
    { 560, 0x03B0 }, { 561, 0x03CE }, { 597, 0x0402 }, { 598, 0x0403 }, { 599, 0x0404 }, { 600, 0x0405 }, { 601, 0x0406 }, { 602, 0x0407 },
    { 603, 0x0408 }, { 604, 0x0409 }, { 605, 0x040A }, { 606, 0x040B }, { 607, 0x040C }, { 608, 0x040E }, { 609, 0x040F }, { 645, 0x0452 },
    { 646, 0x0453 }, { 647, 0x0454 }, { 648, 0x0455 }, { 649, 0x0456 }, { 650, 0x0457 }, { 651, 0x0458 }, { 652, 0x0459 }, { 653, 0x045A },
    { 654, 0x045B }, { 655, 0x045C }, { 656, 0x045E }, { 657, 0x045F }, { 752, 0x00C6 }, { 753, 0x0110 }, { 755, 0x0126 }, { 757, 0x0132 },
    { 759, 0x0141 }, { 760, 0x013F }, { 762, 0x014A }, { 763, 0x00D8 }, { 764, 0x0152 }, { 766, 0x0166 }, { 767, 0x00DE }, { 784, 0x00E6 },
    { 785, 0x0111 }, { 786, 0x00F0 }, { 787, 0x0127 }, { 788, 0x0131 }, { 789, 0x0133 }, { 790, 0x0138 }, { 791, 0x0142 }, { 792, 0x0140 },
    { 793, 0x0149 }, { 794, 0x014B }, { 795, 0x00F8 }, { 796, 0x0153 }, { 797, 0x00DF }, { 798, 0x0167 }, { 799, 0x00FE }, { 846, 0x00C1 },
    { 847, 0x00C0 }, { 848, 0x00C4 }, { 849, 0x00C2 }, { 850, 0x0102 }, { 851, 0x01CD }, { 852, 0x0100 }, { 853, 0x0104 }, { 854, 0x00C5 },
    { 855, 0x00C3 }, { 856, 0x0106 }, { 857, 0x0108 }, { 858, 0x010C }, { 859, 0x00C7 }, { 860, 0x010A }, { 861, 0x010E }, { 862, 0x00C9 },
    { 863, 0x00C8 }, { 864, 0x00CB }, { 865, 0x00CA }, { 866, 0x011A }, { 867, 0x0116 }, { 868, 0x0112 }, { 869, 0x0118 }, { 871, 0x011C },
    { 872, 0x011E }, { 873, 0x0122 }, { 874, 0x0120 }, { 875, 0x0124 }, { 876, 0x00CD }, { 877, 0x00CC }, { 878, 0x00CF }, { 879, 0x00CE },
    { 880, 0x01CF }, { 881, 0x0130 }, { 882, 0x012A }, { 883, 0x012E }, { 884, 0x0128 }, { 885, 0x0134 }, { 886, 0x0136 }, { 887, 0x0139 },
    { 888, 0x013D }, { 889, 0x013B }, { 890, 0x0143 }, { 891, 0x0147 }, { 892, 0x0145 }, { 893, 0x00D1 }, { 894, 0x00D3 }, { 895, 0x00D2 },
    { 896, 0x00D6 }, { 897, 0x00D4 }, { 898, 0x01D1 }, { 899, 0x0150 }, { 900, 0x014C }, { 901, 0x00D5 }, { 902, 0x0154 }, { 903, 0x0158 },
    { 904, 0x0156 }, { 905, 0x015A }, { 906, 0x015C }, { 907, 0x0160 }, { 908, 0x015E }, { 909, 0x0164 }, { 910, 0x0162 }, { 911, 0x00DA },
    { 912, 0x00D9 }, { 913, 0x00DC }, { 914, 0x00DB }, { 915, 0x016C }, { 916, 0x01D3 }, { 917, 0x0170 }, { 918, 0x016A }, { 919, 0x0172 },
    { 920, 0x016E }, { 921, 0x0168 }, { 922, 0x01D7 }, { 923, 0x01DB }, { 924, 0x01D9 }, { 925, 0x01D5 }, { 926, 0x0174 }, { 927, 0x00DD },
    { 928, 0x0178 }, { 929, 0x0176 }, { 930, 0x0179 }, { 931, 0x017D }, { 932, 0x017B }, { 940, 0x00E1 }, { 941, 0x00E0 }, { 942, 0x00E4 },
    { 943, 0x00E2 }, { 944, 0x0103 }, { 945, 0x01CE }, { 946, 0x0101 }, { 947, 0x0105 }, { 948, 0x00E5 }, { 949, 0x00E3 }, { 950, 0x0107 },
    { 951, 0x0109 }, { 952, 0x010D }, { 953, 0x00E7 }, { 954, 0x010B }, { 955, 0x010F }, { 956, 0x00E9 }, { 957, 0x00E8 }, { 958, 0x00EB },
    { 959, 0x00EA }, { 960, 0x011B }, { 961, 0x0117 }, { 962, 0x0113 }, { 963, 0x0119 }, { 964, 0x01F5 }, { 965, 0x011D }, { 966, 0x011F },
    { 968, 0x0121 }, { 969, 0x0125 }, { 970, 0x00ED }, { 971, 0x00EC }, { 972, 0x00EF }, { 973, 0x00EE }, { 974, 0x01D0 }, { 976, 0x012B },
    { 977, 0x012F }, { 978, 0x0129 }, { 979, 0x0135 }, { 980, 0x0137 }, { 981, 0x013A }, { 982, 0x013E }, { 983, 0x013C }, { 984, 0x0144 },
    { 985, 0x0148 }, { 986, 0x0146 }, { 987, 0x00F1 }, { 988, 0x00F3 }, { 989, 0x00F2 }, { 990, 0x00F6 }, { 991, 0x00F4 }, { 992, 0x01D2 },
    { 993, 0x0151 }, { 994, 0x014D }, { 995, 0x00F5 }, { 996, 0x0155 }, { 997, 0x0159 }, { 998, 0x0157 }, { 999, 0x015B }, { 1000, 0x015D },
    { 1001, 0x0161 }, { 1002, 0x015F }, { 1003, 0x0165 }, { 1004, 0x0163 }, { 1005, 0x00FA }, { 1006, 0x00F9 }, { 1007, 0x00FC }, { 1008, 0x00FB },
    { 1009, 0x016D }, { 1010, 0x01D4 }, { 1011, 0x0171 }, { 1012, 0x016B }, { 1013, 0x0173 }, { 1014, 0x016F }, { 1015, 0x0169 }, { 1016, 0x01D8 },
    { 1017, 0x01DC }, { 1018, 0x01DA }, { 1019, 0x01D6 }, { 1020, 0x0175 }, { 1021, 0x00FD }, { 1022, 0x00FF }, { 1023, 0x0177 }, { 1024, 0x017A },
    { 1025, 0x017E }, { 1026, 0x017C }, { 1410, 0x4E02 }, { 1411, 0x4E04 }, { 1412, 0x4E05 }, { 1413, 0x4E0C }, { 1414, 0x4E12 }, { 1415, 0x4E1F },
    { 1416, 0x4E23 }, { 1417, 0x4E24 }, { 1418, 0x4E28 }, { 1419, 0x4E2B }, { 1420, 0x4E2E }, { 1421, 0x4E2F }, { 1422, 0x4E30 }, { 1423, 0x4E35 },
    { 1424, 0x4E40 }, { 1425, 0x4E41 }, { 1426, 0x4E44 }, { 1427, 0x4E47 }, { 1428, 0x4E51 }, { 1429, 0x4E5A }, { 1430, 0x4E5C }, { 1431, 0x4E63 },
    { 1432, 0x4E68 }, { 1433, 0x4E69 }, { 1434, 0x4E74 }, { 1435, 0x4E75 }, { 1436, 0x4E79 }, { 1437, 0x4E7F }, { 1438, 0x4E8D }, { 1439, 0x4E96 },
    { 1440, 0x4E97 }, { 1441, 0x4E9D }, { 1442, 0x4EAF }, { 1443, 0x4EB9 }, { 1444, 0x4EC3 }, { 1445, 0x4ED0 }, { 1446, 0x4EDA }, { 1447, 0x4EDB },
    { 1448, 0x4EE0 }, { 1449, 0x4EE1 }, { 1450, 0x4EE2 }, { 1451, 0x4EE8 }, { 1452, 0x4EEF }, { 1453, 0x4EF1 }, { 1454, 0x4EF3 }, { 1455, 0x4EF5 },
    { 1456, 0x4EFD }, { 1457, 0x4EFE }, { 1458, 0x4EFF }, { 1459, 0x4F00 }, { 1460, 0x4F02 }, { 1461, 0x4F03 }, { 1462, 0x4F08 }, { 1463, 0x4F0B },
    { 1464, 0x4F0C }, { 1465, 0x4F12 }, { 1466, 0x4F15 }, { 1467, 0x4F16 }, { 1468, 0x4F17 }, { 1469, 0x4F19 }, { 1470, 0x4F2E }, { 1471, 0x4F31 },
    { 1472, 0x4F60 }, { 1473, 0x4F33 }, { 1474, 0x4F35 }, { 1475, 0x4F37 }, { 1476, 0x4F39 }, { 1477, 0x4F3B }, { 1478, 0x4F3E }, { 1479, 0x4F40 },
    { 1480, 0x4F42 }, { 1481, 0x4F48 }, { 1482, 0x4F49 }, { 1483, 0x4F4B }, { 1484, 0x4F4C }, { 1485, 0x4F52 }, { 1486, 0x4F54 }, { 1487, 0x4F56 },
    { 1488, 0x4F58 }, { 1489, 0x4F5F }, { 1490, 0x4F63 }, { 1491, 0x4F6A }, { 1492, 0x4F6C }, { 1493, 0x4F6E }, { 1494, 0x4F71 }, { 1495, 0x4F77 },
    { 1496, 0x4F78 }, { 1497, 0x4F79 }, { 1498, 0x4F7A }, { 1499, 0x4F7D }, { 1500, 0x4F7E }, { 1501, 0x4F81 }, { 1502, 0x4F82 }, { 1503, 0x4F84 },
    { 1504, 0x4F85 }, { 1505, 0x4F89 }, { 1506, 0x4F8A }, { 1507, 0x4F8C }, { 1508, 0x4F8E }, { 1509, 0x4F90 }, { 1510, 0x4F92 }, { 1511, 0x4F93 },
    { 1512, 0x4F94 }, { 1513, 0x4F97 }, { 1514, 0x4F99 }, { 1515, 0x4F9A }, { 1516, 0x4F9E }, { 1517, 0x4F9F }, { 1518, 0x4FB2 }, { 1519, 0x4FB7 },
    { 1520, 0x4FB9 }, { 1521, 0x4FBB }, { 1522, 0x4FBC }, { 1523, 0x4FBD }, { 1524, 0x4FBE }, { 1525, 0x4FC0 }, { 1526, 0x4FC1 }, { 1527, 0x4FC5 },
    { 1528, 0x4FC6 }, { 1529, 0x4FC8 }, { 1530, 0x4FC9 }, { 1531, 0x4FCB }, { 1532, 0x4FCC }, { 1533, 0x4FCD }, { 1534, 0x4FCF }, { 1535, 0x4FD2 },
    { 1536, 0x4FDC }, { 1537, 0x4FE0 }, { 1538, 0x4FE2 }, { 1539, 0x4FF0 }, { 1540, 0x4FF2 }, { 1541, 0x4FFC }, { 1542, 0x4FFD }, { 1543, 0x4FFF },
    { 1544, 0x5000 }, { 1545, 0x5001 }, { 1546, 0x5004 }, { 1547, 0x5007 }, { 1548, 0x500A }, { 1549, 0x500C }, { 1550, 0x500E }, { 1551, 0x5010 },
    { 1552, 0x5013 }, { 1553, 0x5017 }, { 1554, 0x5018 }, { 1555, 0x501B }, { 1556, 0x501C }, { 1557, 0x501D }, { 1558, 0x501E }, { 1559, 0x5022 },
    { 1560, 0x5027 }, { 1561, 0x502E }, { 1562, 0x5030 }, { 1563, 0x5032 }, { 1564, 0x5033 }, { 1565, 0x5035 }, { 1566, 0x5040 }, { 1567, 0x5041 },
    { 1568, 0x5042 }, { 1569, 0x5045 }, { 1570, 0x5046 }, { 1571, 0x504A }, { 1572, 0x504C }, { 1573, 0x504E }, { 1574, 0x5051 }, { 1575, 0x5052 },
    { 1576, 0x5053 }, { 1577, 0x5057 }, { 1578, 0x5059 }, { 1579, 0x505F }, { 1580, 0x5060 }, { 1581, 0x5062 }, { 1582, 0x5063 }, { 1583, 0x5066 },
    { 1584, 0x5067 }, { 1585, 0x506A }, { 1586, 0x506D }, { 1587, 0x5070 }, { 1588, 0x5071 }, { 1589, 0x503B }, { 1590, 0x5081 }, { 1591, 0x5083 },
    { 1592, 0x5084 }, { 1593, 0x5086 }, { 1594, 0x508A }, { 1595, 0x508E }, { 1596, 0x508F }, { 1597, 0x5090 }, { 1598, 0x5092 }, { 1599, 0x5093 },
    { 1600, 0x5094 }, { 1601, 0x5096 }, { 1602, 0x509B }, { 1603, 0x509C }, { 1604, 0x509E }, { 1605, 0x509F }, { 1606, 0x50A0 }, { 1607, 0x50A1 },
    { 1608, 0x50A2 }, { 1609, 0x50AA }, { 1610, 0x50AF }, { 1611, 0x50B0 }, { 1612, 0x50B9 }, { 1613, 0x50BA }, { 1614, 0x50BD }, { 1615, 0x50C0 },
    { 1616, 0x50C3 }, { 1617, 0x50C4 }, { 1618, 0x50C7 }, { 1619, 0x50CC }, { 1620, 0x50CE }, { 1621, 0x50D0 }, { 1622, 0x50D3 }, { 1623, 0x50D4 },
    { 1624, 0x50D8 }, { 1625, 0x50DC }, { 1626, 0x50DD }, { 1627, 0x50DF }, { 1628, 0x50E2 }, { 1629, 0x50E4 }, { 1630, 0x50E6 }, { 1631, 0x50E8 },
    { 1632, 0x50E9 }, { 1633, 0x50EF }, { 1634, 0x50F1 }, { 1635, 0x50F6 }, { 1636, 0x50FA }, { 1637, 0x50FE }, { 1638, 0x5103 }, { 1639, 0x5106 },
    { 1640, 0x5107 }, { 1641, 0x5108 }, { 1642, 0x510B }, { 1643, 0x510C }, { 1644, 0x510D }, { 1645, 0x510E }, { 1646, 0x50F2 }, { 1647, 0x5110 },
    { 1648, 0x5117 }, { 1649, 0x5119 }, { 1650, 0x511B }, { 1651, 0x511C }, { 1652, 0x511D }, { 1653, 0x511E }, { 1654, 0x5123 }, { 1655, 0x5127 },
    { 1656, 0x5128 }, { 1657, 0x512C }, { 1658, 0x512D }, { 1659, 0x512F }, { 1660, 0x5131 }, { 1661, 0x5133 }, { 1662, 0x5134 }, { 1663, 0x5135 },
    { 1664, 0x5138 }, { 1665, 0x5139 }, { 1666, 0x5142 }, { 1667, 0x514A }, { 1668, 0x514F }, { 1669, 0x5153 }, { 1670, 0x5155 }, { 1671, 0x5157 },
    { 1672, 0x5158 }, { 1673, 0x515F }, { 1674, 0x5164 }, { 1675, 0x5166 }, { 1676, 0x517E }, { 1677, 0x5183 }, { 1678, 0x5184 }, { 1679, 0x518B },
    { 1680, 0x518E }, { 1681, 0x5198 }, { 1682, 0x519D }, { 1683, 0x51A1 }, { 1684, 0x51A3 }, { 1685, 0x51AD }, { 1686, 0x51B8 }, { 1687, 0x51BA },
    { 1688, 0x51BC }, { 1689, 0x51BE }, { 1690, 0x51BF }, { 1691, 0x51C2 }, { 1692, 0x51C8 }, { 1693, 0x51CF }, { 1694, 0x51D1 }, { 1695, 0x51D2 },
    { 1696, 0x51D3 }, { 1697, 0x51D5 }, { 1698, 0x51D8 }, { 1699, 0x51DE }, { 1700, 0x51E2 }, { 1701, 0x51E5 }, { 1702, 0x51EE }, { 1703, 0x51F2 },
    { 1704, 0x51F3 }, { 1705, 0x51F4 }, { 1706, 0x51F7 }, { 1707, 0x5201 }, { 1708, 0x5202 }, { 1709, 0x5205 }, { 1710, 0x5212 }, { 1711, 0x5213 },
    { 1712, 0x5215 }, { 1713, 0x5216 }, { 1714, 0x5218 }, { 1715, 0x5222 }, { 1716, 0x5228 }, { 1717, 0x5231 }, { 1718, 0x5232 }, { 1719, 0x5235 },
    { 1720, 0x523C }, { 1721, 0x5245 }, { 1722, 0x5249 }, { 1723, 0x5255 }, { 1724, 0x5257 }, { 1725, 0x5258 }, { 1726, 0x525A }, { 1727, 0x525C },
    { 1728, 0x525F }, { 1729, 0x5260 }, { 1730, 0x5261 }, { 1731, 0x5266 }, { 1732, 0x526E }, { 1733, 0x5277 }, { 1734, 0x5278 }, { 1735, 0x5279 },
    { 1736, 0x5280 }, { 1737, 0x5282 }, { 1738, 0x5285 }, { 1739, 0x528A }, { 1740, 0x528C }, { 1741, 0x5293 }, { 1742, 0x5295 }, { 1743, 0x5296 },
    { 1744, 0x5297 }, { 1745, 0x5298 }, { 1746, 0x529A }, { 1747, 0x529C }, { 1748, 0x52A4 }, { 1749, 0x52A5 }, { 1750, 0x52A6 }, { 1751, 0x52A7 },
    { 1752, 0x52AF }, { 1753, 0x52B0 }, { 1754, 0x52B6 }, { 1755, 0x52B7 }, { 1756, 0x52B8 }, { 1757, 0x52BA }, { 1758, 0x52BB }, { 1759, 0x52BD },
    { 1760, 0x52C0 }, { 1761, 0x52C4 }, { 1762, 0x52C6 }, { 1763, 0x52C8 }, { 1764, 0x52CC }, { 1765, 0x52CF }, { 1766, 0x52D1 }, { 1767, 0x52D4 },
    { 1768, 0x52D6 }, { 1769, 0x52DB }, { 1770, 0x52DC }, { 1771, 0x52E1 }, { 1772, 0x52E5 }, { 1773, 0x52E8 }, { 1774, 0x52E9 }, { 1775, 0x52EA },
    { 1776, 0x52EC }, { 1777, 0x52F0 }, { 1778, 0x52F1 }, { 1779, 0x52F4 }, { 1780, 0x52F6 }, { 1781, 0x52F7 }, { 1782, 0x5300 }, { 1783, 0x5303 },
    { 1784, 0x530A }, { 1785, 0x530B }, { 1786, 0x530C }, { 1787, 0x5311 }, { 1788, 0x5313 }, { 1789, 0x5318 }, { 1790, 0x531B }, { 1791, 0x531C },
    { 1792, 0x531E }, { 1793, 0x531F }, { 1794, 0x5325 }, { 1795, 0x5327 }, { 1796, 0x5328 }, { 1797, 0x5329 }, { 1798, 0x532B }, { 1799, 0x532C },
    { 1800, 0x532D }, { 1801, 0x5330 }, { 1802, 0x5332 }, { 1803, 0x5335 }, { 1804, 0x533C }, { 1805, 0x533D }, { 1806, 0x533E }, { 1807, 0x5342 },
    { 1808, 0x534C }, { 1809, 0x534B }, { 1810, 0x5359 }, { 1811, 0x535B }, { 1812, 0x5361 }, { 1813, 0x5363 }, { 1814, 0x5365 }, { 1815, 0x536C },
    { 1816, 0x536D }, { 1817, 0x5372 }, { 1818, 0x5379 }, { 1819, 0x537E }, { 1820, 0x5383 }, { 1821, 0x5387 }, { 1822, 0x5388 }, { 1823, 0x538E },
    { 1824, 0x5393 }, { 1825, 0x5394 }, { 1826, 0x5399 }, { 1827, 0x539D }, { 1828, 0x53A1 }, { 1829, 0x53A4 }, { 1830, 0x53AA }, { 1831, 0x53AB },
    { 1832, 0x53AF }, { 1833, 0x53B2 }, { 1834, 0x53B4 }, { 1835, 0x53B5 }, { 1836, 0x53B7 }, { 1837, 0x53B8 }, { 1838, 0x53BA }, { 1839, 0x53BD },
    { 1840, 0x53C0 }, { 1841, 0x53C5 }, { 1842, 0x53CF }, { 1843, 0x53D2 }, { 1844, 0x53D3 }, { 1845, 0x53D5 }, { 1846, 0x53DA }, { 1847, 0x53DD },
    { 1848, 0x53DE }, { 1849, 0x53E0 }, { 1850, 0x53E6 }, { 1851, 0x53E7 }, { 1852, 0x53F5 }, { 1853, 0x5402 }, { 1854, 0x5413 }, { 1855, 0x541A },
    { 1856, 0x5421 }, { 1857, 0x5427 }, { 1858, 0x5428 }, { 1859, 0x542A }, { 1860, 0x542F }, { 1861, 0x5431 }, { 1862, 0x5434 }, { 1863, 0x5435 },
    { 1864, 0x5443 }, { 1865, 0x5444 }, { 1866, 0x5447 }, { 1867, 0x544D }, { 1868, 0x544F }, { 1869, 0x545E }, { 1870, 0x5462 }, { 1871, 0x5464 },
    { 1872, 0x5466 }, { 1873, 0x5467 }, { 1874, 0x5469 }, { 1875, 0x546B }, { 1876, 0x546D }, { 1877, 0x546E }, { 1878, 0x5474 }, { 1879, 0x547F },
    { 1880, 0x5481 }, { 1881, 0x5483 }, { 1882, 0x5485 }, { 1883, 0x5488 }, { 1884, 0x5489 }, { 1885, 0x548D }, { 1886, 0x5491 }, { 1887, 0x5495 },
    { 1888, 0x5496 }, { 1889, 0x549C }, { 1890, 0x549F }, { 1891, 0x54A1 }, { 1892, 0x54A6 }, { 1893, 0x54A7 }, { 1894, 0x54A9 }, { 1895, 0x54AA },
    { 1896, 0x54AD }, { 1897, 0x54AE }, { 1898, 0x54B1 }, { 1899, 0x54B7 }, { 1900, 0x54B9 }, { 1901, 0x54BA }, { 1902, 0x54BB }, { 1903, 0x54BF },
    { 1904, 0x54C6 }, { 1905, 0x54CA }, { 1906, 0x54CD }, { 1907, 0x54CE }, { 1908, 0x54E0 }, { 1909, 0x54EA }, { 1910, 0x54EC }, { 1911, 0x54EF },
    { 1912, 0x54F6 }, { 1913, 0x54FC }, { 1914, 0x54FE }, { 1915, 0x54FF }, { 1916, 0x5500 }, { 1917, 0x5501 }, { 1918, 0x5505 }, { 1919, 0x5508 },
    { 1920, 0x5509 }, { 1921, 0x550C }, { 1922, 0x550D }, { 1923, 0x550E }, { 1924, 0x5515 }, { 1925, 0x552A }, { 1926, 0x552B }, { 1927, 0x5532 },
    { 1928, 0x5535 }, { 1929, 0x5536 }, { 1930, 0x553B }, { 1931, 0x553C }, { 1932, 0x553D }, { 1933, 0x5541 }, { 1934, 0x5547 }, { 1935, 0x5549 },
    { 1936, 0x554A }, { 1937, 0x554D }, { 1938, 0x5550 }, { 1939, 0x5551 }, { 1940, 0x5558 }, { 1941, 0x555A }, { 1942, 0x555B }, { 1943, 0x555E },
    { 1944, 0x5560 }, { 1945, 0x5561 }, { 1946, 0x5564 }, { 1947, 0x5566 }, { 1948, 0x557F }, { 1949, 0x5581 }, { 1950, 0x5582 }, { 1951, 0x5586 },
    { 1952, 0x5588 }, { 1953, 0x558E }, { 1954, 0x558F }, { 1955, 0x5591 }, { 1956, 0x5592 }, { 1957, 0x5593 }, { 1958, 0x5594 }, { 1959, 0x5597 },
    { 1960, 0x55A3 }, { 1961, 0x55A4 }, { 1962, 0x55AD }, { 1963, 0x55B2 }, { 1964, 0x55BF }, { 1965, 0x55C1 }, { 1966, 0x55C3 }, { 1967, 0x55C6 },
    { 1968, 0x55C9 }, { 1969, 0x55CB }, { 1970, 0x55CC }, { 1971, 0x55CE }, { 1972, 0x55D1 }, { 1973, 0x55D2 }, { 1974, 0x55D3 }, { 1975, 0x55D7 },
    { 1976, 0x55D8 }, { 1977, 0x55DB }, { 1978, 0x55DE }, { 1979, 0x55E2 }, { 1980, 0x55E9 }, { 1981, 0x55F6 }, { 1982, 0x55FF }, { 1983, 0x5605 },
    { 1984, 0x5608 }, { 1985, 0x560A }, { 1986, 0x560D }, { 1987, 0x560E }, { 1988, 0x560F }, { 1989, 0x5610 }, { 1990, 0x5611 }, { 1991, 0x5612 },
    { 1992, 0x5619 }, { 1993, 0x562C }, { 1994, 0x5630 }, { 1995, 0x5633 }, { 1996, 0x5635 }, { 1997, 0x5637 }, { 1998, 0x5639 }, { 1999, 0x563B },
    { 2000, 0x563C }, { 2001, 0x563D }, { 2002, 0x563F }, { 2003, 0x5640 }, { 2004, 0x5641 }, { 2005, 0x5643 }, { 2006, 0x5644 }, { 2007, 0x5646 },
    { 2008, 0x5649 }, { 2009, 0x564B }, { 2010, 0x564D }, { 2011, 0x564F }, { 2012, 0x5654 }, { 2013, 0x565E }, { 2014, 0x5660 }, { 2015, 0x5661 },
    { 2016, 0x5662 }, { 2017, 0x5663 }, { 2018, 0x5666 }, { 2019, 0x5669 }, { 2020, 0x566D }, { 2021, 0x566F }, { 2022, 0x5671 }, { 2023, 0x5672 },
    { 2024, 0x5675 }, { 2025, 0x5684 }, { 2026, 0x5685 }, { 2027, 0x5688 }, { 2028, 0x568B }, { 2029, 0x568C }, { 2030, 0x5695 }, { 2031, 0x5699 },
    { 2032, 0x569A }, { 2033, 0x569D }, { 2034, 0x569E }, { 2035, 0x569F }, { 2036, 0x56A6 }, { 2037, 0x56A7 }, { 2038, 0x56A8 }, { 2039, 0x56A9 },
    { 2040, 0x56AB }, { 2041, 0x56AC }, { 2042, 0x56AD }, { 2043, 0x56B1 }, { 2044, 0x56B3 }, { 2045, 0x56B7 }, { 2046, 0x56BE }, { 2047, 0x56C5 },
    { 2048, 0x56C9 }, { 2049, 0x56CA }, { 2050, 0x56CB }, { 2051, 0x56CF }, { 2052, 0x56D0 }, { 2053, 0x56CC }, { 2054, 0x56CD }, { 2055, 0x56D9 },
    { 2056, 0x56DC }, { 2057, 0x56DD }, { 2058, 0x56DF }, { 2059, 0x56E1 }, { 2060, 0x56E4 }, { 2061, 0x56E5 }, { 2062, 0x56E6 }, { 2063, 0x56E7 },
    { 2064, 0x56E8 }, { 2065, 0x56F1 }, { 2066, 0x56EB }, { 2067, 0x56ED }, { 2068, 0x56F6 }, { 2069, 0x56F7 }, { 2070, 0x5701 }, { 2071, 0x5702 },
    { 2072, 0x5707 }, { 2073, 0x570A }, { 2074, 0x570C }, { 2075, 0x5711 }, { 2076, 0x5715 }, { 2077, 0x571A }, { 2078, 0x571B }, { 2079, 0x571D },
    { 2080, 0x5720 }, { 2081, 0x5722 }, { 2082, 0x5723 }, { 2083, 0x5724 }, { 2084, 0x5725 }, { 2085, 0x5729 }, { 2086, 0x572A }, { 2087, 0x572C },
    { 2088, 0x572E }, { 2089, 0x572F }, { 2090, 0x5733 }, { 2091, 0x5734 }, { 2092, 0x573D }, { 2093, 0x573E }, { 2094, 0x573F }, { 2095, 0x5745 },
    { 2096, 0x5746 }, { 2097, 0x574C }, { 2098, 0x574D }, { 2099, 0x5752 }, { 2100, 0x5762 }, { 2101, 0x5765 }, { 2102, 0x5767 }, { 2103, 0x5768 },
    { 2104, 0x576B }, { 2105, 0x576D }, { 2106, 0x576E }, { 2107, 0x576F }, { 2108, 0x5770 }, { 2109, 0x5771 }, { 2110, 0x5773 }, { 2111, 0x5774 },
    { 2112, 0x5775 }, { 2113, 0x5777 }, { 2114, 0x5779 }, { 2115, 0x577A }, { 2116, 0x577B }, { 2117, 0x577C }, { 2118, 0x577E }, { 2119, 0x5781 },
    { 2120, 0x5783 }, { 2121, 0x578C }, { 2122, 0x5794 }, { 2123, 0x5797 }, { 2124, 0x5799 }, { 2125, 0x579A }, { 2126, 0x579C }, { 2127, 0x579D },
    { 2128, 0x579E }, { 2129, 0x579F }, { 2130, 0x57A1 }, { 2131, 0x5795 }, { 2132, 0x57A7 }, { 2133, 0x57A8 }, { 2134, 0x57A9 }, { 2135, 0x57AC },
    { 2136, 0x57B8 }, { 2137, 0x57BD }, { 2138, 0x57C7 }, { 2139, 0x57C8 }, { 2140, 0x57CC }, { 2141, 0x57CF }, { 2142, 0x57D5 }, { 2143, 0x57DD },
    { 2144, 0x57DE }, { 2145, 0x57E4 }, { 2146, 0x57E6 }, { 2147, 0x57E7 }, { 2148, 0x57E9 }, { 2149, 0x57ED }, { 2150, 0x57F0 }, { 2151, 0x57F5 },
    { 2152, 0x57F6 }, { 2153, 0x57F8 }, { 2154, 0x57FD }, { 2155, 0x57FE }, { 2156, 0x57FF }, { 2157, 0x5803 }, { 2158, 0x5804 }, { 2159, 0x5808 },
    { 2160, 0x5809 }, { 2161, 0x57E1 }, { 2162, 0x580C }, { 2163, 0x580D }, { 2164, 0x581B }, { 2165, 0x581E }, { 2166, 0x581F }, { 2167, 0x5820 },
    { 2168, 0x5826 }, { 2169, 0x5827 }, { 2170, 0x582D }, { 2171, 0x5832 }, { 2172, 0x5839 }, { 2173, 0x583F }, { 2174, 0x5849 }, { 2175, 0x584C },
    { 2176, 0x584D }, { 2177, 0x584F }, { 2178, 0x5850 }, { 2179, 0x5855 }, { 2180, 0x585F }, { 2181, 0x5861 }, { 2182, 0x5864 }, { 2183, 0x5867 },
    { 2184, 0x5868 }, { 2185, 0x5878 }, { 2186, 0x587C }, { 2187, 0x587F }, { 2188, 0x5880 }, { 2189, 0x5881 }, { 2190, 0x5887 }, { 2191, 0x5888 },
    { 2192, 0x5889 }, { 2193, 0x588A }, { 2194, 0x588C }, { 2195, 0x588D }, { 2196, 0x588F }, { 2197, 0x5890 }, { 2198, 0x5894 }, { 2199, 0x5896 },
    { 2200, 0x589D }, { 2201, 0x58A0 }, { 2202, 0x58A1 }, { 2203, 0x58A2 }, { 2204, 0x58A6 }, { 2205, 0x58A9 }, { 2206, 0x58B1 }, { 2207, 0x58B2 },
    { 2208, 0x58C4 }, { 2209, 0x58BC }, { 2210, 0x58C2 }, { 2211, 0x58C8 }, { 2212, 0x58CD }, { 2213, 0x58CE }, { 2214, 0x58D0 }, { 2215, 0x58D2 },
    { 2216, 0x58D4 }, { 2217, 0x58D6 }, { 2218, 0x58DA }, { 2219, 0x58DD }, { 2220, 0x58E1 }, { 2221, 0x58E2 }, { 2222, 0x58E9 }, { 2223, 0x58F3 },
    { 2224, 0x5905 }, { 2225, 0x5906 }, { 2226, 0x590B }, { 2227, 0x590C }, { 2228, 0x5912 }, { 2229, 0x5913 }, { 2230, 0x5914 }, { 2231, 0x8641 },
    { 2232, 0x591D }, { 2233, 0x5921 }, { 2234, 0x5923 }, { 2235, 0x5924 }, { 2236, 0x5928 }, { 2237, 0x592F }, { 2238, 0x5930 }, { 2239, 0x5933 },
    { 2240, 0x5935 }, { 2241, 0x5936 }, { 2242, 0x593F }, { 2243, 0x5943 }, { 2244, 0x5946 }, { 2245, 0x5952 }, { 2246, 0x5953 }, { 2247, 0x5959 },
    { 2248, 0x595B }, { 2249, 0x595D }, { 2250, 0x595E }, { 2251, 0x595F }, { 2252, 0x5961 }, { 2253, 0x5963 }, { 2254, 0x596B }, { 2255, 0x596D },
    { 2256, 0x596F }, { 2257, 0x5972 }, { 2258, 0x5975 }, { 2259, 0x5976 }, { 2260, 0x5979 }, { 2261, 0x597B }, { 2262, 0x597C }, { 2263, 0x598B },
    { 2264, 0x598C }, { 2265, 0x598E }, { 2266, 0x5992 }, { 2267, 0x5995 }, { 2268, 0x5997 }, { 2269, 0x599F }, { 2270, 0x59A4 }, { 2271, 0x59A7 },
    { 2272, 0x59AD }, { 2273, 0x59AE }, { 2274, 0x59AF }, { 2275, 0x59B0 }, { 2276, 0x59B3 }, { 2277, 0x59B7 }, { 2278, 0x59BA }, { 2279, 0x59BC },
    { 2280, 0x59C1 }, { 2281, 0x59C3 }, { 2282, 0x59C4 }, { 2283, 0x59C8 }, { 2284, 0x59CA }, { 2285, 0x59CD }, { 2286, 0x59D2 }, { 2287, 0x59DD },
    { 2288, 0x59DE }, { 2289, 0x59DF }, { 2290, 0x59E3 }, { 2291, 0x59E4 }, { 2292, 0x59E7 }, { 2293, 0x59EE }, { 2294, 0x59EF }, { 2295, 0x59F1 },
    { 2296, 0x59F2 }, { 2297, 0x59F4 }, { 2298, 0x59F7 }, { 2299, 0x5A00 }, { 2300, 0x5A04 }, { 2301, 0x5A0C }, { 2302, 0x5A0D }, { 2303, 0x5A0E },
    { 2304, 0x5A12 }, { 2305, 0x5A13 }, { 2306, 0x5A1E }, { 2307, 0x5A23 }, { 2308, 0x5A24 }, { 2309, 0x5A27 }, { 2310, 0x5A28 }, { 2311, 0x5A2A },
    { 2312, 0x5A2D }, { 2313, 0x5A30 }, { 2314, 0x5A44 }, { 2315, 0x5A45 }, { 2316, 0x5A47 }, { 2317, 0x5A48 }, { 2318, 0x5A4C }, { 2319, 0x5A50 },
    { 2320, 0x5A55 }, { 2321, 0x5A5E }, { 2322, 0x5A63 }, { 2323, 0x5A65 }, { 2324, 0x5A67 }, { 2325, 0x5A6D }, { 2326, 0x5A77 }, { 2327, 0x5A7A },
    { 2328, 0x5A7B }, { 2329, 0x5A7E }, { 2330, 0x5A8B }, { 2331, 0x5A90 }, { 2332, 0x5A93 }, { 2333, 0x5A96 }, { 2334, 0x5A99 }, { 2335, 0x5A9C },
    { 2336, 0x5A9E }, { 2337, 0x5A9F }, { 2338, 0x5AA0 }, { 2339, 0x5AA2 }, { 2340, 0x5AA7 }, { 2341, 0x5AAC }, { 2342, 0x5AB1 }, { 2343, 0x5AB2 },
    { 2344, 0x5AB3 }, { 2345, 0x5AB5 }, { 2346, 0x5AB8 }, { 2347, 0x5ABA }, { 2348, 0x5ABB }, { 2349, 0x5ABF }, { 2350, 0x5AC4 }, { 2351, 0x5AC6 },
    { 2352, 0x5AC8 }, { 2353, 0x5ACF }, { 2354, 0x5ADA }, { 2355, 0x5ADC }, { 2356, 0x5AE0 }, { 2357, 0x5AE5 }, { 2358, 0x5AEA }, { 2359, 0x5AEE },
    { 2360, 0x5AF5 }, { 2361, 0x5AF6 }, { 2362, 0x5AFD }, { 2363, 0x5B00 }, { 2364, 0x5B01 }, { 2365, 0x5B08 }, { 2366, 0x5B17 }, { 2367, 0x5B34 },
    { 2368, 0x5B19 }, { 2369, 0x5B1B }, { 2370, 0x5B1D }, { 2371, 0x5B21 }, { 2372, 0x5B25 }, { 2373, 0x5B2D }, { 2374, 0x5B38 }, { 2375, 0x5B41 },
    { 2376, 0x5B4B }, { 2377, 0x5B4C }, { 2378, 0x5B52 }, { 2379, 0x5B56 }, { 2380, 0x5B5E }, { 2381, 0x5B68 }, { 2382, 0x5B6E }, { 2383, 0x5B6F },
    { 2384, 0x5B7C }, { 2385, 0x5B7D }, { 2386, 0x5B7E }, { 2387, 0x5B7F }, { 2388, 0x5B81 }, { 2389, 0x5B84 }, { 2390, 0x5B86 }, { 2391, 0x5B8A },
    { 2392, 0x5B8E }, { 2393, 0x5B90 }, { 2394, 0x5B91 }, { 2395, 0x5B93 }, { 2396, 0x5B94 }, { 2397, 0x5B96 }, { 2398, 0x5BA8 }, { 2399, 0x5BA9 },
    { 2400, 0x5BAC }, { 2401, 0x5BAD }, { 2402, 0x5BAF }, { 2403, 0x5BB1 }, { 2404, 0x5BB2 }, { 2405, 0x5BB7 }, { 2406, 0x5BBA }, { 2407, 0x5BBC },
    { 2408, 0x5BC0 }, { 2409, 0x5BC1 }, { 2410, 0x5BCD }, { 2411, 0x5BCF }, { 2412, 0x5BD6 }, { 2413, 0x5BD7 }, { 2414, 0x5BD8 }, { 2415, 0x5BD9 },
    { 2416, 0x5BDA }, { 2417, 0x5BE0 }, { 2418, 0x5BEF }, { 2419, 0x5BF1 }, { 2420, 0x5BF4 }, { 2421, 0x5BFD }, { 2422, 0x5C0C }, { 2423, 0x5C17 },
    { 2424, 0x5C1E }, { 2425, 0x5C1F }, { 2426, 0x5C23 }, { 2427, 0x5C26 }, { 2428, 0x5C29 }, { 2429, 0x5C2B }, { 2430, 0x5C2C }, { 2431, 0x5C2E },
    { 2432, 0x5C30 }, { 2433, 0x5C32 }, { 2434, 0x5C35 }, { 2435, 0x5C36 }, { 2436, 0x5C59 }, { 2437, 0x5C5A }, { 2438, 0x5C5C }, { 2439, 0x5C62 },
    { 2440, 0x5C63 }, { 2441, 0x5C67 }, { 2442, 0x5C68 }, { 2443, 0x5C69 }, { 2444, 0x5C6D }, { 2445, 0x5C70 }, { 2446, 0x5C74 }, { 2447, 0x5C75 },
    { 2448, 0x5C7A }, { 2449, 0x5C7B }, { 2450, 0x5C7C }, { 2451, 0x5C7D }, { 2452, 0x5C87 }, { 2453, 0x5C88 }, { 2454, 0x5C8A }, { 2455, 0x5C8F },
    { 2456, 0x5C92 }, { 2457, 0x5C9D }, { 2458, 0x5C9F }, { 2459, 0x5CA0 }, { 2460, 0x5CA2 }, { 2461, 0x5CA3 }, { 2462, 0x5CA6 }, { 2463, 0x5CAA },
    { 2464, 0x5CB2 }, { 2465, 0x5CB4 }, { 2466, 0x5CB5 }, { 2467, 0x5CBA }, { 2468, 0x5CC9 }, { 2469, 0x5CCB }, { 2470, 0x5CD2 }, { 2471, 0x5CDD },
    { 2472, 0x5CD7 }, { 2473, 0x5CEE }, { 2474, 0x5CF1 }, { 2475, 0x5CF2 }, { 2476, 0x5CF4 }, { 2477, 0x5D01 }, { 2478, 0x5D06 }, { 2479, 0x5D0D },
    { 2480, 0x5D12 }, { 2481, 0x5D2B }, { 2482, 0x5D23 }, { 2483, 0x5D24 }, { 2484, 0x5D26 }, { 2485, 0x5D27 }, { 2486, 0x5D31 }, { 2487, 0x5D34 },
    { 2488, 0x5D39 }, { 2489, 0x5D3D }, { 2490, 0x5D3F }, { 2491, 0x5D42 }, { 2492, 0x5D43 }, { 2493, 0x5D46 }, { 2494, 0x5D48 }, { 2495, 0x5D55 },
    { 2496, 0x5D51 }, { 2497, 0x5D59 }, { 2498, 0x5D4A }, { 2499, 0x5D5F }, { 2500, 0x5D60 }, { 2501, 0x5D61 }, { 2502, 0x5D62 }, { 2503, 0x5D64 },
    { 2504, 0x5D6A }, { 2505, 0x5D6D }, { 2506, 0x5D70 }, { 2507, 0x5D79 }, { 2508, 0x5D7A }, { 2509, 0x5D7E }, { 2510, 0x5D7F }, { 2511, 0x5D81 },
    { 2512, 0x5D83 }, { 2513, 0x5D88 }, { 2514, 0x5D8A }, { 2515, 0x5D92 }, { 2516, 0x5D93 }, { 2517, 0x5D94 }, { 2518, 0x5D95 }, { 2519, 0x5D99 },
    { 2520, 0x5D9B }, { 2521, 0x5D9F }, { 2522, 0x5DA0 }, { 2523, 0x5DA7 }, { 2524, 0x5DAB }, { 2525, 0x5DB0 }, { 2526, 0x5DB4 }, { 2527, 0x5DB8 },
    { 2528, 0x5DB9 }, { 2529, 0x5DC3 }, { 2530, 0x5DC7 }, { 2531, 0x5DCB }, { 2532, 0x5DD0 }, { 2533, 0x5DCE }, { 2534, 0x5DD8 }, { 2535, 0x5DD9 },
    { 2536, 0x5DE0 }, { 2537, 0x5DE4 }, { 2538, 0x5DE9 }, { 2539, 0x5DF8 }, { 2540, 0x5DF9 }, { 2541, 0x5E00 }, { 2542, 0x5E07 }, { 2543, 0x5E0D },
    { 2544, 0x5E12 }, { 2545, 0x5E14 }, { 2546, 0x5E15 }, { 2547, 0x5E18 }, { 2548, 0x5E1F }, { 2549, 0x5E20 }, { 2550, 0x5E2E }, { 2551, 0x5E28 },
    { 2552, 0x5E32 }, { 2553, 0x5E35 }, { 2554, 0x5E3E }, { 2555, 0x5E4B }, { 2556, 0x5E50 }, { 2557, 0x5E49 }, { 2558, 0x5E51 }, { 2559, 0x5E56 },
    { 2560, 0x5E58 }, { 2561, 0x5E5B }, { 2562, 0x5E5C }, { 2563, 0x5E5E }, { 2564, 0x5E68 }, { 2565, 0x5E6A }, { 2566, 0x5E6B }, { 2567, 0x5E6C },
    { 2568, 0x5E6D }, { 2569, 0x5E6E }, { 2570, 0x5E70 }, { 2571, 0x5E80 }, { 2572, 0x5E8B }, { 2573, 0x5E8E }, { 2574, 0x5EA2 }, { 2575, 0x5EA4 },
    { 2576, 0x5EA5 }, { 2577, 0x5EA8 }, { 2578, 0x5EAA }, { 2579, 0x5EAC }, { 2580, 0x5EB1 }, { 2581, 0x5EB3 }, { 2582, 0x5EBD }, { 2583, 0x5EBE },
    { 2584, 0x5EBF }, { 2585, 0x5EC6 }, { 2586, 0x5ECC }, { 2587, 0x5ECB }, { 2588, 0x5ECE }, { 2589, 0x5ED1 }, { 2590, 0x5ED2 }, { 2591, 0x5ED4 },
    { 2592, 0x5ED5 }, { 2593, 0x5EDC }, { 2594, 0x5EDE }, { 2595, 0x5EE5 }, { 2596, 0x5EEB }, { 2597, 0x5F02 }, { 2598, 0x5F06 }, { 2599, 0x5F07 },
    { 2600, 0x5F08 }, { 2601, 0x5F0E }, { 2602, 0x5F19 }, { 2603, 0x5F1C }, { 2604, 0x5F1D }, { 2605, 0x5F21 }, { 2606, 0x5F22 }, { 2607, 0x5F23 },
    { 2608, 0x5F24 }, { 2609, 0x5F28 }, { 2610, 0x5F2B }, { 2611, 0x5F2C }, { 2612, 0x5F2E }, { 2613, 0x5F30 }, { 2614, 0x5F34 }, { 2615, 0x5F36 },
    { 2616, 0x5F3B }, { 2617, 0x5F3D }, { 2618, 0x5F3F }, { 2619, 0x5F40 }, { 2620, 0x5F44 }, { 2621, 0x5F45 }, { 2622, 0x5F47 }, { 2623, 0x5F4D },
    { 2624, 0x5F50 }, { 2625, 0x5F54 }, { 2626, 0x5F58 }, { 2627, 0x5F5B }, { 2628, 0x5F60 }, { 2629, 0x5F63 }, { 2630, 0x5F64 }, { 2631, 0x5F67 },
    { 2632, 0x5F6F }, { 2633, 0x5F72 }, { 2634, 0x5F74 }, { 2635, 0x5F75 }, { 2636, 0x5F78 }, { 2637, 0x5F7A }, { 2638, 0x5F7D }, { 2639, 0x5F7E },
    { 2640, 0x5F89 }, { 2641, 0x5F8D }, { 2642, 0x5F8F }, { 2643, 0x5F96 }, { 2644, 0x5F9C }, { 2645, 0x5F9D }, { 2646, 0x5FA2 }, { 2647, 0x5FA7 },
    { 2648, 0x5FAB }, { 2649, 0x5FA4 }, { 2650, 0x5FAC }, { 2651, 0x5FAF }, { 2652, 0x5FB0 }, { 2653, 0x5FB1 }, { 2654, 0x5FB8 }, { 2655, 0x5FC4 },
    { 2656, 0x5FC7 }, { 2657, 0x5FC8 }, { 2658, 0x5FC9 }, { 2659, 0x5FCB }, { 2660, 0x5FD0 }, { 2661, 0x5FD1 }, { 2662, 0x5FD2 }, { 2663, 0x5FD3 },
    { 2664, 0x5FD4 }, { 2665, 0x5FDE }, { 2666, 0x5FE1 }, { 2667, 0x5FE2 }, { 2668, 0x5FE8 }, { 2669, 0x5FE9 }, { 2670, 0x5FEA }, { 2671, 0x5FEC },
    { 2672, 0x5FED }, { 2673, 0x5FEE }, { 2674, 0x5FEF }, { 2675, 0x5FF2 }, { 2676, 0x5FF3 }, { 2677, 0x5FF6 }, { 2678, 0x5FFA }, { 2679, 0x5FFC },
    { 2680, 0x6007 }, { 2681, 0x600A }, { 2682, 0x600D }, { 2683, 0x6013 }, { 2684, 0x6014 }, { 2685, 0x6017 }, { 2686, 0x6018 }, { 2687, 0x601A },
    { 2688, 0x601F }, { 2689, 0x6024 }, { 2690, 0x602D }, { 2691, 0x6033 }, { 2692, 0x6035 }, { 2693, 0x6040 }, { 2694, 0x6047 }, { 2695, 0x6048 },
    { 2696, 0x6049 }, { 2697, 0x604C }, { 2698, 0x6051 }, { 2699, 0x6054 }, { 2700, 0x6056 }, { 2701, 0x6057 }, { 2702, 0x605D }, { 2703, 0x6061 },
    { 2704, 0x6067 }, { 2705, 0x6071 }, { 2706, 0x607E }, { 2707, 0x607F }, { 2708, 0x6082 }, { 2709, 0x6086 }, { 2710, 0x6088 }, { 2711, 0x608A },
    { 2712, 0x608E }, { 2713, 0x6091 }, { 2714, 0x6093 }, { 2715, 0x6095 }, { 2716, 0x6098 }, { 2717, 0x609D }, { 2718, 0x609E }, { 2719, 0x60A2 },
    { 2720, 0x60A4 }, { 2721, 0x60A5 }, { 2722, 0x60A8 }, { 2723, 0x60B0 }, { 2724, 0x60B1 }, { 2725, 0x60B7 }, { 2726, 0x60BB }, { 2727, 0x60BE },
    { 2728, 0x60C2 }, { 2729, 0x60C4 }, { 2730, 0x60C8 }, { 2731, 0x60C9 }, { 2732, 0x60CA }, { 2733, 0x60CB }, { 2734, 0x60CE }, { 2735, 0x60CF },
    { 2736, 0x60D4 }, { 2737, 0x60D5 }, { 2738, 0x60D9 }, { 2739, 0x60DB }, { 2740, 0x60DD }, { 2741, 0x60DE }, { 2742, 0x60E2 }, { 2743, 0x60E5 },
    { 2744, 0x60F2 }, { 2745, 0x60F5 }, { 2746, 0x60F8 }, { 2747, 0x60FC }, { 2748, 0x60FD }, { 2749, 0x6102 }, { 2750, 0x6107 }, { 2751, 0x610A },
    { 2752, 0x610C }, { 2753, 0x6110 }, { 2754, 0x6111 }, { 2755, 0x6112 }, { 2756, 0x6113 }, { 2757, 0x6114 }, { 2758, 0x6116 }, { 2759, 0x6117 },
    { 2760, 0x6119 }, { 2761, 0x611C }, { 2762, 0x611E }, { 2763, 0x6122 }, { 2764, 0x612A }, { 2765, 0x612B }, { 2766, 0x6130 }, { 2767, 0x6131 },
    { 2768, 0x6135 }, { 2769, 0x6136 }, { 2770, 0x6137 }, { 2771, 0x6139 }, { 2772, 0x6141 }, { 2773, 0x6145 }, { 2774, 0x6146 }, { 2775, 0x6149 },
    { 2776, 0x615E }, { 2777, 0x6160 }, { 2778, 0x616C }, { 2779, 0x6172 }, { 2780, 0x6178 }, { 2781, 0x617B }, { 2782, 0x617C }, { 2783, 0x617F },
    { 2784, 0x6180 }, { 2785, 0x6181 }, { 2786, 0x6183 }, { 2787, 0x6184 }, { 2788, 0x618B }, { 2789, 0x618D }, { 2790, 0x6192 }, { 2791, 0x6193 },
    { 2792, 0x6197 }, { 2793, 0x6198 }, { 2794, 0x619C }, { 2795, 0x619D }, { 2796, 0x619F }, { 2797, 0x61A0 }, { 2798, 0x61A5 }, { 2799, 0x61A8 },
    { 2800, 0x61AA }, { 2801, 0x61AD }, { 2802, 0x61B8 }, { 2803, 0x61B9 }, { 2804, 0x61BC }, { 2805, 0x61C0 }, { 2806, 0x61C1 }, { 2807, 0x61C2 },
    { 2808, 0x61CE }, { 2809, 0x61CF }, { 2810, 0x61D5 }, { 2811, 0x61DC }, { 2812, 0x61DD }, { 2813, 0x61DE }, { 2814, 0x61DF }, { 2815, 0x61E1 },
    { 2816, 0x61E2 }, { 2817, 0x61E7 }, { 2818, 0x61E9 }, { 2819, 0x61E5 }, { 2820, 0x61EC }, { 2821, 0x61ED }, { 2822, 0x61EF }, { 2823, 0x6201 },
    { 2824, 0x6203 }, { 2825, 0x6204 }, { 2826, 0x6207 }, { 2827, 0x6213 }, { 2828, 0x6215 }, { 2829, 0x621C }, { 2830, 0x6220 }, { 2831, 0x6222 },
    { 2832, 0x6223 }, { 2833, 0x6227 }, { 2834, 0x6229 }, { 2835, 0x622B }, { 2836, 0x6239 }, { 2837, 0x623D }, { 2838, 0x6242 }, { 2839, 0x6243 },
    { 2840, 0x6244 }, { 2841, 0x6246 }, { 2842, 0x624C }, { 2843, 0x6250 }, { 2844, 0x6251 }, { 2845, 0x6252 }, { 2846, 0x6254 }, { 2847, 0x6256 },
    { 2848, 0x625A }, { 2849, 0x625C }, { 2850, 0x6264 }, { 2851, 0x626D }, { 2852, 0x626F }, { 2853, 0x6273 }, { 2854, 0x627A }, { 2855, 0x627D },
    { 2856, 0x628D }, { 2857, 0x628E }, { 2858, 0x628F }, { 2859, 0x6290 }, { 2860, 0x62A6 }, { 2861, 0x62A8 }, { 2862, 0x62B3 }, { 2863, 0x62B6 },
    { 2864, 0x62B7 }, { 2865, 0x62BA }, { 2866, 0x62BE }, { 2867, 0x62BF }, { 2868, 0x62C4 }, { 2869, 0x62CE }, { 2870, 0x62D5 }, { 2871, 0x62D6 },
    { 2872, 0x62DA }, { 2873, 0x62EA }, { 2874, 0x62F2 }, { 2875, 0x62F4 }, { 2876, 0x62FC }, { 2877, 0x62FD }, { 2878, 0x6303 }, { 2879, 0x6304 },
    { 2880, 0x630A }, { 2881, 0x630B }, { 2882, 0x630D }, { 2883, 0x6310 }, { 2884, 0x6313 }, { 2885, 0x6316 }, { 2886, 0x6318 }, { 2887, 0x6329 },
    { 2888, 0x632A }, { 2889, 0x632D }, { 2890, 0x6335 }, { 2891, 0x6336 }, { 2892, 0x6339 }, { 2893, 0x633C }, { 2894, 0x6341 }, { 2895, 0x6342 },
    { 2896, 0x6343 }, { 2897, 0x6344 }, { 2898, 0x6346 }, { 2899, 0x634A }, { 2900, 0x634B }, { 2901, 0x634E }, { 2902, 0x6352 }, { 2903, 0x6353 },
    { 2904, 0x6354 }, { 2905, 0x6358 }, { 2906, 0x635B }, { 2907, 0x6365 }, { 2908, 0x6366 }, { 2909, 0x636C }, { 2910, 0x636D }, { 2911, 0x6371 },
    { 2912, 0x6374 }, { 2913, 0x6375 }, { 2914, 0x6378 }, { 2915, 0x637C }, { 2916, 0x637D }, { 2917, 0x637F }, { 2918, 0x6382 }, { 2919, 0x6384 },
    { 2920, 0x6387 }, { 2921, 0x638A }, { 2922, 0x6390 }, { 2923, 0x6394 }, { 2924, 0x6395 }, { 2925, 0x6399 }, { 2926, 0x639A }, { 2927, 0x639E },
    { 2928, 0x63A4 }, { 2929, 0x63A6 }, { 2930, 0x63AD }, { 2931, 0x63AE }, { 2932, 0x63AF }, { 2933, 0x63BD }, { 2934, 0x63C1 }, { 2935, 0x63C5 },
    { 2936, 0x63C8 }, { 2937, 0x63CE }, { 2938, 0x63D1 }, { 2939, 0x63D3 }, { 2940, 0x63D4 }, { 2941, 0x63D5 }, { 2942, 0x63DC }, { 2943, 0x63E0 },
    { 2944, 0x63E5 }, { 2945, 0x63EA }, { 2946, 0x63EC }, { 2947, 0x63F2 }, { 2948, 0x63F3 }, { 2949, 0x63F5 }, { 2950, 0x63F8 }, { 2951, 0x63F9 },
    { 2952, 0x6409 }, { 2953, 0x640A }, { 2954, 0x6410 }, { 2955, 0x6412 }, { 2956, 0x6414 }, { 2957, 0x6418 }, { 2958, 0x641E }, { 2959, 0x6420 },
    { 2960, 0x6422 }, { 2961, 0x6424 }, { 2962, 0x6425 }, { 2963, 0x6429 }, { 2964, 0x642A }, { 2965, 0x642F }, { 2966, 0x6430 }, { 2967, 0x6435 },
    { 2968, 0x643D }, { 2969, 0x643F }, { 2970, 0x644B }, { 2971, 0x644F }, { 2972, 0x6451 }, { 2973, 0x6452 }, { 2974, 0x6453 }, { 2975, 0x6454 },
    { 2976, 0x645A }, { 2977, 0x645B }, { 2978, 0x645C }, { 2979, 0x645D }, { 2980, 0x645F }, { 2981, 0x6460 }, { 2982, 0x6461 }, { 2983, 0x6463 },
    { 2984, 0x646D }, { 2985, 0x6473 }, { 2986, 0x6474 }, { 2987, 0x647B }, { 2988, 0x647D }, { 2989, 0x6485 }, { 2990, 0x6487 }, { 2991, 0x648F },
    { 2992, 0x6490 }, { 2993, 0x6491 }, { 2994, 0x6498 }, { 2995, 0x6499 }, { 2996, 0x649B }, { 2997, 0x649D }, { 2998, 0x649F }, { 2999, 0x64A1 },
    { 3000, 0x64A3 }, { 3001, 0x64A6 }, { 3002, 0x64A8 }, { 3003, 0x64AC }, { 3004, 0x64B3 }, { 3005, 0x64BD }, { 3006, 0x64BE }, { 3007, 0x64BF },
    { 3008, 0x64C4 }, { 3009, 0x64C9 }, { 3010, 0x64CA }, { 3011, 0x64CB }, { 3012, 0x64CC }, { 3013, 0x64CE }, { 3014, 0x64D0 }, { 3015, 0x64D1 },
    { 3016, 0x64D5 }, { 3017, 0x64D7 }, { 3018, 0x64E4 }, { 3019, 0x64E5 }, { 3020, 0x64E9 }, { 3021, 0x64EA }, { 3022, 0x64ED }, { 3023, 0x64F0 },
    { 3024, 0x64F5 }, { 3025, 0x64F7 }, { 3026, 0x64FB }, { 3027, 0x64FF }, { 3028, 0x6501 }, { 3029, 0x6504 }, { 3030, 0x6508 }, { 3031, 0x6509 },
    { 3032, 0x650A }, { 3033, 0x650F }, { 3034, 0x6513 }, { 3035, 0x6514 }, { 3036, 0x6516 }, { 3037, 0x6519 }, { 3038, 0x651B }, { 3039, 0x651E },
    { 3040, 0x651F }, { 3041, 0x6522 }, { 3042, 0x6526 }, { 3043, 0x6529 }, { 3044, 0x652E }, { 3045, 0x6531 }, { 3046, 0x653A }, { 3047, 0x653C },
    { 3048, 0x653D }, { 3049, 0x6543 }, { 3050, 0x6547 }, { 3051, 0x6549 }, { 3052, 0x6550 }, { 3053, 0x6552 }, { 3054, 0x6554 }, { 3055, 0x655F },
    { 3056, 0x6560 }, { 3057, 0x6567 }, { 3058, 0x656B }, { 3059, 0x657A }, { 3060, 0x657D }, { 3061, 0x6581 }, { 3062, 0x6585 }, { 3063, 0x658A },
    { 3064, 0x6592 }, { 3065, 0x6595 }, { 3066, 0x6598 }, { 3067, 0x659D }, { 3068, 0x65A0 }, { 3069, 0x65A3 }, { 3070, 0x65A6 }, { 3071, 0x65AE },
    { 3072, 0x65B2 }, { 3073, 0x65B3 }, { 3074, 0x65B4 }, { 3075, 0x65BF }, { 3076, 0x65C2 }, { 3077, 0x65C8 }, { 3078, 0x65C9 }, { 3079, 0x65CE },
    { 3080, 0x65D0 }, { 3081, 0x65D4 }, { 3082, 0x65D6 }, { 3083, 0x65D8 }, { 3084, 0x65DF }, { 3085, 0x65F0 }, { 3086, 0x65F2 }, { 3087, 0x65F4 },
    { 3088, 0x65F5 }, { 3089, 0x65F9 }, { 3090, 0x65FE }, { 3091, 0x65FF }, { 3092, 0x6600 }, { 3093, 0x6604 }, { 3094, 0x6608 }, { 3095, 0x6609 },
    { 3096, 0x660D }, { 3097, 0x6611 }, { 3098, 0x6612 }, { 3099, 0x6615 }, { 3100, 0x6616 }, { 3101, 0x661D }, { 3102, 0x661E }, { 3103, 0x6621 },
    { 3104, 0x6622 }, { 3105, 0x6623 }, { 3106, 0x6624 }, { 3107, 0x6626 }, { 3108, 0x6629 }, { 3109, 0x662A }, { 3110, 0x662B }, { 3111, 0x662C },
    { 3112, 0x662E }, { 3113, 0x6630 }, { 3114, 0x6631 }, { 3115, 0x6633 }, { 3116, 0x6639 }, { 3117, 0x6637 }, { 3118, 0x6640 }, { 3119, 0x6645 },
    { 3120, 0x6646 }, { 3121, 0x664A }, { 3122, 0x664C }, { 3123, 0x6651 }, { 3124, 0x664E }, { 3125, 0x6657 }, { 3126, 0x6658 }, { 3127, 0x6659 },
    { 3128, 0x665B }, { 3129, 0x665C }, { 3130, 0x6660 }, { 3131, 0x6661 }, { 3132, 0x66FB }, { 3133, 0x666A }, { 3134, 0x666B }, { 3135, 0x666C },
    { 3136, 0x667E }, { 3137, 0x6673 }, { 3138, 0x6675 }, { 3139, 0x667F }, { 3140, 0x6677 }, { 3141, 0x6678 }, { 3142, 0x6679 }, { 3143, 0x667B },
    { 3144, 0x6680 }, { 3145, 0x667C }, { 3146, 0x668B }, { 3147, 0x668C }, { 3148, 0x668D }, { 3149, 0x6690 }, { 3150, 0x6692 }, { 3151, 0x6699 },
    { 3152, 0x669A }, { 3153, 0x669B }, { 3154, 0x669C }, { 3155, 0x669F }, { 3156, 0x66A0 }, { 3157, 0x66A4 }, { 3158, 0x66AD }, { 3159, 0x66B1 },
    { 3160, 0x66B2 }, { 3161, 0x66B5 }, { 3162, 0x66BB }, { 3163, 0x66BF }, { 3164, 0x66C0 }, { 3165, 0x66C2 }, { 3166, 0x66C3 }, { 3167, 0x66C8 },
    { 3168, 0x66CC }, { 3169, 0x66CE }, { 3170, 0x66CF }, { 3171, 0x66D4 }, { 3172, 0x66DB }, { 3173, 0x66DF }, { 3174, 0x66E8 }, { 3175, 0x66EB },
    { 3176, 0x66EC }, { 3177, 0x66EE }, { 3178, 0x66FA }, { 3179, 0x6705 }, { 3180, 0x6707 }, { 3181, 0x670E }, { 3182, 0x6713 }, { 3183, 0x6719 },
    { 3184, 0x671C }, { 3185, 0x6720 }, { 3186, 0x6722 }, { 3187, 0x6733 }, { 3188, 0x673E }, { 3189, 0x6745 }, { 3190, 0x6747 }, { 3191, 0x6748 },
    { 3192, 0x674C }, { 3193, 0x6754 }, { 3194, 0x6755 }, { 3195, 0x675D }, { 3196, 0x6766 }, { 3197, 0x676C }, { 3198, 0x676E }, { 3199, 0x6774 },
    { 3200, 0x6776 }, { 3201, 0x677B }, { 3202, 0x6781 }, { 3203, 0x6784 }, { 3204, 0x678E }, { 3205, 0x678F }, { 3206, 0x6791 }, { 3207, 0x6793 },
    { 3208, 0x6796 }, { 3209, 0x6798 }, { 3210, 0x6799 }, { 3211, 0x679B }, { 3212, 0x67B0 }, { 3213, 0x67B1 }, { 3214, 0x67B2 }, { 3215, 0x67B5 },
    { 3216, 0x67BB }, { 3217, 0x67BC }, { 3218, 0x67BD }, { 3219, 0x67F9 }, { 3220, 0x67C0 }, { 3221, 0x67C2 }, { 3222, 0x67C3 }, { 3223, 0x67C5 },
    { 3224, 0x67C8 }, { 3225, 0x67C9 }, { 3226, 0x67D2 }, { 3227, 0x67D7 }, { 3228, 0x67D9 }, { 3229, 0x67DC }, { 3230, 0x67E1 }, { 3231, 0x67E6 },
    { 3232, 0x67F0 }, { 3233, 0x67F2 }, { 3234, 0x67F6 }, { 3235, 0x67F7 }, { 3236, 0x6852 }, { 3237, 0x6814 }, { 3238, 0x6819 }, { 3239, 0x681D },
    { 3240, 0x681F }, { 3241, 0x6828 }, { 3242, 0x6827 }, { 3243, 0x682C }, { 3244, 0x682D }, { 3245, 0x682F }, { 3246, 0x6830 }, { 3247, 0x6831 },
    { 3248, 0x6833 }, { 3249, 0x683B }, { 3250, 0x683F }, { 3251, 0x6844 }, { 3252, 0x6845 }, { 3253, 0x684A }, { 3254, 0x684C }, { 3255, 0x6855 },
    { 3256, 0x6857 }, { 3257, 0x6858 }, { 3258, 0x685B }, { 3259, 0x686B }, { 3260, 0x686E }, { 3261, 0x686F }, { 3262, 0x6870 }, { 3263, 0x6871 },
    { 3264, 0x6872 }, { 3265, 0x6875 }, { 3266, 0x6879 }, { 3267, 0x687A }, { 3268, 0x687B }, { 3269, 0x687C }, { 3270, 0x6882 }, { 3271, 0x6884 },
    { 3272, 0x6886 }, { 3273, 0x6888 }, { 3274, 0x6896 }, { 3275, 0x6898 }, { 3276, 0x689A }, { 3277, 0x689C }, { 3278, 0x68A1 }, { 3279, 0x68A3 },
    { 3280, 0x68A5 }, { 3281, 0x68A9 }, { 3282, 0x68AA }, { 3283, 0x68AE }, { 3284, 0x68B2 }, { 3285, 0x68BB }, { 3286, 0x68C5 }, { 3287, 0x68C8 },
    { 3288, 0x68CC }, { 3289, 0x68CF }, { 3290, 0x68D0 }, { 3291, 0x68D1 }, { 3292, 0x68D3 }, { 3293, 0x68D6 }, { 3294, 0x68D9 }, { 3295, 0x68DC },
    { 3296, 0x68DD }, { 3297, 0x68E5 }, { 3298, 0x68E8 }, { 3299, 0x68EA }, { 3300, 0x68EB }, { 3301, 0x68EC }, { 3302, 0x68ED }, { 3303, 0x68F0 },
    { 3304, 0x68F1 }, { 3305, 0x68F5 }, { 3306, 0x68F6 }, { 3307, 0x68FB }, { 3308, 0x68FC }, { 3309, 0x68FD }, { 3310, 0x6906 }, { 3311, 0x6909 },
    { 3312, 0x690A }, { 3313, 0x6910 }, { 3314, 0x6911 }, { 3315, 0x6913 }, { 3316, 0x6916 }, { 3317, 0x6917 }, { 3318, 0x6931 }, { 3319, 0x6933 },
    { 3320, 0x6935 }, { 3321, 0x6938 }, { 3322, 0x693B }, { 3323, 0x6942 }, { 3324, 0x6945 }, { 3325, 0x6949 }, { 3326, 0x694E }, { 3327, 0x6957 },
    { 3328, 0x695B }, { 3329, 0x6963 }, { 3330, 0x6964 }, { 3331, 0x6965 }, { 3332, 0x6966 }, { 3333, 0x6968 }, { 3334, 0x6969 }, { 3335, 0x696C },
    { 3336, 0x6970 }, { 3337, 0x6971 }, { 3338, 0x6972 }, { 3339, 0x697A }, { 3340, 0x697B }, { 3341, 0x697F }, { 3342, 0x6980 }, { 3343, 0x698D },
    { 3344, 0x6992 }, { 3345, 0x6996 }, { 3346, 0x6998 }, { 3347, 0x69A1 }, { 3348, 0x69A5 }, { 3349, 0x69A6 }, { 3350, 0x69A8 }, { 3351, 0x69AB },
    { 3352, 0x69AD }, { 3353, 0x69AF }, { 3354, 0x69B7 }, { 3355, 0x69B8 }, { 3356, 0x69BA }, { 3357, 0x69BC }, { 3358, 0x69C5 }, { 3359, 0x69C8 },
    { 3360, 0x69D1 }, { 3361, 0x69D6 }, { 3362, 0x69D7 }, { 3363, 0x69E2 }, { 3364, 0x69E5 }, { 3365, 0x69EE }, { 3366, 0x69EF }, { 3367, 0x69F1 },
    { 3368, 0x69F3 }, { 3369, 0x69F5 }, { 3370, 0x69FE }, { 3371, 0x6A00 }, { 3372, 0x6A01 }, { 3373, 0x6A03 }, { 3374, 0x6A0F }, { 3375, 0x6A11 },
    { 3376, 0x6A15 }, { 3377, 0x6A1A }, { 3378, 0x6A1D }, { 3379, 0x6A20 }, { 3380, 0x6A24 }, { 3381, 0x6A28 }, { 3382, 0x6A30 }, { 3383, 0x6A32 },
    { 3384, 0x6A34 }, { 3385, 0x6A37 }, { 3386, 0x6A3B }, { 3387, 0x6A3E }, { 3388, 0x6A3F }, { 3389, 0x6A45 }, { 3390, 0x6A46 }, { 3391, 0x6A49 },
    { 3392, 0x6A4A }, { 3393, 0x6A4E }, { 3394, 0x6A50 }, { 3395, 0x6A51 }, { 3396, 0x6A52 }, { 3397, 0x6A55 }, { 3398, 0x6A56 }, { 3399, 0x6A5B },
    { 3400, 0x6A64 }, { 3401, 0x6A67 }, { 3402, 0x6A6A }, { 3403, 0x6A71 }, { 3404, 0x6A73 }, { 3405, 0x6A7E }, { 3406, 0x6A81 }, { 3407, 0x6A83 },
    { 3408, 0x6A86 }, { 3409, 0x6A87 }, { 3410, 0x6A89 }, { 3411, 0x6A8B }, { 3412, 0x6A91 }, { 3413, 0x6A9B }, { 3414, 0x6A9D }, { 3415, 0x6A9E },
    { 3416, 0x6A9F }, { 3417, 0x6AA5 }, { 3418, 0x6AAB }, { 3419, 0x6AAF }, { 3420, 0x6AB0 }, { 3421, 0x6AB1 }, { 3422, 0x6AB4 }, { 3423, 0x6ABD },
    { 3424, 0x6ABE }, { 3425, 0x6ABF }, { 3426, 0x6AC6 }, { 3427, 0x6AC9 }, { 3428, 0x6AC8 }, { 3429, 0x6ACC }, { 3430, 0x6AD0 }, { 3431, 0x6AD4 },
    { 3432, 0x6AD5 }, { 3433, 0x6AD6 }, { 3434, 0x6ADC }, { 3435, 0x6ADD }, { 3436, 0x6AE4 }, { 3437, 0x6AE7 }, { 3438, 0x6AEC }, { 3439, 0x6AF0 },
    { 3440, 0x6AF1 }, { 3441, 0x6AF2 }, { 3442, 0x6AFC }, { 3443, 0x6AFD }, { 3444, 0x6B02 }, { 3445, 0x6B03 }, { 3446, 0x6B06 }, { 3447, 0x6B07 },
    { 3448, 0x6B09 }, { 3449, 0x6B0F }, { 3450, 0x6B10 }, { 3451, 0x6B11 }, { 3452, 0x6B17 }, { 3453, 0x6B1B }, { 3454, 0x6B1E }, { 3455, 0x6B24 },
    { 3456, 0x6B28 }, { 3457, 0x6B2B }, { 3458, 0x6B2C }, { 3459, 0x6B2F }, { 3460, 0x6B35 }, { 3461, 0x6B36 }, { 3462, 0x6B3B }, { 3463, 0x6B3F },
    { 3464, 0x6B46 }, { 3465, 0x6B4A }, { 3466, 0x6B4D }, { 3467, 0x6B52 }, { 3468, 0x6B56 }, { 3469, 0x6B58 }, { 3470, 0x6B5D }, { 3471, 0x6B60 },
    { 3472, 0x6B67 }, { 3473, 0x6B6B }, { 3474, 0x6B6E }, { 3475, 0x6B70 }, { 3476, 0x6B75 }, { 3477, 0x6B7D }, { 3478, 0x6B7E }, { 3479, 0x6B82 },
    { 3480, 0x6B85 }, { 3481, 0x6B97 }, { 3482, 0x6B9B }, { 3483, 0x6B9F }, { 3484, 0x6BA0 }, { 3485, 0x6BA2 }, { 3486, 0x6BA3 }, { 3487, 0x6BA8 },
    { 3488, 0x6BA9 }, { 3489, 0x6BAC }, { 3490, 0x6BAD }, { 3491, 0x6BAE }, { 3492, 0x6BB0 }, { 3493, 0x6BB8 }, { 3494, 0x6BB9 }, { 3495, 0x6BBD },
    { 3496, 0x6BBE }, { 3497, 0x6BC3 }, { 3498, 0x6BC4 }, { 3499, 0x6BC9 }, { 3500, 0x6BCC }, { 3501, 0x6BD6 }, { 3502, 0x6BDA }, { 3503, 0x6BE1 },
    { 3504, 0x6BE3 }, { 3505, 0x6BE6 }, { 3506, 0x6BE7 }, { 3507, 0x6BEE }, { 3508, 0x6BF1 }, { 3509, 0x6BF7 }, { 3510, 0x6BF9 }, { 3511, 0x6BFF },
    { 3512, 0x6C02 }, { 3513, 0x6C04 }, { 3514, 0x6C05 }, { 3515, 0x6C09 }, { 3516, 0x6C0D }, { 3517, 0x6C0E }, { 3518, 0x6C10 }, { 3519, 0x6C12 },
    { 3520, 0x6C19 }, { 3521, 0x6C1F }, { 3522, 0x6C26 }, { 3523, 0x6C27 }, { 3524, 0x6C28 }, { 3525, 0x6C2C }, { 3526, 0x6C2E }, { 3527, 0x6C33 },
    { 3528, 0x6C35 }, { 3529, 0x6C36 }, { 3530, 0x6C3A }, { 3531, 0x6C3B }, { 3532, 0x6C3F }, { 3533, 0x6C4A }, { 3534, 0x6C4B }, { 3535, 0x6C4D },
    { 3536, 0x6C4F }, { 3537, 0x6C52 }, { 3538, 0x6C54 }, { 3539, 0x6C59 }, { 3540, 0x6C5B }, { 3541, 0x6C5C }, { 3542, 0x6C6B }, { 3543, 0x6C6D },
    { 3544, 0x6C6F }, { 3545, 0x6C74 }, { 3546, 0x6C76 }, { 3547, 0x6C78 }, { 3548, 0x6C79 }, { 3549, 0x6C7B }, { 3550, 0x6C85 }, { 3551, 0x6C86 },
    { 3552, 0x6C87 }, { 3553, 0x6C89 }, { 3554, 0x6C94 }, { 3555, 0x6C95 }, { 3556, 0x6C97 }, { 3557, 0x6C98 }, { 3558, 0x6C9C }, { 3559, 0x6C9F },
    { 3560, 0x6CB0 }, { 3561, 0x6CB2 }, { 3562, 0x6CB4 }, { 3563, 0x6CC2 }, { 3564, 0x6CC6 }, { 3565, 0x6CCD }, { 3566, 0x6CCF }, { 3567, 0x6CD0 },
    { 3568, 0x6CD1 }, { 3569, 0x6CD2 }, { 3570, 0x6CD4 }, { 3571, 0x6CD6 }, { 3572, 0x6CDA }, { 3573, 0x6CDC }, { 3574, 0x6CE0 }, { 3575, 0x6CE7 },
    { 3576, 0x6CE9 }, { 3577, 0x6CEB }, { 3578, 0x6CEC }, { 3579, 0x6CEE }, { 3580, 0x6CF2 }, { 3581, 0x6CF4 }, { 3582, 0x6D04 }, { 3583, 0x6D07 },
    { 3584, 0x6D0A }, { 3585, 0x6D0E }, { 3586, 0x6D0F }, { 3587, 0x6D11 }, { 3588, 0x6D13 }, { 3589, 0x6D1A }, { 3590, 0x6D26 }, { 3591, 0x6D27 },
    { 3592, 0x6D28 }, { 3593, 0x6C67 }, { 3594, 0x6D2E }, { 3595, 0x6D2F }, { 3596, 0x6D31 }, { 3597, 0x6D39 }, { 3598, 0x6D3C }, { 3599, 0x6D3F },
    { 3600, 0x6D57 }, { 3601, 0x6D5E }, { 3602, 0x6D5F }, { 3603, 0x6D61 }, { 3604, 0x6D65 }, { 3605, 0x6D67 }, { 3606, 0x6D6F }, { 3607, 0x6D70 },
    { 3608, 0x6D7C }, { 3609, 0x6D82 }, { 3610, 0x6D87 }, { 3611, 0x6D91 }, { 3612, 0x6D92 }, { 3613, 0x6D94 }, { 3614, 0x6D96 }, { 3615, 0x6D97 },
    { 3616, 0x6D98 }, { 3617, 0x6DAA }, { 3618, 0x6DAC }, { 3619, 0x6DB4 }, { 3620, 0x6DB7 }, { 3621, 0x6DB9 }, { 3622, 0x6DBD }, { 3623, 0x6DBF },
    { 3624, 0x6DC4 }, { 3625, 0x6DC8 }, { 3626, 0x6DCA }, { 3627, 0x6DCE }, { 3628, 0x6DCF }, { 3629, 0x6DD6 }, { 3630, 0x6DDB }, { 3631, 0x6DDD },
    { 3632, 0x6DDF }, { 3633, 0x6DE0 }, { 3634, 0x6DE2 }, { 3635, 0x6DE5 }, { 3636, 0x6DE9 }, { 3637, 0x6DEF }, { 3638, 0x6DF0 }, { 3639, 0x6DF4 },
    { 3640, 0x6DF6 }, { 3641, 0x6DFC }, { 3642, 0x6E00 }, { 3643, 0x6E04 }, { 3644, 0x6E1E }, { 3645, 0x6E22 }, { 3646, 0x6E27 }, { 3647, 0x6E32 },
    { 3648, 0x6E36 }, { 3649, 0x6E39 }, { 3650, 0x6E3B }, { 3651, 0x6E3C }, { 3652, 0x6E44 }, { 3653, 0x6E45 }, { 3654, 0x6E48 }, { 3655, 0x6E49 },
    { 3656, 0x6E4B }, { 3657, 0x6E4F }, { 3658, 0x6E51 }, { 3659, 0x6E52 }, { 3660, 0x6E53 }, { 3661, 0x6E54 }, { 3662, 0x6E57 }, { 3663, 0x6E5C },
    { 3664, 0x6E5D }, { 3665, 0x6E5E }, { 3666, 0x6E62 }, { 3667, 0x6E63 }, { 3668, 0x6E68 }, { 3669, 0x6E73 }, { 3670, 0x6E7B }, { 3671, 0x6E7D },
    { 3672, 0x6E8D }, { 3673, 0x6E93 }, { 3674, 0x6E99 }, { 3675, 0x6EA0 }, { 3676, 0x6EA7 }, { 3677, 0x6EAD }, { 3678, 0x6EAE }, { 3679, 0x6EB1 },
    { 3680, 0x6EB3 }, { 3681, 0x6EBB }, { 3682, 0x6EBF }, { 3683, 0x6EC0 }, { 3684, 0x6EC1 }, { 3685, 0x6EC3 }, { 3686, 0x6EC7 }, { 3687, 0x6EC8 },
    { 3688, 0x6ECA }, { 3689, 0x6ECD }, { 3690, 0x6ECE }, { 3691, 0x6ECF }, { 3692, 0x6EEB }, { 3693, 0x6EED }, { 3694, 0x6EEE }, { 3695, 0x6EF9 },
    { 3696, 0x6EFB }, { 3697, 0x6EFD }, { 3698, 0x6F04 }, { 3699, 0x6F08 }, { 3700, 0x6F0A }, { 3701, 0x6F0C }, { 3702, 0x6F0D }, { 3703, 0x6F16 },
    { 3704, 0x6F18 }, { 3705, 0x6F1A }, { 3706, 0x6F1B }, { 3707, 0x6F26 }, { 3708, 0x6F29 }, { 3709, 0x6F2A }, { 3710, 0x6F2F }, { 3711, 0x6F30 },
    { 3712, 0x6F33 }, { 3713, 0x6F36 }, { 3714, 0x6F3B }, { 3715, 0x6F3C }, { 3716, 0x6F2D }, { 3717, 0x6F4F }, { 3718, 0x6F51 }, { 3719, 0x6F52 },
    { 3720, 0x6F53 }, { 3721, 0x6F57 }, { 3722, 0x6F59 }, { 3723, 0x6F5A }, { 3724, 0x6F5D }, { 3725, 0x6F5E }, { 3726, 0x6F61 }, { 3727, 0x6F62 },
    { 3728, 0x6F68 }, { 3729, 0x6F6C }, { 3730, 0x6F7D }, { 3731, 0x6F7E }, { 3732, 0x6F83 }, { 3733, 0x6F87 }, { 3734, 0x6F88 }, { 3735, 0x6F8B },
    { 3736, 0x6F8C }, { 3737, 0x6F8D }, { 3738, 0x6F90 }, { 3739, 0x6F92 }, { 3740, 0x6F93 }, { 3741, 0x6F94 }, { 3742, 0x6F96 }, { 3743, 0x6F9A },
    { 3744, 0x6F9F }, { 3745, 0x6FA0 }, { 3746, 0x6FA5 }, { 3747, 0x6FA6 }, { 3748, 0x6FA7 }, { 3749, 0x6FA8 }, { 3750, 0x6FAE }, { 3751, 0x6FAF },
    { 3752, 0x6FB0 }, { 3753, 0x6FB5 }, { 3754, 0x6FB6 }, { 3755, 0x6FBC }, { 3756, 0x6FC5 }, { 3757, 0x6FC7 }, { 3758, 0x6FC8 }, { 3759, 0x6FCA },
    { 3760, 0x6FDA }, { 3761, 0x6FDE }, { 3762, 0x6FE8 }, { 3763, 0x6FE9 }, { 3764, 0x6FF0 }, { 3765, 0x6FF5 }, { 3766, 0x6FF9 }, { 3767, 0x6FFC },
    { 3768, 0x6FFD }, { 3769, 0x7000 }, { 3770, 0x7005 }, { 3771, 0x7006 }, { 3772, 0x7007 }, { 3773, 0x700D }, { 3774, 0x7017 }, { 3775, 0x7020 },
    { 3776, 0x7023 }, { 3777, 0x702F }, { 3778, 0x7034 }, { 3779, 0x7037 }, { 3780, 0x7039 }, { 3781, 0x703C }, { 3782, 0x7043 }, { 3783, 0x7044 },
    { 3784, 0x7048 }, { 3785, 0x7049 }, { 3786, 0x704A }, { 3787, 0x704B }, { 3788, 0x7054 }, { 3789, 0x7055 }, { 3790, 0x705D }, { 3791, 0x705E },
    { 3792, 0x704E }, { 3793, 0x7064 }, { 3794, 0x7065 }, { 3795, 0x706C }, { 3796, 0x706E }, { 3797, 0x7075 }, { 3798, 0x7076 }, { 3799, 0x707E },
    { 3800, 0x7081 }, { 3801, 0x7085 }, { 3802, 0x7086 }, { 3803, 0x7094 }, { 3804, 0x7095 }, { 3805, 0x7096 }, { 3806, 0x7097 }, { 3807, 0x7098 },
    { 3808, 0x709B }, { 3809, 0x70A4 }, { 3810, 0x70AB }, { 3811, 0x70B0 }, { 3812, 0x70B1 }, { 3813, 0x70B4 }, { 3814, 0x70B7 }, { 3815, 0x70CA },
    { 3816, 0x70D1 }, { 3817, 0x70D3 }, { 3818, 0x70D4 }, { 3819, 0x70D5 }, { 3820, 0x70D6 }, { 3821, 0x70D8 }, { 3822, 0x70DC }, { 3823, 0x70E4 },
    { 3824, 0x70FA }, { 3825, 0x7103 }, { 3826, 0x7104 }, { 3827, 0x7105 }, { 3828, 0x7106 }, { 3829, 0x7107 }, { 3830, 0x710B }, { 3831, 0x710C },
    { 3832, 0x710F }, { 3833, 0x711E }, { 3834, 0x7120 }, { 3835, 0x712B }, { 3836, 0x712D }, { 3837, 0x712F }, { 3838, 0x7130 }, { 3839, 0x7131 },
    { 3840, 0x7138 }, { 3841, 0x7141 }, { 3842, 0x7145 }, { 3843, 0x7146 }, { 3844, 0x7147 }, { 3845, 0x714A }, { 3846, 0x714B }, { 3847, 0x7150 },
    { 3848, 0x7152 }, { 3849, 0x7157 }, { 3850, 0x715A }, { 3851, 0x715C }, { 3852, 0x715E }, { 3853, 0x7160 }, { 3854, 0x7168 }, { 3855, 0x7179 },
    { 3856, 0x7180 }, { 3857, 0x7185 }, { 3858, 0x7187 }, { 3859, 0x718C }, { 3860, 0x7192 }, { 3861, 0x719A }, { 3862, 0x719B }, { 3863, 0x71A0 },
    { 3864, 0x71A2 }, { 3865, 0x71AF }, { 3866, 0x71B0 }, { 3867, 0x71B2 }, { 3868, 0x71B3 }, { 3869, 0x71BA }, { 3870, 0x71BF }, { 3871, 0x71C0 },
    { 3872, 0x71C1 }, { 3873, 0x71C4 }, { 3874, 0x71CB }, { 3875, 0x71CC }, { 3876, 0x71D3 }, { 3877, 0x71D6 }, { 3878, 0x71D9 }, { 3879, 0x71DA },
    { 3880, 0x71DC }, { 3881, 0x71F8 }, { 3882, 0x71FE }, { 3883, 0x7200 }, { 3884, 0x7207 }, { 3885, 0x7208 }, { 3886, 0x7209 }, { 3887, 0x7213 },
    { 3888, 0x7217 }, { 3889, 0x721A }, { 3890, 0x721D }, { 3891, 0x721F }, { 3892, 0x7224 }, { 3893, 0x722B }, { 3894, 0x722F }, { 3895, 0x7234 },
    { 3896, 0x7238 }, { 3897, 0x7239 }, { 3898, 0x7241 }, { 3899, 0x7242 }, { 3900, 0x7243 }, { 3901, 0x7245 }, { 3902, 0x724E }, { 3903, 0x724F },
    { 3904, 0x7250 }, { 3905, 0x7253 }, { 3906, 0x7255 }, { 3907, 0x7256 }, { 3908, 0x725A }, { 3909, 0x725C }, { 3910, 0x725E }, { 3911, 0x7260 },
    { 3912, 0x7263 }, { 3913, 0x7268 }, { 3914, 0x726B }, { 3915, 0x726E }, { 3916, 0x726F }, { 3917, 0x7271 }, { 3918, 0x7277 }, { 3919, 0x7278 },
    { 3920, 0x727B }, { 3921, 0x727C }, { 3922, 0x727F }, { 3923, 0x7284 }, { 3924, 0x7289 }, { 3925, 0x728D }, { 3926, 0x728E }, { 3927, 0x7293 },
    { 3928, 0x729B }, { 3929, 0x72A8 }, { 3930, 0x72AD }, { 3931, 0x72AE }, { 3932, 0x72B1 }, { 3933, 0x72B4 }, { 3934, 0x72BE }, { 3935, 0x72C1 },
    { 3936, 0x72C7 }, { 3937, 0x72C9 }, { 3938, 0x72CC }, { 3939, 0x72D5 }, { 3940, 0x72D6 }, { 3941, 0x72D8 }, { 3942, 0x72DF }, { 3943, 0x72E5 },
    { 3944, 0x72F3 }, { 3945, 0x72F4 }, { 3946, 0x72FA }, { 3947, 0x72FB }, { 3948, 0x72FE }, { 3949, 0x7302 }, { 3950, 0x7304 }, { 3951, 0x7305 },
    { 3952, 0x7307 }, { 3953, 0x730B }, { 3954, 0x730D }, { 3955, 0x7312 }, { 3956, 0x7313 }, { 3957, 0x7318 }, { 3958, 0x7319 }, { 3959, 0x731E },
    { 3960, 0x7322 }, { 3961, 0x7324 }, { 3962, 0x7327 }, { 3963, 0x7328 }, { 3964, 0x732C }, { 3965, 0x7331 }, { 3966, 0x7332 }, { 3967, 0x7335 },
    { 3968, 0x733A }, { 3969, 0x733B }, { 3970, 0x733D }, { 3971, 0x7343 }, { 3972, 0x734D }, { 3973, 0x7350 }, { 3974, 0x7352 }, { 3975, 0x7356 },
    { 3976, 0x7358 }, { 3977, 0x735D }, { 3978, 0x735E }, { 3979, 0x735F }, { 3980, 0x7360 }, { 3981, 0x7366 }, { 3982, 0x7367 }, { 3983, 0x7369 },
    { 3984, 0x736B }, { 3985, 0x736C }, { 3986, 0x736E }, { 3987, 0x736F }, { 3988, 0x7371 }, { 3989, 0x7377 }, { 3990, 0x7379 }, { 3991, 0x737C },
    { 3992, 0x7380 }, { 3993, 0x7381 }, { 3994, 0x7383 }, { 3995, 0x7385 }, { 3996, 0x7386 }, { 3997, 0x738E }, { 3998, 0x7390 }, { 3999, 0x7393 },
    { 4000, 0x7395 }, { 4001, 0x7397 }, { 4002, 0x7398 }, { 4003, 0x739C }, { 4004, 0x739E }, { 4005, 0x739F }, { 4006, 0x73A0 }, { 4007, 0x73A2 },
    { 4008, 0x73A5 }, { 4009, 0x73A6 }, { 4010, 0x73AA }, { 4011, 0x73AB }, { 4012, 0x73AD }, { 4013, 0x73B5 }, { 4014, 0x73B7 }, { 4015, 0x73B9 },
    { 4016, 0x73BC }, { 4017, 0x73BD }, { 4018, 0x73BF }, { 4019, 0x73C5 }, { 4020, 0x73C6 }, { 4021, 0x73C9 }, { 4022, 0x73CB }, { 4023, 0x73CC },
    { 4024, 0x73CF }, { 4025, 0x73D2 }, { 4026, 0x73D3 }, { 4027, 0x73D6 }, { 4028, 0x73D9 }, { 4029, 0x73DD }, { 4030, 0x73E1 }, { 4031, 0x73E3 },
    { 4032, 0x73E6 }, { 4033, 0x73E7 }, { 4034, 0x73E9 }, { 4035, 0x73F4 }, { 4036, 0x73F5 }, { 4037, 0x73F7 }, { 4038, 0x73F9 }, { 4039, 0x73FA },
    { 4040, 0x73FB }, { 4041, 0x73FD }, { 4042, 0x73FF }, { 4043, 0x7400 }, { 4044, 0x7401 }, { 4045, 0x7404 }, { 4046, 0x7407 }, { 4047, 0x740A },
    { 4048, 0x7411 }, { 4049, 0x741A }, { 4050, 0x741B }, { 4051, 0x7424 }, { 4052, 0x7426 }, { 4053, 0x7428 }, { 4054, 0x7429 }, { 4055, 0x742A },
    { 4056, 0x742B }, { 4057, 0x742C }, { 4058, 0x742D }, { 4059, 0x742E }, { 4060, 0x742F }, { 4061, 0x7430 }, { 4062, 0x7431 }, { 4063, 0x7439 },
    { 4064, 0x7440 }, { 4065, 0x7443 }, { 4066, 0x7444 }, { 4067, 0x7446 }, { 4068, 0x7447 }, { 4069, 0x744B }, { 4070, 0x744D }, { 4071, 0x7451 },
    { 4072, 0x7452 }, { 4073, 0x7457 }, { 4074, 0x745D }, { 4075, 0x7462 }, { 4076, 0x7466 }, { 4077, 0x7467 }, { 4078, 0x7468 }, { 4079, 0x746B },
    { 4080, 0x746D }, { 4081, 0x746E }, { 4082, 0x7471 }, { 4083, 0x7472 }, { 4084, 0x7480 }, { 4085, 0x7481 }, { 4086, 0x7485 }, { 4087, 0x7486 },
    { 4088, 0x7487 }, { 4089, 0x7489 }, { 4090, 0x748F }, { 4091, 0x7490 }, { 4092, 0x7491 }, { 4093, 0x7492 }, { 4094, 0x7498 }, { 4095, 0x7499 },
    { 4096, 0x749A }, { 4097, 0x749C }, { 4098, 0x749F }, { 4099, 0x74A0 }, { 4100, 0x74A1 }, { 4101, 0x74A3 }, { 4102, 0x74A6 }, { 4103, 0x74A8 },
    { 4104, 0x74A9 }, { 4105, 0x74AA }, { 4106, 0x74AB }, { 4107, 0x74AE }, { 4108, 0x74AF }, { 4109, 0x74B1 }, { 4110, 0x74B2 }, { 4111, 0x74B5 },
    { 4112, 0x74B9 }, { 4113, 0x74BB }, { 4114, 0x74BF }, { 4115, 0x74C8 }, { 4116, 0x74C9 }, { 4117, 0x74CC }, { 4118, 0x74D0 }, { 4119, 0x74D3 },
    { 4120, 0x74D8 }, { 4121, 0x74DA }, { 4122, 0x74DB }, { 4123, 0x74DE }, { 4124, 0x74DF }, { 4125, 0x74E4 }, { 4126, 0x74E8 }, { 4127, 0x74EA },
    { 4128, 0x74EB }, { 4129, 0x74EF }, { 4130, 0x74F4 }, { 4131, 0x74FA }, { 4132, 0x74FB }, { 4133, 0x74FC }, { 4134, 0x74FF }, { 4135, 0x7506 },
    { 4136, 0x7512 }, { 4137, 0x7516 }, { 4138, 0x7517 }, { 4139, 0x7520 }, { 4140, 0x7521 }, { 4141, 0x7524 }, { 4142, 0x7527 }, { 4143, 0x7529 },
    { 4144, 0x752A }, { 4145, 0x752F }, { 4146, 0x7536 }, { 4147, 0x7539 }, { 4148, 0x753D }, { 4149, 0x753E }, { 4150, 0x753F }, { 4151, 0x7540 },
    { 4152, 0x7543 }, { 4153, 0x7547 }, { 4154, 0x7548 }, { 4155, 0x754E }, { 4156, 0x7550 }, { 4157, 0x7552 }, { 4158, 0x7557 }, { 4159, 0x755E },
    { 4160, 0x755F }, { 4161, 0x7561 }, { 4162, 0x756F }, { 4163, 0x7571 }, { 4164, 0x7579 }, { 4165, 0x757A }, { 4166, 0x757B }, { 4167, 0x757C },
    { 4168, 0x757D }, { 4169, 0x757E }, { 4170, 0x7581 }, { 4171, 0x7585 }, { 4172, 0x7590 }, { 4173, 0x7592 }, { 4174, 0x7593 }, { 4175, 0x7595 },
    { 4176, 0x7599 }, { 4177, 0x759C }, { 4178, 0x75A2 }, { 4179, 0x75A4 }, { 4180, 0x75B4 }, { 4181, 0x75BA }, { 4182, 0x75BF }, { 4183, 0x75C0 },
    { 4184, 0x75C1 }, { 4185, 0x75C4 }, { 4186, 0x75C6 }, { 4187, 0x75CC }, { 4188, 0x75CE }, { 4189, 0x75CF }, { 4190, 0x75D7 }, { 4191, 0x75DC },
    { 4192, 0x75DF }, { 4193, 0x75E0 }, { 4194, 0x75E1 }, { 4195, 0x75E4 }, { 4196, 0x75E7 }, { 4197, 0x75EC }, { 4198, 0x75EE }, { 4199, 0x75EF },
    { 4200, 0x75F1 }, { 4201, 0x75F9 }, { 4202, 0x7600 }, { 4203, 0x7602 }, { 4204, 0x7603 }, { 4205, 0x7604 }, { 4206, 0x7607 }, { 4207, 0x7608 },
    { 4208, 0x760A }, { 4209, 0x760C }, { 4210, 0x760F }, { 4211, 0x7612 }, { 4212, 0x7613 }, { 4213, 0x7615 }, { 4214, 0x7616 }, { 4215, 0x7619 },
    { 4216, 0x761B }, { 4217, 0x761C }, { 4218, 0x761D }, { 4219, 0x761E }, { 4220, 0x7623 }, { 4221, 0x7625 }, { 4222, 0x7626 }, { 4223, 0x7629 },
    { 4224, 0x762D }, { 4225, 0x7632 }, { 4226, 0x7633 }, { 4227, 0x7635 }, { 4228, 0x7638 }, { 4229, 0x7639 }, { 4230, 0x763A }, { 4231, 0x763C },
    { 4232, 0x764A }, { 4233, 0x7640 }, { 4234, 0x7641 }, { 4235, 0x7643 }, { 4236, 0x7644 }, { 4237, 0x7645 }, { 4238, 0x7649 }, { 4239, 0x764B },
    { 4240, 0x7655 }, { 4241, 0x7659 }, { 4242, 0x765F }, { 4243, 0x7664 }, { 4244, 0x7665 }, { 4245, 0x766D }, { 4246, 0x766E }, { 4247, 0x766F },
    { 4248, 0x7671 }, { 4249, 0x7674 }, { 4250, 0x7681 }, { 4251, 0x7685 }, { 4252, 0x768C }, { 4253, 0x768D }, { 4254, 0x7695 }, { 4255, 0x769B },
    { 4256, 0x769C }, { 4257, 0x769D }, { 4258, 0x769F }, { 4259, 0x76A0 }, { 4260, 0x76A2 }, { 4261, 0x76A3 }, { 4262, 0x76A4 }, { 4263, 0x76A5 },
    { 4264, 0x76A6 }, { 4265, 0x76A7 }, { 4266, 0x76A8 }, { 4267, 0x76AA }, { 4268, 0x76AD }, { 4269, 0x76BD }, { 4270, 0x76C1 }, { 4271, 0x76C5 },
    { 4272, 0x76C9 }, { 4273, 0x76CB }, { 4274, 0x76CC }, { 4275, 0x76CE }, { 4276, 0x76D4 }, { 4277, 0x76D9 }, { 4278, 0x76E0 }, { 4279, 0x76E6 },
    { 4280, 0x76E8 }, { 4281, 0x76EC }, { 4282, 0x76F0 }, { 4283, 0x76F1 }, { 4284, 0x76F6 }, { 4285, 0x76F9 }, { 4286, 0x76FC }, { 4287, 0x7700 },
    { 4288, 0x7706 }, { 4289, 0x770A }, { 4290, 0x770E }, { 4291, 0x7712 }, { 4292, 0x7714 }, { 4293, 0x7715 }, { 4294, 0x7717 }, { 4295, 0x7719 },
    { 4296, 0x771A }, { 4297, 0x771C }, { 4298, 0x7722 }, { 4299, 0x7728 }, { 4300, 0x772D }, { 4301, 0x772E }, { 4302, 0x772F }, { 4303, 0x7734 },
    { 4304, 0x7735 }, { 4305, 0x7736 }, { 4306, 0x7739 }, { 4307, 0x773D }, { 4308, 0x773E }, { 4309, 0x7742 }, { 4310, 0x7745 }, { 4311, 0x7746 },
    { 4312, 0x774A }, { 4313, 0x774D }, { 4314, 0x774E }, { 4315, 0x774F }, { 4316, 0x7752 }, { 4317, 0x7756 }, { 4318, 0x7757 }, { 4319, 0x775C },
    { 4320, 0x775E }, { 4321, 0x775F }, { 4322, 0x7760 }, { 4323, 0x7762 }, { 4324, 0x7764 }, { 4325, 0x7767 }, { 4326, 0x776A }, { 4327, 0x776C },
    { 4328, 0x7770 }, { 4329, 0x7772 }, { 4330, 0x7773 }, { 4331, 0x7774 }, { 4332, 0x777A }, { 4333, 0x777D }, { 4334, 0x7780 }, { 4335, 0x7784 },
    { 4336, 0x778C }, { 4337, 0x778D }, { 4338, 0x7794 }, { 4339, 0x7795 }, { 4340, 0x7796 }, { 4341, 0x779A }, { 4342, 0x779F }, { 4343, 0x77A2 },
    { 4344, 0x77A7 }, { 4345, 0x77AA }, { 4346, 0x77AE }, { 4347, 0x77AF }, { 4348, 0x77B1 }, { 4349, 0x77B5 }, { 4350, 0x77BE }, { 4351, 0x77C3 },
    { 4352, 0x77C9 }, { 4353, 0x77D1 }, { 4354, 0x77D2 }, { 4355, 0x77D5 }, { 4356, 0x77D9 }, { 4357, 0x77DE }, { 4358, 0x77DF }, { 4359, 0x77E0 },
    { 4360, 0x77E4 }, { 4361, 0x77E6 }, { 4362, 0x77EA }, { 4363, 0x77EC }, { 4364, 0x77F0 }, { 4365, 0x77F1 }, { 4366, 0x77F4 }, { 4367, 0x77F8 },
    { 4368, 0x77FB }, { 4369, 0x7805 }, { 4370, 0x7806 }, { 4371, 0x7809 }, { 4372, 0x780D }, { 4373, 0x780E }, { 4374, 0x7811 }, { 4375, 0x781D },
    { 4376, 0x7821 }, { 4377, 0x7822 }, { 4378, 0x7823 }, { 4379, 0x782D }, { 4380, 0x782E }, { 4381, 0x7830 }, { 4382, 0x7835 }, { 4383, 0x7837 },
    { 4384, 0x7843 }, { 4385, 0x7844 }, { 4386, 0x7847 }, { 4387, 0x7848 }, { 4388, 0x784C }, { 4389, 0x784E }, { 4390, 0x7852 }, { 4391, 0x785C },
    { 4392, 0x785E }, { 4393, 0x7860 }, { 4394, 0x7861 }, { 4395, 0x7863 }, { 4396, 0x7864 }, { 4397, 0x7868 }, { 4398, 0x786A }, { 4399, 0x786E },
    { 4400, 0x787A }, { 4401, 0x787E }, { 4402, 0x788A }, { 4403, 0x788F }, { 4404, 0x7894 }, { 4405, 0x7898 }, { 4406, 0x78A1 }, { 4407, 0x789D },
    { 4408, 0x789E }, { 4409, 0x789F }, { 4410, 0x78A4 }, { 4411, 0x78A8 }, { 4412, 0x78AC }, { 4413, 0x78AD }, { 4414, 0x78B0 }, { 4415, 0x78B1 },
    { 4416, 0x78B2 }, { 4417, 0x78B3 }, { 4418, 0x78BB }, { 4419, 0x78BD }, { 4420, 0x78BF }, { 4421, 0x78C7 }, { 4422, 0x78C8 }, { 4423, 0x78C9 },
    { 4424, 0x78CC }, { 4425, 0x78CE }, { 4426, 0x78D2 }, { 4427, 0x78D3 }, { 4428, 0x78D5 }, { 4429, 0x78D6 }, { 4430, 0x78E4 }, { 4431, 0x78DB },
    { 4432, 0x78DF }, { 4433, 0x78E0 }, { 4434, 0x78E1 }, { 4435, 0x78E6 }, { 4436, 0x78EA }, { 4437, 0x78F2 }, { 4438, 0x78F3 }, { 4439, 0x7900 },
    { 4440, 0x78F6 }, { 4441, 0x78F7 }, { 4442, 0x78FA }, { 4443, 0x78FB }, { 4444, 0x78FF }, { 4445, 0x7906 }, { 4446, 0x790C }, { 4447, 0x7910 },
    { 4448, 0x791A }, { 4449, 0x791C }, { 4450, 0x791E }, { 4451, 0x791F }, { 4452, 0x7920 }, { 4453, 0x7925 }, { 4454, 0x7927 }, { 4455, 0x7929 },
    { 4456, 0x792D }, { 4457, 0x7931 }, { 4458, 0x7934 }, { 4459, 0x7935 }, { 4460, 0x793B }, { 4461, 0x793D }, { 4462, 0x793F }, { 4463, 0x7944 },
    { 4464, 0x7945 }, { 4465, 0x7946 }, { 4466, 0x794A }, { 4467, 0x794B }, { 4468, 0x794F }, { 4469, 0x7951 }, { 4470, 0x7954 }, { 4471, 0x7958 },
    { 4472, 0x795B }, { 4473, 0x795C }, { 4474, 0x7967 }, { 4475, 0x7969 }, { 4476, 0x796B }, { 4477, 0x7972 }, { 4478, 0x7979 }, { 4479, 0x797B },
    { 4480, 0x797C }, { 4481, 0x797E }, { 4482, 0x798B }, { 4483, 0x798C }, { 4484, 0x7991 }, { 4485, 0x7993 }, { 4486, 0x7994 }, { 4487, 0x7995 },
    { 4488, 0x7996 }, { 4489, 0x7998 }, { 4490, 0x799B }, { 4491, 0x799C }, { 4492, 0x79A1 }, { 4493, 0x79A8 }, { 4494, 0x79A9 }, { 4495, 0x79AB },
    { 4496, 0x79AF }, { 4497, 0x79B1 }, { 4498, 0x79B4 }, { 4499, 0x79B8 }, { 4500, 0x79BB }, { 4501, 0x79C2 }, { 4502, 0x79C4 }, { 4503, 0x79C7 },
    { 4504, 0x79C8 }, { 4505, 0x79CA }, { 4506, 0x79CF }, { 4507, 0x79D4 }, { 4508, 0x79D6 }, { 4509, 0x79DA }, { 4510, 0x79DD }, { 4511, 0x79DE },
    { 4512, 0x79E0 }, { 4513, 0x79E2 }, { 4514, 0x79E5 }, { 4515, 0x79EA }, { 4516, 0x79EB }, { 4517, 0x79ED }, { 4518, 0x79F1 }, { 4519, 0x79F8 },
    { 4520, 0x79FC }, { 4521, 0x7A02 }, { 4522, 0x7A03 }, { 4523, 0x7A07 }, { 4524, 0x7A09 }, { 4525, 0x7A0A }, { 4526, 0x7A0C }, { 4527, 0x7A11 },
    { 4528, 0x7A15 }, { 4529, 0x7A1B }, { 4530, 0x7A1E }, { 4531, 0x7A21 }, { 4532, 0x7A27 }, { 4533, 0x7A2B }, { 4534, 0x7A2D }, { 4535, 0x7A2F },
    { 4536, 0x7A30 }, { 4537, 0x7A34 }, { 4538, 0x7A35 }, { 4539, 0x7A38 }, { 4540, 0x7A39 }, { 4541, 0x7A3A }, { 4542, 0x7A44 }, { 4543, 0x7A45 },
    { 4544, 0x7A47 }, { 4545, 0x7A48 }, { 4546, 0x7A4C }, { 4547, 0x7A55 }, { 4548, 0x7A56 }, { 4549, 0x7A59 }, { 4550, 0x7A5C }, { 4551, 0x7A5D },
    { 4552, 0x7A5F }, { 4553, 0x7A60 }, { 4554, 0x7A65 }, { 4555, 0x7A67 }, { 4556, 0x7A6A }, { 4557, 0x7A6D }, { 4558, 0x7A75 }, { 4559, 0x7A78 },
    { 4560, 0x7A7E }, { 4561, 0x7A80 }, { 4562, 0x7A82 }, { 4563, 0x7A85 }, { 4564, 0x7A86 }, { 4565, 0x7A8A }, { 4566, 0x7A8B }, { 4567, 0x7A90 },
    { 4568, 0x7A91 }, { 4569, 0x7A94 }, { 4570, 0x7A9E }, { 4571, 0x7AA0 }, { 4572, 0x7AA3 }, { 4573, 0x7AAC }, { 4574, 0x7AB3 }, { 4575, 0x7AB5 },
    { 4576, 0x7AB9 }, { 4577, 0x7ABB }, { 4578, 0x7ABC }, { 4579, 0x7AC6 }, { 4580, 0x7AC9 }, { 4581, 0x7ACC }, { 4582, 0x7ACE }, { 4583, 0x7AD1 },
    { 4584, 0x7ADB }, { 4585, 0x7AE8 }, { 4586, 0x7AE9 }, { 4587, 0x7AEB }, { 4588, 0x7AEC }, { 4589, 0x7AF1 }, { 4590, 0x7AF4 }, { 4591, 0x7AFB },
    { 4592, 0x7AFD }, { 4593, 0x7AFE }, { 4594, 0x7B07 }, { 4595, 0x7B14 }, { 4596, 0x7B1F }, { 4597, 0x7B23 }, { 4598, 0x7B27 }, { 4599, 0x7B29 },
    { 4600, 0x7B2A }, { 4601, 0x7B2B }, { 4602, 0x7B2D }, { 4603, 0x7B2E }, { 4604, 0x7B2F }, { 4605, 0x7B30 }, { 4606, 0x7B31 }, { 4607, 0x7B34 },
    { 4608, 0x7B3D }, { 4609, 0x7B3F }, { 4610, 0x7B40 }, { 4611, 0x7B41 }, { 4612, 0x7B47 }, { 4613, 0x7B4E }, { 4614, 0x7B55 }, { 4615, 0x7B60 },
    { 4616, 0x7B64 }, { 4617, 0x7B66 }, { 4618, 0x7B69 }, { 4619, 0x7B6A }, { 4620, 0x7B6D }, { 4621, 0x7B6F }, { 4622, 0x7B72 }, { 4623, 0x7B73 },
    { 4624, 0x7B77 }, { 4625, 0x7B84 }, { 4626, 0x7B89 }, { 4627, 0x7B8E }, { 4628, 0x7B90 }, { 4629, 0x7B91 }, { 4630, 0x7B96 }, { 4631, 0x7B9B },
    { 4632, 0x7B9E }, { 4633, 0x7BA0 }, { 4634, 0x7BA5 }, { 4635, 0x7BAC }, { 4636, 0x7BAF }, { 4637, 0x7BB0 }, { 4638, 0x7BB2 }, { 4639, 0x7BB5 },
    { 4640, 0x7BB6 }, { 4641, 0x7BBA }, { 4642, 0x7BBB }, { 4643, 0x7BBC }, { 4644, 0x7BBD }, { 4645, 0x7BC2 }, { 4646, 0x7BC5 }, { 4647, 0x7BC8 },
    { 4648, 0x7BCA }, { 4649, 0x7BD4 }, { 4650, 0x7BD6 }, { 4651, 0x7BD7 }, { 4652, 0x7BD9 }, { 4653, 0x7BDA }, { 4654, 0x7BDB }, { 4655, 0x7BE8 },
    { 4656, 0x7BEA }, { 4657, 0x7BF2 }, { 4658, 0x7BF4 }, { 4659, 0x7BF5 }, { 4660, 0x7BF8 }, { 4661, 0x7BF9 }, { 4662, 0x7BFA }, { 4663, 0x7BFC },
    { 4664, 0x7BFE }, { 4665, 0x7C01 }, { 4666, 0x7C02 }, { 4667, 0x7C03 }, { 4668, 0x7C04 }, { 4669, 0x7C06 }, { 4670, 0x7C09 }, { 4671, 0x7C0B },
    { 4672, 0x7C0C }, { 4673, 0x7C0E }, { 4674, 0x7C0F }, { 4675, 0x7C19 }, { 4676, 0x7C1B }, { 4677, 0x7C20 }, { 4678, 0x7C25 }, { 4679, 0x7C26 },
    { 4680, 0x7C28 }, { 4681, 0x7C2C }, { 4682, 0x7C31 }, { 4683, 0x7C33 }, { 4684, 0x7C34 }, { 4685, 0x7C36 }, { 4686, 0x7C39 }, { 4687, 0x7C3A },
    { 4688, 0x7C46 }, { 4689, 0x7C4A }, { 4690, 0x7C55 }, { 4691, 0x7C51 }, { 4692, 0x7C52 }, { 4693, 0x7C53 }, { 4694, 0x7C59 }, { 4695, 0x7C5A },
    { 4696, 0x7C5B }, { 4697, 0x7C5C }, { 4698, 0x7C5D }, { 4699, 0x7C5E }, { 4700, 0x7C61 }, { 4701, 0x7C63 }, { 4702, 0x7C67 }, { 4703, 0x7C69 },
    { 4704, 0x7C6D }, { 4705, 0x7C6E }, { 4706, 0x7C70 }, { 4707, 0x7C72 }, { 4708, 0x7C79 }, { 4709, 0x7C7C }, { 4710, 0x7C7D }, { 4711, 0x7C86 },
    { 4712, 0x7C87 }, { 4713, 0x7C8F }, { 4714, 0x7C94 }, { 4715, 0x7C9E }, { 4716, 0x7CA0 }, { 4717, 0x7CA6 }, { 4718, 0x7CB0 }, { 4719, 0x7CB6 },
    { 4720, 0x7CB7 }, { 4721, 0x7CBA }, { 4722, 0x7CBB }, { 4723, 0x7CBC }, { 4724, 0x7CBF }, { 4725, 0x7CC4 }, { 4726, 0x7CC7 }, { 4727, 0x7CC8 },
    { 4728, 0x7CC9 }, { 4729, 0x7CCD }, { 4730, 0x7CCF }, { 4731, 0x7CD3 }, { 4732, 0x7CD4 }, { 4733, 0x7CD5 }, { 4734, 0x7CD7 }, { 4735, 0x7CD9 },
    { 4736, 0x7CDA }, { 4737, 0x7CDD }, { 4738, 0x7CE6 }, { 4739, 0x7CE9 }, { 4740, 0x7CEB }, { 4741, 0x7CF5 }, { 4742, 0x7D03 }, { 4743, 0x7D07 },
    { 4744, 0x7D08 }, { 4745, 0x7D09 }, { 4746, 0x7D0F }, { 4747, 0x7D11 }, { 4748, 0x7D12 }, { 4749, 0x7D13 }, { 4750, 0x7D16 }, { 4751, 0x7D1D },
    { 4752, 0x7D1E }, { 4753, 0x7D23 }, { 4754, 0x7D26 }, { 4755, 0x7D2A }, { 4756, 0x7D2D }, { 4757, 0x7D31 }, { 4758, 0x7D3C }, { 4759, 0x7D3D },
    { 4760, 0x7D3E }, { 4761, 0x7D40 }, { 4762, 0x7D41 }, { 4763, 0x7D47 }, { 4764, 0x7D48 }, { 4765, 0x7D4D }, { 4766, 0x7D51 }, { 4767, 0x7D53 },
    { 4768, 0x7D57 }, { 4769, 0x7D59 }, { 4770, 0x7D5A }, { 4771, 0x7D5C }, { 4772, 0x7D5D }, { 4773, 0x7D65 }, { 4774, 0x7D67 }, { 4775, 0x7D6A },
    { 4776, 0x7D70 }, { 4777, 0x7D78 }, { 4778, 0x7D7A }, { 4779, 0x7D7B }, { 4780, 0x7D7F }, { 4781, 0x7D81 }, { 4782, 0x7D82 }, { 4783, 0x7D83 },
    { 4784, 0x7D85 }, { 4785, 0x7D86 }, { 4786, 0x7D88 }, { 4787, 0x7D8B }, { 4788, 0x7D8C }, { 4789, 0x7D8D }, { 4790, 0x7D91 }, { 4791, 0x7D96 },
    { 4792, 0x7D97 }, { 4793, 0x7D9D }, { 4794, 0x7D9E }, { 4795, 0x7DA6 }, { 4796, 0x7DA7 }, { 4797, 0x7DAA }, { 4798, 0x7DB3 }, { 4799, 0x7DB6 },
    { 4800, 0x7DB7 }, { 4801, 0x7DB9 }, { 4802, 0x7DC2 }, { 4803, 0x7DC3 }, { 4804, 0x7DC4 }, { 4805, 0x7DC5 }, { 4806, 0x7DC6 }, { 4807, 0x7DCC },
    { 4808, 0x7DCD }, { 4809, 0x7DCE }, { 4810, 0x7DD7 }, { 4811, 0x7DD9 }, { 4812, 0x7E00 }, { 4813, 0x7DE2 }, { 4814, 0x7DE5 }, { 4815, 0x7DE6 },
    { 4816, 0x7DEA }, { 4817, 0x7DEB }, { 4818, 0x7DED }, { 4819, 0x7DF1 }, { 4820, 0x7DF5 }, { 4821, 0x7DF6 }, { 4822, 0x7DF9 }, { 4823, 0x7DFA },
    { 4824, 0x7E08 }, { 4825, 0x7E10 }, { 4826, 0x7E11 }, { 4827, 0x7E15 }, { 4828, 0x7E17 }, { 4829, 0x7E1C }, { 4830, 0x7E1D }, { 4831, 0x7E20 },
    { 4832, 0x7E27 }, { 4833, 0x7E28 }, { 4834, 0x7E2C }, { 4835, 0x7E2D }, { 4836, 0x7E2F }, { 4837, 0x7E33 }, { 4838, 0x7E36 }, { 4839, 0x7E3F },
    { 4840, 0x7E44 }, { 4841, 0x7E45 }, { 4842, 0x7E47 }, { 4843, 0x7E4E }, { 4844, 0x7E50 }, { 4845, 0x7E52 }, { 4846, 0x7E58 }, { 4847, 0x7E5F },
    { 4848, 0x7E61 }, { 4849, 0x7E62 }, { 4850, 0x7E65 }, { 4851, 0x7E6B }, { 4852, 0x7E6E }, { 4853, 0x7E6F }, { 4854, 0x7E73 }, { 4855, 0x7E78 },
    { 4856, 0x7E7E }, { 4857, 0x7E81 }, { 4858, 0x7E86 }, { 4859, 0x7E87 }, { 4860, 0x7E8A }, { 4861, 0x7E8D }, { 4862, 0x7E91 }, { 4863, 0x7E95 },
    { 4864, 0x7E98 }, { 4865, 0x7E9A }, { 4866, 0x7E9D }, { 4867, 0x7E9E }, { 4868, 0x7F3C }, { 4869, 0x7F3B }, { 4870, 0x7F3D }, { 4871, 0x7F3E },
    { 4872, 0x7F3F }, { 4873, 0x7F43 }, { 4874, 0x7F44 }, { 4875, 0x7F47 }, { 4876, 0x7F4F }, { 4877, 0x7F52 }, { 4878, 0x7F53 }, { 4879, 0x7F5B },
    { 4880, 0x7F5C }, { 4881, 0x7F5D }, { 4882, 0x7F61 }, { 4883, 0x7F63 }, { 4884, 0x7F64 }, { 4885, 0x7F65 }, { 4886, 0x7F66 }, { 4887, 0x7F6D },
    { 4888, 0x7F71 }, { 4889, 0x7F7D }, { 4890, 0x7F7E }, { 4891, 0x7F7F }, { 4892, 0x7F80 }, { 4893, 0x7F8B }, { 4894, 0x7F8D }, { 4895, 0x7F8F },
    { 4896, 0x7F90 }, { 4897, 0x7F91 }, { 4898, 0x7F96 }, { 4899, 0x7F97 }, { 4900, 0x7F9C }, { 4901, 0x7FA1 }, { 4902, 0x7FA2 }, { 4903, 0x7FA6 },
    { 4904, 0x7FAA }, { 4905, 0x7FAD }, { 4906, 0x7FB4 }, { 4907, 0x7FBC }, { 4908, 0x7FBF }, { 4909, 0x7FC0 }, { 4910, 0x7FC3 }, { 4911, 0x7FC8 },
    { 4912, 0x7FCE }, { 4913, 0x7FCF }, { 4914, 0x7FDB }, { 4915, 0x7FDF }, { 4916, 0x7FE3 }, { 4917, 0x7FE5 }, { 4918, 0x7FE8 }, { 4919, 0x7FEC },
    { 4920, 0x7FEE }, { 4921, 0x7FEF }, { 4922, 0x7FF2 }, { 4923, 0x7FFA }, { 4924, 0x7FFD }, { 4925, 0x7FFE }, { 4926, 0x7FFF }, { 4927, 0x8007 },
    { 4928, 0x8008 }, { 4929, 0x800A }, { 4930, 0x800D }, { 4931, 0x800E }, { 4932, 0x800F }, { 4933, 0x8011 }, { 4934, 0x8013 }, { 4935, 0x8014 },
    { 4936, 0x8016 }, { 4937, 0x801D }, { 4938, 0x801E }, { 4939, 0x801F }, { 4940, 0x8020 }, { 4941, 0x8024 }, { 4942, 0x8026 }, { 4943, 0x802C },
    { 4944, 0x802E }, { 4945, 0x8030 }, { 4946, 0x8034 }, { 4947, 0x8035 }, { 4948, 0x8037 }, { 4949, 0x8039 }, { 4950, 0x803A }, { 4951, 0x803C },
    { 4952, 0x803E }, { 4953, 0x8040 }, { 4954, 0x8044 }, { 4955, 0x8060 }, { 4956, 0x8064 }, { 4957, 0x8066 }, { 4958, 0x806D }, { 4959, 0x8071 },
    { 4960, 0x8075 }, { 4961, 0x8081 }, { 4962, 0x8088 }, { 4963, 0x808E }, { 4964, 0x809C }, { 4965, 0x809E }, { 4966, 0x80A6 }, { 4967, 0x80A7 },
    { 4968, 0x80AB }, { 4969, 0x80B8 }, { 4970, 0x80B9 }, { 4971, 0x80C8 }, { 4972, 0x80CD }, { 4973, 0x80CF }, { 4974, 0x80D2 }, { 4975, 0x80D4 },
    { 4976, 0x80D5 }, { 4977, 0x80D7 }, { 4978, 0x80D8 }, { 4979, 0x80E0 }, { 4980, 0x80ED }, { 4981, 0x80EE }, { 4982, 0x80F0 }, { 4983, 0x80F2 },
    { 4984, 0x80F3 }, { 4985, 0x80F6 }, { 4986, 0x80F9 }, { 4987, 0x80FA }, { 4988, 0x80FE }, { 4989, 0x8103 }, { 4990, 0x810B }, { 4991, 0x8116 },
    { 4992, 0x8117 }, { 4993, 0x8118 }, { 4994, 0x811C }, { 4995, 0x811E }, { 4996, 0x8120 }, { 4997, 0x8124 }, { 4998, 0x8127 }, { 4999, 0x812C },
    { 5000, 0x8130 }, { 5001, 0x8135 }, { 5002, 0x813A }, { 5003, 0x813C }, { 5004, 0x8145 }, { 5005, 0x8147 }, { 5006, 0x814A }, { 5007, 0x814C },
    { 5008, 0x8152 }, { 5009, 0x8157 }, { 5010, 0x8160 }, { 5011, 0x8161 }, { 5012, 0x8167 }, { 5013, 0x8168 }, { 5014, 0x8169 }, { 5015, 0x816D },
    { 5016, 0x816F }, { 5017, 0x8177 }, { 5018, 0x8181 }, { 5019, 0x8190 }, { 5020, 0x8184 }, { 5021, 0x8185 }, { 5022, 0x8186 }, { 5023, 0x818B },
    { 5024, 0x818E }, { 5025, 0x8196 }, { 5026, 0x8198 }, { 5027, 0x819B }, { 5028, 0x819E }, { 5029, 0x81A2 }, { 5030, 0x81AE }, { 5031, 0x81B2 },
    { 5032, 0x81B4 }, { 5033, 0x81BB }, { 5034, 0x81CB }, { 5035, 0x81C3 }, { 5036, 0x81C5 }, { 5037, 0x81CA }, { 5038, 0x81CE }, { 5039, 0x81CF },
    { 5040, 0x81D5 }, { 5041, 0x81D7 }, { 5042, 0x81DB }, { 5043, 0x81DD }, { 5044, 0x81DE }, { 5045, 0x81E1 }, { 5046, 0x81E4 }, { 5047, 0x81EB },
    { 5048, 0x81EC }, { 5049, 0x81F0 }, { 5050, 0x81F1 }, { 5051, 0x81F2 }, { 5052, 0x81F5 }, { 5053, 0x81F6 }, { 5054, 0x81F8 }, { 5055, 0x81F9 },
    { 5056, 0x81FD }, { 5057, 0x81FF }, { 5058, 0x8200 }, { 5059, 0x8203 }, { 5060, 0x820F }, { 5061, 0x8213 }, { 5062, 0x8214 }, { 5063, 0x8219 },
    { 5064, 0x821A }, { 5065, 0x821D }, { 5066, 0x8221 }, { 5067, 0x8222 }, { 5068, 0x8228 }, { 5069, 0x8232 }, { 5070, 0x8234 }, { 5071, 0x823A },
    { 5072, 0x8243 }, { 5073, 0x8244 }, { 5074, 0x8245 }, { 5075, 0x8246 }, { 5076, 0x824B }, { 5077, 0x824E }, { 5078, 0x824F }, { 5079, 0x8251 },
    { 5080, 0x8256 }, { 5081, 0x825C }, { 5082, 0x8260 }, { 5083, 0x8263 }, { 5084, 0x8267 }, { 5085, 0x826D }, { 5086, 0x8274 }, { 5087, 0x827B },
    { 5088, 0x827D }, { 5089, 0x827F }, { 5090, 0x8280 }, { 5091, 0x8281 }, { 5092, 0x8283 }, { 5093, 0x8284 }, { 5094, 0x8287 }, { 5095, 0x8289 },
    { 5096, 0x828A }, { 5097, 0x828E }, { 5098, 0x8291 }, { 5099, 0x8294 }, { 5100, 0x8296 }, { 5101, 0x8298 }, { 5102, 0x829A }, { 5103, 0x829B },
    { 5104, 0x82A0 }, { 5105, 0x82A1 }, { 5106, 0x82A3 }, { 5107, 0x82A4 }, { 5108, 0x82A7 }, { 5109, 0x82A8 }, { 5110, 0x82A9 }, { 5111, 0x82AA },
    { 5112, 0x82AE }, { 5113, 0x82B0 }, { 5114, 0x82B2 }, { 5115, 0x82B4 }, { 5116, 0x82B7 }, { 5117, 0x82BA }, { 5118, 0x82BC }, { 5119, 0x82BE },
    { 5120, 0x82BF }, { 5121, 0x82C6 }, { 5122, 0x82D0 }, { 5123, 0x82D5 }, { 5124, 0x82DA }, { 5125, 0x82E0 }, { 5126, 0x82E2 }, { 5127, 0x82E4 },
    { 5128, 0x82E8 }, { 5129, 0x82EA }, { 5130, 0x82ED }, { 5131, 0x82EF }, { 5132, 0x82F6 }, { 5133, 0x82F7 }, { 5134, 0x82FD }, { 5135, 0x82FE },
    { 5136, 0x8300 }, { 5137, 0x8301 }, { 5138, 0x8307 }, { 5139, 0x8308 }, { 5140, 0x830A }, { 5141, 0x830B }, { 5142, 0x8354 }, { 5143, 0x831B },
    { 5144, 0x831D }, { 5145, 0x831E }, { 5146, 0x831F }, { 5147, 0x8321 }, { 5148, 0x8322 }, { 5149, 0x832C }, { 5150, 0x832D }, { 5151, 0x832E },
    { 5152, 0x8330 }, { 5153, 0x8333 }, { 5154, 0x8337 }, { 5155, 0x833A }, { 5156, 0x833C }, { 5157, 0x833D }, { 5158, 0x8342 }, { 5159, 0x8343 },
    { 5160, 0x8344 }, { 5161, 0x8347 }, { 5162, 0x834D }, { 5163, 0x834E }, { 5164, 0x8351 }, { 5165, 0x8355 }, { 5166, 0x8356 }, { 5167, 0x8357 },
    { 5168, 0x8370 }, { 5169, 0x8378 }, { 5170, 0x837D }, { 5171, 0x837F }, { 5172, 0x8380 }, { 5173, 0x8382 }, { 5174, 0x8384 }, { 5175, 0x8386 },
    { 5176, 0x838D }, { 5177, 0x8392 }, { 5178, 0x8394 }, { 5179, 0x8395 }, { 5180, 0x8398 }, { 5181, 0x8399 }, { 5182, 0x839B }, { 5183, 0x839C },
    { 5184, 0x839D }, { 5185, 0x83A6 }, { 5186, 0x83A7 }, { 5187, 0x83A9 }, { 5188, 0x83AC }, { 5189, 0x83BE }, { 5190, 0x83BF }, { 5191, 0x83C0 },
    { 5192, 0x83C7 }, { 5193, 0x83C9 }, { 5194, 0x83CF }, { 5195, 0x83D0 }, { 5196, 0x83D1 }, { 5197, 0x83D4 }, { 5198, 0x83DD }, { 5199, 0x8353 },
    { 5200, 0x83E8 }, { 5201, 0x83EA }, { 5202, 0x83F6 }, { 5203, 0x83F8 }, { 5204, 0x83F9 }, { 5205, 0x83FC }, { 5206, 0x8401 }, { 5207, 0x8406 },
    { 5208, 0x840A }, { 5209, 0x840F }, { 5210, 0x8411 }, { 5211, 0x8415 }, { 5212, 0x8419 }, { 5213, 0x83AD }, { 5214, 0x842F }, { 5215, 0x8439 },
    { 5216, 0x8445 }, { 5217, 0x8447 }, { 5218, 0x8448 }, { 5219, 0x844A }, { 5220, 0x844D }, { 5221, 0x844F }, { 5222, 0x8451 }, { 5223, 0x8452 },
    { 5224, 0x8456 }, { 5225, 0x8458 }, { 5226, 0x8459 }, { 5227, 0x845A }, { 5228, 0x845C }, { 5229, 0x8460 }, { 5230, 0x8464 }, { 5231, 0x8465 },
    { 5232, 0x8467 }, { 5233, 0x846A }, { 5234, 0x8470 }, { 5235, 0x8473 }, { 5236, 0x8474 }, { 5237, 0x8476 }, { 5238, 0x8478 }, { 5239, 0x847C },
    { 5240, 0x847D }, { 5241, 0x8481 }, { 5242, 0x8485 }, { 5243, 0x8492 }, { 5244, 0x8493 }, { 5245, 0x8495 }, { 5246, 0x849E }, { 5247, 0x84A6 },
    { 5248, 0x84A8 }, { 5249, 0x84A9 }, { 5250, 0x84AA }, { 5251, 0x84AF }, { 5252, 0x84B1 }, { 5253, 0x84B4 }, { 5254, 0x84BA }, { 5255, 0x84BD },
    { 5256, 0x84BE }, { 5257, 0x84C0 }, { 5258, 0x84C2 }, { 5259, 0x84C7 }, { 5260, 0x84C8 }, { 5261, 0x84CC }, { 5262, 0x84CF }, { 5263, 0x84D3 },
    { 5264, 0x84DC }, { 5265, 0x84E7 }, { 5266, 0x84EA }, { 5267, 0x84EF }, { 5268, 0x84F0 }, { 5269, 0x84F1 }, { 5270, 0x84F2 }, { 5271, 0x84F7 },
    { 5272, 0x8532 }, { 5273, 0x84FA }, { 5274, 0x84FB }, { 5275, 0x84FD }, { 5276, 0x8502 }, { 5277, 0x8503 }, { 5278, 0x8507 }, { 5279, 0x850C },
    { 5280, 0x850E }, { 5281, 0x8510 }, { 5282, 0x851C }, { 5283, 0x851E }, { 5284, 0x8522 }, { 5285, 0x8523 }, { 5286, 0x8524 }, { 5287, 0x8525 },
    { 5288, 0x8527 }, { 5289, 0x852A }, { 5290, 0x852B }, { 5291, 0x852F }, { 5292, 0x8533 }, { 5293, 0x8534 }, { 5294, 0x8536 }, { 5295, 0x853F },
    { 5296, 0x8546 }, { 5297, 0x854F }, { 5298, 0x8550 }, { 5299, 0x8551 }, { 5300, 0x8552 }, { 5301, 0x8553 }, { 5302, 0x8556 }, { 5303, 0x8559 },
    { 5304, 0x855C }, { 5305, 0x855D }, { 5306, 0x855E }, { 5307, 0x855F }, { 5308, 0x8560 }, { 5309, 0x8561 }, { 5310, 0x8562 }, { 5311, 0x8564 },
    { 5312, 0x856B }, { 5313, 0x856F }, { 5314, 0x8579 }, { 5315, 0x857A }, { 5316, 0x857B }, { 5317, 0x857D }, { 5318, 0x857F }, { 5319, 0x8581 },
    { 5320, 0x8585 }, { 5321, 0x8586 }, { 5322, 0x8589 }, { 5323, 0x858B }, { 5324, 0x858C }, { 5325, 0x858F }, { 5326, 0x8593 }, { 5327, 0x8598 },
    { 5328, 0x859D }, { 5329, 0x859F }, { 5330, 0x85A0 }, { 5331, 0x85A2 }, { 5332, 0x85A5 }, { 5333, 0x85A7 }, { 5334, 0x85B4 }, { 5335, 0x85B6 },
    { 5336, 0x85B7 }, { 5337, 0x85B8 }, { 5338, 0x85BC }, { 5339, 0x85BD }, { 5340, 0x85BE }, { 5341, 0x85BF }, { 5342, 0x85C2 }, { 5343, 0x85C7 },
    { 5344, 0x85CA }, { 5345, 0x85CB }, { 5346, 0x85CE }, { 5347, 0x85AD }, { 5348, 0x85D8 }, { 5349, 0x85DA }, { 5350, 0x85DF }, { 5351, 0x85E0 },
    { 5352, 0x85E6 }, { 5353, 0x85E8 }, { 5354, 0x85ED }, { 5355, 0x85F3 }, { 5356, 0x85F6 }, { 5357, 0x85FC }, { 5358, 0x85FF }, { 5359, 0x8600 },
    { 5360, 0x8604 }, { 5361, 0x8605 }, { 5362, 0x860D }, { 5363, 0x860E }, { 5364, 0x8610 }, { 5365, 0x8611 }, { 5366, 0x8612 }, { 5367, 0x8618 },
    { 5368, 0x8619 }, { 5369, 0x861B }, { 5370, 0x861E }, { 5371, 0x8621 }, { 5372, 0x8627 }, { 5373, 0x8629 }, { 5374, 0x8636 }, { 5375, 0x8638 },
    { 5376, 0x863A }, { 5377, 0x863C }, { 5378, 0x863D }, { 5379, 0x8640 }, { 5380, 0x8642 }, { 5381, 0x8646 }, { 5382, 0x8652 }, { 5383, 0x8653 },
    { 5384, 0x8656 }, { 5385, 0x8657 }, { 5386, 0x8658 }, { 5387, 0x8659 }, { 5388, 0x865D }, { 5389, 0x8660 }, { 5390, 0x8661 }, { 5391, 0x8662 },
    { 5392, 0x8663 }, { 5393, 0x8664 }, { 5394, 0x8669 }, { 5395, 0x866C }, { 5396, 0x866F }, { 5397, 0x8675 }, { 5398, 0x8676 }, { 5399, 0x8677 },
    { 5400, 0x867A }, { 5401, 0x868D }, { 5402, 0x8691 }, { 5403, 0x8696 }, { 5404, 0x8698 }, { 5405, 0x869A }, { 5406, 0x869C }, { 5407, 0x86A1 },
    { 5408, 0x86A6 }, { 5409, 0x86A7 }, { 5410, 0x86A8 }, { 5411, 0x86AD }, { 5412, 0x86B1 }, { 5413, 0x86B3 }, { 5414, 0x86B4 }, { 5415, 0x86B5 },
    { 5416, 0x86B7 }, { 5417, 0x86B8 }, { 5418, 0x86B9 }, { 5419, 0x86BF }, { 5420, 0x86C0 }, { 5421, 0x86C1 }, { 5422, 0x86C3 }, { 5423, 0x86C5 },
    { 5424, 0x86D1 }, { 5425, 0x86D2 }, { 5426, 0x86D5 }, { 5427, 0x86D7 }, { 5428, 0x86DA }, { 5429, 0x86DC }, { 5430, 0x86E0 }, { 5431, 0x86E3 },
    { 5432, 0x86E5 }, { 5433, 0x86E7 }, { 5434, 0x8688 }, { 5435, 0x86FA }, { 5436, 0x86FC }, { 5437, 0x86FD }, { 5438, 0x8704 }, { 5439, 0x8705 },
    { 5440, 0x8707 }, { 5441, 0x870B }, { 5442, 0x870E }, { 5443, 0x870F }, { 5444, 0x8710 }, { 5445, 0x8713 }, { 5446, 0x8714 }, { 5447, 0x8719 },
    { 5448, 0x871E }, { 5449, 0x871F }, { 5450, 0x8721 }, { 5451, 0x8723 }, { 5452, 0x8728 }, { 5453, 0x872E }, { 5454, 0x872F }, { 5455, 0x8731 },
    { 5456, 0x8732 }, { 5457, 0x8739 }, { 5458, 0x873A }, { 5459, 0x873C }, { 5460, 0x873D }, { 5461, 0x873E }, { 5462, 0x8740 }, { 5463, 0x8743 },
    { 5464, 0x8745 }, { 5465, 0x874D }, { 5466, 0x8758 }, { 5467, 0x875D }, { 5468, 0x8761 }, { 5469, 0x8764 }, { 5470, 0x8765 }, { 5471, 0x876F },
    { 5472, 0x8771 }, { 5473, 0x8772 }, { 5474, 0x877B }, { 5475, 0x8783 }, { 5476, 0x8784 }, { 5477, 0x8785 }, { 5478, 0x8786 }, { 5479, 0x8787 },
    { 5480, 0x8788 }, { 5481, 0x8789 }, { 5482, 0x878B }, { 5483, 0x878C }, { 5484, 0x8790 }, { 5485, 0x8793 }, { 5486, 0x8795 }, { 5487, 0x8797 },
    { 5488, 0x8798 }, { 5489, 0x8799 }, { 5490, 0x879E }, { 5491, 0x87A0 }, { 5492, 0x87A3 }, { 5493, 0x87A7 }, { 5494, 0x87AC }, { 5495, 0x87AD },
    { 5496, 0x87AE }, { 5497, 0x87B1 }, { 5498, 0x87B5 }, { 5499, 0x87BE }, { 5500, 0x87BF }, { 5501, 0x87C1 }, { 5502, 0x87C8 }, { 5503, 0x87C9 },
    { 5504, 0x87CA }, { 5505, 0x87CE }, { 5506, 0x87D5 }, { 5507, 0x87D6 }, { 5508, 0x87D9 }, { 5509, 0x87DA }, { 5510, 0x87DC }, { 5511, 0x87DF },
    { 5512, 0x87E2 }, { 5513, 0x87E3 }, { 5514, 0x87E4 }, { 5515, 0x87EA }, { 5516, 0x87EB }, { 5517, 0x87ED }, { 5518, 0x87F1 }, { 5519, 0x87F3 },
    { 5520, 0x87F8 }, { 5521, 0x87FA }, { 5522, 0x87FF }, { 5523, 0x8801 }, { 5524, 0x8803 }, { 5525, 0x8806 }, { 5526, 0x8809 }, { 5527, 0x880A },
    { 5528, 0x880B }, { 5529, 0x8810 }, { 5530, 0x8819 }, { 5531, 0x8812 }, { 5532, 0x8813 }, { 5533, 0x8814 }, { 5534, 0x8818 }, { 5535, 0x881A },
    { 5536, 0x881B }, { 5537, 0x881C }, { 5538, 0x881E }, { 5539, 0x881F }, { 5540, 0x8828 }, { 5541, 0x882D }, { 5542, 0x882E }, { 5543, 0x8830 },
    { 5544, 0x8832 }, { 5545, 0x8835 }, { 5546, 0x883A }, { 5547, 0x883C }, { 5548, 0x8841 }, { 5549, 0x8843 }, { 5550, 0x8845 }, { 5551, 0x8848 },
    { 5552, 0x8849 }, { 5553, 0x884A }, { 5554, 0x884B }, { 5555, 0x884E }, { 5556, 0x8851 }, { 5557, 0x8855 }, { 5558, 0x8856 }, { 5559, 0x8858 },
    { 5560, 0x885A }, { 5561, 0x885C }, { 5562, 0x885F }, { 5563, 0x8860 }, { 5564, 0x8864 }, { 5565, 0x8869 }, { 5566, 0x8871 }, { 5567, 0x8879 },
    { 5568, 0x887B }, { 5569, 0x8880 }, { 5570, 0x8898 }, { 5571, 0x889A }, { 5572, 0x889B }, { 5573, 0x889C }, { 5574, 0x889F }, { 5575, 0x88A0 },
    { 5576, 0x88A8 }, { 5577, 0x88AA }, { 5578, 0x88BA }, { 5579, 0x88BD }, { 5580, 0x88BE }, { 5581, 0x88C0 }, { 5582, 0x88CA }, { 5583, 0x88CB },
    { 5584, 0x88CC }, { 5585, 0x88CD }, { 5586, 0x88CE }, { 5587, 0x88D1 }, { 5588, 0x88D2 }, { 5589, 0x88D3 }, { 5590, 0x88DB }, { 5591, 0x88DE },
    { 5592, 0x88E7 }, { 5593, 0x88EF }, { 5594, 0x88F0 }, { 5595, 0x88F1 }, { 5596, 0x88F5 }, { 5597, 0x88F7 }, { 5598, 0x8901 }, { 5599, 0x8906 },
    { 5600, 0x890D }, { 5601, 0x890E }, { 5602, 0x890F }, { 5603, 0x8915 }, { 5604, 0x8916 }, { 5605, 0x8918 }, { 5606, 0x8919 }, { 5607, 0x891A },
    { 5608, 0x891C }, { 5609, 0x8920 }, { 5610, 0x8926 }, { 5611, 0x8927 }, { 5612, 0x8928 }, { 5613, 0x8930 }, { 5614, 0x8931 }, { 5615, 0x8932 },
    { 5616, 0x8935 }, { 5617, 0x8939 }, { 5618, 0x893A }, { 5619, 0x893E }, { 5620, 0x8940 }, { 5621, 0x8942 }, { 5622, 0x8945 }, { 5623, 0x8946 },
    { 5624, 0x8949 }, { 5625, 0x894F }, { 5626, 0x8952 }, { 5627, 0x8957 }, { 5628, 0x895A }, { 5629, 0x895B }, { 5630, 0x895C }, { 5631, 0x8961 },
    { 5632, 0x8962 }, { 5633, 0x8963 }, { 5634, 0x896B }, { 5635, 0x896E }, { 5636, 0x8970 }, { 5637, 0x8973 }, { 5638, 0x8975 }, { 5639, 0x897A },
    { 5640, 0x897B }, { 5641, 0x897C }, { 5642, 0x897D }, { 5643, 0x8989 }, { 5644, 0x898D }, { 5645, 0x8990 }, { 5646, 0x8994 }, { 5647, 0x8995 },
    { 5648, 0x899B }, { 5649, 0x899C }, { 5650, 0x899F }, { 5651, 0x89A0 }, { 5652, 0x89A5 }, { 5653, 0x89B0 }, { 5654, 0x89B4 }, { 5655, 0x89B5 },
    { 5656, 0x89B6 }, { 5657, 0x89B7 }, { 5658, 0x89BC }, { 5659, 0x89D4 }, { 5660, 0x89D5 }, { 5661, 0x89D6 }, { 5662, 0x89D7 }, { 5663, 0x89D8 },
    { 5664, 0x89E5 }, { 5665, 0x89E9 }, { 5666, 0x89EB }, { 5667, 0x89ED }, { 5668, 0x89F1 }, { 5669, 0x89F3 }, { 5670, 0x89F6 }, { 5671, 0x89F9 },
    { 5672, 0x89FD }, { 5673, 0x89FF }, { 5674, 0x8A04 }, { 5675, 0x8A05 }, { 5676, 0x8A07 }, { 5677, 0x8A0F }, { 5678, 0x8A11 }, { 5679, 0x8A12 },
    { 5680, 0x8A14 }, { 5681, 0x8A15 }, { 5682, 0x8A1E }, { 5683, 0x8A20 }, { 5684, 0x8A22 }, { 5685, 0x8A24 }, { 5686, 0x8A26 }, { 5687, 0x8A2B },
    { 5688, 0x8A2C }, { 5689, 0x8A2F }, { 5690, 0x8A35 }, { 5691, 0x8A37 }, { 5692, 0x8A3D }, { 5693, 0x8A3E }, { 5694, 0x8A40 }, { 5695, 0x8A43 },
    { 5696, 0x8A45 }, { 5697, 0x8A47 }, { 5698, 0x8A49 }, { 5699, 0x8A4D }, { 5700, 0x8A4E }, { 5701, 0x8A53 }, { 5702, 0x8A56 }, { 5703, 0x8A57 },
    { 5704, 0x8A58 }, { 5705, 0x8A5C }, { 5706, 0x8A5D }, { 5707, 0x8A61 }, { 5708, 0x8A65 }, { 5709, 0x8A67 }, { 5710, 0x8A75 }, { 5711, 0x8A76 },
    { 5712, 0x8A77 }, { 5713, 0x8A79 }, { 5714, 0x8A7A }, { 5715, 0x8A7B }, { 5716, 0x8A7E }, { 5717, 0x8A7F }, { 5718, 0x8A80 }, { 5719, 0x8A83 },
    { 5720, 0x8A86 }, { 5721, 0x8A8B }, { 5722, 0x8A8F }, { 5723, 0x8A90 }, { 5724, 0x8A92 }, { 5725, 0x8A96 }, { 5726, 0x8A97 }, { 5727, 0x8A99 },
    { 5728, 0x8A9F }, { 5729, 0x8AA7 }, { 5730, 0x8AA9 }, { 5731, 0x8AAE }, { 5732, 0x8AAF }, { 5733, 0x8AB3 }, { 5734, 0x8AB6 }, { 5735, 0x8AB7 },
    { 5736, 0x8ABB }, { 5737, 0x8ABE }, { 5738, 0x8AC3 }, { 5739, 0x8AC6 }, { 5740, 0x8AC8 }, { 5741, 0x8AC9 }, { 5742, 0x8ACA }, { 5743, 0x8AD1 },
    { 5744, 0x8AD3 }, { 5745, 0x8AD4 }, { 5746, 0x8AD5 }, { 5747, 0x8AD7 }, { 5748, 0x8ADD }, { 5749, 0x8ADF }, { 5750, 0x8AEC }, { 5751, 0x8AF0 },
    { 5752, 0x8AF4 }, { 5753, 0x8AF5 }, { 5754, 0x8AF6 }, { 5755, 0x8AFC }, { 5756, 0x8AFF }, { 5757, 0x8B05 }, { 5758, 0x8B06 }, { 5759, 0x8B0B },
    { 5760, 0x8B11 }, { 5761, 0x8B1C }, { 5762, 0x8B1E }, { 5763, 0x8B1F }, { 5764, 0x8B0A }, { 5765, 0x8B2D }, { 5766, 0x8B30 }, { 5767, 0x8B37 },
    { 5768, 0x8B3C }, { 5769, 0x8B42 }, { 5770, 0x8B43 }, { 5771, 0x8B44 }, { 5772, 0x8B45 }, { 5773, 0x8B46 }, { 5774, 0x8B48 }, { 5775, 0x8B52 },
    { 5776, 0x8B53 }, { 5777, 0x8B54 }, { 5778, 0x8B59 }, { 5779, 0x8B4D }, { 5780, 0x8B5E }, { 5781, 0x8B63 }, { 5782, 0x8B6D }, { 5783, 0x8B76 },
    { 5784, 0x8B78 }, { 5785, 0x8B79 }, { 5786, 0x8B7C }, { 5787, 0x8B7E }, { 5788, 0x8B81 }, { 5789, 0x8B84 }, { 5790, 0x8B85 }, { 5791, 0x8B8B },
    { 5792, 0x8B8D }, { 5793, 0x8B8F }, { 5794, 0x8B94 }, { 5795, 0x8B95 }, { 5796, 0x8B9C }, { 5797, 0x8B9E }, { 5798, 0x8B9F }, { 5799, 0x8C38 },
    { 5800, 0x8C39 }, { 5801, 0x8C3D }, { 5802, 0x8C3E }, { 5803, 0x8C45 }, { 5804, 0x8C47 }, { 5805, 0x8C49 }, { 5806, 0x8C4B }, { 5807, 0x8C4F },
    { 5808, 0x8C51 }, { 5809, 0x8C53 }, { 5810, 0x8C54 }, { 5811, 0x8C57 }, { 5812, 0x8C58 }, { 5813, 0x8C5B }, { 5814, 0x8C5D }, { 5815, 0x8C59 },
    { 5816, 0x8C63 }, { 5817, 0x8C64 }, { 5818, 0x8C66 }, { 5819, 0x8C68 }, { 5820, 0x8C69 }, { 5821, 0x8C6D }, { 5822, 0x8C73 }, { 5823, 0x8C75 },
    { 5824, 0x8C76 }, { 5825, 0x8C7B }, { 5826, 0x8C7E }, { 5827, 0x8C86 }, { 5828, 0x8C87 }, { 5829, 0x8C8B }, { 5830, 0x8C90 }, { 5831, 0x8C92 },
    { 5832, 0x8C93 }, { 5833, 0x8C99 }, { 5834, 0x8C9B }, { 5835, 0x8C9C }, { 5836, 0x8CA4 }, { 5837, 0x8CB9 }, { 5838, 0x8CBA }, { 5839, 0x8CC5 },
    { 5840, 0x8CC6 }, { 5841, 0x8CC9 }, { 5842, 0x8CCB }, { 5843, 0x8CCF }, { 5844, 0x8CD6 }, { 5845, 0x8CD5 }, { 5846, 0x8CD9 }, { 5847, 0x8CDD },
    { 5848, 0x8CE1 }, { 5849, 0x8CE8 }, { 5850, 0x8CEC }, { 5851, 0x8CEF }, { 5852, 0x8CF0 }, { 5853, 0x8CF2 }, { 5854, 0x8CF5 }, { 5855, 0x8CF7 },
    { 5856, 0x8CF8 }, { 5857, 0x8CFE }, { 5858, 0x8CFF }, { 5859, 0x8D01 }, { 5860, 0x8D03 }, { 5861, 0x8D09 }, { 5862, 0x8D12 }, { 5863, 0x8D17 },
    { 5864, 0x8D1B }, { 5865, 0x8D65 }, { 5866, 0x8D69 }, { 5867, 0x8D6C }, { 5868, 0x8D6E }, { 5869, 0x8D7F }, { 5870, 0x8D82 }, { 5871, 0x8D84 },
    { 5872, 0x8D88 }, { 5873, 0x8D8D }, { 5874, 0x8D90 }, { 5875, 0x8D91 }, { 5876, 0x8D95 }, { 5877, 0x8D9E }, { 5878, 0x8D9F }, { 5879, 0x8DA0 },
    { 5880, 0x8DA6 }, { 5881, 0x8DAB }, { 5882, 0x8DAC }, { 5883, 0x8DAF }, { 5884, 0x8DB2 }, { 5885, 0x8DB5 }, { 5886, 0x8DB7 }, { 5887, 0x8DB9 },
    { 5888, 0x8DBB }, { 5889, 0x8DC0 }, { 5890, 0x8DC5 }, { 5891, 0x8DC6 }, { 5892, 0x8DC7 }, { 5893, 0x8DC8 }, { 5894, 0x8DCA }, { 5895, 0x8DCE },
    { 5896, 0x8DD1 }, { 5897, 0x8DD4 }, { 5898, 0x8DD5 }, { 5899, 0x8DD7 }, { 5900, 0x8DD9 }, { 5901, 0x8DE4 }, { 5902, 0x8DE5 }, { 5903, 0x8DE7 },
    { 5904, 0x8DEC }, { 5905, 0x8DF0 }, { 5906, 0x8DBC }, { 5907, 0x8DF1 }, { 5908, 0x8DF2 }, { 5909, 0x8DF4 }, { 5910, 0x8DFD }, { 5911, 0x8E01 },
    { 5912, 0x8E04 }, { 5913, 0x8E05 }, { 5914, 0x8E06 }, { 5915, 0x8E0B }, { 5916, 0x8E11 }, { 5917, 0x8E14 }, { 5918, 0x8E16 }, { 5919, 0x8E20 },
    { 5920, 0x8E21 }, { 5921, 0x8E22 }, { 5922, 0x8E23 }, { 5923, 0x8E26 }, { 5924, 0x8E27 }, { 5925, 0x8E31 }, { 5926, 0x8E33 }, { 5927, 0x8E36 },
    { 5928, 0x8E37 }, { 5929, 0x8E38 }, { 5930, 0x8E39 }, { 5931, 0x8E3D }, { 5932, 0x8E40 }, { 5933, 0x8E41 }, { 5934, 0x8E4B }, { 5935, 0x8E4D },
    { 5936, 0x8E4E }, { 5937, 0x8E4F }, { 5938, 0x8E54 }, { 5939, 0x8E5B }, { 5940, 0x8E5C }, { 5941, 0x8E5D }, { 5942, 0x8E5E }, { 5943, 0x8E61 },
    { 5944, 0x8E62 }, { 5945, 0x8E69 }, { 5946, 0x8E6C }, { 5947, 0x8E6D }, { 5948, 0x8E6F }, { 5949, 0x8E70 }, { 5950, 0x8E71 }, { 5951, 0x8E79 },
    { 5952, 0x8E7A }, { 5953, 0x8E7B }, { 5954, 0x8E82 }, { 5955, 0x8E83 }, { 5956, 0x8E89 }, { 5957, 0x8E90 }, { 5958, 0x8E92 }, { 5959, 0x8E95 },
    { 5960, 0x8E9A }, { 5961, 0x8E9B }, { 5962, 0x8E9D }, { 5963, 0x8E9E }, { 5964, 0x8EA2 }, { 5965, 0x8EA7 }, { 5966, 0x8EA9 }, { 5967, 0x8EAD },
    { 5968, 0x8EAE }, { 5969, 0x8EB3 }, { 5970, 0x8EB5 }, { 5971, 0x8EBA }, { 5972, 0x8EBB }, { 5973, 0x8EC0 }, { 5974, 0x8EC1 }, { 5975, 0x8EC3 },
    { 5976, 0x8EC4 }, { 5977, 0x8EC7 }, { 5978, 0x8ECF }, { 5979, 0x8ED1 }, { 5980, 0x8ED4 }, { 5981, 0x8EDC }, { 5982, 0x8EE8 }, { 5983, 0x8EEE },
    { 5984, 0x8EF0 }, { 5985, 0x8EF1 }, { 5986, 0x8EF7 }, { 5987, 0x8EF9 }, { 5988, 0x8EFA }, { 5989, 0x8EED }, { 5990, 0x8F00 }, { 5991, 0x8F02 },
    { 5992, 0x8F07 }, { 5993, 0x8F08 }, { 5994, 0x8F0F }, { 5995, 0x8F10 }, { 5996, 0x8F16 }, { 5997, 0x8F17 }, { 5998, 0x8F18 }, { 5999, 0x8F1E },
    { 6000, 0x8F20 }, { 6001, 0x8F21 }, { 6002, 0x8F23 }, { 6003, 0x8F25 }, { 6004, 0x8F27 }, { 6005, 0x8F28 }, { 6006, 0x8F2C }, { 6007, 0x8F2D },
    { 6008, 0x8F2E }, { 6009, 0x8F34 }, { 6010, 0x8F35 }, { 6011, 0x8F36 }, { 6012, 0x8F37 }, { 6013, 0x8F3A }, { 6014, 0x8F40 }, { 6015, 0x8F41 },
    { 6016, 0x8F43 }, { 6017, 0x8F47 }, { 6018, 0x8F4F }, { 6019, 0x8F51 }, { 6020, 0x8F52 }, { 6021, 0x8F53 }, { 6022, 0x8F54 }, { 6023, 0x8F55 },
    { 6024, 0x8F58 }, { 6025, 0x8F5D }, { 6026, 0x8F5E }, { 6027, 0x8F65 }, { 6028, 0x8F9D }, { 6029, 0x8FA0 }, { 6030, 0x8FA1 }, { 6031, 0x8FA4 },
    { 6032, 0x8FA5 }, { 6033, 0x8FA6 }, { 6034, 0x8FB5 }, { 6035, 0x8FB6 }, { 6036, 0x8FB8 }, { 6037, 0x8FBE }, { 6038, 0x8FC0 }, { 6039, 0x8FC1 },
    { 6040, 0x8FC6 }, { 6041, 0x8FCA }, { 6042, 0x8FCB }, { 6043, 0x8FCD }, { 6044, 0x8FD0 }, { 6045, 0x8FD2 }, { 6046, 0x8FD3 }, { 6047, 0x8FD5 },
    { 6048, 0x8FE0 }, { 6049, 0x8FE3 }, { 6050, 0x8FE4 }, { 6051, 0x8FE8 }, { 6052, 0x8FEE }, { 6053, 0x8FF1 }, { 6054, 0x8FF5 }, { 6055, 0x8FF6 },
    { 6056, 0x8FFB }, { 6057, 0x8FFE }, { 6058, 0x9002 }, { 6059, 0x9004 }, { 6060, 0x9008 }, { 6061, 0x900C }, { 6062, 0x9018 }, { 6063, 0x901B },
    { 6064, 0x9028 }, { 6065, 0x9029 }, { 6066, 0x902F }, { 6067, 0x902A }, { 6068, 0x902C }, { 6069, 0x902D }, { 6070, 0x9033 }, { 6071, 0x9034 },
    { 6072, 0x9037 }, { 6073, 0x903F }, { 6074, 0x9043 }, { 6075, 0x9044 }, { 6076, 0x904C }, { 6077, 0x905B }, { 6078, 0x905D }, { 6079, 0x9062 },
    { 6080, 0x9066 }, { 6081, 0x9067 }, { 6082, 0x906C }, { 6083, 0x9070 }, { 6084, 0x9074 }, { 6085, 0x9079 }, { 6086, 0x9085 }, { 6087, 0x9088 },
    { 6088, 0x908B }, { 6089, 0x908C }, { 6090, 0x908E }, { 6091, 0x9090 }, { 6092, 0x9095 }, { 6093, 0x9097 }, { 6094, 0x9098 }, { 6095, 0x9099 },
    { 6096, 0x909B }, { 6097, 0x90A0 }, { 6098, 0x90A1 }, { 6099, 0x90A2 }, { 6100, 0x90A5 }, { 6101, 0x90B0 }, { 6102, 0x90B2 }, { 6103, 0x90B3 },
    { 6104, 0x90B4 }, { 6105, 0x90B6 }, { 6106, 0x90BD }, { 6107, 0x90CC }, { 6108, 0x90BE }, { 6109, 0x90C3 }, { 6110, 0x90C4 }, { 6111, 0x90C5 },
    { 6112, 0x90C7 }, { 6113, 0x90C8 }, { 6114, 0x90D5 }, { 6115, 0x90D7 }, { 6116, 0x90D8 }, { 6117, 0x90D9 }, { 6118, 0x90DC }, { 6119, 0x90DD },
    { 6120, 0x90DF }, { 6121, 0x90E5 }, { 6122, 0x90D2 }, { 6123, 0x90F6 }, { 6124, 0x90EB }, { 6125, 0x90EF }, { 6126, 0x90F0 }, { 6127, 0x90F4 },
    { 6128, 0x90FE }, { 6129, 0x90FF }, { 6130, 0x9100 }, { 6131, 0x9104 }, { 6132, 0x9105 }, { 6133, 0x9106 }, { 6134, 0x9108 }, { 6135, 0x910D },
    { 6136, 0x9110 }, { 6137, 0x9114 }, { 6138, 0x9116 }, { 6139, 0x9117 }, { 6140, 0x9118 }, { 6141, 0x911A }, { 6142, 0x911C }, { 6143, 0x911E },
    { 6144, 0x9120 }, { 6145, 0x9125 }, { 6146, 0x9122 }, { 6147, 0x9123 }, { 6148, 0x9127 }, { 6149, 0x9129 }, { 6150, 0x912E }, { 6151, 0x912F },
    { 6152, 0x9131 }, { 6153, 0x9134 }, { 6154, 0x9136 }, { 6155, 0x9137 }, { 6156, 0x9139 }, { 6157, 0x913A }, { 6158, 0x913C }, { 6159, 0x913D },
    { 6160, 0x9143 }, { 6161, 0x9147 }, { 6162, 0x9148 }, { 6163, 0x914F }, { 6164, 0x9153 }, { 6165, 0x9157 }, { 6166, 0x9159 }, { 6167, 0x915A },
    { 6168, 0x915B }, { 6169, 0x9161 }, { 6170, 0x9164 }, { 6171, 0x9167 }, { 6172, 0x916D }, { 6173, 0x9174 }, { 6174, 0x9179 }, { 6175, 0x917A },
    { 6176, 0x917B }, { 6177, 0x9181 }, { 6178, 0x9183 }, { 6179, 0x9185 }, { 6180, 0x9186 }, { 6181, 0x918A }, { 6182, 0x918E }, { 6183, 0x9191 },
    { 6184, 0x9193 }, { 6185, 0x9194 }, { 6186, 0x9195 }, { 6187, 0x9198 }, { 6188, 0x919E }, { 6189, 0x91A1 }, { 6190, 0x91A6 }, { 6191, 0x91A8 },
    { 6192, 0x91AC }, { 6193, 0x91AD }, { 6194, 0x91AE }, { 6195, 0x91B0 }, { 6196, 0x91B1 }, { 6197, 0x91B2 }, { 6198, 0x91B3 }, { 6199, 0x91B6 },
    { 6200, 0x91BB }, { 6201, 0x91BC }, { 6202, 0x91BD }, { 6203, 0x91BF }, { 6204, 0x91C2 }, { 6205, 0x91C3 }, { 6206, 0x91C5 }, { 6207, 0x91D3 },
    { 6208, 0x91D4 }, { 6209, 0x91D7 }, { 6210, 0x91D9 }, { 6211, 0x91DA }, { 6212, 0x91DE }, { 6213, 0x91E4 }, { 6214, 0x91E5 }, { 6215, 0x91E9 },
    { 6216, 0x91EA }, { 6217, 0x91EC }, { 6218, 0x91ED }, { 6219, 0x91EE }, { 6220, 0x91EF }, { 6221, 0x91F0 }, { 6222, 0x91F1 }, { 6223, 0x91F7 },
    { 6224, 0x91F9 }, { 6225, 0x91FB }, { 6226, 0x91FD }, { 6227, 0x9200 }, { 6228, 0x9201 }, { 6229, 0x9204 }, { 6230, 0x9205 }, { 6231, 0x9206 },
    { 6232, 0x9207 }, { 6233, 0x9209 }, { 6234, 0x920A }, { 6235, 0x920C }, { 6236, 0x9210 }, { 6237, 0x9212 }, { 6238, 0x9213 }, { 6239, 0x9216 },
    { 6240, 0x9218 }, { 6241, 0x921C }, { 6242, 0x921D }, { 6243, 0x9223 }, { 6244, 0x9224 }, { 6245, 0x9225 }, { 6246, 0x9226 }, { 6247, 0x9228 },
    { 6248, 0x922E }, { 6249, 0x922F }, { 6250, 0x9230 }, { 6251, 0x9233 }, { 6252, 0x9235 }, { 6253, 0x9236 }, { 6254, 0x9238 }, { 6255, 0x9239 },
    { 6256, 0x923A }, { 6257, 0x923C }, { 6258, 0x923E }, { 6259, 0x9240 }, { 6260, 0x9242 }, { 6261, 0x9243 }, { 6262, 0x9246 }, { 6263, 0x9247 },
    { 6264, 0x924A }, { 6265, 0x924D }, { 6266, 0x924E }, { 6267, 0x924F }, { 6268, 0x9251 }, { 6269, 0x9258 }, { 6270, 0x9259 }, { 6271, 0x925C },
    { 6272, 0x925D }, { 6273, 0x9260 }, { 6274, 0x9261 }, { 6275, 0x9265 }, { 6276, 0x9267 }, { 6277, 0x9268 }, { 6278, 0x9269 }, { 6279, 0x926E },
    { 6280, 0x926F }, { 6281, 0x9270 }, { 6282, 0x9275 }, { 6283, 0x9276 }, { 6284, 0x9277 }, { 6285, 0x9278 }, { 6286, 0x9279 }, { 6287, 0x927B },
    { 6288, 0x927C }, { 6289, 0x927D }, { 6290, 0x927F }, { 6291, 0x9288 }, { 6292, 0x9289 }, { 6293, 0x928A }, { 6294, 0x928D }, { 6295, 0x928E },
    { 6296, 0x9292 }, { 6297, 0x9297 }, { 6298, 0x9299 }, { 6299, 0x929F }, { 6300, 0x92A0 }, { 6301, 0x92A4 }, { 6302, 0x92A5 }, { 6303, 0x92A7 },
    { 6304, 0x92A8 }, { 6305, 0x92AB }, { 6306, 0x92AF }, { 6307, 0x92B2 }, { 6308, 0x92B6 }, { 6309, 0x92B8 }, { 6310, 0x92BA }, { 6311, 0x92BB },
    { 6312, 0x92BC }, { 6313, 0x92BD }, { 6314, 0x92BF }, { 6315, 0x92C0 }, { 6316, 0x92C1 }, { 6317, 0x92C2 }, { 6318, 0x92C3 }, { 6319, 0x92C5 },
    { 6320, 0x92C6 }, { 6321, 0x92C7 }, { 6322, 0x92C8 }, { 6323, 0x92CB }, { 6324, 0x92CC }, { 6325, 0x92CD }, { 6326, 0x92CE }, { 6327, 0x92D0 },
    { 6328, 0x92D3 }, { 6329, 0x92D5 }, { 6330, 0x92D7 }, { 6331, 0x92D8 }, { 6332, 0x92D9 }, { 6333, 0x92DC }, { 6334, 0x92DD }, { 6335, 0x92DF },
    { 6336, 0x92E0 }, { 6337, 0x92E1 }, { 6338, 0x92E3 }, { 6339, 0x92E5 }, { 6340, 0x92E7 }, { 6341, 0x92E8 }, { 6342, 0x92EC }, { 6343, 0x92EE },
    { 6344, 0x92F0 }, { 6345, 0x92F9 }, { 6346, 0x92FB }, { 6347, 0x92FF }, { 6348, 0x9300 }, { 6349, 0x9302 }, { 6350, 0x9308 }, { 6351, 0x930D },
    { 6352, 0x9311 }, { 6353, 0x9314 }, { 6354, 0x9315 }, { 6355, 0x931C }, { 6356, 0x931D }, { 6357, 0x931E }, { 6358, 0x931F }, { 6359, 0x9321 },
    { 6360, 0x9324 }, { 6361, 0x9325 }, { 6362, 0x9327 }, { 6363, 0x9329 }, { 6364, 0x932A }, { 6365, 0x9333 }, { 6366, 0x9334 }, { 6367, 0x9336 },
    { 6368, 0x9337 }, { 6369, 0x9347 }, { 6370, 0x9348 }, { 6371, 0x9349 }, { 6372, 0x9350 }, { 6373, 0x9351 }, { 6374, 0x9352 }, { 6375, 0x9355 },
    { 6376, 0x9357 }, { 6377, 0x9358 }, { 6378, 0x935A }, { 6379, 0x935E }, { 6380, 0x9364 }, { 6381, 0x9365 }, { 6382, 0x9367 }, { 6383, 0x9369 },
    { 6384, 0x936A }, { 6385, 0x936D }, { 6386, 0x936F }, { 6387, 0x9370 }, { 6388, 0x9371 }, { 6389, 0x9373 }, { 6390, 0x9374 }, { 6391, 0x9376 },
    { 6392, 0x937A }, { 6393, 0x937D }, { 6394, 0x937F }, { 6395, 0x9380 }, { 6396, 0x9381 }, { 6397, 0x9382 }, { 6398, 0x9388 }, { 6399, 0x938A },
    { 6400, 0x938B }, { 6401, 0x938D }, { 6402, 0x938F }, { 6403, 0x9392 }, { 6404, 0x9395 }, { 6405, 0x9398 }, { 6406, 0x939B }, { 6407, 0x939E },
    { 6408, 0x93A1 }, { 6409, 0x93A3 }, { 6410, 0x93A4 }, { 6411, 0x93A6 }, { 6412, 0x93A8 }, { 6413, 0x93AB }, { 6414, 0x93B4 }, { 6415, 0x93B5 },
    { 6416, 0x93B6 }, { 6417, 0x93BA }, { 6418, 0x93A9 }, { 6419, 0x93C1 }, { 6420, 0x93C4 }, { 6421, 0x93C5 }, { 6422, 0x93C6 }, { 6423, 0x93C7 },
    { 6424, 0x93C9 }, { 6425, 0x93CA }, { 6426, 0x93CB }, { 6427, 0x93CC }, { 6428, 0x93CD }, { 6429, 0x93D3 }, { 6430, 0x93D9 }, { 6431, 0x93DC },
    { 6432, 0x93DE }, { 6433, 0x93DF }, { 6434, 0x93E2 }, { 6435, 0x93E6 }, { 6436, 0x93E7 }, { 6437, 0x93F9 }, { 6438, 0x93F7 }, { 6439, 0x93F8 },
    { 6440, 0x93FA }, { 6441, 0x93FB }, { 6442, 0x93FD }, { 6443, 0x9401 }, { 6444, 0x9402 }, { 6445, 0x9404 }, { 6446, 0x9408 }, { 6447, 0x9409 },
    { 6448, 0x940D }, { 6449, 0x940E }, { 6450, 0x940F }, { 6451, 0x9415 }, { 6452, 0x9416 }, { 6453, 0x9417 }, { 6454, 0x941F }, { 6455, 0x942E },
    { 6456, 0x942F }, { 6457, 0x9431 }, { 6458, 0x9432 }, { 6459, 0x9433 }, { 6460, 0x9434 }, { 6461, 0x943B }, { 6462, 0x943F }, { 6463, 0x943D },
    { 6464, 0x9443 }, { 6465, 0x9445 }, { 6466, 0x9448 }, { 6467, 0x944A }, { 6468, 0x944C }, { 6469, 0x9455 }, { 6470, 0x9459 }, { 6471, 0x945C },
    { 6472, 0x945F }, { 6473, 0x9461 }, { 6474, 0x9463 }, { 6475, 0x9468 }, { 6476, 0x946B }, { 6477, 0x946D }, { 6478, 0x946E }, { 6479, 0x946F },
    { 6480, 0x9471 }, { 6481, 0x9472 }, { 6482, 0x9484 }, { 6483, 0x9483 }, { 6484, 0x9578 }, { 6485, 0x9579 }, { 6486, 0x957E }, { 6487, 0x9584 },
    { 6488, 0x9588 }, { 6489, 0x958C }, { 6490, 0x958D }, { 6491, 0x958E }, { 6492, 0x959D }, { 6493, 0x959E }, { 6494, 0x959F }, { 6495, 0x95A1 },
    { 6496, 0x95A6 }, { 6497, 0x95A9 }, { 6498, 0x95AB }, { 6499, 0x95AC }, { 6500, 0x95B4 }, { 6501, 0x95B6 }, { 6502, 0x95BA }, { 6503, 0x95BD },
    { 6504, 0x95BF }, { 6505, 0x95C6 }, { 6506, 0x95C8 }, { 6507, 0x95C9 }, { 6508, 0x95CB }, { 6509, 0x95D0 }, { 6510, 0x95D1 }, { 6511, 0x95D2 },
    { 6512, 0x95D3 }, { 6513, 0x95D9 }, { 6514, 0x95DA }, { 6515, 0x95DD }, { 6516, 0x95DE }, { 6517, 0x95DF }, { 6518, 0x95E0 }, { 6519, 0x95E4 },
    { 6520, 0x95E6 }, { 6521, 0x961D }, { 6522, 0x961E }, { 6523, 0x9622 }, { 6524, 0x9624 }, { 6525, 0x9625 }, { 6526, 0x9626 }, { 6527, 0x962C },
    { 6528, 0x9631 }, { 6529, 0x9633 }, { 6530, 0x9637 }, { 6531, 0x9638 }, { 6532, 0x9639 }, { 6533, 0x963A }, { 6534, 0x963C }, { 6535, 0x963D },
    { 6536, 0x9641 }, { 6537, 0x9652 }, { 6538, 0x9654 }, { 6539, 0x9656 }, { 6540, 0x9657 }, { 6541, 0x9658 }, { 6542, 0x9661 }, { 6543, 0x966E },
    { 6544, 0x9674 }, { 6545, 0x967B }, { 6546, 0x967C }, { 6547, 0x967E }, { 6548, 0x967F }, { 6549, 0x9681 }, { 6550, 0x9682 }, { 6551, 0x9683 },
    { 6552, 0x9684 }, { 6553, 0x9689 }, { 6554, 0x9691 }, { 6555, 0x9696 }, { 6556, 0x969A }, { 6557, 0x969D }, { 6558, 0x969F }, { 6559, 0x96A4 },
    { 6560, 0x96A5 }, { 6561, 0x96A6 }, { 6562, 0x96A9 }, { 6563, 0x96AE }, { 6564, 0x96AF }, { 6565, 0x96B3 }, { 6566, 0x96BA }, { 6567, 0x96CA },
    { 6568, 0x96D2 }, { 6569, 0x5DB2 }, { 6570, 0x96D8 }, { 6571, 0x96DA }, { 6572, 0x96DD }, { 6573, 0x96DE }, { 6574, 0x96DF }, { 6575, 0x96E9 },
    { 6576, 0x96EF }, { 6577, 0x96F1 }, { 6578, 0x96FA }, { 6579, 0x9702 }, { 6580, 0x9703 }, { 6581, 0x9705 }, { 6582, 0x9709 }, { 6583, 0x971A },
    { 6584, 0x971B }, { 6585, 0x971D }, { 6586, 0x9721 }, { 6587, 0x9722 }, { 6588, 0x9723 }, { 6589, 0x9728 }, { 6590, 0x9731 }, { 6591, 0x9733 },
    { 6592, 0x9741 }, { 6593, 0x9743 }, { 6594, 0x974A }, { 6595, 0x974E }, { 6596, 0x974F }, { 6597, 0x9755 }, { 6598, 0x9757 }, { 6599, 0x9758 },
    { 6600, 0x975A }, { 6601, 0x975B }, { 6602, 0x9763 }, { 6603, 0x9767 }, { 6604, 0x976A }, { 6605, 0x976E }, { 6606, 0x9773 }, { 6607, 0x9776 },
    { 6608, 0x9777 }, { 6609, 0x9778 }, { 6610, 0x977B }, { 6611, 0x977D }, { 6612, 0x977F }, { 6613, 0x9780 }, { 6614, 0x9789 }, { 6615, 0x9795 },
    { 6616, 0x9796 }, { 6617, 0x9797 }, { 6618, 0x9799 }, { 6619, 0x979A }, { 6620, 0x979E }, { 6621, 0x979F }, { 6622, 0x97A2 }, { 6623, 0x97AC },
    { 6624, 0x97AE }, { 6625, 0x97B1 }, { 6626, 0x97B2 }, { 6627, 0x97B5 }, { 6628, 0x97B6 }, { 6629, 0x97B8 }, { 6630, 0x97B9 }, { 6631, 0x97BA },
    { 6632, 0x97BC }, { 6633, 0x97BE }, { 6634, 0x97BF }, { 6635, 0x97C1 }, { 6636, 0x97C4 }, { 6637, 0x97C5 }, { 6638, 0x97C7 }, { 6639, 0x97C9 },
    { 6640, 0x97CA }, { 6641, 0x97CC }, { 6642, 0x97CD }, { 6643, 0x97CE }, { 6644, 0x97D0 }, { 6645, 0x97D1 }, { 6646, 0x97D4 }, { 6647, 0x97D7 },
    { 6648, 0x97D8 }, { 6649, 0x97D9 }, { 6650, 0x97DD }, { 6651, 0x97DE }, { 6652, 0x97E0 }, { 6653, 0x97DB }, { 6654, 0x97E1 }, { 6655, 0x97E4 },
    { 6656, 0x97EF }, { 6657, 0x97F1 }, { 6658, 0x97F4 }, { 6659, 0x97F7 }, { 6660, 0x97F8 }, { 6661, 0x97FA }, { 6662, 0x9807 }, { 6663, 0x980A },
    { 6664, 0x9819 }, { 6665, 0x980D }, { 6666, 0x980E }, { 6667, 0x9814 }, { 6668, 0x9816 }, { 6669, 0x981C }, { 6670, 0x981E }, { 6671, 0x9820 },
    { 6672, 0x9823 }, { 6673, 0x9826 }, { 6674, 0x982B }, { 6675, 0x982E }, { 6676, 0x982F }, { 6677, 0x9830 }, { 6678, 0x9832 }, { 6679, 0x9833 },
    { 6680, 0x9835 }, { 6681, 0x9825 }, { 6682, 0x983E }, { 6683, 0x9844 }, { 6684, 0x9847 }, { 6685, 0x984A }, { 6686, 0x9851 }, { 6687, 0x9852 },
    { 6688, 0x9853 }, { 6689, 0x9856 }, { 6690, 0x9857 }, { 6691, 0x9859 }, { 6692, 0x985A }, { 6693, 0x9862 }, { 6694, 0x9863 }, { 6695, 0x9865 },
    { 6696, 0x9866 }, { 6697, 0x986A }, { 6698, 0x986C }, { 6699, 0x98AB }, { 6700, 0x98AD }, { 6701, 0x98AE }, { 6702, 0x98B0 }, { 6703, 0x98B4 },
    { 6704, 0x98B7 }, { 6705, 0x98B8 }, { 6706, 0x98BA }, { 6707, 0x98BB }, { 6708, 0x98BF }, { 6709, 0x98C2 }, { 6710, 0x98C5 }, { 6711, 0x98C8 },
    { 6712, 0x98CC }, { 6713, 0x98E1 }, { 6714, 0x98E3 }, { 6715, 0x98E5 }, { 6716, 0x98E6 }, { 6717, 0x98E7 }, { 6718, 0x98EA }, { 6719, 0x98F3 },
    { 6720, 0x98F6 }, { 6721, 0x9902 }, { 6722, 0x9907 }, { 6723, 0x9908 }, { 6724, 0x9911 }, { 6725, 0x9915 }, { 6726, 0x9916 }, { 6727, 0x9917 },
    { 6728, 0x991A }, { 6729, 0x991B }, { 6730, 0x991C }, { 6731, 0x991F }, { 6732, 0x9922 }, { 6733, 0x9926 }, { 6734, 0x9927 }, { 6735, 0x992B },
    { 6736, 0x9931 }, { 6737, 0x9932 }, { 6738, 0x9933 }, { 6739, 0x9934 }, { 6740, 0x9935 }, { 6741, 0x9939 }, { 6742, 0x993A }, { 6743, 0x993B },
    { 6744, 0x993C }, { 6745, 0x9940 }, { 6746, 0x9941 }, { 6747, 0x9946 }, { 6748, 0x9947 }, { 6749, 0x9948 }, { 6750, 0x994D }, { 6751, 0x994E },
    { 6752, 0x9954 }, { 6753, 0x9958 }, { 6754, 0x9959 }, { 6755, 0x995B }, { 6756, 0x995C }, { 6757, 0x995E }, { 6758, 0x995F }, { 6759, 0x9960 },
    { 6760, 0x999B }, { 6761, 0x999D }, { 6762, 0x999F }, { 6763, 0x99A6 }, { 6764, 0x99B0 }, { 6765, 0x99B1 }, { 6766, 0x99B2 }, { 6767, 0x99B5 },
    { 6768, 0x99B9 }, { 6769, 0x99BA }, { 6770, 0x99BD }, { 6771, 0x99BF }, { 6772, 0x99C3 }, { 6773, 0x99C9 }, { 6774, 0x99D3 }, { 6775, 0x99D4 },
    { 6776, 0x99D9 }, { 6777, 0x99DA }, { 6778, 0x99DC }, { 6779, 0x99DE }, { 6780, 0x99E7 }, { 6781, 0x99EA }, { 6782, 0x99EB }, { 6783, 0x99EC },
    { 6784, 0x99F0 }, { 6785, 0x99F4 }, { 6786, 0x99F5 }, { 6787, 0x99F9 }, { 6788, 0x99FD }, { 6789, 0x99FE }, { 6790, 0x9A02 }, { 6791, 0x9A03 },
    { 6792, 0x9A04 }, { 6793, 0x9A0B }, { 6794, 0x9A0C }, { 6795, 0x9A10 }, { 6796, 0x9A11 }, { 6797, 0x9A16 }, { 6798, 0x9A1E }, { 6799, 0x9A20 },
    { 6800, 0x9A22 }, { 6801, 0x9A23 }, { 6802, 0x9A24 }, { 6803, 0x9A27 }, { 6804, 0x9A2D }, { 6805, 0x9A2E }, { 6806, 0x9A33 }, { 6807, 0x9A35 },
    { 6808, 0x9A36 }, { 6809, 0x9A38 }, { 6810, 0x9A47 }, { 6811, 0x9A41 }, { 6812, 0x9A44 }, { 6813, 0x9A4A }, { 6814, 0x9A4B }, { 6815, 0x9A4C },
    { 6816, 0x9A4E }, { 6817, 0x9A51 }, { 6818, 0x9A54 }, { 6819, 0x9A56 }, { 6820, 0x9A5D }, { 6821, 0x9AAA }, { 6822, 0x9AAC }, { 6823, 0x9AAE },
    { 6824, 0x9AAF }, { 6825, 0x9AB2 }, { 6826, 0x9AB4 }, { 6827, 0x9AB5 }, { 6828, 0x9AB6 }, { 6829, 0x9AB9 }, { 6830, 0x9ABB }, { 6831, 0x9ABE },
    { 6832, 0x9ABF }, { 6833, 0x9AC1 }, { 6834, 0x9AC3 }, { 6835, 0x9AC6 }, { 6836, 0x9AC8 }, { 6837, 0x9ACE }, { 6838, 0x9AD0 }, { 6839, 0x9AD2 },
    { 6840, 0x9AD5 }, { 6841, 0x9AD6 }, { 6842, 0x9AD7 }, { 6843, 0x9ADB }, { 6844, 0x9ADC }, { 6845, 0x9AE0 }, { 6846, 0x9AE4 }, { 6847, 0x9AE5 },
    { 6848, 0x9AE7 }, { 6849, 0x9AE9 }, { 6850, 0x9AEC }, { 6851, 0x9AF2 }, { 6852, 0x9AF3 }, { 6853, 0x9AF5 }, { 6854, 0x9AF9 }, { 6855, 0x9AFA },
    { 6856, 0x9AFD }, { 6857, 0x9AFF }, { 6858, 0x9B00 }, { 6859, 0x9B01 }, { 6860, 0x9B02 }, { 6861, 0x9B03 }, { 6862, 0x9B04 }, { 6863, 0x9B05 },
    { 6864, 0x9B08 }, { 6865, 0x9B09 }, { 6866, 0x9B0B }, { 6867, 0x9B0C }, { 6868, 0x9B0D }, { 6869, 0x9B0E }, { 6870, 0x9B10 }, { 6871, 0x9B12 },
    { 6872, 0x9B16 }, { 6873, 0x9B19 }, { 6874, 0x9B1B }, { 6875, 0x9B1C }, { 6876, 0x9B20 }, { 6877, 0x9B26 }, { 6878, 0x9B2B }, { 6879, 0x9B2D },
    { 6880, 0x9B33 }, { 6881, 0x9B34 }, { 6882, 0x9B35 }, { 6883, 0x9B37 }, { 6884, 0x9B39 }, { 6885, 0x9B3A }, { 6886, 0x9B3D }, { 6887, 0x9B48 },
    { 6888, 0x9B4B }, { 6889, 0x9B4C }, { 6890, 0x9B55 }, { 6891, 0x9B56 }, { 6892, 0x9B57 }, { 6893, 0x9B5B }, { 6894, 0x9B5E }, { 6895, 0x9B61 },
    { 6896, 0x9B63 }, { 6897, 0x9B65 }, { 6898, 0x9B66 }, { 6899, 0x9B68 }, { 6900, 0x9B6A }, { 6901, 0x9B6B }, { 6902, 0x9B6C }, { 6903, 0x9B6D },
    { 6904, 0x9B6E }, { 6905, 0x9B73 }, { 6906, 0x9B75 }, { 6907, 0x9B77 }, { 6908, 0x9B78 }, { 6909, 0x9B79 }, { 6910, 0x9B7F }, { 6911, 0x9B80 },
    { 6912, 0x9B84 }, { 6913, 0x9B85 }, { 6914, 0x9B86 }, { 6915, 0x9B87 }, { 6916, 0x9B89 }, { 6917, 0x9B8A }, { 6918, 0x9B8B }, { 6919, 0x9B8D },
    { 6920, 0x9B8F }, { 6921, 0x9B90 }, { 6922, 0x9B94 }, { 6923, 0x9B9A }, { 6924, 0x9B9D }, { 6925, 0x9B9E }, { 6926, 0x9BA6 }, { 6927, 0x9BA7 },
    { 6928, 0x9BA9 }, { 6929, 0x9BAC }, { 6930, 0x9BB0 }, { 6931, 0x9BB1 }, { 6932, 0x9BB2 }, { 6933, 0x9BB7 }, { 6934, 0x9BB8 }, { 6935, 0x9BBB },
    { 6936, 0x9BBC }, { 6937, 0x9BBE }, { 6938, 0x9BBF }, { 6939, 0x9BC1 }, { 6940, 0x9BC7 }, { 6941, 0x9BC8 }, { 6942, 0x9BCE }, { 6943, 0x9BD0 },
    { 6944, 0x9BD7 }, { 6945, 0x9BD8 }, { 6946, 0x9BDD }, { 6947, 0x9BDF }, { 6948, 0x9BE5 }, { 6949, 0x9BE7 }, { 6950, 0x9BEA }, { 6951, 0x9BEB },
    { 6952, 0x9BEF }, { 6953, 0x9BF3 }, { 6954, 0x9BF7 }, { 6955, 0x9BF8 }, { 6956, 0x9BF9 }, { 6957, 0x9BFA }, { 6958, 0x9BFD }, { 6959, 0x9BFF },
    { 6960, 0x9C00 }, { 6961, 0x9C02 }, { 6962, 0x9C0B }, { 6963, 0x9C0F }, { 6964, 0x9C11 }, { 6965, 0x9C16 }, { 6966, 0x9C18 }, { 6967, 0x9C19 },
    { 6968, 0x9C1A }, { 6969, 0x9C1C }, { 6970, 0x9C1E }, { 6971, 0x9C22 }, { 6972, 0x9C23 }, { 6973, 0x9C26 }, { 6974, 0x9C27 }, { 6975, 0x9C28 },
    { 6976, 0x9C29 }, { 6977, 0x9C2A }, { 6978, 0x9C31 }, { 6979, 0x9C35 }, { 6980, 0x9C36 }, { 6981, 0x9C37 }, { 6982, 0x9C3D }, { 6983, 0x9C41 },
    { 6984, 0x9C43 }, { 6985, 0x9C44 }, { 6986, 0x9C45 }, { 6987, 0x9C49 }, { 6988, 0x9C4A }, { 6989, 0x9C4E }, { 6990, 0x9C4F }, { 6991, 0x9C50 },
    { 6992, 0x9C53 }, { 6993, 0x9C54 }, { 6994, 0x9C56 }, { 6995, 0x9C58 }, { 6996, 0x9C5B }, { 6997, 0x9C5D }, { 6998, 0x9C5E }, { 6999, 0x9C5F },
    { 7000, 0x9C63 }, { 7001, 0x9C69 }, { 7002, 0x9C6A }, { 7003, 0x9C5C }, { 7004, 0x9C6B }, { 7005, 0x9C68 }, { 7006, 0x9C6E }, { 7007, 0x9C70 },
    { 7008, 0x9C72 }, { 7009, 0x9C75 }, { 7010, 0x9C77 }, { 7011, 0x9C7B }, { 7012, 0x9CE6 }, { 7013, 0x9CF2 }, { 7014, 0x9CF7 }, { 7015, 0x9CF9 },
    { 7016, 0x9D0B }, { 7017, 0x9D02 }, { 7018, 0x9D11 }, { 7019, 0x9D17 }, { 7020, 0x9D18 }, { 7021, 0x9D1C }, { 7022, 0x9D1D }, { 7023, 0x9D1E },
    { 7024, 0x9D2F }, { 7025, 0x9D30 }, { 7026, 0x9D32 }, { 7027, 0x9D33 }, { 7028, 0x9D34 }, { 7029, 0x9D3A }, { 7030, 0x9D3C }, { 7031, 0x9D45 },
    { 7032, 0x9D3D }, { 7033, 0x9D42 }, { 7034, 0x9D43 }, { 7035, 0x9D47 }, { 7036, 0x9D4A }, { 7037, 0x9D53 }, { 7038, 0x9D54 }, { 7039, 0x9D5F },
    { 7040, 0x9D63 }, { 7041, 0x9D62 }, { 7042, 0x9D65 }, { 7043, 0x9D69 }, { 7044, 0x9D6A }, { 7045, 0x9D6B }, { 7046, 0x9D70 }, { 7047, 0x9D76 },
    { 7048, 0x9D77 }, { 7049, 0x9D7B }, { 7050, 0x9D7C }, { 7051, 0x9D7E }, { 7052, 0x9D83 }, { 7053, 0x9D84 }, { 7054, 0x9D86 }, { 7055, 0x9D8A },
    { 7056, 0x9D8D }, { 7057, 0x9D8E }, { 7058, 0x9D92 }, { 7059, 0x9D93 }, { 7060, 0x9D95 }, { 7061, 0x9D96 }, { 7062, 0x9D97 }, { 7063, 0x9D98 },
    { 7064, 0x9DA1 }, { 7065, 0x9DAA }, { 7066, 0x9DAC }, { 7067, 0x9DAE }, { 7068, 0x9DB1 }, { 7069, 0x9DB5 }, { 7070, 0x9DB9 }, { 7071, 0x9DBC },
    { 7072, 0x9DBF }, { 7073, 0x9DC3 }, { 7074, 0x9DC7 }, { 7075, 0x9DC9 }, { 7076, 0x9DCA }, { 7077, 0x9DD4 }, { 7078, 0x9DD5 }, { 7079, 0x9DD6 },
    { 7080, 0x9DD7 }, { 7081, 0x9DDA }, { 7082, 0x9DDE }, { 7083, 0x9DDF }, { 7084, 0x9DE0 }, { 7085, 0x9DE5 }, { 7086, 0x9DE7 }, { 7087, 0x9DE9 },
    { 7088, 0x9DEB }, { 7089, 0x9DEE }, { 7090, 0x9DF0 }, { 7091, 0x9DF3 }, { 7092, 0x9DF4 }, { 7093, 0x9DFE }, { 7094, 0x9E0A }, { 7095, 0x9E02 },
    { 7096, 0x9E07 }, { 7097, 0x9E0E }, { 7098, 0x9E10 }, { 7099, 0x9E11 }, { 7100, 0x9E12 }, { 7101, 0x9E15 }, { 7102, 0x9E16 }, { 7103, 0x9E19 },
    { 7104, 0x9E1C }, { 7105, 0x9E1D }, { 7106, 0x9E7A }, { 7107, 0x9E7B }, { 7108, 0x9E7C }, { 7109, 0x9E80 }, { 7110, 0x9E82 }, { 7111, 0x9E83 },
    { 7112, 0x9E84 }, { 7113, 0x9E85 }, { 7114, 0x9E87 }, { 7115, 0x9E8E }, { 7116, 0x9E8F }, { 7117, 0x9E96 }, { 7118, 0x9E98 }, { 7119, 0x9E9B },
    { 7120, 0x9E9E }, { 7121, 0x9EA4 }, { 7122, 0x9EA8 }, { 7123, 0x9EAC }, { 7124, 0x9EAE }, { 7125, 0x9EAF }, { 7126, 0x9EB0 }, { 7127, 0x9EB3 },
    { 7128, 0x9EB4 }, { 7129, 0x9EB5 }, { 7130, 0x9EC6 }, { 7131, 0x9EC8 }, { 7132, 0x9ECB }, { 7133, 0x9ED5 }, { 7134, 0x9EDF }, { 7135, 0x9EE4 },
    { 7136, 0x9EE7 }, { 7137, 0x9EEC }, { 7138, 0x9EED }, { 7139, 0x9EEE }, { 7140, 0x9EF0 }, { 7141, 0x9EF1 }, { 7142, 0x9EF2 }, { 7143, 0x9EF5 },
    { 7144, 0x9EF8 }, { 7145, 0x9EFF }, { 7146, 0x9F02 }, { 7147, 0x9F03 }, { 7148, 0x9F09 }, { 7149, 0x9F0F }, { 7150, 0x9F10 }, { 7151, 0x9F11 },
    { 7152, 0x9F12 }, { 7153, 0x9F14 }, { 7154, 0x9F16 }, { 7155, 0x9F17 }, { 7156, 0x9F19 }, { 7157, 0x9F1A }, { 7158, 0x9F1B }, { 7159, 0x9F1F },
    { 7160, 0x9F22 }, { 7161, 0x9F26 }, { 7162, 0x9F2A }, { 7163, 0x9F2B }, { 7164, 0x9F2F }, { 7165, 0x9F31 }, { 7166, 0x9F32 }, { 7167, 0x9F34 },
    { 7168, 0x9F37 }, { 7169, 0x9F39 }, { 7170, 0x9F3A }, { 7171, 0x9F3C }, { 7172, 0x9F3D }, { 7173, 0x9F3F }, { 7174, 0x9F41 }, { 7175, 0x9F43 },
    { 7176, 0x9F44 }, { 7177, 0x9F45 }, { 7178, 0x9F46 }, { 7179, 0x9F47 }, { 7180, 0x9F53 }, { 7181, 0x9F55 }, { 7182, 0x9F56 }, { 7183, 0x9F57 },
    { 7184, 0x9F58 }, { 7185, 0x9F5A }, { 7186, 0x9F5D }, { 7187, 0x9F5E }, { 7188, 0x9F68 }, { 7189, 0x9F69 }, { 7190, 0x9F6D }, { 7191, 0x9F6E },
    { 7192, 0x9F6F }, { 7193, 0x9F70 }, { 7194, 0x9F71 }, { 7195, 0x9F73 }, { 7196, 0x9F75 }, { 7197, 0x9F7A }, { 7198, 0x9F7D }, { 7199, 0x9F8F },
    { 7200, 0x9F90 }, { 7201, 0x9F91 }, { 7202, 0x9F92 }, { 7203, 0x9F94 }, { 7204, 0x9F96 }, { 7205, 0x9F97 }, { 7206, 0x9F9E }, { 7207, 0x9FA1 },
    { 7208, 0x9FA2 }, { 7209, 0x9FA3 }, { 7210, 0x9FA5 }
} };
#endif // ASSERT_ENABLED

const std::array<std::pair<uint16_t, UChar>, 6067>& jis0212()
{
    // Allocate this at runtime because building it at compile time would make the binary much larger and this is often not used.
    static std::array<std::pair<uint16_t, UChar>, 6067>* array;
    static std::once_flag flag;
    std::call_once(flag, [] {
        array = new std::array<std::pair<uint16_t, UChar>, 6067>();
        size_t arrayIndex = 0;
        
        UErrorCode error = U_ZERO_ERROR;
        auto icuConverter = ICUConverterPtr { ucnv_open("EUC-JP", &error) };
        ASSERT(!error);

        constexpr size_t range = 94;
        uint8_t icuInput[3];
        UChar icuOutput;
        for (size_t i = 0; i < range; i++) {
            for (size_t j = 0; j < range; j++) {
                icuInput[0] = 0x8F;
                icuInput[1] = 0xA1 + i;
                icuInput[2] = 0xA1 + j;

                UChar* output = &icuOutput;
                const char* input = reinterpret_cast<const char*>(icuInput);
                ucnv_toUnicode(icuConverter.get(), &output, output + 1, &input, input + sizeof(icuInput), nullptr, true, &error);
                ASSERT(!error);
                if (icuOutput != 0xFFFD) {
                    uint16_t pointer = i * range + j;
                    // ICU has some pointers above 7708 that are not in the encoding standard.
                    if (pointer < 7708)
                        (*array)[arrayIndex++] = { pointer, icuOutput };
                }
            }
        }

        RELEASE_ASSERT_WITH_SECURITY_IMPLICATION(arrayIndex == 6067);
        ASSERT(*array == jis0212Reference);
    });
    return *array;
}

#if ASSERT_ENABLED
// From https://encoding.spec.whatwg.org/index-big5.txt
std::array<std::pair<uint16_t, UChar32>, 18590> big5Reference { {
    { 942, 0x43F0 }, { 943, 0x4C32 }, { 944, 0x4603 }, { 945, 0x45A6 }, { 946, 0x4578 }, { 947, 0x27267 }, { 948, 0x4D77 }, { 949, 0x45B3 },
    { 950, 0x27CB1 }, { 951, 0x4CE2 }, { 952, 0x27CC5 }, { 953, 0x3B95 }, { 954, 0x4736 }, { 955, 0x4744 }, { 956, 0x4C47 }, { 957, 0x4C40 },
    { 958, 0x242BF }, { 959, 0x23617 }, { 960, 0x27352 }, { 961, 0x26E8B }, { 962, 0x270D2 }, { 963, 0x4C57 }, { 964, 0x2A351 }, { 965, 0x474F },
    { 966, 0x45DA }, { 967, 0x4C85 }, { 968, 0x27C6C }, { 969, 0x4D07 }, { 970, 0x4AA4 }, { 971, 0x46A1 }, { 972, 0x26B23 }, { 973, 0x7225 },
    { 974, 0x25A54 }, { 975, 0x21A63 }, { 976, 0x23E06 }, { 977, 0x23F61 }, { 978, 0x664D }, { 979, 0x56FB }, { 981, 0x7D95 }, { 982, 0x591D },
    { 983, 0x28BB9 }, { 984, 0x3DF4 }, { 985, 0x9734 }, { 986, 0x27BEF }, { 987, 0x5BDB }, { 988, 0x21D5E }, { 989, 0x5AA4 }, { 990, 0x3625 },
    { 991, 0x29EB0 }, { 992, 0x5AD1 }, { 993, 0x5BB7 }, { 994, 0x5CFC }, { 995, 0x676E }, { 996, 0x8593 }, { 997, 0x29945 }, { 998, 0x7461 },
    { 999, 0x749D }, { 1000, 0x3875 }, { 1001, 0x21D53 }, { 1002, 0x2369E }, { 1003, 0x26021 }, { 1004, 0x3EEC }, { 1005, 0x258DE }, { 1006, 0x3AF5 },
    { 1007, 0x7AFC }, { 1008, 0x9F97 }, { 1009, 0x24161 }, { 1010, 0x2890D }, { 1011, 0x231EA }, { 1012, 0x20A8A }, { 1013, 0x2325E }, { 1014, 0x430A },
    { 1015, 0x8484 }, { 1016, 0x9F96 }, { 1017, 0x942F }, { 1018, 0x4930 }, { 1019, 0x8613 }, { 1020, 0x5896 }, { 1021, 0x974A }, { 1022, 0x9218 },
    { 1023, 0x79D0 }, { 1024, 0x7A32 }, { 1025, 0x6660 }, { 1026, 0x6A29 }, { 1027, 0x889D }, { 1028, 0x744C }, { 1029, 0x7BC5 }, { 1030, 0x6782 },
    { 1031, 0x7A2C }, { 1032, 0x524F }, { 1033, 0x9046 }, { 1034, 0x34E6 }, { 1035, 0x73C4 }, { 1036, 0x25DB9 }, { 1037, 0x74C6 }, { 1038, 0x9FC7 },
    { 1039, 0x57B3 }, { 1040, 0x492F }, { 1041, 0x544C }, { 1042, 0x4131 }, { 1043, 0x2368E }, { 1044, 0x5818 }, { 1045, 0x7A72 }, { 1046, 0x27B65 },
    { 1047, 0x8B8F }, { 1048, 0x46AE }, { 1049, 0x26E88 }, { 1050, 0x4181 }, { 1051, 0x25D99 }, { 1052, 0x7BAE }, { 1053, 0x224BC }, { 1054, 0x9FC8 },
    { 1055, 0x224C1 }, { 1056, 0x224C9 }, { 1057, 0x224CC }, { 1058, 0x9FC9 }, { 1059, 0x8504 }, { 1060, 0x235BB }, { 1061, 0x40B4 }, { 1062, 0x9FCA },
    { 1063, 0x44E1 }, { 1064, 0x2ADFF }, { 1065, 0x62C1 }, { 1066, 0x706E }, { 1067, 0x9FCB }, { 1099, 0x31C0 }, { 1100, 0x31C1 }, { 1101, 0x31C2 },
    { 1102, 0x31C3 }, { 1103, 0x31C4 }, { 1104, 0x2010C }, { 1105, 0x31C5 }, { 1106, 0x200D1 }, { 1107, 0x200CD }, { 1108, 0x31C6 }, { 1109, 0x31C7 },
    { 1110, 0x200CB }, { 1111, 0x21FE8 }, { 1112, 0x31C8 }, { 1113, 0x200CA }, { 1114, 0x31C9 }, { 1115, 0x31CA }, { 1116, 0x31CB }, { 1117, 0x31CC },
    { 1118, 0x2010E }, { 1119, 0x31CD }, { 1120, 0x31CE }, { 1121, 0x0100 }, { 1122, 0x00C1 }, { 1123, 0x01CD }, { 1124, 0x00C0 }, { 1125, 0x0112 },
    { 1126, 0x00C9 }, { 1127, 0x011A }, { 1128, 0x00C8 }, { 1129, 0x014C }, { 1130, 0x00D3 }, { 1131, 0x01D1 }, { 1132, 0x00D2 }, { 1134, 0x1EBE },
    { 1136, 0x1EC0 }, { 1137, 0x00CA }, { 1138, 0x0101 }, { 1139, 0x00E1 }, { 1140, 0x01CE }, { 1141, 0x00E0 }, { 1142, 0x0251 }, { 1143, 0x0113 },
    { 1144, 0x00E9 }, { 1145, 0x011B }, { 1146, 0x00E8 }, { 1147, 0x012B }, { 1148, 0x00ED }, { 1149, 0x01D0 }, { 1150, 0x00EC }, { 1151, 0x014D },
    { 1152, 0x00F3 }, { 1153, 0x01D2 }, { 1154, 0x00F2 }, { 1155, 0x016B }, { 1156, 0x00FA }, { 1157, 0x01D4 }, { 1158, 0x00F9 }, { 1159, 0x01D6 },
    { 1160, 0x01D8 }, { 1161, 0x01DA }, { 1162, 0x01DC }, { 1163, 0x00FC }, { 1165, 0x1EBF }, { 1167, 0x1EC1 }, { 1168, 0x00EA }, { 1169, 0x0261 },
    { 1170, 0x23DA }, { 1171, 0x23DB }, { 1256, 0x2A3A9 }, { 1257, 0x21145 }, { 1259, 0x650A }, { 1262, 0x4E3D }, { 1263, 0x6EDD }, { 1264, 0x9D4E },
    { 1265, 0x91DF }, { 1268, 0x27735 }, { 1269, 0x6491 }, { 1270, 0x4F1A }, { 1271, 0x4F28 }, { 1272, 0x4FA8 }, { 1273, 0x5156 }, { 1274, 0x5174 },
    { 1275, 0x519C }, { 1276, 0x51E4 }, { 1277, 0x52A1 }, { 1278, 0x52A8 }, { 1279, 0x533B }, { 1280, 0x534E }, { 1281, 0x53D1 }, { 1282, 0x53D8 },
    { 1283, 0x56E2 }, { 1284, 0x58F0 }, { 1285, 0x5904 }, { 1286, 0x5907 }, { 1287, 0x5932 }, { 1288, 0x5934 }, { 1289, 0x5B66 }, { 1290, 0x5B9E },
    { 1291, 0x5B9F }, { 1292, 0x5C9A }, { 1293, 0x5E86 }, { 1294, 0x603B }, { 1295, 0x6589 }, { 1296, 0x67FE }, { 1297, 0x6804 }, { 1298, 0x6865 },
    { 1299, 0x6D4E }, { 1300, 0x70BC }, { 1301, 0x7535 }, { 1302, 0x7EA4 }, { 1303, 0x7EAC }, { 1304, 0x7EBA }, { 1305, 0x7EC7 }, { 1306, 0x7ECF },
    { 1307, 0x7EDF }, { 1308, 0x7F06 }, { 1309, 0x7F37 }, { 1310, 0x827A }, { 1311, 0x82CF }, { 1312, 0x836F }, { 1313, 0x89C6 }, { 1314, 0x8BBE },
    { 1315, 0x8BE2 }, { 1316, 0x8F66 }, { 1317, 0x8F67 }, { 1318, 0x8F6E }, { 1319, 0x7411 }, { 1320, 0x7CFC }, { 1321, 0x7DCD }, { 1322, 0x6946 },
    { 1323, 0x7AC9 }, { 1324, 0x5227 }, { 1329, 0x918C }, { 1330, 0x78B8 }, { 1331, 0x915E }, { 1332, 0x80BC }, { 1334, 0x8D0B }, { 1335, 0x80F6 },
    { 1336, 0x209E7 }, { 1339, 0x809F }, { 1340, 0x9EC7 }, { 1341, 0x4CCD }, { 1342, 0x9DC9 }, { 1343, 0x9E0C }, { 1344, 0x4C3E }, { 1345, 0x29DF6 },
    { 1346, 0x2700E }, { 1347, 0x9E0A }, { 1348, 0x2A133 }, { 1349, 0x35C1 }, { 1351, 0x6E9A }, { 1352, 0x823E }, { 1353, 0x7519 }, { 1355, 0x4911 },
    { 1356, 0x9A6C }, { 1357, 0x9A8F }, { 1358, 0x9F99 }, { 1359, 0x7987 }, { 1360, 0x2846C }, { 1361, 0x21DCA }, { 1362, 0x205D0 }, { 1363, 0x22AE6 },
    { 1364, 0x4E24 }, { 1365, 0x4E81 }, { 1366, 0x4E80 }, { 1367, 0x4E87 }, { 1368, 0x4EBF }, { 1369, 0x4EEB }, { 1370, 0x4F37 }, { 1371, 0x344C },
    { 1372, 0x4FBD }, { 1373, 0x3E48 }, { 1374, 0x5003 }, { 1375, 0x5088 }, { 1376, 0x347D }, { 1377, 0x3493 }, { 1378, 0x34A5 }, { 1379, 0x5186 },
    { 1380, 0x5905 }, { 1381, 0x51DB }, { 1382, 0x51FC }, { 1383, 0x5205 }, { 1384, 0x4E89 }, { 1385, 0x5279 }, { 1386, 0x5290 }, { 1387, 0x5327 },
    { 1388, 0x35C7 }, { 1389, 0x53A9 }, { 1390, 0x3551 }, { 1391, 0x53B0 }, { 1392, 0x3553 }, { 1393, 0x53C2 }, { 1394, 0x5423 }, { 1395, 0x356D },
    { 1396, 0x3572 }, { 1397, 0x3681 }, { 1398, 0x5493 }, { 1399, 0x54A3 }, { 1400, 0x54B4 }, { 1401, 0x54B9 }, { 1402, 0x54D0 }, { 1403, 0x54EF },
    { 1404, 0x5518 }, { 1405, 0x5523 }, { 1406, 0x5528 }, { 1407, 0x3598 }, { 1408, 0x553F }, { 1409, 0x35A5 }, { 1410, 0x35BF }, { 1411, 0x55D7 },
    { 1412, 0x35C5 }, { 1413, 0x27D84 }, { 1414, 0x5525 }, { 1416, 0x20C42 }, { 1417, 0x20D15 }, { 1418, 0x2512B }, { 1419, 0x5590 }, { 1420, 0x22CC6 },
    { 1421, 0x39EC }, { 1422, 0x20341 }, { 1423, 0x8E46 }, { 1424, 0x24DB8 }, { 1425, 0x294E5 }, { 1426, 0x4053 }, { 1427, 0x280BE }, { 1428, 0x777A },
    { 1429, 0x22C38 }, { 1430, 0x3A34 }, { 1431, 0x47D5 }, { 1432, 0x2815D }, { 1433, 0x269F2 }, { 1434, 0x24DEA }, { 1435, 0x64DD }, { 1436, 0x20D7C },
    { 1437, 0x20FB4 }, { 1438, 0x20CD5 }, { 1439, 0x210F4 }, { 1440, 0x648D }, { 1441, 0x8E7E }, { 1442, 0x20E96 }, { 1443, 0x20C0B }, { 1444, 0x20F64 },
    { 1445, 0x22CA9 }, { 1446, 0x28256 }, { 1447, 0x244D3 }, { 1449, 0x20D46 }, { 1450, 0x29A4D }, { 1451, 0x280E9 }, { 1452, 0x47F4 }, { 1453, 0x24EA7 },
    { 1454, 0x22CC2 }, { 1455, 0x9AB2 }, { 1456, 0x3A67 }, { 1457, 0x295F4 }, { 1458, 0x3FED }, { 1459, 0x3506 }, { 1460, 0x252C7 }, { 1461, 0x297D4 },
    { 1462, 0x278C8 }, { 1463, 0x22D44 }, { 1464, 0x9D6E }, { 1465, 0x9815 }, { 1467, 0x43D9 }, { 1468, 0x260A5 }, { 1469, 0x64B4 }, { 1470, 0x54E3 },
    { 1471, 0x22D4C }, { 1472, 0x22BCA }, { 1473, 0x21077 }, { 1474, 0x39FB }, { 1475, 0x2106F }, { 1476, 0x266DA }, { 1477, 0x26716 }, { 1478, 0x279A0 },
    { 1479, 0x64EA }, { 1480, 0x25052 }, { 1481, 0x20C43 }, { 1482, 0x8E68 }, { 1483, 0x221A1 }, { 1484, 0x28B4C }, { 1485, 0x20731 }, { 1487, 0x480B },
    { 1488, 0x201A9 }, { 1489, 0x3FFA }, { 1490, 0x5873 }, { 1491, 0x22D8D }, { 1493, 0x245C8 }, { 1494, 0x204FC }, { 1495, 0x26097 }, { 1496, 0x20F4C },
    { 1497, 0x20D96 }, { 1498, 0x5579 }, { 1499, 0x40BB }, { 1500, 0x43BA }, { 1502, 0x4AB4 }, { 1503, 0x22A66 }, { 1504, 0x2109D }, { 1505, 0x81AA },
    { 1506, 0x98F5 }, { 1507, 0x20D9C }, { 1508, 0x6379 }, { 1509, 0x39FE }, { 1510, 0x22775 }, { 1511, 0x8DC0 }, { 1512, 0x56A1 }, { 1513, 0x647C },
    { 1514, 0x3E43 }, { 1516, 0x2A601 }, { 1517, 0x20E09 }, { 1518, 0x22ACF }, { 1519, 0x22CC9 }, { 1521, 0x210C8 }, { 1522, 0x239C2 }, { 1523, 0x3992 },
    { 1524, 0x3A06 }, { 1525, 0x2829B }, { 1526, 0x3578 }, { 1527, 0x25E49 }, { 1528, 0x220C7 }, { 1529, 0x5652 }, { 1530, 0x20F31 }, { 1531, 0x22CB2 },
    { 1532, 0x29720 }, { 1533, 0x34BC }, { 1534, 0x6C3D }, { 1535, 0x24E3B }, { 1538, 0x27574 }, { 1539, 0x22E8B }, { 1540, 0x22208 }, { 1541, 0x2A65B },
    { 1542, 0x28CCD }, { 1543, 0x20E7A }, { 1544, 0x20C34 }, { 1545, 0x2681C }, { 1546, 0x7F93 }, { 1547, 0x210CF }, { 1548, 0x22803 }, { 1549, 0x22939 },
    { 1550, 0x35FB }, { 1551, 0x251E3 }, { 1552, 0x20E8C }, { 1553, 0x20F8D }, { 1554, 0x20EAA }, { 1555, 0x3F93 }, { 1556, 0x20F30 }, { 1557, 0x20D47 },
    { 1558, 0x2114F }, { 1559, 0x20E4C }, { 1561, 0x20EAB }, { 1562, 0x20BA9 }, { 1563, 0x20D48 }, { 1564, 0x210C0 }, { 1565, 0x2113D }, { 1566, 0x3FF9 },
    { 1567, 0x22696 }, { 1568, 0x6432 }, { 1569, 0x20FAD }, { 1570, 0x233F4 }, { 1571, 0x27639 }, { 1572, 0x22BCE }, { 1573, 0x20D7E }, { 1574, 0x20D7F },
    { 1575, 0x22C51 }, { 1576, 0x22C55 }, { 1577, 0x3A18 }, { 1578, 0x20E98 }, { 1579, 0x210C7 }, { 1580, 0x20F2E }, { 1581, 0x2A632 }, { 1582, 0x26B50 },
    { 1583, 0x28CD2 }, { 1584, 0x28D99 }, { 1585, 0x28CCA }, { 1586, 0x95AA }, { 1587, 0x54CC }, { 1588, 0x82C4 }, { 1589, 0x55B9 }, { 1591, 0x29EC3 },
    { 1592, 0x9C26 }, { 1593, 0x9AB6 }, { 1594, 0x2775E }, { 1595, 0x22DEE }, { 1596, 0x7140 }, { 1597, 0x816D }, { 1598, 0x80EC }, { 1599, 0x5C1C },
    { 1600, 0x26572 }, { 1601, 0x8134 }, { 1602, 0x3797 }, { 1603, 0x535F }, { 1604, 0x280BD }, { 1605, 0x91B6 }, { 1606, 0x20EFA }, { 1607, 0x20E0F },
    { 1608, 0x20E77 }, { 1609, 0x20EFB }, { 1610, 0x35DD }, { 1611, 0x24DEB }, { 1612, 0x3609 }, { 1613, 0x20CD6 }, { 1614, 0x56AF }, { 1615, 0x227B5 },
    { 1616, 0x210C9 }, { 1617, 0x20E10 }, { 1618, 0x20E78 }, { 1619, 0x21078 }, { 1620, 0x21148 }, { 1621, 0x28207 }, { 1622, 0x21455 }, { 1623, 0x20E79 },
    { 1624, 0x24E50 }, { 1625, 0x22DA4 }, { 1626, 0x5A54 }, { 1627, 0x2101D }, { 1628, 0x2101E }, { 1629, 0x210F5 }, { 1630, 0x210F6 }, { 1631, 0x579C },
    { 1632, 0x20E11 }, { 1633, 0x27694 }, { 1634, 0x282CD }, { 1635, 0x20FB5 }, { 1636, 0x20E7B }, { 1637, 0x2517E }, { 1638, 0x3703 }, { 1639, 0x20FB6 },
    { 1640, 0x21180 }, { 1641, 0x252D8 }, { 1642, 0x2A2BD }, { 1643, 0x249DA }, { 1644, 0x2183A }, { 1645, 0x24177 }, { 1646, 0x2827C }, { 1647, 0x5899 },
    { 1648, 0x5268 }, { 1649, 0x361A }, { 1650, 0x2573D }, { 1651, 0x7BB2 }, { 1652, 0x5B68 }, { 1653, 0x4800 }, { 1654, 0x4B2C }, { 1655, 0x9F27 },
    { 1656, 0x49E7 }, { 1657, 0x9C1F }, { 1658, 0x9B8D }, { 1659, 0x25B74 }, { 1660, 0x2313D }, { 1661, 0x55FB }, { 1662, 0x35F2 }, { 1663, 0x5689 },
    { 1664, 0x4E28 }, { 1665, 0x5902 }, { 1666, 0x21BC1 }, { 1667, 0x2F878 }, { 1668, 0x9751 }, { 1669, 0x20086 }, { 1670, 0x4E5B }, { 1671, 0x4EBB },
    { 1672, 0x353E }, { 1673, 0x5C23 }, { 1674, 0x5F51 }, { 1675, 0x5FC4 }, { 1676, 0x38FA }, { 1677, 0x624C }, { 1678, 0x6535 }, { 1679, 0x6B7A },
    { 1680, 0x6C35 }, { 1681, 0x6C3A }, { 1682, 0x706C }, { 1683, 0x722B }, { 1684, 0x4E2C }, { 1685, 0x72AD }, { 1686, 0x248E9 }, { 1687, 0x7F52 },
    { 1688, 0x793B }, { 1689, 0x7CF9 }, { 1690, 0x7F53 }, { 1691, 0x2626A }, { 1692, 0x34C1 }, { 1694, 0x2634B }, { 1695, 0x8002 }, { 1696, 0x8080 },
    { 1697, 0x26612 }, { 1698, 0x26951 }, { 1699, 0x535D }, { 1700, 0x8864 }, { 1701, 0x89C1 }, { 1702, 0x278B2 }, { 1703, 0x8BA0 }, { 1704, 0x8D1D },
    { 1705, 0x9485 }, { 1706, 0x9578 }, { 1707, 0x957F }, { 1708, 0x95E8 }, { 1709, 0x28E0F }, { 1710, 0x97E6 }, { 1711, 0x9875 }, { 1712, 0x98CE },
    { 1713, 0x98DE }, { 1714, 0x9963 }, { 1715, 0x29810 }, { 1716, 0x9C7C }, { 1717, 0x9E1F }, { 1718, 0x9EC4 }, { 1719, 0x6B6F }, { 1720, 0xF907 },
    { 1721, 0x4E37 }, { 1722, 0x20087 }, { 1723, 0x961D }, { 1724, 0x6237 }, { 1725, 0x94A2 }, { 1727, 0x503B }, { 1728, 0x6DFE }, { 1729, 0x29C73 },
    { 1730, 0x9FA6 }, { 1731, 0x3DC9 }, { 1732, 0x888F }, { 1733, 0x2414E }, { 1734, 0x7077 }, { 1735, 0x5CF5 }, { 1736, 0x4B20 }, { 1737, 0x251CD },
    { 1738, 0x3559 }, { 1739, 0x25D30 }, { 1740, 0x6122 }, { 1741, 0x28A32 }, { 1742, 0x8FA7 }, { 1743, 0x91F6 }, { 1744, 0x7191 }, { 1745, 0x6719 },
    { 1746, 0x73BA }, { 1747, 0x23281 }, { 1748, 0x2A107 }, { 1749, 0x3C8B }, { 1750, 0x21980 }, { 1751, 0x4B10 }, { 1752, 0x78E4 }, { 1753, 0x7402 },
    { 1754, 0x51AE }, { 1755, 0x2870F }, { 1756, 0x4009 }, { 1757, 0x6A63 }, { 1758, 0x2A2BA }, { 1759, 0x4223 }, { 1760, 0x860F }, { 1761, 0x20A6F },
    { 1762, 0x7A2A }, { 1763, 0x29947 }, { 1764, 0x28AEA }, { 1765, 0x9755 }, { 1766, 0x704D }, { 1767, 0x5324 }, { 1768, 0x2207E }, { 1769, 0x93F4 },
    { 1770, 0x76D9 }, { 1771, 0x289E3 }, { 1772, 0x9FA7 }, { 1773, 0x77DD }, { 1774, 0x4EA3 }, { 1775, 0x4FF0 }, { 1776, 0x50BC }, { 1777, 0x4E2F },
    { 1778, 0x4F17 }, { 1779, 0x9FA8 }, { 1780, 0x5434 }, { 1781, 0x7D8B }, { 1782, 0x5892 }, { 1783, 0x58D0 }, { 1784, 0x21DB6 }, { 1785, 0x5E92 },
    { 1786, 0x5E99 }, { 1787, 0x5FC2 }, { 1788, 0x22712 }, { 1789, 0x658B }, { 1790, 0x233F9 }, { 1791, 0x6919 }, { 1792, 0x6A43 }, { 1793, 0x23C63 },
    { 1794, 0x6CFF }, { 1796, 0x7200 }, { 1797, 0x24505 }, { 1798, 0x738C }, { 1799, 0x3EDB }, { 1800, 0x24A13 }, { 1801, 0x5B15 }, { 1802, 0x74B9 },
    { 1803, 0x8B83 }, { 1804, 0x25CA4 }, { 1805, 0x25695 }, { 1806, 0x7A93 }, { 1807, 0x7BEC }, { 1808, 0x7CC3 }, { 1809, 0x7E6C }, { 1810, 0x82F8 },
    { 1811, 0x8597 }, { 1812, 0x9FA9 }, { 1813, 0x8890 }, { 1814, 0x9FAA }, { 1815, 0x8EB9 }, { 1816, 0x9FAB }, { 1817, 0x8FCF }, { 1818, 0x855F },
    { 1819, 0x99E0 }, { 1820, 0x9221 }, { 1821, 0x9FAC }, { 1822, 0x28DB9 }, { 1823, 0x2143F }, { 1824, 0x4071 }, { 1825, 0x42A2 }, { 1826, 0x5A1A },
    { 1830, 0x9868 }, { 1831, 0x676B }, { 1832, 0x4276 }, { 1833, 0x573D }, { 1835, 0x85D6 }, { 1836, 0x2497B }, { 1837, 0x82BF }, { 1838, 0x2710D },
    { 1839, 0x4C81 }, { 1840, 0x26D74 }, { 1841, 0x5D7B }, { 1842, 0x26B15 }, { 1843, 0x26FBE }, { 1844, 0x9FAD }, { 1845, 0x9FAE }, { 1846, 0x5B96 },
    { 1847, 0x9FAF }, { 1848, 0x66E7 }, { 1849, 0x7E5B }, { 1850, 0x6E57 }, { 1851, 0x79CA }, { 1852, 0x3D88 }, { 1853, 0x44C3 }, { 1854, 0x23256 },
    { 1855, 0x22796 }, { 1856, 0x439A }, { 1857, 0x4536 }, { 1859, 0x5CD5 }, { 1860, 0x23B1A }, { 1861, 0x8AF9 }, { 1862, 0x5C78 }, { 1863, 0x3D12 },
    { 1864, 0x23551 }, { 1865, 0x5D78 }, { 1866, 0x9FB2 }, { 1867, 0x7157 }, { 1868, 0x4558 }, { 1869, 0x240EC }, { 1870, 0x21E23 }, { 1871, 0x4C77 },
    { 1872, 0x3978 }, { 1873, 0x344A }, { 1874, 0x201A4 }, { 1875, 0x26C41 }, { 1876, 0x8ACC }, { 1877, 0x4FB4 }, { 1878, 0x20239 }, { 1879, 0x59BF },
    { 1880, 0x816C }, { 1881, 0x9856 }, { 1882, 0x298FA }, { 1883, 0x5F3B }, { 1884, 0x20B9F }, { 1886, 0x221C1 }, { 1887, 0x2896D }, { 1888, 0x4102 },
    { 1889, 0x46BB }, { 1890, 0x29079 }, { 1891, 0x3F07 }, { 1892, 0x9FB3 }, { 1893, 0x2A1B5 }, { 1894, 0x40F8 }, { 1895, 0x37D6 }, { 1896, 0x46F7 },
    { 1897, 0x26C46 }, { 1898, 0x417C }, { 1899, 0x286B2 }, { 1900, 0x273FF }, { 1901, 0x456D }, { 1902, 0x38D4 }, { 1903, 0x2549A }, { 1904, 0x4561 },
    { 1905, 0x451B }, { 1906, 0x4D89 }, { 1907, 0x4C7B }, { 1908, 0x4D76 }, { 1909, 0x45EA }, { 1910, 0x3FC8 }, { 1911, 0x24B0F }, { 1912, 0x3661 },
    { 1913, 0x44DE }, { 1914, 0x44BD }, { 1915, 0x41ED }, { 1916, 0x5D3E }, { 1917, 0x5D48 }, { 1918, 0x5D56 }, { 1919, 0x3DFC }, { 1920, 0x380F },
    { 1921, 0x5DA4 }, { 1922, 0x5DB9 }, { 1923, 0x3820 }, { 1924, 0x3838 }, { 1925, 0x5E42 }, { 1926, 0x5EBD }, { 1927, 0x5F25 }, { 1928, 0x5F83 },
    { 1929, 0x3908 }, { 1930, 0x3914 }, { 1931, 0x393F }, { 1932, 0x394D }, { 1933, 0x60D7 }, { 1934, 0x613D }, { 1935, 0x5CE5 }, { 1936, 0x3989 },
    { 1937, 0x61B7 }, { 1938, 0x61B9 }, { 1939, 0x61CF }, { 1940, 0x39B8 }, { 1941, 0x622C }, { 1942, 0x6290 }, { 1943, 0x62E5 }, { 1944, 0x6318 },
    { 1945, 0x39F8 }, { 1946, 0x56B1 }, { 1947, 0x3A03 }, { 1948, 0x63E2 }, { 1949, 0x63FB }, { 1950, 0x6407 }, { 1951, 0x645A }, { 1952, 0x3A4B },
    { 1953, 0x64C0 }, { 1954, 0x5D15 }, { 1955, 0x5621 }, { 1956, 0x9F9F }, { 1957, 0x3A97 }, { 1958, 0x6586 }, { 1959, 0x3ABD }, { 1960, 0x65FF },
    { 1961, 0x6653 }, { 1962, 0x3AF2 }, { 1963, 0x6692 }, { 1964, 0x3B22 }, { 1965, 0x6716 }, { 1966, 0x3B42 }, { 1967, 0x67A4 }, { 1968, 0x6800 },
    { 1969, 0x3B58 }, { 1970, 0x684A }, { 1971, 0x6884 }, { 1972, 0x3B72 }, { 1973, 0x3B71 }, { 1974, 0x3B7B }, { 1975, 0x6909 }, { 1976, 0x6943 },
    { 1977, 0x725C }, { 1978, 0x6964 }, { 1979, 0x699F }, { 1980, 0x6985 }, { 1981, 0x3BBC }, { 1982, 0x69D6 }, { 1983, 0x3BDD }, { 1984, 0x6A65 },
    { 1985, 0x6A74 }, { 1986, 0x6A71 }, { 1987, 0x6A82 }, { 1988, 0x3BEC }, { 1989, 0x6A99 }, { 1990, 0x3BF2 }, { 1991, 0x6AAB }, { 1992, 0x6AB5 },
    { 1993, 0x6AD4 }, { 1994, 0x6AF6 }, { 1995, 0x6B81 }, { 1996, 0x6BC1 }, { 1997, 0x6BEA }, { 1998, 0x6C75 }, { 1999, 0x6CAA }, { 2000, 0x3CCB },
    { 2001, 0x6D02 }, { 2002, 0x6D06 }, { 2003, 0x6D26 }, { 2004, 0x6D81 }, { 2005, 0x3CEF }, { 2006, 0x6DA4 }, { 2007, 0x6DB1 }, { 2008, 0x6E15 },
    { 2009, 0x6E18 }, { 2010, 0x6E29 }, { 2011, 0x6E86 }, { 2012, 0x289C0 }, { 2013, 0x6EBB }, { 2014, 0x6EE2 }, { 2015, 0x6EDA }, { 2016, 0x9F7F },
    { 2017, 0x6EE8 }, { 2018, 0x6EE9 }, { 2019, 0x6F24 }, { 2020, 0x6F34 }, { 2021, 0x3D46 }, { 2022, 0x23F41 }, { 2023, 0x6F81 }, { 2024, 0x6FBE },
    { 2025, 0x3D6A }, { 2026, 0x3D75 }, { 2027, 0x71B7 }, { 2028, 0x5C99 }, { 2029, 0x3D8A }, { 2030, 0x702C }, { 2031, 0x3D91 }, { 2032, 0x7050 },
    { 2033, 0x7054 }, { 2034, 0x706F }, { 2035, 0x707F }, { 2036, 0x7089 }, { 2037, 0x20325 }, { 2038, 0x43C1 }, { 2039, 0x35F1 }, { 2040, 0x20ED8 },
    { 2041, 0x23ED7 }, { 2042, 0x57BE }, { 2043, 0x26ED3 }, { 2044, 0x713E }, { 2045, 0x257E0 }, { 2046, 0x364E }, { 2047, 0x69A2 }, { 2048, 0x28BE9 },
    { 2049, 0x5B74 }, { 2050, 0x7A49 }, { 2051, 0x258E1 }, { 2052, 0x294D9 }, { 2053, 0x7A65 }, { 2054, 0x7A7D }, { 2055, 0x259AC }, { 2056, 0x7ABB },
    { 2057, 0x7AB0 }, { 2058, 0x7AC2 }, { 2059, 0x7AC3 }, { 2060, 0x71D1 }, { 2061, 0x2648D }, { 2062, 0x41CA }, { 2063, 0x7ADA }, { 2064, 0x7ADD },
    { 2065, 0x7AEA }, { 2066, 0x41EF }, { 2067, 0x54B2 }, { 2068, 0x25C01 }, { 2069, 0x7B0B }, { 2070, 0x7B55 }, { 2071, 0x7B29 }, { 2072, 0x2530E },
    { 2073, 0x25CFE }, { 2074, 0x7BA2 }, { 2075, 0x7B6F }, { 2076, 0x839C }, { 2077, 0x25BB4 }, { 2078, 0x26C7F }, { 2079, 0x7BD0 }, { 2080, 0x8421 },
    { 2081, 0x7B92 }, { 2082, 0x7BB8 }, { 2083, 0x25D20 }, { 2084, 0x3DAD }, { 2085, 0x25C65 }, { 2086, 0x8492 }, { 2087, 0x7BFA }, { 2088, 0x7C06 },
    { 2089, 0x7C35 }, { 2090, 0x25CC1 }, { 2091, 0x7C44 }, { 2092, 0x7C83 }, { 2093, 0x24882 }, { 2094, 0x7CA6 }, { 2095, 0x667D }, { 2096, 0x24578 },
    { 2097, 0x7CC9 }, { 2098, 0x7CC7 }, { 2099, 0x7CE6 }, { 2100, 0x7C74 }, { 2101, 0x7CF3 }, { 2102, 0x7CF5 }, { 2103, 0x7CCE }, { 2104, 0x7E67 },
    { 2105, 0x451D }, { 2106, 0x26E44 }, { 2107, 0x7D5D }, { 2108, 0x26ED6 }, { 2109, 0x748D }, { 2110, 0x7D89 }, { 2111, 0x7DAB }, { 2112, 0x7135 },
    { 2113, 0x7DB3 }, { 2114, 0x7DD2 }, { 2115, 0x24057 }, { 2116, 0x26029 }, { 2117, 0x7DE4 }, { 2118, 0x3D13 }, { 2119, 0x7DF5 }, { 2120, 0x217F9 },
    { 2121, 0x7DE5 }, { 2122, 0x2836D }, { 2123, 0x7E1D }, { 2124, 0x26121 }, { 2125, 0x2615A }, { 2126, 0x7E6E }, { 2127, 0x7E92 }, { 2128, 0x432B },
    { 2129, 0x946C }, { 2130, 0x7E27 }, { 2131, 0x7F40 }, { 2132, 0x7F41 }, { 2133, 0x7F47 }, { 2134, 0x7936 }, { 2135, 0x262D0 }, { 2136, 0x99E1 },
    { 2137, 0x7F97 }, { 2138, 0x26351 }, { 2139, 0x7FA3 }, { 2140, 0x21661 }, { 2141, 0x20068 }, { 2142, 0x455C }, { 2143, 0x23766 }, { 2144, 0x4503 },
    { 2145, 0x2833A }, { 2146, 0x7FFA }, { 2147, 0x26489 }, { 2148, 0x8005 }, { 2149, 0x8008 }, { 2150, 0x801D }, { 2151, 0x8028 }, { 2152, 0x802F },
    { 2153, 0x2A087 }, { 2154, 0x26CC3 }, { 2155, 0x803B }, { 2156, 0x803C }, { 2157, 0x8061 }, { 2158, 0x22714 }, { 2159, 0x4989 }, { 2160, 0x26626 },
    { 2161, 0x23DE3 }, { 2162, 0x266E8 }, { 2163, 0x6725 }, { 2164, 0x80A7 }, { 2165, 0x28A48 }, { 2166, 0x8107 }, { 2167, 0x811A }, { 2168, 0x58B0 },
    { 2169, 0x226F6 }, { 2170, 0x6C7F }, { 2171, 0x26498 }, { 2172, 0x24FB8 }, { 2173, 0x64E7 }, { 2174, 0x2148A }, { 2175, 0x8218 }, { 2176, 0x2185E },
    { 2177, 0x6A53 }, { 2178, 0x24A65 }, { 2179, 0x24A95 }, { 2180, 0x447A }, { 2181, 0x8229 }, { 2182, 0x20B0D }, { 2183, 0x26A52 }, { 2184, 0x23D7E },
    { 2185, 0x4FF9 }, { 2186, 0x214FD }, { 2187, 0x84E2 }, { 2188, 0x8362 }, { 2189, 0x26B0A }, { 2190, 0x249A7 }, { 2191, 0x23530 }, { 2192, 0x21773 },
    { 2193, 0x23DF8 }, { 2194, 0x82AA }, { 2195, 0x691B }, { 2196, 0x2F994 }, { 2197, 0x41DB }, { 2198, 0x854B }, { 2199, 0x82D0 }, { 2200, 0x831A },
    { 2201, 0x20E16 }, { 2202, 0x217B4 }, { 2203, 0x36C1 }, { 2204, 0x2317D }, { 2205, 0x2355A }, { 2206, 0x827B }, { 2207, 0x82E2 }, { 2208, 0x8318 },
    { 2209, 0x23E8B }, { 2210, 0x26DA3 }, { 2211, 0x26B05 }, { 2212, 0x26B97 }, { 2213, 0x235CE }, { 2214, 0x3DBF }, { 2215, 0x831D }, { 2216, 0x55EC },
    { 2217, 0x8385 }, { 2218, 0x450B }, { 2219, 0x26DA5 }, { 2220, 0x83AC }, { 2221, 0x83C1 }, { 2222, 0x83D3 }, { 2223, 0x347E }, { 2224, 0x26ED4 },
    { 2225, 0x6A57 }, { 2226, 0x855A }, { 2227, 0x3496 }, { 2228, 0x26E42 }, { 2229, 0x22EEF }, { 2230, 0x8458 }, { 2231, 0x25BE4 }, { 2232, 0x8471 },
    { 2233, 0x3DD3 }, { 2234, 0x44E4 }, { 2235, 0x6AA7 }, { 2236, 0x844A }, { 2237, 0x23CB5 }, { 2238, 0x7958 }, { 2239, 0x84A8 }, { 2240, 0x26B96 },
    { 2241, 0x26E77 }, { 2242, 0x26E43 }, { 2243, 0x84DE }, { 2244, 0x840F }, { 2245, 0x8391 }, { 2246, 0x44A0 }, { 2247, 0x8493 }, { 2248, 0x84E4 },
    { 2249, 0x25C91 }, { 2250, 0x4240 }, { 2251, 0x25CC0 }, { 2252, 0x4543 }, { 2253, 0x8534 }, { 2254, 0x5AF2 }, { 2255, 0x26E99 }, { 2256, 0x4527 },
    { 2257, 0x8573 }, { 2258, 0x4516 }, { 2259, 0x67BF }, { 2260, 0x8616 }, { 2261, 0x28625 }, { 2262, 0x2863B }, { 2263, 0x85C1 }, { 2264, 0x27088 },
    { 2265, 0x8602 }, { 2266, 0x21582 }, { 2267, 0x270CD }, { 2268, 0x2F9B2 }, { 2269, 0x456A }, { 2270, 0x8628 }, { 2271, 0x3648 }, { 2272, 0x218A2 },
    { 2273, 0x53F7 }, { 2274, 0x2739A }, { 2275, 0x867E }, { 2276, 0x8771 }, { 2277, 0x2A0F8 }, { 2278, 0x87EE }, { 2279, 0x22C27 }, { 2280, 0x87B1 },
    { 2281, 0x87DA }, { 2282, 0x880F }, { 2283, 0x5661 }, { 2284, 0x866C }, { 2285, 0x6856 }, { 2286, 0x460F }, { 2287, 0x8845 }, { 2288, 0x8846 },
    { 2289, 0x275E0 }, { 2290, 0x23DB9 }, { 2291, 0x275E4 }, { 2292, 0x885E }, { 2293, 0x889C }, { 2294, 0x465B }, { 2295, 0x88B4 }, { 2296, 0x88B5 },
    { 2297, 0x63C1 }, { 2298, 0x88C5 }, { 2299, 0x7777 }, { 2300, 0x2770F }, { 2301, 0x8987 }, { 2302, 0x898A }, { 2303, 0x89A6 }, { 2304, 0x89A9 },
    { 2305, 0x89A7 }, { 2306, 0x89BC }, { 2307, 0x28A25 }, { 2308, 0x89E7 }, { 2309, 0x27924 }, { 2310, 0x27ABD }, { 2311, 0x8A9C }, { 2312, 0x7793 },
    { 2313, 0x91FE }, { 2314, 0x8A90 }, { 2315, 0x27A59 }, { 2316, 0x7AE9 }, { 2317, 0x27B3A }, { 2318, 0x23F8F }, { 2319, 0x4713 }, { 2320, 0x27B38 },
    { 2321, 0x717C }, { 2322, 0x8B0C }, { 2323, 0x8B1F }, { 2324, 0x25430 }, { 2325, 0x25565 }, { 2326, 0x8B3F }, { 2327, 0x8B4C }, { 2328, 0x8B4D },
    { 2329, 0x8AA9 }, { 2330, 0x24A7A }, { 2331, 0x8B90 }, { 2332, 0x8B9B }, { 2333, 0x8AAF }, { 2334, 0x216DF }, { 2335, 0x4615 }, { 2336, 0x884F },
    { 2337, 0x8C9B }, { 2338, 0x27D54 }, { 2339, 0x27D8F }, { 2340, 0x2F9D4 }, { 2341, 0x3725 }, { 2342, 0x27D53 }, { 2343, 0x8CD6 }, { 2344, 0x27D98 },
    { 2345, 0x27DBD }, { 2346, 0x8D12 }, { 2347, 0x8D03 }, { 2348, 0x21910 }, { 2349, 0x8CDB }, { 2350, 0x705C }, { 2351, 0x8D11 }, { 2352, 0x24CC9 },
    { 2353, 0x3ED0 }, { 2354, 0x8D77 }, { 2355, 0x8DA9 }, { 2356, 0x28002 }, { 2357, 0x21014 }, { 2358, 0x2498A }, { 2359, 0x3B7C }, { 2360, 0x281BC },
    { 2361, 0x2710C }, { 2362, 0x7AE7 }, { 2363, 0x8EAD }, { 2364, 0x8EB6 }, { 2365, 0x8EC3 }, { 2366, 0x92D4 }, { 2367, 0x8F19 }, { 2368, 0x8F2D },
    { 2369, 0x28365 }, { 2370, 0x28412 }, { 2371, 0x8FA5 }, { 2372, 0x9303 }, { 2373, 0x2A29F }, { 2374, 0x20A50 }, { 2375, 0x8FB3 }, { 2376, 0x492A },
    { 2377, 0x289DE }, { 2378, 0x2853D }, { 2379, 0x23DBB }, { 2380, 0x5EF8 }, { 2381, 0x23262 }, { 2382, 0x8FF9 }, { 2383, 0x2A014 }, { 2384, 0x286BC },
    { 2385, 0x28501 }, { 2386, 0x22325 }, { 2387, 0x3980 }, { 2388, 0x26ED7 }, { 2389, 0x9037 }, { 2390, 0x2853C }, { 2391, 0x27ABE }, { 2392, 0x9061 },
    { 2393, 0x2856C }, { 2394, 0x2860B }, { 2395, 0x90A8 }, { 2396, 0x28713 }, { 2397, 0x90C4 }, { 2398, 0x286E6 }, { 2399, 0x90AE }, { 2400, 0x90FD },
    { 2401, 0x9167 }, { 2402, 0x3AF0 }, { 2403, 0x91A9 }, { 2404, 0x91C4 }, { 2405, 0x7CAC }, { 2406, 0x28933 }, { 2407, 0x21E89 }, { 2408, 0x920E },
    { 2409, 0x6C9F }, { 2410, 0x9241 }, { 2411, 0x9262 }, { 2412, 0x255B9 }, { 2413, 0x92B9 }, { 2414, 0x28AC6 }, { 2415, 0x23C9B }, { 2416, 0x28B0C },
    { 2417, 0x255DB }, { 2418, 0x20D31 }, { 2419, 0x932C }, { 2420, 0x936B }, { 2421, 0x28AE1 }, { 2422, 0x28BEB }, { 2423, 0x708F }, { 2424, 0x5AC3 },
    { 2425, 0x28AE2 }, { 2426, 0x28AE5 }, { 2427, 0x4965 }, { 2428, 0x9244 }, { 2429, 0x28BEC }, { 2430, 0x28C39 }, { 2431, 0x28BFF }, { 2432, 0x9373 },
    { 2433, 0x945B }, { 2434, 0x8EBC }, { 2435, 0x9585 }, { 2436, 0x95A6 }, { 2437, 0x9426 }, { 2438, 0x95A0 }, { 2439, 0x6FF6 }, { 2440, 0x42B9 },
    { 2441, 0x2267A }, { 2442, 0x286D8 }, { 2443, 0x2127C }, { 2444, 0x23E2E }, { 2445, 0x49DF }, { 2446, 0x6C1C }, { 2447, 0x967B }, { 2448, 0x9696 },
    { 2449, 0x416C }, { 2450, 0x96A3 }, { 2451, 0x26ED5 }, { 2452, 0x61DA }, { 2453, 0x96B6 }, { 2454, 0x78F5 }, { 2455, 0x28AE0 }, { 2456, 0x96BD },
    { 2457, 0x53CC }, { 2458, 0x49A1 }, { 2459, 0x26CB8 }, { 2460, 0x20274 }, { 2461, 0x26410 }, { 2462, 0x290AF }, { 2463, 0x290E5 }, { 2464, 0x24AD1 },
    { 2465, 0x21915 }, { 2466, 0x2330A }, { 2467, 0x9731 }, { 2468, 0x8642 }, { 2469, 0x9736 }, { 2470, 0x4A0F }, { 2471, 0x453D }, { 2472, 0x4585 },
    { 2473, 0x24AE9 }, { 2474, 0x7075 }, { 2475, 0x5B41 }, { 2476, 0x971B }, { 2477, 0x975C }, { 2478, 0x291D5 }, { 2479, 0x9757 }, { 2480, 0x5B4A },
    { 2481, 0x291EB }, { 2482, 0x975F }, { 2483, 0x9425 }, { 2484, 0x50D0 }, { 2485, 0x230B7 }, { 2486, 0x230BC }, { 2487, 0x9789 }, { 2488, 0x979F },
    { 2489, 0x97B1 }, { 2490, 0x97BE }, { 2491, 0x97C0 }, { 2492, 0x97D2 }, { 2493, 0x97E0 }, { 2494, 0x2546C }, { 2495, 0x97EE }, { 2496, 0x741C },
    { 2497, 0x29433 }, { 2498, 0x97FF }, { 2499, 0x97F5 }, { 2500, 0x2941D }, { 2501, 0x2797A }, { 2502, 0x4AD1 }, { 2503, 0x9834 }, { 2504, 0x9833 },
    { 2505, 0x984B }, { 2506, 0x9866 }, { 2507, 0x3B0E }, { 2508, 0x27175 }, { 2509, 0x3D51 }, { 2510, 0x20630 }, { 2511, 0x2415C }, { 2512, 0x25706 },
    { 2513, 0x98CA }, { 2514, 0x98B7 }, { 2515, 0x98C8 }, { 2516, 0x98C7 }, { 2517, 0x4AFF }, { 2518, 0x26D27 }, { 2519, 0x216D3 }, { 2520, 0x55B0 },
    { 2521, 0x98E1 }, { 2522, 0x98E6 }, { 2523, 0x98EC }, { 2524, 0x9378 }, { 2525, 0x9939 }, { 2526, 0x24A29 }, { 2527, 0x4B72 }, { 2528, 0x29857 },
    { 2529, 0x29905 }, { 2530, 0x99F5 }, { 2531, 0x9A0C }, { 2532, 0x9A3B }, { 2533, 0x9A10 }, { 2534, 0x9A58 }, { 2535, 0x25725 }, { 2536, 0x36C4 },
    { 2537, 0x290B1 }, { 2538, 0x29BD5 }, { 2539, 0x9AE0 }, { 2540, 0x9AE2 }, { 2541, 0x29B05 }, { 2542, 0x9AF4 }, { 2543, 0x4C0E }, { 2544, 0x9B14 },
    { 2545, 0x9B2D }, { 2546, 0x28600 }, { 2547, 0x5034 }, { 2548, 0x9B34 }, { 2549, 0x269A8 }, { 2550, 0x38C3 }, { 2551, 0x2307D }, { 2552, 0x9B50 },
    { 2553, 0x9B40 }, { 2554, 0x29D3E }, { 2555, 0x5A45 }, { 2556, 0x21863 }, { 2557, 0x9B8E }, { 2558, 0x2424B }, { 2559, 0x9C02 }, { 2560, 0x9BFF },
    { 2561, 0x9C0C }, { 2562, 0x29E68 }, { 2563, 0x9DD4 }, { 2564, 0x29FB7 }, { 2565, 0x2A192 }, { 2566, 0x2A1AB }, { 2567, 0x2A0E1 }, { 2568, 0x2A123 },
    { 2569, 0x2A1DF }, { 2570, 0x9D7E }, { 2571, 0x9D83 }, { 2572, 0x2A134 }, { 2573, 0x9E0E }, { 2574, 0x6888 }, { 2575, 0x9DC4 }, { 2576, 0x2215B },
    { 2577, 0x2A193 }, { 2578, 0x2A220 }, { 2579, 0x2193B }, { 2580, 0x2A233 }, { 2581, 0x9D39 }, { 2582, 0x2A0B9 }, { 2583, 0x2A2B4 }, { 2584, 0x9E90 },
    { 2585, 0x9E95 }, { 2586, 0x9E9E }, { 2587, 0x9EA2 }, { 2588, 0x4D34 }, { 2589, 0x9EAA }, { 2590, 0x9EAF }, { 2591, 0x24364 }, { 2592, 0x9EC1 },
    { 2593, 0x3B60 }, { 2594, 0x39E5 }, { 2595, 0x3D1D }, { 2596, 0x4F32 }, { 2597, 0x37BE }, { 2598, 0x28C2B }, { 2599, 0x9F02 }, { 2600, 0x9F08 },
    { 2601, 0x4B96 }, { 2602, 0x9424 }, { 2603, 0x26DA2 }, { 2604, 0x9F17 }, { 2605, 0x9F16 }, { 2606, 0x9F39 }, { 2607, 0x569F }, { 2608, 0x568A },
    { 2609, 0x9F45 }, { 2610, 0x99B8 }, { 2611, 0x2908B }, { 2612, 0x97F2 }, { 2613, 0x847F }, { 2614, 0x9F62 }, { 2615, 0x9F69 }, { 2616, 0x7ADC },
    { 2617, 0x9F8E }, { 2618, 0x7216 }, { 2619, 0x4BBE }, { 2620, 0x24975 }, { 2621, 0x249BB }, { 2622, 0x7177 }, { 2623, 0x249F8 }, { 2624, 0x24348 },
    { 2625, 0x24A51 }, { 2626, 0x739E }, { 2627, 0x28BDA }, { 2628, 0x218FA }, { 2629, 0x799F }, { 2630, 0x2897E }, { 2631, 0x28E36 }, { 2632, 0x9369 },
    { 2633, 0x93F3 }, { 2634, 0x28A44 }, { 2635, 0x92EC }, { 2636, 0x9381 }, { 2637, 0x93CB }, { 2638, 0x2896C }, { 2639, 0x244B9 }, { 2640, 0x7217 },
    { 2641, 0x3EEB }, { 2642, 0x7772 }, { 2643, 0x7A43 }, { 2644, 0x70D0 }, { 2645, 0x24473 }, { 2646, 0x243F8 }, { 2647, 0x717E }, { 2648, 0x217EF },
    { 2649, 0x70A3 }, { 2650, 0x218BE }, { 2651, 0x23599 }, { 2652, 0x3EC7 }, { 2653, 0x21885 }, { 2654, 0x2542F }, { 2655, 0x217F8 }, { 2656, 0x3722 },
    { 2657, 0x216FB }, { 2658, 0x21839 }, { 2659, 0x36E1 }, { 2660, 0x21774 }, { 2661, 0x218D1 }, { 2662, 0x25F4B }, { 2663, 0x3723 }, { 2664, 0x216C0 },
    { 2665, 0x575B }, { 2666, 0x24A25 }, { 2667, 0x213FE }, { 2668, 0x212A8 }, { 2669, 0x213C6 }, { 2670, 0x214B6 }, { 2671, 0x8503 }, { 2672, 0x236A6 },
    { 2673, 0x8503 }, { 2674, 0x8455 }, { 2675, 0x24994 }, { 2676, 0x27165 }, { 2677, 0x23E31 }, { 2678, 0x2555C }, { 2679, 0x23EFB }, { 2680, 0x27052 },
    { 2681, 0x44F4 }, { 2682, 0x236EE }, { 2683, 0x2999D }, { 2684, 0x26F26 }, { 2685, 0x67F9 }, { 2686, 0x3733 }, { 2687, 0x3C15 }, { 2688, 0x3DE7 },
    { 2689, 0x586C }, { 2690, 0x21922 }, { 2691, 0x6810 }, { 2692, 0x4057 }, { 2693, 0x2373F }, { 2694, 0x240E1 }, { 2695, 0x2408B }, { 2696, 0x2410F },
    { 2697, 0x26C21 }, { 2698, 0x54CB }, { 2699, 0x569E }, { 2700, 0x266B1 }, { 2701, 0x5692 }, { 2702, 0x20FDF }, { 2703, 0x20BA8 }, { 2704, 0x20E0D },
    { 2705, 0x93C6 }, { 2706, 0x28B13 }, { 2707, 0x939C }, { 2708, 0x4EF8 }, { 2709, 0x512B }, { 2710, 0x3819 }, { 2711, 0x24436 }, { 2712, 0x4EBC },
    { 2713, 0x20465 }, { 2714, 0x2037F }, { 2715, 0x4F4B }, { 2716, 0x4F8A }, { 2717, 0x25651 }, { 2718, 0x5A68 }, { 2719, 0x201AB }, { 2720, 0x203CB },
    { 2721, 0x3999 }, { 2722, 0x2030A }, { 2723, 0x20414 }, { 2724, 0x3435 }, { 2725, 0x4F29 }, { 2726, 0x202C0 }, { 2727, 0x28EB3 }, { 2728, 0x20275 },
    { 2729, 0x8ADA }, { 2730, 0x2020C }, { 2731, 0x4E98 }, { 2732, 0x50CD }, { 2733, 0x510D }, { 2734, 0x4FA2 }, { 2735, 0x4F03 }, { 2736, 0x24A0E },
    { 2737, 0x23E8A }, { 2738, 0x4F42 }, { 2739, 0x502E }, { 2740, 0x506C }, { 2741, 0x5081 }, { 2742, 0x4FCC }, { 2743, 0x4FE5 }, { 2744, 0x5058 },
    { 2745, 0x50FC }, { 2746, 0x5159 }, { 2747, 0x515B }, { 2748, 0x515D }, { 2749, 0x515E }, { 2750, 0x6E76 }, { 2751, 0x23595 }, { 2752, 0x23E39 },
    { 2753, 0x23EBF }, { 2754, 0x6D72 }, { 2755, 0x21884 }, { 2756, 0x23E89 }, { 2757, 0x51A8 }, { 2758, 0x51C3 }, { 2759, 0x205E0 }, { 2760, 0x44DD },
    { 2761, 0x204A3 }, { 2762, 0x20492 }, { 2763, 0x20491 }, { 2764, 0x8D7A }, { 2765, 0x28A9C }, { 2766, 0x2070E }, { 2767, 0x5259 }, { 2768, 0x52A4 },
    { 2769, 0x20873 }, { 2770, 0x52E1 }, { 2771, 0x936E }, { 2772, 0x467A }, { 2773, 0x718C }, { 2774, 0x2438C }, { 2775, 0x20C20 }, { 2776, 0x249AC },
    { 2777, 0x210E4 }, { 2778, 0x69D1 }, { 2779, 0x20E1D }, { 2780, 0x7479 }, { 2781, 0x3EDE }, { 2782, 0x7499 }, { 2783, 0x7414 }, { 2784, 0x7456 },
    { 2785, 0x7398 }, { 2786, 0x4B8E }, { 2787, 0x24ABC }, { 2788, 0x2408D }, { 2789, 0x53D0 }, { 2790, 0x3584 }, { 2791, 0x720F }, { 2792, 0x240C9 },
    { 2793, 0x55B4 }, { 2794, 0x20345 }, { 2795, 0x54CD }, { 2796, 0x20BC6 }, { 2797, 0x571D }, { 2798, 0x925D }, { 2799, 0x96F4 }, { 2800, 0x9366 },
    { 2801, 0x57DD }, { 2802, 0x578D }, { 2803, 0x577F }, { 2804, 0x363E }, { 2805, 0x58CB }, { 2806, 0x5A99 }, { 2807, 0x28A46 }, { 2808, 0x216FA },
    { 2809, 0x2176F }, { 2810, 0x21710 }, { 2811, 0x5A2C }, { 2812, 0x59B8 }, { 2813, 0x928F }, { 2814, 0x5A7E }, { 2815, 0x5ACF }, { 2816, 0x5A12 },
    { 2817, 0x25946 }, { 2818, 0x219F3 }, { 2819, 0x21861 }, { 2820, 0x24295 }, { 2821, 0x36F5 }, { 2822, 0x6D05 }, { 2823, 0x7443 }, { 2824, 0x5A21 },
    { 2825, 0x25E83 }, { 2826, 0x5A81 }, { 2827, 0x28BD7 }, { 2828, 0x20413 }, { 2829, 0x93E0 }, { 2830, 0x748C }, { 2831, 0x21303 }, { 2832, 0x7105 },
    { 2833, 0x4972 }, { 2834, 0x9408 }, { 2835, 0x289FB }, { 2836, 0x93BD }, { 2837, 0x37A0 }, { 2838, 0x5C1E }, { 2839, 0x5C9E }, { 2840, 0x5E5E },
    { 2841, 0x5E48 }, { 2842, 0x21996 }, { 2843, 0x2197C }, { 2844, 0x23AEE }, { 2845, 0x5ECD }, { 2846, 0x5B4F }, { 2847, 0x21903 }, { 2848, 0x21904 },
    { 2849, 0x3701 }, { 2850, 0x218A0 }, { 2851, 0x36DD }, { 2852, 0x216FE }, { 2853, 0x36D3 }, { 2854, 0x812A }, { 2855, 0x28A47 }, { 2856, 0x21DBA },
    { 2857, 0x23472 }, { 2858, 0x289A8 }, { 2859, 0x5F0C }, { 2860, 0x5F0E }, { 2861, 0x21927 }, { 2862, 0x217AB }, { 2863, 0x5A6B }, { 2864, 0x2173B },
    { 2865, 0x5B44 }, { 2866, 0x8614 }, { 2867, 0x275FD }, { 2868, 0x8860 }, { 2869, 0x607E }, { 2870, 0x22860 }, { 2871, 0x2262B }, { 2872, 0x5FDB },
    { 2873, 0x3EB8 }, { 2874, 0x225AF }, { 2875, 0x225BE }, { 2876, 0x29088 }, { 2877, 0x26F73 }, { 2878, 0x61C0 }, { 2879, 0x2003E }, { 2880, 0x20046 },
    { 2881, 0x2261B }, { 2882, 0x6199 }, { 2883, 0x6198 }, { 2884, 0x6075 }, { 2885, 0x22C9B }, { 2886, 0x22D07 }, { 2887, 0x246D4 }, { 2888, 0x2914D },
    { 2889, 0x6471 }, { 2890, 0x24665 }, { 2891, 0x22B6A }, { 2892, 0x3A29 }, { 2893, 0x22B22 }, { 2894, 0x23450 }, { 2895, 0x298EA }, { 2896, 0x22E78 },
    { 2897, 0x6337 }, { 2898, 0x2A45B }, { 2899, 0x64B6 }, { 2900, 0x6331 }, { 2901, 0x63D1 }, { 2902, 0x249E3 }, { 2903, 0x22D67 }, { 2904, 0x62A4 },
    { 2905, 0x22CA1 }, { 2906, 0x643B }, { 2907, 0x656B }, { 2908, 0x6972 }, { 2909, 0x3BF4 }, { 2910, 0x2308E }, { 2911, 0x232AD }, { 2912, 0x24989 },
    { 2913, 0x232AB }, { 2914, 0x550D }, { 2915, 0x232E0 }, { 2916, 0x218D9 }, { 2917, 0x2943F }, { 2918, 0x66CE }, { 2919, 0x23289 }, { 2920, 0x231B3 },
    { 2921, 0x3AE0 }, { 2922, 0x4190 }, { 2923, 0x25584 }, { 2924, 0x28B22 }, { 2925, 0x2558F }, { 2926, 0x216FC }, { 2927, 0x2555B }, { 2928, 0x25425 },
    { 2929, 0x78EE }, { 2930, 0x23103 }, { 2931, 0x2182A }, { 2932, 0x23234 }, { 2933, 0x3464 }, { 2934, 0x2320F }, { 2935, 0x23182 }, { 2936, 0x242C9 },
    { 2937, 0x668E }, { 2938, 0x26D24 }, { 2939, 0x666B }, { 2940, 0x4B93 }, { 2941, 0x6630 }, { 2942, 0x27870 }, { 2943, 0x21DEB }, { 2944, 0x6663 },
    { 2945, 0x232D2 }, { 2946, 0x232E1 }, { 2947, 0x661E }, { 2948, 0x25872 }, { 2949, 0x38D1 }, { 2950, 0x2383A }, { 2951, 0x237BC }, { 2952, 0x3B99 },
    { 2953, 0x237A2 }, { 2954, 0x233FE }, { 2955, 0x74D0 }, { 2956, 0x3B96 }, { 2957, 0x678F }, { 2958, 0x2462A }, { 2959, 0x68B6 }, { 2960, 0x681E },
    { 2961, 0x3BC4 }, { 2962, 0x6ABE }, { 2963, 0x3863 }, { 2964, 0x237D5 }, { 2965, 0x24487 }, { 2966, 0x6A33 }, { 2967, 0x6A52 }, { 2968, 0x6AC9 },
    { 2969, 0x6B05 }, { 2970, 0x21912 }, { 2971, 0x6511 }, { 2972, 0x6898 }, { 2973, 0x6A4C }, { 2974, 0x3BD7 }, { 2975, 0x6A7A }, { 2976, 0x6B57 },
    { 2977, 0x23FC0 }, { 2978, 0x23C9A }, { 2979, 0x93A0 }, { 2980, 0x92F2 }, { 2981, 0x28BEA }, { 2982, 0x28ACB }, { 2983, 0x9289 }, { 2984, 0x2801E },
    { 2985, 0x289DC }, { 2986, 0x9467 }, { 2987, 0x6DA5 }, { 2988, 0x6F0B }, { 2989, 0x249EC }, { 2990, 0x6D67 }, { 2991, 0x23F7F }, { 2992, 0x3D8F },
    { 2993, 0x6E04 }, { 2994, 0x2403C }, { 2995, 0x5A3D }, { 2996, 0x6E0A }, { 2997, 0x5847 }, { 2998, 0x6D24 }, { 2999, 0x7842 }, { 3000, 0x713B },
    { 3001, 0x2431A }, { 3002, 0x24276 }, { 3003, 0x70F1 }, { 3004, 0x7250 }, { 3005, 0x7287 }, { 3006, 0x7294 }, { 3007, 0x2478F }, { 3008, 0x24725 },
    { 3009, 0x5179 }, { 3010, 0x24AA4 }, { 3011, 0x205EB }, { 3012, 0x747A }, { 3013, 0x23EF8 }, { 3014, 0x2365F }, { 3015, 0x24A4A }, { 3016, 0x24917 },
    { 3017, 0x25FE1 }, { 3018, 0x3F06 }, { 3019, 0x3EB1 }, { 3020, 0x24ADF }, { 3021, 0x28C23 }, { 3022, 0x23F35 }, { 3023, 0x60A7 }, { 3024, 0x3EF3 },
    { 3025, 0x74CC }, { 3026, 0x743C }, { 3027, 0x9387 }, { 3028, 0x7437 }, { 3029, 0x449F }, { 3030, 0x26DEA }, { 3031, 0x4551 }, { 3032, 0x7583 },
    { 3033, 0x3F63 }, { 3034, 0x24CD9 }, { 3035, 0x24D06 }, { 3036, 0x3F58 }, { 3037, 0x7555 }, { 3038, 0x7673 }, { 3039, 0x2A5C6 }, { 3040, 0x3B19 },
    { 3041, 0x7468 }, { 3042, 0x28ACC }, { 3043, 0x249AB }, { 3044, 0x2498E }, { 3045, 0x3AFB }, { 3046, 0x3DCD }, { 3047, 0x24A4E }, { 3048, 0x3EFF },
    { 3049, 0x249C5 }, { 3050, 0x248F3 }, { 3051, 0x91FA }, { 3052, 0x5732 }, { 3053, 0x9342 }, { 3054, 0x28AE3 }, { 3055, 0x21864 }, { 3056, 0x50DF },
    { 3057, 0x25221 }, { 3058, 0x251E7 }, { 3059, 0x7778 }, { 3060, 0x23232 }, { 3061, 0x770E }, { 3062, 0x770F }, { 3063, 0x777B }, { 3064, 0x24697 },
    { 3065, 0x23781 }, { 3066, 0x3A5E }, { 3067, 0x248F0 }, { 3068, 0x7438 }, { 3069, 0x749B }, { 3070, 0x3EBF }, { 3071, 0x24ABA }, { 3072, 0x24AC7 },
    { 3073, 0x40C8 }, { 3074, 0x24A96 }, { 3075, 0x261AE }, { 3076, 0x9307 }, { 3077, 0x25581 }, { 3078, 0x781E }, { 3079, 0x788D }, { 3080, 0x7888 },
    { 3081, 0x78D2 }, { 3082, 0x73D0 }, { 3083, 0x7959 }, { 3084, 0x27741 }, { 3085, 0x256E3 }, { 3086, 0x410E }, { 3087, 0x799B }, { 3088, 0x8496 },
    { 3089, 0x79A5 }, { 3090, 0x6A2D }, { 3091, 0x23EFA }, { 3092, 0x7A3A }, { 3093, 0x79F4 }, { 3094, 0x416E }, { 3095, 0x216E6 }, { 3096, 0x4132 },
    { 3097, 0x9235 }, { 3098, 0x79F1 }, { 3099, 0x20D4C }, { 3100, 0x2498C }, { 3101, 0x20299 }, { 3102, 0x23DBA }, { 3103, 0x2176E }, { 3104, 0x3597 },
    { 3105, 0x556B }, { 3106, 0x3570 }, { 3107, 0x36AA }, { 3108, 0x201D4 }, { 3109, 0x20C0D }, { 3110, 0x7AE2 }, { 3111, 0x5A59 }, { 3112, 0x226F5 },
    { 3113, 0x25AAF }, { 3114, 0x25A9C }, { 3115, 0x5A0D }, { 3116, 0x2025B }, { 3117, 0x78F0 }, { 3118, 0x5A2A }, { 3119, 0x25BC6 }, { 3120, 0x7AFE },
    { 3121, 0x41F9 }, { 3122, 0x7C5D }, { 3123, 0x7C6D }, { 3124, 0x4211 }, { 3125, 0x25BB3 }, { 3126, 0x25EBC }, { 3127, 0x25EA6 }, { 3128, 0x7CCD },
    { 3129, 0x249F9 }, { 3130, 0x217B0 }, { 3131, 0x7C8E }, { 3132, 0x7C7C }, { 3133, 0x7CAE }, { 3134, 0x6AB2 }, { 3135, 0x7DDC }, { 3136, 0x7E07 },
    { 3137, 0x7DD3 }, { 3138, 0x7F4E }, { 3139, 0x26261 }, { 3140, 0x2615C }, { 3141, 0x27B48 }, { 3142, 0x7D97 }, { 3143, 0x25E82 }, { 3144, 0x426A },
    { 3145, 0x26B75 }, { 3146, 0x20916 }, { 3147, 0x67D6 }, { 3148, 0x2004E }, { 3149, 0x235CF }, { 3150, 0x57C4 }, { 3151, 0x26412 }, { 3152, 0x263F8 },
    { 3153, 0x24962 }, { 3154, 0x7FDD }, { 3155, 0x7B27 }, { 3156, 0x2082C }, { 3157, 0x25AE9 }, { 3158, 0x25D43 }, { 3159, 0x7B0C }, { 3160, 0x25E0E },
    { 3161, 0x99E6 }, { 3162, 0x8645 }, { 3163, 0x9A63 }, { 3164, 0x6A1C }, { 3165, 0x2343F }, { 3166, 0x39E2 }, { 3167, 0x249F7 }, { 3168, 0x265AD },
    { 3169, 0x9A1F }, { 3170, 0x265A0 }, { 3171, 0x8480 }, { 3172, 0x27127 }, { 3173, 0x26CD1 }, { 3174, 0x44EA }, { 3175, 0x8137 }, { 3176, 0x4402 },
    { 3177, 0x80C6 }, { 3178, 0x8109 }, { 3179, 0x8142 }, { 3180, 0x267B4 }, { 3181, 0x98C3 }, { 3182, 0x26A42 }, { 3183, 0x8262 }, { 3184, 0x8265 },
    { 3185, 0x26A51 }, { 3186, 0x8453 }, { 3187, 0x26DA7 }, { 3188, 0x8610 }, { 3189, 0x2721B }, { 3190, 0x5A86 }, { 3191, 0x417F }, { 3192, 0x21840 },
    { 3193, 0x5B2B }, { 3194, 0x218A1 }, { 3195, 0x5AE4 }, { 3196, 0x218D8 }, { 3197, 0x86A0 }, { 3198, 0x2F9BC }, { 3199, 0x23D8F }, { 3200, 0x882D },
    { 3201, 0x27422 }, { 3202, 0x5A02 }, { 3203, 0x886E }, { 3204, 0x4F45 }, { 3205, 0x8887 }, { 3206, 0x88BF }, { 3207, 0x88E6 }, { 3208, 0x8965 },
    { 3209, 0x894D }, { 3210, 0x25683 }, { 3211, 0x8954 }, { 3212, 0x27785 }, { 3213, 0x27784 }, { 3214, 0x28BF5 }, { 3215, 0x28BD9 }, { 3216, 0x28B9C },
    { 3217, 0x289F9 }, { 3218, 0x3EAD }, { 3219, 0x84A3 }, { 3220, 0x46F5 }, { 3221, 0x46CF }, { 3222, 0x37F2 }, { 3223, 0x8A3D }, { 3224, 0x8A1C },
    { 3225, 0x29448 }, { 3226, 0x5F4D }, { 3227, 0x922B }, { 3228, 0x24284 }, { 3229, 0x65D4 }, { 3230, 0x7129 }, { 3231, 0x70C4 }, { 3232, 0x21845 },
    { 3233, 0x9D6D }, { 3234, 0x8C9F }, { 3235, 0x8CE9 }, { 3236, 0x27DDC }, { 3237, 0x599A }, { 3238, 0x77C3 }, { 3239, 0x59F0 }, { 3240, 0x436E },
    { 3241, 0x36D4 }, { 3242, 0x8E2A }, { 3243, 0x8EA7 }, { 3244, 0x24C09 }, { 3245, 0x8F30 }, { 3246, 0x8F4A }, { 3247, 0x42F4 }, { 3248, 0x6C58 },
    { 3249, 0x6FBB }, { 3250, 0x22321 }, { 3251, 0x489B }, { 3252, 0x6F79 }, { 3253, 0x6E8B }, { 3254, 0x217DA }, { 3255, 0x9BE9 }, { 3256, 0x36B5 },
    { 3257, 0x2492F }, { 3258, 0x90BB }, { 3259, 0x9097 }, { 3260, 0x5571 }, { 3261, 0x4906 }, { 3262, 0x91BB }, { 3263, 0x9404 }, { 3264, 0x28A4B },
    { 3265, 0x4062 }, { 3266, 0x28AFC }, { 3267, 0x9427 }, { 3268, 0x28C1D }, { 3269, 0x28C3B }, { 3270, 0x84E5 }, { 3271, 0x8A2B }, { 3272, 0x9599 },
    { 3273, 0x95A7 }, { 3274, 0x9597 }, { 3275, 0x9596 }, { 3276, 0x28D34 }, { 3277, 0x7445 }, { 3278, 0x3EC2 }, { 3279, 0x248FF }, { 3280, 0x24A42 },
    { 3281, 0x243EA }, { 3282, 0x3EE7 }, { 3283, 0x23225 }, { 3284, 0x968F }, { 3285, 0x28EE7 }, { 3286, 0x28E66 }, { 3287, 0x28E65 }, { 3288, 0x3ECC },
    { 3289, 0x249ED }, { 3290, 0x24A78 }, { 3291, 0x23FEE }, { 3292, 0x7412 }, { 3293, 0x746B }, { 3294, 0x3EFC }, { 3295, 0x9741 }, { 3296, 0x290B0 },
    { 3297, 0x6847 }, { 3298, 0x4A1D }, { 3299, 0x29093 }, { 3300, 0x257DF }, { 3301, 0x975D }, { 3302, 0x9368 }, { 3303, 0x28989 }, { 3304, 0x28C26 },
    { 3305, 0x28B2F }, { 3306, 0x263BE }, { 3307, 0x92BA }, { 3308, 0x5B11 }, { 3309, 0x8B69 }, { 3310, 0x493C }, { 3311, 0x73F9 }, { 3312, 0x2421B },
    { 3313, 0x979B }, { 3314, 0x9771 }, { 3315, 0x9938 }, { 3316, 0x20F26 }, { 3317, 0x5DC1 }, { 3318, 0x28BC5 }, { 3319, 0x24AB2 }, { 3320, 0x981F },
    { 3321, 0x294DA }, { 3322, 0x92F6 }, { 3323, 0x295D7 }, { 3324, 0x91E5 }, { 3325, 0x44C0 }, { 3326, 0x28B50 }, { 3327, 0x24A67 }, { 3328, 0x28B64 },
    { 3329, 0x98DC }, { 3330, 0x28A45 }, { 3331, 0x3F00 }, { 3332, 0x922A }, { 3333, 0x4925 }, { 3334, 0x8414 }, { 3335, 0x993B }, { 3336, 0x994D },
    { 3337, 0x27B06 }, { 3338, 0x3DFD }, { 3339, 0x999B }, { 3340, 0x4B6F }, { 3341, 0x99AA }, { 3342, 0x9A5C }, { 3343, 0x28B65 }, { 3344, 0x258C8 },
    { 3345, 0x6A8F }, { 3346, 0x9A21 }, { 3347, 0x5AFE }, { 3348, 0x9A2F }, { 3349, 0x298F1 }, { 3350, 0x4B90 }, { 3351, 0x29948 }, { 3352, 0x99BC },
    { 3353, 0x4BBD }, { 3354, 0x4B97 }, { 3355, 0x937D }, { 3356, 0x5872 }, { 3357, 0x21302 }, { 3358, 0x5822 }, { 3359, 0x249B8 }, { 3360, 0x214E8 },
    { 3361, 0x7844 }, { 3362, 0x2271F }, { 3363, 0x23DB8 }, { 3364, 0x68C5 }, { 3365, 0x3D7D }, { 3366, 0x9458 }, { 3367, 0x3927 }, { 3368, 0x6150 },
    { 3369, 0x22781 }, { 3370, 0x2296B }, { 3371, 0x6107 }, { 3372, 0x9C4F }, { 3373, 0x9C53 }, { 3374, 0x9C7B }, { 3375, 0x9C35 }, { 3376, 0x9C10 },
    { 3377, 0x9B7F }, { 3378, 0x9BCF }, { 3379, 0x29E2D }, { 3380, 0x9B9F }, { 3381, 0x2A1F5 }, { 3382, 0x2A0FE }, { 3383, 0x9D21 }, { 3384, 0x4CAE },
    { 3385, 0x24104 }, { 3386, 0x9E18 }, { 3387, 0x4CB0 }, { 3388, 0x9D0C }, { 3389, 0x2A1B4 }, { 3390, 0x2A0ED }, { 3391, 0x2A0F3 }, { 3392, 0x2992F },
    { 3393, 0x9DA5 }, { 3394, 0x84BD }, { 3395, 0x26E12 }, { 3396, 0x26FDF }, { 3397, 0x26B82 }, { 3398, 0x85FC }, { 3399, 0x4533 }, { 3400, 0x26DA4 },
    { 3401, 0x26E84 }, { 3402, 0x26DF0 }, { 3403, 0x8420 }, { 3404, 0x85EE }, { 3405, 0x26E00 }, { 3406, 0x237D7 }, { 3407, 0x26064 }, { 3408, 0x79E2 },
    { 3409, 0x2359C }, { 3410, 0x23640 }, { 3411, 0x492D }, { 3412, 0x249DE }, { 3413, 0x3D62 }, { 3414, 0x93DB }, { 3415, 0x92BE }, { 3416, 0x9348 },
    { 3417, 0x202BF }, { 3418, 0x78B9 }, { 3419, 0x9277 }, { 3420, 0x944D }, { 3421, 0x4FE4 }, { 3422, 0x3440 }, { 3423, 0x9064 }, { 3424, 0x2555D },
    { 3425, 0x783D }, { 3426, 0x7854 }, { 3427, 0x78B6 }, { 3428, 0x784B }, { 3429, 0x21757 }, { 3430, 0x231C9 }, { 3431, 0x24941 }, { 3432, 0x369A },
    { 3433, 0x4F72 }, { 3434, 0x6FDA }, { 3435, 0x6FD9 }, { 3436, 0x701E }, { 3437, 0x701E }, { 3438, 0x5414 }, { 3439, 0x241B5 }, { 3440, 0x57BB },
    { 3441, 0x58F3 }, { 3442, 0x578A }, { 3443, 0x9D16 }, { 3444, 0x57D7 }, { 3445, 0x7134 }, { 3446, 0x34AF }, { 3447, 0x241AC }, { 3448, 0x71EB },
    { 3449, 0x26C40 }, { 3450, 0x24F97 }, { 3451, 0x5B28 }, { 3452, 0x217B5 }, { 3453, 0x28A49 }, { 3454, 0x610C }, { 3455, 0x5ACE }, { 3456, 0x5A0B },
    { 3457, 0x42BC }, { 3458, 0x24488 }, { 3459, 0x372C }, { 3460, 0x4B7B }, { 3461, 0x289FC }, { 3462, 0x93BB }, { 3463, 0x93B8 }, { 3464, 0x218D6 },
    { 3465, 0x20F1D }, { 3466, 0x8472 }, { 3467, 0x26CC0 }, { 3468, 0x21413 }, { 3469, 0x242FA }, { 3470, 0x22C26 }, { 3471, 0x243C1 }, { 3472, 0x5994 },
    { 3473, 0x23DB7 }, { 3474, 0x26741 }, { 3475, 0x7DA8 }, { 3476, 0x2615B }, { 3477, 0x260A4 }, { 3478, 0x249B9 }, { 3479, 0x2498B }, { 3480, 0x289FA },
    { 3481, 0x92E5 }, { 3482, 0x73E2 }, { 3483, 0x3EE9 }, { 3484, 0x74B4 }, { 3485, 0x28B63 }, { 3486, 0x2189F }, { 3487, 0x3EE1 }, { 3488, 0x24AB3 },
    { 3489, 0x6AD8 }, { 3490, 0x73F3 }, { 3491, 0x73FB }, { 3492, 0x3ED6 }, { 3493, 0x24A3E }, { 3494, 0x24A94 }, { 3495, 0x217D9 }, { 3496, 0x24A66 },
    { 3497, 0x203A7 }, { 3498, 0x21424 }, { 3499, 0x249E5 }, { 3500, 0x7448 }, { 3501, 0x24916 }, { 3502, 0x70A5 }, { 3503, 0x24976 }, { 3504, 0x9284 },
    { 3505, 0x73E6 }, { 3506, 0x935F }, { 3507, 0x204FE }, { 3508, 0x9331 }, { 3509, 0x28ACE }, { 3510, 0x28A16 }, { 3511, 0x9386 }, { 3512, 0x28BE7 },
    { 3513, 0x255D5 }, { 3514, 0x4935 }, { 3515, 0x28A82 }, { 3516, 0x716B }, { 3517, 0x24943 }, { 3518, 0x20CFF }, { 3519, 0x56A4 }, { 3520, 0x2061A },
    { 3521, 0x20BEB }, { 3522, 0x20CB8 }, { 3523, 0x5502 }, { 3524, 0x79C4 }, { 3525, 0x217FA }, { 3526, 0x7DFE }, { 3527, 0x216C2 }, { 3528, 0x24A50 },
    { 3529, 0x21852 }, { 3530, 0x452E }, { 3531, 0x9401 }, { 3532, 0x370A }, { 3533, 0x28AC0 }, { 3534, 0x249AD }, { 3535, 0x59B0 }, { 3536, 0x218BF },
    { 3537, 0x21883 }, { 3538, 0x27484 }, { 3539, 0x5AA1 }, { 3540, 0x36E2 }, { 3541, 0x23D5B }, { 3542, 0x36B0 }, { 3543, 0x925F }, { 3544, 0x5A79 },
    { 3545, 0x28A81 }, { 3546, 0x21862 }, { 3547, 0x9374 }, { 3548, 0x3CCD }, { 3549, 0x20AB4 }, { 3550, 0x4A96 }, { 3551, 0x398A }, { 3552, 0x50F4 },
    { 3553, 0x3D69 }, { 3554, 0x3D4C }, { 3555, 0x2139C }, { 3556, 0x7175 }, { 3557, 0x42FB }, { 3558, 0x28218 }, { 3559, 0x6E0F }, { 3560, 0x290E4 },
    { 3561, 0x44EB }, { 3562, 0x6D57 }, { 3563, 0x27E4F }, { 3564, 0x7067 }, { 3565, 0x6CAF }, { 3566, 0x3CD6 }, { 3567, 0x23FED }, { 3568, 0x23E2D },
    { 3569, 0x6E02 }, { 3570, 0x6F0C }, { 3571, 0x3D6F }, { 3572, 0x203F5 }, { 3573, 0x7551 }, { 3574, 0x36BC }, { 3575, 0x34C8 }, { 3576, 0x4680 },
    { 3577, 0x3EDA }, { 3578, 0x4871 }, { 3579, 0x59C4 }, { 3580, 0x926E }, { 3581, 0x493E }, { 3582, 0x8F41 }, { 3583, 0x28C1C }, { 3584, 0x26BC0 },
    { 3585, 0x5812 }, { 3586, 0x57C8 }, { 3587, 0x36D6 }, { 3588, 0x21452 }, { 3589, 0x70FE }, { 3590, 0x24362 }, { 3591, 0x24A71 }, { 3592, 0x22FE3 },
    { 3593, 0x212B0 }, { 3594, 0x223BD }, { 3595, 0x68B9 }, { 3596, 0x6967 }, { 3597, 0x21398 }, { 3598, 0x234E5 }, { 3599, 0x27BF4 }, { 3600, 0x236DF },
    { 3601, 0x28A83 }, { 3602, 0x237D6 }, { 3603, 0x233FA }, { 3604, 0x24C9F }, { 3605, 0x6A1A }, { 3606, 0x236AD }, { 3607, 0x26CB7 }, { 3608, 0x843E },
    { 3609, 0x44DF }, { 3610, 0x44CE }, { 3611, 0x26D26 }, { 3612, 0x26D51 }, { 3613, 0x26C82 }, { 3614, 0x26FDE }, { 3615, 0x6F17 }, { 3616, 0x27109 },
    { 3617, 0x833D }, { 3618, 0x2173A }, { 3619, 0x83ED }, { 3620, 0x26C80 }, { 3621, 0x27053 }, { 3622, 0x217DB }, { 3623, 0x5989 }, { 3624, 0x5A82 },
    { 3625, 0x217B3 }, { 3626, 0x5A61 }, { 3627, 0x5A71 }, { 3628, 0x21905 }, { 3629, 0x241FC }, { 3630, 0x372D }, { 3631, 0x59EF }, { 3632, 0x2173C },
    { 3633, 0x36C7 }, { 3634, 0x718E }, { 3635, 0x9390 }, { 3636, 0x669A }, { 3637, 0x242A5 }, { 3638, 0x5A6E }, { 3639, 0x5A2B }, { 3640, 0x24293 },
    { 3641, 0x6A2B }, { 3642, 0x23EF9 }, { 3643, 0x27736 }, { 3644, 0x2445B }, { 3645, 0x242CA }, { 3646, 0x711D }, { 3647, 0x24259 }, { 3648, 0x289E1 },
    { 3649, 0x4FB0 }, { 3650, 0x26D28 }, { 3651, 0x5CC2 }, { 3652, 0x244CE }, { 3653, 0x27E4D }, { 3654, 0x243BD }, { 3655, 0x6A0C }, { 3656, 0x24256 },
    { 3657, 0x21304 }, { 3658, 0x70A6 }, { 3659, 0x7133 }, { 3660, 0x243E9 }, { 3661, 0x3DA5 }, { 3662, 0x6CDF }, { 3663, 0x2F825 }, { 3664, 0x24A4F },
    { 3665, 0x7E65 }, { 3666, 0x59EB }, { 3667, 0x5D2F }, { 3668, 0x3DF3 }, { 3669, 0x5F5C }, { 3670, 0x24A5D }, { 3671, 0x217DF }, { 3672, 0x7DA4 },
    { 3673, 0x8426 }, { 3674, 0x5485 }, { 3675, 0x23AFA }, { 3676, 0x23300 }, { 3677, 0x20214 }, { 3678, 0x577E }, { 3679, 0x208D5 }, { 3680, 0x20619 },
    { 3681, 0x3FE5 }, { 3682, 0x21F9E }, { 3683, 0x2A2B6 }, { 3684, 0x7003 }, { 3685, 0x2915B }, { 3686, 0x5D70 }, { 3687, 0x738F }, { 3688, 0x7CD3 },
    { 3689, 0x28A59 }, { 3690, 0x29420 }, { 3691, 0x4FC8 }, { 3692, 0x7FE7 }, { 3693, 0x72CD }, { 3694, 0x7310 }, { 3695, 0x27AF4 }, { 3696, 0x7338 },
    { 3697, 0x7339 }, { 3698, 0x256F6 }, { 3699, 0x7341 }, { 3700, 0x7348 }, { 3701, 0x3EA9 }, { 3702, 0x27B18 }, { 3703, 0x906C }, { 3704, 0x71F5 },
    { 3705, 0x248F2 }, { 3706, 0x73E1 }, { 3707, 0x81F6 }, { 3708, 0x3ECA }, { 3709, 0x770C }, { 3710, 0x3ED1 }, { 3711, 0x6CA2 }, { 3712, 0x56FD },
    { 3713, 0x7419 }, { 3714, 0x741E }, { 3715, 0x741F }, { 3716, 0x3EE2 }, { 3717, 0x3EF0 }, { 3718, 0x3EF4 }, { 3719, 0x3EFA }, { 3720, 0x74D3 },
    { 3721, 0x3F0E }, { 3722, 0x3F53 }, { 3723, 0x7542 }, { 3724, 0x756D }, { 3725, 0x7572 }, { 3726, 0x758D }, { 3727, 0x3F7C }, { 3728, 0x75C8 },
    { 3729, 0x75DC }, { 3730, 0x3FC0 }, { 3731, 0x764D }, { 3732, 0x3FD7 }, { 3733, 0x7674 }, { 3734, 0x3FDC }, { 3735, 0x767A }, { 3736, 0x24F5C },
    { 3737, 0x7188 }, { 3738, 0x5623 }, { 3739, 0x8980 }, { 3740, 0x5869 }, { 3741, 0x401D }, { 3742, 0x7743 }, { 3743, 0x4039 }, { 3744, 0x6761 },
    { 3745, 0x4045 }, { 3746, 0x35DB }, { 3747, 0x7798 }, { 3748, 0x406A }, { 3749, 0x406F }, { 3750, 0x5C5E }, { 3751, 0x77BE }, { 3752, 0x77CB },
    { 3753, 0x58F2 }, { 3754, 0x7818 }, { 3755, 0x70B9 }, { 3756, 0x781C }, { 3757, 0x40A8 }, { 3758, 0x7839 }, { 3759, 0x7847 }, { 3760, 0x7851 },
    { 3761, 0x7866 }, { 3762, 0x8448 }, { 3763, 0x25535 }, { 3764, 0x7933 }, { 3765, 0x6803 }, { 3766, 0x7932 }, { 3767, 0x4103 }, { 3768, 0x4109 },
    { 3769, 0x7991 }, { 3770, 0x7999 }, { 3771, 0x8FBB }, { 3772, 0x7A06 }, { 3773, 0x8FBC }, { 3774, 0x4167 }, { 3775, 0x7A91 }, { 3776, 0x41B2 },
    { 3777, 0x7ABC }, { 3778, 0x8279 }, { 3779, 0x41C4 }, { 3780, 0x7ACF }, { 3781, 0x7ADB }, { 3782, 0x41CF }, { 3783, 0x4E21 }, { 3784, 0x7B62 },
    { 3785, 0x7B6C }, { 3786, 0x7B7B }, { 3787, 0x7C12 }, { 3788, 0x7C1B }, { 3789, 0x4260 }, { 3790, 0x427A }, { 3791, 0x7C7B }, { 3792, 0x7C9C },
    { 3793, 0x428C }, { 3794, 0x7CB8 }, { 3795, 0x4294 }, { 3796, 0x7CED }, { 3797, 0x8F93 }, { 3798, 0x70C0 }, { 3799, 0x20CCF }, { 3800, 0x7DCF },
    { 3801, 0x7DD4 }, { 3802, 0x7DD0 }, { 3803, 0x7DFD }, { 3804, 0x7FAE }, { 3805, 0x7FB4 }, { 3806, 0x729F }, { 3807, 0x4397 }, { 3808, 0x8020 },
    { 3809, 0x8025 }, { 3810, 0x7B39 }, { 3811, 0x802E }, { 3812, 0x8031 }, { 3813, 0x8054 }, { 3814, 0x3DCC }, { 3815, 0x57B4 }, { 3816, 0x70A0 },
    { 3817, 0x80B7 }, { 3818, 0x80E9 }, { 3819, 0x43ED }, { 3820, 0x810C }, { 3821, 0x732A }, { 3822, 0x810E }, { 3823, 0x8112 }, { 3824, 0x7560 },
    { 3825, 0x8114 }, { 3826, 0x4401 }, { 3827, 0x3B39 }, { 3828, 0x8156 }, { 3829, 0x8159 }, { 3830, 0x815A }, { 3831, 0x4413 }, { 3832, 0x583A },
    { 3833, 0x817C }, { 3834, 0x8184 }, { 3835, 0x4425 }, { 3836, 0x8193 }, { 3837, 0x442D }, { 3838, 0x81A5 }, { 3839, 0x57EF }, { 3840, 0x81C1 },
    { 3841, 0x81E4 }, { 3842, 0x8254 }, { 3843, 0x448F }, { 3844, 0x82A6 }, { 3845, 0x8276 }, { 3846, 0x82CA }, { 3847, 0x82D8 }, { 3848, 0x82FF },
    { 3849, 0x44B0 }, { 3850, 0x8357 }, { 3851, 0x9669 }, { 3852, 0x698A }, { 3853, 0x8405 }, { 3854, 0x70F5 }, { 3855, 0x8464 }, { 3856, 0x60E3 },
    { 3857, 0x8488 }, { 3858, 0x4504 }, { 3859, 0x84BE }, { 3860, 0x84E1 }, { 3861, 0x84F8 }, { 3862, 0x8510 }, { 3863, 0x8538 }, { 3864, 0x8552 },
    { 3865, 0x453B }, { 3866, 0x856F }, { 3867, 0x8570 }, { 3868, 0x85E0 }, { 3869, 0x4577 }, { 3870, 0x8672 }, { 3871, 0x8692 }, { 3872, 0x86B2 },
    { 3873, 0x86EF }, { 3874, 0x9645 }, { 3875, 0x878B }, { 3876, 0x4606 }, { 3877, 0x4617 }, { 3878, 0x88AE }, { 3879, 0x88FF }, { 3880, 0x8924 },
    { 3881, 0x8947 }, { 3882, 0x8991 }, { 3883, 0x27967 }, { 3884, 0x8A29 }, { 3885, 0x8A38 }, { 3886, 0x8A94 }, { 3887, 0x8AB4 }, { 3888, 0x8C51 },
    { 3889, 0x8CD4 }, { 3890, 0x8CF2 }, { 3891, 0x8D1C }, { 3892, 0x4798 }, { 3893, 0x585F }, { 3894, 0x8DC3 }, { 3895, 0x47ED }, { 3896, 0x4EEE },
    { 3897, 0x8E3A }, { 3898, 0x55D8 }, { 3899, 0x5754 }, { 3900, 0x8E71 }, { 3901, 0x55F5 }, { 3902, 0x8EB0 }, { 3903, 0x4837 }, { 3904, 0x8ECE },
    { 3905, 0x8EE2 }, { 3906, 0x8EE4 }, { 3907, 0x8EED }, { 3908, 0x8EF2 }, { 3909, 0x8FB7 }, { 3910, 0x8FC1 }, { 3911, 0x8FCA }, { 3912, 0x8FCC },
    { 3913, 0x9033 }, { 3914, 0x99C4 }, { 3915, 0x48AD }, { 3916, 0x98E0 }, { 3917, 0x9213 }, { 3918, 0x491E }, { 3919, 0x9228 }, { 3920, 0x9258 },
    { 3921, 0x926B }, { 3922, 0x92B1 }, { 3923, 0x92AE }, { 3924, 0x92BF }, { 3925, 0x92E3 }, { 3926, 0x92EB }, { 3927, 0x92F3 }, { 3928, 0x92F4 },
    { 3929, 0x92FD }, { 3930, 0x9343 }, { 3931, 0x9384 }, { 3932, 0x93AD }, { 3933, 0x4945 }, { 3934, 0x4951 }, { 3935, 0x9EBF }, { 3936, 0x9417 },
    { 3937, 0x5301 }, { 3938, 0x941D }, { 3939, 0x942D }, { 3940, 0x943E }, { 3941, 0x496A }, { 3942, 0x9454 }, { 3943, 0x9479 }, { 3944, 0x952D },
    { 3945, 0x95A2 }, { 3946, 0x49A7 }, { 3947, 0x95F4 }, { 3948, 0x9633 }, { 3949, 0x49E5 }, { 3950, 0x67A0 }, { 3951, 0x4A24 }, { 3952, 0x9740 },
    { 3953, 0x4A35 }, { 3954, 0x97B2 }, { 3955, 0x97C2 }, { 3956, 0x5654 }, { 3957, 0x4AE4 }, { 3958, 0x60E8 }, { 3959, 0x98B9 }, { 3960, 0x4B19 },
    { 3961, 0x98F1 }, { 3962, 0x5844 }, { 3963, 0x990E }, { 3964, 0x9919 }, { 3965, 0x51B4 }, { 3966, 0x991C }, { 3967, 0x9937 }, { 3968, 0x9942 },
    { 3969, 0x995D }, { 3970, 0x9962 }, { 3971, 0x4B70 }, { 3972, 0x99C5 }, { 3973, 0x4B9D }, { 3974, 0x9A3C }, { 3975, 0x9B0F }, { 3976, 0x7A83 },
    { 3977, 0x9B69 }, { 3978, 0x9B81 }, { 3979, 0x9BDD }, { 3980, 0x9BF1 }, { 3981, 0x9BF4 }, { 3982, 0x4C6D }, { 3983, 0x9C20 }, { 3984, 0x376F },
    { 3985, 0x21BC2 }, { 3986, 0x9D49 }, { 3987, 0x9C3A }, { 3988, 0x9EFE }, { 3989, 0x5650 }, { 3990, 0x9D93 }, { 3991, 0x9DBD }, { 3992, 0x9DC0 },
    { 3993, 0x9DFC }, { 3994, 0x94F6 }, { 3995, 0x8FB6 }, { 3996, 0x9E7B }, { 3997, 0x9EAC }, { 3998, 0x9EB1 }, { 3999, 0x9EBD }, { 4000, 0x9EC6 },
    { 4001, 0x94DC }, { 4002, 0x9EE2 }, { 4003, 0x9EF1 }, { 4004, 0x9EF8 }, { 4005, 0x7AC8 }, { 4006, 0x9F44 }, { 4007, 0x20094 }, { 4008, 0x202B7 },
    { 4009, 0x203A0 }, { 4010, 0x691A }, { 4011, 0x94C3 }, { 4012, 0x59AC }, { 4013, 0x204D7 }, { 4014, 0x5840 }, { 4015, 0x94C1 }, { 4016, 0x37B9 },
    { 4017, 0x205D5 }, { 4018, 0x20615 }, { 4019, 0x20676 }, { 4020, 0x216BA }, { 4021, 0x5757 }, { 4022, 0x7173 }, { 4023, 0x20AC2 }, { 4024, 0x20ACD },
    { 4025, 0x20BBF }, { 4026, 0x546A }, { 4027, 0x2F83B }, { 4028, 0x20BCB }, { 4029, 0x549E }, { 4030, 0x20BFB }, { 4031, 0x20C3B }, { 4032, 0x20C53 },
    { 4033, 0x20C65 }, { 4034, 0x20C7C }, { 4035, 0x60E7 }, { 4036, 0x20C8D }, { 4037, 0x567A }, { 4038, 0x20CB5 }, { 4039, 0x20CDD }, { 4040, 0x20CED },
    { 4041, 0x20D6F }, { 4042, 0x20DB2 }, { 4043, 0x20DC8 }, { 4044, 0x6955 }, { 4045, 0x9C2F }, { 4046, 0x87A5 }, { 4047, 0x20E04 }, { 4048, 0x20E0E },
    { 4049, 0x20ED7 }, { 4050, 0x20F90 }, { 4051, 0x20F2D }, { 4052, 0x20E73 }, { 4053, 0x5C20 }, { 4054, 0x20FBC }, { 4055, 0x5E0B }, { 4056, 0x2105C },
    { 4057, 0x2104F }, { 4058, 0x21076 }, { 4059, 0x671E }, { 4060, 0x2107B }, { 4061, 0x21088 }, { 4062, 0x21096 }, { 4063, 0x3647 }, { 4064, 0x210BF },
    { 4065, 0x210D3 }, { 4066, 0x2112F }, { 4067, 0x2113B }, { 4068, 0x5364 }, { 4069, 0x84AD }, { 4070, 0x212E3 }, { 4071, 0x21375 }, { 4072, 0x21336 },
    { 4073, 0x8B81 }, { 4074, 0x21577 }, { 4075, 0x21619 }, { 4076, 0x217C3 }, { 4077, 0x217C7 }, { 4078, 0x4E78 }, { 4079, 0x70BB }, { 4080, 0x2182D },
    { 4081, 0x2196A }, { 4082, 0x21A2D }, { 4083, 0x21A45 }, { 4084, 0x21C2A }, { 4085, 0x21C70 }, { 4086, 0x21CAC }, { 4087, 0x21EC8 }, { 4088, 0x62C3 },
    { 4089, 0x21ED5 }, { 4090, 0x21F15 }, { 4091, 0x7198 }, { 4092, 0x6855 }, { 4093, 0x22045 }, { 4094, 0x69E9 }, { 4095, 0x36C8 }, { 4096, 0x2227C },
    { 4097, 0x223D7 }, { 4098, 0x223FA }, { 4099, 0x2272A }, { 4100, 0x22871 }, { 4101, 0x2294F }, { 4102, 0x82FD }, { 4103, 0x22967 }, { 4104, 0x22993 },
    { 4105, 0x22AD5 }, { 4106, 0x89A5 }, { 4107, 0x22AE8 }, { 4108, 0x8FA0 }, { 4109, 0x22B0E }, { 4110, 0x97B8 }, { 4111, 0x22B3F }, { 4112, 0x9847 },
    { 4113, 0x9ABD }, { 4114, 0x22C4C }, { 4116, 0x22C88 }, { 4117, 0x22CB7 }, { 4118, 0x25BE8 }, { 4119, 0x22D08 }, { 4120, 0x22D12 }, { 4121, 0x22DB7 },
    { 4122, 0x22D95 }, { 4123, 0x22E42 }, { 4124, 0x22F74 }, { 4125, 0x22FCC }, { 4126, 0x23033 }, { 4127, 0x23066 }, { 4128, 0x2331F }, { 4129, 0x233DE },
    { 4130, 0x5FB1 }, { 4131, 0x6648 }, { 4132, 0x66BF }, { 4133, 0x27A79 }, { 4134, 0x23567 }, { 4135, 0x235F3 }, { 4136, 0x7201 }, { 4137, 0x249BA },
    { 4138, 0x77D7 }, { 4139, 0x2361A }, { 4140, 0x23716 }, { 4141, 0x7E87 }, { 4142, 0x20346 }, { 4143, 0x58B5 }, { 4144, 0x670E }, { 4145, 0x6918 },
    { 4146, 0x23AA7 }, { 4147, 0x27657 }, { 4148, 0x25FE2 }, { 4149, 0x23E11 }, { 4150, 0x23EB9 }, { 4151, 0x275FE }, { 4152, 0x2209A }, { 4153, 0x48D0 },
    { 4154, 0x4AB8 }, { 4155, 0x24119 }, { 4156, 0x28A9A }, { 4157, 0x242EE }, { 4158, 0x2430D }, { 4159, 0x2403B }, { 4160, 0x24334 }, { 4161, 0x24396 },
    { 4162, 0x24A45 }, { 4163, 0x205CA }, { 4164, 0x51D2 }, { 4165, 0x20611 }, { 4166, 0x599F }, { 4167, 0x21EA8 }, { 4168, 0x3BBE }, { 4169, 0x23CFF },
    { 4170, 0x24404 }, { 4171, 0x244D6 }, { 4172, 0x5788 }, { 4173, 0x24674 }, { 4174, 0x399B }, { 4175, 0x2472F }, { 4176, 0x285E8 }, { 4177, 0x299C9 },
    { 4178, 0x3762 }, { 4179, 0x221C3 }, { 4180, 0x8B5E }, { 4181, 0x28B4E }, { 4182, 0x99D6 }, { 4183, 0x24812 }, { 4184, 0x248FB }, { 4185, 0x24A15 },
    { 4186, 0x7209 }, { 4187, 0x24AC0 }, { 4188, 0x20C78 }, { 4189, 0x5965 }, { 4190, 0x24EA5 }, { 4191, 0x24F86 }, { 4192, 0x20779 }, { 4193, 0x8EDA },
    { 4194, 0x2502C }, { 4195, 0x528F }, { 4196, 0x573F }, { 4197, 0x7171 }, { 4198, 0x25299 }, { 4199, 0x25419 }, { 4200, 0x23F4A }, { 4201, 0x24AA7 },
    { 4202, 0x55BC }, { 4203, 0x25446 }, { 4204, 0x2546E }, { 4205, 0x26B52 }, { 4206, 0x91D4 }, { 4207, 0x3473 }, { 4208, 0x2553F }, { 4209, 0x27632 },
    { 4210, 0x2555E }, { 4211, 0x4718 }, { 4212, 0x25562 }, { 4213, 0x25566 }, { 4214, 0x257C7 }, { 4215, 0x2493F }, { 4216, 0x2585D }, { 4217, 0x5066 },
    { 4218, 0x34FB }, { 4219, 0x233CC }, { 4220, 0x60DE }, { 4221, 0x25903 }, { 4222, 0x477C }, { 4223, 0x28948 }, { 4224, 0x25AAE }, { 4225, 0x25B89 },
    { 4226, 0x25C06 }, { 4227, 0x21D90 }, { 4228, 0x57A1 }, { 4229, 0x7151 }, { 4230, 0x6FB6 }, { 4231, 0x26102 }, { 4232, 0x27C12 }, { 4233, 0x9056 },
    { 4234, 0x261B2 }, { 4235, 0x24F9A }, { 4236, 0x8B62 }, { 4237, 0x26402 }, { 4238, 0x2644A }, { 4239, 0x5D5B }, { 4240, 0x26BF7 }, { 4241, 0x8F36 },
    { 4242, 0x26484 }, { 4243, 0x2191C }, { 4244, 0x8AEA }, { 4245, 0x249F6 }, { 4246, 0x26488 }, { 4247, 0x23FEF }, { 4248, 0x26512 }, { 4249, 0x4BC0 },
    { 4250, 0x265BF }, { 4251, 0x266B5 }, { 4252, 0x2271B }, { 4253, 0x9465 }, { 4254, 0x257E1 }, { 4255, 0x6195 }, { 4256, 0x5A27 }, { 4257, 0x2F8CD },
    { 4258, 0x4FBB }, { 4259, 0x56B9 }, { 4260, 0x24521 }, { 4261, 0x266FC }, { 4262, 0x4E6A }, { 4263, 0x24934 }, { 4264, 0x9656 }, { 4265, 0x6D8F },
    { 4266, 0x26CBD }, { 4267, 0x3618 }, { 4268, 0x8977 }, { 4269, 0x26799 }, { 4270, 0x2686E }, { 4271, 0x26411 }, { 4272, 0x2685E }, { 4273, 0x71DF },
    { 4274, 0x268C7 }, { 4275, 0x7B42 }, { 4276, 0x290C0 }, { 4277, 0x20A11 }, { 4278, 0x26926 }, { 4279, 0x9104 }, { 4280, 0x26939 }, { 4281, 0x7A45 },
    { 4282, 0x9DF0 }, { 4283, 0x269FA }, { 4284, 0x9A26 }, { 4285, 0x26A2D }, { 4286, 0x365F }, { 4287, 0x26469 }, { 4288, 0x20021 }, { 4289, 0x7983 },
    { 4290, 0x26A34 }, { 4291, 0x26B5B }, { 4292, 0x5D2C }, { 4293, 0x23519 }, { 4294, 0x83CF }, { 4295, 0x26B9D }, { 4296, 0x46D0 }, { 4297, 0x26CA4 },
    { 4298, 0x753B }, { 4299, 0x8865 }, { 4300, 0x26DAE }, { 4301, 0x58B6 }, { 4302, 0x371C }, { 4303, 0x2258D }, { 4304, 0x2704B }, { 4305, 0x271CD },
    { 4306, 0x3C54 }, { 4307, 0x27280 }, { 4308, 0x27285 }, { 4309, 0x9281 }, { 4310, 0x2217A }, { 4311, 0x2728B }, { 4312, 0x9330 }, { 4313, 0x272E6 },
    { 4314, 0x249D0 }, { 4315, 0x6C39 }, { 4316, 0x949F }, { 4317, 0x27450 }, { 4318, 0x20EF8 }, { 4319, 0x8827 }, { 4320, 0x88F5 }, { 4321, 0x22926 },
    { 4322, 0x28473 }, { 4323, 0x217B1 }, { 4324, 0x6EB8 }, { 4325, 0x24A2A }, { 4326, 0x21820 }, { 4327, 0x39A4 }, { 4328, 0x36B9 }, { 4329, 0x5C10 },
    { 4330, 0x79E3 }, { 4331, 0x453F }, { 4332, 0x66B6 }, { 4333, 0x29CAD }, { 4334, 0x298A4 }, { 4335, 0x8943 }, { 4336, 0x277CC }, { 4337, 0x27858 },
    { 4338, 0x56D6 }, { 4339, 0x40DF }, { 4340, 0x2160A }, { 4341, 0x39A1 }, { 4342, 0x2372F }, { 4343, 0x280E8 }, { 4344, 0x213C5 }, { 4345, 0x71AD },
    { 4346, 0x8366 }, { 4347, 0x279DD }, { 4348, 0x291A8 }, { 4349, 0x5A67 }, { 4350, 0x4CB7 }, { 4351, 0x270AF }, { 4352, 0x289AB }, { 4353, 0x279FD },
    { 4354, 0x27A0A }, { 4355, 0x27B0B }, { 4356, 0x27D66 }, { 4357, 0x2417A }, { 4358, 0x7B43 }, { 4359, 0x797E }, { 4360, 0x28009 }, { 4361, 0x6FB5 },
    { 4362, 0x2A2DF }, { 4363, 0x6A03 }, { 4364, 0x28318 }, { 4365, 0x53A2 }, { 4366, 0x26E07 }, { 4367, 0x93BF }, { 4368, 0x6836 }, { 4369, 0x975D },
    { 4370, 0x2816F }, { 4371, 0x28023 }, { 4372, 0x269B5 }, { 4373, 0x213ED }, { 4374, 0x2322F }, { 4375, 0x28048 }, { 4376, 0x5D85 }, { 4377, 0x28C30 },
    { 4378, 0x28083 }, { 4379, 0x5715 }, { 4380, 0x9823 }, { 4381, 0x28949 }, { 4382, 0x5DAB }, { 4383, 0x24988 }, { 4384, 0x65BE }, { 4385, 0x69D5 },
    { 4386, 0x53D2 }, { 4387, 0x24AA5 }, { 4388, 0x23F81 }, { 4389, 0x3C11 }, { 4390, 0x6736 }, { 4391, 0x28090 }, { 4392, 0x280F4 }, { 4393, 0x2812E },
    { 4394, 0x21FA1 }, { 4395, 0x2814F }, { 4396, 0x28189 }, { 4397, 0x281AF }, { 4398, 0x2821A }, { 4399, 0x28306 }, { 4400, 0x2832F }, { 4401, 0x2838A },
    { 4402, 0x35CA }, { 4403, 0x28468 }, { 4404, 0x286AA }, { 4405, 0x48FA }, { 4406, 0x63E6 }, { 4407, 0x28956 }, { 4408, 0x7808 }, { 4409, 0x9255 },
    { 4410, 0x289B8 }, { 4411, 0x43F2 }, { 4412, 0x289E7 }, { 4413, 0x43DF }, { 4414, 0x289E8 }, { 4415, 0x28B46 }, { 4416, 0x28BD4 }, { 4417, 0x59F8 },
    { 4418, 0x28C09 }, { 4419, 0x8F0B }, { 4420, 0x28FC5 }, { 4421, 0x290EC }, { 4422, 0x7B51 }, { 4423, 0x29110 }, { 4424, 0x2913C }, { 4425, 0x3DF7 },
    { 4426, 0x2915E }, { 4427, 0x24ACA }, { 4428, 0x8FD0 }, { 4429, 0x728F }, { 4430, 0x568B }, { 4431, 0x294E7 }, { 4432, 0x295E9 }, { 4433, 0x295B0 },
    { 4434, 0x295B8 }, { 4435, 0x29732 }, { 4436, 0x298D1 }, { 4437, 0x29949 }, { 4438, 0x2996A }, { 4439, 0x299C3 }, { 4440, 0x29A28 }, { 4441, 0x29B0E },
    { 4442, 0x29D5A }, { 4443, 0x29D9B }, { 4444, 0x7E9F }, { 4445, 0x29EF8 }, { 4446, 0x29F23 }, { 4447, 0x4CA4 }, { 4448, 0x9547 }, { 4449, 0x2A293 },
    { 4450, 0x71A2 }, { 4451, 0x2A2FF }, { 4452, 0x4D91 }, { 4453, 0x9012 }, { 4454, 0x2A5CB }, { 4455, 0x4D9C }, { 4456, 0x20C9C }, { 4457, 0x8FBE },
    { 4458, 0x55C1 }, { 4459, 0x8FBA }, { 4460, 0x224B0 }, { 4461, 0x8FB9 }, { 4462, 0x24A93 }, { 4463, 0x4509 }, { 4464, 0x7E7F }, { 4465, 0x6F56 },
    { 4466, 0x6AB1 }, { 4467, 0x4EEA }, { 4468, 0x34E4 }, { 4469, 0x28B2C }, { 4470, 0x2789D }, { 4471, 0x373A }, { 4472, 0x8E80 }, { 4473, 0x217F5 },
    { 4474, 0x28024 }, { 4475, 0x28B6C }, { 4476, 0x28B99 }, { 4477, 0x27A3E }, { 4478, 0x266AF }, { 4479, 0x3DEB }, { 4480, 0x27655 }, { 4481, 0x23CB7 },
    { 4482, 0x25635 }, { 4483, 0x25956 }, { 4484, 0x4E9A }, { 4485, 0x25E81 }, { 4486, 0x26258 }, { 4487, 0x56BF }, { 4488, 0x20E6D }, { 4489, 0x8E0E },
    { 4490, 0x5B6D }, { 4491, 0x23E88 }, { 4492, 0x24C9E }, { 4493, 0x63DE }, { 4494, 0x62D0 }, { 4495, 0x217F6 }, { 4496, 0x2187B }, { 4497, 0x6530 },
    { 4498, 0x562D }, { 4499, 0x25C4A }, { 4500, 0x541A }, { 4501, 0x25311 }, { 4502, 0x3DC6 }, { 4503, 0x29D98 }, { 4504, 0x4C7D }, { 4505, 0x5622 },
    { 4506, 0x561E }, { 4507, 0x7F49 }, { 4508, 0x25ED8 }, { 4509, 0x5975 }, { 4510, 0x23D40 }, { 4511, 0x8770 }, { 4512, 0x4E1C }, { 4513, 0x20FEA },
    { 4514, 0x20D49 }, { 4515, 0x236BA }, { 4516, 0x8117 }, { 4517, 0x9D5E }, { 4518, 0x8D18 }, { 4519, 0x763B }, { 4520, 0x9C45 }, { 4521, 0x764E },
    { 4522, 0x77B9 }, { 4523, 0x9345 }, { 4524, 0x5432 }, { 4525, 0x8148 }, { 4526, 0x82F7 }, { 4527, 0x5625 }, { 4528, 0x8132 }, { 4529, 0x8418 },
    { 4530, 0x80BD }, { 4531, 0x55EA }, { 4532, 0x7962 }, { 4533, 0x5643 }, { 4534, 0x5416 }, { 4535, 0x20E9D }, { 4536, 0x35CE }, { 4537, 0x5605 },
    { 4538, 0x55F1 }, { 4539, 0x66F1 }, { 4540, 0x282E2 }, { 4541, 0x362D }, { 4542, 0x7534 }, { 4543, 0x55F0 }, { 4544, 0x55BA }, { 4545, 0x5497 },
    { 4546, 0x5572 }, { 4547, 0x20C41 }, { 4548, 0x20C96 }, { 4549, 0x5ED0 }, { 4550, 0x25148 }, { 4551, 0x20E76 }, { 4552, 0x22C62 }, { 4553, 0x20EA2 },
    { 4554, 0x9EAB }, { 4555, 0x7D5A }, { 4556, 0x55DE }, { 4557, 0x21075 }, { 4558, 0x629D }, { 4559, 0x976D }, { 4560, 0x5494 }, { 4561, 0x8CCD },
    { 4562, 0x71F6 }, { 4563, 0x9176 }, { 4564, 0x63FC }, { 4565, 0x63B9 }, { 4566, 0x63FE }, { 4567, 0x5569 }, { 4568, 0x22B43 }, { 4569, 0x9C72 },
    { 4570, 0x22EB3 }, { 4571, 0x519A }, { 4572, 0x34DF }, { 4573, 0x20DA7 }, { 4574, 0x51A7 }, { 4575, 0x544D }, { 4576, 0x551E }, { 4577, 0x5513 },
    { 4578, 0x7666 }, { 4579, 0x8E2D }, { 4580, 0x2688A }, { 4581, 0x75B1 }, { 4582, 0x80B6 }, { 4583, 0x8804 }, { 4584, 0x8786 }, { 4585, 0x88C7 },
    { 4586, 0x81B6 }, { 4587, 0x841C }, { 4588, 0x210C1 }, { 4589, 0x44EC }, { 4590, 0x7304 }, { 4591, 0x24706 }, { 4592, 0x5B90 }, { 4593, 0x830B },
    { 4594, 0x26893 }, { 4595, 0x567B }, { 4596, 0x226F4 }, { 4597, 0x27D2F }, { 4598, 0x241A3 }, { 4599, 0x27D73 }, { 4600, 0x26ED0 }, { 4601, 0x272B6 },
    { 4602, 0x9170 }, { 4603, 0x211D9 }, { 4604, 0x9208 }, { 4605, 0x23CFC }, { 4606, 0x2A6A9 }, { 4607, 0x20EAC }, { 4608, 0x20EF9 }, { 4609, 0x7266 },
    { 4610, 0x21CA2 }, { 4611, 0x474E }, { 4612, 0x24FC2 }, { 4613, 0x27FF9 }, { 4614, 0x20FEB }, { 4615, 0x40FA }, { 4616, 0x9C5D }, { 4617, 0x651F },
    { 4618, 0x22DA0 }, { 4619, 0x48F3 }, { 4620, 0x247E0 }, { 4621, 0x29D7C }, { 4622, 0x20FEC }, { 4623, 0x20E0A }, { 4624, 0x6062 }, { 4625, 0x275A3 },
    { 4626, 0x20FED }, { 4628, 0x26048 }, { 4629, 0x21187 }, { 4630, 0x71A3 }, { 4631, 0x7E8E }, { 4632, 0x9D50 }, { 4633, 0x4E1A }, { 4634, 0x4E04 },
    { 4635, 0x3577 }, { 4636, 0x5B0D }, { 4637, 0x6CB2 }, { 4638, 0x5367 }, { 4639, 0x36AC }, { 4640, 0x39DC }, { 4641, 0x537D }, { 4642, 0x36A5 },
    { 4643, 0x24618 }, { 4644, 0x589A }, { 4645, 0x24B6E }, { 4646, 0x822D }, { 4647, 0x544B }, { 4648, 0x57AA }, { 4649, 0x25A95 }, { 4650, 0x20979 },
    { 4652, 0x3A52 }, { 4653, 0x22465 }, { 4654, 0x7374 }, { 4655, 0x29EAC }, { 4656, 0x4D09 }, { 4657, 0x9BED }, { 4658, 0x23CFE }, { 4659, 0x29F30 },
    { 4660, 0x4C5B }, { 4661, 0x24FA9 }, { 4662, 0x2959E }, { 4663, 0x29FDE }, { 4664, 0x845C }, { 4665, 0x23DB6 }, { 4666, 0x272B2 }, { 4667, 0x267B3 },
    { 4668, 0x23720 }, { 4669, 0x632E }, { 4670, 0x7D25 }, { 4671, 0x23EF7 }, { 4672, 0x23E2C }, { 4673, 0x3A2A }, { 4674, 0x9008 }, { 4675, 0x52CC },
    { 4676, 0x3E74 }, { 4677, 0x367A }, { 4678, 0x45E9 }, { 4679, 0x2048E }, { 4680, 0x7640 }, { 4681, 0x5AF0 }, { 4682, 0x20EB6 }, { 4683, 0x787A },
    { 4684, 0x27F2E }, { 4685, 0x58A7 }, { 4686, 0x40BF }, { 4687, 0x567C }, { 4688, 0x9B8B }, { 4689, 0x5D74 }, { 4690, 0x7654 }, { 4691, 0x2A434 },
    { 4692, 0x9E85 }, { 4693, 0x4CE1 }, { 4694, 0x75F9 }, { 4695, 0x37FB }, { 4696, 0x6119 }, { 4697, 0x230DA }, { 4698, 0x243F2 }, { 4700, 0x565D },
    { 4701, 0x212A9 }, { 4702, 0x57A7 }, { 4703, 0x24963 }, { 4704, 0x29E06 }, { 4705, 0x5234 }, { 4706, 0x270AE }, { 4707, 0x35AD }, { 4708, 0x6C4A },
    { 4709, 0x9D7C }, { 4710, 0x7C56 }, { 4711, 0x9B39 }, { 4712, 0x57DE }, { 4713, 0x2176C }, { 4714, 0x5C53 }, { 4715, 0x64D3 }, { 4716, 0x294D0 },
    { 4717, 0x26335 }, { 4718, 0x27164 }, { 4719, 0x86AD }, { 4720, 0x20D28 }, { 4721, 0x26D22 }, { 4722, 0x24AE2 }, { 4723, 0x20D71 }, { 4725, 0x51FE },
    { 4726, 0x21F0F }, { 4727, 0x5D8E }, { 4728, 0x9703 }, { 4729, 0x21DD1 }, { 4730, 0x9E81 }, { 4731, 0x904C }, { 4732, 0x7B1F }, { 4733, 0x9B02 },
    { 4734, 0x5CD1 }, { 4735, 0x7BA3 }, { 4736, 0x6268 }, { 4737, 0x6335 }, { 4738, 0x9AFF }, { 4739, 0x7BCF }, { 4740, 0x9B2A }, { 4741, 0x7C7E },
    { 4742, 0x9B2E }, { 4743, 0x7C42 }, { 4744, 0x7C86 }, { 4745, 0x9C15 }, { 4746, 0x7BFC }, { 4747, 0x9B09 }, { 4748, 0x9F17 }, { 4749, 0x9C1B },
    { 4750, 0x2493E }, { 4751, 0x9F5A }, { 4752, 0x5573 }, { 4753, 0x5BC3 }, { 4754, 0x4FFD }, { 4755, 0x9E98 }, { 4756, 0x4FF2 }, { 4757, 0x5260 },
    { 4758, 0x3E06 }, { 4759, 0x52D1 }, { 4760, 0x5767 }, { 4761, 0x5056 }, { 4762, 0x59B7 }, { 4763, 0x5E12 }, { 4764, 0x97C8 }, { 4765, 0x9DAB },
    { 4766, 0x8F5C }, { 4767, 0x5469 }, { 4768, 0x97B4 }, { 4769, 0x9940 }, { 4770, 0x97BA }, { 4771, 0x532C }, { 4772, 0x6130 }, { 4773, 0x692C },
    { 4774, 0x53DA }, { 4775, 0x9C0A }, { 4776, 0x9D02 }, { 4777, 0x4C3B }, { 4778, 0x9641 }, { 4779, 0x6980 }, { 4780, 0x50A6 }, { 4781, 0x7546 },
    { 4782, 0x2176D }, { 4783, 0x99DA }, { 4784, 0x5273 }, { 4786, 0x9159 }, { 4787, 0x9681 }, { 4788, 0x915C }, { 4790, 0x9151 }, { 4791, 0x28E97 },
    { 4792, 0x637F }, { 4793, 0x26D23 }, { 4794, 0x6ACA }, { 4795, 0x5611 }, { 4796, 0x918E }, { 4797, 0x757A }, { 4798, 0x6285 }, { 4799, 0x203FC },
    { 4800, 0x734F }, { 4801, 0x7C70 }, { 4802, 0x25C21 }, { 4803, 0x23CFD }, { 4805, 0x24919 }, { 4806, 0x76D6 }, { 4807, 0x9B9D }, { 4808, 0x4E2A },
    { 4809, 0x20CD4 }, { 4810, 0x83BE }, { 4811, 0x8842 }, { 4813, 0x5C4A }, { 4814, 0x69C0 }, { 4815, 0x50ED }, { 4816, 0x577A }, { 4817, 0x521F },
    { 4818, 0x5DF5 }, { 4819, 0x4ECE }, { 4820, 0x6C31 }, { 4821, 0x201F2 }, { 4822, 0x4F39 }, { 4823, 0x549C }, { 4824, 0x54DA }, { 4825, 0x529A },
    { 4826, 0x8D82 }, { 4827, 0x35FE }, { 4828, 0x5F0C }, { 4829, 0x35F3 }, { 4831, 0x6B52 }, { 4832, 0x917C }, { 4833, 0x9FA5 }, { 4834, 0x9B97 },
    { 4835, 0x982E }, { 4836, 0x98B4 }, { 4837, 0x9ABA }, { 4838, 0x9EA8 }, { 4839, 0x9E84 }, { 4840, 0x717A }, { 4841, 0x7B14 }, { 4843, 0x6BFA },
    { 4844, 0x8818 }, { 4845, 0x7F78 }, { 4847, 0x5620 }, { 4848, 0x2A64A }, { 4849, 0x8E77 }, { 4850, 0x9F53 }, { 4852, 0x8DD4 }, { 4853, 0x8E4F },
    { 4854, 0x9E1C }, { 4855, 0x8E01 }, { 4856, 0x6282 }, { 4857, 0x2837D }, { 4858, 0x8E28 }, { 4859, 0x8E75 }, { 4860, 0x7AD3 }, { 4861, 0x24A77 },
    { 4862, 0x7A3E }, { 4863, 0x78D8 }, { 4864, 0x6CEA }, { 4865, 0x8A67 }, { 4866, 0x7607 }, { 4867, 0x28A5A }, { 4868, 0x9F26 }, { 4869, 0x6CCE },
    { 4870, 0x87D6 }, { 4871, 0x75C3 }, { 4872, 0x2A2B2 }, { 4873, 0x7853 }, { 4874, 0x2F840 }, { 4875, 0x8D0C }, { 4876, 0x72E2 }, { 4877, 0x7371 },
    { 4878, 0x8B2D }, { 4879, 0x7302 }, { 4880, 0x74F1 }, { 4881, 0x8CEB }, { 4882, 0x24ABB }, { 4883, 0x862F }, { 4884, 0x5FBA }, { 4885, 0x88A0 },
    { 4886, 0x44B7 }, { 4888, 0x2183B }, { 4889, 0x26E05 }, { 4891, 0x8A7E }, { 4892, 0x2251B }, { 4894, 0x60FD }, { 4895, 0x7667 }, { 4896, 0x9AD7 },
    { 4897, 0x9D44 }, { 4898, 0x936E }, { 4899, 0x9B8F }, { 4900, 0x87F5 }, { 4902, 0x880F }, { 4903, 0x8CF7 }, { 4904, 0x732C }, { 4905, 0x9721 },
    { 4906, 0x9BB0 }, { 4907, 0x35D6 }, { 4908, 0x72B2 }, { 4909, 0x4C07 }, { 4910, 0x7C51 }, { 4911, 0x994A }, { 4912, 0x26159 }, { 4913, 0x6159 },
    { 4914, 0x4C04 }, { 4915, 0x9E96 }, { 4916, 0x617D }, { 4918, 0x575F }, { 4919, 0x616F }, { 4920, 0x62A6 }, { 4921, 0x6239 }, { 4922, 0x62CE },
    { 4923, 0x3A5C }, { 4924, 0x61E2 }, { 4925, 0x53AA }, { 4926, 0x233F5 }, { 4927, 0x6364 }, { 4928, 0x6802 }, { 4929, 0x35D2 }, { 4930, 0x5D57 },
    { 4931, 0x28BC2 }, { 4932, 0x8FDA }, { 4933, 0x28E39 }, { 4935, 0x50D9 }, { 4936, 0x21D46 }, { 4937, 0x7906 }, { 4938, 0x5332 }, { 4939, 0x9638 },
    { 4940, 0x20F3B }, { 4941, 0x4065 }, { 4943, 0x77FE }, { 4945, 0x7CC2 }, { 4946, 0x25F1A }, { 4947, 0x7CDA }, { 4948, 0x7A2D }, { 4949, 0x8066 },
    { 4950, 0x8063 }, { 4951, 0x7D4D }, { 4952, 0x7505 }, { 4953, 0x74F2 }, { 4954, 0x8994 }, { 4955, 0x821A }, { 4956, 0x670C }, { 4957, 0x8062 },
    { 4958, 0x27486 }, { 4959, 0x805B }, { 4960, 0x74F0 }, { 4961, 0x8103 }, { 4962, 0x7724 }, { 4963, 0x8989 }, { 4964, 0x267CC }, { 4965, 0x7553 },
    { 4966, 0x26ED1 }, { 4967, 0x87A9 }, { 4968, 0x87CE }, { 4969, 0x81C8 }, { 4970, 0x878C }, { 4971, 0x8A49 }, { 4972, 0x8CAD }, { 4973, 0x8B43 },
    { 4974, 0x772B }, { 4975, 0x74F8 }, { 4976, 0x84DA }, { 4977, 0x3635 }, { 4978, 0x69B2 }, { 4979, 0x8DA6 }, { 4981, 0x89A9 }, { 4982, 0x7468 },
    { 4983, 0x6DB9 }, { 4984, 0x87C1 }, { 4985, 0x24011 }, { 4986, 0x74E7 }, { 4987, 0x3DDB }, { 4988, 0x7176 }, { 4989, 0x60A4 }, { 4990, 0x619C },
    { 4991, 0x3CD1 }, { 4992, 0x7162 }, { 4993, 0x6077 }, { 4995, 0x7F71 }, { 4996, 0x28B2D }, { 4997, 0x7250 }, { 4998, 0x60E9 }, { 4999, 0x4B7E },
    { 5000, 0x5220 }, { 5001, 0x3C18 }, { 5002, 0x23CC7 }, { 5003, 0x25ED7 }, { 5004, 0x27656 }, { 5005, 0x25531 }, { 5006, 0x21944 }, { 5007, 0x212FE },
    { 5008, 0x29903 }, { 5009, 0x26DDC }, { 5010, 0x270AD }, { 5011, 0x5CC1 }, { 5012, 0x261AD }, { 5013, 0x28A0F }, { 5014, 0x23677 }, { 5015, 0x200EE },
    { 5016, 0x26846 }, { 5017, 0x24F0E }, { 5018, 0x4562 }, { 5019, 0x5B1F }, { 5020, 0x2634C }, { 5021, 0x9F50 }, { 5022, 0x9EA6 }, { 5023, 0x2626B },
    { 5024, 0x3000 }, { 5025, 0xFF0C }, { 5026, 0x3001 }, { 5027, 0x3002 }, { 5028, 0xFF0E }, { 5029, 0x2027 }, { 5030, 0xFF1B }, { 5031, 0xFF1A },
    { 5032, 0xFF1F }, { 5033, 0xFF01 }, { 5034, 0xFE30 }, { 5035, 0x2026 }, { 5036, 0x2025 }, { 5037, 0xFE50 }, { 5038, 0xFE51 }, { 5039, 0xFE52 },
    { 5040, 0x00B7 }, { 5041, 0xFE54 }, { 5042, 0xFE55 }, { 5043, 0xFE56 }, { 5044, 0xFE57 }, { 5045, 0xFF5C }, { 5046, 0x2013 }, { 5047, 0xFE31 },
    { 5048, 0x2014 }, { 5049, 0xFE33 }, { 5050, 0x2574 }, { 5051, 0xFE34 }, { 5052, 0xFE4F }, { 5053, 0xFF08 }, { 5054, 0xFF09 }, { 5055, 0xFE35 },
    { 5056, 0xFE36 }, { 5057, 0xFF5B }, { 5058, 0xFF5D }, { 5059, 0xFE37 }, { 5060, 0xFE38 }, { 5061, 0x3014 }, { 5062, 0x3015 }, { 5063, 0xFE39 },
    { 5064, 0xFE3A }, { 5065, 0x3010 }, { 5066, 0x3011 }, { 5067, 0xFE3B }, { 5068, 0xFE3C }, { 5069, 0x300A }, { 5070, 0x300B }, { 5071, 0xFE3D },
    { 5072, 0xFE3E }, { 5073, 0x3008 }, { 5074, 0x3009 }, { 5075, 0xFE3F }, { 5076, 0xFE40 }, { 5077, 0x300C }, { 5078, 0x300D }, { 5079, 0xFE41 },
    { 5080, 0xFE42 }, { 5081, 0x300E }, { 5082, 0x300F }, { 5083, 0xFE43 }, { 5084, 0xFE44 }, { 5085, 0xFE59 }, { 5086, 0xFE5A }, { 5087, 0xFE5B },
    { 5088, 0xFE5C }, { 5089, 0xFE5D }, { 5090, 0xFE5E }, { 5091, 0x2018 }, { 5092, 0x2019 }, { 5093, 0x201C }, { 5094, 0x201D }, { 5095, 0x301D },
    { 5096, 0x301E }, { 5097, 0x2035 }, { 5098, 0x2032 }, { 5099, 0xFF03 }, { 5100, 0xFF06 }, { 5101, 0xFF0A }, { 5102, 0x203B }, { 5103, 0x00A7 },
    { 5104, 0x3003 }, { 5105, 0x25CB }, { 5106, 0x25CF }, { 5107, 0x25B3 }, { 5108, 0x25B2 }, { 5109, 0x25CE }, { 5110, 0x2606 }, { 5111, 0x2605 },
    { 5112, 0x25C7 }, { 5113, 0x25C6 }, { 5114, 0x25A1 }, { 5115, 0x25A0 }, { 5116, 0x25BD }, { 5117, 0x25BC }, { 5118, 0x32A3 }, { 5119, 0x2105 },
    { 5120, 0x00AF }, { 5121, 0xFFE3 }, { 5122, 0xFF3F }, { 5123, 0x02CD }, { 5124, 0xFE49 }, { 5125, 0xFE4A }, { 5126, 0xFE4D }, { 5127, 0xFE4E },
    { 5128, 0xFE4B }, { 5129, 0xFE4C }, { 5130, 0xFE5F }, { 5131, 0xFE60 }, { 5132, 0xFE61 }, { 5133, 0xFF0B }, { 5134, 0xFF0D }, { 5135, 0x00D7 },
    { 5136, 0x00F7 }, { 5137, 0x00B1 }, { 5138, 0x221A }, { 5139, 0xFF1C }, { 5140, 0xFF1E }, { 5141, 0xFF1D }, { 5142, 0x2266 }, { 5143, 0x2267 },
    { 5144, 0x2260 }, { 5145, 0x221E }, { 5146, 0x2252 }, { 5147, 0x2261 }, { 5148, 0xFE62 }, { 5149, 0xFE63 }, { 5150, 0xFE64 }, { 5151, 0xFE65 },
    { 5152, 0xFE66 }, { 5153, 0xFF5E }, { 5154, 0x2229 }, { 5155, 0x222A }, { 5156, 0x22A5 }, { 5157, 0x2220 }, { 5158, 0x221F }, { 5159, 0x22BF },
    { 5160, 0x33D2 }, { 5161, 0x33D1 }, { 5162, 0x222B }, { 5163, 0x222E }, { 5164, 0x2235 }, { 5165, 0x2234 }, { 5166, 0x2640 }, { 5167, 0x2642 },
    { 5168, 0x2295 }, { 5169, 0x2299 }, { 5170, 0x2191 }, { 5171, 0x2193 }, { 5172, 0x2190 }, { 5173, 0x2192 }, { 5174, 0x2196 }, { 5175, 0x2197 },
    { 5176, 0x2199 }, { 5177, 0x2198 }, { 5178, 0x2225 }, { 5179, 0x2223 }, { 5180, 0xFF0F }, { 5181, 0xFF3C }, { 5182, 0x2215 }, { 5183, 0xFE68 },
    { 5184, 0xFF04 }, { 5185, 0xFFE5 }, { 5186, 0x3012 }, { 5187, 0xFFE0 }, { 5188, 0xFFE1 }, { 5189, 0xFF05 }, { 5190, 0xFF20 }, { 5191, 0x2103 },
    { 5192, 0x2109 }, { 5193, 0xFE69 }, { 5194, 0xFE6A }, { 5195, 0xFE6B }, { 5196, 0x33D5 }, { 5197, 0x339C }, { 5198, 0x339D }, { 5199, 0x339E },
    { 5200, 0x33CE }, { 5201, 0x33A1 }, { 5202, 0x338E }, { 5203, 0x338F }, { 5204, 0x33C4 }, { 5205, 0x00B0 }, { 5206, 0x5159 }, { 5207, 0x515B },
    { 5208, 0x515E }, { 5209, 0x515D }, { 5210, 0x5161 }, { 5211, 0x5163 }, { 5212, 0x55E7 }, { 5213, 0x74E9 }, { 5214, 0x7CCE }, { 5215, 0x2581 },
    { 5216, 0x2582 }, { 5217, 0x2583 }, { 5218, 0x2584 }, { 5219, 0x2585 }, { 5220, 0x2586 }, { 5221, 0x2587 }, { 5222, 0x2588 }, { 5223, 0x258F },
    { 5224, 0x258E }, { 5225, 0x258D }, { 5226, 0x258C }, { 5227, 0x258B }, { 5228, 0x258A }, { 5229, 0x2589 }, { 5230, 0x253C }, { 5231, 0x2534 },
    { 5232, 0x252C }, { 5233, 0x2524 }, { 5234, 0x251C }, { 5235, 0x2594 }, { 5236, 0x2500 }, { 5237, 0x2502 }, { 5238, 0x2595 }, { 5239, 0x250C },
    { 5240, 0x2510 }, { 5241, 0x2514 }, { 5242, 0x2518 }, { 5243, 0x256D }, { 5244, 0x256E }, { 5245, 0x2570 }, { 5246, 0x256F }, { 5247, 0x2550 },
    { 5248, 0x255E }, { 5249, 0x256A }, { 5250, 0x2561 }, { 5251, 0x25E2 }, { 5252, 0x25E3 }, { 5253, 0x25E5 }, { 5254, 0x25E4 }, { 5255, 0x2571 },
    { 5256, 0x2572 }, { 5257, 0x2573 }, { 5258, 0xFF10 }, { 5259, 0xFF11 }, { 5260, 0xFF12 }, { 5261, 0xFF13 }, { 5262, 0xFF14 }, { 5263, 0xFF15 },
    { 5264, 0xFF16 }, { 5265, 0xFF17 }, { 5266, 0xFF18 }, { 5267, 0xFF19 }, { 5268, 0x2160 }, { 5269, 0x2161 }, { 5270, 0x2162 }, { 5271, 0x2163 },
    { 5272, 0x2164 }, { 5273, 0x2165 }, { 5274, 0x2166 }, { 5275, 0x2167 }, { 5276, 0x2168 }, { 5277, 0x2169 }, { 5278, 0x3021 }, { 5279, 0x3022 },
    { 5280, 0x3023 }, { 5281, 0x3024 }, { 5282, 0x3025 }, { 5283, 0x3026 }, { 5284, 0x3027 }, { 5285, 0x3028 }, { 5286, 0x3029 }, { 5287, 0x5341 },
    { 5288, 0x5344 }, { 5289, 0x5345 }, { 5290, 0xFF21 }, { 5291, 0xFF22 }, { 5292, 0xFF23 }, { 5293, 0xFF24 }, { 5294, 0xFF25 }, { 5295, 0xFF26 },
    { 5296, 0xFF27 }, { 5297, 0xFF28 }, { 5298, 0xFF29 }, { 5299, 0xFF2A }, { 5300, 0xFF2B }, { 5301, 0xFF2C }, { 5302, 0xFF2D }, { 5303, 0xFF2E },
    { 5304, 0xFF2F }, { 5305, 0xFF30 }, { 5306, 0xFF31 }, { 5307, 0xFF32 }, { 5308, 0xFF33 }, { 5309, 0xFF34 }, { 5310, 0xFF35 }, { 5311, 0xFF36 },
    { 5312, 0xFF37 }, { 5313, 0xFF38 }, { 5314, 0xFF39 }, { 5315, 0xFF3A }, { 5316, 0xFF41 }, { 5317, 0xFF42 }, { 5318, 0xFF43 }, { 5319, 0xFF44 },
    { 5320, 0xFF45 }, { 5321, 0xFF46 }, { 5322, 0xFF47 }, { 5323, 0xFF48 }, { 5324, 0xFF49 }, { 5325, 0xFF4A }, { 5326, 0xFF4B }, { 5327, 0xFF4C },
    { 5328, 0xFF4D }, { 5329, 0xFF4E }, { 5330, 0xFF4F }, { 5331, 0xFF50 }, { 5332, 0xFF51 }, { 5333, 0xFF52 }, { 5334, 0xFF53 }, { 5335, 0xFF54 },
    { 5336, 0xFF55 }, { 5337, 0xFF56 }, { 5338, 0xFF57 }, { 5339, 0xFF58 }, { 5340, 0xFF59 }, { 5341, 0xFF5A }, { 5342, 0x0391 }, { 5343, 0x0392 },
    { 5344, 0x0393 }, { 5345, 0x0394 }, { 5346, 0x0395 }, { 5347, 0x0396 }, { 5348, 0x0397 }, { 5349, 0x0398 }, { 5350, 0x0399 }, { 5351, 0x039A },
    { 5352, 0x039B }, { 5353, 0x039C }, { 5354, 0x039D }, { 5355, 0x039E }, { 5356, 0x039F }, { 5357, 0x03A0 }, { 5358, 0x03A1 }, { 5359, 0x03A3 },
    { 5360, 0x03A4 }, { 5361, 0x03A5 }, { 5362, 0x03A6 }, { 5363, 0x03A7 }, { 5364, 0x03A8 }, { 5365, 0x03A9 }, { 5366, 0x03B1 }, { 5367, 0x03B2 },
    { 5368, 0x03B3 }, { 5369, 0x03B4 }, { 5370, 0x03B5 }, { 5371, 0x03B6 }, { 5372, 0x03B7 }, { 5373, 0x03B8 }, { 5374, 0x03B9 }, { 5375, 0x03BA },
    { 5376, 0x03BB }, { 5377, 0x03BC }, { 5378, 0x03BD }, { 5379, 0x03BE }, { 5380, 0x03BF }, { 5381, 0x03C0 }, { 5382, 0x03C1 }, { 5383, 0x03C3 },
    { 5384, 0x03C4 }, { 5385, 0x03C5 }, { 5386, 0x03C6 }, { 5387, 0x03C7 }, { 5388, 0x03C8 }, { 5389, 0x03C9 }, { 5390, 0x3105 }, { 5391, 0x3106 },
    { 5392, 0x3107 }, { 5393, 0x3108 }, { 5394, 0x3109 }, { 5395, 0x310A }, { 5396, 0x310B }, { 5397, 0x310C }, { 5398, 0x310D }, { 5399, 0x310E },
    { 5400, 0x310F }, { 5401, 0x3110 }, { 5402, 0x3111 }, { 5403, 0x3112 }, { 5404, 0x3113 }, { 5405, 0x3114 }, { 5406, 0x3115 }, { 5407, 0x3116 },
    { 5408, 0x3117 }, { 5409, 0x3118 }, { 5410, 0x3119 }, { 5411, 0x311A }, { 5412, 0x311B }, { 5413, 0x311C }, { 5414, 0x311D }, { 5415, 0x311E },
    { 5416, 0x311F }, { 5417, 0x3120 }, { 5418, 0x3121 }, { 5419, 0x3122 }, { 5420, 0x3123 }, { 5421, 0x3124 }, { 5422, 0x3125 }, { 5423, 0x3126 },
    { 5424, 0x3127 }, { 5425, 0x3128 }, { 5426, 0x3129 }, { 5427, 0x02D9 }, { 5428, 0x02C9 }, { 5429, 0x02CA }, { 5430, 0x02C7 }, { 5431, 0x02CB },
    { 5432, 0x2400 }, { 5433, 0x2401 }, { 5434, 0x2402 }, { 5435, 0x2403 }, { 5436, 0x2404 }, { 5437, 0x2405 }, { 5438, 0x2406 }, { 5439, 0x2407 },
    { 5440, 0x2408 }, { 5441, 0x2409 }, { 5442, 0x240A }, { 5443, 0x240B }, { 5444, 0x240C }, { 5445, 0x240D }, { 5446, 0x240E }, { 5447, 0x240F },
    { 5448, 0x2410 }, { 5449, 0x2411 }, { 5450, 0x2412 }, { 5451, 0x2413 }, { 5452, 0x2414 }, { 5453, 0x2415 }, { 5454, 0x2416 }, { 5455, 0x2417 },
    { 5456, 0x2418 }, { 5457, 0x2419 }, { 5458, 0x241A }, { 5459, 0x241B }, { 5460, 0x241C }, { 5461, 0x241D }, { 5462, 0x241E }, { 5463, 0x241F },
    { 5464, 0x2421 }, { 5465, 0x20AC }, { 5495, 0x4E00 }, { 5496, 0x4E59 }, { 5497, 0x4E01 }, { 5498, 0x4E03 }, { 5499, 0x4E43 }, { 5500, 0x4E5D },
    { 5501, 0x4E86 }, { 5502, 0x4E8C }, { 5503, 0x4EBA }, { 5504, 0x513F }, { 5505, 0x5165 }, { 5506, 0x516B }, { 5507, 0x51E0 }, { 5508, 0x5200 },
    { 5509, 0x5201 }, { 5510, 0x529B }, { 5511, 0x5315 }, { 5512, 0x5341 }, { 5513, 0x535C }, { 5514, 0x53C8 }, { 5515, 0x4E09 }, { 5516, 0x4E0B },
    { 5517, 0x4E08 }, { 5518, 0x4E0A }, { 5519, 0x4E2B }, { 5520, 0x4E38 }, { 5521, 0x51E1 }, { 5522, 0x4E45 }, { 5523, 0x4E48 }, { 5524, 0x4E5F },
    { 5525, 0x4E5E }, { 5526, 0x4E8E }, { 5527, 0x4EA1 }, { 5528, 0x5140 }, { 5529, 0x5203 }, { 5530, 0x52FA }, { 5531, 0x5343 }, { 5532, 0x53C9 },
    { 5533, 0x53E3 }, { 5534, 0x571F }, { 5535, 0x58EB }, { 5536, 0x5915 }, { 5537, 0x5927 }, { 5538, 0x5973 }, { 5539, 0x5B50 }, { 5540, 0x5B51 },
    { 5541, 0x5B53 }, { 5542, 0x5BF8 }, { 5543, 0x5C0F }, { 5544, 0x5C22 }, { 5545, 0x5C38 }, { 5546, 0x5C71 }, { 5547, 0x5DDD }, { 5548, 0x5DE5 },
    { 5549, 0x5DF1 }, { 5550, 0x5DF2 }, { 5551, 0x5DF3 }, { 5552, 0x5DFE }, { 5553, 0x5E72 }, { 5554, 0x5EFE }, { 5555, 0x5F0B }, { 5556, 0x5F13 },
    { 5557, 0x624D }, { 5558, 0x4E11 }, { 5559, 0x4E10 }, { 5560, 0x4E0D }, { 5561, 0x4E2D }, { 5562, 0x4E30 }, { 5563, 0x4E39 }, { 5564, 0x4E4B },
    { 5565, 0x5C39 }, { 5566, 0x4E88 }, { 5567, 0x4E91 }, { 5568, 0x4E95 }, { 5569, 0x4E92 }, { 5570, 0x4E94 }, { 5571, 0x4EA2 }, { 5572, 0x4EC1 },
    { 5573, 0x4EC0 }, { 5574, 0x4EC3 }, { 5575, 0x4EC6 }, { 5576, 0x4EC7 }, { 5577, 0x4ECD }, { 5578, 0x4ECA }, { 5579, 0x4ECB }, { 5580, 0x4EC4 },
    { 5581, 0x5143 }, { 5582, 0x5141 }, { 5583, 0x5167 }, { 5584, 0x516D }, { 5585, 0x516E }, { 5586, 0x516C }, { 5587, 0x5197 }, { 5588, 0x51F6 },
    { 5589, 0x5206 }, { 5590, 0x5207 }, { 5591, 0x5208 }, { 5592, 0x52FB }, { 5593, 0x52FE }, { 5594, 0x52FF }, { 5595, 0x5316 }, { 5596, 0x5339 },
    { 5597, 0x5348 }, { 5598, 0x5347 }, { 5599, 0x5345 }, { 5600, 0x535E }, { 5601, 0x5384 }, { 5602, 0x53CB }, { 5603, 0x53CA }, { 5604, 0x53CD },
    { 5605, 0x58EC }, { 5606, 0x5929 }, { 5607, 0x592B }, { 5608, 0x592A }, { 5609, 0x592D }, { 5610, 0x5B54 }, { 5611, 0x5C11 }, { 5612, 0x5C24 },
    { 5613, 0x5C3A }, { 5614, 0x5C6F }, { 5615, 0x5DF4 }, { 5616, 0x5E7B }, { 5617, 0x5EFF }, { 5618, 0x5F14 }, { 5619, 0x5F15 }, { 5620, 0x5FC3 },
    { 5621, 0x6208 }, { 5622, 0x6236 }, { 5623, 0x624B }, { 5624, 0x624E }, { 5625, 0x652F }, { 5626, 0x6587 }, { 5627, 0x6597 }, { 5628, 0x65A4 },
    { 5629, 0x65B9 }, { 5630, 0x65E5 }, { 5631, 0x66F0 }, { 5632, 0x6708 }, { 5633, 0x6728 }, { 5634, 0x6B20 }, { 5635, 0x6B62 }, { 5636, 0x6B79 },
    { 5637, 0x6BCB }, { 5638, 0x6BD4 }, { 5639, 0x6BDB }, { 5640, 0x6C0F }, { 5641, 0x6C34 }, { 5642, 0x706B }, { 5643, 0x722A }, { 5644, 0x7236 },
    { 5645, 0x723B }, { 5646, 0x7247 }, { 5647, 0x7259 }, { 5648, 0x725B }, { 5649, 0x72AC }, { 5650, 0x738B }, { 5651, 0x4E19 }, { 5652, 0x4E16 },
    { 5653, 0x4E15 }, { 5654, 0x4E14 }, { 5655, 0x4E18 }, { 5656, 0x4E3B }, { 5657, 0x4E4D }, { 5658, 0x4E4F }, { 5659, 0x4E4E }, { 5660, 0x4EE5 },
    { 5661, 0x4ED8 }, { 5662, 0x4ED4 }, { 5663, 0x4ED5 }, { 5664, 0x4ED6 }, { 5665, 0x4ED7 }, { 5666, 0x4EE3 }, { 5667, 0x4EE4 }, { 5668, 0x4ED9 },
    { 5669, 0x4EDE }, { 5670, 0x5145 }, { 5671, 0x5144 }, { 5672, 0x5189 }, { 5673, 0x518A }, { 5674, 0x51AC }, { 5675, 0x51F9 }, { 5676, 0x51FA },
    { 5677, 0x51F8 }, { 5678, 0x520A }, { 5679, 0x52A0 }, { 5680, 0x529F }, { 5681, 0x5305 }, { 5682, 0x5306 }, { 5683, 0x5317 }, { 5684, 0x531D },
    { 5685, 0x4EDF }, { 5686, 0x534A }, { 5687, 0x5349 }, { 5688, 0x5361 }, { 5689, 0x5360 }, { 5690, 0x536F }, { 5691, 0x536E }, { 5692, 0x53BB },
    { 5693, 0x53EF }, { 5694, 0x53E4 }, { 5695, 0x53F3 }, { 5696, 0x53EC }, { 5697, 0x53EE }, { 5698, 0x53E9 }, { 5699, 0x53E8 }, { 5700, 0x53FC },
    { 5701, 0x53F8 }, { 5702, 0x53F5 }, { 5703, 0x53EB }, { 5704, 0x53E6 }, { 5705, 0x53EA }, { 5706, 0x53F2 }, { 5707, 0x53F1 }, { 5708, 0x53F0 },
    { 5709, 0x53E5 }, { 5710, 0x53ED }, { 5711, 0x53FB }, { 5712, 0x56DB }, { 5713, 0x56DA }, { 5714, 0x5916 }, { 5715, 0x592E }, { 5716, 0x5931 },
    { 5717, 0x5974 }, { 5718, 0x5976 }, { 5719, 0x5B55 }, { 5720, 0x5B83 }, { 5721, 0x5C3C }, { 5722, 0x5DE8 }, { 5723, 0x5DE7 }, { 5724, 0x5DE6 },
    { 5725, 0x5E02 }, { 5726, 0x5E03 }, { 5727, 0x5E73 }, { 5728, 0x5E7C }, { 5729, 0x5F01 }, { 5730, 0x5F18 }, { 5731, 0x5F17 }, { 5732, 0x5FC5 },
    { 5733, 0x620A }, { 5734, 0x6253 }, { 5735, 0x6254 }, { 5736, 0x6252 }, { 5737, 0x6251 }, { 5738, 0x65A5 }, { 5739, 0x65E6 }, { 5740, 0x672E },
    { 5741, 0x672C }, { 5742, 0x672A }, { 5743, 0x672B }, { 5744, 0x672D }, { 5745, 0x6B63 }, { 5746, 0x6BCD }, { 5747, 0x6C11 }, { 5748, 0x6C10 },
    { 5749, 0x6C38 }, { 5750, 0x6C41 }, { 5751, 0x6C40 }, { 5752, 0x6C3E }, { 5753, 0x72AF }, { 5754, 0x7384 }, { 5755, 0x7389 }, { 5756, 0x74DC },
    { 5757, 0x74E6 }, { 5758, 0x7518 }, { 5759, 0x751F }, { 5760, 0x7528 }, { 5761, 0x7529 }, { 5762, 0x7530 }, { 5763, 0x7531 }, { 5764, 0x7532 },
    { 5765, 0x7533 }, { 5766, 0x758B }, { 5767, 0x767D }, { 5768, 0x76AE }, { 5769, 0x76BF }, { 5770, 0x76EE }, { 5771, 0x77DB }, { 5772, 0x77E2 },
    { 5773, 0x77F3 }, { 5774, 0x793A }, { 5775, 0x79BE }, { 5776, 0x7A74 }, { 5777, 0x7ACB }, { 5778, 0x4E1E }, { 5779, 0x4E1F }, { 5780, 0x4E52 },
    { 5781, 0x4E53 }, { 5782, 0x4E69 }, { 5783, 0x4E99 }, { 5784, 0x4EA4 }, { 5785, 0x4EA6 }, { 5786, 0x4EA5 }, { 5787, 0x4EFF }, { 5788, 0x4F09 },
    { 5789, 0x4F19 }, { 5790, 0x4F0A }, { 5791, 0x4F15 }, { 5792, 0x4F0D }, { 5793, 0x4F10 }, { 5794, 0x4F11 }, { 5795, 0x4F0F }, { 5796, 0x4EF2 },
    { 5797, 0x4EF6 }, { 5798, 0x4EFB }, { 5799, 0x4EF0 }, { 5800, 0x4EF3 }, { 5801, 0x4EFD }, { 5802, 0x4F01 }, { 5803, 0x4F0B }, { 5804, 0x5149 },
    { 5805, 0x5147 }, { 5806, 0x5146 }, { 5807, 0x5148 }, { 5808, 0x5168 }, { 5809, 0x5171 }, { 5810, 0x518D }, { 5811, 0x51B0 }, { 5812, 0x5217 },
    { 5813, 0x5211 }, { 5814, 0x5212 }, { 5815, 0x520E }, { 5816, 0x5216 }, { 5817, 0x52A3 }, { 5818, 0x5308 }, { 5819, 0x5321 }, { 5820, 0x5320 },
    { 5821, 0x5370 }, { 5822, 0x5371 }, { 5823, 0x5409 }, { 5824, 0x540F }, { 5825, 0x540C }, { 5826, 0x540A }, { 5827, 0x5410 }, { 5828, 0x5401 },
    { 5829, 0x540B }, { 5830, 0x5404 }, { 5831, 0x5411 }, { 5832, 0x540D }, { 5833, 0x5408 }, { 5834, 0x5403 }, { 5835, 0x540E }, { 5836, 0x5406 },
    { 5837, 0x5412 }, { 5838, 0x56E0 }, { 5839, 0x56DE }, { 5840, 0x56DD }, { 5841, 0x5733 }, { 5842, 0x5730 }, { 5843, 0x5728 }, { 5844, 0x572D },
    { 5845, 0x572C }, { 5846, 0x572F }, { 5847, 0x5729 }, { 5848, 0x5919 }, { 5849, 0x591A }, { 5850, 0x5937 }, { 5851, 0x5938 }, { 5852, 0x5984 },
    { 5853, 0x5978 }, { 5854, 0x5983 }, { 5855, 0x597D }, { 5856, 0x5979 }, { 5857, 0x5982 }, { 5858, 0x5981 }, { 5859, 0x5B57 }, { 5860, 0x5B58 },
    { 5861, 0x5B87 }, { 5862, 0x5B88 }, { 5863, 0x5B85 }, { 5864, 0x5B89 }, { 5865, 0x5BFA }, { 5866, 0x5C16 }, { 5867, 0x5C79 }, { 5868, 0x5DDE },
    { 5869, 0x5E06 }, { 5870, 0x5E76 }, { 5871, 0x5E74 }, { 5872, 0x5F0F }, { 5873, 0x5F1B }, { 5874, 0x5FD9 }, { 5875, 0x5FD6 }, { 5876, 0x620E },
    { 5877, 0x620C }, { 5878, 0x620D }, { 5879, 0x6210 }, { 5880, 0x6263 }, { 5881, 0x625B }, { 5882, 0x6258 }, { 5883, 0x6536 }, { 5884, 0x65E9 },
    { 5885, 0x65E8 }, { 5886, 0x65EC }, { 5887, 0x65ED }, { 5888, 0x66F2 }, { 5889, 0x66F3 }, { 5890, 0x6709 }, { 5891, 0x673D }, { 5892, 0x6734 },
    { 5893, 0x6731 }, { 5894, 0x6735 }, { 5895, 0x6B21 }, { 5896, 0x6B64 }, { 5897, 0x6B7B }, { 5898, 0x6C16 }, { 5899, 0x6C5D }, { 5900, 0x6C57 },
    { 5901, 0x6C59 }, { 5902, 0x6C5F }, { 5903, 0x6C60 }, { 5904, 0x6C50 }, { 5905, 0x6C55 }, { 5906, 0x6C61 }, { 5907, 0x6C5B }, { 5908, 0x6C4D },
    { 5909, 0x6C4E }, { 5910, 0x7070 }, { 5911, 0x725F }, { 5912, 0x725D }, { 5913, 0x767E }, { 5914, 0x7AF9 }, { 5915, 0x7C73 }, { 5916, 0x7CF8 },
    { 5917, 0x7F36 }, { 5918, 0x7F8A }, { 5919, 0x7FBD }, { 5920, 0x8001 }, { 5921, 0x8003 }, { 5922, 0x800C }, { 5923, 0x8012 }, { 5924, 0x8033 },
    { 5925, 0x807F }, { 5926, 0x8089 }, { 5927, 0x808B }, { 5928, 0x808C }, { 5929, 0x81E3 }, { 5930, 0x81EA }, { 5931, 0x81F3 }, { 5932, 0x81FC },
    { 5933, 0x820C }, { 5934, 0x821B }, { 5935, 0x821F }, { 5936, 0x826E }, { 5937, 0x8272 }, { 5938, 0x827E }, { 5939, 0x866B }, { 5940, 0x8840 },
    { 5941, 0x884C }, { 5942, 0x8863 }, { 5943, 0x897F }, { 5944, 0x9621 }, { 5945, 0x4E32 }, { 5946, 0x4EA8 }, { 5947, 0x4F4D }, { 5948, 0x4F4F },
    { 5949, 0x4F47 }, { 5950, 0x4F57 }, { 5951, 0x4F5E }, { 5952, 0x4F34 }, { 5953, 0x4F5B }, { 5954, 0x4F55 }, { 5955, 0x4F30 }, { 5956, 0x4F50 },
    { 5957, 0x4F51 }, { 5958, 0x4F3D }, { 5959, 0x4F3A }, { 5960, 0x4F38 }, { 5961, 0x4F43 }, { 5962, 0x4F54 }, { 5963, 0x4F3C }, { 5964, 0x4F46 },
    { 5965, 0x4F63 }, { 5966, 0x4F5C }, { 5967, 0x4F60 }, { 5968, 0x4F2F }, { 5969, 0x4F4E }, { 5970, 0x4F36 }, { 5971, 0x4F59 }, { 5972, 0x4F5D },
    { 5973, 0x4F48 }, { 5974, 0x4F5A }, { 5975, 0x514C }, { 5976, 0x514B }, { 5977, 0x514D }, { 5978, 0x5175 }, { 5979, 0x51B6 }, { 5980, 0x51B7 },
    { 5981, 0x5225 }, { 5982, 0x5224 }, { 5983, 0x5229 }, { 5984, 0x522A }, { 5985, 0x5228 }, { 5986, 0x52AB }, { 5987, 0x52A9 }, { 5988, 0x52AA },
    { 5989, 0x52AC }, { 5990, 0x5323 }, { 5991, 0x5373 }, { 5992, 0x5375 }, { 5993, 0x541D }, { 5994, 0x542D }, { 5995, 0x541E }, { 5996, 0x543E },
    { 5997, 0x5426 }, { 5998, 0x544E }, { 5999, 0x5427 }, { 6000, 0x5446 }, { 6001, 0x5443 }, { 6002, 0x5433 }, { 6003, 0x5448 }, { 6004, 0x5442 },
    { 6005, 0x541B }, { 6006, 0x5429 }, { 6007, 0x544A }, { 6008, 0x5439 }, { 6009, 0x543B }, { 6010, 0x5438 }, { 6011, 0x542E }, { 6012, 0x5435 },
    { 6013, 0x5436 }, { 6014, 0x5420 }, { 6015, 0x543C }, { 6016, 0x5440 }, { 6017, 0x5431 }, { 6018, 0x542B }, { 6019, 0x541F }, { 6020, 0x542C },
    { 6021, 0x56EA }, { 6022, 0x56F0 }, { 6023, 0x56E4 }, { 6024, 0x56EB }, { 6025, 0x574A }, { 6026, 0x5751 }, { 6027, 0x5740 }, { 6028, 0x574D },
    { 6029, 0x5747 }, { 6030, 0x574E }, { 6031, 0x573E }, { 6032, 0x5750 }, { 6033, 0x574F }, { 6034, 0x573B }, { 6035, 0x58EF }, { 6036, 0x593E },
    { 6037, 0x599D }, { 6038, 0x5992 }, { 6039, 0x59A8 }, { 6040, 0x599E }, { 6041, 0x59A3 }, { 6042, 0x5999 }, { 6043, 0x5996 }, { 6044, 0x598D },
    { 6045, 0x59A4 }, { 6046, 0x5993 }, { 6047, 0x598A }, { 6048, 0x59A5 }, { 6049, 0x5B5D }, { 6050, 0x5B5C }, { 6051, 0x5B5A }, { 6052, 0x5B5B },
    { 6053, 0x5B8C }, { 6054, 0x5B8B }, { 6055, 0x5B8F }, { 6056, 0x5C2C }, { 6057, 0x5C40 }, { 6058, 0x5C41 }, { 6059, 0x5C3F }, { 6060, 0x5C3E },
    { 6061, 0x5C90 }, { 6062, 0x5C91 }, { 6063, 0x5C94 }, { 6064, 0x5C8C }, { 6065, 0x5DEB }, { 6066, 0x5E0C }, { 6067, 0x5E8F }, { 6068, 0x5E87 },
    { 6069, 0x5E8A }, { 6070, 0x5EF7 }, { 6071, 0x5F04 }, { 6072, 0x5F1F }, { 6073, 0x5F64 }, { 6074, 0x5F62 }, { 6075, 0x5F77 }, { 6076, 0x5F79 },
    { 6077, 0x5FD8 }, { 6078, 0x5FCC }, { 6079, 0x5FD7 }, { 6080, 0x5FCD }, { 6081, 0x5FF1 }, { 6082, 0x5FEB }, { 6083, 0x5FF8 }, { 6084, 0x5FEA },
    { 6085, 0x6212 }, { 6086, 0x6211 }, { 6087, 0x6284 }, { 6088, 0x6297 }, { 6089, 0x6296 }, { 6090, 0x6280 }, { 6091, 0x6276 }, { 6092, 0x6289 },
    { 6093, 0x626D }, { 6094, 0x628A }, { 6095, 0x627C }, { 6096, 0x627E }, { 6097, 0x6279 }, { 6098, 0x6273 }, { 6099, 0x6292 }, { 6100, 0x626F },
    { 6101, 0x6298 }, { 6102, 0x626E }, { 6103, 0x6295 }, { 6104, 0x6293 }, { 6105, 0x6291 }, { 6106, 0x6286 }, { 6107, 0x6539 }, { 6108, 0x653B },
    { 6109, 0x6538 }, { 6110, 0x65F1 }, { 6111, 0x66F4 }, { 6112, 0x675F }, { 6113, 0x674E }, { 6114, 0x674F }, { 6115, 0x6750 }, { 6116, 0x6751 },
    { 6117, 0x675C }, { 6118, 0x6756 }, { 6119, 0x675E }, { 6120, 0x6749 }, { 6121, 0x6746 }, { 6122, 0x6760 }, { 6123, 0x6753 }, { 6124, 0x6757 },
    { 6125, 0x6B65 }, { 6126, 0x6BCF }, { 6127, 0x6C42 }, { 6128, 0x6C5E }, { 6129, 0x6C99 }, { 6130, 0x6C81 }, { 6131, 0x6C88 }, { 6132, 0x6C89 },
    { 6133, 0x6C85 }, { 6134, 0x6C9B }, { 6135, 0x6C6A }, { 6136, 0x6C7A }, { 6137, 0x6C90 }, { 6138, 0x6C70 }, { 6139, 0x6C8C }, { 6140, 0x6C68 },
    { 6141, 0x6C96 }, { 6142, 0x6C92 }, { 6143, 0x6C7D }, { 6144, 0x6C83 }, { 6145, 0x6C72 }, { 6146, 0x6C7E }, { 6147, 0x6C74 }, { 6148, 0x6C86 },
    { 6149, 0x6C76 }, { 6150, 0x6C8D }, { 6151, 0x6C94 }, { 6152, 0x6C98 }, { 6153, 0x6C82 }, { 6154, 0x7076 }, { 6155, 0x707C }, { 6156, 0x707D },
    { 6157, 0x7078 }, { 6158, 0x7262 }, { 6159, 0x7261 }, { 6160, 0x7260 }, { 6161, 0x72C4 }, { 6162, 0x72C2 }, { 6163, 0x7396 }, { 6164, 0x752C },
    { 6165, 0x752B }, { 6166, 0x7537 }, { 6167, 0x7538 }, { 6168, 0x7682 }, { 6169, 0x76EF }, { 6170, 0x77E3 }, { 6171, 0x79C1 }, { 6172, 0x79C0 },
    { 6173, 0x79BF }, { 6174, 0x7A76 }, { 6175, 0x7CFB }, { 6176, 0x7F55 }, { 6177, 0x8096 }, { 6178, 0x8093 }, { 6179, 0x809D }, { 6180, 0x8098 },
    { 6181, 0x809B }, { 6182, 0x809A }, { 6183, 0x80B2 }, { 6184, 0x826F }, { 6185, 0x8292 }, { 6186, 0x828B }, { 6187, 0x828D }, { 6188, 0x898B },
    { 6189, 0x89D2 }, { 6190, 0x8A00 }, { 6191, 0x8C37 }, { 6192, 0x8C46 }, { 6193, 0x8C55 }, { 6194, 0x8C9D }, { 6195, 0x8D64 }, { 6196, 0x8D70 },
    { 6197, 0x8DB3 }, { 6198, 0x8EAB }, { 6199, 0x8ECA }, { 6200, 0x8F9B }, { 6201, 0x8FB0 }, { 6202, 0x8FC2 }, { 6203, 0x8FC6 }, { 6204, 0x8FC5 },
    { 6205, 0x8FC4 }, { 6206, 0x5DE1 }, { 6207, 0x9091 }, { 6208, 0x90A2 }, { 6209, 0x90AA }, { 6210, 0x90A6 }, { 6211, 0x90A3 }, { 6212, 0x9149 },
    { 6213, 0x91C6 }, { 6214, 0x91CC }, { 6215, 0x9632 }, { 6216, 0x962E }, { 6217, 0x9631 }, { 6218, 0x962A }, { 6219, 0x962C }, { 6220, 0x4E26 },
    { 6221, 0x4E56 }, { 6222, 0x4E73 }, { 6223, 0x4E8B }, { 6224, 0x4E9B }, { 6225, 0x4E9E }, { 6226, 0x4EAB }, { 6227, 0x4EAC }, { 6228, 0x4F6F },
    { 6229, 0x4F9D }, { 6230, 0x4F8D }, { 6231, 0x4F73 }, { 6232, 0x4F7F }, { 6233, 0x4F6C }, { 6234, 0x4F9B }, { 6235, 0x4F8B }, { 6236, 0x4F86 },
    { 6237, 0x4F83 }, { 6238, 0x4F70 }, { 6239, 0x4F75 }, { 6240, 0x4F88 }, { 6241, 0x4F69 }, { 6242, 0x4F7B }, { 6243, 0x4F96 }, { 6244, 0x4F7E },
    { 6245, 0x4F8F }, { 6246, 0x4F91 }, { 6247, 0x4F7A }, { 6248, 0x5154 }, { 6249, 0x5152 }, { 6250, 0x5155 }, { 6251, 0x5169 }, { 6252, 0x5177 },
    { 6253, 0x5176 }, { 6254, 0x5178 }, { 6255, 0x51BD }, { 6256, 0x51FD }, { 6257, 0x523B }, { 6258, 0x5238 }, { 6259, 0x5237 }, { 6260, 0x523A },
    { 6261, 0x5230 }, { 6262, 0x522E }, { 6263, 0x5236 }, { 6264, 0x5241 }, { 6265, 0x52BE }, { 6266, 0x52BB }, { 6267, 0x5352 }, { 6268, 0x5354 },
    { 6269, 0x5353 }, { 6270, 0x5351 }, { 6271, 0x5366 }, { 6272, 0x5377 }, { 6273, 0x5378 }, { 6274, 0x5379 }, { 6275, 0x53D6 }, { 6276, 0x53D4 },
    { 6277, 0x53D7 }, { 6278, 0x5473 }, { 6279, 0x5475 }, { 6280, 0x5496 }, { 6281, 0x5478 }, { 6282, 0x5495 }, { 6283, 0x5480 }, { 6284, 0x547B },
    { 6285, 0x5477 }, { 6286, 0x5484 }, { 6287, 0x5492 }, { 6288, 0x5486 }, { 6289, 0x547C }, { 6290, 0x5490 }, { 6291, 0x5471 }, { 6292, 0x5476 },
    { 6293, 0x548C }, { 6294, 0x549A }, { 6295, 0x5462 }, { 6296, 0x5468 }, { 6297, 0x548B }, { 6298, 0x547D }, { 6299, 0x548E }, { 6300, 0x56FA },
    { 6301, 0x5783 }, { 6302, 0x5777 }, { 6303, 0x576A }, { 6304, 0x5769 }, { 6305, 0x5761 }, { 6306, 0x5766 }, { 6307, 0x5764 }, { 6308, 0x577C },
    { 6309, 0x591C }, { 6310, 0x5949 }, { 6311, 0x5947 }, { 6312, 0x5948 }, { 6313, 0x5944 }, { 6314, 0x5954 }, { 6315, 0x59BE }, { 6316, 0x59BB },
    { 6317, 0x59D4 }, { 6318, 0x59B9 }, { 6319, 0x59AE }, { 6320, 0x59D1 }, { 6321, 0x59C6 }, { 6322, 0x59D0 }, { 6323, 0x59CD }, { 6324, 0x59CB },
    { 6325, 0x59D3 }, { 6326, 0x59CA }, { 6327, 0x59AF }, { 6328, 0x59B3 }, { 6329, 0x59D2 }, { 6330, 0x59C5 }, { 6331, 0x5B5F }, { 6332, 0x5B64 },
    { 6333, 0x5B63 }, { 6334, 0x5B97 }, { 6335, 0x5B9A }, { 6336, 0x5B98 }, { 6337, 0x5B9C }, { 6338, 0x5B99 }, { 6339, 0x5B9B }, { 6340, 0x5C1A },
    { 6341, 0x5C48 }, { 6342, 0x5C45 }, { 6343, 0x5C46 }, { 6344, 0x5CB7 }, { 6345, 0x5CA1 }, { 6346, 0x5CB8 }, { 6347, 0x5CA9 }, { 6348, 0x5CAB },
    { 6349, 0x5CB1 }, { 6350, 0x5CB3 }, { 6351, 0x5E18 }, { 6352, 0x5E1A }, { 6353, 0x5E16 }, { 6354, 0x5E15 }, { 6355, 0x5E1B }, { 6356, 0x5E11 },
    { 6357, 0x5E78 }, { 6358, 0x5E9A }, { 6359, 0x5E97 }, { 6360, 0x5E9C }, { 6361, 0x5E95 }, { 6362, 0x5E96 }, { 6363, 0x5EF6 }, { 6364, 0x5F26 },
    { 6365, 0x5F27 }, { 6366, 0x5F29 }, { 6367, 0x5F80 }, { 6368, 0x5F81 }, { 6369, 0x5F7F }, { 6370, 0x5F7C }, { 6371, 0x5FDD }, { 6372, 0x5FE0 },
    { 6373, 0x5FFD }, { 6374, 0x5FF5 }, { 6375, 0x5FFF }, { 6376, 0x600F }, { 6377, 0x6014 }, { 6378, 0x602F }, { 6379, 0x6035 }, { 6380, 0x6016 },
    { 6381, 0x602A }, { 6382, 0x6015 }, { 6383, 0x6021 }, { 6384, 0x6027 }, { 6385, 0x6029 }, { 6386, 0x602B }, { 6387, 0x601B }, { 6388, 0x6216 },
    { 6389, 0x6215 }, { 6390, 0x623F }, { 6391, 0x623E }, { 6392, 0x6240 }, { 6393, 0x627F }, { 6394, 0x62C9 }, { 6395, 0x62CC }, { 6396, 0x62C4 },
    { 6397, 0x62BF }, { 6398, 0x62C2 }, { 6399, 0x62B9 }, { 6400, 0x62D2 }, { 6401, 0x62DB }, { 6402, 0x62AB }, { 6403, 0x62D3 }, { 6404, 0x62D4 },
    { 6405, 0x62CB }, { 6406, 0x62C8 }, { 6407, 0x62A8 }, { 6408, 0x62BD }, { 6409, 0x62BC }, { 6410, 0x62D0 }, { 6411, 0x62D9 }, { 6412, 0x62C7 },
    { 6413, 0x62CD }, { 6414, 0x62B5 }, { 6415, 0x62DA }, { 6416, 0x62B1 }, { 6417, 0x62D8 }, { 6418, 0x62D6 }, { 6419, 0x62D7 }, { 6420, 0x62C6 },
    { 6421, 0x62AC }, { 6422, 0x62CE }, { 6423, 0x653E }, { 6424, 0x65A7 }, { 6425, 0x65BC }, { 6426, 0x65FA }, { 6427, 0x6614 }, { 6428, 0x6613 },
    { 6429, 0x660C }, { 6430, 0x6606 }, { 6431, 0x6602 }, { 6432, 0x660E }, { 6433, 0x6600 }, { 6434, 0x660F }, { 6435, 0x6615 }, { 6436, 0x660A },
    { 6437, 0x6607 }, { 6438, 0x670D }, { 6439, 0x670B }, { 6440, 0x676D }, { 6441, 0x678B }, { 6442, 0x6795 }, { 6443, 0x6771 }, { 6444, 0x679C },
    { 6445, 0x6773 }, { 6446, 0x6777 }, { 6447, 0x6787 }, { 6448, 0x679D }, { 6449, 0x6797 }, { 6450, 0x676F }, { 6451, 0x6770 }, { 6452, 0x677F },
    { 6453, 0x6789 }, { 6454, 0x677E }, { 6455, 0x6790 }, { 6456, 0x6775 }, { 6457, 0x679A }, { 6458, 0x6793 }, { 6459, 0x677C }, { 6460, 0x676A },
    { 6461, 0x6772 }, { 6462, 0x6B23 }, { 6463, 0x6B66 }, { 6464, 0x6B67 }, { 6465, 0x6B7F }, { 6466, 0x6C13 }, { 6467, 0x6C1B }, { 6468, 0x6CE3 },
    { 6469, 0x6CE8 }, { 6470, 0x6CF3 }, { 6471, 0x6CB1 }, { 6472, 0x6CCC }, { 6473, 0x6CE5 }, { 6474, 0x6CB3 }, { 6475, 0x6CBD }, { 6476, 0x6CBE },
    { 6477, 0x6CBC }, { 6478, 0x6CE2 }, { 6479, 0x6CAB }, { 6480, 0x6CD5 }, { 6481, 0x6CD3 }, { 6482, 0x6CB8 }, { 6483, 0x6CC4 }, { 6484, 0x6CB9 },
    { 6485, 0x6CC1 }, { 6486, 0x6CAE }, { 6487, 0x6CD7 }, { 6488, 0x6CC5 }, { 6489, 0x6CF1 }, { 6490, 0x6CBF }, { 6491, 0x6CBB }, { 6492, 0x6CE1 },
    { 6493, 0x6CDB }, { 6494, 0x6CCA }, { 6495, 0x6CAC }, { 6496, 0x6CEF }, { 6497, 0x6CDC }, { 6498, 0x6CD6 }, { 6499, 0x6CE0 }, { 6500, 0x7095 },
    { 6501, 0x708E }, { 6502, 0x7092 }, { 6503, 0x708A }, { 6504, 0x7099 }, { 6505, 0x722C }, { 6506, 0x722D }, { 6507, 0x7238 }, { 6508, 0x7248 },
    { 6509, 0x7267 }, { 6510, 0x7269 }, { 6511, 0x72C0 }, { 6512, 0x72CE }, { 6513, 0x72D9 }, { 6514, 0x72D7 }, { 6515, 0x72D0 }, { 6516, 0x73A9 },
    { 6517, 0x73A8 }, { 6518, 0x739F }, { 6519, 0x73AB }, { 6520, 0x73A5 }, { 6521, 0x753D }, { 6522, 0x759D }, { 6523, 0x7599 }, { 6524, 0x759A },
    { 6525, 0x7684 }, { 6526, 0x76C2 }, { 6527, 0x76F2 }, { 6528, 0x76F4 }, { 6529, 0x77E5 }, { 6530, 0x77FD }, { 6531, 0x793E }, { 6532, 0x7940 },
    { 6533, 0x7941 }, { 6534, 0x79C9 }, { 6535, 0x79C8 }, { 6536, 0x7A7A }, { 6537, 0x7A79 }, { 6538, 0x7AFA }, { 6539, 0x7CFE }, { 6540, 0x7F54 },
    { 6541, 0x7F8C }, { 6542, 0x7F8B }, { 6543, 0x8005 }, { 6544, 0x80BA }, { 6545, 0x80A5 }, { 6546, 0x80A2 }, { 6547, 0x80B1 }, { 6548, 0x80A1 },
    { 6549, 0x80AB }, { 6550, 0x80A9 }, { 6551, 0x80B4 }, { 6552, 0x80AA }, { 6553, 0x80AF }, { 6554, 0x81E5 }, { 6555, 0x81FE }, { 6556, 0x820D },
    { 6557, 0x82B3 }, { 6558, 0x829D }, { 6559, 0x8299 }, { 6560, 0x82AD }, { 6561, 0x82BD }, { 6562, 0x829F }, { 6563, 0x82B9 }, { 6564, 0x82B1 },
    { 6565, 0x82AC }, { 6566, 0x82A5 }, { 6567, 0x82AF }, { 6568, 0x82B8 }, { 6569, 0x82A3 }, { 6570, 0x82B0 }, { 6571, 0x82BE }, { 6572, 0x82B7 },
    { 6573, 0x864E }, { 6574, 0x8671 }, { 6575, 0x521D }, { 6576, 0x8868 }, { 6577, 0x8ECB }, { 6578, 0x8FCE }, { 6579, 0x8FD4 }, { 6580, 0x8FD1 },
    { 6581, 0x90B5 }, { 6582, 0x90B8 }, { 6583, 0x90B1 }, { 6584, 0x90B6 }, { 6585, 0x91C7 }, { 6586, 0x91D1 }, { 6587, 0x9577 }, { 6588, 0x9580 },
    { 6589, 0x961C }, { 6590, 0x9640 }, { 6591, 0x963F }, { 6592, 0x963B }, { 6593, 0x9644 }, { 6594, 0x9642 }, { 6595, 0x96B9 }, { 6596, 0x96E8 },
    { 6597, 0x9752 }, { 6598, 0x975E }, { 6599, 0x4E9F }, { 6600, 0x4EAD }, { 6601, 0x4EAE }, { 6602, 0x4FE1 }, { 6603, 0x4FB5 }, { 6604, 0x4FAF },
    { 6605, 0x4FBF }, { 6606, 0x4FE0 }, { 6607, 0x4FD1 }, { 6608, 0x4FCF }, { 6609, 0x4FDD }, { 6610, 0x4FC3 }, { 6611, 0x4FB6 }, { 6612, 0x4FD8 },
    { 6613, 0x4FDF }, { 6614, 0x4FCA }, { 6615, 0x4FD7 }, { 6616, 0x4FAE }, { 6617, 0x4FD0 }, { 6618, 0x4FC4 }, { 6619, 0x4FC2 }, { 6620, 0x4FDA },
    { 6621, 0x4FCE }, { 6622, 0x4FDE }, { 6623, 0x4FB7 }, { 6624, 0x5157 }, { 6625, 0x5192 }, { 6626, 0x5191 }, { 6627, 0x51A0 }, { 6628, 0x524E },
    { 6629, 0x5243 }, { 6630, 0x524A }, { 6631, 0x524D }, { 6632, 0x524C }, { 6633, 0x524B }, { 6634, 0x5247 }, { 6635, 0x52C7 }, { 6636, 0x52C9 },
    { 6637, 0x52C3 }, { 6638, 0x52C1 }, { 6639, 0x530D }, { 6640, 0x5357 }, { 6641, 0x537B }, { 6642, 0x539A }, { 6643, 0x53DB }, { 6644, 0x54AC },
    { 6645, 0x54C0 }, { 6646, 0x54A8 }, { 6647, 0x54CE }, { 6648, 0x54C9 }, { 6649, 0x54B8 }, { 6650, 0x54A6 }, { 6651, 0x54B3 }, { 6652, 0x54C7 },
    { 6653, 0x54C2 }, { 6654, 0x54BD }, { 6655, 0x54AA }, { 6656, 0x54C1 }, { 6657, 0x54C4 }, { 6658, 0x54C8 }, { 6659, 0x54AF }, { 6660, 0x54AB },
    { 6661, 0x54B1 }, { 6662, 0x54BB }, { 6663, 0x54A9 }, { 6664, 0x54A7 }, { 6665, 0x54BF }, { 6666, 0x56FF }, { 6667, 0x5782 }, { 6668, 0x578B },
    { 6669, 0x57A0 }, { 6670, 0x57A3 }, { 6671, 0x57A2 }, { 6672, 0x57CE }, { 6673, 0x57AE }, { 6674, 0x5793 }, { 6675, 0x5955 }, { 6676, 0x5951 },
    { 6677, 0x594F }, { 6678, 0x594E }, { 6679, 0x5950 }, { 6680, 0x59DC }, { 6681, 0x59D8 }, { 6682, 0x59FF }, { 6683, 0x59E3 }, { 6684, 0x59E8 },
    { 6685, 0x5A03 }, { 6686, 0x59E5 }, { 6687, 0x59EA }, { 6688, 0x59DA }, { 6689, 0x59E6 }, { 6690, 0x5A01 }, { 6691, 0x59FB }, { 6692, 0x5B69 },
    { 6693, 0x5BA3 }, { 6694, 0x5BA6 }, { 6695, 0x5BA4 }, { 6696, 0x5BA2 }, { 6697, 0x5BA5 }, { 6698, 0x5C01 }, { 6699, 0x5C4E }, { 6700, 0x5C4F },
    { 6701, 0x5C4D }, { 6702, 0x5C4B }, { 6703, 0x5CD9 }, { 6704, 0x5CD2 }, { 6705, 0x5DF7 }, { 6706, 0x5E1D }, { 6707, 0x5E25 }, { 6708, 0x5E1F },
    { 6709, 0x5E7D }, { 6710, 0x5EA0 }, { 6711, 0x5EA6 }, { 6712, 0x5EFA }, { 6713, 0x5F08 }, { 6714, 0x5F2D }, { 6715, 0x5F65 }, { 6716, 0x5F88 },
    { 6717, 0x5F85 }, { 6718, 0x5F8A }, { 6719, 0x5F8B }, { 6720, 0x5F87 }, { 6721, 0x5F8C }, { 6722, 0x5F89 }, { 6723, 0x6012 }, { 6724, 0x601D },
    { 6725, 0x6020 }, { 6726, 0x6025 }, { 6727, 0x600E }, { 6728, 0x6028 }, { 6729, 0x604D }, { 6730, 0x6070 }, { 6731, 0x6068 }, { 6732, 0x6062 },
    { 6733, 0x6046 }, { 6734, 0x6043 }, { 6735, 0x606C }, { 6736, 0x606B }, { 6737, 0x606A }, { 6738, 0x6064 }, { 6739, 0x6241 }, { 6740, 0x62DC },
    { 6741, 0x6316 }, { 6742, 0x6309 }, { 6743, 0x62FC }, { 6744, 0x62ED }, { 6745, 0x6301 }, { 6746, 0x62EE }, { 6747, 0x62FD }, { 6748, 0x6307 },
    { 6749, 0x62F1 }, { 6750, 0x62F7 }, { 6751, 0x62EF }, { 6752, 0x62EC }, { 6753, 0x62FE }, { 6754, 0x62F4 }, { 6755, 0x6311 }, { 6756, 0x6302 },
    { 6757, 0x653F }, { 6758, 0x6545 }, { 6759, 0x65AB }, { 6760, 0x65BD }, { 6761, 0x65E2 }, { 6762, 0x6625 }, { 6763, 0x662D }, { 6764, 0x6620 },
    { 6765, 0x6627 }, { 6766, 0x662F }, { 6767, 0x661F }, { 6768, 0x6628 }, { 6769, 0x6631 }, { 6770, 0x6624 }, { 6771, 0x66F7 }, { 6772, 0x67FF },
    { 6773, 0x67D3 }, { 6774, 0x67F1 }, { 6775, 0x67D4 }, { 6776, 0x67D0 }, { 6777, 0x67EC }, { 6778, 0x67B6 }, { 6779, 0x67AF }, { 6780, 0x67F5 },
    { 6781, 0x67E9 }, { 6782, 0x67EF }, { 6783, 0x67C4 }, { 6784, 0x67D1 }, { 6785, 0x67B4 }, { 6786, 0x67DA }, { 6787, 0x67E5 }, { 6788, 0x67B8 },
    { 6789, 0x67CF }, { 6790, 0x67DE }, { 6791, 0x67F3 }, { 6792, 0x67B0 }, { 6793, 0x67D9 }, { 6794, 0x67E2 }, { 6795, 0x67DD }, { 6796, 0x67D2 },
    { 6797, 0x6B6A }, { 6798, 0x6B83 }, { 6799, 0x6B86 }, { 6800, 0x6BB5 }, { 6801, 0x6BD2 }, { 6802, 0x6BD7 }, { 6803, 0x6C1F }, { 6804, 0x6CC9 },
    { 6805, 0x6D0B }, { 6806, 0x6D32 }, { 6807, 0x6D2A }, { 6808, 0x6D41 }, { 6809, 0x6D25 }, { 6810, 0x6D0C }, { 6811, 0x6D31 }, { 6812, 0x6D1E },
    { 6813, 0x6D17 }, { 6814, 0x6D3B }, { 6815, 0x6D3D }, { 6816, 0x6D3E }, { 6817, 0x6D36 }, { 6818, 0x6D1B }, { 6819, 0x6CF5 }, { 6820, 0x6D39 },
    { 6821, 0x6D27 }, { 6822, 0x6D38 }, { 6823, 0x6D29 }, { 6824, 0x6D2E }, { 6825, 0x6D35 }, { 6826, 0x6D0E }, { 6827, 0x6D2B }, { 6828, 0x70AB },
    { 6829, 0x70BA }, { 6830, 0x70B3 }, { 6831, 0x70AC }, { 6832, 0x70AF }, { 6833, 0x70AD }, { 6834, 0x70B8 }, { 6835, 0x70AE }, { 6836, 0x70A4 },
    { 6837, 0x7230 }, { 6838, 0x7272 }, { 6839, 0x726F }, { 6840, 0x7274 }, { 6841, 0x72E9 }, { 6842, 0x72E0 }, { 6843, 0x72E1 }, { 6844, 0x73B7 },
    { 6845, 0x73CA }, { 6846, 0x73BB }, { 6847, 0x73B2 }, { 6848, 0x73CD }, { 6849, 0x73C0 }, { 6850, 0x73B3 }, { 6851, 0x751A }, { 6852, 0x752D },
    { 6853, 0x754F }, { 6854, 0x754C }, { 6855, 0x754E }, { 6856, 0x754B }, { 6857, 0x75AB }, { 6858, 0x75A4 }, { 6859, 0x75A5 }, { 6860, 0x75A2 },
    { 6861, 0x75A3 }, { 6862, 0x7678 }, { 6863, 0x7686 }, { 6864, 0x7687 }, { 6865, 0x7688 }, { 6866, 0x76C8 }, { 6867, 0x76C6 }, { 6868, 0x76C3 },
    { 6869, 0x76C5 }, { 6870, 0x7701 }, { 6871, 0x76F9 }, { 6872, 0x76F8 }, { 6873, 0x7709 }, { 6874, 0x770B }, { 6875, 0x76FE }, { 6876, 0x76FC },
    { 6877, 0x7707 }, { 6878, 0x77DC }, { 6879, 0x7802 }, { 6880, 0x7814 }, { 6881, 0x780C }, { 6882, 0x780D }, { 6883, 0x7946 }, { 6884, 0x7949 },
    { 6885, 0x7948 }, { 6886, 0x7947 }, { 6887, 0x79B9 }, { 6888, 0x79BA }, { 6889, 0x79D1 }, { 6890, 0x79D2 }, { 6891, 0x79CB }, { 6892, 0x7A7F },
    { 6893, 0x7A81 }, { 6894, 0x7AFF }, { 6895, 0x7AFD }, { 6896, 0x7C7D }, { 6897, 0x7D02 }, { 6898, 0x7D05 }, { 6899, 0x7D00 }, { 6900, 0x7D09 },
    { 6901, 0x7D07 }, { 6902, 0x7D04 }, { 6903, 0x7D06 }, { 6904, 0x7F38 }, { 6905, 0x7F8E }, { 6906, 0x7FBF }, { 6907, 0x8004 }, { 6908, 0x8010 },
    { 6909, 0x800D }, { 6910, 0x8011 }, { 6911, 0x8036 }, { 6912, 0x80D6 }, { 6913, 0x80E5 }, { 6914, 0x80DA }, { 6915, 0x80C3 }, { 6916, 0x80C4 },
    { 6917, 0x80CC }, { 6918, 0x80E1 }, { 6919, 0x80DB }, { 6920, 0x80CE }, { 6921, 0x80DE }, { 6922, 0x80E4 }, { 6923, 0x80DD }, { 6924, 0x81F4 },
    { 6925, 0x8222 }, { 6926, 0x82E7 }, { 6927, 0x8303 }, { 6928, 0x8305 }, { 6929, 0x82E3 }, { 6930, 0x82DB }, { 6931, 0x82E6 }, { 6932, 0x8304 },
    { 6933, 0x82E5 }, { 6934, 0x8302 }, { 6935, 0x8309 }, { 6936, 0x82D2 }, { 6937, 0x82D7 }, { 6938, 0x82F1 }, { 6939, 0x8301 }, { 6940, 0x82DC },
    { 6941, 0x82D4 }, { 6942, 0x82D1 }, { 6943, 0x82DE }, { 6944, 0x82D3 }, { 6945, 0x82DF }, { 6946, 0x82EF }, { 6947, 0x8306 }, { 6948, 0x8650 },
    { 6949, 0x8679 }, { 6950, 0x867B }, { 6951, 0x867A }, { 6952, 0x884D }, { 6953, 0x886B }, { 6954, 0x8981 }, { 6955, 0x89D4 }, { 6956, 0x8A08 },
    { 6957, 0x8A02 }, { 6958, 0x8A03 }, { 6959, 0x8C9E }, { 6960, 0x8CA0 }, { 6961, 0x8D74 }, { 6962, 0x8D73 }, { 6963, 0x8DB4 }, { 6964, 0x8ECD },
    { 6965, 0x8ECC }, { 6966, 0x8FF0 }, { 6967, 0x8FE6 }, { 6968, 0x8FE2 }, { 6969, 0x8FEA }, { 6970, 0x8FE5 }, { 6971, 0x8FED }, { 6972, 0x8FEB },
    { 6973, 0x8FE4 }, { 6974, 0x8FE8 }, { 6975, 0x90CA }, { 6976, 0x90CE }, { 6977, 0x90C1 }, { 6978, 0x90C3 }, { 6979, 0x914B }, { 6980, 0x914A },
    { 6981, 0x91CD }, { 6982, 0x9582 }, { 6983, 0x9650 }, { 6984, 0x964B }, { 6985, 0x964C }, { 6986, 0x964D }, { 6987, 0x9762 }, { 6988, 0x9769 },
    { 6989, 0x97CB }, { 6990, 0x97ED }, { 6991, 0x97F3 }, { 6992, 0x9801 }, { 6993, 0x98A8 }, { 6994, 0x98DB }, { 6995, 0x98DF }, { 6996, 0x9996 },
    { 6997, 0x9999 }, { 6998, 0x4E58 }, { 6999, 0x4EB3 }, { 7000, 0x500C }, { 7001, 0x500D }, { 7002, 0x5023 }, { 7003, 0x4FEF }, { 7004, 0x5026 },
    { 7005, 0x5025 }, { 7006, 0x4FF8 }, { 7007, 0x5029 }, { 7008, 0x5016 }, { 7009, 0x5006 }, { 7010, 0x503C }, { 7011, 0x501F }, { 7012, 0x501A },
    { 7013, 0x5012 }, { 7014, 0x5011 }, { 7015, 0x4FFA }, { 7016, 0x5000 }, { 7017, 0x5014 }, { 7018, 0x5028 }, { 7019, 0x4FF1 }, { 7020, 0x5021 },
    { 7021, 0x500B }, { 7022, 0x5019 }, { 7023, 0x5018 }, { 7024, 0x4FF3 }, { 7025, 0x4FEE }, { 7026, 0x502D }, { 7027, 0x502A }, { 7028, 0x4FFE },
    { 7029, 0x502B }, { 7030, 0x5009 }, { 7031, 0x517C }, { 7032, 0x51A4 }, { 7033, 0x51A5 }, { 7034, 0x51A2 }, { 7035, 0x51CD }, { 7036, 0x51CC },
    { 7037, 0x51C6 }, { 7038, 0x51CB }, { 7039, 0x5256 }, { 7040, 0x525C }, { 7041, 0x5254 }, { 7042, 0x525B }, { 7043, 0x525D }, { 7044, 0x532A },
    { 7045, 0x537F }, { 7046, 0x539F }, { 7047, 0x539D }, { 7048, 0x53DF }, { 7049, 0x54E8 }, { 7050, 0x5510 }, { 7051, 0x5501 }, { 7052, 0x5537 },
    { 7053, 0x54FC }, { 7054, 0x54E5 }, { 7055, 0x54F2 }, { 7056, 0x5506 }, { 7057, 0x54FA }, { 7058, 0x5514 }, { 7059, 0x54E9 }, { 7060, 0x54ED },
    { 7061, 0x54E1 }, { 7062, 0x5509 }, { 7063, 0x54EE }, { 7064, 0x54EA }, { 7065, 0x54E6 }, { 7066, 0x5527 }, { 7067, 0x5507 }, { 7068, 0x54FD },
    { 7069, 0x550F }, { 7070, 0x5703 }, { 7071, 0x5704 }, { 7072, 0x57C2 }, { 7073, 0x57D4 }, { 7074, 0x57CB }, { 7075, 0x57C3 }, { 7076, 0x5809 },
    { 7077, 0x590F }, { 7078, 0x5957 }, { 7079, 0x5958 }, { 7080, 0x595A }, { 7081, 0x5A11 }, { 7082, 0x5A18 }, { 7083, 0x5A1C }, { 7084, 0x5A1F },
    { 7085, 0x5A1B }, { 7086, 0x5A13 }, { 7087, 0x59EC }, { 7088, 0x5A20 }, { 7089, 0x5A23 }, { 7090, 0x5A29 }, { 7091, 0x5A25 }, { 7092, 0x5A0C },
    { 7093, 0x5A09 }, { 7094, 0x5B6B }, { 7095, 0x5C58 }, { 7096, 0x5BB0 }, { 7097, 0x5BB3 }, { 7098, 0x5BB6 }, { 7099, 0x5BB4 }, { 7100, 0x5BAE },
    { 7101, 0x5BB5 }, { 7102, 0x5BB9 }, { 7103, 0x5BB8 }, { 7104, 0x5C04 }, { 7105, 0x5C51 }, { 7106, 0x5C55 }, { 7107, 0x5C50 }, { 7108, 0x5CED },
    { 7109, 0x5CFD }, { 7110, 0x5CFB }, { 7111, 0x5CEA }, { 7112, 0x5CE8 }, { 7113, 0x5CF0 }, { 7114, 0x5CF6 }, { 7115, 0x5D01 }, { 7116, 0x5CF4 },
    { 7117, 0x5DEE }, { 7118, 0x5E2D }, { 7119, 0x5E2B }, { 7120, 0x5EAB }, { 7121, 0x5EAD }, { 7122, 0x5EA7 }, { 7123, 0x5F31 }, { 7124, 0x5F92 },
    { 7125, 0x5F91 }, { 7126, 0x5F90 }, { 7127, 0x6059 }, { 7128, 0x6063 }, { 7129, 0x6065 }, { 7130, 0x6050 }, { 7131, 0x6055 }, { 7132, 0x606D },
    { 7133, 0x6069 }, { 7134, 0x606F }, { 7135, 0x6084 }, { 7136, 0x609F }, { 7137, 0x609A }, { 7138, 0x608D }, { 7139, 0x6094 }, { 7140, 0x608C },
    { 7141, 0x6085 }, { 7142, 0x6096 }, { 7143, 0x6247 }, { 7144, 0x62F3 }, { 7145, 0x6308 }, { 7146, 0x62FF }, { 7147, 0x634E }, { 7148, 0x633E },
    { 7149, 0x632F }, { 7150, 0x6355 }, { 7151, 0x6342 }, { 7152, 0x6346 }, { 7153, 0x634F }, { 7154, 0x6349 }, { 7155, 0x633A }, { 7156, 0x6350 },
    { 7157, 0x633D }, { 7158, 0x632A }, { 7159, 0x632B }, { 7160, 0x6328 }, { 7161, 0x634D }, { 7162, 0x634C }, { 7163, 0x6548 }, { 7164, 0x6549 },
    { 7165, 0x6599 }, { 7166, 0x65C1 }, { 7167, 0x65C5 }, { 7168, 0x6642 }, { 7169, 0x6649 }, { 7170, 0x664F }, { 7171, 0x6643 }, { 7172, 0x6652 },
    { 7173, 0x664C }, { 7174, 0x6645 }, { 7175, 0x6641 }, { 7176, 0x66F8 }, { 7177, 0x6714 }, { 7178, 0x6715 }, { 7179, 0x6717 }, { 7180, 0x6821 },
    { 7181, 0x6838 }, { 7182, 0x6848 }, { 7183, 0x6846 }, { 7184, 0x6853 }, { 7185, 0x6839 }, { 7186, 0x6842 }, { 7187, 0x6854 }, { 7188, 0x6829 },
    { 7189, 0x68B3 }, { 7190, 0x6817 }, { 7191, 0x684C }, { 7192, 0x6851 }, { 7193, 0x683D }, { 7194, 0x67F4 }, { 7195, 0x6850 }, { 7196, 0x6840 },
    { 7197, 0x683C }, { 7198, 0x6843 }, { 7199, 0x682A }, { 7200, 0x6845 }, { 7201, 0x6813 }, { 7202, 0x6818 }, { 7203, 0x6841 }, { 7204, 0x6B8A },
    { 7205, 0x6B89 }, { 7206, 0x6BB7 }, { 7207, 0x6C23 }, { 7208, 0x6C27 }, { 7209, 0x6C28 }, { 7210, 0x6C26 }, { 7211, 0x6C24 }, { 7212, 0x6CF0 },
    { 7213, 0x6D6A }, { 7214, 0x6D95 }, { 7215, 0x6D88 }, { 7216, 0x6D87 }, { 7217, 0x6D66 }, { 7218, 0x6D78 }, { 7219, 0x6D77 }, { 7220, 0x6D59 },
    { 7221, 0x6D93 }, { 7222, 0x6D6C }, { 7223, 0x6D89 }, { 7224, 0x6D6E }, { 7225, 0x6D5A }, { 7226, 0x6D74 }, { 7227, 0x6D69 }, { 7228, 0x6D8C },
    { 7229, 0x6D8A }, { 7230, 0x6D79 }, { 7231, 0x6D85 }, { 7232, 0x6D65 }, { 7233, 0x6D94 }, { 7234, 0x70CA }, { 7235, 0x70D8 }, { 7236, 0x70E4 },
    { 7237, 0x70D9 }, { 7238, 0x70C8 }, { 7239, 0x70CF }, { 7240, 0x7239 }, { 7241, 0x7279 }, { 7242, 0x72FC }, { 7243, 0x72F9 }, { 7244, 0x72FD },
    { 7245, 0x72F8 }, { 7246, 0x72F7 }, { 7247, 0x7386 }, { 7248, 0x73ED }, { 7249, 0x7409 }, { 7250, 0x73EE }, { 7251, 0x73E0 }, { 7252, 0x73EA },
    { 7253, 0x73DE }, { 7254, 0x7554 }, { 7255, 0x755D }, { 7256, 0x755C }, { 7257, 0x755A }, { 7258, 0x7559 }, { 7259, 0x75BE }, { 7260, 0x75C5 },
    { 7261, 0x75C7 }, { 7262, 0x75B2 }, { 7263, 0x75B3 }, { 7264, 0x75BD }, { 7265, 0x75BC }, { 7266, 0x75B9 }, { 7267, 0x75C2 }, { 7268, 0x75B8 },
    { 7269, 0x768B }, { 7270, 0x76B0 }, { 7271, 0x76CA }, { 7272, 0x76CD }, { 7273, 0x76CE }, { 7274, 0x7729 }, { 7275, 0x771F }, { 7276, 0x7720 },
    { 7277, 0x7728 }, { 7278, 0x77E9 }, { 7279, 0x7830 }, { 7280, 0x7827 }, { 7281, 0x7838 }, { 7282, 0x781D }, { 7283, 0x7834 }, { 7284, 0x7837 },
    { 7285, 0x7825 }, { 7286, 0x782D }, { 7287, 0x7820 }, { 7288, 0x781F }, { 7289, 0x7832 }, { 7290, 0x7955 }, { 7291, 0x7950 }, { 7292, 0x7960 },
    { 7293, 0x795F }, { 7294, 0x7956 }, { 7295, 0x795E }, { 7296, 0x795D }, { 7297, 0x7957 }, { 7298, 0x795A }, { 7299, 0x79E4 }, { 7300, 0x79E3 },
    { 7301, 0x79E7 }, { 7302, 0x79DF }, { 7303, 0x79E6 }, { 7304, 0x79E9 }, { 7305, 0x79D8 }, { 7306, 0x7A84 }, { 7307, 0x7A88 }, { 7308, 0x7AD9 },
    { 7309, 0x7B06 }, { 7310, 0x7B11 }, { 7311, 0x7C89 }, { 7312, 0x7D21 }, { 7313, 0x7D17 }, { 7314, 0x7D0B }, { 7315, 0x7D0A }, { 7316, 0x7D20 },
    { 7317, 0x7D22 }, { 7318, 0x7D14 }, { 7319, 0x7D10 }, { 7320, 0x7D15 }, { 7321, 0x7D1A }, { 7322, 0x7D1C }, { 7323, 0x7D0D }, { 7324, 0x7D19 },
    { 7325, 0x7D1B }, { 7326, 0x7F3A }, { 7327, 0x7F5F }, { 7328, 0x7F94 }, { 7329, 0x7FC5 }, { 7330, 0x7FC1 }, { 7331, 0x8006 }, { 7332, 0x8018 },
    { 7333, 0x8015 }, { 7334, 0x8019 }, { 7335, 0x8017 }, { 7336, 0x803D }, { 7337, 0x803F }, { 7338, 0x80F1 }, { 7339, 0x8102 }, { 7340, 0x80F0 },
    { 7341, 0x8105 }, { 7342, 0x80ED }, { 7343, 0x80F4 }, { 7344, 0x8106 }, { 7345, 0x80F8 }, { 7346, 0x80F3 }, { 7347, 0x8108 }, { 7348, 0x80FD },
    { 7349, 0x810A }, { 7350, 0x80FC }, { 7351, 0x80EF }, { 7352, 0x81ED }, { 7353, 0x81EC }, { 7354, 0x8200 }, { 7355, 0x8210 }, { 7356, 0x822A },
    { 7357, 0x822B }, { 7358, 0x8228 }, { 7359, 0x822C }, { 7360, 0x82BB }, { 7361, 0x832B }, { 7362, 0x8352 }, { 7363, 0x8354 }, { 7364, 0x834A },
    { 7365, 0x8338 }, { 7366, 0x8350 }, { 7367, 0x8349 }, { 7368, 0x8335 }, { 7369, 0x8334 }, { 7370, 0x834F }, { 7371, 0x8332 }, { 7372, 0x8339 },
    { 7373, 0x8336 }, { 7374, 0x8317 }, { 7375, 0x8340 }, { 7376, 0x8331 }, { 7377, 0x8328 }, { 7378, 0x8343 }, { 7379, 0x8654 }, { 7380, 0x868A },
    { 7381, 0x86AA }, { 7382, 0x8693 }, { 7383, 0x86A4 }, { 7384, 0x86A9 }, { 7385, 0x868C }, { 7386, 0x86A3 }, { 7387, 0x869C }, { 7388, 0x8870 },
    { 7389, 0x8877 }, { 7390, 0x8881 }, { 7391, 0x8882 }, { 7392, 0x887D }, { 7393, 0x8879 }, { 7394, 0x8A18 }, { 7395, 0x8A10 }, { 7396, 0x8A0E },
    { 7397, 0x8A0C }, { 7398, 0x8A15 }, { 7399, 0x8A0A }, { 7400, 0x8A17 }, { 7401, 0x8A13 }, { 7402, 0x8A16 }, { 7403, 0x8A0F }, { 7404, 0x8A11 },
    { 7405, 0x8C48 }, { 7406, 0x8C7A }, { 7407, 0x8C79 }, { 7408, 0x8CA1 }, { 7409, 0x8CA2 }, { 7410, 0x8D77 }, { 7411, 0x8EAC }, { 7412, 0x8ED2 },
    { 7413, 0x8ED4 }, { 7414, 0x8ECF }, { 7415, 0x8FB1 }, { 7416, 0x9001 }, { 7417, 0x9006 }, { 7418, 0x8FF7 }, { 7419, 0x9000 }, { 7420, 0x8FFA },
    { 7421, 0x8FF4 }, { 7422, 0x9003 }, { 7423, 0x8FFD }, { 7424, 0x9005 }, { 7425, 0x8FF8 }, { 7426, 0x9095 }, { 7427, 0x90E1 }, { 7428, 0x90DD },
    { 7429, 0x90E2 }, { 7430, 0x9152 }, { 7431, 0x914D }, { 7432, 0x914C }, { 7433, 0x91D8 }, { 7434, 0x91DD }, { 7435, 0x91D7 }, { 7436, 0x91DC },
    { 7437, 0x91D9 }, { 7438, 0x9583 }, { 7439, 0x9662 }, { 7440, 0x9663 }, { 7441, 0x9661 }, { 7442, 0x965B }, { 7443, 0x965D }, { 7444, 0x9664 },
    { 7445, 0x9658 }, { 7446, 0x965E }, { 7447, 0x96BB }, { 7448, 0x98E2 }, { 7449, 0x99AC }, { 7450, 0x9AA8 }, { 7451, 0x9AD8 }, { 7452, 0x9B25 },
    { 7453, 0x9B32 }, { 7454, 0x9B3C }, { 7455, 0x4E7E }, { 7456, 0x507A }, { 7457, 0x507D }, { 7458, 0x505C }, { 7459, 0x5047 }, { 7460, 0x5043 },
    { 7461, 0x504C }, { 7462, 0x505A }, { 7463, 0x5049 }, { 7464, 0x5065 }, { 7465, 0x5076 }, { 7466, 0x504E }, { 7467, 0x5055 }, { 7468, 0x5075 },
    { 7469, 0x5074 }, { 7470, 0x5077 }, { 7471, 0x504F }, { 7472, 0x500F }, { 7473, 0x506F }, { 7474, 0x506D }, { 7475, 0x515C }, { 7476, 0x5195 },
    { 7477, 0x51F0 }, { 7478, 0x526A }, { 7479, 0x526F }, { 7480, 0x52D2 }, { 7481, 0x52D9 }, { 7482, 0x52D8 }, { 7483, 0x52D5 }, { 7484, 0x5310 },
    { 7485, 0x530F }, { 7486, 0x5319 }, { 7487, 0x533F }, { 7488, 0x5340 }, { 7489, 0x533E }, { 7490, 0x53C3 }, { 7491, 0x66FC }, { 7492, 0x5546 },
    { 7493, 0x556A }, { 7494, 0x5566 }, { 7495, 0x5544 }, { 7496, 0x555E }, { 7497, 0x5561 }, { 7498, 0x5543 }, { 7499, 0x554A }, { 7500, 0x5531 },
    { 7501, 0x5556 }, { 7502, 0x554F }, { 7503, 0x5555 }, { 7504, 0x552F }, { 7505, 0x5564 }, { 7506, 0x5538 }, { 7507, 0x552E }, { 7508, 0x555C },
    { 7509, 0x552C }, { 7510, 0x5563 }, { 7511, 0x5533 }, { 7512, 0x5541 }, { 7513, 0x5557 }, { 7514, 0x5708 }, { 7515, 0x570B }, { 7516, 0x5709 },
    { 7517, 0x57DF }, { 7518, 0x5805 }, { 7519, 0x580A }, { 7520, 0x5806 }, { 7521, 0x57E0 }, { 7522, 0x57E4 }, { 7523, 0x57FA }, { 7524, 0x5802 },
    { 7525, 0x5835 }, { 7526, 0x57F7 }, { 7527, 0x57F9 }, { 7528, 0x5920 }, { 7529, 0x5962 }, { 7530, 0x5A36 }, { 7531, 0x5A41 }, { 7532, 0x5A49 },
    { 7533, 0x5A66 }, { 7534, 0x5A6A }, { 7535, 0x5A40 }, { 7536, 0x5A3C }, { 7537, 0x5A62 }, { 7538, 0x5A5A }, { 7539, 0x5A46 }, { 7540, 0x5A4A },
    { 7541, 0x5B70 }, { 7542, 0x5BC7 }, { 7543, 0x5BC5 }, { 7544, 0x5BC4 }, { 7545, 0x5BC2 }, { 7546, 0x5BBF }, { 7547, 0x5BC6 }, { 7548, 0x5C09 },
    { 7549, 0x5C08 }, { 7550, 0x5C07 }, { 7551, 0x5C60 }, { 7552, 0x5C5C }, { 7553, 0x5C5D }, { 7554, 0x5D07 }, { 7555, 0x5D06 }, { 7556, 0x5D0E },
    { 7557, 0x5D1B }, { 7558, 0x5D16 }, { 7559, 0x5D22 }, { 7560, 0x5D11 }, { 7561, 0x5D29 }, { 7562, 0x5D14 }, { 7563, 0x5D19 }, { 7564, 0x5D24 },
    { 7565, 0x5D27 }, { 7566, 0x5D17 }, { 7567, 0x5DE2 }, { 7568, 0x5E38 }, { 7569, 0x5E36 }, { 7570, 0x5E33 }, { 7571, 0x5E37 }, { 7572, 0x5EB7 },
    { 7573, 0x5EB8 }, { 7574, 0x5EB6 }, { 7575, 0x5EB5 }, { 7576, 0x5EBE }, { 7577, 0x5F35 }, { 7578, 0x5F37 }, { 7579, 0x5F57 }, { 7580, 0x5F6C },
    { 7581, 0x5F69 }, { 7582, 0x5F6B }, { 7583, 0x5F97 }, { 7584, 0x5F99 }, { 7585, 0x5F9E }, { 7586, 0x5F98 }, { 7587, 0x5FA1 }, { 7588, 0x5FA0 },
    { 7589, 0x5F9C }, { 7590, 0x607F }, { 7591, 0x60A3 }, { 7592, 0x6089 }, { 7593, 0x60A0 }, { 7594, 0x60A8 }, { 7595, 0x60CB }, { 7596, 0x60B4 },
    { 7597, 0x60E6 }, { 7598, 0x60BD }, { 7599, 0x60C5 }, { 7600, 0x60BB }, { 7601, 0x60B5 }, { 7602, 0x60DC }, { 7603, 0x60BC }, { 7604, 0x60D8 },
    { 7605, 0x60D5 }, { 7606, 0x60C6 }, { 7607, 0x60DF }, { 7608, 0x60B8 }, { 7609, 0x60DA }, { 7610, 0x60C7 }, { 7611, 0x621A }, { 7612, 0x621B },
    { 7613, 0x6248 }, { 7614, 0x63A0 }, { 7615, 0x63A7 }, { 7616, 0x6372 }, { 7617, 0x6396 }, { 7618, 0x63A2 }, { 7619, 0x63A5 }, { 7620, 0x6377 },
    { 7621, 0x6367 }, { 7622, 0x6398 }, { 7623, 0x63AA }, { 7624, 0x6371 }, { 7625, 0x63A9 }, { 7626, 0x6389 }, { 7627, 0x6383 }, { 7628, 0x639B },
    { 7629, 0x636B }, { 7630, 0x63A8 }, { 7631, 0x6384 }, { 7632, 0x6388 }, { 7633, 0x6399 }, { 7634, 0x63A1 }, { 7635, 0x63AC }, { 7636, 0x6392 },
    { 7637, 0x638F }, { 7638, 0x6380 }, { 7639, 0x637B }, { 7640, 0x6369 }, { 7641, 0x6368 }, { 7642, 0x637A }, { 7643, 0x655D }, { 7644, 0x6556 },
    { 7645, 0x6551 }, { 7646, 0x6559 }, { 7647, 0x6557 }, { 7648, 0x555F }, { 7649, 0x654F }, { 7650, 0x6558 }, { 7651, 0x6555 }, { 7652, 0x6554 },
    { 7653, 0x659C }, { 7654, 0x659B }, { 7655, 0x65AC }, { 7656, 0x65CF }, { 7657, 0x65CB }, { 7658, 0x65CC }, { 7659, 0x65CE }, { 7660, 0x665D },
    { 7661, 0x665A }, { 7662, 0x6664 }, { 7663, 0x6668 }, { 7664, 0x6666 }, { 7665, 0x665E }, { 7666, 0x66F9 }, { 7667, 0x52D7 }, { 7668, 0x671B },
    { 7669, 0x6881 }, { 7670, 0x68AF }, { 7671, 0x68A2 }, { 7672, 0x6893 }, { 7673, 0x68B5 }, { 7674, 0x687F }, { 7675, 0x6876 }, { 7676, 0x68B1 },
    { 7677, 0x68A7 }, { 7678, 0x6897 }, { 7679, 0x68B0 }, { 7680, 0x6883 }, { 7681, 0x68C4 }, { 7682, 0x68AD }, { 7683, 0x6886 }, { 7684, 0x6885 },
    { 7685, 0x6894 }, { 7686, 0x689D }, { 7687, 0x68A8 }, { 7688, 0x689F }, { 7689, 0x68A1 }, { 7690, 0x6882 }, { 7691, 0x6B32 }, { 7692, 0x6BBA },
    { 7693, 0x6BEB }, { 7694, 0x6BEC }, { 7695, 0x6C2B }, { 7696, 0x6D8E }, { 7697, 0x6DBC }, { 7698, 0x6DF3 }, { 7699, 0x6DD9 }, { 7700, 0x6DB2 },
    { 7701, 0x6DE1 }, { 7702, 0x6DCC }, { 7703, 0x6DE4 }, { 7704, 0x6DFB }, { 7705, 0x6DFA }, { 7706, 0x6E05 }, { 7707, 0x6DC7 }, { 7708, 0x6DCB },
    { 7709, 0x6DAF }, { 7710, 0x6DD1 }, { 7711, 0x6DAE }, { 7712, 0x6DDE }, { 7713, 0x6DF9 }, { 7714, 0x6DB8 }, { 7715, 0x6DF7 }, { 7716, 0x6DF5 },
    { 7717, 0x6DC5 }, { 7718, 0x6DD2 }, { 7719, 0x6E1A }, { 7720, 0x6DB5 }, { 7721, 0x6DDA }, { 7722, 0x6DEB }, { 7723, 0x6DD8 }, { 7724, 0x6DEA },
    { 7725, 0x6DF1 }, { 7726, 0x6DEE }, { 7727, 0x6DE8 }, { 7728, 0x6DC6 }, { 7729, 0x6DC4 }, { 7730, 0x6DAA }, { 7731, 0x6DEC }, { 7732, 0x6DBF },
    { 7733, 0x6DE6 }, { 7734, 0x70F9 }, { 7735, 0x7109 }, { 7736, 0x710A }, { 7737, 0x70FD }, { 7738, 0x70EF }, { 7739, 0x723D }, { 7740, 0x727D },
    { 7741, 0x7281 }, { 7742, 0x731C }, { 7743, 0x731B }, { 7744, 0x7316 }, { 7745, 0x7313 }, { 7746, 0x7319 }, { 7747, 0x7387 }, { 7748, 0x7405 },
    { 7749, 0x740A }, { 7750, 0x7403 }, { 7751, 0x7406 }, { 7752, 0x73FE }, { 7753, 0x740D }, { 7754, 0x74E0 }, { 7755, 0x74F6 }, { 7756, 0x74F7 },
    { 7757, 0x751C }, { 7758, 0x7522 }, { 7759, 0x7565 }, { 7760, 0x7566 }, { 7761, 0x7562 }, { 7762, 0x7570 }, { 7763, 0x758F }, { 7764, 0x75D4 },
    { 7765, 0x75D5 }, { 7766, 0x75B5 }, { 7767, 0x75CA }, { 7768, 0x75CD }, { 7769, 0x768E }, { 7770, 0x76D4 }, { 7771, 0x76D2 }, { 7772, 0x76DB },
    { 7773, 0x7737 }, { 7774, 0x773E }, { 7775, 0x773C }, { 7776, 0x7736 }, { 7777, 0x7738 }, { 7778, 0x773A }, { 7779, 0x786B }, { 7780, 0x7843 },
    { 7781, 0x784E }, { 7782, 0x7965 }, { 7783, 0x7968 }, { 7784, 0x796D }, { 7785, 0x79FB }, { 7786, 0x7A92 }, { 7787, 0x7A95 }, { 7788, 0x7B20 },
    { 7789, 0x7B28 }, { 7790, 0x7B1B }, { 7791, 0x7B2C }, { 7792, 0x7B26 }, { 7793, 0x7B19 }, { 7794, 0x7B1E }, { 7795, 0x7B2E }, { 7796, 0x7C92 },
    { 7797, 0x7C97 }, { 7798, 0x7C95 }, { 7799, 0x7D46 }, { 7800, 0x7D43 }, { 7801, 0x7D71 }, { 7802, 0x7D2E }, { 7803, 0x7D39 }, { 7804, 0x7D3C },
    { 7805, 0x7D40 }, { 7806, 0x7D30 }, { 7807, 0x7D33 }, { 7808, 0x7D44 }, { 7809, 0x7D2F }, { 7810, 0x7D42 }, { 7811, 0x7D32 }, { 7812, 0x7D31 },
    { 7813, 0x7F3D }, { 7814, 0x7F9E }, { 7815, 0x7F9A }, { 7816, 0x7FCC }, { 7817, 0x7FCE }, { 7818, 0x7FD2 }, { 7819, 0x801C }, { 7820, 0x804A },
    { 7821, 0x8046 }, { 7822, 0x812F }, { 7823, 0x8116 }, { 7824, 0x8123 }, { 7825, 0x812B }, { 7826, 0x8129 }, { 7827, 0x8130 }, { 7828, 0x8124 },
    { 7829, 0x8202 }, { 7830, 0x8235 }, { 7831, 0x8237 }, { 7832, 0x8236 }, { 7833, 0x8239 }, { 7834, 0x838E }, { 7835, 0x839E }, { 7836, 0x8398 },
    { 7837, 0x8378 }, { 7838, 0x83A2 }, { 7839, 0x8396 }, { 7840, 0x83BD }, { 7841, 0x83AB }, { 7842, 0x8392 }, { 7843, 0x838A }, { 7844, 0x8393 },
    { 7845, 0x8389 }, { 7846, 0x83A0 }, { 7847, 0x8377 }, { 7848, 0x837B }, { 7849, 0x837C }, { 7850, 0x8386 }, { 7851, 0x83A7 }, { 7852, 0x8655 },
    { 7853, 0x5F6A }, { 7854, 0x86C7 }, { 7855, 0x86C0 }, { 7856, 0x86B6 }, { 7857, 0x86C4 }, { 7858, 0x86B5 }, { 7859, 0x86C6 }, { 7860, 0x86CB },
    { 7861, 0x86B1 }, { 7862, 0x86AF }, { 7863, 0x86C9 }, { 7864, 0x8853 }, { 7865, 0x889E }, { 7866, 0x8888 }, { 7867, 0x88AB }, { 7868, 0x8892 },
    { 7869, 0x8896 }, { 7870, 0x888D }, { 7871, 0x888B }, { 7872, 0x8993 }, { 7873, 0x898F }, { 7874, 0x8A2A }, { 7875, 0x8A1D }, { 7876, 0x8A23 },
    { 7877, 0x8A25 }, { 7878, 0x8A31 }, { 7879, 0x8A2D }, { 7880, 0x8A1F }, { 7881, 0x8A1B }, { 7882, 0x8A22 }, { 7883, 0x8C49 }, { 7884, 0x8C5A },
    { 7885, 0x8CA9 }, { 7886, 0x8CAC }, { 7887, 0x8CAB }, { 7888, 0x8CA8 }, { 7889, 0x8CAA }, { 7890, 0x8CA7 }, { 7891, 0x8D67 }, { 7892, 0x8D66 },
    { 7893, 0x8DBE }, { 7894, 0x8DBA }, { 7895, 0x8EDB }, { 7896, 0x8EDF }, { 7897, 0x9019 }, { 7898, 0x900D }, { 7899, 0x901A }, { 7900, 0x9017 },
    { 7901, 0x9023 }, { 7902, 0x901F }, { 7903, 0x901D }, { 7904, 0x9010 }, { 7905, 0x9015 }, { 7906, 0x901E }, { 7907, 0x9020 }, { 7908, 0x900F },
    { 7909, 0x9022 }, { 7910, 0x9016 }, { 7911, 0x901B }, { 7912, 0x9014 }, { 7913, 0x90E8 }, { 7914, 0x90ED }, { 7915, 0x90FD }, { 7916, 0x9157 },
    { 7917, 0x91CE }, { 7918, 0x91F5 }, { 7919, 0x91E6 }, { 7920, 0x91E3 }, { 7921, 0x91E7 }, { 7922, 0x91ED }, { 7923, 0x91E9 }, { 7924, 0x9589 },
    { 7925, 0x966A }, { 7926, 0x9675 }, { 7927, 0x9673 }, { 7928, 0x9678 }, { 7929, 0x9670 }, { 7930, 0x9674 }, { 7931, 0x9676 }, { 7932, 0x9677 },
    { 7933, 0x966C }, { 7934, 0x96C0 }, { 7935, 0x96EA }, { 7936, 0x96E9 }, { 7937, 0x7AE0 }, { 7938, 0x7ADF }, { 7939, 0x9802 }, { 7940, 0x9803 },
    { 7941, 0x9B5A }, { 7942, 0x9CE5 }, { 7943, 0x9E75 }, { 7944, 0x9E7F }, { 7945, 0x9EA5 }, { 7946, 0x9EBB }, { 7947, 0x50A2 }, { 7948, 0x508D },
    { 7949, 0x5085 }, { 7950, 0x5099 }, { 7951, 0x5091 }, { 7952, 0x5080 }, { 7953, 0x5096 }, { 7954, 0x5098 }, { 7955, 0x509A }, { 7956, 0x6700 },
    { 7957, 0x51F1 }, { 7958, 0x5272 }, { 7959, 0x5274 }, { 7960, 0x5275 }, { 7961, 0x5269 }, { 7962, 0x52DE }, { 7963, 0x52DD }, { 7964, 0x52DB },
    { 7965, 0x535A }, { 7966, 0x53A5 }, { 7967, 0x557B }, { 7968, 0x5580 }, { 7969, 0x55A7 }, { 7970, 0x557C }, { 7971, 0x558A }, { 7972, 0x559D },
    { 7973, 0x5598 }, { 7974, 0x5582 }, { 7975, 0x559C }, { 7976, 0x55AA }, { 7977, 0x5594 }, { 7978, 0x5587 }, { 7979, 0x558B }, { 7980, 0x5583 },
    { 7981, 0x55B3 }, { 7982, 0x55AE }, { 7983, 0x559F }, { 7984, 0x553E }, { 7985, 0x55B2 }, { 7986, 0x559A }, { 7987, 0x55BB }, { 7988, 0x55AC },
    { 7989, 0x55B1 }, { 7990, 0x557E }, { 7991, 0x5589 }, { 7992, 0x55AB }, { 7993, 0x5599 }, { 7994, 0x570D }, { 7995, 0x582F }, { 7996, 0x582A },
    { 7997, 0x5834 }, { 7998, 0x5824 }, { 7999, 0x5830 }, { 8000, 0x5831 }, { 8001, 0x5821 }, { 8002, 0x581D }, { 8003, 0x5820 }, { 8004, 0x58F9 },
    { 8005, 0x58FA }, { 8006, 0x5960 }, { 8007, 0x5A77 }, { 8008, 0x5A9A }, { 8009, 0x5A7F }, { 8010, 0x5A92 }, { 8011, 0x5A9B }, { 8012, 0x5AA7 },
    { 8013, 0x5B73 }, { 8014, 0x5B71 }, { 8015, 0x5BD2 }, { 8016, 0x5BCC }, { 8017, 0x5BD3 }, { 8018, 0x5BD0 }, { 8019, 0x5C0A }, { 8020, 0x5C0B },
    { 8021, 0x5C31 }, { 8022, 0x5D4C }, { 8023, 0x5D50 }, { 8024, 0x5D34 }, { 8025, 0x5D47 }, { 8026, 0x5DFD }, { 8027, 0x5E45 }, { 8028, 0x5E3D },
    { 8029, 0x5E40 }, { 8030, 0x5E43 }, { 8031, 0x5E7E }, { 8032, 0x5ECA }, { 8033, 0x5EC1 }, { 8034, 0x5EC2 }, { 8035, 0x5EC4 }, { 8036, 0x5F3C },
    { 8037, 0x5F6D }, { 8038, 0x5FA9 }, { 8039, 0x5FAA }, { 8040, 0x5FA8 }, { 8041, 0x60D1 }, { 8042, 0x60E1 }, { 8043, 0x60B2 }, { 8044, 0x60B6 },
    { 8045, 0x60E0 }, { 8046, 0x611C }, { 8047, 0x6123 }, { 8048, 0x60FA }, { 8049, 0x6115 }, { 8050, 0x60F0 }, { 8051, 0x60FB }, { 8052, 0x60F4 },
    { 8053, 0x6168 }, { 8054, 0x60F1 }, { 8055, 0x610E }, { 8056, 0x60F6 }, { 8057, 0x6109 }, { 8058, 0x6100 }, { 8059, 0x6112 }, { 8060, 0x621F },
    { 8061, 0x6249 }, { 8062, 0x63A3 }, { 8063, 0x638C }, { 8064, 0x63CF }, { 8065, 0x63C0 }, { 8066, 0x63E9 }, { 8067, 0x63C9 }, { 8068, 0x63C6 },
    { 8069, 0x63CD }, { 8070, 0x63D2 }, { 8071, 0x63E3 }, { 8072, 0x63D0 }, { 8073, 0x63E1 }, { 8074, 0x63D6 }, { 8075, 0x63ED }, { 8076, 0x63EE },
    { 8077, 0x6376 }, { 8078, 0x63F4 }, { 8079, 0x63EA }, { 8080, 0x63DB }, { 8081, 0x6452 }, { 8082, 0x63DA }, { 8083, 0x63F9 }, { 8084, 0x655E },
    { 8085, 0x6566 }, { 8086, 0x6562 }, { 8087, 0x6563 }, { 8088, 0x6591 }, { 8089, 0x6590 }, { 8090, 0x65AF }, { 8091, 0x666E }, { 8092, 0x6670 },
    { 8093, 0x6674 }, { 8094, 0x6676 }, { 8095, 0x666F }, { 8096, 0x6691 }, { 8097, 0x667A }, { 8098, 0x667E }, { 8099, 0x6677 }, { 8100, 0x66FE },
    { 8101, 0x66FF }, { 8102, 0x671F }, { 8103, 0x671D }, { 8104, 0x68FA }, { 8105, 0x68D5 }, { 8106, 0x68E0 }, { 8107, 0x68D8 }, { 8108, 0x68D7 },
    { 8109, 0x6905 }, { 8110, 0x68DF }, { 8111, 0x68F5 }, { 8112, 0x68EE }, { 8113, 0x68E7 }, { 8114, 0x68F9 }, { 8115, 0x68D2 }, { 8116, 0x68F2 },
    { 8117, 0x68E3 }, { 8118, 0x68CB }, { 8119, 0x68CD }, { 8120, 0x690D }, { 8121, 0x6912 }, { 8122, 0x690E }, { 8123, 0x68C9 }, { 8124, 0x68DA },
    { 8125, 0x696E }, { 8126, 0x68FB }, { 8127, 0x6B3E }, { 8128, 0x6B3A }, { 8129, 0x6B3D }, { 8130, 0x6B98 }, { 8131, 0x6B96 }, { 8132, 0x6BBC },
    { 8133, 0x6BEF }, { 8134, 0x6C2E }, { 8135, 0x6C2F }, { 8136, 0x6C2C }, { 8137, 0x6E2F }, { 8138, 0x6E38 }, { 8139, 0x6E54 }, { 8140, 0x6E21 },
    { 8141, 0x6E32 }, { 8142, 0x6E67 }, { 8143, 0x6E4A }, { 8144, 0x6E20 }, { 8145, 0x6E25 }, { 8146, 0x6E23 }, { 8147, 0x6E1B }, { 8148, 0x6E5B },
    { 8149, 0x6E58 }, { 8150, 0x6E24 }, { 8151, 0x6E56 }, { 8152, 0x6E6E }, { 8153, 0x6E2D }, { 8154, 0x6E26 }, { 8155, 0x6E6F }, { 8156, 0x6E34 },
    { 8157, 0x6E4D }, { 8158, 0x6E3A }, { 8159, 0x6E2C }, { 8160, 0x6E43 }, { 8161, 0x6E1D }, { 8162, 0x6E3E }, { 8163, 0x6ECB }, { 8164, 0x6E89 },
    { 8165, 0x6E19 }, { 8166, 0x6E4E }, { 8167, 0x6E63 }, { 8168, 0x6E44 }, { 8169, 0x6E72 }, { 8170, 0x6E69 }, { 8171, 0x6E5F }, { 8172, 0x7119 },
    { 8173, 0x711A }, { 8174, 0x7126 }, { 8175, 0x7130 }, { 8176, 0x7121 }, { 8177, 0x7136 }, { 8178, 0x716E }, { 8179, 0x711C }, { 8180, 0x724C },
    { 8181, 0x7284 }, { 8182, 0x7280 }, { 8183, 0x7336 }, { 8184, 0x7325 }, { 8185, 0x7334 }, { 8186, 0x7329 }, { 8187, 0x743A }, { 8188, 0x742A },
    { 8189, 0x7433 }, { 8190, 0x7422 }, { 8191, 0x7425 }, { 8192, 0x7435 }, { 8193, 0x7436 }, { 8194, 0x7434 }, { 8195, 0x742F }, { 8196, 0x741B },
    { 8197, 0x7426 }, { 8198, 0x7428 }, { 8199, 0x7525 }, { 8200, 0x7526 }, { 8201, 0x756B }, { 8202, 0x756A }, { 8203, 0x75E2 }, { 8204, 0x75DB },
    { 8205, 0x75E3 }, { 8206, 0x75D9 }, { 8207, 0x75D8 }, { 8208, 0x75DE }, { 8209, 0x75E0 }, { 8210, 0x767B }, { 8211, 0x767C }, { 8212, 0x7696 },
    { 8213, 0x7693 }, { 8214, 0x76B4 }, { 8215, 0x76DC }, { 8216, 0x774F }, { 8217, 0x77ED }, { 8218, 0x785D }, { 8219, 0x786C }, { 8220, 0x786F },
    { 8221, 0x7A0D }, { 8222, 0x7A08 }, { 8223, 0x7A0B }, { 8224, 0x7A05 }, { 8225, 0x7A00 }, { 8226, 0x7A98 }, { 8227, 0x7A97 }, { 8228, 0x7A96 },
    { 8229, 0x7AE5 }, { 8230, 0x7AE3 }, { 8231, 0x7B49 }, { 8232, 0x7B56 }, { 8233, 0x7B46 }, { 8234, 0x7B50 }, { 8235, 0x7B52 }, { 8236, 0x7B54 },
    { 8237, 0x7B4D }, { 8238, 0x7B4B }, { 8239, 0x7B4F }, { 8240, 0x7B51 }, { 8241, 0x7C9F }, { 8242, 0x7CA5 }, { 8243, 0x7D5E }, { 8244, 0x7D50 },
    { 8245, 0x7D68 }, { 8246, 0x7D55 }, { 8247, 0x7D2B }, { 8248, 0x7D6E }, { 8249, 0x7D72 }, { 8250, 0x7D61 }, { 8251, 0x7D66 }, { 8252, 0x7D62 },
    { 8253, 0x7D70 }, { 8254, 0x7D73 }, { 8255, 0x5584 }, { 8256, 0x7FD4 }, { 8257, 0x7FD5 }, { 8258, 0x800B }, { 8259, 0x8052 }, { 8260, 0x8085 },
    { 8261, 0x8155 }, { 8262, 0x8154 }, { 8263, 0x814B }, { 8264, 0x8151 }, { 8265, 0x814E }, { 8266, 0x8139 }, { 8267, 0x8146 }, { 8268, 0x813E },
    { 8269, 0x814C }, { 8270, 0x8153 }, { 8271, 0x8174 }, { 8272, 0x8212 }, { 8273, 0x821C }, { 8274, 0x83E9 }, { 8275, 0x8403 }, { 8276, 0x83F8 },
    { 8277, 0x840D }, { 8278, 0x83E0 }, { 8279, 0x83C5 }, { 8280, 0x840B }, { 8281, 0x83C1 }, { 8282, 0x83EF }, { 8283, 0x83F1 }, { 8284, 0x83F4 },
    { 8285, 0x8457 }, { 8286, 0x840A }, { 8287, 0x83F0 }, { 8288, 0x840C }, { 8289, 0x83CC }, { 8290, 0x83FD }, { 8291, 0x83F2 }, { 8292, 0x83CA },
    { 8293, 0x8438 }, { 8294, 0x840E }, { 8295, 0x8404 }, { 8296, 0x83DC }, { 8297, 0x8407 }, { 8298, 0x83D4 }, { 8299, 0x83DF }, { 8300, 0x865B },
    { 8301, 0x86DF }, { 8302, 0x86D9 }, { 8303, 0x86ED }, { 8304, 0x86D4 }, { 8305, 0x86DB }, { 8306, 0x86E4 }, { 8307, 0x86D0 }, { 8308, 0x86DE },
    { 8309, 0x8857 }, { 8310, 0x88C1 }, { 8311, 0x88C2 }, { 8312, 0x88B1 }, { 8313, 0x8983 }, { 8314, 0x8996 }, { 8315, 0x8A3B }, { 8316, 0x8A60 },
    { 8317, 0x8A55 }, { 8318, 0x8A5E }, { 8319, 0x8A3C }, { 8320, 0x8A41 }, { 8321, 0x8A54 }, { 8322, 0x8A5B }, { 8323, 0x8A50 }, { 8324, 0x8A46 },
    { 8325, 0x8A34 }, { 8326, 0x8A3A }, { 8327, 0x8A36 }, { 8328, 0x8A56 }, { 8329, 0x8C61 }, { 8330, 0x8C82 }, { 8331, 0x8CAF }, { 8332, 0x8CBC },
    { 8333, 0x8CB3 }, { 8334, 0x8CBD }, { 8335, 0x8CC1 }, { 8336, 0x8CBB }, { 8337, 0x8CC0 }, { 8338, 0x8CB4 }, { 8339, 0x8CB7 }, { 8340, 0x8CB6 },
    { 8341, 0x8CBF }, { 8342, 0x8CB8 }, { 8343, 0x8D8A }, { 8344, 0x8D85 }, { 8345, 0x8D81 }, { 8346, 0x8DCE }, { 8347, 0x8DDD }, { 8348, 0x8DCB },
    { 8349, 0x8DDA }, { 8350, 0x8DD1 }, { 8351, 0x8DCC }, { 8352, 0x8DDB }, { 8353, 0x8DC6 }, { 8354, 0x8EFB }, { 8355, 0x8EF8 }, { 8356, 0x8EFC },
    { 8357, 0x8F9C }, { 8358, 0x902E }, { 8359, 0x9035 }, { 8360, 0x9031 }, { 8361, 0x9038 }, { 8362, 0x9032 }, { 8363, 0x9036 }, { 8364, 0x9102 },
    { 8365, 0x90F5 }, { 8366, 0x9109 }, { 8367, 0x90FE }, { 8368, 0x9163 }, { 8369, 0x9165 }, { 8370, 0x91CF }, { 8371, 0x9214 }, { 8372, 0x9215 },
    { 8373, 0x9223 }, { 8374, 0x9209 }, { 8375, 0x921E }, { 8376, 0x920D }, { 8377, 0x9210 }, { 8378, 0x9207 }, { 8379, 0x9211 }, { 8380, 0x9594 },
    { 8381, 0x958F }, { 8382, 0x958B }, { 8383, 0x9591 }, { 8384, 0x9593 }, { 8385, 0x9592 }, { 8386, 0x958E }, { 8387, 0x968A }, { 8388, 0x968E },
    { 8389, 0x968B }, { 8390, 0x967D }, { 8391, 0x9685 }, { 8392, 0x9686 }, { 8393, 0x968D }, { 8394, 0x9672 }, { 8395, 0x9684 }, { 8396, 0x96C1 },
    { 8397, 0x96C5 }, { 8398, 0x96C4 }, { 8399, 0x96C6 }, { 8400, 0x96C7 }, { 8401, 0x96EF }, { 8402, 0x96F2 }, { 8403, 0x97CC }, { 8404, 0x9805 },
    { 8405, 0x9806 }, { 8406, 0x9808 }, { 8407, 0x98E7 }, { 8408, 0x98EA }, { 8409, 0x98EF }, { 8410, 0x98E9 }, { 8411, 0x98F2 }, { 8412, 0x98ED },
    { 8413, 0x99AE }, { 8414, 0x99AD }, { 8415, 0x9EC3 }, { 8416, 0x9ECD }, { 8417, 0x9ED1 }, { 8418, 0x4E82 }, { 8419, 0x50AD }, { 8420, 0x50B5 },
    { 8421, 0x50B2 }, { 8422, 0x50B3 }, { 8423, 0x50C5 }, { 8424, 0x50BE }, { 8425, 0x50AC }, { 8426, 0x50B7 }, { 8427, 0x50BB }, { 8428, 0x50AF },
    { 8429, 0x50C7 }, { 8430, 0x527F }, { 8431, 0x5277 }, { 8432, 0x527D }, { 8433, 0x52DF }, { 8434, 0x52E6 }, { 8435, 0x52E4 }, { 8436, 0x52E2 },
    { 8437, 0x52E3 }, { 8438, 0x532F }, { 8439, 0x55DF }, { 8440, 0x55E8 }, { 8441, 0x55D3 }, { 8442, 0x55E6 }, { 8443, 0x55CE }, { 8444, 0x55DC },
    { 8445, 0x55C7 }, { 8446, 0x55D1 }, { 8447, 0x55E3 }, { 8448, 0x55E4 }, { 8449, 0x55EF }, { 8450, 0x55DA }, { 8451, 0x55E1 }, { 8452, 0x55C5 },
    { 8453, 0x55C6 }, { 8454, 0x55E5 }, { 8455, 0x55C9 }, { 8456, 0x5712 }, { 8457, 0x5713 }, { 8458, 0x585E }, { 8459, 0x5851 }, { 8460, 0x5858 },
    { 8461, 0x5857 }, { 8462, 0x585A }, { 8463, 0x5854 }, { 8464, 0x586B }, { 8465, 0x584C }, { 8466, 0x586D }, { 8467, 0x584A }, { 8468, 0x5862 },
    { 8469, 0x5852 }, { 8470, 0x584B }, { 8471, 0x5967 }, { 8472, 0x5AC1 }, { 8473, 0x5AC9 }, { 8474, 0x5ACC }, { 8475, 0x5ABE }, { 8476, 0x5ABD },
    { 8477, 0x5ABC }, { 8478, 0x5AB3 }, { 8479, 0x5AC2 }, { 8480, 0x5AB2 }, { 8481, 0x5D69 }, { 8482, 0x5D6F }, { 8483, 0x5E4C }, { 8484, 0x5E79 },
    { 8485, 0x5EC9 }, { 8486, 0x5EC8 }, { 8487, 0x5F12 }, { 8488, 0x5F59 }, { 8489, 0x5FAC }, { 8490, 0x5FAE }, { 8491, 0x611A }, { 8492, 0x610F },
    { 8493, 0x6148 }, { 8494, 0x611F }, { 8495, 0x60F3 }, { 8496, 0x611B }, { 8497, 0x60F9 }, { 8498, 0x6101 }, { 8499, 0x6108 }, { 8500, 0x614E },
    { 8501, 0x614C }, { 8502, 0x6144 }, { 8503, 0x614D }, { 8504, 0x613E }, { 8505, 0x6134 }, { 8506, 0x6127 }, { 8507, 0x610D }, { 8508, 0x6106 },
    { 8509, 0x6137 }, { 8510, 0x6221 }, { 8511, 0x6222 }, { 8512, 0x6413 }, { 8513, 0x643E }, { 8514, 0x641E }, { 8515, 0x642A }, { 8516, 0x642D },
    { 8517, 0x643D }, { 8518, 0x642C }, { 8519, 0x640F }, { 8520, 0x641C }, { 8521, 0x6414 }, { 8522, 0x640D }, { 8523, 0x6436 }, { 8524, 0x6416 },
    { 8525, 0x6417 }, { 8526, 0x6406 }, { 8527, 0x656C }, { 8528, 0x659F }, { 8529, 0x65B0 }, { 8530, 0x6697 }, { 8531, 0x6689 }, { 8532, 0x6687 },
    { 8533, 0x6688 }, { 8534, 0x6696 }, { 8535, 0x6684 }, { 8536, 0x6698 }, { 8537, 0x668D }, { 8538, 0x6703 }, { 8539, 0x6994 }, { 8540, 0x696D },
    { 8541, 0x695A }, { 8542, 0x6977 }, { 8543, 0x6960 }, { 8544, 0x6954 }, { 8545, 0x6975 }, { 8546, 0x6930 }, { 8547, 0x6982 }, { 8548, 0x694A },
    { 8549, 0x6968 }, { 8550, 0x696B }, { 8551, 0x695E }, { 8552, 0x6953 }, { 8553, 0x6979 }, { 8554, 0x6986 }, { 8555, 0x695D }, { 8556, 0x6963 },
    { 8557, 0x695B }, { 8558, 0x6B47 }, { 8559, 0x6B72 }, { 8560, 0x6BC0 }, { 8561, 0x6BBF }, { 8562, 0x6BD3 }, { 8563, 0x6BFD }, { 8564, 0x6EA2 },
    { 8565, 0x6EAF }, { 8566, 0x6ED3 }, { 8567, 0x6EB6 }, { 8568, 0x6EC2 }, { 8569, 0x6E90 }, { 8570, 0x6E9D }, { 8571, 0x6EC7 }, { 8572, 0x6EC5 },
    { 8573, 0x6EA5 }, { 8574, 0x6E98 }, { 8575, 0x6EBC }, { 8576, 0x6EBA }, { 8577, 0x6EAB }, { 8578, 0x6ED1 }, { 8579, 0x6E96 }, { 8580, 0x6E9C },
    { 8581, 0x6EC4 }, { 8582, 0x6ED4 }, { 8583, 0x6EAA }, { 8584, 0x6EA7 }, { 8585, 0x6EB4 }, { 8586, 0x714E }, { 8587, 0x7159 }, { 8588, 0x7169 },
    { 8589, 0x7164 }, { 8590, 0x7149 }, { 8591, 0x7167 }, { 8592, 0x715C }, { 8593, 0x716C }, { 8594, 0x7166 }, { 8595, 0x714C }, { 8596, 0x7165 },
    { 8597, 0x715E }, { 8598, 0x7146 }, { 8599, 0x7168 }, { 8600, 0x7156 }, { 8601, 0x723A }, { 8602, 0x7252 }, { 8603, 0x7337 }, { 8604, 0x7345 },
    { 8605, 0x733F }, { 8606, 0x733E }, { 8607, 0x746F }, { 8608, 0x745A }, { 8609, 0x7455 }, { 8610, 0x745F }, { 8611, 0x745E }, { 8612, 0x7441 },
    { 8613, 0x743F }, { 8614, 0x7459 }, { 8615, 0x745B }, { 8616, 0x745C }, { 8617, 0x7576 }, { 8618, 0x7578 }, { 8619, 0x7600 }, { 8620, 0x75F0 },
    { 8621, 0x7601 }, { 8622, 0x75F2 }, { 8623, 0x75F1 }, { 8624, 0x75FA }, { 8625, 0x75FF }, { 8626, 0x75F4 }, { 8627, 0x75F3 }, { 8628, 0x76DE },
    { 8629, 0x76DF }, { 8630, 0x775B }, { 8631, 0x776B }, { 8632, 0x7766 }, { 8633, 0x775E }, { 8634, 0x7763 }, { 8635, 0x7779 }, { 8636, 0x776A },
    { 8637, 0x776C }, { 8638, 0x775C }, { 8639, 0x7765 }, { 8640, 0x7768 }, { 8641, 0x7762 }, { 8642, 0x77EE }, { 8643, 0x788E }, { 8644, 0x78B0 },
    { 8645, 0x7897 }, { 8646, 0x7898 }, { 8647, 0x788C }, { 8648, 0x7889 }, { 8649, 0x787C }, { 8650, 0x7891 }, { 8651, 0x7893 }, { 8652, 0x787F },
    { 8653, 0x797A }, { 8654, 0x797F }, { 8655, 0x7981 }, { 8656, 0x842C }, { 8657, 0x79BD }, { 8658, 0x7A1C }, { 8659, 0x7A1A }, { 8660, 0x7A20 },
    { 8661, 0x7A14 }, { 8662, 0x7A1F }, { 8663, 0x7A1E }, { 8664, 0x7A9F }, { 8665, 0x7AA0 }, { 8666, 0x7B77 }, { 8667, 0x7BC0 }, { 8668, 0x7B60 },
    { 8669, 0x7B6E }, { 8670, 0x7B67 }, { 8671, 0x7CB1 }, { 8672, 0x7CB3 }, { 8673, 0x7CB5 }, { 8674, 0x7D93 }, { 8675, 0x7D79 }, { 8676, 0x7D91 },
    { 8677, 0x7D81 }, { 8678, 0x7D8F }, { 8679, 0x7D5B }, { 8680, 0x7F6E }, { 8681, 0x7F69 }, { 8682, 0x7F6A }, { 8683, 0x7F72 }, { 8684, 0x7FA9 },
    { 8685, 0x7FA8 }, { 8686, 0x7FA4 }, { 8687, 0x8056 }, { 8688, 0x8058 }, { 8689, 0x8086 }, { 8690, 0x8084 }, { 8691, 0x8171 }, { 8692, 0x8170 },
    { 8693, 0x8178 }, { 8694, 0x8165 }, { 8695, 0x816E }, { 8696, 0x8173 }, { 8697, 0x816B }, { 8698, 0x8179 }, { 8699, 0x817A }, { 8700, 0x8166 },
    { 8701, 0x8205 }, { 8702, 0x8247 }, { 8703, 0x8482 }, { 8704, 0x8477 }, { 8705, 0x843D }, { 8706, 0x8431 }, { 8707, 0x8475 }, { 8708, 0x8466 },
    { 8709, 0x846B }, { 8710, 0x8449 }, { 8711, 0x846C }, { 8712, 0x845B }, { 8713, 0x843C }, { 8714, 0x8435 }, { 8715, 0x8461 }, { 8716, 0x8463 },
    { 8717, 0x8469 }, { 8718, 0x846D }, { 8719, 0x8446 }, { 8720, 0x865E }, { 8721, 0x865C }, { 8722, 0x865F }, { 8723, 0x86F9 }, { 8724, 0x8713 },
    { 8725, 0x8708 }, { 8726, 0x8707 }, { 8727, 0x8700 }, { 8728, 0x86FE }, { 8729, 0x86FB }, { 8730, 0x8702 }, { 8731, 0x8703 }, { 8732, 0x8706 },
    { 8733, 0x870A }, { 8734, 0x8859 }, { 8735, 0x88DF }, { 8736, 0x88D4 }, { 8737, 0x88D9 }, { 8738, 0x88DC }, { 8739, 0x88D8 }, { 8740, 0x88DD },
    { 8741, 0x88E1 }, { 8742, 0x88CA }, { 8743, 0x88D5 }, { 8744, 0x88D2 }, { 8745, 0x899C }, { 8746, 0x89E3 }, { 8747, 0x8A6B }, { 8748, 0x8A72 },
    { 8749, 0x8A73 }, { 8750, 0x8A66 }, { 8751, 0x8A69 }, { 8752, 0x8A70 }, { 8753, 0x8A87 }, { 8754, 0x8A7C }, { 8755, 0x8A63 }, { 8756, 0x8AA0 },
    { 8757, 0x8A71 }, { 8758, 0x8A85 }, { 8759, 0x8A6D }, { 8760, 0x8A62 }, { 8761, 0x8A6E }, { 8762, 0x8A6C }, { 8763, 0x8A79 }, { 8764, 0x8A7B },
    { 8765, 0x8A3E }, { 8766, 0x8A68 }, { 8767, 0x8C62 }, { 8768, 0x8C8A }, { 8769, 0x8C89 }, { 8770, 0x8CCA }, { 8771, 0x8CC7 }, { 8772, 0x8CC8 },
    { 8773, 0x8CC4 }, { 8774, 0x8CB2 }, { 8775, 0x8CC3 }, { 8776, 0x8CC2 }, { 8777, 0x8CC5 }, { 8778, 0x8DE1 }, { 8779, 0x8DDF }, { 8780, 0x8DE8 },
    { 8781, 0x8DEF }, { 8782, 0x8DF3 }, { 8783, 0x8DFA }, { 8784, 0x8DEA }, { 8785, 0x8DE4 }, { 8786, 0x8DE6 }, { 8787, 0x8EB2 }, { 8788, 0x8F03 },
    { 8789, 0x8F09 }, { 8790, 0x8EFE }, { 8791, 0x8F0A }, { 8792, 0x8F9F }, { 8793, 0x8FB2 }, { 8794, 0x904B }, { 8795, 0x904A }, { 8796, 0x9053 },
    { 8797, 0x9042 }, { 8798, 0x9054 }, { 8799, 0x903C }, { 8800, 0x9055 }, { 8801, 0x9050 }, { 8802, 0x9047 }, { 8803, 0x904F }, { 8804, 0x904E },
    { 8805, 0x904D }, { 8806, 0x9051 }, { 8807, 0x903E }, { 8808, 0x9041 }, { 8809, 0x9112 }, { 8810, 0x9117 }, { 8811, 0x916C }, { 8812, 0x916A },
    { 8813, 0x9169 }, { 8814, 0x91C9 }, { 8815, 0x9237 }, { 8816, 0x9257 }, { 8817, 0x9238 }, { 8818, 0x923D }, { 8819, 0x9240 }, { 8820, 0x923E },
    { 8821, 0x925B }, { 8822, 0x924B }, { 8823, 0x9264 }, { 8824, 0x9251 }, { 8825, 0x9234 }, { 8826, 0x9249 }, { 8827, 0x924D }, { 8828, 0x9245 },
    { 8829, 0x9239 }, { 8830, 0x923F }, { 8831, 0x925A }, { 8832, 0x9598 }, { 8833, 0x9698 }, { 8834, 0x9694 }, { 8835, 0x9695 }, { 8836, 0x96CD },
    { 8837, 0x96CB }, { 8838, 0x96C9 }, { 8839, 0x96CA }, { 8840, 0x96F7 }, { 8841, 0x96FB }, { 8842, 0x96F9 }, { 8843, 0x96F6 }, { 8844, 0x9756 },
    { 8845, 0x9774 }, { 8846, 0x9776 }, { 8847, 0x9810 }, { 8848, 0x9811 }, { 8849, 0x9813 }, { 8850, 0x980A }, { 8851, 0x9812 }, { 8852, 0x980C },
    { 8853, 0x98FC }, { 8854, 0x98F4 }, { 8855, 0x98FD }, { 8856, 0x98FE }, { 8857, 0x99B3 }, { 8858, 0x99B1 }, { 8859, 0x99B4 }, { 8860, 0x9AE1 },
    { 8861, 0x9CE9 }, { 8862, 0x9E82 }, { 8863, 0x9F0E }, { 8864, 0x9F13 }, { 8865, 0x9F20 }, { 8866, 0x50E7 }, { 8867, 0x50EE }, { 8868, 0x50E5 },
    { 8869, 0x50D6 }, { 8870, 0x50ED }, { 8871, 0x50DA }, { 8872, 0x50D5 }, { 8873, 0x50CF }, { 8874, 0x50D1 }, { 8875, 0x50F1 }, { 8876, 0x50CE },
    { 8877, 0x50E9 }, { 8878, 0x5162 }, { 8879, 0x51F3 }, { 8880, 0x5283 }, { 8881, 0x5282 }, { 8882, 0x5331 }, { 8883, 0x53AD }, { 8884, 0x55FE },
    { 8885, 0x5600 }, { 8886, 0x561B }, { 8887, 0x5617 }, { 8888, 0x55FD }, { 8889, 0x5614 }, { 8890, 0x5606 }, { 8891, 0x5609 }, { 8892, 0x560D },
    { 8893, 0x560E }, { 8894, 0x55F7 }, { 8895, 0x5616 }, { 8896, 0x561F }, { 8897, 0x5608 }, { 8898, 0x5610 }, { 8899, 0x55F6 }, { 8900, 0x5718 },
    { 8901, 0x5716 }, { 8902, 0x5875 }, { 8903, 0x587E }, { 8904, 0x5883 }, { 8905, 0x5893 }, { 8906, 0x588A }, { 8907, 0x5879 }, { 8908, 0x5885 },
    { 8909, 0x587D }, { 8910, 0x58FD }, { 8911, 0x5925 }, { 8912, 0x5922 }, { 8913, 0x5924 }, { 8914, 0x596A }, { 8915, 0x5969 }, { 8916, 0x5AE1 },
    { 8917, 0x5AE6 }, { 8918, 0x5AE9 }, { 8919, 0x5AD7 }, { 8920, 0x5AD6 }, { 8921, 0x5AD8 }, { 8922, 0x5AE3 }, { 8923, 0x5B75 }, { 8924, 0x5BDE },
    { 8925, 0x5BE7 }, { 8926, 0x5BE1 }, { 8927, 0x5BE5 }, { 8928, 0x5BE6 }, { 8929, 0x5BE8 }, { 8930, 0x5BE2 }, { 8931, 0x5BE4 }, { 8932, 0x5BDF },
    { 8933, 0x5C0D }, { 8934, 0x5C62 }, { 8935, 0x5D84 }, { 8936, 0x5D87 }, { 8937, 0x5E5B }, { 8938, 0x5E63 }, { 8939, 0x5E55 }, { 8940, 0x5E57 },
    { 8941, 0x5E54 }, { 8942, 0x5ED3 }, { 8943, 0x5ED6 }, { 8944, 0x5F0A }, { 8945, 0x5F46 }, { 8946, 0x5F70 }, { 8947, 0x5FB9 }, { 8948, 0x6147 },
    { 8949, 0x613F }, { 8950, 0x614B }, { 8951, 0x6177 }, { 8952, 0x6162 }, { 8953, 0x6163 }, { 8954, 0x615F }, { 8955, 0x615A }, { 8956, 0x6158 },
    { 8957, 0x6175 }, { 8958, 0x622A }, { 8959, 0x6487 }, { 8960, 0x6458 }, { 8961, 0x6454 }, { 8962, 0x64A4 }, { 8963, 0x6478 }, { 8964, 0x645F },
    { 8965, 0x647A }, { 8966, 0x6451 }, { 8967, 0x6467 }, { 8968, 0x6434 }, { 8969, 0x646D }, { 8970, 0x647B }, { 8971, 0x6572 }, { 8972, 0x65A1 },
    { 8973, 0x65D7 }, { 8974, 0x65D6 }, { 8975, 0x66A2 }, { 8976, 0x66A8 }, { 8977, 0x669D }, { 8978, 0x699C }, { 8979, 0x69A8 }, { 8980, 0x6995 },
    { 8981, 0x69C1 }, { 8982, 0x69AE }, { 8983, 0x69D3 }, { 8984, 0x69CB }, { 8985, 0x699B }, { 8986, 0x69B7 }, { 8987, 0x69BB }, { 8988, 0x69AB },
    { 8989, 0x69B4 }, { 8990, 0x69D0 }, { 8991, 0x69CD }, { 8992, 0x69AD }, { 8993, 0x69CC }, { 8994, 0x69A6 }, { 8995, 0x69C3 }, { 8996, 0x69A3 },
    { 8997, 0x6B49 }, { 8998, 0x6B4C }, { 8999, 0x6C33 }, { 9000, 0x6F33 }, { 9001, 0x6F14 }, { 9002, 0x6EFE }, { 9003, 0x6F13 }, { 9004, 0x6EF4 },
    { 9005, 0x6F29 }, { 9006, 0x6F3E }, { 9007, 0x6F20 }, { 9008, 0x6F2C }, { 9009, 0x6F0F }, { 9010, 0x6F02 }, { 9011, 0x6F22 }, { 9012, 0x6EFF },
    { 9013, 0x6EEF }, { 9014, 0x6F06 }, { 9015, 0x6F31 }, { 9016, 0x6F38 }, { 9017, 0x6F32 }, { 9018, 0x6F23 }, { 9019, 0x6F15 }, { 9020, 0x6F2B },
    { 9021, 0x6F2F }, { 9022, 0x6F88 }, { 9023, 0x6F2A }, { 9024, 0x6EEC }, { 9025, 0x6F01 }, { 9026, 0x6EF2 }, { 9027, 0x6ECC }, { 9028, 0x6EF7 },
    { 9029, 0x7194 }, { 9030, 0x7199 }, { 9031, 0x717D }, { 9032, 0x718A }, { 9033, 0x7184 }, { 9034, 0x7192 }, { 9035, 0x723E }, { 9036, 0x7292 },
    { 9037, 0x7296 }, { 9038, 0x7344 }, { 9039, 0x7350 }, { 9040, 0x7464 }, { 9041, 0x7463 }, { 9042, 0x746A }, { 9043, 0x7470 }, { 9044, 0x746D },
    { 9045, 0x7504 }, { 9046, 0x7591 }, { 9047, 0x7627 }, { 9048, 0x760D }, { 9049, 0x760B }, { 9050, 0x7609 }, { 9051, 0x7613 }, { 9052, 0x76E1 },
    { 9053, 0x76E3 }, { 9054, 0x7784 }, { 9055, 0x777D }, { 9056, 0x777F }, { 9057, 0x7761 }, { 9058, 0x78C1 }, { 9059, 0x789F }, { 9060, 0x78A7 },
    { 9061, 0x78B3 }, { 9062, 0x78A9 }, { 9063, 0x78A3 }, { 9064, 0x798E }, { 9065, 0x798F }, { 9066, 0x798D }, { 9067, 0x7A2E }, { 9068, 0x7A31 },
    { 9069, 0x7AAA }, { 9070, 0x7AA9 }, { 9071, 0x7AED }, { 9072, 0x7AEF }, { 9073, 0x7BA1 }, { 9074, 0x7B95 }, { 9075, 0x7B8B }, { 9076, 0x7B75 },
    { 9077, 0x7B97 }, { 9078, 0x7B9D }, { 9079, 0x7B94 }, { 9080, 0x7B8F }, { 9081, 0x7BB8 }, { 9082, 0x7B87 }, { 9083, 0x7B84 }, { 9084, 0x7CB9 },
    { 9085, 0x7CBD }, { 9086, 0x7CBE }, { 9087, 0x7DBB }, { 9088, 0x7DB0 }, { 9089, 0x7D9C }, { 9090, 0x7DBD }, { 9091, 0x7DBE }, { 9092, 0x7DA0 },
    { 9093, 0x7DCA }, { 9094, 0x7DB4 }, { 9095, 0x7DB2 }, { 9096, 0x7DB1 }, { 9097, 0x7DBA }, { 9098, 0x7DA2 }, { 9099, 0x7DBF }, { 9100, 0x7DB5 },
    { 9101, 0x7DB8 }, { 9102, 0x7DAD }, { 9103, 0x7DD2 }, { 9104, 0x7DC7 }, { 9105, 0x7DAC }, { 9106, 0x7F70 }, { 9107, 0x7FE0 }, { 9108, 0x7FE1 },
    { 9109, 0x7FDF }, { 9110, 0x805E }, { 9111, 0x805A }, { 9112, 0x8087 }, { 9113, 0x8150 }, { 9114, 0x8180 }, { 9115, 0x818F }, { 9116, 0x8188 },
    { 9117, 0x818A }, { 9118, 0x817F }, { 9119, 0x8182 }, { 9120, 0x81E7 }, { 9121, 0x81FA }, { 9122, 0x8207 }, { 9123, 0x8214 }, { 9124, 0x821E },
    { 9125, 0x824B }, { 9126, 0x84C9 }, { 9127, 0x84BF }, { 9128, 0x84C6 }, { 9129, 0x84C4 }, { 9130, 0x8499 }, { 9131, 0x849E }, { 9132, 0x84B2 },
    { 9133, 0x849C }, { 9134, 0x84CB }, { 9135, 0x84B8 }, { 9136, 0x84C0 }, { 9137, 0x84D3 }, { 9138, 0x8490 }, { 9139, 0x84BC }, { 9140, 0x84D1 },
    { 9141, 0x84CA }, { 9142, 0x873F }, { 9143, 0x871C }, { 9144, 0x873B }, { 9145, 0x8722 }, { 9146, 0x8725 }, { 9147, 0x8734 }, { 9148, 0x8718 },
    { 9149, 0x8755 }, { 9150, 0x8737 }, { 9151, 0x8729 }, { 9152, 0x88F3 }, { 9153, 0x8902 }, { 9154, 0x88F4 }, { 9155, 0x88F9 }, { 9156, 0x88F8 },
    { 9157, 0x88FD }, { 9158, 0x88E8 }, { 9159, 0x891A }, { 9160, 0x88EF }, { 9161, 0x8AA6 }, { 9162, 0x8A8C }, { 9163, 0x8A9E }, { 9164, 0x8AA3 },
    { 9165, 0x8A8D }, { 9166, 0x8AA1 }, { 9167, 0x8A93 }, { 9168, 0x8AA4 }, { 9169, 0x8AAA }, { 9170, 0x8AA5 }, { 9171, 0x8AA8 }, { 9172, 0x8A98 },
    { 9173, 0x8A91 }, { 9174, 0x8A9A }, { 9175, 0x8AA7 }, { 9176, 0x8C6A }, { 9177, 0x8C8D }, { 9178, 0x8C8C }, { 9179, 0x8CD3 }, { 9180, 0x8CD1 },
    { 9181, 0x8CD2 }, { 9182, 0x8D6B }, { 9183, 0x8D99 }, { 9184, 0x8D95 }, { 9185, 0x8DFC }, { 9186, 0x8F14 }, { 9187, 0x8F12 }, { 9188, 0x8F15 },
    { 9189, 0x8F13 }, { 9190, 0x8FA3 }, { 9191, 0x9060 }, { 9192, 0x9058 }, { 9193, 0x905C }, { 9194, 0x9063 }, { 9195, 0x9059 }, { 9196, 0x905E },
    { 9197, 0x9062 }, { 9198, 0x905D }, { 9199, 0x905B }, { 9200, 0x9119 }, { 9201, 0x9118 }, { 9202, 0x911E }, { 9203, 0x9175 }, { 9204, 0x9178 },
    { 9205, 0x9177 }, { 9206, 0x9174 }, { 9207, 0x9278 }, { 9208, 0x9280 }, { 9209, 0x9285 }, { 9210, 0x9298 }, { 9211, 0x9296 }, { 9212, 0x927B },
    { 9213, 0x9293 }, { 9214, 0x929C }, { 9215, 0x92A8 }, { 9216, 0x927C }, { 9217, 0x9291 }, { 9218, 0x95A1 }, { 9219, 0x95A8 }, { 9220, 0x95A9 },
    { 9221, 0x95A3 }, { 9222, 0x95A5 }, { 9223, 0x95A4 }, { 9224, 0x9699 }, { 9225, 0x969C }, { 9226, 0x969B }, { 9227, 0x96CC }, { 9228, 0x96D2 },
    { 9229, 0x9700 }, { 9230, 0x977C }, { 9231, 0x9785 }, { 9232, 0x97F6 }, { 9233, 0x9817 }, { 9234, 0x9818 }, { 9235, 0x98AF }, { 9236, 0x98B1 },
    { 9237, 0x9903 }, { 9238, 0x9905 }, { 9239, 0x990C }, { 9240, 0x9909 }, { 9241, 0x99C1 }, { 9242, 0x9AAF }, { 9243, 0x9AB0 }, { 9244, 0x9AE6 },
    { 9245, 0x9B41 }, { 9246, 0x9B42 }, { 9247, 0x9CF4 }, { 9248, 0x9CF6 }, { 9249, 0x9CF3 }, { 9250, 0x9EBC }, { 9251, 0x9F3B }, { 9252, 0x9F4A },
    { 9253, 0x5104 }, { 9254, 0x5100 }, { 9255, 0x50FB }, { 9256, 0x50F5 }, { 9257, 0x50F9 }, { 9258, 0x5102 }, { 9259, 0x5108 }, { 9260, 0x5109 },
    { 9261, 0x5105 }, { 9262, 0x51DC }, { 9263, 0x5287 }, { 9264, 0x5288 }, { 9265, 0x5289 }, { 9266, 0x528D }, { 9267, 0x528A }, { 9268, 0x52F0 },
    { 9269, 0x53B2 }, { 9270, 0x562E }, { 9271, 0x563B }, { 9272, 0x5639 }, { 9273, 0x5632 }, { 9274, 0x563F }, { 9275, 0x5634 }, { 9276, 0x5629 },
    { 9277, 0x5653 }, { 9278, 0x564E }, { 9279, 0x5657 }, { 9280, 0x5674 }, { 9281, 0x5636 }, { 9282, 0x562F }, { 9283, 0x5630 }, { 9284, 0x5880 },
    { 9285, 0x589F }, { 9286, 0x589E }, { 9287, 0x58B3 }, { 9288, 0x589C }, { 9289, 0x58AE }, { 9290, 0x58A9 }, { 9291, 0x58A6 }, { 9292, 0x596D },
    { 9293, 0x5B09 }, { 9294, 0x5AFB }, { 9295, 0x5B0B }, { 9296, 0x5AF5 }, { 9297, 0x5B0C }, { 9298, 0x5B08 }, { 9299, 0x5BEE }, { 9300, 0x5BEC },
    { 9301, 0x5BE9 }, { 9302, 0x5BEB }, { 9303, 0x5C64 }, { 9304, 0x5C65 }, { 9305, 0x5D9D }, { 9306, 0x5D94 }, { 9307, 0x5E62 }, { 9308, 0x5E5F },
    { 9309, 0x5E61 }, { 9310, 0x5EE2 }, { 9311, 0x5EDA }, { 9312, 0x5EDF }, { 9313, 0x5EDD }, { 9314, 0x5EE3 }, { 9315, 0x5EE0 }, { 9316, 0x5F48 },
    { 9317, 0x5F71 }, { 9318, 0x5FB7 }, { 9319, 0x5FB5 }, { 9320, 0x6176 }, { 9321, 0x6167 }, { 9322, 0x616E }, { 9323, 0x615D }, { 9324, 0x6155 },
    { 9325, 0x6182 }, { 9326, 0x617C }, { 9327, 0x6170 }, { 9328, 0x616B }, { 9329, 0x617E }, { 9330, 0x61A7 }, { 9331, 0x6190 }, { 9332, 0x61AB },
    { 9333, 0x618E }, { 9334, 0x61AC }, { 9335, 0x619A }, { 9336, 0x61A4 }, { 9337, 0x6194 }, { 9338, 0x61AE }, { 9339, 0x622E }, { 9340, 0x6469 },
    { 9341, 0x646F }, { 9342, 0x6479 }, { 9343, 0x649E }, { 9344, 0x64B2 }, { 9345, 0x6488 }, { 9346, 0x6490 }, { 9347, 0x64B0 }, { 9348, 0x64A5 },
    { 9349, 0x6493 }, { 9350, 0x6495 }, { 9351, 0x64A9 }, { 9352, 0x6492 }, { 9353, 0x64AE }, { 9354, 0x64AD }, { 9355, 0x64AB }, { 9356, 0x649A },
    { 9357, 0x64AC }, { 9358, 0x6499 }, { 9359, 0x64A2 }, { 9360, 0x64B3 }, { 9361, 0x6575 }, { 9362, 0x6577 }, { 9363, 0x6578 }, { 9364, 0x66AE },
    { 9365, 0x66AB }, { 9366, 0x66B4 }, { 9367, 0x66B1 }, { 9368, 0x6A23 }, { 9369, 0x6A1F }, { 9370, 0x69E8 }, { 9371, 0x6A01 }, { 9372, 0x6A1E },
    { 9373, 0x6A19 }, { 9374, 0x69FD }, { 9375, 0x6A21 }, { 9376, 0x6A13 }, { 9377, 0x6A0A }, { 9378, 0x69F3 }, { 9379, 0x6A02 }, { 9380, 0x6A05 },
    { 9381, 0x69ED }, { 9382, 0x6A11 }, { 9383, 0x6B50 }, { 9384, 0x6B4E }, { 9385, 0x6BA4 }, { 9386, 0x6BC5 }, { 9387, 0x6BC6 }, { 9388, 0x6F3F },
    { 9389, 0x6F7C }, { 9390, 0x6F84 }, { 9391, 0x6F51 }, { 9392, 0x6F66 }, { 9393, 0x6F54 }, { 9394, 0x6F86 }, { 9395, 0x6F6D }, { 9396, 0x6F5B },
    { 9397, 0x6F78 }, { 9398, 0x6F6E }, { 9399, 0x6F8E }, { 9400, 0x6F7A }, { 9401, 0x6F70 }, { 9402, 0x6F64 }, { 9403, 0x6F97 }, { 9404, 0x6F58 },
    { 9405, 0x6ED5 }, { 9406, 0x6F6F }, { 9407, 0x6F60 }, { 9408, 0x6F5F }, { 9409, 0x719F }, { 9410, 0x71AC }, { 9411, 0x71B1 }, { 9412, 0x71A8 },
    { 9413, 0x7256 }, { 9414, 0x729B }, { 9415, 0x734E }, { 9416, 0x7357 }, { 9417, 0x7469 }, { 9418, 0x748B }, { 9419, 0x7483 }, { 9420, 0x747E },
    { 9421, 0x7480 }, { 9422, 0x757F }, { 9423, 0x7620 }, { 9424, 0x7629 }, { 9425, 0x761F }, { 9426, 0x7624 }, { 9427, 0x7626 }, { 9428, 0x7621 },
    { 9429, 0x7622 }, { 9430, 0x769A }, { 9431, 0x76BA }, { 9432, 0x76E4 }, { 9433, 0x778E }, { 9434, 0x7787 }, { 9435, 0x778C }, { 9436, 0x7791 },
    { 9437, 0x778B }, { 9438, 0x78CB }, { 9439, 0x78C5 }, { 9440, 0x78BA }, { 9441, 0x78CA }, { 9442, 0x78BE }, { 9443, 0x78D5 }, { 9444, 0x78BC },
    { 9445, 0x78D0 }, { 9446, 0x7A3F }, { 9447, 0x7A3C }, { 9448, 0x7A40 }, { 9449, 0x7A3D }, { 9450, 0x7A37 }, { 9451, 0x7A3B }, { 9452, 0x7AAF },
    { 9453, 0x7AAE }, { 9454, 0x7BAD }, { 9455, 0x7BB1 }, { 9456, 0x7BC4 }, { 9457, 0x7BB4 }, { 9458, 0x7BC6 }, { 9459, 0x7BC7 }, { 9460, 0x7BC1 },
    { 9461, 0x7BA0 }, { 9462, 0x7BCC }, { 9463, 0x7CCA }, { 9464, 0x7DE0 }, { 9465, 0x7DF4 }, { 9466, 0x7DEF }, { 9467, 0x7DFB }, { 9468, 0x7DD8 },
    { 9469, 0x7DEC }, { 9470, 0x7DDD }, { 9471, 0x7DE8 }, { 9472, 0x7DE3 }, { 9473, 0x7DDA }, { 9474, 0x7DDE }, { 9475, 0x7DE9 }, { 9476, 0x7D9E },
    { 9477, 0x7DD9 }, { 9478, 0x7DF2 }, { 9479, 0x7DF9 }, { 9480, 0x7F75 }, { 9481, 0x7F77 }, { 9482, 0x7FAF }, { 9483, 0x7FE9 }, { 9484, 0x8026 },
    { 9485, 0x819B }, { 9486, 0x819C }, { 9487, 0x819D }, { 9488, 0x81A0 }, { 9489, 0x819A }, { 9490, 0x8198 }, { 9491, 0x8517 }, { 9492, 0x853D },
    { 9493, 0x851A }, { 9494, 0x84EE }, { 9495, 0x852C }, { 9496, 0x852D }, { 9497, 0x8513 }, { 9498, 0x8511 }, { 9499, 0x8523 }, { 9500, 0x8521 },
    { 9501, 0x8514 }, { 9502, 0x84EC }, { 9503, 0x8525 }, { 9504, 0x84FF }, { 9505, 0x8506 }, { 9506, 0x8782 }, { 9507, 0x8774 }, { 9508, 0x8776 },
    { 9509, 0x8760 }, { 9510, 0x8766 }, { 9511, 0x8778 }, { 9512, 0x8768 }, { 9513, 0x8759 }, { 9514, 0x8757 }, { 9515, 0x874C }, { 9516, 0x8753 },
    { 9517, 0x885B }, { 9518, 0x885D }, { 9519, 0x8910 }, { 9520, 0x8907 }, { 9521, 0x8912 }, { 9522, 0x8913 }, { 9523, 0x8915 }, { 9524, 0x890A },
    { 9525, 0x8ABC }, { 9526, 0x8AD2 }, { 9527, 0x8AC7 }, { 9528, 0x8AC4 }, { 9529, 0x8A95 }, { 9530, 0x8ACB }, { 9531, 0x8AF8 }, { 9532, 0x8AB2 },
    { 9533, 0x8AC9 }, { 9534, 0x8AC2 }, { 9535, 0x8ABF }, { 9536, 0x8AB0 }, { 9537, 0x8AD6 }, { 9538, 0x8ACD }, { 9539, 0x8AB6 }, { 9540, 0x8AB9 },
    { 9541, 0x8ADB }, { 9542, 0x8C4C }, { 9543, 0x8C4E }, { 9544, 0x8C6C }, { 9545, 0x8CE0 }, { 9546, 0x8CDE }, { 9547, 0x8CE6 }, { 9548, 0x8CE4 },
    { 9549, 0x8CEC }, { 9550, 0x8CED }, { 9551, 0x8CE2 }, { 9552, 0x8CE3 }, { 9553, 0x8CDC }, { 9554, 0x8CEA }, { 9555, 0x8CE1 }, { 9556, 0x8D6D },
    { 9557, 0x8D9F }, { 9558, 0x8DA3 }, { 9559, 0x8E2B }, { 9560, 0x8E10 }, { 9561, 0x8E1D }, { 9562, 0x8E22 }, { 9563, 0x8E0F }, { 9564, 0x8E29 },
    { 9565, 0x8E1F }, { 9566, 0x8E21 }, { 9567, 0x8E1E }, { 9568, 0x8EBA }, { 9569, 0x8F1D }, { 9570, 0x8F1B }, { 9571, 0x8F1F }, { 9572, 0x8F29 },
    { 9573, 0x8F26 }, { 9574, 0x8F2A }, { 9575, 0x8F1C }, { 9576, 0x8F1E }, { 9577, 0x8F25 }, { 9578, 0x9069 }, { 9579, 0x906E }, { 9580, 0x9068 },
    { 9581, 0x906D }, { 9582, 0x9077 }, { 9583, 0x9130 }, { 9584, 0x912D }, { 9585, 0x9127 }, { 9586, 0x9131 }, { 9587, 0x9187 }, { 9588, 0x9189 },
    { 9589, 0x918B }, { 9590, 0x9183 }, { 9591, 0x92C5 }, { 9592, 0x92BB }, { 9593, 0x92B7 }, { 9594, 0x92EA }, { 9595, 0x92AC }, { 9596, 0x92E4 },
    { 9597, 0x92C1 }, { 9598, 0x92B3 }, { 9599, 0x92BC }, { 9600, 0x92D2 }, { 9601, 0x92C7 }, { 9602, 0x92F0 }, { 9603, 0x92B2 }, { 9604, 0x95AD },
    { 9605, 0x95B1 }, { 9606, 0x9704 }, { 9607, 0x9706 }, { 9608, 0x9707 }, { 9609, 0x9709 }, { 9610, 0x9760 }, { 9611, 0x978D }, { 9612, 0x978B },
    { 9613, 0x978F }, { 9614, 0x9821 }, { 9615, 0x982B }, { 9616, 0x981C }, { 9617, 0x98B3 }, { 9618, 0x990A }, { 9619, 0x9913 }, { 9620, 0x9912 },
    { 9621, 0x9918 }, { 9622, 0x99DD }, { 9623, 0x99D0 }, { 9624, 0x99DF }, { 9625, 0x99DB }, { 9626, 0x99D1 }, { 9627, 0x99D5 }, { 9628, 0x99D2 },
    { 9629, 0x99D9 }, { 9630, 0x9AB7 }, { 9631, 0x9AEE }, { 9632, 0x9AEF }, { 9633, 0x9B27 }, { 9634, 0x9B45 }, { 9635, 0x9B44 }, { 9636, 0x9B77 },
    { 9637, 0x9B6F }, { 9638, 0x9D06 }, { 9639, 0x9D09 }, { 9640, 0x9D03 }, { 9641, 0x9EA9 }, { 9642, 0x9EBE }, { 9643, 0x9ECE }, { 9644, 0x58A8 },
    { 9645, 0x9F52 }, { 9646, 0x5112 }, { 9647, 0x5118 }, { 9648, 0x5114 }, { 9649, 0x5110 }, { 9650, 0x5115 }, { 9651, 0x5180 }, { 9652, 0x51AA },
    { 9653, 0x51DD }, { 9654, 0x5291 }, { 9655, 0x5293 }, { 9656, 0x52F3 }, { 9657, 0x5659 }, { 9658, 0x566B }, { 9659, 0x5679 }, { 9660, 0x5669 },
    { 9661, 0x5664 }, { 9662, 0x5678 }, { 9663, 0x566A }, { 9664, 0x5668 }, { 9665, 0x5665 }, { 9666, 0x5671 }, { 9667, 0x566F }, { 9668, 0x566C },
    { 9669, 0x5662 }, { 9670, 0x5676 }, { 9671, 0x58C1 }, { 9672, 0x58BE }, { 9673, 0x58C7 }, { 9674, 0x58C5 }, { 9675, 0x596E }, { 9676, 0x5B1D },
    { 9677, 0x5B34 }, { 9678, 0x5B78 }, { 9679, 0x5BF0 }, { 9680, 0x5C0E }, { 9681, 0x5F4A }, { 9682, 0x61B2 }, { 9683, 0x6191 }, { 9684, 0x61A9 },
    { 9685, 0x618A }, { 9686, 0x61CD }, { 9687, 0x61B6 }, { 9688, 0x61BE }, { 9689, 0x61CA }, { 9690, 0x61C8 }, { 9691, 0x6230 }, { 9692, 0x64C5 },
    { 9693, 0x64C1 }, { 9694, 0x64CB }, { 9695, 0x64BB }, { 9696, 0x64BC }, { 9697, 0x64DA }, { 9698, 0x64C4 }, { 9699, 0x64C7 }, { 9700, 0x64C2 },
    { 9701, 0x64CD }, { 9702, 0x64BF }, { 9703, 0x64D2 }, { 9704, 0x64D4 }, { 9705, 0x64BE }, { 9706, 0x6574 }, { 9707, 0x66C6 }, { 9708, 0x66C9 },
    { 9709, 0x66B9 }, { 9710, 0x66C4 }, { 9711, 0x66C7 }, { 9712, 0x66B8 }, { 9713, 0x6A3D }, { 9714, 0x6A38 }, { 9715, 0x6A3A }, { 9716, 0x6A59 },
    { 9717, 0x6A6B }, { 9718, 0x6A58 }, { 9719, 0x6A39 }, { 9720, 0x6A44 }, { 9721, 0x6A62 }, { 9722, 0x6A61 }, { 9723, 0x6A4B }, { 9724, 0x6A47 },
    { 9725, 0x6A35 }, { 9726, 0x6A5F }, { 9727, 0x6A48 }, { 9728, 0x6B59 }, { 9729, 0x6B77 }, { 9730, 0x6C05 }, { 9731, 0x6FC2 }, { 9732, 0x6FB1 },
    { 9733, 0x6FA1 }, { 9734, 0x6FC3 }, { 9735, 0x6FA4 }, { 9736, 0x6FC1 }, { 9737, 0x6FA7 }, { 9738, 0x6FB3 }, { 9739, 0x6FC0 }, { 9740, 0x6FB9 },
    { 9741, 0x6FB6 }, { 9742, 0x6FA6 }, { 9743, 0x6FA0 }, { 9744, 0x6FB4 }, { 9745, 0x71BE }, { 9746, 0x71C9 }, { 9747, 0x71D0 }, { 9748, 0x71D2 },
    { 9749, 0x71C8 }, { 9750, 0x71D5 }, { 9751, 0x71B9 }, { 9752, 0x71CE }, { 9753, 0x71D9 }, { 9754, 0x71DC }, { 9755, 0x71C3 }, { 9756, 0x71C4 },
    { 9757, 0x7368 }, { 9758, 0x749C }, { 9759, 0x74A3 }, { 9760, 0x7498 }, { 9761, 0x749F }, { 9762, 0x749E }, { 9763, 0x74E2 }, { 9764, 0x750C },
    { 9765, 0x750D }, { 9766, 0x7634 }, { 9767, 0x7638 }, { 9768, 0x763A }, { 9769, 0x76E7 }, { 9770, 0x76E5 }, { 9771, 0x77A0 }, { 9772, 0x779E },
    { 9773, 0x779F }, { 9774, 0x77A5 }, { 9775, 0x78E8 }, { 9776, 0x78DA }, { 9777, 0x78EC }, { 9778, 0x78E7 }, { 9779, 0x79A6 }, { 9780, 0x7A4D },
    { 9781, 0x7A4E }, { 9782, 0x7A46 }, { 9783, 0x7A4C }, { 9784, 0x7A4B }, { 9785, 0x7ABA }, { 9786, 0x7BD9 }, { 9787, 0x7C11 }, { 9788, 0x7BC9 },
    { 9789, 0x7BE4 }, { 9790, 0x7BDB }, { 9791, 0x7BE1 }, { 9792, 0x7BE9 }, { 9793, 0x7BE6 }, { 9794, 0x7CD5 }, { 9795, 0x7CD6 }, { 9796, 0x7E0A },
    { 9797, 0x7E11 }, { 9798, 0x7E08 }, { 9799, 0x7E1B }, { 9800, 0x7E23 }, { 9801, 0x7E1E }, { 9802, 0x7E1D }, { 9803, 0x7E09 }, { 9804, 0x7E10 },
    { 9805, 0x7F79 }, { 9806, 0x7FB2 }, { 9807, 0x7FF0 }, { 9808, 0x7FF1 }, { 9809, 0x7FEE }, { 9810, 0x8028 }, { 9811, 0x81B3 }, { 9812, 0x81A9 },
    { 9813, 0x81A8 }, { 9814, 0x81FB }, { 9815, 0x8208 }, { 9816, 0x8258 }, { 9817, 0x8259 }, { 9818, 0x854A }, { 9819, 0x8559 }, { 9820, 0x8548 },
    { 9821, 0x8568 }, { 9822, 0x8569 }, { 9823, 0x8543 }, { 9824, 0x8549 }, { 9825, 0x856D }, { 9826, 0x856A }, { 9827, 0x855E }, { 9828, 0x8783 },
    { 9829, 0x879F }, { 9830, 0x879E }, { 9831, 0x87A2 }, { 9832, 0x878D }, { 9833, 0x8861 }, { 9834, 0x892A }, { 9835, 0x8932 }, { 9836, 0x8925 },
    { 9837, 0x892B }, { 9838, 0x8921 }, { 9839, 0x89AA }, { 9840, 0x89A6 }, { 9841, 0x8AE6 }, { 9842, 0x8AFA }, { 9843, 0x8AEB }, { 9844, 0x8AF1 },
    { 9845, 0x8B00 }, { 9846, 0x8ADC }, { 9847, 0x8AE7 }, { 9848, 0x8AEE }, { 9849, 0x8AFE }, { 9850, 0x8B01 }, { 9851, 0x8B02 }, { 9852, 0x8AF7 },
    { 9853, 0x8AED }, { 9854, 0x8AF3 }, { 9855, 0x8AF6 }, { 9856, 0x8AFC }, { 9857, 0x8C6B }, { 9858, 0x8C6D }, { 9859, 0x8C93 }, { 9860, 0x8CF4 },
    { 9861, 0x8E44 }, { 9862, 0x8E31 }, { 9863, 0x8E34 }, { 9864, 0x8E42 }, { 9865, 0x8E39 }, { 9866, 0x8E35 }, { 9867, 0x8F3B }, { 9868, 0x8F2F },
    { 9869, 0x8F38 }, { 9870, 0x8F33 }, { 9871, 0x8FA8 }, { 9872, 0x8FA6 }, { 9873, 0x9075 }, { 9874, 0x9074 }, { 9875, 0x9078 }, { 9876, 0x9072 },
    { 9877, 0x907C }, { 9878, 0x907A }, { 9879, 0x9134 }, { 9880, 0x9192 }, { 9881, 0x9320 }, { 9882, 0x9336 }, { 9883, 0x92F8 }, { 9884, 0x9333 },
    { 9885, 0x932F }, { 9886, 0x9322 }, { 9887, 0x92FC }, { 9888, 0x932B }, { 9889, 0x9304 }, { 9890, 0x931A }, { 9891, 0x9310 }, { 9892, 0x9326 },
    { 9893, 0x9321 }, { 9894, 0x9315 }, { 9895, 0x932E }, { 9896, 0x9319 }, { 9897, 0x95BB }, { 9898, 0x96A7 }, { 9899, 0x96A8 }, { 9900, 0x96AA },
    { 9901, 0x96D5 }, { 9902, 0x970E }, { 9903, 0x9711 }, { 9904, 0x9716 }, { 9905, 0x970D }, { 9906, 0x9713 }, { 9907, 0x970F }, { 9908, 0x975B },
    { 9909, 0x975C }, { 9910, 0x9766 }, { 9911, 0x9798 }, { 9912, 0x9830 }, { 9913, 0x9838 }, { 9914, 0x983B }, { 9915, 0x9837 }, { 9916, 0x982D },
    { 9917, 0x9839 }, { 9918, 0x9824 }, { 9919, 0x9910 }, { 9920, 0x9928 }, { 9921, 0x991E }, { 9922, 0x991B }, { 9923, 0x9921 }, { 9924, 0x991A },
    { 9925, 0x99ED }, { 9926, 0x99E2 }, { 9927, 0x99F1 }, { 9928, 0x9AB8 }, { 9929, 0x9ABC }, { 9930, 0x9AFB }, { 9931, 0x9AED }, { 9932, 0x9B28 },
    { 9933, 0x9B91 }, { 9934, 0x9D15 }, { 9935, 0x9D23 }, { 9936, 0x9D26 }, { 9937, 0x9D28 }, { 9938, 0x9D12 }, { 9939, 0x9D1B }, { 9940, 0x9ED8 },
    { 9941, 0x9ED4 }, { 9942, 0x9F8D }, { 9943, 0x9F9C }, { 9944, 0x512A }, { 9945, 0x511F }, { 9946, 0x5121 }, { 9947, 0x5132 }, { 9948, 0x52F5 },
    { 9949, 0x568E }, { 9950, 0x5680 }, { 9951, 0x5690 }, { 9952, 0x5685 }, { 9953, 0x5687 }, { 9954, 0x568F }, { 9955, 0x58D5 }, { 9956, 0x58D3 },
    { 9957, 0x58D1 }, { 9958, 0x58CE }, { 9959, 0x5B30 }, { 9960, 0x5B2A }, { 9961, 0x5B24 }, { 9962, 0x5B7A }, { 9963, 0x5C37 }, { 9964, 0x5C68 },
    { 9965, 0x5DBC }, { 9966, 0x5DBA }, { 9967, 0x5DBD }, { 9968, 0x5DB8 }, { 9969, 0x5E6B }, { 9970, 0x5F4C }, { 9971, 0x5FBD }, { 9972, 0x61C9 },
    { 9973, 0x61C2 }, { 9974, 0x61C7 }, { 9975, 0x61E6 }, { 9976, 0x61CB }, { 9977, 0x6232 }, { 9978, 0x6234 }, { 9979, 0x64CE }, { 9980, 0x64CA },
    { 9981, 0x64D8 }, { 9982, 0x64E0 }, { 9983, 0x64F0 }, { 9984, 0x64E6 }, { 9985, 0x64EC }, { 9986, 0x64F1 }, { 9987, 0x64E2 }, { 9988, 0x64ED },
    { 9989, 0x6582 }, { 9990, 0x6583 }, { 9991, 0x66D9 }, { 9992, 0x66D6 }, { 9993, 0x6A80 }, { 9994, 0x6A94 }, { 9995, 0x6A84 }, { 9996, 0x6AA2 },
    { 9997, 0x6A9C }, { 9998, 0x6ADB }, { 9999, 0x6AA3 }, { 10000, 0x6A7E }, { 10001, 0x6A97 }, { 10002, 0x6A90 }, { 10003, 0x6AA0 }, { 10004, 0x6B5C },
    { 10005, 0x6BAE }, { 10006, 0x6BDA }, { 10007, 0x6C08 }, { 10008, 0x6FD8 }, { 10009, 0x6FF1 }, { 10010, 0x6FDF }, { 10011, 0x6FE0 }, { 10012, 0x6FDB },
    { 10013, 0x6FE4 }, { 10014, 0x6FEB }, { 10015, 0x6FEF }, { 10016, 0x6F80 }, { 10017, 0x6FEC }, { 10018, 0x6FE1 }, { 10019, 0x6FE9 }, { 10020, 0x6FD5 },
    { 10021, 0x6FEE }, { 10022, 0x6FF0 }, { 10023, 0x71E7 }, { 10024, 0x71DF }, { 10025, 0x71EE }, { 10026, 0x71E6 }, { 10027, 0x71E5 }, { 10028, 0x71ED },
    { 10029, 0x71EC }, { 10030, 0x71F4 }, { 10031, 0x71E0 }, { 10032, 0x7235 }, { 10033, 0x7246 }, { 10034, 0x7370 }, { 10035, 0x7372 }, { 10036, 0x74A9 },
    { 10037, 0x74B0 }, { 10038, 0x74A6 }, { 10039, 0x74A8 }, { 10040, 0x7646 }, { 10041, 0x7642 }, { 10042, 0x764C }, { 10043, 0x76EA }, { 10044, 0x77B3 },
    { 10045, 0x77AA }, { 10046, 0x77B0 }, { 10047, 0x77AC }, { 10048, 0x77A7 }, { 10049, 0x77AD }, { 10050, 0x77EF }, { 10051, 0x78F7 }, { 10052, 0x78FA },
    { 10053, 0x78F4 }, { 10054, 0x78EF }, { 10055, 0x7901 }, { 10056, 0x79A7 }, { 10057, 0x79AA }, { 10058, 0x7A57 }, { 10059, 0x7ABF }, { 10060, 0x7C07 },
    { 10061, 0x7C0D }, { 10062, 0x7BFE }, { 10063, 0x7BF7 }, { 10064, 0x7C0C }, { 10065, 0x7BE0 }, { 10066, 0x7CE0 }, { 10067, 0x7CDC }, { 10068, 0x7CDE },
    { 10069, 0x7CE2 }, { 10070, 0x7CDF }, { 10071, 0x7CD9 }, { 10072, 0x7CDD }, { 10073, 0x7E2E }, { 10074, 0x7E3E }, { 10075, 0x7E46 }, { 10076, 0x7E37 },
    { 10077, 0x7E32 }, { 10078, 0x7E43 }, { 10079, 0x7E2B }, { 10080, 0x7E3D }, { 10081, 0x7E31 }, { 10082, 0x7E45 }, { 10083, 0x7E41 }, { 10084, 0x7E34 },
    { 10085, 0x7E39 }, { 10086, 0x7E48 }, { 10087, 0x7E35 }, { 10088, 0x7E3F }, { 10089, 0x7E2F }, { 10090, 0x7F44 }, { 10091, 0x7FF3 }, { 10092, 0x7FFC },
    { 10093, 0x8071 }, { 10094, 0x8072 }, { 10095, 0x8070 }, { 10096, 0x806F }, { 10097, 0x8073 }, { 10098, 0x81C6 }, { 10099, 0x81C3 }, { 10100, 0x81BA },
    { 10101, 0x81C2 }, { 10102, 0x81C0 }, { 10103, 0x81BF }, { 10104, 0x81BD }, { 10105, 0x81C9 }, { 10106, 0x81BE }, { 10107, 0x81E8 }, { 10108, 0x8209 },
    { 10109, 0x8271 }, { 10110, 0x85AA }, { 10111, 0x8584 }, { 10112, 0x857E }, { 10113, 0x859C }, { 10114, 0x8591 }, { 10115, 0x8594 }, { 10116, 0x85AF },
    { 10117, 0x859B }, { 10118, 0x8587 }, { 10119, 0x85A8 }, { 10120, 0x858A }, { 10121, 0x8667 }, { 10122, 0x87C0 }, { 10123, 0x87D1 }, { 10124, 0x87B3 },
    { 10125, 0x87D2 }, { 10126, 0x87C6 }, { 10127, 0x87AB }, { 10128, 0x87BB }, { 10129, 0x87BA }, { 10130, 0x87C8 }, { 10131, 0x87CB }, { 10132, 0x893B },
    { 10133, 0x8936 }, { 10134, 0x8944 }, { 10135, 0x8938 }, { 10136, 0x893D }, { 10137, 0x89AC }, { 10138, 0x8B0E }, { 10139, 0x8B17 }, { 10140, 0x8B19 },
    { 10141, 0x8B1B }, { 10142, 0x8B0A }, { 10143, 0x8B20 }, { 10144, 0x8B1D }, { 10145, 0x8B04 }, { 10146, 0x8B10 }, { 10147, 0x8C41 }, { 10148, 0x8C3F },
    { 10149, 0x8C73 }, { 10150, 0x8CFA }, { 10151, 0x8CFD }, { 10152, 0x8CFC }, { 10153, 0x8CF8 }, { 10154, 0x8CFB }, { 10155, 0x8DA8 }, { 10156, 0x8E49 },
    { 10157, 0x8E4B }, { 10158, 0x8E48 }, { 10159, 0x8E4A }, { 10160, 0x8F44 }, { 10161, 0x8F3E }, { 10162, 0x8F42 }, { 10163, 0x8F45 }, { 10164, 0x8F3F },
    { 10165, 0x907F }, { 10166, 0x907D }, { 10167, 0x9084 }, { 10168, 0x9081 }, { 10169, 0x9082 }, { 10170, 0x9080 }, { 10171, 0x9139 }, { 10172, 0x91A3 },
    { 10173, 0x919E }, { 10174, 0x919C }, { 10175, 0x934D }, { 10176, 0x9382 }, { 10177, 0x9328 }, { 10178, 0x9375 }, { 10179, 0x934A }, { 10180, 0x9365 },
    { 10181, 0x934B }, { 10182, 0x9318 }, { 10183, 0x937E }, { 10184, 0x936C }, { 10185, 0x935B }, { 10186, 0x9370 }, { 10187, 0x935A }, { 10188, 0x9354 },
    { 10189, 0x95CA }, { 10190, 0x95CB }, { 10191, 0x95CC }, { 10192, 0x95C8 }, { 10193, 0x95C6 }, { 10194, 0x96B1 }, { 10195, 0x96B8 }, { 10196, 0x96D6 },
    { 10197, 0x971C }, { 10198, 0x971E }, { 10199, 0x97A0 }, { 10200, 0x97D3 }, { 10201, 0x9846 }, { 10202, 0x98B6 }, { 10203, 0x9935 }, { 10204, 0x9A01 },
    { 10205, 0x99FF }, { 10206, 0x9BAE }, { 10207, 0x9BAB }, { 10208, 0x9BAA }, { 10209, 0x9BAD }, { 10210, 0x9D3B }, { 10211, 0x9D3F }, { 10212, 0x9E8B },
    { 10213, 0x9ECF }, { 10214, 0x9EDE }, { 10215, 0x9EDC }, { 10216, 0x9EDD }, { 10217, 0x9EDB }, { 10218, 0x9F3E }, { 10219, 0x9F4B }, { 10220, 0x53E2 },
    { 10221, 0x5695 }, { 10222, 0x56AE }, { 10223, 0x58D9 }, { 10224, 0x58D8 }, { 10225, 0x5B38 }, { 10226, 0x5F5D }, { 10227, 0x61E3 }, { 10228, 0x6233 },
    { 10229, 0x64F4 }, { 10230, 0x64F2 }, { 10231, 0x64FE }, { 10232, 0x6506 }, { 10233, 0x64FA }, { 10234, 0x64FB }, { 10235, 0x64F7 }, { 10236, 0x65B7 },
    { 10237, 0x66DC }, { 10238, 0x6726 }, { 10239, 0x6AB3 }, { 10240, 0x6AAC }, { 10241, 0x6AC3 }, { 10242, 0x6ABB }, { 10243, 0x6AB8 }, { 10244, 0x6AC2 },
    { 10245, 0x6AAE }, { 10246, 0x6AAF }, { 10247, 0x6B5F }, { 10248, 0x6B78 }, { 10249, 0x6BAF }, { 10250, 0x7009 }, { 10251, 0x700B }, { 10252, 0x6FFE },
    { 10253, 0x7006 }, { 10254, 0x6FFA }, { 10255, 0x7011 }, { 10256, 0x700F }, { 10257, 0x71FB }, { 10258, 0x71FC }, { 10259, 0x71FE }, { 10260, 0x71F8 },
    { 10261, 0x7377 }, { 10262, 0x7375 }, { 10263, 0x74A7 }, { 10264, 0x74BF }, { 10265, 0x7515 }, { 10266, 0x7656 }, { 10267, 0x7658 }, { 10268, 0x7652 },
    { 10269, 0x77BD }, { 10270, 0x77BF }, { 10271, 0x77BB }, { 10272, 0x77BC }, { 10273, 0x790E }, { 10274, 0x79AE }, { 10275, 0x7A61 }, { 10276, 0x7A62 },
    { 10277, 0x7A60 }, { 10278, 0x7AC4 }, { 10279, 0x7AC5 }, { 10280, 0x7C2B }, { 10281, 0x7C27 }, { 10282, 0x7C2A }, { 10283, 0x7C1E }, { 10284, 0x7C23 },
    { 10285, 0x7C21 }, { 10286, 0x7CE7 }, { 10287, 0x7E54 }, { 10288, 0x7E55 }, { 10289, 0x7E5E }, { 10290, 0x7E5A }, { 10291, 0x7E61 }, { 10292, 0x7E52 },
    { 10293, 0x7E59 }, { 10294, 0x7F48 }, { 10295, 0x7FF9 }, { 10296, 0x7FFB }, { 10297, 0x8077 }, { 10298, 0x8076 }, { 10299, 0x81CD }, { 10300, 0x81CF },
    { 10301, 0x820A }, { 10302, 0x85CF }, { 10303, 0x85A9 }, { 10304, 0x85CD }, { 10305, 0x85D0 }, { 10306, 0x85C9 }, { 10307, 0x85B0 }, { 10308, 0x85BA },
    { 10309, 0x85B9 }, { 10310, 0x85A6 }, { 10311, 0x87EF }, { 10312, 0x87EC }, { 10313, 0x87F2 }, { 10314, 0x87E0 }, { 10315, 0x8986 }, { 10316, 0x89B2 },
    { 10317, 0x89F4 }, { 10318, 0x8B28 }, { 10319, 0x8B39 }, { 10320, 0x8B2C }, { 10321, 0x8B2B }, { 10322, 0x8C50 }, { 10323, 0x8D05 }, { 10324, 0x8E59 },
    { 10325, 0x8E63 }, { 10326, 0x8E66 }, { 10327, 0x8E64 }, { 10328, 0x8E5F }, { 10329, 0x8E55 }, { 10330, 0x8EC0 }, { 10331, 0x8F49 }, { 10332, 0x8F4D },
    { 10333, 0x9087 }, { 10334, 0x9083 }, { 10335, 0x9088 }, { 10336, 0x91AB }, { 10337, 0x91AC }, { 10338, 0x91D0 }, { 10339, 0x9394 }, { 10340, 0x938A },
    { 10341, 0x9396 }, { 10342, 0x93A2 }, { 10343, 0x93B3 }, { 10344, 0x93AE }, { 10345, 0x93AC }, { 10346, 0x93B0 }, { 10347, 0x9398 }, { 10348, 0x939A },
    { 10349, 0x9397 }, { 10350, 0x95D4 }, { 10351, 0x95D6 }, { 10352, 0x95D0 }, { 10353, 0x95D5 }, { 10354, 0x96E2 }, { 10355, 0x96DC }, { 10356, 0x96D9 },
    { 10357, 0x96DB }, { 10358, 0x96DE }, { 10359, 0x9724 }, { 10360, 0x97A3 }, { 10361, 0x97A6 }, { 10362, 0x97AD }, { 10363, 0x97F9 }, { 10364, 0x984D },
    { 10365, 0x984F }, { 10366, 0x984C }, { 10367, 0x984E }, { 10368, 0x9853 }, { 10369, 0x98BA }, { 10370, 0x993E }, { 10371, 0x993F }, { 10372, 0x993D },
    { 10373, 0x992E }, { 10374, 0x99A5 }, { 10375, 0x9A0E }, { 10376, 0x9AC1 }, { 10377, 0x9B03 }, { 10378, 0x9B06 }, { 10379, 0x9B4F }, { 10380, 0x9B4E },
    { 10381, 0x9B4D }, { 10382, 0x9BCA }, { 10383, 0x9BC9 }, { 10384, 0x9BFD }, { 10385, 0x9BC8 }, { 10386, 0x9BC0 }, { 10387, 0x9D51 }, { 10388, 0x9D5D },
    { 10389, 0x9D60 }, { 10390, 0x9EE0 }, { 10391, 0x9F15 }, { 10392, 0x9F2C }, { 10393, 0x5133 }, { 10394, 0x56A5 }, { 10395, 0x58DE }, { 10396, 0x58DF },
    { 10397, 0x58E2 }, { 10398, 0x5BF5 }, { 10399, 0x9F90 }, { 10400, 0x5EEC }, { 10401, 0x61F2 }, { 10402, 0x61F7 }, { 10403, 0x61F6 }, { 10404, 0x61F5 },
    { 10405, 0x6500 }, { 10406, 0x650F }, { 10407, 0x66E0 }, { 10408, 0x66DD }, { 10409, 0x6AE5 }, { 10410, 0x6ADD }, { 10411, 0x6ADA }, { 10412, 0x6AD3 },
    { 10413, 0x701B }, { 10414, 0x701F }, { 10415, 0x7028 }, { 10416, 0x701A }, { 10417, 0x701D }, { 10418, 0x7015 }, { 10419, 0x7018 }, { 10420, 0x7206 },
    { 10421, 0x720D }, { 10422, 0x7258 }, { 10423, 0x72A2 }, { 10424, 0x7378 }, { 10425, 0x737A }, { 10426, 0x74BD }, { 10427, 0x74CA }, { 10428, 0x74E3 },
    { 10429, 0x7587 }, { 10430, 0x7586 }, { 10431, 0x765F }, { 10432, 0x7661 }, { 10433, 0x77C7 }, { 10434, 0x7919 }, { 10435, 0x79B1 }, { 10436, 0x7A6B },
    { 10437, 0x7A69 }, { 10438, 0x7C3E }, { 10439, 0x7C3F }, { 10440, 0x7C38 }, { 10441, 0x7C3D }, { 10442, 0x7C37 }, { 10443, 0x7C40 }, { 10444, 0x7E6B },
    { 10445, 0x7E6D }, { 10446, 0x7E79 }, { 10447, 0x7E69 }, { 10448, 0x7E6A }, { 10449, 0x7F85 }, { 10450, 0x7E73 }, { 10451, 0x7FB6 }, { 10452, 0x7FB9 },
    { 10453, 0x7FB8 }, { 10454, 0x81D8 }, { 10455, 0x85E9 }, { 10456, 0x85DD }, { 10457, 0x85EA }, { 10458, 0x85D5 }, { 10459, 0x85E4 }, { 10460, 0x85E5 },
    { 10461, 0x85F7 }, { 10462, 0x87FB }, { 10463, 0x8805 }, { 10464, 0x880D }, { 10465, 0x87F9 }, { 10466, 0x87FE }, { 10467, 0x8960 }, { 10468, 0x895F },
    { 10469, 0x8956 }, { 10470, 0x895E }, { 10471, 0x8B41 }, { 10472, 0x8B5C }, { 10473, 0x8B58 }, { 10474, 0x8B49 }, { 10475, 0x8B5A }, { 10476, 0x8B4E },
    { 10477, 0x8B4F }, { 10478, 0x8B46 }, { 10479, 0x8B59 }, { 10480, 0x8D08 }, { 10481, 0x8D0A }, { 10482, 0x8E7C }, { 10483, 0x8E72 }, { 10484, 0x8E87 },
    { 10485, 0x8E76 }, { 10486, 0x8E6C }, { 10487, 0x8E7A }, { 10488, 0x8E74 }, { 10489, 0x8F54 }, { 10490, 0x8F4E }, { 10491, 0x8FAD }, { 10492, 0x908A },
    { 10493, 0x908B }, { 10494, 0x91B1 }, { 10495, 0x91AE }, { 10496, 0x93E1 }, { 10497, 0x93D1 }, { 10498, 0x93DF }, { 10499, 0x93C3 }, { 10500, 0x93C8 },
    { 10501, 0x93DC }, { 10502, 0x93DD }, { 10503, 0x93D6 }, { 10504, 0x93E2 }, { 10505, 0x93CD }, { 10506, 0x93D8 }, { 10507, 0x93E4 }, { 10508, 0x93D7 },
    { 10509, 0x93E8 }, { 10510, 0x95DC }, { 10511, 0x96B4 }, { 10512, 0x96E3 }, { 10513, 0x972A }, { 10514, 0x9727 }, { 10515, 0x9761 }, { 10516, 0x97DC },
    { 10517, 0x97FB }, { 10518, 0x985E }, { 10519, 0x9858 }, { 10520, 0x985B }, { 10521, 0x98BC }, { 10522, 0x9945 }, { 10523, 0x9949 }, { 10524, 0x9A16 },
    { 10525, 0x9A19 }, { 10526, 0x9B0D }, { 10527, 0x9BE8 }, { 10528, 0x9BE7 }, { 10529, 0x9BD6 }, { 10530, 0x9BDB }, { 10531, 0x9D89 }, { 10532, 0x9D61 },
    { 10533, 0x9D72 }, { 10534, 0x9D6A }, { 10535, 0x9D6C }, { 10536, 0x9E92 }, { 10537, 0x9E97 }, { 10538, 0x9E93 }, { 10539, 0x9EB4 }, { 10540, 0x52F8 },
    { 10541, 0x56A8 }, { 10542, 0x56B7 }, { 10543, 0x56B6 }, { 10544, 0x56B4 }, { 10545, 0x56BC }, { 10546, 0x58E4 }, { 10547, 0x5B40 }, { 10548, 0x5B43 },
    { 10549, 0x5B7D }, { 10550, 0x5BF6 }, { 10551, 0x5DC9 }, { 10552, 0x61F8 }, { 10553, 0x61FA }, { 10554, 0x6518 }, { 10555, 0x6514 }, { 10556, 0x6519 },
    { 10557, 0x66E6 }, { 10558, 0x6727 }, { 10559, 0x6AEC }, { 10560, 0x703E }, { 10561, 0x7030 }, { 10562, 0x7032 }, { 10563, 0x7210 }, { 10564, 0x737B },
    { 10565, 0x74CF }, { 10566, 0x7662 }, { 10567, 0x7665 }, { 10568, 0x7926 }, { 10569, 0x792A }, { 10570, 0x792C }, { 10571, 0x792B }, { 10572, 0x7AC7 },
    { 10573, 0x7AF6 }, { 10574, 0x7C4C }, { 10575, 0x7C43 }, { 10576, 0x7C4D }, { 10577, 0x7CEF }, { 10578, 0x7CF0 }, { 10579, 0x8FAE }, { 10580, 0x7E7D },
    { 10581, 0x7E7C }, { 10582, 0x7E82 }, { 10583, 0x7F4C }, { 10584, 0x8000 }, { 10585, 0x81DA }, { 10586, 0x8266 }, { 10587, 0x85FB }, { 10588, 0x85F9 },
    { 10589, 0x8611 }, { 10590, 0x85FA }, { 10591, 0x8606 }, { 10592, 0x860B }, { 10593, 0x8607 }, { 10594, 0x860A }, { 10595, 0x8814 }, { 10596, 0x8815 },
    { 10597, 0x8964 }, { 10598, 0x89BA }, { 10599, 0x89F8 }, { 10600, 0x8B70 }, { 10601, 0x8B6C }, { 10602, 0x8B66 }, { 10603, 0x8B6F }, { 10604, 0x8B5F },
    { 10605, 0x8B6B }, { 10606, 0x8D0F }, { 10607, 0x8D0D }, { 10608, 0x8E89 }, { 10609, 0x8E81 }, { 10610, 0x8E85 }, { 10611, 0x8E82 }, { 10612, 0x91B4 },
    { 10613, 0x91CB }, { 10614, 0x9418 }, { 10615, 0x9403 }, { 10616, 0x93FD }, { 10617, 0x95E1 }, { 10618, 0x9730 }, { 10619, 0x98C4 }, { 10620, 0x9952 },
    { 10621, 0x9951 }, { 10622, 0x99A8 }, { 10623, 0x9A2B }, { 10624, 0x9A30 }, { 10625, 0x9A37 }, { 10626, 0x9A35 }, { 10627, 0x9C13 }, { 10628, 0x9C0D },
    { 10629, 0x9E79 }, { 10630, 0x9EB5 }, { 10631, 0x9EE8 }, { 10632, 0x9F2F }, { 10633, 0x9F5F }, { 10634, 0x9F63 }, { 10635, 0x9F61 }, { 10636, 0x5137 },
    { 10637, 0x5138 }, { 10638, 0x56C1 }, { 10639, 0x56C0 }, { 10640, 0x56C2 }, { 10641, 0x5914 }, { 10642, 0x5C6C }, { 10643, 0x5DCD }, { 10644, 0x61FC },
    { 10645, 0x61FE }, { 10646, 0x651D }, { 10647, 0x651C }, { 10648, 0x6595 }, { 10649, 0x66E9 }, { 10650, 0x6AFB }, { 10651, 0x6B04 }, { 10652, 0x6AFA },
    { 10653, 0x6BB2 }, { 10654, 0x704C }, { 10655, 0x721B }, { 10656, 0x72A7 }, { 10657, 0x74D6 }, { 10658, 0x74D4 }, { 10659, 0x7669 }, { 10660, 0x77D3 },
    { 10661, 0x7C50 }, { 10662, 0x7E8F }, { 10663, 0x7E8C }, { 10664, 0x7FBC }, { 10665, 0x8617 }, { 10666, 0x862D }, { 10667, 0x861A }, { 10668, 0x8823 },
    { 10669, 0x8822 }, { 10670, 0x8821 }, { 10671, 0x881F }, { 10672, 0x896A }, { 10673, 0x896C }, { 10674, 0x89BD }, { 10675, 0x8B74 }, { 10676, 0x8B77 },
    { 10677, 0x8B7D }, { 10678, 0x8D13 }, { 10679, 0x8E8A }, { 10680, 0x8E8D }, { 10681, 0x8E8B }, { 10682, 0x8F5F }, { 10683, 0x8FAF }, { 10684, 0x91BA },
    { 10685, 0x942E }, { 10686, 0x9433 }, { 10687, 0x9435 }, { 10688, 0x943A }, { 10689, 0x9438 }, { 10690, 0x9432 }, { 10691, 0x942B }, { 10692, 0x95E2 },
    { 10693, 0x9738 }, { 10694, 0x9739 }, { 10695, 0x9732 }, { 10696, 0x97FF }, { 10697, 0x9867 }, { 10698, 0x9865 }, { 10699, 0x9957 }, { 10700, 0x9A45 },
    { 10701, 0x9A43 }, { 10702, 0x9A40 }, { 10703, 0x9A3E }, { 10704, 0x9ACF }, { 10705, 0x9B54 }, { 10706, 0x9B51 }, { 10707, 0x9C2D }, { 10708, 0x9C25 },
    { 10709, 0x9DAF }, { 10710, 0x9DB4 }, { 10711, 0x9DC2 }, { 10712, 0x9DB8 }, { 10713, 0x9E9D }, { 10714, 0x9EEF }, { 10715, 0x9F19 }, { 10716, 0x9F5C },
    { 10717, 0x9F66 }, { 10718, 0x9F67 }, { 10719, 0x513C }, { 10720, 0x513B }, { 10721, 0x56C8 }, { 10722, 0x56CA }, { 10723, 0x56C9 }, { 10724, 0x5B7F },
    { 10725, 0x5DD4 }, { 10726, 0x5DD2 }, { 10727, 0x5F4E }, { 10728, 0x61FF }, { 10729, 0x6524 }, { 10730, 0x6B0A }, { 10731, 0x6B61 }, { 10732, 0x7051 },
    { 10733, 0x7058 }, { 10734, 0x7380 }, { 10735, 0x74E4 }, { 10736, 0x758A }, { 10737, 0x766E }, { 10738, 0x766C }, { 10739, 0x79B3 }, { 10740, 0x7C60 },
    { 10741, 0x7C5F }, { 10742, 0x807E }, { 10743, 0x807D }, { 10744, 0x81DF }, { 10745, 0x8972 }, { 10746, 0x896F }, { 10747, 0x89FC }, { 10748, 0x8B80 },
    { 10749, 0x8D16 }, { 10750, 0x8D17 }, { 10751, 0x8E91 }, { 10752, 0x8E93 }, { 10753, 0x8F61 }, { 10754, 0x9148 }, { 10755, 0x9444 }, { 10756, 0x9451 },
    { 10757, 0x9452 }, { 10758, 0x973D }, { 10759, 0x973E }, { 10760, 0x97C3 }, { 10761, 0x97C1 }, { 10762, 0x986B }, { 10763, 0x9955 }, { 10764, 0x9A55 },
    { 10765, 0x9A4D }, { 10766, 0x9AD2 }, { 10767, 0x9B1A }, { 10768, 0x9C49 }, { 10769, 0x9C31 }, { 10770, 0x9C3E }, { 10771, 0x9C3B }, { 10772, 0x9DD3 },
    { 10773, 0x9DD7 }, { 10774, 0x9F34 }, { 10775, 0x9F6C }, { 10776, 0x9F6A }, { 10777, 0x9F94 }, { 10778, 0x56CC }, { 10779, 0x5DD6 }, { 10780, 0x6200 },
    { 10781, 0x6523 }, { 10782, 0x652B }, { 10783, 0x652A }, { 10784, 0x66EC }, { 10785, 0x6B10 }, { 10786, 0x74DA }, { 10787, 0x7ACA }, { 10788, 0x7C64 },
    { 10789, 0x7C63 }, { 10790, 0x7C65 }, { 10791, 0x7E93 }, { 10792, 0x7E96 }, { 10793, 0x7E94 }, { 10794, 0x81E2 }, { 10795, 0x8638 }, { 10796, 0x863F },
    { 10797, 0x8831 }, { 10798, 0x8B8A }, { 10799, 0x9090 }, { 10800, 0x908F }, { 10801, 0x9463 }, { 10802, 0x9460 }, { 10803, 0x9464 }, { 10804, 0x9768 },
    { 10805, 0x986F }, { 10806, 0x995C }, { 10807, 0x9A5A }, { 10808, 0x9A5B }, { 10809, 0x9A57 }, { 10810, 0x9AD3 }, { 10811, 0x9AD4 }, { 10812, 0x9AD1 },
    { 10813, 0x9C54 }, { 10814, 0x9C57 }, { 10815, 0x9C56 }, { 10816, 0x9DE5 }, { 10817, 0x9E9F }, { 10818, 0x9EF4 }, { 10819, 0x56D1 }, { 10820, 0x58E9 },
    { 10821, 0x652C }, { 10822, 0x705E }, { 10823, 0x7671 }, { 10824, 0x7672 }, { 10825, 0x77D7 }, { 10826, 0x7F50 }, { 10827, 0x7F88 }, { 10828, 0x8836 },
    { 10829, 0x8839 }, { 10830, 0x8862 }, { 10831, 0x8B93 }, { 10832, 0x8B92 }, { 10833, 0x8B96 }, { 10834, 0x8277 }, { 10835, 0x8D1B }, { 10836, 0x91C0 },
    { 10837, 0x946A }, { 10838, 0x9742 }, { 10839, 0x9748 }, { 10840, 0x9744 }, { 10841, 0x97C6 }, { 10842, 0x9870 }, { 10843, 0x9A5F }, { 10844, 0x9B22 },
    { 10845, 0x9B58 }, { 10846, 0x9C5F }, { 10847, 0x9DF9 }, { 10848, 0x9DFA }, { 10849, 0x9E7C }, { 10850, 0x9E7D }, { 10851, 0x9F07 }, { 10852, 0x9F77 },
    { 10853, 0x9F72 }, { 10854, 0x5EF3 }, { 10855, 0x6B16 }, { 10856, 0x7063 }, { 10857, 0x7C6C }, { 10858, 0x7C6E }, { 10859, 0x883B }, { 10860, 0x89C0 },
    { 10861, 0x8EA1 }, { 10862, 0x91C1 }, { 10863, 0x9472 }, { 10864, 0x9470 }, { 10865, 0x9871 }, { 10866, 0x995E }, { 10867, 0x9AD6 }, { 10868, 0x9B23 },
    { 10869, 0x9ECC }, { 10870, 0x7064 }, { 10871, 0x77DA }, { 10872, 0x8B9A }, { 10873, 0x9477 }, { 10874, 0x97C9 }, { 10875, 0x9A62 }, { 10876, 0x9A65 },
    { 10877, 0x7E9C }, { 10878, 0x8B9C }, { 10879, 0x8EAA }, { 10880, 0x91C5 }, { 10881, 0x947D }, { 10882, 0x947E }, { 10883, 0x947C }, { 10884, 0x9C77 },
    { 10885, 0x9C78 }, { 10886, 0x9EF7 }, { 10887, 0x8C54 }, { 10888, 0x947F }, { 10889, 0x9E1A }, { 10890, 0x7228 }, { 10891, 0x9A6A }, { 10892, 0x9B31 },
    { 10893, 0x9E1B }, { 10894, 0x9E1E }, { 10895, 0x7C72 }, { 10896, 0x2460 }, { 10897, 0x2461 }, { 10898, 0x2462 }, { 10899, 0x2463 }, { 10900, 0x2464 },
    { 10901, 0x2465 }, { 10902, 0x2466 }, { 10903, 0x2467 }, { 10904, 0x2468 }, { 10905, 0x2469 }, { 10906, 0x2474 }, { 10907, 0x2475 }, { 10908, 0x2476 },
    { 10909, 0x2477 }, { 10910, 0x2478 }, { 10911, 0x2479 }, { 10912, 0x247A }, { 10913, 0x247B }, { 10914, 0x247C }, { 10915, 0x247D }, { 10916, 0x2170 },
    { 10917, 0x2171 }, { 10918, 0x2172 }, { 10919, 0x2173 }, { 10920, 0x2174 }, { 10921, 0x2175 }, { 10922, 0x2176 }, { 10923, 0x2177 }, { 10924, 0x2178 },
    { 10925, 0x2179 }, { 10926, 0x4E36 }, { 10927, 0x4E3F }, { 10928, 0x4E85 }, { 10929, 0x4EA0 }, { 10930, 0x5182 }, { 10931, 0x5196 }, { 10932, 0x51AB },
    { 10933, 0x52F9 }, { 10934, 0x5338 }, { 10935, 0x5369 }, { 10936, 0x53B6 }, { 10937, 0x590A }, { 10938, 0x5B80 }, { 10939, 0x5DDB }, { 10940, 0x2F33 },
    { 10941, 0x5E7F }, { 10942, 0x5EF4 }, { 10943, 0x5F50 }, { 10944, 0x5F61 }, { 10945, 0x6534 }, { 10946, 0x65E0 }, { 10947, 0x7592 }, { 10948, 0x7676 },
    { 10949, 0x8FB5 }, { 10950, 0x96B6 }, { 10951, 0x00A8 }, { 10952, 0x02C6 }, { 10953, 0x30FD }, { 10954, 0x30FE }, { 10955, 0x309D }, { 10956, 0x309E },
    { 10957, 0x3003 }, { 10958, 0x4EDD }, { 10959, 0x3005 }, { 10960, 0x3006 }, { 10961, 0x3007 }, { 10962, 0x30FC }, { 10963, 0xFF3B }, { 10964, 0xFF3D },
    { 10965, 0x273D }, { 10966, 0x3041 }, { 10967, 0x3042 }, { 10968, 0x3043 }, { 10969, 0x3044 }, { 10970, 0x3045 }, { 10971, 0x3046 }, { 10972, 0x3047 },
    { 10973, 0x3048 }, { 10974, 0x3049 }, { 10975, 0x304A }, { 10976, 0x304B }, { 10977, 0x304C }, { 10978, 0x304D }, { 10979, 0x304E }, { 10980, 0x304F },
    { 10981, 0x3050 }, { 10982, 0x3051 }, { 10983, 0x3052 }, { 10984, 0x3053 }, { 10985, 0x3054 }, { 10986, 0x3055 }, { 10987, 0x3056 }, { 10988, 0x3057 },
    { 10989, 0x3058 }, { 10990, 0x3059 }, { 10991, 0x305A }, { 10992, 0x305B }, { 10993, 0x305C }, { 10994, 0x305D }, { 10995, 0x305E }, { 10996, 0x305F },
    { 10997, 0x3060 }, { 10998, 0x3061 }, { 10999, 0x3062 }, { 11000, 0x3063 }, { 11001, 0x3064 }, { 11002, 0x3065 }, { 11003, 0x3066 }, { 11004, 0x3067 },
    { 11005, 0x3068 }, { 11006, 0x3069 }, { 11007, 0x306A }, { 11008, 0x306B }, { 11009, 0x306C }, { 11010, 0x306D }, { 11011, 0x306E }, { 11012, 0x306F },
    { 11013, 0x3070 }, { 11014, 0x3071 }, { 11015, 0x3072 }, { 11016, 0x3073 }, { 11017, 0x3074 }, { 11018, 0x3075 }, { 11019, 0x3076 }, { 11020, 0x3077 },
    { 11021, 0x3078 }, { 11022, 0x3079 }, { 11023, 0x307A }, { 11024, 0x307B }, { 11025, 0x307C }, { 11026, 0x307D }, { 11027, 0x307E }, { 11028, 0x307F },
    { 11029, 0x3080 }, { 11030, 0x3081 }, { 11031, 0x3082 }, { 11032, 0x3083 }, { 11033, 0x3084 }, { 11034, 0x3085 }, { 11035, 0x3086 }, { 11036, 0x3087 },
    { 11037, 0x3088 }, { 11038, 0x3089 }, { 11039, 0x308A }, { 11040, 0x308B }, { 11041, 0x308C }, { 11042, 0x308D }, { 11043, 0x308E }, { 11044, 0x308F },
    { 11045, 0x3090 }, { 11046, 0x3091 }, { 11047, 0x3092 }, { 11048, 0x3093 }, { 11049, 0x30A1 }, { 11050, 0x30A2 }, { 11051, 0x30A3 }, { 11052, 0x30A4 },
    { 11053, 0x30A5 }, { 11054, 0x30A6 }, { 11055, 0x30A7 }, { 11056, 0x30A8 }, { 11057, 0x30A9 }, { 11058, 0x30AA }, { 11059, 0x30AB }, { 11060, 0x30AC },
    { 11061, 0x30AD }, { 11062, 0x30AE }, { 11063, 0x30AF }, { 11064, 0x30B0 }, { 11065, 0x30B1 }, { 11066, 0x30B2 }, { 11067, 0x30B3 }, { 11068, 0x30B4 },
    { 11069, 0x30B5 }, { 11070, 0x30B6 }, { 11071, 0x30B7 }, { 11072, 0x30B8 }, { 11073, 0x30B9 }, { 11074, 0x30BA }, { 11075, 0x30BB }, { 11076, 0x30BC },
    { 11077, 0x30BD }, { 11078, 0x30BE }, { 11079, 0x30BF }, { 11080, 0x30C0 }, { 11081, 0x30C1 }, { 11082, 0x30C2 }, { 11083, 0x30C3 }, { 11084, 0x30C4 },
    { 11085, 0x30C5 }, { 11086, 0x30C6 }, { 11087, 0x30C7 }, { 11088, 0x30C8 }, { 11089, 0x30C9 }, { 11090, 0x30CA }, { 11091, 0x30CB }, { 11092, 0x30CC },
    { 11093, 0x30CD }, { 11094, 0x30CE }, { 11095, 0x30CF }, { 11096, 0x30D0 }, { 11097, 0x30D1 }, { 11098, 0x30D2 }, { 11099, 0x30D3 }, { 11100, 0x30D4 },
    { 11101, 0x30D5 }, { 11102, 0x30D6 }, { 11103, 0x30D7 }, { 11104, 0x30D8 }, { 11105, 0x30D9 }, { 11106, 0x30DA }, { 11107, 0x30DB }, { 11108, 0x30DC },
    { 11109, 0x30DD }, { 11110, 0x30DE }, { 11111, 0x30DF }, { 11112, 0x30E0 }, { 11113, 0x30E1 }, { 11114, 0x30E2 }, { 11115, 0x30E3 }, { 11116, 0x30E4 },
    { 11117, 0x30E5 }, { 11118, 0x30E6 }, { 11119, 0x30E7 }, { 11120, 0x30E8 }, { 11121, 0x30E9 }, { 11122, 0x30EA }, { 11123, 0x30EB }, { 11124, 0x30EC },
    { 11125, 0x30ED }, { 11126, 0x30EE }, { 11127, 0x30EF }, { 11128, 0x30F0 }, { 11129, 0x30F1 }, { 11130, 0x30F2 }, { 11131, 0x30F3 }, { 11132, 0x30F4 },
    { 11133, 0x30F5 }, { 11134, 0x30F6 }, { 11135, 0x0410 }, { 11136, 0x0411 }, { 11137, 0x0412 }, { 11138, 0x0413 }, { 11139, 0x0414 }, { 11140, 0x0415 },
    { 11141, 0x0401 }, { 11142, 0x0416 }, { 11143, 0x0417 }, { 11144, 0x0418 }, { 11145, 0x0419 }, { 11146, 0x041A }, { 11147, 0x041B }, { 11148, 0x041C },
    { 11149, 0x041D }, { 11150, 0x041E }, { 11151, 0x041F }, { 11152, 0x0420 }, { 11153, 0x0421 }, { 11154, 0x0422 }, { 11155, 0x0423 }, { 11156, 0x0424 },
    { 11157, 0x0425 }, { 11158, 0x0426 }, { 11159, 0x0427 }, { 11160, 0x0428 }, { 11161, 0x0429 }, { 11162, 0x042A }, { 11163, 0x042B }, { 11164, 0x042C },
    { 11165, 0x042D }, { 11166, 0x042E }, { 11167, 0x042F }, { 11168, 0x0430 }, { 11169, 0x0431 }, { 11170, 0x0432 }, { 11171, 0x0433 }, { 11172, 0x0434 },
    { 11173, 0x0435 }, { 11174, 0x0451 }, { 11175, 0x0436 }, { 11176, 0x0437 }, { 11177, 0x0438 }, { 11178, 0x0439 }, { 11179, 0x043A }, { 11180, 0x043B },
    { 11181, 0x043C }, { 11182, 0x043D }, { 11183, 0x043E }, { 11184, 0x043F }, { 11185, 0x0440 }, { 11186, 0x0441 }, { 11187, 0x0442 }, { 11188, 0x0443 },
    { 11189, 0x0444 }, { 11190, 0x0445 }, { 11191, 0x0446 }, { 11192, 0x0447 }, { 11193, 0x0448 }, { 11194, 0x0449 }, { 11195, 0x044A }, { 11196, 0x044B },
    { 11197, 0x044C }, { 11198, 0x044D }, { 11199, 0x044E }, { 11200, 0x044F }, { 11201, 0x21E7 }, { 11202, 0x21B8 }, { 11203, 0x21B9 }, { 11204, 0x31CF },
    { 11205, 0x200CC }, { 11206, 0x4E5A }, { 11207, 0x2008A }, { 11208, 0x5202 }, { 11209, 0x4491 }, { 11210, 0x9FB0 }, { 11211, 0x5188 }, { 11212, 0x9FB1 },
    { 11213, 0x27607 }, { 11254, 0xFFE2 }, { 11255, 0xFFE4 }, { 11256, 0xFF07 }, { 11257, 0xFF02 }, { 11258, 0x3231 }, { 11259, 0x2116 }, { 11260, 0x2121 },
    { 11261, 0x309B }, { 11262, 0x309C }, { 11263, 0x2E80 }, { 11264, 0x2E84 }, { 11265, 0x2E86 }, { 11266, 0x2E87 }, { 11267, 0x2E88 }, { 11268, 0x2E8A },
    { 11269, 0x2E8C }, { 11270, 0x2E8D }, { 11271, 0x2E95 }, { 11272, 0x2E9C }, { 11273, 0x2E9D }, { 11274, 0x2EA5 }, { 11275, 0x2EA7 }, { 11276, 0x2EAA },
    { 11277, 0x2EAC }, { 11278, 0x2EAE }, { 11279, 0x2EB6 }, { 11280, 0x2EBC }, { 11281, 0x2EBE }, { 11282, 0x2EC6 }, { 11283, 0x2ECA }, { 11284, 0x2ECC },
    { 11285, 0x2ECD }, { 11286, 0x2ECF }, { 11287, 0x2ED6 }, { 11288, 0x2ED7 }, { 11289, 0x2EDE }, { 11290, 0x2EE3 }, { 11294, 0x0283 }, { 11295, 0x0250 },
    { 11296, 0x025B }, { 11297, 0x0254 }, { 11298, 0x0275 }, { 11299, 0x0153 }, { 11300, 0x00F8 }, { 11301, 0x014B }, { 11302, 0x028A }, { 11303, 0x026A },
    { 11304, 0x4E42 }, { 11305, 0x4E5C }, { 11306, 0x51F5 }, { 11307, 0x531A }, { 11308, 0x5382 }, { 11309, 0x4E07 }, { 11310, 0x4E0C }, { 11311, 0x4E47 },
    { 11312, 0x4E8D }, { 11313, 0x56D7 }, { 11314, 0xFA0C }, { 11315, 0x5C6E }, { 11316, 0x5F73 }, { 11317, 0x4E0F }, { 11318, 0x5187 }, { 11319, 0x4E0E },
    { 11320, 0x4E2E }, { 11321, 0x4E93 }, { 11322, 0x4EC2 }, { 11323, 0x4EC9 }, { 11324, 0x4EC8 }, { 11325, 0x5198 }, { 11326, 0x52FC }, { 11327, 0x536C },
    { 11328, 0x53B9 }, { 11329, 0x5720 }, { 11330, 0x5903 }, { 11331, 0x592C }, { 11332, 0x5C10 }, { 11333, 0x5DFF }, { 11334, 0x65E1 }, { 11335, 0x6BB3 },
    { 11336, 0x6BCC }, { 11337, 0x6C14 }, { 11338, 0x723F }, { 11339, 0x4E31 }, { 11340, 0x4E3C }, { 11341, 0x4EE8 }, { 11342, 0x4EDC }, { 11343, 0x4EE9 },
    { 11344, 0x4EE1 }, { 11345, 0x4EDD }, { 11346, 0x4EDA }, { 11347, 0x520C }, { 11348, 0x531C }, { 11349, 0x534C }, { 11350, 0x5722 }, { 11351, 0x5723 },
    { 11352, 0x5917 }, { 11353, 0x592F }, { 11354, 0x5B81 }, { 11355, 0x5B84 }, { 11356, 0x5C12 }, { 11357, 0x5C3B }, { 11358, 0x5C74 }, { 11359, 0x5C73 },
    { 11360, 0x5E04 }, { 11361, 0x5E80 }, { 11362, 0x5E82 }, { 11363, 0x5FC9 }, { 11364, 0x6209 }, { 11365, 0x6250 }, { 11366, 0x6C15 }, { 11367, 0x6C36 },
    { 11368, 0x6C43 }, { 11369, 0x6C3F }, { 11370, 0x6C3B }, { 11371, 0x72AE }, { 11372, 0x72B0 }, { 11373, 0x738A }, { 11374, 0x79B8 }, { 11375, 0x808A },
    { 11376, 0x961E }, { 11377, 0x4F0E }, { 11378, 0x4F18 }, { 11379, 0x4F2C }, { 11380, 0x4EF5 }, { 11381, 0x4F14 }, { 11382, 0x4EF1 }, { 11383, 0x4F00 },
    { 11384, 0x4EF7 }, { 11385, 0x4F08 }, { 11386, 0x4F1D }, { 11387, 0x4F02 }, { 11388, 0x4F05 }, { 11389, 0x4F22 }, { 11390, 0x4F13 }, { 11391, 0x4F04 },
    { 11392, 0x4EF4 }, { 11393, 0x4F12 }, { 11394, 0x51B1 }, { 11395, 0x5213 }, { 11396, 0x5209 }, { 11397, 0x5210 }, { 11398, 0x52A6 }, { 11399, 0x5322 },
    { 11400, 0x531F }, { 11401, 0x534D }, { 11402, 0x538A }, { 11403, 0x5407 }, { 11404, 0x56E1 }, { 11405, 0x56DF }, { 11406, 0x572E }, { 11407, 0x572A },
    { 11408, 0x5734 }, { 11409, 0x593C }, { 11410, 0x5980 }, { 11411, 0x597C }, { 11412, 0x5985 }, { 11413, 0x597B }, { 11414, 0x597E }, { 11415, 0x5977 },
    { 11416, 0x597F }, { 11417, 0x5B56 }, { 11418, 0x5C15 }, { 11419, 0x5C25 }, { 11420, 0x5C7C }, { 11421, 0x5C7A }, { 11422, 0x5C7B }, { 11423, 0x5C7E },
    { 11424, 0x5DDF }, { 11425, 0x5E75 }, { 11426, 0x5E84 }, { 11427, 0x5F02 }, { 11428, 0x5F1A }, { 11429, 0x5F74 }, { 11430, 0x5FD5 }, { 11431, 0x5FD4 },
    { 11432, 0x5FCF }, { 11433, 0x625C }, { 11434, 0x625E }, { 11435, 0x6264 }, { 11436, 0x6261 }, { 11437, 0x6266 }, { 11438, 0x6262 }, { 11439, 0x6259 },
    { 11440, 0x6260 }, { 11441, 0x625A }, { 11442, 0x6265 }, { 11443, 0x65EF }, { 11444, 0x65EE }, { 11445, 0x673E }, { 11446, 0x6739 }, { 11447, 0x6738 },
    { 11448, 0x673B }, { 11449, 0x673A }, { 11450, 0x673F }, { 11451, 0x673C }, { 11452, 0x6733 }, { 11453, 0x6C18 }, { 11454, 0x6C46 }, { 11455, 0x6C52 },
    { 11456, 0x6C5C }, { 11457, 0x6C4F }, { 11458, 0x6C4A }, { 11459, 0x6C54 }, { 11460, 0x6C4B }, { 11461, 0x6C4C }, { 11462, 0x7071 }, { 11463, 0x725E },
    { 11464, 0x72B4 }, { 11465, 0x72B5 }, { 11466, 0x738E }, { 11467, 0x752A }, { 11468, 0x767F }, { 11469, 0x7A75 }, { 11470, 0x7F51 }, { 11471, 0x8278 },
    { 11472, 0x827C }, { 11473, 0x8280 }, { 11474, 0x827D }, { 11475, 0x827F }, { 11476, 0x864D }, { 11477, 0x897E }, { 11478, 0x9099 }, { 11479, 0x9097 },
    { 11480, 0x9098 }, { 11481, 0x909B }, { 11482, 0x9094 }, { 11483, 0x9622 }, { 11484, 0x9624 }, { 11485, 0x9620 }, { 11486, 0x9623 }, { 11487, 0x4F56 },
    { 11488, 0x4F3B }, { 11489, 0x4F62 }, { 11490, 0x4F49 }, { 11491, 0x4F53 }, { 11492, 0x4F64 }, { 11493, 0x4F3E }, { 11494, 0x4F67 }, { 11495, 0x4F52 },
    { 11496, 0x4F5F }, { 11497, 0x4F41 }, { 11498, 0x4F58 }, { 11499, 0x4F2D }, { 11500, 0x4F33 }, { 11501, 0x4F3F }, { 11502, 0x4F61 }, { 11503, 0x518F },
    { 11504, 0x51B9 }, { 11505, 0x521C }, { 11506, 0x521E }, { 11507, 0x5221 }, { 11508, 0x52AD }, { 11509, 0x52AE }, { 11510, 0x5309 }, { 11511, 0x5363 },
    { 11512, 0x5372 }, { 11513, 0x538E }, { 11514, 0x538F }, { 11515, 0x5430 }, { 11516, 0x5437 }, { 11517, 0x542A }, { 11518, 0x5454 }, { 11519, 0x5445 },
    { 11520, 0x5419 }, { 11521, 0x541C }, { 11522, 0x5425 }, { 11523, 0x5418 }, { 11524, 0x543D }, { 11525, 0x544F }, { 11526, 0x5441 }, { 11527, 0x5428 },
    { 11528, 0x5424 }, { 11529, 0x5447 }, { 11530, 0x56EE }, { 11531, 0x56E7 }, { 11532, 0x56E5 }, { 11533, 0x5741 }, { 11534, 0x5745 }, { 11535, 0x574C },
    { 11536, 0x5749 }, { 11537, 0x574B }, { 11538, 0x5752 }, { 11539, 0x5906 }, { 11540, 0x5940 }, { 11541, 0x59A6 }, { 11542, 0x5998 }, { 11543, 0x59A0 },
    { 11544, 0x5997 }, { 11545, 0x598E }, { 11546, 0x59A2 }, { 11547, 0x5990 }, { 11548, 0x598F }, { 11549, 0x59A7 }, { 11550, 0x59A1 }, { 11551, 0x5B8E },
    { 11552, 0x5B92 }, { 11553, 0x5C28 }, { 11554, 0x5C2A }, { 11555, 0x5C8D }, { 11556, 0x5C8F }, { 11557, 0x5C88 }, { 11558, 0x5C8B }, { 11559, 0x5C89 },
    { 11560, 0x5C92 }, { 11561, 0x5C8A }, { 11562, 0x5C86 }, { 11563, 0x5C93 }, { 11564, 0x5C95 }, { 11565, 0x5DE0 }, { 11566, 0x5E0A }, { 11567, 0x5E0E },
    { 11568, 0x5E8B }, { 11569, 0x5E89 }, { 11570, 0x5E8C }, { 11571, 0x5E88 }, { 11572, 0x5E8D }, { 11573, 0x5F05 }, { 11574, 0x5F1D }, { 11575, 0x5F78 },
    { 11576, 0x5F76 }, { 11577, 0x5FD2 }, { 11578, 0x5FD1 }, { 11579, 0x5FD0 }, { 11580, 0x5FED }, { 11581, 0x5FE8 }, { 11582, 0x5FEE }, { 11583, 0x5FF3 },
    { 11584, 0x5FE1 }, { 11585, 0x5FE4 }, { 11586, 0x5FE3 }, { 11587, 0x5FFA }, { 11588, 0x5FEF }, { 11589, 0x5FF7 }, { 11590, 0x5FFB }, { 11591, 0x6000 },
    { 11592, 0x5FF4 }, { 11593, 0x623A }, { 11594, 0x6283 }, { 11595, 0x628C }, { 11596, 0x628E }, { 11597, 0x628F }, { 11598, 0x6294 }, { 11599, 0x6287 },
    { 11600, 0x6271 }, { 11601, 0x627B }, { 11602, 0x627A }, { 11603, 0x6270 }, { 11604, 0x6281 }, { 11605, 0x6288 }, { 11606, 0x6277 }, { 11607, 0x627D },
    { 11608, 0x6272 }, { 11609, 0x6274 }, { 11610, 0x6537 }, { 11611, 0x65F0 }, { 11612, 0x65F4 }, { 11613, 0x65F3 }, { 11614, 0x65F2 }, { 11615, 0x65F5 },
    { 11616, 0x6745 }, { 11617, 0x6747 }, { 11618, 0x6759 }, { 11619, 0x6755 }, { 11620, 0x674C }, { 11621, 0x6748 }, { 11622, 0x675D }, { 11623, 0x674D },
    { 11624, 0x675A }, { 11625, 0x674B }, { 11626, 0x6BD0 }, { 11627, 0x6C19 }, { 11628, 0x6C1A }, { 11629, 0x6C78 }, { 11630, 0x6C67 }, { 11631, 0x6C6B },
    { 11632, 0x6C84 }, { 11633, 0x6C8B }, { 11634, 0x6C8F }, { 11635, 0x6C71 }, { 11636, 0x6C6F }, { 11637, 0x6C69 }, { 11638, 0x6C9A }, { 11639, 0x6C6D },
    { 11640, 0x6C87 }, { 11641, 0x6C95 }, { 11642, 0x6C9C }, { 11643, 0x6C66 }, { 11644, 0x6C73 }, { 11645, 0x6C65 }, { 11646, 0x6C7B }, { 11647, 0x6C8E },
    { 11648, 0x7074 }, { 11649, 0x707A }, { 11650, 0x7263 }, { 11651, 0x72BF }, { 11652, 0x72BD }, { 11653, 0x72C3 }, { 11654, 0x72C6 }, { 11655, 0x72C1 },
    { 11656, 0x72BA }, { 11657, 0x72C5 }, { 11658, 0x7395 }, { 11659, 0x7397 }, { 11660, 0x7393 }, { 11661, 0x7394 }, { 11662, 0x7392 }, { 11663, 0x753A },
    { 11664, 0x7539 }, { 11665, 0x7594 }, { 11666, 0x7595 }, { 11667, 0x7681 }, { 11668, 0x793D }, { 11669, 0x8034 }, { 11670, 0x8095 }, { 11671, 0x8099 },
    { 11672, 0x8090 }, { 11673, 0x8092 }, { 11674, 0x809C }, { 11675, 0x8290 }, { 11676, 0x828F }, { 11677, 0x8285 }, { 11678, 0x828E }, { 11679, 0x8291 },
    { 11680, 0x8293 }, { 11681, 0x828A }, { 11682, 0x8283 }, { 11683, 0x8284 }, { 11684, 0x8C78 }, { 11685, 0x8FC9 }, { 11686, 0x8FBF }, { 11687, 0x909F },
    { 11688, 0x90A1 }, { 11689, 0x90A5 }, { 11690, 0x909E }, { 11691, 0x90A7 }, { 11692, 0x90A0 }, { 11693, 0x9630 }, { 11694, 0x9628 }, { 11695, 0x962F },
    { 11696, 0x962D }, { 11697, 0x4E33 }, { 11698, 0x4F98 }, { 11699, 0x4F7C }, { 11700, 0x4F85 }, { 11701, 0x4F7D }, { 11702, 0x4F80 }, { 11703, 0x4F87 },
    { 11704, 0x4F76 }, { 11705, 0x4F74 }, { 11706, 0x4F89 }, { 11707, 0x4F84 }, { 11708, 0x4F77 }, { 11709, 0x4F4C }, { 11710, 0x4F97 }, { 11711, 0x4F6A },
    { 11712, 0x4F9A }, { 11713, 0x4F79 }, { 11714, 0x4F81 }, { 11715, 0x4F78 }, { 11716, 0x4F90 }, { 11717, 0x4F9C }, { 11718, 0x4F94 }, { 11719, 0x4F9E },
    { 11720, 0x4F92 }, { 11721, 0x4F82 }, { 11722, 0x4F95 }, { 11723, 0x4F6B }, { 11724, 0x4F6E }, { 11725, 0x519E }, { 11726, 0x51BC }, { 11727, 0x51BE },
    { 11728, 0x5235 }, { 11729, 0x5232 }, { 11730, 0x5233 }, { 11731, 0x5246 }, { 11732, 0x5231 }, { 11733, 0x52BC }, { 11734, 0x530A }, { 11735, 0x530B },
    { 11736, 0x533C }, { 11737, 0x5392 }, { 11738, 0x5394 }, { 11739, 0x5487 }, { 11740, 0x547F }, { 11741, 0x5481 }, { 11742, 0x5491 }, { 11743, 0x5482 },
    { 11744, 0x5488 }, { 11745, 0x546B }, { 11746, 0x547A }, { 11747, 0x547E }, { 11748, 0x5465 }, { 11749, 0x546C }, { 11750, 0x5474 }, { 11751, 0x5466 },
    { 11752, 0x548D }, { 11753, 0x546F }, { 11754, 0x5461 }, { 11755, 0x5460 }, { 11756, 0x5498 }, { 11757, 0x5463 }, { 11758, 0x5467 }, { 11759, 0x5464 },
    { 11760, 0x56F7 }, { 11761, 0x56F9 }, { 11762, 0x576F }, { 11763, 0x5772 }, { 11764, 0x576D }, { 11765, 0x576B }, { 11766, 0x5771 }, { 11767, 0x5770 },
    { 11768, 0x5776 }, { 11769, 0x5780 }, { 11770, 0x5775 }, { 11771, 0x577B }, { 11772, 0x5773 }, { 11773, 0x5774 }, { 11774, 0x5762 }, { 11775, 0x5768 },
    { 11776, 0x577D }, { 11777, 0x590C }, { 11778, 0x5945 }, { 11779, 0x59B5 }, { 11780, 0x59BA }, { 11781, 0x59CF }, { 11782, 0x59CE }, { 11783, 0x59B2 },
    { 11784, 0x59CC }, { 11785, 0x59C1 }, { 11786, 0x59B6 }, { 11787, 0x59BC }, { 11788, 0x59C3 }, { 11789, 0x59D6 }, { 11790, 0x59B1 }, { 11791, 0x59BD },
    { 11792, 0x59C0 }, { 11793, 0x59C8 }, { 11794, 0x59B4 }, { 11795, 0x59C7 }, { 11796, 0x5B62 }, { 11797, 0x5B65 }, { 11798, 0x5B93 }, { 11799, 0x5B95 },
    { 11800, 0x5C44 }, { 11801, 0x5C47 }, { 11802, 0x5CAE }, { 11803, 0x5CA4 }, { 11804, 0x5CA0 }, { 11805, 0x5CB5 }, { 11806, 0x5CAF }, { 11807, 0x5CA8 },
    { 11808, 0x5CAC }, { 11809, 0x5C9F }, { 11810, 0x5CA3 }, { 11811, 0x5CAD }, { 11812, 0x5CA2 }, { 11813, 0x5CAA }, { 11814, 0x5CA7 }, { 11815, 0x5C9D },
    { 11816, 0x5CA5 }, { 11817, 0x5CB6 }, { 11818, 0x5CB0 }, { 11819, 0x5CA6 }, { 11820, 0x5E17 }, { 11821, 0x5E14 }, { 11822, 0x5E19 }, { 11823, 0x5F28 },
    { 11824, 0x5F22 }, { 11825, 0x5F23 }, { 11826, 0x5F24 }, { 11827, 0x5F54 }, { 11828, 0x5F82 }, { 11829, 0x5F7E }, { 11830, 0x5F7D }, { 11831, 0x5FDE },
    { 11832, 0x5FE5 }, { 11833, 0x602D }, { 11834, 0x6026 }, { 11835, 0x6019 }, { 11836, 0x6032 }, { 11837, 0x600B }, { 11838, 0x6034 }, { 11839, 0x600A },
    { 11840, 0x6017 }, { 11841, 0x6033 }, { 11842, 0x601A }, { 11843, 0x601E }, { 11844, 0x602C }, { 11845, 0x6022 }, { 11846, 0x600D }, { 11847, 0x6010 },
    { 11848, 0x602E }, { 11849, 0x6013 }, { 11850, 0x6011 }, { 11851, 0x600C }, { 11852, 0x6009 }, { 11853, 0x601C }, { 11854, 0x6214 }, { 11855, 0x623D },
    { 11856, 0x62AD }, { 11857, 0x62B4 }, { 11858, 0x62D1 }, { 11859, 0x62BE }, { 11860, 0x62AA }, { 11861, 0x62B6 }, { 11862, 0x62CA }, { 11863, 0x62AE },
    { 11864, 0x62B3 }, { 11865, 0x62AF }, { 11866, 0x62BB }, { 11867, 0x62A9 }, { 11868, 0x62B0 }, { 11869, 0x62B8 }, { 11870, 0x653D }, { 11871, 0x65A8 },
    { 11872, 0x65BB }, { 11873, 0x6609 }, { 11874, 0x65FC }, { 11875, 0x6604 }, { 11876, 0x6612 }, { 11877, 0x6608 }, { 11878, 0x65FB }, { 11879, 0x6603 },
    { 11880, 0x660B }, { 11881, 0x660D }, { 11882, 0x6605 }, { 11883, 0x65FD }, { 11884, 0x6611 }, { 11885, 0x6610 }, { 11886, 0x66F6 }, { 11887, 0x670A },
    { 11888, 0x6785 }, { 11889, 0x676C }, { 11890, 0x678E }, { 11891, 0x6792 }, { 11892, 0x6776 }, { 11893, 0x677B }, { 11894, 0x6798 }, { 11895, 0x6786 },
    { 11896, 0x6784 }, { 11897, 0x6774 }, { 11898, 0x678D }, { 11899, 0x678C }, { 11900, 0x677A }, { 11901, 0x679F }, { 11902, 0x6791 }, { 11903, 0x6799 },
    { 11904, 0x6783 }, { 11905, 0x677D }, { 11906, 0x6781 }, { 11907, 0x6778 }, { 11908, 0x6779 }, { 11909, 0x6794 }, { 11910, 0x6B25 }, { 11911, 0x6B80 },
    { 11912, 0x6B7E }, { 11913, 0x6BDE }, { 11914, 0x6C1D }, { 11915, 0x6C93 }, { 11916, 0x6CEC }, { 11917, 0x6CEB }, { 11918, 0x6CEE }, { 11919, 0x6CD9 },
    { 11920, 0x6CB6 }, { 11921, 0x6CD4 }, { 11922, 0x6CAD }, { 11923, 0x6CE7 }, { 11924, 0x6CB7 }, { 11925, 0x6CD0 }, { 11926, 0x6CC2 }, { 11927, 0x6CBA },
    { 11928, 0x6CC3 }, { 11929, 0x6CC6 }, { 11930, 0x6CED }, { 11931, 0x6CF2 }, { 11932, 0x6CD2 }, { 11933, 0x6CDD }, { 11934, 0x6CB4 }, { 11935, 0x6C8A },
    { 11936, 0x6C9D }, { 11937, 0x6C80 }, { 11938, 0x6CDE }, { 11939, 0x6CC0 }, { 11940, 0x6D30 }, { 11941, 0x6CCD }, { 11942, 0x6CC7 }, { 11943, 0x6CB0 },
    { 11944, 0x6CF9 }, { 11945, 0x6CCF }, { 11946, 0x6CE9 }, { 11947, 0x6CD1 }, { 11948, 0x7094 }, { 11949, 0x7098 }, { 11950, 0x7085 }, { 11951, 0x7093 },
    { 11952, 0x7086 }, { 11953, 0x7084 }, { 11954, 0x7091 }, { 11955, 0x7096 }, { 11956, 0x7082 }, { 11957, 0x709A }, { 11958, 0x7083 }, { 11959, 0x726A },
    { 11960, 0x72D6 }, { 11961, 0x72CB }, { 11962, 0x72D8 }, { 11963, 0x72C9 }, { 11964, 0x72DC }, { 11965, 0x72D2 }, { 11966, 0x72D4 }, { 11967, 0x72DA },
    { 11968, 0x72CC }, { 11969, 0x72D1 }, { 11970, 0x73A4 }, { 11971, 0x73A1 }, { 11972, 0x73AD }, { 11973, 0x73A6 }, { 11974, 0x73A2 }, { 11975, 0x73A0 },
    { 11976, 0x73AC }, { 11977, 0x739D }, { 11978, 0x74DD }, { 11979, 0x74E8 }, { 11980, 0x753F }, { 11981, 0x7540 }, { 11982, 0x753E }, { 11983, 0x758C },
    { 11984, 0x7598 }, { 11985, 0x76AF }, { 11986, 0x76F3 }, { 11987, 0x76F1 }, { 11988, 0x76F0 }, { 11989, 0x76F5 }, { 11990, 0x77F8 }, { 11991, 0x77FC },
    { 11992, 0x77F9 }, { 11993, 0x77FB }, { 11994, 0x77FA }, { 11995, 0x77F7 }, { 11996, 0x7942 }, { 11997, 0x793F }, { 11998, 0x79C5 }, { 11999, 0x7A78 },
    { 12000, 0x7A7B }, { 12001, 0x7AFB }, { 12002, 0x7C75 }, { 12003, 0x7CFD }, { 12004, 0x8035 }, { 12005, 0x808F }, { 12006, 0x80AE }, { 12007, 0x80A3 },
    { 12008, 0x80B8 }, { 12009, 0x80B5 }, { 12010, 0x80AD }, { 12011, 0x8220 }, { 12012, 0x82A0 }, { 12013, 0x82C0 }, { 12014, 0x82AB }, { 12015, 0x829A },
    { 12016, 0x8298 }, { 12017, 0x829B }, { 12018, 0x82B5 }, { 12019, 0x82A7 }, { 12020, 0x82AE }, { 12021, 0x82BC }, { 12022, 0x829E }, { 12023, 0x82BA },
    { 12024, 0x82B4 }, { 12025, 0x82A8 }, { 12026, 0x82A1 }, { 12027, 0x82A9 }, { 12028, 0x82C2 }, { 12029, 0x82A4 }, { 12030, 0x82C3 }, { 12031, 0x82B6 },
    { 12032, 0x82A2 }, { 12033, 0x8670 }, { 12034, 0x866F }, { 12035, 0x866D }, { 12036, 0x866E }, { 12037, 0x8C56 }, { 12038, 0x8FD2 }, { 12039, 0x8FCB },
    { 12040, 0x8FD3 }, { 12041, 0x8FCD }, { 12042, 0x8FD6 }, { 12043, 0x8FD5 }, { 12044, 0x8FD7 }, { 12045, 0x90B2 }, { 12046, 0x90B4 }, { 12047, 0x90AF },
    { 12048, 0x90B3 }, { 12049, 0x90B0 }, { 12050, 0x9639 }, { 12051, 0x963D }, { 12052, 0x963C }, { 12053, 0x963A }, { 12054, 0x9643 }, { 12055, 0x4FCD },
    { 12056, 0x4FC5 }, { 12057, 0x4FD3 }, { 12058, 0x4FB2 }, { 12059, 0x4FC9 }, { 12060, 0x4FCB }, { 12061, 0x4FC1 }, { 12062, 0x4FD4 }, { 12063, 0x4FDC },
    { 12064, 0x4FD9 }, { 12065, 0x4FBB }, { 12066, 0x4FB3 }, { 12067, 0x4FDB }, { 12068, 0x4FC7 }, { 12069, 0x4FD6 }, { 12070, 0x4FBA }, { 12071, 0x4FC0 },
    { 12072, 0x4FB9 }, { 12073, 0x4FEC }, { 12074, 0x5244 }, { 12075, 0x5249 }, { 12076, 0x52C0 }, { 12077, 0x52C2 }, { 12078, 0x533D }, { 12079, 0x537C },
    { 12080, 0x5397 }, { 12081, 0x5396 }, { 12082, 0x5399 }, { 12083, 0x5398 }, { 12084, 0x54BA }, { 12085, 0x54A1 }, { 12086, 0x54AD }, { 12087, 0x54A5 },
    { 12088, 0x54CF }, { 12089, 0x54C3 }, { 12090, 0x830D }, { 12091, 0x54B7 }, { 12092, 0x54AE }, { 12093, 0x54D6 }, { 12094, 0x54B6 }, { 12095, 0x54C5 },
    { 12096, 0x54C6 }, { 12097, 0x54A0 }, { 12098, 0x5470 }, { 12099, 0x54BC }, { 12100, 0x54A2 }, { 12101, 0x54BE }, { 12102, 0x5472 }, { 12103, 0x54DE },
    { 12104, 0x54B0 }, { 12105, 0x57B5 }, { 12106, 0x579E }, { 12107, 0x579F }, { 12108, 0x57A4 }, { 12109, 0x578C }, { 12110, 0x5797 }, { 12111, 0x579D },
    { 12112, 0x579B }, { 12113, 0x5794 }, { 12114, 0x5798 }, { 12115, 0x578F }, { 12116, 0x5799 }, { 12117, 0x57A5 }, { 12118, 0x579A }, { 12119, 0x5795 },
    { 12120, 0x58F4 }, { 12121, 0x590D }, { 12122, 0x5953 }, { 12123, 0x59E1 }, { 12124, 0x59DE }, { 12125, 0x59EE }, { 12126, 0x5A00 }, { 12127, 0x59F1 },
    { 12128, 0x59DD }, { 12129, 0x59FA }, { 12130, 0x59FD }, { 12131, 0x59FC }, { 12132, 0x59F6 }, { 12133, 0x59E4 }, { 12134, 0x59F2 }, { 12135, 0x59F7 },
    { 12136, 0x59DB }, { 12137, 0x59E9 }, { 12138, 0x59F3 }, { 12139, 0x59F5 }, { 12140, 0x59E0 }, { 12141, 0x59FE }, { 12142, 0x59F4 }, { 12143, 0x59ED },
    { 12144, 0x5BA8 }, { 12145, 0x5C4C }, { 12146, 0x5CD0 }, { 12147, 0x5CD8 }, { 12148, 0x5CCC }, { 12149, 0x5CD7 }, { 12150, 0x5CCB }, { 12151, 0x5CDB },
    { 12152, 0x5CDE }, { 12153, 0x5CDA }, { 12154, 0x5CC9 }, { 12155, 0x5CC7 }, { 12156, 0x5CCA }, { 12157, 0x5CD6 }, { 12158, 0x5CD3 }, { 12159, 0x5CD4 },
    { 12160, 0x5CCF }, { 12161, 0x5CC8 }, { 12162, 0x5CC6 }, { 12163, 0x5CCE }, { 12164, 0x5CDF }, { 12165, 0x5CF8 }, { 12166, 0x5DF9 }, { 12167, 0x5E21 },
    { 12168, 0x5E22 }, { 12169, 0x5E23 }, { 12170, 0x5E20 }, { 12171, 0x5E24 }, { 12172, 0x5EB0 }, { 12173, 0x5EA4 }, { 12174, 0x5EA2 }, { 12175, 0x5E9B },
    { 12176, 0x5EA3 }, { 12177, 0x5EA5 }, { 12178, 0x5F07 }, { 12179, 0x5F2E }, { 12180, 0x5F56 }, { 12181, 0x5F86 }, { 12182, 0x6037 }, { 12183, 0x6039 },
    { 12184, 0x6054 }, { 12185, 0x6072 }, { 12186, 0x605E }, { 12187, 0x6045 }, { 12188, 0x6053 }, { 12189, 0x6047 }, { 12190, 0x6049 }, { 12191, 0x605B },
    { 12192, 0x604C }, { 12193, 0x6040 }, { 12194, 0x6042 }, { 12195, 0x605F }, { 12196, 0x6024 }, { 12197, 0x6044 }, { 12198, 0x6058 }, { 12199, 0x6066 },
    { 12200, 0x606E }, { 12201, 0x6242 }, { 12202, 0x6243 }, { 12203, 0x62CF }, { 12204, 0x630D }, { 12205, 0x630B }, { 12206, 0x62F5 }, { 12207, 0x630E },
    { 12208, 0x6303 }, { 12209, 0x62EB }, { 12210, 0x62F9 }, { 12211, 0x630F }, { 12212, 0x630C }, { 12213, 0x62F8 }, { 12214, 0x62F6 }, { 12215, 0x6300 },
    { 12216, 0x6313 }, { 12217, 0x6314 }, { 12218, 0x62FA }, { 12219, 0x6315 }, { 12220, 0x62FB }, { 12221, 0x62F0 }, { 12222, 0x6541 }, { 12223, 0x6543 },
    { 12224, 0x65AA }, { 12225, 0x65BF }, { 12226, 0x6636 }, { 12227, 0x6621 }, { 12228, 0x6632 }, { 12229, 0x6635 }, { 12230, 0x661C }, { 12231, 0x6626 },
    { 12232, 0x6622 }, { 12233, 0x6633 }, { 12234, 0x662B }, { 12235, 0x663A }, { 12236, 0x661D }, { 12237, 0x6634 }, { 12238, 0x6639 }, { 12239, 0x662E },
    { 12240, 0x670F }, { 12241, 0x6710 }, { 12242, 0x67C1 }, { 12243, 0x67F2 }, { 12244, 0x67C8 }, { 12245, 0x67BA }, { 12246, 0x67DC }, { 12247, 0x67BB },
    { 12248, 0x67F8 }, { 12249, 0x67D8 }, { 12250, 0x67C0 }, { 12251, 0x67B7 }, { 12252, 0x67C5 }, { 12253, 0x67EB }, { 12254, 0x67E4 }, { 12255, 0x67DF },
    { 12256, 0x67B5 }, { 12257, 0x67CD }, { 12258, 0x67B3 }, { 12259, 0x67F7 }, { 12260, 0x67F6 }, { 12261, 0x67EE }, { 12262, 0x67E3 }, { 12263, 0x67C2 },
    { 12264, 0x67B9 }, { 12265, 0x67CE }, { 12266, 0x67E7 }, { 12267, 0x67F0 }, { 12268, 0x67B2 }, { 12269, 0x67FC }, { 12270, 0x67C6 }, { 12271, 0x67ED },
    { 12272, 0x67CC }, { 12273, 0x67AE }, { 12274, 0x67E6 }, { 12275, 0x67DB }, { 12276, 0x67FA }, { 12277, 0x67C9 }, { 12278, 0x67CA }, { 12279, 0x67C3 },
    { 12280, 0x67EA }, { 12281, 0x67CB }, { 12282, 0x6B28 }, { 12283, 0x6B82 }, { 12284, 0x6B84 }, { 12285, 0x6BB6 }, { 12286, 0x6BD6 }, { 12287, 0x6BD8 },
    { 12288, 0x6BE0 }, { 12289, 0x6C20 }, { 12290, 0x6C21 }, { 12291, 0x6D28 }, { 12292, 0x6D34 }, { 12293, 0x6D2D }, { 12294, 0x6D1F }, { 12295, 0x6D3C },
    { 12296, 0x6D3F }, { 12297, 0x6D12 }, { 12298, 0x6D0A }, { 12299, 0x6CDA }, { 12300, 0x6D33 }, { 12301, 0x6D04 }, { 12302, 0x6D19 }, { 12303, 0x6D3A },
    { 12304, 0x6D1A }, { 12305, 0x6D11 }, { 12306, 0x6D00 }, { 12307, 0x6D1D }, { 12308, 0x6D42 }, { 12309, 0x6D01 }, { 12310, 0x6D18 }, { 12311, 0x6D37 },
    { 12312, 0x6D03 }, { 12313, 0x6D0F }, { 12314, 0x6D40 }, { 12315, 0x6D07 }, { 12316, 0x6D20 }, { 12317, 0x6D2C }, { 12318, 0x6D08 }, { 12319, 0x6D22 },
    { 12320, 0x6D09 }, { 12321, 0x6D10 }, { 12322, 0x70B7 }, { 12323, 0x709F }, { 12324, 0x70BE }, { 12325, 0x70B1 }, { 12326, 0x70B0 }, { 12327, 0x70A1 },
    { 12328, 0x70B4 }, { 12329, 0x70B5 }, { 12330, 0x70A9 }, { 12331, 0x7241 }, { 12332, 0x7249 }, { 12333, 0x724A }, { 12334, 0x726C }, { 12335, 0x7270 },
    { 12336, 0x7273 }, { 12337, 0x726E }, { 12338, 0x72CA }, { 12339, 0x72E4 }, { 12340, 0x72E8 }, { 12341, 0x72EB }, { 12342, 0x72DF }, { 12343, 0x72EA },
    { 12344, 0x72E6 }, { 12345, 0x72E3 }, { 12346, 0x7385 }, { 12347, 0x73CC }, { 12348, 0x73C2 }, { 12349, 0x73C8 }, { 12350, 0x73C5 }, { 12351, 0x73B9 },
    { 12352, 0x73B6 }, { 12353, 0x73B5 }, { 12354, 0x73B4 }, { 12355, 0x73EB }, { 12356, 0x73BF }, { 12357, 0x73C7 }, { 12358, 0x73BE }, { 12359, 0x73C3 },
    { 12360, 0x73C6 }, { 12361, 0x73B8 }, { 12362, 0x73CB }, { 12363, 0x74EC }, { 12364, 0x74EE }, { 12365, 0x752E }, { 12366, 0x7547 }, { 12367, 0x7548 },
    { 12368, 0x75A7 }, { 12369, 0x75AA }, { 12370, 0x7679 }, { 12371, 0x76C4 }, { 12372, 0x7708 }, { 12373, 0x7703 }, { 12374, 0x7704 }, { 12375, 0x7705 },
    { 12376, 0x770A }, { 12377, 0x76F7 }, { 12378, 0x76FB }, { 12379, 0x76FA }, { 12380, 0x77E7 }, { 12381, 0x77E8 }, { 12382, 0x7806 }, { 12383, 0x7811 },
    { 12384, 0x7812 }, { 12385, 0x7805 }, { 12386, 0x7810 }, { 12387, 0x780F }, { 12388, 0x780E }, { 12389, 0x7809 }, { 12390, 0x7803 }, { 12391, 0x7813 },
    { 12392, 0x794A }, { 12393, 0x794C }, { 12394, 0x794B }, { 12395, 0x7945 }, { 12396, 0x7944 }, { 12397, 0x79D5 }, { 12398, 0x79CD }, { 12399, 0x79CF },
    { 12400, 0x79D6 }, { 12401, 0x79CE }, { 12402, 0x7A80 }, { 12403, 0x7A7E }, { 12404, 0x7AD1 }, { 12405, 0x7B00 }, { 12406, 0x7B01 }, { 12407, 0x7C7A },
    { 12408, 0x7C78 }, { 12409, 0x7C79 }, { 12410, 0x7C7F }, { 12411, 0x7C80 }, { 12412, 0x7C81 }, { 12413, 0x7D03 }, { 12414, 0x7D08 }, { 12415, 0x7D01 },
    { 12416, 0x7F58 }, { 12417, 0x7F91 }, { 12418, 0x7F8D }, { 12419, 0x7FBE }, { 12420, 0x8007 }, { 12421, 0x800E }, { 12422, 0x800F }, { 12423, 0x8014 },
    { 12424, 0x8037 }, { 12425, 0x80D8 }, { 12426, 0x80C7 }, { 12427, 0x80E0 }, { 12428, 0x80D1 }, { 12429, 0x80C8 }, { 12430, 0x80C2 }, { 12431, 0x80D0 },
    { 12432, 0x80C5 }, { 12433, 0x80E3 }, { 12434, 0x80D9 }, { 12435, 0x80DC }, { 12436, 0x80CA }, { 12437, 0x80D5 }, { 12438, 0x80C9 }, { 12439, 0x80CF },
    { 12440, 0x80D7 }, { 12441, 0x80E6 }, { 12442, 0x80CD }, { 12443, 0x81FF }, { 12444, 0x8221 }, { 12445, 0x8294 }, { 12446, 0x82D9 }, { 12447, 0x82FE },
    { 12448, 0x82F9 }, { 12449, 0x8307 }, { 12450, 0x82E8 }, { 12451, 0x8300 }, { 12452, 0x82D5 }, { 12453, 0x833A }, { 12454, 0x82EB }, { 12455, 0x82D6 },
    { 12456, 0x82F4 }, { 12457, 0x82EC }, { 12458, 0x82E1 }, { 12459, 0x82F2 }, { 12460, 0x82F5 }, { 12461, 0x830C }, { 12462, 0x82FB }, { 12463, 0x82F6 },
    { 12464, 0x82F0 }, { 12465, 0x82EA }, { 12466, 0x82E4 }, { 12467, 0x82E0 }, { 12468, 0x82FA }, { 12469, 0x82F3 }, { 12470, 0x82ED }, { 12471, 0x8677 },
    { 12472, 0x8674 }, { 12473, 0x867C }, { 12474, 0x8673 }, { 12475, 0x8841 }, { 12476, 0x884E }, { 12477, 0x8867 }, { 12478, 0x886A }, { 12479, 0x8869 },
    { 12480, 0x89D3 }, { 12481, 0x8A04 }, { 12482, 0x8A07 }, { 12483, 0x8D72 }, { 12484, 0x8FE3 }, { 12485, 0x8FE1 }, { 12486, 0x8FEE }, { 12487, 0x8FE0 },
    { 12488, 0x90F1 }, { 12489, 0x90BD }, { 12490, 0x90BF }, { 12491, 0x90D5 }, { 12492, 0x90C5 }, { 12493, 0x90BE }, { 12494, 0x90C7 }, { 12495, 0x90CB },
    { 12496, 0x90C8 }, { 12497, 0x91D4 }, { 12498, 0x91D3 }, { 12499, 0x9654 }, { 12500, 0x964F }, { 12501, 0x9651 }, { 12502, 0x9653 }, { 12503, 0x964A },
    { 12504, 0x964E }, { 12505, 0x501E }, { 12506, 0x5005 }, { 12507, 0x5007 }, { 12508, 0x5013 }, { 12509, 0x5022 }, { 12510, 0x5030 }, { 12511, 0x501B },
    { 12512, 0x4FF5 }, { 12513, 0x4FF4 }, { 12514, 0x5033 }, { 12515, 0x5037 }, { 12516, 0x502C }, { 12517, 0x4FF6 }, { 12518, 0x4FF7 }, { 12519, 0x5017 },
    { 12520, 0x501C }, { 12521, 0x5020 }, { 12522, 0x5027 }, { 12523, 0x5035 }, { 12524, 0x502F }, { 12525, 0x5031 }, { 12526, 0x500E }, { 12527, 0x515A },
    { 12528, 0x5194 }, { 12529, 0x5193 }, { 12530, 0x51CA }, { 12531, 0x51C4 }, { 12532, 0x51C5 }, { 12533, 0x51C8 }, { 12534, 0x51CE }, { 12535, 0x5261 },
    { 12536, 0x525A }, { 12537, 0x5252 }, { 12538, 0x525E }, { 12539, 0x525F }, { 12540, 0x5255 }, { 12541, 0x5262 }, { 12542, 0x52CD }, { 12543, 0x530E },
    { 12544, 0x539E }, { 12545, 0x5526 }, { 12546, 0x54E2 }, { 12547, 0x5517 }, { 12548, 0x5512 }, { 12549, 0x54E7 }, { 12550, 0x54F3 }, { 12551, 0x54E4 },
    { 12552, 0x551A }, { 12553, 0x54FF }, { 12554, 0x5504 }, { 12555, 0x5508 }, { 12556, 0x54EB }, { 12557, 0x5511 }, { 12558, 0x5505 }, { 12559, 0x54F1 },
    { 12560, 0x550A }, { 12561, 0x54FB }, { 12562, 0x54F7 }, { 12563, 0x54F8 }, { 12564, 0x54E0 }, { 12565, 0x550E }, { 12566, 0x5503 }, { 12567, 0x550B },
    { 12568, 0x5701 }, { 12569, 0x5702 }, { 12570, 0x57CC }, { 12571, 0x5832 }, { 12572, 0x57D5 }, { 12573, 0x57D2 }, { 12574, 0x57BA }, { 12575, 0x57C6 },
    { 12576, 0x57BD }, { 12577, 0x57BC }, { 12578, 0x57B8 }, { 12579, 0x57B6 }, { 12580, 0x57BF }, { 12581, 0x57C7 }, { 12582, 0x57D0 }, { 12583, 0x57B9 },
    { 12584, 0x57C1 }, { 12585, 0x590E }, { 12586, 0x594A }, { 12587, 0x5A19 }, { 12588, 0x5A16 }, { 12589, 0x5A2D }, { 12590, 0x5A2E }, { 12591, 0x5A15 },
    { 12592, 0x5A0F }, { 12593, 0x5A17 }, { 12594, 0x5A0A }, { 12595, 0x5A1E }, { 12596, 0x5A33 }, { 12597, 0x5B6C }, { 12598, 0x5BA7 }, { 12599, 0x5BAD },
    { 12600, 0x5BAC }, { 12601, 0x5C03 }, { 12602, 0x5C56 }, { 12603, 0x5C54 }, { 12604, 0x5CEC }, { 12605, 0x5CFF }, { 12606, 0x5CEE }, { 12607, 0x5CF1 },
    { 12608, 0x5CF7 }, { 12609, 0x5D00 }, { 12610, 0x5CF9 }, { 12611, 0x5E29 }, { 12612, 0x5E28 }, { 12613, 0x5EA8 }, { 12614, 0x5EAE }, { 12615, 0x5EAA },
    { 12616, 0x5EAC }, { 12617, 0x5F33 }, { 12618, 0x5F30 }, { 12619, 0x5F67 }, { 12620, 0x605D }, { 12621, 0x605A }, { 12622, 0x6067 }, { 12623, 0x6041 },
    { 12624, 0x60A2 }, { 12625, 0x6088 }, { 12626, 0x6080 }, { 12627, 0x6092 }, { 12628, 0x6081 }, { 12629, 0x609D }, { 12630, 0x6083 }, { 12631, 0x6095 },
    { 12632, 0x609B }, { 12633, 0x6097 }, { 12634, 0x6087 }, { 12635, 0x609C }, { 12636, 0x608E }, { 12637, 0x6219 }, { 12638, 0x6246 }, { 12639, 0x62F2 },
    { 12640, 0x6310 }, { 12641, 0x6356 }, { 12642, 0x632C }, { 12643, 0x6344 }, { 12644, 0x6345 }, { 12645, 0x6336 }, { 12646, 0x6343 }, { 12647, 0x63E4 },
    { 12648, 0x6339 }, { 12649, 0x634B }, { 12650, 0x634A }, { 12651, 0x633C }, { 12652, 0x6329 }, { 12653, 0x6341 }, { 12654, 0x6334 }, { 12655, 0x6358 },
    { 12656, 0x6354 }, { 12657, 0x6359 }, { 12658, 0x632D }, { 12659, 0x6347 }, { 12660, 0x6333 }, { 12661, 0x635A }, { 12662, 0x6351 }, { 12663, 0x6338 },
    { 12664, 0x6357 }, { 12665, 0x6340 }, { 12666, 0x6348 }, { 12667, 0x654A }, { 12668, 0x6546 }, { 12669, 0x65C6 }, { 12670, 0x65C3 }, { 12671, 0x65C4 },
    { 12672, 0x65C2 }, { 12673, 0x664A }, { 12674, 0x665F }, { 12675, 0x6647 }, { 12676, 0x6651 }, { 12677, 0x6712 }, { 12678, 0x6713 }, { 12679, 0x681F },
    { 12680, 0x681A }, { 12681, 0x6849 }, { 12682, 0x6832 }, { 12683, 0x6833 }, { 12684, 0x683B }, { 12685, 0x684B }, { 12686, 0x684F }, { 12687, 0x6816 },
    { 12688, 0x6831 }, { 12689, 0x681C }, { 12690, 0x6835 }, { 12691, 0x682B }, { 12692, 0x682D }, { 12693, 0x682F }, { 12694, 0x684E }, { 12695, 0x6844 },
    { 12696, 0x6834 }, { 12697, 0x681D }, { 12698, 0x6812 }, { 12699, 0x6814 }, { 12700, 0x6826 }, { 12701, 0x6828 }, { 12702, 0x682E }, { 12703, 0x684D },
    { 12704, 0x683A }, { 12705, 0x6825 }, { 12706, 0x6820 }, { 12707, 0x6B2C }, { 12708, 0x6B2F }, { 12709, 0x6B2D }, { 12710, 0x6B31 }, { 12711, 0x6B34 },
    { 12712, 0x6B6D }, { 12713, 0x8082 }, { 12714, 0x6B88 }, { 12715, 0x6BE6 }, { 12716, 0x6BE4 }, { 12717, 0x6BE8 }, { 12718, 0x6BE3 }, { 12719, 0x6BE2 },
    { 12720, 0x6BE7 }, { 12721, 0x6C25 }, { 12722, 0x6D7A }, { 12723, 0x6D63 }, { 12724, 0x6D64 }, { 12725, 0x6D76 }, { 12726, 0x6D0D }, { 12727, 0x6D61 },
    { 12728, 0x6D92 }, { 12729, 0x6D58 }, { 12730, 0x6D62 }, { 12731, 0x6D6D }, { 12732, 0x6D6F }, { 12733, 0x6D91 }, { 12734, 0x6D8D }, { 12735, 0x6DEF },
    { 12736, 0x6D7F }, { 12737, 0x6D86 }, { 12738, 0x6D5E }, { 12739, 0x6D67 }, { 12740, 0x6D60 }, { 12741, 0x6D97 }, { 12742, 0x6D70 }, { 12743, 0x6D7C },
    { 12744, 0x6D5F }, { 12745, 0x6D82 }, { 12746, 0x6D98 }, { 12747, 0x6D2F }, { 12748, 0x6D68 }, { 12749, 0x6D8B }, { 12750, 0x6D7E }, { 12751, 0x6D80 },
    { 12752, 0x6D84 }, { 12753, 0x6D16 }, { 12754, 0x6D83 }, { 12755, 0x6D7B }, { 12756, 0x6D7D }, { 12757, 0x6D75 }, { 12758, 0x6D90 }, { 12759, 0x70DC },
    { 12760, 0x70D3 }, { 12761, 0x70D1 }, { 12762, 0x70DD }, { 12763, 0x70CB }, { 12764, 0x7F39 }, { 12765, 0x70E2 }, { 12766, 0x70D7 }, { 12767, 0x70D2 },
    { 12768, 0x70DE }, { 12769, 0x70E0 }, { 12770, 0x70D4 }, { 12771, 0x70CD }, { 12772, 0x70C5 }, { 12773, 0x70C6 }, { 12774, 0x70C7 }, { 12775, 0x70DA },
    { 12776, 0x70CE }, { 12777, 0x70E1 }, { 12778, 0x7242 }, { 12779, 0x7278 }, { 12780, 0x7277 }, { 12781, 0x7276 }, { 12782, 0x7300 }, { 12783, 0x72FA },
    { 12784, 0x72F4 }, { 12785, 0x72FE }, { 12786, 0x72F6 }, { 12787, 0x72F3 }, { 12788, 0x72FB }, { 12789, 0x7301 }, { 12790, 0x73D3 }, { 12791, 0x73D9 },
    { 12792, 0x73E5 }, { 12793, 0x73D6 }, { 12794, 0x73BC }, { 12795, 0x73E7 }, { 12796, 0x73E3 }, { 12797, 0x73E9 }, { 12798, 0x73DC }, { 12799, 0x73D2 },
    { 12800, 0x73DB }, { 12801, 0x73D4 }, { 12802, 0x73DD }, { 12803, 0x73DA }, { 12804, 0x73D7 }, { 12805, 0x73D8 }, { 12806, 0x73E8 }, { 12807, 0x74DE },
    { 12808, 0x74DF }, { 12809, 0x74F4 }, { 12810, 0x74F5 }, { 12811, 0x7521 }, { 12812, 0x755B }, { 12813, 0x755F }, { 12814, 0x75B0 }, { 12815, 0x75C1 },
    { 12816, 0x75BB }, { 12817, 0x75C4 }, { 12818, 0x75C0 }, { 12819, 0x75BF }, { 12820, 0x75B6 }, { 12821, 0x75BA }, { 12822, 0x768A }, { 12823, 0x76C9 },
    { 12824, 0x771D }, { 12825, 0x771B }, { 12826, 0x7710 }, { 12827, 0x7713 }, { 12828, 0x7712 }, { 12829, 0x7723 }, { 12830, 0x7711 }, { 12831, 0x7715 },
    { 12832, 0x7719 }, { 12833, 0x771A }, { 12834, 0x7722 }, { 12835, 0x7727 }, { 12836, 0x7823 }, { 12837, 0x782C }, { 12838, 0x7822 }, { 12839, 0x7835 },
    { 12840, 0x782F }, { 12841, 0x7828 }, { 12842, 0x782E }, { 12843, 0x782B }, { 12844, 0x7821 }, { 12845, 0x7829 }, { 12846, 0x7833 }, { 12847, 0x782A },
    { 12848, 0x7831 }, { 12849, 0x7954 }, { 12850, 0x795B }, { 12851, 0x794F }, { 12852, 0x795C }, { 12853, 0x7953 }, { 12854, 0x7952 }, { 12855, 0x7951 },
    { 12856, 0x79EB }, { 12857, 0x79EC }, { 12858, 0x79E0 }, { 12859, 0x79EE }, { 12860, 0x79ED }, { 12861, 0x79EA }, { 12862, 0x79DC }, { 12863, 0x79DE },
    { 12864, 0x79DD }, { 12865, 0x7A86 }, { 12866, 0x7A89 }, { 12867, 0x7A85 }, { 12868, 0x7A8B }, { 12869, 0x7A8C }, { 12870, 0x7A8A }, { 12871, 0x7A87 },
    { 12872, 0x7AD8 }, { 12873, 0x7B10 }, { 12874, 0x7B04 }, { 12875, 0x7B13 }, { 12876, 0x7B05 }, { 12877, 0x7B0F }, { 12878, 0x7B08 }, { 12879, 0x7B0A },
    { 12880, 0x7B0E }, { 12881, 0x7B09 }, { 12882, 0x7B12 }, { 12883, 0x7C84 }, { 12884, 0x7C91 }, { 12885, 0x7C8A }, { 12886, 0x7C8C }, { 12887, 0x7C88 },
    { 12888, 0x7C8D }, { 12889, 0x7C85 }, { 12890, 0x7D1E }, { 12891, 0x7D1D }, { 12892, 0x7D11 }, { 12893, 0x7D0E }, { 12894, 0x7D18 }, { 12895, 0x7D16 },
    { 12896, 0x7D13 }, { 12897, 0x7D1F }, { 12898, 0x7D12 }, { 12899, 0x7D0F }, { 12900, 0x7D0C }, { 12901, 0x7F5C }, { 12902, 0x7F61 }, { 12903, 0x7F5E },
    { 12904, 0x7F60 }, { 12905, 0x7F5D }, { 12906, 0x7F5B }, { 12907, 0x7F96 }, { 12908, 0x7F92 }, { 12909, 0x7FC3 }, { 12910, 0x7FC2 }, { 12911, 0x7FC0 },
    { 12912, 0x8016 }, { 12913, 0x803E }, { 12914, 0x8039 }, { 12915, 0x80FA }, { 12916, 0x80F2 }, { 12917, 0x80F9 }, { 12918, 0x80F5 }, { 12919, 0x8101 },
    { 12920, 0x80FB }, { 12921, 0x8100 }, { 12922, 0x8201 }, { 12923, 0x822F }, { 12924, 0x8225 }, { 12925, 0x8333 }, { 12926, 0x832D }, { 12927, 0x8344 },
    { 12928, 0x8319 }, { 12929, 0x8351 }, { 12930, 0x8325 }, { 12931, 0x8356 }, { 12932, 0x833F }, { 12933, 0x8341 }, { 12934, 0x8326 }, { 12935, 0x831C },
    { 12936, 0x8322 }, { 12937, 0x8342 }, { 12938, 0x834E }, { 12939, 0x831B }, { 12940, 0x832A }, { 12941, 0x8308 }, { 12942, 0x833C }, { 12943, 0x834D },
    { 12944, 0x8316 }, { 12945, 0x8324 }, { 12946, 0x8320 }, { 12947, 0x8337 }, { 12948, 0x832F }, { 12949, 0x8329 }, { 12950, 0x8347 }, { 12951, 0x8345 },
    { 12952, 0x834C }, { 12953, 0x8353 }, { 12954, 0x831E }, { 12955, 0x832C }, { 12956, 0x834B }, { 12957, 0x8327 }, { 12958, 0x8348 }, { 12959, 0x8653 },
    { 12960, 0x8652 }, { 12961, 0x86A2 }, { 12962, 0x86A8 }, { 12963, 0x8696 }, { 12964, 0x868D }, { 12965, 0x8691 }, { 12966, 0x869E }, { 12967, 0x8687 },
    { 12968, 0x8697 }, { 12969, 0x8686 }, { 12970, 0x868B }, { 12971, 0x869A }, { 12972, 0x8685 }, { 12973, 0x86A5 }, { 12974, 0x8699 }, { 12975, 0x86A1 },
    { 12976, 0x86A7 }, { 12977, 0x8695 }, { 12978, 0x8698 }, { 12979, 0x868E }, { 12980, 0x869D }, { 12981, 0x8690 }, { 12982, 0x8694 }, { 12983, 0x8843 },
    { 12984, 0x8844 }, { 12985, 0x886D }, { 12986, 0x8875 }, { 12987, 0x8876 }, { 12988, 0x8872 }, { 12989, 0x8880 }, { 12990, 0x8871 }, { 12991, 0x887F },
    { 12992, 0x886F }, { 12993, 0x8883 }, { 12994, 0x887E }, { 12995, 0x8874 }, { 12996, 0x887C }, { 12997, 0x8A12 }, { 12998, 0x8C47 }, { 12999, 0x8C57 },
    { 13000, 0x8C7B }, { 13001, 0x8CA4 }, { 13002, 0x8CA3 }, { 13003, 0x8D76 }, { 13004, 0x8D78 }, { 13005, 0x8DB5 }, { 13006, 0x8DB7 }, { 13007, 0x8DB6 },
    { 13008, 0x8ED1 }, { 13009, 0x8ED3 }, { 13010, 0x8FFE }, { 13011, 0x8FF5 }, { 13012, 0x9002 }, { 13013, 0x8FFF }, { 13014, 0x8FFB }, { 13015, 0x9004 },
    { 13016, 0x8FFC }, { 13017, 0x8FF6 }, { 13018, 0x90D6 }, { 13019, 0x90E0 }, { 13020, 0x90D9 }, { 13021, 0x90DA }, { 13022, 0x90E3 }, { 13023, 0x90DF },
    { 13024, 0x90E5 }, { 13025, 0x90D8 }, { 13026, 0x90DB }, { 13027, 0x90D7 }, { 13028, 0x90DC }, { 13029, 0x90E4 }, { 13030, 0x9150 }, { 13031, 0x914E },
    { 13032, 0x914F }, { 13033, 0x91D5 }, { 13034, 0x91E2 }, { 13035, 0x91DA }, { 13036, 0x965C }, { 13037, 0x965F }, { 13038, 0x96BC }, { 13039, 0x98E3 },
    { 13040, 0x9ADF }, { 13041, 0x9B2F }, { 13042, 0x4E7F }, { 13043, 0x5070 }, { 13044, 0x506A }, { 13045, 0x5061 }, { 13046, 0x505E }, { 13047, 0x5060 },
    { 13048, 0x5053 }, { 13049, 0x504B }, { 13050, 0x505D }, { 13051, 0x5072 }, { 13052, 0x5048 }, { 13053, 0x504D }, { 13054, 0x5041 }, { 13055, 0x505B },
    { 13056, 0x504A }, { 13057, 0x5062 }, { 13058, 0x5015 }, { 13059, 0x5045 }, { 13060, 0x505F }, { 13061, 0x5069 }, { 13062, 0x506B }, { 13063, 0x5063 },
    { 13064, 0x5064 }, { 13065, 0x5046 }, { 13066, 0x5040 }, { 13067, 0x506E }, { 13068, 0x5073 }, { 13069, 0x5057 }, { 13070, 0x5051 }, { 13071, 0x51D0 },
    { 13072, 0x526B }, { 13073, 0x526D }, { 13074, 0x526C }, { 13075, 0x526E }, { 13076, 0x52D6 }, { 13077, 0x52D3 }, { 13078, 0x532D }, { 13079, 0x539C },
    { 13080, 0x5575 }, { 13081, 0x5576 }, { 13082, 0x553C }, { 13083, 0x554D }, { 13084, 0x5550 }, { 13085, 0x5534 }, { 13086, 0x552A }, { 13087, 0x5551 },
    { 13088, 0x5562 }, { 13089, 0x5536 }, { 13090, 0x5535 }, { 13091, 0x5530 }, { 13092, 0x5552 }, { 13093, 0x5545 }, { 13094, 0x550C }, { 13095, 0x5532 },
    { 13096, 0x5565 }, { 13097, 0x554E }, { 13098, 0x5539 }, { 13099, 0x5548 }, { 13100, 0x552D }, { 13101, 0x553B }, { 13102, 0x5540 }, { 13103, 0x554B },
    { 13104, 0x570A }, { 13105, 0x5707 }, { 13106, 0x57FB }, { 13107, 0x5814 }, { 13108, 0x57E2 }, { 13109, 0x57F6 }, { 13110, 0x57DC }, { 13111, 0x57F4 },
    { 13112, 0x5800 }, { 13113, 0x57ED }, { 13114, 0x57FD }, { 13115, 0x5808 }, { 13116, 0x57F8 }, { 13117, 0x580B }, { 13118, 0x57F3 }, { 13119, 0x57CF },
    { 13120, 0x5807 }, { 13121, 0x57EE }, { 13122, 0x57E3 }, { 13123, 0x57F2 }, { 13124, 0x57E5 }, { 13125, 0x57EC }, { 13126, 0x57E1 }, { 13127, 0x580E },
    { 13128, 0x57FC }, { 13129, 0x5810 }, { 13130, 0x57E7 }, { 13131, 0x5801 }, { 13132, 0x580C }, { 13133, 0x57F1 }, { 13134, 0x57E9 }, { 13135, 0x57F0 },
    { 13136, 0x580D }, { 13137, 0x5804 }, { 13138, 0x595C }, { 13139, 0x5A60 }, { 13140, 0x5A58 }, { 13141, 0x5A55 }, { 13142, 0x5A67 }, { 13143, 0x5A5E },
    { 13144, 0x5A38 }, { 13145, 0x5A35 }, { 13146, 0x5A6D }, { 13147, 0x5A50 }, { 13148, 0x5A5F }, { 13149, 0x5A65 }, { 13150, 0x5A6C }, { 13151, 0x5A53 },
    { 13152, 0x5A64 }, { 13153, 0x5A57 }, { 13154, 0x5A43 }, { 13155, 0x5A5D }, { 13156, 0x5A52 }, { 13157, 0x5A44 }, { 13158, 0x5A5B }, { 13159, 0x5A48 },
    { 13160, 0x5A8E }, { 13161, 0x5A3E }, { 13162, 0x5A4D }, { 13163, 0x5A39 }, { 13164, 0x5A4C }, { 13165, 0x5A70 }, { 13166, 0x5A69 }, { 13167, 0x5A47 },
    { 13168, 0x5A51 }, { 13169, 0x5A56 }, { 13170, 0x5A42 }, { 13171, 0x5A5C }, { 13172, 0x5B72 }, { 13173, 0x5B6E }, { 13174, 0x5BC1 }, { 13175, 0x5BC0 },
    { 13176, 0x5C59 }, { 13177, 0x5D1E }, { 13178, 0x5D0B }, { 13179, 0x5D1D }, { 13180, 0x5D1A }, { 13181, 0x5D20 }, { 13182, 0x5D0C }, { 13183, 0x5D28 },
    { 13184, 0x5D0D }, { 13185, 0x5D26 }, { 13186, 0x5D25 }, { 13187, 0x5D0F }, { 13188, 0x5D30 }, { 13189, 0x5D12 }, { 13190, 0x5D23 }, { 13191, 0x5D1F },
    { 13192, 0x5D2E }, { 13193, 0x5E3E }, { 13194, 0x5E34 }, { 13195, 0x5EB1 }, { 13196, 0x5EB4 }, { 13197, 0x5EB9 }, { 13198, 0x5EB2 }, { 13199, 0x5EB3 },
    { 13200, 0x5F36 }, { 13201, 0x5F38 }, { 13202, 0x5F9B }, { 13203, 0x5F96 }, { 13204, 0x5F9F }, { 13205, 0x608A }, { 13206, 0x6090 }, { 13207, 0x6086 },
    { 13208, 0x60BE }, { 13209, 0x60B0 }, { 13210, 0x60BA }, { 13211, 0x60D3 }, { 13212, 0x60D4 }, { 13213, 0x60CF }, { 13214, 0x60E4 }, { 13215, 0x60D9 },
    { 13216, 0x60DD }, { 13217, 0x60C8 }, { 13218, 0x60B1 }, { 13219, 0x60DB }, { 13220, 0x60B7 }, { 13221, 0x60CA }, { 13222, 0x60BF }, { 13223, 0x60C3 },
    { 13224, 0x60CD }, { 13225, 0x60C0 }, { 13226, 0x6332 }, { 13227, 0x6365 }, { 13228, 0x638A }, { 13229, 0x6382 }, { 13230, 0x637D }, { 13231, 0x63BD },
    { 13232, 0x639E }, { 13233, 0x63AD }, { 13234, 0x639D }, { 13235, 0x6397 }, { 13236, 0x63AB }, { 13237, 0x638E }, { 13238, 0x636F }, { 13239, 0x6387 },
    { 13240, 0x6390 }, { 13241, 0x636E }, { 13242, 0x63AF }, { 13243, 0x6375 }, { 13244, 0x639C }, { 13245, 0x636D }, { 13246, 0x63AE }, { 13247, 0x637C },
    { 13248, 0x63A4 }, { 13249, 0x633B }, { 13250, 0x639F }, { 13251, 0x6378 }, { 13252, 0x6385 }, { 13253, 0x6381 }, { 13254, 0x6391 }, { 13255, 0x638D },
    { 13256, 0x6370 }, { 13257, 0x6553 }, { 13258, 0x65CD }, { 13259, 0x6665 }, { 13260, 0x6661 }, { 13261, 0x665B }, { 13262, 0x6659 }, { 13263, 0x665C },
    { 13264, 0x6662 }, { 13265, 0x6718 }, { 13266, 0x6879 }, { 13267, 0x6887 }, { 13268, 0x6890 }, { 13269, 0x689C }, { 13270, 0x686D }, { 13271, 0x686E },
    { 13272, 0x68AE }, { 13273, 0x68AB }, { 13274, 0x6956 }, { 13275, 0x686F }, { 13276, 0x68A3 }, { 13277, 0x68AC }, { 13278, 0x68A9 }, { 13279, 0x6875 },
    { 13280, 0x6874 }, { 13281, 0x68B2 }, { 13282, 0x688F }, { 13283, 0x6877 }, { 13284, 0x6892 }, { 13285, 0x687C }, { 13286, 0x686B }, { 13287, 0x6872 },
    { 13288, 0x68AA }, { 13289, 0x6880 }, { 13290, 0x6871 }, { 13291, 0x687E }, { 13292, 0x689B }, { 13293, 0x6896 }, { 13294, 0x688B }, { 13295, 0x68A0 },
    { 13296, 0x6889 }, { 13297, 0x68A4 }, { 13298, 0x6878 }, { 13299, 0x687B }, { 13300, 0x6891 }, { 13301, 0x688C }, { 13302, 0x688A }, { 13303, 0x687D },
    { 13304, 0x6B36 }, { 13305, 0x6B33 }, { 13306, 0x6B37 }, { 13307, 0x6B38 }, { 13308, 0x6B91 }, { 13309, 0x6B8F }, { 13310, 0x6B8D }, { 13311, 0x6B8E },
    { 13312, 0x6B8C }, { 13313, 0x6C2A }, { 13314, 0x6DC0 }, { 13315, 0x6DAB }, { 13316, 0x6DB4 }, { 13317, 0x6DB3 }, { 13318, 0x6E74 }, { 13319, 0x6DAC },
    { 13320, 0x6DE9 }, { 13321, 0x6DE2 }, { 13322, 0x6DB7 }, { 13323, 0x6DF6 }, { 13324, 0x6DD4 }, { 13325, 0x6E00 }, { 13326, 0x6DC8 }, { 13327, 0x6DE0 },
    { 13328, 0x6DDF }, { 13329, 0x6DD6 }, { 13330, 0x6DBE }, { 13331, 0x6DE5 }, { 13332, 0x6DDC }, { 13333, 0x6DDD }, { 13334, 0x6DDB }, { 13335, 0x6DF4 },
    { 13336, 0x6DCA }, { 13337, 0x6DBD }, { 13338, 0x6DED }, { 13339, 0x6DF0 }, { 13340, 0x6DBA }, { 13341, 0x6DD5 }, { 13342, 0x6DC2 }, { 13343, 0x6DCF },
    { 13344, 0x6DC9 }, { 13345, 0x6DD0 }, { 13346, 0x6DF2 }, { 13347, 0x6DD3 }, { 13348, 0x6DFD }, { 13349, 0x6DD7 }, { 13350, 0x6DCD }, { 13351, 0x6DE3 },
    { 13352, 0x6DBB }, { 13353, 0x70FA }, { 13354, 0x710D }, { 13355, 0x70F7 }, { 13356, 0x7117 }, { 13357, 0x70F4 }, { 13358, 0x710C }, { 13359, 0x70F0 },
    { 13360, 0x7104 }, { 13361, 0x70F3 }, { 13362, 0x7110 }, { 13363, 0x70FC }, { 13364, 0x70FF }, { 13365, 0x7106 }, { 13366, 0x7113 }, { 13367, 0x7100 },
    { 13368, 0x70F8 }, { 13369, 0x70F6 }, { 13370, 0x710B }, { 13371, 0x7102 }, { 13372, 0x710E }, { 13373, 0x727E }, { 13374, 0x727B }, { 13375, 0x727C },
    { 13376, 0x727F }, { 13377, 0x731D }, { 13378, 0x7317 }, { 13379, 0x7307 }, { 13380, 0x7311 }, { 13381, 0x7318 }, { 13382, 0x730A }, { 13383, 0x7308 },
    { 13384, 0x72FF }, { 13385, 0x730F }, { 13386, 0x731E }, { 13387, 0x7388 }, { 13388, 0x73F6 }, { 13389, 0x73F8 }, { 13390, 0x73F5 }, { 13391, 0x7404 },
    { 13392, 0x7401 }, { 13393, 0x73FD }, { 13394, 0x7407 }, { 13395, 0x7400 }, { 13396, 0x73FA }, { 13397, 0x73FC }, { 13398, 0x73FF }, { 13399, 0x740C },
    { 13400, 0x740B }, { 13401, 0x73F4 }, { 13402, 0x7408 }, { 13403, 0x7564 }, { 13404, 0x7563 }, { 13405, 0x75CE }, { 13406, 0x75D2 }, { 13407, 0x75CF },
    { 13408, 0x75CB }, { 13409, 0x75CC }, { 13410, 0x75D1 }, { 13411, 0x75D0 }, { 13412, 0x768F }, { 13413, 0x7689 }, { 13414, 0x76D3 }, { 13415, 0x7739 },
    { 13416, 0x772F }, { 13417, 0x772D }, { 13418, 0x7731 }, { 13419, 0x7732 }, { 13420, 0x7734 }, { 13421, 0x7733 }, { 13422, 0x773D }, { 13423, 0x7725 },
    { 13424, 0x773B }, { 13425, 0x7735 }, { 13426, 0x7848 }, { 13427, 0x7852 }, { 13428, 0x7849 }, { 13429, 0x784D }, { 13430, 0x784A }, { 13431, 0x784C },
    { 13432, 0x7826 }, { 13433, 0x7845 }, { 13434, 0x7850 }, { 13435, 0x7964 }, { 13436, 0x7967 }, { 13437, 0x7969 }, { 13438, 0x796A }, { 13439, 0x7963 },
    { 13440, 0x796B }, { 13441, 0x7961 }, { 13442, 0x79BB }, { 13443, 0x79FA }, { 13444, 0x79F8 }, { 13445, 0x79F6 }, { 13446, 0x79F7 }, { 13447, 0x7A8F },
    { 13448, 0x7A94 }, { 13449, 0x7A90 }, { 13450, 0x7B35 }, { 13451, 0x7B47 }, { 13452, 0x7B34 }, { 13453, 0x7B25 }, { 13454, 0x7B30 }, { 13455, 0x7B22 },
    { 13456, 0x7B24 }, { 13457, 0x7B33 }, { 13458, 0x7B18 }, { 13459, 0x7B2A }, { 13460, 0x7B1D }, { 13461, 0x7B31 }, { 13462, 0x7B2B }, { 13463, 0x7B2D },
    { 13464, 0x7B2F }, { 13465, 0x7B32 }, { 13466, 0x7B38 }, { 13467, 0x7B1A }, { 13468, 0x7B23 }, { 13469, 0x7C94 }, { 13470, 0x7C98 }, { 13471, 0x7C96 },
    { 13472, 0x7CA3 }, { 13473, 0x7D35 }, { 13474, 0x7D3D }, { 13475, 0x7D38 }, { 13476, 0x7D36 }, { 13477, 0x7D3A }, { 13478, 0x7D45 }, { 13479, 0x7D2C },
    { 13480, 0x7D29 }, { 13481, 0x7D41 }, { 13482, 0x7D47 }, { 13483, 0x7D3E }, { 13484, 0x7D3F }, { 13485, 0x7D4A }, { 13486, 0x7D3B }, { 13487, 0x7D28 },
    { 13488, 0x7F63 }, { 13489, 0x7F95 }, { 13490, 0x7F9C }, { 13491, 0x7F9D }, { 13492, 0x7F9B }, { 13493, 0x7FCA }, { 13494, 0x7FCB }, { 13495, 0x7FCD },
    { 13496, 0x7FD0 }, { 13497, 0x7FD1 }, { 13498, 0x7FC7 }, { 13499, 0x7FCF }, { 13500, 0x7FC9 }, { 13501, 0x801F }, { 13502, 0x801E }, { 13503, 0x801B },
    { 13504, 0x8047 }, { 13505, 0x8043 }, { 13506, 0x8048 }, { 13507, 0x8118 }, { 13508, 0x8125 }, { 13509, 0x8119 }, { 13510, 0x811B }, { 13511, 0x812D },
    { 13512, 0x811F }, { 13513, 0x812C }, { 13514, 0x811E }, { 13515, 0x8121 }, { 13516, 0x8115 }, { 13517, 0x8127 }, { 13518, 0x811D }, { 13519, 0x8122 },
    { 13520, 0x8211 }, { 13521, 0x8238 }, { 13522, 0x8233 }, { 13523, 0x823A }, { 13524, 0x8234 }, { 13525, 0x8232 }, { 13526, 0x8274 }, { 13527, 0x8390 },
    { 13528, 0x83A3 }, { 13529, 0x83A8 }, { 13530, 0x838D }, { 13531, 0x837A }, { 13532, 0x8373 }, { 13533, 0x83A4 }, { 13534, 0x8374 }, { 13535, 0x838F },
    { 13536, 0x8381 }, { 13537, 0x8395 }, { 13538, 0x8399 }, { 13539, 0x8375 }, { 13540, 0x8394 }, { 13541, 0x83A9 }, { 13542, 0x837D }, { 13543, 0x8383 },
    { 13544, 0x838C }, { 13545, 0x839D }, { 13546, 0x839B }, { 13547, 0x83AA }, { 13548, 0x838B }, { 13549, 0x837E }, { 13550, 0x83A5 }, { 13551, 0x83AF },
    { 13552, 0x8388 }, { 13553, 0x8397 }, { 13554, 0x83B0 }, { 13555, 0x837F }, { 13556, 0x83A6 }, { 13557, 0x8387 }, { 13558, 0x83AE }, { 13559, 0x8376 },
    { 13560, 0x839A }, { 13561, 0x8659 }, { 13562, 0x8656 }, { 13563, 0x86BF }, { 13564, 0x86B7 }, { 13565, 0x86C2 }, { 13566, 0x86C1 }, { 13567, 0x86C5 },
    { 13568, 0x86BA }, { 13569, 0x86B0 }, { 13570, 0x86C8 }, { 13571, 0x86B9 }, { 13572, 0x86B3 }, { 13573, 0x86B8 }, { 13574, 0x86CC }, { 13575, 0x86B4 },
    { 13576, 0x86BB }, { 13577, 0x86BC }, { 13578, 0x86C3 }, { 13579, 0x86BD }, { 13580, 0x86BE }, { 13581, 0x8852 }, { 13582, 0x8889 }, { 13583, 0x8895 },
    { 13584, 0x88A8 }, { 13585, 0x88A2 }, { 13586, 0x88AA }, { 13587, 0x889A }, { 13588, 0x8891 }, { 13589, 0x88A1 }, { 13590, 0x889F }, { 13591, 0x8898 },
    { 13592, 0x88A7 }, { 13593, 0x8899 }, { 13594, 0x889B }, { 13595, 0x8897 }, { 13596, 0x88A4 }, { 13597, 0x88AC }, { 13598, 0x888C }, { 13599, 0x8893 },
    { 13600, 0x888E }, { 13601, 0x8982 }, { 13602, 0x89D6 }, { 13603, 0x89D9 }, { 13604, 0x89D5 }, { 13605, 0x8A30 }, { 13606, 0x8A27 }, { 13607, 0x8A2C },
    { 13608, 0x8A1E }, { 13609, 0x8C39 }, { 13610, 0x8C3B }, { 13611, 0x8C5C }, { 13612, 0x8C5D }, { 13613, 0x8C7D }, { 13614, 0x8CA5 }, { 13615, 0x8D7D },
    { 13616, 0x8D7B }, { 13617, 0x8D79 }, { 13618, 0x8DBC }, { 13619, 0x8DC2 }, { 13620, 0x8DB9 }, { 13621, 0x8DBF }, { 13622, 0x8DC1 }, { 13623, 0x8ED8 },
    { 13624, 0x8EDE }, { 13625, 0x8EDD }, { 13626, 0x8EDC }, { 13627, 0x8ED7 }, { 13628, 0x8EE0 }, { 13629, 0x8EE1 }, { 13630, 0x9024 }, { 13631, 0x900B },
    { 13632, 0x9011 }, { 13633, 0x901C }, { 13634, 0x900C }, { 13635, 0x9021 }, { 13636, 0x90EF }, { 13637, 0x90EA }, { 13638, 0x90F0 }, { 13639, 0x90F4 },
    { 13640, 0x90F2 }, { 13641, 0x90F3 }, { 13642, 0x90D4 }, { 13643, 0x90EB }, { 13644, 0x90EC }, { 13645, 0x90E9 }, { 13646, 0x9156 }, { 13647, 0x9158 },
    { 13648, 0x915A }, { 13649, 0x9153 }, { 13650, 0x9155 }, { 13651, 0x91EC }, { 13652, 0x91F4 }, { 13653, 0x91F1 }, { 13654, 0x91F3 }, { 13655, 0x91F8 },
    { 13656, 0x91E4 }, { 13657, 0x91F9 }, { 13658, 0x91EA }, { 13659, 0x91EB }, { 13660, 0x91F7 }, { 13661, 0x91E8 }, { 13662, 0x91EE }, { 13663, 0x957A },
    { 13664, 0x9586 }, { 13665, 0x9588 }, { 13666, 0x967C }, { 13667, 0x966D }, { 13668, 0x966B }, { 13669, 0x9671 }, { 13670, 0x966F }, { 13671, 0x96BF },
    { 13672, 0x976A }, { 13673, 0x9804 }, { 13674, 0x98E5 }, { 13675, 0x9997 }, { 13676, 0x509B }, { 13677, 0x5095 }, { 13678, 0x5094 }, { 13679, 0x509E },
    { 13680, 0x508B }, { 13681, 0x50A3 }, { 13682, 0x5083 }, { 13683, 0x508C }, { 13684, 0x508E }, { 13685, 0x509D }, { 13686, 0x5068 }, { 13687, 0x509C },
    { 13688, 0x5092 }, { 13689, 0x5082 }, { 13690, 0x5087 }, { 13691, 0x515F }, { 13692, 0x51D4 }, { 13693, 0x5312 }, { 13694, 0x5311 }, { 13695, 0x53A4 },
    { 13696, 0x53A7 }, { 13697, 0x5591 }, { 13698, 0x55A8 }, { 13699, 0x55A5 }, { 13700, 0x55AD }, { 13701, 0x5577 }, { 13702, 0x5645 }, { 13703, 0x55A2 },
    { 13704, 0x5593 }, { 13705, 0x5588 }, { 13706, 0x558F }, { 13707, 0x55B5 }, { 13708, 0x5581 }, { 13709, 0x55A3 }, { 13710, 0x5592 }, { 13711, 0x55A4 },
    { 13712, 0x557D }, { 13713, 0x558C }, { 13714, 0x55A6 }, { 13715, 0x557F }, { 13716, 0x5595 }, { 13717, 0x55A1 }, { 13718, 0x558E }, { 13719, 0x570C },
    { 13720, 0x5829 }, { 13721, 0x5837 }, { 13722, 0x5819 }, { 13723, 0x581E }, { 13724, 0x5827 }, { 13725, 0x5823 }, { 13726, 0x5828 }, { 13727, 0x57F5 },
    { 13728, 0x5848 }, { 13729, 0x5825 }, { 13730, 0x581C }, { 13731, 0x581B }, { 13732, 0x5833 }, { 13733, 0x583F }, { 13734, 0x5836 }, { 13735, 0x582E },
    { 13736, 0x5839 }, { 13737, 0x5838 }, { 13738, 0x582D }, { 13739, 0x582C }, { 13740, 0x583B }, { 13741, 0x5961 }, { 13742, 0x5AAF }, { 13743, 0x5A94 },
    { 13744, 0x5A9F }, { 13745, 0x5A7A }, { 13746, 0x5AA2 }, { 13747, 0x5A9E }, { 13748, 0x5A78 }, { 13749, 0x5AA6 }, { 13750, 0x5A7C }, { 13751, 0x5AA5 },
    { 13752, 0x5AAC }, { 13753, 0x5A95 }, { 13754, 0x5AAE }, { 13755, 0x5A37 }, { 13756, 0x5A84 }, { 13757, 0x5A8A }, { 13758, 0x5A97 }, { 13759, 0x5A83 },
    { 13760, 0x5A8B }, { 13761, 0x5AA9 }, { 13762, 0x5A7B }, { 13763, 0x5A7D }, { 13764, 0x5A8C }, { 13765, 0x5A9C }, { 13766, 0x5A8F }, { 13767, 0x5A93 },
    { 13768, 0x5A9D }, { 13769, 0x5BEA }, { 13770, 0x5BCD }, { 13771, 0x5BCB }, { 13772, 0x5BD4 }, { 13773, 0x5BD1 }, { 13774, 0x5BCA }, { 13775, 0x5BCE },
    { 13776, 0x5C0C }, { 13777, 0x5C30 }, { 13778, 0x5D37 }, { 13779, 0x5D43 }, { 13780, 0x5D6B }, { 13781, 0x5D41 }, { 13782, 0x5D4B }, { 13783, 0x5D3F },
    { 13784, 0x5D35 }, { 13785, 0x5D51 }, { 13786, 0x5D4E }, { 13787, 0x5D55 }, { 13788, 0x5D33 }, { 13789, 0x5D3A }, { 13790, 0x5D52 }, { 13791, 0x5D3D },
    { 13792, 0x5D31 }, { 13793, 0x5D59 }, { 13794, 0x5D42 }, { 13795, 0x5D39 }, { 13796, 0x5D49 }, { 13797, 0x5D38 }, { 13798, 0x5D3C }, { 13799, 0x5D32 },
    { 13800, 0x5D36 }, { 13801, 0x5D40 }, { 13802, 0x5D45 }, { 13803, 0x5E44 }, { 13804, 0x5E41 }, { 13805, 0x5F58 }, { 13806, 0x5FA6 }, { 13807, 0x5FA5 },
    { 13808, 0x5FAB }, { 13809, 0x60C9 }, { 13810, 0x60B9 }, { 13811, 0x60CC }, { 13812, 0x60E2 }, { 13813, 0x60CE }, { 13814, 0x60C4 }, { 13815, 0x6114 },
    { 13816, 0x60F2 }, { 13817, 0x610A }, { 13818, 0x6116 }, { 13819, 0x6105 }, { 13820, 0x60F5 }, { 13821, 0x6113 }, { 13822, 0x60F8 }, { 13823, 0x60FC },
    { 13824, 0x60FE }, { 13825, 0x60C1 }, { 13826, 0x6103 }, { 13827, 0x6118 }, { 13828, 0x611D }, { 13829, 0x6110 }, { 13830, 0x60FF }, { 13831, 0x6104 },
    { 13832, 0x610B }, { 13833, 0x624A }, { 13834, 0x6394 }, { 13835, 0x63B1 }, { 13836, 0x63B0 }, { 13837, 0x63CE }, { 13838, 0x63E5 }, { 13839, 0x63E8 },
    { 13840, 0x63EF }, { 13841, 0x63C3 }, { 13842, 0x649D }, { 13843, 0x63F3 }, { 13844, 0x63CA }, { 13845, 0x63E0 }, { 13846, 0x63F6 }, { 13847, 0x63D5 },
    { 13848, 0x63F2 }, { 13849, 0x63F5 }, { 13850, 0x6461 }, { 13851, 0x63DF }, { 13852, 0x63BE }, { 13853, 0x63DD }, { 13854, 0x63DC }, { 13855, 0x63C4 },
    { 13856, 0x63D8 }, { 13857, 0x63D3 }, { 13858, 0x63C2 }, { 13859, 0x63C7 }, { 13860, 0x63CC }, { 13861, 0x63CB }, { 13862, 0x63C8 }, { 13863, 0x63F0 },
    { 13864, 0x63D7 }, { 13865, 0x63D9 }, { 13866, 0x6532 }, { 13867, 0x6567 }, { 13868, 0x656A }, { 13869, 0x6564 }, { 13870, 0x655C }, { 13871, 0x6568 },
    { 13872, 0x6565 }, { 13873, 0x658C }, { 13874, 0x659D }, { 13875, 0x659E }, { 13876, 0x65AE }, { 13877, 0x65D0 }, { 13878, 0x65D2 }, { 13879, 0x667C },
    { 13880, 0x666C }, { 13881, 0x667B }, { 13882, 0x6680 }, { 13883, 0x6671 }, { 13884, 0x6679 }, { 13885, 0x666A }, { 13886, 0x6672 }, { 13887, 0x6701 },
    { 13888, 0x690C }, { 13889, 0x68D3 }, { 13890, 0x6904 }, { 13891, 0x68DC }, { 13892, 0x692A }, { 13893, 0x68EC }, { 13894, 0x68EA }, { 13895, 0x68F1 },
    { 13896, 0x690F }, { 13897, 0x68D6 }, { 13898, 0x68F7 }, { 13899, 0x68EB }, { 13900, 0x68E4 }, { 13901, 0x68F6 }, { 13902, 0x6913 }, { 13903, 0x6910 },
    { 13904, 0x68F3 }, { 13905, 0x68E1 }, { 13906, 0x6907 }, { 13907, 0x68CC }, { 13908, 0x6908 }, { 13909, 0x6970 }, { 13910, 0x68B4 }, { 13911, 0x6911 },
    { 13912, 0x68EF }, { 13913, 0x68C6 }, { 13914, 0x6914 }, { 13915, 0x68F8 }, { 13916, 0x68D0 }, { 13917, 0x68FD }, { 13918, 0x68FC }, { 13919, 0x68E8 },
    { 13920, 0x690B }, { 13921, 0x690A }, { 13922, 0x6917 }, { 13923, 0x68CE }, { 13924, 0x68C8 }, { 13925, 0x68DD }, { 13926, 0x68DE }, { 13927, 0x68E6 },
    { 13928, 0x68F4 }, { 13929, 0x68D1 }, { 13930, 0x6906 }, { 13931, 0x68D4 }, { 13932, 0x68E9 }, { 13933, 0x6915 }, { 13934, 0x6925 }, { 13935, 0x68C7 },
    { 13936, 0x6B39 }, { 13937, 0x6B3B }, { 13938, 0x6B3F }, { 13939, 0x6B3C }, { 13940, 0x6B94 }, { 13941, 0x6B97 }, { 13942, 0x6B99 }, { 13943, 0x6B95 },
    { 13944, 0x6BBD }, { 13945, 0x6BF0 }, { 13946, 0x6BF2 }, { 13947, 0x6BF3 }, { 13948, 0x6C30 }, { 13949, 0x6DFC }, { 13950, 0x6E46 }, { 13951, 0x6E47 },
    { 13952, 0x6E1F }, { 13953, 0x6E49 }, { 13954, 0x6E88 }, { 13955, 0x6E3C }, { 13956, 0x6E3D }, { 13957, 0x6E45 }, { 13958, 0x6E62 }, { 13959, 0x6E2B },
    { 13960, 0x6E3F }, { 13961, 0x6E41 }, { 13962, 0x6E5D }, { 13963, 0x6E73 }, { 13964, 0x6E1C }, { 13965, 0x6E33 }, { 13966, 0x6E4B }, { 13967, 0x6E40 },
    { 13968, 0x6E51 }, { 13969, 0x6E3B }, { 13970, 0x6E03 }, { 13971, 0x6E2E }, { 13972, 0x6E5E }, { 13973, 0x6E68 }, { 13974, 0x6E5C }, { 13975, 0x6E61 },
    { 13976, 0x6E31 }, { 13977, 0x6E28 }, { 13978, 0x6E60 }, { 13979, 0x6E71 }, { 13980, 0x6E6B }, { 13981, 0x6E39 }, { 13982, 0x6E22 }, { 13983, 0x6E30 },
    { 13984, 0x6E53 }, { 13985, 0x6E65 }, { 13986, 0x6E27 }, { 13987, 0x6E78 }, { 13988, 0x6E64 }, { 13989, 0x6E77 }, { 13990, 0x6E55 }, { 13991, 0x6E79 },
    { 13992, 0x6E52 }, { 13993, 0x6E66 }, { 13994, 0x6E35 }, { 13995, 0x6E36 }, { 13996, 0x6E5A }, { 13997, 0x7120 }, { 13998, 0x711E }, { 13999, 0x712F },
    { 14000, 0x70FB }, { 14001, 0x712E }, { 14002, 0x7131 }, { 14003, 0x7123 }, { 14004, 0x7125 }, { 14005, 0x7122 }, { 14006, 0x7132 }, { 14007, 0x711F },
    { 14008, 0x7128 }, { 14009, 0x713A }, { 14010, 0x711B }, { 14011, 0x724B }, { 14012, 0x725A }, { 14013, 0x7288 }, { 14014, 0x7289 }, { 14015, 0x7286 },
    { 14016, 0x7285 }, { 14017, 0x728B }, { 14018, 0x7312 }, { 14019, 0x730B }, { 14020, 0x7330 }, { 14021, 0x7322 }, { 14022, 0x7331 }, { 14023, 0x7333 },
    { 14024, 0x7327 }, { 14025, 0x7332 }, { 14026, 0x732D }, { 14027, 0x7326 }, { 14028, 0x7323 }, { 14029, 0x7335 }, { 14030, 0x730C }, { 14031, 0x742E },
    { 14032, 0x742C }, { 14033, 0x7430 }, { 14034, 0x742B }, { 14035, 0x7416 }, { 14036, 0x741A }, { 14037, 0x7421 }, { 14038, 0x742D }, { 14039, 0x7431 },
    { 14040, 0x7424 }, { 14041, 0x7423 }, { 14042, 0x741D }, { 14043, 0x7429 }, { 14044, 0x7420 }, { 14045, 0x7432 }, { 14046, 0x74FB }, { 14047, 0x752F },
    { 14048, 0x756F }, { 14049, 0x756C }, { 14050, 0x75E7 }, { 14051, 0x75DA }, { 14052, 0x75E1 }, { 14053, 0x75E6 }, { 14054, 0x75DD }, { 14055, 0x75DF },
    { 14056, 0x75E4 }, { 14057, 0x75D7 }, { 14058, 0x7695 }, { 14059, 0x7692 }, { 14060, 0x76DA }, { 14061, 0x7746 }, { 14062, 0x7747 }, { 14063, 0x7744 },
    { 14064, 0x774D }, { 14065, 0x7745 }, { 14066, 0x774A }, { 14067, 0x774E }, { 14068, 0x774B }, { 14069, 0x774C }, { 14070, 0x77DE }, { 14071, 0x77EC },
    { 14072, 0x7860 }, { 14073, 0x7864 }, { 14074, 0x7865 }, { 14075, 0x785C }, { 14076, 0x786D }, { 14077, 0x7871 }, { 14078, 0x786A }, { 14079, 0x786E },
    { 14080, 0x7870 }, { 14081, 0x7869 }, { 14082, 0x7868 }, { 14083, 0x785E }, { 14084, 0x7862 }, { 14085, 0x7974 }, { 14086, 0x7973 }, { 14087, 0x7972 },
    { 14088, 0x7970 }, { 14089, 0x7A02 }, { 14090, 0x7A0A }, { 14091, 0x7A03 }, { 14092, 0x7A0C }, { 14093, 0x7A04 }, { 14094, 0x7A99 }, { 14095, 0x7AE6 },
    { 14096, 0x7AE4 }, { 14097, 0x7B4A }, { 14098, 0x7B3B }, { 14099, 0x7B44 }, { 14100, 0x7B48 }, { 14101, 0x7B4C }, { 14102, 0x7B4E }, { 14103, 0x7B40 },
    { 14104, 0x7B58 }, { 14105, 0x7B45 }, { 14106, 0x7CA2 }, { 14107, 0x7C9E }, { 14108, 0x7CA8 }, { 14109, 0x7CA1 }, { 14110, 0x7D58 }, { 14111, 0x7D6F },
    { 14112, 0x7D63 }, { 14113, 0x7D53 }, { 14114, 0x7D56 }, { 14115, 0x7D67 }, { 14116, 0x7D6A }, { 14117, 0x7D4F }, { 14118, 0x7D6D }, { 14119, 0x7D5C },
    { 14120, 0x7D6B }, { 14121, 0x7D52 }, { 14122, 0x7D54 }, { 14123, 0x7D69 }, { 14124, 0x7D51 }, { 14125, 0x7D5F }, { 14126, 0x7D4E }, { 14127, 0x7F3E },
    { 14128, 0x7F3F }, { 14129, 0x7F65 }, { 14130, 0x7F66 }, { 14131, 0x7FA2 }, { 14132, 0x7FA0 }, { 14133, 0x7FA1 }, { 14134, 0x7FD7 }, { 14135, 0x8051 },
    { 14136, 0x804F }, { 14137, 0x8050 }, { 14138, 0x80FE }, { 14139, 0x80D4 }, { 14140, 0x8143 }, { 14141, 0x814A }, { 14142, 0x8152 }, { 14143, 0x814F },
    { 14144, 0x8147 }, { 14145, 0x813D }, { 14146, 0x814D }, { 14147, 0x813A }, { 14148, 0x81E6 }, { 14149, 0x81EE }, { 14150, 0x81F7 }, { 14151, 0x81F8 },
    { 14152, 0x81F9 }, { 14153, 0x8204 }, { 14154, 0x823C }, { 14155, 0x823D }, { 14156, 0x823F }, { 14157, 0x8275 }, { 14158, 0x833B }, { 14159, 0x83CF },
    { 14160, 0x83F9 }, { 14161, 0x8423 }, { 14162, 0x83C0 }, { 14163, 0x83E8 }, { 14164, 0x8412 }, { 14165, 0x83E7 }, { 14166, 0x83E4 }, { 14167, 0x83FC },
    { 14168, 0x83F6 }, { 14169, 0x8410 }, { 14170, 0x83C6 }, { 14171, 0x83C8 }, { 14172, 0x83EB }, { 14173, 0x83E3 }, { 14174, 0x83BF }, { 14175, 0x8401 },
    { 14176, 0x83DD }, { 14177, 0x83E5 }, { 14178, 0x83D8 }, { 14179, 0x83FF }, { 14180, 0x83E1 }, { 14181, 0x83CB }, { 14182, 0x83CE }, { 14183, 0x83D6 },
    { 14184, 0x83F5 }, { 14185, 0x83C9 }, { 14186, 0x8409 }, { 14187, 0x840F }, { 14188, 0x83DE }, { 14189, 0x8411 }, { 14190, 0x8406 }, { 14191, 0x83C2 },
    { 14192, 0x83F3 }, { 14193, 0x83D5 }, { 14194, 0x83FA }, { 14195, 0x83C7 }, { 14196, 0x83D1 }, { 14197, 0x83EA }, { 14198, 0x8413 }, { 14199, 0x83C3 },
    { 14200, 0x83EC }, { 14201, 0x83EE }, { 14202, 0x83C4 }, { 14203, 0x83FB }, { 14204, 0x83D7 }, { 14205, 0x83E2 }, { 14206, 0x841B }, { 14207, 0x83DB },
    { 14208, 0x83FE }, { 14209, 0x86D8 }, { 14210, 0x86E2 }, { 14211, 0x86E6 }, { 14212, 0x86D3 }, { 14213, 0x86E3 }, { 14214, 0x86DA }, { 14215, 0x86EA },
    { 14216, 0x86DD }, { 14217, 0x86EB }, { 14218, 0x86DC }, { 14219, 0x86EC }, { 14220, 0x86E9 }, { 14221, 0x86D7 }, { 14222, 0x86E8 }, { 14223, 0x86D1 },
    { 14224, 0x8848 }, { 14225, 0x8856 }, { 14226, 0x8855 }, { 14227, 0x88BA }, { 14228, 0x88D7 }, { 14229, 0x88B9 }, { 14230, 0x88B8 }, { 14231, 0x88C0 },
    { 14232, 0x88BE }, { 14233, 0x88B6 }, { 14234, 0x88BC }, { 14235, 0x88B7 }, { 14236, 0x88BD }, { 14237, 0x88B2 }, { 14238, 0x8901 }, { 14239, 0x88C9 },
    { 14240, 0x8995 }, { 14241, 0x8998 }, { 14242, 0x8997 }, { 14243, 0x89DD }, { 14244, 0x89DA }, { 14245, 0x89DB }, { 14246, 0x8A4E }, { 14247, 0x8A4D },
    { 14248, 0x8A39 }, { 14249, 0x8A59 }, { 14250, 0x8A40 }, { 14251, 0x8A57 }, { 14252, 0x8A58 }, { 14253, 0x8A44 }, { 14254, 0x8A45 }, { 14255, 0x8A52 },
    { 14256, 0x8A48 }, { 14257, 0x8A51 }, { 14258, 0x8A4A }, { 14259, 0x8A4C }, { 14260, 0x8A4F }, { 14261, 0x8C5F }, { 14262, 0x8C81 }, { 14263, 0x8C80 },
    { 14264, 0x8CBA }, { 14265, 0x8CBE }, { 14266, 0x8CB0 }, { 14267, 0x8CB9 }, { 14268, 0x8CB5 }, { 14269, 0x8D84 }, { 14270, 0x8D80 }, { 14271, 0x8D89 },
    { 14272, 0x8DD8 }, { 14273, 0x8DD3 }, { 14274, 0x8DCD }, { 14275, 0x8DC7 }, { 14276, 0x8DD6 }, { 14277, 0x8DDC }, { 14278, 0x8DCF }, { 14279, 0x8DD5 },
    { 14280, 0x8DD9 }, { 14281, 0x8DC8 }, { 14282, 0x8DD7 }, { 14283, 0x8DC5 }, { 14284, 0x8EEF }, { 14285, 0x8EF7 }, { 14286, 0x8EFA }, { 14287, 0x8EF9 },
    { 14288, 0x8EE6 }, { 14289, 0x8EEE }, { 14290, 0x8EE5 }, { 14291, 0x8EF5 }, { 14292, 0x8EE7 }, { 14293, 0x8EE8 }, { 14294, 0x8EF6 }, { 14295, 0x8EEB },
    { 14296, 0x8EF1 }, { 14297, 0x8EEC }, { 14298, 0x8EF4 }, { 14299, 0x8EE9 }, { 14300, 0x902D }, { 14301, 0x9034 }, { 14302, 0x902F }, { 14303, 0x9106 },
    { 14304, 0x912C }, { 14305, 0x9104 }, { 14306, 0x90FF }, { 14307, 0x90FC }, { 14308, 0x9108 }, { 14309, 0x90F9 }, { 14310, 0x90FB }, { 14311, 0x9101 },
    { 14312, 0x9100 }, { 14313, 0x9107 }, { 14314, 0x9105 }, { 14315, 0x9103 }, { 14316, 0x9161 }, { 14317, 0x9164 }, { 14318, 0x915F }, { 14319, 0x9162 },
    { 14320, 0x9160 }, { 14321, 0x9201 }, { 14322, 0x920A }, { 14323, 0x9225 }, { 14324, 0x9203 }, { 14325, 0x921A }, { 14326, 0x9226 }, { 14327, 0x920F },
    { 14328, 0x920C }, { 14329, 0x9200 }, { 14330, 0x9212 }, { 14331, 0x91FF }, { 14332, 0x91FD }, { 14333, 0x9206 }, { 14334, 0x9204 }, { 14335, 0x9227 },
    { 14336, 0x9202 }, { 14337, 0x921C }, { 14338, 0x9224 }, { 14339, 0x9219 }, { 14340, 0x9217 }, { 14341, 0x9205 }, { 14342, 0x9216 }, { 14343, 0x957B },
    { 14344, 0x958D }, { 14345, 0x958C }, { 14346, 0x9590 }, { 14347, 0x9687 }, { 14348, 0x967E }, { 14349, 0x9688 }, { 14350, 0x9689 }, { 14351, 0x9683 },
    { 14352, 0x9680 }, { 14353, 0x96C2 }, { 14354, 0x96C8 }, { 14355, 0x96C3 }, { 14356, 0x96F1 }, { 14357, 0x96F0 }, { 14358, 0x976C }, { 14359, 0x9770 },
    { 14360, 0x976E }, { 14361, 0x9807 }, { 14362, 0x98A9 }, { 14363, 0x98EB }, { 14364, 0x9CE6 }, { 14365, 0x9EF9 }, { 14366, 0x4E83 }, { 14367, 0x4E84 },
    { 14368, 0x4EB6 }, { 14369, 0x50BD }, { 14370, 0x50BF }, { 14371, 0x50C6 }, { 14372, 0x50AE }, { 14373, 0x50C4 }, { 14374, 0x50CA }, { 14375, 0x50B4 },
    { 14376, 0x50C8 }, { 14377, 0x50C2 }, { 14378, 0x50B0 }, { 14379, 0x50C1 }, { 14380, 0x50BA }, { 14381, 0x50B1 }, { 14382, 0x50CB }, { 14383, 0x50C9 },
    { 14384, 0x50B6 }, { 14385, 0x50B8 }, { 14386, 0x51D7 }, { 14387, 0x527A }, { 14388, 0x5278 }, { 14389, 0x527B }, { 14390, 0x527C }, { 14391, 0x55C3 },
    { 14392, 0x55DB }, { 14393, 0x55CC }, { 14394, 0x55D0 }, { 14395, 0x55CB }, { 14396, 0x55CA }, { 14397, 0x55DD }, { 14398, 0x55C0 }, { 14399, 0x55D4 },
    { 14400, 0x55C4 }, { 14401, 0x55E9 }, { 14402, 0x55BF }, { 14403, 0x55D2 }, { 14404, 0x558D }, { 14405, 0x55CF }, { 14406, 0x55D5 }, { 14407, 0x55E2 },
    { 14408, 0x55D6 }, { 14409, 0x55C8 }, { 14410, 0x55F2 }, { 14411, 0x55CD }, { 14412, 0x55D9 }, { 14413, 0x55C2 }, { 14414, 0x5714 }, { 14415, 0x5853 },
    { 14416, 0x5868 }, { 14417, 0x5864 }, { 14418, 0x584F }, { 14419, 0x584D }, { 14420, 0x5849 }, { 14421, 0x586F }, { 14422, 0x5855 }, { 14423, 0x584E },
    { 14424, 0x585D }, { 14425, 0x5859 }, { 14426, 0x5865 }, { 14427, 0x585B }, { 14428, 0x583D }, { 14429, 0x5863 }, { 14430, 0x5871 }, { 14431, 0x58FC },
    { 14432, 0x5AC7 }, { 14433, 0x5AC4 }, { 14434, 0x5ACB }, { 14435, 0x5ABA }, { 14436, 0x5AB8 }, { 14437, 0x5AB1 }, { 14438, 0x5AB5 }, { 14439, 0x5AB0 },
    { 14440, 0x5ABF }, { 14441, 0x5AC8 }, { 14442, 0x5ABB }, { 14443, 0x5AC6 }, { 14444, 0x5AB7 }, { 14445, 0x5AC0 }, { 14446, 0x5ACA }, { 14447, 0x5AB4 },
    { 14448, 0x5AB6 }, { 14449, 0x5ACD }, { 14450, 0x5AB9 }, { 14451, 0x5A90 }, { 14452, 0x5BD6 }, { 14453, 0x5BD8 }, { 14454, 0x5BD9 }, { 14455, 0x5C1F },
    { 14456, 0x5C33 }, { 14457, 0x5D71 }, { 14458, 0x5D63 }, { 14459, 0x5D4A }, { 14460, 0x5D65 }, { 14461, 0x5D72 }, { 14462, 0x5D6C }, { 14463, 0x5D5E },
    { 14464, 0x5D68 }, { 14465, 0x5D67 }, { 14466, 0x5D62 }, { 14467, 0x5DF0 }, { 14468, 0x5E4F }, { 14469, 0x5E4E }, { 14470, 0x5E4A }, { 14471, 0x5E4D },
    { 14472, 0x5E4B }, { 14473, 0x5EC5 }, { 14474, 0x5ECC }, { 14475, 0x5EC6 }, { 14476, 0x5ECB }, { 14477, 0x5EC7 }, { 14478, 0x5F40 }, { 14479, 0x5FAF },
    { 14480, 0x5FAD }, { 14481, 0x60F7 }, { 14482, 0x6149 }, { 14483, 0x614A }, { 14484, 0x612B }, { 14485, 0x6145 }, { 14486, 0x6136 }, { 14487, 0x6132 },
    { 14488, 0x612E }, { 14489, 0x6146 }, { 14490, 0x612F }, { 14491, 0x614F }, { 14492, 0x6129 }, { 14493, 0x6140 }, { 14494, 0x6220 }, { 14495, 0x9168 },
    { 14496, 0x6223 }, { 14497, 0x6225 }, { 14498, 0x6224 }, { 14499, 0x63C5 }, { 14500, 0x63F1 }, { 14501, 0x63EB }, { 14502, 0x6410 }, { 14503, 0x6412 },
    { 14504, 0x6409 }, { 14505, 0x6420 }, { 14506, 0x6424 }, { 14507, 0x6433 }, { 14508, 0x6443 }, { 14509, 0x641F }, { 14510, 0x6415 }, { 14511, 0x6418 },
    { 14512, 0x6439 }, { 14513, 0x6437 }, { 14514, 0x6422 }, { 14515, 0x6423 }, { 14516, 0x640C }, { 14517, 0x6426 }, { 14518, 0x6430 }, { 14519, 0x6428 },
    { 14520, 0x6441 }, { 14521, 0x6435 }, { 14522, 0x642F }, { 14523, 0x640A }, { 14524, 0x641A }, { 14525, 0x6440 }, { 14526, 0x6425 }, { 14527, 0x6427 },
    { 14528, 0x640B }, { 14529, 0x63E7 }, { 14530, 0x641B }, { 14531, 0x642E }, { 14532, 0x6421 }, { 14533, 0x640E }, { 14534, 0x656F }, { 14535, 0x6592 },
    { 14536, 0x65D3 }, { 14537, 0x6686 }, { 14538, 0x668C }, { 14539, 0x6695 }, { 14540, 0x6690 }, { 14541, 0x668B }, { 14542, 0x668A }, { 14543, 0x6699 },
    { 14544, 0x6694 }, { 14545, 0x6678 }, { 14546, 0x6720 }, { 14547, 0x6966 }, { 14548, 0x695F }, { 14549, 0x6938 }, { 14550, 0x694E }, { 14551, 0x6962 },
    { 14552, 0x6971 }, { 14553, 0x693F }, { 14554, 0x6945 }, { 14555, 0x696A }, { 14556, 0x6939 }, { 14557, 0x6942 }, { 14558, 0x6957 }, { 14559, 0x6959 },
    { 14560, 0x697A }, { 14561, 0x6948 }, { 14562, 0x6949 }, { 14563, 0x6935 }, { 14564, 0x696C }, { 14565, 0x6933 }, { 14566, 0x693D }, { 14567, 0x6965 },
    { 14568, 0x68F0 }, { 14569, 0x6978 }, { 14570, 0x6934 }, { 14571, 0x6969 }, { 14572, 0x6940 }, { 14573, 0x696F }, { 14574, 0x6944 }, { 14575, 0x6976 },
    { 14576, 0x6958 }, { 14577, 0x6941 }, { 14578, 0x6974 }, { 14579, 0x694C }, { 14580, 0x693B }, { 14581, 0x694B }, { 14582, 0x6937 }, { 14583, 0x695C },
    { 14584, 0x694F }, { 14585, 0x6951 }, { 14586, 0x6932 }, { 14587, 0x6952 }, { 14588, 0x692F }, { 14589, 0x697B }, { 14590, 0x693C }, { 14591, 0x6B46 },
    { 14592, 0x6B45 }, { 14593, 0x6B43 }, { 14594, 0x6B42 }, { 14595, 0x6B48 }, { 14596, 0x6B41 }, { 14597, 0x6B9B }, { 14598, 0xFA0D }, { 14599, 0x6BFB },
    { 14600, 0x6BFC }, { 14601, 0x6BF9 }, { 14602, 0x6BF7 }, { 14603, 0x6BF8 }, { 14604, 0x6E9B }, { 14605, 0x6ED6 }, { 14606, 0x6EC8 }, { 14607, 0x6E8F },
    { 14608, 0x6EC0 }, { 14609, 0x6E9F }, { 14610, 0x6E93 }, { 14611, 0x6E94 }, { 14612, 0x6EA0 }, { 14613, 0x6EB1 }, { 14614, 0x6EB9 }, { 14615, 0x6EC6 },
    { 14616, 0x6ED2 }, { 14617, 0x6EBD }, { 14618, 0x6EC1 }, { 14619, 0x6E9E }, { 14620, 0x6EC9 }, { 14621, 0x6EB7 }, { 14622, 0x6EB0 }, { 14623, 0x6ECD },
    { 14624, 0x6EA6 }, { 14625, 0x6ECF }, { 14626, 0x6EB2 }, { 14627, 0x6EBE }, { 14628, 0x6EC3 }, { 14629, 0x6EDC }, { 14630, 0x6ED8 }, { 14631, 0x6E99 },
    { 14632, 0x6E92 }, { 14633, 0x6E8E }, { 14634, 0x6E8D }, { 14635, 0x6EA4 }, { 14636, 0x6EA1 }, { 14637, 0x6EBF }, { 14638, 0x6EB3 }, { 14639, 0x6ED0 },
    { 14640, 0x6ECA }, { 14641, 0x6E97 }, { 14642, 0x6EAE }, { 14643, 0x6EA3 }, { 14644, 0x7147 }, { 14645, 0x7154 }, { 14646, 0x7152 }, { 14647, 0x7163 },
    { 14648, 0x7160 }, { 14649, 0x7141 }, { 14650, 0x715D }, { 14651, 0x7162 }, { 14652, 0x7172 }, { 14653, 0x7178 }, { 14654, 0x716A }, { 14655, 0x7161 },
    { 14656, 0x7142 }, { 14657, 0x7158 }, { 14658, 0x7143 }, { 14659, 0x714B }, { 14660, 0x7170 }, { 14661, 0x715F }, { 14662, 0x7150 }, { 14663, 0x7153 },
    { 14664, 0x7144 }, { 14665, 0x714D }, { 14666, 0x715A }, { 14667, 0x724F }, { 14668, 0x728D }, { 14669, 0x728C }, { 14670, 0x7291 }, { 14671, 0x7290 },
    { 14672, 0x728E }, { 14673, 0x733C }, { 14674, 0x7342 }, { 14675, 0x733B }, { 14676, 0x733A }, { 14677, 0x7340 }, { 14678, 0x734A }, { 14679, 0x7349 },
    { 14680, 0x7444 }, { 14681, 0x744A }, { 14682, 0x744B }, { 14683, 0x7452 }, { 14684, 0x7451 }, { 14685, 0x7457 }, { 14686, 0x7440 }, { 14687, 0x744F },
    { 14688, 0x7450 }, { 14689, 0x744E }, { 14690, 0x7442 }, { 14691, 0x7446 }, { 14692, 0x744D }, { 14693, 0x7454 }, { 14694, 0x74E1 }, { 14695, 0x74FF },
    { 14696, 0x74FE }, { 14697, 0x74FD }, { 14698, 0x751D }, { 14699, 0x7579 }, { 14700, 0x7577 }, { 14701, 0x6983 }, { 14702, 0x75EF }, { 14703, 0x760F },
    { 14704, 0x7603 }, { 14705, 0x75F7 }, { 14706, 0x75FE }, { 14707, 0x75FC }, { 14708, 0x75F9 }, { 14709, 0x75F8 }, { 14710, 0x7610 }, { 14711, 0x75FB },
    { 14712, 0x75F6 }, { 14713, 0x75ED }, { 14714, 0x75F5 }, { 14715, 0x75FD }, { 14716, 0x7699 }, { 14717, 0x76B5 }, { 14718, 0x76DD }, { 14719, 0x7755 },
    { 14720, 0x775F }, { 14721, 0x7760 }, { 14722, 0x7752 }, { 14723, 0x7756 }, { 14724, 0x775A }, { 14725, 0x7769 }, { 14726, 0x7767 }, { 14727, 0x7754 },
    { 14728, 0x7759 }, { 14729, 0x776D }, { 14730, 0x77E0 }, { 14731, 0x7887 }, { 14732, 0x789A }, { 14733, 0x7894 }, { 14734, 0x788F }, { 14735, 0x7884 },
    { 14736, 0x7895 }, { 14737, 0x7885 }, { 14738, 0x7886 }, { 14739, 0x78A1 }, { 14740, 0x7883 }, { 14741, 0x7879 }, { 14742, 0x7899 }, { 14743, 0x7880 },
    { 14744, 0x7896 }, { 14745, 0x787B }, { 14746, 0x797C }, { 14747, 0x7982 }, { 14748, 0x797D }, { 14749, 0x7979 }, { 14750, 0x7A11 }, { 14751, 0x7A18 },
    { 14752, 0x7A19 }, { 14753, 0x7A12 }, { 14754, 0x7A17 }, { 14755, 0x7A15 }, { 14756, 0x7A22 }, { 14757, 0x7A13 }, { 14758, 0x7A1B }, { 14759, 0x7A10 },
    { 14760, 0x7AA3 }, { 14761, 0x7AA2 }, { 14762, 0x7A9E }, { 14763, 0x7AEB }, { 14764, 0x7B66 }, { 14765, 0x7B64 }, { 14766, 0x7B6D }, { 14767, 0x7B74 },
    { 14768, 0x7B69 }, { 14769, 0x7B72 }, { 14770, 0x7B65 }, { 14771, 0x7B73 }, { 14772, 0x7B71 }, { 14773, 0x7B70 }, { 14774, 0x7B61 }, { 14775, 0x7B78 },
    { 14776, 0x7B76 }, { 14777, 0x7B63 }, { 14778, 0x7CB2 }, { 14779, 0x7CB4 }, { 14780, 0x7CAF }, { 14781, 0x7D88 }, { 14782, 0x7D86 }, { 14783, 0x7D80 },
    { 14784, 0x7D8D }, { 14785, 0x7D7F }, { 14786, 0x7D85 }, { 14787, 0x7D7A }, { 14788, 0x7D8E }, { 14789, 0x7D7B }, { 14790, 0x7D83 }, { 14791, 0x7D7C },
    { 14792, 0x7D8C }, { 14793, 0x7D94 }, { 14794, 0x7D84 }, { 14795, 0x7D7D }, { 14796, 0x7D92 }, { 14797, 0x7F6D }, { 14798, 0x7F6B }, { 14799, 0x7F67 },
    { 14800, 0x7F68 }, { 14801, 0x7F6C }, { 14802, 0x7FA6 }, { 14803, 0x7FA5 }, { 14804, 0x7FA7 }, { 14805, 0x7FDB }, { 14806, 0x7FDC }, { 14807, 0x8021 },
    { 14808, 0x8164 }, { 14809, 0x8160 }, { 14810, 0x8177 }, { 14811, 0x815C }, { 14812, 0x8169 }, { 14813, 0x815B }, { 14814, 0x8162 }, { 14815, 0x8172 },
    { 14816, 0x6721 }, { 14817, 0x815E }, { 14818, 0x8176 }, { 14819, 0x8167 }, { 14820, 0x816F }, { 14821, 0x8144 }, { 14822, 0x8161 }, { 14823, 0x821D },
    { 14824, 0x8249 }, { 14825, 0x8244 }, { 14826, 0x8240 }, { 14827, 0x8242 }, { 14828, 0x8245 }, { 14829, 0x84F1 }, { 14830, 0x843F }, { 14831, 0x8456 },
    { 14832, 0x8476 }, { 14833, 0x8479 }, { 14834, 0x848F }, { 14835, 0x848D }, { 14836, 0x8465 }, { 14837, 0x8451 }, { 14838, 0x8440 }, { 14839, 0x8486 },
    { 14840, 0x8467 }, { 14841, 0x8430 }, { 14842, 0x844D }, { 14843, 0x847D }, { 14844, 0x845A }, { 14845, 0x8459 }, { 14846, 0x8474 }, { 14847, 0x8473 },
    { 14848, 0x845D }, { 14849, 0x8507 }, { 14850, 0x845E }, { 14851, 0x8437 }, { 14852, 0x843A }, { 14853, 0x8434 }, { 14854, 0x847A }, { 14855, 0x8443 },
    { 14856, 0x8478 }, { 14857, 0x8432 }, { 14858, 0x8445 }, { 14859, 0x8429 }, { 14860, 0x83D9 }, { 14861, 0x844B }, { 14862, 0x842F }, { 14863, 0x8442 },
    { 14864, 0x842D }, { 14865, 0x845F }, { 14866, 0x8470 }, { 14867, 0x8439 }, { 14868, 0x844E }, { 14869, 0x844C }, { 14870, 0x8452 }, { 14871, 0x846F },
    { 14872, 0x84C5 }, { 14873, 0x848E }, { 14874, 0x843B }, { 14875, 0x8447 }, { 14876, 0x8436 }, { 14877, 0x8433 }, { 14878, 0x8468 }, { 14879, 0x847E },
    { 14880, 0x8444 }, { 14881, 0x842B }, { 14882, 0x8460 }, { 14883, 0x8454 }, { 14884, 0x846E }, { 14885, 0x8450 }, { 14886, 0x870B }, { 14887, 0x8704 },
    { 14888, 0x86F7 }, { 14889, 0x870C }, { 14890, 0x86FA }, { 14891, 0x86D6 }, { 14892, 0x86F5 }, { 14893, 0x874D }, { 14894, 0x86F8 }, { 14895, 0x870E },
    { 14896, 0x8709 }, { 14897, 0x8701 }, { 14898, 0x86F6 }, { 14899, 0x870D }, { 14900, 0x8705 }, { 14901, 0x88D6 }, { 14902, 0x88CB }, { 14903, 0x88CD },
    { 14904, 0x88CE }, { 14905, 0x88DE }, { 14906, 0x88DB }, { 14907, 0x88DA }, { 14908, 0x88CC }, { 14909, 0x88D0 }, { 14910, 0x8985 }, { 14911, 0x899B },
    { 14912, 0x89DF }, { 14913, 0x89E5 }, { 14914, 0x89E4 }, { 14915, 0x89E1 }, { 14916, 0x89E0 }, { 14917, 0x89E2 }, { 14918, 0x89DC }, { 14919, 0x89E6 },
    { 14920, 0x8A76 }, { 14921, 0x8A86 }, { 14922, 0x8A7F }, { 14923, 0x8A61 }, { 14924, 0x8A3F }, { 14925, 0x8A77 }, { 14926, 0x8A82 }, { 14927, 0x8A84 },
    { 14928, 0x8A75 }, { 14929, 0x8A83 }, { 14930, 0x8A81 }, { 14931, 0x8A74 }, { 14932, 0x8A7A }, { 14933, 0x8C3C }, { 14934, 0x8C4B }, { 14935, 0x8C4A },
    { 14936, 0x8C65 }, { 14937, 0x8C64 }, { 14938, 0x8C66 }, { 14939, 0x8C86 }, { 14940, 0x8C84 }, { 14941, 0x8C85 }, { 14942, 0x8CCC }, { 14943, 0x8D68 },
    { 14944, 0x8D69 }, { 14945, 0x8D91 }, { 14946, 0x8D8C }, { 14947, 0x8D8E }, { 14948, 0x8D8F }, { 14949, 0x8D8D }, { 14950, 0x8D93 }, { 14951, 0x8D94 },
    { 14952, 0x8D90 }, { 14953, 0x8D92 }, { 14954, 0x8DF0 }, { 14955, 0x8DE0 }, { 14956, 0x8DEC }, { 14957, 0x8DF1 }, { 14958, 0x8DEE }, { 14959, 0x8DD0 },
    { 14960, 0x8DE9 }, { 14961, 0x8DE3 }, { 14962, 0x8DE2 }, { 14963, 0x8DE7 }, { 14964, 0x8DF2 }, { 14965, 0x8DEB }, { 14966, 0x8DF4 }, { 14967, 0x8F06 },
    { 14968, 0x8EFF }, { 14969, 0x8F01 }, { 14970, 0x8F00 }, { 14971, 0x8F05 }, { 14972, 0x8F07 }, { 14973, 0x8F08 }, { 14974, 0x8F02 }, { 14975, 0x8F0B },
    { 14976, 0x9052 }, { 14977, 0x903F }, { 14978, 0x9044 }, { 14979, 0x9049 }, { 14980, 0x903D }, { 14981, 0x9110 }, { 14982, 0x910D }, { 14983, 0x910F },
    { 14984, 0x9111 }, { 14985, 0x9116 }, { 14986, 0x9114 }, { 14987, 0x910B }, { 14988, 0x910E }, { 14989, 0x916E }, { 14990, 0x916F }, { 14991, 0x9248 },
    { 14992, 0x9252 }, { 14993, 0x9230 }, { 14994, 0x923A }, { 14995, 0x9266 }, { 14996, 0x9233 }, { 14997, 0x9265 }, { 14998, 0x925E }, { 14999, 0x9283 },
    { 15000, 0x922E }, { 15001, 0x924A }, { 15002, 0x9246 }, { 15003, 0x926D }, { 15004, 0x926C }, { 15005, 0x924F }, { 15006, 0x9260 }, { 15007, 0x9267 },
    { 15008, 0x926F }, { 15009, 0x9236 }, { 15010, 0x9261 }, { 15011, 0x9270 }, { 15012, 0x9231 }, { 15013, 0x9254 }, { 15014, 0x9263 }, { 15015, 0x9250 },
    { 15016, 0x9272 }, { 15017, 0x924E }, { 15018, 0x9253 }, { 15019, 0x924C }, { 15020, 0x9256 }, { 15021, 0x9232 }, { 15022, 0x959F }, { 15023, 0x959C },
    { 15024, 0x959E }, { 15025, 0x959B }, { 15026, 0x9692 }, { 15027, 0x9693 }, { 15028, 0x9691 }, { 15029, 0x9697 }, { 15030, 0x96CE }, { 15031, 0x96FA },
    { 15032, 0x96FD }, { 15033, 0x96F8 }, { 15034, 0x96F5 }, { 15035, 0x9773 }, { 15036, 0x9777 }, { 15037, 0x9778 }, { 15038, 0x9772 }, { 15039, 0x980F },
    { 15040, 0x980D }, { 15041, 0x980E }, { 15042, 0x98AC }, { 15043, 0x98F6 }, { 15044, 0x98F9 }, { 15045, 0x99AF }, { 15046, 0x99B2 }, { 15047, 0x99B0 },
    { 15048, 0x99B5 }, { 15049, 0x9AAD }, { 15050, 0x9AAB }, { 15051, 0x9B5B }, { 15052, 0x9CEA }, { 15053, 0x9CED }, { 15054, 0x9CE7 }, { 15055, 0x9E80 },
    { 15056, 0x9EFD }, { 15057, 0x50E6 }, { 15058, 0x50D4 }, { 15059, 0x50D7 }, { 15060, 0x50E8 }, { 15061, 0x50F3 }, { 15062, 0x50DB }, { 15063, 0x50EA },
    { 15064, 0x50DD }, { 15065, 0x50E4 }, { 15066, 0x50D3 }, { 15067, 0x50EC }, { 15068, 0x50F0 }, { 15069, 0x50EF }, { 15070, 0x50E3 }, { 15071, 0x50E0 },
    { 15072, 0x51D8 }, { 15073, 0x5280 }, { 15074, 0x5281 }, { 15075, 0x52E9 }, { 15076, 0x52EB }, { 15077, 0x5330 }, { 15078, 0x53AC }, { 15079, 0x5627 },
    { 15080, 0x5615 }, { 15081, 0x560C }, { 15082, 0x5612 }, { 15083, 0x55FC }, { 15084, 0x560F }, { 15085, 0x561C }, { 15086, 0x5601 }, { 15087, 0x5613 },
    { 15088, 0x5602 }, { 15089, 0x55FA }, { 15090, 0x561D }, { 15091, 0x5604 }, { 15092, 0x55FF }, { 15093, 0x55F9 }, { 15094, 0x5889 }, { 15095, 0x587C },
    { 15096, 0x5890 }, { 15097, 0x5898 }, { 15098, 0x5886 }, { 15099, 0x5881 }, { 15100, 0x587F }, { 15101, 0x5874 }, { 15102, 0x588B }, { 15103, 0x587A },
    { 15104, 0x5887 }, { 15105, 0x5891 }, { 15106, 0x588E }, { 15107, 0x5876 }, { 15108, 0x5882 }, { 15109, 0x5888 }, { 15110, 0x587B }, { 15111, 0x5894 },
    { 15112, 0x588F }, { 15113, 0x58FE }, { 15114, 0x596B }, { 15115, 0x5ADC }, { 15116, 0x5AEE }, { 15117, 0x5AE5 }, { 15118, 0x5AD5 }, { 15119, 0x5AEA },
    { 15120, 0x5ADA }, { 15121, 0x5AED }, { 15122, 0x5AEB }, { 15123, 0x5AF3 }, { 15124, 0x5AE2 }, { 15125, 0x5AE0 }, { 15126, 0x5ADB }, { 15127, 0x5AEC },
    { 15128, 0x5ADE }, { 15129, 0x5ADD }, { 15130, 0x5AD9 }, { 15131, 0x5AE8 }, { 15132, 0x5ADF }, { 15133, 0x5B77 }, { 15134, 0x5BE0 }, { 15135, 0x5BE3 },
    { 15136, 0x5C63 }, { 15137, 0x5D82 }, { 15138, 0x5D80 }, { 15139, 0x5D7D }, { 15140, 0x5D86 }, { 15141, 0x5D7A }, { 15142, 0x5D81 }, { 15143, 0x5D77 },
    { 15144, 0x5D8A }, { 15145, 0x5D89 }, { 15146, 0x5D88 }, { 15147, 0x5D7E }, { 15148, 0x5D7C }, { 15149, 0x5D8D }, { 15150, 0x5D79 }, { 15151, 0x5D7F },
    { 15152, 0x5E58 }, { 15153, 0x5E59 }, { 15154, 0x5E53 }, { 15155, 0x5ED8 }, { 15156, 0x5ED1 }, { 15157, 0x5ED7 }, { 15158, 0x5ECE }, { 15159, 0x5EDC },
    { 15160, 0x5ED5 }, { 15161, 0x5ED9 }, { 15162, 0x5ED2 }, { 15163, 0x5ED4 }, { 15164, 0x5F44 }, { 15165, 0x5F43 }, { 15166, 0x5F6F }, { 15167, 0x5FB6 },
    { 15168, 0x612C }, { 15169, 0x6128 }, { 15170, 0x6141 }, { 15171, 0x615E }, { 15172, 0x6171 }, { 15173, 0x6173 }, { 15174, 0x6152 }, { 15175, 0x6153 },
    { 15176, 0x6172 }, { 15177, 0x616C }, { 15178, 0x6180 }, { 15179, 0x6174 }, { 15180, 0x6154 }, { 15181, 0x617A }, { 15182, 0x615B }, { 15183, 0x6165 },
    { 15184, 0x613B }, { 15185, 0x616A }, { 15186, 0x6161 }, { 15187, 0x6156 }, { 15188, 0x6229 }, { 15189, 0x6227 }, { 15190, 0x622B }, { 15191, 0x642B },
    { 15192, 0x644D }, { 15193, 0x645B }, { 15194, 0x645D }, { 15195, 0x6474 }, { 15196, 0x6476 }, { 15197, 0x6472 }, { 15198, 0x6473 }, { 15199, 0x647D },
    { 15200, 0x6475 }, { 15201, 0x6466 }, { 15202, 0x64A6 }, { 15203, 0x644E }, { 15204, 0x6482 }, { 15205, 0x645E }, { 15206, 0x645C }, { 15207, 0x644B },
    { 15208, 0x6453 }, { 15209, 0x6460 }, { 15210, 0x6450 }, { 15211, 0x647F }, { 15212, 0x643F }, { 15213, 0x646C }, { 15214, 0x646B }, { 15215, 0x6459 },
    { 15216, 0x6465 }, { 15217, 0x6477 }, { 15218, 0x6573 }, { 15219, 0x65A0 }, { 15220, 0x66A1 }, { 15221, 0x66A0 }, { 15222, 0x669F }, { 15223, 0x6705 },
    { 15224, 0x6704 }, { 15225, 0x6722 }, { 15226, 0x69B1 }, { 15227, 0x69B6 }, { 15228, 0x69C9 }, { 15229, 0x69A0 }, { 15230, 0x69CE }, { 15231, 0x6996 },
    { 15232, 0x69B0 }, { 15233, 0x69AC }, { 15234, 0x69BC }, { 15235, 0x6991 }, { 15236, 0x6999 }, { 15237, 0x698E }, { 15238, 0x69A7 }, { 15239, 0x698D },
    { 15240, 0x69A9 }, { 15241, 0x69BE }, { 15242, 0x69AF }, { 15243, 0x69BF }, { 15244, 0x69C4 }, { 15245, 0x69BD }, { 15246, 0x69A4 }, { 15247, 0x69D4 },
    { 15248, 0x69B9 }, { 15249, 0x69CA }, { 15250, 0x699A }, { 15251, 0x69CF }, { 15252, 0x69B3 }, { 15253, 0x6993 }, { 15254, 0x69AA }, { 15255, 0x69A1 },
    { 15256, 0x699E }, { 15257, 0x69D9 }, { 15258, 0x6997 }, { 15259, 0x6990 }, { 15260, 0x69C2 }, { 15261, 0x69B5 }, { 15262, 0x69A5 }, { 15263, 0x69C6 },
    { 15264, 0x6B4A }, { 15265, 0x6B4D }, { 15266, 0x6B4B }, { 15267, 0x6B9E }, { 15268, 0x6B9F }, { 15269, 0x6BA0 }, { 15270, 0x6BC3 }, { 15271, 0x6BC4 },
    { 15272, 0x6BFE }, { 15273, 0x6ECE }, { 15274, 0x6EF5 }, { 15275, 0x6EF1 }, { 15276, 0x6F03 }, { 15277, 0x6F25 }, { 15278, 0x6EF8 }, { 15279, 0x6F37 },
    { 15280, 0x6EFB }, { 15281, 0x6F2E }, { 15282, 0x6F09 }, { 15283, 0x6F4E }, { 15284, 0x6F19 }, { 15285, 0x6F1A }, { 15286, 0x6F27 }, { 15287, 0x6F18 },
    { 15288, 0x6F3B }, { 15289, 0x6F12 }, { 15290, 0x6EED }, { 15291, 0x6F0A }, { 15292, 0x6F36 }, { 15293, 0x6F73 }, { 15294, 0x6EF9 }, { 15295, 0x6EEE },
    { 15296, 0x6F2D }, { 15297, 0x6F40 }, { 15298, 0x6F30 }, { 15299, 0x6F3C }, { 15300, 0x6F35 }, { 15301, 0x6EEB }, { 15302, 0x6F07 }, { 15303, 0x6F0E },
    { 15304, 0x6F43 }, { 15305, 0x6F05 }, { 15306, 0x6EFD }, { 15307, 0x6EF6 }, { 15308, 0x6F39 }, { 15309, 0x6F1C }, { 15310, 0x6EFC }, { 15311, 0x6F3A },
    { 15312, 0x6F1F }, { 15313, 0x6F0D }, { 15314, 0x6F1E }, { 15315, 0x6F08 }, { 15316, 0x6F21 }, { 15317, 0x7187 }, { 15318, 0x7190 }, { 15319, 0x7189 },
    { 15320, 0x7180 }, { 15321, 0x7185 }, { 15322, 0x7182 }, { 15323, 0x718F }, { 15324, 0x717B }, { 15325, 0x7186 }, { 15326, 0x7181 }, { 15327, 0x7197 },
    { 15328, 0x7244 }, { 15329, 0x7253 }, { 15330, 0x7297 }, { 15331, 0x7295 }, { 15332, 0x7293 }, { 15333, 0x7343 }, { 15334, 0x734D }, { 15335, 0x7351 },
    { 15336, 0x734C }, { 15337, 0x7462 }, { 15338, 0x7473 }, { 15339, 0x7471 }, { 15340, 0x7475 }, { 15341, 0x7472 }, { 15342, 0x7467 }, { 15343, 0x746E },
    { 15344, 0x7500 }, { 15345, 0x7502 }, { 15346, 0x7503 }, { 15347, 0x757D }, { 15348, 0x7590 }, { 15349, 0x7616 }, { 15350, 0x7608 }, { 15351, 0x760C },
    { 15352, 0x7615 }, { 15353, 0x7611 }, { 15354, 0x760A }, { 15355, 0x7614 }, { 15356, 0x76B8 }, { 15357, 0x7781 }, { 15358, 0x777C }, { 15359, 0x7785 },
    { 15360, 0x7782 }, { 15361, 0x776E }, { 15362, 0x7780 }, { 15363, 0x776F }, { 15364, 0x777E }, { 15365, 0x7783 }, { 15366, 0x78B2 }, { 15367, 0x78AA },
    { 15368, 0x78B4 }, { 15369, 0x78AD }, { 15370, 0x78A8 }, { 15371, 0x787E }, { 15372, 0x78AB }, { 15373, 0x789E }, { 15374, 0x78A5 }, { 15375, 0x78A0 },
    { 15376, 0x78AC }, { 15377, 0x78A2 }, { 15378, 0x78A4 }, { 15379, 0x7998 }, { 15380, 0x798A }, { 15381, 0x798B }, { 15382, 0x7996 }, { 15383, 0x7995 },
    { 15384, 0x7994 }, { 15385, 0x7993 }, { 15386, 0x7997 }, { 15387, 0x7988 }, { 15388, 0x7992 }, { 15389, 0x7990 }, { 15390, 0x7A2B }, { 15391, 0x7A4A },
    { 15392, 0x7A30 }, { 15393, 0x7A2F }, { 15394, 0x7A28 }, { 15395, 0x7A26 }, { 15396, 0x7AA8 }, { 15397, 0x7AAB }, { 15398, 0x7AAC }, { 15399, 0x7AEE },
    { 15400, 0x7B88 }, { 15401, 0x7B9C }, { 15402, 0x7B8A }, { 15403, 0x7B91 }, { 15404, 0x7B90 }, { 15405, 0x7B96 }, { 15406, 0x7B8D }, { 15407, 0x7B8C },
    { 15408, 0x7B9B }, { 15409, 0x7B8E }, { 15410, 0x7B85 }, { 15411, 0x7B98 }, { 15412, 0x5284 }, { 15413, 0x7B99 }, { 15414, 0x7BA4 }, { 15415, 0x7B82 },
    { 15416, 0x7CBB }, { 15417, 0x7CBF }, { 15418, 0x7CBC }, { 15419, 0x7CBA }, { 15420, 0x7DA7 }, { 15421, 0x7DB7 }, { 15422, 0x7DC2 }, { 15423, 0x7DA3 },
    { 15424, 0x7DAA }, { 15425, 0x7DC1 }, { 15426, 0x7DC0 }, { 15427, 0x7DC5 }, { 15428, 0x7D9D }, { 15429, 0x7DCE }, { 15430, 0x7DC4 }, { 15431, 0x7DC6 },
    { 15432, 0x7DCB }, { 15433, 0x7DCC }, { 15434, 0x7DAF }, { 15435, 0x7DB9 }, { 15436, 0x7D96 }, { 15437, 0x7DBC }, { 15438, 0x7D9F }, { 15439, 0x7DA6 },
    { 15440, 0x7DAE }, { 15441, 0x7DA9 }, { 15442, 0x7DA1 }, { 15443, 0x7DC9 }, { 15444, 0x7F73 }, { 15445, 0x7FE2 }, { 15446, 0x7FE3 }, { 15447, 0x7FE5 },
    { 15448, 0x7FDE }, { 15449, 0x8024 }, { 15450, 0x805D }, { 15451, 0x805C }, { 15452, 0x8189 }, { 15453, 0x8186 }, { 15454, 0x8183 }, { 15455, 0x8187 },
    { 15456, 0x818D }, { 15457, 0x818C }, { 15458, 0x818B }, { 15459, 0x8215 }, { 15460, 0x8497 }, { 15461, 0x84A4 }, { 15462, 0x84A1 }, { 15463, 0x849F },
    { 15464, 0x84BA }, { 15465, 0x84CE }, { 15466, 0x84C2 }, { 15467, 0x84AC }, { 15468, 0x84AE }, { 15469, 0x84AB }, { 15470, 0x84B9 }, { 15471, 0x84B4 },
    { 15472, 0x84C1 }, { 15473, 0x84CD }, { 15474, 0x84AA }, { 15475, 0x849A }, { 15476, 0x84B1 }, { 15477, 0x84D0 }, { 15478, 0x849D }, { 15479, 0x84A7 },
    { 15480, 0x84BB }, { 15481, 0x84A2 }, { 15482, 0x8494 }, { 15483, 0x84C7 }, { 15484, 0x84CC }, { 15485, 0x849B }, { 15486, 0x84A9 }, { 15487, 0x84AF },
    { 15488, 0x84A8 }, { 15489, 0x84D6 }, { 15490, 0x8498 }, { 15491, 0x84B6 }, { 15492, 0x84CF }, { 15493, 0x84A0 }, { 15494, 0x84D7 }, { 15495, 0x84D4 },
    { 15496, 0x84D2 }, { 15497, 0x84DB }, { 15498, 0x84B0 }, { 15499, 0x8491 }, { 15500, 0x8661 }, { 15501, 0x8733 }, { 15502, 0x8723 }, { 15503, 0x8728 },
    { 15504, 0x876B }, { 15505, 0x8740 }, { 15506, 0x872E }, { 15507, 0x871E }, { 15508, 0x8721 }, { 15509, 0x8719 }, { 15510, 0x871B }, { 15511, 0x8743 },
    { 15512, 0x872C }, { 15513, 0x8741 }, { 15514, 0x873E }, { 15515, 0x8746 }, { 15516, 0x8720 }, { 15517, 0x8732 }, { 15518, 0x872A }, { 15519, 0x872D },
    { 15520, 0x873C }, { 15521, 0x8712 }, { 15522, 0x873A }, { 15523, 0x8731 }, { 15524, 0x8735 }, { 15525, 0x8742 }, { 15526, 0x8726 }, { 15527, 0x8727 },
    { 15528, 0x8738 }, { 15529, 0x8724 }, { 15530, 0x871A }, { 15531, 0x8730 }, { 15532, 0x8711 }, { 15533, 0x88F7 }, { 15534, 0x88E7 }, { 15535, 0x88F1 },
    { 15536, 0x88F2 }, { 15537, 0x88FA }, { 15538, 0x88FE }, { 15539, 0x88EE }, { 15540, 0x88FC }, { 15541, 0x88F6 }, { 15542, 0x88FB }, { 15543, 0x88F0 },
    { 15544, 0x88EC }, { 15545, 0x88EB }, { 15546, 0x899D }, { 15547, 0x89A1 }, { 15548, 0x899F }, { 15549, 0x899E }, { 15550, 0x89E9 }, { 15551, 0x89EB },
    { 15552, 0x89E8 }, { 15553, 0x8AAB }, { 15554, 0x8A99 }, { 15555, 0x8A8B }, { 15556, 0x8A92 }, { 15557, 0x8A8F }, { 15558, 0x8A96 }, { 15559, 0x8C3D },
    { 15560, 0x8C68 }, { 15561, 0x8C69 }, { 15562, 0x8CD5 }, { 15563, 0x8CCF }, { 15564, 0x8CD7 }, { 15565, 0x8D96 }, { 15566, 0x8E09 }, { 15567, 0x8E02 },
    { 15568, 0x8DFF }, { 15569, 0x8E0D }, { 15570, 0x8DFD }, { 15571, 0x8E0A }, { 15572, 0x8E03 }, { 15573, 0x8E07 }, { 15574, 0x8E06 }, { 15575, 0x8E05 },
    { 15576, 0x8DFE }, { 15577, 0x8E00 }, { 15578, 0x8E04 }, { 15579, 0x8F10 }, { 15580, 0x8F11 }, { 15581, 0x8F0E }, { 15582, 0x8F0D }, { 15583, 0x9123 },
    { 15584, 0x911C }, { 15585, 0x9120 }, { 15586, 0x9122 }, { 15587, 0x911F }, { 15588, 0x911D }, { 15589, 0x911A }, { 15590, 0x9124 }, { 15591, 0x9121 },
    { 15592, 0x911B }, { 15593, 0x917A }, { 15594, 0x9172 }, { 15595, 0x9179 }, { 15596, 0x9173 }, { 15597, 0x92A5 }, { 15598, 0x92A4 }, { 15599, 0x9276 },
    { 15600, 0x929B }, { 15601, 0x927A }, { 15602, 0x92A0 }, { 15603, 0x9294 }, { 15604, 0x92AA }, { 15605, 0x928D }, { 15606, 0x92A6 }, { 15607, 0x929A },
    { 15608, 0x92AB }, { 15609, 0x9279 }, { 15610, 0x9297 }, { 15611, 0x927F }, { 15612, 0x92A3 }, { 15613, 0x92EE }, { 15614, 0x928E }, { 15615, 0x9282 },
    { 15616, 0x9295 }, { 15617, 0x92A2 }, { 15618, 0x927D }, { 15619, 0x9288 }, { 15620, 0x92A1 }, { 15621, 0x928A }, { 15622, 0x9286 }, { 15623, 0x928C },
    { 15624, 0x9299 }, { 15625, 0x92A7 }, { 15626, 0x927E }, { 15627, 0x9287 }, { 15628, 0x92A9 }, { 15629, 0x929D }, { 15630, 0x928B }, { 15631, 0x922D },
    { 15632, 0x969E }, { 15633, 0x96A1 }, { 15634, 0x96FF }, { 15635, 0x9758 }, { 15636, 0x977D }, { 15637, 0x977A }, { 15638, 0x977E }, { 15639, 0x9783 },
    { 15640, 0x9780 }, { 15641, 0x9782 }, { 15642, 0x977B }, { 15643, 0x9784 }, { 15644, 0x9781 }, { 15645, 0x977F }, { 15646, 0x97CE }, { 15647, 0x97CD },
    { 15648, 0x9816 }, { 15649, 0x98AD }, { 15650, 0x98AE }, { 15651, 0x9902 }, { 15652, 0x9900 }, { 15653, 0x9907 }, { 15654, 0x999D }, { 15655, 0x999C },
    { 15656, 0x99C3 }, { 15657, 0x99B9 }, { 15658, 0x99BB }, { 15659, 0x99BA }, { 15660, 0x99C2 }, { 15661, 0x99BD }, { 15662, 0x99C7 }, { 15663, 0x9AB1 },
    { 15664, 0x9AE3 }, { 15665, 0x9AE7 }, { 15666, 0x9B3E }, { 15667, 0x9B3F }, { 15668, 0x9B60 }, { 15669, 0x9B61 }, { 15670, 0x9B5F }, { 15671, 0x9CF1 },
    { 15672, 0x9CF2 }, { 15673, 0x9CF5 }, { 15674, 0x9EA7 }, { 15675, 0x50FF }, { 15676, 0x5103 }, { 15677, 0x5130 }, { 15678, 0x50F8 }, { 15679, 0x5106 },
    { 15680, 0x5107 }, { 15681, 0x50F6 }, { 15682, 0x50FE }, { 15683, 0x510B }, { 15684, 0x510C }, { 15685, 0x50FD }, { 15686, 0x510A }, { 15687, 0x528B },
    { 15688, 0x528C }, { 15689, 0x52F1 }, { 15690, 0x52EF }, { 15691, 0x5648 }, { 15692, 0x5642 }, { 15693, 0x564C }, { 15694, 0x5635 }, { 15695, 0x5641 },
    { 15696, 0x564A }, { 15697, 0x5649 }, { 15698, 0x5646 }, { 15699, 0x5658 }, { 15700, 0x565A }, { 15701, 0x5640 }, { 15702, 0x5633 }, { 15703, 0x563D },
    { 15704, 0x562C }, { 15705, 0x563E }, { 15706, 0x5638 }, { 15707, 0x562A }, { 15708, 0x563A }, { 15709, 0x571A }, { 15710, 0x58AB }, { 15711, 0x589D },
    { 15712, 0x58B1 }, { 15713, 0x58A0 }, { 15714, 0x58A3 }, { 15715, 0x58AF }, { 15716, 0x58AC }, { 15717, 0x58A5 }, { 15718, 0x58A1 }, { 15719, 0x58FF },
    { 15720, 0x5AFF }, { 15721, 0x5AF4 }, { 15722, 0x5AFD }, { 15723, 0x5AF7 }, { 15724, 0x5AF6 }, { 15725, 0x5B03 }, { 15726, 0x5AF8 }, { 15727, 0x5B02 },
    { 15728, 0x5AF9 }, { 15729, 0x5B01 }, { 15730, 0x5B07 }, { 15731, 0x5B05 }, { 15732, 0x5B0F }, { 15733, 0x5C67 }, { 15734, 0x5D99 }, { 15735, 0x5D97 },
    { 15736, 0x5D9F }, { 15737, 0x5D92 }, { 15738, 0x5DA2 }, { 15739, 0x5D93 }, { 15740, 0x5D95 }, { 15741, 0x5DA0 }, { 15742, 0x5D9C }, { 15743, 0x5DA1 },
    { 15744, 0x5D9A }, { 15745, 0x5D9E }, { 15746, 0x5E69 }, { 15747, 0x5E5D }, { 15748, 0x5E60 }, { 15749, 0x5E5C }, { 15750, 0x7DF3 }, { 15751, 0x5EDB },
    { 15752, 0x5EDE }, { 15753, 0x5EE1 }, { 15754, 0x5F49 }, { 15755, 0x5FB2 }, { 15756, 0x618B }, { 15757, 0x6183 }, { 15758, 0x6179 }, { 15759, 0x61B1 },
    { 15760, 0x61B0 }, { 15761, 0x61A2 }, { 15762, 0x6189 }, { 15763, 0x619B }, { 15764, 0x6193 }, { 15765, 0x61AF }, { 15766, 0x61AD }, { 15767, 0x619F },
    { 15768, 0x6192 }, { 15769, 0x61AA }, { 15770, 0x61A1 }, { 15771, 0x618D }, { 15772, 0x6166 }, { 15773, 0x61B3 }, { 15774, 0x622D }, { 15775, 0x646E },
    { 15776, 0x6470 }, { 15777, 0x6496 }, { 15778, 0x64A0 }, { 15779, 0x6485 }, { 15780, 0x6497 }, { 15781, 0x649C }, { 15782, 0x648F }, { 15783, 0x648B },
    { 15784, 0x648A }, { 15785, 0x648C }, { 15786, 0x64A3 }, { 15787, 0x649F }, { 15788, 0x6468 }, { 15789, 0x64B1 }, { 15790, 0x6498 }, { 15791, 0x6576 },
    { 15792, 0x657A }, { 15793, 0x6579 }, { 15794, 0x657B }, { 15795, 0x65B2 }, { 15796, 0x65B3 }, { 15797, 0x66B5 }, { 15798, 0x66B0 }, { 15799, 0x66A9 },
    { 15800, 0x66B2 }, { 15801, 0x66B7 }, { 15802, 0x66AA }, { 15803, 0x66AF }, { 15804, 0x6A00 }, { 15805, 0x6A06 }, { 15806, 0x6A17 }, { 15807, 0x69E5 },
    { 15808, 0x69F8 }, { 15809, 0x6A15 }, { 15810, 0x69F1 }, { 15811, 0x69E4 }, { 15812, 0x6A20 }, { 15813, 0x69FF }, { 15814, 0x69EC }, { 15815, 0x69E2 },
    { 15816, 0x6A1B }, { 15817, 0x6A1D }, { 15818, 0x69FE }, { 15819, 0x6A27 }, { 15820, 0x69F2 }, { 15821, 0x69EE }, { 15822, 0x6A14 }, { 15823, 0x69F7 },
    { 15824, 0x69E7 }, { 15825, 0x6A40 }, { 15826, 0x6A08 }, { 15827, 0x69E6 }, { 15828, 0x69FB }, { 15829, 0x6A0D }, { 15830, 0x69FC }, { 15831, 0x69EB },
    { 15832, 0x6A09 }, { 15833, 0x6A04 }, { 15834, 0x6A18 }, { 15835, 0x6A25 }, { 15836, 0x6A0F }, { 15837, 0x69F6 }, { 15838, 0x6A26 }, { 15839, 0x6A07 },
    { 15840, 0x69F4 }, { 15841, 0x6A16 }, { 15842, 0x6B51 }, { 15843, 0x6BA5 }, { 15844, 0x6BA3 }, { 15845, 0x6BA2 }, { 15846, 0x6BA6 }, { 15847, 0x6C01 },
    { 15848, 0x6C00 }, { 15849, 0x6BFF }, { 15850, 0x6C02 }, { 15851, 0x6F41 }, { 15852, 0x6F26 }, { 15853, 0x6F7E }, { 15854, 0x6F87 }, { 15855, 0x6FC6 },
    { 15856, 0x6F92 }, { 15857, 0x6F8D }, { 15858, 0x6F89 }, { 15859, 0x6F8C }, { 15860, 0x6F62 }, { 15861, 0x6F4F }, { 15862, 0x6F85 }, { 15863, 0x6F5A },
    { 15864, 0x6F96 }, { 15865, 0x6F76 }, { 15866, 0x6F6C }, { 15867, 0x6F82 }, { 15868, 0x6F55 }, { 15869, 0x6F72 }, { 15870, 0x6F52 }, { 15871, 0x6F50 },
    { 15872, 0x6F57 }, { 15873, 0x6F94 }, { 15874, 0x6F93 }, { 15875, 0x6F5D }, { 15876, 0x6F00 }, { 15877, 0x6F61 }, { 15878, 0x6F6B }, { 15879, 0x6F7D },
    { 15880, 0x6F67 }, { 15881, 0x6F90 }, { 15882, 0x6F53 }, { 15883, 0x6F8B }, { 15884, 0x6F69 }, { 15885, 0x6F7F }, { 15886, 0x6F95 }, { 15887, 0x6F63 },
    { 15888, 0x6F77 }, { 15889, 0x6F6A }, { 15890, 0x6F7B }, { 15891, 0x71B2 }, { 15892, 0x71AF }, { 15893, 0x719B }, { 15894, 0x71B0 }, { 15895, 0x71A0 },
    { 15896, 0x719A }, { 15897, 0x71A9 }, { 15898, 0x71B5 }, { 15899, 0x719D }, { 15900, 0x71A5 }, { 15901, 0x719E }, { 15902, 0x71A4 }, { 15903, 0x71A1 },
    { 15904, 0x71AA }, { 15905, 0x719C }, { 15906, 0x71A7 }, { 15907, 0x71B3 }, { 15908, 0x7298 }, { 15909, 0x729A }, { 15910, 0x7358 }, { 15911, 0x7352 },
    { 15912, 0x735E }, { 15913, 0x735F }, { 15914, 0x7360 }, { 15915, 0x735D }, { 15916, 0x735B }, { 15917, 0x7361 }, { 15918, 0x735A }, { 15919, 0x7359 },
    { 15920, 0x7362 }, { 15921, 0x7487 }, { 15922, 0x7489 }, { 15923, 0x748A }, { 15924, 0x7486 }, { 15925, 0x7481 }, { 15926, 0x747D }, { 15927, 0x7485 },
    { 15928, 0x7488 }, { 15929, 0x747C }, { 15930, 0x7479 }, { 15931, 0x7508 }, { 15932, 0x7507 }, { 15933, 0x757E }, { 15934, 0x7625 }, { 15935, 0x761E },
    { 15936, 0x7619 }, { 15937, 0x761D }, { 15938, 0x761C }, { 15939, 0x7623 }, { 15940, 0x761A }, { 15941, 0x7628 }, { 15942, 0x761B }, { 15943, 0x769C },
    { 15944, 0x769D }, { 15945, 0x769E }, { 15946, 0x769B }, { 15947, 0x778D }, { 15948, 0x778F }, { 15949, 0x7789 }, { 15950, 0x7788 }, { 15951, 0x78CD },
    { 15952, 0x78BB }, { 15953, 0x78CF }, { 15954, 0x78CC }, { 15955, 0x78D1 }, { 15956, 0x78CE }, { 15957, 0x78D4 }, { 15958, 0x78C8 }, { 15959, 0x78C3 },
    { 15960, 0x78C4 }, { 15961, 0x78C9 }, { 15962, 0x799A }, { 15963, 0x79A1 }, { 15964, 0x79A0 }, { 15965, 0x799C }, { 15966, 0x79A2 }, { 15967, 0x799B },
    { 15968, 0x6B76 }, { 15969, 0x7A39 }, { 15970, 0x7AB2 }, { 15971, 0x7AB4 }, { 15972, 0x7AB3 }, { 15973, 0x7BB7 }, { 15974, 0x7BCB }, { 15975, 0x7BBE },
    { 15976, 0x7BAC }, { 15977, 0x7BCE }, { 15978, 0x7BAF }, { 15979, 0x7BB9 }, { 15980, 0x7BCA }, { 15981, 0x7BB5 }, { 15982, 0x7CC5 }, { 15983, 0x7CC8 },
    { 15984, 0x7CCC }, { 15985, 0x7CCB }, { 15986, 0x7DF7 }, { 15987, 0x7DDB }, { 15988, 0x7DEA }, { 15989, 0x7DE7 }, { 15990, 0x7DD7 }, { 15991, 0x7DE1 },
    { 15992, 0x7E03 }, { 15993, 0x7DFA }, { 15994, 0x7DE6 }, { 15995, 0x7DF6 }, { 15996, 0x7DF1 }, { 15997, 0x7DF0 }, { 15998, 0x7DEE }, { 15999, 0x7DDF },
    { 16000, 0x7F76 }, { 16001, 0x7FAC }, { 16002, 0x7FB0 }, { 16003, 0x7FAD }, { 16004, 0x7FED }, { 16005, 0x7FEB }, { 16006, 0x7FEA }, { 16007, 0x7FEC },
    { 16008, 0x7FE6 }, { 16009, 0x7FE8 }, { 16010, 0x8064 }, { 16011, 0x8067 }, { 16012, 0x81A3 }, { 16013, 0x819F }, { 16014, 0x819E }, { 16015, 0x8195 },
    { 16016, 0x81A2 }, { 16017, 0x8199 }, { 16018, 0x8197 }, { 16019, 0x8216 }, { 16020, 0x824F }, { 16021, 0x8253 }, { 16022, 0x8252 }, { 16023, 0x8250 },
    { 16024, 0x824E }, { 16025, 0x8251 }, { 16026, 0x8524 }, { 16027, 0x853B }, { 16028, 0x850F }, { 16029, 0x8500 }, { 16030, 0x8529 }, { 16031, 0x850E },
    { 16032, 0x8509 }, { 16033, 0x850D }, { 16034, 0x851F }, { 16035, 0x850A }, { 16036, 0x8527 }, { 16037, 0x851C }, { 16038, 0x84FB }, { 16039, 0x852B },
    { 16040, 0x84FA }, { 16041, 0x8508 }, { 16042, 0x850C }, { 16043, 0x84F4 }, { 16044, 0x852A }, { 16045, 0x84F2 }, { 16046, 0x8515 }, { 16047, 0x84F7 },
    { 16048, 0x84EB }, { 16049, 0x84F3 }, { 16050, 0x84FC }, { 16051, 0x8512 }, { 16052, 0x84EA }, { 16053, 0x84E9 }, { 16054, 0x8516 }, { 16055, 0x84FE },
    { 16056, 0x8528 }, { 16057, 0x851D }, { 16058, 0x852E }, { 16059, 0x8502 }, { 16060, 0x84FD }, { 16061, 0x851E }, { 16062, 0x84F6 }, { 16063, 0x8531 },
    { 16064, 0x8526 }, { 16065, 0x84E7 }, { 16066, 0x84E8 }, { 16067, 0x84F0 }, { 16068, 0x84EF }, { 16069, 0x84F9 }, { 16070, 0x8518 }, { 16071, 0x8520 },
    { 16072, 0x8530 }, { 16073, 0x850B }, { 16074, 0x8519 }, { 16075, 0x852F }, { 16076, 0x8662 }, { 16077, 0x8756 }, { 16078, 0x8763 }, { 16079, 0x8764 },
    { 16080, 0x8777 }, { 16081, 0x87E1 }, { 16082, 0x8773 }, { 16083, 0x8758 }, { 16084, 0x8754 }, { 16085, 0x875B }, { 16086, 0x8752 }, { 16087, 0x8761 },
    { 16088, 0x875A }, { 16089, 0x8751 }, { 16090, 0x875E }, { 16091, 0x876D }, { 16092, 0x876A }, { 16093, 0x8750 }, { 16094, 0x874E }, { 16095, 0x875F },
    { 16096, 0x875D }, { 16097, 0x876F }, { 16098, 0x876C }, { 16099, 0x877A }, { 16100, 0x876E }, { 16101, 0x875C }, { 16102, 0x8765 }, { 16103, 0x874F },
    { 16104, 0x877B }, { 16105, 0x8775 }, { 16106, 0x8762 }, { 16107, 0x8767 }, { 16108, 0x8769 }, { 16109, 0x885A }, { 16110, 0x8905 }, { 16111, 0x890C },
    { 16112, 0x8914 }, { 16113, 0x890B }, { 16114, 0x8917 }, { 16115, 0x8918 }, { 16116, 0x8919 }, { 16117, 0x8906 }, { 16118, 0x8916 }, { 16119, 0x8911 },
    { 16120, 0x890E }, { 16121, 0x8909 }, { 16122, 0x89A2 }, { 16123, 0x89A4 }, { 16124, 0x89A3 }, { 16125, 0x89ED }, { 16126, 0x89F0 }, { 16127, 0x89EC },
    { 16128, 0x8ACF }, { 16129, 0x8AC6 }, { 16130, 0x8AB8 }, { 16131, 0x8AD3 }, { 16132, 0x8AD1 }, { 16133, 0x8AD4 }, { 16134, 0x8AD5 }, { 16135, 0x8ABB },
    { 16136, 0x8AD7 }, { 16137, 0x8ABE }, { 16138, 0x8AC0 }, { 16139, 0x8AC5 }, { 16140, 0x8AD8 }, { 16141, 0x8AC3 }, { 16142, 0x8ABA }, { 16143, 0x8ABD },
    { 16144, 0x8AD9 }, { 16145, 0x8C3E }, { 16146, 0x8C4D }, { 16147, 0x8C8F }, { 16148, 0x8CE5 }, { 16149, 0x8CDF }, { 16150, 0x8CD9 }, { 16151, 0x8CE8 },
    { 16152, 0x8CDA }, { 16153, 0x8CDD }, { 16154, 0x8CE7 }, { 16155, 0x8DA0 }, { 16156, 0x8D9C }, { 16157, 0x8DA1 }, { 16158, 0x8D9B }, { 16159, 0x8E20 },
    { 16160, 0x8E23 }, { 16161, 0x8E25 }, { 16162, 0x8E24 }, { 16163, 0x8E2E }, { 16164, 0x8E15 }, { 16165, 0x8E1B }, { 16166, 0x8E16 }, { 16167, 0x8E11 },
    { 16168, 0x8E19 }, { 16169, 0x8E26 }, { 16170, 0x8E27 }, { 16171, 0x8E14 }, { 16172, 0x8E12 }, { 16173, 0x8E18 }, { 16174, 0x8E13 }, { 16175, 0x8E1C },
    { 16176, 0x8E17 }, { 16177, 0x8E1A }, { 16178, 0x8F2C }, { 16179, 0x8F24 }, { 16180, 0x8F18 }, { 16181, 0x8F1A }, { 16182, 0x8F20 }, { 16183, 0x8F23 },
    { 16184, 0x8F16 }, { 16185, 0x8F17 }, { 16186, 0x9073 }, { 16187, 0x9070 }, { 16188, 0x906F }, { 16189, 0x9067 }, { 16190, 0x906B }, { 16191, 0x912F },
    { 16192, 0x912B }, { 16193, 0x9129 }, { 16194, 0x912A }, { 16195, 0x9132 }, { 16196, 0x9126 }, { 16197, 0x912E }, { 16198, 0x9185 }, { 16199, 0x9186 },
    { 16200, 0x918A }, { 16201, 0x9181 }, { 16202, 0x9182 }, { 16203, 0x9184 }, { 16204, 0x9180 }, { 16205, 0x92D0 }, { 16206, 0x92C3 }, { 16207, 0x92C4 },
    { 16208, 0x92C0 }, { 16209, 0x92D9 }, { 16210, 0x92B6 }, { 16211, 0x92CF }, { 16212, 0x92F1 }, { 16213, 0x92DF }, { 16214, 0x92D8 }, { 16215, 0x92E9 },
    { 16216, 0x92D7 }, { 16217, 0x92DD }, { 16218, 0x92CC }, { 16219, 0x92EF }, { 16220, 0x92C2 }, { 16221, 0x92E8 }, { 16222, 0x92CA }, { 16223, 0x92C8 },
    { 16224, 0x92CE }, { 16225, 0x92E6 }, { 16226, 0x92CD }, { 16227, 0x92D5 }, { 16228, 0x92C9 }, { 16229, 0x92E0 }, { 16230, 0x92DE }, { 16231, 0x92E7 },
    { 16232, 0x92D1 }, { 16233, 0x92D3 }, { 16234, 0x92B5 }, { 16235, 0x92E1 }, { 16236, 0x92C6 }, { 16237, 0x92B4 }, { 16238, 0x957C }, { 16239, 0x95AC },
    { 16240, 0x95AB }, { 16241, 0x95AE }, { 16242, 0x95B0 }, { 16243, 0x96A4 }, { 16244, 0x96A2 }, { 16245, 0x96D3 }, { 16246, 0x9705 }, { 16247, 0x9708 },
    { 16248, 0x9702 }, { 16249, 0x975A }, { 16250, 0x978A }, { 16251, 0x978E }, { 16252, 0x9788 }, { 16253, 0x97D0 }, { 16254, 0x97CF }, { 16255, 0x981E },
    { 16256, 0x981D }, { 16257, 0x9826 }, { 16258, 0x9829 }, { 16259, 0x9828 }, { 16260, 0x9820 }, { 16261, 0x981B }, { 16262, 0x9827 }, { 16263, 0x98B2 },
    { 16264, 0x9908 }, { 16265, 0x98FA }, { 16266, 0x9911 }, { 16267, 0x9914 }, { 16268, 0x9916 }, { 16269, 0x9917 }, { 16270, 0x9915 }, { 16271, 0x99DC },
    { 16272, 0x99CD }, { 16273, 0x99CF }, { 16274, 0x99D3 }, { 16275, 0x99D4 }, { 16276, 0x99CE }, { 16277, 0x99C9 }, { 16278, 0x99D6 }, { 16279, 0x99D8 },
    { 16280, 0x99CB }, { 16281, 0x99D7 }, { 16282, 0x99CC }, { 16283, 0x9AB3 }, { 16284, 0x9AEC }, { 16285, 0x9AEB }, { 16286, 0x9AF3 }, { 16287, 0x9AF2 },
    { 16288, 0x9AF1 }, { 16289, 0x9B46 }, { 16290, 0x9B43 }, { 16291, 0x9B67 }, { 16292, 0x9B74 }, { 16293, 0x9B71 }, { 16294, 0x9B66 }, { 16295, 0x9B76 },
    { 16296, 0x9B75 }, { 16297, 0x9B70 }, { 16298, 0x9B68 }, { 16299, 0x9B64 }, { 16300, 0x9B6C }, { 16301, 0x9CFC }, { 16302, 0x9CFA }, { 16303, 0x9CFD },
    { 16304, 0x9CFF }, { 16305, 0x9CF7 }, { 16306, 0x9D07 }, { 16307, 0x9D00 }, { 16308, 0x9CF9 }, { 16309, 0x9CFB }, { 16310, 0x9D08 }, { 16311, 0x9D05 },
    { 16312, 0x9D04 }, { 16313, 0x9E83 }, { 16314, 0x9ED3 }, { 16315, 0x9F0F }, { 16316, 0x9F10 }, { 16317, 0x511C }, { 16318, 0x5113 }, { 16319, 0x5117 },
    { 16320, 0x511A }, { 16321, 0x5111 }, { 16322, 0x51DE }, { 16323, 0x5334 }, { 16324, 0x53E1 }, { 16325, 0x5670 }, { 16326, 0x5660 }, { 16327, 0x566E },
    { 16328, 0x5673 }, { 16329, 0x5666 }, { 16330, 0x5663 }, { 16331, 0x566D }, { 16332, 0x5672 }, { 16333, 0x565E }, { 16334, 0x5677 }, { 16335, 0x571C },
    { 16336, 0x571B }, { 16337, 0x58C8 }, { 16338, 0x58BD }, { 16339, 0x58C9 }, { 16340, 0x58BF }, { 16341, 0x58BA }, { 16342, 0x58C2 }, { 16343, 0x58BC },
    { 16344, 0x58C6 }, { 16345, 0x5B17 }, { 16346, 0x5B19 }, { 16347, 0x5B1B }, { 16348, 0x5B21 }, { 16349, 0x5B14 }, { 16350, 0x5B13 }, { 16351, 0x5B10 },
    { 16352, 0x5B16 }, { 16353, 0x5B28 }, { 16354, 0x5B1A }, { 16355, 0x5B20 }, { 16356, 0x5B1E }, { 16357, 0x5BEF }, { 16358, 0x5DAC }, { 16359, 0x5DB1 },
    { 16360, 0x5DA9 }, { 16361, 0x5DA7 }, { 16362, 0x5DB5 }, { 16363, 0x5DB0 }, { 16364, 0x5DAE }, { 16365, 0x5DAA }, { 16366, 0x5DA8 }, { 16367, 0x5DB2 },
    { 16368, 0x5DAD }, { 16369, 0x5DAF }, { 16370, 0x5DB4 }, { 16371, 0x5E67 }, { 16372, 0x5E68 }, { 16373, 0x5E66 }, { 16374, 0x5E6F }, { 16375, 0x5EE9 },
    { 16376, 0x5EE7 }, { 16377, 0x5EE6 }, { 16378, 0x5EE8 }, { 16379, 0x5EE5 }, { 16380, 0x5F4B }, { 16381, 0x5FBC }, { 16382, 0x619D }, { 16383, 0x61A8 },
    { 16384, 0x6196 }, { 16385, 0x61C5 }, { 16386, 0x61B4 }, { 16387, 0x61C6 }, { 16388, 0x61C1 }, { 16389, 0x61CC }, { 16390, 0x61BA }, { 16391, 0x61BF },
    { 16392, 0x61B8 }, { 16393, 0x618C }, { 16394, 0x64D7 }, { 16395, 0x64D6 }, { 16396, 0x64D0 }, { 16397, 0x64CF }, { 16398, 0x64C9 }, { 16399, 0x64BD },
    { 16400, 0x6489 }, { 16401, 0x64C3 }, { 16402, 0x64DB }, { 16403, 0x64F3 }, { 16404, 0x64D9 }, { 16405, 0x6533 }, { 16406, 0x657F }, { 16407, 0x657C },
    { 16408, 0x65A2 }, { 16409, 0x66C8 }, { 16410, 0x66BE }, { 16411, 0x66C0 }, { 16412, 0x66CA }, { 16413, 0x66CB }, { 16414, 0x66CF }, { 16415, 0x66BD },
    { 16416, 0x66BB }, { 16417, 0x66BA }, { 16418, 0x66CC }, { 16419, 0x6723 }, { 16420, 0x6A34 }, { 16421, 0x6A66 }, { 16422, 0x6A49 }, { 16423, 0x6A67 },
    { 16424, 0x6A32 }, { 16425, 0x6A68 }, { 16426, 0x6A3E }, { 16427, 0x6A5D }, { 16428, 0x6A6D }, { 16429, 0x6A76 }, { 16430, 0x6A5B }, { 16431, 0x6A51 },
    { 16432, 0x6A28 }, { 16433, 0x6A5A }, { 16434, 0x6A3B }, { 16435, 0x6A3F }, { 16436, 0x6A41 }, { 16437, 0x6A6A }, { 16438, 0x6A64 }, { 16439, 0x6A50 },
    { 16440, 0x6A4F }, { 16441, 0x6A54 }, { 16442, 0x6A6F }, { 16443, 0x6A69 }, { 16444, 0x6A60 }, { 16445, 0x6A3C }, { 16446, 0x6A5E }, { 16447, 0x6A56 },
    { 16448, 0x6A55 }, { 16449, 0x6A4D }, { 16450, 0x6A4E }, { 16451, 0x6A46 }, { 16452, 0x6B55 }, { 16453, 0x6B54 }, { 16454, 0x6B56 }, { 16455, 0x6BA7 },
    { 16456, 0x6BAA }, { 16457, 0x6BAB }, { 16458, 0x6BC8 }, { 16459, 0x6BC7 }, { 16460, 0x6C04 }, { 16461, 0x6C03 }, { 16462, 0x6C06 }, { 16463, 0x6FAD },
    { 16464, 0x6FCB }, { 16465, 0x6FA3 }, { 16466, 0x6FC7 }, { 16467, 0x6FBC }, { 16468, 0x6FCE }, { 16469, 0x6FC8 }, { 16470, 0x6F5E }, { 16471, 0x6FC4 },
    { 16472, 0x6FBD }, { 16473, 0x6F9E }, { 16474, 0x6FCA }, { 16475, 0x6FA8 }, { 16476, 0x7004 }, { 16477, 0x6FA5 }, { 16478, 0x6FAE }, { 16479, 0x6FBA },
    { 16480, 0x6FAC }, { 16481, 0x6FAA }, { 16482, 0x6FCF }, { 16483, 0x6FBF }, { 16484, 0x6FB8 }, { 16485, 0x6FA2 }, { 16486, 0x6FC9 }, { 16487, 0x6FAB },
    { 16488, 0x6FCD }, { 16489, 0x6FAF }, { 16490, 0x6FB2 }, { 16491, 0x6FB0 }, { 16492, 0x71C5 }, { 16493, 0x71C2 }, { 16494, 0x71BF }, { 16495, 0x71B8 },
    { 16496, 0x71D6 }, { 16497, 0x71C0 }, { 16498, 0x71C1 }, { 16499, 0x71CB }, { 16500, 0x71D4 }, { 16501, 0x71CA }, { 16502, 0x71C7 }, { 16503, 0x71CF },
    { 16504, 0x71BD }, { 16505, 0x71D8 }, { 16506, 0x71BC }, { 16507, 0x71C6 }, { 16508, 0x71DA }, { 16509, 0x71DB }, { 16510, 0x729D }, { 16511, 0x729E },
    { 16512, 0x7369 }, { 16513, 0x7366 }, { 16514, 0x7367 }, { 16515, 0x736C }, { 16516, 0x7365 }, { 16517, 0x736B }, { 16518, 0x736A }, { 16519, 0x747F },
    { 16520, 0x749A }, { 16521, 0x74A0 }, { 16522, 0x7494 }, { 16523, 0x7492 }, { 16524, 0x7495 }, { 16525, 0x74A1 }, { 16526, 0x750B }, { 16527, 0x7580 },
    { 16528, 0x762F }, { 16529, 0x762D }, { 16530, 0x7631 }, { 16531, 0x763D }, { 16532, 0x7633 }, { 16533, 0x763C }, { 16534, 0x7635 }, { 16535, 0x7632 },
    { 16536, 0x7630 }, { 16537, 0x76BB }, { 16538, 0x76E6 }, { 16539, 0x779A }, { 16540, 0x779D }, { 16541, 0x77A1 }, { 16542, 0x779C }, { 16543, 0x779B },
    { 16544, 0x77A2 }, { 16545, 0x77A3 }, { 16546, 0x7795 }, { 16547, 0x7799 }, { 16548, 0x7797 }, { 16549, 0x78DD }, { 16550, 0x78E9 }, { 16551, 0x78E5 },
    { 16552, 0x78EA }, { 16553, 0x78DE }, { 16554, 0x78E3 }, { 16555, 0x78DB }, { 16556, 0x78E1 }, { 16557, 0x78E2 }, { 16558, 0x78ED }, { 16559, 0x78DF },
    { 16560, 0x78E0 }, { 16561, 0x79A4 }, { 16562, 0x7A44 }, { 16563, 0x7A48 }, { 16564, 0x7A47 }, { 16565, 0x7AB6 }, { 16566, 0x7AB8 }, { 16567, 0x7AB5 },
    { 16568, 0x7AB1 }, { 16569, 0x7AB7 }, { 16570, 0x7BDE }, { 16571, 0x7BE3 }, { 16572, 0x7BE7 }, { 16573, 0x7BDD }, { 16574, 0x7BD5 }, { 16575, 0x7BE5 },
    { 16576, 0x7BDA }, { 16577, 0x7BE8 }, { 16578, 0x7BF9 }, { 16579, 0x7BD4 }, { 16580, 0x7BEA }, { 16581, 0x7BE2 }, { 16582, 0x7BDC }, { 16583, 0x7BEB },
    { 16584, 0x7BD8 }, { 16585, 0x7BDF }, { 16586, 0x7CD2 }, { 16587, 0x7CD4 }, { 16588, 0x7CD7 }, { 16589, 0x7CD0 }, { 16590, 0x7CD1 }, { 16591, 0x7E12 },
    { 16592, 0x7E21 }, { 16593, 0x7E17 }, { 16594, 0x7E0C }, { 16595, 0x7E1F }, { 16596, 0x7E20 }, { 16597, 0x7E13 }, { 16598, 0x7E0E }, { 16599, 0x7E1C },
    { 16600, 0x7E15 }, { 16601, 0x7E1A }, { 16602, 0x7E22 }, { 16603, 0x7E0B }, { 16604, 0x7E0F }, { 16605, 0x7E16 }, { 16606, 0x7E0D }, { 16607, 0x7E14 },
    { 16608, 0x7E25 }, { 16609, 0x7E24 }, { 16610, 0x7F43 }, { 16611, 0x7F7B }, { 16612, 0x7F7C }, { 16613, 0x7F7A }, { 16614, 0x7FB1 }, { 16615, 0x7FEF },
    { 16616, 0x802A }, { 16617, 0x8029 }, { 16618, 0x806C }, { 16619, 0x81B1 }, { 16620, 0x81A6 }, { 16621, 0x81AE }, { 16622, 0x81B9 }, { 16623, 0x81B5 },
    { 16624, 0x81AB }, { 16625, 0x81B0 }, { 16626, 0x81AC }, { 16627, 0x81B4 }, { 16628, 0x81B2 }, { 16629, 0x81B7 }, { 16630, 0x81A7 }, { 16631, 0x81F2 },
    { 16632, 0x8255 }, { 16633, 0x8256 }, { 16634, 0x8257 }, { 16635, 0x8556 }, { 16636, 0x8545 }, { 16637, 0x856B }, { 16638, 0x854D }, { 16639, 0x8553 },
    { 16640, 0x8561 }, { 16641, 0x8558 }, { 16642, 0x8540 }, { 16643, 0x8546 }, { 16644, 0x8564 }, { 16645, 0x8541 }, { 16646, 0x8562 }, { 16647, 0x8544 },
    { 16648, 0x8551 }, { 16649, 0x8547 }, { 16650, 0x8563 }, { 16651, 0x853E }, { 16652, 0x855B }, { 16653, 0x8571 }, { 16654, 0x854E }, { 16655, 0x856E },
    { 16656, 0x8575 }, { 16657, 0x8555 }, { 16658, 0x8567 }, { 16659, 0x8560 }, { 16660, 0x858C }, { 16661, 0x8566 }, { 16662, 0x855D }, { 16663, 0x8554 },
    { 16664, 0x8565 }, { 16665, 0x856C }, { 16666, 0x8663 }, { 16667, 0x8665 }, { 16668, 0x8664 }, { 16669, 0x879B }, { 16670, 0x878F }, { 16671, 0x8797 },
    { 16672, 0x8793 }, { 16673, 0x8792 }, { 16674, 0x8788 }, { 16675, 0x8781 }, { 16676, 0x8796 }, { 16677, 0x8798 }, { 16678, 0x8779 }, { 16679, 0x8787 },
    { 16680, 0x87A3 }, { 16681, 0x8785 }, { 16682, 0x8790 }, { 16683, 0x8791 }, { 16684, 0x879D }, { 16685, 0x8784 }, { 16686, 0x8794 }, { 16687, 0x879C },
    { 16688, 0x879A }, { 16689, 0x8789 }, { 16690, 0x891E }, { 16691, 0x8926 }, { 16692, 0x8930 }, { 16693, 0x892D }, { 16694, 0x892E }, { 16695, 0x8927 },
    { 16696, 0x8931 }, { 16697, 0x8922 }, { 16698, 0x8929 }, { 16699, 0x8923 }, { 16700, 0x892F }, { 16701, 0x892C }, { 16702, 0x891F }, { 16703, 0x89F1 },
    { 16704, 0x8AE0 }, { 16705, 0x8AE2 }, { 16706, 0x8AF2 }, { 16707, 0x8AF4 }, { 16708, 0x8AF5 }, { 16709, 0x8ADD }, { 16710, 0x8B14 }, { 16711, 0x8AE4 },
    { 16712, 0x8ADF }, { 16713, 0x8AF0 }, { 16714, 0x8AC8 }, { 16715, 0x8ADE }, { 16716, 0x8AE1 }, { 16717, 0x8AE8 }, { 16718, 0x8AFF }, { 16719, 0x8AEF },
    { 16720, 0x8AFB }, { 16721, 0x8C91 }, { 16722, 0x8C92 }, { 16723, 0x8C90 }, { 16724, 0x8CF5 }, { 16725, 0x8CEE }, { 16726, 0x8CF1 }, { 16727, 0x8CF0 },
    { 16728, 0x8CF3 }, { 16729, 0x8D6C }, { 16730, 0x8D6E }, { 16731, 0x8DA5 }, { 16732, 0x8DA7 }, { 16733, 0x8E33 }, { 16734, 0x8E3E }, { 16735, 0x8E38 },
    { 16736, 0x8E40 }, { 16737, 0x8E45 }, { 16738, 0x8E36 }, { 16739, 0x8E3C }, { 16740, 0x8E3D }, { 16741, 0x8E41 }, { 16742, 0x8E30 }, { 16743, 0x8E3F },
    { 16744, 0x8EBD }, { 16745, 0x8F36 }, { 16746, 0x8F2E }, { 16747, 0x8F35 }, { 16748, 0x8F32 }, { 16749, 0x8F39 }, { 16750, 0x8F37 }, { 16751, 0x8F34 },
    { 16752, 0x9076 }, { 16753, 0x9079 }, { 16754, 0x907B }, { 16755, 0x9086 }, { 16756, 0x90FA }, { 16757, 0x9133 }, { 16758, 0x9135 }, { 16759, 0x9136 },
    { 16760, 0x9193 }, { 16761, 0x9190 }, { 16762, 0x9191 }, { 16763, 0x918D }, { 16764, 0x918F }, { 16765, 0x9327 }, { 16766, 0x931E }, { 16767, 0x9308 },
    { 16768, 0x931F }, { 16769, 0x9306 }, { 16770, 0x930F }, { 16771, 0x937A }, { 16772, 0x9338 }, { 16773, 0x933C }, { 16774, 0x931B }, { 16775, 0x9323 },
    { 16776, 0x9312 }, { 16777, 0x9301 }, { 16778, 0x9346 }, { 16779, 0x932D }, { 16780, 0x930E }, { 16781, 0x930D }, { 16782, 0x92CB }, { 16783, 0x931D },
    { 16784, 0x92FA }, { 16785, 0x9325 }, { 16786, 0x9313 }, { 16787, 0x92F9 }, { 16788, 0x92F7 }, { 16789, 0x9334 }, { 16790, 0x9302 }, { 16791, 0x9324 },
    { 16792, 0x92FF }, { 16793, 0x9329 }, { 16794, 0x9339 }, { 16795, 0x9335 }, { 16796, 0x932A }, { 16797, 0x9314 }, { 16798, 0x930C }, { 16799, 0x930B },
    { 16800, 0x92FE }, { 16801, 0x9309 }, { 16802, 0x9300 }, { 16803, 0x92FB }, { 16804, 0x9316 }, { 16805, 0x95BC }, { 16806, 0x95CD }, { 16807, 0x95BE },
    { 16808, 0x95B9 }, { 16809, 0x95BA }, { 16810, 0x95B6 }, { 16811, 0x95BF }, { 16812, 0x95B5 }, { 16813, 0x95BD }, { 16814, 0x96A9 }, { 16815, 0x96D4 },
    { 16816, 0x970B }, { 16817, 0x9712 }, { 16818, 0x9710 }, { 16819, 0x9799 }, { 16820, 0x9797 }, { 16821, 0x9794 }, { 16822, 0x97F0 }, { 16823, 0x97F8 },
    { 16824, 0x9835 }, { 16825, 0x982F }, { 16826, 0x9832 }, { 16827, 0x9924 }, { 16828, 0x991F }, { 16829, 0x9927 }, { 16830, 0x9929 }, { 16831, 0x999E },
    { 16832, 0x99EE }, { 16833, 0x99EC }, { 16834, 0x99E5 }, { 16835, 0x99E4 }, { 16836, 0x99F0 }, { 16837, 0x99E3 }, { 16838, 0x99EA }, { 16839, 0x99E9 },
    { 16840, 0x99E7 }, { 16841, 0x9AB9 }, { 16842, 0x9ABF }, { 16843, 0x9AB4 }, { 16844, 0x9ABB }, { 16845, 0x9AF6 }, { 16846, 0x9AFA }, { 16847, 0x9AF9 },
    { 16848, 0x9AF7 }, { 16849, 0x9B33 }, { 16850, 0x9B80 }, { 16851, 0x9B85 }, { 16852, 0x9B87 }, { 16853, 0x9B7C }, { 16854, 0x9B7E }, { 16855, 0x9B7B },
    { 16856, 0x9B82 }, { 16857, 0x9B93 }, { 16858, 0x9B92 }, { 16859, 0x9B90 }, { 16860, 0x9B7A }, { 16861, 0x9B95 }, { 16862, 0x9B7D }, { 16863, 0x9B88 },
    { 16864, 0x9D25 }, { 16865, 0x9D17 }, { 16866, 0x9D20 }, { 16867, 0x9D1E }, { 16868, 0x9D14 }, { 16869, 0x9D29 }, { 16870, 0x9D1D }, { 16871, 0x9D18 },
    { 16872, 0x9D22 }, { 16873, 0x9D10 }, { 16874, 0x9D19 }, { 16875, 0x9D1F }, { 16876, 0x9E88 }, { 16877, 0x9E86 }, { 16878, 0x9E87 }, { 16879, 0x9EAE },
    { 16880, 0x9EAD }, { 16881, 0x9ED5 }, { 16882, 0x9ED6 }, { 16883, 0x9EFA }, { 16884, 0x9F12 }, { 16885, 0x9F3D }, { 16886, 0x5126 }, { 16887, 0x5125 },
    { 16888, 0x5122 }, { 16889, 0x5124 }, { 16890, 0x5120 }, { 16891, 0x5129 }, { 16892, 0x52F4 }, { 16893, 0x5693 }, { 16894, 0x568C }, { 16895, 0x568D },
    { 16896, 0x5686 }, { 16897, 0x5684 }, { 16898, 0x5683 }, { 16899, 0x567E }, { 16900, 0x5682 }, { 16901, 0x567F }, { 16902, 0x5681 }, { 16903, 0x58D6 },
    { 16904, 0x58D4 }, { 16905, 0x58CF }, { 16906, 0x58D2 }, { 16907, 0x5B2D }, { 16908, 0x5B25 }, { 16909, 0x5B32 }, { 16910, 0x5B23 }, { 16911, 0x5B2C },
    { 16912, 0x5B27 }, { 16913, 0x5B26 }, { 16914, 0x5B2F }, { 16915, 0x5B2E }, { 16916, 0x5B7B }, { 16917, 0x5BF1 }, { 16918, 0x5BF2 }, { 16919, 0x5DB7 },
    { 16920, 0x5E6C }, { 16921, 0x5E6A }, { 16922, 0x5FBE }, { 16923, 0x5FBB }, { 16924, 0x61C3 }, { 16925, 0x61B5 }, { 16926, 0x61BC }, { 16927, 0x61E7 },
    { 16928, 0x61E0 }, { 16929, 0x61E5 }, { 16930, 0x61E4 }, { 16931, 0x61E8 }, { 16932, 0x61DE }, { 16933, 0x64EF }, { 16934, 0x64E9 }, { 16935, 0x64E3 },
    { 16936, 0x64EB }, { 16937, 0x64E4 }, { 16938, 0x64E8 }, { 16939, 0x6581 }, { 16940, 0x6580 }, { 16941, 0x65B6 }, { 16942, 0x65DA }, { 16943, 0x66D2 },
    { 16944, 0x6A8D }, { 16945, 0x6A96 }, { 16946, 0x6A81 }, { 16947, 0x6AA5 }, { 16948, 0x6A89 }, { 16949, 0x6A9F }, { 16950, 0x6A9B }, { 16951, 0x6AA1 },
    { 16952, 0x6A9E }, { 16953, 0x6A87 }, { 16954, 0x6A93 }, { 16955, 0x6A8E }, { 16956, 0x6A95 }, { 16957, 0x6A83 }, { 16958, 0x6AA8 }, { 16959, 0x6AA4 },
    { 16960, 0x6A91 }, { 16961, 0x6A7F }, { 16962, 0x6AA6 }, { 16963, 0x6A9A }, { 16964, 0x6A85 }, { 16965, 0x6A8C }, { 16966, 0x6A92 }, { 16967, 0x6B5B },
    { 16968, 0x6BAD }, { 16969, 0x6C09 }, { 16970, 0x6FCC }, { 16971, 0x6FA9 }, { 16972, 0x6FF4 }, { 16973, 0x6FD4 }, { 16974, 0x6FE3 }, { 16975, 0x6FDC },
    { 16976, 0x6FED }, { 16977, 0x6FE7 }, { 16978, 0x6FE6 }, { 16979, 0x6FDE }, { 16980, 0x6FF2 }, { 16981, 0x6FDD }, { 16982, 0x6FE2 }, { 16983, 0x6FE8 },
    { 16984, 0x71E1 }, { 16985, 0x71F1 }, { 16986, 0x71E8 }, { 16987, 0x71F2 }, { 16988, 0x71E4 }, { 16989, 0x71F0 }, { 16990, 0x71E2 }, { 16991, 0x7373 },
    { 16992, 0x736E }, { 16993, 0x736F }, { 16994, 0x7497 }, { 16995, 0x74B2 }, { 16996, 0x74AB }, { 16997, 0x7490 }, { 16998, 0x74AA }, { 16999, 0x74AD },
    { 17000, 0x74B1 }, { 17001, 0x74A5 }, { 17002, 0x74AF }, { 17003, 0x7510 }, { 17004, 0x7511 }, { 17005, 0x7512 }, { 17006, 0x750F }, { 17007, 0x7584 },
    { 17008, 0x7643 }, { 17009, 0x7648 }, { 17010, 0x7649 }, { 17011, 0x7647 }, { 17012, 0x76A4 }, { 17013, 0x76E9 }, { 17014, 0x77B5 }, { 17015, 0x77AB },
    { 17016, 0x77B2 }, { 17017, 0x77B7 }, { 17018, 0x77B6 }, { 17019, 0x77B4 }, { 17020, 0x77B1 }, { 17021, 0x77A8 }, { 17022, 0x77F0 }, { 17023, 0x78F3 },
    { 17024, 0x78FD }, { 17025, 0x7902 }, { 17026, 0x78FB }, { 17027, 0x78FC }, { 17028, 0x78F2 }, { 17029, 0x7905 }, { 17030, 0x78F9 }, { 17031, 0x78FE },
    { 17032, 0x7904 }, { 17033, 0x79AB }, { 17034, 0x79A8 }, { 17035, 0x7A5C }, { 17036, 0x7A5B }, { 17037, 0x7A56 }, { 17038, 0x7A58 }, { 17039, 0x7A54 },
    { 17040, 0x7A5A }, { 17041, 0x7ABE }, { 17042, 0x7AC0 }, { 17043, 0x7AC1 }, { 17044, 0x7C05 }, { 17045, 0x7C0F }, { 17046, 0x7BF2 }, { 17047, 0x7C00 },
    { 17048, 0x7BFF }, { 17049, 0x7BFB }, { 17050, 0x7C0E }, { 17051, 0x7BF4 }, { 17052, 0x7C0B }, { 17053, 0x7BF3 }, { 17054, 0x7C02 }, { 17055, 0x7C09 },
    { 17056, 0x7C03 }, { 17057, 0x7C01 }, { 17058, 0x7BF8 }, { 17059, 0x7BFD }, { 17060, 0x7C06 }, { 17061, 0x7BF0 }, { 17062, 0x7BF1 }, { 17063, 0x7C10 },
    { 17064, 0x7C0A }, { 17065, 0x7CE8 }, { 17066, 0x7E2D }, { 17067, 0x7E3C }, { 17068, 0x7E42 }, { 17069, 0x7E33 }, { 17070, 0x9848 }, { 17071, 0x7E38 },
    { 17072, 0x7E2A }, { 17073, 0x7E49 }, { 17074, 0x7E40 }, { 17075, 0x7E47 }, { 17076, 0x7E29 }, { 17077, 0x7E4C }, { 17078, 0x7E30 }, { 17079, 0x7E3B },
    { 17080, 0x7E36 }, { 17081, 0x7E44 }, { 17082, 0x7E3A }, { 17083, 0x7F45 }, { 17084, 0x7F7F }, { 17085, 0x7F7E }, { 17086, 0x7F7D }, { 17087, 0x7FF4 },
    { 17088, 0x7FF2 }, { 17089, 0x802C }, { 17090, 0x81BB }, { 17091, 0x81C4 }, { 17092, 0x81CC }, { 17093, 0x81CA }, { 17094, 0x81C5 }, { 17095, 0x81C7 },
    { 17096, 0x81BC }, { 17097, 0x81E9 }, { 17098, 0x825B }, { 17099, 0x825A }, { 17100, 0x825C }, { 17101, 0x8583 }, { 17102, 0x8580 }, { 17103, 0x858F },
    { 17104, 0x85A7 }, { 17105, 0x8595 }, { 17106, 0x85A0 }, { 17107, 0x858B }, { 17108, 0x85A3 }, { 17109, 0x857B }, { 17110, 0x85A4 }, { 17111, 0x859A },
    { 17112, 0x859E }, { 17113, 0x8577 }, { 17114, 0x857C }, { 17115, 0x8589 }, { 17116, 0x85A1 }, { 17117, 0x857A }, { 17118, 0x8578 }, { 17119, 0x8557 },
    { 17120, 0x858E }, { 17121, 0x8596 }, { 17122, 0x8586 }, { 17123, 0x858D }, { 17124, 0x8599 }, { 17125, 0x859D }, { 17126, 0x8581 }, { 17127, 0x85A2 },
    { 17128, 0x8582 }, { 17129, 0x8588 }, { 17130, 0x8585 }, { 17131, 0x8579 }, { 17132, 0x8576 }, { 17133, 0x8598 }, { 17134, 0x8590 }, { 17135, 0x859F },
    { 17136, 0x8668 }, { 17137, 0x87BE }, { 17138, 0x87AA }, { 17139, 0x87AD }, { 17140, 0x87C5 }, { 17141, 0x87B0 }, { 17142, 0x87AC }, { 17143, 0x87B9 },
    { 17144, 0x87B5 }, { 17145, 0x87BC }, { 17146, 0x87AE }, { 17147, 0x87C9 }, { 17148, 0x87C3 }, { 17149, 0x87C2 }, { 17150, 0x87CC }, { 17151, 0x87B7 },
    { 17152, 0x87AF }, { 17153, 0x87C4 }, { 17154, 0x87CA }, { 17155, 0x87B4 }, { 17156, 0x87B6 }, { 17157, 0x87BF }, { 17158, 0x87B8 }, { 17159, 0x87BD },
    { 17160, 0x87DE }, { 17161, 0x87B2 }, { 17162, 0x8935 }, { 17163, 0x8933 }, { 17164, 0x893C }, { 17165, 0x893E }, { 17166, 0x8941 }, { 17167, 0x8952 },
    { 17168, 0x8937 }, { 17169, 0x8942 }, { 17170, 0x89AD }, { 17171, 0x89AF }, { 17172, 0x89AE }, { 17173, 0x89F2 }, { 17174, 0x89F3 }, { 17175, 0x8B1E },
    { 17176, 0x8B18 }, { 17177, 0x8B16 }, { 17178, 0x8B11 }, { 17179, 0x8B05 }, { 17180, 0x8B0B }, { 17181, 0x8B22 }, { 17182, 0x8B0F }, { 17183, 0x8B12 },
    { 17184, 0x8B15 }, { 17185, 0x8B07 }, { 17186, 0x8B0D }, { 17187, 0x8B08 }, { 17188, 0x8B06 }, { 17189, 0x8B1C }, { 17190, 0x8B13 }, { 17191, 0x8B1A },
    { 17192, 0x8C4F }, { 17193, 0x8C70 }, { 17194, 0x8C72 }, { 17195, 0x8C71 }, { 17196, 0x8C6F }, { 17197, 0x8C95 }, { 17198, 0x8C94 }, { 17199, 0x8CF9 },
    { 17200, 0x8D6F }, { 17201, 0x8E4E }, { 17202, 0x8E4D }, { 17203, 0x8E53 }, { 17204, 0x8E50 }, { 17205, 0x8E4C }, { 17206, 0x8E47 }, { 17207, 0x8F43 },
    { 17208, 0x8F40 }, { 17209, 0x9085 }, { 17210, 0x907E }, { 17211, 0x9138 }, { 17212, 0x919A }, { 17213, 0x91A2 }, { 17214, 0x919B }, { 17215, 0x9199 },
    { 17216, 0x919F }, { 17217, 0x91A1 }, { 17218, 0x919D }, { 17219, 0x91A0 }, { 17220, 0x93A1 }, { 17221, 0x9383 }, { 17222, 0x93AF }, { 17223, 0x9364 },
    { 17224, 0x9356 }, { 17225, 0x9347 }, { 17226, 0x937C }, { 17227, 0x9358 }, { 17228, 0x935C }, { 17229, 0x9376 }, { 17230, 0x9349 }, { 17231, 0x9350 },
    { 17232, 0x9351 }, { 17233, 0x9360 }, { 17234, 0x936D }, { 17235, 0x938F }, { 17236, 0x934C }, { 17237, 0x936A }, { 17238, 0x9379 }, { 17239, 0x9357 },
    { 17240, 0x9355 }, { 17241, 0x9352 }, { 17242, 0x934F }, { 17243, 0x9371 }, { 17244, 0x9377 }, { 17245, 0x937B }, { 17246, 0x9361 }, { 17247, 0x935E },
    { 17248, 0x9363 }, { 17249, 0x9367 }, { 17250, 0x9380 }, { 17251, 0x934E }, { 17252, 0x9359 }, { 17253, 0x95C7 }, { 17254, 0x95C0 }, { 17255, 0x95C9 },
    { 17256, 0x95C3 }, { 17257, 0x95C5 }, { 17258, 0x95B7 }, { 17259, 0x96AE }, { 17260, 0x96B0 }, { 17261, 0x96AC }, { 17262, 0x9720 }, { 17263, 0x971F },
    { 17264, 0x9718 }, { 17265, 0x971D }, { 17266, 0x9719 }, { 17267, 0x979A }, { 17268, 0x97A1 }, { 17269, 0x979C }, { 17270, 0x979E }, { 17271, 0x979D },
    { 17272, 0x97D5 }, { 17273, 0x97D4 }, { 17274, 0x97F1 }, { 17275, 0x9841 }, { 17276, 0x9844 }, { 17277, 0x984A }, { 17278, 0x9849 }, { 17279, 0x9845 },
    { 17280, 0x9843 }, { 17281, 0x9925 }, { 17282, 0x992B }, { 17283, 0x992C }, { 17284, 0x992A }, { 17285, 0x9933 }, { 17286, 0x9932 }, { 17287, 0x992F },
    { 17288, 0x992D }, { 17289, 0x9931 }, { 17290, 0x9930 }, { 17291, 0x9998 }, { 17292, 0x99A3 }, { 17293, 0x99A1 }, { 17294, 0x9A02 }, { 17295, 0x99FA },
    { 17296, 0x99F4 }, { 17297, 0x99F7 }, { 17298, 0x99F9 }, { 17299, 0x99F8 }, { 17300, 0x99F6 }, { 17301, 0x99FB }, { 17302, 0x99FD }, { 17303, 0x99FE },
    { 17304, 0x99FC }, { 17305, 0x9A03 }, { 17306, 0x9ABE }, { 17307, 0x9AFE }, { 17308, 0x9AFD }, { 17309, 0x9B01 }, { 17310, 0x9AFC }, { 17311, 0x9B48 },
    { 17312, 0x9B9A }, { 17313, 0x9BA8 }, { 17314, 0x9B9E }, { 17315, 0x9B9B }, { 17316, 0x9BA6 }, { 17317, 0x9BA1 }, { 17318, 0x9BA5 }, { 17319, 0x9BA4 },
    { 17320, 0x9B86 }, { 17321, 0x9BA2 }, { 17322, 0x9BA0 }, { 17323, 0x9BAF }, { 17324, 0x9D33 }, { 17325, 0x9D41 }, { 17326, 0x9D67 }, { 17327, 0x9D36 },
    { 17328, 0x9D2E }, { 17329, 0x9D2F }, { 17330, 0x9D31 }, { 17331, 0x9D38 }, { 17332, 0x9D30 }, { 17333, 0x9D45 }, { 17334, 0x9D42 }, { 17335, 0x9D43 },
    { 17336, 0x9D3E }, { 17337, 0x9D37 }, { 17338, 0x9D40 }, { 17339, 0x9D3D }, { 17340, 0x7FF5 }, { 17341, 0x9D2D }, { 17342, 0x9E8A }, { 17343, 0x9E89 },
    { 17344, 0x9E8D }, { 17345, 0x9EB0 }, { 17346, 0x9EC8 }, { 17347, 0x9EDA }, { 17348, 0x9EFB }, { 17349, 0x9EFF }, { 17350, 0x9F24 }, { 17351, 0x9F23 },
    { 17352, 0x9F22 }, { 17353, 0x9F54 }, { 17354, 0x9FA0 }, { 17355, 0x5131 }, { 17356, 0x512D }, { 17357, 0x512E }, { 17358, 0x5698 }, { 17359, 0x569C },
    { 17360, 0x5697 }, { 17361, 0x569A }, { 17362, 0x569D }, { 17363, 0x5699 }, { 17364, 0x5970 }, { 17365, 0x5B3C }, { 17366, 0x5C69 }, { 17367, 0x5C6A },
    { 17368, 0x5DC0 }, { 17369, 0x5E6D }, { 17370, 0x5E6E }, { 17371, 0x61D8 }, { 17372, 0x61DF }, { 17373, 0x61ED }, { 17374, 0x61EE }, { 17375, 0x61F1 },
    { 17376, 0x61EA }, { 17377, 0x61F0 }, { 17378, 0x61EB }, { 17379, 0x61D6 }, { 17380, 0x61E9 }, { 17381, 0x64FF }, { 17382, 0x6504 }, { 17383, 0x64FD },
    { 17384, 0x64F8 }, { 17385, 0x6501 }, { 17386, 0x6503 }, { 17387, 0x64FC }, { 17388, 0x6594 }, { 17389, 0x65DB }, { 17390, 0x66DA }, { 17391, 0x66DB },
    { 17392, 0x66D8 }, { 17393, 0x6AC5 }, { 17394, 0x6AB9 }, { 17395, 0x6ABD }, { 17396, 0x6AE1 }, { 17397, 0x6AC6 }, { 17398, 0x6ABA }, { 17399, 0x6AB6 },
    { 17400, 0x6AB7 }, { 17401, 0x6AC7 }, { 17402, 0x6AB4 }, { 17403, 0x6AAD }, { 17404, 0x6B5E }, { 17405, 0x6BC9 }, { 17406, 0x6C0B }, { 17407, 0x7007 },
    { 17408, 0x700C }, { 17409, 0x700D }, { 17410, 0x7001 }, { 17411, 0x7005 }, { 17412, 0x7014 }, { 17413, 0x700E }, { 17414, 0x6FFF }, { 17415, 0x7000 },
    { 17416, 0x6FFB }, { 17417, 0x7026 }, { 17418, 0x6FFC }, { 17419, 0x6FF7 }, { 17420, 0x700A }, { 17421, 0x7201 }, { 17422, 0x71FF }, { 17423, 0x71F9 },
    { 17424, 0x7203 }, { 17425, 0x71FD }, { 17426, 0x7376 }, { 17427, 0x74B8 }, { 17428, 0x74C0 }, { 17429, 0x74B5 }, { 17430, 0x74C1 }, { 17431, 0x74BE },
    { 17432, 0x74B6 }, { 17433, 0x74BB }, { 17434, 0x74C2 }, { 17435, 0x7514 }, { 17436, 0x7513 }, { 17437, 0x765C }, { 17438, 0x7664 }, { 17439, 0x7659 },
    { 17440, 0x7650 }, { 17441, 0x7653 }, { 17442, 0x7657 }, { 17443, 0x765A }, { 17444, 0x76A6 }, { 17445, 0x76BD }, { 17446, 0x76EC }, { 17447, 0x77C2 },
    { 17448, 0x77BA }, { 17449, 0x78FF }, { 17450, 0x790C }, { 17451, 0x7913 }, { 17452, 0x7914 }, { 17453, 0x7909 }, { 17454, 0x7910 }, { 17455, 0x7912 },
    { 17456, 0x7911 }, { 17457, 0x79AD }, { 17458, 0x79AC }, { 17459, 0x7A5F }, { 17460, 0x7C1C }, { 17461, 0x7C29 }, { 17462, 0x7C19 }, { 17463, 0x7C20 },
    { 17464, 0x7C1F }, { 17465, 0x7C2D }, { 17466, 0x7C1D }, { 17467, 0x7C26 }, { 17468, 0x7C28 }, { 17469, 0x7C22 }, { 17470, 0x7C25 }, { 17471, 0x7C30 },
    { 17472, 0x7E5C }, { 17473, 0x7E50 }, { 17474, 0x7E56 }, { 17475, 0x7E63 }, { 17476, 0x7E58 }, { 17477, 0x7E62 }, { 17478, 0x7E5F }, { 17479, 0x7E51 },
    { 17480, 0x7E60 }, { 17481, 0x7E57 }, { 17482, 0x7E53 }, { 17483, 0x7FB5 }, { 17484, 0x7FB3 }, { 17485, 0x7FF7 }, { 17486, 0x7FF8 }, { 17487, 0x8075 },
    { 17488, 0x81D1 }, { 17489, 0x81D2 }, { 17490, 0x81D0 }, { 17491, 0x825F }, { 17492, 0x825E }, { 17493, 0x85B4 }, { 17494, 0x85C6 }, { 17495, 0x85C0 },
    { 17496, 0x85C3 }, { 17497, 0x85C2 }, { 17498, 0x85B3 }, { 17499, 0x85B5 }, { 17500, 0x85BD }, { 17501, 0x85C7 }, { 17502, 0x85C4 }, { 17503, 0x85BF },
    { 17504, 0x85CB }, { 17505, 0x85CE }, { 17506, 0x85C8 }, { 17507, 0x85C5 }, { 17508, 0x85B1 }, { 17509, 0x85B6 }, { 17510, 0x85D2 }, { 17511, 0x8624 },
    { 17512, 0x85B8 }, { 17513, 0x85B7 }, { 17514, 0x85BE }, { 17515, 0x8669 }, { 17516, 0x87E7 }, { 17517, 0x87E6 }, { 17518, 0x87E2 }, { 17519, 0x87DB },
    { 17520, 0x87EB }, { 17521, 0x87EA }, { 17522, 0x87E5 }, { 17523, 0x87DF }, { 17524, 0x87F3 }, { 17525, 0x87E4 }, { 17526, 0x87D4 }, { 17527, 0x87DC },
    { 17528, 0x87D3 }, { 17529, 0x87ED }, { 17530, 0x87D8 }, { 17531, 0x87E3 }, { 17532, 0x87A4 }, { 17533, 0x87D7 }, { 17534, 0x87D9 }, { 17535, 0x8801 },
    { 17536, 0x87F4 }, { 17537, 0x87E8 }, { 17538, 0x87DD }, { 17539, 0x8953 }, { 17540, 0x894B }, { 17541, 0x894F }, { 17542, 0x894C }, { 17543, 0x8946 },
    { 17544, 0x8950 }, { 17545, 0x8951 }, { 17546, 0x8949 }, { 17547, 0x8B2A }, { 17548, 0x8B27 }, { 17549, 0x8B23 }, { 17550, 0x8B33 }, { 17551, 0x8B30 },
    { 17552, 0x8B35 }, { 17553, 0x8B47 }, { 17554, 0x8B2F }, { 17555, 0x8B3C }, { 17556, 0x8B3E }, { 17557, 0x8B31 }, { 17558, 0x8B25 }, { 17559, 0x8B37 },
    { 17560, 0x8B26 }, { 17561, 0x8B36 }, { 17562, 0x8B2E }, { 17563, 0x8B24 }, { 17564, 0x8B3B }, { 17565, 0x8B3D }, { 17566, 0x8B3A }, { 17567, 0x8C42 },
    { 17568, 0x8C75 }, { 17569, 0x8C99 }, { 17570, 0x8C98 }, { 17571, 0x8C97 }, { 17572, 0x8CFE }, { 17573, 0x8D04 }, { 17574, 0x8D02 }, { 17575, 0x8D00 },
    { 17576, 0x8E5C }, { 17577, 0x8E62 }, { 17578, 0x8E60 }, { 17579, 0x8E57 }, { 17580, 0x8E56 }, { 17581, 0x8E5E }, { 17582, 0x8E65 }, { 17583, 0x8E67 },
    { 17584, 0x8E5B }, { 17585, 0x8E5A }, { 17586, 0x8E61 }, { 17587, 0x8E5D }, { 17588, 0x8E69 }, { 17589, 0x8E54 }, { 17590, 0x8F46 }, { 17591, 0x8F47 },
    { 17592, 0x8F48 }, { 17593, 0x8F4B }, { 17594, 0x9128 }, { 17595, 0x913A }, { 17596, 0x913B }, { 17597, 0x913E }, { 17598, 0x91A8 }, { 17599, 0x91A5 },
    { 17600, 0x91A7 }, { 17601, 0x91AF }, { 17602, 0x91AA }, { 17603, 0x93B5 }, { 17604, 0x938C }, { 17605, 0x9392 }, { 17606, 0x93B7 }, { 17607, 0x939B },
    { 17608, 0x939D }, { 17609, 0x9389 }, { 17610, 0x93A7 }, { 17611, 0x938E }, { 17612, 0x93AA }, { 17613, 0x939E }, { 17614, 0x93A6 }, { 17615, 0x9395 },
    { 17616, 0x9388 }, { 17617, 0x9399 }, { 17618, 0x939F }, { 17619, 0x938D }, { 17620, 0x93B1 }, { 17621, 0x9391 }, { 17622, 0x93B2 }, { 17623, 0x93A4 },
    { 17624, 0x93A8 }, { 17625, 0x93B4 }, { 17626, 0x93A3 }, { 17627, 0x93A5 }, { 17628, 0x95D2 }, { 17629, 0x95D3 }, { 17630, 0x95D1 }, { 17631, 0x96B3 },
    { 17632, 0x96D7 }, { 17633, 0x96DA }, { 17634, 0x5DC2 }, { 17635, 0x96DF }, { 17636, 0x96D8 }, { 17637, 0x96DD }, { 17638, 0x9723 }, { 17639, 0x9722 },
    { 17640, 0x9725 }, { 17641, 0x97AC }, { 17642, 0x97AE }, { 17643, 0x97A8 }, { 17644, 0x97AB }, { 17645, 0x97A4 }, { 17646, 0x97AA }, { 17647, 0x97A2 },
    { 17648, 0x97A5 }, { 17649, 0x97D7 }, { 17650, 0x97D9 }, { 17651, 0x97D6 }, { 17652, 0x97D8 }, { 17653, 0x97FA }, { 17654, 0x9850 }, { 17655, 0x9851 },
    { 17656, 0x9852 }, { 17657, 0x98B8 }, { 17658, 0x9941 }, { 17659, 0x993C }, { 17660, 0x993A }, { 17661, 0x9A0F }, { 17662, 0x9A0B }, { 17663, 0x9A09 },
    { 17664, 0x9A0D }, { 17665, 0x9A04 }, { 17666, 0x9A11 }, { 17667, 0x9A0A }, { 17668, 0x9A05 }, { 17669, 0x9A07 }, { 17670, 0x9A06 }, { 17671, 0x9AC0 },
    { 17672, 0x9ADC }, { 17673, 0x9B08 }, { 17674, 0x9B04 }, { 17675, 0x9B05 }, { 17676, 0x9B29 }, { 17677, 0x9B35 }, { 17678, 0x9B4A }, { 17679, 0x9B4C },
    { 17680, 0x9B4B }, { 17681, 0x9BC7 }, { 17682, 0x9BC6 }, { 17683, 0x9BC3 }, { 17684, 0x9BBF }, { 17685, 0x9BC1 }, { 17686, 0x9BB5 }, { 17687, 0x9BB8 },
    { 17688, 0x9BD3 }, { 17689, 0x9BB6 }, { 17690, 0x9BC4 }, { 17691, 0x9BB9 }, { 17692, 0x9BBD }, { 17693, 0x9D5C }, { 17694, 0x9D53 }, { 17695, 0x9D4F },
    { 17696, 0x9D4A }, { 17697, 0x9D5B }, { 17698, 0x9D4B }, { 17699, 0x9D59 }, { 17700, 0x9D56 }, { 17701, 0x9D4C }, { 17702, 0x9D57 }, { 17703, 0x9D52 },
    { 17704, 0x9D54 }, { 17705, 0x9D5F }, { 17706, 0x9D58 }, { 17707, 0x9D5A }, { 17708, 0x9E8E }, { 17709, 0x9E8C }, { 17710, 0x9EDF }, { 17711, 0x9F01 },
    { 17712, 0x9F00 }, { 17713, 0x9F16 }, { 17714, 0x9F25 }, { 17715, 0x9F2B }, { 17716, 0x9F2A }, { 17717, 0x9F29 }, { 17718, 0x9F28 }, { 17719, 0x9F4C },
    { 17720, 0x9F55 }, { 17721, 0x5134 }, { 17722, 0x5135 }, { 17723, 0x5296 }, { 17724, 0x52F7 }, { 17725, 0x53B4 }, { 17726, 0x56AB }, { 17727, 0x56AD },
    { 17728, 0x56A6 }, { 17729, 0x56A7 }, { 17730, 0x56AA }, { 17731, 0x56AC }, { 17732, 0x58DA }, { 17733, 0x58DD }, { 17734, 0x58DB }, { 17735, 0x5912 },
    { 17736, 0x5B3D }, { 17737, 0x5B3E }, { 17738, 0x5B3F }, { 17739, 0x5DC3 }, { 17740, 0x5E70 }, { 17741, 0x5FBF }, { 17742, 0x61FB }, { 17743, 0x6507 },
    { 17744, 0x6510 }, { 17745, 0x650D }, { 17746, 0x6509 }, { 17747, 0x650C }, { 17748, 0x650E }, { 17749, 0x6584 }, { 17750, 0x65DE }, { 17751, 0x65DD },
    { 17752, 0x66DE }, { 17753, 0x6AE7 }, { 17754, 0x6AE0 }, { 17755, 0x6ACC }, { 17756, 0x6AD1 }, { 17757, 0x6AD9 }, { 17758, 0x6ACB }, { 17759, 0x6ADF },
    { 17760, 0x6ADC }, { 17761, 0x6AD0 }, { 17762, 0x6AEB }, { 17763, 0x6ACF }, { 17764, 0x6ACD }, { 17765, 0x6ADE }, { 17766, 0x6B60 }, { 17767, 0x6BB0 },
    { 17768, 0x6C0C }, { 17769, 0x7019 }, { 17770, 0x7027 }, { 17771, 0x7020 }, { 17772, 0x7016 }, { 17773, 0x702B }, { 17774, 0x7021 }, { 17775, 0x7022 },
    { 17776, 0x7023 }, { 17777, 0x7029 }, { 17778, 0x7017 }, { 17779, 0x7024 }, { 17780, 0x701C }, { 17781, 0x702A }, { 17782, 0x720C }, { 17783, 0x720A },
    { 17784, 0x7207 }, { 17785, 0x7202 }, { 17786, 0x7205 }, { 17787, 0x72A5 }, { 17788, 0x72A6 }, { 17789, 0x72A4 }, { 17790, 0x72A3 }, { 17791, 0x72A1 },
    { 17792, 0x74CB }, { 17793, 0x74C5 }, { 17794, 0x74B7 }, { 17795, 0x74C3 }, { 17796, 0x7516 }, { 17797, 0x7660 }, { 17798, 0x77C9 }, { 17799, 0x77CA },
    { 17800, 0x77C4 }, { 17801, 0x77F1 }, { 17802, 0x791D }, { 17803, 0x791B }, { 17804, 0x7921 }, { 17805, 0x791C }, { 17806, 0x7917 }, { 17807, 0x791E },
    { 17808, 0x79B0 }, { 17809, 0x7A67 }, { 17810, 0x7A68 }, { 17811, 0x7C33 }, { 17812, 0x7C3C }, { 17813, 0x7C39 }, { 17814, 0x7C2C }, { 17815, 0x7C3B },
    { 17816, 0x7CEC }, { 17817, 0x7CEA }, { 17818, 0x7E76 }, { 17819, 0x7E75 }, { 17820, 0x7E78 }, { 17821, 0x7E70 }, { 17822, 0x7E77 }, { 17823, 0x7E6F },
    { 17824, 0x7E7A }, { 17825, 0x7E72 }, { 17826, 0x7E74 }, { 17827, 0x7E68 }, { 17828, 0x7F4B }, { 17829, 0x7F4A }, { 17830, 0x7F83 }, { 17831, 0x7F86 },
    { 17832, 0x7FB7 }, { 17833, 0x7FFD }, { 17834, 0x7FFE }, { 17835, 0x8078 }, { 17836, 0x81D7 }, { 17837, 0x81D5 }, { 17838, 0x8264 }, { 17839, 0x8261 },
    { 17840, 0x8263 }, { 17841, 0x85EB }, { 17842, 0x85F1 }, { 17843, 0x85ED }, { 17844, 0x85D9 }, { 17845, 0x85E1 }, { 17846, 0x85E8 }, { 17847, 0x85DA },
    { 17848, 0x85D7 }, { 17849, 0x85EC }, { 17850, 0x85F2 }, { 17851, 0x85F8 }, { 17852, 0x85D8 }, { 17853, 0x85DF }, { 17854, 0x85E3 }, { 17855, 0x85DC },
    { 17856, 0x85D1 }, { 17857, 0x85F0 }, { 17858, 0x85E6 }, { 17859, 0x85EF }, { 17860, 0x85DE }, { 17861, 0x85E2 }, { 17862, 0x8800 }, { 17863, 0x87FA },
    { 17864, 0x8803 }, { 17865, 0x87F6 }, { 17866, 0x87F7 }, { 17867, 0x8809 }, { 17868, 0x880C }, { 17869, 0x880B }, { 17870, 0x8806 }, { 17871, 0x87FC },
    { 17872, 0x8808 }, { 17873, 0x87FF }, { 17874, 0x880A }, { 17875, 0x8802 }, { 17876, 0x8962 }, { 17877, 0x895A }, { 17878, 0x895B }, { 17879, 0x8957 },
    { 17880, 0x8961 }, { 17881, 0x895C }, { 17882, 0x8958 }, { 17883, 0x895D }, { 17884, 0x8959 }, { 17885, 0x8988 }, { 17886, 0x89B7 }, { 17887, 0x89B6 },
    { 17888, 0x89F6 }, { 17889, 0x8B50 }, { 17890, 0x8B48 }, { 17891, 0x8B4A }, { 17892, 0x8B40 }, { 17893, 0x8B53 }, { 17894, 0x8B56 }, { 17895, 0x8B54 },
    { 17896, 0x8B4B }, { 17897, 0x8B55 }, { 17898, 0x8B51 }, { 17899, 0x8B42 }, { 17900, 0x8B52 }, { 17901, 0x8B57 }, { 17902, 0x8C43 }, { 17903, 0x8C77 },
    { 17904, 0x8C76 }, { 17905, 0x8C9A }, { 17906, 0x8D06 }, { 17907, 0x8D07 }, { 17908, 0x8D09 }, { 17909, 0x8DAC }, { 17910, 0x8DAA }, { 17911, 0x8DAD },
    { 17912, 0x8DAB }, { 17913, 0x8E6D }, { 17914, 0x8E78 }, { 17915, 0x8E73 }, { 17916, 0x8E6A }, { 17917, 0x8E6F }, { 17918, 0x8E7B }, { 17919, 0x8EC2 },
    { 17920, 0x8F52 }, { 17921, 0x8F51 }, { 17922, 0x8F4F }, { 17923, 0x8F50 }, { 17924, 0x8F53 }, { 17925, 0x8FB4 }, { 17926, 0x9140 }, { 17927, 0x913F },
    { 17928, 0x91B0 }, { 17929, 0x91AD }, { 17930, 0x93DE }, { 17931, 0x93C7 }, { 17932, 0x93CF }, { 17933, 0x93C2 }, { 17934, 0x93DA }, { 17935, 0x93D0 },
    { 17936, 0x93F9 }, { 17937, 0x93EC }, { 17938, 0x93CC }, { 17939, 0x93D9 }, { 17940, 0x93A9 }, { 17941, 0x93E6 }, { 17942, 0x93CA }, { 17943, 0x93D4 },
    { 17944, 0x93EE }, { 17945, 0x93E3 }, { 17946, 0x93D5 }, { 17947, 0x93C4 }, { 17948, 0x93CE }, { 17949, 0x93C0 }, { 17950, 0x93D2 }, { 17951, 0x93E7 },
    { 17952, 0x957D }, { 17953, 0x95DA }, { 17954, 0x95DB }, { 17955, 0x96E1 }, { 17956, 0x9729 }, { 17957, 0x972B }, { 17958, 0x972C }, { 17959, 0x9728 },
    { 17960, 0x9726 }, { 17961, 0x97B3 }, { 17962, 0x97B7 }, { 17963, 0x97B6 }, { 17964, 0x97DD }, { 17965, 0x97DE }, { 17966, 0x97DF }, { 17967, 0x985C },
    { 17968, 0x9859 }, { 17969, 0x985D }, { 17970, 0x9857 }, { 17971, 0x98BF }, { 17972, 0x98BD }, { 17973, 0x98BB }, { 17974, 0x98BE }, { 17975, 0x9948 },
    { 17976, 0x9947 }, { 17977, 0x9943 }, { 17978, 0x99A6 }, { 17979, 0x99A7 }, { 17980, 0x9A1A }, { 17981, 0x9A15 }, { 17982, 0x9A25 }, { 17983, 0x9A1D },
    { 17984, 0x9A24 }, { 17985, 0x9A1B }, { 17986, 0x9A22 }, { 17987, 0x9A20 }, { 17988, 0x9A27 }, { 17989, 0x9A23 }, { 17990, 0x9A1E }, { 17991, 0x9A1C },
    { 17992, 0x9A14 }, { 17993, 0x9AC2 }, { 17994, 0x9B0B }, { 17995, 0x9B0A }, { 17996, 0x9B0E }, { 17997, 0x9B0C }, { 17998, 0x9B37 }, { 17999, 0x9BEA },
    { 18000, 0x9BEB }, { 18001, 0x9BE0 }, { 18002, 0x9BDE }, { 18003, 0x9BE4 }, { 18004, 0x9BE6 }, { 18005, 0x9BE2 }, { 18006, 0x9BF0 }, { 18007, 0x9BD4 },
    { 18008, 0x9BD7 }, { 18009, 0x9BEC }, { 18010, 0x9BDC }, { 18011, 0x9BD9 }, { 18012, 0x9BE5 }, { 18013, 0x9BD5 }, { 18014, 0x9BE1 }, { 18015, 0x9BDA },
    { 18016, 0x9D77 }, { 18017, 0x9D81 }, { 18018, 0x9D8A }, { 18019, 0x9D84 }, { 18020, 0x9D88 }, { 18021, 0x9D71 }, { 18022, 0x9D80 }, { 18023, 0x9D78 },
    { 18024, 0x9D86 }, { 18025, 0x9D8B }, { 18026, 0x9D8C }, { 18027, 0x9D7D }, { 18028, 0x9D6B }, { 18029, 0x9D74 }, { 18030, 0x9D75 }, { 18031, 0x9D70 },
    { 18032, 0x9D69 }, { 18033, 0x9D85 }, { 18034, 0x9D73 }, { 18035, 0x9D7B }, { 18036, 0x9D82 }, { 18037, 0x9D6F }, { 18038, 0x9D79 }, { 18039, 0x9D7F },
    { 18040, 0x9D87 }, { 18041, 0x9D68 }, { 18042, 0x9E94 }, { 18043, 0x9E91 }, { 18044, 0x9EC0 }, { 18045, 0x9EFC }, { 18046, 0x9F2D }, { 18047, 0x9F40 },
    { 18048, 0x9F41 }, { 18049, 0x9F4D }, { 18050, 0x9F56 }, { 18051, 0x9F57 }, { 18052, 0x9F58 }, { 18053, 0x5337 }, { 18054, 0x56B2 }, { 18055, 0x56B5 },
    { 18056, 0x56B3 }, { 18057, 0x58E3 }, { 18058, 0x5B45 }, { 18059, 0x5DC6 }, { 18060, 0x5DC7 }, { 18061, 0x5EEE }, { 18062, 0x5EEF }, { 18063, 0x5FC0 },
    { 18064, 0x5FC1 }, { 18065, 0x61F9 }, { 18066, 0x6517 }, { 18067, 0x6516 }, { 18068, 0x6515 }, { 18069, 0x6513 }, { 18070, 0x65DF }, { 18071, 0x66E8 },
    { 18072, 0x66E3 }, { 18073, 0x66E4 }, { 18074, 0x6AF3 }, { 18075, 0x6AF0 }, { 18076, 0x6AEA }, { 18077, 0x6AE8 }, { 18078, 0x6AF9 }, { 18079, 0x6AF1 },
    { 18080, 0x6AEE }, { 18081, 0x6AEF }, { 18082, 0x703C }, { 18083, 0x7035 }, { 18084, 0x702F }, { 18085, 0x7037 }, { 18086, 0x7034 }, { 18087, 0x7031 },
    { 18088, 0x7042 }, { 18089, 0x7038 }, { 18090, 0x703F }, { 18091, 0x703A }, { 18092, 0x7039 }, { 18093, 0x7040 }, { 18094, 0x703B }, { 18095, 0x7033 },
    { 18096, 0x7041 }, { 18097, 0x7213 }, { 18098, 0x7214 }, { 18099, 0x72A8 }, { 18100, 0x737D }, { 18101, 0x737C }, { 18102, 0x74BA }, { 18103, 0x76AB },
    { 18104, 0x76AA }, { 18105, 0x76BE }, { 18106, 0x76ED }, { 18107, 0x77CC }, { 18108, 0x77CE }, { 18109, 0x77CF }, { 18110, 0x77CD }, { 18111, 0x77F2 },
    { 18112, 0x7925 }, { 18113, 0x7923 }, { 18114, 0x7927 }, { 18115, 0x7928 }, { 18116, 0x7924 }, { 18117, 0x7929 }, { 18118, 0x79B2 }, { 18119, 0x7A6E },
    { 18120, 0x7A6C }, { 18121, 0x7A6D }, { 18122, 0x7AF7 }, { 18123, 0x7C49 }, { 18124, 0x7C48 }, { 18125, 0x7C4A }, { 18126, 0x7C47 }, { 18127, 0x7C45 },
    { 18128, 0x7CEE }, { 18129, 0x7E7B }, { 18130, 0x7E7E }, { 18131, 0x7E81 }, { 18132, 0x7E80 }, { 18133, 0x7FBA }, { 18134, 0x7FFF }, { 18135, 0x8079 },
    { 18136, 0x81DB }, { 18137, 0x81D9 }, { 18138, 0x820B }, { 18139, 0x8268 }, { 18140, 0x8269 }, { 18141, 0x8622 }, { 18142, 0x85FF }, { 18143, 0x8601 },
    { 18144, 0x85FE }, { 18145, 0x861B }, { 18146, 0x8600 }, { 18147, 0x85F6 }, { 18148, 0x8604 }, { 18149, 0x8609 }, { 18150, 0x8605 }, { 18151, 0x860C },
    { 18152, 0x85FD }, { 18153, 0x8819 }, { 18154, 0x8810 }, { 18155, 0x8811 }, { 18156, 0x8817 }, { 18157, 0x8813 }, { 18158, 0x8816 }, { 18159, 0x8963 },
    { 18160, 0x8966 }, { 18161, 0x89B9 }, { 18162, 0x89F7 }, { 18163, 0x8B60 }, { 18164, 0x8B6A }, { 18165, 0x8B5D }, { 18166, 0x8B68 }, { 18167, 0x8B63 },
    { 18168, 0x8B65 }, { 18169, 0x8B67 }, { 18170, 0x8B6D }, { 18171, 0x8DAE }, { 18172, 0x8E86 }, { 18173, 0x8E88 }, { 18174, 0x8E84 }, { 18175, 0x8F59 },
    { 18176, 0x8F56 }, { 18177, 0x8F57 }, { 18178, 0x8F55 }, { 18179, 0x8F58 }, { 18180, 0x8F5A }, { 18181, 0x908D }, { 18182, 0x9143 }, { 18183, 0x9141 },
    { 18184, 0x91B7 }, { 18185, 0x91B5 }, { 18186, 0x91B2 }, { 18187, 0x91B3 }, { 18188, 0x940B }, { 18189, 0x9413 }, { 18190, 0x93FB }, { 18191, 0x9420 },
    { 18192, 0x940F }, { 18193, 0x9414 }, { 18194, 0x93FE }, { 18195, 0x9415 }, { 18196, 0x9410 }, { 18197, 0x9428 }, { 18198, 0x9419 }, { 18199, 0x940D },
    { 18200, 0x93F5 }, { 18201, 0x9400 }, { 18202, 0x93F7 }, { 18203, 0x9407 }, { 18204, 0x940E }, { 18205, 0x9416 }, { 18206, 0x9412 }, { 18207, 0x93FA },
    { 18208, 0x9409 }, { 18209, 0x93F8 }, { 18210, 0x940A }, { 18211, 0x93FF }, { 18212, 0x93FC }, { 18213, 0x940C }, { 18214, 0x93F6 }, { 18215, 0x9411 },
    { 18216, 0x9406 }, { 18217, 0x95DE }, { 18218, 0x95E0 }, { 18219, 0x95DF }, { 18220, 0x972E }, { 18221, 0x972F }, { 18222, 0x97B9 }, { 18223, 0x97BB },
    { 18224, 0x97FD }, { 18225, 0x97FE }, { 18226, 0x9860 }, { 18227, 0x9862 }, { 18228, 0x9863 }, { 18229, 0x985F }, { 18230, 0x98C1 }, { 18231, 0x98C2 },
    { 18232, 0x9950 }, { 18233, 0x994E }, { 18234, 0x9959 }, { 18235, 0x994C }, { 18236, 0x994B }, { 18237, 0x9953 }, { 18238, 0x9A32 }, { 18239, 0x9A34 },
    { 18240, 0x9A31 }, { 18241, 0x9A2C }, { 18242, 0x9A2A }, { 18243, 0x9A36 }, { 18244, 0x9A29 }, { 18245, 0x9A2E }, { 18246, 0x9A38 }, { 18247, 0x9A2D },
    { 18248, 0x9AC7 }, { 18249, 0x9ACA }, { 18250, 0x9AC6 }, { 18251, 0x9B10 }, { 18252, 0x9B12 }, { 18253, 0x9B11 }, { 18254, 0x9C0B }, { 18255, 0x9C08 },
    { 18256, 0x9BF7 }, { 18257, 0x9C05 }, { 18258, 0x9C12 }, { 18259, 0x9BF8 }, { 18260, 0x9C40 }, { 18261, 0x9C07 }, { 18262, 0x9C0E }, { 18263, 0x9C06 },
    { 18264, 0x9C17 }, { 18265, 0x9C14 }, { 18266, 0x9C09 }, { 18267, 0x9D9F }, { 18268, 0x9D99 }, { 18269, 0x9DA4 }, { 18270, 0x9D9D }, { 18271, 0x9D92 },
    { 18272, 0x9D98 }, { 18273, 0x9D90 }, { 18274, 0x9D9B }, { 18275, 0x9DA0 }, { 18276, 0x9D94 }, { 18277, 0x9D9C }, { 18278, 0x9DAA }, { 18279, 0x9D97 },
    { 18280, 0x9DA1 }, { 18281, 0x9D9A }, { 18282, 0x9DA2 }, { 18283, 0x9DA8 }, { 18284, 0x9D9E }, { 18285, 0x9DA3 }, { 18286, 0x9DBF }, { 18287, 0x9DA9 },
    { 18288, 0x9D96 }, { 18289, 0x9DA6 }, { 18290, 0x9DA7 }, { 18291, 0x9E99 }, { 18292, 0x9E9B }, { 18293, 0x9E9A }, { 18294, 0x9EE5 }, { 18295, 0x9EE4 },
    { 18296, 0x9EE7 }, { 18297, 0x9EE6 }, { 18298, 0x9F30 }, { 18299, 0x9F2E }, { 18300, 0x9F5B }, { 18301, 0x9F60 }, { 18302, 0x9F5E }, { 18303, 0x9F5D },
    { 18304, 0x9F59 }, { 18305, 0x9F91 }, { 18306, 0x513A }, { 18307, 0x5139 }, { 18308, 0x5298 }, { 18309, 0x5297 }, { 18310, 0x56C3 }, { 18311, 0x56BD },
    { 18312, 0x56BE }, { 18313, 0x5B48 }, { 18314, 0x5B47 }, { 18315, 0x5DCB }, { 18316, 0x5DCF }, { 18317, 0x5EF1 }, { 18318, 0x61FD }, { 18319, 0x651B },
    { 18320, 0x6B02 }, { 18321, 0x6AFC }, { 18322, 0x6B03 }, { 18323, 0x6AF8 }, { 18324, 0x6B00 }, { 18325, 0x7043 }, { 18326, 0x7044 }, { 18327, 0x704A },
    { 18328, 0x7048 }, { 18329, 0x7049 }, { 18330, 0x7045 }, { 18331, 0x7046 }, { 18332, 0x721D }, { 18333, 0x721A }, { 18334, 0x7219 }, { 18335, 0x737E },
    { 18336, 0x7517 }, { 18337, 0x766A }, { 18338, 0x77D0 }, { 18339, 0x792D }, { 18340, 0x7931 }, { 18341, 0x792F }, { 18342, 0x7C54 }, { 18343, 0x7C53 },
    { 18344, 0x7CF2 }, { 18345, 0x7E8A }, { 18346, 0x7E87 }, { 18347, 0x7E88 }, { 18348, 0x7E8B }, { 18349, 0x7E86 }, { 18350, 0x7E8D }, { 18351, 0x7F4D },
    { 18352, 0x7FBB }, { 18353, 0x8030 }, { 18354, 0x81DD }, { 18355, 0x8618 }, { 18356, 0x862A }, { 18357, 0x8626 }, { 18358, 0x861F }, { 18359, 0x8623 },
    { 18360, 0x861C }, { 18361, 0x8619 }, { 18362, 0x8627 }, { 18363, 0x862E }, { 18364, 0x8621 }, { 18365, 0x8620 }, { 18366, 0x8629 }, { 18367, 0x861E },
    { 18368, 0x8625 }, { 18369, 0x8829 }, { 18370, 0x881D }, { 18371, 0x881B }, { 18372, 0x8820 }, { 18373, 0x8824 }, { 18374, 0x881C }, { 18375, 0x882B },
    { 18376, 0x884A }, { 18377, 0x896D }, { 18378, 0x8969 }, { 18379, 0x896E }, { 18380, 0x896B }, { 18381, 0x89FA }, { 18382, 0x8B79 }, { 18383, 0x8B78 },
    { 18384, 0x8B45 }, { 18385, 0x8B7A }, { 18386, 0x8B7B }, { 18387, 0x8D10 }, { 18388, 0x8D14 }, { 18389, 0x8DAF }, { 18390, 0x8E8E }, { 18391, 0x8E8C },
    { 18392, 0x8F5E }, { 18393, 0x8F5B }, { 18394, 0x8F5D }, { 18395, 0x9146 }, { 18396, 0x9144 }, { 18397, 0x9145 }, { 18398, 0x91B9 }, { 18399, 0x943F },
    { 18400, 0x943B }, { 18401, 0x9436 }, { 18402, 0x9429 }, { 18403, 0x943D }, { 18404, 0x943C }, { 18405, 0x9430 }, { 18406, 0x9439 }, { 18407, 0x942A },
    { 18408, 0x9437 }, { 18409, 0x942C }, { 18410, 0x9440 }, { 18411, 0x9431 }, { 18412, 0x95E5 }, { 18413, 0x95E4 }, { 18414, 0x95E3 }, { 18415, 0x9735 },
    { 18416, 0x973A }, { 18417, 0x97BF }, { 18418, 0x97E1 }, { 18419, 0x9864 }, { 18420, 0x98C9 }, { 18421, 0x98C6 }, { 18422, 0x98C0 }, { 18423, 0x9958 },
    { 18424, 0x9956 }, { 18425, 0x9A39 }, { 18426, 0x9A3D }, { 18427, 0x9A46 }, { 18428, 0x9A44 }, { 18429, 0x9A42 }, { 18430, 0x9A41 }, { 18431, 0x9A3A },
    { 18432, 0x9A3F }, { 18433, 0x9ACD }, { 18434, 0x9B15 }, { 18435, 0x9B17 }, { 18436, 0x9B18 }, { 18437, 0x9B16 }, { 18438, 0x9B3A }, { 18439, 0x9B52 },
    { 18440, 0x9C2B }, { 18441, 0x9C1D }, { 18442, 0x9C1C }, { 18443, 0x9C2C }, { 18444, 0x9C23 }, { 18445, 0x9C28 }, { 18446, 0x9C29 }, { 18447, 0x9C24 },
    { 18448, 0x9C21 }, { 18449, 0x9DB7 }, { 18450, 0x9DB6 }, { 18451, 0x9DBC }, { 18452, 0x9DC1 }, { 18453, 0x9DC7 }, { 18454, 0x9DCA }, { 18455, 0x9DCF },
    { 18456, 0x9DBE }, { 18457, 0x9DC5 }, { 18458, 0x9DC3 }, { 18459, 0x9DBB }, { 18460, 0x9DB5 }, { 18461, 0x9DCE }, { 18462, 0x9DB9 }, { 18463, 0x9DBA },
    { 18464, 0x9DAC }, { 18465, 0x9DC8 }, { 18466, 0x9DB1 }, { 18467, 0x9DAD }, { 18468, 0x9DCC }, { 18469, 0x9DB3 }, { 18470, 0x9DCD }, { 18471, 0x9DB2 },
    { 18472, 0x9E7A }, { 18473, 0x9E9C }, { 18474, 0x9EEB }, { 18475, 0x9EEE }, { 18476, 0x9EED }, { 18477, 0x9F1B }, { 18478, 0x9F18 }, { 18479, 0x9F1A },
    { 18480, 0x9F31 }, { 18481, 0x9F4E }, { 18482, 0x9F65 }, { 18483, 0x9F64 }, { 18484, 0x9F92 }, { 18485, 0x4EB9 }, { 18486, 0x56C6 }, { 18487, 0x56C5 },
    { 18488, 0x56CB }, { 18489, 0x5971 }, { 18490, 0x5B4B }, { 18491, 0x5B4C }, { 18492, 0x5DD5 }, { 18493, 0x5DD1 }, { 18494, 0x5EF2 }, { 18495, 0x6521 },
    { 18496, 0x6520 }, { 18497, 0x6526 }, { 18498, 0x6522 }, { 18499, 0x6B0B }, { 18500, 0x6B08 }, { 18501, 0x6B09 }, { 18502, 0x6C0D }, { 18503, 0x7055 },
    { 18504, 0x7056 }, { 18505, 0x7057 }, { 18506, 0x7052 }, { 18507, 0x721E }, { 18508, 0x721F }, { 18509, 0x72A9 }, { 18510, 0x737F }, { 18511, 0x74D8 },
    { 18512, 0x74D5 }, { 18513, 0x74D9 }, { 18514, 0x74D7 }, { 18515, 0x766D }, { 18516, 0x76AD }, { 18517, 0x7935 }, { 18518, 0x79B4 }, { 18519, 0x7A70 },
    { 18520, 0x7A71 }, { 18521, 0x7C57 }, { 18522, 0x7C5C }, { 18523, 0x7C59 }, { 18524, 0x7C5B }, { 18525, 0x7C5A }, { 18526, 0x7CF4 }, { 18527, 0x7CF1 },
    { 18528, 0x7E91 }, { 18529, 0x7F4F }, { 18530, 0x7F87 }, { 18531, 0x81DE }, { 18532, 0x826B }, { 18533, 0x8634 }, { 18534, 0x8635 }, { 18535, 0x8633 },
    { 18536, 0x862C }, { 18537, 0x8632 }, { 18538, 0x8636 }, { 18539, 0x882C }, { 18540, 0x8828 }, { 18541, 0x8826 }, { 18542, 0x882A }, { 18543, 0x8825 },
    { 18544, 0x8971 }, { 18545, 0x89BF }, { 18546, 0x89BE }, { 18547, 0x89FB }, { 18548, 0x8B7E }, { 18549, 0x8B84 }, { 18550, 0x8B82 }, { 18551, 0x8B86 },
    { 18552, 0x8B85 }, { 18553, 0x8B7F }, { 18554, 0x8D15 }, { 18555, 0x8E95 }, { 18556, 0x8E94 }, { 18557, 0x8E9A }, { 18558, 0x8E92 }, { 18559, 0x8E90 },
    { 18560, 0x8E96 }, { 18561, 0x8E97 }, { 18562, 0x8F60 }, { 18563, 0x8F62 }, { 18564, 0x9147 }, { 18565, 0x944C }, { 18566, 0x9450 }, { 18567, 0x944A },
    { 18568, 0x944B }, { 18569, 0x944F }, { 18570, 0x9447 }, { 18571, 0x9445 }, { 18572, 0x9448 }, { 18573, 0x9449 }, { 18574, 0x9446 }, { 18575, 0x973F },
    { 18576, 0x97E3 }, { 18577, 0x986A }, { 18578, 0x9869 }, { 18579, 0x98CB }, { 18580, 0x9954 }, { 18581, 0x995B }, { 18582, 0x9A4E }, { 18583, 0x9A53 },
    { 18584, 0x9A54 }, { 18585, 0x9A4C }, { 18586, 0x9A4F }, { 18587, 0x9A48 }, { 18588, 0x9A4A }, { 18589, 0x9A49 }, { 18590, 0x9A52 }, { 18591, 0x9A50 },
    { 18592, 0x9AD0 }, { 18593, 0x9B19 }, { 18594, 0x9B2B }, { 18595, 0x9B3B }, { 18596, 0x9B56 }, { 18597, 0x9B55 }, { 18598, 0x9C46 }, { 18599, 0x9C48 },
    { 18600, 0x9C3F }, { 18601, 0x9C44 }, { 18602, 0x9C39 }, { 18603, 0x9C33 }, { 18604, 0x9C41 }, { 18605, 0x9C3C }, { 18606, 0x9C37 }, { 18607, 0x9C34 },
    { 18608, 0x9C32 }, { 18609, 0x9C3D }, { 18610, 0x9C36 }, { 18611, 0x9DDB }, { 18612, 0x9DD2 }, { 18613, 0x9DDE }, { 18614, 0x9DDA }, { 18615, 0x9DCB },
    { 18616, 0x9DD0 }, { 18617, 0x9DDC }, { 18618, 0x9DD1 }, { 18619, 0x9DDF }, { 18620, 0x9DE9 }, { 18621, 0x9DD9 }, { 18622, 0x9DD8 }, { 18623, 0x9DD6 },
    { 18624, 0x9DF5 }, { 18625, 0x9DD5 }, { 18626, 0x9DDD }, { 18627, 0x9EB6 }, { 18628, 0x9EF0 }, { 18629, 0x9F35 }, { 18630, 0x9F33 }, { 18631, 0x9F32 },
    { 18632, 0x9F42 }, { 18633, 0x9F6B }, { 18634, 0x9F95 }, { 18635, 0x9FA2 }, { 18636, 0x513D }, { 18637, 0x5299 }, { 18638, 0x58E8 }, { 18639, 0x58E7 },
    { 18640, 0x5972 }, { 18641, 0x5B4D }, { 18642, 0x5DD8 }, { 18643, 0x882F }, { 18644, 0x5F4F }, { 18645, 0x6201 }, { 18646, 0x6203 }, { 18647, 0x6204 },
    { 18648, 0x6529 }, { 18649, 0x6525 }, { 18650, 0x6596 }, { 18651, 0x66EB }, { 18652, 0x6B11 }, { 18653, 0x6B12 }, { 18654, 0x6B0F }, { 18655, 0x6BCA },
    { 18656, 0x705B }, { 18657, 0x705A }, { 18658, 0x7222 }, { 18659, 0x7382 }, { 18660, 0x7381 }, { 18661, 0x7383 }, { 18662, 0x7670 }, { 18663, 0x77D4 },
    { 18664, 0x7C67 }, { 18665, 0x7C66 }, { 18666, 0x7E95 }, { 18667, 0x826C }, { 18668, 0x863A }, { 18669, 0x8640 }, { 18670, 0x8639 }, { 18671, 0x863C },
    { 18672, 0x8631 }, { 18673, 0x863B }, { 18674, 0x863E }, { 18675, 0x8830 }, { 18676, 0x8832 }, { 18677, 0x882E }, { 18678, 0x8833 }, { 18679, 0x8976 },
    { 18680, 0x8974 }, { 18681, 0x8973 }, { 18682, 0x89FE }, { 18683, 0x8B8C }, { 18684, 0x8B8E }, { 18685, 0x8B8B }, { 18686, 0x8B88 }, { 18687, 0x8C45 },
    { 18688, 0x8D19 }, { 18689, 0x8E98 }, { 18690, 0x8F64 }, { 18691, 0x8F63 }, { 18692, 0x91BC }, { 18693, 0x9462 }, { 18694, 0x9455 }, { 18695, 0x945D },
    { 18696, 0x9457 }, { 18697, 0x945E }, { 18698, 0x97C4 }, { 18699, 0x97C5 }, { 18700, 0x9800 }, { 18701, 0x9A56 }, { 18702, 0x9A59 }, { 18703, 0x9B1E },
    { 18704, 0x9B1F }, { 18705, 0x9B20 }, { 18706, 0x9C52 }, { 18707, 0x9C58 }, { 18708, 0x9C50 }, { 18709, 0x9C4A }, { 18710, 0x9C4D }, { 18711, 0x9C4B },
    { 18712, 0x9C55 }, { 18713, 0x9C59 }, { 18714, 0x9C4C }, { 18715, 0x9C4E }, { 18716, 0x9DFB }, { 18717, 0x9DF7 }, { 18718, 0x9DEF }, { 18719, 0x9DE3 },
    { 18720, 0x9DEB }, { 18721, 0x9DF8 }, { 18722, 0x9DE4 }, { 18723, 0x9DF6 }, { 18724, 0x9DE1 }, { 18725, 0x9DEE }, { 18726, 0x9DE6 }, { 18727, 0x9DF2 },
    { 18728, 0x9DF0 }, { 18729, 0x9DE2 }, { 18730, 0x9DEC }, { 18731, 0x9DF4 }, { 18732, 0x9DF3 }, { 18733, 0x9DE8 }, { 18734, 0x9DED }, { 18735, 0x9EC2 },
    { 18736, 0x9ED0 }, { 18737, 0x9EF2 }, { 18738, 0x9EF3 }, { 18739, 0x9F06 }, { 18740, 0x9F1C }, { 18741, 0x9F38 }, { 18742, 0x9F37 }, { 18743, 0x9F36 },
    { 18744, 0x9F43 }, { 18745, 0x9F4F }, { 18746, 0x9F71 }, { 18747, 0x9F70 }, { 18748, 0x9F6E }, { 18749, 0x9F6F }, { 18750, 0x56D3 }, { 18751, 0x56CD },
    { 18752, 0x5B4E }, { 18753, 0x5C6D }, { 18754, 0x652D }, { 18755, 0x66ED }, { 18756, 0x66EE }, { 18757, 0x6B13 }, { 18758, 0x705F }, { 18759, 0x7061 },
    { 18760, 0x705D }, { 18761, 0x7060 }, { 18762, 0x7223 }, { 18763, 0x74DB }, { 18764, 0x74E5 }, { 18765, 0x77D5 }, { 18766, 0x7938 }, { 18767, 0x79B7 },
    { 18768, 0x79B6 }, { 18769, 0x7C6A }, { 18770, 0x7E97 }, { 18771, 0x7F89 }, { 18772, 0x826D }, { 18773, 0x8643 }, { 18774, 0x8838 }, { 18775, 0x8837 },
    { 18776, 0x8835 }, { 18777, 0x884B }, { 18778, 0x8B94 }, { 18779, 0x8B95 }, { 18780, 0x8E9E }, { 18781, 0x8E9F }, { 18782, 0x8EA0 }, { 18783, 0x8E9D },
    { 18784, 0x91BE }, { 18785, 0x91BD }, { 18786, 0x91C2 }, { 18787, 0x946B }, { 18788, 0x9468 }, { 18789, 0x9469 }, { 18790, 0x96E5 }, { 18791, 0x9746 },
    { 18792, 0x9743 }, { 18793, 0x9747 }, { 18794, 0x97C7 }, { 18795, 0x97E5 }, { 18796, 0x9A5E }, { 18797, 0x9AD5 }, { 18798, 0x9B59 }, { 18799, 0x9C63 },
    { 18800, 0x9C67 }, { 18801, 0x9C66 }, { 18802, 0x9C62 }, { 18803, 0x9C5E }, { 18804, 0x9C60 }, { 18805, 0x9E02 }, { 18806, 0x9DFE }, { 18807, 0x9E07 },
    { 18808, 0x9E03 }, { 18809, 0x9E06 }, { 18810, 0x9E05 }, { 18811, 0x9E00 }, { 18812, 0x9E01 }, { 18813, 0x9E09 }, { 18814, 0x9DFF }, { 18815, 0x9DFD },
    { 18816, 0x9E04 }, { 18817, 0x9EA0 }, { 18818, 0x9F1E }, { 18819, 0x9F46 }, { 18820, 0x9F74 }, { 18821, 0x9F75 }, { 18822, 0x9F76 }, { 18823, 0x56D4 },
    { 18824, 0x652E }, { 18825, 0x65B8 }, { 18826, 0x6B18 }, { 18827, 0x6B19 }, { 18828, 0x6B17 }, { 18829, 0x6B1A }, { 18830, 0x7062 }, { 18831, 0x7226 },
    { 18832, 0x72AA }, { 18833, 0x77D8 }, { 18834, 0x77D9 }, { 18835, 0x7939 }, { 18836, 0x7C69 }, { 18837, 0x7C6B }, { 18838, 0x7CF6 }, { 18839, 0x7E9A },
    { 18840, 0x7E98 }, { 18841, 0x7E9B }, { 18842, 0x7E99 }, { 18843, 0x81E0 }, { 18844, 0x81E1 }, { 18845, 0x8646 }, { 18846, 0x8647 }, { 18847, 0x8648 },
    { 18848, 0x8979 }, { 18849, 0x897A }, { 18850, 0x897C }, { 18851, 0x897B }, { 18852, 0x89FF }, { 18853, 0x8B98 }, { 18854, 0x8B99 }, { 18855, 0x8EA5 },
    { 18856, 0x8EA4 }, { 18857, 0x8EA3 }, { 18858, 0x946E }, { 18859, 0x946D }, { 18860, 0x946F }, { 18861, 0x9471 }, { 18862, 0x9473 }, { 18863, 0x9749 },
    { 18864, 0x9872 }, { 18865, 0x995F }, { 18866, 0x9C68 }, { 18867, 0x9C6E }, { 18868, 0x9C6D }, { 18869, 0x9E0B }, { 18870, 0x9E0D }, { 18871, 0x9E10 },
    { 18872, 0x9E0F }, { 18873, 0x9E12 }, { 18874, 0x9E11 }, { 18875, 0x9EA1 }, { 18876, 0x9EF5 }, { 18877, 0x9F09 }, { 18878, 0x9F47 }, { 18879, 0x9F78 },
    { 18880, 0x9F7B }, { 18881, 0x9F7A }, { 18882, 0x9F79 }, { 18883, 0x571E }, { 18884, 0x7066 }, { 18885, 0x7C6F }, { 18886, 0x883C }, { 18887, 0x8DB2 },
    { 18888, 0x8EA6 }, { 18889, 0x91C3 }, { 18890, 0x9474 }, { 18891, 0x9478 }, { 18892, 0x9476 }, { 18893, 0x9475 }, { 18894, 0x9A60 }, { 18895, 0x9C74 },
    { 18896, 0x9C73 }, { 18897, 0x9C71 }, { 18898, 0x9C75 }, { 18899, 0x9E14 }, { 18900, 0x9E13 }, { 18901, 0x9EF6 }, { 18902, 0x9F0A }, { 18903, 0x9FA4 },
    { 18904, 0x7068 }, { 18905, 0x7065 }, { 18906, 0x7CF7 }, { 18907, 0x866A }, { 18908, 0x883E }, { 18909, 0x883D }, { 18910, 0x883F }, { 18911, 0x8B9E },
    { 18912, 0x8C9C }, { 18913, 0x8EA9 }, { 18914, 0x8EC9 }, { 18915, 0x974B }, { 18916, 0x9873 }, { 18917, 0x9874 }, { 18918, 0x98CC }, { 18919, 0x9961 },
    { 18920, 0x99AB }, { 18921, 0x9A64 }, { 18922, 0x9A66 }, { 18923, 0x9A67 }, { 18924, 0x9B24 }, { 18925, 0x9E15 }, { 18926, 0x9E17 }, { 18927, 0x9F48 },
    { 18928, 0x6207 }, { 18929, 0x6B1E }, { 18930, 0x7227 }, { 18931, 0x864C }, { 18932, 0x8EA8 }, { 18933, 0x9482 }, { 18934, 0x9480 }, { 18935, 0x9481 },
    { 18936, 0x9A69 }, { 18937, 0x9A68 }, { 18938, 0x9B2E }, { 18939, 0x9E19 }, { 18940, 0x7229 }, { 18941, 0x864B }, { 18942, 0x8B9F }, { 18943, 0x9483 },
    { 18944, 0x9C79 }, { 18945, 0x9EB7 }, { 18946, 0x7675 }, { 18947, 0x9A6B }, { 18948, 0x9C7A }, { 18949, 0x9E1D }, { 18950, 0x7069 }, { 18951, 0x706A },
    { 18952, 0x9EA4 }, { 18953, 0x9F7E }, { 18954, 0x9F49 }, { 18955, 0x9F98 }, { 18956, 0x7881 }, { 18957, 0x92B9 }, { 18958, 0x88CF }, { 18959, 0x58BB },
    { 18960, 0x6052 }, { 18961, 0x7CA7 }, { 18962, 0x5AFA }, { 18963, 0x2554 }, { 18964, 0x2566 }, { 18965, 0x2557 }, { 18966, 0x2560 }, { 18967, 0x256C },
    { 18968, 0x2563 }, { 18969, 0x255A }, { 18970, 0x2569 }, { 18971, 0x255D }, { 18972, 0x2552 }, { 18973, 0x2564 }, { 18974, 0x2555 }, { 18975, 0x255E },
    { 18976, 0x256A }, { 18977, 0x2561 }, { 18978, 0x2558 }, { 18979, 0x2567 }, { 18980, 0x255B }, { 18981, 0x2553 }, { 18982, 0x2565 }, { 18983, 0x2556 },
    { 18984, 0x255F }, { 18985, 0x256B }, { 18986, 0x2562 }, { 18987, 0x2559 }, { 18988, 0x2568 }, { 18989, 0x255C }, { 18990, 0x2551 }, { 18991, 0x2550 },
    { 18992, 0x256D }, { 18993, 0x256E }, { 18994, 0x2570 }, { 18995, 0x256F }, { 18996, 0xFFED }, { 18997, 0x20547 }, { 18998, 0x92DB }, { 18999, 0x205DF },
    { 19000, 0x23FC5 }, { 19001, 0x854C }, { 19002, 0x42B5 }, { 19003, 0x73EF }, { 19004, 0x51B5 }, { 19005, 0x3649 }, { 19006, 0x24942 }, { 19007, 0x289E4 },
    { 19008, 0x9344 }, { 19009, 0x219DB }, { 19010, 0x82EE }, { 19011, 0x23CC8 }, { 19012, 0x783C }, { 19013, 0x6744 }, { 19014, 0x62DF }, { 19015, 0x24933 },
    { 19016, 0x289AA }, { 19017, 0x202A0 }, { 19018, 0x26BB3 }, { 19019, 0x21305 }, { 19020, 0x4FAB }, { 19021, 0x224ED }, { 19022, 0x5008 }, { 19023, 0x26D29 },
    { 19024, 0x27A84 }, { 19025, 0x23600 }, { 19026, 0x24AB1 }, { 19027, 0x22513 }, { 19028, 0x5029 }, { 19029, 0x2037E }, { 19030, 0x5FA4 }, { 19031, 0x20380 },
    { 19032, 0x20347 }, { 19033, 0x6EDB }, { 19034, 0x2041F }, { 19035, 0x507D }, { 19036, 0x5101 }, { 19037, 0x347A }, { 19038, 0x510E }, { 19039, 0x986C },
    { 19040, 0x3743 }, { 19041, 0x8416 }, { 19042, 0x249A4 }, { 19043, 0x20487 }, { 19044, 0x5160 }, { 19045, 0x233B4 }, { 19046, 0x516A }, { 19047, 0x20BFF },
    { 19048, 0x220FC }, { 19049, 0x202E5 }, { 19050, 0x22530 }, { 19051, 0x2058E }, { 19052, 0x23233 }, { 19053, 0x21983 }, { 19054, 0x5B82 }, { 19055, 0x877D },
    { 19056, 0x205B3 }, { 19057, 0x23C99 }, { 19058, 0x51B2 }, { 19059, 0x51B8 }, { 19060, 0x9D34 }, { 19061, 0x51C9 }, { 19062, 0x51CF }, { 19063, 0x51D1 },
    { 19064, 0x3CDC }, { 19065, 0x51D3 }, { 19066, 0x24AA6 }, { 19067, 0x51B3 }, { 19068, 0x51E2 }, { 19069, 0x5342 }, { 19070, 0x51ED }, { 19071, 0x83CD },
    { 19072, 0x693E }, { 19073, 0x2372D }, { 19074, 0x5F7B }, { 19075, 0x520B }, { 19076, 0x5226 }, { 19077, 0x523C }, { 19078, 0x52B5 }, { 19079, 0x5257 },
    { 19080, 0x5294 }, { 19081, 0x52B9 }, { 19082, 0x52C5 }, { 19083, 0x7C15 }, { 19084, 0x8542 }, { 19085, 0x52E0 }, { 19086, 0x860D }, { 19087, 0x26B13 },
    { 19088, 0x5305 }, { 19089, 0x28ADE }, { 19090, 0x5549 }, { 19091, 0x6ED9 }, { 19092, 0x23F80 }, { 19093, 0x20954 }, { 19094, 0x23FEC }, { 19095, 0x5333 },
    { 19096, 0x5344 }, { 19097, 0x20BE2 }, { 19098, 0x6CCB }, { 19099, 0x21726 }, { 19100, 0x681B }, { 19101, 0x73D5 }, { 19102, 0x604A }, { 19103, 0x3EAA },
    { 19104, 0x38CC }, { 19105, 0x216E8 }, { 19106, 0x71DD }, { 19107, 0x44A2 }, { 19108, 0x536D }, { 19109, 0x5374 }, { 19110, 0x286AB }, { 19111, 0x537E },
    { 19112, 0x537F }, { 19113, 0x21596 }, { 19114, 0x21613 }, { 19115, 0x77E6 }, { 19116, 0x5393 }, { 19117, 0x28A9B }, { 19118, 0x53A0 }, { 19119, 0x53AB },
    { 19120, 0x53AE }, { 19121, 0x73A7 }, { 19122, 0x25772 }, { 19123, 0x3F59 }, { 19124, 0x739C }, { 19125, 0x53C1 }, { 19126, 0x53C5 }, { 19127, 0x6C49 },
    { 19128, 0x4E49 }, { 19129, 0x57FE }, { 19130, 0x53D9 }, { 19131, 0x3AAB }, { 19132, 0x20B8F }, { 19133, 0x53E0 }, { 19134, 0x23FEB }, { 19135, 0x22DA3 },
    { 19136, 0x53F6 }, { 19137, 0x20C77 }, { 19138, 0x5413 }, { 19139, 0x7079 }, { 19140, 0x552B }, { 19141, 0x6657 }, { 19142, 0x6D5B }, { 19143, 0x546D },
    { 19144, 0x26B53 }, { 19145, 0x20D74 }, { 19146, 0x555D }, { 19147, 0x548F }, { 19148, 0x54A4 }, { 19149, 0x47A6 }, { 19150, 0x2170D }, { 19151, 0x20EDD },
    { 19152, 0x3DB4 }, { 19153, 0x20D4D }, { 19154, 0x289BC }, { 19155, 0x22698 }, { 19156, 0x5547 }, { 19157, 0x4CED }, { 19158, 0x542F }, { 19159, 0x7417 },
    { 19160, 0x5586 }, { 19161, 0x55A9 }, { 19162, 0x5605 }, { 19163, 0x218D7 }, { 19164, 0x2403A }, { 19165, 0x4552 }, { 19166, 0x24435 }, { 19167, 0x66B3 },
    { 19168, 0x210B4 }, { 19169, 0x5637 }, { 19170, 0x66CD }, { 19171, 0x2328A }, { 19172, 0x66A4 }, { 19173, 0x66AD }, { 19174, 0x564D }, { 19175, 0x564F },
    { 19176, 0x78F1 }, { 19177, 0x56F1 }, { 19178, 0x9787 }, { 19179, 0x53FE }, { 19180, 0x5700 }, { 19181, 0x56EF }, { 19182, 0x56ED }, { 19183, 0x28B66 },
    { 19184, 0x3623 }, { 19185, 0x2124F }, { 19186, 0x5746 }, { 19187, 0x241A5 }, { 19188, 0x6C6E }, { 19189, 0x708B }, { 19190, 0x5742 }, { 19191, 0x36B1 },
    { 19192, 0x26C7E }, { 19193, 0x57E6 }, { 19194, 0x21416 }, { 19195, 0x5803 }, { 19196, 0x21454 }, { 19197, 0x24363 }, { 19198, 0x5826 }, { 19199, 0x24BF5 },
    { 19200, 0x585C }, { 19201, 0x58AA }, { 19202, 0x3561 }, { 19203, 0x58E0 }, { 19204, 0x58DC }, { 19205, 0x2123C }, { 19206, 0x58FB }, { 19207, 0x5BFF },
    { 19208, 0x5743 }, { 19209, 0x2A150 }, { 19210, 0x24278 }, { 19211, 0x93D3 }, { 19212, 0x35A1 }, { 19213, 0x591F }, { 19214, 0x68A6 }, { 19215, 0x36C3 },
    { 19216, 0x6E59 }, { 19217, 0x2163E }, { 19218, 0x5A24 }, { 19219, 0x5553 }, { 19220, 0x21692 }, { 19221, 0x8505 }, { 19222, 0x59C9 }, { 19223, 0x20D4E },
    { 19224, 0x26C81 }, { 19225, 0x26D2A }, { 19226, 0x217DC }, { 19227, 0x59D9 }, { 19228, 0x217FB }, { 19229, 0x217B2 }, { 19230, 0x26DA6 }, { 19231, 0x6D71 },
    { 19232, 0x21828 }, { 19233, 0x216D5 }, { 19234, 0x59F9 }, { 19235, 0x26E45 }, { 19236, 0x5AAB }, { 19237, 0x5A63 }, { 19238, 0x36E6 }, { 19239, 0x249A9 },
    { 19240, 0x5A77 }, { 19241, 0x3708 }, { 19242, 0x5A96 }, { 19243, 0x7465 }, { 19244, 0x5AD3 }, { 19245, 0x26FA1 }, { 19246, 0x22554 }, { 19247, 0x3D85 },
    { 19248, 0x21911 }, { 19249, 0x3732 }, { 19250, 0x216B8 }, { 19251, 0x5E83 }, { 19252, 0x52D0 }, { 19253, 0x5B76 }, { 19254, 0x6588 }, { 19255, 0x5B7C },
    { 19256, 0x27A0E }, { 19257, 0x4004 }, { 19258, 0x485D }, { 19259, 0x20204 }, { 19260, 0x5BD5 }, { 19261, 0x6160 }, { 19262, 0x21A34 }, { 19263, 0x259CC },
    { 19264, 0x205A5 }, { 19265, 0x5BF3 }, { 19266, 0x5B9D }, { 19267, 0x4D10 }, { 19268, 0x5C05 }, { 19269, 0x21B44 }, { 19270, 0x5C13 }, { 19271, 0x73CE },
    { 19272, 0x5C14 }, { 19273, 0x21CA5 }, { 19274, 0x26B28 }, { 19275, 0x5C49 }, { 19276, 0x48DD }, { 19277, 0x5C85 }, { 19278, 0x5CE9 }, { 19279, 0x5CEF },
    { 19280, 0x5D8B }, { 19281, 0x21DF9 }, { 19282, 0x21E37 }, { 19283, 0x5D10 }, { 19284, 0x5D18 }, { 19285, 0x5D46 }, { 19286, 0x21EA4 }, { 19287, 0x5CBA },
    { 19288, 0x5DD7 }, { 19289, 0x82FC }, { 19290, 0x382D }, { 19291, 0x24901 }, { 19292, 0x22049 }, { 19293, 0x22173 }, { 19294, 0x8287 }, { 19295, 0x3836 },
    { 19296, 0x3BC2 }, { 19297, 0x5E2E }, { 19298, 0x6A8A }, { 19299, 0x5E75 }, { 19300, 0x5E7A }, { 19301, 0x244BC }, { 19302, 0x20CD3 }, { 19303, 0x53A6 },
    { 19304, 0x4EB7 }, { 19305, 0x5ED0 }, { 19306, 0x53A8 }, { 19307, 0x21771 }, { 19308, 0x5E09 }, { 19309, 0x5EF4 }, { 19310, 0x28482 }, { 19311, 0x5EF9 },
    { 19312, 0x5EFB }, { 19313, 0x38A0 }, { 19314, 0x5EFC }, { 19315, 0x683E }, { 19316, 0x941B }, { 19317, 0x5F0D }, { 19318, 0x201C1 }, { 19319, 0x2F894 },
    { 19320, 0x3ADE }, { 19321, 0x48AE }, { 19322, 0x2133A }, { 19323, 0x5F3A }, { 19324, 0x26888 }, { 19325, 0x223D0 }, { 19326, 0x5F58 }, { 19327, 0x22471 },
    { 19328, 0x5F63 }, { 19329, 0x97BD }, { 19330, 0x26E6E }, { 19331, 0x5F72 }, { 19332, 0x9340 }, { 19333, 0x28A36 }, { 19334, 0x5FA7 }, { 19335, 0x5DB6 },
    { 19336, 0x3D5F }, { 19337, 0x25250 }, { 19338, 0x21F6A }, { 19339, 0x270F8 }, { 19340, 0x22668 }, { 19341, 0x91D6 }, { 19342, 0x2029E }, { 19343, 0x28A29 },
    { 19344, 0x6031 }, { 19345, 0x6685 }, { 19346, 0x21877 }, { 19347, 0x3963 }, { 19348, 0x3DC7 }, { 19349, 0x3639 }, { 19350, 0x5790 }, { 19351, 0x227B4 },
    { 19352, 0x7971 }, { 19353, 0x3E40 }, { 19354, 0x609E }, { 19355, 0x60A4 }, { 19356, 0x60B3 }, { 19357, 0x24982 }, { 19358, 0x2498F }, { 19359, 0x27A53 },
    { 19360, 0x74A4 }, { 19361, 0x50E1 }, { 19362, 0x5AA0 }, { 19363, 0x6164 }, { 19364, 0x8424 }, { 19365, 0x6142 }, { 19366, 0x2F8A6 }, { 19367, 0x26ED2 },
    { 19368, 0x6181 }, { 19369, 0x51F4 }, { 19370, 0x20656 }, { 19371, 0x6187 }, { 19372, 0x5BAA }, { 19373, 0x23FB7 }, { 19374, 0x2285F }, { 19375, 0x61D3 },
    { 19376, 0x28B9D }, { 19377, 0x2995D }, { 19378, 0x61D0 }, { 19379, 0x3932 }, { 19380, 0x22980 }, { 19381, 0x228C1 }, { 19382, 0x6023 }, { 19383, 0x615C },
    { 19384, 0x651E }, { 19385, 0x638B }, { 19386, 0x20118 }, { 19387, 0x62C5 }, { 19388, 0x21770 }, { 19389, 0x62D5 }, { 19390, 0x22E0D }, { 19391, 0x636C },
    { 19392, 0x249DF }, { 19393, 0x3A17 }, { 19394, 0x6438 }, { 19395, 0x63F8 }, { 19396, 0x2138E }, { 19397, 0x217FC }, { 19398, 0x6490 }, { 19399, 0x6F8A },
    { 19400, 0x22E36 }, { 19401, 0x9814 }, { 19402, 0x2408C }, { 19403, 0x2571D }, { 19404, 0x64E1 }, { 19405, 0x64E5 }, { 19406, 0x947B }, { 19407, 0x3A66 },
    { 19408, 0x643A }, { 19409, 0x3A57 }, { 19410, 0x654D }, { 19411, 0x6F16 }, { 19412, 0x24A28 }, { 19413, 0x24A23 }, { 19414, 0x6585 }, { 19415, 0x656D },
    { 19416, 0x655F }, { 19417, 0x2307E }, { 19418, 0x65B5 }, { 19419, 0x24940 }, { 19420, 0x4B37 }, { 19421, 0x65D1 }, { 19422, 0x40D8 }, { 19423, 0x21829 },
    { 19424, 0x65E0 }, { 19425, 0x65E3 }, { 19426, 0x5FDF }, { 19427, 0x23400 }, { 19428, 0x6618 }, { 19429, 0x231F7 }, { 19430, 0x231F8 }, { 19431, 0x6644 },
    { 19432, 0x231A4 }, { 19433, 0x231A5 }, { 19434, 0x664B }, { 19435, 0x20E75 }, { 19436, 0x6667 }, { 19437, 0x251E6 }, { 19438, 0x6673 }, { 19439, 0x6674 },
    { 19440, 0x21E3D }, { 19441, 0x23231 }, { 19442, 0x285F4 }, { 19443, 0x231C8 }, { 19444, 0x25313 }, { 19445, 0x77C5 }, { 19446, 0x228F7 }, { 19447, 0x99A4 },
    { 19448, 0x6702 }, { 19449, 0x2439C }, { 19450, 0x24A21 }, { 19451, 0x3B2B }, { 19452, 0x69FA }, { 19453, 0x237C2 }, { 19454, 0x675E }, { 19455, 0x6767 },
    { 19456, 0x6762 }, { 19457, 0x241CD }, { 19458, 0x290ED }, { 19459, 0x67D7 }, { 19460, 0x44E9 }, { 19461, 0x6822 }, { 19462, 0x6E50 }, { 19463, 0x923C },
    { 19464, 0x6801 }, { 19465, 0x233E6 }, { 19466, 0x26DA0 }, { 19467, 0x685D }, { 19468, 0x2346F }, { 19469, 0x69E1 }, { 19470, 0x6A0B }, { 19471, 0x28ADF },
    { 19472, 0x6973 }, { 19473, 0x68C3 }, { 19474, 0x235CD }, { 19475, 0x6901 }, { 19476, 0x6900 }, { 19477, 0x3D32 }, { 19478, 0x3A01 }, { 19479, 0x2363C },
    { 19480, 0x3B80 }, { 19481, 0x67AC }, { 19482, 0x6961 }, { 19483, 0x28A4A }, { 19484, 0x42FC }, { 19485, 0x6936 }, { 19486, 0x6998 }, { 19487, 0x3BA1 },
    { 19488, 0x203C9 }, { 19489, 0x8363 }, { 19490, 0x5090 }, { 19491, 0x69F9 }, { 19492, 0x23659 }, { 19493, 0x2212A }, { 19494, 0x6A45 }, { 19495, 0x23703 },
    { 19496, 0x6A9D }, { 19497, 0x3BF3 }, { 19498, 0x67B1 }, { 19499, 0x6AC8 }, { 19500, 0x2919C }, { 19501, 0x3C0D }, { 19502, 0x6B1D }, { 19503, 0x20923 },
    { 19504, 0x60DE }, { 19505, 0x6B35 }, { 19506, 0x6B74 }, { 19507, 0x227CD }, { 19508, 0x6EB5 }, { 19509, 0x23ADB }, { 19510, 0x203B5 }, { 19511, 0x21958 },
    { 19512, 0x3740 }, { 19513, 0x5421 }, { 19514, 0x23B5A }, { 19515, 0x6BE1 }, { 19516, 0x23EFC }, { 19517, 0x6BDC }, { 19518, 0x6C37 }, { 19519, 0x2248B },
    { 19520, 0x248F1 }, { 19521, 0x26B51 }, { 19522, 0x6C5A }, { 19523, 0x8226 }, { 19524, 0x6C79 }, { 19525, 0x23DBC }, { 19526, 0x44C5 }, { 19527, 0x23DBD },
    { 19528, 0x241A4 }, { 19529, 0x2490C }, { 19530, 0x24900 }, { 19531, 0x23CC9 }, { 19532, 0x36E5 }, { 19533, 0x3CEB }, { 19534, 0x20D32 }, { 19535, 0x9B83 },
    { 19536, 0x231F9 }, { 19537, 0x22491 }, { 19538, 0x7F8F }, { 19539, 0x6837 }, { 19540, 0x26D25 }, { 19541, 0x26DA1 }, { 19542, 0x26DEB }, { 19543, 0x6D96 },
    { 19544, 0x6D5C }, { 19545, 0x6E7C }, { 19546, 0x6F04 }, { 19547, 0x2497F }, { 19548, 0x24085 }, { 19549, 0x26E72 }, { 19550, 0x8533 }, { 19551, 0x26F74 },
    { 19552, 0x51C7 }, { 19553, 0x6C9C }, { 19554, 0x6E1D }, { 19555, 0x842E }, { 19556, 0x28B21 }, { 19557, 0x6E2F }, { 19558, 0x23E2F }, { 19559, 0x7453 },
    { 19560, 0x23F82 }, { 19561, 0x79CC }, { 19562, 0x6E4F }, { 19563, 0x5A91 }, { 19564, 0x2304B }, { 19565, 0x6FF8 }, { 19566, 0x370D }, { 19567, 0x6F9D },
    { 19568, 0x23E30 }, { 19569, 0x6EFA }, { 19570, 0x21497 }, { 19571, 0x2403D }, { 19572, 0x4555 }, { 19573, 0x93F0 }, { 19574, 0x6F44 }, { 19575, 0x6F5C },
    { 19576, 0x3D4E }, { 19577, 0x6F74 }, { 19578, 0x29170 }, { 19579, 0x3D3B }, { 19580, 0x6F9F }, { 19581, 0x24144 }, { 19582, 0x6FD3 }, { 19583, 0x24091 },
    { 19584, 0x24155 }, { 19585, 0x24039 }, { 19586, 0x23FF0 }, { 19587, 0x23FB4 }, { 19588, 0x2413F }, { 19589, 0x51DF }, { 19590, 0x24156 }, { 19591, 0x24157 },
    { 19592, 0x24140 }, { 19593, 0x261DD }, { 19594, 0x704B }, { 19595, 0x707E }, { 19596, 0x70A7 }, { 19597, 0x7081 }, { 19598, 0x70CC }, { 19599, 0x70D5 },
    { 19600, 0x70D6 }, { 19601, 0x70DF }, { 19602, 0x4104 }, { 19603, 0x3DE8 }, { 19604, 0x71B4 }, { 19605, 0x7196 }, { 19606, 0x24277 }, { 19607, 0x712B },
    { 19608, 0x7145 }, { 19609, 0x5A88 }, { 19610, 0x714A }, { 19611, 0x716E }, { 19612, 0x5C9C }, { 19613, 0x24365 }, { 19614, 0x714F }, { 19615, 0x9362 },
    { 19616, 0x242C1 }, { 19617, 0x712C }, { 19618, 0x2445A }, { 19619, 0x24A27 }, { 19620, 0x24A22 }, { 19621, 0x71BA }, { 19622, 0x28BE8 }, { 19623, 0x70BD },
    { 19624, 0x720E }, { 19625, 0x9442 }, { 19626, 0x7215 }, { 19627, 0x5911 }, { 19628, 0x9443 }, { 19629, 0x7224 }, { 19630, 0x9341 }, { 19631, 0x25605 },
    { 19632, 0x722E }, { 19633, 0x7240 }, { 19634, 0x24974 }, { 19635, 0x68BD }, { 19636, 0x7255 }, { 19637, 0x7257 }, { 19638, 0x3E55 }, { 19639, 0x23044 },
    { 19640, 0x680D }, { 19641, 0x6F3D }, { 19642, 0x7282 }, { 19643, 0x732A }, { 19644, 0x732B }, { 19645, 0x24823 }, { 19646, 0x2882B }, { 19647, 0x48ED },
    { 19648, 0x28804 }, { 19649, 0x7328 }, { 19650, 0x732E }, { 19651, 0x73CF }, { 19652, 0x73AA }, { 19653, 0x20C3A }, { 19654, 0x26A2E }, { 19655, 0x73C9 },
    { 19656, 0x7449 }, { 19657, 0x241E2 }, { 19658, 0x216E7 }, { 19659, 0x24A24 }, { 19660, 0x6623 }, { 19661, 0x36C5 }, { 19662, 0x249B7 }, { 19663, 0x2498D },
    { 19664, 0x249FB }, { 19665, 0x73F7 }, { 19666, 0x7415 }, { 19667, 0x6903 }, { 19668, 0x24A26 }, { 19669, 0x7439 }, { 19670, 0x205C3 }, { 19671, 0x3ED7 },
    { 19672, 0x745C }, { 19673, 0x228AD }, { 19674, 0x7460 }, { 19675, 0x28EB2 }, { 19676, 0x7447 }, { 19677, 0x73E4 }, { 19678, 0x7476 }, { 19679, 0x83B9 },
    { 19680, 0x746C }, { 19681, 0x3730 }, { 19682, 0x7474 }, { 19683, 0x93F1 }, { 19684, 0x6A2C }, { 19685, 0x7482 }, { 19686, 0x4953 }, { 19687, 0x24A8C },
    { 19688, 0x2415F }, { 19689, 0x24A79 }, { 19690, 0x28B8F }, { 19691, 0x5B46 }, { 19692, 0x28C03 }, { 19693, 0x2189E }, { 19694, 0x74C8 }, { 19695, 0x21988 },
    { 19696, 0x750E }, { 19697, 0x74E9 }, { 19698, 0x751E }, { 19699, 0x28ED9 }, { 19700, 0x21A4B }, { 19701, 0x5BD7 }, { 19702, 0x28EAC }, { 19703, 0x9385 },
    { 19704, 0x754D }, { 19705, 0x754A }, { 19706, 0x7567 }, { 19707, 0x756E }, { 19708, 0x24F82 }, { 19709, 0x3F04 }, { 19710, 0x24D13 }, { 19711, 0x758E },
    { 19712, 0x745D }, { 19713, 0x759E }, { 19714, 0x75B4 }, { 19715, 0x7602 }, { 19716, 0x762C }, { 19717, 0x7651 }, { 19718, 0x764F }, { 19719, 0x766F },
    { 19720, 0x7676 }, { 19721, 0x263F5 }, { 19722, 0x7690 }, { 19723, 0x81EF }, { 19724, 0x37F8 }, { 19725, 0x26911 }, { 19726, 0x2690E }, { 19727, 0x76A1 },
    { 19728, 0x76A5 }, { 19729, 0x76B7 }, { 19730, 0x76CC }, { 19731, 0x26F9F }, { 19732, 0x8462 }, { 19733, 0x2509D }, { 19734, 0x2517D }, { 19735, 0x21E1C },
    { 19736, 0x771E }, { 19737, 0x7726 }, { 19738, 0x7740 }, { 19739, 0x64AF }, { 19740, 0x25220 }, { 19741, 0x7758 }, { 19742, 0x232AC }, { 19743, 0x77AF },
    { 19744, 0x28964 }, { 19745, 0x28968 }, { 19746, 0x216C1 }, { 19747, 0x77F4 }, { 19748, 0x7809 }, { 19749, 0x21376 }, { 19750, 0x24A12 }, { 19751, 0x68CA },
    { 19752, 0x78AF }, { 19753, 0x78C7 }, { 19754, 0x78D3 }, { 19755, 0x96A5 }, { 19756, 0x792E }, { 19757, 0x255E0 }, { 19758, 0x78D7 }, { 19759, 0x7934 },
    { 19760, 0x78B1 }, { 19761, 0x2760C }, { 19762, 0x8FB8 }, { 19763, 0x8884 }, { 19764, 0x28B2B }, { 19765, 0x26083 }, { 19766, 0x2261C }, { 19767, 0x7986 },
    { 19768, 0x8900 }, { 19769, 0x6902 }, { 19770, 0x7980 }, { 19771, 0x25857 }, { 19772, 0x799D }, { 19773, 0x27B39 }, { 19774, 0x793C }, { 19775, 0x79A9 },
    { 19776, 0x6E2A }, { 19777, 0x27126 }, { 19778, 0x3EA8 }, { 19779, 0x79C6 }, { 19780, 0x2910D }, { 19781, 0x79D4 },
} };
#endif // ASSERT_ENABLED

// These are values from https://encoding.spec.whatwg.org/index-big5.txt that are not in ICU.
constexpr std::array<std::pair<uint16_t, UChar32>, 5088> big5Extras { {
    { 942, 0x43f0 }, { 943, 0x4c32 }, { 944, 0x4603 }, { 945, 0x45a6 }, { 946, 0x4578 }, { 947, 0x27267 }, { 948, 0x4d77 }, { 949, 0x45b3 },
    { 950, 0x27cb1 }, { 951, 0x4ce2 }, { 952, 0x27cc5 }, { 953, 0x3b95 }, { 954, 0x4736 }, { 955, 0x4744 }, { 956, 0x4c47 }, { 957, 0x4c40 },
    { 958, 0x242bf }, { 959, 0x23617 }, { 960, 0x27352 }, { 961, 0x26e8b }, { 962, 0x270d2 }, { 963, 0x4c57 }, { 964, 0x2a351 }, { 965, 0x474f },
    { 966, 0x45da }, { 967, 0x4c85 }, { 968, 0x27c6c }, { 969, 0x4d07 }, { 970, 0x4aa4 }, { 971, 0x46a1 }, { 972, 0x26b23 }, { 973, 0x7225 },
    { 974, 0x25a54 }, { 975, 0x21a63 }, { 976, 0x23e06 }, { 977, 0x23f61 }, { 978, 0x664d }, { 979, 0x56fb }, { 981, 0x7d95 }, { 982, 0x591d },
    { 983, 0x28bb9 }, { 984, 0x3df4 }, { 985, 0x9734 }, { 986, 0x27bef }, { 987, 0x5bdb }, { 988, 0x21d5e }, { 989, 0x5aa4 }, { 990, 0x3625 },
    { 991, 0x29eb0 }, { 992, 0x5ad1 }, { 993, 0x5bb7 }, { 994, 0x5cfc }, { 995, 0x676e }, { 996, 0x8593 }, { 997, 0x29945 }, { 998, 0x7461 },
    { 999, 0x749d }, { 1000, 0x3875 }, { 1001, 0x21d53 }, { 1002, 0x2369e }, { 1003, 0x26021 }, { 1004, 0x3eec }, { 1005, 0x258de }, { 1006, 0x3af5 },
    { 1007, 0x7afc }, { 1008, 0x9f97 }, { 1009, 0x24161 }, { 1010, 0x2890d }, { 1011, 0x231ea }, { 1012, 0x20a8a }, { 1013, 0x2325e }, { 1014, 0x430a },
    { 1015, 0x8484 }, { 1016, 0x9f96 }, { 1017, 0x942f }, { 1018, 0x4930 }, { 1019, 0x8613 }, { 1020, 0x5896 }, { 1021, 0x974a }, { 1022, 0x9218 },
    { 1023, 0x79d0 }, { 1024, 0x7a32 }, { 1025, 0x6660 }, { 1026, 0x6a29 }, { 1027, 0x889d }, { 1028, 0x744c }, { 1029, 0x7bc5 }, { 1030, 0x6782 },
    { 1031, 0x7a2c }, { 1032, 0x524f }, { 1033, 0x9046 }, { 1034, 0x34e6 }, { 1035, 0x73c4 }, { 1036, 0x25db9 }, { 1037, 0x74c6 }, { 1038, 0x9fc7 },
    { 1039, 0x57b3 }, { 1040, 0x492f }, { 1041, 0x544c }, { 1042, 0x4131 }, { 1043, 0x2368e }, { 1044, 0x5818 }, { 1045, 0x7a72 }, { 1046, 0x27b65 },
    { 1047, 0x8b8f }, { 1048, 0x46ae }, { 1049, 0x26e88 }, { 1050, 0x4181 }, { 1051, 0x25d99 }, { 1052, 0x7bae }, { 1053, 0x224bc }, { 1054, 0x9fc8 },
    { 1055, 0x224c1 }, { 1056, 0x224c9 }, { 1057, 0x224cc }, { 1058, 0x9fc9 }, { 1059, 0x8504 }, { 1060, 0x235bb }, { 1061, 0x40b4 }, { 1062, 0x9fca },
    { 1063, 0x44e1 }, { 1064, 0x2adff }, { 1065, 0x62c1 }, { 1066, 0x706e }, { 1067, 0x9fcb }, { 1099, 0x31c0 }, { 1100, 0x31c1 }, { 1101, 0x31c2 },
    { 1102, 0x31c3 }, { 1103, 0x31c4 }, { 1104, 0x2010c }, { 1105, 0x31c5 }, { 1106, 0x200d1 }, { 1107, 0x200cd }, { 1108, 0x31c6 }, { 1109, 0x31c7 },
    { 1110, 0x200cb }, { 1111, 0x21fe8 }, { 1112, 0x31c8 }, { 1113, 0x200ca }, { 1114, 0x31c9 }, { 1115, 0x31ca }, { 1116, 0x31cb }, { 1117, 0x31cc },
    { 1118, 0x2010e }, { 1119, 0x31cd }, { 1120, 0x31ce }, { 1121, 0x100 }, { 1122, 0xc1 }, { 1123, 0x1cd }, { 1124, 0xc0 }, { 1125, 0x112 },
    { 1126, 0xc9 }, { 1127, 0x11a }, { 1128, 0xc8 }, { 1129, 0x14c }, { 1130, 0xd3 }, { 1131, 0x1d1 }, { 1132, 0xd2 }, { 1134, 0x1ebe },
    { 1136, 0x1ec0 }, { 1137, 0xca }, { 1138, 0x101 }, { 1139, 0xe1 }, { 1140, 0x1ce }, { 1141, 0xe0 }, { 1142, 0x251 }, { 1143, 0x113 },
    { 1144, 0xe9 }, { 1145, 0x11b }, { 1146, 0xe8 }, { 1147, 0x12b }, { 1148, 0xed }, { 1149, 0x1d0 }, { 1150, 0xec }, { 1151, 0x14d },
    { 1152, 0xf3 }, { 1153, 0x1d2 }, { 1154, 0xf2 }, { 1155, 0x16b }, { 1156, 0xfa }, { 1157, 0x1d4 }, { 1158, 0xf9 }, { 1159, 0x1d6 },
    { 1160, 0x1d8 }, { 1161, 0x1da }, { 1162, 0x1dc }, { 1163, 0xfc }, { 1165, 0x1ebf }, { 1167, 0x1ec1 }, { 1168, 0xea }, { 1169, 0x261 },
    { 1170, 0x23da }, { 1171, 0x23db }, { 1256, 0x2a3a9 }, { 1257, 0x21145 }, { 1259, 0x650a }, { 1262, 0x4e3d }, { 1263, 0x6edd }, { 1264, 0x9d4e },
    { 1265, 0x91df }, { 1268, 0x27735 }, { 1269, 0x6491 }, { 1270, 0x4f1a }, { 1271, 0x4f28 }, { 1272, 0x4fa8 }, { 1273, 0x5156 }, { 1274, 0x5174 },
    { 1275, 0x519c }, { 1276, 0x51e4 }, { 1277, 0x52a1 }, { 1278, 0x52a8 }, { 1279, 0x533b }, { 1280, 0x534e }, { 1281, 0x53d1 }, { 1282, 0x53d8 },
    { 1283, 0x56e2 }, { 1284, 0x58f0 }, { 1285, 0x5904 }, { 1286, 0x5907 }, { 1287, 0x5932 }, { 1288, 0x5934 }, { 1289, 0x5b66 }, { 1290, 0x5b9e },
    { 1291, 0x5b9f }, { 1292, 0x5c9a }, { 1293, 0x5e86 }, { 1294, 0x603b }, { 1295, 0x6589 }, { 1296, 0x67fe }, { 1297, 0x6804 }, { 1298, 0x6865 },
    { 1299, 0x6d4e }, { 1300, 0x70bc }, { 1301, 0x7535 }, { 1302, 0x7ea4 }, { 1303, 0x7eac }, { 1304, 0x7eba }, { 1305, 0x7ec7 }, { 1306, 0x7ecf },
    { 1307, 0x7edf }, { 1308, 0x7f06 }, { 1309, 0x7f37 }, { 1310, 0x827a }, { 1311, 0x82cf }, { 1312, 0x836f }, { 1313, 0x89c6 }, { 1314, 0x8bbe },
    { 1315, 0x8be2 }, { 1316, 0x8f66 }, { 1317, 0x8f67 }, { 1318, 0x8f6e }, { 1319, 0x7411 }, { 1320, 0x7cfc }, { 1321, 0x7dcd }, { 1322, 0x6946 },
    { 1323, 0x7ac9 }, { 1324, 0x5227 }, { 1329, 0x918c }, { 1330, 0x78b8 }, { 1331, 0x915e }, { 1332, 0x80bc }, { 1334, 0x8d0b }, { 1335, 0x80f6 },
    { 1336, 0x209e7 }, { 1339, 0x809f }, { 1340, 0x9ec7 }, { 1341, 0x4ccd }, { 1342, 0x9dc9 }, { 1343, 0x9e0c }, { 1344, 0x4c3e }, { 1345, 0x29df6 },
    { 1346, 0x2700e }, { 1347, 0x9e0a }, { 1348, 0x2a133 }, { 1349, 0x35c1 }, { 1351, 0x6e9a }, { 1352, 0x823e }, { 1353, 0x7519 }, { 1355, 0x4911 },
    { 1356, 0x9a6c }, { 1357, 0x9a8f }, { 1358, 0x9f99 }, { 1359, 0x7987 }, { 1360, 0x2846c }, { 1361, 0x21dca }, { 1362, 0x205d0 }, { 1363, 0x22ae6 },
    { 1364, 0x4e24 }, { 1365, 0x4e81 }, { 1366, 0x4e80 }, { 1367, 0x4e87 }, { 1368, 0x4ebf }, { 1369, 0x4eeb }, { 1370, 0x4f37 }, { 1371, 0x344c },
    { 1372, 0x4fbd }, { 1373, 0x3e48 }, { 1374, 0x5003 }, { 1375, 0x5088 }, { 1376, 0x347d }, { 1377, 0x3493 }, { 1378, 0x34a5 }, { 1379, 0x5186 },
    { 1380, 0x5905 }, { 1381, 0x51db }, { 1382, 0x51fc }, { 1383, 0x5205 }, { 1384, 0x4e89 }, { 1385, 0x5279 }, { 1386, 0x5290 }, { 1387, 0x5327 },
    { 1388, 0x35c7 }, { 1389, 0x53a9 }, { 1390, 0x3551 }, { 1391, 0x53b0 }, { 1392, 0x3553 }, { 1393, 0x53c2 }, { 1394, 0x5423 }, { 1395, 0x356d },
    { 1396, 0x3572 }, { 1397, 0x3681 }, { 1398, 0x5493 }, { 1399, 0x54a3 }, { 1400, 0x54b4 }, { 1401, 0x54b9 }, { 1402, 0x54d0 }, { 1403, 0x54ef },
    { 1404, 0x5518 }, { 1405, 0x5523 }, { 1406, 0x5528 }, { 1407, 0x3598 }, { 1408, 0x553f }, { 1409, 0x35a5 }, { 1410, 0x35bf }, { 1411, 0x55d7 },
    { 1412, 0x35c5 }, { 1413, 0x27d84 }, { 1414, 0x5525 }, { 1416, 0x20c42 }, { 1417, 0x20d15 }, { 1418, 0x2512b }, { 1419, 0x5590 }, { 1420, 0x22cc6 },
    { 1421, 0x39ec }, { 1422, 0x20341 }, { 1423, 0x8e46 }, { 1424, 0x24db8 }, { 1425, 0x294e5 }, { 1426, 0x4053 }, { 1427, 0x280be }, { 1428, 0x777a },
    { 1429, 0x22c38 }, { 1430, 0x3a34 }, { 1431, 0x47d5 }, { 1432, 0x2815d }, { 1433, 0x269f2 }, { 1434, 0x24dea }, { 1435, 0x64dd }, { 1436, 0x20d7c },
    { 1437, 0x20fb4 }, { 1438, 0x20cd5 }, { 1439, 0x210f4 }, { 1440, 0x648d }, { 1441, 0x8e7e }, { 1442, 0x20e96 }, { 1443, 0x20c0b }, { 1444, 0x20f64 },
    { 1445, 0x22ca9 }, { 1446, 0x28256 }, { 1447, 0x244d3 }, { 1449, 0x20d46 }, { 1450, 0x29a4d }, { 1451, 0x280e9 }, { 1452, 0x47f4 }, { 1453, 0x24ea7 },
    { 1454, 0x22cc2 }, { 1455, 0x9ab2 }, { 1456, 0x3a67 }, { 1457, 0x295f4 }, { 1458, 0x3fed }, { 1459, 0x3506 }, { 1460, 0x252c7 }, { 1461, 0x297d4 },
    { 1462, 0x278c8 }, { 1463, 0x22d44 }, { 1464, 0x9d6e }, { 1465, 0x9815 }, { 1467, 0x43d9 }, { 1468, 0x260a5 }, { 1469, 0x64b4 }, { 1470, 0x54e3 },
    { 1471, 0x22d4c }, { 1472, 0x22bca }, { 1473, 0x21077 }, { 1474, 0x39fb }, { 1475, 0x2106f }, { 1476, 0x266da }, { 1477, 0x26716 }, { 1478, 0x279a0 },
    { 1479, 0x64ea }, { 1480, 0x25052 }, { 1481, 0x20c43 }, { 1482, 0x8e68 }, { 1483, 0x221a1 }, { 1484, 0x28b4c }, { 1485, 0x20731 }, { 1487, 0x480b },
    { 1488, 0x201a9 }, { 1489, 0x3ffa }, { 1490, 0x5873 }, { 1491, 0x22d8d }, { 1493, 0x245c8 }, { 1494, 0x204fc }, { 1495, 0x26097 }, { 1496, 0x20f4c },
    { 1497, 0x20d96 }, { 1498, 0x5579 }, { 1499, 0x40bb }, { 1500, 0x43ba }, { 1502, 0x4ab4 }, { 1503, 0x22a66 }, { 1504, 0x2109d }, { 1505, 0x81aa },
    { 1506, 0x98f5 }, { 1507, 0x20d9c }, { 1508, 0x6379 }, { 1509, 0x39fe }, { 1510, 0x22775 }, { 1511, 0x8dc0 }, { 1512, 0x56a1 }, { 1513, 0x647c },
    { 1514, 0x3e43 }, { 1516, 0x2a601 }, { 1517, 0x20e09 }, { 1518, 0x22acf }, { 1519, 0x22cc9 }, { 1521, 0x210c8 }, { 1522, 0x239c2 }, { 1523, 0x3992 },
    { 1524, 0x3a06 }, { 1525, 0x2829b }, { 1526, 0x3578 }, { 1527, 0x25e49 }, { 1528, 0x220c7 }, { 1529, 0x5652 }, { 1530, 0x20f31 }, { 1531, 0x22cb2 },
    { 1532, 0x29720 }, { 1533, 0x34bc }, { 1534, 0x6c3d }, { 1535, 0x24e3b }, { 1538, 0x27574 }, { 1539, 0x22e8b }, { 1540, 0x22208 }, { 1541, 0x2a65b },
    { 1542, 0x28ccd }, { 1543, 0x20e7a }, { 1544, 0x20c34 }, { 1545, 0x2681c }, { 1546, 0x7f93 }, { 1547, 0x210cf }, { 1548, 0x22803 }, { 1549, 0x22939 },
    { 1550, 0x35fb }, { 1551, 0x251e3 }, { 1552, 0x20e8c }, { 1553, 0x20f8d }, { 1554, 0x20eaa }, { 1555, 0x3f93 }, { 1556, 0x20f30 }, { 1557, 0x20d47 },
    { 1558, 0x2114f }, { 1559, 0x20e4c }, { 1561, 0x20eab }, { 1562, 0x20ba9 }, { 1563, 0x20d48 }, { 1564, 0x210c0 }, { 1565, 0x2113d }, { 1566, 0x3ff9 },
    { 1567, 0x22696 }, { 1568, 0x6432 }, { 1569, 0x20fad }, { 1570, 0x233f4 }, { 1571, 0x27639 }, { 1572, 0x22bce }, { 1573, 0x20d7e }, { 1574, 0x20d7f },
    { 1575, 0x22c51 }, { 1576, 0x22c55 }, { 1577, 0x3a18 }, { 1578, 0x20e98 }, { 1579, 0x210c7 }, { 1580, 0x20f2e }, { 1581, 0x2a632 }, { 1582, 0x26b50 },
    { 1583, 0x28cd2 }, { 1584, 0x28d99 }, { 1585, 0x28cca }, { 1586, 0x95aa }, { 1587, 0x54cc }, { 1588, 0x82c4 }, { 1589, 0x55b9 }, { 1591, 0x29ec3 },
    { 1592, 0x9c26 }, { 1593, 0x9ab6 }, { 1594, 0x2775e }, { 1595, 0x22dee }, { 1596, 0x7140 }, { 1597, 0x816d }, { 1598, 0x80ec }, { 1599, 0x5c1c },
    { 1600, 0x26572 }, { 1601, 0x8134 }, { 1602, 0x3797 }, { 1603, 0x535f }, { 1604, 0x280bd }, { 1605, 0x91b6 }, { 1606, 0x20efa }, { 1607, 0x20e0f },
    { 1608, 0x20e77 }, { 1609, 0x20efb }, { 1610, 0x35dd }, { 1611, 0x24deb }, { 1612, 0x3609 }, { 1613, 0x20cd6 }, { 1614, 0x56af }, { 1615, 0x227b5 },
    { 1616, 0x210c9 }, { 1617, 0x20e10 }, { 1618, 0x20e78 }, { 1619, 0x21078 }, { 1620, 0x21148 }, { 1621, 0x28207 }, { 1622, 0x21455 }, { 1623, 0x20e79 },
    { 1624, 0x24e50 }, { 1625, 0x22da4 }, { 1626, 0x5a54 }, { 1627, 0x2101d }, { 1628, 0x2101e }, { 1629, 0x210f5 }, { 1630, 0x210f6 }, { 1631, 0x579c },
    { 1632, 0x20e11 }, { 1633, 0x27694 }, { 1634, 0x282cd }, { 1635, 0x20fb5 }, { 1636, 0x20e7b }, { 1637, 0x2517e }, { 1638, 0x3703 }, { 1639, 0x20fb6 },
    { 1640, 0x21180 }, { 1641, 0x252d8 }, { 1642, 0x2a2bd }, { 1643, 0x249da }, { 1644, 0x2183a }, { 1645, 0x24177 }, { 1646, 0x2827c }, { 1647, 0x5899 },
    { 1648, 0x5268 }, { 1649, 0x361a }, { 1650, 0x2573d }, { 1651, 0x7bb2 }, { 1652, 0x5b68 }, { 1653, 0x4800 }, { 1654, 0x4b2c }, { 1655, 0x9f27 },
    { 1656, 0x49e7 }, { 1657, 0x9c1f }, { 1658, 0x9b8d }, { 1659, 0x25b74 }, { 1660, 0x2313d }, { 1661, 0x55fb }, { 1662, 0x35f2 }, { 1663, 0x5689 },
    { 1664, 0x4e28 }, { 1665, 0x5902 }, { 1666, 0x21bc1 }, { 1667, 0x2f878 }, { 1668, 0x9751 }, { 1669, 0x20086 }, { 1670, 0x4e5b }, { 1671, 0x4ebb },
    { 1672, 0x353e }, { 1673, 0x5c23 }, { 1674, 0x5f51 }, { 1675, 0x5fc4 }, { 1676, 0x38fa }, { 1677, 0x624c }, { 1678, 0x6535 }, { 1679, 0x6b7a },
    { 1680, 0x6c35 }, { 1681, 0x6c3a }, { 1682, 0x706c }, { 1683, 0x722b }, { 1684, 0x4e2c }, { 1685, 0x72ad }, { 1686, 0x248e9 }, { 1687, 0x7f52 },
    { 1688, 0x793b }, { 1689, 0x7cf9 }, { 1690, 0x7f53 }, { 1691, 0x2626a }, { 1692, 0x34c1 }, { 1694, 0x2634b }, { 1695, 0x8002 }, { 1696, 0x8080 },
    { 1697, 0x26612 }, { 1698, 0x26951 }, { 1699, 0x535d }, { 1700, 0x8864 }, { 1701, 0x89c1 }, { 1702, 0x278b2 }, { 1703, 0x8ba0 }, { 1704, 0x8d1d },
    { 1705, 0x9485 }, { 1706, 0x9578 }, { 1707, 0x957f }, { 1708, 0x95e8 }, { 1709, 0x28e0f }, { 1710, 0x97e6 }, { 1711, 0x9875 }, { 1712, 0x98ce },
    { 1713, 0x98de }, { 1714, 0x9963 }, { 1715, 0x29810 }, { 1716, 0x9c7c }, { 1717, 0x9e1f }, { 1718, 0x9ec4 }, { 1719, 0x6b6f }, { 1720, 0xf907 },
    { 1721, 0x4e37 }, { 1722, 0x20087 }, { 1723, 0x961d }, { 1724, 0x6237 }, { 1725, 0x94a2 }, { 1727, 0x503b }, { 1728, 0x6dfe }, { 1729, 0x29c73 },
    { 1730, 0x9fa6 }, { 1731, 0x3dc9 }, { 1732, 0x888f }, { 1733, 0x2414e }, { 1734, 0x7077 }, { 1735, 0x5cf5 }, { 1736, 0x4b20 }, { 1737, 0x251cd },
    { 1738, 0x3559 }, { 1739, 0x25d30 }, { 1740, 0x6122 }, { 1741, 0x28a32 }, { 1742, 0x8fa7 }, { 1743, 0x91f6 }, { 1744, 0x7191 }, { 1745, 0x6719 },
    { 1746, 0x73ba }, { 1747, 0x23281 }, { 1748, 0x2a107 }, { 1749, 0x3c8b }, { 1750, 0x21980 }, { 1751, 0x4b10 }, { 1752, 0x78e4 }, { 1753, 0x7402 },
    { 1754, 0x51ae }, { 1755, 0x2870f }, { 1756, 0x4009 }, { 1757, 0x6a63 }, { 1758, 0x2a2ba }, { 1759, 0x4223 }, { 1760, 0x860f }, { 1761, 0x20a6f },
    { 1762, 0x7a2a }, { 1763, 0x29947 }, { 1764, 0x28aea }, { 1765, 0x9755 }, { 1766, 0x704d }, { 1767, 0x5324 }, { 1768, 0x2207e }, { 1769, 0x93f4 },
    { 1770, 0x76d9 }, { 1771, 0x289e3 }, { 1772, 0x9fa7 }, { 1773, 0x77dd }, { 1774, 0x4ea3 }, { 1775, 0x4ff0 }, { 1776, 0x50bc }, { 1777, 0x4e2f },
    { 1778, 0x4f17 }, { 1779, 0x9fa8 }, { 1780, 0x5434 }, { 1781, 0x7d8b }, { 1782, 0x5892 }, { 1783, 0x58d0 }, { 1784, 0x21db6 }, { 1785, 0x5e92 },
    { 1786, 0x5e99 }, { 1787, 0x5fc2 }, { 1788, 0x22712 }, { 1789, 0x658b }, { 1790, 0x233f9 }, { 1791, 0x6919 }, { 1792, 0x6a43 }, { 1793, 0x23c63 },
    { 1794, 0x6cff }, { 1796, 0x7200 }, { 1797, 0x24505 }, { 1798, 0x738c }, { 1799, 0x3edb }, { 1800, 0x24a13 }, { 1801, 0x5b15 }, { 1802, 0x74b9 },
    { 1803, 0x8b83 }, { 1804, 0x25ca4 }, { 1805, 0x25695 }, { 1806, 0x7a93 }, { 1807, 0x7bec }, { 1808, 0x7cc3 }, { 1809, 0x7e6c }, { 1810, 0x82f8 },
    { 1811, 0x8597 }, { 1812, 0x9fa9 }, { 1813, 0x8890 }, { 1814, 0x9faa }, { 1815, 0x8eb9 }, { 1816, 0x9fab }, { 1817, 0x8fcf }, { 1818, 0x855f },
    { 1819, 0x99e0 }, { 1820, 0x9221 }, { 1821, 0x9fac }, { 1822, 0x28db9 }, { 1823, 0x2143f }, { 1824, 0x4071 }, { 1825, 0x42a2 }, { 1826, 0x5a1a },
    { 1830, 0x9868 }, { 1831, 0x676b }, { 1832, 0x4276 }, { 1833, 0x573d }, { 1835, 0x85d6 }, { 1836, 0x2497b }, { 1837, 0x82bf }, { 1838, 0x2710d },
    { 1839, 0x4c81 }, { 1840, 0x26d74 }, { 1841, 0x5d7b }, { 1842, 0x26b15 }, { 1843, 0x26fbe }, { 1844, 0x9fad }, { 1845, 0x9fae }, { 1846, 0x5b96 },
    { 1847, 0x9faf }, { 1848, 0x66e7 }, { 1849, 0x7e5b }, { 1850, 0x6e57 }, { 1851, 0x79ca }, { 1852, 0x3d88 }, { 1853, 0x44c3 }, { 1854, 0x23256 },
    { 1855, 0x22796 }, { 1856, 0x439a }, { 1857, 0x4536 }, { 1859, 0x5cd5 }, { 1860, 0x23b1a }, { 1861, 0x8af9 }, { 1862, 0x5c78 }, { 1863, 0x3d12 },
    { 1864, 0x23551 }, { 1865, 0x5d78 }, { 1866, 0x9fb2 }, { 1867, 0x7157 }, { 1868, 0x4558 }, { 1869, 0x240ec }, { 1870, 0x21e23 }, { 1871, 0x4c77 },
    { 1872, 0x3978 }, { 1873, 0x344a }, { 1874, 0x201a4 }, { 1875, 0x26c41 }, { 1876, 0x8acc }, { 1877, 0x4fb4 }, { 1878, 0x20239 }, { 1879, 0x59bf },
    { 1880, 0x816c }, { 1881, 0x9856 }, { 1882, 0x298fa }, { 1883, 0x5f3b }, { 1884, 0x20b9f }, { 1886, 0x221c1 }, { 1887, 0x2896d }, { 1888, 0x4102 },
    { 1889, 0x46bb }, { 1890, 0x29079 }, { 1891, 0x3f07 }, { 1892, 0x9fb3 }, { 1893, 0x2a1b5 }, { 1894, 0x40f8 }, { 1895, 0x37d6 }, { 1896, 0x46f7 },
    { 1897, 0x26c46 }, { 1898, 0x417c }, { 1899, 0x286b2 }, { 1900, 0x273ff }, { 1901, 0x456d }, { 1902, 0x38d4 }, { 1903, 0x2549a }, { 1904, 0x4561 },
    { 1905, 0x451b }, { 1906, 0x4d89 }, { 1907, 0x4c7b }, { 1908, 0x4d76 }, { 1909, 0x45ea }, { 1910, 0x3fc8 }, { 1911, 0x24b0f }, { 1912, 0x3661 },
    { 1913, 0x44de }, { 1914, 0x44bd }, { 1915, 0x41ed }, { 1916, 0x5d3e }, { 1917, 0x5d48 }, { 1918, 0x5d56 }, { 1919, 0x3dfc }, { 1920, 0x380f },
    { 1921, 0x5da4 }, { 1922, 0x5db9 }, { 1923, 0x3820 }, { 1924, 0x3838 }, { 1925, 0x5e42 }, { 1926, 0x5ebd }, { 1927, 0x5f25 }, { 1928, 0x5f83 },
    { 1929, 0x3908 }, { 1930, 0x3914 }, { 1931, 0x393f }, { 1932, 0x394d }, { 1933, 0x60d7 }, { 1934, 0x613d }, { 1935, 0x5ce5 }, { 1936, 0x3989 },
    { 1937, 0x61b7 }, { 1938, 0x61b9 }, { 1939, 0x61cf }, { 1940, 0x39b8 }, { 1941, 0x622c }, { 1942, 0x6290 }, { 1943, 0x62e5 }, { 1944, 0x6318 },
    { 1945, 0x39f8 }, { 1946, 0x56b1 }, { 1947, 0x3a03 }, { 1948, 0x63e2 }, { 1949, 0x63fb }, { 1950, 0x6407 }, { 1951, 0x645a }, { 1952, 0x3a4b },
    { 1953, 0x64c0 }, { 1954, 0x5d15 }, { 1955, 0x5621 }, { 1956, 0x9f9f }, { 1957, 0x3a97 }, { 1958, 0x6586 }, { 1959, 0x3abd }, { 1960, 0x65ff },
    { 1961, 0x6653 }, { 1962, 0x3af2 }, { 1963, 0x6692 }, { 1964, 0x3b22 }, { 1965, 0x6716 }, { 1966, 0x3b42 }, { 1967, 0x67a4 }, { 1968, 0x6800 },
    { 1969, 0x3b58 }, { 1970, 0x684a }, { 1971, 0x6884 }, { 1972, 0x3b72 }, { 1973, 0x3b71 }, { 1974, 0x3b7b }, { 1975, 0x6909 }, { 1976, 0x6943 },
    { 1977, 0x725c }, { 1978, 0x6964 }, { 1979, 0x699f }, { 1980, 0x6985 }, { 1981, 0x3bbc }, { 1982, 0x69d6 }, { 1983, 0x3bdd }, { 1984, 0x6a65 },
    { 1985, 0x6a74 }, { 1986, 0x6a71 }, { 1987, 0x6a82 }, { 1988, 0x3bec }, { 1989, 0x6a99 }, { 1990, 0x3bf2 }, { 1991, 0x6aab }, { 1992, 0x6ab5 },
    { 1993, 0x6ad4 }, { 1994, 0x6af6 }, { 1995, 0x6b81 }, { 1996, 0x6bc1 }, { 1997, 0x6bea }, { 1998, 0x6c75 }, { 1999, 0x6caa }, { 2000, 0x3ccb },
    { 2001, 0x6d02 }, { 2002, 0x6d06 }, { 2003, 0x6d26 }, { 2004, 0x6d81 }, { 2005, 0x3cef }, { 2006, 0x6da4 }, { 2007, 0x6db1 }, { 2008, 0x6e15 },
    { 2009, 0x6e18 }, { 2010, 0x6e29 }, { 2011, 0x6e86 }, { 2012, 0x289c0 }, { 2013, 0x6ebb }, { 2014, 0x6ee2 }, { 2015, 0x6eda }, { 2016, 0x9f7f },
    { 2017, 0x6ee8 }, { 2018, 0x6ee9 }, { 2019, 0x6f24 }, { 2020, 0x6f34 }, { 2021, 0x3d46 }, { 2022, 0x23f41 }, { 2023, 0x6f81 }, { 2024, 0x6fbe },
    { 2025, 0x3d6a }, { 2026, 0x3d75 }, { 2027, 0x71b7 }, { 2028, 0x5c99 }, { 2029, 0x3d8a }, { 2030, 0x702c }, { 2031, 0x3d91 }, { 2032, 0x7050 },
    { 2033, 0x7054 }, { 2034, 0x706f }, { 2035, 0x707f }, { 2036, 0x7089 }, { 2037, 0x20325 }, { 2038, 0x43c1 }, { 2039, 0x35f1 }, { 2040, 0x20ed8 },
    { 2041, 0x23ed7 }, { 2042, 0x57be }, { 2043, 0x26ed3 }, { 2044, 0x713e }, { 2045, 0x257e0 }, { 2046, 0x364e }, { 2047, 0x69a2 }, { 2048, 0x28be9 },
    { 2049, 0x5b74 }, { 2050, 0x7a49 }, { 2051, 0x258e1 }, { 2052, 0x294d9 }, { 2053, 0x7a65 }, { 2054, 0x7a7d }, { 2055, 0x259ac }, { 2056, 0x7abb },
    { 2057, 0x7ab0 }, { 2058, 0x7ac2 }, { 2059, 0x7ac3 }, { 2060, 0x71d1 }, { 2061, 0x2648d }, { 2062, 0x41ca }, { 2063, 0x7ada }, { 2064, 0x7add },
    { 2065, 0x7aea }, { 2066, 0x41ef }, { 2067, 0x54b2 }, { 2068, 0x25c01 }, { 2069, 0x7b0b }, { 2070, 0x7b55 }, { 2071, 0x7b29 }, { 2072, 0x2530e },
    { 2073, 0x25cfe }, { 2074, 0x7ba2 }, { 2075, 0x7b6f }, { 2076, 0x839c }, { 2077, 0x25bb4 }, { 2078, 0x26c7f }, { 2079, 0x7bd0 }, { 2080, 0x8421 },
    { 2081, 0x7b92 }, { 2082, 0x7bb8 }, { 2083, 0x25d20 }, { 2084, 0x3dad }, { 2085, 0x25c65 }, { 2086, 0x8492 }, { 2087, 0x7bfa }, { 2088, 0x7c06 },
    { 2089, 0x7c35 }, { 2090, 0x25cc1 }, { 2091, 0x7c44 }, { 2092, 0x7c83 }, { 2093, 0x24882 }, { 2094, 0x7ca6 }, { 2095, 0x667d }, { 2096, 0x24578 },
    { 2097, 0x7cc9 }, { 2098, 0x7cc7 }, { 2099, 0x7ce6 }, { 2100, 0x7c74 }, { 2101, 0x7cf3 }, { 2102, 0x7cf5 }, { 2103, 0x7cce }, { 2104, 0x7e67 },
    { 2105, 0x451d }, { 2106, 0x26e44 }, { 2107, 0x7d5d }, { 2108, 0x26ed6 }, { 2109, 0x748d }, { 2110, 0x7d89 }, { 2111, 0x7dab }, { 2112, 0x7135 },
    { 2113, 0x7db3 }, { 2114, 0x7dd2 }, { 2115, 0x24057 }, { 2116, 0x26029 }, { 2117, 0x7de4 }, { 2118, 0x3d13 }, { 2119, 0x7df5 }, { 2120, 0x217f9 },
    { 2121, 0x7de5 }, { 2122, 0x2836d }, { 2123, 0x7e1d }, { 2124, 0x26121 }, { 2125, 0x2615a }, { 2126, 0x7e6e }, { 2127, 0x7e92 }, { 2128, 0x432b },
    { 2129, 0x946c }, { 2130, 0x7e27 }, { 2131, 0x7f40 }, { 2132, 0x7f41 }, { 2133, 0x7f47 }, { 2134, 0x7936 }, { 2135, 0x262d0 }, { 2136, 0x99e1 },
    { 2137, 0x7f97 }, { 2138, 0x26351 }, { 2139, 0x7fa3 }, { 2140, 0x21661 }, { 2141, 0x20068 }, { 2142, 0x455c }, { 2143, 0x23766 }, { 2144, 0x4503 },
    { 2145, 0x2833a }, { 2146, 0x7ffa }, { 2147, 0x26489 }, { 2148, 0x8005 }, { 2149, 0x8008 }, { 2150, 0x801d }, { 2151, 0x8028 }, { 2152, 0x802f },
    { 2153, 0x2a087 }, { 2154, 0x26cc3 }, { 2155, 0x803b }, { 2156, 0x803c }, { 2157, 0x8061 }, { 2158, 0x22714 }, { 2159, 0x4989 }, { 2160, 0x26626 },
    { 2161, 0x23de3 }, { 2162, 0x266e8 }, { 2163, 0x6725 }, { 2164, 0x80a7 }, { 2165, 0x28a48 }, { 2166, 0x8107 }, { 2167, 0x811a }, { 2168, 0x58b0 },
    { 2169, 0x226f6 }, { 2170, 0x6c7f }, { 2171, 0x26498 }, { 2172, 0x24fb8 }, { 2173, 0x64e7 }, { 2174, 0x2148a }, { 2175, 0x8218 }, { 2176, 0x2185e },
    { 2177, 0x6a53 }, { 2178, 0x24a65 }, { 2179, 0x24a95 }, { 2180, 0x447a }, { 2181, 0x8229 }, { 2182, 0x20b0d }, { 2183, 0x26a52 }, { 2184, 0x23d7e },
    { 2185, 0x4ff9 }, { 2186, 0x214fd }, { 2187, 0x84e2 }, { 2188, 0x8362 }, { 2189, 0x26b0a }, { 2190, 0x249a7 }, { 2191, 0x23530 }, { 2192, 0x21773 },
    { 2193, 0x23df8 }, { 2194, 0x82aa }, { 2195, 0x691b }, { 2196, 0x2f994 }, { 2197, 0x41db }, { 2198, 0x854b }, { 2199, 0x82d0 }, { 2200, 0x831a },
    { 2201, 0x20e16 }, { 2202, 0x217b4 }, { 2203, 0x36c1 }, { 2204, 0x2317d }, { 2205, 0x2355a }, { 2206, 0x827b }, { 2207, 0x82e2 }, { 2208, 0x8318 },
    { 2209, 0x23e8b }, { 2210, 0x26da3 }, { 2211, 0x26b05 }, { 2212, 0x26b97 }, { 2213, 0x235ce }, { 2214, 0x3dbf }, { 2215, 0x831d }, { 2216, 0x55ec },
    { 2217, 0x8385 }, { 2218, 0x450b }, { 2219, 0x26da5 }, { 2220, 0x83ac }, { 2221, 0x83c1 }, { 2222, 0x83d3 }, { 2223, 0x347e }, { 2224, 0x26ed4 },
    { 2225, 0x6a57 }, { 2226, 0x855a }, { 2227, 0x3496 }, { 2228, 0x26e42 }, { 2229, 0x22eef }, { 2230, 0x8458 }, { 2231, 0x25be4 }, { 2232, 0x8471 },
    { 2233, 0x3dd3 }, { 2234, 0x44e4 }, { 2235, 0x6aa7 }, { 2236, 0x844a }, { 2237, 0x23cb5 }, { 2238, 0x7958 }, { 2239, 0x84a8 }, { 2240, 0x26b96 },
    { 2241, 0x26e77 }, { 2242, 0x26e43 }, { 2243, 0x84de }, { 2244, 0x840f }, { 2245, 0x8391 }, { 2246, 0x44a0 }, { 2247, 0x8493 }, { 2248, 0x84e4 },
    { 2249, 0x25c91 }, { 2250, 0x4240 }, { 2251, 0x25cc0 }, { 2252, 0x4543 }, { 2253, 0x8534 }, { 2254, 0x5af2 }, { 2255, 0x26e99 }, { 2256, 0x4527 },
    { 2257, 0x8573 }, { 2258, 0x4516 }, { 2259, 0x67bf }, { 2260, 0x8616 }, { 2261, 0x28625 }, { 2262, 0x2863b }, { 2263, 0x85c1 }, { 2264, 0x27088 },
    { 2265, 0x8602 }, { 2266, 0x21582 }, { 2267, 0x270cd }, { 2268, 0x2f9b2 }, { 2269, 0x456a }, { 2270, 0x8628 }, { 2271, 0x3648 }, { 2272, 0x218a2 },
    { 2273, 0x53f7 }, { 2274, 0x2739a }, { 2275, 0x867e }, { 2276, 0x8771 }, { 2277, 0x2a0f8 }, { 2278, 0x87ee }, { 2279, 0x22c27 }, { 2280, 0x87b1 },
    { 2281, 0x87da }, { 2282, 0x880f }, { 2283, 0x5661 }, { 2284, 0x866c }, { 2285, 0x6856 }, { 2286, 0x460f }, { 2287, 0x8845 }, { 2288, 0x8846 },
    { 2289, 0x275e0 }, { 2290, 0x23db9 }, { 2291, 0x275e4 }, { 2292, 0x885e }, { 2293, 0x889c }, { 2294, 0x465b }, { 2295, 0x88b4 }, { 2296, 0x88b5 },
    { 2297, 0x63c1 }, { 2298, 0x88c5 }, { 2299, 0x7777 }, { 2300, 0x2770f }, { 2301, 0x8987 }, { 2302, 0x898a }, { 2303, 0x89a6 }, { 2304, 0x89a9 },
    { 2305, 0x89a7 }, { 2306, 0x89bc }, { 2307, 0x28a25 }, { 2308, 0x89e7 }, { 2309, 0x27924 }, { 2310, 0x27abd }, { 2311, 0x8a9c }, { 2312, 0x7793 },
    { 2313, 0x91fe }, { 2314, 0x8a90 }, { 2315, 0x27a59 }, { 2316, 0x7ae9 }, { 2317, 0x27b3a }, { 2318, 0x23f8f }, { 2319, 0x4713 }, { 2320, 0x27b38 },
    { 2321, 0x717c }, { 2322, 0x8b0c }, { 2323, 0x8b1f }, { 2324, 0x25430 }, { 2325, 0x25565 }, { 2326, 0x8b3f }, { 2327, 0x8b4c }, { 2328, 0x8b4d },
    { 2329, 0x8aa9 }, { 2330, 0x24a7a }, { 2331, 0x8b90 }, { 2332, 0x8b9b }, { 2333, 0x8aaf }, { 2334, 0x216df }, { 2335, 0x4615 }, { 2336, 0x884f },
    { 2337, 0x8c9b }, { 2338, 0x27d54 }, { 2339, 0x27d8f }, { 2340, 0x2f9d4 }, { 2341, 0x3725 }, { 2342, 0x27d53 }, { 2343, 0x8cd6 }, { 2344, 0x27d98 },
    { 2345, 0x27dbd }, { 2346, 0x8d12 }, { 2347, 0x8d03 }, { 2348, 0x21910 }, { 2349, 0x8cdb }, { 2350, 0x705c }, { 2351, 0x8d11 }, { 2352, 0x24cc9 },
    { 2353, 0x3ed0 }, { 2354, 0x8d77 }, { 2355, 0x8da9 }, { 2356, 0x28002 }, { 2357, 0x21014 }, { 2358, 0x2498a }, { 2359, 0x3b7c }, { 2360, 0x281bc },
    { 2361, 0x2710c }, { 2362, 0x7ae7 }, { 2363, 0x8ead }, { 2364, 0x8eb6 }, { 2365, 0x8ec3 }, { 2366, 0x92d4 }, { 2367, 0x8f19 }, { 2368, 0x8f2d },
    { 2369, 0x28365 }, { 2370, 0x28412 }, { 2371, 0x8fa5 }, { 2372, 0x9303 }, { 2373, 0x2a29f }, { 2374, 0x20a50 }, { 2375, 0x8fb3 }, { 2376, 0x492a },
    { 2377, 0x289de }, { 2378, 0x2853d }, { 2379, 0x23dbb }, { 2380, 0x5ef8 }, { 2381, 0x23262 }, { 2382, 0x8ff9 }, { 2383, 0x2a014 }, { 2384, 0x286bc },
    { 2385, 0x28501 }, { 2386, 0x22325 }, { 2387, 0x3980 }, { 2388, 0x26ed7 }, { 2389, 0x9037 }, { 2390, 0x2853c }, { 2391, 0x27abe }, { 2392, 0x9061 },
    { 2393, 0x2856c }, { 2394, 0x2860b }, { 2395, 0x90a8 }, { 2396, 0x28713 }, { 2397, 0x90c4 }, { 2398, 0x286e6 }, { 2399, 0x90ae }, { 2400, 0x90fd },
    { 2401, 0x9167 }, { 2402, 0x3af0 }, { 2403, 0x91a9 }, { 2404, 0x91c4 }, { 2405, 0x7cac }, { 2406, 0x28933 }, { 2407, 0x21e89 }, { 2408, 0x920e },
    { 2409, 0x6c9f }, { 2410, 0x9241 }, { 2411, 0x9262 }, { 2412, 0x255b9 }, { 2413, 0x92b9 }, { 2414, 0x28ac6 }, { 2415, 0x23c9b }, { 2416, 0x28b0c },
    { 2417, 0x255db }, { 2418, 0x20d31 }, { 2419, 0x932c }, { 2420, 0x936b }, { 2421, 0x28ae1 }, { 2422, 0x28beb }, { 2423, 0x708f }, { 2424, 0x5ac3 },
    { 2425, 0x28ae2 }, { 2426, 0x28ae5 }, { 2427, 0x4965 }, { 2428, 0x9244 }, { 2429, 0x28bec }, { 2430, 0x28c39 }, { 2431, 0x28bff }, { 2432, 0x9373 },
    { 2433, 0x945b }, { 2434, 0x8ebc }, { 2435, 0x9585 }, { 2436, 0x95a6 }, { 2437, 0x9426 }, { 2438, 0x95a0 }, { 2439, 0x6ff6 }, { 2440, 0x42b9 },
    { 2441, 0x2267a }, { 2442, 0x286d8 }, { 2443, 0x2127c }, { 2444, 0x23e2e }, { 2445, 0x49df }, { 2446, 0x6c1c }, { 2447, 0x967b }, { 2448, 0x9696 },
    { 2449, 0x416c }, { 2450, 0x96a3 }, { 2451, 0x26ed5 }, { 2452, 0x61da }, { 2453, 0x96b6 }, { 2454, 0x78f5 }, { 2455, 0x28ae0 }, { 2456, 0x96bd },
    { 2457, 0x53cc }, { 2458, 0x49a1 }, { 2459, 0x26cb8 }, { 2460, 0x20274 }, { 2461, 0x26410 }, { 2462, 0x290af }, { 2463, 0x290e5 }, { 2464, 0x24ad1 },
    { 2465, 0x21915 }, { 2466, 0x2330a }, { 2467, 0x9731 }, { 2468, 0x8642 }, { 2469, 0x9736 }, { 2470, 0x4a0f }, { 2471, 0x453d }, { 2472, 0x4585 },
    { 2473, 0x24ae9 }, { 2474, 0x7075 }, { 2475, 0x5b41 }, { 2476, 0x971b }, { 2477, 0x975c }, { 2478, 0x291d5 }, { 2479, 0x9757 }, { 2480, 0x5b4a },
    { 2481, 0x291eb }, { 2482, 0x975f }, { 2483, 0x9425 }, { 2484, 0x50d0 }, { 2485, 0x230b7 }, { 2486, 0x230bc }, { 2487, 0x9789 }, { 2488, 0x979f },
    { 2489, 0x97b1 }, { 2490, 0x97be }, { 2491, 0x97c0 }, { 2492, 0x97d2 }, { 2493, 0x97e0 }, { 2494, 0x2546c }, { 2495, 0x97ee }, { 2496, 0x741c },
    { 2497, 0x29433 }, { 2498, 0x97ff }, { 2499, 0x97f5 }, { 2500, 0x2941d }, { 2501, 0x2797a }, { 2502, 0x4ad1 }, { 2503, 0x9834 }, { 2504, 0x9833 },
    { 2505, 0x984b }, { 2506, 0x9866 }, { 2507, 0x3b0e }, { 2508, 0x27175 }, { 2509, 0x3d51 }, { 2510, 0x20630 }, { 2511, 0x2415c }, { 2512, 0x25706 },
    { 2513, 0x98ca }, { 2514, 0x98b7 }, { 2515, 0x98c8 }, { 2516, 0x98c7 }, { 2517, 0x4aff }, { 2518, 0x26d27 }, { 2519, 0x216d3 }, { 2520, 0x55b0 },
    { 2521, 0x98e1 }, { 2522, 0x98e6 }, { 2523, 0x98ec }, { 2524, 0x9378 }, { 2525, 0x9939 }, { 2526, 0x24a29 }, { 2527, 0x4b72 }, { 2528, 0x29857 },
    { 2529, 0x29905 }, { 2530, 0x99f5 }, { 2531, 0x9a0c }, { 2532, 0x9a3b }, { 2533, 0x9a10 }, { 2534, 0x9a58 }, { 2535, 0x25725 }, { 2536, 0x36c4 },
    { 2537, 0x290b1 }, { 2538, 0x29bd5 }, { 2539, 0x9ae0 }, { 2540, 0x9ae2 }, { 2541, 0x29b05 }, { 2542, 0x9af4 }, { 2543, 0x4c0e }, { 2544, 0x9b14 },
    { 2545, 0x9b2d }, { 2546, 0x28600 }, { 2547, 0x5034 }, { 2548, 0x9b34 }, { 2549, 0x269a8 }, { 2550, 0x38c3 }, { 2551, 0x2307d }, { 2552, 0x9b50 },
    { 2553, 0x9b40 }, { 2554, 0x29d3e }, { 2555, 0x5a45 }, { 2556, 0x21863 }, { 2557, 0x9b8e }, { 2558, 0x2424b }, { 2559, 0x9c02 }, { 2560, 0x9bff },
    { 2561, 0x9c0c }, { 2562, 0x29e68 }, { 2563, 0x9dd4 }, { 2564, 0x29fb7 }, { 2565, 0x2a192 }, { 2566, 0x2a1ab }, { 2567, 0x2a0e1 }, { 2568, 0x2a123 },
    { 2569, 0x2a1df }, { 2570, 0x9d7e }, { 2571, 0x9d83 }, { 2572, 0x2a134 }, { 2573, 0x9e0e }, { 2574, 0x6888 }, { 2575, 0x9dc4 }, { 2576, 0x2215b },
    { 2577, 0x2a193 }, { 2578, 0x2a220 }, { 2579, 0x2193b }, { 2580, 0x2a233 }, { 2581, 0x9d39 }, { 2582, 0x2a0b9 }, { 2583, 0x2a2b4 }, { 2584, 0x9e90 },
    { 2585, 0x9e95 }, { 2586, 0x9e9e }, { 2587, 0x9ea2 }, { 2588, 0x4d34 }, { 2589, 0x9eaa }, { 2590, 0x9eaf }, { 2591, 0x24364 }, { 2592, 0x9ec1 },
    { 2593, 0x3b60 }, { 2594, 0x39e5 }, { 2595, 0x3d1d }, { 2596, 0x4f32 }, { 2597, 0x37be }, { 2598, 0x28c2b }, { 2599, 0x9f02 }, { 2600, 0x9f08 },
    { 2601, 0x4b96 }, { 2602, 0x9424 }, { 2603, 0x26da2 }, { 2604, 0x9f17 }, { 2605, 0x9f16 }, { 2606, 0x9f39 }, { 2607, 0x569f }, { 2608, 0x568a },
    { 2609, 0x9f45 }, { 2610, 0x99b8 }, { 2611, 0x2908b }, { 2612, 0x97f2 }, { 2613, 0x847f }, { 2614, 0x9f62 }, { 2615, 0x9f69 }, { 2616, 0x7adc },
    { 2617, 0x9f8e }, { 2618, 0x7216 }, { 2619, 0x4bbe }, { 2620, 0x24975 }, { 2621, 0x249bb }, { 2622, 0x7177 }, { 2623, 0x249f8 }, { 2624, 0x24348 },
    { 2625, 0x24a51 }, { 2626, 0x739e }, { 2627, 0x28bda }, { 2628, 0x218fa }, { 2629, 0x799f }, { 2630, 0x2897e }, { 2631, 0x28e36 }, { 2632, 0x9369 },
    { 2633, 0x93f3 }, { 2634, 0x28a44 }, { 2635, 0x92ec }, { 2636, 0x9381 }, { 2637, 0x93cb }, { 2638, 0x2896c }, { 2639, 0x244b9 }, { 2640, 0x7217 },
    { 2641, 0x3eeb }, { 2642, 0x7772 }, { 2643, 0x7a43 }, { 2644, 0x70d0 }, { 2645, 0x24473 }, { 2646, 0x243f8 }, { 2647, 0x717e }, { 2648, 0x217ef },
    { 2649, 0x70a3 }, { 2650, 0x218be }, { 2651, 0x23599 }, { 2652, 0x3ec7 }, { 2653, 0x21885 }, { 2654, 0x2542f }, { 2655, 0x217f8 }, { 2656, 0x3722 },
    { 2657, 0x216fb }, { 2658, 0x21839 }, { 2659, 0x36e1 }, { 2660, 0x21774 }, { 2661, 0x218d1 }, { 2662, 0x25f4b }, { 2663, 0x3723 }, { 2664, 0x216c0 },
    { 2665, 0x575b }, { 2666, 0x24a25 }, { 2667, 0x213fe }, { 2668, 0x212a8 }, { 2669, 0x213c6 }, { 2670, 0x214b6 }, { 2671, 0x8503 }, { 2672, 0x236a6 },
    { 2673, 0x8503 }, { 2674, 0x8455 }, { 2675, 0x24994 }, { 2676, 0x27165 }, { 2677, 0x23e31 }, { 2678, 0x2555c }, { 2679, 0x23efb }, { 2680, 0x27052 },
    { 2681, 0x44f4 }, { 2682, 0x236ee }, { 2683, 0x2999d }, { 2684, 0x26f26 }, { 2685, 0x67f9 }, { 2686, 0x3733 }, { 2687, 0x3c15 }, { 2688, 0x3de7 },
    { 2689, 0x586c }, { 2690, 0x21922 }, { 2691, 0x6810 }, { 2692, 0x4057 }, { 2693, 0x2373f }, { 2694, 0x240e1 }, { 2695, 0x2408b }, { 2696, 0x2410f },
    { 2697, 0x26c21 }, { 2698, 0x54cb }, { 2699, 0x569e }, { 2700, 0x266b1 }, { 2701, 0x5692 }, { 2702, 0x20fdf }, { 2703, 0x20ba8 }, { 2704, 0x20e0d },
    { 2705, 0x93c6 }, { 2706, 0x28b13 }, { 2707, 0x939c }, { 2708, 0x4ef8 }, { 2709, 0x512b }, { 2710, 0x3819 }, { 2711, 0x24436 }, { 2712, 0x4ebc },
    { 2713, 0x20465 }, { 2714, 0x2037f }, { 2715, 0x4f4b }, { 2716, 0x4f8a }, { 2717, 0x25651 }, { 2718, 0x5a68 }, { 2719, 0x201ab }, { 2720, 0x203cb },
    { 2721, 0x3999 }, { 2722, 0x2030a }, { 2723, 0x20414 }, { 2724, 0x3435 }, { 2725, 0x4f29 }, { 2726, 0x202c0 }, { 2727, 0x28eb3 }, { 2728, 0x20275 },
    { 2729, 0x8ada }, { 2730, 0x2020c }, { 2731, 0x4e98 }, { 2732, 0x50cd }, { 2733, 0x510d }, { 2734, 0x4fa2 }, { 2735, 0x4f03 }, { 2736, 0x24a0e },
    { 2737, 0x23e8a }, { 2738, 0x4f42 }, { 2739, 0x502e }, { 2740, 0x506c }, { 2741, 0x5081 }, { 2742, 0x4fcc }, { 2743, 0x4fe5 }, { 2744, 0x5058 },
    { 2745, 0x50fc }, { 2746, 0x5159 }, { 2747, 0x515b }, { 2748, 0x515d }, { 2749, 0x515e }, { 2750, 0x6e76 }, { 2751, 0x23595 }, { 2752, 0x23e39 },
    { 2753, 0x23ebf }, { 2754, 0x6d72 }, { 2755, 0x21884 }, { 2756, 0x23e89 }, { 2757, 0x51a8 }, { 2758, 0x51c3 }, { 2759, 0x205e0 }, { 2760, 0x44dd },
    { 2761, 0x204a3 }, { 2762, 0x20492 }, { 2763, 0x20491 }, { 2764, 0x8d7a }, { 2765, 0x28a9c }, { 2766, 0x2070e }, { 2767, 0x5259 }, { 2768, 0x52a4 },
    { 2769, 0x20873 }, { 2770, 0x52e1 }, { 2771, 0x936e }, { 2772, 0x467a }, { 2773, 0x718c }, { 2774, 0x2438c }, { 2775, 0x20c20 }, { 2776, 0x249ac },
    { 2777, 0x210e4 }, { 2778, 0x69d1 }, { 2779, 0x20e1d }, { 2780, 0x7479 }, { 2781, 0x3ede }, { 2782, 0x7499 }, { 2783, 0x7414 }, { 2784, 0x7456 },
    { 2785, 0x7398 }, { 2786, 0x4b8e }, { 2787, 0x24abc }, { 2788, 0x2408d }, { 2789, 0x53d0 }, { 2790, 0x3584 }, { 2791, 0x720f }, { 2792, 0x240c9 },
    { 2793, 0x55b4 }, { 2794, 0x20345 }, { 2795, 0x54cd }, { 2796, 0x20bc6 }, { 2797, 0x571d }, { 2798, 0x925d }, { 2799, 0x96f4 }, { 2800, 0x9366 },
    { 2801, 0x57dd }, { 2802, 0x578d }, { 2803, 0x577f }, { 2804, 0x363e }, { 2805, 0x58cb }, { 2806, 0x5a99 }, { 2807, 0x28a46 }, { 2808, 0x216fa },
    { 2809, 0x2176f }, { 2810, 0x21710 }, { 2811, 0x5a2c }, { 2812, 0x59b8 }, { 2813, 0x928f }, { 2814, 0x5a7e }, { 2815, 0x5acf }, { 2816, 0x5a12 },
    { 2817, 0x25946 }, { 2818, 0x219f3 }, { 2819, 0x21861 }, { 2820, 0x24295 }, { 2821, 0x36f5 }, { 2822, 0x6d05 }, { 2823, 0x7443 }, { 2824, 0x5a21 },
    { 2825, 0x25e83 }, { 2826, 0x5a81 }, { 2827, 0x28bd7 }, { 2828, 0x20413 }, { 2829, 0x93e0 }, { 2830, 0x748c }, { 2831, 0x21303 }, { 2832, 0x7105 },
    { 2833, 0x4972 }, { 2834, 0x9408 }, { 2835, 0x289fb }, { 2836, 0x93bd }, { 2837, 0x37a0 }, { 2838, 0x5c1e }, { 2839, 0x5c9e }, { 2840, 0x5e5e },
    { 2841, 0x5e48 }, { 2842, 0x21996 }, { 2843, 0x2197c }, { 2844, 0x23aee }, { 2845, 0x5ecd }, { 2846, 0x5b4f }, { 2847, 0x21903 }, { 2848, 0x21904 },
    { 2849, 0x3701 }, { 2850, 0x218a0 }, { 2851, 0x36dd }, { 2852, 0x216fe }, { 2853, 0x36d3 }, { 2854, 0x812a }, { 2855, 0x28a47 }, { 2856, 0x21dba },
    { 2857, 0x23472 }, { 2858, 0x289a8 }, { 2859, 0x5f0c }, { 2860, 0x5f0e }, { 2861, 0x21927 }, { 2862, 0x217ab }, { 2863, 0x5a6b }, { 2864, 0x2173b },
    { 2865, 0x5b44 }, { 2866, 0x8614 }, { 2867, 0x275fd }, { 2868, 0x8860 }, { 2869, 0x607e }, { 2870, 0x22860 }, { 2871, 0x2262b }, { 2872, 0x5fdb },
    { 2873, 0x3eb8 }, { 2874, 0x225af }, { 2875, 0x225be }, { 2876, 0x29088 }, { 2877, 0x26f73 }, { 2878, 0x61c0 }, { 2879, 0x2003e }, { 2880, 0x20046 },
    { 2881, 0x2261b }, { 2882, 0x6199 }, { 2883, 0x6198 }, { 2884, 0x6075 }, { 2885, 0x22c9b }, { 2886, 0x22d07 }, { 2887, 0x246d4 }, { 2888, 0x2914d },
    { 2889, 0x6471 }, { 2890, 0x24665 }, { 2891, 0x22b6a }, { 2892, 0x3a29 }, { 2893, 0x22b22 }, { 2894, 0x23450 }, { 2895, 0x298ea }, { 2896, 0x22e78 },
    { 2897, 0x6337 }, { 2898, 0x2a45b }, { 2899, 0x64b6 }, { 2900, 0x6331 }, { 2901, 0x63d1 }, { 2902, 0x249e3 }, { 2903, 0x22d67 }, { 2904, 0x62a4 },
    { 2905, 0x22ca1 }, { 2906, 0x643b }, { 2907, 0x656b }, { 2908, 0x6972 }, { 2909, 0x3bf4 }, { 2910, 0x2308e }, { 2911, 0x232ad }, { 2912, 0x24989 },
    { 2913, 0x232ab }, { 2914, 0x550d }, { 2915, 0x232e0 }, { 2916, 0x218d9 }, { 2917, 0x2943f }, { 2918, 0x66ce }, { 2919, 0x23289 }, { 2920, 0x231b3 },
    { 2921, 0x3ae0 }, { 2922, 0x4190 }, { 2923, 0x25584 }, { 2924, 0x28b22 }, { 2925, 0x2558f }, { 2926, 0x216fc }, { 2927, 0x2555b }, { 2928, 0x25425 },
    { 2929, 0x78ee }, { 2930, 0x23103 }, { 2931, 0x2182a }, { 2932, 0x23234 }, { 2933, 0x3464 }, { 2934, 0x2320f }, { 2935, 0x23182 }, { 2936, 0x242c9 },
    { 2937, 0x668e }, { 2938, 0x26d24 }, { 2939, 0x666b }, { 2940, 0x4b93 }, { 2941, 0x6630 }, { 2942, 0x27870 }, { 2943, 0x21deb }, { 2944, 0x6663 },
    { 2945, 0x232d2 }, { 2946, 0x232e1 }, { 2947, 0x661e }, { 2948, 0x25872 }, { 2949, 0x38d1 }, { 2950, 0x2383a }, { 2951, 0x237bc }, { 2952, 0x3b99 },
    { 2953, 0x237a2 }, { 2954, 0x233fe }, { 2955, 0x74d0 }, { 2956, 0x3b96 }, { 2957, 0x678f }, { 2958, 0x2462a }, { 2959, 0x68b6 }, { 2960, 0x681e },
    { 2961, 0x3bc4 }, { 2962, 0x6abe }, { 2963, 0x3863 }, { 2964, 0x237d5 }, { 2965, 0x24487 }, { 2966, 0x6a33 }, { 2967, 0x6a52 }, { 2968, 0x6ac9 },
    { 2969, 0x6b05 }, { 2970, 0x21912 }, { 2971, 0x6511 }, { 2972, 0x6898 }, { 2973, 0x6a4c }, { 2974, 0x3bd7 }, { 2975, 0x6a7a }, { 2976, 0x6b57 },
    { 2977, 0x23fc0 }, { 2978, 0x23c9a }, { 2979, 0x93a0 }, { 2980, 0x92f2 }, { 2981, 0x28bea }, { 2982, 0x28acb }, { 2983, 0x9289 }, { 2984, 0x2801e },
    { 2985, 0x289dc }, { 2986, 0x9467 }, { 2987, 0x6da5 }, { 2988, 0x6f0b }, { 2989, 0x249ec }, { 2990, 0x6d67 }, { 2991, 0x23f7f }, { 2992, 0x3d8f },
    { 2993, 0x6e04 }, { 2994, 0x2403c }, { 2995, 0x5a3d }, { 2996, 0x6e0a }, { 2997, 0x5847 }, { 2998, 0x6d24 }, { 2999, 0x7842 }, { 3000, 0x713b },
    { 3001, 0x2431a }, { 3002, 0x24276 }, { 3003, 0x70f1 }, { 3004, 0x7250 }, { 3005, 0x7287 }, { 3006, 0x7294 }, { 3007, 0x2478f }, { 3008, 0x24725 },
    { 3009, 0x5179 }, { 3010, 0x24aa4 }, { 3011, 0x205eb }, { 3012, 0x747a }, { 3013, 0x23ef8 }, { 3014, 0x2365f }, { 3015, 0x24a4a }, { 3016, 0x24917 },
    { 3017, 0x25fe1 }, { 3018, 0x3f06 }, { 3019, 0x3eb1 }, { 3020, 0x24adf }, { 3021, 0x28c23 }, { 3022, 0x23f35 }, { 3023, 0x60a7 }, { 3024, 0x3ef3 },
    { 3025, 0x74cc }, { 3026, 0x743c }, { 3027, 0x9387 }, { 3028, 0x7437 }, { 3029, 0x449f }, { 3030, 0x26dea }, { 3031, 0x4551 }, { 3032, 0x7583 },
    { 3033, 0x3f63 }, { 3034, 0x24cd9 }, { 3035, 0x24d06 }, { 3036, 0x3f58 }, { 3037, 0x7555 }, { 3038, 0x7673 }, { 3039, 0x2a5c6 }, { 3040, 0x3b19 },
    { 3041, 0x7468 }, { 3042, 0x28acc }, { 3043, 0x249ab }, { 3044, 0x2498e }, { 3045, 0x3afb }, { 3046, 0x3dcd }, { 3047, 0x24a4e }, { 3048, 0x3eff },
    { 3049, 0x249c5 }, { 3050, 0x248f3 }, { 3051, 0x91fa }, { 3052, 0x5732 }, { 3053, 0x9342 }, { 3054, 0x28ae3 }, { 3055, 0x21864 }, { 3056, 0x50df },
    { 3057, 0x25221 }, { 3058, 0x251e7 }, { 3059, 0x7778 }, { 3060, 0x23232 }, { 3061, 0x770e }, { 3062, 0x770f }, { 3063, 0x777b }, { 3064, 0x24697 },
    { 3065, 0x23781 }, { 3066, 0x3a5e }, { 3067, 0x248f0 }, { 3068, 0x7438 }, { 3069, 0x749b }, { 3070, 0x3ebf }, { 3071, 0x24aba }, { 3072, 0x24ac7 },
    { 3073, 0x40c8 }, { 3074, 0x24a96 }, { 3075, 0x261ae }, { 3076, 0x9307 }, { 3077, 0x25581 }, { 3078, 0x781e }, { 3079, 0x788d }, { 3080, 0x7888 },
    { 3081, 0x78d2 }, { 3082, 0x73d0 }, { 3083, 0x7959 }, { 3084, 0x27741 }, { 3085, 0x256e3 }, { 3086, 0x410e }, { 3087, 0x799b }, { 3088, 0x8496 },
    { 3089, 0x79a5 }, { 3090, 0x6a2d }, { 3091, 0x23efa }, { 3092, 0x7a3a }, { 3093, 0x79f4 }, { 3094, 0x416e }, { 3095, 0x216e6 }, { 3096, 0x4132 },
    { 3097, 0x9235 }, { 3098, 0x79f1 }, { 3099, 0x20d4c }, { 3100, 0x2498c }, { 3101, 0x20299 }, { 3102, 0x23dba }, { 3103, 0x2176e }, { 3104, 0x3597 },
    { 3105, 0x556b }, { 3106, 0x3570 }, { 3107, 0x36aa }, { 3108, 0x201d4 }, { 3109, 0x20c0d }, { 3110, 0x7ae2 }, { 3111, 0x5a59 }, { 3112, 0x226f5 },
    { 3113, 0x25aaf }, { 3114, 0x25a9c }, { 3115, 0x5a0d }, { 3116, 0x2025b }, { 3117, 0x78f0 }, { 3118, 0x5a2a }, { 3119, 0x25bc6 }, { 3120, 0x7afe },
    { 3121, 0x41f9 }, { 3122, 0x7c5d }, { 3123, 0x7c6d }, { 3124, 0x4211 }, { 3125, 0x25bb3 }, { 3126, 0x25ebc }, { 3127, 0x25ea6 }, { 3128, 0x7ccd },
    { 3129, 0x249f9 }, { 3130, 0x217b0 }, { 3131, 0x7c8e }, { 3132, 0x7c7c }, { 3133, 0x7cae }, { 3134, 0x6ab2 }, { 3135, 0x7ddc }, { 3136, 0x7e07 },
    { 3137, 0x7dd3 }, { 3138, 0x7f4e }, { 3139, 0x26261 }, { 3140, 0x2615c }, { 3141, 0x27b48 }, { 3142, 0x7d97 }, { 3143, 0x25e82 }, { 3144, 0x426a },
    { 3145, 0x26b75 }, { 3146, 0x20916 }, { 3147, 0x67d6 }, { 3148, 0x2004e }, { 3149, 0x235cf }, { 3150, 0x57c4 }, { 3151, 0x26412 }, { 3152, 0x263f8 },
    { 3153, 0x24962 }, { 3154, 0x7fdd }, { 3155, 0x7b27 }, { 3156, 0x2082c }, { 3157, 0x25ae9 }, { 3158, 0x25d43 }, { 3159, 0x7b0c }, { 3160, 0x25e0e },
    { 3161, 0x99e6 }, { 3162, 0x8645 }, { 3163, 0x9a63 }, { 3164, 0x6a1c }, { 3165, 0x2343f }, { 3166, 0x39e2 }, { 3167, 0x249f7 }, { 3168, 0x265ad },
    { 3169, 0x9a1f }, { 3170, 0x265a0 }, { 3171, 0x8480 }, { 3172, 0x27127 }, { 3173, 0x26cd1 }, { 3174, 0x44ea }, { 3175, 0x8137 }, { 3176, 0x4402 },
    { 3177, 0x80c6 }, { 3178, 0x8109 }, { 3179, 0x8142 }, { 3180, 0x267b4 }, { 3181, 0x98c3 }, { 3182, 0x26a42 }, { 3183, 0x8262 }, { 3184, 0x8265 },
    { 3185, 0x26a51 }, { 3186, 0x8453 }, { 3187, 0x26da7 }, { 3188, 0x8610 }, { 3189, 0x2721b }, { 3190, 0x5a86 }, { 3191, 0x417f }, { 3192, 0x21840 },
    { 3193, 0x5b2b }, { 3194, 0x218a1 }, { 3195, 0x5ae4 }, { 3196, 0x218d8 }, { 3197, 0x86a0 }, { 3198, 0x2f9bc }, { 3199, 0x23d8f }, { 3200, 0x882d },
    { 3201, 0x27422 }, { 3202, 0x5a02 }, { 3203, 0x886e }, { 3204, 0x4f45 }, { 3205, 0x8887 }, { 3206, 0x88bf }, { 3207, 0x88e6 }, { 3208, 0x8965 },
    { 3209, 0x894d }, { 3210, 0x25683 }, { 3211, 0x8954 }, { 3212, 0x27785 }, { 3213, 0x27784 }, { 3214, 0x28bf5 }, { 3215, 0x28bd9 }, { 3216, 0x28b9c },
    { 3217, 0x289f9 }, { 3218, 0x3ead }, { 3219, 0x84a3 }, { 3220, 0x46f5 }, { 3221, 0x46cf }, { 3222, 0x37f2 }, { 3223, 0x8a3d }, { 3224, 0x8a1c },
    { 3225, 0x29448 }, { 3226, 0x5f4d }, { 3227, 0x922b }, { 3228, 0x24284 }, { 3229, 0x65d4 }, { 3230, 0x7129 }, { 3231, 0x70c4 }, { 3232, 0x21845 },
    { 3233, 0x9d6d }, { 3234, 0x8c9f }, { 3235, 0x8ce9 }, { 3236, 0x27ddc }, { 3237, 0x599a }, { 3238, 0x77c3 }, { 3239, 0x59f0 }, { 3240, 0x436e },
    { 3241, 0x36d4 }, { 3242, 0x8e2a }, { 3243, 0x8ea7 }, { 3244, 0x24c09 }, { 3245, 0x8f30 }, { 3246, 0x8f4a }, { 3247, 0x42f4 }, { 3248, 0x6c58 },
    { 3249, 0x6fbb }, { 3250, 0x22321 }, { 3251, 0x489b }, { 3252, 0x6f79 }, { 3253, 0x6e8b }, { 3254, 0x217da }, { 3255, 0x9be9 }, { 3256, 0x36b5 },
    { 3257, 0x2492f }, { 3258, 0x90bb }, { 3259, 0x9097 }, { 3260, 0x5571 }, { 3261, 0x4906 }, { 3262, 0x91bb }, { 3263, 0x9404 }, { 3264, 0x28a4b },
    { 3265, 0x4062 }, { 3266, 0x28afc }, { 3267, 0x9427 }, { 3268, 0x28c1d }, { 3269, 0x28c3b }, { 3270, 0x84e5 }, { 3271, 0x8a2b }, { 3272, 0x9599 },
    { 3273, 0x95a7 }, { 3274, 0x9597 }, { 3275, 0x9596 }, { 3276, 0x28d34 }, { 3277, 0x7445 }, { 3278, 0x3ec2 }, { 3279, 0x248ff }, { 3280, 0x24a42 },
    { 3281, 0x243ea }, { 3282, 0x3ee7 }, { 3283, 0x23225 }, { 3284, 0x968f }, { 3285, 0x28ee7 }, { 3286, 0x28e66 }, { 3287, 0x28e65 }, { 3288, 0x3ecc },
    { 3289, 0x249ed }, { 3290, 0x24a78 }, { 3291, 0x23fee }, { 3292, 0x7412 }, { 3293, 0x746b }, { 3294, 0x3efc }, { 3295, 0x9741 }, { 3296, 0x290b0 },
    { 3297, 0x6847 }, { 3298, 0x4a1d }, { 3299, 0x29093 }, { 3300, 0x257df }, { 3301, 0x975d }, { 3302, 0x9368 }, { 3303, 0x28989 }, { 3304, 0x28c26 },
    { 3305, 0x28b2f }, { 3306, 0x263be }, { 3307, 0x92ba }, { 3308, 0x5b11 }, { 3309, 0x8b69 }, { 3310, 0x493c }, { 3311, 0x73f9 }, { 3312, 0x2421b },
    { 3313, 0x979b }, { 3314, 0x9771 }, { 3315, 0x9938 }, { 3316, 0x20f26 }, { 3317, 0x5dc1 }, { 3318, 0x28bc5 }, { 3319, 0x24ab2 }, { 3320, 0x981f },
    { 3321, 0x294da }, { 3322, 0x92f6 }, { 3323, 0x295d7 }, { 3324, 0x91e5 }, { 3325, 0x44c0 }, { 3326, 0x28b50 }, { 3327, 0x24a67 }, { 3328, 0x28b64 },
    { 3329, 0x98dc }, { 3330, 0x28a45 }, { 3331, 0x3f00 }, { 3332, 0x922a }, { 3333, 0x4925 }, { 3334, 0x8414 }, { 3335, 0x993b }, { 3336, 0x994d },
    { 3337, 0x27b06 }, { 3338, 0x3dfd }, { 3339, 0x999b }, { 3340, 0x4b6f }, { 3341, 0x99aa }, { 3342, 0x9a5c }, { 3343, 0x28b65 }, { 3344, 0x258c8 },
    { 3345, 0x6a8f }, { 3346, 0x9a21 }, { 3347, 0x5afe }, { 3348, 0x9a2f }, { 3349, 0x298f1 }, { 3350, 0x4b90 }, { 3351, 0x29948 }, { 3352, 0x99bc },
    { 3353, 0x4bbd }, { 3354, 0x4b97 }, { 3355, 0x937d }, { 3356, 0x5872 }, { 3357, 0x21302 }, { 3358, 0x5822 }, { 3359, 0x249b8 }, { 3360, 0x214e8 },
    { 3361, 0x7844 }, { 3362, 0x2271f }, { 3363, 0x23db8 }, { 3364, 0x68c5 }, { 3365, 0x3d7d }, { 3366, 0x9458 }, { 3367, 0x3927 }, { 3368, 0x6150 },
    { 3369, 0x22781 }, { 3370, 0x2296b }, { 3371, 0x6107 }, { 3372, 0x9c4f }, { 3373, 0x9c53 }, { 3374, 0x9c7b }, { 3375, 0x9c35 }, { 3376, 0x9c10 },
    { 3377, 0x9b7f }, { 3378, 0x9bcf }, { 3379, 0x29e2d }, { 3380, 0x9b9f }, { 3381, 0x2a1f5 }, { 3382, 0x2a0fe }, { 3383, 0x9d21 }, { 3384, 0x4cae },
    { 3385, 0x24104 }, { 3386, 0x9e18 }, { 3387, 0x4cb0 }, { 3388, 0x9d0c }, { 3389, 0x2a1b4 }, { 3390, 0x2a0ed }, { 3391, 0x2a0f3 }, { 3392, 0x2992f },
    { 3393, 0x9da5 }, { 3394, 0x84bd }, { 3395, 0x26e12 }, { 3396, 0x26fdf }, { 3397, 0x26b82 }, { 3398, 0x85fc }, { 3399, 0x4533 }, { 3400, 0x26da4 },
    { 3401, 0x26e84 }, { 3402, 0x26df0 }, { 3403, 0x8420 }, { 3404, 0x85ee }, { 3405, 0x26e00 }, { 3406, 0x237d7 }, { 3407, 0x26064 }, { 3408, 0x79e2 },
    { 3409, 0x2359c }, { 3410, 0x23640 }, { 3411, 0x492d }, { 3412, 0x249de }, { 3413, 0x3d62 }, { 3414, 0x93db }, { 3415, 0x92be }, { 3416, 0x9348 },
    { 3417, 0x202bf }, { 3418, 0x78b9 }, { 3419, 0x9277 }, { 3420, 0x944d }, { 3421, 0x4fe4 }, { 3422, 0x3440 }, { 3423, 0x9064 }, { 3424, 0x2555d },
    { 3425, 0x783d }, { 3426, 0x7854 }, { 3427, 0x78b6 }, { 3428, 0x784b }, { 3429, 0x21757 }, { 3430, 0x231c9 }, { 3431, 0x24941 }, { 3432, 0x369a },
    { 3433, 0x4f72 }, { 3434, 0x6fda }, { 3435, 0x6fd9 }, { 3436, 0x701e }, { 3437, 0x701e }, { 3438, 0x5414 }, { 3439, 0x241b5 }, { 3440, 0x57bb },
    { 3441, 0x58f3 }, { 3442, 0x578a }, { 3443, 0x9d16 }, { 3444, 0x57d7 }, { 3445, 0x7134 }, { 3446, 0x34af }, { 3447, 0x241ac }, { 3448, 0x71eb },
    { 3449, 0x26c40 }, { 3450, 0x24f97 }, { 3451, 0x5b28 }, { 3452, 0x217b5 }, { 3453, 0x28a49 }, { 3454, 0x610c }, { 3455, 0x5ace }, { 3456, 0x5a0b },
    { 3457, 0x42bc }, { 3458, 0x24488 }, { 3459, 0x372c }, { 3460, 0x4b7b }, { 3461, 0x289fc }, { 3462, 0x93bb }, { 3463, 0x93b8 }, { 3464, 0x218d6 },
    { 3465, 0x20f1d }, { 3466, 0x8472 }, { 3467, 0x26cc0 }, { 3468, 0x21413 }, { 3469, 0x242fa }, { 3470, 0x22c26 }, { 3471, 0x243c1 }, { 3472, 0x5994 },
    { 3473, 0x23db7 }, { 3474, 0x26741 }, { 3475, 0x7da8 }, { 3476, 0x2615b }, { 3477, 0x260a4 }, { 3478, 0x249b9 }, { 3479, 0x2498b }, { 3480, 0x289fa },
    { 3481, 0x92e5 }, { 3482, 0x73e2 }, { 3483, 0x3ee9 }, { 3484, 0x74b4 }, { 3485, 0x28b63 }, { 3486, 0x2189f }, { 3487, 0x3ee1 }, { 3488, 0x24ab3 },
    { 3489, 0x6ad8 }, { 3490, 0x73f3 }, { 3491, 0x73fb }, { 3492, 0x3ed6 }, { 3493, 0x24a3e }, { 3494, 0x24a94 }, { 3495, 0x217d9 }, { 3496, 0x24a66 },
    { 3497, 0x203a7 }, { 3498, 0x21424 }, { 3499, 0x249e5 }, { 3500, 0x7448 }, { 3501, 0x24916 }, { 3502, 0x70a5 }, { 3503, 0x24976 }, { 3504, 0x9284 },
    { 3505, 0x73e6 }, { 3506, 0x935f }, { 3507, 0x204fe }, { 3508, 0x9331 }, { 3509, 0x28ace }, { 3510, 0x28a16 }, { 3511, 0x9386 }, { 3512, 0x28be7 },
    { 3513, 0x255d5 }, { 3514, 0x4935 }, { 3515, 0x28a82 }, { 3516, 0x716b }, { 3517, 0x24943 }, { 3518, 0x20cff }, { 3519, 0x56a4 }, { 3520, 0x2061a },
    { 3521, 0x20beb }, { 3522, 0x20cb8 }, { 3523, 0x5502 }, { 3524, 0x79c4 }, { 3525, 0x217fa }, { 3526, 0x7dfe }, { 3527, 0x216c2 }, { 3528, 0x24a50 },
    { 3529, 0x21852 }, { 3530, 0x452e }, { 3531, 0x9401 }, { 3532, 0x370a }, { 3533, 0x28ac0 }, { 3534, 0x249ad }, { 3535, 0x59b0 }, { 3536, 0x218bf },
    { 3537, 0x21883 }, { 3538, 0x27484 }, { 3539, 0x5aa1 }, { 3540, 0x36e2 }, { 3541, 0x23d5b }, { 3542, 0x36b0 }, { 3543, 0x925f }, { 3544, 0x5a79 },
    { 3545, 0x28a81 }, { 3546, 0x21862 }, { 3547, 0x9374 }, { 3548, 0x3ccd }, { 3549, 0x20ab4 }, { 3550, 0x4a96 }, { 3551, 0x398a }, { 3552, 0x50f4 },
    { 3553, 0x3d69 }, { 3554, 0x3d4c }, { 3555, 0x2139c }, { 3556, 0x7175 }, { 3557, 0x42fb }, { 3558, 0x28218 }, { 3559, 0x6e0f }, { 3560, 0x290e4 },
    { 3561, 0x44eb }, { 3562, 0x6d57 }, { 3563, 0x27e4f }, { 3564, 0x7067 }, { 3565, 0x6caf }, { 3566, 0x3cd6 }, { 3567, 0x23fed }, { 3568, 0x23e2d },
    { 3569, 0x6e02 }, { 3570, 0x6f0c }, { 3571, 0x3d6f }, { 3572, 0x203f5 }, { 3573, 0x7551 }, { 3574, 0x36bc }, { 3575, 0x34c8 }, { 3576, 0x4680 },
    { 3577, 0x3eda }, { 3578, 0x4871 }, { 3579, 0x59c4 }, { 3580, 0x926e }, { 3581, 0x493e }, { 3582, 0x8f41 }, { 3583, 0x28c1c }, { 3584, 0x26bc0 },
    { 3585, 0x5812 }, { 3586, 0x57c8 }, { 3587, 0x36d6 }, { 3588, 0x21452 }, { 3589, 0x70fe }, { 3590, 0x24362 }, { 3591, 0x24a71 }, { 3592, 0x22fe3 },
    { 3593, 0x212b0 }, { 3594, 0x223bd }, { 3595, 0x68b9 }, { 3596, 0x6967 }, { 3597, 0x21398 }, { 3598, 0x234e5 }, { 3599, 0x27bf4 }, { 3600, 0x236df },
    { 3601, 0x28a83 }, { 3602, 0x237d6 }, { 3603, 0x233fa }, { 3604, 0x24c9f }, { 3605, 0x6a1a }, { 3606, 0x236ad }, { 3607, 0x26cb7 }, { 3608, 0x843e },
    { 3609, 0x44df }, { 3610, 0x44ce }, { 3611, 0x26d26 }, { 3612, 0x26d51 }, { 3613, 0x26c82 }, { 3614, 0x26fde }, { 3615, 0x6f17 }, { 3616, 0x27109 },
    { 3617, 0x833d }, { 3618, 0x2173a }, { 3619, 0x83ed }, { 3620, 0x26c80 }, { 3621, 0x27053 }, { 3622, 0x217db }, { 3623, 0x5989 }, { 3624, 0x5a82 },
    { 3625, 0x217b3 }, { 3626, 0x5a61 }, { 3627, 0x5a71 }, { 3628, 0x21905 }, { 3629, 0x241fc }, { 3630, 0x372d }, { 3631, 0x59ef }, { 3632, 0x2173c },
    { 3633, 0x36c7 }, { 3634, 0x718e }, { 3635, 0x9390 }, { 3636, 0x669a }, { 3637, 0x242a5 }, { 3638, 0x5a6e }, { 3639, 0x5a2b }, { 3640, 0x24293 },
    { 3641, 0x6a2b }, { 3642, 0x23ef9 }, { 3643, 0x27736 }, { 3644, 0x2445b }, { 3645, 0x242ca }, { 3646, 0x711d }, { 3647, 0x24259 }, { 3648, 0x289e1 },
    { 3649, 0x4fb0 }, { 3650, 0x26d28 }, { 3651, 0x5cc2 }, { 3652, 0x244ce }, { 3653, 0x27e4d }, { 3654, 0x243bd }, { 3655, 0x6a0c }, { 3656, 0x24256 },
    { 3657, 0x21304 }, { 3658, 0x70a6 }, { 3659, 0x7133 }, { 3660, 0x243e9 }, { 3661, 0x3da5 }, { 3662, 0x6cdf }, { 3663, 0x2f825 }, { 3664, 0x24a4f },
    { 3665, 0x7e65 }, { 3666, 0x59eb }, { 3667, 0x5d2f }, { 3668, 0x3df3 }, { 3669, 0x5f5c }, { 3670, 0x24a5d }, { 3671, 0x217df }, { 3672, 0x7da4 },
    { 3673, 0x8426 }, { 3674, 0x5485 }, { 3675, 0x23afa }, { 3676, 0x23300 }, { 3677, 0x20214 }, { 3678, 0x577e }, { 3679, 0x208d5 }, { 3680, 0x20619 },
    { 3681, 0x3fe5 }, { 3682, 0x21f9e }, { 3683, 0x2a2b6 }, { 3684, 0x7003 }, { 3685, 0x2915b }, { 3686, 0x5d70 }, { 3687, 0x738f }, { 3688, 0x7cd3 },
    { 3689, 0x28a59 }, { 3690, 0x29420 }, { 3691, 0x4fc8 }, { 3692, 0x7fe7 }, { 3693, 0x72cd }, { 3694, 0x7310 }, { 3695, 0x27af4 }, { 3696, 0x7338 },
    { 3697, 0x7339 }, { 3698, 0x256f6 }, { 3699, 0x7341 }, { 3700, 0x7348 }, { 3701, 0x3ea9 }, { 3702, 0x27b18 }, { 3703, 0x906c }, { 3704, 0x71f5 },
    { 3705, 0x248f2 }, { 3706, 0x73e1 }, { 3707, 0x81f6 }, { 3708, 0x3eca }, { 3709, 0x770c }, { 3710, 0x3ed1 }, { 3711, 0x6ca2 }, { 3712, 0x56fd },
    { 3713, 0x7419 }, { 3714, 0x741e }, { 3715, 0x741f }, { 3716, 0x3ee2 }, { 3717, 0x3ef0 }, { 3718, 0x3ef4 }, { 3719, 0x3efa }, { 3720, 0x74d3 },
    { 3721, 0x3f0e }, { 3722, 0x3f53 }, { 3723, 0x7542 }, { 3724, 0x756d }, { 3725, 0x7572 }, { 3726, 0x758d }, { 3727, 0x3f7c }, { 3728, 0x75c8 },
    { 3729, 0x75dc }, { 3730, 0x3fc0 }, { 3731, 0x764d }, { 3732, 0x3fd7 }, { 3733, 0x7674 }, { 3734, 0x3fdc }, { 3735, 0x767a }, { 3736, 0x24f5c },
    { 3737, 0x7188 }, { 3738, 0x5623 }, { 3739, 0x8980 }, { 3740, 0x5869 }, { 3741, 0x401d }, { 3742, 0x7743 }, { 3743, 0x4039 }, { 3744, 0x6761 },
    { 3745, 0x4045 }, { 3746, 0x35db }, { 3747, 0x7798 }, { 3748, 0x406a }, { 3749, 0x406f }, { 3750, 0x5c5e }, { 3751, 0x77be }, { 3752, 0x77cb },
    { 3753, 0x58f2 }, { 3754, 0x7818 }, { 3755, 0x70b9 }, { 3756, 0x781c }, { 3757, 0x40a8 }, { 3758, 0x7839 }, { 3759, 0x7847 }, { 3760, 0x7851 },
    { 3761, 0x7866 }, { 3762, 0x8448 }, { 3763, 0x25535 }, { 3764, 0x7933 }, { 3765, 0x6803 }, { 3766, 0x7932 }, { 3767, 0x4103 }, { 3768, 0x4109 },
    { 3769, 0x7991 }, { 3770, 0x7999 }, { 3771, 0x8fbb }, { 3772, 0x7a06 }, { 3773, 0x8fbc }, { 3774, 0x4167 }, { 3775, 0x7a91 }, { 3776, 0x41b2 },
    { 3777, 0x7abc }, { 3778, 0x8279 }, { 3779, 0x41c4 }, { 3780, 0x7acf }, { 3781, 0x7adb }, { 3782, 0x41cf }, { 3783, 0x4e21 }, { 3784, 0x7b62 },
    { 3785, 0x7b6c }, { 3786, 0x7b7b }, { 3787, 0x7c12 }, { 3788, 0x7c1b }, { 3789, 0x4260 }, { 3790, 0x427a }, { 3791, 0x7c7b }, { 3792, 0x7c9c },
    { 3793, 0x428c }, { 3794, 0x7cb8 }, { 3795, 0x4294 }, { 3796, 0x7ced }, { 3797, 0x8f93 }, { 3798, 0x70c0 }, { 3799, 0x20ccf }, { 3800, 0x7dcf },
    { 3801, 0x7dd4 }, { 3802, 0x7dd0 }, { 3803, 0x7dfd }, { 3804, 0x7fae }, { 3805, 0x7fb4 }, { 3806, 0x729f }, { 3807, 0x4397 }, { 3808, 0x8020 },
    { 3809, 0x8025 }, { 3810, 0x7b39 }, { 3811, 0x802e }, { 3812, 0x8031 }, { 3813, 0x8054 }, { 3814, 0x3dcc }, { 3815, 0x57b4 }, { 3816, 0x70a0 },
    { 3817, 0x80b7 }, { 3818, 0x80e9 }, { 3819, 0x43ed }, { 3820, 0x810c }, { 3821, 0x732a }, { 3822, 0x810e }, { 3823, 0x8112 }, { 3824, 0x7560 },
    { 3825, 0x8114 }, { 3826, 0x4401 }, { 3827, 0x3b39 }, { 3828, 0x8156 }, { 3829, 0x8159 }, { 3830, 0x815a }, { 3831, 0x4413 }, { 3832, 0x583a },
    { 3833, 0x817c }, { 3834, 0x8184 }, { 3835, 0x4425 }, { 3836, 0x8193 }, { 3837, 0x442d }, { 3838, 0x81a5 }, { 3839, 0x57ef }, { 3840, 0x81c1 },
    { 3841, 0x81e4 }, { 3842, 0x8254 }, { 3843, 0x448f }, { 3844, 0x82a6 }, { 3845, 0x8276 }, { 3846, 0x82ca }, { 3847, 0x82d8 }, { 3848, 0x82ff },
    { 3849, 0x44b0 }, { 3850, 0x8357 }, { 3851, 0x9669 }, { 3852, 0x698a }, { 3853, 0x8405 }, { 3854, 0x70f5 }, { 3855, 0x8464 }, { 3856, 0x60e3 },
    { 3857, 0x8488 }, { 3858, 0x4504 }, { 3859, 0x84be }, { 3860, 0x84e1 }, { 3861, 0x84f8 }, { 3862, 0x8510 }, { 3863, 0x8538 }, { 3864, 0x8552 },
    { 3865, 0x453b }, { 3866, 0x856f }, { 3867, 0x8570 }, { 3868, 0x85e0 }, { 3869, 0x4577 }, { 3870, 0x8672 }, { 3871, 0x8692 }, { 3872, 0x86b2 },
    { 3873, 0x86ef }, { 3874, 0x9645 }, { 3875, 0x878b }, { 3876, 0x4606 }, { 3877, 0x4617 }, { 3878, 0x88ae }, { 3879, 0x88ff }, { 3880, 0x8924 },
    { 3881, 0x8947 }, { 3882, 0x8991 }, { 3883, 0x27967 }, { 3884, 0x8a29 }, { 3885, 0x8a38 }, { 3886, 0x8a94 }, { 3887, 0x8ab4 }, { 3888, 0x8c51 },
    { 3889, 0x8cd4 }, { 3890, 0x8cf2 }, { 3891, 0x8d1c }, { 3892, 0x4798 }, { 3893, 0x585f }, { 3894, 0x8dc3 }, { 3895, 0x47ed }, { 3896, 0x4eee },
    { 3897, 0x8e3a }, { 3898, 0x55d8 }, { 3899, 0x5754 }, { 3900, 0x8e71 }, { 3901, 0x55f5 }, { 3902, 0x8eb0 }, { 3903, 0x4837 }, { 3904, 0x8ece },
    { 3905, 0x8ee2 }, { 3906, 0x8ee4 }, { 3907, 0x8eed }, { 3908, 0x8ef2 }, { 3909, 0x8fb7 }, { 3910, 0x8fc1 }, { 3911, 0x8fca }, { 3912, 0x8fcc },
    { 3913, 0x9033 }, { 3914, 0x99c4 }, { 3915, 0x48ad }, { 3916, 0x98e0 }, { 3917, 0x9213 }, { 3918, 0x491e }, { 3919, 0x9228 }, { 3920, 0x9258 },
    { 3921, 0x926b }, { 3922, 0x92b1 }, { 3923, 0x92ae }, { 3924, 0x92bf }, { 3925, 0x92e3 }, { 3926, 0x92eb }, { 3927, 0x92f3 }, { 3928, 0x92f4 },
    { 3929, 0x92fd }, { 3930, 0x9343 }, { 3931, 0x9384 }, { 3932, 0x93ad }, { 3933, 0x4945 }, { 3934, 0x4951 }, { 3935, 0x9ebf }, { 3936, 0x9417 },
    { 3937, 0x5301 }, { 3938, 0x941d }, { 3939, 0x942d }, { 3940, 0x943e }, { 3941, 0x496a }, { 3942, 0x9454 }, { 3943, 0x9479 }, { 3944, 0x952d },
    { 3945, 0x95a2 }, { 3946, 0x49a7 }, { 3947, 0x95f4 }, { 3948, 0x9633 }, { 3949, 0x49e5 }, { 3950, 0x67a0 }, { 3951, 0x4a24 }, { 3952, 0x9740 },
    { 3953, 0x4a35 }, { 3954, 0x97b2 }, { 3955, 0x97c2 }, { 3956, 0x5654 }, { 3957, 0x4ae4 }, { 3958, 0x60e8 }, { 3959, 0x98b9 }, { 3960, 0x4b19 },
    { 3961, 0x98f1 }, { 3962, 0x5844 }, { 3963, 0x990e }, { 3964, 0x9919 }, { 3965, 0x51b4 }, { 3966, 0x991c }, { 3967, 0x9937 }, { 3968, 0x9942 },
    { 3969, 0x995d }, { 3970, 0x9962 }, { 3971, 0x4b70 }, { 3972, 0x99c5 }, { 3973, 0x4b9d }, { 3974, 0x9a3c }, { 3975, 0x9b0f }, { 3976, 0x7a83 },
    { 3977, 0x9b69 }, { 3978, 0x9b81 }, { 3979, 0x9bdd }, { 3980, 0x9bf1 }, { 3981, 0x9bf4 }, { 3982, 0x4c6d }, { 3983, 0x9c20 }, { 3984, 0x376f },
    { 3985, 0x21bc2 }, { 3986, 0x9d49 }, { 3987, 0x9c3a }, { 3988, 0x9efe }, { 3989, 0x5650 }, { 3990, 0x9d93 }, { 3991, 0x9dbd }, { 3992, 0x9dc0 },
    { 3993, 0x9dfc }, { 3994, 0x94f6 }, { 3995, 0x8fb6 }, { 3996, 0x9e7b }, { 3997, 0x9eac }, { 3998, 0x9eb1 }, { 3999, 0x9ebd }, { 4000, 0x9ec6 },
    { 4001, 0x94dc }, { 4002, 0x9ee2 }, { 4003, 0x9ef1 }, { 4004, 0x9ef8 }, { 4005, 0x7ac8 }, { 4006, 0x9f44 }, { 4007, 0x20094 }, { 4008, 0x202b7 },
    { 4009, 0x203a0 }, { 4010, 0x691a }, { 4011, 0x94c3 }, { 4012, 0x59ac }, { 4013, 0x204d7 }, { 4014, 0x5840 }, { 4015, 0x94c1 }, { 4016, 0x37b9 },
    { 4017, 0x205d5 }, { 4018, 0x20615 }, { 4019, 0x20676 }, { 4020, 0x216ba }, { 4021, 0x5757 }, { 4022, 0x7173 }, { 4023, 0x20ac2 }, { 4024, 0x20acd },
    { 4025, 0x20bbf }, { 4026, 0x546a }, { 4027, 0x2f83b }, { 4028, 0x20bcb }, { 4029, 0x549e }, { 4030, 0x20bfb }, { 4031, 0x20c3b }, { 4032, 0x20c53 },
    { 4033, 0x20c65 }, { 4034, 0x20c7c }, { 4035, 0x60e7 }, { 4036, 0x20c8d }, { 4037, 0x567a }, { 4038, 0x20cb5 }, { 4039, 0x20cdd }, { 4040, 0x20ced },
    { 4041, 0x20d6f }, { 4042, 0x20db2 }, { 4043, 0x20dc8 }, { 4044, 0x6955 }, { 4045, 0x9c2f }, { 4046, 0x87a5 }, { 4047, 0x20e04 }, { 4048, 0x20e0e },
    { 4049, 0x20ed7 }, { 4050, 0x20f90 }, { 4051, 0x20f2d }, { 4052, 0x20e73 }, { 4053, 0x5c20 }, { 4054, 0x20fbc }, { 4055, 0x5e0b }, { 4056, 0x2105c },
    { 4057, 0x2104f }, { 4058, 0x21076 }, { 4059, 0x671e }, { 4060, 0x2107b }, { 4061, 0x21088 }, { 4062, 0x21096 }, { 4063, 0x3647 }, { 4064, 0x210bf },
    { 4065, 0x210d3 }, { 4066, 0x2112f }, { 4067, 0x2113b }, { 4068, 0x5364 }, { 4069, 0x84ad }, { 4070, 0x212e3 }, { 4071, 0x21375 }, { 4072, 0x21336 },
    { 4073, 0x8b81 }, { 4074, 0x21577 }, { 4075, 0x21619 }, { 4076, 0x217c3 }, { 4077, 0x217c7 }, { 4078, 0x4e78 }, { 4079, 0x70bb }, { 4080, 0x2182d },
    { 4081, 0x2196a }, { 4082, 0x21a2d }, { 4083, 0x21a45 }, { 4084, 0x21c2a }, { 4085, 0x21c70 }, { 4086, 0x21cac }, { 4087, 0x21ec8 }, { 4088, 0x62c3 },
    { 4089, 0x21ed5 }, { 4090, 0x21f15 }, { 4091, 0x7198 }, { 4092, 0x6855 }, { 4093, 0x22045 }, { 4094, 0x69e9 }, { 4095, 0x36c8 }, { 4096, 0x2227c },
    { 4097, 0x223d7 }, { 4098, 0x223fa }, { 4099, 0x2272a }, { 4100, 0x22871 }, { 4101, 0x2294f }, { 4102, 0x82fd }, { 4103, 0x22967 }, { 4104, 0x22993 },
    { 4105, 0x22ad5 }, { 4106, 0x89a5 }, { 4107, 0x22ae8 }, { 4108, 0x8fa0 }, { 4109, 0x22b0e }, { 4110, 0x97b8 }, { 4111, 0x22b3f }, { 4112, 0x9847 },
    { 4113, 0x9abd }, { 4114, 0x22c4c }, { 4116, 0x22c88 }, { 4117, 0x22cb7 }, { 4118, 0x25be8 }, { 4119, 0x22d08 }, { 4120, 0x22d12 }, { 4121, 0x22db7 },
    { 4122, 0x22d95 }, { 4123, 0x22e42 }, { 4124, 0x22f74 }, { 4125, 0x22fcc }, { 4126, 0x23033 }, { 4127, 0x23066 }, { 4128, 0x2331f }, { 4129, 0x233de },
    { 4130, 0x5fb1 }, { 4131, 0x6648 }, { 4132, 0x66bf }, { 4133, 0x27a79 }, { 4134, 0x23567 }, { 4135, 0x235f3 }, { 4136, 0x7201 }, { 4137, 0x249ba },
    { 4138, 0x77d7 }, { 4139, 0x2361a }, { 4140, 0x23716 }, { 4141, 0x7e87 }, { 4142, 0x20346 }, { 4143, 0x58b5 }, { 4144, 0x670e }, { 4145, 0x6918 },
    { 4146, 0x23aa7 }, { 4147, 0x27657 }, { 4148, 0x25fe2 }, { 4149, 0x23e11 }, { 4150, 0x23eb9 }, { 4151, 0x275fe }, { 4152, 0x2209a }, { 4153, 0x48d0 },
    { 4154, 0x4ab8 }, { 4155, 0x24119 }, { 4156, 0x28a9a }, { 4157, 0x242ee }, { 4158, 0x2430d }, { 4159, 0x2403b }, { 4160, 0x24334 }, { 4161, 0x24396 },
    { 4162, 0x24a45 }, { 4163, 0x205ca }, { 4164, 0x51d2 }, { 4165, 0x20611 }, { 4166, 0x599f }, { 4167, 0x21ea8 }, { 4168, 0x3bbe }, { 4169, 0x23cff },
    { 4170, 0x24404 }, { 4171, 0x244d6 }, { 4172, 0x5788 }, { 4173, 0x24674 }, { 4174, 0x399b }, { 4175, 0x2472f }, { 4176, 0x285e8 }, { 4177, 0x299c9 },
    { 4178, 0x3762 }, { 4179, 0x221c3 }, { 4180, 0x8b5e }, { 4181, 0x28b4e }, { 4182, 0x99d6 }, { 4183, 0x24812 }, { 4184, 0x248fb }, { 4185, 0x24a15 },
    { 4186, 0x7209 }, { 4187, 0x24ac0 }, { 4188, 0x20c78 }, { 4189, 0x5965 }, { 4190, 0x24ea5 }, { 4191, 0x24f86 }, { 4192, 0x20779 }, { 4193, 0x8eda },
    { 4194, 0x2502c }, { 4195, 0x528f }, { 4196, 0x573f }, { 4197, 0x7171 }, { 4198, 0x25299 }, { 4199, 0x25419 }, { 4200, 0x23f4a }, { 4201, 0x24aa7 },
    { 4202, 0x55bc }, { 4203, 0x25446 }, { 4204, 0x2546e }, { 4205, 0x26b52 }, { 4206, 0x91d4 }, { 4207, 0x3473 }, { 4208, 0x2553f }, { 4209, 0x27632 },
    { 4210, 0x2555e }, { 4211, 0x4718 }, { 4212, 0x25562 }, { 4213, 0x25566 }, { 4214, 0x257c7 }, { 4215, 0x2493f }, { 4216, 0x2585d }, { 4217, 0x5066 },
    { 4218, 0x34fb }, { 4219, 0x233cc }, { 4220, 0x60de }, { 4221, 0x25903 }, { 4222, 0x477c }, { 4223, 0x28948 }, { 4224, 0x25aae }, { 4225, 0x25b89 },
    { 4226, 0x25c06 }, { 4227, 0x21d90 }, { 4228, 0x57a1 }, { 4229, 0x7151 }, { 4230, 0x6fb6 }, { 4231, 0x26102 }, { 4232, 0x27c12 }, { 4233, 0x9056 },
    { 4234, 0x261b2 }, { 4235, 0x24f9a }, { 4236, 0x8b62 }, { 4237, 0x26402 }, { 4238, 0x2644a }, { 4239, 0x5d5b }, { 4240, 0x26bf7 }, { 4241, 0x8f36 },
    { 4242, 0x26484 }, { 4243, 0x2191c }, { 4244, 0x8aea }, { 4245, 0x249f6 }, { 4246, 0x26488 }, { 4247, 0x23fef }, { 4248, 0x26512 }, { 4249, 0x4bc0 },
    { 4250, 0x265bf }, { 4251, 0x266b5 }, { 4252, 0x2271b }, { 4253, 0x9465 }, { 4254, 0x257e1 }, { 4255, 0x6195 }, { 4256, 0x5a27 }, { 4257, 0x2f8cd },
    { 4258, 0x4fbb }, { 4259, 0x56b9 }, { 4260, 0x24521 }, { 4261, 0x266fc }, { 4262, 0x4e6a }, { 4263, 0x24934 }, { 4264, 0x9656 }, { 4265, 0x6d8f },
    { 4266, 0x26cbd }, { 4267, 0x3618 }, { 4268, 0x8977 }, { 4269, 0x26799 }, { 4270, 0x2686e }, { 4271, 0x26411 }, { 4272, 0x2685e }, { 4273, 0x71df },
    { 4274, 0x268c7 }, { 4275, 0x7b42 }, { 4276, 0x290c0 }, { 4277, 0x20a11 }, { 4278, 0x26926 }, { 4279, 0x9104 }, { 4280, 0x26939 }, { 4281, 0x7a45 },
    { 4282, 0x9df0 }, { 4283, 0x269fa }, { 4284, 0x9a26 }, { 4285, 0x26a2d }, { 4286, 0x365f }, { 4287, 0x26469 }, { 4288, 0x20021 }, { 4289, 0x7983 },
    { 4290, 0x26a34 }, { 4291, 0x26b5b }, { 4292, 0x5d2c }, { 4293, 0x23519 }, { 4294, 0x83cf }, { 4295, 0x26b9d }, { 4296, 0x46d0 }, { 4297, 0x26ca4 },
    { 4298, 0x753b }, { 4299, 0x8865 }, { 4300, 0x26dae }, { 4301, 0x58b6 }, { 4302, 0x371c }, { 4303, 0x2258d }, { 4304, 0x2704b }, { 4305, 0x271cd },
    { 4306, 0x3c54 }, { 4307, 0x27280 }, { 4308, 0x27285 }, { 4309, 0x9281 }, { 4310, 0x2217a }, { 4311, 0x2728b }, { 4312, 0x9330 }, { 4313, 0x272e6 },
    { 4314, 0x249d0 }, { 4315, 0x6c39 }, { 4316, 0x949f }, { 4317, 0x27450 }, { 4318, 0x20ef8 }, { 4319, 0x8827 }, { 4320, 0x88f5 }, { 4321, 0x22926 },
    { 4322, 0x28473 }, { 4323, 0x217b1 }, { 4324, 0x6eb8 }, { 4325, 0x24a2a }, { 4326, 0x21820 }, { 4327, 0x39a4 }, { 4328, 0x36b9 }, { 4329, 0x5c10 },
    { 4330, 0x79e3 }, { 4331, 0x453f }, { 4332, 0x66b6 }, { 4333, 0x29cad }, { 4334, 0x298a4 }, { 4335, 0x8943 }, { 4336, 0x277cc }, { 4337, 0x27858 },
    { 4338, 0x56d6 }, { 4339, 0x40df }, { 4340, 0x2160a }, { 4341, 0x39a1 }, { 4342, 0x2372f }, { 4343, 0x280e8 }, { 4344, 0x213c5 }, { 4345, 0x71ad },
    { 4346, 0x8366 }, { 4347, 0x279dd }, { 4348, 0x291a8 }, { 4349, 0x5a67 }, { 4350, 0x4cb7 }, { 4351, 0x270af }, { 4352, 0x289ab }, { 4353, 0x279fd },
    { 4354, 0x27a0a }, { 4355, 0x27b0b }, { 4356, 0x27d66 }, { 4357, 0x2417a }, { 4358, 0x7b43 }, { 4359, 0x797e }, { 4360, 0x28009 }, { 4361, 0x6fb5 },
    { 4362, 0x2a2df }, { 4363, 0x6a03 }, { 4364, 0x28318 }, { 4365, 0x53a2 }, { 4366, 0x26e07 }, { 4367, 0x93bf }, { 4368, 0x6836 }, { 4369, 0x975d },
    { 4370, 0x2816f }, { 4371, 0x28023 }, { 4372, 0x269b5 }, { 4373, 0x213ed }, { 4374, 0x2322f }, { 4375, 0x28048 }, { 4376, 0x5d85 }, { 4377, 0x28c30 },
    { 4378, 0x28083 }, { 4379, 0x5715 }, { 4380, 0x9823 }, { 4381, 0x28949 }, { 4382, 0x5dab }, { 4383, 0x24988 }, { 4384, 0x65be }, { 4385, 0x69d5 },
    { 4386, 0x53d2 }, { 4387, 0x24aa5 }, { 4388, 0x23f81 }, { 4389, 0x3c11 }, { 4390, 0x6736 }, { 4391, 0x28090 }, { 4392, 0x280f4 }, { 4393, 0x2812e },
    { 4394, 0x21fa1 }, { 4395, 0x2814f }, { 4396, 0x28189 }, { 4397, 0x281af }, { 4398, 0x2821a }, { 4399, 0x28306 }, { 4400, 0x2832f }, { 4401, 0x2838a },
    { 4402, 0x35ca }, { 4403, 0x28468 }, { 4404, 0x286aa }, { 4405, 0x48fa }, { 4406, 0x63e6 }, { 4407, 0x28956 }, { 4408, 0x7808 }, { 4409, 0x9255 },
    { 4410, 0x289b8 }, { 4411, 0x43f2 }, { 4412, 0x289e7 }, { 4413, 0x43df }, { 4414, 0x289e8 }, { 4415, 0x28b46 }, { 4416, 0x28bd4 }, { 4417, 0x59f8 },
    { 4418, 0x28c09 }, { 4419, 0x8f0b }, { 4420, 0x28fc5 }, { 4421, 0x290ec }, { 4422, 0x7b51 }, { 4423, 0x29110 }, { 4424, 0x2913c }, { 4425, 0x3df7 },
    { 4426, 0x2915e }, { 4427, 0x24aca }, { 4428, 0x8fd0 }, { 4429, 0x728f }, { 4430, 0x568b }, { 4431, 0x294e7 }, { 4432, 0x295e9 }, { 4433, 0x295b0 },
    { 4434, 0x295b8 }, { 4435, 0x29732 }, { 4436, 0x298d1 }, { 4437, 0x29949 }, { 4438, 0x2996a }, { 4439, 0x299c3 }, { 4440, 0x29a28 }, { 4441, 0x29b0e },
    { 4442, 0x29d5a }, { 4443, 0x29d9b }, { 4444, 0x7e9f }, { 4445, 0x29ef8 }, { 4446, 0x29f23 }, { 4447, 0x4ca4 }, { 4448, 0x9547 }, { 4449, 0x2a293 },
    { 4450, 0x71a2 }, { 4451, 0x2a2ff }, { 4452, 0x4d91 }, { 4453, 0x9012 }, { 4454, 0x2a5cb }, { 4455, 0x4d9c }, { 4456, 0x20c9c }, { 4457, 0x8fbe },
    { 4458, 0x55c1 }, { 4459, 0x8fba }, { 4460, 0x224b0 }, { 4461, 0x8fb9 }, { 4462, 0x24a93 }, { 4463, 0x4509 }, { 4464, 0x7e7f }, { 4465, 0x6f56 },
    { 4466, 0x6ab1 }, { 4467, 0x4eea }, { 4468, 0x34e4 }, { 4469, 0x28b2c }, { 4470, 0x2789d }, { 4471, 0x373a }, { 4472, 0x8e80 }, { 4473, 0x217f5 },
    { 4474, 0x28024 }, { 4475, 0x28b6c }, { 4476, 0x28b99 }, { 4477, 0x27a3e }, { 4478, 0x266af }, { 4479, 0x3deb }, { 4480, 0x27655 }, { 4481, 0x23cb7 },
    { 4482, 0x25635 }, { 4483, 0x25956 }, { 4484, 0x4e9a }, { 4485, 0x25e81 }, { 4486, 0x26258 }, { 4487, 0x56bf }, { 4488, 0x20e6d }, { 4489, 0x8e0e },
    { 4490, 0x5b6d }, { 4491, 0x23e88 }, { 4492, 0x24c9e }, { 4493, 0x63de }, { 4494, 0x62d0 }, { 4495, 0x217f6 }, { 4496, 0x2187b }, { 4497, 0x6530 },
    { 4498, 0x562d }, { 4499, 0x25c4a }, { 4500, 0x541a }, { 4501, 0x25311 }, { 4502, 0x3dc6 }, { 4503, 0x29d98 }, { 4504, 0x4c7d }, { 4505, 0x5622 },
    { 4506, 0x561e }, { 4507, 0x7f49 }, { 4508, 0x25ed8 }, { 4509, 0x5975 }, { 4510, 0x23d40 }, { 4511, 0x8770 }, { 4512, 0x4e1c }, { 4513, 0x20fea },
    { 4514, 0x20d49 }, { 4515, 0x236ba }, { 4516, 0x8117 }, { 4517, 0x9d5e }, { 4518, 0x8d18 }, { 4519, 0x763b }, { 4520, 0x9c45 }, { 4521, 0x764e },
    { 4522, 0x77b9 }, { 4523, 0x9345 }, { 4524, 0x5432 }, { 4525, 0x8148 }, { 4526, 0x82f7 }, { 4527, 0x5625 }, { 4528, 0x8132 }, { 4529, 0x8418 },
    { 4530, 0x80bd }, { 4531, 0x55ea }, { 4532, 0x7962 }, { 4533, 0x5643 }, { 4534, 0x5416 }, { 4535, 0x20e9d }, { 4536, 0x35ce }, { 4537, 0x5605 },
    { 4538, 0x55f1 }, { 4539, 0x66f1 }, { 4540, 0x282e2 }, { 4541, 0x362d }, { 4542, 0x7534 }, { 4543, 0x55f0 }, { 4544, 0x55ba }, { 4545, 0x5497 },
    { 4546, 0x5572 }, { 4547, 0x20c41 }, { 4548, 0x20c96 }, { 4549, 0x5ed0 }, { 4550, 0x25148 }, { 4551, 0x20e76 }, { 4552, 0x22c62 }, { 4553, 0x20ea2 },
    { 4554, 0x9eab }, { 4555, 0x7d5a }, { 4556, 0x55de }, { 4557, 0x21075 }, { 4558, 0x629d }, { 4559, 0x976d }, { 4560, 0x5494 }, { 4561, 0x8ccd },
    { 4562, 0x71f6 }, { 4563, 0x9176 }, { 4564, 0x63fc }, { 4565, 0x63b9 }, { 4566, 0x63fe }, { 4567, 0x5569 }, { 4568, 0x22b43 }, { 4569, 0x9c72 },
    { 4570, 0x22eb3 }, { 4571, 0x519a }, { 4572, 0x34df }, { 4573, 0x20da7 }, { 4574, 0x51a7 }, { 4575, 0x544d }, { 4576, 0x551e }, { 4577, 0x5513 },
    { 4578, 0x7666 }, { 4579, 0x8e2d }, { 4580, 0x2688a }, { 4581, 0x75b1 }, { 4582, 0x80b6 }, { 4583, 0x8804 }, { 4584, 0x8786 }, { 4585, 0x88c7 },
    { 4586, 0x81b6 }, { 4587, 0x841c }, { 4588, 0x210c1 }, { 4589, 0x44ec }, { 4590, 0x7304 }, { 4591, 0x24706 }, { 4592, 0x5b90 }, { 4593, 0x830b },
    { 4594, 0x26893 }, { 4595, 0x567b }, { 4596, 0x226f4 }, { 4597, 0x27d2f }, { 4598, 0x241a3 }, { 4599, 0x27d73 }, { 4600, 0x26ed0 }, { 4601, 0x272b6 },
    { 4602, 0x9170 }, { 4603, 0x211d9 }, { 4604, 0x9208 }, { 4605, 0x23cfc }, { 4606, 0x2a6a9 }, { 4607, 0x20eac }, { 4608, 0x20ef9 }, { 4609, 0x7266 },
    { 4610, 0x21ca2 }, { 4611, 0x474e }, { 4612, 0x24fc2 }, { 4613, 0x27ff9 }, { 4614, 0x20feb }, { 4615, 0x40fa }, { 4616, 0x9c5d }, { 4617, 0x651f },
    { 4618, 0x22da0 }, { 4619, 0x48f3 }, { 4620, 0x247e0 }, { 4621, 0x29d7c }, { 4622, 0x20fec }, { 4623, 0x20e0a }, { 4624, 0x6062 }, { 4625, 0x275a3 },
    { 4626, 0x20fed }, { 4628, 0x26048 }, { 4629, 0x21187 }, { 4630, 0x71a3 }, { 4631, 0x7e8e }, { 4632, 0x9d50 }, { 4633, 0x4e1a }, { 4634, 0x4e04 },
    { 4635, 0x3577 }, { 4636, 0x5b0d }, { 4637, 0x6cb2 }, { 4638, 0x5367 }, { 4639, 0x36ac }, { 4640, 0x39dc }, { 4641, 0x537d }, { 4642, 0x36a5 },
    { 4643, 0x24618 }, { 4644, 0x589a }, { 4645, 0x24b6e }, { 4646, 0x822d }, { 4647, 0x544b }, { 4648, 0x57aa }, { 4649, 0x25a95 }, { 4650, 0x20979 },
    { 4652, 0x3a52 }, { 4653, 0x22465 }, { 4654, 0x7374 }, { 4655, 0x29eac }, { 4656, 0x4d09 }, { 4657, 0x9bed }, { 4658, 0x23cfe }, { 4659, 0x29f30 },
    { 4660, 0x4c5b }, { 4661, 0x24fa9 }, { 4662, 0x2959e }, { 4663, 0x29fde }, { 4664, 0x845c }, { 4665, 0x23db6 }, { 4666, 0x272b2 }, { 4667, 0x267b3 },
    { 4668, 0x23720 }, { 4669, 0x632e }, { 4670, 0x7d25 }, { 4671, 0x23ef7 }, { 4672, 0x23e2c }, { 4673, 0x3a2a }, { 4674, 0x9008 }, { 4675, 0x52cc },
    { 4676, 0x3e74 }, { 4677, 0x367a }, { 4678, 0x45e9 }, { 4679, 0x2048e }, { 4680, 0x7640 }, { 4681, 0x5af0 }, { 4682, 0x20eb6 }, { 4683, 0x787a },
    { 4684, 0x27f2e }, { 4685, 0x58a7 }, { 4686, 0x40bf }, { 4687, 0x567c }, { 4688, 0x9b8b }, { 4689, 0x5d74 }, { 4690, 0x7654 }, { 4691, 0x2a434 },
    { 4692, 0x9e85 }, { 4693, 0x4ce1 }, { 4694, 0x75f9 }, { 4695, 0x37fb }, { 4696, 0x6119 }, { 4697, 0x230da }, { 4698, 0x243f2 }, { 4700, 0x565d },
    { 4701, 0x212a9 }, { 4702, 0x57a7 }, { 4703, 0x24963 }, { 4704, 0x29e06 }, { 4705, 0x5234 }, { 4706, 0x270ae }, { 4707, 0x35ad }, { 4708, 0x6c4a },
    { 4709, 0x9d7c }, { 4710, 0x7c56 }, { 4711, 0x9b39 }, { 4712, 0x57de }, { 4713, 0x2176c }, { 4714, 0x5c53 }, { 4715, 0x64d3 }, { 4716, 0x294d0 },
    { 4717, 0x26335 }, { 4718, 0x27164 }, { 4719, 0x86ad }, { 4720, 0x20d28 }, { 4721, 0x26d22 }, { 4722, 0x24ae2 }, { 4723, 0x20d71 }, { 4725, 0x51fe },
    { 4726, 0x21f0f }, { 4727, 0x5d8e }, { 4728, 0x9703 }, { 4729, 0x21dd1 }, { 4730, 0x9e81 }, { 4731, 0x904c }, { 4732, 0x7b1f }, { 4733, 0x9b02 },
    { 4734, 0x5cd1 }, { 4735, 0x7ba3 }, { 4736, 0x6268 }, { 4737, 0x6335 }, { 4738, 0x9aff }, { 4739, 0x7bcf }, { 4740, 0x9b2a }, { 4741, 0x7c7e },
    { 4742, 0x9b2e }, { 4743, 0x7c42 }, { 4744, 0x7c86 }, { 4745, 0x9c15 }, { 4746, 0x7bfc }, { 4747, 0x9b09 }, { 4748, 0x9f17 }, { 4749, 0x9c1b },
    { 4750, 0x2493e }, { 4751, 0x9f5a }, { 4752, 0x5573 }, { 4753, 0x5bc3 }, { 4754, 0x4ffd }, { 4755, 0x9e98 }, { 4756, 0x4ff2 }, { 4757, 0x5260 },
    { 4758, 0x3e06 }, { 4759, 0x52d1 }, { 4760, 0x5767 }, { 4761, 0x5056 }, { 4762, 0x59b7 }, { 4763, 0x5e12 }, { 4764, 0x97c8 }, { 4765, 0x9dab },
    { 4766, 0x8f5c }, { 4767, 0x5469 }, { 4768, 0x97b4 }, { 4769, 0x9940 }, { 4770, 0x97ba }, { 4771, 0x532c }, { 4772, 0x6130 }, { 4773, 0x692c },
    { 4774, 0x53da }, { 4775, 0x9c0a }, { 4776, 0x9d02 }, { 4777, 0x4c3b }, { 4778, 0x9641 }, { 4779, 0x6980 }, { 4780, 0x50a6 }, { 4781, 0x7546 },
    { 4782, 0x2176d }, { 4783, 0x99da }, { 4784, 0x5273 }, { 4786, 0x9159 }, { 4787, 0x9681 }, { 4788, 0x915c }, { 4790, 0x9151 }, { 4791, 0x28e97 },
    { 4792, 0x637f }, { 4793, 0x26d23 }, { 4794, 0x6aca }, { 4795, 0x5611 }, { 4796, 0x918e }, { 4797, 0x757a }, { 4798, 0x6285 }, { 4799, 0x203fc },
    { 4800, 0x734f }, { 4801, 0x7c70 }, { 4802, 0x25c21 }, { 4803, 0x23cfd }, { 4805, 0x24919 }, { 4806, 0x76d6 }, { 4807, 0x9b9d }, { 4808, 0x4e2a },
    { 4809, 0x20cd4 }, { 4810, 0x83be }, { 4811, 0x8842 }, { 4813, 0x5c4a }, { 4814, 0x69c0 }, { 4815, 0x50ed }, { 4816, 0x577a }, { 4817, 0x521f },
    { 4818, 0x5df5 }, { 4819, 0x4ece }, { 4820, 0x6c31 }, { 4821, 0x201f2 }, { 4822, 0x4f39 }, { 4823, 0x549c }, { 4824, 0x54da }, { 4825, 0x529a },
    { 4826, 0x8d82 }, { 4827, 0x35fe }, { 4828, 0x5f0c }, { 4829, 0x35f3 }, { 4831, 0x6b52 }, { 4832, 0x917c }, { 4833, 0x9fa5 }, { 4834, 0x9b97 },
    { 4835, 0x982e }, { 4836, 0x98b4 }, { 4837, 0x9aba }, { 4838, 0x9ea8 }, { 4839, 0x9e84 }, { 4840, 0x717a }, { 4841, 0x7b14 }, { 4843, 0x6bfa },
    { 4844, 0x8818 }, { 4845, 0x7f78 }, { 4847, 0x5620 }, { 4848, 0x2a64a }, { 4849, 0x8e77 }, { 4850, 0x9f53 }, { 4852, 0x8dd4 }, { 4853, 0x8e4f },
    { 4854, 0x9e1c }, { 4855, 0x8e01 }, { 4856, 0x6282 }, { 4857, 0x2837d }, { 4858, 0x8e28 }, { 4859, 0x8e75 }, { 4860, 0x7ad3 }, { 4861, 0x24a77 },
    { 4862, 0x7a3e }, { 4863, 0x78d8 }, { 4864, 0x6cea }, { 4865, 0x8a67 }, { 4866, 0x7607 }, { 4867, 0x28a5a }, { 4868, 0x9f26 }, { 4869, 0x6cce },
    { 4870, 0x87d6 }, { 4871, 0x75c3 }, { 4872, 0x2a2b2 }, { 4873, 0x7853 }, { 4874, 0x2f840 }, { 4875, 0x8d0c }, { 4876, 0x72e2 }, { 4877, 0x7371 },
    { 4878, 0x8b2d }, { 4879, 0x7302 }, { 4880, 0x74f1 }, { 4881, 0x8ceb }, { 4882, 0x24abb }, { 4883, 0x862f }, { 4884, 0x5fba }, { 4885, 0x88a0 },
    { 4886, 0x44b7 }, { 4888, 0x2183b }, { 4889, 0x26e05 }, { 4891, 0x8a7e }, { 4892, 0x2251b }, { 4894, 0x60fd }, { 4895, 0x7667 }, { 4896, 0x9ad7 },
    { 4897, 0x9d44 }, { 4898, 0x936e }, { 4899, 0x9b8f }, { 4900, 0x87f5 }, { 4902, 0x880f }, { 4903, 0x8cf7 }, { 4904, 0x732c }, { 4905, 0x9721 },
    { 4906, 0x9bb0 }, { 4907, 0x35d6 }, { 4908, 0x72b2 }, { 4909, 0x4c07 }, { 4910, 0x7c51 }, { 4911, 0x994a }, { 4912, 0x26159 }, { 4913, 0x6159 },
    { 4914, 0x4c04 }, { 4915, 0x9e96 }, { 4916, 0x617d }, { 4918, 0x575f }, { 4919, 0x616f }, { 4920, 0x62a6 }, { 4921, 0x6239 }, { 4922, 0x62ce },
    { 4923, 0x3a5c }, { 4924, 0x61e2 }, { 4925, 0x53aa }, { 4926, 0x233f5 }, { 4927, 0x6364 }, { 4928, 0x6802 }, { 4929, 0x35d2 }, { 4930, 0x5d57 },
    { 4931, 0x28bc2 }, { 4932, 0x8fda }, { 4933, 0x28e39 }, { 4935, 0x50d9 }, { 4936, 0x21d46 }, { 4937, 0x7906 }, { 4938, 0x5332 }, { 4939, 0x9638 },
    { 4940, 0x20f3b }, { 4941, 0x4065 }, { 4943, 0x77fe }, { 4945, 0x7cc2 }, { 4946, 0x25f1a }, { 4947, 0x7cda }, { 4948, 0x7a2d }, { 4949, 0x8066 },
    { 4950, 0x8063 }, { 4951, 0x7d4d }, { 4952, 0x7505 }, { 4953, 0x74f2 }, { 4954, 0x8994 }, { 4955, 0x821a }, { 4956, 0x670c }, { 4957, 0x8062 },
    { 4958, 0x27486 }, { 4959, 0x805b }, { 4960, 0x74f0 }, { 4961, 0x8103 }, { 4962, 0x7724 }, { 4963, 0x8989 }, { 4964, 0x267cc }, { 4965, 0x7553 },
    { 4966, 0x26ed1 }, { 4967, 0x87a9 }, { 4968, 0x87ce }, { 4969, 0x81c8 }, { 4970, 0x878c }, { 4971, 0x8a49 }, { 4972, 0x8cad }, { 4973, 0x8b43 },
    { 4974, 0x772b }, { 4975, 0x74f8 }, { 4976, 0x84da }, { 4977, 0x3635 }, { 4978, 0x69b2 }, { 4979, 0x8da6 }, { 4981, 0x89a9 }, { 4982, 0x7468 },
    { 4983, 0x6db9 }, { 4984, 0x87c1 }, { 4985, 0x24011 }, { 4986, 0x74e7 }, { 4987, 0x3ddb }, { 4988, 0x7176 }, { 4989, 0x60a4 }, { 4990, 0x619c },
    { 4991, 0x3cd1 }, { 4992, 0x7162 }, { 4993, 0x6077 }, { 4995, 0x7f71 }, { 4996, 0x28b2d }, { 4997, 0x7250 }, { 4998, 0x60e9 }, { 4999, 0x4b7e },
    { 5000, 0x5220 }, { 5001, 0x3c18 }, { 5002, 0x23cc7 }, { 5003, 0x25ed7 }, { 5004, 0x27656 }, { 5005, 0x25531 }, { 5006, 0x21944 }, { 5007, 0x212fe },
    { 5008, 0x29903 }, { 5009, 0x26ddc }, { 5010, 0x270ad }, { 5011, 0x5cc1 }, { 5012, 0x261ad }, { 5013, 0x28a0f }, { 5014, 0x23677 }, { 5015, 0x200ee },
    { 5016, 0x26846 }, { 5017, 0x24f0e }, { 5018, 0x4562 }, { 5019, 0x5b1f }, { 5020, 0x2634c }, { 5021, 0x9f50 }, { 5022, 0x9ea6 }, { 5023, 0x2626b },
    { 5432, 0x2400 }, { 5433, 0x2401 }, { 5434, 0x2402 }, { 5435, 0x2403 }, { 5436, 0x2404 }, { 5437, 0x2405 }, { 5438, 0x2406 }, { 5439, 0x2407 },
    { 5440, 0x2408 }, { 5441, 0x2409 }, { 5442, 0x240a }, { 5443, 0x240b }, { 5444, 0x240c }, { 5445, 0x240d }, { 5446, 0x240e }, { 5447, 0x240f },
    { 5448, 0x2410 }, { 5449, 0x2411 }, { 5450, 0x2412 }, { 5451, 0x2413 }, { 5452, 0x2414 }, { 5453, 0x2415 }, { 5454, 0x2416 }, { 5455, 0x2417 },
    { 5456, 0x2418 }, { 5457, 0x2419 }, { 5458, 0x241a }, { 5459, 0x241b }, { 5460, 0x241c }, { 5461, 0x241d }, { 5462, 0x241e }, { 5463, 0x241f },
    { 5464, 0x2421 }, { 10896, 0x2460 }, { 10897, 0x2461 }, { 10898, 0x2462 }, { 10899, 0x2463 }, { 10900, 0x2464 }, { 10901, 0x2465 }, { 10902, 0x2466 },
    { 10903, 0x2467 }, { 10904, 0x2468 }, { 10905, 0x2469 }, { 10906, 0x2474 }, { 10907, 0x2475 }, { 10908, 0x2476 }, { 10909, 0x2477 }, { 10910, 0x2478 },
    { 10911, 0x2479 }, { 10912, 0x247a }, { 10913, 0x247b }, { 10914, 0x247c }, { 10915, 0x247d }, { 10916, 0x2170 }, { 10917, 0x2171 }, { 10918, 0x2172 },
    { 10919, 0x2173 }, { 10920, 0x2174 }, { 10921, 0x2175 }, { 10922, 0x2176 }, { 10923, 0x2177 }, { 10924, 0x2178 }, { 10925, 0x2179 }, { 10926, 0x4e36 },
    { 10927, 0x4e3f }, { 10928, 0x4e85 }, { 10929, 0x4ea0 }, { 10930, 0x5182 }, { 10931, 0x5196 }, { 10932, 0x51ab }, { 10933, 0x52f9 }, { 10934, 0x5338 },
    { 10935, 0x5369 }, { 10936, 0x53b6 }, { 10937, 0x590a }, { 10938, 0x5b80 }, { 10939, 0x5ddb }, { 10940, 0x2f33 }, { 10941, 0x5e7f }, { 10942, 0x5ef4 },
    { 10943, 0x5f50 }, { 10944, 0x5f61 }, { 10945, 0x6534 }, { 10946, 0x65e0 }, { 10947, 0x7592 }, { 10948, 0x7676 }, { 10949, 0x8fb5 }, { 10950, 0x96b6 },
    { 10951, 0xa8 }, { 10952, 0x2c6 }, { 10953, 0x30fd }, { 10954, 0x30fe }, { 10955, 0x309d }, { 10956, 0x309e }, { 10957, 0x3003 }, { 10958, 0x4edd },
    { 10959, 0x3005 }, { 10960, 0x3006 }, { 10961, 0x3007 }, { 10962, 0x30fc }, { 10963, 0xff3b }, { 10964, 0xff3d }, { 10965, 0x273d }, { 10966, 0x3041 },
    { 10967, 0x3042 }, { 10968, 0x3043 }, { 10969, 0x3044 }, { 10970, 0x3045 }, { 10971, 0x3046 }, { 10972, 0x3047 }, { 10973, 0x3048 }, { 10974, 0x3049 },
    { 10975, 0x304a }, { 10976, 0x304b }, { 10977, 0x304c }, { 10978, 0x304d }, { 10979, 0x304e }, { 10980, 0x304f }, { 10981, 0x3050 }, { 10982, 0x3051 },
    { 10983, 0x3052 }, { 10984, 0x3053 }, { 10985, 0x3054 }, { 10986, 0x3055 }, { 10987, 0x3056 }, { 10988, 0x3057 }, { 10989, 0x3058 }, { 10990, 0x3059 },
    { 10991, 0x305a }, { 10992, 0x305b }, { 10993, 0x305c }, { 10994, 0x305d }, { 10995, 0x305e }, { 10996, 0x305f }, { 10997, 0x3060 }, { 10998, 0x3061 },
    { 10999, 0x3062 }, { 11000, 0x3063 }, { 11001, 0x3064 }, { 11002, 0x3065 }, { 11003, 0x3066 }, { 11004, 0x3067 }, { 11005, 0x3068 }, { 11006, 0x3069 },
    { 11007, 0x306a }, { 11008, 0x306b }, { 11009, 0x306c }, { 11010, 0x306d }, { 11011, 0x306e }, { 11012, 0x306f }, { 11013, 0x3070 }, { 11014, 0x3071 },
    { 11015, 0x3072 }, { 11016, 0x3073 }, { 11017, 0x3074 }, { 11018, 0x3075 }, { 11019, 0x3076 }, { 11020, 0x3077 }, { 11021, 0x3078 }, { 11022, 0x3079 },
    { 11023, 0x307a }, { 11024, 0x307b }, { 11025, 0x307c }, { 11026, 0x307d }, { 11027, 0x307e }, { 11028, 0x307f }, { 11029, 0x3080 }, { 11030, 0x3081 },
    { 11031, 0x3082 }, { 11032, 0x3083 }, { 11033, 0x3084 }, { 11034, 0x3085 }, { 11035, 0x3086 }, { 11036, 0x3087 }, { 11037, 0x3088 }, { 11038, 0x3089 },
    { 11039, 0x308a }, { 11040, 0x308b }, { 11041, 0x308c }, { 11042, 0x308d }, { 11043, 0x308e }, { 11044, 0x308f }, { 11045, 0x3090 }, { 11046, 0x3091 },
    { 11047, 0x3092 }, { 11048, 0x3093 }, { 11049, 0x30a1 }, { 11050, 0x30a2 }, { 11051, 0x30a3 }, { 11052, 0x30a4 }, { 11053, 0x30a5 }, { 11054, 0x30a6 },
    { 11055, 0x30a7 }, { 11056, 0x30a8 }, { 11057, 0x30a9 }, { 11058, 0x30aa }, { 11059, 0x30ab }, { 11060, 0x30ac }, { 11061, 0x30ad }, { 11062, 0x30ae },
    { 11063, 0x30af }, { 11064, 0x30b0 }, { 11065, 0x30b1 }, { 11066, 0x30b2 }, { 11067, 0x30b3 }, { 11068, 0x30b4 }, { 11069, 0x30b5 }, { 11070, 0x30b6 },
    { 11071, 0x30b7 }, { 11072, 0x30b8 }, { 11073, 0x30b9 }, { 11074, 0x30ba }, { 11075, 0x30bb }, { 11076, 0x30bc }, { 11077, 0x30bd }, { 11078, 0x30be },
    { 11079, 0x30bf }, { 11080, 0x30c0 }, { 11081, 0x30c1 }, { 11082, 0x30c2 }, { 11083, 0x30c3 }, { 11084, 0x30c4 }, { 11085, 0x30c5 }, { 11086, 0x30c6 },
    { 11087, 0x30c7 }, { 11088, 0x30c8 }, { 11089, 0x30c9 }, { 11090, 0x30ca }, { 11091, 0x30cb }, { 11092, 0x30cc }, { 11093, 0x30cd }, { 11094, 0x30ce },
    { 11095, 0x30cf }, { 11096, 0x30d0 }, { 11097, 0x30d1 }, { 11098, 0x30d2 }, { 11099, 0x30d3 }, { 11100, 0x30d4 }, { 11101, 0x30d5 }, { 11102, 0x30d6 },
    { 11103, 0x30d7 }, { 11104, 0x30d8 }, { 11105, 0x30d9 }, { 11106, 0x30da }, { 11107, 0x30db }, { 11108, 0x30dc }, { 11109, 0x30dd }, { 11110, 0x30de },
    { 11111, 0x30df }, { 11112, 0x30e0 }, { 11113, 0x30e1 }, { 11114, 0x30e2 }, { 11115, 0x30e3 }, { 11116, 0x30e4 }, { 11117, 0x30e5 }, { 11118, 0x30e6 },
    { 11119, 0x30e7 }, { 11120, 0x30e8 }, { 11121, 0x30e9 }, { 11122, 0x30ea }, { 11123, 0x30eb }, { 11124, 0x30ec }, { 11125, 0x30ed }, { 11126, 0x30ee },
    { 11127, 0x30ef }, { 11128, 0x30f0 }, { 11129, 0x30f1 }, { 11130, 0x30f2 }, { 11131, 0x30f3 }, { 11132, 0x30f4 }, { 11133, 0x30f5 }, { 11134, 0x30f6 },
    { 11135, 0x410 }, { 11136, 0x411 }, { 11137, 0x412 }, { 11138, 0x413 }, { 11139, 0x414 }, { 11140, 0x415 }, { 11141, 0x401 }, { 11142, 0x416 },
    { 11143, 0x417 }, { 11144, 0x418 }, { 11145, 0x419 }, { 11146, 0x41a }, { 11147, 0x41b }, { 11148, 0x41c }, { 11149, 0x41d }, { 11150, 0x41e },
    { 11151, 0x41f }, { 11152, 0x420 }, { 11153, 0x421 }, { 11154, 0x422 }, { 11155, 0x423 }, { 11156, 0x424 }, { 11157, 0x425 }, { 11158, 0x426 },
    { 11159, 0x427 }, { 11160, 0x428 }, { 11161, 0x429 }, { 11162, 0x42a }, { 11163, 0x42b }, { 11164, 0x42c }, { 11165, 0x42d }, { 11166, 0x42e },
    { 11167, 0x42f }, { 11168, 0x430 }, { 11169, 0x431 }, { 11170, 0x432 }, { 11171, 0x433 }, { 11172, 0x434 }, { 11173, 0x435 }, { 11174, 0x451 },
    { 11175, 0x436 }, { 11176, 0x437 }, { 11177, 0x438 }, { 11178, 0x439 }, { 11179, 0x43a }, { 11180, 0x43b }, { 11181, 0x43c }, { 11182, 0x43d },
    { 11183, 0x43e }, { 11184, 0x43f }, { 11185, 0x440 }, { 11186, 0x441 }, { 11187, 0x442 }, { 11188, 0x443 }, { 11189, 0x444 }, { 11190, 0x445 },
    { 11191, 0x446 }, { 11192, 0x447 }, { 11193, 0x448 }, { 11194, 0x449 }, { 11195, 0x44a }, { 11196, 0x44b }, { 11197, 0x44c }, { 11198, 0x44d },
    { 11199, 0x44e }, { 11200, 0x44f }, { 11201, 0x21e7 }, { 11202, 0x21b8 }, { 11203, 0x21b9 }, { 11204, 0x31cf }, { 11205, 0x200cc }, { 11206, 0x4e5a },
    { 11207, 0x2008a }, { 11208, 0x5202 }, { 11209, 0x4491 }, { 11210, 0x9fb0 }, { 11211, 0x5188 }, { 11212, 0x9fb1 }, { 11213, 0x27607 }, { 11254, 0xffe2 },
    { 11255, 0xffe4 }, { 11256, 0xff07 }, { 11257, 0xff02 }, { 11258, 0x3231 }, { 11259, 0x2116 }, { 11260, 0x2121 }, { 11261, 0x309b }, { 11262, 0x309c },
    { 11263, 0x2e80 }, { 11264, 0x2e84 }, { 11265, 0x2e86 }, { 11266, 0x2e87 }, { 11267, 0x2e88 }, { 11268, 0x2e8a }, { 11269, 0x2e8c }, { 11270, 0x2e8d },
    { 11271, 0x2e95 }, { 11272, 0x2e9c }, { 11273, 0x2e9d }, { 11274, 0x2ea5 }, { 11275, 0x2ea7 }, { 11276, 0x2eaa }, { 11277, 0x2eac }, { 11278, 0x2eae },
    { 11279, 0x2eb6 }, { 11280, 0x2ebc }, { 11281, 0x2ebe }, { 11282, 0x2ec6 }, { 11283, 0x2eca }, { 11284, 0x2ecc }, { 11285, 0x2ecd }, { 11286, 0x2ecf },
    { 11287, 0x2ed6 }, { 11288, 0x2ed7 }, { 11289, 0x2ede }, { 11290, 0x2ee3 }, { 11294, 0x283 }, { 11295, 0x250 }, { 11296, 0x25b }, { 11297, 0x254 },
    { 11298, 0x275 }, { 11299, 0x153 }, { 11300, 0xf8 }, { 11301, 0x14b }, { 11302, 0x28a }, { 11303, 0x26a }, { 18996, 0xffed }, { 18997, 0x20547 },
    { 18998, 0x92db }, { 18999, 0x205df }, { 19000, 0x23fc5 }, { 19001, 0x854c }, { 19002, 0x42b5 }, { 19003, 0x73ef }, { 19004, 0x51b5 }, { 19005, 0x3649 },
    { 19006, 0x24942 }, { 19007, 0x289e4 }, { 19008, 0x9344 }, { 19009, 0x219db }, { 19010, 0x82ee }, { 19011, 0x23cc8 }, { 19012, 0x783c }, { 19013, 0x6744 },
    { 19014, 0x62df }, { 19015, 0x24933 }, { 19016, 0x289aa }, { 19017, 0x202a0 }, { 19018, 0x26bb3 }, { 19019, 0x21305 }, { 19020, 0x4fab }, { 19021, 0x224ed },
    { 19022, 0x5008 }, { 19023, 0x26d29 }, { 19024, 0x27a84 }, { 19025, 0x23600 }, { 19026, 0x24ab1 }, { 19027, 0x22513 }, { 19028, 0x5029 }, { 19029, 0x2037e },
    { 19030, 0x5fa4 }, { 19031, 0x20380 }, { 19032, 0x20347 }, { 19033, 0x6edb }, { 19034, 0x2041f }, { 19035, 0x507d }, { 19036, 0x5101 }, { 19037, 0x347a },
    { 19038, 0x510e }, { 19039, 0x986c }, { 19040, 0x3743 }, { 19041, 0x8416 }, { 19042, 0x249a4 }, { 19043, 0x20487 }, { 19044, 0x5160 }, { 19045, 0x233b4 },
    { 19046, 0x516a }, { 19047, 0x20bff }, { 19048, 0x220fc }, { 19049, 0x202e5 }, { 19050, 0x22530 }, { 19051, 0x2058e }, { 19052, 0x23233 }, { 19053, 0x21983 },
    { 19054, 0x5b82 }, { 19055, 0x877d }, { 19056, 0x205b3 }, { 19057, 0x23c99 }, { 19058, 0x51b2 }, { 19059, 0x51b8 }, { 19060, 0x9d34 }, { 19061, 0x51c9 },
    { 19062, 0x51cf }, { 19063, 0x51d1 }, { 19064, 0x3cdc }, { 19065, 0x51d3 }, { 19066, 0x24aa6 }, { 19067, 0x51b3 }, { 19068, 0x51e2 }, { 19069, 0x5342 },
    { 19070, 0x51ed }, { 19071, 0x83cd }, { 19072, 0x693e }, { 19073, 0x2372d }, { 19074, 0x5f7b }, { 19075, 0x520b }, { 19076, 0x5226 }, { 19077, 0x523c },
    { 19078, 0x52b5 }, { 19079, 0x5257 }, { 19080, 0x5294 }, { 19081, 0x52b9 }, { 19082, 0x52c5 }, { 19083, 0x7c15 }, { 19084, 0x8542 }, { 19085, 0x52e0 },
    { 19086, 0x860d }, { 19087, 0x26b13 }, { 19088, 0x5305 }, { 19089, 0x28ade }, { 19090, 0x5549 }, { 19091, 0x6ed9 }, { 19092, 0x23f80 }, { 19093, 0x20954 },
    { 19094, 0x23fec }, { 19095, 0x5333 }, { 19096, 0x5344 }, { 19097, 0x20be2 }, { 19098, 0x6ccb }, { 19099, 0x21726 }, { 19100, 0x681b }, { 19101, 0x73d5 },
    { 19102, 0x604a }, { 19103, 0x3eaa }, { 19104, 0x38cc }, { 19105, 0x216e8 }, { 19106, 0x71dd }, { 19107, 0x44a2 }, { 19108, 0x536d }, { 19109, 0x5374 },
    { 19110, 0x286ab }, { 19111, 0x537e }, { 19112, 0x537f }, { 19113, 0x21596 }, { 19114, 0x21613 }, { 19115, 0x77e6 }, { 19116, 0x5393 }, { 19117, 0x28a9b },
    { 19118, 0x53a0 }, { 19119, 0x53ab }, { 19120, 0x53ae }, { 19121, 0x73a7 }, { 19122, 0x25772 }, { 19123, 0x3f59 }, { 19124, 0x739c }, { 19125, 0x53c1 },
    { 19126, 0x53c5 }, { 19127, 0x6c49 }, { 19128, 0x4e49 }, { 19129, 0x57fe }, { 19130, 0x53d9 }, { 19131, 0x3aab }, { 19132, 0x20b8f }, { 19133, 0x53e0 },
    { 19134, 0x23feb }, { 19135, 0x22da3 }, { 19136, 0x53f6 }, { 19137, 0x20c77 }, { 19138, 0x5413 }, { 19139, 0x7079 }, { 19140, 0x552b }, { 19141, 0x6657 },
    { 19142, 0x6d5b }, { 19143, 0x546d }, { 19144, 0x26b53 }, { 19145, 0x20d74 }, { 19146, 0x555d }, { 19147, 0x548f }, { 19148, 0x54a4 }, { 19149, 0x47a6 },
    { 19150, 0x2170d }, { 19151, 0x20edd }, { 19152, 0x3db4 }, { 19153, 0x20d4d }, { 19154, 0x289bc }, { 19155, 0x22698 }, { 19156, 0x5547 }, { 19157, 0x4ced },
    { 19158, 0x542f }, { 19159, 0x7417 }, { 19160, 0x5586 }, { 19161, 0x55a9 }, { 19162, 0x5605 }, { 19163, 0x218d7 }, { 19164, 0x2403a }, { 19165, 0x4552 },
    { 19166, 0x24435 }, { 19167, 0x66b3 }, { 19168, 0x210b4 }, { 19169, 0x5637 }, { 19170, 0x66cd }, { 19171, 0x2328a }, { 19172, 0x66a4 }, { 19173, 0x66ad },
    { 19174, 0x564d }, { 19175, 0x564f }, { 19176, 0x78f1 }, { 19177, 0x56f1 }, { 19178, 0x9787 }, { 19179, 0x53fe }, { 19180, 0x5700 }, { 19181, 0x56ef },
    { 19182, 0x56ed }, { 19183, 0x28b66 }, { 19184, 0x3623 }, { 19185, 0x2124f }, { 19186, 0x5746 }, { 19187, 0x241a5 }, { 19188, 0x6c6e }, { 19189, 0x708b },
    { 19190, 0x5742 }, { 19191, 0x36b1 }, { 19192, 0x26c7e }, { 19193, 0x57e6 }, { 19194, 0x21416 }, { 19195, 0x5803 }, { 19196, 0x21454 }, { 19197, 0x24363 },
    { 19198, 0x5826 }, { 19199, 0x24bf5 }, { 19200, 0x585c }, { 19201, 0x58aa }, { 19202, 0x3561 }, { 19203, 0x58e0 }, { 19204, 0x58dc }, { 19205, 0x2123c },
    { 19206, 0x58fb }, { 19207, 0x5bff }, { 19208, 0x5743 }, { 19209, 0x2a150 }, { 19210, 0x24278 }, { 19211, 0x93d3 }, { 19212, 0x35a1 }, { 19213, 0x591f },
    { 19214, 0x68a6 }, { 19215, 0x36c3 }, { 19216, 0x6e59 }, { 19217, 0x2163e }, { 19218, 0x5a24 }, { 19219, 0x5553 }, { 19220, 0x21692 }, { 19221, 0x8505 },
    { 19222, 0x59c9 }, { 19223, 0x20d4e }, { 19224, 0x26c81 }, { 19225, 0x26d2a }, { 19226, 0x217dc }, { 19227, 0x59d9 }, { 19228, 0x217fb }, { 19229, 0x217b2 },
    { 19230, 0x26da6 }, { 19231, 0x6d71 }, { 19232, 0x21828 }, { 19233, 0x216d5 }, { 19234, 0x59f9 }, { 19235, 0x26e45 }, { 19236, 0x5aab }, { 19237, 0x5a63 },
    { 19238, 0x36e6 }, { 19239, 0x249a9 }, { 19240, 0x5a77 }, { 19241, 0x3708 }, { 19242, 0x5a96 }, { 19243, 0x7465 }, { 19244, 0x5ad3 }, { 19245, 0x26fa1 },
    { 19246, 0x22554 }, { 19247, 0x3d85 }, { 19248, 0x21911 }, { 19249, 0x3732 }, { 19250, 0x216b8 }, { 19251, 0x5e83 }, { 19252, 0x52d0 }, { 19253, 0x5b76 },
    { 19254, 0x6588 }, { 19255, 0x5b7c }, { 19256, 0x27a0e }, { 19257, 0x4004 }, { 19258, 0x485d }, { 19259, 0x20204 }, { 19260, 0x5bd5 }, { 19261, 0x6160 },
    { 19262, 0x21a34 }, { 19263, 0x259cc }, { 19264, 0x205a5 }, { 19265, 0x5bf3 }, { 19266, 0x5b9d }, { 19267, 0x4d10 }, { 19268, 0x5c05 }, { 19269, 0x21b44 },
    { 19270, 0x5c13 }, { 19271, 0x73ce }, { 19272, 0x5c14 }, { 19273, 0x21ca5 }, { 19274, 0x26b28 }, { 19275, 0x5c49 }, { 19276, 0x48dd }, { 19277, 0x5c85 },
    { 19278, 0x5ce9 }, { 19279, 0x5cef }, { 19280, 0x5d8b }, { 19281, 0x21df9 }, { 19282, 0x21e37 }, { 19283, 0x5d10 }, { 19284, 0x5d18 }, { 19285, 0x5d46 },
    { 19286, 0x21ea4 }, { 19287, 0x5cba }, { 19288, 0x5dd7 }, { 19289, 0x82fc }, { 19290, 0x382d }, { 19291, 0x24901 }, { 19292, 0x22049 }, { 19293, 0x22173 },
    { 19294, 0x8287 }, { 19295, 0x3836 }, { 19296, 0x3bc2 }, { 19297, 0x5e2e }, { 19298, 0x6a8a }, { 19299, 0x5e75 }, { 19300, 0x5e7a }, { 19301, 0x244bc },
    { 19302, 0x20cd3 }, { 19303, 0x53a6 }, { 19304, 0x4eb7 }, { 19305, 0x5ed0 }, { 19306, 0x53a8 }, { 19307, 0x21771 }, { 19308, 0x5e09 }, { 19309, 0x5ef4 },
    { 19310, 0x28482 }, { 19311, 0x5ef9 }, { 19312, 0x5efb }, { 19313, 0x38a0 }, { 19314, 0x5efc }, { 19315, 0x683e }, { 19316, 0x941b }, { 19317, 0x5f0d },
    { 19318, 0x201c1 }, { 19319, 0x2f894 }, { 19320, 0x3ade }, { 19321, 0x48ae }, { 19322, 0x2133a }, { 19323, 0x5f3a }, { 19324, 0x26888 }, { 19325, 0x223d0 },
    { 19326, 0x5f58 }, { 19327, 0x22471 }, { 19328, 0x5f63 }, { 19329, 0x97bd }, { 19330, 0x26e6e }, { 19331, 0x5f72 }, { 19332, 0x9340 }, { 19333, 0x28a36 },
    { 19334, 0x5fa7 }, { 19335, 0x5db6 }, { 19336, 0x3d5f }, { 19337, 0x25250 }, { 19338, 0x21f6a }, { 19339, 0x270f8 }, { 19340, 0x22668 }, { 19341, 0x91d6 },
    { 19342, 0x2029e }, { 19343, 0x28a29 }, { 19344, 0x6031 }, { 19345, 0x6685 }, { 19346, 0x21877 }, { 19347, 0x3963 }, { 19348, 0x3dc7 }, { 19349, 0x3639 },
    { 19350, 0x5790 }, { 19351, 0x227b4 }, { 19352, 0x7971 }, { 19353, 0x3e40 }, { 19354, 0x609e }, { 19355, 0x60a4 }, { 19356, 0x60b3 }, { 19357, 0x24982 },
    { 19358, 0x2498f }, { 19359, 0x27a53 }, { 19360, 0x74a4 }, { 19361, 0x50e1 }, { 19362, 0x5aa0 }, { 19363, 0x6164 }, { 19364, 0x8424 }, { 19365, 0x6142 },
    { 19366, 0x2f8a6 }, { 19367, 0x26ed2 }, { 19368, 0x6181 }, { 19369, 0x51f4 }, { 19370, 0x20656 }, { 19371, 0x6187 }, { 19372, 0x5baa }, { 19373, 0x23fb7 },
    { 19374, 0x2285f }, { 19375, 0x61d3 }, { 19376, 0x28b9d }, { 19377, 0x2995d }, { 19378, 0x61d0 }, { 19379, 0x3932 }, { 19380, 0x22980 }, { 19381, 0x228c1 },
    { 19382, 0x6023 }, { 19383, 0x615c }, { 19384, 0x651e }, { 19385, 0x638b }, { 19386, 0x20118 }, { 19387, 0x62c5 }, { 19388, 0x21770 }, { 19389, 0x62d5 },
    { 19390, 0x22e0d }, { 19391, 0x636c }, { 19392, 0x249df }, { 19393, 0x3a17 }, { 19394, 0x6438 }, { 19395, 0x63f8 }, { 19396, 0x2138e }, { 19397, 0x217fc },
    { 19398, 0x6490 }, { 19399, 0x6f8a }, { 19400, 0x22e36 }, { 19401, 0x9814 }, { 19402, 0x2408c }, { 19403, 0x2571d }, { 19404, 0x64e1 }, { 19405, 0x64e5 },
    { 19406, 0x947b }, { 19407, 0x3a66 }, { 19408, 0x643a }, { 19409, 0x3a57 }, { 19410, 0x654d }, { 19411, 0x6f16 }, { 19412, 0x24a28 }, { 19413, 0x24a23 },
    { 19414, 0x6585 }, { 19415, 0x656d }, { 19416, 0x655f }, { 19417, 0x2307e }, { 19418, 0x65b5 }, { 19419, 0x24940 }, { 19420, 0x4b37 }, { 19421, 0x65d1 },
    { 19422, 0x40d8 }, { 19423, 0x21829 }, { 19424, 0x65e0 }, { 19425, 0x65e3 }, { 19426, 0x5fdf }, { 19427, 0x23400 }, { 19428, 0x6618 }, { 19429, 0x231f7 },
    { 19430, 0x231f8 }, { 19431, 0x6644 }, { 19432, 0x231a4 }, { 19433, 0x231a5 }, { 19434, 0x664b }, { 19435, 0x20e75 }, { 19436, 0x6667 }, { 19437, 0x251e6 },
    { 19438, 0x6673 }, { 19439, 0x6674 }, { 19440, 0x21e3d }, { 19441, 0x23231 }, { 19442, 0x285f4 }, { 19443, 0x231c8 }, { 19444, 0x25313 }, { 19445, 0x77c5 },
    { 19446, 0x228f7 }, { 19447, 0x99a4 }, { 19448, 0x6702 }, { 19449, 0x2439c }, { 19450, 0x24a21 }, { 19451, 0x3b2b }, { 19452, 0x69fa }, { 19453, 0x237c2 },
    { 19454, 0x675e }, { 19455, 0x6767 }, { 19456, 0x6762 }, { 19457, 0x241cd }, { 19458, 0x290ed }, { 19459, 0x67d7 }, { 19460, 0x44e9 }, { 19461, 0x6822 },
    { 19462, 0x6e50 }, { 19463, 0x923c }, { 19464, 0x6801 }, { 19465, 0x233e6 }, { 19466, 0x26da0 }, { 19467, 0x685d }, { 19468, 0x2346f }, { 19469, 0x69e1 },
    { 19470, 0x6a0b }, { 19471, 0x28adf }, { 19472, 0x6973 }, { 19473, 0x68c3 }, { 19474, 0x235cd }, { 19475, 0x6901 }, { 19476, 0x6900 }, { 19477, 0x3d32 },
    { 19478, 0x3a01 }, { 19479, 0x2363c }, { 19480, 0x3b80 }, { 19481, 0x67ac }, { 19482, 0x6961 }, { 19483, 0x28a4a }, { 19484, 0x42fc }, { 19485, 0x6936 },
    { 19486, 0x6998 }, { 19487, 0x3ba1 }, { 19488, 0x203c9 }, { 19489, 0x8363 }, { 19490, 0x5090 }, { 19491, 0x69f9 }, { 19492, 0x23659 }, { 19493, 0x2212a },
    { 19494, 0x6a45 }, { 19495, 0x23703 }, { 19496, 0x6a9d }, { 19497, 0x3bf3 }, { 19498, 0x67b1 }, { 19499, 0x6ac8 }, { 19500, 0x2919c }, { 19501, 0x3c0d },
    { 19502, 0x6b1d }, { 19503, 0x20923 }, { 19504, 0x60de }, { 19505, 0x6b35 }, { 19506, 0x6b74 }, { 19507, 0x227cd }, { 19508, 0x6eb5 }, { 19509, 0x23adb },
    { 19510, 0x203b5 }, { 19511, 0x21958 }, { 19512, 0x3740 }, { 19513, 0x5421 }, { 19514, 0x23b5a }, { 19515, 0x6be1 }, { 19516, 0x23efc }, { 19517, 0x6bdc },
    { 19518, 0x6c37 }, { 19519, 0x2248b }, { 19520, 0x248f1 }, { 19521, 0x26b51 }, { 19522, 0x6c5a }, { 19523, 0x8226 }, { 19524, 0x6c79 }, { 19525, 0x23dbc },
    { 19526, 0x44c5 }, { 19527, 0x23dbd }, { 19528, 0x241a4 }, { 19529, 0x2490c }, { 19530, 0x24900 }, { 19531, 0x23cc9 }, { 19532, 0x36e5 }, { 19533, 0x3ceb },
    { 19534, 0x20d32 }, { 19535, 0x9b83 }, { 19536, 0x231f9 }, { 19537, 0x22491 }, { 19538, 0x7f8f }, { 19539, 0x6837 }, { 19540, 0x26d25 }, { 19541, 0x26da1 },
    { 19542, 0x26deb }, { 19543, 0x6d96 }, { 19544, 0x6d5c }, { 19545, 0x6e7c }, { 19546, 0x6f04 }, { 19547, 0x2497f }, { 19548, 0x24085 }, { 19549, 0x26e72 },
    { 19550, 0x8533 }, { 19551, 0x26f74 }, { 19552, 0x51c7 }, { 19553, 0x6c9c }, { 19554, 0x6e1d }, { 19555, 0x842e }, { 19556, 0x28b21 }, { 19557, 0x6e2f },
    { 19558, 0x23e2f }, { 19559, 0x7453 }, { 19560, 0x23f82 }, { 19561, 0x79cc }, { 19562, 0x6e4f }, { 19563, 0x5a91 }, { 19564, 0x2304b }, { 19565, 0x6ff8 },
    { 19566, 0x370d }, { 19567, 0x6f9d }, { 19568, 0x23e30 }, { 19569, 0x6efa }, { 19570, 0x21497 }, { 19571, 0x2403d }, { 19572, 0x4555 }, { 19573, 0x93f0 },
    { 19574, 0x6f44 }, { 19575, 0x6f5c }, { 19576, 0x3d4e }, { 19577, 0x6f74 }, { 19578, 0x29170 }, { 19579, 0x3d3b }, { 19580, 0x6f9f }, { 19581, 0x24144 },
    { 19582, 0x6fd3 }, { 19583, 0x24091 }, { 19584, 0x24155 }, { 19585, 0x24039 }, { 19586, 0x23ff0 }, { 19587, 0x23fb4 }, { 19588, 0x2413f }, { 19589, 0x51df },
    { 19590, 0x24156 }, { 19591, 0x24157 }, { 19592, 0x24140 }, { 19593, 0x261dd }, { 19594, 0x704b }, { 19595, 0x707e }, { 19596, 0x70a7 }, { 19597, 0x7081 },
    { 19598, 0x70cc }, { 19599, 0x70d5 }, { 19600, 0x70d6 }, { 19601, 0x70df }, { 19602, 0x4104 }, { 19603, 0x3de8 }, { 19604, 0x71b4 }, { 19605, 0x7196 },
    { 19606, 0x24277 }, { 19607, 0x712b }, { 19608, 0x7145 }, { 19609, 0x5a88 }, { 19610, 0x714a }, { 19611, 0x716e }, { 19612, 0x5c9c }, { 19613, 0x24365 },
    { 19614, 0x714f }, { 19615, 0x9362 }, { 19616, 0x242c1 }, { 19617, 0x712c }, { 19618, 0x2445a }, { 19619, 0x24a27 }, { 19620, 0x24a22 }, { 19621, 0x71ba },
    { 19622, 0x28be8 }, { 19623, 0x70bd }, { 19624, 0x720e }, { 19625, 0x9442 }, { 19626, 0x7215 }, { 19627, 0x5911 }, { 19628, 0x9443 }, { 19629, 0x7224 },
    { 19630, 0x9341 }, { 19631, 0x25605 }, { 19632, 0x722e }, { 19633, 0x7240 }, { 19634, 0x24974 }, { 19635, 0x68bd }, { 19636, 0x7255 }, { 19637, 0x7257 },
    { 19638, 0x3e55 }, { 19639, 0x23044 }, { 19640, 0x680d }, { 19641, 0x6f3d }, { 19642, 0x7282 }, { 19643, 0x732a }, { 19644, 0x732b }, { 19645, 0x24823 },
    { 19646, 0x2882b }, { 19647, 0x48ed }, { 19648, 0x28804 }, { 19649, 0x7328 }, { 19650, 0x732e }, { 19651, 0x73cf }, { 19652, 0x73aa }, { 19653, 0x20c3a },
    { 19654, 0x26a2e }, { 19655, 0x73c9 }, { 19656, 0x7449 }, { 19657, 0x241e2 }, { 19658, 0x216e7 }, { 19659, 0x24a24 }, { 19660, 0x6623 }, { 19661, 0x36c5 },
    { 19662, 0x249b7 }, { 19663, 0x2498d }, { 19664, 0x249fb }, { 19665, 0x73f7 }, { 19666, 0x7415 }, { 19667, 0x6903 }, { 19668, 0x24a26 }, { 19669, 0x7439 },
    { 19670, 0x205c3 }, { 19671, 0x3ed7 }, { 19672, 0x745c }, { 19673, 0x228ad }, { 19674, 0x7460 }, { 19675, 0x28eb2 }, { 19676, 0x7447 }, { 19677, 0x73e4 },
    { 19678, 0x7476 }, { 19679, 0x83b9 }, { 19680, 0x746c }, { 19681, 0x3730 }, { 19682, 0x7474 }, { 19683, 0x93f1 }, { 19684, 0x6a2c }, { 19685, 0x7482 },
    { 19686, 0x4953 }, { 19687, 0x24a8c }, { 19688, 0x2415f }, { 19689, 0x24a79 }, { 19690, 0x28b8f }, { 19691, 0x5b46 }, { 19692, 0x28c03 }, { 19693, 0x2189e },
    { 19694, 0x74c8 }, { 19695, 0x21988 }, { 19696, 0x750e }, { 19697, 0x74e9 }, { 19698, 0x751e }, { 19699, 0x28ed9 }, { 19700, 0x21a4b }, { 19701, 0x5bd7 },
    { 19702, 0x28eac }, { 19703, 0x9385 }, { 19704, 0x754d }, { 19705, 0x754a }, { 19706, 0x7567 }, { 19707, 0x756e }, { 19708, 0x24f82 }, { 19709, 0x3f04 },
    { 19710, 0x24d13 }, { 19711, 0x758e }, { 19712, 0x745d }, { 19713, 0x759e }, { 19714, 0x75b4 }, { 19715, 0x7602 }, { 19716, 0x762c }, { 19717, 0x7651 },
    { 19718, 0x764f }, { 19719, 0x766f }, { 19720, 0x7676 }, { 19721, 0x263f5 }, { 19722, 0x7690 }, { 19723, 0x81ef }, { 19724, 0x37f8 }, { 19725, 0x26911 },
    { 19726, 0x2690e }, { 19727, 0x76a1 }, { 19728, 0x76a5 }, { 19729, 0x76b7 }, { 19730, 0x76cc }, { 19731, 0x26f9f }, { 19732, 0x8462 }, { 19733, 0x2509d },
    { 19734, 0x2517d }, { 19735, 0x21e1c }, { 19736, 0x771e }, { 19737, 0x7726 }, { 19738, 0x7740 }, { 19739, 0x64af }, { 19740, 0x25220 }, { 19741, 0x7758 },
    { 19742, 0x232ac }, { 19743, 0x77af }, { 19744, 0x28964 }, { 19745, 0x28968 }, { 19746, 0x216c1 }, { 19747, 0x77f4 }, { 19748, 0x7809 }, { 19749, 0x21376 },
    { 19750, 0x24a12 }, { 19751, 0x68ca }, { 19752, 0x78af }, { 19753, 0x78c7 }, { 19754, 0x78d3 }, { 19755, 0x96a5 }, { 19756, 0x792e }, { 19757, 0x255e0 },
    { 19758, 0x78d7 }, { 19759, 0x7934 }, { 19760, 0x78b1 }, { 19761, 0x2760c }, { 19762, 0x8fb8 }, { 19763, 0x8884 }, { 19764, 0x28b2b }, { 19765, 0x26083 },
    { 19766, 0x2261c }, { 19767, 0x7986 }, { 19768, 0x8900 }, { 19769, 0x6902 }, { 19770, 0x7980 }, { 19771, 0x25857 }, { 19772, 0x799d }, { 19773, 0x27b39 },
    { 19774, 0x793c }, { 19775, 0x79a9 }, { 19776, 0x6e2a }, { 19777, 0x27126 }, { 19778, 0x3ea8 }, { 19779, 0x79c6 }, { 19780, 0x2910d }, { 19781, 0x79d4 }
} };

const std::array<std::pair<uint16_t, UChar32>, 18590>& big5()
{
    // Allocate this at runtime because building it at compile time would make the binary much larger and this is often not used.
    static std::array<std::pair<uint16_t, UChar32>, 18590>* array;
    static std::once_flag flag;
    std::call_once(flag, [] {
        array = new std::array<std::pair<uint16_t, UChar32>, 18590>();
        size_t arrayIndex = 0;
        
        UErrorCode error = U_ZERO_ERROR;
        auto icuConverter = ICUConverterPtr { ucnv_open("Big-5", &error) };
        ASSERT(!error);

        uint8_t icuInput[2];
        UChar icuOutput;

        // These are the ranges from https://encoding.spec.whatwg.org/index-big5.txt that have valid pointers.
        constexpr std::array<std::pair<uint16_t, uint16_t>, 60> big5PointerRanges { {
            { 942, 980 }, { 981, 1068 }, { 1099, 1133 }, { 1134, 1135 }, { 1136, 1164 }, { 1165, 1166 }, { 1167, 1172 }, { 1256, 1258 },
            { 1259, 1260 }, { 1262, 1266 }, { 1268, 1325 }, { 1329, 1333 }, { 1334, 1337 }, { 1339, 1350 }, { 1351, 1354 }, { 1355, 1415 },
            { 1416, 1448 }, { 1449, 1466 }, { 1467, 1486 }, { 1487, 1492 }, { 1493, 1501 }, { 1502, 1515 }, { 1516, 1520 }, { 1521, 1536 },
            { 1538, 1560 }, { 1561, 1590 }, { 1591, 1693 }, { 1694, 1726 }, { 1727, 1795 }, { 1796, 1827 }, { 1830, 1834 }, { 1835, 1858 },
            { 1859, 1885 }, { 1886, 4115 }, { 4116, 4627 }, { 4628, 4651 }, { 4652, 4699 }, { 4700, 4724 }, { 4725, 4785 }, { 4786, 4789 },
            { 4790, 4804 }, { 4805, 4812 }, { 4813, 4830 }, { 4831, 4842 }, { 4843, 4846 }, { 4847, 4851 }, { 4852, 4887 }, { 4888, 4890 },
            { 4891, 4893 }, { 4894, 4901 }, { 4902, 4917 }, { 4918, 4934 }, { 4935, 4942 }, { 4943, 4944 }, { 4945, 4980 }, { 4981, 4994 },
            { 4995, 5466 }, { 5495, 11214 }, { 11254, 11291 }, { 11294, 19782 }
        } };

        for (auto& range : big5PointerRanges) {
            for (uint16_t pointer = range.first; pointer < range.second; pointer++) {
                uint8_t lead = pointer / 157 + 0x81;
                uint8_t trail = pointer % 157;
                uint8_t offset = trail < 0x3F ? 0x40 : 0x62;
                icuInput[0] = lead;
                icuInput[1] = trail + offset;
                UChar* output = &icuOutput;
                const char* input = reinterpret_cast<const char*>(icuInput);
                ucnv_toUnicode(icuConverter.get(), &output, output + 1, &input, input + sizeof(icuInput), nullptr, true, &error);
                ASSERT(!error);
                (*array)[arrayIndex++] = { pointer, icuOutput };
            }
        }
        
        for (auto& pair : big5Extras) {
            auto range = std::equal_range(array->begin(), array->end(), pair, CompareFirst { });
            ASSERT(range.first + 1 == range.second);
            range.first->second = pair.second;
        }
        
        RELEASE_ASSERT_WITH_SECURITY_IMPLICATION(arrayIndex == 18590);
        ASSERT(*array == big5Reference);
    });
    return *array;
}

#if ASSERT_ENABLED
// From https://encoding.spec.whatwg.org/index-euc-kr.txt
const std::array<std::pair<uint16_t, UChar>, 17048> eucKRDecodingIndexReference { {
    { 0, 0xAC02 }, { 1, 0xAC03 }, { 2, 0xAC05 }, { 3, 0xAC06 }, { 4, 0xAC0B }, { 5, 0xAC0C }, { 6, 0xAC0D }, { 7, 0xAC0E },
    { 8, 0xAC0F }, { 9, 0xAC18 }, { 10, 0xAC1E }, { 11, 0xAC1F }, { 12, 0xAC21 }, { 13, 0xAC22 }, { 14, 0xAC23 }, { 15, 0xAC25 },
    { 16, 0xAC26 }, { 17, 0xAC27 }, { 18, 0xAC28 }, { 19, 0xAC29 }, { 20, 0xAC2A }, { 21, 0xAC2B }, { 22, 0xAC2E }, { 23, 0xAC32 },
    { 24, 0xAC33 }, { 25, 0xAC34 }, { 32, 0xAC35 }, { 33, 0xAC36 }, { 34, 0xAC37 }, { 35, 0xAC3A }, { 36, 0xAC3B }, { 37, 0xAC3D },
    { 38, 0xAC3E }, { 39, 0xAC3F }, { 40, 0xAC41 }, { 41, 0xAC42 }, { 42, 0xAC43 }, { 43, 0xAC44 }, { 44, 0xAC45 }, { 45, 0xAC46 },
    { 46, 0xAC47 }, { 47, 0xAC48 }, { 48, 0xAC49 }, { 49, 0xAC4A }, { 50, 0xAC4C }, { 51, 0xAC4E }, { 52, 0xAC4F }, { 53, 0xAC50 },
    { 54, 0xAC51 }, { 55, 0xAC52 }, { 56, 0xAC53 }, { 57, 0xAC55 }, { 64, 0xAC56 }, { 65, 0xAC57 }, { 66, 0xAC59 }, { 67, 0xAC5A },
    { 68, 0xAC5B }, { 69, 0xAC5D }, { 70, 0xAC5E }, { 71, 0xAC5F }, { 72, 0xAC60 }, { 73, 0xAC61 }, { 74, 0xAC62 }, { 75, 0xAC63 },
    { 76, 0xAC64 }, { 77, 0xAC65 }, { 78, 0xAC66 }, { 79, 0xAC67 }, { 80, 0xAC68 }, { 81, 0xAC69 }, { 82, 0xAC6A }, { 83, 0xAC6B },
    { 84, 0xAC6C }, { 85, 0xAC6D }, { 86, 0xAC6E }, { 87, 0xAC6F }, { 88, 0xAC72 }, { 89, 0xAC73 }, { 90, 0xAC75 }, { 91, 0xAC76 },
    { 92, 0xAC79 }, { 93, 0xAC7B }, { 94, 0xAC7C }, { 95, 0xAC7D }, { 96, 0xAC7E }, { 97, 0xAC7F }, { 98, 0xAC82 }, { 99, 0xAC87 },
    { 100, 0xAC88 }, { 101, 0xAC8D }, { 102, 0xAC8E }, { 103, 0xAC8F }, { 104, 0xAC91 }, { 105, 0xAC92 }, { 106, 0xAC93 }, { 107, 0xAC95 },
    { 108, 0xAC96 }, { 109, 0xAC97 }, { 110, 0xAC98 }, { 111, 0xAC99 }, { 112, 0xAC9A }, { 113, 0xAC9B }, { 114, 0xAC9E }, { 115, 0xACA2 },
    { 116, 0xACA3 }, { 117, 0xACA4 }, { 118, 0xACA5 }, { 119, 0xACA6 }, { 120, 0xACA7 }, { 121, 0xACAB }, { 122, 0xACAD }, { 123, 0xACAE },
    { 124, 0xACB1 }, { 125, 0xACB2 }, { 126, 0xACB3 }, { 127, 0xACB4 }, { 128, 0xACB5 }, { 129, 0xACB6 }, { 130, 0xACB7 }, { 131, 0xACBA },
    { 132, 0xACBE }, { 133, 0xACBF }, { 134, 0xACC0 }, { 135, 0xACC2 }, { 136, 0xACC3 }, { 137, 0xACC5 }, { 138, 0xACC6 }, { 139, 0xACC7 },
    { 140, 0xACC9 }, { 141, 0xACCA }, { 142, 0xACCB }, { 143, 0xACCD }, { 144, 0xACCE }, { 145, 0xACCF }, { 146, 0xACD0 }, { 147, 0xACD1 },
    { 148, 0xACD2 }, { 149, 0xACD3 }, { 150, 0xACD4 }, { 151, 0xACD6 }, { 152, 0xACD8 }, { 153, 0xACD9 }, { 154, 0xACDA }, { 155, 0xACDB },
    { 156, 0xACDC }, { 157, 0xACDD }, { 158, 0xACDE }, { 159, 0xACDF }, { 160, 0xACE2 }, { 161, 0xACE3 }, { 162, 0xACE5 }, { 163, 0xACE6 },
    { 164, 0xACE9 }, { 165, 0xACEB }, { 166, 0xACED }, { 167, 0xACEE }, { 168, 0xACF2 }, { 169, 0xACF4 }, { 170, 0xACF7 }, { 171, 0xACF8 },
    { 172, 0xACF9 }, { 173, 0xACFA }, { 174, 0xACFB }, { 175, 0xACFE }, { 176, 0xACFF }, { 177, 0xAD01 }, { 178, 0xAD02 }, { 179, 0xAD03 },
    { 180, 0xAD05 }, { 181, 0xAD07 }, { 182, 0xAD08 }, { 183, 0xAD09 }, { 184, 0xAD0A }, { 185, 0xAD0B }, { 186, 0xAD0E }, { 187, 0xAD10 },
    { 188, 0xAD12 }, { 189, 0xAD13 }, { 190, 0xAD14 }, { 191, 0xAD15 }, { 192, 0xAD16 }, { 193, 0xAD17 }, { 194, 0xAD19 }, { 195, 0xAD1A },
    { 196, 0xAD1B }, { 197, 0xAD1D }, { 198, 0xAD1E }, { 199, 0xAD1F }, { 200, 0xAD21 }, { 201, 0xAD22 }, { 202, 0xAD23 }, { 203, 0xAD24 },
    { 204, 0xAD25 }, { 205, 0xAD26 }, { 206, 0xAD27 }, { 207, 0xAD28 }, { 208, 0xAD2A }, { 209, 0xAD2B }, { 210, 0xAD2E }, { 211, 0xAD2F },
    { 212, 0xAD30 }, { 213, 0xAD31 }, { 214, 0xAD32 }, { 215, 0xAD33 }, { 222, 0xAD36 }, { 223, 0xAD37 }, { 224, 0xAD39 }, { 225, 0xAD3A },
    { 226, 0xAD3B }, { 227, 0xAD3D }, { 228, 0xAD3E }, { 229, 0xAD3F }, { 230, 0xAD40 }, { 231, 0xAD41 }, { 232, 0xAD42 }, { 233, 0xAD43 },
    { 234, 0xAD46 }, { 235, 0xAD48 }, { 236, 0xAD4A }, { 237, 0xAD4B }, { 238, 0xAD4C }, { 239, 0xAD4D }, { 240, 0xAD4E }, { 241, 0xAD4F },
    { 242, 0xAD51 }, { 243, 0xAD52 }, { 244, 0xAD53 }, { 245, 0xAD55 }, { 246, 0xAD56 }, { 247, 0xAD57 }, { 254, 0xAD59 }, { 255, 0xAD5A },
    { 256, 0xAD5B }, { 257, 0xAD5C }, { 258, 0xAD5D }, { 259, 0xAD5E }, { 260, 0xAD5F }, { 261, 0xAD60 }, { 262, 0xAD62 }, { 263, 0xAD64 },
    { 264, 0xAD65 }, { 265, 0xAD66 }, { 266, 0xAD67 }, { 267, 0xAD68 }, { 268, 0xAD69 }, { 269, 0xAD6A }, { 270, 0xAD6B }, { 271, 0xAD6E },
    { 272, 0xAD6F }, { 273, 0xAD71 }, { 274, 0xAD72 }, { 275, 0xAD77 }, { 276, 0xAD78 }, { 277, 0xAD79 }, { 278, 0xAD7A }, { 279, 0xAD7E },
    { 280, 0xAD80 }, { 281, 0xAD83 }, { 282, 0xAD84 }, { 283, 0xAD85 }, { 284, 0xAD86 }, { 285, 0xAD87 }, { 286, 0xAD8A }, { 287, 0xAD8B },
    { 288, 0xAD8D }, { 289, 0xAD8E }, { 290, 0xAD8F }, { 291, 0xAD91 }, { 292, 0xAD92 }, { 293, 0xAD93 }, { 294, 0xAD94 }, { 295, 0xAD95 },
    { 296, 0xAD96 }, { 297, 0xAD97 }, { 298, 0xAD98 }, { 299, 0xAD99 }, { 300, 0xAD9A }, { 301, 0xAD9B }, { 302, 0xAD9E }, { 303, 0xAD9F },
    { 304, 0xADA0 }, { 305, 0xADA1 }, { 306, 0xADA2 }, { 307, 0xADA3 }, { 308, 0xADA5 }, { 309, 0xADA6 }, { 310, 0xADA7 }, { 311, 0xADA8 },
    { 312, 0xADA9 }, { 313, 0xADAA }, { 314, 0xADAB }, { 315, 0xADAC }, { 316, 0xADAD }, { 317, 0xADAE }, { 318, 0xADAF }, { 319, 0xADB0 },
    { 320, 0xADB1 }, { 321, 0xADB2 }, { 322, 0xADB3 }, { 323, 0xADB4 }, { 324, 0xADB5 }, { 325, 0xADB6 }, { 326, 0xADB8 }, { 327, 0xADB9 },
    { 328, 0xADBA }, { 329, 0xADBB }, { 330, 0xADBC }, { 331, 0xADBD }, { 332, 0xADBE }, { 333, 0xADBF }, { 334, 0xADC2 }, { 335, 0xADC3 },
    { 336, 0xADC5 }, { 337, 0xADC6 }, { 338, 0xADC7 }, { 339, 0xADC9 }, { 340, 0xADCA }, { 341, 0xADCB }, { 342, 0xADCC }, { 343, 0xADCD },
    { 344, 0xADCE }, { 345, 0xADCF }, { 346, 0xADD2 }, { 347, 0xADD4 }, { 348, 0xADD5 }, { 349, 0xADD6 }, { 350, 0xADD7 }, { 351, 0xADD8 },
    { 352, 0xADD9 }, { 353, 0xADDA }, { 354, 0xADDB }, { 355, 0xADDD }, { 356, 0xADDE }, { 357, 0xADDF }, { 358, 0xADE1 }, { 359, 0xADE2 },
    { 360, 0xADE3 }, { 361, 0xADE5 }, { 362, 0xADE6 }, { 363, 0xADE7 }, { 364, 0xADE8 }, { 365, 0xADE9 }, { 366, 0xADEA }, { 367, 0xADEB },
    { 368, 0xADEC }, { 369, 0xADED }, { 370, 0xADEE }, { 371, 0xADEF }, { 372, 0xADF0 }, { 373, 0xADF1 }, { 374, 0xADF2 }, { 375, 0xADF3 },
    { 376, 0xADF4 }, { 377, 0xADF5 }, { 378, 0xADF6 }, { 379, 0xADF7 }, { 380, 0xADFA }, { 381, 0xADFB }, { 382, 0xADFD }, { 383, 0xADFE },
    { 384, 0xAE02 }, { 385, 0xAE03 }, { 386, 0xAE04 }, { 387, 0xAE05 }, { 388, 0xAE06 }, { 389, 0xAE07 }, { 390, 0xAE0A }, { 391, 0xAE0C },
    { 392, 0xAE0E }, { 393, 0xAE0F }, { 394, 0xAE10 }, { 395, 0xAE11 }, { 396, 0xAE12 }, { 397, 0xAE13 }, { 398, 0xAE15 }, { 399, 0xAE16 },
    { 400, 0xAE17 }, { 401, 0xAE18 }, { 402, 0xAE19 }, { 403, 0xAE1A }, { 404, 0xAE1B }, { 405, 0xAE1C }, { 412, 0xAE1D }, { 413, 0xAE1E },
    { 414, 0xAE1F }, { 415, 0xAE20 }, { 416, 0xAE21 }, { 417, 0xAE22 }, { 418, 0xAE23 }, { 419, 0xAE24 }, { 420, 0xAE25 }, { 421, 0xAE26 },
    { 422, 0xAE27 }, { 423, 0xAE28 }, { 424, 0xAE29 }, { 425, 0xAE2A }, { 426, 0xAE2B }, { 427, 0xAE2C }, { 428, 0xAE2D }, { 429, 0xAE2E },
    { 430, 0xAE2F }, { 431, 0xAE32 }, { 432, 0xAE33 }, { 433, 0xAE35 }, { 434, 0xAE36 }, { 435, 0xAE39 }, { 436, 0xAE3B }, { 437, 0xAE3C },
    { 444, 0xAE3D }, { 445, 0xAE3E }, { 446, 0xAE3F }, { 447, 0xAE42 }, { 448, 0xAE44 }, { 449, 0xAE47 }, { 450, 0xAE48 }, { 451, 0xAE49 },
    { 452, 0xAE4B }, { 453, 0xAE4F }, { 454, 0xAE51 }, { 455, 0xAE52 }, { 456, 0xAE53 }, { 457, 0xAE55 }, { 458, 0xAE57 }, { 459, 0xAE58 },
    { 460, 0xAE59 }, { 461, 0xAE5A }, { 462, 0xAE5B }, { 463, 0xAE5E }, { 464, 0xAE62 }, { 465, 0xAE63 }, { 466, 0xAE64 }, { 467, 0xAE66 },
    { 468, 0xAE67 }, { 469, 0xAE6A }, { 470, 0xAE6B }, { 471, 0xAE6D }, { 472, 0xAE6E }, { 473, 0xAE6F }, { 474, 0xAE71 }, { 475, 0xAE72 },
    { 476, 0xAE73 }, { 477, 0xAE74 }, { 478, 0xAE75 }, { 479, 0xAE76 }, { 480, 0xAE77 }, { 481, 0xAE7A }, { 482, 0xAE7E }, { 483, 0xAE7F },
    { 484, 0xAE80 }, { 485, 0xAE81 }, { 486, 0xAE82 }, { 487, 0xAE83 }, { 488, 0xAE86 }, { 489, 0xAE87 }, { 490, 0xAE88 }, { 491, 0xAE89 },
    { 492, 0xAE8A }, { 493, 0xAE8B }, { 494, 0xAE8D }, { 495, 0xAE8E }, { 496, 0xAE8F }, { 497, 0xAE90 }, { 498, 0xAE91 }, { 499, 0xAE92 },
    { 500, 0xAE93 }, { 501, 0xAE94 }, { 502, 0xAE95 }, { 503, 0xAE96 }, { 504, 0xAE97 }, { 505, 0xAE98 }, { 506, 0xAE99 }, { 507, 0xAE9A },
    { 508, 0xAE9B }, { 509, 0xAE9C }, { 510, 0xAE9D }, { 511, 0xAE9E }, { 512, 0xAE9F }, { 513, 0xAEA0 }, { 514, 0xAEA1 }, { 515, 0xAEA2 },
    { 516, 0xAEA3 }, { 517, 0xAEA4 }, { 518, 0xAEA5 }, { 519, 0xAEA6 }, { 520, 0xAEA7 }, { 521, 0xAEA8 }, { 522, 0xAEA9 }, { 523, 0xAEAA },
    { 524, 0xAEAB }, { 525, 0xAEAC }, { 526, 0xAEAD }, { 527, 0xAEAE }, { 528, 0xAEAF }, { 529, 0xAEB0 }, { 530, 0xAEB1 }, { 531, 0xAEB2 },
    { 532, 0xAEB3 }, { 533, 0xAEB4 }, { 534, 0xAEB5 }, { 535, 0xAEB6 }, { 536, 0xAEB7 }, { 537, 0xAEB8 }, { 538, 0xAEB9 }, { 539, 0xAEBA },
    { 540, 0xAEBB }, { 541, 0xAEBF }, { 542, 0xAEC1 }, { 543, 0xAEC2 }, { 544, 0xAEC3 }, { 545, 0xAEC5 }, { 546, 0xAEC6 }, { 547, 0xAEC7 },
    { 548, 0xAEC8 }, { 549, 0xAEC9 }, { 550, 0xAECA }, { 551, 0xAECB }, { 552, 0xAECE }, { 553, 0xAED2 }, { 554, 0xAED3 }, { 555, 0xAED4 },
    { 556, 0xAED5 }, { 557, 0xAED6 }, { 558, 0xAED7 }, { 559, 0xAEDA }, { 560, 0xAEDB }, { 561, 0xAEDD }, { 562, 0xAEDE }, { 563, 0xAEDF },
    { 564, 0xAEE0 }, { 565, 0xAEE1 }, { 566, 0xAEE2 }, { 567, 0xAEE3 }, { 568, 0xAEE4 }, { 569, 0xAEE5 }, { 570, 0xAEE6 }, { 571, 0xAEE7 },
    { 572, 0xAEE9 }, { 573, 0xAEEA }, { 574, 0xAEEC }, { 575, 0xAEEE }, { 576, 0xAEEF }, { 577, 0xAEF0 }, { 578, 0xAEF1 }, { 579, 0xAEF2 },
    { 580, 0xAEF3 }, { 581, 0xAEF5 }, { 582, 0xAEF6 }, { 583, 0xAEF7 }, { 584, 0xAEF9 }, { 585, 0xAEFA }, { 586, 0xAEFB }, { 587, 0xAEFD },
    { 588, 0xAEFE }, { 589, 0xAEFF }, { 590, 0xAF00 }, { 591, 0xAF01 }, { 592, 0xAF02 }, { 593, 0xAF03 }, { 594, 0xAF04 }, { 595, 0xAF05 },
    { 602, 0xAF06 }, { 603, 0xAF09 }, { 604, 0xAF0A }, { 605, 0xAF0B }, { 606, 0xAF0C }, { 607, 0xAF0E }, { 608, 0xAF0F }, { 609, 0xAF11 },
    { 610, 0xAF12 }, { 611, 0xAF13 }, { 612, 0xAF14 }, { 613, 0xAF15 }, { 614, 0xAF16 }, { 615, 0xAF17 }, { 616, 0xAF18 }, { 617, 0xAF19 },
    { 618, 0xAF1A }, { 619, 0xAF1B }, { 620, 0xAF1C }, { 621, 0xAF1D }, { 622, 0xAF1E }, { 623, 0xAF1F }, { 624, 0xAF20 }, { 625, 0xAF21 },
    { 626, 0xAF22 }, { 627, 0xAF23 }, { 634, 0xAF24 }, { 635, 0xAF25 }, { 636, 0xAF26 }, { 637, 0xAF27 }, { 638, 0xAF28 }, { 639, 0xAF29 },
    { 640, 0xAF2A }, { 641, 0xAF2B }, { 642, 0xAF2E }, { 643, 0xAF2F }, { 644, 0xAF31 }, { 645, 0xAF33 }, { 646, 0xAF35 }, { 647, 0xAF36 },
    { 648, 0xAF37 }, { 649, 0xAF38 }, { 650, 0xAF39 }, { 651, 0xAF3A }, { 652, 0xAF3B }, { 653, 0xAF3E }, { 654, 0xAF40 }, { 655, 0xAF44 },
    { 656, 0xAF45 }, { 657, 0xAF46 }, { 658, 0xAF47 }, { 659, 0xAF4A }, { 660, 0xAF4B }, { 661, 0xAF4C }, { 662, 0xAF4D }, { 663, 0xAF4E },
    { 664, 0xAF4F }, { 665, 0xAF51 }, { 666, 0xAF52 }, { 667, 0xAF53 }, { 668, 0xAF54 }, { 669, 0xAF55 }, { 670, 0xAF56 }, { 671, 0xAF57 },
    { 672, 0xAF58 }, { 673, 0xAF59 }, { 674, 0xAF5A }, { 675, 0xAF5B }, { 676, 0xAF5E }, { 677, 0xAF5F }, { 678, 0xAF60 }, { 679, 0xAF61 },
    { 680, 0xAF62 }, { 681, 0xAF63 }, { 682, 0xAF66 }, { 683, 0xAF67 }, { 684, 0xAF68 }, { 685, 0xAF69 }, { 686, 0xAF6A }, { 687, 0xAF6B },
    { 688, 0xAF6C }, { 689, 0xAF6D }, { 690, 0xAF6E }, { 691, 0xAF6F }, { 692, 0xAF70 }, { 693, 0xAF71 }, { 694, 0xAF72 }, { 695, 0xAF73 },
    { 696, 0xAF74 }, { 697, 0xAF75 }, { 698, 0xAF76 }, { 699, 0xAF77 }, { 700, 0xAF78 }, { 701, 0xAF7A }, { 702, 0xAF7B }, { 703, 0xAF7C },
    { 704, 0xAF7D }, { 705, 0xAF7E }, { 706, 0xAF7F }, { 707, 0xAF81 }, { 708, 0xAF82 }, { 709, 0xAF83 }, { 710, 0xAF85 }, { 711, 0xAF86 },
    { 712, 0xAF87 }, { 713, 0xAF89 }, { 714, 0xAF8A }, { 715, 0xAF8B }, { 716, 0xAF8C }, { 717, 0xAF8D }, { 718, 0xAF8E }, { 719, 0xAF8F },
    { 720, 0xAF92 }, { 721, 0xAF93 }, { 722, 0xAF94 }, { 723, 0xAF96 }, { 724, 0xAF97 }, { 725, 0xAF98 }, { 726, 0xAF99 }, { 727, 0xAF9A },
    { 728, 0xAF9B }, { 729, 0xAF9D }, { 730, 0xAF9E }, { 731, 0xAF9F }, { 732, 0xAFA0 }, { 733, 0xAFA1 }, { 734, 0xAFA2 }, { 735, 0xAFA3 },
    { 736, 0xAFA4 }, { 737, 0xAFA5 }, { 738, 0xAFA6 }, { 739, 0xAFA7 }, { 740, 0xAFA8 }, { 741, 0xAFA9 }, { 742, 0xAFAA }, { 743, 0xAFAB },
    { 744, 0xAFAC }, { 745, 0xAFAD }, { 746, 0xAFAE }, { 747, 0xAFAF }, { 748, 0xAFB0 }, { 749, 0xAFB1 }, { 750, 0xAFB2 }, { 751, 0xAFB3 },
    { 752, 0xAFB4 }, { 753, 0xAFB5 }, { 754, 0xAFB6 }, { 755, 0xAFB7 }, { 756, 0xAFBA }, { 757, 0xAFBB }, { 758, 0xAFBD }, { 759, 0xAFBE },
    { 760, 0xAFBF }, { 761, 0xAFC1 }, { 762, 0xAFC2 }, { 763, 0xAFC3 }, { 764, 0xAFC4 }, { 765, 0xAFC5 }, { 766, 0xAFC6 }, { 767, 0xAFCA },
    { 768, 0xAFCC }, { 769, 0xAFCF }, { 770, 0xAFD0 }, { 771, 0xAFD1 }, { 772, 0xAFD2 }, { 773, 0xAFD3 }, { 774, 0xAFD5 }, { 775, 0xAFD6 },
    { 776, 0xAFD7 }, { 777, 0xAFD8 }, { 778, 0xAFD9 }, { 779, 0xAFDA }, { 780, 0xAFDB }, { 781, 0xAFDD }, { 782, 0xAFDE }, { 783, 0xAFDF },
    { 784, 0xAFE0 }, { 785, 0xAFE1 }, { 792, 0xAFE2 }, { 793, 0xAFE3 }, { 794, 0xAFE4 }, { 795, 0xAFE5 }, { 796, 0xAFE6 }, { 797, 0xAFE7 },
    { 798, 0xAFEA }, { 799, 0xAFEB }, { 800, 0xAFEC }, { 801, 0xAFED }, { 802, 0xAFEE }, { 803, 0xAFEF }, { 804, 0xAFF2 }, { 805, 0xAFF3 },
    { 806, 0xAFF5 }, { 807, 0xAFF6 }, { 808, 0xAFF7 }, { 809, 0xAFF9 }, { 810, 0xAFFA }, { 811, 0xAFFB }, { 812, 0xAFFC }, { 813, 0xAFFD },
    { 814, 0xAFFE }, { 815, 0xAFFF }, { 816, 0xB002 }, { 817, 0xB003 }, { 824, 0xB005 }, { 825, 0xB006 }, { 826, 0xB007 }, { 827, 0xB008 },
    { 828, 0xB009 }, { 829, 0xB00A }, { 830, 0xB00B }, { 831, 0xB00D }, { 832, 0xB00E }, { 833, 0xB00F }, { 834, 0xB011 }, { 835, 0xB012 },
    { 836, 0xB013 }, { 837, 0xB015 }, { 838, 0xB016 }, { 839, 0xB017 }, { 840, 0xB018 }, { 841, 0xB019 }, { 842, 0xB01A }, { 843, 0xB01B },
    { 844, 0xB01E }, { 845, 0xB01F }, { 846, 0xB020 }, { 847, 0xB021 }, { 848, 0xB022 }, { 849, 0xB023 }, { 850, 0xB024 }, { 851, 0xB025 },
    { 852, 0xB026 }, { 853, 0xB027 }, { 854, 0xB029 }, { 855, 0xB02A }, { 856, 0xB02B }, { 857, 0xB02C }, { 858, 0xB02D }, { 859, 0xB02E },
    { 860, 0xB02F }, { 861, 0xB030 }, { 862, 0xB031 }, { 863, 0xB032 }, { 864, 0xB033 }, { 865, 0xB034 }, { 866, 0xB035 }, { 867, 0xB036 },
    { 868, 0xB037 }, { 869, 0xB038 }, { 870, 0xB039 }, { 871, 0xB03A }, { 872, 0xB03B }, { 873, 0xB03C }, { 874, 0xB03D }, { 875, 0xB03E },
    { 876, 0xB03F }, { 877, 0xB040 }, { 878, 0xB041 }, { 879, 0xB042 }, { 880, 0xB043 }, { 881, 0xB046 }, { 882, 0xB047 }, { 883, 0xB049 },
    { 884, 0xB04B }, { 885, 0xB04D }, { 886, 0xB04F }, { 887, 0xB050 }, { 888, 0xB051 }, { 889, 0xB052 }, { 890, 0xB056 }, { 891, 0xB058 },
    { 892, 0xB05A }, { 893, 0xB05B }, { 894, 0xB05C }, { 895, 0xB05E }, { 896, 0xB05F }, { 897, 0xB060 }, { 898, 0xB061 }, { 899, 0xB062 },
    { 900, 0xB063 }, { 901, 0xB064 }, { 902, 0xB065 }, { 903, 0xB066 }, { 904, 0xB067 }, { 905, 0xB068 }, { 906, 0xB069 }, { 907, 0xB06A },
    { 908, 0xB06B }, { 909, 0xB06C }, { 910, 0xB06D }, { 911, 0xB06E }, { 912, 0xB06F }, { 913, 0xB070 }, { 914, 0xB071 }, { 915, 0xB072 },
    { 916, 0xB073 }, { 917, 0xB074 }, { 918, 0xB075 }, { 919, 0xB076 }, { 920, 0xB077 }, { 921, 0xB078 }, { 922, 0xB079 }, { 923, 0xB07A },
    { 924, 0xB07B }, { 925, 0xB07E }, { 926, 0xB07F }, { 927, 0xB081 }, { 928, 0xB082 }, { 929, 0xB083 }, { 930, 0xB085 }, { 931, 0xB086 },
    { 932, 0xB087 }, { 933, 0xB088 }, { 934, 0xB089 }, { 935, 0xB08A }, { 936, 0xB08B }, { 937, 0xB08E }, { 938, 0xB090 }, { 939, 0xB092 },
    { 940, 0xB093 }, { 941, 0xB094 }, { 942, 0xB095 }, { 943, 0xB096 }, { 944, 0xB097 }, { 945, 0xB09B }, { 946, 0xB09D }, { 947, 0xB09E },
    { 948, 0xB0A3 }, { 949, 0xB0A4 }, { 950, 0xB0A5 }, { 951, 0xB0A6 }, { 952, 0xB0A7 }, { 953, 0xB0AA }, { 954, 0xB0B0 }, { 955, 0xB0B2 },
    { 956, 0xB0B6 }, { 957, 0xB0B7 }, { 958, 0xB0B9 }, { 959, 0xB0BA }, { 960, 0xB0BB }, { 961, 0xB0BD }, { 962, 0xB0BE }, { 963, 0xB0BF },
    { 964, 0xB0C0 }, { 965, 0xB0C1 }, { 966, 0xB0C2 }, { 967, 0xB0C3 }, { 968, 0xB0C6 }, { 969, 0xB0CA }, { 970, 0xB0CB }, { 971, 0xB0CC },
    { 972, 0xB0CD }, { 973, 0xB0CE }, { 974, 0xB0CF }, { 975, 0xB0D2 }, { 982, 0xB0D3 }, { 983, 0xB0D5 }, { 984, 0xB0D6 }, { 985, 0xB0D7 },
    { 986, 0xB0D9 }, { 987, 0xB0DA }, { 988, 0xB0DB }, { 989, 0xB0DC }, { 990, 0xB0DD }, { 991, 0xB0DE }, { 992, 0xB0DF }, { 993, 0xB0E1 },
    { 994, 0xB0E2 }, { 995, 0xB0E3 }, { 996, 0xB0E4 }, { 997, 0xB0E6 }, { 998, 0xB0E7 }, { 999, 0xB0E8 }, { 1000, 0xB0E9 }, { 1001, 0xB0EA },
    { 1002, 0xB0EB }, { 1003, 0xB0EC }, { 1004, 0xB0ED }, { 1005, 0xB0EE }, { 1006, 0xB0EF }, { 1007, 0xB0F0 }, { 1014, 0xB0F1 }, { 1015, 0xB0F2 },
    { 1016, 0xB0F3 }, { 1017, 0xB0F4 }, { 1018, 0xB0F5 }, { 1019, 0xB0F6 }, { 1020, 0xB0F7 }, { 1021, 0xB0F8 }, { 1022, 0xB0F9 }, { 1023, 0xB0FA },
    { 1024, 0xB0FB }, { 1025, 0xB0FC }, { 1026, 0xB0FD }, { 1027, 0xB0FE }, { 1028, 0xB0FF }, { 1029, 0xB100 }, { 1030, 0xB101 }, { 1031, 0xB102 },
    { 1032, 0xB103 }, { 1033, 0xB104 }, { 1034, 0xB105 }, { 1035, 0xB106 }, { 1036, 0xB107 }, { 1037, 0xB10A }, { 1038, 0xB10D }, { 1039, 0xB10E },
    { 1040, 0xB10F }, { 1041, 0xB111 }, { 1042, 0xB114 }, { 1043, 0xB115 }, { 1044, 0xB116 }, { 1045, 0xB117 }, { 1046, 0xB11A }, { 1047, 0xB11E },
    { 1048, 0xB11F }, { 1049, 0xB120 }, { 1050, 0xB121 }, { 1051, 0xB122 }, { 1052, 0xB126 }, { 1053, 0xB127 }, { 1054, 0xB129 }, { 1055, 0xB12A },
    { 1056, 0xB12B }, { 1057, 0xB12D }, { 1058, 0xB12E }, { 1059, 0xB12F }, { 1060, 0xB130 }, { 1061, 0xB131 }, { 1062, 0xB132 }, { 1063, 0xB133 },
    { 1064, 0xB136 }, { 1065, 0xB13A }, { 1066, 0xB13B }, { 1067, 0xB13C }, { 1068, 0xB13D }, { 1069, 0xB13E }, { 1070, 0xB13F }, { 1071, 0xB142 },
    { 1072, 0xB143 }, { 1073, 0xB145 }, { 1074, 0xB146 }, { 1075, 0xB147 }, { 1076, 0xB149 }, { 1077, 0xB14A }, { 1078, 0xB14B }, { 1079, 0xB14C },
    { 1080, 0xB14D }, { 1081, 0xB14E }, { 1082, 0xB14F }, { 1083, 0xB152 }, { 1084, 0xB153 }, { 1085, 0xB156 }, { 1086, 0xB157 }, { 1087, 0xB159 },
    { 1088, 0xB15A }, { 1089, 0xB15B }, { 1090, 0xB15D }, { 1091, 0xB15E }, { 1092, 0xB15F }, { 1093, 0xB161 }, { 1094, 0xB162 }, { 1095, 0xB163 },
    { 1096, 0xB164 }, { 1097, 0xB165 }, { 1098, 0xB166 }, { 1099, 0xB167 }, { 1100, 0xB168 }, { 1101, 0xB169 }, { 1102, 0xB16A }, { 1103, 0xB16B },
    { 1104, 0xB16C }, { 1105, 0xB16D }, { 1106, 0xB16E }, { 1107, 0xB16F }, { 1108, 0xB170 }, { 1109, 0xB171 }, { 1110, 0xB172 }, { 1111, 0xB173 },
    { 1112, 0xB174 }, { 1113, 0xB175 }, { 1114, 0xB176 }, { 1115, 0xB177 }, { 1116, 0xB17A }, { 1117, 0xB17B }, { 1118, 0xB17D }, { 1119, 0xB17E },
    { 1120, 0xB17F }, { 1121, 0xB181 }, { 1122, 0xB183 }, { 1123, 0xB184 }, { 1124, 0xB185 }, { 1125, 0xB186 }, { 1126, 0xB187 }, { 1127, 0xB18A },
    { 1128, 0xB18C }, { 1129, 0xB18E }, { 1130, 0xB18F }, { 1131, 0xB190 }, { 1132, 0xB191 }, { 1133, 0xB195 }, { 1134, 0xB196 }, { 1135, 0xB197 },
    { 1136, 0xB199 }, { 1137, 0xB19A }, { 1138, 0xB19B }, { 1139, 0xB19D }, { 1140, 0xB19E }, { 1141, 0xB19F }, { 1142, 0xB1A0 }, { 1143, 0xB1A1 },
    { 1144, 0xB1A2 }, { 1145, 0xB1A3 }, { 1146, 0xB1A4 }, { 1147, 0xB1A5 }, { 1148, 0xB1A6 }, { 1149, 0xB1A7 }, { 1150, 0xB1A9 }, { 1151, 0xB1AA },
    { 1152, 0xB1AB }, { 1153, 0xB1AC }, { 1154, 0xB1AD }, { 1155, 0xB1AE }, { 1156, 0xB1AF }, { 1157, 0xB1B0 }, { 1158, 0xB1B1 }, { 1159, 0xB1B2 },
    { 1160, 0xB1B3 }, { 1161, 0xB1B4 }, { 1162, 0xB1B5 }, { 1163, 0xB1B6 }, { 1164, 0xB1B7 }, { 1165, 0xB1B8 }, { 1172, 0xB1B9 }, { 1173, 0xB1BA },
    { 1174, 0xB1BB }, { 1175, 0xB1BC }, { 1176, 0xB1BD }, { 1177, 0xB1BE }, { 1178, 0xB1BF }, { 1179, 0xB1C0 }, { 1180, 0xB1C1 }, { 1181, 0xB1C2 },
    { 1182, 0xB1C3 }, { 1183, 0xB1C4 }, { 1184, 0xB1C5 }, { 1185, 0xB1C6 }, { 1186, 0xB1C7 }, { 1187, 0xB1C8 }, { 1188, 0xB1C9 }, { 1189, 0xB1CA },
    { 1190, 0xB1CB }, { 1191, 0xB1CD }, { 1192, 0xB1CE }, { 1193, 0xB1CF }, { 1194, 0xB1D1 }, { 1195, 0xB1D2 }, { 1196, 0xB1D3 }, { 1197, 0xB1D5 },
    { 1204, 0xB1D6 }, { 1205, 0xB1D7 }, { 1206, 0xB1D8 }, { 1207, 0xB1D9 }, { 1208, 0xB1DA }, { 1209, 0xB1DB }, { 1210, 0xB1DE }, { 1211, 0xB1E0 },
    { 1212, 0xB1E1 }, { 1213, 0xB1E2 }, { 1214, 0xB1E3 }, { 1215, 0xB1E4 }, { 1216, 0xB1E5 }, { 1217, 0xB1E6 }, { 1218, 0xB1E7 }, { 1219, 0xB1EA },
    { 1220, 0xB1EB }, { 1221, 0xB1ED }, { 1222, 0xB1EE }, { 1223, 0xB1EF }, { 1224, 0xB1F1 }, { 1225, 0xB1F2 }, { 1226, 0xB1F3 }, { 1227, 0xB1F4 },
    { 1228, 0xB1F5 }, { 1229, 0xB1F6 }, { 1230, 0xB1F7 }, { 1231, 0xB1F8 }, { 1232, 0xB1FA }, { 1233, 0xB1FC }, { 1234, 0xB1FE }, { 1235, 0xB1FF },
    { 1236, 0xB200 }, { 1237, 0xB201 }, { 1238, 0xB202 }, { 1239, 0xB203 }, { 1240, 0xB206 }, { 1241, 0xB207 }, { 1242, 0xB209 }, { 1243, 0xB20A },
    { 1244, 0xB20D }, { 1245, 0xB20E }, { 1246, 0xB20F }, { 1247, 0xB210 }, { 1248, 0xB211 }, { 1249, 0xB212 }, { 1250, 0xB213 }, { 1251, 0xB216 },
    { 1252, 0xB218 }, { 1253, 0xB21A }, { 1254, 0xB21B }, { 1255, 0xB21C }, { 1256, 0xB21D }, { 1257, 0xB21E }, { 1258, 0xB21F }, { 1259, 0xB221 },
    { 1260, 0xB222 }, { 1261, 0xB223 }, { 1262, 0xB224 }, { 1263, 0xB225 }, { 1264, 0xB226 }, { 1265, 0xB227 }, { 1266, 0xB228 }, { 1267, 0xB229 },
    { 1268, 0xB22A }, { 1269, 0xB22B }, { 1270, 0xB22C }, { 1271, 0xB22D }, { 1272, 0xB22E }, { 1273, 0xB22F }, { 1274, 0xB230 }, { 1275, 0xB231 },
    { 1276, 0xB232 }, { 1277, 0xB233 }, { 1278, 0xB235 }, { 1279, 0xB236 }, { 1280, 0xB237 }, { 1281, 0xB238 }, { 1282, 0xB239 }, { 1283, 0xB23A },
    { 1284, 0xB23B }, { 1285, 0xB23D }, { 1286, 0xB23E }, { 1287, 0xB23F }, { 1288, 0xB240 }, { 1289, 0xB241 }, { 1290, 0xB242 }, { 1291, 0xB243 },
    { 1292, 0xB244 }, { 1293, 0xB245 }, { 1294, 0xB246 }, { 1295, 0xB247 }, { 1296, 0xB248 }, { 1297, 0xB249 }, { 1298, 0xB24A }, { 1299, 0xB24B },
    { 1300, 0xB24C }, { 1301, 0xB24D }, { 1302, 0xB24E }, { 1303, 0xB24F }, { 1304, 0xB250 }, { 1305, 0xB251 }, { 1306, 0xB252 }, { 1307, 0xB253 },
    { 1308, 0xB254 }, { 1309, 0xB255 }, { 1310, 0xB256 }, { 1311, 0xB257 }, { 1312, 0xB259 }, { 1313, 0xB25A }, { 1314, 0xB25B }, { 1315, 0xB25D },
    { 1316, 0xB25E }, { 1317, 0xB25F }, { 1318, 0xB261 }, { 1319, 0xB262 }, { 1320, 0xB263 }, { 1321, 0xB264 }, { 1322, 0xB265 }, { 1323, 0xB266 },
    { 1324, 0xB267 }, { 1325, 0xB26A }, { 1326, 0xB26B }, { 1327, 0xB26C }, { 1328, 0xB26D }, { 1329, 0xB26E }, { 1330, 0xB26F }, { 1331, 0xB270 },
    { 1332, 0xB271 }, { 1333, 0xB272 }, { 1334, 0xB273 }, { 1335, 0xB276 }, { 1336, 0xB277 }, { 1337, 0xB278 }, { 1338, 0xB279 }, { 1339, 0xB27A },
    { 1340, 0xB27B }, { 1341, 0xB27D }, { 1342, 0xB27E }, { 1343, 0xB27F }, { 1344, 0xB280 }, { 1345, 0xB281 }, { 1346, 0xB282 }, { 1347, 0xB283 },
    { 1348, 0xB286 }, { 1349, 0xB287 }, { 1350, 0xB288 }, { 1351, 0xB28A }, { 1352, 0xB28B }, { 1353, 0xB28C }, { 1354, 0xB28D }, { 1355, 0xB28E },
    { 1362, 0xB28F }, { 1363, 0xB292 }, { 1364, 0xB293 }, { 1365, 0xB295 }, { 1366, 0xB296 }, { 1367, 0xB297 }, { 1368, 0xB29B }, { 1369, 0xB29C },
    { 1370, 0xB29D }, { 1371, 0xB29E }, { 1372, 0xB29F }, { 1373, 0xB2A2 }, { 1374, 0xB2A4 }, { 1375, 0xB2A7 }, { 1376, 0xB2A8 }, { 1377, 0xB2A9 },
    { 1378, 0xB2AB }, { 1379, 0xB2AD }, { 1380, 0xB2AE }, { 1381, 0xB2AF }, { 1382, 0xB2B1 }, { 1383, 0xB2B2 }, { 1384, 0xB2B3 }, { 1385, 0xB2B5 },
    { 1386, 0xB2B6 }, { 1387, 0xB2B7 }, { 1394, 0xB2B8 }, { 1395, 0xB2B9 }, { 1396, 0xB2BA }, { 1397, 0xB2BB }, { 1398, 0xB2BC }, { 1399, 0xB2BD },
    { 1400, 0xB2BE }, { 1401, 0xB2BF }, { 1402, 0xB2C0 }, { 1403, 0xB2C1 }, { 1404, 0xB2C2 }, { 1405, 0xB2C3 }, { 1406, 0xB2C4 }, { 1407, 0xB2C5 },
    { 1408, 0xB2C6 }, { 1409, 0xB2C7 }, { 1410, 0xB2CA }, { 1411, 0xB2CB }, { 1412, 0xB2CD }, { 1413, 0xB2CE }, { 1414, 0xB2CF }, { 1415, 0xB2D1 },
    { 1416, 0xB2D3 }, { 1417, 0xB2D4 }, { 1418, 0xB2D5 }, { 1419, 0xB2D6 }, { 1420, 0xB2D7 }, { 1421, 0xB2DA }, { 1422, 0xB2DC }, { 1423, 0xB2DE },
    { 1424, 0xB2DF }, { 1425, 0xB2E0 }, { 1426, 0xB2E1 }, { 1427, 0xB2E3 }, { 1428, 0xB2E7 }, { 1429, 0xB2E9 }, { 1430, 0xB2EA }, { 1431, 0xB2F0 },
    { 1432, 0xB2F1 }, { 1433, 0xB2F2 }, { 1434, 0xB2F6 }, { 1435, 0xB2FC }, { 1436, 0xB2FD }, { 1437, 0xB2FE }, { 1438, 0xB302 }, { 1439, 0xB303 },
    { 1440, 0xB305 }, { 1441, 0xB306 }, { 1442, 0xB307 }, { 1443, 0xB309 }, { 1444, 0xB30A }, { 1445, 0xB30B }, { 1446, 0xB30C }, { 1447, 0xB30D },
    { 1448, 0xB30E }, { 1449, 0xB30F }, { 1450, 0xB312 }, { 1451, 0xB316 }, { 1452, 0xB317 }, { 1453, 0xB318 }, { 1454, 0xB319 }, { 1455, 0xB31A },
    { 1456, 0xB31B }, { 1457, 0xB31D }, { 1458, 0xB31E }, { 1459, 0xB31F }, { 1460, 0xB320 }, { 1461, 0xB321 }, { 1462, 0xB322 }, { 1463, 0xB323 },
    { 1464, 0xB324 }, { 1465, 0xB325 }, { 1466, 0xB326 }, { 1467, 0xB327 }, { 1468, 0xB328 }, { 1469, 0xB329 }, { 1470, 0xB32A }, { 1471, 0xB32B },
    { 1472, 0xB32C }, { 1473, 0xB32D }, { 1474, 0xB32E }, { 1475, 0xB32F }, { 1476, 0xB330 }, { 1477, 0xB331 }, { 1478, 0xB332 }, { 1479, 0xB333 },
    { 1480, 0xB334 }, { 1481, 0xB335 }, { 1482, 0xB336 }, { 1483, 0xB337 }, { 1484, 0xB338 }, { 1485, 0xB339 }, { 1486, 0xB33A }, { 1487, 0xB33B },
    { 1488, 0xB33C }, { 1489, 0xB33D }, { 1490, 0xB33E }, { 1491, 0xB33F }, { 1492, 0xB340 }, { 1493, 0xB341 }, { 1494, 0xB342 }, { 1495, 0xB343 },
    { 1496, 0xB344 }, { 1497, 0xB345 }, { 1498, 0xB346 }, { 1499, 0xB347 }, { 1500, 0xB348 }, { 1501, 0xB349 }, { 1502, 0xB34A }, { 1503, 0xB34B },
    { 1504, 0xB34C }, { 1505, 0xB34D }, { 1506, 0xB34E }, { 1507, 0xB34F }, { 1508, 0xB350 }, { 1509, 0xB351 }, { 1510, 0xB352 }, { 1511, 0xB353 },
    { 1512, 0xB357 }, { 1513, 0xB359 }, { 1514, 0xB35A }, { 1515, 0xB35D }, { 1516, 0xB360 }, { 1517, 0xB361 }, { 1518, 0xB362 }, { 1519, 0xB363 },
    { 1520, 0xB366 }, { 1521, 0xB368 }, { 1522, 0xB36A }, { 1523, 0xB36C }, { 1524, 0xB36D }, { 1525, 0xB36F }, { 1526, 0xB372 }, { 1527, 0xB373 },
    { 1528, 0xB375 }, { 1529, 0xB376 }, { 1530, 0xB377 }, { 1531, 0xB379 }, { 1532, 0xB37A }, { 1533, 0xB37B }, { 1534, 0xB37C }, { 1535, 0xB37D },
    { 1536, 0xB37E }, { 1537, 0xB37F }, { 1538, 0xB382 }, { 1539, 0xB386 }, { 1540, 0xB387 }, { 1541, 0xB388 }, { 1542, 0xB389 }, { 1543, 0xB38A },
    { 1544, 0xB38B }, { 1545, 0xB38D }, { 1552, 0xB38E }, { 1553, 0xB38F }, { 1554, 0xB391 }, { 1555, 0xB392 }, { 1556, 0xB393 }, { 1557, 0xB395 },
    { 1558, 0xB396 }, { 1559, 0xB397 }, { 1560, 0xB398 }, { 1561, 0xB399 }, { 1562, 0xB39A }, { 1563, 0xB39B }, { 1564, 0xB39C }, { 1565, 0xB39D },
    { 1566, 0xB39E }, { 1567, 0xB39F }, { 1568, 0xB3A2 }, { 1569, 0xB3A3 }, { 1570, 0xB3A4 }, { 1571, 0xB3A5 }, { 1572, 0xB3A6 }, { 1573, 0xB3A7 },
    { 1574, 0xB3A9 }, { 1575, 0xB3AA }, { 1576, 0xB3AB }, { 1577, 0xB3AD }, { 1584, 0xB3AE }, { 1585, 0xB3AF }, { 1586, 0xB3B0 }, { 1587, 0xB3B1 },
    { 1588, 0xB3B2 }, { 1589, 0xB3B3 }, { 1590, 0xB3B4 }, { 1591, 0xB3B5 }, { 1592, 0xB3B6 }, { 1593, 0xB3B7 }, { 1594, 0xB3B8 }, { 1595, 0xB3B9 },
    { 1596, 0xB3BA }, { 1597, 0xB3BB }, { 1598, 0xB3BC }, { 1599, 0xB3BD }, { 1600, 0xB3BE }, { 1601, 0xB3BF }, { 1602, 0xB3C0 }, { 1603, 0xB3C1 },
    { 1604, 0xB3C2 }, { 1605, 0xB3C3 }, { 1606, 0xB3C6 }, { 1607, 0xB3C7 }, { 1608, 0xB3C9 }, { 1609, 0xB3CA }, { 1610, 0xB3CD }, { 1611, 0xB3CF },
    { 1612, 0xB3D1 }, { 1613, 0xB3D2 }, { 1614, 0xB3D3 }, { 1615, 0xB3D6 }, { 1616, 0xB3D8 }, { 1617, 0xB3DA }, { 1618, 0xB3DC }, { 1619, 0xB3DE },
    { 1620, 0xB3DF }, { 1621, 0xB3E1 }, { 1622, 0xB3E2 }, { 1623, 0xB3E3 }, { 1624, 0xB3E5 }, { 1625, 0xB3E6 }, { 1626, 0xB3E7 }, { 1627, 0xB3E9 },
    { 1628, 0xB3EA }, { 1629, 0xB3EB }, { 1630, 0xB3EC }, { 1631, 0xB3ED }, { 1632, 0xB3EE }, { 1633, 0xB3EF }, { 1634, 0xB3F0 }, { 1635, 0xB3F1 },
    { 1636, 0xB3F2 }, { 1637, 0xB3F3 }, { 1638, 0xB3F4 }, { 1639, 0xB3F5 }, { 1640, 0xB3F6 }, { 1641, 0xB3F7 }, { 1642, 0xB3F8 }, { 1643, 0xB3F9 },
    { 1644, 0xB3FA }, { 1645, 0xB3FB }, { 1646, 0xB3FD }, { 1647, 0xB3FE }, { 1648, 0xB3FF }, { 1649, 0xB400 }, { 1650, 0xB401 }, { 1651, 0xB402 },
    { 1652, 0xB403 }, { 1653, 0xB404 }, { 1654, 0xB405 }, { 1655, 0xB406 }, { 1656, 0xB407 }, { 1657, 0xB408 }, { 1658, 0xB409 }, { 1659, 0xB40A },
    { 1660, 0xB40B }, { 1661, 0xB40C }, { 1662, 0xB40D }, { 1663, 0xB40E }, { 1664, 0xB40F }, { 1665, 0xB411 }, { 1666, 0xB412 }, { 1667, 0xB413 },
    { 1668, 0xB414 }, { 1669, 0xB415 }, { 1670, 0xB416 }, { 1671, 0xB417 }, { 1672, 0xB419 }, { 1673, 0xB41A }, { 1674, 0xB41B }, { 1675, 0xB41D },
    { 1676, 0xB41E }, { 1677, 0xB41F }, { 1678, 0xB421 }, { 1679, 0xB422 }, { 1680, 0xB423 }, { 1681, 0xB424 }, { 1682, 0xB425 }, { 1683, 0xB426 },
    { 1684, 0xB427 }, { 1685, 0xB42A }, { 1686, 0xB42C }, { 1687, 0xB42D }, { 1688, 0xB42E }, { 1689, 0xB42F }, { 1690, 0xB430 }, { 1691, 0xB431 },
    { 1692, 0xB432 }, { 1693, 0xB433 }, { 1694, 0xB435 }, { 1695, 0xB436 }, { 1696, 0xB437 }, { 1697, 0xB438 }, { 1698, 0xB439 }, { 1699, 0xB43A },
    { 1700, 0xB43B }, { 1701, 0xB43C }, { 1702, 0xB43D }, { 1703, 0xB43E }, { 1704, 0xB43F }, { 1705, 0xB440 }, { 1706, 0xB441 }, { 1707, 0xB442 },
    { 1708, 0xB443 }, { 1709, 0xB444 }, { 1710, 0xB445 }, { 1711, 0xB446 }, { 1712, 0xB447 }, { 1713, 0xB448 }, { 1714, 0xB449 }, { 1715, 0xB44A },
    { 1716, 0xB44B }, { 1717, 0xB44C }, { 1718, 0xB44D }, { 1719, 0xB44E }, { 1720, 0xB44F }, { 1721, 0xB452 }, { 1722, 0xB453 }, { 1723, 0xB455 },
    { 1724, 0xB456 }, { 1725, 0xB457 }, { 1726, 0xB459 }, { 1727, 0xB45A }, { 1728, 0xB45B }, { 1729, 0xB45C }, { 1730, 0xB45D }, { 1731, 0xB45E },
    { 1732, 0xB45F }, { 1733, 0xB462 }, { 1734, 0xB464 }, { 1735, 0xB466 }, { 1742, 0xB467 }, { 1743, 0xB468 }, { 1744, 0xB469 }, { 1745, 0xB46A },
    { 1746, 0xB46B }, { 1747, 0xB46D }, { 1748, 0xB46E }, { 1749, 0xB46F }, { 1750, 0xB470 }, { 1751, 0xB471 }, { 1752, 0xB472 }, { 1753, 0xB473 },
    { 1754, 0xB474 }, { 1755, 0xB475 }, { 1756, 0xB476 }, { 1757, 0xB477 }, { 1758, 0xB478 }, { 1759, 0xB479 }, { 1760, 0xB47A }, { 1761, 0xB47B },
    { 1762, 0xB47C }, { 1763, 0xB47D }, { 1764, 0xB47E }, { 1765, 0xB47F }, { 1766, 0xB481 }, { 1767, 0xB482 }, { 1774, 0xB483 }, { 1775, 0xB484 },
    { 1776, 0xB485 }, { 1777, 0xB486 }, { 1778, 0xB487 }, { 1779, 0xB489 }, { 1780, 0xB48A }, { 1781, 0xB48B }, { 1782, 0xB48C }, { 1783, 0xB48D },
    { 1784, 0xB48E }, { 1785, 0xB48F }, { 1786, 0xB490 }, { 1787, 0xB491 }, { 1788, 0xB492 }, { 1789, 0xB493 }, { 1790, 0xB494 }, { 1791, 0xB495 },
    { 1792, 0xB496 }, { 1793, 0xB497 }, { 1794, 0xB498 }, { 1795, 0xB499 }, { 1796, 0xB49A }, { 1797, 0xB49B }, { 1798, 0xB49C }, { 1799, 0xB49E },
    { 1800, 0xB49F }, { 1801, 0xB4A0 }, { 1802, 0xB4A1 }, { 1803, 0xB4A2 }, { 1804, 0xB4A3 }, { 1805, 0xB4A5 }, { 1806, 0xB4A6 }, { 1807, 0xB4A7 },
    { 1808, 0xB4A9 }, { 1809, 0xB4AA }, { 1810, 0xB4AB }, { 1811, 0xB4AD }, { 1812, 0xB4AE }, { 1813, 0xB4AF }, { 1814, 0xB4B0 }, { 1815, 0xB4B1 },
    { 1816, 0xB4B2 }, { 1817, 0xB4B3 }, { 1818, 0xB4B4 }, { 1819, 0xB4B6 }, { 1820, 0xB4B8 }, { 1821, 0xB4BA }, { 1822, 0xB4BB }, { 1823, 0xB4BC },
    { 1824, 0xB4BD }, { 1825, 0xB4BE }, { 1826, 0xB4BF }, { 1827, 0xB4C1 }, { 1828, 0xB4C2 }, { 1829, 0xB4C3 }, { 1830, 0xB4C5 }, { 1831, 0xB4C6 },
    { 1832, 0xB4C7 }, { 1833, 0xB4C9 }, { 1834, 0xB4CA }, { 1835, 0xB4CB }, { 1836, 0xB4CC }, { 1837, 0xB4CD }, { 1838, 0xB4CE }, { 1839, 0xB4CF },
    { 1840, 0xB4D1 }, { 1841, 0xB4D2 }, { 1842, 0xB4D3 }, { 1843, 0xB4D4 }, { 1844, 0xB4D6 }, { 1845, 0xB4D7 }, { 1846, 0xB4D8 }, { 1847, 0xB4D9 },
    { 1848, 0xB4DA }, { 1849, 0xB4DB }, { 1850, 0xB4DE }, { 1851, 0xB4DF }, { 1852, 0xB4E1 }, { 1853, 0xB4E2 }, { 1854, 0xB4E5 }, { 1855, 0xB4E7 },
    { 1856, 0xB4E8 }, { 1857, 0xB4E9 }, { 1858, 0xB4EA }, { 1859, 0xB4EB }, { 1860, 0xB4EE }, { 1861, 0xB4F0 }, { 1862, 0xB4F2 }, { 1863, 0xB4F3 },
    { 1864, 0xB4F4 }, { 1865, 0xB4F5 }, { 1866, 0xB4F6 }, { 1867, 0xB4F7 }, { 1868, 0xB4F9 }, { 1869, 0xB4FA }, { 1870, 0xB4FB }, { 1871, 0xB4FC },
    { 1872, 0xB4FD }, { 1873, 0xB4FE }, { 1874, 0xB4FF }, { 1875, 0xB500 }, { 1876, 0xB501 }, { 1877, 0xB502 }, { 1878, 0xB503 }, { 1879, 0xB504 },
    { 1880, 0xB505 }, { 1881, 0xB506 }, { 1882, 0xB507 }, { 1883, 0xB508 }, { 1884, 0xB509 }, { 1885, 0xB50A }, { 1886, 0xB50B }, { 1887, 0xB50C },
    { 1888, 0xB50D }, { 1889, 0xB50E }, { 1890, 0xB50F }, { 1891, 0xB510 }, { 1892, 0xB511 }, { 1893, 0xB512 }, { 1894, 0xB513 }, { 1895, 0xB516 },
    { 1896, 0xB517 }, { 1897, 0xB519 }, { 1898, 0xB51A }, { 1899, 0xB51D }, { 1900, 0xB51E }, { 1901, 0xB51F }, { 1902, 0xB520 }, { 1903, 0xB521 },
    { 1904, 0xB522 }, { 1905, 0xB523 }, { 1906, 0xB526 }, { 1907, 0xB52B }, { 1908, 0xB52C }, { 1909, 0xB52D }, { 1910, 0xB52E }, { 1911, 0xB52F },
    { 1912, 0xB532 }, { 1913, 0xB533 }, { 1914, 0xB535 }, { 1915, 0xB536 }, { 1916, 0xB537 }, { 1917, 0xB539 }, { 1918, 0xB53A }, { 1919, 0xB53B },
    { 1920, 0xB53C }, { 1921, 0xB53D }, { 1922, 0xB53E }, { 1923, 0xB53F }, { 1924, 0xB542 }, { 1925, 0xB546 }, { 1932, 0xB547 }, { 1933, 0xB548 },
    { 1934, 0xB549 }, { 1935, 0xB54A }, { 1936, 0xB54E }, { 1937, 0xB54F }, { 1938, 0xB551 }, { 1939, 0xB552 }, { 1940, 0xB553 }, { 1941, 0xB555 },
    { 1942, 0xB556 }, { 1943, 0xB557 }, { 1944, 0xB558 }, { 1945, 0xB559 }, { 1946, 0xB55A }, { 1947, 0xB55B }, { 1948, 0xB55E }, { 1949, 0xB562 },
    { 1950, 0xB563 }, { 1951, 0xB564 }, { 1952, 0xB565 }, { 1953, 0xB566 }, { 1954, 0xB567 }, { 1955, 0xB568 }, { 1956, 0xB569 }, { 1957, 0xB56A },
    { 1964, 0xB56B }, { 1965, 0xB56C }, { 1966, 0xB56D }, { 1967, 0xB56E }, { 1968, 0xB56F }, { 1969, 0xB570 }, { 1970, 0xB571 }, { 1971, 0xB572 },
    { 1972, 0xB573 }, { 1973, 0xB574 }, { 1974, 0xB575 }, { 1975, 0xB576 }, { 1976, 0xB577 }, { 1977, 0xB578 }, { 1978, 0xB579 }, { 1979, 0xB57A },
    { 1980, 0xB57B }, { 1981, 0xB57C }, { 1982, 0xB57D }, { 1983, 0xB57E }, { 1984, 0xB57F }, { 1985, 0xB580 }, { 1986, 0xB581 }, { 1987, 0xB582 },
    { 1988, 0xB583 }, { 1989, 0xB584 }, { 1990, 0xB585 }, { 1991, 0xB586 }, { 1992, 0xB587 }, { 1993, 0xB588 }, { 1994, 0xB589 }, { 1995, 0xB58A },
    { 1996, 0xB58B }, { 1997, 0xB58C }, { 1998, 0xB58D }, { 1999, 0xB58E }, { 2000, 0xB58F }, { 2001, 0xB590 }, { 2002, 0xB591 }, { 2003, 0xB592 },
    { 2004, 0xB593 }, { 2005, 0xB594 }, { 2006, 0xB595 }, { 2007, 0xB596 }, { 2008, 0xB597 }, { 2009, 0xB598 }, { 2010, 0xB599 }, { 2011, 0xB59A },
    { 2012, 0xB59B }, { 2013, 0xB59C }, { 2014, 0xB59D }, { 2015, 0xB59E }, { 2016, 0xB59F }, { 2017, 0xB5A2 }, { 2018, 0xB5A3 }, { 2019, 0xB5A5 },
    { 2020, 0xB5A6 }, { 2021, 0xB5A7 }, { 2022, 0xB5A9 }, { 2023, 0xB5AC }, { 2024, 0xB5AD }, { 2025, 0xB5AE }, { 2026, 0xB5AF }, { 2027, 0xB5B2 },
    { 2028, 0xB5B6 }, { 2029, 0xB5B7 }, { 2030, 0xB5B8 }, { 2031, 0xB5B9 }, { 2032, 0xB5BA }, { 2033, 0xB5BE }, { 2034, 0xB5BF }, { 2035, 0xB5C1 },
    { 2036, 0xB5C2 }, { 2037, 0xB5C3 }, { 2038, 0xB5C5 }, { 2039, 0xB5C6 }, { 2040, 0xB5C7 }, { 2041, 0xB5C8 }, { 2042, 0xB5C9 }, { 2043, 0xB5CA },
    { 2044, 0xB5CB }, { 2045, 0xB5CE }, { 2046, 0xB5D2 }, { 2047, 0xB5D3 }, { 2048, 0xB5D4 }, { 2049, 0xB5D5 }, { 2050, 0xB5D6 }, { 2051, 0xB5D7 },
    { 2052, 0xB5D9 }, { 2053, 0xB5DA }, { 2054, 0xB5DB }, { 2055, 0xB5DC }, { 2056, 0xB5DD }, { 2057, 0xB5DE }, { 2058, 0xB5DF }, { 2059, 0xB5E0 },
    { 2060, 0xB5E1 }, { 2061, 0xB5E2 }, { 2062, 0xB5E3 }, { 2063, 0xB5E4 }, { 2064, 0xB5E5 }, { 2065, 0xB5E6 }, { 2066, 0xB5E7 }, { 2067, 0xB5E8 },
    { 2068, 0xB5E9 }, { 2069, 0xB5EA }, { 2070, 0xB5EB }, { 2071, 0xB5ED }, { 2072, 0xB5EE }, { 2073, 0xB5EF }, { 2074, 0xB5F0 }, { 2075, 0xB5F1 },
    { 2076, 0xB5F2 }, { 2077, 0xB5F3 }, { 2078, 0xB5F4 }, { 2079, 0xB5F5 }, { 2080, 0xB5F6 }, { 2081, 0xB5F7 }, { 2082, 0xB5F8 }, { 2083, 0xB5F9 },
    { 2084, 0xB5FA }, { 2085, 0xB5FB }, { 2086, 0xB5FC }, { 2087, 0xB5FD }, { 2088, 0xB5FE }, { 2089, 0xB5FF }, { 2090, 0xB600 }, { 2091, 0xB601 },
    { 2092, 0xB602 }, { 2093, 0xB603 }, { 2094, 0xB604 }, { 2095, 0xB605 }, { 2096, 0xB606 }, { 2097, 0xB607 }, { 2098, 0xB608 }, { 2099, 0xB609 },
    { 2100, 0xB60A }, { 2101, 0xB60B }, { 2102, 0xB60C }, { 2103, 0xB60D }, { 2104, 0xB60E }, { 2105, 0xB60F }, { 2106, 0xB612 }, { 2107, 0xB613 },
    { 2108, 0xB615 }, { 2109, 0xB616 }, { 2110, 0xB617 }, { 2111, 0xB619 }, { 2112, 0xB61A }, { 2113, 0xB61B }, { 2114, 0xB61C }, { 2115, 0xB61D },
    { 2122, 0xB61E }, { 2123, 0xB61F }, { 2124, 0xB620 }, { 2125, 0xB621 }, { 2126, 0xB622 }, { 2127, 0xB623 }, { 2128, 0xB624 }, { 2129, 0xB626 },
    { 2130, 0xB627 }, { 2131, 0xB628 }, { 2132, 0xB629 }, { 2133, 0xB62A }, { 2134, 0xB62B }, { 2135, 0xB62D }, { 2136, 0xB62E }, { 2137, 0xB62F },
    { 2138, 0xB630 }, { 2139, 0xB631 }, { 2140, 0xB632 }, { 2141, 0xB633 }, { 2142, 0xB635 }, { 2143, 0xB636 }, { 2144, 0xB637 }, { 2145, 0xB638 },
    { 2146, 0xB639 }, { 2147, 0xB63A }, { 2154, 0xB63B }, { 2155, 0xB63C }, { 2156, 0xB63D }, { 2157, 0xB63E }, { 2158, 0xB63F }, { 2159, 0xB640 },
    { 2160, 0xB641 }, { 2161, 0xB642 }, { 2162, 0xB643 }, { 2163, 0xB644 }, { 2164, 0xB645 }, { 2165, 0xB646 }, { 2166, 0xB647 }, { 2167, 0xB649 },
    { 2168, 0xB64A }, { 2169, 0xB64B }, { 2170, 0xB64C }, { 2171, 0xB64D }, { 2172, 0xB64E }, { 2173, 0xB64F }, { 2174, 0xB650 }, { 2175, 0xB651 },
    { 2176, 0xB652 }, { 2177, 0xB653 }, { 2178, 0xB654 }, { 2179, 0xB655 }, { 2180, 0xB656 }, { 2181, 0xB657 }, { 2182, 0xB658 }, { 2183, 0xB659 },
    { 2184, 0xB65A }, { 2185, 0xB65B }, { 2186, 0xB65C }, { 2187, 0xB65D }, { 2188, 0xB65E }, { 2189, 0xB65F }, { 2190, 0xB660 }, { 2191, 0xB661 },
    { 2192, 0xB662 }, { 2193, 0xB663 }, { 2194, 0xB665 }, { 2195, 0xB666 }, { 2196, 0xB667 }, { 2197, 0xB669 }, { 2198, 0xB66A }, { 2199, 0xB66B },
    { 2200, 0xB66C }, { 2201, 0xB66D }, { 2202, 0xB66E }, { 2203, 0xB66F }, { 2204, 0xB670 }, { 2205, 0xB671 }, { 2206, 0xB672 }, { 2207, 0xB673 },
    { 2208, 0xB674 }, { 2209, 0xB675 }, { 2210, 0xB676 }, { 2211, 0xB677 }, { 2212, 0xB678 }, { 2213, 0xB679 }, { 2214, 0xB67A }, { 2215, 0xB67B },
    { 2216, 0xB67C }, { 2217, 0xB67D }, { 2218, 0xB67E }, { 2219, 0xB67F }, { 2220, 0xB680 }, { 2221, 0xB681 }, { 2222, 0xB682 }, { 2223, 0xB683 },
    { 2224, 0xB684 }, { 2225, 0xB685 }, { 2226, 0xB686 }, { 2227, 0xB687 }, { 2228, 0xB688 }, { 2229, 0xB689 }, { 2230, 0xB68A }, { 2231, 0xB68B },
    { 2232, 0xB68C }, { 2233, 0xB68D }, { 2234, 0xB68E }, { 2235, 0xB68F }, { 2236, 0xB690 }, { 2237, 0xB691 }, { 2238, 0xB692 }, { 2239, 0xB693 },
    { 2240, 0xB694 }, { 2241, 0xB695 }, { 2242, 0xB696 }, { 2243, 0xB697 }, { 2244, 0xB698 }, { 2245, 0xB699 }, { 2246, 0xB69A }, { 2247, 0xB69B },
    { 2248, 0xB69E }, { 2249, 0xB69F }, { 2250, 0xB6A1 }, { 2251, 0xB6A2 }, { 2252, 0xB6A3 }, { 2253, 0xB6A5 }, { 2254, 0xB6A6 }, { 2255, 0xB6A7 },
    { 2256, 0xB6A8 }, { 2257, 0xB6A9 }, { 2258, 0xB6AA }, { 2259, 0xB6AD }, { 2260, 0xB6AE }, { 2261, 0xB6AF }, { 2262, 0xB6B0 }, { 2263, 0xB6B2 },
    { 2264, 0xB6B3 }, { 2265, 0xB6B4 }, { 2266, 0xB6B5 }, { 2267, 0xB6B6 }, { 2268, 0xB6B7 }, { 2269, 0xB6B8 }, { 2270, 0xB6B9 }, { 2271, 0xB6BA },
    { 2272, 0xB6BB }, { 2273, 0xB6BC }, { 2274, 0xB6BD }, { 2275, 0xB6BE }, { 2276, 0xB6BF }, { 2277, 0xB6C0 }, { 2278, 0xB6C1 }, { 2279, 0xB6C2 },
    { 2280, 0xB6C3 }, { 2281, 0xB6C4 }, { 2282, 0xB6C5 }, { 2283, 0xB6C6 }, { 2284, 0xB6C7 }, { 2285, 0xB6C8 }, { 2286, 0xB6C9 }, { 2287, 0xB6CA },
    { 2288, 0xB6CB }, { 2289, 0xB6CC }, { 2290, 0xB6CD }, { 2291, 0xB6CE }, { 2292, 0xB6CF }, { 2293, 0xB6D0 }, { 2294, 0xB6D1 }, { 2295, 0xB6D2 },
    { 2296, 0xB6D3 }, { 2297, 0xB6D5 }, { 2298, 0xB6D6 }, { 2299, 0xB6D7 }, { 2300, 0xB6D8 }, { 2301, 0xB6D9 }, { 2302, 0xB6DA }, { 2303, 0xB6DB },
    { 2304, 0xB6DC }, { 2305, 0xB6DD }, { 2312, 0xB6DE }, { 2313, 0xB6DF }, { 2314, 0xB6E0 }, { 2315, 0xB6E1 }, { 2316, 0xB6E2 }, { 2317, 0xB6E3 },
    { 2318, 0xB6E4 }, { 2319, 0xB6E5 }, { 2320, 0xB6E6 }, { 2321, 0xB6E7 }, { 2322, 0xB6E8 }, { 2323, 0xB6E9 }, { 2324, 0xB6EA }, { 2325, 0xB6EB },
    { 2326, 0xB6EC }, { 2327, 0xB6ED }, { 2328, 0xB6EE }, { 2329, 0xB6EF }, { 2330, 0xB6F1 }, { 2331, 0xB6F2 }, { 2332, 0xB6F3 }, { 2333, 0xB6F5 },
    { 2334, 0xB6F6 }, { 2335, 0xB6F7 }, { 2336, 0xB6F9 }, { 2337, 0xB6FA }, { 2344, 0xB6FB }, { 2345, 0xB6FC }, { 2346, 0xB6FD }, { 2347, 0xB6FE },
    { 2348, 0xB6FF }, { 2349, 0xB702 }, { 2350, 0xB703 }, { 2351, 0xB704 }, { 2352, 0xB706 }, { 2353, 0xB707 }, { 2354, 0xB708 }, { 2355, 0xB709 },
    { 2356, 0xB70A }, { 2357, 0xB70B }, { 2358, 0xB70C }, { 2359, 0xB70D }, { 2360, 0xB70E }, { 2361, 0xB70F }, { 2362, 0xB710 }, { 2363, 0xB711 },
    { 2364, 0xB712 }, { 2365, 0xB713 }, { 2366, 0xB714 }, { 2367, 0xB715 }, { 2368, 0xB716 }, { 2369, 0xB717 }, { 2370, 0xB718 }, { 2371, 0xB719 },
    { 2372, 0xB71A }, { 2373, 0xB71B }, { 2374, 0xB71C }, { 2375, 0xB71D }, { 2376, 0xB71E }, { 2377, 0xB71F }, { 2378, 0xB720 }, { 2379, 0xB721 },
    { 2380, 0xB722 }, { 2381, 0xB723 }, { 2382, 0xB724 }, { 2383, 0xB725 }, { 2384, 0xB726 }, { 2385, 0xB727 }, { 2386, 0xB72A }, { 2387, 0xB72B },
    { 2388, 0xB72D }, { 2389, 0xB72E }, { 2390, 0xB731 }, { 2391, 0xB732 }, { 2392, 0xB733 }, { 2393, 0xB734 }, { 2394, 0xB735 }, { 2395, 0xB736 },
    { 2396, 0xB737 }, { 2397, 0xB73A }, { 2398, 0xB73C }, { 2399, 0xB73D }, { 2400, 0xB73E }, { 2401, 0xB73F }, { 2402, 0xB740 }, { 2403, 0xB741 },
    { 2404, 0xB742 }, { 2405, 0xB743 }, { 2406, 0xB745 }, { 2407, 0xB746 }, { 2408, 0xB747 }, { 2409, 0xB749 }, { 2410, 0xB74A }, { 2411, 0xB74B },
    { 2412, 0xB74D }, { 2413, 0xB74E }, { 2414, 0xB74F }, { 2415, 0xB750 }, { 2416, 0xB751 }, { 2417, 0xB752 }, { 2418, 0xB753 }, { 2419, 0xB756 },
    { 2420, 0xB757 }, { 2421, 0xB758 }, { 2422, 0xB759 }, { 2423, 0xB75A }, { 2424, 0xB75B }, { 2425, 0xB75C }, { 2426, 0xB75D }, { 2427, 0xB75E },
    { 2428, 0xB75F }, { 2429, 0xB761 }, { 2430, 0xB762 }, { 2431, 0xB763 }, { 2432, 0xB765 }, { 2433, 0xB766 }, { 2434, 0xB767 }, { 2435, 0xB769 },
    { 2436, 0xB76A }, { 2437, 0xB76B }, { 2438, 0xB76C }, { 2439, 0xB76D }, { 2440, 0xB76E }, { 2441, 0xB76F }, { 2442, 0xB772 }, { 2443, 0xB774 },
    { 2444, 0xB776 }, { 2445, 0xB777 }, { 2446, 0xB778 }, { 2447, 0xB779 }, { 2448, 0xB77A }, { 2449, 0xB77B }, { 2450, 0xB77E }, { 2451, 0xB77F },
    { 2452, 0xB781 }, { 2453, 0xB782 }, { 2454, 0xB783 }, { 2455, 0xB785 }, { 2456, 0xB786 }, { 2457, 0xB787 }, { 2458, 0xB788 }, { 2459, 0xB789 },
    { 2460, 0xB78A }, { 2461, 0xB78B }, { 2462, 0xB78E }, { 2463, 0xB793 }, { 2464, 0xB794 }, { 2465, 0xB795 }, { 2466, 0xB79A }, { 2467, 0xB79B },
    { 2468, 0xB79D }, { 2469, 0xB79E }, { 2470, 0xB79F }, { 2471, 0xB7A1 }, { 2472, 0xB7A2 }, { 2473, 0xB7A3 }, { 2474, 0xB7A4 }, { 2475, 0xB7A5 },
    { 2476, 0xB7A6 }, { 2477, 0xB7A7 }, { 2478, 0xB7AA }, { 2479, 0xB7AE }, { 2480, 0xB7AF }, { 2481, 0xB7B0 }, { 2482, 0xB7B1 }, { 2483, 0xB7B2 },
    { 2484, 0xB7B3 }, { 2485, 0xB7B6 }, { 2486, 0xB7B7 }, { 2487, 0xB7B9 }, { 2488, 0xB7BA }, { 2489, 0xB7BB }, { 2490, 0xB7BC }, { 2491, 0xB7BD },
    { 2492, 0xB7BE }, { 2493, 0xB7BF }, { 2494, 0xB7C0 }, { 2495, 0xB7C1 }, { 2502, 0xB7C2 }, { 2503, 0xB7C3 }, { 2504, 0xB7C4 }, { 2505, 0xB7C5 },
    { 2506, 0xB7C6 }, { 2507, 0xB7C8 }, { 2508, 0xB7CA }, { 2509, 0xB7CB }, { 2510, 0xB7CC }, { 2511, 0xB7CD }, { 2512, 0xB7CE }, { 2513, 0xB7CF },
    { 2514, 0xB7D0 }, { 2515, 0xB7D1 }, { 2516, 0xB7D2 }, { 2517, 0xB7D3 }, { 2518, 0xB7D4 }, { 2519, 0xB7D5 }, { 2520, 0xB7D6 }, { 2521, 0xB7D7 },
    { 2522, 0xB7D8 }, { 2523, 0xB7D9 }, { 2524, 0xB7DA }, { 2525, 0xB7DB }, { 2526, 0xB7DC }, { 2527, 0xB7DD }, { 2534, 0xB7DE }, { 2535, 0xB7DF },
    { 2536, 0xB7E0 }, { 2537, 0xB7E1 }, { 2538, 0xB7E2 }, { 2539, 0xB7E3 }, { 2540, 0xB7E4 }, { 2541, 0xB7E5 }, { 2542, 0xB7E6 }, { 2543, 0xB7E7 },
    { 2544, 0xB7E8 }, { 2545, 0xB7E9 }, { 2546, 0xB7EA }, { 2547, 0xB7EB }, { 2548, 0xB7EE }, { 2549, 0xB7EF }, { 2550, 0xB7F1 }, { 2551, 0xB7F2 },
    { 2552, 0xB7F3 }, { 2553, 0xB7F5 }, { 2554, 0xB7F6 }, { 2555, 0xB7F7 }, { 2556, 0xB7F8 }, { 2557, 0xB7F9 }, { 2558, 0xB7FA }, { 2559, 0xB7FB },
    { 2560, 0xB7FE }, { 2561, 0xB802 }, { 2562, 0xB803 }, { 2563, 0xB804 }, { 2564, 0xB805 }, { 2565, 0xB806 }, { 2566, 0xB80A }, { 2567, 0xB80B },
    { 2568, 0xB80D }, { 2569, 0xB80E }, { 2570, 0xB80F }, { 2571, 0xB811 }, { 2572, 0xB812 }, { 2573, 0xB813 }, { 2574, 0xB814 }, { 2575, 0xB815 },
    { 2576, 0xB816 }, { 2577, 0xB817 }, { 2578, 0xB81A }, { 2579, 0xB81C }, { 2580, 0xB81E }, { 2581, 0xB81F }, { 2582, 0xB820 }, { 2583, 0xB821 },
    { 2584, 0xB822 }, { 2585, 0xB823 }, { 2586, 0xB826 }, { 2587, 0xB827 }, { 2588, 0xB829 }, { 2589, 0xB82A }, { 2590, 0xB82B }, { 2591, 0xB82D },
    { 2592, 0xB82E }, { 2593, 0xB82F }, { 2594, 0xB830 }, { 2595, 0xB831 }, { 2596, 0xB832 }, { 2597, 0xB833 }, { 2598, 0xB836 }, { 2599, 0xB83A },
    { 2600, 0xB83B }, { 2601, 0xB83C }, { 2602, 0xB83D }, { 2603, 0xB83E }, { 2604, 0xB83F }, { 2605, 0xB841 }, { 2606, 0xB842 }, { 2607, 0xB843 },
    { 2608, 0xB845 }, { 2609, 0xB846 }, { 2610, 0xB847 }, { 2611, 0xB848 }, { 2612, 0xB849 }, { 2613, 0xB84A }, { 2614, 0xB84B }, { 2615, 0xB84C },
    { 2616, 0xB84D }, { 2617, 0xB84E }, { 2618, 0xB84F }, { 2619, 0xB850 }, { 2620, 0xB852 }, { 2621, 0xB854 }, { 2622, 0xB855 }, { 2623, 0xB856 },
    { 2624, 0xB857 }, { 2625, 0xB858 }, { 2626, 0xB859 }, { 2627, 0xB85A }, { 2628, 0xB85B }, { 2629, 0xB85E }, { 2630, 0xB85F }, { 2631, 0xB861 },
    { 2632, 0xB862 }, { 2633, 0xB863 }, { 2634, 0xB865 }, { 2635, 0xB866 }, { 2636, 0xB867 }, { 2637, 0xB868 }, { 2638, 0xB869 }, { 2639, 0xB86A },
    { 2640, 0xB86B }, { 2641, 0xB86E }, { 2642, 0xB870 }, { 2643, 0xB872 }, { 2644, 0xB873 }, { 2645, 0xB874 }, { 2646, 0xB875 }, { 2647, 0xB876 },
    { 2648, 0xB877 }, { 2649, 0xB879 }, { 2650, 0xB87A }, { 2651, 0xB87B }, { 2652, 0xB87D }, { 2653, 0xB87E }, { 2654, 0xB87F }, { 2655, 0xB880 },
    { 2656, 0xB881 }, { 2657, 0xB882 }, { 2658, 0xB883 }, { 2659, 0xB884 }, { 2660, 0xB885 }, { 2661, 0xB886 }, { 2662, 0xB887 }, { 2663, 0xB888 },
    { 2664, 0xB889 }, { 2665, 0xB88A }, { 2666, 0xB88B }, { 2667, 0xB88C }, { 2668, 0xB88E }, { 2669, 0xB88F }, { 2670, 0xB890 }, { 2671, 0xB891 },
    { 2672, 0xB892 }, { 2673, 0xB893 }, { 2674, 0xB894 }, { 2675, 0xB895 }, { 2676, 0xB896 }, { 2677, 0xB897 }, { 2678, 0xB898 }, { 2679, 0xB899 },
    { 2680, 0xB89A }, { 2681, 0xB89B }, { 2682, 0xB89C }, { 2683, 0xB89D }, { 2684, 0xB89E }, { 2685, 0xB89F }, { 2692, 0xB8A0 }, { 2693, 0xB8A1 },
    { 2694, 0xB8A2 }, { 2695, 0xB8A3 }, { 2696, 0xB8A4 }, { 2697, 0xB8A5 }, { 2698, 0xB8A6 }, { 2699, 0xB8A7 }, { 2700, 0xB8A9 }, { 2701, 0xB8AA },
    { 2702, 0xB8AB }, { 2703, 0xB8AC }, { 2704, 0xB8AD }, { 2705, 0xB8AE }, { 2706, 0xB8AF }, { 2707, 0xB8B1 }, { 2708, 0xB8B2 }, { 2709, 0xB8B3 },
    { 2710, 0xB8B5 }, { 2711, 0xB8B6 }, { 2712, 0xB8B7 }, { 2713, 0xB8B9 }, { 2714, 0xB8BA }, { 2715, 0xB8BB }, { 2716, 0xB8BC }, { 2717, 0xB8BD },
    { 2724, 0xB8BE }, { 2725, 0xB8BF }, { 2726, 0xB8C2 }, { 2727, 0xB8C4 }, { 2728, 0xB8C6 }, { 2729, 0xB8C7 }, { 2730, 0xB8C8 }, { 2731, 0xB8C9 },
    { 2732, 0xB8CA }, { 2733, 0xB8CB }, { 2734, 0xB8CD }, { 2735, 0xB8CE }, { 2736, 0xB8CF }, { 2737, 0xB8D1 }, { 2738, 0xB8D2 }, { 2739, 0xB8D3 },
    { 2740, 0xB8D5 }, { 2741, 0xB8D6 }, { 2742, 0xB8D7 }, { 2743, 0xB8D8 }, { 2744, 0xB8D9 }, { 2745, 0xB8DA }, { 2746, 0xB8DB }, { 2747, 0xB8DC },
    { 2748, 0xB8DE }, { 2749, 0xB8E0 }, { 2750, 0xB8E2 }, { 2751, 0xB8E3 }, { 2752, 0xB8E4 }, { 2753, 0xB8E5 }, { 2754, 0xB8E6 }, { 2755, 0xB8E7 },
    { 2756, 0xB8EA }, { 2757, 0xB8EB }, { 2758, 0xB8ED }, { 2759, 0xB8EE }, { 2760, 0xB8EF }, { 2761, 0xB8F1 }, { 2762, 0xB8F2 }, { 2763, 0xB8F3 },
    { 2764, 0xB8F4 }, { 2765, 0xB8F5 }, { 2766, 0xB8F6 }, { 2767, 0xB8F7 }, { 2768, 0xB8FA }, { 2769, 0xB8FC }, { 2770, 0xB8FE }, { 2771, 0xB8FF },
    { 2772, 0xB900 }, { 2773, 0xB901 }, { 2774, 0xB902 }, { 2775, 0xB903 }, { 2776, 0xB905 }, { 2777, 0xB906 }, { 2778, 0xB907 }, { 2779, 0xB908 },
    { 2780, 0xB909 }, { 2781, 0xB90A }, { 2782, 0xB90B }, { 2783, 0xB90C }, { 2784, 0xB90D }, { 2785, 0xB90E }, { 2786, 0xB90F }, { 2787, 0xB910 },
    { 2788, 0xB911 }, { 2789, 0xB912 }, { 2790, 0xB913 }, { 2791, 0xB914 }, { 2792, 0xB915 }, { 2793, 0xB916 }, { 2794, 0xB917 }, { 2795, 0xB919 },
    { 2796, 0xB91A }, { 2797, 0xB91B }, { 2798, 0xB91C }, { 2799, 0xB91D }, { 2800, 0xB91E }, { 2801, 0xB91F }, { 2802, 0xB921 }, { 2803, 0xB922 },
    { 2804, 0xB923 }, { 2805, 0xB924 }, { 2806, 0xB925 }, { 2807, 0xB926 }, { 2808, 0xB927 }, { 2809, 0xB928 }, { 2810, 0xB929 }, { 2811, 0xB92A },
    { 2812, 0xB92B }, { 2813, 0xB92C }, { 2814, 0xB92D }, { 2815, 0xB92E }, { 2816, 0xB92F }, { 2817, 0xB930 }, { 2818, 0xB931 }, { 2819, 0xB932 },
    { 2820, 0xB933 }, { 2821, 0xB934 }, { 2822, 0xB935 }, { 2823, 0xB936 }, { 2824, 0xB937 }, { 2825, 0xB938 }, { 2826, 0xB939 }, { 2827, 0xB93A },
    { 2828, 0xB93B }, { 2829, 0xB93E }, { 2830, 0xB93F }, { 2831, 0xB941 }, { 2832, 0xB942 }, { 2833, 0xB943 }, { 2834, 0xB945 }, { 2835, 0xB946 },
    { 2836, 0xB947 }, { 2837, 0xB948 }, { 2838, 0xB949 }, { 2839, 0xB94A }, { 2840, 0xB94B }, { 2841, 0xB94D }, { 2842, 0xB94E }, { 2843, 0xB950 },
    { 2844, 0xB952 }, { 2845, 0xB953 }, { 2846, 0xB954 }, { 2847, 0xB955 }, { 2848, 0xB956 }, { 2849, 0xB957 }, { 2850, 0xB95A }, { 2851, 0xB95B },
    { 2852, 0xB95D }, { 2853, 0xB95E }, { 2854, 0xB95F }, { 2855, 0xB961 }, { 2856, 0xB962 }, { 2857, 0xB963 }, { 2858, 0xB964 }, { 2859, 0xB965 },
    { 2860, 0xB966 }, { 2861, 0xB967 }, { 2862, 0xB96A }, { 2863, 0xB96C }, { 2864, 0xB96E }, { 2865, 0xB96F }, { 2866, 0xB970 }, { 2867, 0xB971 },
    { 2868, 0xB972 }, { 2869, 0xB973 }, { 2870, 0xB976 }, { 2871, 0xB977 }, { 2872, 0xB979 }, { 2873, 0xB97A }, { 2874, 0xB97B }, { 2875, 0xB97D },
    { 2882, 0xB97E }, { 2883, 0xB97F }, { 2884, 0xB980 }, { 2885, 0xB981 }, { 2886, 0xB982 }, { 2887, 0xB983 }, { 2888, 0xB986 }, { 2889, 0xB988 },
    { 2890, 0xB98B }, { 2891, 0xB98C }, { 2892, 0xB98F }, { 2893, 0xB990 }, { 2894, 0xB991 }, { 2895, 0xB992 }, { 2896, 0xB993 }, { 2897, 0xB994 },
    { 2898, 0xB995 }, { 2899, 0xB996 }, { 2900, 0xB997 }, { 2901, 0xB998 }, { 2902, 0xB999 }, { 2903, 0xB99A }, { 2904, 0xB99B }, { 2905, 0xB99C },
    { 2906, 0xB99D }, { 2907, 0xB99E }, { 2914, 0xB99F }, { 2915, 0xB9A0 }, { 2916, 0xB9A1 }, { 2917, 0xB9A2 }, { 2918, 0xB9A3 }, { 2919, 0xB9A4 },
    { 2920, 0xB9A5 }, { 2921, 0xB9A6 }, { 2922, 0xB9A7 }, { 2923, 0xB9A8 }, { 2924, 0xB9A9 }, { 2925, 0xB9AA }, { 2926, 0xB9AB }, { 2927, 0xB9AE },
    { 2928, 0xB9AF }, { 2929, 0xB9B1 }, { 2930, 0xB9B2 }, { 2931, 0xB9B3 }, { 2932, 0xB9B5 }, { 2933, 0xB9B6 }, { 2934, 0xB9B7 }, { 2935, 0xB9B8 },
    { 2936, 0xB9B9 }, { 2937, 0xB9BA }, { 2938, 0xB9BB }, { 2939, 0xB9BE }, { 2940, 0xB9C0 }, { 2941, 0xB9C2 }, { 2942, 0xB9C3 }, { 2943, 0xB9C4 },
    { 2944, 0xB9C5 }, { 2945, 0xB9C6 }, { 2946, 0xB9C7 }, { 2947, 0xB9CA }, { 2948, 0xB9CB }, { 2949, 0xB9CD }, { 2950, 0xB9D3 }, { 2951, 0xB9D4 },
    { 2952, 0xB9D5 }, { 2953, 0xB9D6 }, { 2954, 0xB9D7 }, { 2955, 0xB9DA }, { 2956, 0xB9DC }, { 2957, 0xB9DF }, { 2958, 0xB9E0 }, { 2959, 0xB9E2 },
    { 2960, 0xB9E6 }, { 2961, 0xB9E7 }, { 2962, 0xB9E9 }, { 2963, 0xB9EA }, { 2964, 0xB9EB }, { 2965, 0xB9ED }, { 2966, 0xB9EE }, { 2967, 0xB9EF },
    { 2968, 0xB9F0 }, { 2969, 0xB9F1 }, { 2970, 0xB9F2 }, { 2971, 0xB9F3 }, { 2972, 0xB9F6 }, { 2973, 0xB9FB }, { 2974, 0xB9FC }, { 2975, 0xB9FD },
    { 2976, 0xB9FE }, { 2977, 0xB9FF }, { 2978, 0xBA02 }, { 2979, 0xBA03 }, { 2980, 0xBA04 }, { 2981, 0xBA05 }, { 2982, 0xBA06 }, { 2983, 0xBA07 },
    { 2984, 0xBA09 }, { 2985, 0xBA0A }, { 2986, 0xBA0B }, { 2987, 0xBA0C }, { 2988, 0xBA0D }, { 2989, 0xBA0E }, { 2990, 0xBA0F }, { 2991, 0xBA10 },
    { 2992, 0xBA11 }, { 2993, 0xBA12 }, { 2994, 0xBA13 }, { 2995, 0xBA14 }, { 2996, 0xBA16 }, { 2997, 0xBA17 }, { 2998, 0xBA18 }, { 2999, 0xBA19 },
    { 3000, 0xBA1A }, { 3001, 0xBA1B }, { 3002, 0xBA1C }, { 3003, 0xBA1D }, { 3004, 0xBA1E }, { 3005, 0xBA1F }, { 3006, 0xBA20 }, { 3007, 0xBA21 },
    { 3008, 0xBA22 }, { 3009, 0xBA23 }, { 3010, 0xBA24 }, { 3011, 0xBA25 }, { 3012, 0xBA26 }, { 3013, 0xBA27 }, { 3014, 0xBA28 }, { 3015, 0xBA29 },
    { 3016, 0xBA2A }, { 3017, 0xBA2B }, { 3018, 0xBA2C }, { 3019, 0xBA2D }, { 3020, 0xBA2E }, { 3021, 0xBA2F }, { 3022, 0xBA30 }, { 3023, 0xBA31 },
    { 3024, 0xBA32 }, { 3025, 0xBA33 }, { 3026, 0xBA34 }, { 3027, 0xBA35 }, { 3028, 0xBA36 }, { 3029, 0xBA37 }, { 3030, 0xBA3A }, { 3031, 0xBA3B },
    { 3032, 0xBA3D }, { 3033, 0xBA3E }, { 3034, 0xBA3F }, { 3035, 0xBA41 }, { 3036, 0xBA43 }, { 3037, 0xBA44 }, { 3038, 0xBA45 }, { 3039, 0xBA46 },
    { 3040, 0xBA47 }, { 3041, 0xBA4A }, { 3042, 0xBA4C }, { 3043, 0xBA4F }, { 3044, 0xBA50 }, { 3045, 0xBA51 }, { 3046, 0xBA52 }, { 3047, 0xBA56 },
    { 3048, 0xBA57 }, { 3049, 0xBA59 }, { 3050, 0xBA5A }, { 3051, 0xBA5B }, { 3052, 0xBA5D }, { 3053, 0xBA5E }, { 3054, 0xBA5F }, { 3055, 0xBA60 },
    { 3056, 0xBA61 }, { 3057, 0xBA62 }, { 3058, 0xBA63 }, { 3059, 0xBA66 }, { 3060, 0xBA6A }, { 3061, 0xBA6B }, { 3062, 0xBA6C }, { 3063, 0xBA6D },
    { 3064, 0xBA6E }, { 3065, 0xBA6F }, { 3072, 0xBA72 }, { 3073, 0xBA73 }, { 3074, 0xBA75 }, { 3075, 0xBA76 }, { 3076, 0xBA77 }, { 3077, 0xBA79 },
    { 3078, 0xBA7A }, { 3079, 0xBA7B }, { 3080, 0xBA7C }, { 3081, 0xBA7D }, { 3082, 0xBA7E }, { 3083, 0xBA7F }, { 3084, 0xBA80 }, { 3085, 0xBA81 },
    { 3086, 0xBA82 }, { 3087, 0xBA86 }, { 3088, 0xBA88 }, { 3089, 0xBA89 }, { 3090, 0xBA8A }, { 3091, 0xBA8B }, { 3092, 0xBA8D }, { 3093, 0xBA8E },
    { 3094, 0xBA8F }, { 3095, 0xBA90 }, { 3096, 0xBA91 }, { 3097, 0xBA92 }, { 3104, 0xBA93 }, { 3105, 0xBA94 }, { 3106, 0xBA95 }, { 3107, 0xBA96 },
    { 3108, 0xBA97 }, { 3109, 0xBA98 }, { 3110, 0xBA99 }, { 3111, 0xBA9A }, { 3112, 0xBA9B }, { 3113, 0xBA9C }, { 3114, 0xBA9D }, { 3115, 0xBA9E },
    { 3116, 0xBA9F }, { 3117, 0xBAA0 }, { 3118, 0xBAA1 }, { 3119, 0xBAA2 }, { 3120, 0xBAA3 }, { 3121, 0xBAA4 }, { 3122, 0xBAA5 }, { 3123, 0xBAA6 },
    { 3124, 0xBAA7 }, { 3125, 0xBAAA }, { 3126, 0xBAAD }, { 3127, 0xBAAE }, { 3128, 0xBAAF }, { 3129, 0xBAB1 }, { 3130, 0xBAB3 }, { 3131, 0xBAB4 },
    { 3132, 0xBAB5 }, { 3133, 0xBAB6 }, { 3134, 0xBAB7 }, { 3135, 0xBABA }, { 3136, 0xBABC }, { 3137, 0xBABE }, { 3138, 0xBABF }, { 3139, 0xBAC0 },
    { 3140, 0xBAC1 }, { 3141, 0xBAC2 }, { 3142, 0xBAC3 }, { 3143, 0xBAC5 }, { 3144, 0xBAC6 }, { 3145, 0xBAC7 }, { 3146, 0xBAC9 }, { 3147, 0xBACA },
    { 3148, 0xBACB }, { 3149, 0xBACC }, { 3150, 0xBACD }, { 3151, 0xBACE }, { 3152, 0xBACF }, { 3153, 0xBAD0 }, { 3154, 0xBAD1 }, { 3155, 0xBAD2 },
    { 3156, 0xBAD3 }, { 3157, 0xBAD4 }, { 3158, 0xBAD5 }, { 3159, 0xBAD6 }, { 3160, 0xBAD7 }, { 3161, 0xBADA }, { 3162, 0xBADB }, { 3163, 0xBADC },
    { 3164, 0xBADD }, { 3165, 0xBADE }, { 3166, 0xBADF }, { 3167, 0xBAE0 }, { 3168, 0xBAE1 }, { 3169, 0xBAE2 }, { 3170, 0xBAE3 }, { 3171, 0xBAE4 },
    { 3172, 0xBAE5 }, { 3173, 0xBAE6 }, { 3174, 0xBAE7 }, { 3175, 0xBAE8 }, { 3176, 0xBAE9 }, { 3177, 0xBAEA }, { 3178, 0xBAEB }, { 3179, 0xBAEC },
    { 3180, 0xBAED }, { 3181, 0xBAEE }, { 3182, 0xBAEF }, { 3183, 0xBAF0 }, { 3184, 0xBAF1 }, { 3185, 0xBAF2 }, { 3186, 0xBAF3 }, { 3187, 0xBAF4 },
    { 3188, 0xBAF5 }, { 3189, 0xBAF6 }, { 3190, 0xBAF7 }, { 3191, 0xBAF8 }, { 3192, 0xBAF9 }, { 3193, 0xBAFA }, { 3194, 0xBAFB }, { 3195, 0xBAFD },
    { 3196, 0xBAFE }, { 3197, 0xBAFF }, { 3198, 0xBB01 }, { 3199, 0xBB02 }, { 3200, 0xBB03 }, { 3201, 0xBB05 }, { 3202, 0xBB06 }, { 3203, 0xBB07 },
    { 3204, 0xBB08 }, { 3205, 0xBB09 }, { 3206, 0xBB0A }, { 3207, 0xBB0B }, { 3208, 0xBB0C }, { 3209, 0xBB0E }, { 3210, 0xBB10 }, { 3211, 0xBB12 },
    { 3212, 0xBB13 }, { 3213, 0xBB14 }, { 3214, 0xBB15 }, { 3215, 0xBB16 }, { 3216, 0xBB17 }, { 3217, 0xBB19 }, { 3218, 0xBB1A }, { 3219, 0xBB1B },
    { 3220, 0xBB1D }, { 3221, 0xBB1E }, { 3222, 0xBB1F }, { 3223, 0xBB21 }, { 3224, 0xBB22 }, { 3225, 0xBB23 }, { 3226, 0xBB24 }, { 3227, 0xBB25 },
    { 3228, 0xBB26 }, { 3229, 0xBB27 }, { 3230, 0xBB28 }, { 3231, 0xBB2A }, { 3232, 0xBB2C }, { 3233, 0xBB2D }, { 3234, 0xBB2E }, { 3235, 0xBB2F },
    { 3236, 0xBB30 }, { 3237, 0xBB31 }, { 3238, 0xBB32 }, { 3239, 0xBB33 }, { 3240, 0xBB37 }, { 3241, 0xBB39 }, { 3242, 0xBB3A }, { 3243, 0xBB3F },
    { 3244, 0xBB40 }, { 3245, 0xBB41 }, { 3246, 0xBB42 }, { 3247, 0xBB43 }, { 3248, 0xBB46 }, { 3249, 0xBB48 }, { 3250, 0xBB4A }, { 3251, 0xBB4B },
    { 3252, 0xBB4C }, { 3253, 0xBB4E }, { 3254, 0xBB51 }, { 3255, 0xBB52 }, { 3262, 0xBB53 }, { 3263, 0xBB55 }, { 3264, 0xBB56 }, { 3265, 0xBB57 },
    { 3266, 0xBB59 }, { 3267, 0xBB5A }, { 3268, 0xBB5B }, { 3269, 0xBB5C }, { 3270, 0xBB5D }, { 3271, 0xBB5E }, { 3272, 0xBB5F }, { 3273, 0xBB60 },
    { 3274, 0xBB62 }, { 3275, 0xBB64 }, { 3276, 0xBB65 }, { 3277, 0xBB66 }, { 3278, 0xBB67 }, { 3279, 0xBB68 }, { 3280, 0xBB69 }, { 3281, 0xBB6A },
    { 3282, 0xBB6B }, { 3283, 0xBB6D }, { 3284, 0xBB6E }, { 3285, 0xBB6F }, { 3286, 0xBB70 }, { 3287, 0xBB71 }, { 3294, 0xBB72 }, { 3295, 0xBB73 },
    { 3296, 0xBB74 }, { 3297, 0xBB75 }, { 3298, 0xBB76 }, { 3299, 0xBB77 }, { 3300, 0xBB78 }, { 3301, 0xBB79 }, { 3302, 0xBB7A }, { 3303, 0xBB7B },
    { 3304, 0xBB7C }, { 3305, 0xBB7D }, { 3306, 0xBB7E }, { 3307, 0xBB7F }, { 3308, 0xBB80 }, { 3309, 0xBB81 }, { 3310, 0xBB82 }, { 3311, 0xBB83 },
    { 3312, 0xBB84 }, { 3313, 0xBB85 }, { 3314, 0xBB86 }, { 3315, 0xBB87 }, { 3316, 0xBB89 }, { 3317, 0xBB8A }, { 3318, 0xBB8B }, { 3319, 0xBB8D },
    { 3320, 0xBB8E }, { 3321, 0xBB8F }, { 3322, 0xBB91 }, { 3323, 0xBB92 }, { 3324, 0xBB93 }, { 3325, 0xBB94 }, { 3326, 0xBB95 }, { 3327, 0xBB96 },
    { 3328, 0xBB97 }, { 3329, 0xBB98 }, { 3330, 0xBB99 }, { 3331, 0xBB9A }, { 3332, 0xBB9B }, { 3333, 0xBB9C }, { 3334, 0xBB9D }, { 3335, 0xBB9E },
    { 3336, 0xBB9F }, { 3337, 0xBBA0 }, { 3338, 0xBBA1 }, { 3339, 0xBBA2 }, { 3340, 0xBBA3 }, { 3341, 0xBBA5 }, { 3342, 0xBBA6 }, { 3343, 0xBBA7 },
    { 3344, 0xBBA9 }, { 3345, 0xBBAA }, { 3346, 0xBBAB }, { 3347, 0xBBAD }, { 3348, 0xBBAE }, { 3349, 0xBBAF }, { 3350, 0xBBB0 }, { 3351, 0xBBB1 },
    { 3352, 0xBBB2 }, { 3353, 0xBBB3 }, { 3354, 0xBBB5 }, { 3355, 0xBBB6 }, { 3356, 0xBBB8 }, { 3357, 0xBBB9 }, { 3358, 0xBBBA }, { 3359, 0xBBBB },
    { 3360, 0xBBBC }, { 3361, 0xBBBD }, { 3362, 0xBBBE }, { 3363, 0xBBBF }, { 3364, 0xBBC1 }, { 3365, 0xBBC2 }, { 3366, 0xBBC3 }, { 3367, 0xBBC5 },
    { 3368, 0xBBC6 }, { 3369, 0xBBC7 }, { 3370, 0xBBC9 }, { 3371, 0xBBCA }, { 3372, 0xBBCB }, { 3373, 0xBBCC }, { 3374, 0xBBCD }, { 3375, 0xBBCE },
    { 3376, 0xBBCF }, { 3377, 0xBBD1 }, { 3378, 0xBBD2 }, { 3379, 0xBBD4 }, { 3380, 0xBBD5 }, { 3381, 0xBBD6 }, { 3382, 0xBBD7 }, { 3383, 0xBBD8 },
    { 3384, 0xBBD9 }, { 3385, 0xBBDA }, { 3386, 0xBBDB }, { 3387, 0xBBDC }, { 3388, 0xBBDD }, { 3389, 0xBBDE }, { 3390, 0xBBDF }, { 3391, 0xBBE0 },
    { 3392, 0xBBE1 }, { 3393, 0xBBE2 }, { 3394, 0xBBE3 }, { 3395, 0xBBE4 }, { 3396, 0xBBE5 }, { 3397, 0xBBE6 }, { 3398, 0xBBE7 }, { 3399, 0xBBE8 },
    { 3400, 0xBBE9 }, { 3401, 0xBBEA }, { 3402, 0xBBEB }, { 3403, 0xBBEC }, { 3404, 0xBBED }, { 3405, 0xBBEE }, { 3406, 0xBBEF }, { 3407, 0xBBF0 },
    { 3408, 0xBBF1 }, { 3409, 0xBBF2 }, { 3410, 0xBBF3 }, { 3411, 0xBBF4 }, { 3412, 0xBBF5 }, { 3413, 0xBBF6 }, { 3414, 0xBBF7 }, { 3415, 0xBBFA },
    { 3416, 0xBBFB }, { 3417, 0xBBFD }, { 3418, 0xBBFE }, { 3419, 0xBC01 }, { 3420, 0xBC03 }, { 3421, 0xBC04 }, { 3422, 0xBC05 }, { 3423, 0xBC06 },
    { 3424, 0xBC07 }, { 3425, 0xBC0A }, { 3426, 0xBC0E }, { 3427, 0xBC10 }, { 3428, 0xBC12 }, { 3429, 0xBC13 }, { 3430, 0xBC19 }, { 3431, 0xBC1A },
    { 3432, 0xBC20 }, { 3433, 0xBC21 }, { 3434, 0xBC22 }, { 3435, 0xBC23 }, { 3436, 0xBC26 }, { 3437, 0xBC28 }, { 3438, 0xBC2A }, { 3439, 0xBC2B },
    { 3440, 0xBC2C }, { 3441, 0xBC2E }, { 3442, 0xBC2F }, { 3443, 0xBC32 }, { 3444, 0xBC33 }, { 3445, 0xBC35 }, { 3452, 0xBC36 }, { 3453, 0xBC37 },
    { 3454, 0xBC39 }, { 3455, 0xBC3A }, { 3456, 0xBC3B }, { 3457, 0xBC3C }, { 3458, 0xBC3D }, { 3459, 0xBC3E }, { 3460, 0xBC3F }, { 3461, 0xBC42 },
    { 3462, 0xBC46 }, { 3463, 0xBC47 }, { 3464, 0xBC48 }, { 3465, 0xBC4A }, { 3466, 0xBC4B }, { 3467, 0xBC4E }, { 3468, 0xBC4F }, { 3469, 0xBC51 },
    { 3470, 0xBC52 }, { 3471, 0xBC53 }, { 3472, 0xBC54 }, { 3473, 0xBC55 }, { 3474, 0xBC56 }, { 3475, 0xBC57 }, { 3476, 0xBC58 }, { 3477, 0xBC59 },
    { 3484, 0xBC5A }, { 3485, 0xBC5B }, { 3486, 0xBC5C }, { 3487, 0xBC5E }, { 3488, 0xBC5F }, { 3489, 0xBC60 }, { 3490, 0xBC61 }, { 3491, 0xBC62 },
    { 3492, 0xBC63 }, { 3493, 0xBC64 }, { 3494, 0xBC65 }, { 3495, 0xBC66 }, { 3496, 0xBC67 }, { 3497, 0xBC68 }, { 3498, 0xBC69 }, { 3499, 0xBC6A },
    { 3500, 0xBC6B }, { 3501, 0xBC6C }, { 3502, 0xBC6D }, { 3503, 0xBC6E }, { 3504, 0xBC6F }, { 3505, 0xBC70 }, { 3506, 0xBC71 }, { 3507, 0xBC72 },
    { 3508, 0xBC73 }, { 3509, 0xBC74 }, { 3510, 0xBC75 }, { 3511, 0xBC76 }, { 3512, 0xBC77 }, { 3513, 0xBC78 }, { 3514, 0xBC79 }, { 3515, 0xBC7A },
    { 3516, 0xBC7B }, { 3517, 0xBC7C }, { 3518, 0xBC7D }, { 3519, 0xBC7E }, { 3520, 0xBC7F }, { 3521, 0xBC80 }, { 3522, 0xBC81 }, { 3523, 0xBC82 },
    { 3524, 0xBC83 }, { 3525, 0xBC86 }, { 3526, 0xBC87 }, { 3527, 0xBC89 }, { 3528, 0xBC8A }, { 3529, 0xBC8D }, { 3530, 0xBC8F }, { 3531, 0xBC90 },
    { 3532, 0xBC91 }, { 3533, 0xBC92 }, { 3534, 0xBC93 }, { 3535, 0xBC96 }, { 3536, 0xBC98 }, { 3537, 0xBC9B }, { 3538, 0xBC9C }, { 3539, 0xBC9D },
    { 3540, 0xBC9E }, { 3541, 0xBC9F }, { 3542, 0xBCA2 }, { 3543, 0xBCA3 }, { 3544, 0xBCA5 }, { 3545, 0xBCA6 }, { 3546, 0xBCA9 }, { 3547, 0xBCAA },
    { 3548, 0xBCAB }, { 3549, 0xBCAC }, { 3550, 0xBCAD }, { 3551, 0xBCAE }, { 3552, 0xBCAF }, { 3553, 0xBCB2 }, { 3554, 0xBCB6 }, { 3555, 0xBCB7 },
    { 3556, 0xBCB8 }, { 3557, 0xBCB9 }, { 3558, 0xBCBA }, { 3559, 0xBCBB }, { 3560, 0xBCBE }, { 3561, 0xBCBF }, { 3562, 0xBCC1 }, { 3563, 0xBCC2 },
    { 3564, 0xBCC3 }, { 3565, 0xBCC5 }, { 3566, 0xBCC6 }, { 3567, 0xBCC7 }, { 3568, 0xBCC8 }, { 3569, 0xBCC9 }, { 3570, 0xBCCA }, { 3571, 0xBCCB },
    { 3572, 0xBCCC }, { 3573, 0xBCCE }, { 3574, 0xBCD2 }, { 3575, 0xBCD3 }, { 3576, 0xBCD4 }, { 3577, 0xBCD6 }, { 3578, 0xBCD7 }, { 3579, 0xBCD9 },
    { 3580, 0xBCDA }, { 3581, 0xBCDB }, { 3582, 0xBCDD }, { 3583, 0xBCDE }, { 3584, 0xBCDF }, { 3585, 0xBCE0 }, { 3586, 0xBCE1 }, { 3587, 0xBCE2 },
    { 3588, 0xBCE3 }, { 3589, 0xBCE4 }, { 3590, 0xBCE5 }, { 3591, 0xBCE6 }, { 3592, 0xBCE7 }, { 3593, 0xBCE8 }, { 3594, 0xBCE9 }, { 3595, 0xBCEA },
    { 3596, 0xBCEB }, { 3597, 0xBCEC }, { 3598, 0xBCED }, { 3599, 0xBCEE }, { 3600, 0xBCEF }, { 3601, 0xBCF0 }, { 3602, 0xBCF1 }, { 3603, 0xBCF2 },
    { 3604, 0xBCF3 }, { 3605, 0xBCF7 }, { 3606, 0xBCF9 }, { 3607, 0xBCFA }, { 3608, 0xBCFB }, { 3609, 0xBCFD }, { 3610, 0xBCFE }, { 3611, 0xBCFF },
    { 3612, 0xBD00 }, { 3613, 0xBD01 }, { 3614, 0xBD02 }, { 3615, 0xBD03 }, { 3616, 0xBD06 }, { 3617, 0xBD08 }, { 3618, 0xBD0A }, { 3619, 0xBD0B },
    { 3620, 0xBD0C }, { 3621, 0xBD0D }, { 3622, 0xBD0E }, { 3623, 0xBD0F }, { 3624, 0xBD11 }, { 3625, 0xBD12 }, { 3626, 0xBD13 }, { 3627, 0xBD15 },
    { 3628, 0xBD16 }, { 3629, 0xBD17 }, { 3630, 0xBD18 }, { 3631, 0xBD19 }, { 3632, 0xBD1A }, { 3633, 0xBD1B }, { 3634, 0xBD1C }, { 3635, 0xBD1D },
    { 3642, 0xBD1E }, { 3643, 0xBD1F }, { 3644, 0xBD20 }, { 3645, 0xBD21 }, { 3646, 0xBD22 }, { 3647, 0xBD23 }, { 3648, 0xBD25 }, { 3649, 0xBD26 },
    { 3650, 0xBD27 }, { 3651, 0xBD28 }, { 3652, 0xBD29 }, { 3653, 0xBD2A }, { 3654, 0xBD2B }, { 3655, 0xBD2D }, { 3656, 0xBD2E }, { 3657, 0xBD2F },
    { 3658, 0xBD30 }, { 3659, 0xBD31 }, { 3660, 0xBD32 }, { 3661, 0xBD33 }, { 3662, 0xBD34 }, { 3663, 0xBD35 }, { 3664, 0xBD36 }, { 3665, 0xBD37 },
    { 3666, 0xBD38 }, { 3667, 0xBD39 }, { 3674, 0xBD3A }, { 3675, 0xBD3B }, { 3676, 0xBD3C }, { 3677, 0xBD3D }, { 3678, 0xBD3E }, { 3679, 0xBD3F },
    { 3680, 0xBD41 }, { 3681, 0xBD42 }, { 3682, 0xBD43 }, { 3683, 0xBD44 }, { 3684, 0xBD45 }, { 3685, 0xBD46 }, { 3686, 0xBD47 }, { 3687, 0xBD4A },
    { 3688, 0xBD4B }, { 3689, 0xBD4D }, { 3690, 0xBD4E }, { 3691, 0xBD4F }, { 3692, 0xBD51 }, { 3693, 0xBD52 }, { 3694, 0xBD53 }, { 3695, 0xBD54 },
    { 3696, 0xBD55 }, { 3697, 0xBD56 }, { 3698, 0xBD57 }, { 3699, 0xBD5A }, { 3700, 0xBD5B }, { 3701, 0xBD5C }, { 3702, 0xBD5D }, { 3703, 0xBD5E },
    { 3704, 0xBD5F }, { 3705, 0xBD60 }, { 3706, 0xBD61 }, { 3707, 0xBD62 }, { 3708, 0xBD63 }, { 3709, 0xBD65 }, { 3710, 0xBD66 }, { 3711, 0xBD67 },
    { 3712, 0xBD69 }, { 3713, 0xBD6A }, { 3714, 0xBD6B }, { 3715, 0xBD6C }, { 3716, 0xBD6D }, { 3717, 0xBD6E }, { 3718, 0xBD6F }, { 3719, 0xBD70 },
    { 3720, 0xBD71 }, { 3721, 0xBD72 }, { 3722, 0xBD73 }, { 3723, 0xBD74 }, { 3724, 0xBD75 }, { 3725, 0xBD76 }, { 3726, 0xBD77 }, { 3727, 0xBD78 },
    { 3728, 0xBD79 }, { 3729, 0xBD7A }, { 3730, 0xBD7B }, { 3731, 0xBD7C }, { 3732, 0xBD7D }, { 3733, 0xBD7E }, { 3734, 0xBD7F }, { 3735, 0xBD82 },
    { 3736, 0xBD83 }, { 3737, 0xBD85 }, { 3738, 0xBD86 }, { 3739, 0xBD8B }, { 3740, 0xBD8C }, { 3741, 0xBD8D }, { 3742, 0xBD8E }, { 3743, 0xBD8F },
    { 3744, 0xBD92 }, { 3745, 0xBD94 }, { 3746, 0xBD96 }, { 3747, 0xBD97 }, { 3748, 0xBD98 }, { 3749, 0xBD9B }, { 3750, 0xBD9D }, { 3751, 0xBD9E },
    { 3752, 0xBD9F }, { 3753, 0xBDA0 }, { 3754, 0xBDA1 }, { 3755, 0xBDA2 }, { 3756, 0xBDA3 }, { 3757, 0xBDA5 }, { 3758, 0xBDA6 }, { 3759, 0xBDA7 },
    { 3760, 0xBDA8 }, { 3761, 0xBDA9 }, { 3762, 0xBDAA }, { 3763, 0xBDAB }, { 3764, 0xBDAC }, { 3765, 0xBDAD }, { 3766, 0xBDAE }, { 3767, 0xBDAF },
    { 3768, 0xBDB1 }, { 3769, 0xBDB2 }, { 3770, 0xBDB3 }, { 3771, 0xBDB4 }, { 3772, 0xBDB5 }, { 3773, 0xBDB6 }, { 3774, 0xBDB7 }, { 3775, 0xBDB9 },
    { 3776, 0xBDBA }, { 3777, 0xBDBB }, { 3778, 0xBDBC }, { 3779, 0xBDBD }, { 3780, 0xBDBE }, { 3781, 0xBDBF }, { 3782, 0xBDC0 }, { 3783, 0xBDC1 },
    { 3784, 0xBDC2 }, { 3785, 0xBDC3 }, { 3786, 0xBDC4 }, { 3787, 0xBDC5 }, { 3788, 0xBDC6 }, { 3789, 0xBDC7 }, { 3790, 0xBDC8 }, { 3791, 0xBDC9 },
    { 3792, 0xBDCA }, { 3793, 0xBDCB }, { 3794, 0xBDCC }, { 3795, 0xBDCD }, { 3796, 0xBDCE }, { 3797, 0xBDCF }, { 3798, 0xBDD0 }, { 3799, 0xBDD1 },
    { 3800, 0xBDD2 }, { 3801, 0xBDD3 }, { 3802, 0xBDD6 }, { 3803, 0xBDD7 }, { 3804, 0xBDD9 }, { 3805, 0xBDDA }, { 3806, 0xBDDB }, { 3807, 0xBDDD },
    { 3808, 0xBDDE }, { 3809, 0xBDDF }, { 3810, 0xBDE0 }, { 3811, 0xBDE1 }, { 3812, 0xBDE2 }, { 3813, 0xBDE3 }, { 3814, 0xBDE4 }, { 3815, 0xBDE5 },
    { 3816, 0xBDE6 }, { 3817, 0xBDE7 }, { 3818, 0xBDE8 }, { 3819, 0xBDEA }, { 3820, 0xBDEB }, { 3821, 0xBDEC }, { 3822, 0xBDED }, { 3823, 0xBDEE },
    { 3824, 0xBDEF }, { 3825, 0xBDF1 }, { 3832, 0xBDF2 }, { 3833, 0xBDF3 }, { 3834, 0xBDF5 }, { 3835, 0xBDF6 }, { 3836, 0xBDF7 }, { 3837, 0xBDF9 },
    { 3838, 0xBDFA }, { 3839, 0xBDFB }, { 3840, 0xBDFC }, { 3841, 0xBDFD }, { 3842, 0xBDFE }, { 3843, 0xBDFF }, { 3844, 0xBE01 }, { 3845, 0xBE02 },
    { 3846, 0xBE04 }, { 3847, 0xBE06 }, { 3848, 0xBE07 }, { 3849, 0xBE08 }, { 3850, 0xBE09 }, { 3851, 0xBE0A }, { 3852, 0xBE0B }, { 3853, 0xBE0E },
    { 3854, 0xBE0F }, { 3855, 0xBE11 }, { 3856, 0xBE12 }, { 3857, 0xBE13 }, { 3864, 0xBE15 }, { 3865, 0xBE16 }, { 3866, 0xBE17 }, { 3867, 0xBE18 },
    { 3868, 0xBE19 }, { 3869, 0xBE1A }, { 3870, 0xBE1B }, { 3871, 0xBE1E }, { 3872, 0xBE20 }, { 3873, 0xBE21 }, { 3874, 0xBE22 }, { 3875, 0xBE23 },
    { 3876, 0xBE24 }, { 3877, 0xBE25 }, { 3878, 0xBE26 }, { 3879, 0xBE27 }, { 3880, 0xBE28 }, { 3881, 0xBE29 }, { 3882, 0xBE2A }, { 3883, 0xBE2B },
    { 3884, 0xBE2C }, { 3885, 0xBE2D }, { 3886, 0xBE2E }, { 3887, 0xBE2F }, { 3888, 0xBE30 }, { 3889, 0xBE31 }, { 3890, 0xBE32 }, { 3891, 0xBE33 },
    { 3892, 0xBE34 }, { 3893, 0xBE35 }, { 3894, 0xBE36 }, { 3895, 0xBE37 }, { 3896, 0xBE38 }, { 3897, 0xBE39 }, { 3898, 0xBE3A }, { 3899, 0xBE3B },
    { 3900, 0xBE3C }, { 3901, 0xBE3D }, { 3902, 0xBE3E }, { 3903, 0xBE3F }, { 3904, 0xBE40 }, { 3905, 0xBE41 }, { 3906, 0xBE42 }, { 3907, 0xBE43 },
    { 3908, 0xBE46 }, { 3909, 0xBE47 }, { 3910, 0xBE49 }, { 3911, 0xBE4A }, { 3912, 0xBE4B }, { 3913, 0xBE4D }, { 3914, 0xBE4F }, { 3915, 0xBE50 },
    { 3916, 0xBE51 }, { 3917, 0xBE52 }, { 3918, 0xBE53 }, { 3919, 0xBE56 }, { 3920, 0xBE58 }, { 3921, 0xBE5C }, { 3922, 0xBE5D }, { 3923, 0xBE5E },
    { 3924, 0xBE5F }, { 3925, 0xBE62 }, { 3926, 0xBE63 }, { 3927, 0xBE65 }, { 3928, 0xBE66 }, { 3929, 0xBE67 }, { 3930, 0xBE69 }, { 3931, 0xBE6B },
    { 3932, 0xBE6C }, { 3933, 0xBE6D }, { 3934, 0xBE6E }, { 3935, 0xBE6F }, { 3936, 0xBE72 }, { 3937, 0xBE76 }, { 3938, 0xBE77 }, { 3939, 0xBE78 },
    { 3940, 0xBE79 }, { 3941, 0xBE7A }, { 3942, 0xBE7E }, { 3943, 0xBE7F }, { 3944, 0xBE81 }, { 3945, 0xBE82 }, { 3946, 0xBE83 }, { 3947, 0xBE85 },
    { 3948, 0xBE86 }, { 3949, 0xBE87 }, { 3950, 0xBE88 }, { 3951, 0xBE89 }, { 3952, 0xBE8A }, { 3953, 0xBE8B }, { 3954, 0xBE8E }, { 3955, 0xBE92 },
    { 3956, 0xBE93 }, { 3957, 0xBE94 }, { 3958, 0xBE95 }, { 3959, 0xBE96 }, { 3960, 0xBE97 }, { 3961, 0xBE9A }, { 3962, 0xBE9B }, { 3963, 0xBE9C },
    { 3964, 0xBE9D }, { 3965, 0xBE9E }, { 3966, 0xBE9F }, { 3967, 0xBEA0 }, { 3968, 0xBEA1 }, { 3969, 0xBEA2 }, { 3970, 0xBEA3 }, { 3971, 0xBEA4 },
    { 3972, 0xBEA5 }, { 3973, 0xBEA6 }, { 3974, 0xBEA7 }, { 3975, 0xBEA9 }, { 3976, 0xBEAA }, { 3977, 0xBEAB }, { 3978, 0xBEAC }, { 3979, 0xBEAD },
    { 3980, 0xBEAE }, { 3981, 0xBEAF }, { 3982, 0xBEB0 }, { 3983, 0xBEB1 }, { 3984, 0xBEB2 }, { 3985, 0xBEB3 }, { 3986, 0xBEB4 }, { 3987, 0xBEB5 },
    { 3988, 0xBEB6 }, { 3989, 0xBEB7 }, { 3990, 0xBEB8 }, { 3991, 0xBEB9 }, { 3992, 0xBEBA }, { 3993, 0xBEBB }, { 3994, 0xBEBC }, { 3995, 0xBEBD },
    { 3996, 0xBEBE }, { 3997, 0xBEBF }, { 3998, 0xBEC0 }, { 3999, 0xBEC1 }, { 4000, 0xBEC2 }, { 4001, 0xBEC3 }, { 4002, 0xBEC4 }, { 4003, 0xBEC5 },
    { 4004, 0xBEC6 }, { 4005, 0xBEC7 }, { 4006, 0xBEC8 }, { 4007, 0xBEC9 }, { 4008, 0xBECA }, { 4009, 0xBECB }, { 4010, 0xBECC }, { 4011, 0xBECD },
    { 4012, 0xBECE }, { 4013, 0xBECF }, { 4014, 0xBED2 }, { 4015, 0xBED3 }, { 4022, 0xBED5 }, { 4023, 0xBED6 }, { 4024, 0xBED9 }, { 4025, 0xBEDA },
    { 4026, 0xBEDB }, { 4027, 0xBEDC }, { 4028, 0xBEDD }, { 4029, 0xBEDE }, { 4030, 0xBEDF }, { 4031, 0xBEE1 }, { 4032, 0xBEE2 }, { 4033, 0xBEE6 },
    { 4034, 0xBEE7 }, { 4035, 0xBEE8 }, { 4036, 0xBEE9 }, { 4037, 0xBEEA }, { 4038, 0xBEEB }, { 4039, 0xBEED }, { 4040, 0xBEEE }, { 4041, 0xBEEF },
    { 4042, 0xBEF0 }, { 4043, 0xBEF1 }, { 4044, 0xBEF2 }, { 4045, 0xBEF3 }, { 4046, 0xBEF4 }, { 4047, 0xBEF5 }, { 4054, 0xBEF6 }, { 4055, 0xBEF7 },
    { 4056, 0xBEF8 }, { 4057, 0xBEF9 }, { 4058, 0xBEFA }, { 4059, 0xBEFB }, { 4060, 0xBEFC }, { 4061, 0xBEFD }, { 4062, 0xBEFE }, { 4063, 0xBEFF },
    { 4064, 0xBF00 }, { 4065, 0xBF02 }, { 4066, 0xBF03 }, { 4067, 0xBF04 }, { 4068, 0xBF05 }, { 4069, 0xBF06 }, { 4070, 0xBF07 }, { 4071, 0xBF0A },
    { 4072, 0xBF0B }, { 4073, 0xBF0C }, { 4074, 0xBF0D }, { 4075, 0xBF0E }, { 4076, 0xBF0F }, { 4077, 0xBF10 }, { 4078, 0xBF11 }, { 4079, 0xBF12 },
    { 4080, 0xBF13 }, { 4081, 0xBF14 }, { 4082, 0xBF15 }, { 4083, 0xBF16 }, { 4084, 0xBF17 }, { 4085, 0xBF1A }, { 4086, 0xBF1E }, { 4087, 0xBF1F },
    { 4088, 0xBF20 }, { 4089, 0xBF21 }, { 4090, 0xBF22 }, { 4091, 0xBF23 }, { 4092, 0xBF24 }, { 4093, 0xBF25 }, { 4094, 0xBF26 }, { 4095, 0xBF27 },
    { 4096, 0xBF28 }, { 4097, 0xBF29 }, { 4098, 0xBF2A }, { 4099, 0xBF2B }, { 4100, 0xBF2C }, { 4101, 0xBF2D }, { 4102, 0xBF2E }, { 4103, 0xBF2F },
    { 4104, 0xBF30 }, { 4105, 0xBF31 }, { 4106, 0xBF32 }, { 4107, 0xBF33 }, { 4108, 0xBF34 }, { 4109, 0xBF35 }, { 4110, 0xBF36 }, { 4111, 0xBF37 },
    { 4112, 0xBF38 }, { 4113, 0xBF39 }, { 4114, 0xBF3A }, { 4115, 0xBF3B }, { 4116, 0xBF3C }, { 4117, 0xBF3D }, { 4118, 0xBF3E }, { 4119, 0xBF3F },
    { 4120, 0xBF42 }, { 4121, 0xBF43 }, { 4122, 0xBF45 }, { 4123, 0xBF46 }, { 4124, 0xBF47 }, { 4125, 0xBF49 }, { 4126, 0xBF4A }, { 4127, 0xBF4B },
    { 4128, 0xBF4C }, { 4129, 0xBF4D }, { 4130, 0xBF4E }, { 4131, 0xBF4F }, { 4132, 0xBF52 }, { 4133, 0xBF53 }, { 4134, 0xBF54 }, { 4135, 0xBF56 },
    { 4136, 0xBF57 }, { 4137, 0xBF58 }, { 4138, 0xBF59 }, { 4139, 0xBF5A }, { 4140, 0xBF5B }, { 4141, 0xBF5C }, { 4142, 0xBF5D }, { 4143, 0xBF5E },
    { 4144, 0xBF5F }, { 4145, 0xBF60 }, { 4146, 0xBF61 }, { 4147, 0xBF62 }, { 4148, 0xBF63 }, { 4149, 0xBF64 }, { 4150, 0xBF65 }, { 4151, 0xBF66 },
    { 4152, 0xBF67 }, { 4153, 0xBF68 }, { 4154, 0xBF69 }, { 4155, 0xBF6A }, { 4156, 0xBF6B }, { 4157, 0xBF6C }, { 4158, 0xBF6D }, { 4159, 0xBF6E },
    { 4160, 0xBF6F }, { 4161, 0xBF70 }, { 4162, 0xBF71 }, { 4163, 0xBF72 }, { 4164, 0xBF73 }, { 4165, 0xBF74 }, { 4166, 0xBF75 }, { 4167, 0xBF76 },
    { 4168, 0xBF77 }, { 4169, 0xBF78 }, { 4170, 0xBF79 }, { 4171, 0xBF7A }, { 4172, 0xBF7B }, { 4173, 0xBF7C }, { 4174, 0xBF7D }, { 4175, 0xBF7E },
    { 4176, 0xBF7F }, { 4177, 0xBF80 }, { 4178, 0xBF81 }, { 4179, 0xBF82 }, { 4180, 0xBF83 }, { 4181, 0xBF84 }, { 4182, 0xBF85 }, { 4183, 0xBF86 },
    { 4184, 0xBF87 }, { 4185, 0xBF88 }, { 4186, 0xBF89 }, { 4187, 0xBF8A }, { 4188, 0xBF8B }, { 4189, 0xBF8C }, { 4190, 0xBF8D }, { 4191, 0xBF8E },
    { 4192, 0xBF8F }, { 4193, 0xBF90 }, { 4194, 0xBF91 }, { 4195, 0xBF92 }, { 4196, 0xBF93 }, { 4197, 0xBF95 }, { 4198, 0xBF96 }, { 4199, 0xBF97 },
    { 4200, 0xBF98 }, { 4201, 0xBF99 }, { 4202, 0xBF9A }, { 4203, 0xBF9B }, { 4204, 0xBF9C }, { 4205, 0xBF9D }, { 4212, 0xBF9E }, { 4213, 0xBF9F },
    { 4214, 0xBFA0 }, { 4215, 0xBFA1 }, { 4216, 0xBFA2 }, { 4217, 0xBFA3 }, { 4218, 0xBFA4 }, { 4219, 0xBFA5 }, { 4220, 0xBFA6 }, { 4221, 0xBFA7 },
    { 4222, 0xBFA8 }, { 4223, 0xBFA9 }, { 4224, 0xBFAA }, { 4225, 0xBFAB }, { 4226, 0xBFAC }, { 4227, 0xBFAD }, { 4228, 0xBFAE }, { 4229, 0xBFAF },
    { 4230, 0xBFB1 }, { 4231, 0xBFB2 }, { 4232, 0xBFB3 }, { 4233, 0xBFB4 }, { 4234, 0xBFB5 }, { 4235, 0xBFB6 }, { 4236, 0xBFB7 }, { 4237, 0xBFB8 },
    { 4244, 0xBFB9 }, { 4245, 0xBFBA }, { 4246, 0xBFBB }, { 4247, 0xBFBC }, { 4248, 0xBFBD }, { 4249, 0xBFBE }, { 4250, 0xBFBF }, { 4251, 0xBFC0 },
    { 4252, 0xBFC1 }, { 4253, 0xBFC2 }, { 4254, 0xBFC3 }, { 4255, 0xBFC4 }, { 4256, 0xBFC6 }, { 4257, 0xBFC7 }, { 4258, 0xBFC8 }, { 4259, 0xBFC9 },
    { 4260, 0xBFCA }, { 4261, 0xBFCB }, { 4262, 0xBFCE }, { 4263, 0xBFCF }, { 4264, 0xBFD1 }, { 4265, 0xBFD2 }, { 4266, 0xBFD3 }, { 4267, 0xBFD5 },
    { 4268, 0xBFD6 }, { 4269, 0xBFD7 }, { 4270, 0xBFD8 }, { 4271, 0xBFD9 }, { 4272, 0xBFDA }, { 4273, 0xBFDB }, { 4274, 0xBFDD }, { 4275, 0xBFDE },
    { 4276, 0xBFE0 }, { 4277, 0xBFE2 }, { 4278, 0xBFE3 }, { 4279, 0xBFE4 }, { 4280, 0xBFE5 }, { 4281, 0xBFE6 }, { 4282, 0xBFE7 }, { 4283, 0xBFE8 },
    { 4284, 0xBFE9 }, { 4285, 0xBFEA }, { 4286, 0xBFEB }, { 4287, 0xBFEC }, { 4288, 0xBFED }, { 4289, 0xBFEE }, { 4290, 0xBFEF }, { 4291, 0xBFF0 },
    { 4292, 0xBFF1 }, { 4293, 0xBFF2 }, { 4294, 0xBFF3 }, { 4295, 0xBFF4 }, { 4296, 0xBFF5 }, { 4297, 0xBFF6 }, { 4298, 0xBFF7 }, { 4299, 0xBFF8 },
    { 4300, 0xBFF9 }, { 4301, 0xBFFA }, { 4302, 0xBFFB }, { 4303, 0xBFFC }, { 4304, 0xBFFD }, { 4305, 0xBFFE }, { 4306, 0xBFFF }, { 4307, 0xC000 },
    { 4308, 0xC001 }, { 4309, 0xC002 }, { 4310, 0xC003 }, { 4311, 0xC004 }, { 4312, 0xC005 }, { 4313, 0xC006 }, { 4314, 0xC007 }, { 4315, 0xC008 },
    { 4316, 0xC009 }, { 4317, 0xC00A }, { 4318, 0xC00B }, { 4319, 0xC00C }, { 4320, 0xC00D }, { 4321, 0xC00E }, { 4322, 0xC00F }, { 4323, 0xC010 },
    { 4324, 0xC011 }, { 4325, 0xC012 }, { 4326, 0xC013 }, { 4327, 0xC014 }, { 4328, 0xC015 }, { 4329, 0xC016 }, { 4330, 0xC017 }, { 4331, 0xC018 },
    { 4332, 0xC019 }, { 4333, 0xC01A }, { 4334, 0xC01B }, { 4335, 0xC01C }, { 4336, 0xC01D }, { 4337, 0xC01E }, { 4338, 0xC01F }, { 4339, 0xC020 },
    { 4340, 0xC021 }, { 4341, 0xC022 }, { 4342, 0xC023 }, { 4343, 0xC024 }, { 4344, 0xC025 }, { 4345, 0xC026 }, { 4346, 0xC027 }, { 4347, 0xC028 },
    { 4348, 0xC029 }, { 4349, 0xC02A }, { 4350, 0xC02B }, { 4351, 0xC02C }, { 4352, 0xC02D }, { 4353, 0xC02E }, { 4354, 0xC02F }, { 4355, 0xC030 },
    { 4356, 0xC031 }, { 4357, 0xC032 }, { 4358, 0xC033 }, { 4359, 0xC034 }, { 4360, 0xC035 }, { 4361, 0xC036 }, { 4362, 0xC037 }, { 4363, 0xC038 },
    { 4364, 0xC039 }, { 4365, 0xC03A }, { 4366, 0xC03B }, { 4367, 0xC03D }, { 4368, 0xC03E }, { 4369, 0xC03F }, { 4370, 0xC040 }, { 4371, 0xC041 },
    { 4372, 0xC042 }, { 4373, 0xC043 }, { 4374, 0xC044 }, { 4375, 0xC045 }, { 4376, 0xC046 }, { 4377, 0xC047 }, { 4378, 0xC048 }, { 4379, 0xC049 },
    { 4380, 0xC04A }, { 4381, 0xC04B }, { 4382, 0xC04C }, { 4383, 0xC04D }, { 4384, 0xC04E }, { 4385, 0xC04F }, { 4386, 0xC050 }, { 4387, 0xC052 },
    { 4388, 0xC053 }, { 4389, 0xC054 }, { 4390, 0xC055 }, { 4391, 0xC056 }, { 4392, 0xC057 }, { 4393, 0xC059 }, { 4394, 0xC05A }, { 4395, 0xC05B },
    { 4402, 0xC05D }, { 4403, 0xC05E }, { 4404, 0xC05F }, { 4405, 0xC061 }, { 4406, 0xC062 }, { 4407, 0xC063 }, { 4408, 0xC064 }, { 4409, 0xC065 },
    { 4410, 0xC066 }, { 4411, 0xC067 }, { 4412, 0xC06A }, { 4413, 0xC06B }, { 4414, 0xC06C }, { 4415, 0xC06D }, { 4416, 0xC06E }, { 4417, 0xC06F },
    { 4418, 0xC070 }, { 4419, 0xC071 }, { 4420, 0xC072 }, { 4421, 0xC073 }, { 4422, 0xC074 }, { 4423, 0xC075 }, { 4424, 0xC076 }, { 4425, 0xC077 },
    { 4426, 0xC078 }, { 4427, 0xC079 }, { 4434, 0xC07A }, { 4435, 0xC07B }, { 4436, 0xC07C }, { 4437, 0xC07D }, { 4438, 0xC07E }, { 4439, 0xC07F },
    { 4440, 0xC080 }, { 4441, 0xC081 }, { 4442, 0xC082 }, { 4443, 0xC083 }, { 4444, 0xC084 }, { 4445, 0xC085 }, { 4446, 0xC086 }, { 4447, 0xC087 },
    { 4448, 0xC088 }, { 4449, 0xC089 }, { 4450, 0xC08A }, { 4451, 0xC08B }, { 4452, 0xC08C }, { 4453, 0xC08D }, { 4454, 0xC08E }, { 4455, 0xC08F },
    { 4456, 0xC092 }, { 4457, 0xC093 }, { 4458, 0xC095 }, { 4459, 0xC096 }, { 4460, 0xC097 }, { 4461, 0xC099 }, { 4462, 0xC09A }, { 4463, 0xC09B },
    { 4464, 0xC09C }, { 4465, 0xC09D }, { 4466, 0xC09E }, { 4467, 0xC09F }, { 4468, 0xC0A2 }, { 4469, 0xC0A4 }, { 4470, 0xC0A6 }, { 4471, 0xC0A7 },
    { 4472, 0xC0A8 }, { 4473, 0xC0A9 }, { 4474, 0xC0AA }, { 4475, 0xC0AB }, { 4476, 0xC0AE }, { 4477, 0xC0B1 }, { 4478, 0xC0B2 }, { 4479, 0xC0B7 },
    { 4480, 0xC0B8 }, { 4481, 0xC0B9 }, { 4482, 0xC0BA }, { 4483, 0xC0BB }, { 4484, 0xC0BE }, { 4485, 0xC0C2 }, { 4486, 0xC0C3 }, { 4487, 0xC0C4 },
    { 4488, 0xC0C6 }, { 4489, 0xC0C7 }, { 4490, 0xC0CA }, { 4491, 0xC0CB }, { 4492, 0xC0CD }, { 4493, 0xC0CE }, { 4494, 0xC0CF }, { 4495, 0xC0D1 },
    { 4496, 0xC0D2 }, { 4497, 0xC0D3 }, { 4498, 0xC0D4 }, { 4499, 0xC0D5 }, { 4500, 0xC0D6 }, { 4501, 0xC0D7 }, { 4502, 0xC0DA }, { 4503, 0xC0DE },
    { 4504, 0xC0DF }, { 4505, 0xC0E0 }, { 4506, 0xC0E1 }, { 4507, 0xC0E2 }, { 4508, 0xC0E3 }, { 4509, 0xC0E6 }, { 4510, 0xC0E7 }, { 4511, 0xC0E9 },
    { 4512, 0xC0EA }, { 4513, 0xC0EB }, { 4514, 0xC0ED }, { 4515, 0xC0EE }, { 4516, 0xC0EF }, { 4517, 0xC0F0 }, { 4518, 0xC0F1 }, { 4519, 0xC0F2 },
    { 4520, 0xC0F3 }, { 4521, 0xC0F6 }, { 4522, 0xC0F8 }, { 4523, 0xC0FA }, { 4524, 0xC0FB }, { 4525, 0xC0FC }, { 4526, 0xC0FD }, { 4527, 0xC0FE },
    { 4528, 0xC0FF }, { 4529, 0xC101 }, { 4530, 0xC102 }, { 4531, 0xC103 }, { 4532, 0xC105 }, { 4533, 0xC106 }, { 4534, 0xC107 }, { 4535, 0xC109 },
    { 4536, 0xC10A }, { 4537, 0xC10B }, { 4538, 0xC10C }, { 4539, 0xC10D }, { 4540, 0xC10E }, { 4541, 0xC10F }, { 4542, 0xC111 }, { 4543, 0xC112 },
    { 4544, 0xC113 }, { 4545, 0xC114 }, { 4546, 0xC116 }, { 4547, 0xC117 }, { 4548, 0xC118 }, { 4549, 0xC119 }, { 4550, 0xC11A }, { 4551, 0xC11B },
    { 4552, 0xC121 }, { 4553, 0xC122 }, { 4554, 0xC125 }, { 4555, 0xC128 }, { 4556, 0xC129 }, { 4557, 0xC12A }, { 4558, 0xC12B }, { 4559, 0xC12E },
    { 4560, 0xC132 }, { 4561, 0xC133 }, { 4562, 0xC134 }, { 4563, 0xC135 }, { 4564, 0xC137 }, { 4565, 0xC13A }, { 4566, 0xC13B }, { 4567, 0xC13D },
    { 4568, 0xC13E }, { 4569, 0xC13F }, { 4570, 0xC141 }, { 4571, 0xC142 }, { 4572, 0xC143 }, { 4573, 0xC144 }, { 4574, 0xC145 }, { 4575, 0xC146 },
    { 4576, 0xC147 }, { 4577, 0xC14A }, { 4578, 0xC14E }, { 4579, 0xC14F }, { 4580, 0xC150 }, { 4581, 0xC151 }, { 4582, 0xC152 }, { 4583, 0xC153 },
    { 4584, 0xC156 }, { 4585, 0xC157 }, { 4592, 0xC159 }, { 4593, 0xC15A }, { 4594, 0xC15B }, { 4595, 0xC15D }, { 4596, 0xC15E }, { 4597, 0xC15F },
    { 4598, 0xC160 }, { 4599, 0xC161 }, { 4600, 0xC162 }, { 4601, 0xC163 }, { 4602, 0xC166 }, { 4603, 0xC16A }, { 4604, 0xC16B }, { 4605, 0xC16C },
    { 4606, 0xC16D }, { 4607, 0xC16E }, { 4608, 0xC16F }, { 4609, 0xC171 }, { 4610, 0xC172 }, { 4611, 0xC173 }, { 4612, 0xC175 }, { 4613, 0xC176 },
    { 4614, 0xC177 }, { 4615, 0xC179 }, { 4616, 0xC17A }, { 4617, 0xC17B }, { 4624, 0xC17C }, { 4625, 0xC17D }, { 4626, 0xC17E }, { 4627, 0xC17F },
    { 4628, 0xC180 }, { 4629, 0xC181 }, { 4630, 0xC182 }, { 4631, 0xC183 }, { 4632, 0xC184 }, { 4633, 0xC186 }, { 4634, 0xC187 }, { 4635, 0xC188 },
    { 4636, 0xC189 }, { 4637, 0xC18A }, { 4638, 0xC18B }, { 4639, 0xC18F }, { 4640, 0xC191 }, { 4641, 0xC192 }, { 4642, 0xC193 }, { 4643, 0xC195 },
    { 4644, 0xC197 }, { 4645, 0xC198 }, { 4646, 0xC199 }, { 4647, 0xC19A }, { 4648, 0xC19B }, { 4649, 0xC19E }, { 4650, 0xC1A0 }, { 4651, 0xC1A2 },
    { 4652, 0xC1A3 }, { 4653, 0xC1A4 }, { 4654, 0xC1A6 }, { 4655, 0xC1A7 }, { 4656, 0xC1AA }, { 4657, 0xC1AB }, { 4658, 0xC1AD }, { 4659, 0xC1AE },
    { 4660, 0xC1AF }, { 4661, 0xC1B1 }, { 4662, 0xC1B2 }, { 4663, 0xC1B3 }, { 4664, 0xC1B4 }, { 4665, 0xC1B5 }, { 4666, 0xC1B6 }, { 4667, 0xC1B7 },
    { 4668, 0xC1B8 }, { 4669, 0xC1B9 }, { 4670, 0xC1BA }, { 4671, 0xC1BB }, { 4672, 0xC1BC }, { 4673, 0xC1BE }, { 4674, 0xC1BF }, { 4675, 0xC1C0 },
    { 4676, 0xC1C1 }, { 4677, 0xC1C2 }, { 4678, 0xC1C3 }, { 4679, 0xC1C5 }, { 4680, 0xC1C6 }, { 4681, 0xC1C7 }, { 4682, 0xC1C9 }, { 4683, 0xC1CA },
    { 4684, 0xC1CB }, { 4685, 0xC1CD }, { 4686, 0xC1CE }, { 4687, 0xC1CF }, { 4688, 0xC1D0 }, { 4689, 0xC1D1 }, { 4690, 0xC1D2 }, { 4691, 0xC1D3 },
    { 4692, 0xC1D5 }, { 4693, 0xC1D6 }, { 4694, 0xC1D9 }, { 4695, 0xC1DA }, { 4696, 0xC1DB }, { 4697, 0xC1DC }, { 4698, 0xC1DD }, { 4699, 0xC1DE },
    { 4700, 0xC1DF }, { 4701, 0xC1E1 }, { 4702, 0xC1E2 }, { 4703, 0xC1E3 }, { 4704, 0xC1E5 }, { 4705, 0xC1E6 }, { 4706, 0xC1E7 }, { 4707, 0xC1E9 },
    { 4708, 0xC1EA }, { 4709, 0xC1EB }, { 4710, 0xC1EC }, { 4711, 0xC1ED }, { 4712, 0xC1EE }, { 4713, 0xC1EF }, { 4714, 0xC1F2 }, { 4715, 0xC1F4 },
    { 4716, 0xC1F5 }, { 4717, 0xC1F6 }, { 4718, 0xC1F7 }, { 4719, 0xC1F8 }, { 4720, 0xC1F9 }, { 4721, 0xC1FA }, { 4722, 0xC1FB }, { 4723, 0xC1FE },
    { 4724, 0xC1FF }, { 4725, 0xC201 }, { 4726, 0xC202 }, { 4727, 0xC203 }, { 4728, 0xC205 }, { 4729, 0xC206 }, { 4730, 0xC207 }, { 4731, 0xC208 },
    { 4732, 0xC209 }, { 4733, 0xC20A }, { 4734, 0xC20B }, { 4735, 0xC20E }, { 4736, 0xC210 }, { 4737, 0xC212 }, { 4738, 0xC213 }, { 4739, 0xC214 },
    { 4740, 0xC215 }, { 4741, 0xC216 }, { 4742, 0xC217 }, { 4743, 0xC21A }, { 4744, 0xC21B }, { 4745, 0xC21D }, { 4746, 0xC21E }, { 4747, 0xC221 },
    { 4748, 0xC222 }, { 4749, 0xC223 }, { 4750, 0xC224 }, { 4751, 0xC225 }, { 4752, 0xC226 }, { 4753, 0xC227 }, { 4754, 0xC22A }, { 4755, 0xC22C },
    { 4756, 0xC22E }, { 4757, 0xC230 }, { 4758, 0xC233 }, { 4759, 0xC235 }, { 4760, 0xC236 }, { 4761, 0xC237 }, { 4762, 0xC238 }, { 4763, 0xC239 },
    { 4764, 0xC23A }, { 4765, 0xC23B }, { 4766, 0xC23C }, { 4767, 0xC23D }, { 4768, 0xC23E }, { 4769, 0xC23F }, { 4770, 0xC240 }, { 4771, 0xC241 },
    { 4772, 0xC242 }, { 4773, 0xC243 }, { 4774, 0xC244 }, { 4775, 0xC245 }, { 4782, 0xC246 }, { 4783, 0xC247 }, { 4784, 0xC249 }, { 4785, 0xC24A },
    { 4786, 0xC24B }, { 4787, 0xC24C }, { 4788, 0xC24D }, { 4789, 0xC24E }, { 4790, 0xC24F }, { 4791, 0xC252 }, { 4792, 0xC253 }, { 4793, 0xC255 },
    { 4794, 0xC256 }, { 4795, 0xC257 }, { 4796, 0xC259 }, { 4797, 0xC25A }, { 4798, 0xC25B }, { 4799, 0xC25C }, { 4800, 0xC25D }, { 4801, 0xC25E },
    { 4802, 0xC25F }, { 4803, 0xC261 }, { 4804, 0xC262 }, { 4805, 0xC263 }, { 4806, 0xC264 }, { 4807, 0xC266 }, { 4814, 0xC267 }, { 4815, 0xC268 },
    { 4816, 0xC269 }, { 4817, 0xC26A }, { 4818, 0xC26B }, { 4819, 0xC26E }, { 4820, 0xC26F }, { 4821, 0xC271 }, { 4822, 0xC272 }, { 4823, 0xC273 },
    { 4824, 0xC275 }, { 4825, 0xC276 }, { 4826, 0xC277 }, { 4827, 0xC278 }, { 4828, 0xC279 }, { 4829, 0xC27A }, { 4830, 0xC27B }, { 4831, 0xC27E },
    { 4832, 0xC280 }, { 4833, 0xC282 }, { 4834, 0xC283 }, { 4835, 0xC284 }, { 4836, 0xC285 }, { 4837, 0xC286 }, { 4838, 0xC287 }, { 4839, 0xC28A },
    { 4840, 0xC28B }, { 4841, 0xC28C }, { 4842, 0xC28D }, { 4843, 0xC28E }, { 4844, 0xC28F }, { 4845, 0xC291 }, { 4846, 0xC292 }, { 4847, 0xC293 },
    { 4848, 0xC294 }, { 4849, 0xC295 }, { 4850, 0xC296 }, { 4851, 0xC297 }, { 4852, 0xC299 }, { 4853, 0xC29A }, { 4854, 0xC29C }, { 4855, 0xC29E },
    { 4856, 0xC29F }, { 4857, 0xC2A0 }, { 4858, 0xC2A1 }, { 4859, 0xC2A2 }, { 4860, 0xC2A3 }, { 4861, 0xC2A6 }, { 4862, 0xC2A7 }, { 4863, 0xC2A9 },
    { 4864, 0xC2AA }, { 4865, 0xC2AB }, { 4866, 0xC2AE }, { 4867, 0xC2AF }, { 4868, 0xC2B0 }, { 4869, 0xC2B1 }, { 4870, 0xC2B2 }, { 4871, 0xC2B3 },
    { 4872, 0xC2B6 }, { 4873, 0xC2B8 }, { 4874, 0xC2BA }, { 4875, 0xC2BB }, { 4876, 0xC2BC }, { 4877, 0xC2BD }, { 4878, 0xC2BE }, { 4879, 0xC2BF },
    { 4880, 0xC2C0 }, { 4881, 0xC2C1 }, { 4882, 0xC2C2 }, { 4883, 0xC2C3 }, { 4884, 0xC2C4 }, { 4885, 0xC2C5 }, { 4886, 0xC2C6 }, { 4887, 0xC2C7 },
    { 4888, 0xC2C8 }, { 4889, 0xC2C9 }, { 4890, 0xC2CA }, { 4891, 0xC2CB }, { 4892, 0xC2CC }, { 4893, 0xC2CD }, { 4894, 0xC2CE }, { 4895, 0xC2CF },
    { 4896, 0xC2D0 }, { 4897, 0xC2D1 }, { 4898, 0xC2D2 }, { 4899, 0xC2D3 }, { 4900, 0xC2D4 }, { 4901, 0xC2D5 }, { 4902, 0xC2D6 }, { 4903, 0xC2D7 },
    { 4904, 0xC2D8 }, { 4905, 0xC2D9 }, { 4906, 0xC2DA }, { 4907, 0xC2DB }, { 4908, 0xC2DE }, { 4909, 0xC2DF }, { 4910, 0xC2E1 }, { 4911, 0xC2E2 },
    { 4912, 0xC2E5 }, { 4913, 0xC2E6 }, { 4914, 0xC2E7 }, { 4915, 0xC2E8 }, { 4916, 0xC2E9 }, { 4917, 0xC2EA }, { 4918, 0xC2EE }, { 4919, 0xC2F0 },
    { 4920, 0xC2F2 }, { 4921, 0xC2F3 }, { 4922, 0xC2F4 }, { 4923, 0xC2F5 }, { 4924, 0xC2F7 }, { 4925, 0xC2FA }, { 4926, 0xC2FD }, { 4927, 0xC2FE },
    { 4928, 0xC2FF }, { 4929, 0xC301 }, { 4930, 0xC302 }, { 4931, 0xC303 }, { 4932, 0xC304 }, { 4933, 0xC305 }, { 4934, 0xC306 }, { 4935, 0xC307 },
    { 4936, 0xC30A }, { 4937, 0xC30B }, { 4938, 0xC30E }, { 4939, 0xC30F }, { 4940, 0xC310 }, { 4941, 0xC311 }, { 4942, 0xC312 }, { 4943, 0xC316 },
    { 4944, 0xC317 }, { 4945, 0xC319 }, { 4946, 0xC31A }, { 4947, 0xC31B }, { 4948, 0xC31D }, { 4949, 0xC31E }, { 4950, 0xC31F }, { 4951, 0xC320 },
    { 4952, 0xC321 }, { 4953, 0xC322 }, { 4954, 0xC323 }, { 4955, 0xC326 }, { 4956, 0xC327 }, { 4957, 0xC32A }, { 4958, 0xC32B }, { 4959, 0xC32C },
    { 4960, 0xC32D }, { 4961, 0xC32E }, { 4962, 0xC32F }, { 4963, 0xC330 }, { 4964, 0xC331 }, { 4965, 0xC332 }, { 4972, 0xC333 }, { 4973, 0xC334 },
    { 4974, 0xC335 }, { 4975, 0xC336 }, { 4976, 0xC337 }, { 4977, 0xC338 }, { 4978, 0xC339 }, { 4979, 0xC33A }, { 4980, 0xC33B }, { 4981, 0xC33C },
    { 4982, 0xC33D }, { 4983, 0xC33E }, { 4984, 0xC33F }, { 4985, 0xC340 }, { 4986, 0xC341 }, { 4987, 0xC342 }, { 4988, 0xC343 }, { 4989, 0xC344 },
    { 4990, 0xC346 }, { 4991, 0xC347 }, { 4992, 0xC348 }, { 4993, 0xC349 }, { 4994, 0xC34A }, { 4995, 0xC34B }, { 4996, 0xC34C }, { 4997, 0xC34D },
    { 5004, 0xC34E }, { 5005, 0xC34F }, { 5006, 0xC350 }, { 5007, 0xC351 }, { 5008, 0xC352 }, { 5009, 0xC353 }, { 5010, 0xC354 }, { 5011, 0xC355 },
    { 5012, 0xC356 }, { 5013, 0xC357 }, { 5014, 0xC358 }, { 5015, 0xC359 }, { 5016, 0xC35A }, { 5017, 0xC35B }, { 5018, 0xC35C }, { 5019, 0xC35D },
    { 5020, 0xC35E }, { 5021, 0xC35F }, { 5022, 0xC360 }, { 5023, 0xC361 }, { 5024, 0xC362 }, { 5025, 0xC363 }, { 5026, 0xC364 }, { 5027, 0xC365 },
    { 5028, 0xC366 }, { 5029, 0xC367 }, { 5030, 0xC36A }, { 5031, 0xC36B }, { 5032, 0xC36D }, { 5033, 0xC36E }, { 5034, 0xC36F }, { 5035, 0xC371 },
    { 5036, 0xC373 }, { 5037, 0xC374 }, { 5038, 0xC375 }, { 5039, 0xC376 }, { 5040, 0xC377 }, { 5041, 0xC37A }, { 5042, 0xC37B }, { 5043, 0xC37E },
    { 5044, 0xC37F }, { 5045, 0xC380 }, { 5046, 0xC381 }, { 5047, 0xC382 }, { 5048, 0xC383 }, { 5049, 0xC385 }, { 5050, 0xC386 }, { 5051, 0xC387 },
    { 5052, 0xC389 }, { 5053, 0xC38A }, { 5054, 0xC38B }, { 5055, 0xC38D }, { 5056, 0xC38E }, { 5057, 0xC38F }, { 5058, 0xC390 }, { 5059, 0xC391 },
    { 5060, 0xC392 }, { 5061, 0xC393 }, { 5062, 0xC394 }, { 5063, 0xC395 }, { 5064, 0xC396 }, { 5065, 0xC397 }, { 5066, 0xC398 }, { 5067, 0xC399 },
    { 5068, 0xC39A }, { 5069, 0xC39B }, { 5070, 0xC39C }, { 5071, 0xC39D }, { 5072, 0xC39E }, { 5073, 0xC39F }, { 5074, 0xC3A0 }, { 5075, 0xC3A1 },
    { 5076, 0xC3A2 }, { 5077, 0xC3A3 }, { 5078, 0xC3A4 }, { 5079, 0xC3A5 }, { 5080, 0xC3A6 }, { 5081, 0xC3A7 }, { 5082, 0xC3A8 }, { 5083, 0xC3A9 },
    { 5084, 0xC3AA }, { 5085, 0xC3AB }, { 5086, 0xC3AC }, { 5087, 0xC3AD }, { 5088, 0xC3AE }, { 5089, 0xC3AF }, { 5090, 0xC3B0 }, { 5091, 0xC3B1 },
    { 5092, 0xC3B2 }, { 5093, 0xC3B3 }, { 5094, 0xC3B4 }, { 5095, 0xC3B5 }, { 5096, 0xC3B6 }, { 5097, 0xC3B7 }, { 5098, 0xC3B8 }, { 5099, 0xC3B9 },
    { 5100, 0xC3BA }, { 5101, 0xC3BB }, { 5102, 0xC3BC }, { 5103, 0xC3BD }, { 5104, 0xC3BE }, { 5105, 0xC3BF }, { 5106, 0xC3C1 }, { 5107, 0xC3C2 },
    { 5108, 0xC3C3 }, { 5109, 0xC3C4 }, { 5110, 0xC3C5 }, { 5111, 0xC3C6 }, { 5112, 0xC3C7 }, { 5113, 0xC3C8 }, { 5114, 0xC3C9 }, { 5115, 0xC3CA },
    { 5116, 0xC3CB }, { 5117, 0xC3CC }, { 5118, 0xC3CD }, { 5119, 0xC3CE }, { 5120, 0xC3CF }, { 5121, 0xC3D0 }, { 5122, 0xC3D1 }, { 5123, 0xC3D2 },
    { 5124, 0xC3D3 }, { 5125, 0xC3D4 }, { 5126, 0xC3D5 }, { 5127, 0xC3D6 }, { 5128, 0xC3D7 }, { 5129, 0xC3DA }, { 5130, 0xC3DB }, { 5131, 0xC3DD },
    { 5132, 0xC3DE }, { 5133, 0xC3E1 }, { 5134, 0xC3E3 }, { 5135, 0xC3E4 }, { 5136, 0xC3E5 }, { 5137, 0xC3E6 }, { 5138, 0xC3E7 }, { 5139, 0xC3EA },
    { 5140, 0xC3EB }, { 5141, 0xC3EC }, { 5142, 0xC3EE }, { 5143, 0xC3EF }, { 5144, 0xC3F0 }, { 5145, 0xC3F1 }, { 5146, 0xC3F2 }, { 5147, 0xC3F3 },
    { 5148, 0xC3F6 }, { 5149, 0xC3F7 }, { 5150, 0xC3F9 }, { 5151, 0xC3FA }, { 5152, 0xC3FB }, { 5153, 0xC3FC }, { 5154, 0xC3FD }, { 5155, 0xC3FE },
    { 5162, 0xC3FF }, { 5163, 0xC400 }, { 5164, 0xC401 }, { 5165, 0xC402 }, { 5166, 0xC403 }, { 5167, 0xC404 }, { 5168, 0xC405 }, { 5169, 0xC406 },
    { 5170, 0xC407 }, { 5171, 0xC409 }, { 5172, 0xC40A }, { 5173, 0xC40B }, { 5174, 0xC40C }, { 5175, 0xC40D }, { 5176, 0xC40E }, { 5177, 0xC40F },
    { 5178, 0xC411 }, { 5179, 0xC412 }, { 5180, 0xC413 }, { 5181, 0xC414 }, { 5182, 0xC415 }, { 5183, 0xC416 }, { 5184, 0xC417 }, { 5185, 0xC418 },
    { 5186, 0xC419 }, { 5187, 0xC41A }, { 5194, 0xC41B }, { 5195, 0xC41C }, { 5196, 0xC41D }, { 5197, 0xC41E }, { 5198, 0xC41F }, { 5199, 0xC420 },
    { 5200, 0xC421 }, { 5201, 0xC422 }, { 5202, 0xC423 }, { 5203, 0xC425 }, { 5204, 0xC426 }, { 5205, 0xC427 }, { 5206, 0xC428 }, { 5207, 0xC429 },
    { 5208, 0xC42A }, { 5209, 0xC42B }, { 5210, 0xC42D }, { 5211, 0xC42E }, { 5212, 0xC42F }, { 5213, 0xC431 }, { 5214, 0xC432 }, { 5215, 0xC433 },
    { 5216, 0xC435 }, { 5217, 0xC436 }, { 5218, 0xC437 }, { 5219, 0xC438 }, { 5220, 0xC439 }, { 5221, 0xC43A }, { 5222, 0xC43B }, { 5223, 0xC43E },
    { 5224, 0xC43F }, { 5225, 0xC440 }, { 5226, 0xC441 }, { 5227, 0xC442 }, { 5228, 0xC443 }, { 5229, 0xC444 }, { 5230, 0xC445 }, { 5231, 0xC446 },
    { 5232, 0xC447 }, { 5233, 0xC449 }, { 5234, 0xC44A }, { 5235, 0xC44B }, { 5236, 0xC44C }, { 5237, 0xC44D }, { 5238, 0xC44E }, { 5239, 0xC44F },
    { 5240, 0xC450 }, { 5241, 0xC451 }, { 5242, 0xC452 }, { 5243, 0xC453 }, { 5244, 0xC454 }, { 5245, 0xC455 }, { 5246, 0xC456 }, { 5247, 0xC457 },
    { 5248, 0xC458 }, { 5249, 0xC459 }, { 5250, 0xC45A }, { 5251, 0xC45B }, { 5252, 0xC45C }, { 5253, 0xC45D }, { 5254, 0xC45E }, { 5255, 0xC45F },
    { 5256, 0xC460 }, { 5257, 0xC461 }, { 5258, 0xC462 }, { 5259, 0xC463 }, { 5260, 0xC466 }, { 5261, 0xC467 }, { 5262, 0xC469 }, { 5263, 0xC46A },
    { 5264, 0xC46B }, { 5265, 0xC46D }, { 5266, 0xC46E }, { 5267, 0xC46F }, { 5268, 0xC470 }, { 5269, 0xC471 }, { 5270, 0xC472 }, { 5271, 0xC473 },
    { 5272, 0xC476 }, { 5273, 0xC477 }, { 5274, 0xC478 }, { 5275, 0xC47A }, { 5276, 0xC47B }, { 5277, 0xC47C }, { 5278, 0xC47D }, { 5279, 0xC47E },
    { 5280, 0xC47F }, { 5281, 0xC481 }, { 5282, 0xC482 }, { 5283, 0xC483 }, { 5284, 0xC484 }, { 5285, 0xC485 }, { 5286, 0xC486 }, { 5287, 0xC487 },
    { 5288, 0xC488 }, { 5289, 0xC489 }, { 5290, 0xC48A }, { 5291, 0xC48B }, { 5292, 0xC48C }, { 5293, 0xC48D }, { 5294, 0xC48E }, { 5295, 0xC48F },
    { 5296, 0xC490 }, { 5297, 0xC491 }, { 5298, 0xC492 }, { 5299, 0xC493 }, { 5300, 0xC495 }, { 5301, 0xC496 }, { 5302, 0xC497 }, { 5303, 0xC498 },
    { 5304, 0xC499 }, { 5305, 0xC49A }, { 5306, 0xC49B }, { 5307, 0xC49D }, { 5308, 0xC49E }, { 5309, 0xC49F }, { 5310, 0xC4A0 }, { 5311, 0xC4A1 },
    { 5312, 0xC4A2 }, { 5313, 0xC4A3 }, { 5314, 0xC4A4 }, { 5315, 0xC4A5 }, { 5316, 0xC4A6 }, { 5317, 0xC4A7 }, { 5318, 0xC4A8 }, { 5319, 0xC4A9 },
    { 5320, 0xC4AA }, { 5321, 0xC4AB }, { 5322, 0xC4AC }, { 5323, 0xC4AD }, { 5324, 0xC4AE }, { 5325, 0xC4AF }, { 5326, 0xC4B0 }, { 5327, 0xC4B1 },
    { 5328, 0xC4B2 }, { 5329, 0xC4B3 }, { 5330, 0xC4B4 }, { 5331, 0xC4B5 }, { 5332, 0xC4B6 }, { 5333, 0xC4B7 }, { 5334, 0xC4B9 }, { 5335, 0xC4BA },
    { 5336, 0xC4BB }, { 5337, 0xC4BD }, { 5338, 0xC4BE }, { 5339, 0xC4BF }, { 5340, 0xC4C0 }, { 5341, 0xC4C1 }, { 5342, 0xC4C2 }, { 5343, 0xC4C3 },
    { 5344, 0xC4C4 }, { 5345, 0xC4C5 }, { 5352, 0xC4C6 }, { 5353, 0xC4C7 }, { 5354, 0xC4C8 }, { 5355, 0xC4C9 }, { 5356, 0xC4CA }, { 5357, 0xC4CB },
    { 5358, 0xC4CC }, { 5359, 0xC4CD }, { 5360, 0xC4CE }, { 5361, 0xC4CF }, { 5362, 0xC4D0 }, { 5363, 0xC4D1 }, { 5364, 0xC4D2 }, { 5365, 0xC4D3 },
    { 5366, 0xC4D4 }, { 5367, 0xC4D5 }, { 5368, 0xC4D6 }, { 5369, 0xC4D7 }, { 5370, 0xC4D8 }, { 5371, 0xC4D9 }, { 5372, 0xC4DA }, { 5373, 0xC4DB },
    { 5374, 0xC4DC }, { 5375, 0xC4DD }, { 5376, 0xC4DE }, { 5377, 0xC4DF }, { 5384, 0xC4E0 }, { 5385, 0xC4E1 }, { 5386, 0xC4E2 }, { 5387, 0xC4E3 },
    { 5388, 0xC4E4 }, { 5389, 0xC4E5 }, { 5390, 0xC4E6 }, { 5391, 0xC4E7 }, { 5392, 0xC4E8 }, { 5393, 0xC4EA }, { 5394, 0xC4EB }, { 5395, 0xC4EC },
    { 5396, 0xC4ED }, { 5397, 0xC4EE }, { 5398, 0xC4EF }, { 5399, 0xC4F2 }, { 5400, 0xC4F3 }, { 5401, 0xC4F5 }, { 5402, 0xC4F6 }, { 5403, 0xC4F7 },
    { 5404, 0xC4F9 }, { 5405, 0xC4FB }, { 5406, 0xC4FC }, { 5407, 0xC4FD }, { 5408, 0xC4FE }, { 5409, 0xC502 }, { 5410, 0xC503 }, { 5411, 0xC504 },
    { 5412, 0xC505 }, { 5413, 0xC506 }, { 5414, 0xC507 }, { 5415, 0xC508 }, { 5416, 0xC509 }, { 5417, 0xC50A }, { 5418, 0xC50B }, { 5419, 0xC50D },
    { 5420, 0xC50E }, { 5421, 0xC50F }, { 5422, 0xC511 }, { 5423, 0xC512 }, { 5424, 0xC513 }, { 5425, 0xC515 }, { 5426, 0xC516 }, { 5427, 0xC517 },
    { 5428, 0xC518 }, { 5429, 0xC519 }, { 5430, 0xC51A }, { 5431, 0xC51B }, { 5432, 0xC51D }, { 5433, 0xC51E }, { 5434, 0xC51F }, { 5435, 0xC520 },
    { 5436, 0xC521 }, { 5437, 0xC522 }, { 5438, 0xC523 }, { 5439, 0xC524 }, { 5440, 0xC525 }, { 5441, 0xC526 }, { 5442, 0xC527 }, { 5443, 0xC52A },
    { 5444, 0xC52B }, { 5445, 0xC52D }, { 5446, 0xC52E }, { 5447, 0xC52F }, { 5448, 0xC531 }, { 5449, 0xC532 }, { 5450, 0xC533 }, { 5451, 0xC534 },
    { 5452, 0xC535 }, { 5453, 0xC536 }, { 5454, 0xC537 }, { 5455, 0xC53A }, { 5456, 0xC53C }, { 5457, 0xC53E }, { 5458, 0xC53F }, { 5459, 0xC540 },
    { 5460, 0xC541 }, { 5461, 0xC542 }, { 5462, 0xC543 }, { 5463, 0xC546 }, { 5464, 0xC547 }, { 5465, 0xC54B }, { 5466, 0xC54F }, { 5467, 0xC550 },
    { 5468, 0xC551 }, { 5469, 0xC552 }, { 5470, 0xC556 }, { 5471, 0xC55A }, { 5472, 0xC55B }, { 5473, 0xC55C }, { 5474, 0xC55F }, { 5475, 0xC562 },
    { 5476, 0xC563 }, { 5477, 0xC565 }, { 5478, 0xC566 }, { 5479, 0xC567 }, { 5480, 0xC569 }, { 5481, 0xC56A }, { 5482, 0xC56B }, { 5483, 0xC56C },
    { 5484, 0xC56D }, { 5485, 0xC56E }, { 5486, 0xC56F }, { 5487, 0xC572 }, { 5488, 0xC576 }, { 5489, 0xC577 }, { 5490, 0xC578 }, { 5491, 0xC579 },
    { 5492, 0xC57A }, { 5493, 0xC57B }, { 5494, 0xC57E }, { 5495, 0xC57F }, { 5496, 0xC581 }, { 5497, 0xC582 }, { 5498, 0xC583 }, { 5499, 0xC585 },
    { 5500, 0xC586 }, { 5501, 0xC588 }, { 5502, 0xC589 }, { 5503, 0xC58A }, { 5504, 0xC58B }, { 5505, 0xC58E }, { 5506, 0xC590 }, { 5507, 0xC592 },
    { 5508, 0xC593 }, { 5509, 0xC594 }, { 5510, 0xC596 }, { 5511, 0xC599 }, { 5512, 0xC59A }, { 5513, 0xC59B }, { 5514, 0xC59D }, { 5515, 0xC59E },
    { 5516, 0xC59F }, { 5517, 0xC5A1 }, { 5518, 0xC5A2 }, { 5519, 0xC5A3 }, { 5520, 0xC5A4 }, { 5521, 0xC5A5 }, { 5522, 0xC5A6 }, { 5523, 0xC5A7 },
    { 5524, 0xC5A8 }, { 5525, 0xC5AA }, { 5526, 0xC5AB }, { 5527, 0xC5AC }, { 5528, 0xC5AD }, { 5529, 0xC5AE }, { 5530, 0xC5AF }, { 5531, 0xC5B0 },
    { 5532, 0xC5B1 }, { 5533, 0xC5B2 }, { 5534, 0xC5B3 }, { 5535, 0xC5B6 }, { 5542, 0xC5B7 }, { 5543, 0xC5BA }, { 5544, 0xC5BF }, { 5545, 0xC5C0 },
    { 5546, 0xC5C1 }, { 5547, 0xC5C2 }, { 5548, 0xC5C3 }, { 5549, 0xC5CB }, { 5550, 0xC5CD }, { 5551, 0xC5CF }, { 5552, 0xC5D2 }, { 5553, 0xC5D3 },
    { 5554, 0xC5D5 }, { 5555, 0xC5D6 }, { 5556, 0xC5D7 }, { 5557, 0xC5D9 }, { 5558, 0xC5DA }, { 5559, 0xC5DB }, { 5560, 0xC5DC }, { 5561, 0xC5DD },
    { 5562, 0xC5DE }, { 5563, 0xC5DF }, { 5564, 0xC5E2 }, { 5565, 0xC5E4 }, { 5566, 0xC5E6 }, { 5567, 0xC5E7 }, { 5574, 0xC5E8 }, { 5575, 0xC5E9 },
    { 5576, 0xC5EA }, { 5577, 0xC5EB }, { 5578, 0xC5EF }, { 5579, 0xC5F1 }, { 5580, 0xC5F2 }, { 5581, 0xC5F3 }, { 5582, 0xC5F5 }, { 5583, 0xC5F8 },
    { 5584, 0xC5F9 }, { 5585, 0xC5FA }, { 5586, 0xC5FB }, { 5587, 0xC602 }, { 5588, 0xC603 }, { 5589, 0xC604 }, { 5590, 0xC609 }, { 5591, 0xC60A },
    { 5592, 0xC60B }, { 5593, 0xC60D }, { 5594, 0xC60E }, { 5595, 0xC60F }, { 5596, 0xC611 }, { 5597, 0xC612 }, { 5598, 0xC613 }, { 5599, 0xC614 },
    { 5600, 0xC615 }, { 5601, 0xC616 }, { 5602, 0xC617 }, { 5603, 0xC61A }, { 5604, 0xC61D }, { 5605, 0xC61E }, { 5606, 0xC61F }, { 5607, 0xC620 },
    { 5608, 0xC621 }, { 5609, 0xC622 }, { 5610, 0xC623 }, { 5611, 0xC626 }, { 5612, 0xC627 }, { 5613, 0xC629 }, { 5614, 0xC62A }, { 5615, 0xC62B },
    { 5616, 0xC62F }, { 5617, 0xC631 }, { 5618, 0xC632 }, { 5619, 0xC636 }, { 5620, 0xC638 }, { 5621, 0xC63A }, { 5622, 0xC63C }, { 5623, 0xC63D },
    { 5624, 0xC63E }, { 5625, 0xC63F }, { 5626, 0xC642 }, { 5627, 0xC643 }, { 5628, 0xC645 }, { 5629, 0xC646 }, { 5630, 0xC647 }, { 5631, 0xC649 },
    { 5632, 0xC64A }, { 5633, 0xC64B }, { 5634, 0xC64C }, { 5635, 0xC64D }, { 5636, 0xC64E }, { 5637, 0xC64F }, { 5638, 0xC652 }, { 5639, 0xC656 },
    { 5640, 0xC657 }, { 5641, 0xC658 }, { 5642, 0xC659 }, { 5643, 0xC65A }, { 5644, 0xC65B }, { 5645, 0xC65E }, { 5646, 0xC65F }, { 5647, 0xC661 },
    { 5648, 0xC662 }, { 5649, 0xC663 }, { 5650, 0xC664 }, { 5651, 0xC665 }, { 5652, 0xC666 }, { 5653, 0xC667 }, { 5654, 0xC668 }, { 5655, 0xC669 },
    { 5656, 0xC66A }, { 5657, 0xC66B }, { 5658, 0xC66D }, { 5659, 0xC66E }, { 5660, 0xC670 }, { 5661, 0xC672 }, { 5662, 0xC673 }, { 5663, 0xC674 },
    { 5664, 0xC675 }, { 5665, 0xC676 }, { 5666, 0xC677 }, { 5667, 0xC67A }, { 5668, 0xC67B }, { 5669, 0xC67D }, { 5670, 0xC67E }, { 5671, 0xC67F },
    { 5672, 0xC681 }, { 5673, 0xC682 }, { 5674, 0xC683 }, { 5675, 0xC684 }, { 5676, 0xC685 }, { 5677, 0xC686 }, { 5678, 0xC687 }, { 5679, 0xC68A },
    { 5680, 0xC68C }, { 5681, 0xC68E }, { 5682, 0xC68F }, { 5683, 0xC690 }, { 5684, 0xC691 }, { 5685, 0xC692 }, { 5686, 0xC693 }, { 5687, 0xC696 },
    { 5688, 0xC697 }, { 5689, 0xC699 }, { 5690, 0xC69A }, { 5691, 0xC69B }, { 5692, 0xC69D }, { 5693, 0xC69E }, { 5694, 0xC69F }, { 5695, 0xC6A0 },
    { 5696, 0xC6A1 }, { 5697, 0xC6A2 }, { 5698, 0xC6A3 }, { 5699, 0xC6A6 }, { 5700, 0xC6A8 }, { 5701, 0xC6AA }, { 5702, 0xC6AB }, { 5703, 0xC6AC },
    { 5704, 0xC6AD }, { 5705, 0xC6AE }, { 5706, 0xC6AF }, { 5707, 0xC6B2 }, { 5708, 0xC6B3 }, { 5709, 0xC6B5 }, { 5710, 0xC6B6 }, { 5711, 0xC6B7 },
    { 5712, 0xC6BB }, { 5713, 0xC6BC }, { 5714, 0xC6BD }, { 5715, 0xC6BE }, { 5716, 0xC6BF }, { 5717, 0xC6C2 }, { 5718, 0xC6C4 }, { 5719, 0xC6C6 },
    { 5720, 0xC6C7 }, { 5721, 0xC6C8 }, { 5722, 0xC6C9 }, { 5723, 0xC6CA }, { 5724, 0xC6CB }, { 5725, 0xC6CE }, { 5732, 0xC6CF }, { 5733, 0xC6D1 },
    { 5734, 0xC6D2 }, { 5735, 0xC6D3 }, { 5736, 0xC6D5 }, { 5737, 0xC6D6 }, { 5738, 0xC6D7 }, { 5739, 0xC6D8 }, { 5740, 0xC6D9 }, { 5741, 0xC6DA },
    { 5742, 0xC6DB }, { 5743, 0xC6DE }, { 5744, 0xC6DF }, { 5745, 0xC6E2 }, { 5746, 0xC6E3 }, { 5747, 0xC6E4 }, { 5748, 0xC6E5 }, { 5749, 0xC6E6 },
    { 5750, 0xC6E7 }, { 5751, 0xC6EA }, { 5752, 0xC6EB }, { 5753, 0xC6ED }, { 5754, 0xC6EE }, { 5755, 0xC6EF }, { 5756, 0xC6F1 }, { 5757, 0xC6F2 },
    { 5764, 0xC6F3 }, { 5765, 0xC6F4 }, { 5766, 0xC6F5 }, { 5767, 0xC6F6 }, { 5768, 0xC6F7 }, { 5769, 0xC6FA }, { 5770, 0xC6FB }, { 5771, 0xC6FC },
    { 5772, 0xC6FE }, { 5773, 0xC6FF }, { 5774, 0xC700 }, { 5775, 0xC701 }, { 5776, 0xC702 }, { 5777, 0xC703 }, { 5778, 0xC706 }, { 5779, 0xC707 },
    { 5780, 0xC709 }, { 5781, 0xC70A }, { 5782, 0xC70B }, { 5783, 0xC70D }, { 5784, 0xC70E }, { 5785, 0xC70F }, { 5786, 0xC710 }, { 5787, 0xC711 },
    { 5788, 0xC712 }, { 5789, 0xC713 }, { 5790, 0xC716 }, { 5791, 0xC718 }, { 5792, 0xC71A }, { 5793, 0xC71B }, { 5794, 0xC71C }, { 5795, 0xC71D },
    { 5796, 0xC71E }, { 5797, 0xC71F }, { 5798, 0xC722 }, { 5799, 0xC723 }, { 5800, 0xC725 }, { 5801, 0xC726 }, { 5802, 0xC727 }, { 5803, 0xC729 },
    { 5804, 0xC72A }, { 5805, 0xC72B }, { 5806, 0xC72C }, { 5807, 0xC72D }, { 5808, 0xC72E }, { 5809, 0xC72F }, { 5810, 0xC732 }, { 5811, 0xC734 },
    { 5812, 0xC736 }, { 5813, 0xC738 }, { 5814, 0xC739 }, { 5815, 0xC73A }, { 5816, 0xC73B }, { 5817, 0xC73E }, { 5818, 0xC73F }, { 5819, 0xC741 },
    { 5820, 0xC742 }, { 5821, 0xC743 }, { 5822, 0xC745 }, { 5823, 0xC746 }, { 5824, 0xC747 }, { 5825, 0xC748 }, { 5826, 0xC749 }, { 5827, 0xC74B },
    { 5828, 0xC74E }, { 5829, 0xC750 }, { 5830, 0xC759 }, { 5831, 0xC75A }, { 5832, 0xC75B }, { 5833, 0xC75D }, { 5834, 0xC75E }, { 5835, 0xC75F },
    { 5836, 0xC761 }, { 5837, 0xC762 }, { 5838, 0xC763 }, { 5839, 0xC764 }, { 5840, 0xC765 }, { 5841, 0xC766 }, { 5842, 0xC767 }, { 5843, 0xC769 },
    { 5844, 0xC76A }, { 5845, 0xC76C }, { 5846, 0xC76D }, { 5847, 0xC76E }, { 5848, 0xC76F }, { 5849, 0xC770 }, { 5850, 0xC771 }, { 5851, 0xC772 },
    { 5852, 0xC773 }, { 5853, 0xC776 }, { 5854, 0xC777 }, { 5855, 0xC779 }, { 5856, 0xC77A }, { 5857, 0xC77B }, { 5858, 0xC77F }, { 5859, 0xC780 },
    { 5860, 0xC781 }, { 5861, 0xC782 }, { 5862, 0xC786 }, { 5863, 0xC78B }, { 5864, 0xC78C }, { 5865, 0xC78D }, { 5866, 0xC78F }, { 5867, 0xC792 },
    { 5868, 0xC793 }, { 5869, 0xC795 }, { 5870, 0xC799 }, { 5871, 0xC79B }, { 5872, 0xC79C }, { 5873, 0xC79D }, { 5874, 0xC79E }, { 5875, 0xC79F },
    { 5876, 0xC7A2 }, { 5877, 0xC7A7 }, { 5878, 0xC7A8 }, { 5879, 0xC7A9 }, { 5880, 0xC7AA }, { 5881, 0xC7AB }, { 5882, 0xC7AE }, { 5883, 0xC7AF },
    { 5884, 0xC7B1 }, { 5885, 0xC7B2 }, { 5886, 0xC7B3 }, { 5887, 0xC7B5 }, { 5888, 0xC7B6 }, { 5889, 0xC7B7 }, { 5890, 0xC7B8 }, { 5891, 0xC7B9 },
    { 5892, 0xC7BA }, { 5893, 0xC7BB }, { 5894, 0xC7BE }, { 5895, 0xC7C2 }, { 5896, 0xC7C3 }, { 5897, 0xC7C4 }, { 5898, 0xC7C5 }, { 5899, 0xC7C6 },
    { 5900, 0xC7C7 }, { 5901, 0xC7CA }, { 5902, 0xC7CB }, { 5903, 0xC7CD }, { 5904, 0xC7CF }, { 5905, 0xC7D1 }, { 5906, 0xC7D2 }, { 5907, 0xC7D3 },
    { 5908, 0xC7D4 }, { 5909, 0xC7D5 }, { 5910, 0xC7D6 }, { 5911, 0xC7D7 }, { 5912, 0xC7D9 }, { 5913, 0xC7DA }, { 5914, 0xC7DB }, { 5915, 0xC7DC },
    { 5922, 0xC7DE }, { 5923, 0xC7DF }, { 5924, 0xC7E0 }, { 5925, 0xC7E1 }, { 5926, 0xC7E2 }, { 5927, 0xC7E3 }, { 5928, 0xC7E5 }, { 5929, 0xC7E6 },
    { 5930, 0xC7E7 }, { 5931, 0xC7E9 }, { 5932, 0xC7EA }, { 5933, 0xC7EB }, { 5934, 0xC7ED }, { 5935, 0xC7EE }, { 5936, 0xC7EF }, { 5937, 0xC7F0 },
    { 5938, 0xC7F1 }, { 5939, 0xC7F2 }, { 5940, 0xC7F3 }, { 5941, 0xC7F4 }, { 5942, 0xC7F5 }, { 5943, 0xC7F6 }, { 5944, 0xC7F7 }, { 5945, 0xC7F8 },
    { 5946, 0xC7F9 }, { 5947, 0xC7FA }, { 5954, 0xC7FB }, { 5955, 0xC7FC }, { 5956, 0xC7FD }, { 5957, 0xC7FE }, { 5958, 0xC7FF }, { 5959, 0xC802 },
    { 5960, 0xC803 }, { 5961, 0xC805 }, { 5962, 0xC806 }, { 5963, 0xC807 }, { 5964, 0xC809 }, { 5965, 0xC80B }, { 5966, 0xC80C }, { 5967, 0xC80D },
    { 5968, 0xC80E }, { 5969, 0xC80F }, { 5970, 0xC812 }, { 5971, 0xC814 }, { 5972, 0xC817 }, { 5973, 0xC818 }, { 5974, 0xC819 }, { 5975, 0xC81A },
    { 5976, 0xC81B }, { 5977, 0xC81E }, { 5978, 0xC81F }, { 5979, 0xC821 }, { 5980, 0xC822 }, { 5981, 0xC823 }, { 5982, 0xC825 }, { 5983, 0xC826 },
    { 5984, 0xC827 }, { 5985, 0xC828 }, { 5986, 0xC829 }, { 5987, 0xC82A }, { 5988, 0xC82B }, { 5989, 0xC82E }, { 5990, 0xC830 }, { 5991, 0xC832 },
    { 5992, 0xC833 }, { 5993, 0xC834 }, { 5994, 0xC835 }, { 5995, 0xC836 }, { 5996, 0xC837 }, { 5997, 0xC839 }, { 5998, 0xC83A }, { 5999, 0xC83B },
    { 6000, 0xC83D }, { 6001, 0xC83E }, { 6002, 0xC83F }, { 6003, 0xC841 }, { 6004, 0xC842 }, { 6005, 0xC843 }, { 6006, 0xC844 }, { 6007, 0xC845 },
    { 6008, 0xC846 }, { 6009, 0xC847 }, { 6010, 0xC84A }, { 6011, 0xC84B }, { 6012, 0xC84E }, { 6013, 0xC84F }, { 6014, 0xC850 }, { 6015, 0xC851 },
    { 6016, 0xC852 }, { 6017, 0xC853 }, { 6018, 0xC855 }, { 6019, 0xC856 }, { 6020, 0xC857 }, { 6021, 0xC858 }, { 6022, 0xC859 }, { 6023, 0xC85A },
    { 6024, 0xC85B }, { 6025, 0xC85C }, { 6026, 0xC85D }, { 6027, 0xC85E }, { 6028, 0xC85F }, { 6029, 0xC860 }, { 6030, 0xC861 }, { 6031, 0xC862 },
    { 6032, 0xC863 }, { 6033, 0xC864 }, { 6034, 0xC865 }, { 6035, 0xC866 }, { 6036, 0xC867 }, { 6037, 0xC868 }, { 6038, 0xC869 }, { 6039, 0xC86A },
    { 6040, 0xC86B }, { 6041, 0xC86C }, { 6042, 0xC86D }, { 6043, 0xC86E }, { 6044, 0xC86F }, { 6045, 0xC872 }, { 6046, 0xC873 }, { 6047, 0xC875 },
    { 6048, 0xC876 }, { 6049, 0xC877 }, { 6050, 0xC879 }, { 6051, 0xC87B }, { 6052, 0xC87C }, { 6053, 0xC87D }, { 6054, 0xC87E }, { 6055, 0xC87F },
    { 6056, 0xC882 }, { 6057, 0xC884 }, { 6058, 0xC888 }, { 6059, 0xC889 }, { 6060, 0xC88A }, { 6061, 0xC88E }, { 6062, 0xC88F }, { 6063, 0xC890 },
    { 6064, 0xC891 }, { 6065, 0xC892 }, { 6066, 0xC893 }, { 6067, 0xC895 }, { 6068, 0xC896 }, { 6069, 0xC897 }, { 6070, 0xC898 }, { 6071, 0xC899 },
    { 6072, 0xC89A }, { 6073, 0xC89B }, { 6074, 0xC89C }, { 6075, 0xC89E }, { 6076, 0xC8A0 }, { 6077, 0xC8A2 }, { 6078, 0xC8A3 }, { 6079, 0xC8A4 },
    { 6080, 0xC8A5 }, { 6081, 0xC8A6 }, { 6082, 0xC8A7 }, { 6083, 0xC8A9 }, { 6084, 0xC8AA }, { 6085, 0xC8AB }, { 6086, 0xC8AC }, { 6087, 0xC8AD },
    { 6088, 0xC8AE }, { 6089, 0xC8AF }, { 6090, 0xC8B0 }, { 6091, 0xC8B1 }, { 6092, 0xC8B2 }, { 6093, 0xC8B3 }, { 6094, 0xC8B4 }, { 6095, 0xC8B5 },
    { 6096, 0xC8B6 }, { 6097, 0xC8B7 }, { 6098, 0xC8B8 }, { 6099, 0xC8B9 }, { 6100, 0xC8BA }, { 6101, 0xC8BB }, { 6102, 0xC8BE }, { 6103, 0xC8BF },
    { 6104, 0xC8C0 }, { 6105, 0xC8C1 }, { 6112, 0xC8C2 }, { 6113, 0xC8C3 }, { 6114, 0xC8C5 }, { 6115, 0xC8C6 }, { 6116, 0xC8C7 }, { 6117, 0xC8C9 },
    { 6118, 0xC8CA }, { 6119, 0xC8CB }, { 6120, 0xC8CD }, { 6121, 0xC8CE }, { 6122, 0xC8CF }, { 6123, 0xC8D0 }, { 6124, 0xC8D1 }, { 6125, 0xC8D2 },
    { 6126, 0xC8D3 }, { 6127, 0xC8D6 }, { 6128, 0xC8D8 }, { 6129, 0xC8DA }, { 6130, 0xC8DB }, { 6131, 0xC8DC }, { 6132, 0xC8DD }, { 6133, 0xC8DE },
    { 6134, 0xC8DF }, { 6135, 0xC8E2 }, { 6136, 0xC8E3 }, { 6137, 0xC8E5 }, { 6144, 0xC8E6 }, { 6145, 0xC8E7 }, { 6146, 0xC8E8 }, { 6147, 0xC8E9 },
    { 6148, 0xC8EA }, { 6149, 0xC8EB }, { 6150, 0xC8EC }, { 6151, 0xC8ED }, { 6152, 0xC8EE }, { 6153, 0xC8EF }, { 6154, 0xC8F0 }, { 6155, 0xC8F1 },
    { 6156, 0xC8F2 }, { 6157, 0xC8F3 }, { 6158, 0xC8F4 }, { 6159, 0xC8F6 }, { 6160, 0xC8F7 }, { 6161, 0xC8F8 }, { 6162, 0xC8F9 }, { 6163, 0xC8FA },
    { 6164, 0xC8FB }, { 6165, 0xC8FE }, { 6166, 0xC8FF }, { 6167, 0xC901 }, { 6168, 0xC902 }, { 6169, 0xC903 }, { 6170, 0xC907 }, { 6171, 0xC908 },
    { 6172, 0xC909 }, { 6173, 0xC90A }, { 6174, 0xC90B }, { 6175, 0xC90E }, { 6176, 0x3000 }, { 6177, 0x3001 }, { 6178, 0x3002 }, { 6179, 0x00B7 },
    { 6180, 0x2025 }, { 6181, 0x2026 }, { 6182, 0x00A8 }, { 6183, 0x3003 }, { 6184, 0x00AD }, { 6185, 0x2015 }, { 6186, 0x2225 }, { 6187, 0xFF3C },
    { 6188, 0x223C }, { 6189, 0x2018 }, { 6190, 0x2019 }, { 6191, 0x201C }, { 6192, 0x201D }, { 6193, 0x3014 }, { 6194, 0x3015 }, { 6195, 0x3008 },
    { 6196, 0x3009 }, { 6197, 0x300A }, { 6198, 0x300B }, { 6199, 0x300C }, { 6200, 0x300D }, { 6201, 0x300E }, { 6202, 0x300F }, { 6203, 0x3010 },
    { 6204, 0x3011 }, { 6205, 0x00B1 }, { 6206, 0x00D7 }, { 6207, 0x00F7 }, { 6208, 0x2260 }, { 6209, 0x2264 }, { 6210, 0x2265 }, { 6211, 0x221E },
    { 6212, 0x2234 }, { 6213, 0x00B0 }, { 6214, 0x2032 }, { 6215, 0x2033 }, { 6216, 0x2103 }, { 6217, 0x212B }, { 6218, 0xFFE0 }, { 6219, 0xFFE1 },
    { 6220, 0xFFE5 }, { 6221, 0x2642 }, { 6222, 0x2640 }, { 6223, 0x2220 }, { 6224, 0x22A5 }, { 6225, 0x2312 }, { 6226, 0x2202 }, { 6227, 0x2207 },
    { 6228, 0x2261 }, { 6229, 0x2252 }, { 6230, 0x00A7 }, { 6231, 0x203B }, { 6232, 0x2606 }, { 6233, 0x2605 }, { 6234, 0x25CB }, { 6235, 0x25CF },
    { 6236, 0x25CE }, { 6237, 0x25C7 }, { 6238, 0x25C6 }, { 6239, 0x25A1 }, { 6240, 0x25A0 }, { 6241, 0x25B3 }, { 6242, 0x25B2 }, { 6243, 0x25BD },
    { 6244, 0x25BC }, { 6245, 0x2192 }, { 6246, 0x2190 }, { 6247, 0x2191 }, { 6248, 0x2193 }, { 6249, 0x2194 }, { 6250, 0x3013 }, { 6251, 0x226A },
    { 6252, 0x226B }, { 6253, 0x221A }, { 6254, 0x223D }, { 6255, 0x221D }, { 6256, 0x2235 }, { 6257, 0x222B }, { 6258, 0x222C }, { 6259, 0x2208 },
    { 6260, 0x220B }, { 6261, 0x2286 }, { 6262, 0x2287 }, { 6263, 0x2282 }, { 6264, 0x2283 }, { 6265, 0x222A }, { 6266, 0x2229 }, { 6267, 0x2227 },
    { 6268, 0x2228 }, { 6269, 0xFFE2 }, { 6270, 0xC910 }, { 6271, 0xC912 }, { 6272, 0xC913 }, { 6273, 0xC914 }, { 6274, 0xC915 }, { 6275, 0xC916 },
    { 6276, 0xC917 }, { 6277, 0xC919 }, { 6278, 0xC91A }, { 6279, 0xC91B }, { 6280, 0xC91C }, { 6281, 0xC91D }, { 6282, 0xC91E }, { 6283, 0xC91F },
    { 6284, 0xC920 }, { 6285, 0xC921 }, { 6286, 0xC922 }, { 6287, 0xC923 }, { 6288, 0xC924 }, { 6289, 0xC925 }, { 6290, 0xC926 }, { 6291, 0xC927 },
    { 6292, 0xC928 }, { 6293, 0xC929 }, { 6294, 0xC92A }, { 6295, 0xC92B }, { 6302, 0xC92D }, { 6303, 0xC92E }, { 6304, 0xC92F }, { 6305, 0xC930 },
    { 6306, 0xC931 }, { 6307, 0xC932 }, { 6308, 0xC933 }, { 6309, 0xC935 }, { 6310, 0xC936 }, { 6311, 0xC937 }, { 6312, 0xC938 }, { 6313, 0xC939 },
    { 6314, 0xC93A }, { 6315, 0xC93B }, { 6316, 0xC93C }, { 6317, 0xC93D }, { 6318, 0xC93E }, { 6319, 0xC93F }, { 6320, 0xC940 }, { 6321, 0xC941 },
    { 6322, 0xC942 }, { 6323, 0xC943 }, { 6324, 0xC944 }, { 6325, 0xC945 }, { 6326, 0xC946 }, { 6327, 0xC947 }, { 6334, 0xC948 }, { 6335, 0xC949 },
    { 6336, 0xC94A }, { 6337, 0xC94B }, { 6338, 0xC94C }, { 6339, 0xC94D }, { 6340, 0xC94E }, { 6341, 0xC94F }, { 6342, 0xC952 }, { 6343, 0xC953 },
    { 6344, 0xC955 }, { 6345, 0xC956 }, { 6346, 0xC957 }, { 6347, 0xC959 }, { 6348, 0xC95A }, { 6349, 0xC95B }, { 6350, 0xC95C }, { 6351, 0xC95D },
    { 6352, 0xC95E }, { 6353, 0xC95F }, { 6354, 0xC962 }, { 6355, 0xC964 }, { 6356, 0xC965 }, { 6357, 0xC966 }, { 6358, 0xC967 }, { 6359, 0xC968 },
    { 6360, 0xC969 }, { 6361, 0xC96A }, { 6362, 0xC96B }, { 6363, 0xC96D }, { 6364, 0xC96E }, { 6365, 0xC96F }, { 6366, 0x21D2 }, { 6367, 0x21D4 },
    { 6368, 0x2200 }, { 6369, 0x2203 }, { 6370, 0x00B4 }, { 6371, 0xFF5E }, { 6372, 0x02C7 }, { 6373, 0x02D8 }, { 6374, 0x02DD }, { 6375, 0x02DA },
    { 6376, 0x02D9 }, { 6377, 0x00B8 }, { 6378, 0x02DB }, { 6379, 0x00A1 }, { 6380, 0x00BF }, { 6381, 0x02D0 }, { 6382, 0x222E }, { 6383, 0x2211 },
    { 6384, 0x220F }, { 6385, 0x00A4 }, { 6386, 0x2109 }, { 6387, 0x2030 }, { 6388, 0x25C1 }, { 6389, 0x25C0 }, { 6390, 0x25B7 }, { 6391, 0x25B6 },
    { 6392, 0x2664 }, { 6393, 0x2660 }, { 6394, 0x2661 }, { 6395, 0x2665 }, { 6396, 0x2667 }, { 6397, 0x2663 }, { 6398, 0x2299 }, { 6399, 0x25C8 },
    { 6400, 0x25A3 }, { 6401, 0x25D0 }, { 6402, 0x25D1 }, { 6403, 0x2592 }, { 6404, 0x25A4 }, { 6405, 0x25A5 }, { 6406, 0x25A8 }, { 6407, 0x25A7 },
    { 6408, 0x25A6 }, { 6409, 0x25A9 }, { 6410, 0x2668 }, { 6411, 0x260F }, { 6412, 0x260E }, { 6413, 0x261C }, { 6414, 0x261E }, { 6415, 0x00B6 },
    { 6416, 0x2020 }, { 6417, 0x2021 }, { 6418, 0x2195 }, { 6419, 0x2197 }, { 6420, 0x2199 }, { 6421, 0x2196 }, { 6422, 0x2198 }, { 6423, 0x266D },
    { 6424, 0x2669 }, { 6425, 0x266A }, { 6426, 0x266C }, { 6427, 0x327F }, { 6428, 0x321C }, { 6429, 0x2116 }, { 6430, 0x33C7 }, { 6431, 0x2122 },
    { 6432, 0x33C2 }, { 6433, 0x33D8 }, { 6434, 0x2121 }, { 6435, 0x20AC }, { 6436, 0x00AE }, { 6460, 0xC971 }, { 6461, 0xC972 }, { 6462, 0xC973 },
    { 6463, 0xC975 }, { 6464, 0xC976 }, { 6465, 0xC977 }, { 6466, 0xC978 }, { 6467, 0xC979 }, { 6468, 0xC97A }, { 6469, 0xC97B }, { 6470, 0xC97D },
    { 6471, 0xC97E }, { 6472, 0xC97F }, { 6473, 0xC980 }, { 6474, 0xC981 }, { 6475, 0xC982 }, { 6476, 0xC983 }, { 6477, 0xC984 }, { 6478, 0xC985 },
    { 6479, 0xC986 }, { 6480, 0xC987 }, { 6481, 0xC98A }, { 6482, 0xC98B }, { 6483, 0xC98D }, { 6484, 0xC98E }, { 6485, 0xC98F }, { 6492, 0xC991 },
    { 6493, 0xC992 }, { 6494, 0xC993 }, { 6495, 0xC994 }, { 6496, 0xC995 }, { 6497, 0xC996 }, { 6498, 0xC997 }, { 6499, 0xC99A }, { 6500, 0xC99C },
    { 6501, 0xC99E }, { 6502, 0xC99F }, { 6503, 0xC9A0 }, { 6504, 0xC9A1 }, { 6505, 0xC9A2 }, { 6506, 0xC9A3 }, { 6507, 0xC9A4 }, { 6508, 0xC9A5 },
    { 6509, 0xC9A6 }, { 6510, 0xC9A7 }, { 6511, 0xC9A8 }, { 6512, 0xC9A9 }, { 6513, 0xC9AA }, { 6514, 0xC9AB }, { 6515, 0xC9AC }, { 6516, 0xC9AD },
    { 6517, 0xC9AE }, { 6524, 0xC9AF }, { 6525, 0xC9B0 }, { 6526, 0xC9B1 }, { 6527, 0xC9B2 }, { 6528, 0xC9B3 }, { 6529, 0xC9B4 }, { 6530, 0xC9B5 },
    { 6531, 0xC9B6 }, { 6532, 0xC9B7 }, { 6533, 0xC9B8 }, { 6534, 0xC9B9 }, { 6535, 0xC9BA }, { 6536, 0xC9BB }, { 6537, 0xC9BC }, { 6538, 0xC9BD },
    { 6539, 0xC9BE }, { 6540, 0xC9BF }, { 6541, 0xC9C2 }, { 6542, 0xC9C3 }, { 6543, 0xC9C5 }, { 6544, 0xC9C6 }, { 6545, 0xC9C9 }, { 6546, 0xC9CB },
    { 6547, 0xC9CC }, { 6548, 0xC9CD }, { 6549, 0xC9CE }, { 6550, 0xC9CF }, { 6551, 0xC9D2 }, { 6552, 0xC9D4 }, { 6553, 0xC9D7 }, { 6554, 0xC9D8 },
    { 6555, 0xC9DB }, { 6556, 0xFF01 }, { 6557, 0xFF02 }, { 6558, 0xFF03 }, { 6559, 0xFF04 }, { 6560, 0xFF05 }, { 6561, 0xFF06 }, { 6562, 0xFF07 },
    { 6563, 0xFF08 }, { 6564, 0xFF09 }, { 6565, 0xFF0A }, { 6566, 0xFF0B }, { 6567, 0xFF0C }, { 6568, 0xFF0D }, { 6569, 0xFF0E }, { 6570, 0xFF0F },
    { 6571, 0xFF10 }, { 6572, 0xFF11 }, { 6573, 0xFF12 }, { 6574, 0xFF13 }, { 6575, 0xFF14 }, { 6576, 0xFF15 }, { 6577, 0xFF16 }, { 6578, 0xFF17 },
    { 6579, 0xFF18 }, { 6580, 0xFF19 }, { 6581, 0xFF1A }, { 6582, 0xFF1B }, { 6583, 0xFF1C }, { 6584, 0xFF1D }, { 6585, 0xFF1E }, { 6586, 0xFF1F },
    { 6587, 0xFF20 }, { 6588, 0xFF21 }, { 6589, 0xFF22 }, { 6590, 0xFF23 }, { 6591, 0xFF24 }, { 6592, 0xFF25 }, { 6593, 0xFF26 }, { 6594, 0xFF27 },
    { 6595, 0xFF28 }, { 6596, 0xFF29 }, { 6597, 0xFF2A }, { 6598, 0xFF2B }, { 6599, 0xFF2C }, { 6600, 0xFF2D }, { 6601, 0xFF2E }, { 6602, 0xFF2F },
    { 6603, 0xFF30 }, { 6604, 0xFF31 }, { 6605, 0xFF32 }, { 6606, 0xFF33 }, { 6607, 0xFF34 }, { 6608, 0xFF35 }, { 6609, 0xFF36 }, { 6610, 0xFF37 },
    { 6611, 0xFF38 }, { 6612, 0xFF39 }, { 6613, 0xFF3A }, { 6614, 0xFF3B }, { 6615, 0xFFE6 }, { 6616, 0xFF3D }, { 6617, 0xFF3E }, { 6618, 0xFF3F },
    { 6619, 0xFF40 }, { 6620, 0xFF41 }, { 6621, 0xFF42 }, { 6622, 0xFF43 }, { 6623, 0xFF44 }, { 6624, 0xFF45 }, { 6625, 0xFF46 }, { 6626, 0xFF47 },
    { 6627, 0xFF48 }, { 6628, 0xFF49 }, { 6629, 0xFF4A }, { 6630, 0xFF4B }, { 6631, 0xFF4C }, { 6632, 0xFF4D }, { 6633, 0xFF4E }, { 6634, 0xFF4F },
    { 6635, 0xFF50 }, { 6636, 0xFF51 }, { 6637, 0xFF52 }, { 6638, 0xFF53 }, { 6639, 0xFF54 }, { 6640, 0xFF55 }, { 6641, 0xFF56 }, { 6642, 0xFF57 },
    { 6643, 0xFF58 }, { 6644, 0xFF59 }, { 6645, 0xFF5A }, { 6646, 0xFF5B }, { 6647, 0xFF5C }, { 6648, 0xFF5D }, { 6649, 0xFFE3 }, { 6650, 0xC9DE },
    { 6651, 0xC9DF }, { 6652, 0xC9E1 }, { 6653, 0xC9E3 }, { 6654, 0xC9E5 }, { 6655, 0xC9E6 }, { 6656, 0xC9E8 }, { 6657, 0xC9E9 }, { 6658, 0xC9EA },
    { 6659, 0xC9EB }, { 6660, 0xC9EE }, { 6661, 0xC9F2 }, { 6662, 0xC9F3 }, { 6663, 0xC9F4 }, { 6664, 0xC9F5 }, { 6665, 0xC9F6 }, { 6666, 0xC9F7 },
    { 6667, 0xC9FA }, { 6668, 0xC9FB }, { 6669, 0xC9FD }, { 6670, 0xC9FE }, { 6671, 0xC9FF }, { 6672, 0xCA01 }, { 6673, 0xCA02 }, { 6674, 0xCA03 },
    { 6675, 0xCA04 }, { 6682, 0xCA05 }, { 6683, 0xCA06 }, { 6684, 0xCA07 }, { 6685, 0xCA0A }, { 6686, 0xCA0E }, { 6687, 0xCA0F }, { 6688, 0xCA10 },
    { 6689, 0xCA11 }, { 6690, 0xCA12 }, { 6691, 0xCA13 }, { 6692, 0xCA15 }, { 6693, 0xCA16 }, { 6694, 0xCA17 }, { 6695, 0xCA19 }, { 6696, 0xCA1A },
    { 6697, 0xCA1B }, { 6698, 0xCA1C }, { 6699, 0xCA1D }, { 6700, 0xCA1E }, { 6701, 0xCA1F }, { 6702, 0xCA20 }, { 6703, 0xCA21 }, { 6704, 0xCA22 },
    { 6705, 0xCA23 }, { 6706, 0xCA24 }, { 6707, 0xCA25 }, { 6714, 0xCA26 }, { 6715, 0xCA27 }, { 6716, 0xCA28 }, { 6717, 0xCA2A }, { 6718, 0xCA2B },
    { 6719, 0xCA2C }, { 6720, 0xCA2D }, { 6721, 0xCA2E }, { 6722, 0xCA2F }, { 6723, 0xCA30 }, { 6724, 0xCA31 }, { 6725, 0xCA32 }, { 6726, 0xCA33 },
    { 6727, 0xCA34 }, { 6728, 0xCA35 }, { 6729, 0xCA36 }, { 6730, 0xCA37 }, { 6731, 0xCA38 }, { 6732, 0xCA39 }, { 6733, 0xCA3A }, { 6734, 0xCA3B },
    { 6735, 0xCA3C }, { 6736, 0xCA3D }, { 6737, 0xCA3E }, { 6738, 0xCA3F }, { 6739, 0xCA40 }, { 6740, 0xCA41 }, { 6741, 0xCA42 }, { 6742, 0xCA43 },
    { 6743, 0xCA44 }, { 6744, 0xCA45 }, { 6745, 0xCA46 }, { 6746, 0x3131 }, { 6747, 0x3132 }, { 6748, 0x3133 }, { 6749, 0x3134 }, { 6750, 0x3135 },
    { 6751, 0x3136 }, { 6752, 0x3137 }, { 6753, 0x3138 }, { 6754, 0x3139 }, { 6755, 0x313A }, { 6756, 0x313B }, { 6757, 0x313C }, { 6758, 0x313D },
    { 6759, 0x313E }, { 6760, 0x313F }, { 6761, 0x3140 }, { 6762, 0x3141 }, { 6763, 0x3142 }, { 6764, 0x3143 }, { 6765, 0x3144 }, { 6766, 0x3145 },
    { 6767, 0x3146 }, { 6768, 0x3147 }, { 6769, 0x3148 }, { 6770, 0x3149 }, { 6771, 0x314A }, { 6772, 0x314B }, { 6773, 0x314C }, { 6774, 0x314D },
    { 6775, 0x314E }, { 6776, 0x314F }, { 6777, 0x3150 }, { 6778, 0x3151 }, { 6779, 0x3152 }, { 6780, 0x3153 }, { 6781, 0x3154 }, { 6782, 0x3155 },
    { 6783, 0x3156 }, { 6784, 0x3157 }, { 6785, 0x3158 }, { 6786, 0x3159 }, { 6787, 0x315A }, { 6788, 0x315B }, { 6789, 0x315C }, { 6790, 0x315D },
    { 6791, 0x315E }, { 6792, 0x315F }, { 6793, 0x3160 }, { 6794, 0x3161 }, { 6795, 0x3162 }, { 6796, 0x3163 }, { 6797, 0x3164 }, { 6798, 0x3165 },
    { 6799, 0x3166 }, { 6800, 0x3167 }, { 6801, 0x3168 }, { 6802, 0x3169 }, { 6803, 0x316A }, { 6804, 0x316B }, { 6805, 0x316C }, { 6806, 0x316D },
    { 6807, 0x316E }, { 6808, 0x316F }, { 6809, 0x3170 }, { 6810, 0x3171 }, { 6811, 0x3172 }, { 6812, 0x3173 }, { 6813, 0x3174 }, { 6814, 0x3175 },
    { 6815, 0x3176 }, { 6816, 0x3177 }, { 6817, 0x3178 }, { 6818, 0x3179 }, { 6819, 0x317A }, { 6820, 0x317B }, { 6821, 0x317C }, { 6822, 0x317D },
    { 6823, 0x317E }, { 6824, 0x317F }, { 6825, 0x3180 }, { 6826, 0x3181 }, { 6827, 0x3182 }, { 6828, 0x3183 }, { 6829, 0x3184 }, { 6830, 0x3185 },
    { 6831, 0x3186 }, { 6832, 0x3187 }, { 6833, 0x3188 }, { 6834, 0x3189 }, { 6835, 0x318A }, { 6836, 0x318B }, { 6837, 0x318C }, { 6838, 0x318D },
    { 6839, 0x318E }, { 6840, 0xCA47 }, { 6841, 0xCA48 }, { 6842, 0xCA49 }, { 6843, 0xCA4A }, { 6844, 0xCA4B }, { 6845, 0xCA4E }, { 6846, 0xCA4F },
    { 6847, 0xCA51 }, { 6848, 0xCA52 }, { 6849, 0xCA53 }, { 6850, 0xCA55 }, { 6851, 0xCA56 }, { 6852, 0xCA57 }, { 6853, 0xCA58 }, { 6854, 0xCA59 },
    { 6855, 0xCA5A }, { 6856, 0xCA5B }, { 6857, 0xCA5E }, { 6858, 0xCA62 }, { 6859, 0xCA63 }, { 6860, 0xCA64 }, { 6861, 0xCA65 }, { 6862, 0xCA66 },
    { 6863, 0xCA67 }, { 6864, 0xCA69 }, { 6865, 0xCA6A }, { 6872, 0xCA6B }, { 6873, 0xCA6C }, { 6874, 0xCA6D }, { 6875, 0xCA6E }, { 6876, 0xCA6F },
    { 6877, 0xCA70 }, { 6878, 0xCA71 }, { 6879, 0xCA72 }, { 6880, 0xCA73 }, { 6881, 0xCA74 }, { 6882, 0xCA75 }, { 6883, 0xCA76 }, { 6884, 0xCA77 },
    { 6885, 0xCA78 }, { 6886, 0xCA79 }, { 6887, 0xCA7A }, { 6888, 0xCA7B }, { 6889, 0xCA7C }, { 6890, 0xCA7E }, { 6891, 0xCA7F }, { 6892, 0xCA80 },
    { 6893, 0xCA81 }, { 6894, 0xCA82 }, { 6895, 0xCA83 }, { 6896, 0xCA85 }, { 6897, 0xCA86 }, { 6904, 0xCA87 }, { 6905, 0xCA88 }, { 6906, 0xCA89 },
    { 6907, 0xCA8A }, { 6908, 0xCA8B }, { 6909, 0xCA8C }, { 6910, 0xCA8D }, { 6911, 0xCA8E }, { 6912, 0xCA8F }, { 6913, 0xCA90 }, { 6914, 0xCA91 },
    { 6915, 0xCA92 }, { 6916, 0xCA93 }, { 6917, 0xCA94 }, { 6918, 0xCA95 }, { 6919, 0xCA96 }, { 6920, 0xCA97 }, { 6921, 0xCA99 }, { 6922, 0xCA9A },
    { 6923, 0xCA9B }, { 6924, 0xCA9C }, { 6925, 0xCA9D }, { 6926, 0xCA9E }, { 6927, 0xCA9F }, { 6928, 0xCAA0 }, { 6929, 0xCAA1 }, { 6930, 0xCAA2 },
    { 6931, 0xCAA3 }, { 6932, 0xCAA4 }, { 6933, 0xCAA5 }, { 6934, 0xCAA6 }, { 6935, 0xCAA7 }, { 6936, 0x2170 }, { 6937, 0x2171 }, { 6938, 0x2172 },
    { 6939, 0x2173 }, { 6940, 0x2174 }, { 6941, 0x2175 }, { 6942, 0x2176 }, { 6943, 0x2177 }, { 6944, 0x2178 }, { 6945, 0x2179 }, { 6951, 0x2160 },
    { 6952, 0x2161 }, { 6953, 0x2162 }, { 6954, 0x2163 }, { 6955, 0x2164 }, { 6956, 0x2165 }, { 6957, 0x2166 }, { 6958, 0x2167 }, { 6959, 0x2168 },
    { 6960, 0x2169 }, { 6968, 0x0391 }, { 6969, 0x0392 }, { 6970, 0x0393 }, { 6971, 0x0394 }, { 6972, 0x0395 }, { 6973, 0x0396 }, { 6974, 0x0397 },
    { 6975, 0x0398 }, { 6976, 0x0399 }, { 6977, 0x039A }, { 6978, 0x039B }, { 6979, 0x039C }, { 6980, 0x039D }, { 6981, 0x039E }, { 6982, 0x039F },
    { 6983, 0x03A0 }, { 6984, 0x03A1 }, { 6985, 0x03A3 }, { 6986, 0x03A4 }, { 6987, 0x03A5 }, { 6988, 0x03A6 }, { 6989, 0x03A7 }, { 6990, 0x03A8 },
    { 6991, 0x03A9 }, { 7000, 0x03B1 }, { 7001, 0x03B2 }, { 7002, 0x03B3 }, { 7003, 0x03B4 }, { 7004, 0x03B5 }, { 7005, 0x03B6 }, { 7006, 0x03B7 },
    { 7007, 0x03B8 }, { 7008, 0x03B9 }, { 7009, 0x03BA }, { 7010, 0x03BB }, { 7011, 0x03BC }, { 7012, 0x03BD }, { 7013, 0x03BE }, { 7014, 0x03BF },
    { 7015, 0x03C0 }, { 7016, 0x03C1 }, { 7017, 0x03C3 }, { 7018, 0x03C4 }, { 7019, 0x03C5 }, { 7020, 0x03C6 }, { 7021, 0x03C7 }, { 7022, 0x03C8 },
    { 7023, 0x03C9 }, { 7030, 0xCAA8 }, { 7031, 0xCAA9 }, { 7032, 0xCAAA }, { 7033, 0xCAAB }, { 7034, 0xCAAC }, { 7035, 0xCAAD }, { 7036, 0xCAAE },
    { 7037, 0xCAAF }, { 7038, 0xCAB0 }, { 7039, 0xCAB1 }, { 7040, 0xCAB2 }, { 7041, 0xCAB3 }, { 7042, 0xCAB4 }, { 7043, 0xCAB5 }, { 7044, 0xCAB6 },
    { 7045, 0xCAB7 }, { 7046, 0xCAB8 }, { 7047, 0xCAB9 }, { 7048, 0xCABA }, { 7049, 0xCABB }, { 7050, 0xCABE }, { 7051, 0xCABF }, { 7052, 0xCAC1 },
    { 7053, 0xCAC2 }, { 7054, 0xCAC3 }, { 7055, 0xCAC5 }, { 7062, 0xCAC6 }, { 7063, 0xCAC7 }, { 7064, 0xCAC8 }, { 7065, 0xCAC9 }, { 7066, 0xCACA },
    { 7067, 0xCACB }, { 7068, 0xCACE }, { 7069, 0xCAD0 }, { 7070, 0xCAD2 }, { 7071, 0xCAD4 }, { 7072, 0xCAD5 }, { 7073, 0xCAD6 }, { 7074, 0xCAD7 },
    { 7075, 0xCADA }, { 7076, 0xCADB }, { 7077, 0xCADC }, { 7078, 0xCADD }, { 7079, 0xCADE }, { 7080, 0xCADF }, { 7081, 0xCAE1 }, { 7082, 0xCAE2 },
    { 7083, 0xCAE3 }, { 7084, 0xCAE4 }, { 7085, 0xCAE5 }, { 7086, 0xCAE6 }, { 7087, 0xCAE7 }, { 7094, 0xCAE8 }, { 7095, 0xCAE9 }, { 7096, 0xCAEA },
    { 7097, 0xCAEB }, { 7098, 0xCAED }, { 7099, 0xCAEE }, { 7100, 0xCAEF }, { 7101, 0xCAF0 }, { 7102, 0xCAF1 }, { 7103, 0xCAF2 }, { 7104, 0xCAF3 },
    { 7105, 0xCAF5 }, { 7106, 0xCAF6 }, { 7107, 0xCAF7 }, { 7108, 0xCAF8 }, { 7109, 0xCAF9 }, { 7110, 0xCAFA }, { 7111, 0xCAFB }, { 7112, 0xCAFC },
    { 7113, 0xCAFD }, { 7114, 0xCAFE }, { 7115, 0xCAFF }, { 7116, 0xCB00 }, { 7117, 0xCB01 }, { 7118, 0xCB02 }, { 7119, 0xCB03 }, { 7120, 0xCB04 },
    { 7121, 0xCB05 }, { 7122, 0xCB06 }, { 7123, 0xCB07 }, { 7124, 0xCB09 }, { 7125, 0xCB0A }, { 7126, 0x2500 }, { 7127, 0x2502 }, { 7128, 0x250C },
    { 7129, 0x2510 }, { 7130, 0x2518 }, { 7131, 0x2514 }, { 7132, 0x251C }, { 7133, 0x252C }, { 7134, 0x2524 }, { 7135, 0x2534 }, { 7136, 0x253C },
    { 7137, 0x2501 }, { 7138, 0x2503 }, { 7139, 0x250F }, { 7140, 0x2513 }, { 7141, 0x251B }, { 7142, 0x2517 }, { 7143, 0x2523 }, { 7144, 0x2533 },
    { 7145, 0x252B }, { 7146, 0x253B }, { 7147, 0x254B }, { 7148, 0x2520 }, { 7149, 0x252F }, { 7150, 0x2528 }, { 7151, 0x2537 }, { 7152, 0x253F },
    { 7153, 0x251D }, { 7154, 0x2530 }, { 7155, 0x2525 }, { 7156, 0x2538 }, { 7157, 0x2542 }, { 7158, 0x2512 }, { 7159, 0x2511 }, { 7160, 0x251A },
    { 7161, 0x2519 }, { 7162, 0x2516 }, { 7163, 0x2515 }, { 7164, 0x250E }, { 7165, 0x250D }, { 7166, 0x251E }, { 7167, 0x251F }, { 7168, 0x2521 },
    { 7169, 0x2522 }, { 7170, 0x2526 }, { 7171, 0x2527 }, { 7172, 0x2529 }, { 7173, 0x252A }, { 7174, 0x252D }, { 7175, 0x252E }, { 7176, 0x2531 },
    { 7177, 0x2532 }, { 7178, 0x2535 }, { 7179, 0x2536 }, { 7180, 0x2539 }, { 7181, 0x253A }, { 7182, 0x253D }, { 7183, 0x253E }, { 7184, 0x2540 },
    { 7185, 0x2541 }, { 7186, 0x2543 }, { 7187, 0x2544 }, { 7188, 0x2545 }, { 7189, 0x2546 }, { 7190, 0x2547 }, { 7191, 0x2548 }, { 7192, 0x2549 },
    { 7193, 0x254A }, { 7220, 0xCB0B }, { 7221, 0xCB0C }, { 7222, 0xCB0D }, { 7223, 0xCB0E }, { 7224, 0xCB0F }, { 7225, 0xCB11 }, { 7226, 0xCB12 },
    { 7227, 0xCB13 }, { 7228, 0xCB15 }, { 7229, 0xCB16 }, { 7230, 0xCB17 }, { 7231, 0xCB19 }, { 7232, 0xCB1A }, { 7233, 0xCB1B }, { 7234, 0xCB1C },
    { 7235, 0xCB1D }, { 7236, 0xCB1E }, { 7237, 0xCB1F }, { 7238, 0xCB22 }, { 7239, 0xCB23 }, { 7240, 0xCB24 }, { 7241, 0xCB25 }, { 7242, 0xCB26 },
    { 7243, 0xCB27 }, { 7244, 0xCB28 }, { 7245, 0xCB29 }, { 7252, 0xCB2A }, { 7253, 0xCB2B }, { 7254, 0xCB2C }, { 7255, 0xCB2D }, { 7256, 0xCB2E },
    { 7257, 0xCB2F }, { 7258, 0xCB30 }, { 7259, 0xCB31 }, { 7260, 0xCB32 }, { 7261, 0xCB33 }, { 7262, 0xCB34 }, { 7263, 0xCB35 }, { 7264, 0xCB36 },
    { 7265, 0xCB37 }, { 7266, 0xCB38 }, { 7267, 0xCB39 }, { 7268, 0xCB3A }, { 7269, 0xCB3B }, { 7270, 0xCB3C }, { 7271, 0xCB3D }, { 7272, 0xCB3E },
    { 7273, 0xCB3F }, { 7274, 0xCB40 }, { 7275, 0xCB42 }, { 7276, 0xCB43 }, { 7277, 0xCB44 }, { 7284, 0xCB45 }, { 7285, 0xCB46 }, { 7286, 0xCB47 },
    { 7287, 0xCB4A }, { 7288, 0xCB4B }, { 7289, 0xCB4D }, { 7290, 0xCB4E }, { 7291, 0xCB4F }, { 7292, 0xCB51 }, { 7293, 0xCB52 }, { 7294, 0xCB53 },
    { 7295, 0xCB54 }, { 7296, 0xCB55 }, { 7297, 0xCB56 }, { 7298, 0xCB57 }, { 7299, 0xCB5A }, { 7300, 0xCB5B }, { 7301, 0xCB5C }, { 7302, 0xCB5E },
    { 7303, 0xCB5F }, { 7304, 0xCB60 }, { 7305, 0xCB61 }, { 7306, 0xCB62 }, { 7307, 0xCB63 }, { 7308, 0xCB65 }, { 7309, 0xCB66 }, { 7310, 0xCB67 },
    { 7311, 0xCB68 }, { 7312, 0xCB69 }, { 7313, 0xCB6A }, { 7314, 0xCB6B }, { 7315, 0xCB6C }, { 7316, 0x3395 }, { 7317, 0x3396 }, { 7318, 0x3397 },
    { 7319, 0x2113 }, { 7320, 0x3398 }, { 7321, 0x33C4 }, { 7322, 0x33A3 }, { 7323, 0x33A4 }, { 7324, 0x33A5 }, { 7325, 0x33A6 }, { 7326, 0x3399 },
    { 7327, 0x339A }, { 7328, 0x339B }, { 7329, 0x339C }, { 7330, 0x339D }, { 7331, 0x339E }, { 7332, 0x339F }, { 7333, 0x33A0 }, { 7334, 0x33A1 },
    { 7335, 0x33A2 }, { 7336, 0x33CA }, { 7337, 0x338D }, { 7338, 0x338E }, { 7339, 0x338F }, { 7340, 0x33CF }, { 7341, 0x3388 }, { 7342, 0x3389 },
    { 7343, 0x33C8 }, { 7344, 0x33A7 }, { 7345, 0x33A8 }, { 7346, 0x33B0 }, { 7347, 0x33B1 }, { 7348, 0x33B2 }, { 7349, 0x33B3 }, { 7350, 0x33B4 },
    { 7351, 0x33B5 }, { 7352, 0x33B6 }, { 7353, 0x33B7 }, { 7354, 0x33B8 }, { 7355, 0x33B9 }, { 7356, 0x3380 }, { 7357, 0x3381 }, { 7358, 0x3382 },
    { 7359, 0x3383 }, { 7360, 0x3384 }, { 7361, 0x33BA }, { 7362, 0x33BB }, { 7363, 0x33BC }, { 7364, 0x33BD }, { 7365, 0x33BE }, { 7366, 0x33BF },
    { 7367, 0x3390 }, { 7368, 0x3391 }, { 7369, 0x3392 }, { 7370, 0x3393 }, { 7371, 0x3394 }, { 7372, 0x2126 }, { 7373, 0x33C0 }, { 7374, 0x33C1 },
    { 7375, 0x338A }, { 7376, 0x338B }, { 7377, 0x338C }, { 7378, 0x33D6 }, { 7379, 0x33C5 }, { 7380, 0x33AD }, { 7381, 0x33AE }, { 7382, 0x33AF },
    { 7383, 0x33DB }, { 7384, 0x33A9 }, { 7385, 0x33AA }, { 7386, 0x33AB }, { 7387, 0x33AC }, { 7388, 0x33DD }, { 7389, 0x33D0 }, { 7390, 0x33D3 },
    { 7391, 0x33C3 }, { 7392, 0x33C9 }, { 7393, 0x33DC }, { 7394, 0x33C6 }, { 7410, 0xCB6D }, { 7411, 0xCB6E }, { 7412, 0xCB6F }, { 7413, 0xCB70 },
    { 7414, 0xCB71 }, { 7415, 0xCB72 }, { 7416, 0xCB73 }, { 7417, 0xCB74 }, { 7418, 0xCB75 }, { 7419, 0xCB76 }, { 7420, 0xCB77 }, { 7421, 0xCB7A },
    { 7422, 0xCB7B }, { 7423, 0xCB7C }, { 7424, 0xCB7D }, { 7425, 0xCB7E }, { 7426, 0xCB7F }, { 7427, 0xCB80 }, { 7428, 0xCB81 }, { 7429, 0xCB82 },
    { 7430, 0xCB83 }, { 7431, 0xCB84 }, { 7432, 0xCB85 }, { 7433, 0xCB86 }, { 7434, 0xCB87 }, { 7435, 0xCB88 }, { 7442, 0xCB89 }, { 7443, 0xCB8A },
    { 7444, 0xCB8B }, { 7445, 0xCB8C }, { 7446, 0xCB8D }, { 7447, 0xCB8E }, { 7448, 0xCB8F }, { 7449, 0xCB90 }, { 7450, 0xCB91 }, { 7451, 0xCB92 },
    { 7452, 0xCB93 }, { 7453, 0xCB94 }, { 7454, 0xCB95 }, { 7455, 0xCB96 }, { 7456, 0xCB97 }, { 7457, 0xCB98 }, { 7458, 0xCB99 }, { 7459, 0xCB9A },
    { 7460, 0xCB9B }, { 7461, 0xCB9D }, { 7462, 0xCB9E }, { 7463, 0xCB9F }, { 7464, 0xCBA0 }, { 7465, 0xCBA1 }, { 7466, 0xCBA2 }, { 7467, 0xCBA3 },
    { 7474, 0xCBA4 }, { 7475, 0xCBA5 }, { 7476, 0xCBA6 }, { 7477, 0xCBA7 }, { 7478, 0xCBA8 }, { 7479, 0xCBA9 }, { 7480, 0xCBAA }, { 7481, 0xCBAB },
    { 7482, 0xCBAC }, { 7483, 0xCBAD }, { 7484, 0xCBAE }, { 7485, 0xCBAF }, { 7486, 0xCBB0 }, { 7487, 0xCBB1 }, { 7488, 0xCBB2 }, { 7489, 0xCBB3 },
    { 7490, 0xCBB4 }, { 7491, 0xCBB5 }, { 7492, 0xCBB6 }, { 7493, 0xCBB7 }, { 7494, 0xCBB9 }, { 7495, 0xCBBA }, { 7496, 0xCBBB }, { 7497, 0xCBBC },
    { 7498, 0xCBBD }, { 7499, 0xCBBE }, { 7500, 0xCBBF }, { 7501, 0xCBC0 }, { 7502, 0xCBC1 }, { 7503, 0xCBC2 }, { 7504, 0xCBC3 }, { 7505, 0xCBC4 },
    { 7506, 0x00C6 }, { 7507, 0x00D0 }, { 7508, 0x00AA }, { 7509, 0x0126 }, { 7511, 0x0132 }, { 7513, 0x013F }, { 7514, 0x0141 }, { 7515, 0x00D8 },
    { 7516, 0x0152 }, { 7517, 0x00BA }, { 7518, 0x00DE }, { 7519, 0x0166 }, { 7520, 0x014A }, { 7522, 0x3260 }, { 7523, 0x3261 }, { 7524, 0x3262 },
    { 7525, 0x3263 }, { 7526, 0x3264 }, { 7527, 0x3265 }, { 7528, 0x3266 }, { 7529, 0x3267 }, { 7530, 0x3268 }, { 7531, 0x3269 }, { 7532, 0x326A },
    { 7533, 0x326B }, { 7534, 0x326C }, { 7535, 0x326D }, { 7536, 0x326E }, { 7537, 0x326F }, { 7538, 0x3270 }, { 7539, 0x3271 }, { 7540, 0x3272 },
    { 7541, 0x3273 }, { 7542, 0x3274 }, { 7543, 0x3275 }, { 7544, 0x3276 }, { 7545, 0x3277 }, { 7546, 0x3278 }, { 7547, 0x3279 }, { 7548, 0x327A },
    { 7549, 0x327B }, { 7550, 0x24D0 }, { 7551, 0x24D1 }, { 7552, 0x24D2 }, { 7553, 0x24D3 }, { 7554, 0x24D4 }, { 7555, 0x24D5 }, { 7556, 0x24D6 },
    { 7557, 0x24D7 }, { 7558, 0x24D8 }, { 7559, 0x24D9 }, { 7560, 0x24DA }, { 7561, 0x24DB }, { 7562, 0x24DC }, { 7563, 0x24DD }, { 7564, 0x24DE },
    { 7565, 0x24DF }, { 7566, 0x24E0 }, { 7567, 0x24E1 }, { 7568, 0x24E2 }, { 7569, 0x24E3 }, { 7570, 0x24E4 }, { 7571, 0x24E5 }, { 7572, 0x24E6 },
    { 7573, 0x24E7 }, { 7574, 0x24E8 }, { 7575, 0x24E9 }, { 7576, 0x2460 }, { 7577, 0x2461 }, { 7578, 0x2462 }, { 7579, 0x2463 }, { 7580, 0x2464 },
    { 7581, 0x2465 }, { 7582, 0x2466 }, { 7583, 0x2467 }, { 7584, 0x2468 }, { 7585, 0x2469 }, { 7586, 0x246A }, { 7587, 0x246B }, { 7588, 0x246C },
    { 7589, 0x246D }, { 7590, 0x246E }, { 7591, 0x00BD }, { 7592, 0x2153 }, { 7593, 0x2154 }, { 7594, 0x00BC }, { 7595, 0x00BE }, { 7596, 0x215B },
    { 7597, 0x215C }, { 7598, 0x215D }, { 7599, 0x215E }, { 7600, 0xCBC5 }, { 7601, 0xCBC6 }, { 7602, 0xCBC7 }, { 7603, 0xCBC8 }, { 7604, 0xCBC9 },
    { 7605, 0xCBCA }, { 7606, 0xCBCB }, { 7607, 0xCBCC }, { 7608, 0xCBCD }, { 7609, 0xCBCE }, { 7610, 0xCBCF }, { 7611, 0xCBD0 }, { 7612, 0xCBD1 },
    { 7613, 0xCBD2 }, { 7614, 0xCBD3 }, { 7615, 0xCBD5 }, { 7616, 0xCBD6 }, { 7617, 0xCBD7 }, { 7618, 0xCBD8 }, { 7619, 0xCBD9 }, { 7620, 0xCBDA },
    { 7621, 0xCBDB }, { 7622, 0xCBDC }, { 7623, 0xCBDD }, { 7624, 0xCBDE }, { 7625, 0xCBDF }, { 7632, 0xCBE0 }, { 7633, 0xCBE1 }, { 7634, 0xCBE2 },
    { 7635, 0xCBE3 }, { 7636, 0xCBE5 }, { 7637, 0xCBE6 }, { 7638, 0xCBE8 }, { 7639, 0xCBEA }, { 7640, 0xCBEB }, { 7641, 0xCBEC }, { 7642, 0xCBED },
    { 7643, 0xCBEE }, { 7644, 0xCBEF }, { 7645, 0xCBF0 }, { 7646, 0xCBF1 }, { 7647, 0xCBF2 }, { 7648, 0xCBF3 }, { 7649, 0xCBF4 }, { 7650, 0xCBF5 },
    { 7651, 0xCBF6 }, { 7652, 0xCBF7 }, { 7653, 0xCBF8 }, { 7654, 0xCBF9 }, { 7655, 0xCBFA }, { 7656, 0xCBFB }, { 7657, 0xCBFC }, { 7664, 0xCBFD },
    { 7665, 0xCBFE }, { 7666, 0xCBFF }, { 7667, 0xCC00 }, { 7668, 0xCC01 }, { 7669, 0xCC02 }, { 7670, 0xCC03 }, { 7671, 0xCC04 }, { 7672, 0xCC05 },
    { 7673, 0xCC06 }, { 7674, 0xCC07 }, { 7675, 0xCC08 }, { 7676, 0xCC09 }, { 7677, 0xCC0A }, { 7678, 0xCC0B }, { 7679, 0xCC0E }, { 7680, 0xCC0F },
    { 7681, 0xCC11 }, { 7682, 0xCC12 }, { 7683, 0xCC13 }, { 7684, 0xCC15 }, { 7685, 0xCC16 }, { 7686, 0xCC17 }, { 7687, 0xCC18 }, { 7688, 0xCC19 },
    { 7689, 0xCC1A }, { 7690, 0xCC1B }, { 7691, 0xCC1E }, { 7692, 0xCC1F }, { 7693, 0xCC20 }, { 7694, 0xCC23 }, { 7695, 0xCC24 }, { 7696, 0x00E6 },
    { 7697, 0x0111 }, { 7698, 0x00F0 }, { 7699, 0x0127 }, { 7700, 0x0131 }, { 7701, 0x0133 }, { 7702, 0x0138 }, { 7703, 0x0140 }, { 7704, 0x0142 },
    { 7705, 0x00F8 }, { 7706, 0x0153 }, { 7707, 0x00DF }, { 7708, 0x00FE }, { 7709, 0x0167 }, { 7710, 0x014B }, { 7711, 0x0149 }, { 7712, 0x3200 },
    { 7713, 0x3201 }, { 7714, 0x3202 }, { 7715, 0x3203 }, { 7716, 0x3204 }, { 7717, 0x3205 }, { 7718, 0x3206 }, { 7719, 0x3207 }, { 7720, 0x3208 },
    { 7721, 0x3209 }, { 7722, 0x320A }, { 7723, 0x320B }, { 7724, 0x320C }, { 7725, 0x320D }, { 7726, 0x320E }, { 7727, 0x320F }, { 7728, 0x3210 },
    { 7729, 0x3211 }, { 7730, 0x3212 }, { 7731, 0x3213 }, { 7732, 0x3214 }, { 7733, 0x3215 }, { 7734, 0x3216 }, { 7735, 0x3217 }, { 7736, 0x3218 },
    { 7737, 0x3219 }, { 7738, 0x321A }, { 7739, 0x321B }, { 7740, 0x249C }, { 7741, 0x249D }, { 7742, 0x249E }, { 7743, 0x249F }, { 7744, 0x24A0 },
    { 7745, 0x24A1 }, { 7746, 0x24A2 }, { 7747, 0x24A3 }, { 7748, 0x24A4 }, { 7749, 0x24A5 }, { 7750, 0x24A6 }, { 7751, 0x24A7 }, { 7752, 0x24A8 },
    { 7753, 0x24A9 }, { 7754, 0x24AA }, { 7755, 0x24AB }, { 7756, 0x24AC }, { 7757, 0x24AD }, { 7758, 0x24AE }, { 7759, 0x24AF }, { 7760, 0x24B0 },
    { 7761, 0x24B1 }, { 7762, 0x24B2 }, { 7763, 0x24B3 }, { 7764, 0x24B4 }, { 7765, 0x24B5 }, { 7766, 0x2474 }, { 7767, 0x2475 }, { 7768, 0x2476 },
    { 7769, 0x2477 }, { 7770, 0x2478 }, { 7771, 0x2479 }, { 7772, 0x247A }, { 7773, 0x247B }, { 7774, 0x247C }, { 7775, 0x247D }, { 7776, 0x247E },
    { 7777, 0x247F }, { 7778, 0x2480 }, { 7779, 0x2481 }, { 7780, 0x2482 }, { 7781, 0x00B9 }, { 7782, 0x00B2 }, { 7783, 0x00B3 }, { 7784, 0x2074 },
    { 7785, 0x207F }, { 7786, 0x2081 }, { 7787, 0x2082 }, { 7788, 0x2083 }, { 7789, 0x2084 }, { 7790, 0xCC25 }, { 7791, 0xCC26 }, { 7792, 0xCC2A },
    { 7793, 0xCC2B }, { 7794, 0xCC2D }, { 7795, 0xCC2F }, { 7796, 0xCC31 }, { 7797, 0xCC32 }, { 7798, 0xCC33 }, { 7799, 0xCC34 }, { 7800, 0xCC35 },
    { 7801, 0xCC36 }, { 7802, 0xCC37 }, { 7803, 0xCC3A }, { 7804, 0xCC3F }, { 7805, 0xCC40 }, { 7806, 0xCC41 }, { 7807, 0xCC42 }, { 7808, 0xCC43 },
    { 7809, 0xCC46 }, { 7810, 0xCC47 }, { 7811, 0xCC49 }, { 7812, 0xCC4A }, { 7813, 0xCC4B }, { 7814, 0xCC4D }, { 7815, 0xCC4E }, { 7822, 0xCC4F },
    { 7823, 0xCC50 }, { 7824, 0xCC51 }, { 7825, 0xCC52 }, { 7826, 0xCC53 }, { 7827, 0xCC56 }, { 7828, 0xCC5A }, { 7829, 0xCC5B }, { 7830, 0xCC5C },
    { 7831, 0xCC5D }, { 7832, 0xCC5E }, { 7833, 0xCC5F }, { 7834, 0xCC61 }, { 7835, 0xCC62 }, { 7836, 0xCC63 }, { 7837, 0xCC65 }, { 7838, 0xCC67 },
    { 7839, 0xCC69 }, { 7840, 0xCC6A }, { 7841, 0xCC6B }, { 7842, 0xCC6C }, { 7843, 0xCC6D }, { 7844, 0xCC6E }, { 7845, 0xCC6F }, { 7846, 0xCC71 },
    { 7847, 0xCC72 }, { 7854, 0xCC73 }, { 7855, 0xCC74 }, { 7856, 0xCC76 }, { 7857, 0xCC77 }, { 7858, 0xCC78 }, { 7859, 0xCC79 }, { 7860, 0xCC7A },
    { 7861, 0xCC7B }, { 7862, 0xCC7C }, { 7863, 0xCC7D }, { 7864, 0xCC7E }, { 7865, 0xCC7F }, { 7866, 0xCC80 }, { 7867, 0xCC81 }, { 7868, 0xCC82 },
    { 7869, 0xCC83 }, { 7870, 0xCC84 }, { 7871, 0xCC85 }, { 7872, 0xCC86 }, { 7873, 0xCC87 }, { 7874, 0xCC88 }, { 7875, 0xCC89 }, { 7876, 0xCC8A },
    { 7877, 0xCC8B }, { 7878, 0xCC8C }, { 7879, 0xCC8D }, { 7880, 0xCC8E }, { 7881, 0xCC8F }, { 7882, 0xCC90 }, { 7883, 0xCC91 }, { 7884, 0xCC92 },
    { 7885, 0xCC93 }, { 7886, 0x3041 }, { 7887, 0x3042 }, { 7888, 0x3043 }, { 7889, 0x3044 }, { 7890, 0x3045 }, { 7891, 0x3046 }, { 7892, 0x3047 },
    { 7893, 0x3048 }, { 7894, 0x3049 }, { 7895, 0x304A }, { 7896, 0x304B }, { 7897, 0x304C }, { 7898, 0x304D }, { 7899, 0x304E }, { 7900, 0x304F },
    { 7901, 0x3050 }, { 7902, 0x3051 }, { 7903, 0x3052 }, { 7904, 0x3053 }, { 7905, 0x3054 }, { 7906, 0x3055 }, { 7907, 0x3056 }, { 7908, 0x3057 },
    { 7909, 0x3058 }, { 7910, 0x3059 }, { 7911, 0x305A }, { 7912, 0x305B }, { 7913, 0x305C }, { 7914, 0x305D }, { 7915, 0x305E }, { 7916, 0x305F },
    { 7917, 0x3060 }, { 7918, 0x3061 }, { 7919, 0x3062 }, { 7920, 0x3063 }, { 7921, 0x3064 }, { 7922, 0x3065 }, { 7923, 0x3066 }, { 7924, 0x3067 },
    { 7925, 0x3068 }, { 7926, 0x3069 }, { 7927, 0x306A }, { 7928, 0x306B }, { 7929, 0x306C }, { 7930, 0x306D }, { 7931, 0x306E }, { 7932, 0x306F },
    { 7933, 0x3070 }, { 7934, 0x3071 }, { 7935, 0x3072 }, { 7936, 0x3073 }, { 7937, 0x3074 }, { 7938, 0x3075 }, { 7939, 0x3076 }, { 7940, 0x3077 },
    { 7941, 0x3078 }, { 7942, 0x3079 }, { 7943, 0x307A }, { 7944, 0x307B }, { 7945, 0x307C }, { 7946, 0x307D }, { 7947, 0x307E }, { 7948, 0x307F },
    { 7949, 0x3080 }, { 7950, 0x3081 }, { 7951, 0x3082 }, { 7952, 0x3083 }, { 7953, 0x3084 }, { 7954, 0x3085 }, { 7955, 0x3086 }, { 7956, 0x3087 },
    { 7957, 0x3088 }, { 7958, 0x3089 }, { 7959, 0x308A }, { 7960, 0x308B }, { 7961, 0x308C }, { 7962, 0x308D }, { 7963, 0x308E }, { 7964, 0x308F },
    { 7965, 0x3090 }, { 7966, 0x3091 }, { 7967, 0x3092 }, { 7968, 0x3093 }, { 7980, 0xCC94 }, { 7981, 0xCC95 }, { 7982, 0xCC96 }, { 7983, 0xCC97 },
    { 7984, 0xCC9A }, { 7985, 0xCC9B }, { 7986, 0xCC9D }, { 7987, 0xCC9E }, { 7988, 0xCC9F }, { 7989, 0xCCA1 }, { 7990, 0xCCA2 }, { 7991, 0xCCA3 },
    { 7992, 0xCCA4 }, { 7993, 0xCCA5 }, { 7994, 0xCCA6 }, { 7995, 0xCCA7 }, { 7996, 0xCCAA }, { 7997, 0xCCAE }, { 7998, 0xCCAF }, { 7999, 0xCCB0 },
    { 8000, 0xCCB1 }, { 8001, 0xCCB2 }, { 8002, 0xCCB3 }, { 8003, 0xCCB6 }, { 8004, 0xCCB7 }, { 8005, 0xCCB9 }, { 8012, 0xCCBA }, { 8013, 0xCCBB },
    { 8014, 0xCCBD }, { 8015, 0xCCBE }, { 8016, 0xCCBF }, { 8017, 0xCCC0 }, { 8018, 0xCCC1 }, { 8019, 0xCCC2 }, { 8020, 0xCCC3 }, { 8021, 0xCCC6 },
    { 8022, 0xCCC8 }, { 8023, 0xCCCA }, { 8024, 0xCCCB }, { 8025, 0xCCCC }, { 8026, 0xCCCD }, { 8027, 0xCCCE }, { 8028, 0xCCCF }, { 8029, 0xCCD1 },
    { 8030, 0xCCD2 }, { 8031, 0xCCD3 }, { 8032, 0xCCD5 }, { 8033, 0xCCD6 }, { 8034, 0xCCD7 }, { 8035, 0xCCD8 }, { 8036, 0xCCD9 }, { 8037, 0xCCDA },
    { 8044, 0xCCDB }, { 8045, 0xCCDC }, { 8046, 0xCCDD }, { 8047, 0xCCDE }, { 8048, 0xCCDF }, { 8049, 0xCCE0 }, { 8050, 0xCCE1 }, { 8051, 0xCCE2 },
    { 8052, 0xCCE3 }, { 8053, 0xCCE5 }, { 8054, 0xCCE6 }, { 8055, 0xCCE7 }, { 8056, 0xCCE8 }, { 8057, 0xCCE9 }, { 8058, 0xCCEA }, { 8059, 0xCCEB },
    { 8060, 0xCCED }, { 8061, 0xCCEE }, { 8062, 0xCCEF }, { 8063, 0xCCF1 }, { 8064, 0xCCF2 }, { 8065, 0xCCF3 }, { 8066, 0xCCF4 }, { 8067, 0xCCF5 },
    { 8068, 0xCCF6 }, { 8069, 0xCCF7 }, { 8070, 0xCCF8 }, { 8071, 0xCCF9 }, { 8072, 0xCCFA }, { 8073, 0xCCFB }, { 8074, 0xCCFC }, { 8075, 0xCCFD },
    { 8076, 0x30A1 }, { 8077, 0x30A2 }, { 8078, 0x30A3 }, { 8079, 0x30A4 }, { 8080, 0x30A5 }, { 8081, 0x30A6 }, { 8082, 0x30A7 }, { 8083, 0x30A8 },
    { 8084, 0x30A9 }, { 8085, 0x30AA }, { 8086, 0x30AB }, { 8087, 0x30AC }, { 8088, 0x30AD }, { 8089, 0x30AE }, { 8090, 0x30AF }, { 8091, 0x30B0 },
    { 8092, 0x30B1 }, { 8093, 0x30B2 }, { 8094, 0x30B3 }, { 8095, 0x30B4 }, { 8096, 0x30B5 }, { 8097, 0x30B6 }, { 8098, 0x30B7 }, { 8099, 0x30B8 },
    { 8100, 0x30B9 }, { 8101, 0x30BA }, { 8102, 0x30BB }, { 8103, 0x30BC }, { 8104, 0x30BD }, { 8105, 0x30BE }, { 8106, 0x30BF }, { 8107, 0x30C0 },
    { 8108, 0x30C1 }, { 8109, 0x30C2 }, { 8110, 0x30C3 }, { 8111, 0x30C4 }, { 8112, 0x30C5 }, { 8113, 0x30C6 }, { 8114, 0x30C7 }, { 8115, 0x30C8 },
    { 8116, 0x30C9 }, { 8117, 0x30CA }, { 8118, 0x30CB }, { 8119, 0x30CC }, { 8120, 0x30CD }, { 8121, 0x30CE }, { 8122, 0x30CF }, { 8123, 0x30D0 },
    { 8124, 0x30D1 }, { 8125, 0x30D2 }, { 8126, 0x30D3 }, { 8127, 0x30D4 }, { 8128, 0x30D5 }, { 8129, 0x30D6 }, { 8130, 0x30D7 }, { 8131, 0x30D8 },
    { 8132, 0x30D9 }, { 8133, 0x30DA }, { 8134, 0x30DB }, { 8135, 0x30DC }, { 8136, 0x30DD }, { 8137, 0x30DE }, { 8138, 0x30DF }, { 8139, 0x30E0 },
    { 8140, 0x30E1 }, { 8141, 0x30E2 }, { 8142, 0x30E3 }, { 8143, 0x30E4 }, { 8144, 0x30E5 }, { 8145, 0x30E6 }, { 8146, 0x30E7 }, { 8147, 0x30E8 },
    { 8148, 0x30E9 }, { 8149, 0x30EA }, { 8150, 0x30EB }, { 8151, 0x30EC }, { 8152, 0x30ED }, { 8153, 0x30EE }, { 8154, 0x30EF }, { 8155, 0x30F0 },
    { 8156, 0x30F1 }, { 8157, 0x30F2 }, { 8158, 0x30F3 }, { 8159, 0x30F4 }, { 8160, 0x30F5 }, { 8161, 0x30F6 }, { 8170, 0xCCFE }, { 8171, 0xCCFF },
    { 8172, 0xCD00 }, { 8173, 0xCD02 }, { 8174, 0xCD03 }, { 8175, 0xCD04 }, { 8176, 0xCD05 }, { 8177, 0xCD06 }, { 8178, 0xCD07 }, { 8179, 0xCD0A },
    { 8180, 0xCD0B }, { 8181, 0xCD0D }, { 8182, 0xCD0E }, { 8183, 0xCD0F }, { 8184, 0xCD11 }, { 8185, 0xCD12 }, { 8186, 0xCD13 }, { 8187, 0xCD14 },
    { 8188, 0xCD15 }, { 8189, 0xCD16 }, { 8190, 0xCD17 }, { 8191, 0xCD1A }, { 8192, 0xCD1C }, { 8193, 0xCD1E }, { 8194, 0xCD1F }, { 8195, 0xCD20 },
    { 8202, 0xCD21 }, { 8203, 0xCD22 }, { 8204, 0xCD23 }, { 8205, 0xCD25 }, { 8206, 0xCD26 }, { 8207, 0xCD27 }, { 8208, 0xCD29 }, { 8209, 0xCD2A },
    { 8210, 0xCD2B }, { 8211, 0xCD2D }, { 8212, 0xCD2E }, { 8213, 0xCD2F }, { 8214, 0xCD30 }, { 8215, 0xCD31 }, { 8216, 0xCD32 }, { 8217, 0xCD33 },
    { 8218, 0xCD34 }, { 8219, 0xCD35 }, { 8220, 0xCD36 }, { 8221, 0xCD37 }, { 8222, 0xCD38 }, { 8223, 0xCD3A }, { 8224, 0xCD3B }, { 8225, 0xCD3C },
    { 8226, 0xCD3D }, { 8227, 0xCD3E }, { 8234, 0xCD3F }, { 8235, 0xCD40 }, { 8236, 0xCD41 }, { 8237, 0xCD42 }, { 8238, 0xCD43 }, { 8239, 0xCD44 },
    { 8240, 0xCD45 }, { 8241, 0xCD46 }, { 8242, 0xCD47 }, { 8243, 0xCD48 }, { 8244, 0xCD49 }, { 8245, 0xCD4A }, { 8246, 0xCD4B }, { 8247, 0xCD4C },
    { 8248, 0xCD4D }, { 8249, 0xCD4E }, { 8250, 0xCD4F }, { 8251, 0xCD50 }, { 8252, 0xCD51 }, { 8253, 0xCD52 }, { 8254, 0xCD53 }, { 8255, 0xCD54 },
    { 8256, 0xCD55 }, { 8257, 0xCD56 }, { 8258, 0xCD57 }, { 8259, 0xCD58 }, { 8260, 0xCD59 }, { 8261, 0xCD5A }, { 8262, 0xCD5B }, { 8263, 0xCD5D },
    { 8264, 0xCD5E }, { 8265, 0xCD5F }, { 8266, 0x0410 }, { 8267, 0x0411 }, { 8268, 0x0412 }, { 8269, 0x0413 }, { 8270, 0x0414 }, { 8271, 0x0415 },
    { 8272, 0x0401 }, { 8273, 0x0416 }, { 8274, 0x0417 }, { 8275, 0x0418 }, { 8276, 0x0419 }, { 8277, 0x041A }, { 8278, 0x041B }, { 8279, 0x041C },
    { 8280, 0x041D }, { 8281, 0x041E }, { 8282, 0x041F }, { 8283, 0x0420 }, { 8284, 0x0421 }, { 8285, 0x0422 }, { 8286, 0x0423 }, { 8287, 0x0424 },
    { 8288, 0x0425 }, { 8289, 0x0426 }, { 8290, 0x0427 }, { 8291, 0x0428 }, { 8292, 0x0429 }, { 8293, 0x042A }, { 8294, 0x042B }, { 8295, 0x042C },
    { 8296, 0x042D }, { 8297, 0x042E }, { 8298, 0x042F }, { 8314, 0x0430 }, { 8315, 0x0431 }, { 8316, 0x0432 }, { 8317, 0x0433 }, { 8318, 0x0434 },
    { 8319, 0x0435 }, { 8320, 0x0451 }, { 8321, 0x0436 }, { 8322, 0x0437 }, { 8323, 0x0438 }, { 8324, 0x0439 }, { 8325, 0x043A }, { 8326, 0x043B },
    { 8327, 0x043C }, { 8328, 0x043D }, { 8329, 0x043E }, { 8330, 0x043F }, { 8331, 0x0440 }, { 8332, 0x0441 }, { 8333, 0x0442 }, { 8334, 0x0443 },
    { 8335, 0x0444 }, { 8336, 0x0445 }, { 8337, 0x0446 }, { 8338, 0x0447 }, { 8339, 0x0448 }, { 8340, 0x0449 }, { 8341, 0x044A }, { 8342, 0x044B },
    { 8343, 0x044C }, { 8344, 0x044D }, { 8345, 0x044E }, { 8346, 0x044F }, { 8360, 0xCD61 }, { 8361, 0xCD62 }, { 8362, 0xCD63 }, { 8363, 0xCD65 },
    { 8364, 0xCD66 }, { 8365, 0xCD67 }, { 8366, 0xCD68 }, { 8367, 0xCD69 }, { 8368, 0xCD6A }, { 8369, 0xCD6B }, { 8370, 0xCD6E }, { 8371, 0xCD70 },
    { 8372, 0xCD72 }, { 8373, 0xCD73 }, { 8374, 0xCD74 }, { 8375, 0xCD75 }, { 8376, 0xCD76 }, { 8377, 0xCD77 }, { 8378, 0xCD79 }, { 8379, 0xCD7A },
    { 8380, 0xCD7B }, { 8381, 0xCD7C }, { 8382, 0xCD7D }, { 8383, 0xCD7E }, { 8384, 0xCD7F }, { 8385, 0xCD80 }, { 8392, 0xCD81 }, { 8393, 0xCD82 },
    { 8394, 0xCD83 }, { 8395, 0xCD84 }, { 8396, 0xCD85 }, { 8397, 0xCD86 }, { 8398, 0xCD87 }, { 8399, 0xCD89 }, { 8400, 0xCD8A }, { 8401, 0xCD8B },
    { 8402, 0xCD8C }, { 8403, 0xCD8D }, { 8404, 0xCD8E }, { 8405, 0xCD8F }, { 8406, 0xCD90 }, { 8407, 0xCD91 }, { 8408, 0xCD92 }, { 8409, 0xCD93 },
    { 8410, 0xCD96 }, { 8411, 0xCD97 }, { 8412, 0xCD99 }, { 8413, 0xCD9A }, { 8414, 0xCD9B }, { 8415, 0xCD9D }, { 8416, 0xCD9E }, { 8417, 0xCD9F },
    { 8424, 0xCDA0 }, { 8425, 0xCDA1 }, { 8426, 0xCDA2 }, { 8427, 0xCDA3 }, { 8428, 0xCDA6 }, { 8429, 0xCDA8 }, { 8430, 0xCDAA }, { 8431, 0xCDAB },
    { 8432, 0xCDAC }, { 8433, 0xCDAD }, { 8434, 0xCDAE }, { 8435, 0xCDAF }, { 8436, 0xCDB1 }, { 8437, 0xCDB2 }, { 8438, 0xCDB3 }, { 8439, 0xCDB4 },
    { 8440, 0xCDB5 }, { 8441, 0xCDB6 }, { 8442, 0xCDB7 }, { 8443, 0xCDB8 }, { 8444, 0xCDB9 }, { 8445, 0xCDBA }, { 8446, 0xCDBB }, { 8447, 0xCDBC },
    { 8448, 0xCDBD }, { 8449, 0xCDBE }, { 8450, 0xCDBF }, { 8451, 0xCDC0 }, { 8452, 0xCDC1 }, { 8453, 0xCDC2 }, { 8454, 0xCDC3 }, { 8455, 0xCDC5 },
    { 8550, 0xCDC6 }, { 8551, 0xCDC7 }, { 8552, 0xCDC8 }, { 8553, 0xCDC9 }, { 8554, 0xCDCA }, { 8555, 0xCDCB }, { 8556, 0xCDCD }, { 8557, 0xCDCE },
    { 8558, 0xCDCF }, { 8559, 0xCDD1 }, { 8560, 0xCDD2 }, { 8561, 0xCDD3 }, { 8562, 0xCDD4 }, { 8563, 0xCDD5 }, { 8564, 0xCDD6 }, { 8565, 0xCDD7 },
    { 8566, 0xCDD8 }, { 8567, 0xCDD9 }, { 8568, 0xCDDA }, { 8569, 0xCDDB }, { 8570, 0xCDDC }, { 8571, 0xCDDD }, { 8572, 0xCDDE }, { 8573, 0xCDDF },
    { 8574, 0xCDE0 }, { 8575, 0xCDE1 }, { 8582, 0xCDE2 }, { 8583, 0xCDE3 }, { 8584, 0xCDE4 }, { 8585, 0xCDE5 }, { 8586, 0xCDE6 }, { 8587, 0xCDE7 },
    { 8588, 0xCDE9 }, { 8589, 0xCDEA }, { 8590, 0xCDEB }, { 8591, 0xCDED }, { 8592, 0xCDEE }, { 8593, 0xCDEF }, { 8594, 0xCDF1 }, { 8595, 0xCDF2 },
    { 8596, 0xCDF3 }, { 8597, 0xCDF4 }, { 8598, 0xCDF5 }, { 8599, 0xCDF6 }, { 8600, 0xCDF7 }, { 8601, 0xCDFA }, { 8602, 0xCDFC }, { 8603, 0xCDFE },
    { 8604, 0xCDFF }, { 8605, 0xCE00 }, { 8606, 0xCE01 }, { 8607, 0xCE02 }, { 8614, 0xCE03 }, { 8615, 0xCE05 }, { 8616, 0xCE06 }, { 8617, 0xCE07 },
    { 8618, 0xCE09 }, { 8619, 0xCE0A }, { 8620, 0xCE0B }, { 8621, 0xCE0D }, { 8622, 0xCE0E }, { 8623, 0xCE0F }, { 8624, 0xCE10 }, { 8625, 0xCE11 },
    { 8626, 0xCE12 }, { 8627, 0xCE13 }, { 8628, 0xCE15 }, { 8629, 0xCE16 }, { 8630, 0xCE17 }, { 8631, 0xCE18 }, { 8632, 0xCE1A }, { 8633, 0xCE1B },
    { 8634, 0xCE1C }, { 8635, 0xCE1D }, { 8636, 0xCE1E }, { 8637, 0xCE1F }, { 8638, 0xCE22 }, { 8639, 0xCE23 }, { 8640, 0xCE25 }, { 8641, 0xCE26 },
    { 8642, 0xCE27 }, { 8643, 0xCE29 }, { 8644, 0xCE2A }, { 8645, 0xCE2B }, { 8740, 0xCE2C }, { 8741, 0xCE2D }, { 8742, 0xCE2E }, { 8743, 0xCE2F },
    { 8744, 0xCE32 }, { 8745, 0xCE34 }, { 8746, 0xCE36 }, { 8747, 0xCE37 }, { 8748, 0xCE38 }, { 8749, 0xCE39 }, { 8750, 0xCE3A }, { 8751, 0xCE3B },
    { 8752, 0xCE3C }, { 8753, 0xCE3D }, { 8754, 0xCE3E }, { 8755, 0xCE3F }, { 8756, 0xCE40 }, { 8757, 0xCE41 }, { 8758, 0xCE42 }, { 8759, 0xCE43 },
    { 8760, 0xCE44 }, { 8761, 0xCE45 }, { 8762, 0xCE46 }, { 8763, 0xCE47 }, { 8764, 0xCE48 }, { 8765, 0xCE49 }, { 8772, 0xCE4A }, { 8773, 0xCE4B },
    { 8774, 0xCE4C }, { 8775, 0xCE4D }, { 8776, 0xCE4E }, { 8777, 0xCE4F }, { 8778, 0xCE50 }, { 8779, 0xCE51 }, { 8780, 0xCE52 }, { 8781, 0xCE53 },
    { 8782, 0xCE54 }, { 8783, 0xCE55 }, { 8784, 0xCE56 }, { 8785, 0xCE57 }, { 8786, 0xCE5A }, { 8787, 0xCE5B }, { 8788, 0xCE5D }, { 8789, 0xCE5E },
    { 8790, 0xCE62 }, { 8791, 0xCE63 }, { 8792, 0xCE64 }, { 8793, 0xCE65 }, { 8794, 0xCE66 }, { 8795, 0xCE67 }, { 8796, 0xCE6A }, { 8797, 0xCE6C },
    { 8804, 0xCE6E }, { 8805, 0xCE6F }, { 8806, 0xCE70 }, { 8807, 0xCE71 }, { 8808, 0xCE72 }, { 8809, 0xCE73 }, { 8810, 0xCE76 }, { 8811, 0xCE77 },
    { 8812, 0xCE79 }, { 8813, 0xCE7A }, { 8814, 0xCE7B }, { 8815, 0xCE7D }, { 8816, 0xCE7E }, { 8817, 0xCE7F }, { 8818, 0xCE80 }, { 8819, 0xCE81 },
    { 8820, 0xCE82 }, { 8821, 0xCE83 }, { 8822, 0xCE86 }, { 8823, 0xCE88 }, { 8824, 0xCE8A }, { 8825, 0xCE8B }, { 8826, 0xCE8C }, { 8827, 0xCE8D },
    { 8828, 0xCE8E }, { 8829, 0xCE8F }, { 8830, 0xCE92 }, { 8831, 0xCE93 }, { 8832, 0xCE95 }, { 8833, 0xCE96 }, { 8834, 0xCE97 }, { 8835, 0xCE99 },
    { 8930, 0xCE9A }, { 8931, 0xCE9B }, { 8932, 0xCE9C }, { 8933, 0xCE9D }, { 8934, 0xCE9E }, { 8935, 0xCE9F }, { 8936, 0xCEA2 }, { 8937, 0xCEA6 },
    { 8938, 0xCEA7 }, { 8939, 0xCEA8 }, { 8940, 0xCEA9 }, { 8941, 0xCEAA }, { 8942, 0xCEAB }, { 8943, 0xCEAE }, { 8944, 0xCEAF }, { 8945, 0xCEB0 },
    { 8946, 0xCEB1 }, { 8947, 0xCEB2 }, { 8948, 0xCEB3 }, { 8949, 0xCEB4 }, { 8950, 0xCEB5 }, { 8951, 0xCEB6 }, { 8952, 0xCEB7 }, { 8953, 0xCEB8 },
    { 8954, 0xCEB9 }, { 8955, 0xCEBA }, { 8962, 0xCEBB }, { 8963, 0xCEBC }, { 8964, 0xCEBD }, { 8965, 0xCEBE }, { 8966, 0xCEBF }, { 8967, 0xCEC0 },
    { 8968, 0xCEC2 }, { 8969, 0xCEC3 }, { 8970, 0xCEC4 }, { 8971, 0xCEC5 }, { 8972, 0xCEC6 }, { 8973, 0xCEC7 }, { 8974, 0xCEC8 }, { 8975, 0xCEC9 },
    { 8976, 0xCECA }, { 8977, 0xCECB }, { 8978, 0xCECC }, { 8979, 0xCECD }, { 8980, 0xCECE }, { 8981, 0xCECF }, { 8982, 0xCED0 }, { 8983, 0xCED1 },
    { 8984, 0xCED2 }, { 8985, 0xCED3 }, { 8986, 0xCED4 }, { 8987, 0xCED5 }, { 8994, 0xCED6 }, { 8995, 0xCED7 }, { 8996, 0xCED8 }, { 8997, 0xCED9 },
    { 8998, 0xCEDA }, { 8999, 0xCEDB }, { 9000, 0xCEDC }, { 9001, 0xCEDD }, { 9002, 0xCEDE }, { 9003, 0xCEDF }, { 9004, 0xCEE0 }, { 9005, 0xCEE1 },
    { 9006, 0xCEE2 }, { 9007, 0xCEE3 }, { 9008, 0xCEE6 }, { 9009, 0xCEE7 }, { 9010, 0xCEE9 }, { 9011, 0xCEEA }, { 9012, 0xCEED }, { 9013, 0xCEEE },
    { 9014, 0xCEEF }, { 9015, 0xCEF0 }, { 9016, 0xCEF1 }, { 9017, 0xCEF2 }, { 9018, 0xCEF3 }, { 9019, 0xCEF6 }, { 9020, 0xCEFA }, { 9021, 0xCEFB },
    { 9022, 0xCEFC }, { 9023, 0xCEFD }, { 9024, 0xCEFE }, { 9025, 0xCEFF }, { 9026, 0xAC00 }, { 9027, 0xAC01 }, { 9028, 0xAC04 }, { 9029, 0xAC07 },
    { 9030, 0xAC08 }, { 9031, 0xAC09 }, { 9032, 0xAC0A }, { 9033, 0xAC10 }, { 9034, 0xAC11 }, { 9035, 0xAC12 }, { 9036, 0xAC13 }, { 9037, 0xAC14 },
    { 9038, 0xAC15 }, { 9039, 0xAC16 }, { 9040, 0xAC17 }, { 9041, 0xAC19 }, { 9042, 0xAC1A }, { 9043, 0xAC1B }, { 9044, 0xAC1C }, { 9045, 0xAC1D },
    { 9046, 0xAC20 }, { 9047, 0xAC24 }, { 9048, 0xAC2C }, { 9049, 0xAC2D }, { 9050, 0xAC2F }, { 9051, 0xAC30 }, { 9052, 0xAC31 }, { 9053, 0xAC38 },
    { 9054, 0xAC39 }, { 9055, 0xAC3C }, { 9056, 0xAC40 }, { 9057, 0xAC4B }, { 9058, 0xAC4D }, { 9059, 0xAC54 }, { 9060, 0xAC58 }, { 9061, 0xAC5C },
    { 9062, 0xAC70 }, { 9063, 0xAC71 }, { 9064, 0xAC74 }, { 9065, 0xAC77 }, { 9066, 0xAC78 }, { 9067, 0xAC7A }, { 9068, 0xAC80 }, { 9069, 0xAC81 },
    { 9070, 0xAC83 }, { 9071, 0xAC84 }, { 9072, 0xAC85 }, { 9073, 0xAC86 }, { 9074, 0xAC89 }, { 9075, 0xAC8A }, { 9076, 0xAC8B }, { 9077, 0xAC8C },
    { 9078, 0xAC90 }, { 9079, 0xAC94 }, { 9080, 0xAC9C }, { 9081, 0xAC9D }, { 9082, 0xAC9F }, { 9083, 0xACA0 }, { 9084, 0xACA1 }, { 9085, 0xACA8 },
    { 9086, 0xACA9 }, { 9087, 0xACAA }, { 9088, 0xACAC }, { 9089, 0xACAF }, { 9090, 0xACB0 }, { 9091, 0xACB8 }, { 9092, 0xACB9 }, { 9093, 0xACBB },
    { 9094, 0xACBC }, { 9095, 0xACBD }, { 9096, 0xACC1 }, { 9097, 0xACC4 }, { 9098, 0xACC8 }, { 9099, 0xACCC }, { 9100, 0xACD5 }, { 9101, 0xACD7 },
    { 9102, 0xACE0 }, { 9103, 0xACE1 }, { 9104, 0xACE4 }, { 9105, 0xACE7 }, { 9106, 0xACE8 }, { 9107, 0xACEA }, { 9108, 0xACEC }, { 9109, 0xACEF },
    { 9110, 0xACF0 }, { 9111, 0xACF1 }, { 9112, 0xACF3 }, { 9113, 0xACF5 }, { 9114, 0xACF6 }, { 9115, 0xACFC }, { 9116, 0xACFD }, { 9117, 0xAD00 },
    { 9118, 0xAD04 }, { 9119, 0xAD06 }, { 9120, 0xCF02 }, { 9121, 0xCF03 }, { 9122, 0xCF05 }, { 9123, 0xCF06 }, { 9124, 0xCF07 }, { 9125, 0xCF09 },
    { 9126, 0xCF0A }, { 9127, 0xCF0B }, { 9128, 0xCF0C }, { 9129, 0xCF0D }, { 9130, 0xCF0E }, { 9131, 0xCF0F }, { 9132, 0xCF12 }, { 9133, 0xCF14 },
    { 9134, 0xCF16 }, { 9135, 0xCF17 }, { 9136, 0xCF18 }, { 9137, 0xCF19 }, { 9138, 0xCF1A }, { 9139, 0xCF1B }, { 9140, 0xCF1D }, { 9141, 0xCF1E },
    { 9142, 0xCF1F }, { 9143, 0xCF21 }, { 9144, 0xCF22 }, { 9145, 0xCF23 }, { 9152, 0xCF25 }, { 9153, 0xCF26 }, { 9154, 0xCF27 }, { 9155, 0xCF28 },
    { 9156, 0xCF29 }, { 9157, 0xCF2A }, { 9158, 0xCF2B }, { 9159, 0xCF2E }, { 9160, 0xCF32 }, { 9161, 0xCF33 }, { 9162, 0xCF34 }, { 9163, 0xCF35 },
    { 9164, 0xCF36 }, { 9165, 0xCF37 }, { 9166, 0xCF39 }, { 9167, 0xCF3A }, { 9168, 0xCF3B }, { 9169, 0xCF3C }, { 9170, 0xCF3D }, { 9171, 0xCF3E },
    { 9172, 0xCF3F }, { 9173, 0xCF40 }, { 9174, 0xCF41 }, { 9175, 0xCF42 }, { 9176, 0xCF43 }, { 9177, 0xCF44 }, { 9184, 0xCF45 }, { 9185, 0xCF46 },
    { 9186, 0xCF47 }, { 9187, 0xCF48 }, { 9188, 0xCF49 }, { 9189, 0xCF4A }, { 9190, 0xCF4B }, { 9191, 0xCF4C }, { 9192, 0xCF4D }, { 9193, 0xCF4E },
    { 9194, 0xCF4F }, { 9195, 0xCF50 }, { 9196, 0xCF51 }, { 9197, 0xCF52 }, { 9198, 0xCF53 }, { 9199, 0xCF56 }, { 9200, 0xCF57 }, { 9201, 0xCF59 },
    { 9202, 0xCF5A }, { 9203, 0xCF5B }, { 9204, 0xCF5D }, { 9205, 0xCF5E }, { 9206, 0xCF5F }, { 9207, 0xCF60 }, { 9208, 0xCF61 }, { 9209, 0xCF62 },
    { 9210, 0xCF63 }, { 9211, 0xCF66 }, { 9212, 0xCF68 }, { 9213, 0xCF6A }, { 9214, 0xCF6B }, { 9215, 0xCF6C }, { 9216, 0xAD0C }, { 9217, 0xAD0D },
    { 9218, 0xAD0F }, { 9219, 0xAD11 }, { 9220, 0xAD18 }, { 9221, 0xAD1C }, { 9222, 0xAD20 }, { 9223, 0xAD29 }, { 9224, 0xAD2C }, { 9225, 0xAD2D },
    { 9226, 0xAD34 }, { 9227, 0xAD35 }, { 9228, 0xAD38 }, { 9229, 0xAD3C }, { 9230, 0xAD44 }, { 9231, 0xAD45 }, { 9232, 0xAD47 }, { 9233, 0xAD49 },
    { 9234, 0xAD50 }, { 9235, 0xAD54 }, { 9236, 0xAD58 }, { 9237, 0xAD61 }, { 9238, 0xAD63 }, { 9239, 0xAD6C }, { 9240, 0xAD6D }, { 9241, 0xAD70 },
    { 9242, 0xAD73 }, { 9243, 0xAD74 }, { 9244, 0xAD75 }, { 9245, 0xAD76 }, { 9246, 0xAD7B }, { 9247, 0xAD7C }, { 9248, 0xAD7D }, { 9249, 0xAD7F },
    { 9250, 0xAD81 }, { 9251, 0xAD82 }, { 9252, 0xAD88 }, { 9253, 0xAD89 }, { 9254, 0xAD8C }, { 9255, 0xAD90 }, { 9256, 0xAD9C }, { 9257, 0xAD9D },
    { 9258, 0xADA4 }, { 9259, 0xADB7 }, { 9260, 0xADC0 }, { 9261, 0xADC1 }, { 9262, 0xADC4 }, { 9263, 0xADC8 }, { 9264, 0xADD0 }, { 9265, 0xADD1 },
    { 9266, 0xADD3 }, { 9267, 0xADDC }, { 9268, 0xADE0 }, { 9269, 0xADE4 }, { 9270, 0xADF8 }, { 9271, 0xADF9 }, { 9272, 0xADFC }, { 9273, 0xADFF },
    { 9274, 0xAE00 }, { 9275, 0xAE01 }, { 9276, 0xAE08 }, { 9277, 0xAE09 }, { 9278, 0xAE0B }, { 9279, 0xAE0D }, { 9280, 0xAE14 }, { 9281, 0xAE30 },
    { 9282, 0xAE31 }, { 9283, 0xAE34 }, { 9284, 0xAE37 }, { 9285, 0xAE38 }, { 9286, 0xAE3A }, { 9287, 0xAE40 }, { 9288, 0xAE41 }, { 9289, 0xAE43 },
    { 9290, 0xAE45 }, { 9291, 0xAE46 }, { 9292, 0xAE4A }, { 9293, 0xAE4C }, { 9294, 0xAE4D }, { 9295, 0xAE4E }, { 9296, 0xAE50 }, { 9297, 0xAE54 },
    { 9298, 0xAE56 }, { 9299, 0xAE5C }, { 9300, 0xAE5D }, { 9301, 0xAE5F }, { 9302, 0xAE60 }, { 9303, 0xAE61 }, { 9304, 0xAE65 }, { 9305, 0xAE68 },
    { 9306, 0xAE69 }, { 9307, 0xAE6C }, { 9308, 0xAE70 }, { 9309, 0xAE78 }, { 9310, 0xCF6D }, { 9311, 0xCF6E }, { 9312, 0xCF6F }, { 9313, 0xCF72 },
    { 9314, 0xCF73 }, { 9315, 0xCF75 }, { 9316, 0xCF76 }, { 9317, 0xCF77 }, { 9318, 0xCF79 }, { 9319, 0xCF7A }, { 9320, 0xCF7B }, { 9321, 0xCF7C },
    { 9322, 0xCF7D }, { 9323, 0xCF7E }, { 9324, 0xCF7F }, { 9325, 0xCF81 }, { 9326, 0xCF82 }, { 9327, 0xCF83 }, { 9328, 0xCF84 }, { 9329, 0xCF86 },
    { 9330, 0xCF87 }, { 9331, 0xCF88 }, { 9332, 0xCF89 }, { 9333, 0xCF8A }, { 9334, 0xCF8B }, { 9335, 0xCF8D }, { 9342, 0xCF8E }, { 9343, 0xCF8F },
    { 9344, 0xCF90 }, { 9345, 0xCF91 }, { 9346, 0xCF92 }, { 9347, 0xCF93 }, { 9348, 0xCF94 }, { 9349, 0xCF95 }, { 9350, 0xCF96 }, { 9351, 0xCF97 },
    { 9352, 0xCF98 }, { 9353, 0xCF99 }, { 9354, 0xCF9A }, { 9355, 0xCF9B }, { 9356, 0xCF9C }, { 9357, 0xCF9D }, { 9358, 0xCF9E }, { 9359, 0xCF9F },
    { 9360, 0xCFA0 }, { 9361, 0xCFA2 }, { 9362, 0xCFA3 }, { 9363, 0xCFA4 }, { 9364, 0xCFA5 }, { 9365, 0xCFA6 }, { 9366, 0xCFA7 }, { 9367, 0xCFA9 },
    { 9374, 0xCFAA }, { 9375, 0xCFAB }, { 9376, 0xCFAC }, { 9377, 0xCFAD }, { 9378, 0xCFAE }, { 9379, 0xCFAF }, { 9380, 0xCFB1 }, { 9381, 0xCFB2 },
    { 9382, 0xCFB3 }, { 9383, 0xCFB4 }, { 9384, 0xCFB5 }, { 9385, 0xCFB6 }, { 9386, 0xCFB7 }, { 9387, 0xCFB8 }, { 9388, 0xCFB9 }, { 9389, 0xCFBA },
    { 9390, 0xCFBB }, { 9391, 0xCFBC }, { 9392, 0xCFBD }, { 9393, 0xCFBE }, { 9394, 0xCFBF }, { 9395, 0xCFC0 }, { 9396, 0xCFC1 }, { 9397, 0xCFC2 },
    { 9398, 0xCFC3 }, { 9399, 0xCFC5 }, { 9400, 0xCFC6 }, { 9401, 0xCFC7 }, { 9402, 0xCFC8 }, { 9403, 0xCFC9 }, { 9404, 0xCFCA }, { 9405, 0xCFCB },
    { 9406, 0xAE79 }, { 9407, 0xAE7B }, { 9408, 0xAE7C }, { 9409, 0xAE7D }, { 9410, 0xAE84 }, { 9411, 0xAE85 }, { 9412, 0xAE8C }, { 9413, 0xAEBC },
    { 9414, 0xAEBD }, { 9415, 0xAEBE }, { 9416, 0xAEC0 }, { 9417, 0xAEC4 }, { 9418, 0xAECC }, { 9419, 0xAECD }, { 9420, 0xAECF }, { 9421, 0xAED0 },
    { 9422, 0xAED1 }, { 9423, 0xAED8 }, { 9424, 0xAED9 }, { 9425, 0xAEDC }, { 9426, 0xAEE8 }, { 9427, 0xAEEB }, { 9428, 0xAEED }, { 9429, 0xAEF4 },
    { 9430, 0xAEF8 }, { 9431, 0xAEFC }, { 9432, 0xAF07 }, { 9433, 0xAF08 }, { 9434, 0xAF0D }, { 9435, 0xAF10 }, { 9436, 0xAF2C }, { 9437, 0xAF2D },
    { 9438, 0xAF30 }, { 9439, 0xAF32 }, { 9440, 0xAF34 }, { 9441, 0xAF3C }, { 9442, 0xAF3D }, { 9443, 0xAF3F }, { 9444, 0xAF41 }, { 9445, 0xAF42 },
    { 9446, 0xAF43 }, { 9447, 0xAF48 }, { 9448, 0xAF49 }, { 9449, 0xAF50 }, { 9450, 0xAF5C }, { 9451, 0xAF5D }, { 9452, 0xAF64 }, { 9453, 0xAF65 },
    { 9454, 0xAF79 }, { 9455, 0xAF80 }, { 9456, 0xAF84 }, { 9457, 0xAF88 }, { 9458, 0xAF90 }, { 9459, 0xAF91 }, { 9460, 0xAF95 }, { 9461, 0xAF9C },
    { 9462, 0xAFB8 }, { 9463, 0xAFB9 }, { 9464, 0xAFBC }, { 9465, 0xAFC0 }, { 9466, 0xAFC7 }, { 9467, 0xAFC8 }, { 9468, 0xAFC9 }, { 9469, 0xAFCB },
    { 9470, 0xAFCD }, { 9471, 0xAFCE }, { 9472, 0xAFD4 }, { 9473, 0xAFDC }, { 9474, 0xAFE8 }, { 9475, 0xAFE9 }, { 9476, 0xAFF0 }, { 9477, 0xAFF1 },
    { 9478, 0xAFF4 }, { 9479, 0xAFF8 }, { 9480, 0xB000 }, { 9481, 0xB001 }, { 9482, 0xB004 }, { 9483, 0xB00C }, { 9484, 0xB010 }, { 9485, 0xB014 },
    { 9486, 0xB01C }, { 9487, 0xB01D }, { 9488, 0xB028 }, { 9489, 0xB044 }, { 9490, 0xB045 }, { 9491, 0xB048 }, { 9492, 0xB04A }, { 9493, 0xB04C },
    { 9494, 0xB04E }, { 9495, 0xB053 }, { 9496, 0xB054 }, { 9497, 0xB055 }, { 9498, 0xB057 }, { 9499, 0xB059 }, { 9500, 0xCFCC }, { 9501, 0xCFCD },
    { 9502, 0xCFCE }, { 9503, 0xCFCF }, { 9504, 0xCFD0 }, { 9505, 0xCFD1 }, { 9506, 0xCFD2 }, { 9507, 0xCFD3 }, { 9508, 0xCFD4 }, { 9509, 0xCFD5 },
    { 9510, 0xCFD6 }, { 9511, 0xCFD7 }, { 9512, 0xCFD8 }, { 9513, 0xCFD9 }, { 9514, 0xCFDA }, { 9515, 0xCFDB }, { 9516, 0xCFDC }, { 9517, 0xCFDD },
    { 9518, 0xCFDE }, { 9519, 0xCFDF }, { 9520, 0xCFE2 }, { 9521, 0xCFE3 }, { 9522, 0xCFE5 }, { 9523, 0xCFE6 }, { 9524, 0xCFE7 }, { 9525, 0xCFE9 },
    { 9532, 0xCFEA }, { 9533, 0xCFEB }, { 9534, 0xCFEC }, { 9535, 0xCFED }, { 9536, 0xCFEE }, { 9537, 0xCFEF }, { 9538, 0xCFF2 }, { 9539, 0xCFF4 },
    { 9540, 0xCFF6 }, { 9541, 0xCFF7 }, { 9542, 0xCFF8 }, { 9543, 0xCFF9 }, { 9544, 0xCFFA }, { 9545, 0xCFFB }, { 9546, 0xCFFD }, { 9547, 0xCFFE },
    { 9548, 0xCFFF }, { 9549, 0xD001 }, { 9550, 0xD002 }, { 9551, 0xD003 }, { 9552, 0xD005 }, { 9553, 0xD006 }, { 9554, 0xD007 }, { 9555, 0xD008 },
    { 9556, 0xD009 }, { 9557, 0xD00A }, { 9564, 0xD00B }, { 9565, 0xD00C }, { 9566, 0xD00D }, { 9567, 0xD00E }, { 9568, 0xD00F }, { 9569, 0xD010 },
    { 9570, 0xD012 }, { 9571, 0xD013 }, { 9572, 0xD014 }, { 9573, 0xD015 }, { 9574, 0xD016 }, { 9575, 0xD017 }, { 9576, 0xD019 }, { 9577, 0xD01A },
    { 9578, 0xD01B }, { 9579, 0xD01C }, { 9580, 0xD01D }, { 9581, 0xD01E }, { 9582, 0xD01F }, { 9583, 0xD020 }, { 9584, 0xD021 }, { 9585, 0xD022 },
    { 9586, 0xD023 }, { 9587, 0xD024 }, { 9588, 0xD025 }, { 9589, 0xD026 }, { 9590, 0xD027 }, { 9591, 0xD028 }, { 9592, 0xD029 }, { 9593, 0xD02A },
    { 9594, 0xD02B }, { 9595, 0xD02C }, { 9596, 0xB05D }, { 9597, 0xB07C }, { 9598, 0xB07D }, { 9599, 0xB080 }, { 9600, 0xB084 }, { 9601, 0xB08C },
    { 9602, 0xB08D }, { 9603, 0xB08F }, { 9604, 0xB091 }, { 9605, 0xB098 }, { 9606, 0xB099 }, { 9607, 0xB09A }, { 9608, 0xB09C }, { 9609, 0xB09F },
    { 9610, 0xB0A0 }, { 9611, 0xB0A1 }, { 9612, 0xB0A2 }, { 9613, 0xB0A8 }, { 9614, 0xB0A9 }, { 9615, 0xB0AB }, { 9616, 0xB0AC }, { 9617, 0xB0AD },
    { 9618, 0xB0AE }, { 9619, 0xB0AF }, { 9620, 0xB0B1 }, { 9621, 0xB0B3 }, { 9622, 0xB0B4 }, { 9623, 0xB0B5 }, { 9624, 0xB0B8 }, { 9625, 0xB0BC },
    { 9626, 0xB0C4 }, { 9627, 0xB0C5 }, { 9628, 0xB0C7 }, { 9629, 0xB0C8 }, { 9630, 0xB0C9 }, { 9631, 0xB0D0 }, { 9632, 0xB0D1 }, { 9633, 0xB0D4 },
    { 9634, 0xB0D8 }, { 9635, 0xB0E0 }, { 9636, 0xB0E5 }, { 9637, 0xB108 }, { 9638, 0xB109 }, { 9639, 0xB10B }, { 9640, 0xB10C }, { 9641, 0xB110 },
    { 9642, 0xB112 }, { 9643, 0xB113 }, { 9644, 0xB118 }, { 9645, 0xB119 }, { 9646, 0xB11B }, { 9647, 0xB11C }, { 9648, 0xB11D }, { 9649, 0xB123 },
    { 9650, 0xB124 }, { 9651, 0xB125 }, { 9652, 0xB128 }, { 9653, 0xB12C }, { 9654, 0xB134 }, { 9655, 0xB135 }, { 9656, 0xB137 }, { 9657, 0xB138 },
    { 9658, 0xB139 }, { 9659, 0xB140 }, { 9660, 0xB141 }, { 9661, 0xB144 }, { 9662, 0xB148 }, { 9663, 0xB150 }, { 9664, 0xB151 }, { 9665, 0xB154 },
    { 9666, 0xB155 }, { 9667, 0xB158 }, { 9668, 0xB15C }, { 9669, 0xB160 }, { 9670, 0xB178 }, { 9671, 0xB179 }, { 9672, 0xB17C }, { 9673, 0xB180 },
    { 9674, 0xB182 }, { 9675, 0xB188 }, { 9676, 0xB189 }, { 9677, 0xB18B }, { 9678, 0xB18D }, { 9679, 0xB192 }, { 9680, 0xB193 }, { 9681, 0xB194 },
    { 9682, 0xB198 }, { 9683, 0xB19C }, { 9684, 0xB1A8 }, { 9685, 0xB1CC }, { 9686, 0xB1D0 }, { 9687, 0xB1D4 }, { 9688, 0xB1DC }, { 9689, 0xB1DD },
    { 9690, 0xD02E }, { 9691, 0xD02F }, { 9692, 0xD030 }, { 9693, 0xD031 }, { 9694, 0xD032 }, { 9695, 0xD033 }, { 9696, 0xD036 }, { 9697, 0xD037 },
    { 9698, 0xD039 }, { 9699, 0xD03A }, { 9700, 0xD03B }, { 9701, 0xD03D }, { 9702, 0xD03E }, { 9703, 0xD03F }, { 9704, 0xD040 }, { 9705, 0xD041 },
    { 9706, 0xD042 }, { 9707, 0xD043 }, { 9708, 0xD046 }, { 9709, 0xD048 }, { 9710, 0xD04A }, { 9711, 0xD04B }, { 9712, 0xD04C }, { 9713, 0xD04D },
    { 9714, 0xD04E }, { 9715, 0xD04F }, { 9722, 0xD051 }, { 9723, 0xD052 }, { 9724, 0xD053 }, { 9725, 0xD055 }, { 9726, 0xD056 }, { 9727, 0xD057 },
    { 9728, 0xD059 }, { 9729, 0xD05A }, { 9730, 0xD05B }, { 9731, 0xD05C }, { 9732, 0xD05D }, { 9733, 0xD05E }, { 9734, 0xD05F }, { 9735, 0xD061 },
    { 9736, 0xD062 }, { 9737, 0xD063 }, { 9738, 0xD064 }, { 9739, 0xD065 }, { 9740, 0xD066 }, { 9741, 0xD067 }, { 9742, 0xD068 }, { 9743, 0xD069 },
    { 9744, 0xD06A }, { 9745, 0xD06B }, { 9746, 0xD06E }, { 9747, 0xD06F }, { 9754, 0xD071 }, { 9755, 0xD072 }, { 9756, 0xD073 }, { 9757, 0xD075 },
    { 9758, 0xD076 }, { 9759, 0xD077 }, { 9760, 0xD078 }, { 9761, 0xD079 }, { 9762, 0xD07A }, { 9763, 0xD07B }, { 9764, 0xD07E }, { 9765, 0xD07F },
    { 9766, 0xD080 }, { 9767, 0xD082 }, { 9768, 0xD083 }, { 9769, 0xD084 }, { 9770, 0xD085 }, { 9771, 0xD086 }, { 9772, 0xD087 }, { 9773, 0xD088 },
    { 9774, 0xD089 }, { 9775, 0xD08A }, { 9776, 0xD08B }, { 9777, 0xD08C }, { 9778, 0xD08D }, { 9779, 0xD08E }, { 9780, 0xD08F }, { 9781, 0xD090 },
    { 9782, 0xD091 }, { 9783, 0xD092 }, { 9784, 0xD093 }, { 9785, 0xD094 }, { 9786, 0xB1DF }, { 9787, 0xB1E8 }, { 9788, 0xB1E9 }, { 9789, 0xB1EC },
    { 9790, 0xB1F0 }, { 9791, 0xB1F9 }, { 9792, 0xB1FB }, { 9793, 0xB1FD }, { 9794, 0xB204 }, { 9795, 0xB205 }, { 9796, 0xB208 }, { 9797, 0xB20B },
    { 9798, 0xB20C }, { 9799, 0xB214 }, { 9800, 0xB215 }, { 9801, 0xB217 }, { 9802, 0xB219 }, { 9803, 0xB220 }, { 9804, 0xB234 }, { 9805, 0xB23C },
    { 9806, 0xB258 }, { 9807, 0xB25C }, { 9808, 0xB260 }, { 9809, 0xB268 }, { 9810, 0xB269 }, { 9811, 0xB274 }, { 9812, 0xB275 }, { 9813, 0xB27C },
    { 9814, 0xB284 }, { 9815, 0xB285 }, { 9816, 0xB289 }, { 9817, 0xB290 }, { 9818, 0xB291 }, { 9819, 0xB294 }, { 9820, 0xB298 }, { 9821, 0xB299 },
    { 9822, 0xB29A }, { 9823, 0xB2A0 }, { 9824, 0xB2A1 }, { 9825, 0xB2A3 }, { 9826, 0xB2A5 }, { 9827, 0xB2A6 }, { 9828, 0xB2AA }, { 9829, 0xB2AC },
    { 9830, 0xB2B0 }, { 9831, 0xB2B4 }, { 9832, 0xB2C8 }, { 9833, 0xB2C9 }, { 9834, 0xB2CC }, { 9835, 0xB2D0 }, { 9836, 0xB2D2 }, { 9837, 0xB2D8 },
    { 9838, 0xB2D9 }, { 9839, 0xB2DB }, { 9840, 0xB2DD }, { 9841, 0xB2E2 }, { 9842, 0xB2E4 }, { 9843, 0xB2E5 }, { 9844, 0xB2E6 }, { 9845, 0xB2E8 },
    { 9846, 0xB2EB }, { 9847, 0xB2EC }, { 9848, 0xB2ED }, { 9849, 0xB2EE }, { 9850, 0xB2EF }, { 9851, 0xB2F3 }, { 9852, 0xB2F4 }, { 9853, 0xB2F5 },
    { 9854, 0xB2F7 }, { 9855, 0xB2F8 }, { 9856, 0xB2F9 }, { 9857, 0xB2FA }, { 9858, 0xB2FB }, { 9859, 0xB2FF }, { 9860, 0xB300 }, { 9861, 0xB301 },
    { 9862, 0xB304 }, { 9863, 0xB308 }, { 9864, 0xB310 }, { 9865, 0xB311 }, { 9866, 0xB313 }, { 9867, 0xB314 }, { 9868, 0xB315 }, { 9869, 0xB31C },
    { 9870, 0xB354 }, { 9871, 0xB355 }, { 9872, 0xB356 }, { 9873, 0xB358 }, { 9874, 0xB35B }, { 9875, 0xB35C }, { 9876, 0xB35E }, { 9877, 0xB35F },
    { 9878, 0xB364 }, { 9879, 0xB365 }, { 9880, 0xD095 }, { 9881, 0xD096 }, { 9882, 0xD097 }, { 9883, 0xD098 }, { 9884, 0xD099 }, { 9885, 0xD09A },
    { 9886, 0xD09B }, { 9887, 0xD09C }, { 9888, 0xD09D }, { 9889, 0xD09E }, { 9890, 0xD09F }, { 9891, 0xD0A0 }, { 9892, 0xD0A1 }, { 9893, 0xD0A2 },
    { 9894, 0xD0A3 }, { 9895, 0xD0A6 }, { 9896, 0xD0A7 }, { 9897, 0xD0A9 }, { 9898, 0xD0AA }, { 9899, 0xD0AB }, { 9900, 0xD0AD }, { 9901, 0xD0AE },
    { 9902, 0xD0AF }, { 9903, 0xD0B0 }, { 9904, 0xD0B1 }, { 9905, 0xD0B2 }, { 9912, 0xD0B3 }, { 9913, 0xD0B6 }, { 9914, 0xD0B8 }, { 9915, 0xD0BA },
    { 9916, 0xD0BB }, { 9917, 0xD0BC }, { 9918, 0xD0BD }, { 9919, 0xD0BE }, { 9920, 0xD0BF }, { 9921, 0xD0C2 }, { 9922, 0xD0C3 }, { 9923, 0xD0C5 },
    { 9924, 0xD0C6 }, { 9925, 0xD0C7 }, { 9926, 0xD0CA }, { 9927, 0xD0CB }, { 9928, 0xD0CC }, { 9929, 0xD0CD }, { 9930, 0xD0CE }, { 9931, 0xD0CF },
    { 9932, 0xD0D2 }, { 9933, 0xD0D6 }, { 9934, 0xD0D7 }, { 9935, 0xD0D8 }, { 9936, 0xD0D9 }, { 9937, 0xD0DA }, { 9944, 0xD0DB }, { 9945, 0xD0DE },
    { 9946, 0xD0DF }, { 9947, 0xD0E1 }, { 9948, 0xD0E2 }, { 9949, 0xD0E3 }, { 9950, 0xD0E5 }, { 9951, 0xD0E6 }, { 9952, 0xD0E7 }, { 9953, 0xD0E8 },
    { 9954, 0xD0E9 }, { 9955, 0xD0EA }, { 9956, 0xD0EB }, { 9957, 0xD0EE }, { 9958, 0xD0F2 }, { 9959, 0xD0F3 }, { 9960, 0xD0F4 }, { 9961, 0xD0F5 },
    { 9962, 0xD0F6 }, { 9963, 0xD0F7 }, { 9964, 0xD0F9 }, { 9965, 0xD0FA }, { 9966, 0xD0FB }, { 9967, 0xD0FC }, { 9968, 0xD0FD }, { 9969, 0xD0FE },
    { 9970, 0xD0FF }, { 9971, 0xD100 }, { 9972, 0xD101 }, { 9973, 0xD102 }, { 9974, 0xD103 }, { 9975, 0xD104 }, { 9976, 0xB367 }, { 9977, 0xB369 },
    { 9978, 0xB36B }, { 9979, 0xB36E }, { 9980, 0xB370 }, { 9981, 0xB371 }, { 9982, 0xB374 }, { 9983, 0xB378 }, { 9984, 0xB380 }, { 9985, 0xB381 },
    { 9986, 0xB383 }, { 9987, 0xB384 }, { 9988, 0xB385 }, { 9989, 0xB38C }, { 9990, 0xB390 }, { 9991, 0xB394 }, { 9992, 0xB3A0 }, { 9993, 0xB3A1 },
    { 9994, 0xB3A8 }, { 9995, 0xB3AC }, { 9996, 0xB3C4 }, { 9997, 0xB3C5 }, { 9998, 0xB3C8 }, { 9999, 0xB3CB }, { 10000, 0xB3CC }, { 10001, 0xB3CE },
    { 10002, 0xB3D0 }, { 10003, 0xB3D4 }, { 10004, 0xB3D5 }, { 10005, 0xB3D7 }, { 10006, 0xB3D9 }, { 10007, 0xB3DB }, { 10008, 0xB3DD }, { 10009, 0xB3E0 },
    { 10010, 0xB3E4 }, { 10011, 0xB3E8 }, { 10012, 0xB3FC }, { 10013, 0xB410 }, { 10014, 0xB418 }, { 10015, 0xB41C }, { 10016, 0xB420 }, { 10017, 0xB428 },
    { 10018, 0xB429 }, { 10019, 0xB42B }, { 10020, 0xB434 }, { 10021, 0xB450 }, { 10022, 0xB451 }, { 10023, 0xB454 }, { 10024, 0xB458 }, { 10025, 0xB460 },
    { 10026, 0xB461 }, { 10027, 0xB463 }, { 10028, 0xB465 }, { 10029, 0xB46C }, { 10030, 0xB480 }, { 10031, 0xB488 }, { 10032, 0xB49D }, { 10033, 0xB4A4 },
    { 10034, 0xB4A8 }, { 10035, 0xB4AC }, { 10036, 0xB4B5 }, { 10037, 0xB4B7 }, { 10038, 0xB4B9 }, { 10039, 0xB4C0 }, { 10040, 0xB4C4 }, { 10041, 0xB4C8 },
    { 10042, 0xB4D0 }, { 10043, 0xB4D5 }, { 10044, 0xB4DC }, { 10045, 0xB4DD }, { 10046, 0xB4E0 }, { 10047, 0xB4E3 }, { 10048, 0xB4E4 }, { 10049, 0xB4E6 },
    { 10050, 0xB4EC }, { 10051, 0xB4ED }, { 10052, 0xB4EF }, { 10053, 0xB4F1 }, { 10054, 0xB4F8 }, { 10055, 0xB514 }, { 10056, 0xB515 }, { 10057, 0xB518 },
    { 10058, 0xB51B }, { 10059, 0xB51C }, { 10060, 0xB524 }, { 10061, 0xB525 }, { 10062, 0xB527 }, { 10063, 0xB528 }, { 10064, 0xB529 }, { 10065, 0xB52A },
    { 10066, 0xB530 }, { 10067, 0xB531 }, { 10068, 0xB534 }, { 10069, 0xB538 }, { 10070, 0xD105 }, { 10071, 0xD106 }, { 10072, 0xD107 }, { 10073, 0xD108 },
    { 10074, 0xD109 }, { 10075, 0xD10A }, { 10076, 0xD10B }, { 10077, 0xD10C }, { 10078, 0xD10E }, { 10079, 0xD10F }, { 10080, 0xD110 }, { 10081, 0xD111 },
    { 10082, 0xD112 }, { 10083, 0xD113 }, { 10084, 0xD114 }, { 10085, 0xD115 }, { 10086, 0xD116 }, { 10087, 0xD117 }, { 10088, 0xD118 }, { 10089, 0xD119 },
    { 10090, 0xD11A }, { 10091, 0xD11B }, { 10092, 0xD11C }, { 10093, 0xD11D }, { 10094, 0xD11E }, { 10095, 0xD11F }, { 10102, 0xD120 }, { 10103, 0xD121 },
    { 10104, 0xD122 }, { 10105, 0xD123 }, { 10106, 0xD124 }, { 10107, 0xD125 }, { 10108, 0xD126 }, { 10109, 0xD127 }, { 10110, 0xD128 }, { 10111, 0xD129 },
    { 10112, 0xD12A }, { 10113, 0xD12B }, { 10114, 0xD12C }, { 10115, 0xD12D }, { 10116, 0xD12E }, { 10117, 0xD12F }, { 10118, 0xD132 }, { 10119, 0xD133 },
    { 10120, 0xD135 }, { 10121, 0xD136 }, { 10122, 0xD137 }, { 10123, 0xD139 }, { 10124, 0xD13B }, { 10125, 0xD13C }, { 10126, 0xD13D }, { 10127, 0xD13E },
    { 10134, 0xD13F }, { 10135, 0xD142 }, { 10136, 0xD146 }, { 10137, 0xD147 }, { 10138, 0xD148 }, { 10139, 0xD149 }, { 10140, 0xD14A }, { 10141, 0xD14B },
    { 10142, 0xD14E }, { 10143, 0xD14F }, { 10144, 0xD151 }, { 10145, 0xD152 }, { 10146, 0xD153 }, { 10147, 0xD155 }, { 10148, 0xD156 }, { 10149, 0xD157 },
    { 10150, 0xD158 }, { 10151, 0xD159 }, { 10152, 0xD15A }, { 10153, 0xD15B }, { 10154, 0xD15E }, { 10155, 0xD160 }, { 10156, 0xD162 }, { 10157, 0xD163 },
    { 10158, 0xD164 }, { 10159, 0xD165 }, { 10160, 0xD166 }, { 10161, 0xD167 }, { 10162, 0xD169 }, { 10163, 0xD16A }, { 10164, 0xD16B }, { 10165, 0xD16D },
    { 10166, 0xB540 }, { 10167, 0xB541 }, { 10168, 0xB543 }, { 10169, 0xB544 }, { 10170, 0xB545 }, { 10171, 0xB54B }, { 10172, 0xB54C }, { 10173, 0xB54D },
    { 10174, 0xB550 }, { 10175, 0xB554 }, { 10176, 0xB55C }, { 10177, 0xB55D }, { 10178, 0xB55F }, { 10179, 0xB560 }, { 10180, 0xB561 }, { 10181, 0xB5A0 },
    { 10182, 0xB5A1 }, { 10183, 0xB5A4 }, { 10184, 0xB5A8 }, { 10185, 0xB5AA }, { 10186, 0xB5AB }, { 10187, 0xB5B0 }, { 10188, 0xB5B1 }, { 10189, 0xB5B3 },
    { 10190, 0xB5B4 }, { 10191, 0xB5B5 }, { 10192, 0xB5BB }, { 10193, 0xB5BC }, { 10194, 0xB5BD }, { 10195, 0xB5C0 }, { 10196, 0xB5C4 }, { 10197, 0xB5CC },
    { 10198, 0xB5CD }, { 10199, 0xB5CF }, { 10200, 0xB5D0 }, { 10201, 0xB5D1 }, { 10202, 0xB5D8 }, { 10203, 0xB5EC }, { 10204, 0xB610 }, { 10205, 0xB611 },
    { 10206, 0xB614 }, { 10207, 0xB618 }, { 10208, 0xB625 }, { 10209, 0xB62C }, { 10210, 0xB634 }, { 10211, 0xB648 }, { 10212, 0xB664 }, { 10213, 0xB668 },
    { 10214, 0xB69C }, { 10215, 0xB69D }, { 10216, 0xB6A0 }, { 10217, 0xB6A4 }, { 10218, 0xB6AB }, { 10219, 0xB6AC }, { 10220, 0xB6B1 }, { 10221, 0xB6D4 },
    { 10222, 0xB6F0 }, { 10223, 0xB6F4 }, { 10224, 0xB6F8 }, { 10225, 0xB700 }, { 10226, 0xB701 }, { 10227, 0xB705 }, { 10228, 0xB728 }, { 10229, 0xB729 },
    { 10230, 0xB72C }, { 10231, 0xB72F }, { 10232, 0xB730 }, { 10233, 0xB738 }, { 10234, 0xB739 }, { 10235, 0xB73B }, { 10236, 0xB744 }, { 10237, 0xB748 },
    { 10238, 0xB74C }, { 10239, 0xB754 }, { 10240, 0xB755 }, { 10241, 0xB760 }, { 10242, 0xB764 }, { 10243, 0xB768 }, { 10244, 0xB770 }, { 10245, 0xB771 },
    { 10246, 0xB773 }, { 10247, 0xB775 }, { 10248, 0xB77C }, { 10249, 0xB77D }, { 10250, 0xB780 }, { 10251, 0xB784 }, { 10252, 0xB78C }, { 10253, 0xB78D },
    { 10254, 0xB78F }, { 10255, 0xB790 }, { 10256, 0xB791 }, { 10257, 0xB792 }, { 10258, 0xB796 }, { 10259, 0xB797 }, { 10260, 0xD16E }, { 10261, 0xD16F },
    { 10262, 0xD170 }, { 10263, 0xD171 }, { 10264, 0xD172 }, { 10265, 0xD173 }, { 10266, 0xD174 }, { 10267, 0xD175 }, { 10268, 0xD176 }, { 10269, 0xD177 },
    { 10270, 0xD178 }, { 10271, 0xD179 }, { 10272, 0xD17A }, { 10273, 0xD17B }, { 10274, 0xD17D }, { 10275, 0xD17E }, { 10276, 0xD17F }, { 10277, 0xD180 },
    { 10278, 0xD181 }, { 10279, 0xD182 }, { 10280, 0xD183 }, { 10281, 0xD185 }, { 10282, 0xD186 }, { 10283, 0xD187 }, { 10284, 0xD189 }, { 10285, 0xD18A },
    { 10292, 0xD18B }, { 10293, 0xD18C }, { 10294, 0xD18D }, { 10295, 0xD18E }, { 10296, 0xD18F }, { 10297, 0xD190 }, { 10298, 0xD191 }, { 10299, 0xD192 },
    { 10300, 0xD193 }, { 10301, 0xD194 }, { 10302, 0xD195 }, { 10303, 0xD196 }, { 10304, 0xD197 }, { 10305, 0xD198 }, { 10306, 0xD199 }, { 10307, 0xD19A },
    { 10308, 0xD19B }, { 10309, 0xD19C }, { 10310, 0xD19D }, { 10311, 0xD19E }, { 10312, 0xD19F }, { 10313, 0xD1A2 }, { 10314, 0xD1A3 }, { 10315, 0xD1A5 },
    { 10316, 0xD1A6 }, { 10317, 0xD1A7 }, { 10324, 0xD1A9 }, { 10325, 0xD1AA }, { 10326, 0xD1AB }, { 10327, 0xD1AC }, { 10328, 0xD1AD }, { 10329, 0xD1AE },
    { 10330, 0xD1AF }, { 10331, 0xD1B2 }, { 10332, 0xD1B4 }, { 10333, 0xD1B6 }, { 10334, 0xD1B7 }, { 10335, 0xD1B8 }, { 10336, 0xD1B9 }, { 10337, 0xD1BB },
    { 10338, 0xD1BD }, { 10339, 0xD1BE }, { 10340, 0xD1BF }, { 10341, 0xD1C1 }, { 10342, 0xD1C2 }, { 10343, 0xD1C3 }, { 10344, 0xD1C4 }, { 10345, 0xD1C5 },
    { 10346, 0xD1C6 }, { 10347, 0xD1C7 }, { 10348, 0xD1C8 }, { 10349, 0xD1C9 }, { 10350, 0xD1CA }, { 10351, 0xD1CB }, { 10352, 0xD1CC }, { 10353, 0xD1CD },
    { 10354, 0xD1CE }, { 10355, 0xD1CF }, { 10356, 0xB798 }, { 10357, 0xB799 }, { 10358, 0xB79C }, { 10359, 0xB7A0 }, { 10360, 0xB7A8 }, { 10361, 0xB7A9 },
    { 10362, 0xB7AB }, { 10363, 0xB7AC }, { 10364, 0xB7AD }, { 10365, 0xB7B4 }, { 10366, 0xB7B5 }, { 10367, 0xB7B8 }, { 10368, 0xB7C7 }, { 10369, 0xB7C9 },
    { 10370, 0xB7EC }, { 10371, 0xB7ED }, { 10372, 0xB7F0 }, { 10373, 0xB7F4 }, { 10374, 0xB7FC }, { 10375, 0xB7FD }, { 10376, 0xB7FF }, { 10377, 0xB800 },
    { 10378, 0xB801 }, { 10379, 0xB807 }, { 10380, 0xB808 }, { 10381, 0xB809 }, { 10382, 0xB80C }, { 10383, 0xB810 }, { 10384, 0xB818 }, { 10385, 0xB819 },
    { 10386, 0xB81B }, { 10387, 0xB81D }, { 10388, 0xB824 }, { 10389, 0xB825 }, { 10390, 0xB828 }, { 10391, 0xB82C }, { 10392, 0xB834 }, { 10393, 0xB835 },
    { 10394, 0xB837 }, { 10395, 0xB838 }, { 10396, 0xB839 }, { 10397, 0xB840 }, { 10398, 0xB844 }, { 10399, 0xB851 }, { 10400, 0xB853 }, { 10401, 0xB85C },
    { 10402, 0xB85D }, { 10403, 0xB860 }, { 10404, 0xB864 }, { 10405, 0xB86C }, { 10406, 0xB86D }, { 10407, 0xB86F }, { 10408, 0xB871 }, { 10409, 0xB878 },
    { 10410, 0xB87C }, { 10411, 0xB88D }, { 10412, 0xB8A8 }, { 10413, 0xB8B0 }, { 10414, 0xB8B4 }, { 10415, 0xB8B8 }, { 10416, 0xB8C0 }, { 10417, 0xB8C1 },
    { 10418, 0xB8C3 }, { 10419, 0xB8C5 }, { 10420, 0xB8CC }, { 10421, 0xB8D0 }, { 10422, 0xB8D4 }, { 10423, 0xB8DD }, { 10424, 0xB8DF }, { 10425, 0xB8E1 },
    { 10426, 0xB8E8 }, { 10427, 0xB8E9 }, { 10428, 0xB8EC }, { 10429, 0xB8F0 }, { 10430, 0xB8F8 }, { 10431, 0xB8F9 }, { 10432, 0xB8FB }, { 10433, 0xB8FD },
    { 10434, 0xB904 }, { 10435, 0xB918 }, { 10436, 0xB920 }, { 10437, 0xB93C }, { 10438, 0xB93D }, { 10439, 0xB940 }, { 10440, 0xB944 }, { 10441, 0xB94C },
    { 10442, 0xB94F }, { 10443, 0xB951 }, { 10444, 0xB958 }, { 10445, 0xB959 }, { 10446, 0xB95C }, { 10447, 0xB960 }, { 10448, 0xB968 }, { 10449, 0xB969 },
    { 10450, 0xD1D0 }, { 10451, 0xD1D1 }, { 10452, 0xD1D2 }, { 10453, 0xD1D3 }, { 10454, 0xD1D4 }, { 10455, 0xD1D5 }, { 10456, 0xD1D6 }, { 10457, 0xD1D7 },
    { 10458, 0xD1D9 }, { 10459, 0xD1DA }, { 10460, 0xD1DB }, { 10461, 0xD1DC }, { 10462, 0xD1DD }, { 10463, 0xD1DE }, { 10464, 0xD1DF }, { 10465, 0xD1E0 },
    { 10466, 0xD1E1 }, { 10467, 0xD1E2 }, { 10468, 0xD1E3 }, { 10469, 0xD1E4 }, { 10470, 0xD1E5 }, { 10471, 0xD1E6 }, { 10472, 0xD1E7 }, { 10473, 0xD1E8 },
    { 10474, 0xD1E9 }, { 10475, 0xD1EA }, { 10482, 0xD1EB }, { 10483, 0xD1EC }, { 10484, 0xD1ED }, { 10485, 0xD1EE }, { 10486, 0xD1EF }, { 10487, 0xD1F0 },
    { 10488, 0xD1F1 }, { 10489, 0xD1F2 }, { 10490, 0xD1F3 }, { 10491, 0xD1F5 }, { 10492, 0xD1F6 }, { 10493, 0xD1F7 }, { 10494, 0xD1F9 }, { 10495, 0xD1FA },
    { 10496, 0xD1FB }, { 10497, 0xD1FC }, { 10498, 0xD1FD }, { 10499, 0xD1FE }, { 10500, 0xD1FF }, { 10501, 0xD200 }, { 10502, 0xD201 }, { 10503, 0xD202 },
    { 10504, 0xD203 }, { 10505, 0xD204 }, { 10506, 0xD205 }, { 10507, 0xD206 }, { 10514, 0xD208 }, { 10515, 0xD20A }, { 10516, 0xD20B }, { 10517, 0xD20C },
    { 10518, 0xD20D }, { 10519, 0xD20E }, { 10520, 0xD20F }, { 10521, 0xD211 }, { 10522, 0xD212 }, { 10523, 0xD213 }, { 10524, 0xD214 }, { 10525, 0xD215 },
    { 10526, 0xD216 }, { 10527, 0xD217 }, { 10528, 0xD218 }, { 10529, 0xD219 }, { 10530, 0xD21A }, { 10531, 0xD21B }, { 10532, 0xD21C }, { 10533, 0xD21D },
    { 10534, 0xD21E }, { 10535, 0xD21F }, { 10536, 0xD220 }, { 10537, 0xD221 }, { 10538, 0xD222 }, { 10539, 0xD223 }, { 10540, 0xD224 }, { 10541, 0xD225 },
    { 10542, 0xD226 }, { 10543, 0xD227 }, { 10544, 0xD228 }, { 10545, 0xD229 }, { 10546, 0xB96B }, { 10547, 0xB96D }, { 10548, 0xB974 }, { 10549, 0xB975 },
    { 10550, 0xB978 }, { 10551, 0xB97C }, { 10552, 0xB984 }, { 10553, 0xB985 }, { 10554, 0xB987 }, { 10555, 0xB989 }, { 10556, 0xB98A }, { 10557, 0xB98D },
    { 10558, 0xB98E }, { 10559, 0xB9AC }, { 10560, 0xB9AD }, { 10561, 0xB9B0 }, { 10562, 0xB9B4 }, { 10563, 0xB9BC }, { 10564, 0xB9BD }, { 10565, 0xB9BF },
    { 10566, 0xB9C1 }, { 10567, 0xB9C8 }, { 10568, 0xB9C9 }, { 10569, 0xB9CC }, { 10570, 0xB9CE }, { 10571, 0xB9CF }, { 10572, 0xB9D0 }, { 10573, 0xB9D1 },
    { 10574, 0xB9D2 }, { 10575, 0xB9D8 }, { 10576, 0xB9D9 }, { 10577, 0xB9DB }, { 10578, 0xB9DD }, { 10579, 0xB9DE }, { 10580, 0xB9E1 }, { 10581, 0xB9E3 },
    { 10582, 0xB9E4 }, { 10583, 0xB9E5 }, { 10584, 0xB9E8 }, { 10585, 0xB9EC }, { 10586, 0xB9F4 }, { 10587, 0xB9F5 }, { 10588, 0xB9F7 }, { 10589, 0xB9F8 },
    { 10590, 0xB9F9 }, { 10591, 0xB9FA }, { 10592, 0xBA00 }, { 10593, 0xBA01 }, { 10594, 0xBA08 }, { 10595, 0xBA15 }, { 10596, 0xBA38 }, { 10597, 0xBA39 },
    { 10598, 0xBA3C }, { 10599, 0xBA40 }, { 10600, 0xBA42 }, { 10601, 0xBA48 }, { 10602, 0xBA49 }, { 10603, 0xBA4B }, { 10604, 0xBA4D }, { 10605, 0xBA4E },
    { 10606, 0xBA53 }, { 10607, 0xBA54 }, { 10608, 0xBA55 }, { 10609, 0xBA58 }, { 10610, 0xBA5C }, { 10611, 0xBA64 }, { 10612, 0xBA65 }, { 10613, 0xBA67 },
    { 10614, 0xBA68 }, { 10615, 0xBA69 }, { 10616, 0xBA70 }, { 10617, 0xBA71 }, { 10618, 0xBA74 }, { 10619, 0xBA78 }, { 10620, 0xBA83 }, { 10621, 0xBA84 },
    { 10622, 0xBA85 }, { 10623, 0xBA87 }, { 10624, 0xBA8C }, { 10625, 0xBAA8 }, { 10626, 0xBAA9 }, { 10627, 0xBAAB }, { 10628, 0xBAAC }, { 10629, 0xBAB0 },
    { 10630, 0xBAB2 }, { 10631, 0xBAB8 }, { 10632, 0xBAB9 }, { 10633, 0xBABB }, { 10634, 0xBABD }, { 10635, 0xBAC4 }, { 10636, 0xBAC8 }, { 10637, 0xBAD8 },
    { 10638, 0xBAD9 }, { 10639, 0xBAFC }, { 10640, 0xD22A }, { 10641, 0xD22B }, { 10642, 0xD22E }, { 10643, 0xD22F }, { 10644, 0xD231 }, { 10645, 0xD232 },
    { 10646, 0xD233 }, { 10647, 0xD235 }, { 10648, 0xD236 }, { 10649, 0xD237 }, { 10650, 0xD238 }, { 10651, 0xD239 }, { 10652, 0xD23A }, { 10653, 0xD23B },
    { 10654, 0xD23E }, { 10655, 0xD240 }, { 10656, 0xD242 }, { 10657, 0xD243 }, { 10658, 0xD244 }, { 10659, 0xD245 }, { 10660, 0xD246 }, { 10661, 0xD247 },
    { 10662, 0xD249 }, { 10663, 0xD24A }, { 10664, 0xD24B }, { 10665, 0xD24C }, { 10672, 0xD24D }, { 10673, 0xD24E }, { 10674, 0xD24F }, { 10675, 0xD250 },
    { 10676, 0xD251 }, { 10677, 0xD252 }, { 10678, 0xD253 }, { 10679, 0xD254 }, { 10680, 0xD255 }, { 10681, 0xD256 }, { 10682, 0xD257 }, { 10683, 0xD258 },
    { 10684, 0xD259 }, { 10685, 0xD25A }, { 10686, 0xD25B }, { 10687, 0xD25D }, { 10688, 0xD25E }, { 10689, 0xD25F }, { 10690, 0xD260 }, { 10691, 0xD261 },
    { 10692, 0xD262 }, { 10693, 0xD263 }, { 10694, 0xD265 }, { 10695, 0xD266 }, { 10696, 0xD267 }, { 10697, 0xD268 }, { 10704, 0xD269 }, { 10705, 0xD26A },
    { 10706, 0xD26B }, { 10707, 0xD26C }, { 10708, 0xD26D }, { 10709, 0xD26E }, { 10710, 0xD26F }, { 10711, 0xD270 }, { 10712, 0xD271 }, { 10713, 0xD272 },
    { 10714, 0xD273 }, { 10715, 0xD274 }, { 10716, 0xD275 }, { 10717, 0xD276 }, { 10718, 0xD277 }, { 10719, 0xD278 }, { 10720, 0xD279 }, { 10721, 0xD27A },
    { 10722, 0xD27B }, { 10723, 0xD27C }, { 10724, 0xD27D }, { 10725, 0xD27E }, { 10726, 0xD27F }, { 10727, 0xD282 }, { 10728, 0xD283 }, { 10729, 0xD285 },
    { 10730, 0xD286 }, { 10731, 0xD287 }, { 10732, 0xD289 }, { 10733, 0xD28A }, { 10734, 0xD28B }, { 10735, 0xD28C }, { 10736, 0xBB00 }, { 10737, 0xBB04 },
    { 10738, 0xBB0D }, { 10739, 0xBB0F }, { 10740, 0xBB11 }, { 10741, 0xBB18 }, { 10742, 0xBB1C }, { 10743, 0xBB20 }, { 10744, 0xBB29 }, { 10745, 0xBB2B },
    { 10746, 0xBB34 }, { 10747, 0xBB35 }, { 10748, 0xBB36 }, { 10749, 0xBB38 }, { 10750, 0xBB3B }, { 10751, 0xBB3C }, { 10752, 0xBB3D }, { 10753, 0xBB3E },
    { 10754, 0xBB44 }, { 10755, 0xBB45 }, { 10756, 0xBB47 }, { 10757, 0xBB49 }, { 10758, 0xBB4D }, { 10759, 0xBB4F }, { 10760, 0xBB50 }, { 10761, 0xBB54 },
    { 10762, 0xBB58 }, { 10763, 0xBB61 }, { 10764, 0xBB63 }, { 10765, 0xBB6C }, { 10766, 0xBB88 }, { 10767, 0xBB8C }, { 10768, 0xBB90 }, { 10769, 0xBBA4 },
    { 10770, 0xBBA8 }, { 10771, 0xBBAC }, { 10772, 0xBBB4 }, { 10773, 0xBBB7 }, { 10774, 0xBBC0 }, { 10775, 0xBBC4 }, { 10776, 0xBBC8 }, { 10777, 0xBBD0 },
    { 10778, 0xBBD3 }, { 10779, 0xBBF8 }, { 10780, 0xBBF9 }, { 10781, 0xBBFC }, { 10782, 0xBBFF }, { 10783, 0xBC00 }, { 10784, 0xBC02 }, { 10785, 0xBC08 },
    { 10786, 0xBC09 }, { 10787, 0xBC0B }, { 10788, 0xBC0C }, { 10789, 0xBC0D }, { 10790, 0xBC0F }, { 10791, 0xBC11 }, { 10792, 0xBC14 }, { 10793, 0xBC15 },
    { 10794, 0xBC16 }, { 10795, 0xBC17 }, { 10796, 0xBC18 }, { 10797, 0xBC1B }, { 10798, 0xBC1C }, { 10799, 0xBC1D }, { 10800, 0xBC1E }, { 10801, 0xBC1F },
    { 10802, 0xBC24 }, { 10803, 0xBC25 }, { 10804, 0xBC27 }, { 10805, 0xBC29 }, { 10806, 0xBC2D }, { 10807, 0xBC30 }, { 10808, 0xBC31 }, { 10809, 0xBC34 },
    { 10810, 0xBC38 }, { 10811, 0xBC40 }, { 10812, 0xBC41 }, { 10813, 0xBC43 }, { 10814, 0xBC44 }, { 10815, 0xBC45 }, { 10816, 0xBC49 }, { 10817, 0xBC4C },
    { 10818, 0xBC4D }, { 10819, 0xBC50 }, { 10820, 0xBC5D }, { 10821, 0xBC84 }, { 10822, 0xBC85 }, { 10823, 0xBC88 }, { 10824, 0xBC8B }, { 10825, 0xBC8C },
    { 10826, 0xBC8E }, { 10827, 0xBC94 }, { 10828, 0xBC95 }, { 10829, 0xBC97 }, { 10830, 0xD28D }, { 10831, 0xD28E }, { 10832, 0xD28F }, { 10833, 0xD292 },
    { 10834, 0xD293 }, { 10835, 0xD294 }, { 10836, 0xD296 }, { 10837, 0xD297 }, { 10838, 0xD298 }, { 10839, 0xD299 }, { 10840, 0xD29A }, { 10841, 0xD29B },
    { 10842, 0xD29D }, { 10843, 0xD29E }, { 10844, 0xD29F }, { 10845, 0xD2A1 }, { 10846, 0xD2A2 }, { 10847, 0xD2A3 }, { 10848, 0xD2A5 }, { 10849, 0xD2A6 },
    { 10850, 0xD2A7 }, { 10851, 0xD2A8 }, { 10852, 0xD2A9 }, { 10853, 0xD2AA }, { 10854, 0xD2AB }, { 10855, 0xD2AD }, { 10862, 0xD2AE }, { 10863, 0xD2AF },
    { 10864, 0xD2B0 }, { 10865, 0xD2B2 }, { 10866, 0xD2B3 }, { 10867, 0xD2B4 }, { 10868, 0xD2B5 }, { 10869, 0xD2B6 }, { 10870, 0xD2B7 }, { 10871, 0xD2BA },
    { 10872, 0xD2BB }, { 10873, 0xD2BD }, { 10874, 0xD2BE }, { 10875, 0xD2C1 }, { 10876, 0xD2C3 }, { 10877, 0xD2C4 }, { 10878, 0xD2C5 }, { 10879, 0xD2C6 },
    { 10880, 0xD2C7 }, { 10881, 0xD2CA }, { 10882, 0xD2CC }, { 10883, 0xD2CD }, { 10884, 0xD2CE }, { 10885, 0xD2CF }, { 10886, 0xD2D0 }, { 10887, 0xD2D1 },
    { 10894, 0xD2D2 }, { 10895, 0xD2D3 }, { 10896, 0xD2D5 }, { 10897, 0xD2D6 }, { 10898, 0xD2D7 }, { 10899, 0xD2D9 }, { 10900, 0xD2DA }, { 10901, 0xD2DB },
    { 10902, 0xD2DD }, { 10903, 0xD2DE }, { 10904, 0xD2DF }, { 10905, 0xD2E0 }, { 10906, 0xD2E1 }, { 10907, 0xD2E2 }, { 10908, 0xD2E3 }, { 10909, 0xD2E6 },
    { 10910, 0xD2E7 }, { 10911, 0xD2E8 }, { 10912, 0xD2E9 }, { 10913, 0xD2EA }, { 10914, 0xD2EB }, { 10915, 0xD2EC }, { 10916, 0xD2ED }, { 10917, 0xD2EE },
    { 10918, 0xD2EF }, { 10919, 0xD2F2 }, { 10920, 0xD2F3 }, { 10921, 0xD2F5 }, { 10922, 0xD2F6 }, { 10923, 0xD2F7 }, { 10924, 0xD2F9 }, { 10925, 0xD2FA },
    { 10926, 0xBC99 }, { 10927, 0xBC9A }, { 10928, 0xBCA0 }, { 10929, 0xBCA1 }, { 10930, 0xBCA4 }, { 10931, 0xBCA7 }, { 10932, 0xBCA8 }, { 10933, 0xBCB0 },
    { 10934, 0xBCB1 }, { 10935, 0xBCB3 }, { 10936, 0xBCB4 }, { 10937, 0xBCB5 }, { 10938, 0xBCBC }, { 10939, 0xBCBD }, { 10940, 0xBCC0 }, { 10941, 0xBCC4 },
    { 10942, 0xBCCD }, { 10943, 0xBCCF }, { 10944, 0xBCD0 }, { 10945, 0xBCD1 }, { 10946, 0xBCD5 }, { 10947, 0xBCD8 }, { 10948, 0xBCDC }, { 10949, 0xBCF4 },
    { 10950, 0xBCF5 }, { 10951, 0xBCF6 }, { 10952, 0xBCF8 }, { 10953, 0xBCFC }, { 10954, 0xBD04 }, { 10955, 0xBD05 }, { 10956, 0xBD07 }, { 10957, 0xBD09 },
    { 10958, 0xBD10 }, { 10959, 0xBD14 }, { 10960, 0xBD24 }, { 10961, 0xBD2C }, { 10962, 0xBD40 }, { 10963, 0xBD48 }, { 10964, 0xBD49 }, { 10965, 0xBD4C },
    { 10966, 0xBD50 }, { 10967, 0xBD58 }, { 10968, 0xBD59 }, { 10969, 0xBD64 }, { 10970, 0xBD68 }, { 10971, 0xBD80 }, { 10972, 0xBD81 }, { 10973, 0xBD84 },
    { 10974, 0xBD87 }, { 10975, 0xBD88 }, { 10976, 0xBD89 }, { 10977, 0xBD8A }, { 10978, 0xBD90 }, { 10979, 0xBD91 }, { 10980, 0xBD93 }, { 10981, 0xBD95 },
    { 10982, 0xBD99 }, { 10983, 0xBD9A }, { 10984, 0xBD9C }, { 10985, 0xBDA4 }, { 10986, 0xBDB0 }, { 10987, 0xBDB8 }, { 10988, 0xBDD4 }, { 10989, 0xBDD5 },
    { 10990, 0xBDD8 }, { 10991, 0xBDDC }, { 10992, 0xBDE9 }, { 10993, 0xBDF0 }, { 10994, 0xBDF4 }, { 10995, 0xBDF8 }, { 10996, 0xBE00 }, { 10997, 0xBE03 },
    { 10998, 0xBE05 }, { 10999, 0xBE0C }, { 11000, 0xBE0D }, { 11001, 0xBE10 }, { 11002, 0xBE14 }, { 11003, 0xBE1C }, { 11004, 0xBE1D }, { 11005, 0xBE1F },
    { 11006, 0xBE44 }, { 11007, 0xBE45 }, { 11008, 0xBE48 }, { 11009, 0xBE4C }, { 11010, 0xBE4E }, { 11011, 0xBE54 }, { 11012, 0xBE55 }, { 11013, 0xBE57 },
    { 11014, 0xBE59 }, { 11015, 0xBE5A }, { 11016, 0xBE5B }, { 11017, 0xBE60 }, { 11018, 0xBE61 }, { 11019, 0xBE64 }, { 11020, 0xD2FB }, { 11021, 0xD2FC },
    { 11022, 0xD2FD }, { 11023, 0xD2FE }, { 11024, 0xD2FF }, { 11025, 0xD302 }, { 11026, 0xD304 }, { 11027, 0xD306 }, { 11028, 0xD307 }, { 11029, 0xD308 },
    { 11030, 0xD309 }, { 11031, 0xD30A }, { 11032, 0xD30B }, { 11033, 0xD30F }, { 11034, 0xD311 }, { 11035, 0xD312 }, { 11036, 0xD313 }, { 11037, 0xD315 },
    { 11038, 0xD317 }, { 11039, 0xD318 }, { 11040, 0xD319 }, { 11041, 0xD31A }, { 11042, 0xD31B }, { 11043, 0xD31E }, { 11044, 0xD322 }, { 11045, 0xD323 },
    { 11052, 0xD324 }, { 11053, 0xD326 }, { 11054, 0xD327 }, { 11055, 0xD32A }, { 11056, 0xD32B }, { 11057, 0xD32D }, { 11058, 0xD32E }, { 11059, 0xD32F },
    { 11060, 0xD331 }, { 11061, 0xD332 }, { 11062, 0xD333 }, { 11063, 0xD334 }, { 11064, 0xD335 }, { 11065, 0xD336 }, { 11066, 0xD337 }, { 11067, 0xD33A },
    { 11068, 0xD33E }, { 11069, 0xD33F }, { 11070, 0xD340 }, { 11071, 0xD341 }, { 11072, 0xD342 }, { 11073, 0xD343 }, { 11074, 0xD346 }, { 11075, 0xD347 },
    { 11076, 0xD348 }, { 11077, 0xD349 }, { 11084, 0xD34A }, { 11085, 0xD34B }, { 11086, 0xD34C }, { 11087, 0xD34D }, { 11088, 0xD34E }, { 11089, 0xD34F },
    { 11090, 0xD350 }, { 11091, 0xD351 }, { 11092, 0xD352 }, { 11093, 0xD353 }, { 11094, 0xD354 }, { 11095, 0xD355 }, { 11096, 0xD356 }, { 11097, 0xD357 },
    { 11098, 0xD358 }, { 11099, 0xD359 }, { 11100, 0xD35A }, { 11101, 0xD35B }, { 11102, 0xD35C }, { 11103, 0xD35D }, { 11104, 0xD35E }, { 11105, 0xD35F },
    { 11106, 0xD360 }, { 11107, 0xD361 }, { 11108, 0xD362 }, { 11109, 0xD363 }, { 11110, 0xD364 }, { 11111, 0xD365 }, { 11112, 0xD366 }, { 11113, 0xD367 },
    { 11114, 0xD368 }, { 11115, 0xD369 }, { 11116, 0xBE68 }, { 11117, 0xBE6A }, { 11118, 0xBE70 }, { 11119, 0xBE71 }, { 11120, 0xBE73 }, { 11121, 0xBE74 },
    { 11122, 0xBE75 }, { 11123, 0xBE7B }, { 11124, 0xBE7C }, { 11125, 0xBE7D }, { 11126, 0xBE80 }, { 11127, 0xBE84 }, { 11128, 0xBE8C }, { 11129, 0xBE8D },
    { 11130, 0xBE8F }, { 11131, 0xBE90 }, { 11132, 0xBE91 }, { 11133, 0xBE98 }, { 11134, 0xBE99 }, { 11135, 0xBEA8 }, { 11136, 0xBED0 }, { 11137, 0xBED1 },
    { 11138, 0xBED4 }, { 11139, 0xBED7 }, { 11140, 0xBED8 }, { 11141, 0xBEE0 }, { 11142, 0xBEE3 }, { 11143, 0xBEE4 }, { 11144, 0xBEE5 }, { 11145, 0xBEEC },
    { 11146, 0xBF01 }, { 11147, 0xBF08 }, { 11148, 0xBF09 }, { 11149, 0xBF18 }, { 11150, 0xBF19 }, { 11151, 0xBF1B }, { 11152, 0xBF1C }, { 11153, 0xBF1D },
    { 11154, 0xBF40 }, { 11155, 0xBF41 }, { 11156, 0xBF44 }, { 11157, 0xBF48 }, { 11158, 0xBF50 }, { 11159, 0xBF51 }, { 11160, 0xBF55 }, { 11161, 0xBF94 },
    { 11162, 0xBFB0 }, { 11163, 0xBFC5 }, { 11164, 0xBFCC }, { 11165, 0xBFCD }, { 11166, 0xBFD0 }, { 11167, 0xBFD4 }, { 11168, 0xBFDC }, { 11169, 0xBFDF },
    { 11170, 0xBFE1 }, { 11171, 0xC03C }, { 11172, 0xC051 }, { 11173, 0xC058 }, { 11174, 0xC05C }, { 11175, 0xC060 }, { 11176, 0xC068 }, { 11177, 0xC069 },
    { 11178, 0xC090 }, { 11179, 0xC091 }, { 11180, 0xC094 }, { 11181, 0xC098 }, { 11182, 0xC0A0 }, { 11183, 0xC0A1 }, { 11184, 0xC0A3 }, { 11185, 0xC0A5 },
    { 11186, 0xC0AC }, { 11187, 0xC0AD }, { 11188, 0xC0AF }, { 11189, 0xC0B0 }, { 11190, 0xC0B3 }, { 11191, 0xC0B4 }, { 11192, 0xC0B5 }, { 11193, 0xC0B6 },
    { 11194, 0xC0BC }, { 11195, 0xC0BD }, { 11196, 0xC0BF }, { 11197, 0xC0C0 }, { 11198, 0xC0C1 }, { 11199, 0xC0C5 }, { 11200, 0xC0C8 }, { 11201, 0xC0C9 },
    { 11202, 0xC0CC }, { 11203, 0xC0D0 }, { 11204, 0xC0D8 }, { 11205, 0xC0D9 }, { 11206, 0xC0DB }, { 11207, 0xC0DC }, { 11208, 0xC0DD }, { 11209, 0xC0E4 },
    { 11210, 0xD36A }, { 11211, 0xD36B }, { 11212, 0xD36C }, { 11213, 0xD36D }, { 11214, 0xD36E }, { 11215, 0xD36F }, { 11216, 0xD370 }, { 11217, 0xD371 },
    { 11218, 0xD372 }, { 11219, 0xD373 }, { 11220, 0xD374 }, { 11221, 0xD375 }, { 11222, 0xD376 }, { 11223, 0xD377 }, { 11224, 0xD378 }, { 11225, 0xD379 },
    { 11226, 0xD37A }, { 11227, 0xD37B }, { 11228, 0xD37E }, { 11229, 0xD37F }, { 11230, 0xD381 }, { 11231, 0xD382 }, { 11232, 0xD383 }, { 11233, 0xD385 },
    { 11234, 0xD386 }, { 11235, 0xD387 }, { 11242, 0xD388 }, { 11243, 0xD389 }, { 11244, 0xD38A }, { 11245, 0xD38B }, { 11246, 0xD38E }, { 11247, 0xD392 },
    { 11248, 0xD393 }, { 11249, 0xD394 }, { 11250, 0xD395 }, { 11251, 0xD396 }, { 11252, 0xD397 }, { 11253, 0xD39A }, { 11254, 0xD39B }, { 11255, 0xD39D },
    { 11256, 0xD39E }, { 11257, 0xD39F }, { 11258, 0xD3A1 }, { 11259, 0xD3A2 }, { 11260, 0xD3A3 }, { 11261, 0xD3A4 }, { 11262, 0xD3A5 }, { 11263, 0xD3A6 },
    { 11264, 0xD3A7 }, { 11265, 0xD3AA }, { 11266, 0xD3AC }, { 11267, 0xD3AE }, { 11274, 0xD3AF }, { 11275, 0xD3B0 }, { 11276, 0xD3B1 }, { 11277, 0xD3B2 },
    { 11278, 0xD3B3 }, { 11279, 0xD3B5 }, { 11280, 0xD3B6 }, { 11281, 0xD3B7 }, { 11282, 0xD3B9 }, { 11283, 0xD3BA }, { 11284, 0xD3BB }, { 11285, 0xD3BD },
    { 11286, 0xD3BE }, { 11287, 0xD3BF }, { 11288, 0xD3C0 }, { 11289, 0xD3C1 }, { 11290, 0xD3C2 }, { 11291, 0xD3C3 }, { 11292, 0xD3C6 }, { 11293, 0xD3C7 },
    { 11294, 0xD3CA }, { 11295, 0xD3CB }, { 11296, 0xD3CC }, { 11297, 0xD3CD }, { 11298, 0xD3CE }, { 11299, 0xD3CF }, { 11300, 0xD3D1 }, { 11301, 0xD3D2 },
    { 11302, 0xD3D3 }, { 11303, 0xD3D4 }, { 11304, 0xD3D5 }, { 11305, 0xD3D6 }, { 11306, 0xC0E5 }, { 11307, 0xC0E8 }, { 11308, 0xC0EC }, { 11309, 0xC0F4 },
    { 11310, 0xC0F5 }, { 11311, 0xC0F7 }, { 11312, 0xC0F9 }, { 11313, 0xC100 }, { 11314, 0xC104 }, { 11315, 0xC108 }, { 11316, 0xC110 }, { 11317, 0xC115 },
    { 11318, 0xC11C }, { 11319, 0xC11D }, { 11320, 0xC11E }, { 11321, 0xC11F }, { 11322, 0xC120 }, { 11323, 0xC123 }, { 11324, 0xC124 }, { 11325, 0xC126 },
    { 11326, 0xC127 }, { 11327, 0xC12C }, { 11328, 0xC12D }, { 11329, 0xC12F }, { 11330, 0xC130 }, { 11331, 0xC131 }, { 11332, 0xC136 }, { 11333, 0xC138 },
    { 11334, 0xC139 }, { 11335, 0xC13C }, { 11336, 0xC140 }, { 11337, 0xC148 }, { 11338, 0xC149 }, { 11339, 0xC14B }, { 11340, 0xC14C }, { 11341, 0xC14D },
    { 11342, 0xC154 }, { 11343, 0xC155 }, { 11344, 0xC158 }, { 11345, 0xC15C }, { 11346, 0xC164 }, { 11347, 0xC165 }, { 11348, 0xC167 }, { 11349, 0xC168 },
    { 11350, 0xC169 }, { 11351, 0xC170 }, { 11352, 0xC174 }, { 11353, 0xC178 }, { 11354, 0xC185 }, { 11355, 0xC18C }, { 11356, 0xC18D }, { 11357, 0xC18E },
    { 11358, 0xC190 }, { 11359, 0xC194 }, { 11360, 0xC196 }, { 11361, 0xC19C }, { 11362, 0xC19D }, { 11363, 0xC19F }, { 11364, 0xC1A1 }, { 11365, 0xC1A5 },
    { 11366, 0xC1A8 }, { 11367, 0xC1A9 }, { 11368, 0xC1AC }, { 11369, 0xC1B0 }, { 11370, 0xC1BD }, { 11371, 0xC1C4 }, { 11372, 0xC1C8 }, { 11373, 0xC1CC },
    { 11374, 0xC1D4 }, { 11375, 0xC1D7 }, { 11376, 0xC1D8 }, { 11377, 0xC1E0 }, { 11378, 0xC1E4 }, { 11379, 0xC1E8 }, { 11380, 0xC1F0 }, { 11381, 0xC1F1 },
    { 11382, 0xC1F3 }, { 11383, 0xC1FC }, { 11384, 0xC1FD }, { 11385, 0xC200 }, { 11386, 0xC204 }, { 11387, 0xC20C }, { 11388, 0xC20D }, { 11389, 0xC20F },
    { 11390, 0xC211 }, { 11391, 0xC218 }, { 11392, 0xC219 }, { 11393, 0xC21C }, { 11394, 0xC21F }, { 11395, 0xC220 }, { 11396, 0xC228 }, { 11397, 0xC229 },
    { 11398, 0xC22B }, { 11399, 0xC22D }, { 11400, 0xD3D7 }, { 11401, 0xD3D9 }, { 11402, 0xD3DA }, { 11403, 0xD3DB }, { 11404, 0xD3DC }, { 11405, 0xD3DD },
    { 11406, 0xD3DE }, { 11407, 0xD3DF }, { 11408, 0xD3E0 }, { 11409, 0xD3E2 }, { 11410, 0xD3E4 }, { 11411, 0xD3E5 }, { 11412, 0xD3E6 }, { 11413, 0xD3E7 },
    { 11414, 0xD3E8 }, { 11415, 0xD3E9 }, { 11416, 0xD3EA }, { 11417, 0xD3EB }, { 11418, 0xD3EE }, { 11419, 0xD3EF }, { 11420, 0xD3F1 }, { 11421, 0xD3F2 },
    { 11422, 0xD3F3 }, { 11423, 0xD3F5 }, { 11424, 0xD3F6 }, { 11425, 0xD3F7 }, { 11432, 0xD3F8 }, { 11433, 0xD3F9 }, { 11434, 0xD3FA }, { 11435, 0xD3FB },
    { 11436, 0xD3FE }, { 11437, 0xD400 }, { 11438, 0xD402 }, { 11439, 0xD403 }, { 11440, 0xD404 }, { 11441, 0xD405 }, { 11442, 0xD406 }, { 11443, 0xD407 },
    { 11444, 0xD409 }, { 11445, 0xD40A }, { 11446, 0xD40B }, { 11447, 0xD40C }, { 11448, 0xD40D }, { 11449, 0xD40E }, { 11450, 0xD40F }, { 11451, 0xD410 },
    { 11452, 0xD411 }, { 11453, 0xD412 }, { 11454, 0xD413 }, { 11455, 0xD414 }, { 11456, 0xD415 }, { 11457, 0xD416 }, { 11464, 0xD417 }, { 11465, 0xD418 },
    { 11466, 0xD419 }, { 11467, 0xD41A }, { 11468, 0xD41B }, { 11469, 0xD41C }, { 11470, 0xD41E }, { 11471, 0xD41F }, { 11472, 0xD420 }, { 11473, 0xD421 },
    { 11474, 0xD422 }, { 11475, 0xD423 }, { 11476, 0xD424 }, { 11477, 0xD425 }, { 11478, 0xD426 }, { 11479, 0xD427 }, { 11480, 0xD428 }, { 11481, 0xD429 },
    { 11482, 0xD42A }, { 11483, 0xD42B }, { 11484, 0xD42C }, { 11485, 0xD42D }, { 11486, 0xD42E }, { 11487, 0xD42F }, { 11488, 0xD430 }, { 11489, 0xD431 },
    { 11490, 0xD432 }, { 11491, 0xD433 }, { 11492, 0xD434 }, { 11493, 0xD435 }, { 11494, 0xD436 }, { 11495, 0xD437 }, { 11496, 0xC22F }, { 11497, 0xC231 },
    { 11498, 0xC232 }, { 11499, 0xC234 }, { 11500, 0xC248 }, { 11501, 0xC250 }, { 11502, 0xC251 }, { 11503, 0xC254 }, { 11504, 0xC258 }, { 11505, 0xC260 },
    { 11506, 0xC265 }, { 11507, 0xC26C }, { 11508, 0xC26D }, { 11509, 0xC270 }, { 11510, 0xC274 }, { 11511, 0xC27C }, { 11512, 0xC27D }, { 11513, 0xC27F },
    { 11514, 0xC281 }, { 11515, 0xC288 }, { 11516, 0xC289 }, { 11517, 0xC290 }, { 11518, 0xC298 }, { 11519, 0xC29B }, { 11520, 0xC29D }, { 11521, 0xC2A4 },
    { 11522, 0xC2A5 }, { 11523, 0xC2A8 }, { 11524, 0xC2AC }, { 11525, 0xC2AD }, { 11526, 0xC2B4 }, { 11527, 0xC2B5 }, { 11528, 0xC2B7 }, { 11529, 0xC2B9 },
    { 11530, 0xC2DC }, { 11531, 0xC2DD }, { 11532, 0xC2E0 }, { 11533, 0xC2E3 }, { 11534, 0xC2E4 }, { 11535, 0xC2EB }, { 11536, 0xC2EC }, { 11537, 0xC2ED },
    { 11538, 0xC2EF }, { 11539, 0xC2F1 }, { 11540, 0xC2F6 }, { 11541, 0xC2F8 }, { 11542, 0xC2F9 }, { 11543, 0xC2FB }, { 11544, 0xC2FC }, { 11545, 0xC300 },
    { 11546, 0xC308 }, { 11547, 0xC309 }, { 11548, 0xC30C }, { 11549, 0xC30D }, { 11550, 0xC313 }, { 11551, 0xC314 }, { 11552, 0xC315 }, { 11553, 0xC318 },
    { 11554, 0xC31C }, { 11555, 0xC324 }, { 11556, 0xC325 }, { 11557, 0xC328 }, { 11558, 0xC329 }, { 11559, 0xC345 }, { 11560, 0xC368 }, { 11561, 0xC369 },
    { 11562, 0xC36C }, { 11563, 0xC370 }, { 11564, 0xC372 }, { 11565, 0xC378 }, { 11566, 0xC379 }, { 11567, 0xC37C }, { 11568, 0xC37D }, { 11569, 0xC384 },
    { 11570, 0xC388 }, { 11571, 0xC38C }, { 11572, 0xC3C0 }, { 11573, 0xC3D8 }, { 11574, 0xC3D9 }, { 11575, 0xC3DC }, { 11576, 0xC3DF }, { 11577, 0xC3E0 },
    { 11578, 0xC3E2 }, { 11579, 0xC3E8 }, { 11580, 0xC3E9 }, { 11581, 0xC3ED }, { 11582, 0xC3F4 }, { 11583, 0xC3F5 }, { 11584, 0xC3F8 }, { 11585, 0xC408 },
    { 11586, 0xC410 }, { 11587, 0xC424 }, { 11588, 0xC42C }, { 11589, 0xC430 }, { 11590, 0xD438 }, { 11591, 0xD439 }, { 11592, 0xD43A }, { 11593, 0xD43B },
    { 11594, 0xD43C }, { 11595, 0xD43D }, { 11596, 0xD43E }, { 11597, 0xD43F }, { 11598, 0xD441 }, { 11599, 0xD442 }, { 11600, 0xD443 }, { 11601, 0xD445 },
    { 11602, 0xD446 }, { 11603, 0xD447 }, { 11604, 0xD448 }, { 11605, 0xD449 }, { 11606, 0xD44A }, { 11607, 0xD44B }, { 11608, 0xD44C }, { 11609, 0xD44D },
    { 11610, 0xD44E }, { 11611, 0xD44F }, { 11612, 0xD450 }, { 11613, 0xD451 }, { 11614, 0xD452 }, { 11615, 0xD453 }, { 11622, 0xD454 }, { 11623, 0xD455 },
    { 11624, 0xD456 }, { 11625, 0xD457 }, { 11626, 0xD458 }, { 11627, 0xD459 }, { 11628, 0xD45A }, { 11629, 0xD45B }, { 11630, 0xD45D }, { 11631, 0xD45E },
    { 11632, 0xD45F }, { 11633, 0xD461 }, { 11634, 0xD462 }, { 11635, 0xD463 }, { 11636, 0xD465 }, { 11637, 0xD466 }, { 11638, 0xD467 }, { 11639, 0xD468 },
    { 11640, 0xD469 }, { 11641, 0xD46A }, { 11642, 0xD46B }, { 11643, 0xD46C }, { 11644, 0xD46E }, { 11645, 0xD470 }, { 11646, 0xD471 }, { 11647, 0xD472 },
    { 11654, 0xD473 }, { 11655, 0xD474 }, { 11656, 0xD475 }, { 11657, 0xD476 }, { 11658, 0xD477 }, { 11659, 0xD47A }, { 11660, 0xD47B }, { 11661, 0xD47D },
    { 11662, 0xD47E }, { 11663, 0xD481 }, { 11664, 0xD483 }, { 11665, 0xD484 }, { 11666, 0xD485 }, { 11667, 0xD486 }, { 11668, 0xD487 }, { 11669, 0xD48A },
    { 11670, 0xD48C }, { 11671, 0xD48E }, { 11672, 0xD48F }, { 11673, 0xD490 }, { 11674, 0xD491 }, { 11675, 0xD492 }, { 11676, 0xD493 }, { 11677, 0xD495 },
    { 11678, 0xD496 }, { 11679, 0xD497 }, { 11680, 0xD498 }, { 11681, 0xD499 }, { 11682, 0xD49A }, { 11683, 0xD49B }, { 11684, 0xD49C }, { 11685, 0xD49D },
    { 11686, 0xC434 }, { 11687, 0xC43C }, { 11688, 0xC43D }, { 11689, 0xC448 }, { 11690, 0xC464 }, { 11691, 0xC465 }, { 11692, 0xC468 }, { 11693, 0xC46C },
    { 11694, 0xC474 }, { 11695, 0xC475 }, { 11696, 0xC479 }, { 11697, 0xC480 }, { 11698, 0xC494 }, { 11699, 0xC49C }, { 11700, 0xC4B8 }, { 11701, 0xC4BC },
    { 11702, 0xC4E9 }, { 11703, 0xC4F0 }, { 11704, 0xC4F1 }, { 11705, 0xC4F4 }, { 11706, 0xC4F8 }, { 11707, 0xC4FA }, { 11708, 0xC4FF }, { 11709, 0xC500 },
    { 11710, 0xC501 }, { 11711, 0xC50C }, { 11712, 0xC510 }, { 11713, 0xC514 }, { 11714, 0xC51C }, { 11715, 0xC528 }, { 11716, 0xC529 }, { 11717, 0xC52C },
    { 11718, 0xC530 }, { 11719, 0xC538 }, { 11720, 0xC539 }, { 11721, 0xC53B }, { 11722, 0xC53D }, { 11723, 0xC544 }, { 11724, 0xC545 }, { 11725, 0xC548 },
    { 11726, 0xC549 }, { 11727, 0xC54A }, { 11728, 0xC54C }, { 11729, 0xC54D }, { 11730, 0xC54E }, { 11731, 0xC553 }, { 11732, 0xC554 }, { 11733, 0xC555 },
    { 11734, 0xC557 }, { 11735, 0xC558 }, { 11736, 0xC559 }, { 11737, 0xC55D }, { 11738, 0xC55E }, { 11739, 0xC560 }, { 11740, 0xC561 }, { 11741, 0xC564 },
    { 11742, 0xC568 }, { 11743, 0xC570 }, { 11744, 0xC571 }, { 11745, 0xC573 }, { 11746, 0xC574 }, { 11747, 0xC575 }, { 11748, 0xC57C }, { 11749, 0xC57D },
    { 11750, 0xC580 }, { 11751, 0xC584 }, { 11752, 0xC587 }, { 11753, 0xC58C }, { 11754, 0xC58D }, { 11755, 0xC58F }, { 11756, 0xC591 }, { 11757, 0xC595 },
    { 11758, 0xC597 }, { 11759, 0xC598 }, { 11760, 0xC59C }, { 11761, 0xC5A0 }, { 11762, 0xC5A9 }, { 11763, 0xC5B4 }, { 11764, 0xC5B5 }, { 11765, 0xC5B8 },
    { 11766, 0xC5B9 }, { 11767, 0xC5BB }, { 11768, 0xC5BC }, { 11769, 0xC5BD }, { 11770, 0xC5BE }, { 11771, 0xC5C4 }, { 11772, 0xC5C5 }, { 11773, 0xC5C6 },
    { 11774, 0xC5C7 }, { 11775, 0xC5C8 }, { 11776, 0xC5C9 }, { 11777, 0xC5CA }, { 11778, 0xC5CC }, { 11779, 0xC5CE }, { 11780, 0xD49E }, { 11781, 0xD49F },
    { 11782, 0xD4A0 }, { 11783, 0xD4A1 }, { 11784, 0xD4A2 }, { 11785, 0xD4A3 }, { 11786, 0xD4A4 }, { 11787, 0xD4A5 }, { 11788, 0xD4A6 }, { 11789, 0xD4A7 },
    { 11790, 0xD4A8 }, { 11791, 0xD4AA }, { 11792, 0xD4AB }, { 11793, 0xD4AC }, { 11794, 0xD4AD }, { 11795, 0xD4AE }, { 11796, 0xD4AF }, { 11797, 0xD4B0 },
    { 11798, 0xD4B1 }, { 11799, 0xD4B2 }, { 11800, 0xD4B3 }, { 11801, 0xD4B4 }, { 11802, 0xD4B5 }, { 11803, 0xD4B6 }, { 11804, 0xD4B7 }, { 11805, 0xD4B8 },
    { 11812, 0xD4B9 }, { 11813, 0xD4BA }, { 11814, 0xD4BB }, { 11815, 0xD4BC }, { 11816, 0xD4BD }, { 11817, 0xD4BE }, { 11818, 0xD4BF }, { 11819, 0xD4C0 },
    { 11820, 0xD4C1 }, { 11821, 0xD4C2 }, { 11822, 0xD4C3 }, { 11823, 0xD4C4 }, { 11824, 0xD4C5 }, { 11825, 0xD4C6 }, { 11826, 0xD4C7 }, { 11827, 0xD4C8 },
    { 11828, 0xD4C9 }, { 11829, 0xD4CA }, { 11830, 0xD4CB }, { 11831, 0xD4CD }, { 11832, 0xD4CE }, { 11833, 0xD4CF }, { 11834, 0xD4D1 }, { 11835, 0xD4D2 },
    { 11836, 0xD4D3 }, { 11837, 0xD4D5 }, { 11844, 0xD4D6 }, { 11845, 0xD4D7 }, { 11846, 0xD4D8 }, { 11847, 0xD4D9 }, { 11848, 0xD4DA }, { 11849, 0xD4DB },
    { 11850, 0xD4DD }, { 11851, 0xD4DE }, { 11852, 0xD4E0 }, { 11853, 0xD4E1 }, { 11854, 0xD4E2 }, { 11855, 0xD4E3 }, { 11856, 0xD4E4 }, { 11857, 0xD4E5 },
    { 11858, 0xD4E6 }, { 11859, 0xD4E7 }, { 11860, 0xD4E9 }, { 11861, 0xD4EA }, { 11862, 0xD4EB }, { 11863, 0xD4ED }, { 11864, 0xD4EE }, { 11865, 0xD4EF },
    { 11866, 0xD4F1 }, { 11867, 0xD4F2 }, { 11868, 0xD4F3 }, { 11869, 0xD4F4 }, { 11870, 0xD4F5 }, { 11871, 0xD4F6 }, { 11872, 0xD4F7 }, { 11873, 0xD4F9 },
    { 11874, 0xD4FA }, { 11875, 0xD4FC }, { 11876, 0xC5D0 }, { 11877, 0xC5D1 }, { 11878, 0xC5D4 }, { 11879, 0xC5D8 }, { 11880, 0xC5E0 }, { 11881, 0xC5E1 },
    { 11882, 0xC5E3 }, { 11883, 0xC5E5 }, { 11884, 0xC5EC }, { 11885, 0xC5ED }, { 11886, 0xC5EE }, { 11887, 0xC5F0 }, { 11888, 0xC5F4 }, { 11889, 0xC5F6 },
    { 11890, 0xC5F7 }, { 11891, 0xC5FC }, { 11892, 0xC5FD }, { 11893, 0xC5FE }, { 11894, 0xC5FF }, { 11895, 0xC600 }, { 11896, 0xC601 }, { 11897, 0xC605 },
    { 11898, 0xC606 }, { 11899, 0xC607 }, { 11900, 0xC608 }, { 11901, 0xC60C }, { 11902, 0xC610 }, { 11903, 0xC618 }, { 11904, 0xC619 }, { 11905, 0xC61B },
    { 11906, 0xC61C }, { 11907, 0xC624 }, { 11908, 0xC625 }, { 11909, 0xC628 }, { 11910, 0xC62C }, { 11911, 0xC62D }, { 11912, 0xC62E }, { 11913, 0xC630 },
    { 11914, 0xC633 }, { 11915, 0xC634 }, { 11916, 0xC635 }, { 11917, 0xC637 }, { 11918, 0xC639 }, { 11919, 0xC63B }, { 11920, 0xC640 }, { 11921, 0xC641 },
    { 11922, 0xC644 }, { 11923, 0xC648 }, { 11924, 0xC650 }, { 11925, 0xC651 }, { 11926, 0xC653 }, { 11927, 0xC654 }, { 11928, 0xC655 }, { 11929, 0xC65C },
    { 11930, 0xC65D }, { 11931, 0xC660 }, { 11932, 0xC66C }, { 11933, 0xC66F }, { 11934, 0xC671 }, { 11935, 0xC678 }, { 11936, 0xC679 }, { 11937, 0xC67C },
    { 11938, 0xC680 }, { 11939, 0xC688 }, { 11940, 0xC689 }, { 11941, 0xC68B }, { 11942, 0xC68D }, { 11943, 0xC694 }, { 11944, 0xC695 }, { 11945, 0xC698 },
    { 11946, 0xC69C }, { 11947, 0xC6A4 }, { 11948, 0xC6A5 }, { 11949, 0xC6A7 }, { 11950, 0xC6A9 }, { 11951, 0xC6B0 }, { 11952, 0xC6B1 }, { 11953, 0xC6B4 },
    { 11954, 0xC6B8 }, { 11955, 0xC6B9 }, { 11956, 0xC6BA }, { 11957, 0xC6C0 }, { 11958, 0xC6C1 }, { 11959, 0xC6C3 }, { 11960, 0xC6C5 }, { 11961, 0xC6CC },
    { 11962, 0xC6CD }, { 11963, 0xC6D0 }, { 11964, 0xC6D4 }, { 11965, 0xC6DC }, { 11966, 0xC6DD }, { 11967, 0xC6E0 }, { 11968, 0xC6E1 }, { 11969, 0xC6E8 },
    { 11970, 0xD4FE }, { 11971, 0xD4FF }, { 11972, 0xD500 }, { 11973, 0xD501 }, { 11974, 0xD502 }, { 11975, 0xD503 }, { 11976, 0xD505 }, { 11977, 0xD506 },
    { 11978, 0xD507 }, { 11979, 0xD509 }, { 11980, 0xD50A }, { 11981, 0xD50B }, { 11982, 0xD50D }, { 11983, 0xD50E }, { 11984, 0xD50F }, { 11985, 0xD510 },
    { 11986, 0xD511 }, { 11987, 0xD512 }, { 11988, 0xD513 }, { 11989, 0xD516 }, { 11990, 0xD518 }, { 11991, 0xD519 }, { 11992, 0xD51A }, { 11993, 0xD51B },
    { 11994, 0xD51C }, { 11995, 0xD51D }, { 12002, 0xD51E }, { 12003, 0xD51F }, { 12004, 0xD520 }, { 12005, 0xD521 }, { 12006, 0xD522 }, { 12007, 0xD523 },
    { 12008, 0xD524 }, { 12009, 0xD525 }, { 12010, 0xD526 }, { 12011, 0xD527 }, { 12012, 0xD528 }, { 12013, 0xD529 }, { 12014, 0xD52A }, { 12015, 0xD52B },
    { 12016, 0xD52C }, { 12017, 0xD52D }, { 12018, 0xD52E }, { 12019, 0xD52F }, { 12020, 0xD530 }, { 12021, 0xD531 }, { 12022, 0xD532 }, { 12023, 0xD533 },
    { 12024, 0xD534 }, { 12025, 0xD535 }, { 12026, 0xD536 }, { 12027, 0xD537 }, { 12034, 0xD538 }, { 12035, 0xD539 }, { 12036, 0xD53A }, { 12037, 0xD53B },
    { 12038, 0xD53E }, { 12039, 0xD53F }, { 12040, 0xD541 }, { 12041, 0xD542 }, { 12042, 0xD543 }, { 12043, 0xD545 }, { 12044, 0xD546 }, { 12045, 0xD547 },
    { 12046, 0xD548 }, { 12047, 0xD549 }, { 12048, 0xD54A }, { 12049, 0xD54B }, { 12050, 0xD54E }, { 12051, 0xD550 }, { 12052, 0xD552 }, { 12053, 0xD553 },
    { 12054, 0xD554 }, { 12055, 0xD555 }, { 12056, 0xD556 }, { 12057, 0xD557 }, { 12058, 0xD55A }, { 12059, 0xD55B }, { 12060, 0xD55D }, { 12061, 0xD55E },
    { 12062, 0xD55F }, { 12063, 0xD561 }, { 12064, 0xD562 }, { 12065, 0xD563 }, { 12066, 0xC6E9 }, { 12067, 0xC6EC }, { 12068, 0xC6F0 }, { 12069, 0xC6F8 },
    { 12070, 0xC6F9 }, { 12071, 0xC6FD }, { 12072, 0xC704 }, { 12073, 0xC705 }, { 12074, 0xC708 }, { 12075, 0xC70C }, { 12076, 0xC714 }, { 12077, 0xC715 },
    { 12078, 0xC717 }, { 12079, 0xC719 }, { 12080, 0xC720 }, { 12081, 0xC721 }, { 12082, 0xC724 }, { 12083, 0xC728 }, { 12084, 0xC730 }, { 12085, 0xC731 },
    { 12086, 0xC733 }, { 12087, 0xC735 }, { 12088, 0xC737 }, { 12089, 0xC73C }, { 12090, 0xC73D }, { 12091, 0xC740 }, { 12092, 0xC744 }, { 12093, 0xC74A },
    { 12094, 0xC74C }, { 12095, 0xC74D }, { 12096, 0xC74F }, { 12097, 0xC751 }, { 12098, 0xC752 }, { 12099, 0xC753 }, { 12100, 0xC754 }, { 12101, 0xC755 },
    { 12102, 0xC756 }, { 12103, 0xC757 }, { 12104, 0xC758 }, { 12105, 0xC75C }, { 12106, 0xC760 }, { 12107, 0xC768 }, { 12108, 0xC76B }, { 12109, 0xC774 },
    { 12110, 0xC775 }, { 12111, 0xC778 }, { 12112, 0xC77C }, { 12113, 0xC77D }, { 12114, 0xC77E }, { 12115, 0xC783 }, { 12116, 0xC784 }, { 12117, 0xC785 },
    { 12118, 0xC787 }, { 12119, 0xC788 }, { 12120, 0xC789 }, { 12121, 0xC78A }, { 12122, 0xC78E }, { 12123, 0xC790 }, { 12124, 0xC791 }, { 12125, 0xC794 },
    { 12126, 0xC796 }, { 12127, 0xC797 }, { 12128, 0xC798 }, { 12129, 0xC79A }, { 12130, 0xC7A0 }, { 12131, 0xC7A1 }, { 12132, 0xC7A3 }, { 12133, 0xC7A4 },
    { 12134, 0xC7A5 }, { 12135, 0xC7A6 }, { 12136, 0xC7AC }, { 12137, 0xC7AD }, { 12138, 0xC7B0 }, { 12139, 0xC7B4 }, { 12140, 0xC7BC }, { 12141, 0xC7BD },
    { 12142, 0xC7BF }, { 12143, 0xC7C0 }, { 12144, 0xC7C1 }, { 12145, 0xC7C8 }, { 12146, 0xC7C9 }, { 12147, 0xC7CC }, { 12148, 0xC7CE }, { 12149, 0xC7D0 },
    { 12150, 0xC7D8 }, { 12151, 0xC7DD }, { 12152, 0xC7E4 }, { 12153, 0xC7E8 }, { 12154, 0xC7EC }, { 12155, 0xC800 }, { 12156, 0xC801 }, { 12157, 0xC804 },
    { 12158, 0xC808 }, { 12159, 0xC80A }, { 12160, 0xD564 }, { 12161, 0xD566 }, { 12162, 0xD567 }, { 12163, 0xD56A }, { 12164, 0xD56C }, { 12165, 0xD56E },
    { 12166, 0xD56F }, { 12167, 0xD570 }, { 12168, 0xD571 }, { 12169, 0xD572 }, { 12170, 0xD573 }, { 12171, 0xD576 }, { 12172, 0xD577 }, { 12173, 0xD579 },
    { 12174, 0xD57A }, { 12175, 0xD57B }, { 12176, 0xD57D }, { 12177, 0xD57E }, { 12178, 0xD57F }, { 12179, 0xD580 }, { 12180, 0xD581 }, { 12181, 0xD582 },
    { 12182, 0xD583 }, { 12183, 0xD586 }, { 12184, 0xD58A }, { 12185, 0xD58B }, { 12192, 0xD58C }, { 12193, 0xD58D }, { 12194, 0xD58E }, { 12195, 0xD58F },
    { 12196, 0xD591 }, { 12197, 0xD592 }, { 12198, 0xD593 }, { 12199, 0xD594 }, { 12200, 0xD595 }, { 12201, 0xD596 }, { 12202, 0xD597 }, { 12203, 0xD598 },
    { 12204, 0xD599 }, { 12205, 0xD59A }, { 12206, 0xD59B }, { 12207, 0xD59C }, { 12208, 0xD59D }, { 12209, 0xD59E }, { 12210, 0xD59F }, { 12211, 0xD5A0 },
    { 12212, 0xD5A1 }, { 12213, 0xD5A2 }, { 12214, 0xD5A3 }, { 12215, 0xD5A4 }, { 12216, 0xD5A6 }, { 12217, 0xD5A7 }, { 12224, 0xD5A8 }, { 12225, 0xD5A9 },
    { 12226, 0xD5AA }, { 12227, 0xD5AB }, { 12228, 0xD5AC }, { 12229, 0xD5AD }, { 12230, 0xD5AE }, { 12231, 0xD5AF }, { 12232, 0xD5B0 }, { 12233, 0xD5B1 },
    { 12234, 0xD5B2 }, { 12235, 0xD5B3 }, { 12236, 0xD5B4 }, { 12237, 0xD5B5 }, { 12238, 0xD5B6 }, { 12239, 0xD5B7 }, { 12240, 0xD5B8 }, { 12241, 0xD5B9 },
    { 12242, 0xD5BA }, { 12243, 0xD5BB }, { 12244, 0xD5BC }, { 12245, 0xD5BD }, { 12246, 0xD5BE }, { 12247, 0xD5BF }, { 12248, 0xD5C0 }, { 12249, 0xD5C1 },
    { 12250, 0xD5C2 }, { 12251, 0xD5C3 }, { 12252, 0xD5C4 }, { 12253, 0xD5C5 }, { 12254, 0xD5C6 }, { 12255, 0xD5C7 }, { 12256, 0xC810 }, { 12257, 0xC811 },
    { 12258, 0xC813 }, { 12259, 0xC815 }, { 12260, 0xC816 }, { 12261, 0xC81C }, { 12262, 0xC81D }, { 12263, 0xC820 }, { 12264, 0xC824 }, { 12265, 0xC82C },
    { 12266, 0xC82D }, { 12267, 0xC82F }, { 12268, 0xC831 }, { 12269, 0xC838 }, { 12270, 0xC83C }, { 12271, 0xC840 }, { 12272, 0xC848 }, { 12273, 0xC849 },
    { 12274, 0xC84C }, { 12275, 0xC84D }, { 12276, 0xC854 }, { 12277, 0xC870 }, { 12278, 0xC871 }, { 12279, 0xC874 }, { 12280, 0xC878 }, { 12281, 0xC87A },
    { 12282, 0xC880 }, { 12283, 0xC881 }, { 12284, 0xC883 }, { 12285, 0xC885 }, { 12286, 0xC886 }, { 12287, 0xC887 }, { 12288, 0xC88B }, { 12289, 0xC88C },
    { 12290, 0xC88D }, { 12291, 0xC894 }, { 12292, 0xC89D }, { 12293, 0xC89F }, { 12294, 0xC8A1 }, { 12295, 0xC8A8 }, { 12296, 0xC8BC }, { 12297, 0xC8BD },
    { 12298, 0xC8C4 }, { 12299, 0xC8C8 }, { 12300, 0xC8CC }, { 12301, 0xC8D4 }, { 12302, 0xC8D5 }, { 12303, 0xC8D7 }, { 12304, 0xC8D9 }, { 12305, 0xC8E0 },
    { 12306, 0xC8E1 }, { 12307, 0xC8E4 }, { 12308, 0xC8F5 }, { 12309, 0xC8FC }, { 12310, 0xC8FD }, { 12311, 0xC900 }, { 12312, 0xC904 }, { 12313, 0xC905 },
    { 12314, 0xC906 }, { 12315, 0xC90C }, { 12316, 0xC90D }, { 12317, 0xC90F }, { 12318, 0xC911 }, { 12319, 0xC918 }, { 12320, 0xC92C }, { 12321, 0xC934 },
    { 12322, 0xC950 }, { 12323, 0xC951 }, { 12324, 0xC954 }, { 12325, 0xC958 }, { 12326, 0xC960 }, { 12327, 0xC961 }, { 12328, 0xC963 }, { 12329, 0xC96C },
    { 12330, 0xC970 }, { 12331, 0xC974 }, { 12332, 0xC97C }, { 12333, 0xC988 }, { 12334, 0xC989 }, { 12335, 0xC98C }, { 12336, 0xC990 }, { 12337, 0xC998 },
    { 12338, 0xC999 }, { 12339, 0xC99B }, { 12340, 0xC99D }, { 12341, 0xC9C0 }, { 12342, 0xC9C1 }, { 12343, 0xC9C4 }, { 12344, 0xC9C7 }, { 12345, 0xC9C8 },
    { 12346, 0xC9CA }, { 12347, 0xC9D0 }, { 12348, 0xC9D1 }, { 12349, 0xC9D3 }, { 12350, 0xD5CA }, { 12351, 0xD5CB }, { 12352, 0xD5CD }, { 12353, 0xD5CE },
    { 12354, 0xD5CF }, { 12355, 0xD5D1 }, { 12356, 0xD5D3 }, { 12357, 0xD5D4 }, { 12358, 0xD5D5 }, { 12359, 0xD5D6 }, { 12360, 0xD5D7 }, { 12361, 0xD5DA },
    { 12362, 0xD5DC }, { 12363, 0xD5DE }, { 12364, 0xD5DF }, { 12365, 0xD5E0 }, { 12366, 0xD5E1 }, { 12367, 0xD5E2 }, { 12368, 0xD5E3 }, { 12369, 0xD5E6 },
    { 12370, 0xD5E7 }, { 12371, 0xD5E9 }, { 12372, 0xD5EA }, { 12373, 0xD5EB }, { 12374, 0xD5ED }, { 12375, 0xD5EE }, { 12382, 0xD5EF }, { 12383, 0xD5F0 },
    { 12384, 0xD5F1 }, { 12385, 0xD5F2 }, { 12386, 0xD5F3 }, { 12387, 0xD5F6 }, { 12388, 0xD5F8 }, { 12389, 0xD5FA }, { 12390, 0xD5FB }, { 12391, 0xD5FC },
    { 12392, 0xD5FD }, { 12393, 0xD5FE }, { 12394, 0xD5FF }, { 12395, 0xD602 }, { 12396, 0xD603 }, { 12397, 0xD605 }, { 12398, 0xD606 }, { 12399, 0xD607 },
    { 12400, 0xD609 }, { 12401, 0xD60A }, { 12402, 0xD60B }, { 12403, 0xD60C }, { 12404, 0xD60D }, { 12405, 0xD60E }, { 12406, 0xD60F }, { 12407, 0xD612 },
    { 12414, 0xD616 }, { 12415, 0xD617 }, { 12416, 0xD618 }, { 12417, 0xD619 }, { 12418, 0xD61A }, { 12419, 0xD61B }, { 12420, 0xD61D }, { 12421, 0xD61E },
    { 12422, 0xD61F }, { 12423, 0xD621 }, { 12424, 0xD622 }, { 12425, 0xD623 }, { 12426, 0xD625 }, { 12427, 0xD626 }, { 12428, 0xD627 }, { 12429, 0xD628 },
    { 12430, 0xD629 }, { 12431, 0xD62A }, { 12432, 0xD62B }, { 12433, 0xD62C }, { 12434, 0xD62E }, { 12435, 0xD62F }, { 12436, 0xD630 }, { 12437, 0xD631 },
    { 12438, 0xD632 }, { 12439, 0xD633 }, { 12440, 0xD634 }, { 12441, 0xD635 }, { 12442, 0xD636 }, { 12443, 0xD637 }, { 12444, 0xD63A }, { 12445, 0xD63B },
    { 12446, 0xC9D5 }, { 12447, 0xC9D6 }, { 12448, 0xC9D9 }, { 12449, 0xC9DA }, { 12450, 0xC9DC }, { 12451, 0xC9DD }, { 12452, 0xC9E0 }, { 12453, 0xC9E2 },
    { 12454, 0xC9E4 }, { 12455, 0xC9E7 }, { 12456, 0xC9EC }, { 12457, 0xC9ED }, { 12458, 0xC9EF }, { 12459, 0xC9F0 }, { 12460, 0xC9F1 }, { 12461, 0xC9F8 },
    { 12462, 0xC9F9 }, { 12463, 0xC9FC }, { 12464, 0xCA00 }, { 12465, 0xCA08 }, { 12466, 0xCA09 }, { 12467, 0xCA0B }, { 12468, 0xCA0C }, { 12469, 0xCA0D },
    { 12470, 0xCA14 }, { 12471, 0xCA18 }, { 12472, 0xCA29 }, { 12473, 0xCA4C }, { 12474, 0xCA4D }, { 12475, 0xCA50 }, { 12476, 0xCA54 }, { 12477, 0xCA5C },
    { 12478, 0xCA5D }, { 12479, 0xCA5F }, { 12480, 0xCA60 }, { 12481, 0xCA61 }, { 12482, 0xCA68 }, { 12483, 0xCA7D }, { 12484, 0xCA84 }, { 12485, 0xCA98 },
    { 12486, 0xCABC }, { 12487, 0xCABD }, { 12488, 0xCAC0 }, { 12489, 0xCAC4 }, { 12490, 0xCACC }, { 12491, 0xCACD }, { 12492, 0xCACF }, { 12493, 0xCAD1 },
    { 12494, 0xCAD3 }, { 12495, 0xCAD8 }, { 12496, 0xCAD9 }, { 12497, 0xCAE0 }, { 12498, 0xCAEC }, { 12499, 0xCAF4 }, { 12500, 0xCB08 }, { 12501, 0xCB10 },
    { 12502, 0xCB14 }, { 12503, 0xCB18 }, { 12504, 0xCB20 }, { 12505, 0xCB21 }, { 12506, 0xCB41 }, { 12507, 0xCB48 }, { 12508, 0xCB49 }, { 12509, 0xCB4C },
    { 12510, 0xCB50 }, { 12511, 0xCB58 }, { 12512, 0xCB59 }, { 12513, 0xCB5D }, { 12514, 0xCB64 }, { 12515, 0xCB78 }, { 12516, 0xCB79 }, { 12517, 0xCB9C },
    { 12518, 0xCBB8 }, { 12519, 0xCBD4 }, { 12520, 0xCBE4 }, { 12521, 0xCBE7 }, { 12522, 0xCBE9 }, { 12523, 0xCC0C }, { 12524, 0xCC0D }, { 12525, 0xCC10 },
    { 12526, 0xCC14 }, { 12527, 0xCC1C }, { 12528, 0xCC1D }, { 12529, 0xCC21 }, { 12530, 0xCC22 }, { 12531, 0xCC27 }, { 12532, 0xCC28 }, { 12533, 0xCC29 },
    { 12534, 0xCC2C }, { 12535, 0xCC2E }, { 12536, 0xCC30 }, { 12537, 0xCC38 }, { 12538, 0xCC39 }, { 12539, 0xCC3B }, { 12540, 0xD63D }, { 12541, 0xD63E },
    { 12542, 0xD63F }, { 12543, 0xD641 }, { 12544, 0xD642 }, { 12545, 0xD643 }, { 12546, 0xD644 }, { 12547, 0xD646 }, { 12548, 0xD647 }, { 12549, 0xD64A },
    { 12550, 0xD64C }, { 12551, 0xD64E }, { 12552, 0xD64F }, { 12553, 0xD650 }, { 12554, 0xD652 }, { 12555, 0xD653 }, { 12556, 0xD656 }, { 12557, 0xD657 },
    { 12558, 0xD659 }, { 12559, 0xD65A }, { 12560, 0xD65B }, { 12561, 0xD65D }, { 12562, 0xD65E }, { 12563, 0xD65F }, { 12564, 0xD660 }, { 12565, 0xD661 },
    { 12572, 0xD662 }, { 12573, 0xD663 }, { 12574, 0xD664 }, { 12575, 0xD665 }, { 12576, 0xD666 }, { 12577, 0xD668 }, { 12578, 0xD66A }, { 12579, 0xD66B },
    { 12580, 0xD66C }, { 12581, 0xD66D }, { 12582, 0xD66E }, { 12583, 0xD66F }, { 12584, 0xD672 }, { 12585, 0xD673 }, { 12586, 0xD675 }, { 12587, 0xD676 },
    { 12588, 0xD677 }, { 12589, 0xD678 }, { 12590, 0xD679 }, { 12591, 0xD67A }, { 12592, 0xD67B }, { 12593, 0xD67C }, { 12594, 0xD67D }, { 12595, 0xD67E },
    { 12596, 0xD67F }, { 12597, 0xD680 }, { 12604, 0xD681 }, { 12605, 0xD682 }, { 12606, 0xD684 }, { 12607, 0xD686 }, { 12608, 0xD687 }, { 12609, 0xD688 },
    { 12610, 0xD689 }, { 12611, 0xD68A }, { 12612, 0xD68B }, { 12613, 0xD68E }, { 12614, 0xD68F }, { 12615, 0xD691 }, { 12616, 0xD692 }, { 12617, 0xD693 },
    { 12618, 0xD695 }, { 12619, 0xD696 }, { 12620, 0xD697 }, { 12621, 0xD698 }, { 12622, 0xD699 }, { 12623, 0xD69A }, { 12624, 0xD69B }, { 12625, 0xD69C },
    { 12626, 0xD69E }, { 12627, 0xD6A0 }, { 12628, 0xD6A2 }, { 12629, 0xD6A3 }, { 12630, 0xD6A4 }, { 12631, 0xD6A5 }, { 12632, 0xD6A6 }, { 12633, 0xD6A7 },
    { 12634, 0xD6A9 }, { 12635, 0xD6AA }, { 12636, 0xCC3C }, { 12637, 0xCC3D }, { 12638, 0xCC3E }, { 12639, 0xCC44 }, { 12640, 0xCC45 }, { 12641, 0xCC48 },
    { 12642, 0xCC4C }, { 12643, 0xCC54 }, { 12644, 0xCC55 }, { 12645, 0xCC57 }, { 12646, 0xCC58 }, { 12647, 0xCC59 }, { 12648, 0xCC60 }, { 12649, 0xCC64 },
    { 12650, 0xCC66 }, { 12651, 0xCC68 }, { 12652, 0xCC70 }, { 12653, 0xCC75 }, { 12654, 0xCC98 }, { 12655, 0xCC99 }, { 12656, 0xCC9C }, { 12657, 0xCCA0 },
    { 12658, 0xCCA8 }, { 12659, 0xCCA9 }, { 12660, 0xCCAB }, { 12661, 0xCCAC }, { 12662, 0xCCAD }, { 12663, 0xCCB4 }, { 12664, 0xCCB5 }, { 12665, 0xCCB8 },
    { 12666, 0xCCBC }, { 12667, 0xCCC4 }, { 12668, 0xCCC5 }, { 12669, 0xCCC7 }, { 12670, 0xCCC9 }, { 12671, 0xCCD0 }, { 12672, 0xCCD4 }, { 12673, 0xCCE4 },
    { 12674, 0xCCEC }, { 12675, 0xCCF0 }, { 12676, 0xCD01 }, { 12677, 0xCD08 }, { 12678, 0xCD09 }, { 12679, 0xCD0C }, { 12680, 0xCD10 }, { 12681, 0xCD18 },
    { 12682, 0xCD19 }, { 12683, 0xCD1B }, { 12684, 0xCD1D }, { 12685, 0xCD24 }, { 12686, 0xCD28 }, { 12687, 0xCD2C }, { 12688, 0xCD39 }, { 12689, 0xCD5C },
    { 12690, 0xCD60 }, { 12691, 0xCD64 }, { 12692, 0xCD6C }, { 12693, 0xCD6D }, { 12694, 0xCD6F }, { 12695, 0xCD71 }, { 12696, 0xCD78 }, { 12697, 0xCD88 },
    { 12698, 0xCD94 }, { 12699, 0xCD95 }, { 12700, 0xCD98 }, { 12701, 0xCD9C }, { 12702, 0xCDA4 }, { 12703, 0xCDA5 }, { 12704, 0xCDA7 }, { 12705, 0xCDA9 },
    { 12706, 0xCDB0 }, { 12707, 0xCDC4 }, { 12708, 0xCDCC }, { 12709, 0xCDD0 }, { 12710, 0xCDE8 }, { 12711, 0xCDEC }, { 12712, 0xCDF0 }, { 12713, 0xCDF8 },
    { 12714, 0xCDF9 }, { 12715, 0xCDFB }, { 12716, 0xCDFD }, { 12717, 0xCE04 }, { 12718, 0xCE08 }, { 12719, 0xCE0C }, { 12720, 0xCE14 }, { 12721, 0xCE19 },
    { 12722, 0xCE20 }, { 12723, 0xCE21 }, { 12724, 0xCE24 }, { 12725, 0xCE28 }, { 12726, 0xCE30 }, { 12727, 0xCE31 }, { 12728, 0xCE33 }, { 12729, 0xCE35 },
    { 12730, 0xD6AB }, { 12731, 0xD6AD }, { 12732, 0xD6AE }, { 12733, 0xD6AF }, { 12734, 0xD6B1 }, { 12735, 0xD6B2 }, { 12736, 0xD6B3 }, { 12737, 0xD6B4 },
    { 12738, 0xD6B5 }, { 12739, 0xD6B6 }, { 12740, 0xD6B7 }, { 12741, 0xD6B8 }, { 12742, 0xD6BA }, { 12743, 0xD6BC }, { 12744, 0xD6BD }, { 12745, 0xD6BE },
    { 12746, 0xD6BF }, { 12747, 0xD6C0 }, { 12748, 0xD6C1 }, { 12749, 0xD6C2 }, { 12750, 0xD6C3 }, { 12751, 0xD6C6 }, { 12752, 0xD6C7 }, { 12753, 0xD6C9 },
    { 12754, 0xD6CA }, { 12755, 0xD6CB }, { 12762, 0xD6CD }, { 12763, 0xD6CE }, { 12764, 0xD6CF }, { 12765, 0xD6D0 }, { 12766, 0xD6D2 }, { 12767, 0xD6D3 },
    { 12768, 0xD6D5 }, { 12769, 0xD6D6 }, { 12770, 0xD6D8 }, { 12771, 0xD6DA }, { 12772, 0xD6DB }, { 12773, 0xD6DC }, { 12774, 0xD6DD }, { 12775, 0xD6DE },
    { 12776, 0xD6DF }, { 12777, 0xD6E1 }, { 12778, 0xD6E2 }, { 12779, 0xD6E3 }, { 12780, 0xD6E5 }, { 12781, 0xD6E6 }, { 12782, 0xD6E7 }, { 12783, 0xD6E9 },
    { 12784, 0xD6EA }, { 12785, 0xD6EB }, { 12786, 0xD6EC }, { 12787, 0xD6ED }, { 12794, 0xD6EE }, { 12795, 0xD6EF }, { 12796, 0xD6F1 }, { 12797, 0xD6F2 },
    { 12798, 0xD6F3 }, { 12799, 0xD6F4 }, { 12800, 0xD6F6 }, { 12801, 0xD6F7 }, { 12802, 0xD6F8 }, { 12803, 0xD6F9 }, { 12804, 0xD6FA }, { 12805, 0xD6FB },
    { 12806, 0xD6FE }, { 12807, 0xD6FF }, { 12808, 0xD701 }, { 12809, 0xD702 }, { 12810, 0xD703 }, { 12811, 0xD705 }, { 12812, 0xD706 }, { 12813, 0xD707 },
    { 12814, 0xD708 }, { 12815, 0xD709 }, { 12816, 0xD70A }, { 12817, 0xD70B }, { 12818, 0xD70C }, { 12819, 0xD70D }, { 12820, 0xD70E }, { 12821, 0xD70F },
    { 12822, 0xD710 }, { 12823, 0xD712 }, { 12824, 0xD713 }, { 12825, 0xD714 }, { 12826, 0xCE58 }, { 12827, 0xCE59 }, { 12828, 0xCE5C }, { 12829, 0xCE5F },
    { 12830, 0xCE60 }, { 12831, 0xCE61 }, { 12832, 0xCE68 }, { 12833, 0xCE69 }, { 12834, 0xCE6B }, { 12835, 0xCE6D }, { 12836, 0xCE74 }, { 12837, 0xCE75 },
    { 12838, 0xCE78 }, { 12839, 0xCE7C }, { 12840, 0xCE84 }, { 12841, 0xCE85 }, { 12842, 0xCE87 }, { 12843, 0xCE89 }, { 12844, 0xCE90 }, { 12845, 0xCE91 },
    { 12846, 0xCE94 }, { 12847, 0xCE98 }, { 12848, 0xCEA0 }, { 12849, 0xCEA1 }, { 12850, 0xCEA3 }, { 12851, 0xCEA4 }, { 12852, 0xCEA5 }, { 12853, 0xCEAC },
    { 12854, 0xCEAD }, { 12855, 0xCEC1 }, { 12856, 0xCEE4 }, { 12857, 0xCEE5 }, { 12858, 0xCEE8 }, { 12859, 0xCEEB }, { 12860, 0xCEEC }, { 12861, 0xCEF4 },
    { 12862, 0xCEF5 }, { 12863, 0xCEF7 }, { 12864, 0xCEF8 }, { 12865, 0xCEF9 }, { 12866, 0xCF00 }, { 12867, 0xCF01 }, { 12868, 0xCF04 }, { 12869, 0xCF08 },
    { 12870, 0xCF10 }, { 12871, 0xCF11 }, { 12872, 0xCF13 }, { 12873, 0xCF15 }, { 12874, 0xCF1C }, { 12875, 0xCF20 }, { 12876, 0xCF24 }, { 12877, 0xCF2C },
    { 12878, 0xCF2D }, { 12879, 0xCF2F }, { 12880, 0xCF30 }, { 12881, 0xCF31 }, { 12882, 0xCF38 }, { 12883, 0xCF54 }, { 12884, 0xCF55 }, { 12885, 0xCF58 },
    { 12886, 0xCF5C }, { 12887, 0xCF64 }, { 12888, 0xCF65 }, { 12889, 0xCF67 }, { 12890, 0xCF69 }, { 12891, 0xCF70 }, { 12892, 0xCF71 }, { 12893, 0xCF74 },
    { 12894, 0xCF78 }, { 12895, 0xCF80 }, { 12896, 0xCF85 }, { 12897, 0xCF8C }, { 12898, 0xCFA1 }, { 12899, 0xCFA8 }, { 12900, 0xCFB0 }, { 12901, 0xCFC4 },
    { 12902, 0xCFE0 }, { 12903, 0xCFE1 }, { 12904, 0xCFE4 }, { 12905, 0xCFE8 }, { 12906, 0xCFF0 }, { 12907, 0xCFF1 }, { 12908, 0xCFF3 }, { 12909, 0xCFF5 },
    { 12910, 0xCFFC }, { 12911, 0xD000 }, { 12912, 0xD004 }, { 12913, 0xD011 }, { 12914, 0xD018 }, { 12915, 0xD02D }, { 12916, 0xD034 }, { 12917, 0xD035 },
    { 12918, 0xD038 }, { 12919, 0xD03C }, { 12920, 0xD715 }, { 12921, 0xD716 }, { 12922, 0xD717 }, { 12923, 0xD71A }, { 12924, 0xD71B }, { 12925, 0xD71D },
    { 12926, 0xD71E }, { 12927, 0xD71F }, { 12928, 0xD721 }, { 12929, 0xD722 }, { 12930, 0xD723 }, { 12931, 0xD724 }, { 12932, 0xD725 }, { 12933, 0xD726 },
    { 12934, 0xD727 }, { 12935, 0xD72A }, { 12936, 0xD72C }, { 12937, 0xD72E }, { 12938, 0xD72F }, { 12939, 0xD730 }, { 12940, 0xD731 }, { 12941, 0xD732 },
    { 12942, 0xD733 }, { 12943, 0xD736 }, { 12944, 0xD737 }, { 12945, 0xD739 }, { 12952, 0xD73A }, { 12953, 0xD73B }, { 12954, 0xD73D }, { 12955, 0xD73E },
    { 12956, 0xD73F }, { 12957, 0xD740 }, { 12958, 0xD741 }, { 12959, 0xD742 }, { 12960, 0xD743 }, { 12961, 0xD745 }, { 12962, 0xD746 }, { 12963, 0xD748 },
    { 12964, 0xD74A }, { 12965, 0xD74B }, { 12966, 0xD74C }, { 12967, 0xD74D }, { 12968, 0xD74E }, { 12969, 0xD74F }, { 12970, 0xD752 }, { 12971, 0xD753 },
    { 12972, 0xD755 }, { 12973, 0xD75A }, { 12974, 0xD75B }, { 12975, 0xD75C }, { 12976, 0xD75D }, { 12977, 0xD75E }, { 12984, 0xD75F }, { 12985, 0xD762 },
    { 12986, 0xD764 }, { 12987, 0xD766 }, { 12988, 0xD767 }, { 12989, 0xD768 }, { 12990, 0xD76A }, { 12991, 0xD76B }, { 12992, 0xD76D }, { 12993, 0xD76E },
    { 12994, 0xD76F }, { 12995, 0xD771 }, { 12996, 0xD772 }, { 12997, 0xD773 }, { 12998, 0xD775 }, { 12999, 0xD776 }, { 13000, 0xD777 }, { 13001, 0xD778 },
    { 13002, 0xD779 }, { 13003, 0xD77A }, { 13004, 0xD77B }, { 13005, 0xD77E }, { 13006, 0xD77F }, { 13007, 0xD780 }, { 13008, 0xD782 }, { 13009, 0xD783 },
    { 13010, 0xD784 }, { 13011, 0xD785 }, { 13012, 0xD786 }, { 13013, 0xD787 }, { 13014, 0xD78A }, { 13015, 0xD78B }, { 13016, 0xD044 }, { 13017, 0xD045 },
    { 13018, 0xD047 }, { 13019, 0xD049 }, { 13020, 0xD050 }, { 13021, 0xD054 }, { 13022, 0xD058 }, { 13023, 0xD060 }, { 13024, 0xD06C }, { 13025, 0xD06D },
    { 13026, 0xD070 }, { 13027, 0xD074 }, { 13028, 0xD07C }, { 13029, 0xD07D }, { 13030, 0xD081 }, { 13031, 0xD0A4 }, { 13032, 0xD0A5 }, { 13033, 0xD0A8 },
    { 13034, 0xD0AC }, { 13035, 0xD0B4 }, { 13036, 0xD0B5 }, { 13037, 0xD0B7 }, { 13038, 0xD0B9 }, { 13039, 0xD0C0 }, { 13040, 0xD0C1 }, { 13041, 0xD0C4 },
    { 13042, 0xD0C8 }, { 13043, 0xD0C9 }, { 13044, 0xD0D0 }, { 13045, 0xD0D1 }, { 13046, 0xD0D3 }, { 13047, 0xD0D4 }, { 13048, 0xD0D5 }, { 13049, 0xD0DC },
    { 13050, 0xD0DD }, { 13051, 0xD0E0 }, { 13052, 0xD0E4 }, { 13053, 0xD0EC }, { 13054, 0xD0ED }, { 13055, 0xD0EF }, { 13056, 0xD0F0 }, { 13057, 0xD0F1 },
    { 13058, 0xD0F8 }, { 13059, 0xD10D }, { 13060, 0xD130 }, { 13061, 0xD131 }, { 13062, 0xD134 }, { 13063, 0xD138 }, { 13064, 0xD13A }, { 13065, 0xD140 },
    { 13066, 0xD141 }, { 13067, 0xD143 }, { 13068, 0xD144 }, { 13069, 0xD145 }, { 13070, 0xD14C }, { 13071, 0xD14D }, { 13072, 0xD150 }, { 13073, 0xD154 },
    { 13074, 0xD15C }, { 13075, 0xD15D }, { 13076, 0xD15F }, { 13077, 0xD161 }, { 13078, 0xD168 }, { 13079, 0xD16C }, { 13080, 0xD17C }, { 13081, 0xD184 },
    { 13082, 0xD188 }, { 13083, 0xD1A0 }, { 13084, 0xD1A1 }, { 13085, 0xD1A4 }, { 13086, 0xD1A8 }, { 13087, 0xD1B0 }, { 13088, 0xD1B1 }, { 13089, 0xD1B3 },
    { 13090, 0xD1B5 }, { 13091, 0xD1BA }, { 13092, 0xD1BC }, { 13093, 0xD1C0 }, { 13094, 0xD1D8 }, { 13095, 0xD1F4 }, { 13096, 0xD1F8 }, { 13097, 0xD207 },
    { 13098, 0xD209 }, { 13099, 0xD210 }, { 13100, 0xD22C }, { 13101, 0xD22D }, { 13102, 0xD230 }, { 13103, 0xD234 }, { 13104, 0xD23C }, { 13105, 0xD23D },
    { 13106, 0xD23F }, { 13107, 0xD241 }, { 13108, 0xD248 }, { 13109, 0xD25C }, { 13110, 0xD78D }, { 13111, 0xD78E }, { 13112, 0xD78F }, { 13113, 0xD791 },
    { 13114, 0xD792 }, { 13115, 0xD793 }, { 13116, 0xD794 }, { 13117, 0xD795 }, { 13118, 0xD796 }, { 13119, 0xD797 }, { 13120, 0xD79A }, { 13121, 0xD79C },
    { 13122, 0xD79E }, { 13123, 0xD79F }, { 13124, 0xD7A0 }, { 13125, 0xD7A1 }, { 13126, 0xD7A2 }, { 13127, 0xD7A3 }, { 13206, 0xD264 }, { 13207, 0xD280 },
    { 13208, 0xD281 }, { 13209, 0xD284 }, { 13210, 0xD288 }, { 13211, 0xD290 }, { 13212, 0xD291 }, { 13213, 0xD295 }, { 13214, 0xD29C }, { 13215, 0xD2A0 },
    { 13216, 0xD2A4 }, { 13217, 0xD2AC }, { 13218, 0xD2B1 }, { 13219, 0xD2B8 }, { 13220, 0xD2B9 }, { 13221, 0xD2BC }, { 13222, 0xD2BF }, { 13223, 0xD2C0 },
    { 13224, 0xD2C2 }, { 13225, 0xD2C8 }, { 13226, 0xD2C9 }, { 13227, 0xD2CB }, { 13228, 0xD2D4 }, { 13229, 0xD2D8 }, { 13230, 0xD2DC }, { 13231, 0xD2E4 },
    { 13232, 0xD2E5 }, { 13233, 0xD2F0 }, { 13234, 0xD2F1 }, { 13235, 0xD2F4 }, { 13236, 0xD2F8 }, { 13237, 0xD300 }, { 13238, 0xD301 }, { 13239, 0xD303 },
    { 13240, 0xD305 }, { 13241, 0xD30C }, { 13242, 0xD30D }, { 13243, 0xD30E }, { 13244, 0xD310 }, { 13245, 0xD314 }, { 13246, 0xD316 }, { 13247, 0xD31C },
    { 13248, 0xD31D }, { 13249, 0xD31F }, { 13250, 0xD320 }, { 13251, 0xD321 }, { 13252, 0xD325 }, { 13253, 0xD328 }, { 13254, 0xD329 }, { 13255, 0xD32C },
    { 13256, 0xD330 }, { 13257, 0xD338 }, { 13258, 0xD339 }, { 13259, 0xD33B }, { 13260, 0xD33C }, { 13261, 0xD33D }, { 13262, 0xD344 }, { 13263, 0xD345 },
    { 13264, 0xD37C }, { 13265, 0xD37D }, { 13266, 0xD380 }, { 13267, 0xD384 }, { 13268, 0xD38C }, { 13269, 0xD38D }, { 13270, 0xD38F }, { 13271, 0xD390 },
    { 13272, 0xD391 }, { 13273, 0xD398 }, { 13274, 0xD399 }, { 13275, 0xD39C }, { 13276, 0xD3A0 }, { 13277, 0xD3A8 }, { 13278, 0xD3A9 }, { 13279, 0xD3AB },
    { 13280, 0xD3AD }, { 13281, 0xD3B4 }, { 13282, 0xD3B8 }, { 13283, 0xD3BC }, { 13284, 0xD3C4 }, { 13285, 0xD3C5 }, { 13286, 0xD3C8 }, { 13287, 0xD3C9 },
    { 13288, 0xD3D0 }, { 13289, 0xD3D8 }, { 13290, 0xD3E1 }, { 13291, 0xD3E3 }, { 13292, 0xD3EC }, { 13293, 0xD3ED }, { 13294, 0xD3F0 }, { 13295, 0xD3F4 },
    { 13296, 0xD3FC }, { 13297, 0xD3FD }, { 13298, 0xD3FF }, { 13299, 0xD401 }, { 13396, 0xD408 }, { 13397, 0xD41D }, { 13398, 0xD440 }, { 13399, 0xD444 },
    { 13400, 0xD45C }, { 13401, 0xD460 }, { 13402, 0xD464 }, { 13403, 0xD46D }, { 13404, 0xD46F }, { 13405, 0xD478 }, { 13406, 0xD479 }, { 13407, 0xD47C },
    { 13408, 0xD47F }, { 13409, 0xD480 }, { 13410, 0xD482 }, { 13411, 0xD488 }, { 13412, 0xD489 }, { 13413, 0xD48B }, { 13414, 0xD48D }, { 13415, 0xD494 },
    { 13416, 0xD4A9 }, { 13417, 0xD4CC }, { 13418, 0xD4D0 }, { 13419, 0xD4D4 }, { 13420, 0xD4DC }, { 13421, 0xD4DF }, { 13422, 0xD4E8 }, { 13423, 0xD4EC },
    { 13424, 0xD4F0 }, { 13425, 0xD4F8 }, { 13426, 0xD4FB }, { 13427, 0xD4FD }, { 13428, 0xD504 }, { 13429, 0xD508 }, { 13430, 0xD50C }, { 13431, 0xD514 },
    { 13432, 0xD515 }, { 13433, 0xD517 }, { 13434, 0xD53C }, { 13435, 0xD53D }, { 13436, 0xD540 }, { 13437, 0xD544 }, { 13438, 0xD54C }, { 13439, 0xD54D },
    { 13440, 0xD54F }, { 13441, 0xD551 }, { 13442, 0xD558 }, { 13443, 0xD559 }, { 13444, 0xD55C }, { 13445, 0xD560 }, { 13446, 0xD565 }, { 13447, 0xD568 },
    { 13448, 0xD569 }, { 13449, 0xD56B }, { 13450, 0xD56D }, { 13451, 0xD574 }, { 13452, 0xD575 }, { 13453, 0xD578 }, { 13454, 0xD57C }, { 13455, 0xD584 },
    { 13456, 0xD585 }, { 13457, 0xD587 }, { 13458, 0xD588 }, { 13459, 0xD589 }, { 13460, 0xD590 }, { 13461, 0xD5A5 }, { 13462, 0xD5C8 }, { 13463, 0xD5C9 },
    { 13464, 0xD5CC }, { 13465, 0xD5D0 }, { 13466, 0xD5D2 }, { 13467, 0xD5D8 }, { 13468, 0xD5D9 }, { 13469, 0xD5DB }, { 13470, 0xD5DD }, { 13471, 0xD5E4 },
    { 13472, 0xD5E5 }, { 13473, 0xD5E8 }, { 13474, 0xD5EC }, { 13475, 0xD5F4 }, { 13476, 0xD5F5 }, { 13477, 0xD5F7 }, { 13478, 0xD5F9 }, { 13479, 0xD600 },
    { 13480, 0xD601 }, { 13481, 0xD604 }, { 13482, 0xD608 }, { 13483, 0xD610 }, { 13484, 0xD611 }, { 13485, 0xD613 }, { 13486, 0xD614 }, { 13487, 0xD615 },
    { 13488, 0xD61C }, { 13489, 0xD620 }, { 13586, 0xD624 }, { 13587, 0xD62D }, { 13588, 0xD638 }, { 13589, 0xD639 }, { 13590, 0xD63C }, { 13591, 0xD640 },
    { 13592, 0xD645 }, { 13593, 0xD648 }, { 13594, 0xD649 }, { 13595, 0xD64B }, { 13596, 0xD64D }, { 13597, 0xD651 }, { 13598, 0xD654 }, { 13599, 0xD655 },
    { 13600, 0xD658 }, { 13601, 0xD65C }, { 13602, 0xD667 }, { 13603, 0xD669 }, { 13604, 0xD670 }, { 13605, 0xD671 }, { 13606, 0xD674 }, { 13607, 0xD683 },
    { 13608, 0xD685 }, { 13609, 0xD68C }, { 13610, 0xD68D }, { 13611, 0xD690 }, { 13612, 0xD694 }, { 13613, 0xD69D }, { 13614, 0xD69F }, { 13615, 0xD6A1 },
    { 13616, 0xD6A8 }, { 13617, 0xD6AC }, { 13618, 0xD6B0 }, { 13619, 0xD6B9 }, { 13620, 0xD6BB }, { 13621, 0xD6C4 }, { 13622, 0xD6C5 }, { 13623, 0xD6C8 },
    { 13624, 0xD6CC }, { 13625, 0xD6D1 }, { 13626, 0xD6D4 }, { 13627, 0xD6D7 }, { 13628, 0xD6D9 }, { 13629, 0xD6E0 }, { 13630, 0xD6E4 }, { 13631, 0xD6E8 },
    { 13632, 0xD6F0 }, { 13633, 0xD6F5 }, { 13634, 0xD6FC }, { 13635, 0xD6FD }, { 13636, 0xD700 }, { 13637, 0xD704 }, { 13638, 0xD711 }, { 13639, 0xD718 },
    { 13640, 0xD719 }, { 13641, 0xD71C }, { 13642, 0xD720 }, { 13643, 0xD728 }, { 13644, 0xD729 }, { 13645, 0xD72B }, { 13646, 0xD72D }, { 13647, 0xD734 },
    { 13648, 0xD735 }, { 13649, 0xD738 }, { 13650, 0xD73C }, { 13651, 0xD744 }, { 13652, 0xD747 }, { 13653, 0xD749 }, { 13654, 0xD750 }, { 13655, 0xD751 },
    { 13656, 0xD754 }, { 13657, 0xD756 }, { 13658, 0xD757 }, { 13659, 0xD758 }, { 13660, 0xD759 }, { 13661, 0xD760 }, { 13662, 0xD761 }, { 13663, 0xD763 },
    { 13664, 0xD765 }, { 13665, 0xD769 }, { 13666, 0xD76C }, { 13667, 0xD770 }, { 13668, 0xD774 }, { 13669, 0xD77C }, { 13670, 0xD77D }, { 13671, 0xD781 },
    { 13672, 0xD788 }, { 13673, 0xD789 }, { 13674, 0xD78C }, { 13675, 0xD790 }, { 13676, 0xD798 }, { 13677, 0xD799 }, { 13678, 0xD79B }, { 13679, 0xD79D },
    { 13966, 0x4F3D }, { 13967, 0x4F73 }, { 13968, 0x5047 }, { 13969, 0x50F9 }, { 13970, 0x52A0 }, { 13971, 0x53EF }, { 13972, 0x5475 }, { 13973, 0x54E5 },
    { 13974, 0x5609 }, { 13975, 0x5AC1 }, { 13976, 0x5BB6 }, { 13977, 0x6687 }, { 13978, 0x67B6 }, { 13979, 0x67B7 }, { 13980, 0x67EF }, { 13981, 0x6B4C },
    { 13982, 0x73C2 }, { 13983, 0x75C2 }, { 13984, 0x7A3C }, { 13985, 0x82DB }, { 13986, 0x8304 }, { 13987, 0x8857 }, { 13988, 0x8888 }, { 13989, 0x8A36 },
    { 13990, 0x8CC8 }, { 13991, 0x8DCF }, { 13992, 0x8EFB }, { 13993, 0x8FE6 }, { 13994, 0x99D5 }, { 13995, 0x523B }, { 13996, 0x5374 }, { 13997, 0x5404 },
    { 13998, 0x606A }, { 13999, 0x6164 }, { 14000, 0x6BBC }, { 14001, 0x73CF }, { 14002, 0x811A }, { 14003, 0x89BA }, { 14004, 0x89D2 }, { 14005, 0x95A3 },
    { 14006, 0x4F83 }, { 14007, 0x520A }, { 14008, 0x58BE }, { 14009, 0x5978 }, { 14010, 0x59E6 }, { 14011, 0x5E72 }, { 14012, 0x5E79 }, { 14013, 0x61C7 },
    { 14014, 0x63C0 }, { 14015, 0x6746 }, { 14016, 0x67EC }, { 14017, 0x687F }, { 14018, 0x6F97 }, { 14019, 0x764E }, { 14020, 0x770B }, { 14021, 0x78F5 },
    { 14022, 0x7A08 }, { 14023, 0x7AFF }, { 14024, 0x7C21 }, { 14025, 0x809D }, { 14026, 0x826E }, { 14027, 0x8271 }, { 14028, 0x8AEB }, { 14029, 0x9593 },
    { 14030, 0x4E6B }, { 14031, 0x559D }, { 14032, 0x66F7 }, { 14033, 0x6E34 }, { 14034, 0x78A3 }, { 14035, 0x7AED }, { 14036, 0x845B }, { 14037, 0x8910 },
    { 14038, 0x874E }, { 14039, 0x97A8 }, { 14040, 0x52D8 }, { 14041, 0x574E }, { 14042, 0x582A }, { 14043, 0x5D4C }, { 14044, 0x611F }, { 14045, 0x61BE },
    { 14046, 0x6221 }, { 14047, 0x6562 }, { 14048, 0x67D1 }, { 14049, 0x6A44 }, { 14050, 0x6E1B }, { 14051, 0x7518 }, { 14052, 0x75B3 }, { 14053, 0x76E3 },
    { 14054, 0x77B0 }, { 14055, 0x7D3A }, { 14056, 0x90AF }, { 14057, 0x9451 }, { 14058, 0x9452 }, { 14059, 0x9F95 }, { 14156, 0x5323 }, { 14157, 0x5CAC },
    { 14158, 0x7532 }, { 14159, 0x80DB }, { 14160, 0x9240 }, { 14161, 0x9598 }, { 14162, 0x525B }, { 14163, 0x5808 }, { 14164, 0x59DC }, { 14165, 0x5CA1 },
    { 14166, 0x5D17 }, { 14167, 0x5EB7 }, { 14168, 0x5F3A }, { 14169, 0x5F4A }, { 14170, 0x6177 }, { 14171, 0x6C5F }, { 14172, 0x757A }, { 14173, 0x7586 },
    { 14174, 0x7CE0 }, { 14175, 0x7D73 }, { 14176, 0x7DB1 }, { 14177, 0x7F8C }, { 14178, 0x8154 }, { 14179, 0x8221 }, { 14180, 0x8591 }, { 14181, 0x8941 },
    { 14182, 0x8B1B }, { 14183, 0x92FC }, { 14184, 0x964D }, { 14185, 0x9C47 }, { 14186, 0x4ECB }, { 14187, 0x4EF7 }, { 14188, 0x500B }, { 14189, 0x51F1 },
    { 14190, 0x584F }, { 14191, 0x6137 }, { 14192, 0x613E }, { 14193, 0x6168 }, { 14194, 0x6539 }, { 14195, 0x69EA }, { 14196, 0x6F11 }, { 14197, 0x75A5 },
    { 14198, 0x7686 }, { 14199, 0x76D6 }, { 14200, 0x7B87 }, { 14201, 0x82A5 }, { 14202, 0x84CB }, { 14203, 0xF900 }, { 14204, 0x93A7 }, { 14205, 0x958B },
    { 14206, 0x5580 }, { 14207, 0x5BA2 }, { 14208, 0x5751 }, { 14209, 0xF901 }, { 14210, 0x7CB3 }, { 14211, 0x7FB9 }, { 14212, 0x91B5 }, { 14213, 0x5028 },
    { 14214, 0x53BB }, { 14215, 0x5C45 }, { 14216, 0x5DE8 }, { 14217, 0x62D2 }, { 14218, 0x636E }, { 14219, 0x64DA }, { 14220, 0x64E7 }, { 14221, 0x6E20 },
    { 14222, 0x70AC }, { 14223, 0x795B }, { 14224, 0x8DDD }, { 14225, 0x8E1E }, { 14226, 0xF902 }, { 14227, 0x907D }, { 14228, 0x9245 }, { 14229, 0x92F8 },
    { 14230, 0x4E7E }, { 14231, 0x4EF6 }, { 14232, 0x5065 }, { 14233, 0x5DFE }, { 14234, 0x5EFA }, { 14235, 0x6106 }, { 14236, 0x6957 }, { 14237, 0x8171 },
    { 14238, 0x8654 }, { 14239, 0x8E47 }, { 14240, 0x9375 }, { 14241, 0x9A2B }, { 14242, 0x4E5E }, { 14243, 0x5091 }, { 14244, 0x6770 }, { 14245, 0x6840 },
    { 14246, 0x5109 }, { 14247, 0x528D }, { 14248, 0x5292 }, { 14249, 0x6AA2 }, { 14346, 0x77BC }, { 14347, 0x9210 }, { 14348, 0x9ED4 }, { 14349, 0x52AB },
    { 14350, 0x602F }, { 14351, 0x8FF2 }, { 14352, 0x5048 }, { 14353, 0x61A9 }, { 14354, 0x63ED }, { 14355, 0x64CA }, { 14356, 0x683C }, { 14357, 0x6A84 },
    { 14358, 0x6FC0 }, { 14359, 0x8188 }, { 14360, 0x89A1 }, { 14361, 0x9694 }, { 14362, 0x5805 }, { 14363, 0x727D }, { 14364, 0x72AC }, { 14365, 0x7504 },
    { 14366, 0x7D79 }, { 14367, 0x7E6D }, { 14368, 0x80A9 }, { 14369, 0x898B }, { 14370, 0x8B74 }, { 14371, 0x9063 }, { 14372, 0x9D51 }, { 14373, 0x6289 },
    { 14374, 0x6C7A }, { 14375, 0x6F54 }, { 14376, 0x7D50 }, { 14377, 0x7F3A }, { 14378, 0x8A23 }, { 14379, 0x517C }, { 14380, 0x614A }, { 14381, 0x7B9D },
    { 14382, 0x8B19 }, { 14383, 0x9257 }, { 14384, 0x938C }, { 14385, 0x4EAC }, { 14386, 0x4FD3 }, { 14387, 0x501E }, { 14388, 0x50BE }, { 14389, 0x5106 },
    { 14390, 0x52C1 }, { 14391, 0x52CD }, { 14392, 0x537F }, { 14393, 0x5770 }, { 14394, 0x5883 }, { 14395, 0x5E9A }, { 14396, 0x5F91 }, { 14397, 0x6176 },
    { 14398, 0x61AC }, { 14399, 0x64CE }, { 14400, 0x656C }, { 14401, 0x666F }, { 14402, 0x66BB }, { 14403, 0x66F4 }, { 14404, 0x6897 }, { 14405, 0x6D87 },
    { 14406, 0x7085 }, { 14407, 0x70F1 }, { 14408, 0x749F }, { 14409, 0x74A5 }, { 14410, 0x74CA }, { 14411, 0x75D9 }, { 14412, 0x786C }, { 14413, 0x78EC },
    { 14414, 0x7ADF }, { 14415, 0x7AF6 }, { 14416, 0x7D45 }, { 14417, 0x7D93 }, { 14418, 0x8015 }, { 14419, 0x803F }, { 14420, 0x811B }, { 14421, 0x8396 },
    { 14422, 0x8B66 }, { 14423, 0x8F15 }, { 14424, 0x9015 }, { 14425, 0x93E1 }, { 14426, 0x9803 }, { 14427, 0x9838 }, { 14428, 0x9A5A }, { 14429, 0x9BE8 },
    { 14430, 0x4FC2 }, { 14431, 0x5553 }, { 14432, 0x583A }, { 14433, 0x5951 }, { 14434, 0x5B63 }, { 14435, 0x5C46 }, { 14436, 0x60B8 }, { 14437, 0x6212 },
    { 14438, 0x6842 }, { 14439, 0x68B0 }, { 14536, 0x68E8 }, { 14537, 0x6EAA }, { 14538, 0x754C }, { 14539, 0x7678 }, { 14540, 0x78CE }, { 14541, 0x7A3D },
    { 14542, 0x7CFB }, { 14543, 0x7E6B }, { 14544, 0x7E7C }, { 14545, 0x8A08 }, { 14546, 0x8AA1 }, { 14547, 0x8C3F }, { 14548, 0x968E }, { 14549, 0x9DC4 },
    { 14550, 0x53E4 }, { 14551, 0x53E9 }, { 14552, 0x544A }, { 14553, 0x5471 }, { 14554, 0x56FA }, { 14555, 0x59D1 }, { 14556, 0x5B64 }, { 14557, 0x5C3B },
    { 14558, 0x5EAB }, { 14559, 0x62F7 }, { 14560, 0x6537 }, { 14561, 0x6545 }, { 14562, 0x6572 }, { 14563, 0x66A0 }, { 14564, 0x67AF }, { 14565, 0x69C1 },
    { 14566, 0x6CBD }, { 14567, 0x75FC }, { 14568, 0x7690 }, { 14569, 0x777E }, { 14570, 0x7A3F }, { 14571, 0x7F94 }, { 14572, 0x8003 }, { 14573, 0x80A1 },
    { 14574, 0x818F }, { 14575, 0x82E6 }, { 14576, 0x82FD }, { 14577, 0x83F0 }, { 14578, 0x85C1 }, { 14579, 0x8831 }, { 14580, 0x88B4 }, { 14581, 0x8AA5 },
    { 14582, 0xF903 }, { 14583, 0x8F9C }, { 14584, 0x932E }, { 14585, 0x96C7 }, { 14586, 0x9867 }, { 14587, 0x9AD8 }, { 14588, 0x9F13 }, { 14589, 0x54ED },
    { 14590, 0x659B }, { 14591, 0x66F2 }, { 14592, 0x688F }, { 14593, 0x7A40 }, { 14594, 0x8C37 }, { 14595, 0x9D60 }, { 14596, 0x56F0 }, { 14597, 0x5764 },
    { 14598, 0x5D11 }, { 14599, 0x6606 }, { 14600, 0x68B1 }, { 14601, 0x68CD }, { 14602, 0x6EFE }, { 14603, 0x7428 }, { 14604, 0x889E }, { 14605, 0x9BE4 },
    { 14606, 0x6C68 }, { 14607, 0xF904 }, { 14608, 0x9AA8 }, { 14609, 0x4F9B }, { 14610, 0x516C }, { 14611, 0x5171 }, { 14612, 0x529F }, { 14613, 0x5B54 },
    { 14614, 0x5DE5 }, { 14615, 0x6050 }, { 14616, 0x606D }, { 14617, 0x62F1 }, { 14618, 0x63A7 }, { 14619, 0x653B }, { 14620, 0x73D9 }, { 14621, 0x7A7A },
    { 14622, 0x86A3 }, { 14623, 0x8CA2 }, { 14624, 0x978F }, { 14625, 0x4E32 }, { 14626, 0x5BE1 }, { 14627, 0x6208 }, { 14628, 0x679C }, { 14629, 0x74DC },
    { 14726, 0x79D1 }, { 14727, 0x83D3 }, { 14728, 0x8A87 }, { 14729, 0x8AB2 }, { 14730, 0x8DE8 }, { 14731, 0x904E }, { 14732, 0x934B }, { 14733, 0x9846 },
    { 14734, 0x5ED3 }, { 14735, 0x69E8 }, { 14736, 0x85FF }, { 14737, 0x90ED }, { 14738, 0xF905 }, { 14739, 0x51A0 }, { 14740, 0x5B98 }, { 14741, 0x5BEC },
    { 14742, 0x6163 }, { 14743, 0x68FA }, { 14744, 0x6B3E }, { 14745, 0x704C }, { 14746, 0x742F }, { 14747, 0x74D8 }, { 14748, 0x7BA1 }, { 14749, 0x7F50 },
    { 14750, 0x83C5 }, { 14751, 0x89C0 }, { 14752, 0x8CAB }, { 14753, 0x95DC }, { 14754, 0x9928 }, { 14755, 0x522E }, { 14756, 0x605D }, { 14757, 0x62EC },
    { 14758, 0x9002 }, { 14759, 0x4F8A }, { 14760, 0x5149 }, { 14761, 0x5321 }, { 14762, 0x58D9 }, { 14763, 0x5EE3 }, { 14764, 0x66E0 }, { 14765, 0x6D38 },
    { 14766, 0x709A }, { 14767, 0x72C2 }, { 14768, 0x73D6 }, { 14769, 0x7B50 }, { 14770, 0x80F1 }, { 14771, 0x945B }, { 14772, 0x5366 }, { 14773, 0x639B },
    { 14774, 0x7F6B }, { 14775, 0x4E56 }, { 14776, 0x5080 }, { 14777, 0x584A }, { 14778, 0x58DE }, { 14779, 0x602A }, { 14780, 0x6127 }, { 14781, 0x62D0 },
    { 14782, 0x69D0 }, { 14783, 0x9B41 }, { 14784, 0x5B8F }, { 14785, 0x7D18 }, { 14786, 0x80B1 }, { 14787, 0x8F5F }, { 14788, 0x4EA4 }, { 14789, 0x50D1 },
    { 14790, 0x54AC }, { 14791, 0x55AC }, { 14792, 0x5B0C }, { 14793, 0x5DA0 }, { 14794, 0x5DE7 }, { 14795, 0x652A }, { 14796, 0x654E }, { 14797, 0x6821 },
    { 14798, 0x6A4B }, { 14799, 0x72E1 }, { 14800, 0x768E }, { 14801, 0x77EF }, { 14802, 0x7D5E }, { 14803, 0x7FF9 }, { 14804, 0x81A0 }, { 14805, 0x854E },
    { 14806, 0x86DF }, { 14807, 0x8F03 }, { 14808, 0x8F4E }, { 14809, 0x90CA }, { 14810, 0x9903 }, { 14811, 0x9A55 }, { 14812, 0x9BAB }, { 14813, 0x4E18 },
    { 14814, 0x4E45 }, { 14815, 0x4E5D }, { 14816, 0x4EC7 }, { 14817, 0x4FF1 }, { 14818, 0x5177 }, { 14819, 0x52FE }, { 14916, 0x5340 }, { 14917, 0x53E3 },
    { 14918, 0x53E5 }, { 14919, 0x548E }, { 14920, 0x5614 }, { 14921, 0x5775 }, { 14922, 0x57A2 }, { 14923, 0x5BC7 }, { 14924, 0x5D87 }, { 14925, 0x5ED0 },
    { 14926, 0x61FC }, { 14927, 0x62D8 }, { 14928, 0x6551 }, { 14929, 0x67B8 }, { 14930, 0x67E9 }, { 14931, 0x69CB }, { 14932, 0x6B50 }, { 14933, 0x6BC6 },
    { 14934, 0x6BEC }, { 14935, 0x6C42 }, { 14936, 0x6E9D }, { 14937, 0x7078 }, { 14938, 0x72D7 }, { 14939, 0x7396 }, { 14940, 0x7403 }, { 14941, 0x77BF },
    { 14942, 0x77E9 }, { 14943, 0x7A76 }, { 14944, 0x7D7F }, { 14945, 0x8009 }, { 14946, 0x81FC }, { 14947, 0x8205 }, { 14948, 0x820A }, { 14949, 0x82DF },
    { 14950, 0x8862 }, { 14951, 0x8B33 }, { 14952, 0x8CFC }, { 14953, 0x8EC0 }, { 14954, 0x9011 }, { 14955, 0x90B1 }, { 14956, 0x9264 }, { 14957, 0x92B6 },
    { 14958, 0x99D2 }, { 14959, 0x9A45 }, { 14960, 0x9CE9 }, { 14961, 0x9DD7 }, { 14962, 0x9F9C }, { 14963, 0x570B }, { 14964, 0x5C40 }, { 14965, 0x83CA },
    { 14966, 0x97A0 }, { 14967, 0x97AB }, { 14968, 0x9EB4 }, { 14969, 0x541B }, { 14970, 0x7A98 }, { 14971, 0x7FA4 }, { 14972, 0x88D9 }, { 14973, 0x8ECD },
    { 14974, 0x90E1 }, { 14975, 0x5800 }, { 14976, 0x5C48 }, { 14977, 0x6398 }, { 14978, 0x7A9F }, { 14979, 0x5BAE }, { 14980, 0x5F13 }, { 14981, 0x7A79 },
    { 14982, 0x7AAE }, { 14983, 0x828E }, { 14984, 0x8EAC }, { 14985, 0x5026 }, { 14986, 0x5238 }, { 14987, 0x52F8 }, { 14988, 0x5377 }, { 14989, 0x5708 },
    { 14990, 0x62F3 }, { 14991, 0x6372 }, { 14992, 0x6B0A }, { 14993, 0x6DC3 }, { 14994, 0x7737 }, { 14995, 0x53A5 }, { 14996, 0x7357 }, { 14997, 0x8568 },
    { 14998, 0x8E76 }, { 14999, 0x95D5 }, { 15000, 0x673A }, { 15001, 0x6AC3 }, { 15002, 0x6F70 }, { 15003, 0x8A6D }, { 15004, 0x8ECC }, { 15005, 0x994B },
    { 15006, 0xF906 }, { 15007, 0x6677 }, { 15008, 0x6B78 }, { 15009, 0x8CB4 }, { 15106, 0x9B3C }, { 15107, 0xF907 }, { 15108, 0x53EB }, { 15109, 0x572D },
    { 15110, 0x594E }, { 15111, 0x63C6 }, { 15112, 0x69FB }, { 15113, 0x73EA }, { 15114, 0x7845 }, { 15115, 0x7ABA }, { 15116, 0x7AC5 }, { 15117, 0x7CFE },
    { 15118, 0x8475 }, { 15119, 0x898F }, { 15120, 0x8D73 }, { 15121, 0x9035 }, { 15122, 0x95A8 }, { 15123, 0x52FB }, { 15124, 0x5747 }, { 15125, 0x7547 },
    { 15126, 0x7B60 }, { 15127, 0x83CC }, { 15128, 0x921E }, { 15129, 0xF908 }, { 15130, 0x6A58 }, { 15131, 0x514B }, { 15132, 0x524B }, { 15133, 0x5287 },
    { 15134, 0x621F }, { 15135, 0x68D8 }, { 15136, 0x6975 }, { 15137, 0x9699 }, { 15138, 0x50C5 }, { 15139, 0x52A4 }, { 15140, 0x52E4 }, { 15141, 0x61C3 },
    { 15142, 0x65A4 }, { 15143, 0x6839 }, { 15144, 0x69FF }, { 15145, 0x747E }, { 15146, 0x7B4B }, { 15147, 0x82B9 }, { 15148, 0x83EB }, { 15149, 0x89B2 },
    { 15150, 0x8B39 }, { 15151, 0x8FD1 }, { 15152, 0x9949 }, { 15153, 0xF909 }, { 15154, 0x4ECA }, { 15155, 0x5997 }, { 15156, 0x64D2 }, { 15157, 0x6611 },
    { 15158, 0x6A8E }, { 15159, 0x7434 }, { 15160, 0x7981 }, { 15161, 0x79BD }, { 15162, 0x82A9 }, { 15163, 0x887E }, { 15164, 0x887F }, { 15165, 0x895F },
    { 15166, 0xF90A }, { 15167, 0x9326 }, { 15168, 0x4F0B }, { 15169, 0x53CA }, { 15170, 0x6025 }, { 15171, 0x6271 }, { 15172, 0x6C72 }, { 15173, 0x7D1A },
    { 15174, 0x7D66 }, { 15175, 0x4E98 }, { 15176, 0x5162 }, { 15177, 0x77DC }, { 15178, 0x80AF }, { 15179, 0x4F01 }, { 15180, 0x4F0E }, { 15181, 0x5176 },
    { 15182, 0x5180 }, { 15183, 0x55DC }, { 15184, 0x5668 }, { 15185, 0x573B }, { 15186, 0x57FA }, { 15187, 0x57FC }, { 15188, 0x5914 }, { 15189, 0x5947 },
    { 15190, 0x5993 }, { 15191, 0x5BC4 }, { 15192, 0x5C90 }, { 15193, 0x5D0E }, { 15194, 0x5DF1 }, { 15195, 0x5E7E }, { 15196, 0x5FCC }, { 15197, 0x6280 },
    { 15198, 0x65D7 }, { 15199, 0x65E3 }, { 15296, 0x671E }, { 15297, 0x671F }, { 15298, 0x675E }, { 15299, 0x68CB }, { 15300, 0x68C4 }, { 15301, 0x6A5F },
    { 15302, 0x6B3A }, { 15303, 0x6C23 }, { 15304, 0x6C7D }, { 15305, 0x6C82 }, { 15306, 0x6DC7 }, { 15307, 0x7398 }, { 15308, 0x7426 }, { 15309, 0x742A },
    { 15310, 0x7482 }, { 15311, 0x74A3 }, { 15312, 0x7578 }, { 15313, 0x757F }, { 15314, 0x7881 }, { 15315, 0x78EF }, { 15316, 0x7941 }, { 15317, 0x7947 },
    { 15318, 0x7948 }, { 15319, 0x797A }, { 15320, 0x7B95 }, { 15321, 0x7D00 }, { 15322, 0x7DBA }, { 15323, 0x7F88 }, { 15324, 0x8006 }, { 15325, 0x802D },
    { 15326, 0x808C }, { 15327, 0x8A18 }, { 15328, 0x8B4F }, { 15329, 0x8C48 }, { 15330, 0x8D77 }, { 15331, 0x9321 }, { 15332, 0x9324 }, { 15333, 0x98E2 },
    { 15334, 0x9951 }, { 15335, 0x9A0E }, { 15336, 0x9A0F }, { 15337, 0x9A65 }, { 15338, 0x9E92 }, { 15339, 0x7DCA }, { 15340, 0x4F76 }, { 15341, 0x5409 },
    { 15342, 0x62EE }, { 15343, 0x6854 }, { 15344, 0x91D1 }, { 15345, 0x55AB }, { 15346, 0x513A }, { 15347, 0xF90B }, { 15348, 0xF90C }, { 15349, 0x5A1C },
    { 15350, 0x61E6 }, { 15351, 0xF90D }, { 15352, 0x62CF }, { 15353, 0x62FF }, { 15354, 0xF90E }, { 15355, 0xF90F }, { 15356, 0xF910 }, { 15357, 0xF911 },
    { 15358, 0xF912 }, { 15359, 0xF913 }, { 15360, 0x90A3 }, { 15361, 0xF914 }, { 15362, 0xF915 }, { 15363, 0xF916 }, { 15364, 0xF917 }, { 15365, 0xF918 },
    { 15366, 0x8AFE }, { 15367, 0xF919 }, { 15368, 0xF91A }, { 15369, 0xF91B }, { 15370, 0xF91C }, { 15371, 0x6696 }, { 15372, 0xF91D }, { 15373, 0x7156 },
    { 15374, 0xF91E }, { 15375, 0xF91F }, { 15376, 0x96E3 }, { 15377, 0xF920 }, { 15378, 0x634F }, { 15379, 0x637A }, { 15380, 0x5357 }, { 15381, 0xF921 },
    { 15382, 0x678F }, { 15383, 0x6960 }, { 15384, 0x6E73 }, { 15385, 0xF922 }, { 15386, 0x7537 }, { 15387, 0xF923 }, { 15388, 0xF924 }, { 15389, 0xF925 },
    { 15486, 0x7D0D }, { 15487, 0xF926 }, { 15488, 0xF927 }, { 15489, 0x8872 }, { 15490, 0x56CA }, { 15491, 0x5A18 }, { 15492, 0xF928 }, { 15493, 0xF929 },
    { 15494, 0xF92A }, { 15495, 0xF92B }, { 15496, 0xF92C }, { 15497, 0x4E43 }, { 15498, 0xF92D }, { 15499, 0x5167 }, { 15500, 0x5948 }, { 15501, 0x67F0 },
    { 15502, 0x8010 }, { 15503, 0xF92E }, { 15504, 0x5973 }, { 15505, 0x5E74 }, { 15506, 0x649A }, { 15507, 0x79CA }, { 15508, 0x5FF5 }, { 15509, 0x606C },
    { 15510, 0x62C8 }, { 15511, 0x637B }, { 15512, 0x5BE7 }, { 15513, 0x5BD7 }, { 15514, 0x52AA }, { 15515, 0xF92F }, { 15516, 0x5974 }, { 15517, 0x5F29 },
    { 15518, 0x6012 }, { 15519, 0xF930 }, { 15520, 0xF931 }, { 15521, 0xF932 }, { 15522, 0x7459 }, { 15523, 0xF933 }, { 15524, 0xF934 }, { 15525, 0xF935 },
    { 15526, 0xF936 }, { 15527, 0xF937 }, { 15528, 0xF938 }, { 15529, 0x99D1 }, { 15530, 0xF939 }, { 15531, 0xF93A }, { 15532, 0xF93B }, { 15533, 0xF93C },
    { 15534, 0xF93D }, { 15535, 0xF93E }, { 15536, 0xF93F }, { 15537, 0xF940 }, { 15538, 0xF941 }, { 15539, 0xF942 }, { 15540, 0xF943 }, { 15541, 0x6FC3 },
    { 15542, 0xF944 }, { 15543, 0xF945 }, { 15544, 0x81BF }, { 15545, 0x8FB2 }, { 15546, 0x60F1 }, { 15547, 0xF946 }, { 15548, 0xF947 }, { 15549, 0x8166 },
    { 15550, 0xF948 }, { 15551, 0xF949 }, { 15552, 0x5C3F }, { 15553, 0xF94A }, { 15554, 0xF94B }, { 15555, 0xF94C }, { 15556, 0xF94D }, { 15557, 0xF94E },
    { 15558, 0xF94F }, { 15559, 0xF950 }, { 15560, 0xF951 }, { 15561, 0x5AE9 }, { 15562, 0x8A25 }, { 15563, 0x677B }, { 15564, 0x7D10 }, { 15565, 0xF952 },
    { 15566, 0xF953 }, { 15567, 0xF954 }, { 15568, 0xF955 }, { 15569, 0xF956 }, { 15570, 0xF957 }, { 15571, 0x80FD }, { 15572, 0xF958 }, { 15573, 0xF959 },
    { 15574, 0x5C3C }, { 15575, 0x6CE5 }, { 15576, 0x533F }, { 15577, 0x6EBA }, { 15578, 0x591A }, { 15579, 0x8336 }, { 15676, 0x4E39 }, { 15677, 0x4EB6 },
    { 15678, 0x4F46 }, { 15679, 0x55AE }, { 15680, 0x5718 }, { 15681, 0x58C7 }, { 15682, 0x5F56 }, { 15683, 0x65B7 }, { 15684, 0x65E6 }, { 15685, 0x6A80 },
    { 15686, 0x6BB5 }, { 15687, 0x6E4D }, { 15688, 0x77ED }, { 15689, 0x7AEF }, { 15690, 0x7C1E }, { 15691, 0x7DDE }, { 15692, 0x86CB }, { 15693, 0x8892 },
    { 15694, 0x9132 }, { 15695, 0x935B }, { 15696, 0x64BB }, { 15697, 0x6FBE }, { 15698, 0x737A }, { 15699, 0x75B8 }, { 15700, 0x9054 }, { 15701, 0x5556 },
    { 15702, 0x574D }, { 15703, 0x61BA }, { 15704, 0x64D4 }, { 15705, 0x66C7 }, { 15706, 0x6DE1 }, { 15707, 0x6E5B }, { 15708, 0x6F6D }, { 15709, 0x6FB9 },
    { 15710, 0x75F0 }, { 15711, 0x8043 }, { 15712, 0x81BD }, { 15713, 0x8541 }, { 15714, 0x8983 }, { 15715, 0x8AC7 }, { 15716, 0x8B5A }, { 15717, 0x931F },
    { 15718, 0x6C93 }, { 15719, 0x7553 }, { 15720, 0x7B54 }, { 15721, 0x8E0F }, { 15722, 0x905D }, { 15723, 0x5510 }, { 15724, 0x5802 }, { 15725, 0x5858 },
    { 15726, 0x5E62 }, { 15727, 0x6207 }, { 15728, 0x649E }, { 15729, 0x68E0 }, { 15730, 0x7576 }, { 15731, 0x7CD6 }, { 15732, 0x87B3 }, { 15733, 0x9EE8 },
    { 15734, 0x4EE3 }, { 15735, 0x5788 }, { 15736, 0x576E }, { 15737, 0x5927 }, { 15738, 0x5C0D }, { 15739, 0x5CB1 }, { 15740, 0x5E36 }, { 15741, 0x5F85 },
    { 15742, 0x6234 }, { 15743, 0x64E1 }, { 15744, 0x73B3 }, { 15745, 0x81FA }, { 15746, 0x888B }, { 15747, 0x8CB8 }, { 15748, 0x968A }, { 15749, 0x9EDB },
    { 15750, 0x5B85 }, { 15751, 0x5FB7 }, { 15752, 0x60B3 }, { 15753, 0x5012 }, { 15754, 0x5200 }, { 15755, 0x5230 }, { 15756, 0x5716 }, { 15757, 0x5835 },
    { 15758, 0x5857 }, { 15759, 0x5C0E }, { 15760, 0x5C60 }, { 15761, 0x5CF6 }, { 15762, 0x5D8B }, { 15763, 0x5EA6 }, { 15764, 0x5F92 }, { 15765, 0x60BC },
    { 15766, 0x6311 }, { 15767, 0x6389 }, { 15768, 0x6417 }, { 15769, 0x6843 }, { 15866, 0x68F9 }, { 15867, 0x6AC2 }, { 15868, 0x6DD8 }, { 15869, 0x6E21 },
    { 15870, 0x6ED4 }, { 15871, 0x6FE4 }, { 15872, 0x71FE }, { 15873, 0x76DC }, { 15874, 0x7779 }, { 15875, 0x79B1 }, { 15876, 0x7A3B }, { 15877, 0x8404 },
    { 15878, 0x89A9 }, { 15879, 0x8CED }, { 15880, 0x8DF3 }, { 15881, 0x8E48 }, { 15882, 0x9003 }, { 15883, 0x9014 }, { 15884, 0x9053 }, { 15885, 0x90FD },
    { 15886, 0x934D }, { 15887, 0x9676 }, { 15888, 0x97DC }, { 15889, 0x6BD2 }, { 15890, 0x7006 }, { 15891, 0x7258 }, { 15892, 0x72A2 }, { 15893, 0x7368 },
    { 15894, 0x7763 }, { 15895, 0x79BF }, { 15896, 0x7BE4 }, { 15897, 0x7E9B }, { 15898, 0x8B80 }, { 15899, 0x58A9 }, { 15900, 0x60C7 }, { 15901, 0x6566 },
    { 15902, 0x65FD }, { 15903, 0x66BE }, { 15904, 0x6C8C }, { 15905, 0x711E }, { 15906, 0x71C9 }, { 15907, 0x8C5A }, { 15908, 0x9813 }, { 15909, 0x4E6D },
    { 15910, 0x7A81 }, { 15911, 0x4EDD }, { 15912, 0x51AC }, { 15913, 0x51CD }, { 15914, 0x52D5 }, { 15915, 0x540C }, { 15916, 0x61A7 }, { 15917, 0x6771 },
    { 15918, 0x6850 }, { 15919, 0x68DF }, { 15920, 0x6D1E }, { 15921, 0x6F7C }, { 15922, 0x75BC }, { 15923, 0x77B3 }, { 15924, 0x7AE5 }, { 15925, 0x80F4 },
    { 15926, 0x8463 }, { 15927, 0x9285 }, { 15928, 0x515C }, { 15929, 0x6597 }, { 15930, 0x675C }, { 15931, 0x6793 }, { 15932, 0x75D8 }, { 15933, 0x7AC7 },
    { 15934, 0x8373 }, { 15935, 0xF95A }, { 15936, 0x8C46 }, { 15937, 0x9017 }, { 15938, 0x982D }, { 15939, 0x5C6F }, { 15940, 0x81C0 }, { 15941, 0x829A },
    { 15942, 0x9041 }, { 15943, 0x906F }, { 15944, 0x920D }, { 15945, 0x5F97 }, { 15946, 0x5D9D }, { 15947, 0x6A59 }, { 15948, 0x71C8 }, { 15949, 0x767B },
    { 15950, 0x7B49 }, { 15951, 0x85E4 }, { 15952, 0x8B04 }, { 15953, 0x9127 }, { 15954, 0x9A30 }, { 15955, 0x5587 }, { 15956, 0x61F6 }, { 15957, 0xF95B },
    { 15958, 0x7669 }, { 15959, 0x7F85 }, { 16056, 0x863F }, { 16057, 0x87BA }, { 16058, 0x88F8 }, { 16059, 0x908F }, { 16060, 0xF95C }, { 16061, 0x6D1B },
    { 16062, 0x70D9 }, { 16063, 0x73DE }, { 16064, 0x7D61 }, { 16065, 0x843D }, { 16066, 0xF95D }, { 16067, 0x916A }, { 16068, 0x99F1 }, { 16069, 0xF95E },
    { 16070, 0x4E82 }, { 16071, 0x5375 }, { 16072, 0x6B04 }, { 16073, 0x6B12 }, { 16074, 0x703E }, { 16075, 0x721B }, { 16076, 0x862D }, { 16077, 0x9E1E },
    { 16078, 0x524C }, { 16079, 0x8FA3 }, { 16080, 0x5D50 }, { 16081, 0x64E5 }, { 16082, 0x652C }, { 16083, 0x6B16 }, { 16084, 0x6FEB }, { 16085, 0x7C43 },
    { 16086, 0x7E9C }, { 16087, 0x85CD }, { 16088, 0x8964 }, { 16089, 0x89BD }, { 16090, 0x62C9 }, { 16091, 0x81D8 }, { 16092, 0x881F }, { 16093, 0x5ECA },
    { 16094, 0x6717 }, { 16095, 0x6D6A }, { 16096, 0x72FC }, { 16097, 0x7405 }, { 16098, 0x746F }, { 16099, 0x8782 }, { 16100, 0x90DE }, { 16101, 0x4F86 },
    { 16102, 0x5D0D }, { 16103, 0x5FA0 }, { 16104, 0x840A }, { 16105, 0x51B7 }, { 16106, 0x63A0 }, { 16107, 0x7565 }, { 16108, 0x4EAE }, { 16109, 0x5006 },
    { 16110, 0x5169 }, { 16111, 0x51C9 }, { 16112, 0x6881 }, { 16113, 0x6A11 }, { 16114, 0x7CAE }, { 16115, 0x7CB1 }, { 16116, 0x7CE7 }, { 16117, 0x826F },
    { 16118, 0x8AD2 }, { 16119, 0x8F1B }, { 16120, 0x91CF }, { 16121, 0x4FB6 }, { 16122, 0x5137 }, { 16123, 0x52F5 }, { 16124, 0x5442 }, { 16125, 0x5EEC },
    { 16126, 0x616E }, { 16127, 0x623E }, { 16128, 0x65C5 }, { 16129, 0x6ADA }, { 16130, 0x6FFE }, { 16131, 0x792A }, { 16132, 0x85DC }, { 16133, 0x8823 },
    { 16134, 0x95AD }, { 16135, 0x9A62 }, { 16136, 0x9A6A }, { 16137, 0x9E97 }, { 16138, 0x9ECE }, { 16139, 0x529B }, { 16140, 0x66C6 }, { 16141, 0x6B77 },
    { 16142, 0x701D }, { 16143, 0x792B }, { 16144, 0x8F62 }, { 16145, 0x9742 }, { 16146, 0x6190 }, { 16147, 0x6200 }, { 16148, 0x6523 }, { 16149, 0x6F23 },
    { 16246, 0x7149 }, { 16247, 0x7489 }, { 16248, 0x7DF4 }, { 16249, 0x806F }, { 16250, 0x84EE }, { 16251, 0x8F26 }, { 16252, 0x9023 }, { 16253, 0x934A },
    { 16254, 0x51BD }, { 16255, 0x5217 }, { 16256, 0x52A3 }, { 16257, 0x6D0C }, { 16258, 0x70C8 }, { 16259, 0x88C2 }, { 16260, 0x5EC9 }, { 16261, 0x6582 },
    { 16262, 0x6BAE }, { 16263, 0x6FC2 }, { 16264, 0x7C3E }, { 16265, 0x7375 }, { 16266, 0x4EE4 }, { 16267, 0x4F36 }, { 16268, 0x56F9 }, { 16269, 0xF95F },
    { 16270, 0x5CBA }, { 16271, 0x5DBA }, { 16272, 0x601C }, { 16273, 0x73B2 }, { 16274, 0x7B2D }, { 16275, 0x7F9A }, { 16276, 0x7FCE }, { 16277, 0x8046 },
    { 16278, 0x901E }, { 16279, 0x9234 }, { 16280, 0x96F6 }, { 16281, 0x9748 }, { 16282, 0x9818 }, { 16283, 0x9F61 }, { 16284, 0x4F8B }, { 16285, 0x6FA7 },
    { 16286, 0x79AE }, { 16287, 0x91B4 }, { 16288, 0x96B7 }, { 16289, 0x52DE }, { 16290, 0xF960 }, { 16291, 0x6488 }, { 16292, 0x64C4 }, { 16293, 0x6AD3 },
    { 16294, 0x6F5E }, { 16295, 0x7018 }, { 16296, 0x7210 }, { 16297, 0x76E7 }, { 16298, 0x8001 }, { 16299, 0x8606 }, { 16300, 0x865C }, { 16301, 0x8DEF },
    { 16302, 0x8F05 }, { 16303, 0x9732 }, { 16304, 0x9B6F }, { 16305, 0x9DFA }, { 16306, 0x9E75 }, { 16307, 0x788C }, { 16308, 0x797F }, { 16309, 0x7DA0 },
    { 16310, 0x83C9 }, { 16311, 0x9304 }, { 16312, 0x9E7F }, { 16313, 0x9E93 }, { 16314, 0x8AD6 }, { 16315, 0x58DF }, { 16316, 0x5F04 }, { 16317, 0x6727 },
    { 16318, 0x7027 }, { 16319, 0x74CF }, { 16320, 0x7C60 }, { 16321, 0x807E }, { 16322, 0x5121 }, { 16323, 0x7028 }, { 16324, 0x7262 }, { 16325, 0x78CA },
    { 16326, 0x8CC2 }, { 16327, 0x8CDA }, { 16328, 0x8CF4 }, { 16329, 0x96F7 }, { 16330, 0x4E86 }, { 16331, 0x50DA }, { 16332, 0x5BEE }, { 16333, 0x5ED6 },
    { 16334, 0x6599 }, { 16335, 0x71CE }, { 16336, 0x7642 }, { 16337, 0x77AD }, { 16338, 0x804A }, { 16339, 0x84FC }, { 16436, 0x907C }, { 16437, 0x9B27 },
    { 16438, 0x9F8D }, { 16439, 0x58D8 }, { 16440, 0x5A41 }, { 16441, 0x5C62 }, { 16442, 0x6A13 }, { 16443, 0x6DDA }, { 16444, 0x6F0F }, { 16445, 0x763B },
    { 16446, 0x7D2F }, { 16447, 0x7E37 }, { 16448, 0x851E }, { 16449, 0x8938 }, { 16450, 0x93E4 }, { 16451, 0x964B }, { 16452, 0x5289 }, { 16453, 0x65D2 },
    { 16454, 0x67F3 }, { 16455, 0x69B4 }, { 16456, 0x6D41 }, { 16457, 0x6E9C }, { 16458, 0x700F }, { 16459, 0x7409 }, { 16460, 0x7460 }, { 16461, 0x7559 },
    { 16462, 0x7624 }, { 16463, 0x786B }, { 16464, 0x8B2C }, { 16465, 0x985E }, { 16466, 0x516D }, { 16467, 0x622E }, { 16468, 0x9678 }, { 16469, 0x4F96 },
    { 16470, 0x502B }, { 16471, 0x5D19 }, { 16472, 0x6DEA }, { 16473, 0x7DB8 }, { 16474, 0x8F2A }, { 16475, 0x5F8B }, { 16476, 0x6144 }, { 16477, 0x6817 },
    { 16478, 0xF961 }, { 16479, 0x9686 }, { 16480, 0x52D2 }, { 16481, 0x808B }, { 16482, 0x51DC }, { 16483, 0x51CC }, { 16484, 0x695E }, { 16485, 0x7A1C },
    { 16486, 0x7DBE }, { 16487, 0x83F1 }, { 16488, 0x9675 }, { 16489, 0x4FDA }, { 16490, 0x5229 }, { 16491, 0x5398 }, { 16492, 0x540F }, { 16493, 0x550E },
    { 16494, 0x5C65 }, { 16495, 0x60A7 }, { 16496, 0x674E }, { 16497, 0x68A8 }, { 16498, 0x6D6C }, { 16499, 0x7281 }, { 16500, 0x72F8 }, { 16501, 0x7406 },
    { 16502, 0x7483 }, { 16503, 0xF962 }, { 16504, 0x75E2 }, { 16505, 0x7C6C }, { 16506, 0x7F79 }, { 16507, 0x7FB8 }, { 16508, 0x8389 }, { 16509, 0x88CF },
    { 16510, 0x88E1 }, { 16511, 0x91CC }, { 16512, 0x91D0 }, { 16513, 0x96E2 }, { 16514, 0x9BC9 }, { 16515, 0x541D }, { 16516, 0x6F7E }, { 16517, 0x71D0 },
    { 16518, 0x7498 }, { 16519, 0x85FA }, { 16520, 0x8EAA }, { 16521, 0x96A3 }, { 16522, 0x9C57 }, { 16523, 0x9E9F }, { 16524, 0x6797 }, { 16525, 0x6DCB },
    { 16526, 0x7433 }, { 16527, 0x81E8 }, { 16528, 0x9716 }, { 16529, 0x782C }, { 16626, 0x7ACB }, { 16627, 0x7B20 }, { 16628, 0x7C92 }, { 16629, 0x6469 },
    { 16630, 0x746A }, { 16631, 0x75F2 }, { 16632, 0x78BC }, { 16633, 0x78E8 }, { 16634, 0x99AC }, { 16635, 0x9B54 }, { 16636, 0x9EBB }, { 16637, 0x5BDE },
    { 16638, 0x5E55 }, { 16639, 0x6F20 }, { 16640, 0x819C }, { 16641, 0x83AB }, { 16642, 0x9088 }, { 16643, 0x4E07 }, { 16644, 0x534D }, { 16645, 0x5A29 },
    { 16646, 0x5DD2 }, { 16647, 0x5F4E }, { 16648, 0x6162 }, { 16649, 0x633D }, { 16650, 0x6669 }, { 16651, 0x66FC }, { 16652, 0x6EFF }, { 16653, 0x6F2B },
    { 16654, 0x7063 }, { 16655, 0x779E }, { 16656, 0x842C }, { 16657, 0x8513 }, { 16658, 0x883B }, { 16659, 0x8F13 }, { 16660, 0x9945 }, { 16661, 0x9C3B },
    { 16662, 0x551C }, { 16663, 0x62B9 }, { 16664, 0x672B }, { 16665, 0x6CAB }, { 16666, 0x8309 }, { 16667, 0x896A }, { 16668, 0x977A }, { 16669, 0x4EA1 },
    { 16670, 0x5984 }, { 16671, 0x5FD8 }, { 16672, 0x5FD9 }, { 16673, 0x671B }, { 16674, 0x7DB2 }, { 16675, 0x7F54 }, { 16676, 0x8292 }, { 16677, 0x832B },
    { 16678, 0x83BD }, { 16679, 0x8F1E }, { 16680, 0x9099 }, { 16681, 0x57CB }, { 16682, 0x59B9 }, { 16683, 0x5A92 }, { 16684, 0x5BD0 }, { 16685, 0x6627 },
    { 16686, 0x679A }, { 16687, 0x6885 }, { 16688, 0x6BCF }, { 16689, 0x7164 }, { 16690, 0x7F75 }, { 16691, 0x8CB7 }, { 16692, 0x8CE3 }, { 16693, 0x9081 },
    { 16694, 0x9B45 }, { 16695, 0x8108 }, { 16696, 0x8C8A }, { 16697, 0x964C }, { 16698, 0x9A40 }, { 16699, 0x9EA5 }, { 16700, 0x5B5F }, { 16701, 0x6C13 },
    { 16702, 0x731B }, { 16703, 0x76F2 }, { 16704, 0x76DF }, { 16705, 0x840C }, { 16706, 0x51AA }, { 16707, 0x8993 }, { 16708, 0x514D }, { 16709, 0x5195 },
    { 16710, 0x52C9 }, { 16711, 0x68C9 }, { 16712, 0x6C94 }, { 16713, 0x7704 }, { 16714, 0x7720 }, { 16715, 0x7DBF }, { 16716, 0x7DEC }, { 16717, 0x9762 },
    { 16718, 0x9EB5 }, { 16719, 0x6EC5 }, { 16816, 0x8511 }, { 16817, 0x51A5 }, { 16818, 0x540D }, { 16819, 0x547D }, { 16820, 0x660E }, { 16821, 0x669D },
    { 16822, 0x6927 }, { 16823, 0x6E9F }, { 16824, 0x76BF }, { 16825, 0x7791 }, { 16826, 0x8317 }, { 16827, 0x84C2 }, { 16828, 0x879F }, { 16829, 0x9169 },
    { 16830, 0x9298 }, { 16831, 0x9CF4 }, { 16832, 0x8882 }, { 16833, 0x4FAE }, { 16834, 0x5192 }, { 16835, 0x52DF }, { 16836, 0x59C6 }, { 16837, 0x5E3D },
    { 16838, 0x6155 }, { 16839, 0x6478 }, { 16840, 0x6479 }, { 16841, 0x66AE }, { 16842, 0x67D0 }, { 16843, 0x6A21 }, { 16844, 0x6BCD }, { 16845, 0x6BDB },
    { 16846, 0x725F }, { 16847, 0x7261 }, { 16848, 0x7441 }, { 16849, 0x7738 }, { 16850, 0x77DB }, { 16851, 0x8017 }, { 16852, 0x82BC }, { 16853, 0x8305 },
    { 16854, 0x8B00 }, { 16855, 0x8B28 }, { 16856, 0x8C8C }, { 16857, 0x6728 }, { 16858, 0x6C90 }, { 16859, 0x7267 }, { 16860, 0x76EE }, { 16861, 0x7766 },
    { 16862, 0x7A46 }, { 16863, 0x9DA9 }, { 16864, 0x6B7F }, { 16865, 0x6C92 }, { 16866, 0x5922 }, { 16867, 0x6726 }, { 16868, 0x8499 }, { 16869, 0x536F },
    { 16870, 0x5893 }, { 16871, 0x5999 }, { 16872, 0x5EDF }, { 16873, 0x63CF }, { 16874, 0x6634 }, { 16875, 0x6773 }, { 16876, 0x6E3A }, { 16877, 0x732B },
    { 16878, 0x7AD7 }, { 16879, 0x82D7 }, { 16880, 0x9328 }, { 16881, 0x52D9 }, { 16882, 0x5DEB }, { 16883, 0x61AE }, { 16884, 0x61CB }, { 16885, 0x620A },
    { 16886, 0x62C7 }, { 16887, 0x64AB }, { 16888, 0x65E0 }, { 16889, 0x6959 }, { 16890, 0x6B66 }, { 16891, 0x6BCB }, { 16892, 0x7121 }, { 16893, 0x73F7 },
    { 16894, 0x755D }, { 16895, 0x7E46 }, { 16896, 0x821E }, { 16897, 0x8302 }, { 16898, 0x856A }, { 16899, 0x8AA3 }, { 16900, 0x8CBF }, { 16901, 0x9727 },
    { 16902, 0x9D61 }, { 16903, 0x58A8 }, { 16904, 0x9ED8 }, { 16905, 0x5011 }, { 16906, 0x520E }, { 16907, 0x543B }, { 16908, 0x554F }, { 16909, 0x6587 },
    { 17006, 0x6C76 }, { 17007, 0x7D0A }, { 17008, 0x7D0B }, { 17009, 0x805E }, { 17010, 0x868A }, { 17011, 0x9580 }, { 17012, 0x96EF }, { 17013, 0x52FF },
    { 17014, 0x6C95 }, { 17015, 0x7269 }, { 17016, 0x5473 }, { 17017, 0x5A9A }, { 17018, 0x5C3E }, { 17019, 0x5D4B }, { 17020, 0x5F4C }, { 17021, 0x5FAE },
    { 17022, 0x672A }, { 17023, 0x68B6 }, { 17024, 0x6963 }, { 17025, 0x6E3C }, { 17026, 0x6E44 }, { 17027, 0x7709 }, { 17028, 0x7C73 }, { 17029, 0x7F8E },
    { 17030, 0x8587 }, { 17031, 0x8B0E }, { 17032, 0x8FF7 }, { 17033, 0x9761 }, { 17034, 0x9EF4 }, { 17035, 0x5CB7 }, { 17036, 0x60B6 }, { 17037, 0x610D },
    { 17038, 0x61AB }, { 17039, 0x654F }, { 17040, 0x65FB }, { 17041, 0x65FC }, { 17042, 0x6C11 }, { 17043, 0x6CEF }, { 17044, 0x739F }, { 17045, 0x73C9 },
    { 17046, 0x7DE1 }, { 17047, 0x9594 }, { 17048, 0x5BC6 }, { 17049, 0x871C }, { 17050, 0x8B10 }, { 17051, 0x525D }, { 17052, 0x535A }, { 17053, 0x62CD },
    { 17054, 0x640F }, { 17055, 0x64B2 }, { 17056, 0x6734 }, { 17057, 0x6A38 }, { 17058, 0x6CCA }, { 17059, 0x73C0 }, { 17060, 0x749E }, { 17061, 0x7B94 },
    { 17062, 0x7C95 }, { 17063, 0x7E1B }, { 17064, 0x818A }, { 17065, 0x8236 }, { 17066, 0x8584 }, { 17067, 0x8FEB }, { 17068, 0x96F9 }, { 17069, 0x99C1 },
    { 17070, 0x4F34 }, { 17071, 0x534A }, { 17072, 0x53CD }, { 17073, 0x53DB }, { 17074, 0x62CC }, { 17075, 0x642C }, { 17076, 0x6500 }, { 17077, 0x6591 },
    { 17078, 0x69C3 }, { 17079, 0x6CEE }, { 17080, 0x6F58 }, { 17081, 0x73ED }, { 17082, 0x7554 }, { 17083, 0x7622 }, { 17084, 0x76E4 }, { 17085, 0x76FC },
    { 17086, 0x78D0 }, { 17087, 0x78FB }, { 17088, 0x792C }, { 17089, 0x7D46 }, { 17090, 0x822C }, { 17091, 0x87E0 }, { 17092, 0x8FD4 }, { 17093, 0x9812 },
    { 17094, 0x98EF }, { 17095, 0x52C3 }, { 17096, 0x62D4 }, { 17097, 0x64A5 }, { 17098, 0x6E24 }, { 17099, 0x6F51 }, { 17196, 0x767C }, { 17197, 0x8DCB },
    { 17198, 0x91B1 }, { 17199, 0x9262 }, { 17200, 0x9AEE }, { 17201, 0x9B43 }, { 17202, 0x5023 }, { 17203, 0x508D }, { 17204, 0x574A }, { 17205, 0x59A8 },
    { 17206, 0x5C28 }, { 17207, 0x5E47 }, { 17208, 0x5F77 }, { 17209, 0x623F }, { 17210, 0x653E }, { 17211, 0x65B9 }, { 17212, 0x65C1 }, { 17213, 0x6609 },
    { 17214, 0x678B }, { 17215, 0x699C }, { 17216, 0x6EC2 }, { 17217, 0x78C5 }, { 17218, 0x7D21 }, { 17219, 0x80AA }, { 17220, 0x8180 }, { 17221, 0x822B },
    { 17222, 0x82B3 }, { 17223, 0x84A1 }, { 17224, 0x868C }, { 17225, 0x8A2A }, { 17226, 0x8B17 }, { 17227, 0x90A6 }, { 17228, 0x9632 }, { 17229, 0x9F90 },
    { 17230, 0x500D }, { 17231, 0x4FF3 }, { 17232, 0xF963 }, { 17233, 0x57F9 }, { 17234, 0x5F98 }, { 17235, 0x62DC }, { 17236, 0x6392 }, { 17237, 0x676F },
    { 17238, 0x6E43 }, { 17239, 0x7119 }, { 17240, 0x76C3 }, { 17241, 0x80CC }, { 17242, 0x80DA }, { 17243, 0x88F4 }, { 17244, 0x88F5 }, { 17245, 0x8919 },
    { 17246, 0x8CE0 }, { 17247, 0x8F29 }, { 17248, 0x914D }, { 17249, 0x966A }, { 17250, 0x4F2F }, { 17251, 0x4F70 }, { 17252, 0x5E1B }, { 17253, 0x67CF },
    { 17254, 0x6822 }, { 17255, 0x767D }, { 17256, 0x767E }, { 17257, 0x9B44 }, { 17258, 0x5E61 }, { 17259, 0x6A0A }, { 17260, 0x7169 }, { 17261, 0x71D4 },
    { 17262, 0x756A }, { 17263, 0xF964 }, { 17264, 0x7E41 }, { 17265, 0x8543 }, { 17266, 0x85E9 }, { 17267, 0x98DC }, { 17268, 0x4F10 }, { 17269, 0x7B4F },
    { 17270, 0x7F70 }, { 17271, 0x95A5 }, { 17272, 0x51E1 }, { 17273, 0x5E06 }, { 17274, 0x68B5 }, { 17275, 0x6C3E }, { 17276, 0x6C4E }, { 17277, 0x6CDB },
    { 17278, 0x72AF }, { 17279, 0x7BC4 }, { 17280, 0x8303 }, { 17281, 0x6CD5 }, { 17282, 0x743A }, { 17283, 0x50FB }, { 17284, 0x5288 }, { 17285, 0x58C1 },
    { 17286, 0x64D8 }, { 17287, 0x6A97 }, { 17288, 0x74A7 }, { 17289, 0x7656 }, { 17386, 0x78A7 }, { 17387, 0x8617 }, { 17388, 0x95E2 }, { 17389, 0x9739 },
    { 17390, 0xF965 }, { 17391, 0x535E }, { 17392, 0x5F01 }, { 17393, 0x8B8A }, { 17394, 0x8FA8 }, { 17395, 0x8FAF }, { 17396, 0x908A }, { 17397, 0x5225 },
    { 17398, 0x77A5 }, { 17399, 0x9C49 }, { 17400, 0x9F08 }, { 17401, 0x4E19 }, { 17402, 0x5002 }, { 17403, 0x5175 }, { 17404, 0x5C5B }, { 17405, 0x5E77 },
    { 17406, 0x661E }, { 17407, 0x663A }, { 17408, 0x67C4 }, { 17409, 0x68C5 }, { 17410, 0x70B3 }, { 17411, 0x7501 }, { 17412, 0x75C5 }, { 17413, 0x79C9 },
    { 17414, 0x7ADD }, { 17415, 0x8F27 }, { 17416, 0x9920 }, { 17417, 0x9A08 }, { 17418, 0x4FDD }, { 17419, 0x5821 }, { 17420, 0x5831 }, { 17421, 0x5BF6 },
    { 17422, 0x666E }, { 17423, 0x6B65 }, { 17424, 0x6D11 }, { 17425, 0x6E7A }, { 17426, 0x6F7D }, { 17427, 0x73E4 }, { 17428, 0x752B }, { 17429, 0x83E9 },
    { 17430, 0x88DC }, { 17431, 0x8913 }, { 17432, 0x8B5C }, { 17433, 0x8F14 }, { 17434, 0x4F0F }, { 17435, 0x50D5 }, { 17436, 0x5310 }, { 17437, 0x535C },
    { 17438, 0x5B93 }, { 17439, 0x5FA9 }, { 17440, 0x670D }, { 17441, 0x798F }, { 17442, 0x8179 }, { 17443, 0x832F }, { 17444, 0x8514 }, { 17445, 0x8907 },
    { 17446, 0x8986 }, { 17447, 0x8F39 }, { 17448, 0x8F3B }, { 17449, 0x99A5 }, { 17450, 0x9C12 }, { 17451, 0x672C }, { 17452, 0x4E76 }, { 17453, 0x4FF8 },
    { 17454, 0x5949 }, { 17455, 0x5C01 }, { 17456, 0x5CEF }, { 17457, 0x5CF0 }, { 17458, 0x6367 }, { 17459, 0x68D2 }, { 17460, 0x70FD }, { 17461, 0x71A2 },
    { 17462, 0x742B }, { 17463, 0x7E2B }, { 17464, 0x84EC }, { 17465, 0x8702 }, { 17466, 0x9022 }, { 17467, 0x92D2 }, { 17468, 0x9CF3 }, { 17469, 0x4E0D },
    { 17470, 0x4ED8 }, { 17471, 0x4FEF }, { 17472, 0x5085 }, { 17473, 0x5256 }, { 17474, 0x526F }, { 17475, 0x5426 }, { 17476, 0x5490 }, { 17477, 0x57E0 },
    { 17478, 0x592B }, { 17479, 0x5A66 }, { 17576, 0x5B5A }, { 17577, 0x5B75 }, { 17578, 0x5BCC }, { 17579, 0x5E9C }, { 17580, 0xF966 }, { 17581, 0x6276 },
    { 17582, 0x6577 }, { 17583, 0x65A7 }, { 17584, 0x6D6E }, { 17585, 0x6EA5 }, { 17586, 0x7236 }, { 17587, 0x7B26 }, { 17588, 0x7C3F }, { 17589, 0x7F36 },
    { 17590, 0x8150 }, { 17591, 0x8151 }, { 17592, 0x819A }, { 17593, 0x8240 }, { 17594, 0x8299 }, { 17595, 0x83A9 }, { 17596, 0x8A03 }, { 17597, 0x8CA0 },
    { 17598, 0x8CE6 }, { 17599, 0x8CFB }, { 17600, 0x8D74 }, { 17601, 0x8DBA }, { 17602, 0x90E8 }, { 17603, 0x91DC }, { 17604, 0x961C }, { 17605, 0x9644 },
    { 17606, 0x99D9 }, { 17607, 0x9CE7 }, { 17608, 0x5317 }, { 17609, 0x5206 }, { 17610, 0x5429 }, { 17611, 0x5674 }, { 17612, 0x58B3 }, { 17613, 0x5954 },
    { 17614, 0x596E }, { 17615, 0x5FFF }, { 17616, 0x61A4 }, { 17617, 0x626E }, { 17618, 0x6610 }, { 17619, 0x6C7E }, { 17620, 0x711A }, { 17621, 0x76C6 },
    { 17622, 0x7C89 }, { 17623, 0x7CDE }, { 17624, 0x7D1B }, { 17625, 0x82AC }, { 17626, 0x8CC1 }, { 17627, 0x96F0 }, { 17628, 0xF967 }, { 17629, 0x4F5B },
    { 17630, 0x5F17 }, { 17631, 0x5F7F }, { 17632, 0x62C2 }, { 17633, 0x5D29 }, { 17634, 0x670B }, { 17635, 0x68DA }, { 17636, 0x787C }, { 17637, 0x7E43 },
    { 17638, 0x9D6C }, { 17639, 0x4E15 }, { 17640, 0x5099 }, { 17641, 0x5315 }, { 17642, 0x532A }, { 17643, 0x5351 }, { 17644, 0x5983 }, { 17645, 0x5A62 },
    { 17646, 0x5E87 }, { 17647, 0x60B2 }, { 17648, 0x618A }, { 17649, 0x6249 }, { 17650, 0x6279 }, { 17651, 0x6590 }, { 17652, 0x6787 }, { 17653, 0x69A7 },
    { 17654, 0x6BD4 }, { 17655, 0x6BD6 }, { 17656, 0x6BD7 }, { 17657, 0x6BD8 }, { 17658, 0x6CB8 }, { 17659, 0xF968 }, { 17660, 0x7435 }, { 17661, 0x75FA },
    { 17662, 0x7812 }, { 17663, 0x7891 }, { 17664, 0x79D5 }, { 17665, 0x79D8 }, { 17666, 0x7C83 }, { 17667, 0x7DCB }, { 17668, 0x7FE1 }, { 17669, 0x80A5 },
    { 17766, 0x813E }, { 17767, 0x81C2 }, { 17768, 0x83F2 }, { 17769, 0x871A }, { 17770, 0x88E8 }, { 17771, 0x8AB9 }, { 17772, 0x8B6C }, { 17773, 0x8CBB },
    { 17774, 0x9119 }, { 17775, 0x975E }, { 17776, 0x98DB }, { 17777, 0x9F3B }, { 17778, 0x56AC }, { 17779, 0x5B2A }, { 17780, 0x5F6C }, { 17781, 0x658C },
    { 17782, 0x6AB3 }, { 17783, 0x6BAF }, { 17784, 0x6D5C }, { 17785, 0x6FF1 }, { 17786, 0x7015 }, { 17787, 0x725D }, { 17788, 0x73AD }, { 17789, 0x8CA7 },
    { 17790, 0x8CD3 }, { 17791, 0x983B }, { 17792, 0x6191 }, { 17793, 0x6C37 }, { 17794, 0x8058 }, { 17795, 0x9A01 }, { 17796, 0x4E4D }, { 17797, 0x4E8B },
    { 17798, 0x4E9B }, { 17799, 0x4ED5 }, { 17800, 0x4F3A }, { 17801, 0x4F3C }, { 17802, 0x4F7F }, { 17803, 0x4FDF }, { 17804, 0x50FF }, { 17805, 0x53F2 },
    { 17806, 0x53F8 }, { 17807, 0x5506 }, { 17808, 0x55E3 }, { 17809, 0x56DB }, { 17810, 0x58EB }, { 17811, 0x5962 }, { 17812, 0x5A11 }, { 17813, 0x5BEB },
    { 17814, 0x5BFA }, { 17815, 0x5C04 }, { 17816, 0x5DF3 }, { 17817, 0x5E2B }, { 17818, 0x5F99 }, { 17819, 0x601D }, { 17820, 0x6368 }, { 17821, 0x659C },
    { 17822, 0x65AF }, { 17823, 0x67F6 }, { 17824, 0x67FB }, { 17825, 0x68AD }, { 17826, 0x6B7B }, { 17827, 0x6C99 }, { 17828, 0x6CD7 }, { 17829, 0x6E23 },
    { 17830, 0x7009 }, { 17831, 0x7345 }, { 17832, 0x7802 }, { 17833, 0x793E }, { 17834, 0x7940 }, { 17835, 0x7960 }, { 17836, 0x79C1 }, { 17837, 0x7BE9 },
    { 17838, 0x7D17 }, { 17839, 0x7D72 }, { 17840, 0x8086 }, { 17841, 0x820D }, { 17842, 0x838E }, { 17843, 0x84D1 }, { 17844, 0x86C7 }, { 17845, 0x88DF },
    { 17846, 0x8A50 }, { 17847, 0x8A5E }, { 17848, 0x8B1D }, { 17849, 0x8CDC }, { 17850, 0x8D66 }, { 17851, 0x8FAD }, { 17852, 0x90AA }, { 17853, 0x98FC },
    { 17854, 0x99DF }, { 17855, 0x9E9D }, { 17856, 0x524A }, { 17857, 0xF969 }, { 17858, 0x6714 }, { 17859, 0xF96A }, { 17956, 0x5098 }, { 17957, 0x522A },
    { 17958, 0x5C71 }, { 17959, 0x6563 }, { 17960, 0x6C55 }, { 17961, 0x73CA }, { 17962, 0x7523 }, { 17963, 0x759D }, { 17964, 0x7B97 }, { 17965, 0x849C },
    { 17966, 0x9178 }, { 17967, 0x9730 }, { 17968, 0x4E77 }, { 17969, 0x6492 }, { 17970, 0x6BBA }, { 17971, 0x715E }, { 17972, 0x85A9 }, { 17973, 0x4E09 },
    { 17974, 0xF96B }, { 17975, 0x6749 }, { 17976, 0x68EE }, { 17977, 0x6E17 }, { 17978, 0x829F }, { 17979, 0x8518 }, { 17980, 0x886B }, { 17981, 0x63F7 },
    { 17982, 0x6F81 }, { 17983, 0x9212 }, { 17984, 0x98AF }, { 17985, 0x4E0A }, { 17986, 0x50B7 }, { 17987, 0x50CF }, { 17988, 0x511F }, { 17989, 0x5546 },
    { 17990, 0x55AA }, { 17991, 0x5617 }, { 17992, 0x5B40 }, { 17993, 0x5C19 }, { 17994, 0x5CE0 }, { 17995, 0x5E38 }, { 17996, 0x5E8A }, { 17997, 0x5EA0 },
    { 17998, 0x5EC2 }, { 17999, 0x60F3 }, { 18000, 0x6851 }, { 18001, 0x6A61 }, { 18002, 0x6E58 }, { 18003, 0x723D }, { 18004, 0x7240 }, { 18005, 0x72C0 },
    { 18006, 0x76F8 }, { 18007, 0x7965 }, { 18008, 0x7BB1 }, { 18009, 0x7FD4 }, { 18010, 0x88F3 }, { 18011, 0x89F4 }, { 18012, 0x8A73 }, { 18013, 0x8C61 },
    { 18014, 0x8CDE }, { 18015, 0x971C }, { 18016, 0x585E }, { 18017, 0x74BD }, { 18018, 0x8CFD }, { 18019, 0x55C7 }, { 18020, 0xF96C }, { 18021, 0x7A61 },
    { 18022, 0x7D22 }, { 18023, 0x8272 }, { 18024, 0x7272 }, { 18025, 0x751F }, { 18026, 0x7525 }, { 18027, 0xF96D }, { 18028, 0x7B19 }, { 18029, 0x5885 },
    { 18030, 0x58FB }, { 18031, 0x5DBC }, { 18032, 0x5E8F }, { 18033, 0x5EB6 }, { 18034, 0x5F90 }, { 18035, 0x6055 }, { 18036, 0x6292 }, { 18037, 0x637F },
    { 18038, 0x654D }, { 18039, 0x6691 }, { 18040, 0x66D9 }, { 18041, 0x66F8 }, { 18042, 0x6816 }, { 18043, 0x68F2 }, { 18044, 0x7280 }, { 18045, 0x745E },
    { 18046, 0x7B6E }, { 18047, 0x7D6E }, { 18048, 0x7DD6 }, { 18049, 0x7F72 }, { 18146, 0x80E5 }, { 18147, 0x8212 }, { 18148, 0x85AF }, { 18149, 0x897F },
    { 18150, 0x8A93 }, { 18151, 0x901D }, { 18152, 0x92E4 }, { 18153, 0x9ECD }, { 18154, 0x9F20 }, { 18155, 0x5915 }, { 18156, 0x596D }, { 18157, 0x5E2D },
    { 18158, 0x60DC }, { 18159, 0x6614 }, { 18160, 0x6673 }, { 18161, 0x6790 }, { 18162, 0x6C50 }, { 18163, 0x6DC5 }, { 18164, 0x6F5F }, { 18165, 0x77F3 },
    { 18166, 0x78A9 }, { 18167, 0x84C6 }, { 18168, 0x91CB }, { 18169, 0x932B }, { 18170, 0x4ED9 }, { 18171, 0x50CA }, { 18172, 0x5148 }, { 18173, 0x5584 },
    { 18174, 0x5B0B }, { 18175, 0x5BA3 }, { 18176, 0x6247 }, { 18177, 0x657E }, { 18178, 0x65CB }, { 18179, 0x6E32 }, { 18180, 0x717D }, { 18181, 0x7401 },
    { 18182, 0x7444 }, { 18183, 0x7487 }, { 18184, 0x74BF }, { 18185, 0x766C }, { 18186, 0x79AA }, { 18187, 0x7DDA }, { 18188, 0x7E55 }, { 18189, 0x7FA8 },
    { 18190, 0x817A }, { 18191, 0x81B3 }, { 18192, 0x8239 }, { 18193, 0x861A }, { 18194, 0x87EC }, { 18195, 0x8A75 }, { 18196, 0x8DE3 }, { 18197, 0x9078 },
    { 18198, 0x9291 }, { 18199, 0x9425 }, { 18200, 0x994D }, { 18201, 0x9BAE }, { 18202, 0x5368 }, { 18203, 0x5C51 }, { 18204, 0x6954 }, { 18205, 0x6CC4 },
    { 18206, 0x6D29 }, { 18207, 0x6E2B }, { 18208, 0x820C }, { 18209, 0x859B }, { 18210, 0x893B }, { 18211, 0x8A2D }, { 18212, 0x8AAA }, { 18213, 0x96EA },
    { 18214, 0x9F67 }, { 18215, 0x5261 }, { 18216, 0x66B9 }, { 18217, 0x6BB2 }, { 18218, 0x7E96 }, { 18219, 0x87FE }, { 18220, 0x8D0D }, { 18221, 0x9583 },
    { 18222, 0x965D }, { 18223, 0x651D }, { 18224, 0x6D89 }, { 18225, 0x71EE }, { 18226, 0xF96E }, { 18227, 0x57CE }, { 18228, 0x59D3 }, { 18229, 0x5BAC },
    { 18230, 0x6027 }, { 18231, 0x60FA }, { 18232, 0x6210 }, { 18233, 0x661F }, { 18234, 0x665F }, { 18235, 0x7329 }, { 18236, 0x73F9 }, { 18237, 0x76DB },
    { 18238, 0x7701 }, { 18239, 0x7B6C }, { 18336, 0x8056 }, { 18337, 0x8072 }, { 18338, 0x8165 }, { 18339, 0x8AA0 }, { 18340, 0x9192 }, { 18341, 0x4E16 },
    { 18342, 0x52E2 }, { 18343, 0x6B72 }, { 18344, 0x6D17 }, { 18345, 0x7A05 }, { 18346, 0x7B39 }, { 18347, 0x7D30 }, { 18348, 0xF96F }, { 18349, 0x8CB0 },
    { 18350, 0x53EC }, { 18351, 0x562F }, { 18352, 0x5851 }, { 18353, 0x5BB5 }, { 18354, 0x5C0F }, { 18355, 0x5C11 }, { 18356, 0x5DE2 }, { 18357, 0x6240 },
    { 18358, 0x6383 }, { 18359, 0x6414 }, { 18360, 0x662D }, { 18361, 0x68B3 }, { 18362, 0x6CBC }, { 18363, 0x6D88 }, { 18364, 0x6EAF }, { 18365, 0x701F },
    { 18366, 0x70A4 }, { 18367, 0x71D2 }, { 18368, 0x7526 }, { 18369, 0x758F }, { 18370, 0x758E }, { 18371, 0x7619 }, { 18372, 0x7B11 }, { 18373, 0x7BE0 },
    { 18374, 0x7C2B }, { 18375, 0x7D20 }, { 18376, 0x7D39 }, { 18377, 0x852C }, { 18378, 0x856D }, { 18379, 0x8607 }, { 18380, 0x8A34 }, { 18381, 0x900D },
    { 18382, 0x9061 }, { 18383, 0x90B5 }, { 18384, 0x92B7 }, { 18385, 0x97F6 }, { 18386, 0x9A37 }, { 18387, 0x4FD7 }, { 18388, 0x5C6C }, { 18389, 0x675F },
    { 18390, 0x6D91 }, { 18391, 0x7C9F }, { 18392, 0x7E8C }, { 18393, 0x8B16 }, { 18394, 0x8D16 }, { 18395, 0x901F }, { 18396, 0x5B6B }, { 18397, 0x5DFD },
    { 18398, 0x640D }, { 18399, 0x84C0 }, { 18400, 0x905C }, { 18401, 0x98E1 }, { 18402, 0x7387 }, { 18403, 0x5B8B }, { 18404, 0x609A }, { 18405, 0x677E },
    { 18406, 0x6DDE }, { 18407, 0x8A1F }, { 18408, 0x8AA6 }, { 18409, 0x9001 }, { 18410, 0x980C }, { 18411, 0x5237 }, { 18412, 0xF970 }, { 18413, 0x7051 },
    { 18414, 0x788E }, { 18415, 0x9396 }, { 18416, 0x8870 }, { 18417, 0x91D7 }, { 18418, 0x4FEE }, { 18419, 0x53D7 }, { 18420, 0x55FD }, { 18421, 0x56DA },
    { 18422, 0x5782 }, { 18423, 0x58FD }, { 18424, 0x5AC2 }, { 18425, 0x5B88 }, { 18426, 0x5CAB }, { 18427, 0x5CC0 }, { 18428, 0x5E25 }, { 18429, 0x6101 },
    { 18526, 0x620D }, { 18527, 0x624B }, { 18528, 0x6388 }, { 18529, 0x641C }, { 18530, 0x6536 }, { 18531, 0x6578 }, { 18532, 0x6A39 }, { 18533, 0x6B8A },
    { 18534, 0x6C34 }, { 18535, 0x6D19 }, { 18536, 0x6F31 }, { 18537, 0x71E7 }, { 18538, 0x72E9 }, { 18539, 0x7378 }, { 18540, 0x7407 }, { 18541, 0x74B2 },
    { 18542, 0x7626 }, { 18543, 0x7761 }, { 18544, 0x79C0 }, { 18545, 0x7A57 }, { 18546, 0x7AEA }, { 18547, 0x7CB9 }, { 18548, 0x7D8F }, { 18549, 0x7DAC },
    { 18550, 0x7E61 }, { 18551, 0x7F9E }, { 18552, 0x8129 }, { 18553, 0x8331 }, { 18554, 0x8490 }, { 18555, 0x84DA }, { 18556, 0x85EA }, { 18557, 0x8896 },
    { 18558, 0x8AB0 }, { 18559, 0x8B90 }, { 18560, 0x8F38 }, { 18561, 0x9042 }, { 18562, 0x9083 }, { 18563, 0x916C }, { 18564, 0x9296 }, { 18565, 0x92B9 },
    { 18566, 0x968B }, { 18567, 0x96A7 }, { 18568, 0x96A8 }, { 18569, 0x96D6 }, { 18570, 0x9700 }, { 18571, 0x9808 }, { 18572, 0x9996 }, { 18573, 0x9AD3 },
    { 18574, 0x9B1A }, { 18575, 0x53D4 }, { 18576, 0x587E }, { 18577, 0x5919 }, { 18578, 0x5B70 }, { 18579, 0x5BBF }, { 18580, 0x6DD1 }, { 18581, 0x6F5A },
    { 18582, 0x719F }, { 18583, 0x7421 }, { 18584, 0x74B9 }, { 18585, 0x8085 }, { 18586, 0x83FD }, { 18587, 0x5DE1 }, { 18588, 0x5F87 }, { 18589, 0x5FAA },
    { 18590, 0x6042 }, { 18591, 0x65EC }, { 18592, 0x6812 }, { 18593, 0x696F }, { 18594, 0x6A53 }, { 18595, 0x6B89 }, { 18596, 0x6D35 }, { 18597, 0x6DF3 },
    { 18598, 0x73E3 }, { 18599, 0x76FE }, { 18600, 0x77AC }, { 18601, 0x7B4D }, { 18602, 0x7D14 }, { 18603, 0x8123 }, { 18604, 0x821C }, { 18605, 0x8340 },
    { 18606, 0x84F4 }, { 18607, 0x8563 }, { 18608, 0x8A62 }, { 18609, 0x8AC4 }, { 18610, 0x9187 }, { 18611, 0x931E }, { 18612, 0x9806 }, { 18613, 0x99B4 },
    { 18614, 0x620C }, { 18615, 0x8853 }, { 18616, 0x8FF0 }, { 18617, 0x9265 }, { 18618, 0x5D07 }, { 18619, 0x5D27 }, { 18716, 0x5D69 }, { 18717, 0x745F },
    { 18718, 0x819D }, { 18719, 0x8768 }, { 18720, 0x6FD5 }, { 18721, 0x62FE }, { 18722, 0x7FD2 }, { 18723, 0x8936 }, { 18724, 0x8972 }, { 18725, 0x4E1E },
    { 18726, 0x4E58 }, { 18727, 0x50E7 }, { 18728, 0x52DD }, { 18729, 0x5347 }, { 18730, 0x627F }, { 18731, 0x6607 }, { 18732, 0x7E69 }, { 18733, 0x8805 },
    { 18734, 0x965E }, { 18735, 0x4F8D }, { 18736, 0x5319 }, { 18737, 0x5636 }, { 18738, 0x59CB }, { 18739, 0x5AA4 }, { 18740, 0x5C38 }, { 18741, 0x5C4E },
    { 18742, 0x5C4D }, { 18743, 0x5E02 }, { 18744, 0x5F11 }, { 18745, 0x6043 }, { 18746, 0x65BD }, { 18747, 0x662F }, { 18748, 0x6642 }, { 18749, 0x67BE },
    { 18750, 0x67F4 }, { 18751, 0x731C }, { 18752, 0x77E2 }, { 18753, 0x793A }, { 18754, 0x7FC5 }, { 18755, 0x8494 }, { 18756, 0x84CD }, { 18757, 0x8996 },
    { 18758, 0x8A66 }, { 18759, 0x8A69 }, { 18760, 0x8AE1 }, { 18761, 0x8C55 }, { 18762, 0x8C7A }, { 18763, 0x57F4 }, { 18764, 0x5BD4 }, { 18765, 0x5F0F },
    { 18766, 0x606F }, { 18767, 0x62ED }, { 18768, 0x690D }, { 18769, 0x6B96 }, { 18770, 0x6E5C }, { 18771, 0x7184 }, { 18772, 0x7BD2 }, { 18773, 0x8755 },
    { 18774, 0x8B58 }, { 18775, 0x8EFE }, { 18776, 0x98DF }, { 18777, 0x98FE }, { 18778, 0x4F38 }, { 18779, 0x4F81 }, { 18780, 0x4FE1 }, { 18781, 0x547B },
    { 18782, 0x5A20 }, { 18783, 0x5BB8 }, { 18784, 0x613C }, { 18785, 0x65B0 }, { 18786, 0x6668 }, { 18787, 0x71FC }, { 18788, 0x7533 }, { 18789, 0x795E },
    { 18790, 0x7D33 }, { 18791, 0x814E }, { 18792, 0x81E3 }, { 18793, 0x8398 }, { 18794, 0x85AA }, { 18795, 0x85CE }, { 18796, 0x8703 }, { 18797, 0x8A0A },
    { 18798, 0x8EAB }, { 18799, 0x8F9B }, { 18800, 0xF971 }, { 18801, 0x8FC5 }, { 18802, 0x5931 }, { 18803, 0x5BA4 }, { 18804, 0x5BE6 }, { 18805, 0x6089 },
    { 18806, 0x5BE9 }, { 18807, 0x5C0B }, { 18808, 0x5FC3 }, { 18809, 0x6C81 }, { 18906, 0xF972 }, { 18907, 0x6DF1 }, { 18908, 0x700B }, { 18909, 0x751A },
    { 18910, 0x82AF }, { 18911, 0x8AF6 }, { 18912, 0x4EC0 }, { 18913, 0x5341 }, { 18914, 0xF973 }, { 18915, 0x96D9 }, { 18916, 0x6C0F }, { 18917, 0x4E9E },
    { 18918, 0x4FC4 }, { 18919, 0x5152 }, { 18920, 0x555E }, { 18921, 0x5A25 }, { 18922, 0x5CE8 }, { 18923, 0x6211 }, { 18924, 0x7259 }, { 18925, 0x82BD },
    { 18926, 0x83AA }, { 18927, 0x86FE }, { 18928, 0x8859 }, { 18929, 0x8A1D }, { 18930, 0x963F }, { 18931, 0x96C5 }, { 18932, 0x9913 }, { 18933, 0x9D09 },
    { 18934, 0x9D5D }, { 18935, 0x580A }, { 18936, 0x5CB3 }, { 18937, 0x5DBD }, { 18938, 0x5E44 }, { 18939, 0x60E1 }, { 18940, 0x6115 }, { 18941, 0x63E1 },
    { 18942, 0x6A02 }, { 18943, 0x6E25 }, { 18944, 0x9102 }, { 18945, 0x9354 }, { 18946, 0x984E }, { 18947, 0x9C10 }, { 18948, 0x9F77 }, { 18949, 0x5B89 },
    { 18950, 0x5CB8 }, { 18951, 0x6309 }, { 18952, 0x664F }, { 18953, 0x6848 }, { 18954, 0x773C }, { 18955, 0x96C1 }, { 18956, 0x978D }, { 18957, 0x9854 },
    { 18958, 0x9B9F }, { 18959, 0x65A1 }, { 18960, 0x8B01 }, { 18961, 0x8ECB }, { 18962, 0x95BC }, { 18963, 0x5535 }, { 18964, 0x5CA9 }, { 18965, 0x5DD6 },
    { 18966, 0x5EB5 }, { 18967, 0x6697 }, { 18968, 0x764C }, { 18969, 0x83F4 }, { 18970, 0x95C7 }, { 18971, 0x58D3 }, { 18972, 0x62BC }, { 18973, 0x72CE },
    { 18974, 0x9D28 }, { 18975, 0x4EF0 }, { 18976, 0x592E }, { 18977, 0x600F }, { 18978, 0x663B }, { 18979, 0x6B83 }, { 18980, 0x79E7 }, { 18981, 0x9D26 },
    { 18982, 0x5393 }, { 18983, 0x54C0 }, { 18984, 0x57C3 }, { 18985, 0x5D16 }, { 18986, 0x611B }, { 18987, 0x66D6 }, { 18988, 0x6DAF }, { 18989, 0x788D },
    { 18990, 0x827E }, { 18991, 0x9698 }, { 18992, 0x9744 }, { 18993, 0x5384 }, { 18994, 0x627C }, { 18995, 0x6396 }, { 18996, 0x6DB2 }, { 18997, 0x7E0A },
    { 18998, 0x814B }, { 18999, 0x984D }, { 19096, 0x6AFB }, { 19097, 0x7F4C }, { 19098, 0x9DAF }, { 19099, 0x9E1A }, { 19100, 0x4E5F }, { 19101, 0x503B },
    { 19102, 0x51B6 }, { 19103, 0x591C }, { 19104, 0x60F9 }, { 19105, 0x63F6 }, { 19106, 0x6930 }, { 19107, 0x723A }, { 19108, 0x8036 }, { 19109, 0xF974 },
    { 19110