/*
 * ATRAC9 decoder
 * Copyright (c) 2018 Rostislav Pehlivanov <atomnuker@gmail.com>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_ATRAC9TAB_H
#define AVCODEC_ATRAC9TAB_H

#include <stdint.h>

#include "libavutil/channel_layout.h"

enum ATRAC9BlockType {
    ATRAC9_BLOCK_TYPE_SCE, /* Single channel */
    ATRAC9_BLOCK_TYPE_CPE, /* 2 coupled channels */
    ATRAC9_BLOCK_TYPE_LFE, /* Single LFE channel */
};

typedef struct ATRAC9BlockConfig {
    uint64_t channel_layout;
    enum ATRAC9BlockType type[5];
    int plane_map[5][2];
    int count;
} ATRAC9BlockConfig;

static const ATRAC9BlockConfig at9_block_layout[] = {
    { /* Mono */
        .channel_layout  = AV_CH_LAYOUT_MONO,
        .type            = { ATRAC9_BLOCK_TYPE_SCE, },
        .count           = 1,
    },
    { /* Dual Mono */
        .channel_layout  = AV_CH_LAYOUT_STEREO,
        .type            = { ATRAC9_BLOCK_TYPE_SCE,
                             ATRAC9_BLOCK_TYPE_SCE, },
        .plane_map       = { { 0 }, { 1 }, },
        .count           = 2,
    },
    { /* Stereo */
        .channel_layout  = AV_CH_LAYOUT_STEREO,
        .type            = { ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, },
        .count           = 1,
    },
    { /* 5.1 */
        .channel_layout  = AV_CH_LAYOUT_5POINT1,
        .type            = { ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_SCE,
                             ATRAC9_BLOCK_TYPE_LFE,
                             ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, { 2 }, { 3 }, { 4, 5 }, },
        .count           = 4,
    },
    { /* 7.1 */
        .channel_layout  = AV_CH_LAYOUT_7POINT1,
        .type            = { ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_SCE,
                             ATRAC9_BLOCK_TYPE_LFE,
                             ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, { 2 }, { 3 }, { 4, 5 }, { 6, 7 }, },
        .count           = 5,
    },
    { /* Quad */
        .channel_layout  = AV_CH_LAYOUT_QUAD,
        .type            = { ATRAC9_BLOCK_TYPE_CPE,
                             ATRAC9_BLOCK_TYPE_CPE, },
        .plane_map       = { { 0, 1 }, { 2, 3 }, },
        .count           = 2,
    },
};

static const uint8_t at9_tab_sri_frame_log2[] = {
    6, 6, 7, 7, 7, 8, 8, 8, 6, 6, 7, 7, 7, 8, 8, 8,
};

static const uint8_t at9_tab_band_q_unit_map[] = {
    0, 4, 8, 10, 12, 13, 14, 15, 16, 18, 20, 21, 22, 23, 24, 25, 26, 28, 30,
};

static const uint8_t at9_q_unit_to_coeff_cnt[] = {
    2, 2, 2, 2, 2,  2,  2,  2,  4,  4,  4,  4,  8,  8,  8,
    8, 8, 8, 8, 8, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
};

static const int at9_q_unit_to_coeff_idx[] = {
    0, 2, 4, 6, 8, 10, 12, 14, 16, 20, 24, 28, 32, 40, 48, 56, 64,
    72, 80, 88, 96, 112, 128, 144, 160, 176, 192, 208, 224, 240, 256
};

const uint8_t at9_q_unit_to_codebookidx[] = {
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2,
    2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
};

static const uint8_t at9_tab_sri_max_bands[] = {
    8, 8, 12, 12, 12, 18, 18, 18, 8, 8, 12, 12, 12, 16, 16, 16,
};

static const int at9_tab_samplerates[] = {
    11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000, 44100, 48000,
    64000, 88200, 96000, 128000, 176400, 192000,
};

static const uint8_t at9_tab_band_ext_cnt[][6] = {
    { 0, 0, 0, 4, 4, 2 },
    { 0, 0, 0, 0, 0, 0 },
    { 0, 0, 0, 2, 2, 1 },
    { 0, 0, 0, 2, 2, 2 },
    { 1, 1, 1, 0, 0, 0 },
};

/* B unit, C unit, Band count */
static const uint8_t at9_tab_band_ext_group[][3] = {
    { 16, 21, 0 },
    { 18, 22, 1 },
    { 20, 22, 2 },
    { 21, 22, 3 },
    { 21, 22, 3 },
    { 23, 24, 4 },
    { 23, 24, 4 },
    { 24, 24, 5 },
};

static const uint8_t at9_tab_band_ext_lengths[][6][4] = {
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 5, 4, 3, 3 },
        { 4, 4, 3, 4 },
        { 4, 5, 0, 0 },
    },
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
    },
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 6, 6, 0, 0 },
        { 6, 6, 0, 0 },
        { 6, 0, 0, 0 },
    },
    {
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 4, 4, 0, 0 },
        { 4, 4, 0, 0 },
        { 4, 4, 0, 0 },
    },
    {
        { 3, 0, 0, 0 },
        { 3, 0, 0, 0 },
        { 3, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
        { 0, 0, 0, 0 },
    },
};

static const float at9_band_ext_scales_m0[][5][32] = {
    {
        {
            0.000000e+0f, 1.988220e-1f, 2.514343e-1f, 2.960510e-1f,
            3.263550e-1f, 3.771362e-1f, 3.786926e-1f, 4.540405e-1f,
            4.877625e-1f, 5.262451e-1f, 5.447083e-1f, 5.737000e-1f,
            6.212158e-1f, 6.222839e-1f, 6.560974e-1f, 6.896667e-1f,
            7.555542e-1f, 7.677917e-1f, 7.918091e-1f, 7.971497e-1f,
            8.188171e-1f, 8.446045e-1f, 9.790649e-1f, 9.822083e-1f,
            9.846191e-1f, 9.859314e-1f, 9.863586e-1f, 9.863892e-1f,
            9.873352e-1f, 9.881287e-1f, 9.898682e-1f, 9.913330e-1f,
        }, {
            0.000000e+0f, 9.982910e-1f, 7.592773e-2f, 7.179565e-1f,
            9.851379e-1f, 5.340271e-1f, 9.013672e-1f, 6.349182e-1f,
            7.226257e-1f, 1.948547e-1f, 7.628174e-1f, 9.873657e-1f,
            8.112183e-1f, 2.715454e-1f, 9.734192e-1f, 1.443787e-1f,
            4.640198e-1f, 3.249207e-1f, 3.790894e-1f, 8.276367e-2f,
            5.954590e-1f, 2.864380e-1f, 9.806824e-1f, 7.929077e-1f,
            6.292114e-1f, 4.887085e-1f, 2.905273e-1f, 1.301880e-1f,
            3.140869e-1f, 5.482483e-1f, 4.210815e-1f, 1.182861e-1f,
        }, {
            0.000000e+0f, 3.155518e-2f, 8.581543e-2f, 1.364746e-1f,
            1.858826e-1f, 2.368469e-1f, 2.888184e-1f, 3.432617e-1f,
            4.012451e-1f, 4.623108e-1f, 5.271301e-1f, 5.954895e-1f,
            6.681213e-1f, 7.448425e-1f, 8.245239e-1f, 9.097290e-1f,
        }, {
            0.000000e+0f, 4.418945e-2f, 1.303711e-1f, 2.273560e-1f,
            3.395996e-1f, 4.735718e-1f, 6.267090e-1f, 8.003845e-1f,
        }, {
            0.000000e+0f, 2.804565e-2f, 9.683228e-2f, 1.849976e-1f,
            3.005981e-1f, 4.470520e-1f, 6.168518e-1f, 8.007813e-1f,
        },
    },
    {
        {
            0.000000e+0f, 2.708740e-1f, 3.479614e-1f, 3.578186e-1f,
            5.083618e-1f, 5.299072e-1f, 5.819092e-1f, 6.381836e-1f,
            7.276917e-1f, 7.595520e-1f, 7.878723e-1f, 9.707336e-1f,
            9.713135e-1f, 9.736023e-1f, 9.759827e-1f, 9.832458e-1f,
        }, {
            0.000000e+0f, 2.330627e-1f, 5.891418e-1f, 7.170410e-1f,
            2.036438e-1f, 1.613464e-1f, 6.668701e-1f, 9.481201e-1f,
            9.769897e-1f, 5.111694e-1f, 3.522644e-1f, 8.209534e-1f,
            2.933960e-1f, 9.757690e-1f, 5.289917e-1f, 4.372253e-1f,
        }, {
            0.000000e+0f, 4.360962e-2f, 1.056519e-1f, 1.590576e-1f,
            2.078857e-1f, 2.572937e-1f, 3.082581e-1f, 3.616028e-1f,
            4.191589e-1f, 4.792175e-1f, 5.438538e-1f, 6.125183e-1f,
            6.841125e-1f, 7.589417e-1f, 8.365173e-1f, 9.148254e-1f,
        }, {
            0.000000e+0f, 4.074097e-2f, 1.164551e-1f, 2.077026e-1f,
            3.184509e-1f, 4.532166e-1f, 6.124268e-1f, 7.932129e-1f,
        }, {
            0.000000e+0f, 8.880615e-3f, 2.932739e-2f, 5.593872e-2f,
            8.825684e-2f, 1.259155e-1f, 1.721497e-1f, 2.270813e-1f,
            2.901611e-1f, 3.579712e-1f, 4.334106e-1f, 5.147095e-1f,
            6.023254e-1f, 6.956177e-1f, 7.952881e-1f, 8.977356e-1f,
        },
    },
    {
        {
            0.000000e+0f, 7.379150e-2f, 1.806335e-1f, 2.687073e-1f,
            3.407898e-1f, 4.047546e-1f, 4.621887e-1f, 5.168762e-1f,
            5.703125e-1f, 6.237488e-1f, 6.763611e-1f, 7.288208e-1f,
            7.808533e-1f, 8.337708e-1f, 8.874512e-1f, 9.418030e-1f,
        }, {
            0.000000e+0f, 7.980347e-2f, 1.615295e-1f, 1.665649e-1f,
            1.822205e-1f, 2.185669e-1f, 2.292175e-1f, 2.456665e-1f,
            2.666321e-1f, 3.306580e-1f, 3.330688e-1f, 3.765259e-1f,
            4.085083e-1f, 4.400024e-1f, 4.407654e-1f, 4.817505e-1f,
            4.924011e-1f, 5.320740e-1f, 5.893860e-1f, 6.131287e-1f,
            6.212463e-1f, 6.278076e-1f, 6.308899e-1f, 7.660828e-1f,
            7.850647e-1f, 7.910461e-1f, 7.929382e-1f, 8.038330e-1f,
            9.834900e-1f, 9.846191e-1f, 9.852295e-1f, 9.862671e-1f,
        }, {
            0.000000e+0f, 6.084290e-1f, 3.672791e-1f, 3.151855e-1f,
            1.488953e-1f, 2.571716e-1f, 5.103455e-1f, 3.311157e-1f,
            5.426025e-2f, 4.254456e-1f, 7.998352e-1f, 7.873230e-1f,
            5.418701e-1f, 2.925110e-1f, 8.468628e-2f, 1.410522e-1f,
            9.819641e-1f, 9.609070e-1f, 3.530884e-2f, 9.729004e-2f,
            5.758362e-1f, 9.941711e-1f, 7.215576e-1f, 7.183228e-1f,
            2.028809e-1f, 9.588623e-2f, 2.032166e-1f, 1.338806e-1f,
            5.003357e-1f, 1.874390e-1f, 9.804993e-1f, 1.107788e-1f,
        },
    },
};

static const float at9_band_ext_scales_m2[] = {
    4.272461e-4f, 1.312256e-3f, 2.441406e-3f, 3.692627e-3f,
    4.913330e-3f, 6.134033e-3f, 7.507324e-3f, 8.972168e-3f,
    1.049805e-2f, 1.223755e-2f, 1.406860e-2f, 1.599121e-2f,
    1.800537e-2f, 2.026367e-2f, 2.264404e-2f, 2.517700e-2f,
    2.792358e-2f, 3.073120e-2f, 3.344727e-2f, 3.631592e-2f,
    3.952026e-2f, 4.275513e-2f, 4.608154e-2f, 4.968262e-2f,
    5.355835e-2f, 5.783081e-2f, 6.195068e-2f, 6.677246e-2f,
    7.196045e-2f, 7.745361e-2f, 8.319092e-2f, 8.993530e-2f,
    9.759521e-2f, 1.056213e-1f, 1.138916e-1f, 1.236267e-1f,
    1.348267e-1f, 1.470337e-1f, 1.603394e-1f, 1.755676e-1f,
    1.905823e-1f, 2.071228e-1f, 2.245178e-1f, 2.444153e-1f,
    2.658997e-1f, 2.897644e-1f, 3.146057e-1f, 3.450012e-1f,
    3.766174e-1f, 4.122620e-1f, 4.505615e-1f, 4.893799e-1f,
    5.305481e-1f, 5.731201e-1f, 6.157837e-1f, 6.580811e-1f,
    6.985168e-1f, 7.435303e-1f, 7.865906e-1f, 8.302612e-1f,
    8.718567e-1f, 9.125671e-1f, 9.575806e-1f, 9.996643e-1f,
};

static const float at9_band_ext_scales_m3[][2] = {
    { 3.491211e-1f, -2.913818e-1f, }, { 5.371094e-1f, -2.541504e-1f, },
    { 6.782227e-1f, -1.664429e-1f, }, { 7.910156e-1f, -1.476440e-1f, },
    { 9.057617e-1f, -1.342163e-1f, }, { 1.024902e+0f, -1.220703e-1f, },
    { 1.156250e+0f, -1.117554e-1f, }, { 1.290527e+0f, -1.026611e-1f, },
    { 1.458984e+0f, -9.436035e-2f, }, { 1.664551e+0f, -8.483887e-2f, },
    { 1.929688e+0f, -7.476807e-2f, }, { 2.278320e+0f, -6.304932e-2f, },
    { 2.831543e+0f, -4.492188e-2f, }, { 3.659180e+0f, -2.447510e-2f, },
    { 5.257813e+0f, +1.831055e-4f, }, { 8.373047e+0f, +4.174805e-2f, },
};

static const float at9_band_ext_scales_m4[] = {
    3.610229e-2f, 1.260681e-1f, 2.227478e-1f, 3.338318e-1f,
    4.662170e-1f, 6.221313e-1f, 7.989197e-1f, 9.939575e-1f,
};

static const float at9_quant_step_coarse[] = {
    2.0000000000000000e+0f, 6.6666666666666663e-1f, 2.8571428571428570e-1f,
    1.3333333333333333e-1f, 6.4516129032258063e-2f, 3.1746031746031744e-2f,
    1.5748031496062992e-2f, 7.8431372549019607e-3f, 3.9138943248532287e-3f,
    1.9550342130987292e-3f, 9.7703957010258913e-4f, 4.8840048840048840e-4f,
    2.4417043096081065e-4f, 1.2207776353537203e-4f, 6.1037018951994385e-5f,
    3.0518043793392844e-5f,
};

static const float at9_quant_step_fine[] = {
    3.0518043793392844e-05f, 1.0172681264464281e-05f, 4.3597205419132631e-06f,
    2.0345362528928561e-06f, 9.8445302559331759e-07f, 4.8441339354591809e-07f,
    2.4029955742829012e-07f, 1.1967860311134448e-07f, 5.9722199204291275e-08f,
    2.9831909866464167e-08f, 1.4908668194134265e-08f, 7.4525137468602791e-09f,
    3.7258019525568114e-09f, 1.8627872668859698e-09f, 9.3136520869755679e-10f,
    4.6567549848772173e-10f,
};

static const float at9_scalefactor_c[] = {
    3.0517578125e-5f, 6.1035156250e-5f, 1.2207031250e-4f, 2.4414062500e-4f,
    4.8828125000e-4f, 9.7656250000e-4f, 1.9531250000e-3f, 3.9062500000e-3f,
    7.8125000000e-3f, 1.5625000000e-2f, 3.1250000000e-2f, 6.2500000000e-2f,
    1.2500000000e-1f, 2.5000000000e-1f, 5.0000000000e-1f, 1.0000000000e+0f,
    2.0000000000e+0f, 4.0000000000e+0f, 8.0000000000e+0f, 1.6000000000e+1f,
    3.2000000000e+1f, 6.4000000000e+1f, 1.2800000000e+2f, 2.5600000000e+2f,
    5.1200000000e+2f, 1.0240000000e+3f, 2.0480000000e+3f, 4.0960000000e+3f,
    8.1920000000e+3f, 1.6384000000e+4f, 3.2768000000e+4f, 6.5536000000e+4f,
};

static const uint8_t at9_tab_sf_weights[][32] = {
    {
        0,  0,  0,  1,  1,  2,  2,  2,  2,  2,  2,  3,  2,  3,  3,  4,  4,  4,
        4,  4,  4,  5,  5,  6,  6,  7,  7,  8, 10, 12, 12, 12,
    },
    {
        3,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  1,  0,  1,
        1,  1,  1,  1,  1,  2,  3,  3,  4,  5,  7, 10, 10, 10,
    },
    {
        0,  2,  4,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
        6,  6,  6,  6,  6,  7,  7,  7,  7,  8,  9, 12, 12, 12,
    },
    {
        0,  1,  1,  2,  2,  2,  3,  3,  3,  3,  3,  4,  4,  4,  5,  5,  5,  6,
        6,  6,  6,  7,  8,  8, 10, 11, 11, 12, 13, 13, 13, 13,
    },
    {
        0,  2,  2,  3,  3,  4,  4,  5,  4,  5,  5,  5,  5,  6,  7,  8,  8,  8,
        8,  9,  9,  9, 10, 10, 11, 12, 12, 13, 13, 14, 14, 14,
    },
    {
        1,  1,  0,  0,  0,  0,  1,  0,  0,  1,  1,  1,  1,  1,  2,  2,  2,  2,
        2,  3,  3,  3,  4,  4,  5,  6,  7,  7,  9, 11, 11, 11,
    },
    {
        0,  5,  8, 10, 11, 11, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13,
        13, 13, 13, 13, 13, 13, 12, 12, 12, 12, 13, 15, 15, 15,
    },
    {
        0,  2,  3,  4,  5,  6,  6,  7,  7,  8,  8,  8,  9,  9, 10, 10, 10, 11,
        11, 11, 11, 11, 11, 12, 12, 12, 12, 13, 13, 15, 15, 15,
    },
};

static const uint8_t at9_tab_b_dist[] = {
         1,  1,  1,  1,  2,  2,  2,  2,  3,  3,  3,  4,  4,  5,  5,  6,  7,  8,
         9, 10, 11, 12, 13, 15,  6, 18, 19, 20, 21, 22, 23, 24, 25, 26, 26, 27,
        27, 28, 28, 28, 29, 29, 29, 29, 30, 30, 30, 30,
};

static const uint8_t huff_sfb_a1_bits[] = {
    1, 1,
};

static const uint16_t huff_sfb_a1_codes[] = {
    0x00, 0x01,
};

static const uint8_t huff_sfb_a2_bits[] = {
    1, 3, 3, 2,
};

static const uint16_t huff_sfb_a2_codes[] = {
    0x00, 0x06, 0x07, 0x02,
};

static const uint8_t huff_sfb_a3_bits[] = {
    2, 2, 4, 6, 6, 5, 3, 2,
};

static const uint16_t huff_sfb_a3_codes[] = {
    0x00, 0x01, 0x0E, 0x3E, 0x3F, 0x1E, 0x06, 0x02,
};

static const uint8_t huff_sfb_a4_bits[] = {
    2, 2, 4, 5, 6, 7, 8, 8, 8, 8, 8, 8, 6, 5, 4, 2,
};

static const uint16_t huff_sfb_a4_codes[] = {
    0x01, 0x02, 0x00, 0x06, 0x0F, 0x13, 0x23, 0x24,
    0x25, 0x22, 0x21, 0x20, 0x0E, 0x05, 0x01, 0x03,
};

static const uint8_t huff_sfb_a5_bits[] = {
    2, 3, 3, 4, 5, 5, 6, 7, 7, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 6, 5, 5, 4, 3,
};

static const uint16_t huff_sfb_a5_codes[] = {
    0x02, 0x01, 0x07, 0x0D, 0x0C, 0x18, 0x1B, 0x21, 0x3F, 0x6A, 0x6B, 0x68,
    0x73, 0x79, 0x7C, 0x7D, 0x7A, 0x7B, 0x78, 0x72, 0x44, 0x45, 0x47, 0x46,
    0x69, 0x38, 0x20, 0x1D, 0x19, 0x09, 0x05, 0x00,
};

static const uint8_t huff_sfb_a6_bits[] = {
    3, 3, 4, 4, 5, 5, 6, 6, 6, 7, 7, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 7, 7, 7, 6, 6, 5, 5, 5, 4, 4, 4,
};

static const uint16_t huff_sfb_a6_codes[] = {
    0x00, 0x01, 0x04, 0x05, 0x12, 0x13, 0x2E, 0x2F, 0x30, 0x66, 0x67, 0xD6,
    0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2,
    0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE,
    0xEF, 0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA,
    0xFB, 0xFC, 0xFD, 0xFE, 0xFF, 0x68, 0x69, 0x6A, 0x31, 0x32, 0x14, 0x15,
    0x16, 0x06, 0x07, 0x08,
};

static const uint8_t huff_sfb_b2_bits[] = {
    1, 2, 0, 2,
};

static const uint16_t huff_sfb_b2_codes[] = {
    0x00, 0x03, 0x00, 0x02,
};

static const uint8_t huff_sfb_b3_bits[] = {
    1, 3, 5, 6, 0, 6, 4, 2,
};

static const uint16_t huff_sfb_b3_codes[] = {
    0x01, 0x00, 0x04, 0x0B, 0x00, 0x0A, 0x03, 0x01,
};

static const uint8_t huff_sfb_b4_bits[] = {
    1, 3, 4, 5, 5, 7, 8, 8, 0, 8, 8, 7, 6, 6, 4, 3,
};

static const uint16_t huff_sfb_b4_codes[] = {
    0x01, 0x01, 0x04, 0x0E, 0x0F, 0x2C, 0x5A, 0x5D, 0x00, 0x5C, 0x5B, 0x2F,
    0x15, 0x14, 0x06, 0x00,
};

static const uint8_t huff_sfb_b5_bits[] = {
    3, 3, 4, 4, 4, 4, 4, 4, 4, 5, 6, 7, 7, 7, 8, 8,
    8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 6, 3,
};

static const uint16_t huff_sfb_b5_codes[] = {
    0x00, 0x05, 0x07, 0x0C, 0x04, 0x02, 0x03, 0x05, 0x09, 0x10, 0x23, 0x33,
    0x36, 0x6E, 0x60, 0x65, 0x62, 0x61, 0x63, 0x64, 0x6F, 0x6D, 0x6C, 0x6B,
    0x6A, 0x68, 0x69, 0x45, 0x44, 0x37, 0x1A, 0x07,
};

typedef struct HuffmanCodebook {
    const uint8_t *bits;
    const uint16_t *codes;
    const int size;
    const int value_cnt;
    const int value_cnt_pow;
    const int value_bits;
    const int max_bit_size;
} HuffmanCodebook;

static const HuffmanCodebook at9_huffman_sf_unsigned[] = {
    { 0 },
    { huff_sfb_a1_bits, huff_sfb_a1_codes,  2,  1,  0,  1,  1, },
    { huff_sfb_a2_bits, huff_sfb_a2_codes,  4,  1,  0,  2,  3, },
    { huff_sfb_a3_bits, huff_sfb_a3_codes,  8,  1,  0,  3,  6, },
    { huff_sfb_a4_bits, huff_sfb_a4_codes, 16,  1,  0,  4,  8, },
    { huff_sfb_a5_bits, huff_sfb_a5_codes, 32,  1,  0,  5,  8, },
    { huff_sfb_a6_bits, huff_sfb_a6_codes, 64,  1,  0,  6,  8, },
};

static const HuffmanCodebook at9_huffman_sf_signed[] = {
    { 0 },
    { 0 },
    { huff_sfb_b2_bits, huff_sfb_b2_codes,  4,  1,  0,  2,  2, },
    { huff_sfb_b3_bits, huff_sfb_b3_codes,  8,  1,  0,  3,  6, },
    { huff_sfb_b4_bits, huff_sfb_b4_codes, 16,  1,  0,  4,  8, },
    { huff_sfb_b5_bits, huff_sfb_b5_codes, 32,  1,  0,  5,  8, },
};

static const uint8_t huff_spec_a21_bits[] = {
    0, 3, 0, 3, 3, 3, 0, 3, 0, 0, 0, 0, 3, 3, 0, 3,
};

static const uint16_t huff_spec_a21_codes[] = {
    0x00, 0x00, 0x00, 0x01, 0x03, 0x07, 0x00, 0x04,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x05, 0x00, 0x06,
};

static const uint8_t huff_spec_a22_bits[] = {
    0, 4, 0, 4, 5, 6, 0, 6, 0, 0, 0, 0, 5, 6, 0, 6,
    5, 6, 0, 6, 6, 7, 0, 7, 0, 0, 0, 0, 6, 7, 0, 7,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    5, 6, 0, 6, 6, 7, 0, 7, 0, 0, 0, 0, 6, 7, 0, 7,
    5, 6, 0, 6, 7, 7, 0, 7, 0, 0, 0, 0, 6, 7, 0, 7,
    6, 7, 0, 7, 7, 8, 0, 8, 0, 0, 0, 0, 7, 8, 0, 7,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    6, 7, 0, 7, 7, 8, 0, 8, 0, 0, 0, 0, 7, 7, 0, 8,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    5, 6, 0, 6, 6, 7, 0, 7, 0, 0, 0, 0, 7, 7, 0, 7,
    6, 7, 0, 7, 7, 8, 0, 7, 0, 0, 0, 0, 7, 8, 0, 8,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    6, 7, 0, 7, 7, 7, 0, 8, 0, 0, 0, 0, 7, 8, 0, 8,
};

static const uint16_t huff_spec_a22_codes[] = {
    0x00, 0x02, 0x00, 0x03, 0x10, 0x3C, 0x00, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x11, 0x3E, 0x00, 0x3D,
    0x0E, 0x00, 0x00, 0x39, 0x18, 0x26, 0x00, 0x75, 0x00, 0x00, 0x00, 0x00, 0x1B, 0x24, 0x00, 0x6D,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0F, 0x38, 0x00, 0x01, 0x1A, 0x6C, 0x00, 0x25, 0x00, 0x00, 0x00, 0x00, 0x19, 0x74, 0x00, 0x27,
    0x16, 0x14, 0x00, 0x17, 0x76, 0x06, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x35, 0x64, 0x00, 0x6F,
    0x26, 0x04, 0x00, 0x63, 0x22, 0xA2, 0x00, 0x97, 0x00, 0x00, 0x00, 0x00, 0x67, 0xA0, 0x00, 0x0D,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x2B, 0x52, 0x00, 0x0B, 0x20, 0x92, 0x00, 0x91, 0x00, 0x00, 0x00, 0x00, 0x61, 0x0E, 0x00, 0x95,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x17, 0x16, 0x00, 0x15, 0x34, 0x6E, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x77, 0x08, 0x00, 0x07,
    0x2A, 0x0A, 0x00, 0x53, 0x60, 0x94, 0x00, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x21, 0x90, 0x00, 0x93,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x27, 0x62, 0x00, 0x05, 0x66, 0x0C, 0x00, 0xA1, 0x00, 0x00, 0x00, 0x00, 0x23, 0x96, 0x00, 0xA3,
};

static const uint8_t huff_spec_a23_bits[] = {
    3, 4, 0, 4, 5, 6, 0, 6, 0, 0, 0, 0, 5, 6, 0, 6,
    5, 7, 0, 6, 6, 8, 0, 7, 0, 0, 0, 0, 6, 8, 0, 7,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    5, 6, 0, 7, 6, 7, 0, 8, 0, 0, 0, 0, 6, 7, 0, 8,
    5, 6, 0, 6, 7, 8, 0, 8, 0, 0, 0, 0, 6, 7, 0, 7,
    6, 8, 0, 7, 8, 9, 0, 9, 0, 0, 0, 0, 7, 9, 0, 8,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    6, 8, 0, 8, 8, 9, 0, 9, 0, 0, 0, 0, 7, 8, 0, 9,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    5, 6, 0, 6, 6, 7, 0, 7, 0, 0, 0, 0, 7, 8, 0, 8,
    6, 8, 0, 8, 7, 9, 0, 8, 0, 0, 0, 0, 8, 9, 0, 9,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    6, 7, 0, 8, 7, 8, 0, 9, 0, 0, 0, 0, 8, 9, 0, 9,
};

static const uint16_t huff_spec_a23_codes[] = {
    0x006, 0x002, 0x000, 0x003, 0x016, 0x01E, 0x000, 0x021, 0x000, 0x000, 0x000, 0x000,
    0x017, 0x020, 0x000, 0x01F, 0x01C, 0x054, 0x000, 0x027, 0x010, 0x0A6, 0x000, 0x027,
    0x000, 0x000, 0x000, 0x000, 0x015, 0x0A4, 0x000, 0x02D, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x01D, 0x026, 0x000, 0x055, 0x014, 0x02C, 0x000, 0x0A5, 0x000, 0x000, 0x000, 0x000,
    0x011, 0x026, 0x000, 0x0A7, 0x01E, 0x000, 0x000, 0x003, 0x04A, 0x074, 0x000, 0x071,
    0x000, 0x000, 0x000, 0x000, 0x023, 0x00A, 0x000, 0x009, 0x018, 0x072, 0x000, 0x00D,
    0x0A2, 0x15A, 0x000, 0x123, 0x000, 0x000, 0x000, 0x000, 0x00F, 0x158, 0x000, 0x05D,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x01B, 0x0AE, 0x000, 0x077, 0x092, 0x140, 0x000, 0x121,
    0x000, 0x000, 0x000, 0x000, 0x025, 0x05E, 0x000, 0x143, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x01F, 0x002, 0x000, 0x001, 0x022, 0x008, 0x000, 0x00B, 0x000, 0x000, 0x000, 0x000,
    0x04B, 0x070, 0x000, 0x075, 0x01A, 0x076, 0x000, 0x0AF, 0x024, 0x142, 0x000, 0x05F,
    0x000, 0x000, 0x000, 0x000, 0x093, 0x120, 0x000, 0x141, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x019, 0x00C, 0x000, 0x073, 0x00E, 0x05C, 0x000, 0x159, 0x000, 0x000, 0x000, 0x000,
    0x0A3, 0x122, 0x000, 0x15B,
};

static const uint8_t huff_spec_a24_bits[] = {
    2,  4,  0,  4,  5,  6,  0,  6,  0,  0,  0,  0,  5,  6,  0,  6,
    5,  7,  0,  6,  6,  8,  0,  8,  0,  0,  0,  0,  6,  8,  0,  8,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    5,  6,  0,  7,  6,  8,  0,  8,  0,  0,  0,  0,  6,  8,  0,  8,
    5,  7,  0,  7,  7,  9,  0,  9,  0,  0,  0,  0,  6,  8,  0,  8,
    6,  9,  0,  8,  8, 10,  0, 10,  0,  0,  0,  0,  8, 10,  0,  9,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    6,  8,  0,  9,  9, 10,  0, 10,  0,  0,  0,  0,  8,  9,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    5,  7,  0,  7,  6,  8,  0,  8,  0,  0,  0,  0,  7,  9,  0,  9,
    6,  9,  0,  8,  8, 10,  0,  9,  0,  0,  0,  0,  9, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    6,  8,  0,  9,  8,  9,  0, 10,  0,  0,  0,  0,  8, 10,  0, 10,
};

static const uint16_t huff_spec_a24_codes[] = {
    0x002, 0x002, 0x000, 0x003, 0x01E, 0x010, 0x000, 0x013, 0x000, 0x000, 0x000, 0x000,
    0x01F, 0x012, 0x000, 0x011, 0x01A, 0x030, 0x000, 0x01B, 0x000, 0x064, 0x000, 0x0C1,
    0x000, 0x000, 0x000, 0x000, 0x003, 0x052, 0x000, 0x07D, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x01B, 0x01A, 0x000, 0x031, 0x002, 0x07C, 0x000, 0x053, 0x000, 0x000, 0x000, 0x000,
    0x001, 0x0C0, 0x000, 0x065, 0x01C, 0x062, 0x000, 0x065, 0x02A, 0x198, 0x000, 0x19B,
    0x000, 0x000, 0x000, 0x000, 0x017, 0x078, 0x000, 0x07B, 0x004, 0x0FE, 0x000, 0x077,
    0x050, 0x33A, 0x000, 0x1F9, 0x000, 0x000, 0x000, 0x000, 0x073, 0x338, 0x000, 0x0E1,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x007, 0x066, 0x000, 0x187, 0x19E, 0x308, 0x000, 0x30B,
    0x000, 0x000, 0x000, 0x000, 0x075, 0x0E2, 0x000, 0x1FB, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x01D, 0x064, 0x000, 0x063, 0x016, 0x07A, 0x000, 0x079, 0x000, 0x000, 0x000, 0x000,
    0x02B, 0x19A, 0x000, 0x199, 0x006, 0x186, 0x000, 0x067, 0x074, 0x1FA, 0x000, 0x0E3,
    0x000, 0x000, 0x000, 0x000, 0x19F, 0x30A, 0x000, 0x309, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x005, 0x076, 0x000, 0x0FF, 0x072, 0x0E0, 0x000, 0x339, 0x000, 0x000, 0x000, 0x000,
    0x051, 0x1F8, 0x000, 0x33B,
};

static const uint8_t huff_spec_a31_bits[] = {
    0, 0, 4, 5, 0, 5, 4, 0, 0, 0, 5, 5, 0, 5, 5, 0,
    5, 5, 6, 6, 0, 6, 5, 5, 5, 6, 6, 7, 0, 7, 6, 6,
    0, 0, 0, 0, 0, 0, 0, 0, 5, 6, 6, 7, 0, 7, 6, 6,
    5, 5, 5, 6, 0, 6, 6, 5, 0, 0, 5, 5, 0, 5, 5, 0,
};

static const uint16_t huff_spec_a31_codes[] = {
    0x00, 0x00, 0x02, 0x18, 0x00, 0x19, 0x03, 0x00, 0x00, 0x00, 0x12, 0x02, 0x00, 0x09, 0x15, 0x00,
    0x1A, 0x0A, 0x3E, 0x2C, 0x00, 0x2F, 0x01, 0x0D, 0x0E, 0x38, 0x20, 0x78, 0x00, 0x7B, 0x23, 0x3B,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x3A, 0x22, 0x7A, 0x00, 0x79, 0x21, 0x39,
    0x1B, 0x0C, 0x00, 0x2E, 0x00, 0x2D, 0x3F, 0x0B, 0x00, 0x00, 0x14, 0x08, 0x00, 0x03, 0x13, 0x00,
};

static const uint8_t huff_spec_a32_bits[] = {
    4, 5, 5, 6, 0, 6, 5, 5, 5, 6, 5, 6, 0, 6, 5, 5,
    5, 5, 6, 7, 0, 7, 6, 5, 6, 6, 7, 7, 0, 7, 7, 6,
    0, 0, 0, 0, 0, 0, 0, 0, 6, 6, 7, 7, 0, 7, 7, 6,
    5, 5, 6, 7, 0, 7, 6, 5, 5, 5, 5, 6, 0, 6, 5, 6,
};

static const uint16_t huff_spec_a32_codes[] = {
    0x0D, 0x18, 0x16, 0x3A, 0x00, 0x3B, 0x17, 0x19, 0x12, 0x3E, 0x08, 0x1C, 0x00, 0x1B, 0x07, 0x01,
    0x10, 0x02, 0x28, 0x78, 0x00, 0x7B, 0x1F, 0x05, 0x2A, 0x16, 0x72, 0x2A, 0x00, 0x29, 0x71, 0x19,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2B, 0x18, 0x70, 0x28, 0x00, 0x2B, 0x73, 0x17,
    0x11, 0x04, 0x1E, 0x7A, 0x00, 0x79, 0x29, 0x03, 0x13, 0x00, 0x06, 0x1A, 0x00, 0x1D, 0x09, 0x3F,
};

static const uint8_t huff_spec_a33_bits[] = {
    3, 4, 5, 6, 0, 6, 5, 4, 4, 5, 6, 7, 0, 7, 6, 5,
    5, 6, 6, 7, 0, 7, 6, 6, 6, 7, 8, 8, 0, 8, 8, 7,
    0, 0, 0, 0, 0, 0, 0, 0, 6, 7, 8, 8, 0, 8, 8, 7,
    5, 6, 6, 7, 0, 7, 6, 6, 4, 5, 6, 7, 0, 7, 6, 5,
};

static const uint16_t huff_spec_a33_codes[] = {
    0x05, 0x06, 0x10, 0x08, 0x00, 0x09, 0x11, 0x07, 0x04, 0x12, 0x3E, 0x6A, 0x00, 0x6D, 0x3D, 0x19,
    0x06, 0x3A, 0x06, 0x02, 0x00, 0x01, 0x05, 0x39, 0x02, 0x16, 0xDC, 0x2A, 0x00, 0x29, 0xDF, 0x69,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x68, 0xDE, 0x28, 0x00, 0x2B, 0xDD, 0x17,
    0x07, 0x38, 0x04, 0x00, 0x00, 0x03, 0x07, 0x3B, 0x05, 0x18, 0x3C, 0x6C, 0x00, 0x6B, 0x3F, 0x13,
};

static const uint8_t huff_spec_a34_bits[] = {
    2,  4,  5,  7,  0,  7,  5,  4,  4,  5,  6,  8,  0,  8,  6,  5,
    5,  6,  7,  8,  0,  8,  7,  6,  7,  8,  8, 10,  0, 10,  9,  8,
    0,  0,  0,  0,  0,  0,  0,  0,  7,  8,  9, 10,  0, 10,  8,  8,
    5,  6,  7,  8,  0,  8,  7,  6,  4,  5,  6,  8,  0,  8,  6,  5,
};

static const uint16_t huff_spec_a34_codes[] = {
    0x000, 0x00A, 0x00A, 0x034, 0x000, 0x035, 0x00B, 0x00B, 0x008, 0x01C, 0x032, 0x0DA,
    0x000, 0x0DD, 0x035, 0x01F, 0x008, 0x01E, 0x03A, 0x06C, 0x000, 0x063, 0x039, 0x031,
    0x032, 0x06E, 0x060, 0x37A, 0x000, 0x379, 0x1BF, 0x0D9, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x033, 0x0D8, 0x1BE, 0x378, 0x000, 0x37B, 0x061, 0x06F,
    0x009, 0x030, 0x038, 0x062, 0x000, 0x06D, 0x03B, 0x01F, 0x009, 0x01E, 0x034, 0x0DC,
    0x000, 0x0DB, 0x033, 0x01D,
};

static const uint8_t huff_spec_a41_bits[] = {
    0, 0, 0, 0, 6, 6, 7, 7, 0, 7, 7, 6, 6, 0, 0, 0,
    0, 0, 0, 0, 7, 7, 7, 7, 0, 7, 7, 7, 6, 0, 0, 0,
    0, 0, 0, 0, 7, 7, 7, 8, 0, 8, 7, 7, 7, 0, 0, 0,
    0, 0, 0, 0, 7, 7, 8, 8, 0, 8, 8, 7, 7, 0, 0, 0,
    7, 7, 7, 8, 7, 8, 8, 8, 0, 8, 8, 8, 7, 8, 7, 7,
    7, 7, 7, 7, 8, 8, 8, 9, 0, 8, 8, 8, 8, 7, 7, 7,
    7, 7, 8, 8, 8, 8, 9, 9, 0, 9, 8, 8, 8, 8, 8, 7,
    8, 8, 8, 8, 8, 9, 9, 9, 0, 9, 9, 9, 8, 8, 8, 8,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    8, 8, 8, 8, 8, 9, 9, 9, 0, 9, 9, 9, 8, 8, 8, 8,
    7, 7, 8, 8, 8, 8, 8, 9, 0, 9, 9, 8, 8, 8, 8, 7,
    7, 7, 7, 7, 8, 8, 8, 8, 0, 9, 8, 8, 8, 7, 7, 7,
    7, 7, 7, 8, 7, 8, 8, 8, 0, 8, 8, 8, 7, 8, 7, 7,
    0, 0, 0, 0, 7, 7, 8, 8, 0, 8, 8, 7, 7, 0, 0, 0,
    0, 0, 0, 0, 7, 7, 7, 8, 0, 8, 7, 7, 7, 0, 0, 0,
    0, 0, 0, 0, 6, 7, 7, 7, 0, 7, 7, 7, 7, 0, 0, 0,
};

static const uint16_t huff_spec_a41_codes[] = {
    0x000, 0x000, 0x000, 0x000, 0x018, 0x00E, 0x05E, 0x028, 0x000, 0x029, 0x05F, 0x00F,
    0x019, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x076, 0x06E, 0x03E, 0x004,
    0x000, 0x017, 0x045, 0x07B, 0x013, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x04A, 0x048, 0x010, 0x0CE, 0x000, 0x0E1, 0x023, 0x055, 0x053, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x008, 0x018, 0x0D6, 0x09E, 0x000, 0x09D, 0x0E5, 0x02B,
    0x01B, 0x000, 0x000, 0x000, 0x07C, 0x05C, 0x038, 0x0FC, 0x002, 0x0D2, 0x09A, 0x05C,
    0x000, 0x06B, 0x0A3, 0x0D9, 0x00F, 0x0FF, 0x03D, 0x061, 0x074, 0x056, 0x036, 0x000,
    0x0CC, 0x08C, 0x058, 0x1E2, 0x000, 0x00F, 0x05F, 0x0A1, 0x0D5, 0x00D, 0x03B, 0x059,
    0x040, 0x014, 0x0DA, 0x0B6, 0x084, 0x040, 0x1E0, 0x196, 0x000, 0x1A1, 0x00D, 0x043,
    0x087, 0x0C7, 0x0E3, 0x00B, 0x0F2, 0x0C4, 0x08E, 0x05A, 0x024, 0x1CC, 0x194, 0x168,
    0x000, 0x16B, 0x1A3, 0x1CF, 0x027, 0x069, 0x099, 0x0C9, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x0F3, 0x0C8, 0x098, 0x068, 0x026, 0x1CE, 0x1A2, 0x16A, 0x000, 0x169, 0x195, 0x1CD,
    0x025, 0x05B, 0x08F, 0x0C5, 0x041, 0x00A, 0x0E2, 0x0C6, 0x086, 0x042, 0x00C, 0x1A0,
    0x000, 0x197, 0x1E1, 0x041, 0x085, 0x0B7, 0x0DB, 0x015, 0x075, 0x058, 0x03A, 0x00C,
    0x0D4, 0x0A0, 0x05E, 0x00E, 0x000, 0x1E3, 0x059, 0x08D, 0x0CD, 0x001, 0x037, 0x057,
    0x07D, 0x060, 0x03C, 0x0FE, 0x00E, 0x0D8, 0x0A2, 0x06A, 0x000, 0x05D, 0x09B, 0x0D3,
    0x003, 0x0FD, 0x039, 0x05D, 0x000, 0x000, 0x000, 0x000, 0x01A, 0x02A, 0x0E4, 0x09C,
    0x000, 0x09F, 0x0D7, 0x019, 0x009, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x052, 0x054, 0x022, 0x0E0, 0x000, 0x0CF, 0x011, 0x049, 0x04B, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x012, 0x07A, 0x044, 0x016, 0x000, 0x005, 0x03F, 0x06F,
    0x077, 0x000, 0x000, 0x000,
};

static const uint8_t huff_spec_a42_bits[] = {
    5,  6,  7,  7,  7,  7,  8,  8,  0,  8,  8,  7,  7,  7,  7,  6,
    6,  7,  7,  8,  7,  7,  8,  8,  0,  8,  8,  7,  7,  8,  7,  7,
    7,  7,  8,  8,  7,  8,  8,  9,  0,  9,  8,  8,  7,  8,  8,  7,
    8,  8,  8,  8,  8,  8,  8,  9,  0,  9,  8,  8,  8,  8,  8,  8,
    7,  7,  7,  8,  8,  8,  9,  9,  0,  9,  9,  8,  8,  8,  7,  7,
    7,  7,  8,  8,  8,  9,  9,  9,  0,  9,  9,  9,  8,  8,  8,  7,
    8,  8,  8,  8,  9,  9,  9, 10,  0, 10,  9,  9,  9,  8,  8,  8,
    8,  8,  9,  9,  9,  9, 10, 10,  0, 10, 10,  9,  9,  9,  9,  9,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    8,  9,  9,  9,  9,  9, 10, 10,  0, 10, 10,  9,  9,  9,  9,  8,
    8,  8,  8,  8,  9,  9,  9, 10,  0, 10,  9,  9,  9,  8,  8,  8,
    7,  7,  8,  8,  8,  9,  9,  9,  0,  9,  9,  9,  8,  8,  8,  7,
    7,  7,  7,  8,  8,  8,  9,  9,  0,  9,  9,  8,  8,  8,  7,  7,
    8,  8,  8,  8,  8,  8,  8,  9,  0,  9,  8,  8,  8,  8,  8,  8,
    7,  7,  8,  8,  7,  8,  8,  9,  0,  9,  8,  8,  7,  8,  8,  7,
    6,  7,  7,  8,  7,  7,  8,  8,  0,  8,  8,  7,  7,  8,  7,  7,
};

static const uint16_t huff_spec_a42_codes[] = {
    0x003, 0x018, 0x058, 0x000, 0x066, 0x03C, 0x0D6, 0x07C, 0x000, 0x07D, 0x0D7, 0x03D,
    0x067, 0x001, 0x059, 0x019, 0x002, 0x064, 0x036, 0x0DA, 0x04C, 0x01C, 0x0BE, 0x02C,
    0x000, 0x037, 0x0C5, 0x029, 0x04B, 0x0E7, 0x03B, 0x069, 0x044, 0x02E, 0x0FA, 0x092,
    0x020, 0x0F8, 0x086, 0x1FC, 0x000, 0x1E7, 0x07F, 0x0F5, 0x023, 0x0AD, 0x0FD, 0x02D,
    0x0F6, 0x0DC, 0x09C, 0x03E, 0x0F0, 0x0B6, 0x026, 0x186, 0x000, 0x18D, 0x02F, 0x0B5,
    0x0E1, 0x03D, 0x0AF, 0x0D9, 0x054, 0x040, 0x014, 0x0EC, 0x0BC, 0x054, 0x1C6, 0x108,
    0x000, 0x10B, 0x1C5, 0x069, 0x0B9, 0x0DF, 0x019, 0x047, 0x026, 0x008, 0x0E4, 0x0A2,
    0x056, 0x1DC, 0x142, 0x06A, 0x000, 0x091, 0x123, 0x1DF, 0x04B, 0x0A7, 0x0EB, 0x00B,
    0x0C0, 0x09E, 0x06A, 0x022, 0x1AA, 0x140, 0x092, 0x3CA, 0x000, 0x3A7, 0x04B, 0x121,
    0x18F, 0x007, 0x071, 0x0A5, 0x020, 0x004, 0x1A8, 0x174, 0x0E4, 0x068, 0x3A4, 0x2EE,
    0x000, 0x2ED, 0x3C9, 0x049, 0x0E7, 0x185, 0x1D1, 0x1FF, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x021, 0x1FE, 0x1D0, 0x184, 0x0E6, 0x048, 0x3C8, 0x2EC, 0x000, 0x2EF, 0x3A5, 0x069,
    0x0E5, 0x175, 0x1A9, 0x005, 0x0C1, 0x0A4, 0x070, 0x006, 0x18E, 0x120, 0x04A, 0x3A6,
    0x000, 0x3CB, 0x093, 0x141, 0x1AB, 0x023, 0x06B, 0x09F, 0x027, 0x00A, 0x0EA, 0x0A6,
    0x04A, 0x1DE, 0x122, 0x090, 0x000, 0x06B, 0x143, 0x1DD, 0x057, 0x0A3, 0x0E5, 0x009,
    0x055, 0x046, 0x018, 0x0DE, 0x0B8, 0x068, 0x1C4, 0x10A, 0x000, 0x109, 0x1C7, 0x055,
    0x0BD, 0x0ED, 0x015, 0x041, 0x0F7, 0x0D8, 0x0AE, 0x03C, 0x0E0, 0x0B4, 0x02E, 0x18C,
    0x000, 0x187, 0x027, 0x0B7, 0x0F1, 0x03F, 0x09D, 0x0DD, 0x045, 0x02C, 0x0FC, 0x0AC,
    0x022, 0x0F4, 0x07E, 0x1E6, 0x000, 0x1FD, 0x087, 0x0F9, 0x021, 0x093, 0x0FB, 0x02F,
    0x003, 0x068, 0x03A, 0x0E6, 0x04A, 0x028, 0x0C4, 0x036, 0x000, 0x02D, 0x0BF, 0x01D,
    0x04D, 0x0DB, 0x037, 0x065,
};

static const uint8_t huff_spec_a43_bits[] = {
    4,  6,  6,  7,  7,  8,  8,  9,  0,  9,  8,  8,  7,  7,  6,  6,
    5,  6,  7,  7,  7,  8,  8,  9,  0,  9,  8,  8,  7,  7,  7,  6,
    6,  7,  7,  7,  8,  8,  9,  9,  0,  9,  9,  8,  8,  7,  7,  7,
    7,  7,  7,  8,  8,  8,  9, 10,  0, 10,  9,  9,  8,  8,  7,  7,
    7,  7,  8,  8,  8,  9, 10, 10,  0, 10, 10,  9,  8,  8,  8,  7,
    8,  8,  8,  9,  9,  9, 10, 10,  0, 10, 10,  9,  9,  9,  8,  8,
    8,  9,  9,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  9,  9,
    9,  9, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  9,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    9,  9, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  9,
    8,  9,  9,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  9,  9,
    8,  8,  8,  9,  9,  9, 10, 10,  0, 10, 10,  9,  9,  9,  8,  8,
    7,  7,  8,  8,  8,  9, 10, 10,  0, 10, 10,  9,  8,  8,  8,  7,
    7,  7,  7,  8,  8,  9,  9, 10,  0, 10,  9,  8,  8,  8,  7,  7,
    6,  7,  7,  7,  8,  8,  9,  9,  0,  9,  9,  8,  8,  7,  7,  7,
    5,  6,  7,  7,  7,  8,  8,  9,  0,  9,  8,  8,  7,  7,  7,  6,
};

static const uint16_t huff_spec_a43_codes[] = {
    0x002, 0x03E, 0x016, 0x060, 0x04E, 0x0DC, 0x04A, 0x130, 0x000, 0x131, 0x04B, 0x0DD,
    0x04F, 0x061, 0x017, 0x03F, 0x002, 0x02C, 0x076, 0x042, 0x034, 0x0CE, 0x002, 0x0E8,
    0x000, 0x0CF, 0x001, 0x0D1, 0x037, 0x045, 0x07B, 0x02F, 0x014, 0x072, 0x052, 0x01A,
    0x0E0, 0x080, 0x198, 0x01E, 0x000, 0x01D, 0x19B, 0x083, 0x0DF, 0x019, 0x055, 0x079,
    0x050, 0x03C, 0x004, 0x0C4, 0x096, 0x00C, 0x0EA, 0x34A, 0x000, 0x34F, 0x0ED, 0x1D7,
    0x095, 0x0AF, 0x003, 0x03F, 0x046, 0x026, 0x0D6, 0x092, 0x046, 0x15A, 0x3A8, 0x108,
    0x000, 0x10F, 0x3A3, 0x135, 0x039, 0x091, 0x0D9, 0x031, 0x0D4, 0x0CA, 0x072, 0x1C6,
    0x136, 0x090, 0x2B2, 0x104, 0x000, 0x103, 0x111, 0x08B, 0x133, 0x1D3, 0x071, 0x0C9,
    0x03E, 0x1B4, 0x18C, 0x0CC, 0x38A, 0x2B0, 0x106, 0x0F2, 0x000, 0x0EF, 0x101, 0x113,
    0x3A1, 0x0CB, 0x18F, 0x1B7, 0x0EE, 0x092, 0x388, 0x348, 0x10A, 0x0F4, 0x0F0, 0x0EA,
    0x000, 0x0E9, 0x0ED, 0x0F7, 0x10D, 0x34D, 0x3AB, 0x0C9, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x0EF, 0x0C8, 0x3AA, 0x34C, 0x10C, 0x0F6, 0x0EC, 0x0E8, 0x000, 0x0EB, 0x0F1, 0x0F5,
    0x10B, 0x349, 0x389, 0x093, 0x03F, 0x1B6, 0x18E, 0x0CA, 0x3A0, 0x112, 0x100, 0x0EE,
    0x000, 0x0F3, 0x107, 0x2B1, 0x38B, 0x0CD, 0x18D, 0x1B5, 0x0D5, 0x0C8, 0x070, 0x1D2,
    0x132, 0x08A, 0x110, 0x102, 0x000, 0x105, 0x2B3, 0x091, 0x137, 0x1C7, 0x073, 0x0CB,
    0x047, 0x030, 0x0D8, 0x090, 0x038, 0x134, 0x3A2, 0x10E, 0x000, 0x109, 0x3A9, 0x15B,
    0x047, 0x093, 0x0D7, 0x027, 0x051, 0x03E, 0x002, 0x0AE, 0x094, 0x1D6, 0x0EC, 0x34E,
    0x000, 0x34B, 0x0EB, 0x00D, 0x097, 0x0C5, 0x005, 0x03D, 0x015, 0x078, 0x054, 0x018,
    0x0DE, 0x082, 0x19A, 0x01C, 0x000, 0x01F, 0x199, 0x081, 0x0E1, 0x01B, 0x053, 0x073,
    0x003, 0x02E, 0x07A, 0x044, 0x036, 0x0D0, 0x000, 0x0CE, 0x000, 0x0E9, 0x003, 0x0CF,
    0x035, 0x043, 0x077, 0x02D,
};

static const uint8_t huff_spec_a44_bits[] = {
    4,  5,  6,  7,  7,  8,  9, 10,  0, 10,  9,  8,  7,  7,  6,  5,
    5,  6,  6,  7,  7,  8,  9, 10,  0, 10,  9,  8,  7,  7,  6,  6,
    6,  6,  7,  7,  8,  9, 10, 10,  0, 10, 10,  9,  8,  7,  7,  6,
    7,  7,  7,  8,  8,  9, 10, 10,  0, 10, 10,  9,  8,  8,  7,  7,
    7,  8,  8,  8,  9, 10, 10, 10,  0, 10, 10, 10,  9,  8,  8,  7,
    8,  8,  9,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  9,  8,
    9,  9, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  9,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    9,  9, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  9,
    8,  8,  9,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  9,  8,
    7,  7,  8,  8,  9, 10, 10, 10,  0, 10, 10, 10,  9,  8,  8,  8,
    7,  7,  7,  8,  8,  9, 10, 10,  0, 10, 10,  9,  8,  8,  7,  7,
    6,  6,  7,  7,  8,  9, 10, 10,  0, 10, 10,  9,  8,  7,  7,  6,
    5,  6,  6,  7,  7,  8,  9, 10,  0, 10,  9,  8,  7,  7,  6,  6,
};

static const uint16_t huff_spec_a44_codes[] = {
    0x00A, 0x012, 0x030, 0x06E, 0x024, 0x074, 0x0EC, 0x07E, 0x000, 0x07F, 0x0ED, 0x075,
    0x025, 0x06F, 0x031, 0x013, 0x010, 0x03C, 0x018, 0x05A, 0x002, 0x046, 0x09E, 0x07C,
    0x000, 0x079, 0x0E5, 0x04D, 0x007, 0x065, 0x01B, 0x03F, 0x02E, 0x016, 0x072, 0x01A,
    0x0D6, 0x1C6, 0x3B4, 0x066, 0x000, 0x06B, 0x3B7, 0x1D9, 0x0D5, 0x021, 0x075, 0x015,
    0x06C, 0x03E, 0x01E, 0x0CC, 0x044, 0x0F2, 0x082, 0x05C, 0x000, 0x05F, 0x087, 0x0F5,
    0x031, 0x0CF, 0x017, 0x059, 0x01C, 0x0EE, 0x0D0, 0x024, 0x1C0, 0x08E, 0x06E, 0x048,
    0x000, 0x04D, 0x06D, 0x089, 0x0F7, 0x033, 0x0D3, 0x001, 0x070, 0x028, 0x1C2, 0x0F0,
    0x08A, 0x074, 0x054, 0x040, 0x000, 0x043, 0x053, 0x073, 0x099, 0x0EF, 0x1C5, 0x02B,
    0x0E6, 0x04E, 0x08C, 0x080, 0x068, 0x058, 0x046, 0x02A, 0x000, 0x029, 0x045, 0x051,
    0x065, 0x085, 0x09B, 0x09D, 0x07A, 0x076, 0x060, 0x056, 0x04E, 0x02C, 0x024, 0x022,
    0x000, 0x021, 0x027, 0x02F, 0x04B, 0x05B, 0x063, 0x071, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x07B, 0x070, 0x062, 0x05A, 0x04A, 0x02E, 0x026, 0x020, 0x000, 0x023, 0x025, 0x02D,
    0x04F, 0x057, 0x061, 0x077, 0x0E7, 0x09C, 0x09A, 0x084, 0x064, 0x050, 0x044, 0x028,
    0x000, 0x02B, 0x047, 0x059, 0x069, 0x081, 0x08D, 0x04F, 0x071, 0x02A, 0x1C4, 0x0EE,
    0x098, 0x072, 0x052, 0x042, 0x000, 0x041, 0x055, 0x075, 0x08B, 0x0F1, 0x1C3, 0x029,
    0x01D, 0x000, 0x0D2, 0x032, 0x0F6, 0x088, 0x06C, 0x04C, 0x000, 0x049, 0x06F, 0x08F,
    0x1C1, 0x025, 0x0D1, 0x0EF, 0x06D, 0x058, 0x016, 0x0CE, 0x030, 0x0F4, 0x086, 0x05E,
    0x000, 0x05D, 0x083, 0x0F3, 0x045, 0x0CD, 0x01F, 0x03F, 0x02F, 0x014, 0x074, 0x020,
    0x0D4, 0x1D8, 0x3B6, 0x06A, 0x000, 0x067, 0x3B5, 0x1C7, 0x0D7, 0x01B, 0x073, 0x017,
    0x011, 0x03E, 0x01A, 0x064, 0x006, 0x04C, 0x0E4, 0x078, 0x000, 0x07D, 0x09F, 0x047,
    0x003, 0x05B, 0x019, 0x03D,
};

static const uint8_t huff_spec_a51_bits[] = {
    5, 5, 5, 5, 5, 6, 6, 6, 4, 4, 5, 5, 5, 5, 5, 5,
    0, 5, 5, 5, 5, 5, 5, 4, 4, 6, 6, 6, 5, 5, 5, 5,
};

static const uint16_t huff_spec_a51_codes[] = {
    0x19, 0x16, 0x12, 0x0E, 0x06, 0x3A, 0x38, 0x30, 0x00, 0x04, 0x1E, 0x1A,
    0x14, 0x10, 0x0C, 0x04, 0x00, 0x05, 0x0D, 0x11, 0x15, 0x1B, 0x1F, 0x05,
    0x01, 0x31, 0x39, 0x3B, 0x07, 0x0F, 0x13, 0x17,
};

static const uint8_t huff_spec_a52_bits[] = {
    4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6,
    0, 6, 6, 6, 6, 5, 5, 5, 5, 5, 5, 5, 5, 5, 4, 4,
};

static const uint16_t huff_spec_a52_codes[] = {
    0x09, 0x04, 0x00, 0x1E, 0x1A, 0x14, 0x0C, 0x06, 0x18, 0x16, 0x0E, 0x04,
    0x3A, 0x38, 0x22, 0x20, 0x00, 0x21, 0x23, 0x39, 0x3B, 0x05, 0x0F, 0x17,
    0x19, 0x07, 0x0D, 0x15, 0x1B, 0x1F, 0x01, 0x05,
};

static const uint8_t huff_spec_a53_bits[] = {
    3, 4, 4, 4, 5, 5, 5, 5, 5, 5, 6, 6, 7, 7, 7, 7,
    0, 7, 7, 7, 7, 6, 6, 5, 5, 5, 5, 5, 5, 4, 4, 4,
};

static const uint16_t huff_spec_a53_codes[] = {
    0x00, 0x0C, 0x08, 0x04, 0x1E, 0x16, 0x14, 0x06, 0x0C, 0x04, 0x38, 0x1E,
    0x76, 0x74, 0x3A, 0x38, 0x00, 0x39, 0x3B, 0x75, 0x77, 0x1F, 0x39, 0x05,
    0x0D, 0x07, 0x15, 0x17, 0x1F, 0x05, 0x09, 0x0D,
};

static const uint8_t huff_spec_a54_bits[] = {
    3, 4, 4, 4, 4, 5, 5, 5, 5, 6, 6, 7, 7, 7, 8, 8,
    0, 8, 8, 7, 7, 7, 6, 6, 5, 5, 5, 5, 4, 4, 4, 4,
};

static const uint16_t huff_spec_a54_codes[] = {
    0x02, 0x0E, 0x0A, 0x08, 0x02, 0x1A, 0x0E, 0x02, 0x00, 0x30, 0x18, 0x66,
    0x36, 0x34, 0xCA, 0xC8, 0x00, 0xC9, 0xCB, 0x35, 0x37, 0x67, 0x19, 0x31,
    0x01, 0x03, 0x0F, 0x1B, 0x03, 0x09, 0x0B, 0x0F,
};

static const uint8_t huff_spec_a61_bits[] = {
    6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7,
    5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    0, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 5, 5, 5,
    5, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 6, 6,
};

static const uint16_t huff_spec_a61_codes[] = {
    0x35, 0x30, 0x2A, 0x28, 0x24, 0x20, 0x18, 0x0E, 0x0C, 0x7E, 0x7C, 0x72,
    0x70, 0x68, 0x5E, 0x5C, 0x04, 0x0E, 0x08, 0x00, 0x3C, 0x3A, 0x36, 0x32,
    0x2C, 0x26, 0x22, 0x1A, 0x16, 0x14, 0x06, 0x04, 0x00, 0x05, 0x07, 0x15,
    0x17, 0x1B, 0x23, 0x27, 0x2D, 0x33, 0x37, 0x3B, 0x3D, 0x01, 0x09, 0x0F,
    0x05, 0x5D, 0x5F, 0x69, 0x71, 0x73, 0x7D, 0x7F, 0x0D, 0x0F, 0x19, 0x21,
    0x25, 0x29, 0x2B, 0x31,
};

static const uint8_t huff_spec_a62_bits[] = {
    5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7,
    0, 7, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 5, 5, 5, 5,
};

static const uint16_t huff_spec_a62_codes[] = {
    0x14, 0x0E, 0x08, 0x04, 0x02, 0x3E, 0x3C, 0x38, 0x34, 0x30, 0x2A, 0x24,
    0x1A, 0x18, 0x0E, 0x02, 0x32, 0x36, 0x2C, 0x26, 0x20, 0x16, 0x0C, 0x00,
    0x76, 0x74, 0x5E, 0x5C, 0x46, 0x44, 0x2A, 0x28, 0x00, 0x29, 0x2B, 0x45,
    0x47, 0x5D, 0x5F, 0x75, 0x77, 0x01, 0x0D, 0x17, 0x21, 0x27, 0x2D, 0x37,
    0x33, 0x03, 0x0F, 0x19, 0x1B, 0x25, 0x2B, 0x31, 0x35, 0x39, 0x3D, 0x3F,
    0x03, 0x05, 0x09, 0x0F,
};

static const uint8_t huff_spec_a63_bits[] = {
    4, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8,
    0, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 5, 5, 5, 5, 5, 5, 5,
};

static const uint16_t huff_spec_a63_codes[] = {
    0x00, 0x1C, 0x18, 0x14, 0x10, 0x0A, 0x08, 0x02, 0x3E, 0x36, 0x2E, 0x2C,
    0x24, 0x1C, 0x0E, 0x08, 0x1E, 0x1A, 0x0C, 0x7A, 0x6A, 0x68, 0x4C, 0x32,
    0x16, 0x14, 0xF2, 0xF0, 0x9E, 0x9C, 0x62, 0x60, 0x00, 0x61, 0x63, 0x9D,
    0x9F, 0xF1, 0xF3, 0x15, 0x17, 0x33, 0x4D, 0x69, 0x6B, 0x7B, 0x0D, 0x1B,
    0x1F, 0x09, 0x0F, 0x1D, 0x25, 0x2D, 0x2F, 0x37, 0x3F, 0x03, 0x09, 0x0B,
    0x11, 0x15, 0x19, 0x1D,
};

static const uint8_t huff_spec_a64_bits[] = {
    4, 4, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 7,
    6, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9,
    0, 9, 9, 9, 9, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7,
    6, 7, 6, 6, 6, 6, 6, 6, 5, 5, 5, 5, 5, 5, 5, 4,
};

static const uint16_t huff_spec_a64_codes[] = {
    0x006, 0x002, 0x01C, 0x01A, 0x016, 0x012, 0x00E, 0x00A, 0x002, 0x03E,
    0x032, 0x02A, 0x022, 0x020, 0x010, 0x07A, 0x000, 0x078, 0x060, 0x050,
    0x024, 0x006, 0x0C6, 0x0C4, 0x0A4, 0x04E, 0x00A, 0x008, 0x14E, 0x14C,
    0x09A, 0x098, 0x000, 0x099, 0x09B, 0x14D, 0x14F, 0x009, 0x00B, 0x04F,
    0x0A5, 0x0C5, 0x0C7, 0x007, 0x025, 0x051, 0x061, 0x079, 0x001, 0x07B,
    0x011, 0x021, 0x023, 0x02B, 0x033, 0x03F, 0x003, 0x00B, 0x00F, 0x013,
    0x017, 0x01B, 0x01D, 0x003,
};

static const uint8_t huff_spec_a71_bits[] = {
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    0, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6,
    6, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
};

static const uint16_t huff_spec_a71_codes[] = {
    0x6C, 0x66, 0x62, 0x5C, 0x56, 0x50, 0x52, 0x4E, 0x48, 0x3E, 0x36, 0x34, 0x2A, 0x26, 0x1E, 0x16,
    0x0E, 0x08, 0x00, 0xF6, 0xF4, 0xEE, 0xEC, 0xE2, 0xE0, 0xDA, 0xD2, 0xD0, 0xBE, 0xBC, 0xB2, 0xB0,
    0x0C, 0x20, 0x1C, 0x16, 0x10, 0x08, 0x02, 0x7E, 0x7C, 0x78, 0x74, 0x72, 0x6E, 0x6A, 0x64, 0x60,
    0x5A, 0x54, 0x4C, 0x4A, 0x46, 0x44, 0x3C, 0x32, 0x30, 0x28, 0x24, 0x1C, 0x14, 0x0C, 0x0A, 0x02,
    0x00, 0x03, 0x0B, 0x0D, 0x15, 0x1D, 0x25, 0x29, 0x31, 0x33, 0x3D, 0x45, 0x47, 0x4B, 0x4D, 0x55,
    0x5B, 0x61, 0x65, 0x6B, 0x6F, 0x73, 0x75, 0x79, 0x7D, 0x7F, 0x03, 0x09, 0x11, 0x17, 0x1D, 0x21,
    0x0D, 0xB1, 0xB3, 0xBD, 0xBF, 0xD1, 0xD3, 0xDB, 0xE1, 0xE3, 0xED, 0xEF, 0xF5, 0xF7, 0x01, 0x09,
    0x0F, 0x17, 0x1F, 0x27, 0x2B, 0x35, 0x37, 0x3F, 0x49, 0x4F, 0x53, 0x51, 0x57, 0x5D, 0x63, 0x67,
};

static const uint8_t huff_spec_a72_bits[] = {
    6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    0, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 6, 6,
};

static const uint16_t huff_spec_a72_codes[] = {
    0x2A, 0x24, 0x1C, 0x18, 0x12, 0x0E, 0x0A, 0x06, 0x02, 0x7E, 0x7C, 0x7A, 0x76, 0x72, 0x70, 0x6A,
    0x68, 0x62, 0x5C, 0x5A, 0x52, 0x4E, 0x46, 0x42, 0x3C, 0x34, 0x2A, 0x28, 0x20, 0x12, 0x10, 0x08,
    0x66, 0x74, 0x6C, 0x64, 0x5E, 0x58, 0x50, 0x44, 0x40, 0x36, 0x2C, 0x22, 0x1A, 0x0A, 0x02, 0x00,
    0xF2, 0xF0, 0xDE, 0xDC, 0xC2, 0xC0, 0xAE, 0xAC, 0x9A, 0x98, 0x7E, 0x7C, 0x5E, 0x5C, 0x32, 0x30,
    0x00, 0x31, 0x33, 0x5D, 0x5F, 0x7D, 0x7F, 0x99, 0x9B, 0xAD, 0xAF, 0xC1, 0xC3, 0xDD, 0xDF, 0xF1,
    0xF3, 0x01, 0x03, 0x0B, 0x1B, 0x23, 0x2D, 0x37, 0x41, 0x45, 0x51, 0x59, 0x5F, 0x65, 0x6D, 0x75,
    0x67, 0x09, 0x11, 0x13, 0x21, 0x29, 0x2B, 0x35, 0x3D, 0x43, 0x47, 0x4F, 0x53, 0x5B, 0x5D, 0x63,
    0x69, 0x6B, 0x71, 0x73, 0x77, 0x7B, 0x7D, 0x7F, 0x03, 0x07, 0x0B, 0x0F, 0x13, 0x19, 0x1D, 0x25,
};

static const uint8_t huff_spec_a73_bits[] = {
    5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
    0, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
};

static const uint16_t huff_spec_a73_codes[] = {
    0x003, 0x03E, 0x038, 0x034, 0x030, 0x02C, 0x028, 0x024, 0x020, 0x01C, 0x016, 0x014,
    0x00E, 0x00A, 0x004, 0x000, 0x07A, 0x076, 0x06E, 0x06C, 0x064, 0x05E, 0x056, 0x04E,
    0x04C, 0x044, 0x036, 0x030, 0x022, 0x018, 0x012, 0x004, 0x03C, 0x03E, 0x032, 0x024,
    0x020, 0x010, 0x0F2, 0x0F0, 0x0E8, 0x0CE, 0x0BA, 0x0B8, 0x0A8, 0x08C, 0x06A, 0x04E,
    0x04C, 0x034, 0x00E, 0x00C, 0x1D6, 0x1D4, 0x19A, 0x198, 0x156, 0x154, 0x11E, 0x11C,
    0x0D2, 0x0D0, 0x06E, 0x06C, 0x000, 0x06D, 0x06F, 0x0D1, 0x0D3, 0x11D, 0x11F, 0x155,
    0x157, 0x199, 0x19B, 0x1D5, 0x1D7, 0x00D, 0x00F, 0x035, 0x04D, 0x04F, 0x06B, 0x08D,
    0x0A9, 0x0B9, 0x0BB, 0x0CF, 0x0E9, 0x0F1, 0x0F3, 0x011, 0x021, 0x025, 0x033, 0x03F,
    0x03D, 0x005, 0x013, 0x019, 0x023, 0x031, 0x037, 0x045, 0x04D, 0x04F, 0x057, 0x05F,
    0x065, 0x06D, 0x06F, 0x077, 0x07B, 0x001, 0x005, 0x00B, 0x00F, 0x015, 0x017, 0x01D,
    0x021, 0x025, 0x029, 0x02D, 0x031, 0x035, 0x039, 0x03F,
};

static const uint8_t huff_spec_a74_bits[] = {
    5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
    6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,
    7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,
    9,  9,  9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  9,  9,  9,  9,  9,
    9,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
    7,  8,  8,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,
    6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  5,  5,  5,
};

static const uint16_t huff_spec_a74_codes[] = {
    0x00D, 0x00A, 0x004, 0x000, 0x03A, 0x036, 0x032, 0x030, 0x02C, 0x028, 0x026, 0x022,
    0x01E, 0x018, 0x012, 0x00E, 0x006, 0x07E, 0x07A, 0x070, 0x06A, 0x05E, 0x056, 0x054,
    0x048, 0x040, 0x038, 0x022, 0x01A, 0x00A, 0x0F8, 0x0E6, 0x008, 0x0FA, 0x0F0, 0x0D2,
    0x0BA, 0x0B8, 0x094, 0x084, 0x074, 0x042, 0x032, 0x1E6, 0x1CA, 0x1C8, 0x1A2, 0x12E,
    0x10E, 0x10C, 0x0EC, 0x082, 0x062, 0x060, 0x3CA, 0x3C8, 0x342, 0x340, 0x25A, 0x258,
    0x1DE, 0x1DC, 0x102, 0x100, 0x000, 0x101, 0x103, 0x1DD, 0x1DF, 0x259, 0x25B, 0x341,
    0x343, 0x3C9, 0x3CB, 0x061, 0x063, 0x083, 0x0ED, 0x10D, 0x10F, 0x12F, 0x1A3, 0x1C9,
    0x1CB, 0x1E7, 0x033, 0x043, 0x075, 0x085, 0x095, 0x0B9, 0x0BB, 0x0D3, 0x0F1, 0x0FB,
    0x009, 0x0E7, 0x0F9, 0x00B, 0x01B, 0x023, 0x039, 0x041, 0x049, 0x055, 0x057, 0x05F,
    0x06B, 0x071, 0x07B, 0x07F, 0x007, 0x00F, 0x013, 0x019, 0x01F, 0x023, 0x027, 0x029,
    0x02D, 0x031, 0x033, 0x037, 0x03B, 0x001, 0x005, 0x00B,
};

static const uint8_t huff_spec_b22_bits[] = {
    0,  4,  0,  4,  4,  5,  0,  5,  0,  0,  0,  0,  4,  5,  0,  5,
    4,  7,  0,  6,  6,  9,  0,  7,  0,  0,  0,  0,  6,  9,  0,  7,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    4,  6,  0,  7,  6,  7,  0,  9,  0,  0,  0,  0,  6,  7,  0,  9,
    4,  8,  0,  8,  8, 10,  0, 10,  0,  0,  0,  0,  6,  9,  0,  9,
    5, 10,  0,  9,  9, 10,  0, 10,  0,  0,  0,  0,  7, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    6,  9,  0, 10,  9, 10,  0, 10,  0,  0,  0,  0,  7, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    4,  8,  0,  8,  6,  9,  0,  9,  0,  0,  0,  0,  8, 10,  0, 10,
    6, 10,  0,  9,  7, 10,  0, 10,  0,  0,  0,  0,  9, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    5,  9,  0, 10,  7, 10,  0, 10,  0,  0,  0,  0,  9, 10,  0, 10,
};

static const uint16_t huff_spec_b22_codes[] = {
    0x000, 0x00E, 0x000, 0x00F, 0x008, 0x006, 0x000, 0x00B, 0x000, 0x000, 0x000, 0x000,
    0x009, 0x00A, 0x000, 0x007, 0x006, 0x00A, 0x000, 0x029, 0x006, 0x158, 0x000, 0x023,
    0x000, 0x000, 0x000, 0x000, 0x013, 0x174, 0x000, 0x021, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x007, 0x028, 0x000, 0x00B, 0x012, 0x020, 0x000, 0x175, 0x000, 0x000, 0x000, 0x000,
    0x007, 0x022, 0x000, 0x159, 0x00C, 0x0BC, 0x000, 0x0BF, 0x022, 0x2B8, 0x000, 0x2BB,
    0x000, 0x000, 0x000, 0x000, 0x00B, 0x170, 0x000, 0x15B, 0x000, 0x04E, 0x000, 0x15F,
    0x042, 0x04A, 0x000, 0x041, 0x000, 0x000, 0x000, 0x000, 0x055, 0x044, 0x000, 0x04D,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x02D, 0x172, 0x000, 0x2ED, 0x040, 0x042, 0x000, 0x047,
    0x000, 0x000, 0x000, 0x000, 0x013, 0x2EE, 0x000, 0x049, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x00D, 0x0BE, 0x000, 0x0BD, 0x00A, 0x15A, 0x000, 0x171, 0x000, 0x000, 0x000, 0x000,
    0x023, 0x2BA, 0x000, 0x2B9, 0x02C, 0x2EC, 0x000, 0x173, 0x012, 0x048, 0x000, 0x2EF,
    0x000, 0x000, 0x000, 0x000, 0x041, 0x046, 0x000, 0x043, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x001, 0x15E, 0x000, 0x04F, 0x054, 0x04C, 0x000, 0x045, 0x000, 0x000, 0x000, 0x000,
    0x043, 0x040, 0x000, 0x04B,
};

static const uint8_t huff_spec_b23_bits[] = {
    2,  4,  0,  4,  4,  6,  0,  6,  0,  0,  0,  0,  4,  6,  0,  6,
    4,  9,  0,  7,  7,  9,  0,  8,  0,  0,  0,  0,  7,  9,  0,  8,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    4,  7,  0,  9,  7,  8,  0,  9,  0,  0,  0,  0,  7,  8,  0,  9,
    4,  8,  0,  8,  9, 10,  0, 10,  0,  0,  0,  0,  7, 10,  0, 10,
    7, 10,  0, 10, 10, 10,  0, 10,  0,  0,  0,  0,  9, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    7, 10,  0, 10, 10, 10,  0, 10,  0,  0,  0,  0,  8, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    4,  8,  0,  8,  7, 10,  0, 10,  0,  0,  0,  0,  9, 10,  0, 10,
    7, 10,  0, 10,  8, 10,  0, 10,  0,  0,  0,  0, 10, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    7, 10,  0, 10,  9, 10,  0, 10,  0,  0,  0,  0, 10, 10,  0, 10,
};

static const uint16_t huff_spec_b23_codes[] = {
    0x003, 0x008, 0x000, 0x009, 0x002, 0x018, 0x000, 0x01B, 0x000, 0x000, 0x000, 0x000,
    0x003, 0x01A, 0x000, 0x019, 0x000, 0x17C, 0x000, 0x055, 0x056, 0x0E8, 0x000, 0x07D,
    0x000, 0x000, 0x000, 0x000, 0x059, 0x0F6, 0x000, 0x07F, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x001, 0x054, 0x000, 0x17D, 0x058, 0x07E, 0x000, 0x0F7, 0x000, 0x000, 0x000, 0x000,
    0x057, 0x07C, 0x000, 0x0E9, 0x004, 0x0A2, 0x000, 0x0A1, 0x17A, 0x1DA, 0x000, 0x1D9,
    0x000, 0x000, 0x000, 0x000, 0x053, 0x1E8, 0x000, 0x2F3, 0x05C, 0x1D6, 0x000, 0x1E7,
    0x1EA, 0x1E2, 0x000, 0x1CF, 0x000, 0x000, 0x000, 0x000, 0x17F, 0x1CA, 0x000, 0x1DD,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x05B, 0x2F0, 0x000, 0x1DF, 0x1E4, 0x1CC, 0x000, 0x1D5,
    0x000, 0x000, 0x000, 0x000, 0x071, 0x1E0, 0x000, 0x1C9, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x005, 0x0A0, 0x000, 0x0A3, 0x052, 0x2F2, 0x000, 0x1E9, 0x000, 0x000, 0x000, 0x000,
    0x17B, 0x1D8, 0x000, 0x1DB, 0x05A, 0x1DE, 0x000, 0x2F1, 0x070, 0x1C8, 0x000, 0x1E1,
    0x000, 0x000, 0x000, 0x000, 0x1E5, 0x1D4, 0x000, 0x1CD, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x05D, 0x1E6, 0x000, 0x1D7, 0x17E, 0x1DC, 0x000, 0x1CB, 0x000, 0x000, 0x000, 0x000,
    0x1EB, 0x1CE, 0x000, 0x1E3,
};

static const uint8_t huff_spec_b24_bits[] = {
    1,  4,  0,  4,  5,  7,  0,  7,  0,  0,  0,  0,  5,  7,  0,  7,
    5,  9,  0,  7,  8, 10,  0,  9,  0,  0,  0,  0,  7, 10,  0,  9,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    5,  7,  0,  9,  7,  9,  0, 10,  0,  0,  0,  0,  8,  9,  0, 10,
    5,  9,  0,  8,  9, 10,  0, 10,  0,  0,  0,  0,  7, 10,  0, 10,
    7, 10,  0, 10, 10, 10,  0, 10,  0,  0,  0,  0, 10, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    7, 10,  0, 10, 10, 10,  0, 10,  0,  0,  0,  0, 10, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    5,  8,  0,  9,  7, 10,  0, 10,  0,  0,  0,  0,  9, 10,  0, 10,
    7, 10,  0, 10, 10, 10,  0, 10,  0,  0,  0,  0, 10, 10,  0, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    7, 10,  0, 10, 10, 10,  0, 10,  0,  0,  0,  0, 10, 10,  0, 10,
};

static const uint16_t huff_spec_b24_codes[] = {
    0x001, 0x000, 0x000, 0x001, 0x00A, 0x01C, 0x000, 0x033, 0x000, 0x000, 0x000, 0x000,
    0x00B, 0x032, 0x000, 0x01D, 0x008, 0x0D8, 0x000, 0x031, 0x06E, 0x0FA, 0x000, 0x0D7,
    0x000, 0x000, 0x000, 0x000, 0x011, 0x0F4, 0x000, 0x0D5, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x009, 0x030, 0x000, 0x0D9, 0x010, 0x0D4, 0x000, 0x0F5, 0x000, 0x000, 0x000, 0x000,
    0x06F, 0x0D6, 0x000, 0x0FB, 0x00E, 0x0DA, 0x000, 0x025, 0x0D2, 0x0D4, 0x000, 0x0DB,
    0x000, 0x000, 0x000, 0x000, 0x017, 0x0FE, 0x000, 0x0FD, 0x014, 0x0DC, 0x000, 0x0F9,
    0x0F2, 0x0D6, 0x000, 0x09B, 0x000, 0x000, 0x000, 0x000, 0x1A3, 0x09C, 0x000, 0x0D3,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x019, 0x0F6, 0x000, 0x0D9, 0x0F0, 0x09E, 0x000, 0x0D1,
    0x000, 0x000, 0x000, 0x000, 0x1A1, 0x0DE, 0x000, 0x099, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x00F, 0x024, 0x000, 0x0DB, 0x016, 0x0FC, 0x000, 0x0FF, 0x000, 0x000, 0x000, 0x000,
    0x0D3, 0x0DA, 0x000, 0x0D5, 0x018, 0x0D8, 0x000, 0x0F7, 0x1A0, 0x098, 0x000, 0x0DF,
    0x000, 0x000, 0x000, 0x000, 0x0F1, 0x0D0, 0x000, 0x09F, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x015, 0x0F8, 0x000, 0x0DD, 0x1A2, 0x0D2, 0x000, 0x09D, 0x000, 0x000, 0x000, 0x000,
    0x0F3, 0x09A, 0x000, 0x0D7
};

static const uint8_t huff_spec_b32_bits[] = {
    2, 4, 5, 6, 0, 6, 5, 4, 5, 6, 6, 7, 0, 6, 5, 6,
    5, 6, 7, 7, 0, 8, 7, 6, 6, 7, 8, 9, 0, 9, 8, 7,
    0, 0, 0, 0, 0, 0, 0, 0, 6, 7, 8, 9, 0, 9, 8, 7,
    5, 6, 7, 8, 0, 7, 7, 6, 5, 6, 5, 6, 0, 7, 6, 6,
};

static const uint16_t huff_spec_b32_codes[] = {
    0x001, 0x002, 0x01E, 0x02A, 0x000, 0x02B, 0x01F, 0x003, 0x016, 0x020, 0x03A, 0x064,
    0x000, 0x005, 0x001, 0x023, 0x01A, 0x026, 0x070, 0x00C, 0x000, 0x0CF, 0x073, 0x031,
    0x024, 0x00E, 0x0CC, 0x146, 0x000, 0x145, 0x0A1, 0x053, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x025, 0x052, 0x0A0, 0x144, 0x000, 0x147, 0x0CD, 0x00F,
    0x01B, 0x030, 0x072, 0x0CE, 0x000, 0x00D, 0x071, 0x027, 0x017, 0x022, 0x000, 0x004,
    0x000, 0x065, 0x03B, 0x021,
};

static const uint8_t huff_spec_b33_bits[] = {
    2,  4,  5,  7,  0,  7,  5,  4,  4,  5,  6,  8,  0,  7,  6,  5,
    5,  6,  7,  9,  0,  8,  7,  6,  7,  8,  9, 10,  0, 10,  9,  8,
    0,  0,  0,  0,  0,  0,  0,  0,  7,  8,  9, 10,  0, 10,  9,  8,
    5,  6,  7,  8,  0,  9,  7,  6,  4,  5,  6,  7,  0,  8,  6,  5,
};

static const uint16_t huff_spec_b33_codes[] = {
    0x003, 0x008, 0x014, 0x05E, 0x000, 0x05F, 0x015, 0x009, 0x004, 0x002, 0x01C, 0x0BA,
    0x000, 0x011, 0x01F, 0x001, 0x00C, 0x00C, 0x014, 0x166, 0x000, 0x02D, 0x013, 0x00F,
    0x05A, 0x0B0, 0x05E, 0x0B8, 0x000, 0x0BB, 0x165, 0x0B9, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x05B, 0x0B8, 0x164, 0x0BA, 0x000, 0x0B9, 0x05F, 0x0B1,
    0x00D, 0x00E, 0x012, 0x02C, 0x000, 0x167, 0x015, 0x00D, 0x005, 0x000, 0x01E, 0x010,
    0x000, 0x0BB, 0x01D, 0x003
};

static const uint8_t huff_spec_b34_bits[] = {
    1,  4,  6,  8,  0,  8,  6,  4,  4,  6,  7,  9,  0,  8,  7,  6,
    6,  7,  8, 10,  0, 10,  8,  7,  8,  9, 10, 10,  0, 10, 10,  9,
    0,  0,  0,  0,  0,  0,  0,  0,  8,  9, 10, 10,  0, 10, 10,  9,
    6,  7,  8, 10,  0, 10,  8,  7,  4,  6,  7,  8,  0,  9,  7,  6,
};

static const uint16_t huff_spec_b34_codes[] = {
    0x000, 0x00A, 0x038, 0x0EE, 0x000, 0x0EF, 0x039, 0x00B, 0x008, 0x03C, 0x06E, 0x1D8,
    0x000, 0x0C1, 0x075, 0x03F, 0x032, 0x068, 0x0C4, 0x358, 0x000, 0x30F, 0x0C7, 0x06D,
    0x0D4, 0x1AE, 0x30C, 0x308, 0x000, 0x30B, 0x35B, 0x1DB, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x0D5, 0x1DA, 0x35A, 0x30A, 0x000, 0x309, 0x30D, 0x1AF,
    0x033, 0x06C, 0x0C6, 0x30E, 0x000, 0x359, 0x0C5, 0x069, 0x009, 0x03E, 0x074, 0x0C0,
    0x000, 0x1D9, 0x06F, 0x03D,
};

static const uint8_t huff_spec_b42_bits[] = {
    4,  5,  6,  8,  6,  7,  8,  8,  0,  8,  8,  7,  6,  8,  6,  5,
    5,  6,  7,  8,  7,  7,  8,  9,  0,  8,  8,  7,  7,  8,  7,  6,
    7,  7,  8,  9,  7,  8,  9,  9,  0,  9,  9,  8,  7,  9,  8,  7,
    8,  9,  9, 10,  8,  8,  9, 10,  0, 10,  9,  8,  8, 10,  9,  8,
    6,  7,  8,  8,  9,  9, 10, 10,  0, 10, 10,  9,  9,  8,  8,  7,
    7,  7,  8,  9,  9, 10, 10, 10,  0, 10, 10, 10,  9,  9,  8,  7,
    8,  8,  9,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  9,  8,
    8,  9,  9, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10,  9,  9,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    8,  9,  9, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10,  9,  9,
    8,  8,  9,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  9,  8,
    7,  7,  8,  9,  9, 10, 10, 10,  0, 10, 10, 10,  9,  9,  8,  7,
    6,  7,  8,  8,  9,  9, 10, 10,  0, 10, 10,  9,  9,  8,  8,  7,
    8,  8,  9, 10,  8,  8,  9, 10,  0, 10,  9,  8,  8, 10,  9,  9,
    7,  7,  8,  9,  7,  8,  9,  9,  0,  9,  9,  8,  7,  9,  8,  7,
    5,  6,  7,  8,  7,  7,  8,  8,  0,  9,  8,  7,  7,  8,  7,  6,
};

static const uint16_t huff_spec_b42_codes[] = {
    0x00E, 0x018, 0x010, 0x0F0, 0x024, 0x05A, 0x0F6, 0x078, 0x000, 0x079, 0x0F7, 0x05B,
    0x025, 0x0F1, 0x011, 0x019, 0x00C, 0x014, 0x01C, 0x036, 0x05C, 0x012, 0x09E, 0x1E4,
    0x000, 0x00B, 0x0A9, 0x03B, 0x05F, 0x071, 0x019, 0x017, 0x06E, 0x000, 0x03E, 0x114,
    0x002, 0x0B0, 0x1AA, 0x07A, 0x000, 0x099, 0x1E7, 0x0B3, 0x00B, 0x131, 0x07F, 0x00D,
    0x0D8, 0x1FE, 0x112, 0x22E, 0x086, 0x010, 0x134, 0x35C, 0x000, 0x35F, 0x133, 0x013,
    0x081, 0x22D, 0x119, 0x07B, 0x00A, 0x050, 0x0F8, 0x04E, 0x1B4, 0x154, 0x3EC, 0x0D2,
    0x000, 0x0D7, 0x3D7, 0x137, 0x1FD, 0x073, 0x0FD, 0x057, 0x052, 0x010, 0x08E, 0x1E8,
    0x11A, 0x3EE, 0x0F2, 0x03C, 0x000, 0x03F, 0x0F1, 0x3D5, 0x111, 0x1F5, 0x09D, 0x025,
    0x0D2, 0x082, 0x1A0, 0x0F8, 0x36E, 0x0D4, 0x072, 0x03A, 0x000, 0x027, 0x071, 0x07D,
    0x36D, 0x0FB, 0x1AD, 0x085, 0x00C, 0x1A8, 0x03C, 0x346, 0x0D0, 0x076, 0x024, 0x020,
    0x000, 0x023, 0x039, 0x075, 0x07F, 0x345, 0x09B, 0x157, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x00D, 0x156, 0x09A, 0x344, 0x07E, 0x074, 0x038, 0x022, 0x000, 0x021, 0x025, 0x077,
    0x0D1, 0x347, 0x03D, 0x1A9, 0x0D3, 0x084, 0x1AC, 0x0FA, 0x36C, 0x07C, 0x070, 0x026,
    0x000, 0x03B, 0x073, 0x0D5, 0x36F, 0x0F9, 0x1A1, 0x083, 0x053, 0x024, 0x09C, 0x1F4,
    0x110, 0x3D4, 0x0F0, 0x03E, 0x000, 0x03D, 0x0F3, 0x3EF, 0x11B, 0x1E9, 0x08F, 0x011,
    0x00B, 0x056, 0x0FC, 0x072, 0x1FC, 0x136, 0x3D6, 0x0D6, 0x000, 0x0D3, 0x3ED, 0x155,
    0x1B5, 0x04F, 0x0F9, 0x051, 0x0D9, 0x07A, 0x118, 0x22C, 0x080, 0x012, 0x132, 0x35E,
    0x000, 0x35D, 0x135, 0x011, 0x087, 0x22F, 0x113, 0x1FF, 0x06F, 0x00C, 0x07E, 0x130,
    0x00A, 0x0B2, 0x1E6, 0x098, 0x000, 0x07B, 0x1AB, 0x0B1, 0x003, 0x115, 0x03F, 0x001,
    0x00D, 0x016, 0x018, 0x070, 0x05E, 0x03A, 0x0A8, 0x00A, 0x000, 0x1E5, 0x09F, 0x013,
    0x05D, 0x037, 0x01D, 0x015,
};

static const uint8_t huff_spec_b43_bits[] = {
    2,  5,  6,  7,  7,  8,  8,  9,  0,  9,  8,  8,  7,  7,  6,  5,
    5,  6,  7,  8,  7,  8,  9, 10,  0, 10,  9,  8,  7,  8,  7,  6,
    6,  7,  8,  9,  8,  9, 10, 10,  0, 10, 10,  9,  8,  9,  8,  7,
    7,  8,  9, 10,  9,  9, 10, 10,  0, 10, 10, 10,  9, 10,  9,  8,
    7,  8,  8,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  8,  7,
    8,  8,  9, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10,  9,  8,
    9,  9, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  9,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    9,  9, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  9,
    8,  8,  9, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10,  9,  8,
    7,  7,  8,  9, 10, 10, 10, 10,  0, 10, 10, 10, 10,  9,  8,  8,
    7,  8,  9, 10,  9, 10, 10, 10,  0, 10, 10,  9,  9, 10,  9,  8,
    6,  7,  8,  9,  8,  9, 10, 10,  0, 10, 10,  9,  8,  9,  8,  7,
    5,  6,  7,  8,  7,  8,  9, 10,  0, 10,  9,  8,  7,  8,  7,  6,
};

static const uint16_t huff_spec_b43_codes[] = {
    0x001, 0x01E, 0x022, 0x018, 0x064, 0x0EC, 0x008, 0x100, 0x000, 0x101, 0x009, 0x0ED,
    0x065, 0x019, 0x023, 0x01F, 0x01A, 0x030, 0x056, 0x09A, 0x00A, 0x090, 0x12C, 0x0A6,
    0x000, 0x0A9, 0x12F, 0x093, 0x00F, 0x09F, 0x059, 0x039, 0x00E, 0x054, 0x0BC, 0x19E,
    0x082, 0x176, 0x0AC, 0x088, 0x000, 0x08B, 0x0AF, 0x19D, 0x095, 0x1D1, 0x0BF, 0x051,
    0x002, 0x098, 0x1D4, 0x0B8, 0x170, 0x046, 0x090, 0x060, 0x000, 0x067, 0x095, 0x0BD,
    0x173, 0x0B5, 0x1D3, 0x09D, 0x052, 0x0EE, 0x034, 0x174, 0x0BA, 0x09C, 0x080, 0x044,
    0x000, 0x047, 0x06D, 0x099, 0x0BF, 0x16F, 0x085, 0x001, 0x0CC, 0x036, 0x16C, 0x0B0,
    0x09A, 0x084, 0x04E, 0x03E, 0x000, 0x037, 0x04B, 0x06B, 0x0A1, 0x0B3, 0x16B, 0x087,
    0x1D6, 0x102, 0x0A4, 0x092, 0x068, 0x04C, 0x034, 0x030, 0x000, 0x02D, 0x03D, 0x049,
    0x083, 0x097, 0x0AB, 0x169, 0x0B6, 0x09E, 0x06E, 0x064, 0x040, 0x038, 0x02E, 0x02A,
    0x000, 0x029, 0x033, 0x03B, 0x043, 0x063, 0x087, 0x0A3, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x0B7, 0x0A2, 0x086, 0x062, 0x042, 0x03A, 0x032, 0x028, 0x000, 0x02B, 0x02F, 0x039,
    0x041, 0x065, 0x06F, 0x09F, 0x1D7, 0x168, 0x0AA, 0x096, 0x082, 0x048, 0x03C, 0x02C,
    0x000, 0x031, 0x035, 0x04D, 0x069, 0x093, 0x0A5, 0x103, 0x0CD, 0x086, 0x16A, 0x0B2,
    0x0A0, 0x06A, 0x04A, 0x036, 0x000, 0x03F, 0x04F, 0x085, 0x09B, 0x0B1, 0x16D, 0x037,
    0x053, 0x000, 0x084, 0x16E, 0x0BE, 0x098, 0x06C, 0x046, 0x000, 0x045, 0x081, 0x09D,
    0x0BB, 0x175, 0x035, 0x0EF, 0x003, 0x09C, 0x1D2, 0x0B4, 0x172, 0x0BC, 0x094, 0x066,
    0x000, 0x061, 0x091, 0x047, 0x171, 0x0B9, 0x1D5, 0x099, 0x00F, 0x050, 0x0BE, 0x1D0,
    0x094, 0x19C, 0x0AE, 0x08A, 0x000, 0x089, 0x0AD, 0x177, 0x083, 0x19F, 0x0BD, 0x055,
    0x01B, 0x038, 0x058, 0x09E, 0x00E, 0x092, 0x12E, 0x0A8, 0x000, 0x0A7, 0x12D, 0x091,
    0x00B, 0x09B, 0x057, 0x031,
};

static const uint8_t huff_spec_b44_bits[] = {
    2,  4,  6,  7,  7,  8, 10, 10,  0, 10, 10,  8,  7,  7,  6,  4,
    5,  5,  7,  8,  8, 10, 10, 10,  0, 10, 10, 10,  8,  8,  7,  5,
    6,  7,  8,  9,  9, 10, 10, 10,  0, 10, 10, 10, 10,  9,  8,  7,
    8,  8,  9, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  8,
    8,  8, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  8,
    9, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    9, 10, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10, 10,
    8,  8, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10, 10,  8,
    8,  8, 10, 10, 10, 10, 10, 10,  0, 10, 10, 10, 10, 10,  9,  8,
    6,  7,  8,  9, 10, 10, 10, 10,  0, 10, 10, 10,  9,  9,  8,  7,
    5,  5,  7,  8,  8, 10, 10, 10,  0, 10, 10, 10,  8,  8,  7,  5,
};

static const uint16_t huff_spec_b44_codes[] = {
    0x002, 0x002, 0x030, 0x000, 0x002, 0x00C, 0x1D2, 0x1AE, 0x000, 0x1AF, 0x1D3, 0x00D,
    0x003, 0x001, 0x031, 0x003, 0x01E, 0x002, 0x070, 0x0C8, 0x07E, 0x1E8, 0x1C0, 0x176,
    0x000, 0x17F, 0x1C3, 0x1EB, 0x0CF, 0x0D3, 0x073, 0x009, 0x018, 0x06A, 0x0EC, 0x1DE,
    0x1A2, 0x1CA, 0x1AA, 0x164, 0x000, 0x16D, 0x1AD, 0x1D1, 0x1EF, 0x1DD, 0x0EB, 0x06D,
    0x0E8, 0x0CA, 0x1BE, 0x1CE, 0x1DA, 0x1B6, 0x170, 0x154, 0x000, 0x153, 0x173, 0x1B1,
    0x1D7, 0x1D5, 0x343, 0x0CD, 0x0DC, 0x078, 0x340, 0x1CC, 0x1BA, 0x1A8, 0x156, 0x148,
    0x000, 0x145, 0x15F, 0x1A1, 0x1BD, 0x1D9, 0x1ED, 0x07D, 0x1BC, 0x1DC, 0x1C4, 0x1B2,
    0x17C, 0x15A, 0x14A, 0x03A, 0x000, 0x039, 0x147, 0x16B, 0x17B, 0x1B5, 0x1C9, 0x1DF,
    0x1C6, 0x1B8, 0x1A2, 0x168, 0x160, 0x14C, 0x02E, 0x024, 0x000, 0x027, 0x03D, 0x151,
    0x15D, 0x16F, 0x1A7, 0x1BF, 0x1A4, 0x174, 0x162, 0x14E, 0x140, 0x02C, 0x02A, 0x022,
    0x000, 0x021, 0x029, 0x03F, 0x143, 0x159, 0x167, 0x179, 0x000, 0x000, 0x000, 0x000,
    0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000, 0x000,
    0x1A5, 0x178, 0x166, 0x158, 0x142, 0x03E, 0x028, 0x020, 0x000, 0x023, 0x02B, 0x02D,
    0x141, 0x14F, 0x163, 0x175, 0x1C7, 0x1BE, 0x1A6, 0x16E, 0x15C, 0x150, 0x03C, 0x026,
    0x000, 0x025, 0x02F, 0x14D, 0x161, 0x169, 0x1A3, 0x1B9, 0x1BD, 0x1DE, 0x1C8, 0x1B4,
    0x17A, 0x16A, 0x146, 0x038, 0x000, 0x03B, 0x14B, 0x15B, 0x17D, 0x1B3, 0x1C5, 0x1DD,
    0x0DD, 0x07C, 0x1EC, 0x1D8, 0x1BC, 0x1A0, 0x15E, 0x144, 0x000, 0x149, 0x157, 0x1A9,
    0x1BB, 0x1CD, 0x341, 0x079, 0x0E9, 0x0CC, 0x342, 0x1D4, 0x1D6, 0x1B0, 0x172, 0x152,
    0x000, 0x155, 0x171, 0x1B7, 0x1DB, 0x1CF, 0x1BF, 0x0CB, 0x019, 0x06C, 0x0EA, 0x1DC,
    0x1EE, 0x1D0, 0x1AC, 0x16C, 0x000, 0x165, 0x1AB, 0x1CB, 0x1A3, 0x1DF, 0x0ED, 0x06B,
    0x01F, 0x008, 0x072, 0x0D2, 0x0CE, 0x1EA, 0x1C2, 0x17E, 0x000, 0x177, 0x1C1, 0x1E9,
    0x07F, 0x0C9, 0x071, 0x003,
};

static const uint8_t huff_spec_b52_bits[] = {
    3, 4, 4, 4, 5, 5, 6, 6, 5, 5, 5, 6, 6, 6, 7, 7,
    0, 7, 7, 6, 6, 6, 5, 5, 5, 6, 6, 5, 5, 4, 4, 4,
};

static const uint16_t huff_spec_b52_codes[] = {
    0x06, 0x0E, 0x06, 0x00, 0x0A, 0x04, 0x2C, 0x12, 0x14, 0x10, 0x06, 0x2E, 0x24, 0x10, 0x4E, 0x4C,
    0x00, 0x4D, 0x4F, 0x11, 0x25, 0x2F, 0x07, 0x11, 0x15, 0x13, 0x2D, 0x05, 0x0B, 0x01, 0x07, 0x0F,
};

static const uint8_t huff_spec_b53_bits[] = {
    2, 3, 4, 5, 5, 6, 6, 6, 6, 6, 6, 7, 7, 7, 8, 8,
    0, 8, 8, 7, 7, 7, 6, 6, 6, 6, 6, 6, 5, 5, 4, 3,
};

static const uint16_t huff_spec_b53_codes[] = {
    0x02, 0x00, 0x06, 0x1C, 0x18, 0x3E, 0x16, 0x10, 0x3C, 0x36, 0x14, 0x6A, 0x26, 0x24, 0xD2, 0xD0,
    0x00, 0xD1, 0xD3, 0x25, 0x27, 0x6B, 0x15, 0x37, 0x3D, 0x11, 0x17, 0x3F, 0x19, 0x1D, 0x07, 0x01,
};

static const uint8_t huff_spec_b54_bits[] = {
    2, 3, 4, 4, 5, 6, 6, 7, 6, 6, 7, 8, 8, 8, 9, 9,
    0, 9, 9, 8, 8, 8, 7, 6, 6, 7, 6, 6, 5, 4, 4, 3,
};

static const uint16_t huff_spec_b54_codes[] = {
    0x003, 0x002, 0x008, 0x000, 0x014, 0x02E, 0x00E, 0x05A, 0x00A, 0x008, 0x01A, 0x0B2,
    0x032, 0x030, 0x162, 0x160, 0x000, 0x161, 0x163, 0x031, 0x033, 0x0B3, 0x01B, 0x009,
    0x00B, 0x05B, 0x00F, 0x02F, 0x015, 0x001, 0x009, 0x003,
};

static const uint8_t huff_spec_b62_bits[] = {
    4, 4, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7,
    6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8,
    0, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6,
    6, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 5, 5, 5, 5, 4,
};

static const uint16_t huff_spec_b62_codes[] = {
    0x0D, 0x06, 0x1C, 0x14, 0x0A, 0x04, 0x3E, 0x2E, 0x22, 0x0E, 0x06, 0x00, 0x5A, 0x4E, 0x40, 0x20,
    0x30, 0x32, 0x24, 0x12, 0x0C, 0x02, 0x78, 0x58, 0x42, 0x22, 0x0A, 0x08, 0xF6, 0xF4, 0x9A, 0x98,
    0x00, 0x99, 0x9B, 0xF5, 0xF7, 0x09, 0x0B, 0x23, 0x43, 0x59, 0x79, 0x03, 0x0D, 0x13, 0x25, 0x33,
    0x31, 0x21, 0x41, 0x4F, 0x5B, 0x01, 0x07, 0x0F, 0x23, 0x2F, 0x3F, 0x05, 0x0B, 0x15, 0x1D, 0x07,
};

static const uint8_t huff_spec_b63_bits[] = {
    3, 4, 4, 5, 5, 5, 5, 6, 6, 6, 6, 7, 7, 7, 7, 8,
    6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9,
    0, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 7, 7, 7, 7, 6,
    6, 8, 7, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 5, 4, 4,
};

static const uint16_t huff_spec_b63_codes[] = {
    0x006, 0x00E, 0x004, 0x014, 0x010, 0x006, 0x000, 0x026, 0x01C, 0x018, 0x004, 0x05C,
    0x04A, 0x03C, 0x016, 0x0BC, 0x006, 0x008, 0x058, 0x03E, 0x036, 0x014, 0x0B6, 0x0B4,
    0x090, 0x068, 0x17E, 0x17C, 0x126, 0x124, 0x0D6, 0x0D4, 0x000, 0x0D5, 0x0D7, 0x125,
    0x127, 0x17D, 0x17F, 0x069, 0x091, 0x0B5, 0x0B7, 0x015, 0x037, 0x03F, 0x059, 0x009,
    0x007, 0x0BD, 0x017, 0x03D, 0x04B, 0x05D, 0x005, 0x019, 0x01D, 0x027, 0x001, 0x007,
    0x011, 0x015, 0x005, 0x00F,
};

static const uint8_t huff_spec_b64_bits[] = {
    3,  3,  4,  5,  5,  5,  6,  6,  6,  6,  7,  7,  7,  7,  7,  8,
    7,  7,  7,  8,  8,  8,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10,
    0, 10, 10, 10, 10,  9,  9,  9,  9,  9,  9,  8,  8,  8,  7,  7,
    7,  8,  7,  7,  7,  7,  7,  6,  6,  6,  6,  5,  5,  5,  4,  3,
};

static const uint16_t huff_spec_b64_codes[] = {
    0x007, 0x000, 0x008, 0x01A, 0x014, 0x00C, 0x032, 0x02E, 0x01E, 0x014, 0x062, 0x05A,
    0x03A, 0x026, 0x020, 0x0B2, 0x038, 0x02C, 0x022, 0x0C0, 0x05E, 0x04A, 0x186, 0x184,
    0x160, 0x0BA, 0x092, 0x090, 0x2C6, 0x2C4, 0x172, 0x170, 0x000, 0x171, 0x173, 0x2C5,
    0x2C7, 0x091, 0x093, 0x0BB, 0x161, 0x185, 0x187, 0x04B, 0x05F, 0x0C1, 0x023, 0x02D,
    0x039, 0x0B3, 0x021, 0x027, 0x03B, 0x05B, 0x063, 0x015, 0x01F, 0x02F, 0x033, 0x00D,
    0x015, 0x01B, 0x009, 0x001,
};

static const uint8_t huff_spec_b72_bits[] = {
    5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
    0, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 6, 6, 5, 5, 5,
};

static const uint16_t huff_spec_b72_codes[] = {
    0x01E, 0x016, 0x00C, 0x000, 0x038, 0x032, 0x028, 0x022, 0x01C, 0x012, 0x00E, 0x006,
    0x076, 0x06C, 0x060, 0x04E, 0x03E, 0x02A, 0x022, 0x01A, 0x012, 0x00A, 0x0FC, 0x0DC,
    0x0C6, 0x0A8, 0x094, 0x086, 0x058, 0x042, 0x040, 0x02A, 0x068, 0x07C, 0x06A, 0x056,
    0x048, 0x040, 0x02E, 0x028, 0x016, 0x010, 0x008, 0x0EA, 0x0DE, 0x0AA, 0x09A, 0x096,
    0x07A, 0x078, 0x05A, 0x032, 0x030, 0x028, 0x1FE, 0x1FC, 0x1D2, 0x1D0, 0x18A, 0x188,
    0x132, 0x130, 0x10A, 0x108, 0x000, 0x109, 0x10B, 0x131, 0x133, 0x189, 0x18B, 0x1D1,
    0x1D3, 0x1FD, 0x1FF, 0x029, 0x031, 0x033, 0x05B, 0x079, 0x07B, 0x097, 0x09B, 0x0AB,
    0x0DF, 0x0EB, 0x009, 0x011, 0x017, 0x029, 0x02F, 0x041, 0x049, 0x057, 0x06B, 0x07D,
    0x069, 0x02B, 0x041, 0x043, 0x059, 0x087, 0x095, 0x0A9, 0x0C7, 0x0DD, 0x0FD, 0x00B,
    0x013, 0x01B, 0x023, 0x02B, 0x03F, 0x04F, 0x061, 0x06D, 0x077, 0x007, 0x00F, 0x013,
    0x01D, 0x023, 0x029, 0x033, 0x039, 0x001, 0x00D, 0x017,
};

static const uint8_t huff_spec_b73_bits[] = {
    3,  4,  5,  5,  5,  6,  6,  6,  6,  6,  6,  7,  7,  7,  7,  7,
    7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,
    8,  7,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9,
    9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  9,  9,  9,
    9,  9,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,  8,  8,  7,
    8,  9,  9,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  7,  7,  7,
    7,  7,  7,  7,  7,  7,  6,  6,  6,  6,  6,  6,  5,  5,  5,  4,
};

static const uint16_t huff_spec_b73_codes[] = {
    0x000, 0x006, 0x018, 0x010, 0x004, 0x03A, 0x034, 0x02A, 0x026, 0x014, 0x010, 0x07E,
    0x072, 0x06E, 0x05C, 0x052, 0x04A, 0x02C, 0x024, 0x018, 0x0F4, 0x0E0, 0x0DA, 0x0B6,
    0x0B2, 0x0A0, 0x05E, 0x04E, 0x038, 0x034, 0x1E6, 0x1B2, 0x0FA, 0x01E, 0x0F8, 0x0F0,
    0x0BE, 0x0B4, 0x0A2, 0x090, 0x04C, 0x03A, 0x1EE, 0x1E4, 0x1C6, 0x1B0, 0x178, 0x162,
    0x126, 0x124, 0x0B8, 0x06C, 0x3DA, 0x3D8, 0x38A, 0x388, 0x2F6, 0x2F4, 0x2C2, 0x2C0,
    0x176, 0x174, 0x0DC, 0x0DE, 0x000, 0x0DF, 0x0DD, 0x175, 0x177, 0x2C1, 0x2C3, 0x2F5,
    0x2F7, 0x389, 0x38B, 0x3D9, 0x3DB, 0x06D, 0x0B9, 0x125, 0x127, 0x163, 0x179, 0x1B1,
    0x1C7, 0x1E5, 0x1EF, 0x03B, 0x04D, 0x091, 0x0A3, 0x0B5, 0x0BF, 0x0F1, 0x0F9, 0x01F,
    0x0FB, 0x1B3, 0x1E7, 0x035, 0x039, 0x04F, 0x05F, 0x0A1, 0x0B3, 0x0B7, 0x0DB, 0x0E1,
    0x0F5, 0x019, 0x025, 0x02D, 0x04B, 0x053, 0x05D, 0x06F, 0x073, 0x07F, 0x011, 0x015,
    0x027, 0x02B, 0x035, 0x03B, 0x005, 0x011, 0x019, 0x007,
};

static const uint8_t huff_spec_b74_bits[] = {
    3,  4,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  7,  7,  7,  7,
    7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,
    8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10,  9,  9,  9,  9,  9,  9,  9,  8,  8,  8,  8,
    8,  9,  9,  9,  8,  8,  8,  8,  8,  8,  8,  8,  8,  7,  7,  7,
    7,  7,  7,  7,  7,  6,  6,  6,  6,  6,  6,  5,  5,  5,  5,  4,
};

static const uint16_t huff_spec_b74_codes[] = {
    0x001, 0x008, 0x01E, 0x018, 0x00C, 0x002, 0x03A, 0x034, 0x02C, 0x01E, 0x016, 0x012,
    0x072, 0x06E, 0x05E, 0x056, 0x050, 0x038, 0x022, 0x004, 0x0E2, 0x0DA, 0x0BA, 0x0A8,
    0x076, 0x054, 0x050, 0x002, 0x000, 0x1C0, 0x1B0, 0x156, 0x0A4, 0x0A6, 0x074, 0x052,
    0x004, 0x1C2, 0x1B2, 0x170, 0x154, 0x0AE, 0x0AC, 0x086, 0x2E6, 0x2E4, 0x10A, 0x108,
    0x106, 0x104, 0x102, 0x100, 0x03E, 0x03A, 0x03C, 0x038, 0x036, 0x034, 0x032, 0x030,
    0x01E, 0x01A, 0x01C, 0x018, 0x000, 0x019, 0x01D, 0x01B, 0x01F, 0x031, 0x033, 0x035,
    0x037, 0x039, 0x03D, 0x03B, 0x03F, 0x101, 0x103, 0x105, 0x107, 0x109, 0x10B, 0x2E5,
    0x2E7, 0x087, 0x0AD, 0x0AF, 0x155, 0x171, 0x1B3, 0x1C3, 0x005, 0x053, 0x075, 0x0A7,
    0x0A5, 0x157, 0x1B1, 0x1C1, 0x001, 0x003, 0x051, 0x055, 0x077, 0x0A9, 0x0BB, 0x0DB,
    0x0E3, 0x005, 0x023, 0x039, 0x051, 0x057, 0x05F, 0x06F, 0x073, 0x013, 0x017, 0x01F,
    0x02D, 0x035, 0x03B, 0x003, 0x00D, 0x019, 0x01F, 0x009,
};

static const HuffmanCodebook at9_huffman_coeffs[][8][4] = {
    {
        { { 0 } },
        { { 0 } },
        {
            { huff_spec_a21_bits, huff_spec_a21_codes,  16,   2,   1,   2,   3, },
            { huff_spec_a22_bits, huff_spec_a22_codes, 256,   4,   2,   2,   8, },
            { huff_spec_a23_bits, huff_spec_a23_codes, 256,   4,   2,   2,   9, },
            { huff_spec_a24_bits, huff_spec_a24_codes, 256,   4,   2,   2,  10, },
        },
        {
            { huff_spec_a31_bits, huff_spec_a31_codes,  64,   2,   1,   3,   7, },
            { huff_spec_a32_bits, huff_spec_a32_codes,  64,   2,   1,   3,   7, },
            { huff_spec_a33_bits, huff_spec_a33_codes,  64,   2,   1,   3,   8, },
            { huff_spec_a34_bits, huff_spec_a34_codes,  64,   2,   1,   3,  10, },
        },
        {
            { huff_spec_a41_bits, huff_spec_a41_codes, 256,   2,   1,   4,   9, },
            { huff_spec_a42_bits, huff_spec_a42_codes, 256,   2,   1,   4,  10, },
            { huff_spec_a43_bits, huff_spec_a43_codes, 256,   2,   1,   4,  10, },
            { huff_spec_a44_bits, huff_spec_a44_codes, 256,   2,   1,   4,  10, },
        },
        {
            { huff_spec_a51_bits, huff_spec_a51_codes,  32,   1,   0,   5,   6, },
            { huff_spec_a52_bits, huff_spec_a52_codes,  32,   1,   0,   5,   6, },
            { huff_spec_a53_bits, huff_spec_a53_codes,  32,   1,   0,   5,   7, },
            { huff_spec_a54_bits, huff_spec_a54_codes,  32,   1,   0,   5,   8, },
        },
        {
            { huff_spec_a61_bits, huff_spec_a61_codes,  64,   1,   0,   6,   7, },
            { huff_spec_a62_bits, huff_spec_a62_codes,  64,   1,   0,   6,   7, },
            { huff_spec_a63_bits, huff_spec_a63_codes,  64,   1,   0,   6,   8, },
            { huff_spec_a64_bits, huff_spec_a64_codes,  64,   1,   0,   6,   9, },
        },
        {
            { huff_spec_a71_bits, huff_spec_a71_codes, 128,   1,   0,   7,   8, },
            { huff_spec_a72_bits, huff_spec_a72_codes, 128,   1,   0,   7,   8, },
            { huff_spec_a73_bits, huff_spec_a73_codes, 128,   1,   0,   7,   9, },
            { huff_spec_a74_bits, huff_spec_a74_codes, 128,   1,   0,   7,  10, },
        },
    },
    {
        { { 0 } },
        { { 0 } },
        {
            { 0 },
            { huff_spec_b22_bits, huff_spec_b22_codes,  256,  4,   2,   2,  10, },
            { huff_spec_b23_bits, huff_spec_b23_codes,  256,  4,   2,   2,  10, },
            { huff_spec_b24_bits, huff_spec_b24_codes,  256,  4,   2,   2,  10, },
        },
        {
            { 0 },
            { huff_spec_b32_bits, huff_spec_b32_codes,  64,   2,   1,   3,   9, },
            { huff_spec_b33_bits, huff_spec_b33_codes,  64,   2,   1,   3,  10, },
            { huff_spec_b34_bits, huff_spec_b34_codes,  64,   2,   1,   3,  10, },
        },
        {
            { 0 },
            { huff_spec_b42_bits, huff_spec_b42_codes, 256,   2,   1,   4,  10, },
            { huff_spec_b43_bits, huff_spec_b43_codes, 256,   2,   1,   4,  10, },
            { huff_spec_b44_bits, huff_spec_b44_codes, 256,   2,   1,   4,  10, },
        },
        {
            { 0 },
            { huff_spec_b52_bits, huff_spec_b52_codes,  32,   1,   0,   5,   7, },
            { huff_spec_b53_bits, huff_spec_b53_codes,  32,   1,   0,   5,   8, },
            { huff_spec_b54_bits, huff_spec_b54_codes,  32,   1,   0,   5,   9, },
        },
        {
            { 0 },
            { huff_spec_b62_bits, huff_spec_b62_codes,  64,   1,   0,   6,   8, },
            { huff_spec_b63_bits, huff_spec_b63_codes,  64,   1,   0,   6,   9, },
            { huff_spec_b64_bits, huff_spec_b64_codes,  64,   1,   0,   6,  10, },
        },
        {
            { 0 },
            { huff_spec_b72_bits, huff_spec_b72_codes, 128,   1,   0,   7,   9, },
            { huff_spec_b73_bits, huff_spec_b73_codes, 128,   1,   0,   7,  10, },
            { huff_spec_b74_bits, huff_spec_b74_codes, 128,   1,   0,   7,  10, },
        },
    },
};

#endif /* AVCODEC_ATRAC9TAB_H */
