//
//	Integer Wavelet Classes
//
//  $Id: Wavelet.C,v 4.1.2.3.2.1 1999/07/20 16:15:52 geert Exp $
//
//  Copyright (C) 1996-1999 Department of Computer Science, K.U.Leuven, Belgium
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#include <waili/Util.h>
#include <waili/Channel.h>


//
//	Biorthogonal Cohen-Daubechies-Feauveau Wavelet Filters
//
//
//	D = detail (high freqency)
//	S = signal (low freqency)
//
//
//	(1,1):	g~ = ( 1 -1 ) / 2
//			start = 0
//		h~ = ( 1 1 ) / 2
//			start = 0
//
//		D1 = ( -1 0 )
//		S1 = ( 0 1 ) / 2
//
//		NS = 1
//		ND = -1/2
//
//
//	(1,3):	g~ = ( 1 -1 ) / 2
//			start = 0
//		h~ = ( -1 1 8 8 1 -1 ) / 16
//			start = -2
//
//		D1 = ( -1 0 )
//		S1 = ( 0 1 8 -1 ) / 16
//
//		NS = 1
//		ND = -1/2
//
//
//	(1,5):	g~ = ( 1 -1 ) / 2
//			start = 0
//		h~ = ( 3 -3 -22 22 128 128 22 -22 -3 3 ) / 256
//			start = -4
//
//		D1 = ( -1 0 )
//		S1 = ( 0 -3 22 128 -22 3 ) / 256
//
//		NS = 1
//		ND = -1/2
//
// ----------------------------------------------------------------------------
//
//	(2,2):	g~ = ( 1 -2 1 ) / 4
//			start = 0
//		h~ = ( -1 2 6 2 -1 ) / 8
//			start = -2
//
//		D1 = ( -1 -1 ) / 2
//		S1 = ( 1 1) / 4
//
//		NS = 1
//		ND = -1/2
//
//
//	(2,4):	g~ = ( 1 -2 1 ) / 4
//			start = 0
//		h~ = ( 3 -6 -16 38 90 38 -16 -6 3 ) / 128
//			start = -4
//
//		D1 = ( -1 -1 ) / 2
//		S1 = ( -3 19 19 -3 ) / 64
//
//		NS = 1
//		ND = -1/2
//
//
//	(2,6):	g~ = ( 1 -2 1 ) / 4
//			start = 0
//		h~ = ( -5 10 34 -78 -123 324 700 324 -123 -78 34 10 -5) / 1024
//			start = -6
//
//		D1 = ( -1 -1 ) / 2
//		S1 = ( 5 -39 162 162 -39 5 ) / 512
//
//		NS = 1
//		ND = -1/2
//
// ----------------------------------------------------------------------------
//
//	(3,1):	g~ = ( -1 3 -3 1 ) / 8
//			start = -1
//		h~ = ( -1 3 3 -1 ) / 4
//			start = -1
//
//		S1 = ( -1 0 ) / 3
//		D1 = ( -9 -3 ) / 8
//		S2 = ( 0 4 ) / 9
//
//		NS = 3/2
//		ND = -1/3
//
//
//	(3,3):	g~ = ( -1 3 -3 1 ) / 8
//			start = -1
//		h~ = ( 3 -9 -7 45 45 -7 -9 3 ) / 64
//			start = -3
//
//		S1 = ( -1 0 ) / 3
//		D1 = ( -9 -3 ) / 8
//		S2 = ( 0 3 16 -3 ) / 36
//
//		NS = 3/2
//		ND = -1/3
//
//
//	(3,5):	g~ = ( -1 3 -3 1 ) / 8
//			start = -1
//		h~ = ( -5 15 19 -97 -26 350 350 -26 -97 19 15 -5 ) / 512
//			start = -5
//
//		S1 = ( -1 0 ) / 3
//		D1 = ( -9 -3 ) / 8
//		S2 = ( 0 -5 34 128 -34 5 ) / 288
//
//		NS = 3/2
//		ND = -1/3
//
//
// ----------------------------------------------------------------------------
//
//	(4,2):	g~ = ( -1 4 -6 4 -1 ) / 16
//			start = -1
//		h~ = ( 3 -12 5 40 5 -12 3 ) / 32
//			start = -3
//
//		S1 = ( -1 -1 ) / 4
//		D1 = ( -1 -1 )
//		S2 = ( 3 3 ) / 16
//
//		NS = 2
//		ND = -1/4
//
//
//	(4,4):	g~ = ( -1 4 -6 4 -1 ) / 16
//			start = -1
//		h~ = ( -10 40 -2 -192 140 560 140 -192 -2 40 -10 ) / 512
//			start = -5
//
//		S1 = ( -1 -1 ) / 4
//		D1 = ( -1 -1 )
//		S2 = ( -5 29 29 -5 ) / 128
//
//		NS = 2
//		ND = -1/4
//
//
//	(4,6):	g~ = ( -1 4 -6 4 -1 ) / 16
//			start = -1
//		h~ = ( 35 -140 -55 920 -557 -2932 2625 8400 2625 -2932
//			      -557 920 -55 -140 35) / 8192
//			start = -7
//
//		S1 = ( -1 -1 ) / 4
//		D1 = ( -1 -1 )
//		S2 = ( 35 -265 998 998 -265 35) / 4096
//
//		NS = 2
//		ND = -1/4
//
// ----------------------------------------------------------------------------
//
//	(5,1):	g~ = ( 1 -5 10 -10 5 -1 ) / 32
//			start = -2
//		h~ = ( 3 -15 20 20 -15 3 ) / 16
//			start = -2
//
//		D1 = ( -1 0 ) / 5
//		S1 = ( -15 -5 ) / 24
//		D2 = ( -15 -9 ) / 10
//		S2 = ( 0 1 ) / 3
//
//		NS = 3
//		ND = -1/6
//
//
//	(5,3):	g~ = ( 1 -5 10 -10 5 -1 ) / 32
//			start = -2
//		h~ = ( -5 25 -26 -70 140 140 -70 -26 25 -5 ) / 128
//			start = -4
//
//		D1 = ( -1 0 ) / 5
//		S1 = ( -15 -5 ) / 24
//		D2 = ( -15 -9 ) / 10
//		S2 = ( 0 5 24 -5 ) / 72
//
//		NS = 3
//		ND = -1/6
//
//
//	(5,5):	g~ = ( 1 -5 10 -10 5 -1 ) / 32
//			start = -2
//		h~ = ( 35 -175 120 800 -1357 -1575 4200 4200 -1575 -1357 800
//		       120 -175 35 ) / 4096
//			start = -6
//
//		D1 = ( -1 0 ) / 5
//		S1 = ( -15 -5 ) / 24
//		D2 = ( -15 -9 ) / 10
//		S2 = ( 0 -35 230 768 -230 35 ) / 2304
//
//		NS = 3
//		ND = -1/6
//
//
// ----------------------------------------------------------------------------
//
//	(6,2):	g~ = ( 1 -6 15 -20 15 -6 1 ) / 64
//			start = -2
//		h~ = ( -5 30 -56 -14 154 -14 -56 30 -5 ) / 64
//			start = -4
//
//		D1 = ( -1 -1 ) / 6
//		S1 = ( -9 -9 ) / 16
//		D2 = ( -4 -4 ) / 3
//		S2 = ( 5 5 ) / 32
//
//		NS = 4
//		ND = -1/8
//
//
//	(6,4):	g~ = ( 1 -6 15 -20 15 -6 1 ) / 64
//			start = -2
//		h~ = ( 35 -210 330 470 -1827 252 3948 252 -1827 470 330 -210
//		       35 ) / 2048
//			start = -6
//
//		D1 = ( -1 -1 ) / 6
//		S1 = ( -9 -9 ) / 16
//		D2 = ( -4 -4 ) / 3
//		S2 = ( -35 195 195 -35 ) / 1024
//
//		NS = 4
//		ND = -1/8
//
//
//	(6,6):	g~ = ( 1 -6 15 -20 15 -6 1 ) / 64
//			start = -2
//		h~ = ( -63 378 -476 -1554 4404 1114 -13860 4158 28182 4158
//		       -13860 1114 4404 -1554 -476 378 -63 ) / 16384
//			start = -8
//
//		D1 = ( -1 -1 ) / 6
//		S1 = ( -9 -9 ) / 16
//		D2 = ( -4 -4 ) / 3
//		S2 = ( 63 -469 1686 1686 -469 63 ) / 8192
//
//		NS = 4
//		ND = -1/8
//
//
// ----------------------------------------------------------------------------
//
//
//	Some more wavelets used by JPEG2000
//
//	Biorthogonal CRF (13, 7):
//
//		g~ = ( 1 0 -9 16 -9 0 1 ) / 16
//			start = -4
//		h~ = ( -1 0 14 -16 -31 80 164 80 -31 -16 14 0 -1 ) / 256
//			start = -6
//
//		D1 = ( 1 -9 -9 1 ) / 16
//		S1 = ( -1 5 5 -1 ) / 16
//
//		NS = 1
//		ND = 1
//
//
//	Biorthogonal SWE (13, 7):
//
//		g~ = ( 1 0 -9 16 -9 0 1 ) / 16
//			start = -4
//		h~ = ( -1 0 18 -16 -63 144 348 144 -63 -16 18 0 -1 ) / 512
//			start = -6
//
//		D1 = ( 1 -9 -9 1 ) / 16
//		S1 = ( -1, 9, 9, -1 ) / 32
//
//		NS = 1
//		ND = 1
//
//
// ----------------------------------------------------------------------------
//
//	Growth of the wavelet coefficients for 8-bit (-128..127) input data
//	(obtained by experiments)
//
//
//	CDF (1, 1):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-128..127	-255..255
//			    2		-128..127	-255..255
//			    3		-128..127	-255..255
//			    4		-128..127	-255..255
//			    5		-128..127	-255..255
//			    6		-128..127	-255..255
//			    7		-128..127	-255..255
//			    8		-128..127	-255..255
//			    9		-128..127	-255..255
//			   10		-128..127	-255..255
//
//	CDF (1, 3):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-159..159	-255..255
//			    2		-162..163	-286..286
//			    3		-164..165	-291..291
//			    4		-164..165	-293..293
//			    5		-164..165	-293..293
//			    6		-164..165	-293..293
//			    7		-165..166	-293..293
//			    8		-164..166	-293..293
//			    9		-165..166	-293..293
//			   10		-165..166	-293..293
//
//	CDF (1, 5):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-177..177	-255..255
//			    2		-179..180	-310..310
//			    3		-182..184	-315..315
//			    4		-182..184	-319..319
//			    5		-183..184	-319..319
//			    6		-183..184	-319..319
//			    7		-183..184	-319..319
//			    8		-182..184	-319..319
//			    9		-183..184	-319..319
//
//	CDF (2, 2):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-191..191	-510..510
//			    2		-206..207	-636..638
//			    3		-214..214	-700..698
//			    4		-215..216	-712..714
//			    5		-217..217	-718..718
//			    6		-218..218	-724..720
//			    7		-217..218	-724..722
//			    8		-217..218	-724..720
//			    9		-218..218	-724..726
//			   10		-216..218	-722..724
//
//	CDF (2, 4):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-215..215	-510..510
//			    2		-222..223	-708..708
//			    3		-232..232	-776..776
//			    4		-232..234	-796..800
//			    5		-232..234	-802..802
//			    6		-233..234	-804..804
//			    7		-233..234	-804..804
//			    8		-233..234	-802..802
//			    9		-233..234	-804..806
//			   10		-233..234	-804..804
//
//	CDF (2, 6):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-230..230	-510..510
//			    2		-236..237	-758..758
//			    3		-245..246	-828..828
//			    4		-245..248	-848..848
//			    5		-245..247	-852..856
//			    6		-246..247	-854..856
//
//	CDF (4, 2):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-318..319	-1020..1018
//			    2		-524..527	-2096..2094
//			    3		-949..951	-3498..3496
//			    4		-1580..1586	-6890..6880
//			    5		-2726..2731	-11102..11094
//			    6		-4623..4629	-19380..19380
//			    7		-7928..7933	-32482..32488
//			    8		-13522..13530	-55888..55892
//			    9		-23105..23116	-94794..94794
//			   10		-39428..39438	-162448..162446
//
//	CDF (4, 4):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-330..331	-1020..1018
//			    2		-456..456	-2238..2236
//			    3		-583..587	-3410..3410
//			    4		-713..715	-4212..4216
//			    5		-859..865	-5212..5210
//			    6		-1015..1019	-6274..6272
//			    7		-1180..1187	-7300..7304
//			    8		-1370..1377	-8574..8566
//			    9		-1582..1590	-9948..9948
//			   10		-1819..1829	-11434..11450
//
//	CDF (4, 6):	# levels	Low-Pass	High-Pass
//			--------	--------	---------
//			    1		-356..357	-1020..1018
//			    2		-463..464	-2388..2386
//			    3		-499..502	-3366..3364
//			    4		-539..544	-3742..3746
//			    5		-563..568	-3980..3978
//			    6		-576..582	-4148..4136
//			    7		-583..589	-4228..4228
//


const char *Wavelet::rcsid = "$Id: Wavelet.C,v 4.1.2.3.2.1 1999/07/20 16:15:52 geert Exp $";


    //  Biorthogonal Cohen-Daubechies-Feauveau

Wavelet *Wavelet::CreateCDF(u_int np, u_int nd)
{
    switch (np) {
	case 0:
	    switch (nd) {
		case 0:
		    return(new Wavelet_Lazy);
	    }
	    break;
	case 1:
	    switch (nd) {
		case 1:
		    return(new Wavelet_CDF_1_1);
		case 3:
		    return(new Wavelet_CDF_1_3);
		case 5:
		    return(new Wavelet_CDF_1_5);
	    }
	    break;
	case 2:
	    switch (nd) {
		case 2:
		    return(new Wavelet_CDF_2_2);
		case 4:
		    return(new Wavelet_CDF_2_4);
		case 6:
		    return(new Wavelet_CDF_2_6);
	    }
	    break;
	case 4:
	    switch (nd) {
		case 2:
		    return(new Wavelet_CDF_4_2);
		case 4:
		    return(new Wavelet_CDF_4_4);
		case 6:
		    return(new Wavelet_CDF_4_6);
	    }
	    break;
    }
    if ((np & 1) == (nd & 1))
	Die("%s: Cohen-Daubechies-Feauveau (%d, %d) wasn't implemented\n",
	    __FUNCTION__, np, nd);
    else
	Die("%s: Cohen-Daubechies-Feauveau (%d, %d) does not exist\n",
	    __FUNCTION__, np, nd);
}


    //  Create a Wavelet from an ID

Wavelet *Wavelet::CreateFromID(u8 id)
{
    switch (id) {
	case ID_CDF_1_1:
	    return new Wavelet_CDF_1_1;
	case ID_CDF_1_3:
	    return new Wavelet_CDF_1_3;
	case ID_CDF_1_5:
	    return new Wavelet_CDF_1_5;
	case ID_CDF_2_2:
	    return new Wavelet_CDF_2_2;
	case ID_CDF_2_4:
	    return new Wavelet_CDF_2_4;
	case ID_CDF_2_6:
	    return new Wavelet_CDF_2_6;
	case ID_CDF_4_2:
	    return new Wavelet_CDF_4_2;
	case ID_CDF_4_4:
	    return new Wavelet_CDF_4_4;
	case ID_CDF_4_6:
	    return new Wavelet_CDF_4_6;
	case ID_CRF_13_7:
	    return new Wavelet_CRF_13_7;
	case ID_SWE_13_7:
	    return new Wavelet_SWE_13_7;
	default:
	    Die("%s: Illegal ID %u\n", __FUNCTION__, id);
    }
}
