!{\src2tex{textfont=tt}}
!!****f* ABINIT/mklocl_wavelets
!!
!! NAME
!! mklocl_wavelets
!!
!! FUNCTION
!! Compute the ionic local potential when the pseudo-potentials are GTH, using
!! the special decomposition of these pseudo. The resulting potential is computed with
!! free boundary conditions. It gives the same result than mklocl_realspace for the
!! GTH pseudo only with a different way to compute the potential.
!!
!! Optionally compute :
!!  option=1 : local ionic potential throughout unit cell
!!  option=2 : contribution of local ionic potential to E gradient wrt xred
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (DC)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  dtset <type(dataset_type)>=internal variables used by wavelets, describing
!!   | wvl_internal=desciption of the wavelet box.
!!   | natom=number of atoms.
!!  mpi_enreg=informations about MPI parallelization
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!
!! OUTPUT
!!  (if option==1) vpsp(dtset%nfft)=the potential resulting from the ionic
!!                 density of charge.
!!  (if option==2) grtn(3,natom)=grads of Etot wrt tn. These gradients are in
!!                 reduced coordinates. Multiply them by rprimd to get
!!                 gradients in cartesian coordinates.
!!
!! SIDE EFFECTS
!!  xred(3,natom)=reduced dimensionless atomic coordinates (in fact IN but here
!!                because of INOUT xredxcart() behavior).
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE
#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine mklocl_wavelets(dtset, grtn, mpi_enreg, option, psps, rhor, rprimd, &
     & vpsp, xred)

  use defs_basis
  use defs_datatypes
#if defined HAVE_BIGDFT
  use libbigdft
#endif

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_12geometry
 use interfaces_12poisson
#endif
!End of the abilint section

  implicit none

!Arguments ------------------------------------
  !scalars
  integer, intent(in)                       :: option
  type(dataset_type), intent(in)            :: dtset
  type(MPI_type), intent(in)                :: mpi_enreg
  type(pseudopotential_type), intent(in)    :: psps
  !arrays
  real(dp), intent(inout)                     :: grtn(3, dtset%natom)
  real(dp), intent(in)                      :: rprimd(3, 3)
  real(dp), intent(in)                      :: rhor(dtset%nfft, dtset%nspden)
  real(dp), intent(inout)                     :: vpsp(dtset%nfft)
  real(dp), intent(inout)                   :: xred(3, dtset%natom)

!Local variables-------------------------------
  !scalars
  integer               :: i, ia, igeo
  real(dp)              :: energy
  character(len=500)    :: message
  !arrays
  integer, allocatable  :: nelpsp(:)
  real(dp), allocatable :: xcart(:,:)
  real(dp), allocatable :: ionDensity(:), vhartr(:), gxyz(:, :)
  character*20          :: atomnames(100)

! *********************************************************************

#if defined HAVE_BIGDFT
  ! Allocations
  allocate(nelpsp(dtset%ntypat))

  ! Store xcart for each atom
  allocate(xcart(3, dtset%natom))
  call xredxcart(dtset%natom, 1, rprimd, xcart, xred)

  ! Dummy arguments
  atomnames(:) = "'Unknown name'"

  ! Need to transtype from real to integer
  do i = 1, dtset%ntypat, 1
     nelpsp(i) = int(psps%ziontypat(i))
  end do

  if (option == 1) then
     write(message, '(a,a)' ) ch10,&
          &  ' mklocl_wavelets : Create local potential from ions.'
     call wrtout(6,message,'COLL')

     ! Allocations
     allocate(ionDensity(dtset%wvl_internal%nDpPoints))

     ! Precalculate ionic potential from PSP charge densities and local Gaussian terms
     call input_rho_ion(mpi_enreg%me, dtset%ntypat, dtset%natom, dtset%typat, &
          & atomnames, xcart, psps%gth_params%psppar, nelpsp, &
          & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
          & dtset%wvl_internal%nSize(3), dtset%wvl_hgrid, ionDensity, energy)

#if defined MPI
     ! NOT IMPLEMENTED YET
     !  call ParPSolver_Kernel(2 * box%nSize(1) + box%buffer, 2 * box%nSize(2) + box%buffer, &
     !                       & 2 * box%nSize(3) + box%buffer,nfft1,nfft2,nfft3,hgridh,pkernel,.false.,  &
     !               rhopot,pot_ion,ehart,eexcu,vexcu,iproc,nproc)
#else
     call PSolver_hartree(dtset, 1, dtset%nfft, dtset%ngfft, ionDensity, rprimd, vpsp)
#endif

     call addlocgauspsp(mpi_enreg%me, dtset%ntypat, dtset%natom, dtset%typat, &
          & atomnames, xcart, psps%gth_params%psppar, dtset%wvl_internal%nSize(1), &
          & dtset%wvl_internal%nSize(2), dtset%wvl_internal%nSize(3), &
          & dtset%wvl_hgrid, vpsp)

     ! Deallocations
     deallocate(ionDensity)

  else if (option == 2) then
     ! Compute forces
     write(message, '(a)' ) 'mklocl_wavelets() : compute local forces.'
     call wrtout(6,message,'COLL')

     ! Compute Hartree's potential from rhor.
     allocate(vhartr(dtset%nfft))
#if defined MPI
     ! NOT IMPLEMENTED YET
     !  call ParPSolver_Kernel(2 * box%nSize(1) + box%buffer, 2 * box%nSize(2) + box%buffer, &
     !                       & 2 * box%nSize(3) + box%buffer,nfft1,nfft2,nfft3,hgridh,pkernel,.false.,  &
     !               rhopot,pot_ion,ehart,eexcu,vexcu,iproc,nproc)
#else
     call PSolver_hartree(dtset, 1, dtset%nfft, dtset%ngfft, rhor, rprimd, vhartr)
#endif

     ! Allocate temporary array for forces.
     allocate(gxyz(3, dtset%natom))

     ! calculate local part of the forces grtn (BigDFT routine)
     call local_forces(mpi_enreg%me, mpi_enreg%nproc, dtset%ntypat, dtset%natom, &
          & dtset%typat, atomnames, xcart, psps%gth_params%psppar, nelpsp, &
          & dtset%wvl_hgrid, dtset%wvl_internal%nSize(1), &
          & dtset%wvl_internal%nSize(2), dtset%wvl_internal%nSize(3), rhor, vhartr, &
          & gxyz)
     deallocate(vhartr)

     ! Forces should be in reduced coordinates.
     do ia = 1, dtset%natom, 1
        do igeo = 1, 3, 1
           grtn(igeo, ia) = rprimd(1, igeo) * gxyz(1, ia) + &
                & rprimd(2, igeo) * gxyz(2, ia) + &
                & rprimd(3, igeo) * gxyz(3, ia)
        end do
     end do

     ! Deallocate local variables
     deallocate(gxyz)
  else ! option switch
     write(message, '(a,a,a,a)' ) ch10,&
          &  ' mklocl_wavelets : internal error, option should be 1 or 2.'
     call wrtout(6,message,'COLL')
     call leave_new('COLL')
  end if
  
  ! Deallocations
  deallocate(xcart)
  deallocate(nelpsp)

#else
  write(message, '(a,a,a,a)' ) ch10,&
    &  ' mklocl_wavelets : BigDFT library is not compiled.', ch10, &
    &  '   Action, used the flag --enable-bigdft when configuring.'
  call wrtout(6,message,'COLL')
  call leave_new('COLL')
#endif
end subroutine mklocl_wavelets
!!***
