/******************************************************************************
 * $Id$
 *
 * Project:  Arc/Info Binary Grid Translator
 * Purpose:  Code for decoding CCITT RLE (G1) compressed data.
 * Author:   Frank Warmerdam, warmerdam@pobox.com
 *           Code derived from libtiff (tif_fax3), which originally was
 *           derived from code by Frank Cringle in viewfax.
 *
 ******************************************************************************
 * Copyright (c) 2002 Frank Warmerdam <warmerdam@pobox.com>
 * Copyright (c) 1990-1997 Sam Leffler
 * Copyright (c) 1991-1997 Silicon Graphics, Inc.
 * Copyright (c) 2009-2013, Even Rouault <even dot rouault at spatialys.com>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 ****************************************************************************/

#include <assert.h>
#include "aigrid.h"

/* ==================================================================== */
/*      Various declarations for the decompression state machine        */
/*      cribbed from tif_fax3.h in libtiff.                             */
/* ==================================================================== */

#define FAXMODE_CLASSIC 0x0000       /* default, include RTC */
#define FAXMODE_NORTC 0x0001         /* no RTC at end of data */
#define FAXMODE_NOEOL 0x0002         /* no EOL code at end of row */
#define FAXMODE_BYTEALIGN 0x0004     /* byte align row */
#define FAXMODE_WORDALIGN 0x0008     /* word align row */
#define FAXMODE_CLASSF FAXMODE_NORTC /* TIFF Class F */

/*
 * To override the default routine used to image decoded
 * spans one can use the pseudo tag TIFFTAG_FAXFILLFUNC.
 * The routine must have the type signature given below;
 * for example:
 *
 * fillruns(unsigned char* buf, uint32* runs, uint32* erun, uint32 lastx)
 *
 * where buf is place to set the bits, runs is the array of b&w run
 * lengths (white then black), erun is the last run in the array, and
 * lastx is the width of the row in pixels.  Fill routines can assume
 * the run array has room for at least lastx runs and can overwrite
 * data in the run array as needed (e.g. to append zero runs to bring
 * the count up to a nice multiple).
 */
typedef void (*TIFFFaxFillFunc)(unsigned char *, GUInt32 *, GUInt32 *, GUInt32);

/* finite state machine codes */

#define S_Null 0
#define S_Pass 1
#define S_Horiz 2
#define S_V0 3
#define S_VR 4
#define S_VL 5
#define S_Ext 6
#define S_TermW 7
#define S_TermB 8
#define S_MakeUpW 9
#define S_MakeUpB 10
#define S_MakeUp 11
#define S_EOL 12

typedef struct
{                        /* state table entry */
    unsigned char State; /* see above */
    unsigned char Width; /* width of code in bits */
    GUInt16 Param;       /* unsigned 16-bit run length in bits */
} TIFFFaxTabEnt;

#if 0  /* Unused */
static const TIFFFaxTabEnt aig_TIFFFaxMainTable[128] = {
{12,7,0},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},{1,4,0},{3,1,0},{5,3,1},{3,1,0},
{2,3,0},{3,1,0},{4,3,1},{3,1,0},{5,6,2},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},
{1,4,0},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},{5,7,3},{3,1,0},{5,3,1},{3,1,0},
{2,3,0},{3,1,0},{4,3,1},{3,1,0},{1,4,0},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},
{4,6,2},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},{1,4,0},{3,1,0},{5,3,1},{3,1,0},
{2,3,0},{3,1,0},{4,3,1},{3,1,0},{6,7,0},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},
{1,4,0},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},{5,6,2},{3,1,0},{5,3,1},{3,1,0},
{2,3,0},{3,1,0},{4,3,1},{3,1,0},{1,4,0},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},
{4,7,3},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},{1,4,0},{3,1,0},{5,3,1},{3,1,0},
{2,3,0},{3,1,0},{4,3,1},{3,1,0},{4,6,2},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0},
{1,4,0},{3,1,0},{5,3,1},{3,1,0},{2,3,0},{3,1,0},{4,3,1},{3,1,0}
};
#endif /* Unused */

static const TIFFFaxTabEnt aig_TIFFFaxWhiteTable[4096] = {
    {12, 11, 0},    {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 11, 1792}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 11, 1856}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2112}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2368}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 1984}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 11, 1920}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2240}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2496}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {12, 11, 0},    {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 11, 1792}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 11, 1856}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2176}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2432}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2048}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 11, 1920}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1472}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1216}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 960},  {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 704},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2304}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 832},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1600}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1344}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1088}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {0, 0, 0},      {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 39},     {7, 6, 16},  {9, 8, 576},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 55},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 45},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 53},     {7, 5, 9},   {9, 8, 448},  {7, 4, 6},
    {7, 8, 35},     {9, 5, 128}, {7, 8, 51},   {7, 6, 15},
    {7, 8, 63},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1536}, {7, 4, 5},
    {7, 8, 43},     {7, 6, 17},  {9, 9, 1280}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 29},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 33},     {9, 5, 128}, {7, 8, 49},   {7, 6, 14},
    {7, 8, 61},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 47},     {7, 4, 3},   {7, 8, 59},   {7, 4, 5},
    {7, 8, 41},     {7, 6, 16},  {9, 9, 1024}, {7, 4, 6},
    {7, 8, 31},     {7, 5, 8},   {7, 8, 57},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 9, 768},  {7, 4, 6},
    {7, 8, 37},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 320},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {11, 12, 2560}, {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 7, 20},     {9, 5, 128}, {7, 7, 24},   {7, 6, 14},
    {7, 7, 28},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 23},     {7, 4, 3},   {7, 7, 27},   {7, 4, 5},
    {7, 8, 40},     {7, 6, 16},  {9, 9, 896},  {7, 4, 6},
    {7, 7, 19},     {7, 5, 8},   {7, 8, 56},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 46},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 8, 54},     {7, 5, 9},   {9, 8, 512},  {7, 4, 6},
    {7, 8, 36},     {9, 5, 128}, {7, 8, 52},   {7, 6, 15},
    {7, 8, 0},      {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {9, 9, 1728}, {7, 4, 5},
    {7, 8, 44},     {7, 6, 17},  {9, 9, 1408}, {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 30},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 6, 12},     {7, 5, 9},   {9, 6, 1664}, {7, 4, 6},
    {7, 8, 34},     {9, 5, 128}, {7, 8, 50},   {7, 6, 14},
    {7, 8, 62},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 8, 48},     {7, 4, 3},   {7, 8, 60},   {7, 4, 5},
    {7, 8, 42},     {7, 6, 16},  {9, 9, 1152}, {7, 4, 6},
    {7, 8, 32},     {7, 5, 8},   {7, 8, 58},   {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 7, 22},     {7, 4, 3},   {7, 5, 11},   {7, 4, 5},
    {7, 7, 26},     {7, 5, 9},   {9, 8, 640},  {7, 4, 6},
    {7, 8, 38},     {9, 5, 128}, {7, 7, 25},   {7, 6, 15},
    {9, 8, 384},    {7, 4, 4},   {7, 4, 2},    {7, 4, 7},
    {7, 6, 13},     {7, 4, 3},   {7, 7, 18},   {7, 4, 5},
    {7, 7, 21},     {7, 6, 17},  {9, 7, 256},  {7, 4, 6},
    {7, 6, 1},      {7, 5, 8},   {9, 6, 192},  {9, 5, 64},
    {7, 5, 10},     {7, 4, 4},   {7, 4, 2},    {7, 4, 7}};
static const TIFFFaxTabEnt aig_TIFFFaxBlackTable[8192] = {
    {12, 11, 0},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1792}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 23},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 20},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 25},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 128},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 56},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 30},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1856}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 57},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 21},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 54},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 52},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 48},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2112}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 44},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 36},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 384},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 28},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 60},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 40},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2368}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 1984}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 50},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 34},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1664}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 26},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1408}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 32},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1920}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 61},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 42},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1024}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 768},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 62},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2240}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 46},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 38},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 512},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 19},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 24},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 22},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2496}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {12, 11, 0},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1792}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 23},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 20},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 25},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 192},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1280}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 31},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1856}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 58},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 21},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 896},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 640},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 49},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2176}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 45},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 37},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 448},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 29},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1536}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 41},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2432}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2048}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 51},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 35},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 320},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 27},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 59},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 33},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1920}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 256},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 43},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1152}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 55},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 63},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2304}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 47},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 39},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 53},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 19},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 24},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 22},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2560}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {12, 11, 0},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1792}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 23},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 20},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 25},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 128},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 56},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 30},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1856}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 57},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 21},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 54},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 52},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 48},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2112}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 44},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 36},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 384},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 28},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 60},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 40},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2368}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 1984}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 50},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 34},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1728}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 26},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1472}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 32},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1920}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 61},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 42},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1088}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 832},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 62},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2240}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 46},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 38},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 576},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 19},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 24},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 22},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2496}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {12, 11, 0},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1792}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 23},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 20},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 25},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 192},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1344}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 31},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1856}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 58},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 21},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 960},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 704},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 49},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2176}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 45},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 37},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 448},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 29},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1600}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 41},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2432}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 18},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 17},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2048}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 51},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 35},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 320},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 27},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 59},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 33},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 11, 1920}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 12, 256},  {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 43},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 13, 1216}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 9, 15},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 55},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 63},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2304}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 47},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 39},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 12, 53},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {0, 0, 0},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 13},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 19},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 24},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 11, 22},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {11, 12, 2560}, {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 10},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 16},    {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 10, 0},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {10, 10, 64},   {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 9},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 11},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 8, 14},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 6, 8},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 7, 12},     {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 6},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2},
    {8, 5, 7},      {8, 2, 3}, {8, 3, 1}, {8, 2, 2},
    {8, 4, 5},      {8, 2, 3}, {8, 3, 4}, {8, 2, 2}};

static const unsigned char aig_TIFFBitRevTable[256] = {
    0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0, 0x10, 0x90, 0x50, 0xd0,
    0x30, 0xb0, 0x70, 0xf0, 0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
    0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8, 0x04, 0x84, 0x44, 0xc4,
    0x24, 0xa4, 0x64, 0xe4, 0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
    0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec, 0x1c, 0x9c, 0x5c, 0xdc,
    0x3c, 0xbc, 0x7c, 0xfc, 0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
    0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2, 0x0a, 0x8a, 0x4a, 0xca,
    0x2a, 0xaa, 0x6a, 0xea, 0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
    0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6, 0x16, 0x96, 0x56, 0xd6,
    0x36, 0xb6, 0x76, 0xf6, 0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
    0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe, 0x01, 0x81, 0x41, 0xc1,
    0x21, 0xa1, 0x61, 0xe1, 0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
    0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9, 0x19, 0x99, 0x59, 0xd9,
    0x39, 0xb9, 0x79, 0xf9, 0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
    0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5, 0x0d, 0x8d, 0x4d, 0xcd,
    0x2d, 0xad, 0x6d, 0xed, 0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
    0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3, 0x13, 0x93, 0x53, 0xd3,
    0x33, 0xb3, 0x73, 0xf3, 0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
    0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb, 0x07, 0x87, 0x47, 0xc7,
    0x27, 0xa7, 0x67, 0xe7, 0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
    0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef, 0x1f, 0x9f, 0x5f, 0xdf,
    0x3f, 0xbf, 0x7f, 0xff};

#if 0  /* Unused */
static const unsigned char aig_TIFFNoBitRevTable[256] = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
    0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
    0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
    0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
    0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
    0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
    0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
    0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
    0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
    0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
    0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
    0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
    0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
    0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
    0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
    0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
    0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};
#endif /* Unused */

/*
 * The following macros define the majority of the G3/G4 decoder
 * algorithm using the state tables defined elsewhere.  To build
 * a decoder you need some setup code and some glue code. Note
 * that you may also need/want to change the way the NeedBits*
 * macros get input data if, for example, you know the data to be
 * decoded is properly aligned and oriented (doing so before running
 * the decoder can be a big performance win).
 *
 * Consult the decoder in the TIFF library for an idea of what you
 * need to define and setup to make use of these definitions.
 *
 * NB: to enable a debugging version of these macros define FAX3_DEBUG
 *     before including this file.  Trace output goes to stdout.
 */

#ifndef EndOfData
#define EndOfData() (cp >= ep)
#endif
/*
 * Need <=8 or <=16 bits of input data.  Unlike viewfax we
 * cannot use/assume a word-aligned, properly bit swizzled
 * input data set because data may come from an arbitrarily
 * aligned, read-only source such as a memory-mapped file.
 * Note also that the viewfax decoder does not check for
 * running off the end of the input data buffer.  This is
 * possible for G3-encoded data because it prescans the input
 * data to count EOL markers, but can cause problems for G4
 * data.  In any event, we don't prescan and must watch for
 * running out of data since we can't permit the library to
 * scan past the end of the input data buffer.
 *
 * Finally, note that we must handle remaindered data at the end
 * of a strip specially.  The coder asks for a fixed number of
 * bits when scanning for the next code.  This may be more bits
 * than are actually present in the data stream.  If we appear
 * to run out of data but still have some number of valid bits
 * remaining then we makeup the requested amount with zeros and
 * return successfully.  If the returned data is incorrect then
 * we should be called again and get a premature EOF error;
 * otherwise we should get the right answer.
 */
#ifndef NeedBits8
#define NeedBits8(n, eoflab)                                                   \
    do                                                                         \
    {                                                                          \
        if (BitsAvail < (n))                                                   \
        {                                                                      \
            if (EndOfData())                                                   \
            {                                                                  \
                if (BitsAvail == 0) /* no valid bits */                        \
                    goto eoflab;                                               \
                BitsAvail = (n); /* pad with zeros */                          \
            }                                                                  \
            else                                                               \
            {                                                                  \
                BitAcc |= ((GUInt32)bitmap[*cp++]) << BitsAvail;               \
                BitsAvail += 8;                                                \
            }                                                                  \
        }                                                                      \
    } while (0)
#endif
#ifndef NeedBits16
#define NeedBits16(n, eoflab)                                                  \
    do                                                                         \
    {                                                                          \
        if (BitsAvail < (n))                                                   \
        {                                                                      \
            if (EndOfData())                                                   \
            {                                                                  \
                if (BitsAvail == 0) /* no valid bits */                        \
                    goto eoflab;                                               \
                BitsAvail = (n); /* pad with zeros */                          \
            }                                                                  \
            else                                                               \
            {                                                                  \
                BitAcc |= ((GUInt32)bitmap[*cp++]) << BitsAvail;               \
                if ((BitsAvail += 8) < (n))                                    \
                {                                                              \
                    if (EndOfData())                                           \
                    {                                                          \
                        /* NB: we know BitsAvail is non-zero here */           \
                        BitsAvail = (n); /* pad with zeros */                  \
                    }                                                          \
                    else                                                       \
                    {                                                          \
                        BitAcc |= ((GUInt32)bitmap[*cp++]) << BitsAvail;       \
                        BitsAvail += 8;                                        \
                    }                                                          \
                }                                                              \
            }                                                                  \
        }                                                                      \
    } while (0)
#endif
#define GetBits(n) (BitAcc & ((1 << (n)) - 1))
#define ClrBits(n)                                                             \
    do                                                                         \
    {                                                                          \
        BitsAvail -= (n);                                                      \
        BitAcc >>= (n);                                                        \
    } while (0)

#ifdef FAX3_DEBUG
static const char *StateNames[] = {
    "Null   ", "Pass   ", "Horiz  ", "V0     ", "VR     ", "VL     ", "Ext    ",
    "TermW  ", "TermB  ", "MakeUpW", "MakeUpB", "MakeUp ", "EOL    ",
};
#define DEBUG_SHOW putchar(BitAcc & (1 << t) ? '1' : '0')
#define LOOKUP8(wid, tab, eoflab)                                              \
    do                                                                         \
    {                                                                          \
        int t;                                                                 \
        NeedBits8(wid, eoflab);                                                \
        TabEnt = tab + GetBits(wid);                                           \
        printf("%08lX/%d: %s%5d\t", (long)BitAcc, BitsAvail, /*ok*/            \
               StateNames[TabEnt->State], TabEnt->Param);                      \
        for (t = 0; t < TabEnt->Width; t++)                                    \
            DEBUG_SHOW;                                                        \
        putchar('\n');                                                         \
        fflush(stdout);                                                        \
        ClrBits(TabEnt->Width);                                                \
    } while (0)
#define LOOKUP16(wid, tab, eoflab)                                             \
    do                                                                         \
    {                                                                          \
        int t;                                                                 \
        NeedBits16(wid, eoflab);                                               \
        TabEnt = tab + GetBits(wid);                                           \
        printf("%08lX/%d: %s%5d\t", (long)BitAcc, BitsAvail, /*ok*/            \
               StateNames[TabEnt->State], TabEnt->Param);                      \
        for (t = 0; t < TabEnt->Width; t++)                                    \
            DEBUG_SHOW;                                                        \
        putchar('\n');                                                         \
        fflush(stdout);                                                        \
        ClrBits(TabEnt->Width);                                                \
    } while (0)

#define SETVAL(x)                                                              \
    do                                                                         \
    {                                                                          \
        *pa++ = RunLength + (x);                                               \
        printf("SETVAL: %d\t%d\n", RunLength + (x), a0); /*ok*/                \
        a0 += x;                                                               \
        RunLength = 0;                                                         \
    } while (0)
#else
#define LOOKUP8(wid, tab, eoflab)                                              \
    do                                                                         \
    {                                                                          \
        NeedBits8(wid, eoflab);                                                \
        TabEnt = tab + GetBits(wid);                                           \
        ClrBits(TabEnt->Width);                                                \
    } while (0)
#define LOOKUP16(wid, tab, eoflab)                                             \
    do                                                                         \
    {                                                                          \
        NeedBits16(wid, eoflab);                                               \
        TabEnt = tab + GetBits(wid);                                           \
        ClrBits(TabEnt->Width);                                                \
    } while (0)

/*
 * Append a run to the run length array for the
 * current row and reset decoding state.
 */
#define SETVAL(x)                                                              \
    do                                                                         \
    {                                                                          \
        *pa++ = RunLength + (x);                                               \
        a0 += (x);                                                             \
        RunLength = 0;                                                         \
    } while (0)
#endif

/*
 * Synchronize input decoding at the start of each
 * row by scanning for an EOL (if appropriate) and
 * skipping any trash data that might be present
 * after a decoding error.  Note that the decoding
 * done elsewhere that recognizes an EOL only consumes
 * 11 consecutive zero bits.  This means that if EOLcnt
 * is non-zero then we still need to scan for the final flag
 * bit that is part of the EOL code.
 */
#define SYNC_EOL(eoflab)                                                       \
    do                                                                         \
    {                                                                          \
        if (EOLcnt == 0)                                                       \
        {                                                                      \
            for (;;)                                                           \
            {                                                                  \
                NeedBits16(11, eoflab);                                        \
                if (GetBits(11) == 0)                                          \
                    break;                                                     \
                ClrBits(1);                                                    \
            }                                                                  \
        }                                                                      \
        for (;;)                                                               \
        {                                                                      \
            NeedBits8(8, eoflab);                                              \
            if (GetBits(8))                                                    \
                break;                                                         \
            ClrBits(8);                                                        \
        }                                                                      \
        while (GetBits(1) == 0)                                                \
            ClrBits(1);                                                        \
        ClrBits(1); /* EOL bit */                                              \
        EOLcnt = 0; /* reset EOL counter/flag */                               \
    } while (0)

/*
 * Cleanup the array of runs after decoding a row.
 * We adjust final runs to insure the user buffer is not
 * overwritten and/or undecoded area is white filled.
 */
#define CLEANUP_RUNS()                                                         \
    do                                                                         \
    {                                                                          \
        if (RunLength)                                                         \
            SETVAL(0);                                                         \
        if (a0 != lastx)                                                       \
        {                                                                      \
            badlength(a0, lastx);                                              \
            while (a0 > lastx && pa > thisrun)                                 \
                a0 -= *--pa;                                                   \
            if (a0 < lastx)                                                    \
            {                                                                  \
                if (a0 < 0)                                                    \
                    a0 = 0;                                                    \
                if ((pa - thisrun) & 1)                                        \
                    SETVAL(0);                                                 \
                SETVAL(lastx - a0);                                            \
            }                                                                  \
            else if (a0 > lastx)                                               \
            {                                                                  \
                SETVAL(lastx);                                                 \
                SETVAL(0);                                                     \
            }                                                                  \
        }                                                                      \
    } while (0)

/*
 * Decode a line of 1D-encoded data.
 *
 * The line expanders are written as macros so that they can be reused
 * but still have direct access to the local variables of the "calling"
 * function.
 *
 * Note that unlike the original version we have to explicitly test for
 * a0 >= lastx after each black/white run is decoded.  This is because
 * the original code depended on the input data being zero-padded to
 * insure the decoder recognized an EOL before running out of data.
 */

/*
 * Update the value of b1 using the array
 * of runs for the reference line.
 */
#define CHECK_b1                                                               \
    do                                                                         \
    {                                                                          \
        if (pa != thisrun)                                                     \
            while (b1 <= a0 && b1 < lastx)                                     \
            {                                                                  \
                b1 += pb[0] + pb[1];                                           \
                pb += 2;                                                       \
            }                                                                  \
    } while (0)

/* ==================================================================== */
/*      Declarations from tif_fax3.c                                    */
/* ==================================================================== */

/*
 * Compression+decompression state blocks are
 * derived from this ``base state'' block.
 */
typedef struct
{
    int rw_mode;       /* O_RDONLY for decode, else encode */
    int mode;          /* operating mode */
    GUInt32 rowbytes;  /* bytes in a decoded scanline */
    GUInt32 rowpixels; /* pixels in a scanline */

    GUInt16 cleanfaxdata; /* CleanFaxData tag */
    GUInt32 badfaxrun;    /* BadFaxRun tag */
    GUInt32 badfaxlines;  /* BadFaxLines tag */
    GUInt32 groupoptions; /* Group 3/4 options tag */
    GUInt32 recvparams;   /* encoded Class 2 session params */
    char *subaddress;     /* subaddress string */
    GUInt32 recvtime;     /* time spent receiving (secs) */
} Fax3BaseState;

#define Fax3State(tif) ((Fax3BaseState *)tif)

typedef struct
{
    Fax3BaseState b;
    const unsigned char *bitmap; /* bit reversal table */
    GUInt32 data;                /* current i/o byte/word */
    int bit;                     /* current i/o bit in byte */
    int EOLcnt;                  /* count of EOL codes recognized */
    TIFFFaxFillFunc fill;        /* fill routine */
    GUInt32 *runs;               /* b&w runs for current/previous row */
    GUInt32 *refruns;            /* runs for reference line */
    GUInt32 *curruns;            /* runs for current line */
} Fax3DecodeState;

#define DecoderState(tif) ((Fax3DecodeState *)Fax3State(tif))

typedef enum
{
    G3_1D,
    G3_2D
} Ttag;
#ifdef notdef
typedef struct
{
    Fax3BaseState b;
    int data;               /* current i/o byte */
    int bit;                /* current i/o bit in byte */
    Ttag tag;               /* encoding state */
    unsigned char *refline; /* reference line for 2d decoding */
    int k;                  /* #rows left that can be 2d encoded */
    int maxk;               /* max #rows that can be 2d encoded */
} Fax3EncodeState;

#define EncoderState(tif) ((Fax3EncodeState *)Fax3State(tif))
#endif

#define is2DEncoding(sp) (sp->b.groupoptions & GROUP3OPT_2DENCODING)
#define isAligned(p, t) ((((size_t)(p)) & (sizeof(t) - 1)) == 0)

/*
 * Group 3 and Group 4 Decoding.
 */

/*
 * These macros glue the TIFF library state to
 * the state expected by Frank's decoder.
 */
#define DECLARE_STATE(tif, sp)                                                 \
    Fax3DecodeState *sp = DecoderState(tif);                                   \
    int a0;                                   /* reference element */          \
    int lastx = sp->b.rowpixels;              /* last element in row */        \
    GUInt32 BitAcc;                           /* bit accumulator */            \
    int BitsAvail;                            /* # valid bits in BitAcc */     \
    int RunLength;                            /* length of current run */      \
    unsigned char *cp;                        /* next byte of input data */    \
    unsigned char *ep;                        /* end of input data */          \
    GUInt32 *pa;                              /* place to stuff next run */    \
    GUInt32 *thisrun;                         /* current row's run array */    \
    int EOLcnt;                               /* # EOL codes recognized */     \
    const unsigned char *bitmap = sp->bitmap; /* input data bit reverser */    \
    const TIFFFaxTabEnt *TabEnt

#define DECLARE_STATE_2D(tif, sp, mod)                                         \
    DECLARE_STATE(tif, sp, mod);                                               \
    int b1;     /* next change on prev line */                                 \
    GUInt32 *pb /* next run in reference line */

/**
 * Load any state that may be hanged during decoding.
 */
#define CACHE_STATE(sp)                                                        \
    do                                                                         \
    {                                                                          \
        BitAcc = sp->data;                                                     \
        BitsAvail = sp->bit;                                                   \
        EOLcnt = sp->EOLcnt;                                                   \
        cp = (unsigned char *)rawcp;                                           \
        ep = cp + rawcc;                                                       \
    } while (0)
/*
 * Save state possibly changed during decoding.
 */
#define UNCACHE_STATE(sp)                                                      \
    do                                                                         \
    {                                                                          \
        sp->bit = BitsAvail;                                                   \
        sp->data = BitAcc;                                                     \
        sp->EOLcnt = EOLcnt;                                                   \
        /* below updates not needed in AIG case */                             \
        /* rawcc -= (int)((unsigned char *) cp - rawcp); */                    \
        /* rawcp = (unsigned char *) cp; */                                    \
    } while (0)

/*
 * Routine for handling various errors/conditions.
 * Note how they are "glued into the decoder" by
 * overriding the definitions used by the decoder.
 */

static void Fax3Unexpected()
{
    CPLError(CE_Failure, CPLE_AppDefined, "Bad code word");
}

#define unexpected(table, a0) Fax3Unexpected()

static void Fax3BadLength(GUInt32 a0, GUInt32 lastx)
{
    CPLError(CE_Warning, CPLE_AppDefined, "%s (got %lu, expected %lu)",
             a0 < lastx ? "Premature EOL" : "Line length mismatch",
             (unsigned long)a0, (unsigned long)lastx);
}

#define badlength(a0, lastx) Fax3BadLength(a0, lastx)

static void Fax3PrematureEOF()
{
    CPLError(CE_Warning, CPLE_AppDefined, "Premature EOF");
}

#define prematureEOF(a0) Fax3PrematureEOF()

#define Nop

/*
 * The ZERO & FILL macros must handle spans < 2*sizeof(long) bytes.
 * For machines with 64-bit longs this is <16 bytes; otherwise
 * this is <8 bytes.  We optimize the code here to reflect the
 * machine characteristics.
 */

#if __GNUC__ >= 7
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wimplicit-fallthrough"
#endif

#if SIZEOF_UNSIGNED_LONG == 8
#define FILL(n, cp)                                                            \
    switch (n)                                                                 \
    {                                                                          \
        case 15:                                                               \
            (cp)[14] = 0xff;                                                   \
        case 14:                                                               \
            (cp)[13] = 0xff;                                                   \
        case 13:                                                               \
            (cp)[12] = 0xff;                                                   \
        case 12:                                                               \
            (cp)[11] = 0xff;                                                   \
        case 11:                                                               \
            (cp)[10] = 0xff;                                                   \
        case 10:                                                               \
            (cp)[9] = 0xff;                                                    \
        case 9:                                                                \
            (cp)[8] = 0xff;                                                    \
        case 8:                                                                \
            (cp)[7] = 0xff;                                                    \
        case 7:                                                                \
            (cp)[6] = 0xff;                                                    \
        case 6:                                                                \
            (cp)[5] = 0xff;                                                    \
        case 5:                                                                \
            (cp)[4] = 0xff;                                                    \
        case 4:                                                                \
            (cp)[3] = 0xff;                                                    \
        case 3:                                                                \
            (cp)[2] = 0xff;                                                    \
        case 2:                                                                \
            (cp)[1] = 0xff;                                                    \
        case 1:                                                                \
            (cp)[0] = 0xff;                                                    \
            (cp) += (n);                                                       \
        case 0:;                                                               \
    }
#define ZERO(n, cp)                                                            \
    switch (n)                                                                 \
    {                                                                          \
        case 15:                                                               \
            (cp)[14] = 0;                                                      \
        case 14:                                                               \
            (cp)[13] = 0;                                                      \
        case 13:                                                               \
            (cp)[12] = 0;                                                      \
        case 12:                                                               \
            (cp)[11] = 0;                                                      \
        case 11:                                                               \
            (cp)[10] = 0;                                                      \
        case 10:                                                               \
            (cp)[9] = 0;                                                       \
        case 9:                                                                \
            (cp)[8] = 0;                                                       \
        case 8:                                                                \
            (cp)[7] = 0;                                                       \
        case 7:                                                                \
            (cp)[6] = 0;                                                       \
        case 6:                                                                \
            (cp)[5] = 0;                                                       \
        case 5:                                                                \
            (cp)[4] = 0;                                                       \
        case 4:                                                                \
            (cp)[3] = 0;                                                       \
        case 3:                                                                \
            (cp)[2] = 0;                                                       \
        case 2:                                                                \
            (cp)[1] = 0;                                                       \
        case 1:                                                                \
            (cp)[0] = 0;                                                       \
            (cp) += (n);                                                       \
        case 0:;                                                               \
    }
#else
#define FILL(n, cp)                                                            \
    switch (n)                                                                 \
    {                                                                          \
        case 7:                                                                \
            (cp)[6] = 0xff;                                                    \
        case 6:                                                                \
            (cp)[5] = 0xff;                                                    \
        case 5:                                                                \
            (cp)[4] = 0xff;                                                    \
        case 4:                                                                \
            (cp)[3] = 0xff;                                                    \
        case 3:                                                                \
            (cp)[2] = 0xff;                                                    \
        case 2:                                                                \
            (cp)[1] = 0xff;                                                    \
        case 1:                                                                \
            (cp)[0] = 0xff;                                                    \
            (cp) += (n);                                                       \
        case 0:;                                                               \
    }
#define ZERO(n, cp)                                                            \
    switch (n)                                                                 \
    {                                                                          \
        case 7:                                                                \
            (cp)[6] = 0;                                                       \
        case 6:                                                                \
            (cp)[5] = 0;                                                       \
        case 5:                                                                \
            (cp)[4] = 0;                                                       \
        case 4:                                                                \
            (cp)[3] = 0;                                                       \
        case 3:                                                                \
            (cp)[2] = 0;                                                       \
        case 2:                                                                \
            (cp)[1] = 0;                                                       \
        case 1:                                                                \
            (cp)[0] = 0;                                                       \
            (cp) += (n);                                                       \
        case 0:;                                                               \
    }
#endif

/************************************************************************/
/*                         _TIFFFax3fillruns()                          */
/*                                                                      */
/*      Bit-fill a row according to the white/black                     */
/*      runs generated during G3/G4 decoding.                           */
/************************************************************************/

static void aig_TIFFFax3fillruns(unsigned char *buf, GUInt32 *runs,
                                 GUInt32 *erun, GUInt32 lastx)
{
    static const unsigned char _fillmasks[] = {0x00, 0x80, 0xc0, 0xe0, 0xf0,
                                               0xf8, 0xfc, 0xfe, 0xff};
    unsigned char *cp;
    GUInt32 x, bx, run;
    GInt32 n, nw;
    long *lp;

    if ((erun - runs) & 1)
        *erun++ = 0;
    x = 0;
    for (; runs < erun; runs += 2)
    {
        run = runs[0];
        if (x + run > lastx || run > lastx)
            run = runs[0] = (GUInt32)(lastx - x);
        if (run)
        {
            cp = buf + (x >> 3);
            bx = x & 7;
            if (run > 8 - bx)
            {
                if (bx)
                { /* align to byte boundary */
                    *cp++ &= 0xff << (8 - bx);
                    run -= 8 - bx;
                }
                if ((n = run >> 3) != 0)
                { /* multiple bytes to fill */
                    if ((n / sizeof(long)) > 1)
                    {
                        /*
                         * Align to longword boundary and fill.
                         */
                        for (; n && !isAligned(cp, long); n--)
                            *cp++ = 0x00;
                        lp = (long *)(void *)cp;
                        nw = (GInt32)(n / sizeof(long));
                        n -= nw * sizeof(long);
                        do
                        {
                            *lp++ = 0L;
                        } while (--nw);
                        cp = (unsigned char *)(void *)lp;
                    }
                    ZERO(n, cp);
                    run &= 7;
                }
                if (run)
                    cp[0] &= 0xff >> run;
            }
            else
                cp[0] &= ~(_fillmasks[run] >> bx);
            x += runs[0];
        }
        run = runs[1];
        if (x + run > lastx || run > lastx)
            run = runs[1] = lastx - x;
        if (run)
        {
            cp = buf + (x >> 3);
            bx = x & 7;
            if (run > 8 - bx)
            {
                if (bx)
                { /* align to byte boundary */
                    *cp++ |= 0xff >> bx;
                    run -= 8 - bx;
                }
                if ((n = run >> 3) != 0)
                { /* multiple bytes to fill */
                    if ((n / sizeof(long)) > 1)
                    {
                        /*
                         * Align to longword boundary and fill.
                         */
                        for (; n && !isAligned(cp, long); n--)
                            *cp++ = 0xff;
                        lp = (long *)(void *)cp;
                        nw = (GInt32)(n / sizeof(long));
                        n -= nw * sizeof(long);
                        do
                        {
                            *lp++ = -1L;
                        } while (--nw);
                        cp = (unsigned char *)(void *)lp;
                    }
                    FILL(n, cp);
                    run &= 7;
                }
                if (run)
                    cp[0] |= 0xff00 >> run;
            }
            else
                cp[0] |= _fillmasks[run] >> bx;
            x += runs[1];
        }
    }
    assert(x == lastx);
}

#undef ZERO
#undef FILL

#if __GNUC__ >= 7
#pragma GCC diagnostic pop
#endif

/************************************************************************/
/*                           Fax3DecodeRLE()                            */
/*                                                                      */
/*      Decode the requested amount of RLE-encoded data.                */
/************************************************************************/

static int Fax3DecodeRLE(Fax3BaseState *tif, unsigned char *buf, int occ,
                         unsigned char *rawcp, int rawcc)
{
    DECLARE_STATE(tif, sp);
    int mode = sp->b.mode;
    if (occ % sp->b.rowbytes)
    {
        CPLError(CE_Failure, CPLE_AppDefined,
                 "Fractional scanlines cannot be read");
        return (-1);
    }

    CACHE_STATE(sp);
    thisrun = sp->curruns;
    while ((long)occ > 0)
    {
        a0 = 0;
        RunLength = 0;
        pa = thisrun;
#ifdef FAX3_DEBUG
        printf("\nBitAcc=%08X, BitsAvail = %d\n", BitAcc, BitsAvail); /*ok*/
        printf("-------------------- \n");                            /*ok*/
        fflush(stdout);
#endif

        /* --------------------------------------------------------------------
         */
        /*      EXPAND1D() */
        /*                                                                      */
        /*                                                                      */
        /*       Decode a line of 1D-encoded data. */
        /*                                                                      */
        /*      The line expanders are written as macros so that they can */
        /*      be  reused but still have direct access to the local */
        /*      variables of the "calling" function. */
        /*                                                                      */
        /*      Note that unlike the original version we have to explicitly */
        /*      test for a0 >= lastx after each black/white run is decoded. */
        /*      This is because the original code depended on the input data */
        /*      being zero-padded to insure the decoder recognized an EOL */
        /*      before running out of data. */
        /* --------------------------------------------------------------------
         */
        do
        {
            for (;;)
            {
                for (;;)
                {
                    LOOKUP16(12, aig_TIFFFaxWhiteTable, eof1d);
                    switch (TabEnt->State)
                    {
                        case S_EOL:
                            EOLcnt = 1;
                            goto done1d;
                        case S_TermW:
                            SETVAL(TabEnt->Param);
                            goto doneWhite1d;
                        case S_MakeUpW:
                        case S_MakeUp:
                            a0 += TabEnt->Param;
                            RunLength += TabEnt->Param;
                            break;
                        default:
                            unexpected("WhiteTable", a0);
                            goto done1d;
                    }
                }
            doneWhite1d:
                if (a0 >= lastx)
                    goto done1d;
                for (;;)
                {
                    LOOKUP16(13, aig_TIFFFaxBlackTable, eof1d);
                    switch (TabEnt->State)
                    {
                        case S_EOL:
                            EOLcnt = 1;
                            goto done1d;
                        case S_TermB:
                            SETVAL(TabEnt->Param);
                            goto doneBlack1d;
                        case S_MakeUpB:
                        case S_MakeUp:
                            a0 += TabEnt->Param;
                            RunLength += TabEnt->Param;
                            break;
                        default:
                            unexpected("BlackTable", a0);
                            goto done1d;
                    }
                }
            doneBlack1d:
                if (a0 >= lastx)
                    goto done1d;
                if (*(pa - 1) == 0 && *(pa - 2) == 0)
                    pa -= 2;
            }
        eof1d:
            prematureEOF(a0);
            CLEANUP_RUNS();
            goto EOFRLE;
        done1d:
            CLEANUP_RUNS();
        } while (0);

        /* --------------------------------------------------------------------
         */
        /*      Fill */
        /* --------------------------------------------------------------------
         */
        (*sp->fill)(buf, thisrun, pa, lastx);
        /*
         * Cleanup at the end of the row.
         */
        if (mode & FAXMODE_BYTEALIGN)
        {
            int n = BitsAvail - (BitsAvail & ~7);
            ClrBits(n);
        }
        else if (mode & FAXMODE_WORDALIGN)
        {
            int n = BitsAvail - (BitsAvail & ~15);
            ClrBits(n);
            if (BitsAvail == 0 && !isAligned(cp, GUInt16))
                cp++;
        }
        buf += sp->b.rowbytes;
        occ -= sp->b.rowbytes;
#ifdef notdef
        if (occ != 0)
            tif->tif_row++;
#endif
        continue;
    EOFRLE: /* premature EOF */
        (*sp->fill)(buf, thisrun, pa, lastx);
        UNCACHE_STATE(sp);
        return (-1);
    }
    UNCACHE_STATE(sp);

    return (1);
}

/************************************************************************/
/*                       DecompressCCITTRLETile()                       */
/************************************************************************/

CPLErr DecompressCCITTRLETile(unsigned char *pabySrcData, int nSrcBytes,
                              unsigned char *pabyDstData, int nDstBytes,
                              int nBlockXSize, CPL_UNUSED int nBlockYSize)
{
    Fax3DecodeState sDecoderState;
    Fax3BaseState *sp = (Fax3BaseState *)&sDecoderState;
    unsigned char runs_buf[4000];
    long rowbytes, rowpixels;

    memset(&sDecoderState, 0, sizeof(sDecoderState));

    sp->groupoptions = 0;
    sp->recvparams = 0;
    sp->subaddress = NULL;

    DecoderState(sp)->runs = NULL;
    DecoderState(sp)->fill = aig_TIFFFax3fillruns;

    if (sizeof(runs_buf) < (size_t)(nBlockXSize * 2 + 3))
    {
        CPLError(CE_Failure, CPLE_AppDefined, "Run buffer too small");
        return CE_Failure;
    }

    /* -------------------------------------------------------------------- */
    /*                                                                      */
    /* -------------------------------------------------------------------- */
    /*
     * Calculate the scanline/tile widths.
     */
    rowbytes = (nBlockXSize + 7) / 8;
    rowpixels = nBlockXSize;

    sp->rowbytes = (GUInt32)rowbytes;
    sp->rowpixels = (GUInt32)rowpixels;
    sp->mode |= FAXMODE_BYTEALIGN;
    /*
     * Allocate any additional space required for decoding/encoding.
     */
    {
        Fax3DecodeState *dsp = DecoderState(sp);

        dsp->runs = (GUInt32 *)runs_buf;
        dsp->curruns = dsp->runs;
        dsp->refruns = NULL;
    }

    /* -------------------------------------------------------------------- */
    /*                                                                      */
    /* -------------------------------------------------------------------- */
    DecoderState(sp)->bit = 0; /* force initial read */
    DecoderState(sp)->data = 0;
    DecoderState(sp)->EOLcnt = 0; /* force initial scan for EOL */

    DecoderState(sp)->bitmap = aig_TIFFBitRevTable;

    if (DecoderState(sp)->refruns)
    { /* init reference line to white */
        DecoderState(sp)->refruns[0] = (GUInt32)DecoderState(sp)->b.rowpixels;
        DecoderState(sp)->refruns[1] = 0;
    }

    if (Fax3DecodeRLE(sp, pabyDstData, nDstBytes, pabySrcData, nSrcBytes) == 1)
        return CE_None;
    else
        return CE_Failure;
}
