//===-- Half-precision tanh(x) function -----------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "src/math/tanhf16.h"
#include "expxf16.h"
#include "hdr/fenv_macros.h"
#include "src/__support/CPP/array.h"
#include "src/__support/FPUtil/FEnvImpl.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/PolyEval.h"
#include "src/__support/FPUtil/cast.h"
#include "src/__support/FPUtil/except_value_utils.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/FPUtil/nearest_integer.h"
#include "src/__support/FPUtil/rounding_mode.h"
#include "src/__support/common.h"
#include "src/__support/macros/config.h"
#include "src/__support/macros/optimization.h"

namespace LIBC_NAMESPACE_DECL {

static constexpr fputil::ExceptValues<float16, 2> TANHF16_EXCEPTS = {{
    // x = 0x1.f54p+0, tanhf16(x) = 0x1.ecp-1 (RZ)
    {0x3fd5U, 0x3bb0U, 1U, 0U, 0U},
    // x = -0x1.f54p+0, tanhf16(x) = -0x1.ecp-1 (RZ)
    {0xbfd5U, 0xbbb0U, 0U, 1U, 0U},
}};

LLVM_LIBC_FUNCTION(float16, tanhf16, (float16 x)) {
  using FPBits = fputil::FPBits<float16>;
  FPBits x_bits(x);

  uint16_t x_u = x_bits.uintval();
  uint16_t x_abs = x_u & 0x7fffU;

  // When -2^(-14) <= x <= -2^(-9), or |x| <= 0x1.d2p-4,
  // or |x| >= atanh(1 - 2^(-11)), or x is NaN.
  if (LIBC_UNLIKELY(x_abs <= 0x2f48U || x_abs >= 0x4429U)) {
    // tanh(NaN) = NaN
    if (x_bits.is_nan()) {
      if (x_bits.is_signaling_nan()) {
        fputil::raise_except_if_required(FE_INVALID);
        return FPBits::quiet_nan().get_val();
      }

      return x;
    }

    // When -2^(-14) <= x <= -2^(-9).
    if (x_u >= 0x8400U && x_u <= 0x9800U) {
      switch (fputil::quick_get_round()) {
      case FE_TONEAREST:
      case FE_DOWNWARD:
        return x;
      default:
        return FPBits(static_cast<uint16_t>(x_u - 1U)).get_val();
      }
    }

    // When |x| <= 0x1.d2p-4.
    if (x_abs <= 0x2f48U) {
      if (LIBC_UNLIKELY(x_abs == 0))
        return x;

      float xf = x;
      float xf_sq = xf * xf;
      // Degree-7 Taylor expansion generated by Sollya with the following
      // commands:
      //   > taylor(tanh(x), 7, 0);
      //   > display = hexadecimal;
      //   > // For each coefficient:
      //   > round(/* put coefficient here */, SG, RN);
      return fputil::cast<float16>(
          xf * fputil::polyeval(xf_sq, 0x1p+0f, -0x1.555556p-2f, 0x1.111112p-3f,
                                -0x1.ba1ba2p-5f));
    }

    // tanh(+/-inf) = +/-1
    if (x_bits.is_inf())
      return FPBits::one(x_bits.sign()).get_val();

    // When |x| >= atanh(1 - 2^(-11)).
    fputil::raise_except_if_required(FE_INEXACT);

    int rounding_mode = fputil::quick_get_round();
    if ((rounding_mode == FE_TONEAREST && x_abs >= 0x4482U) ||
        (rounding_mode == FE_UPWARD && x_bits.is_pos()) ||
        (rounding_mode == FE_DOWNWARD && x_bits.is_neg())) {
      return FPBits::one(x_bits.sign()).get_val();
    }
    if (x_bits.is_pos())
      return fputil::cast<float16>(0x1.ffcp-1);
    return fputil::cast<float16>(-0x1.ffcp-1);
  }

  if (auto r = TANHF16_EXCEPTS.lookup(x_u); LIBC_UNLIKELY(r.has_value()))
    return r.value();

  // For atanh(-1 + 2^(-11)) < x < atanh(1 - 2^(-11)), to compute tanh(x), we
  // perform the following range reduction: find hi, mid, lo, such that:
  //   x = (hi + mid) * log(2) * 0.5 + lo, in which
  //     hi is an integer,
  //     mid * 2^5 is an integer,
  //     -2^(-5) <= lo < 2^(-5).
  // In particular,
  //   hi + mid = round(x * log2(e) * 2 * 2^5) * 2^(-5).
  // Then,
  //   tanh(x) = sinh(x)/cosh(x)
  //           = (e^x - e^(-x)) / (e^x + e^(-x))
  //           = (e^(2x) - 1) / (e^(2x) + 1)
  //           = (2^(hi + mid) * e^(2*lo) - 1) / (2^(hi + mid) * e^(2*lo) + 1)
  //           = (e^(2*lo) - 2^(-hi - mid)) / (e^(2*lo) + 2^(-hi - mid))
  // We store 2^(-mid) in the lookup table EXP2_MID_5_BITS, and compute
  // 2^(-hi - mid) by adding -hi to the exponent field of 2^(-mid).
  // e^lo is computed using a degree-3 minimax polynomial generated by Sollya.

  float xf = x;
  float kf = fputil::nearest_integer(xf * (LOG2F_E * 2.0f * 0x1.0p+5f));
  int x_hi_mid = -static_cast<int>(kf);
  unsigned x_hi = static_cast<unsigned>(x_hi_mid) >> 5;
  unsigned x_mid = static_cast<unsigned>(x_hi_mid) & 0x1f;
  // lo = x - (hi + mid)
  //    = round(x * log2(e) * 2 * 2^5) * log(2) * 0.5 * (-2^(-5)) + x
  float lo = fputil::multiply_add(kf, LOGF_2 * 0.5f * -0x1.0p-5f, xf);

  uint32_t exp2_hi_mid_bits =
      EXP2_MID_5_BITS[x_mid] +
      static_cast<uint32_t>(x_hi << fputil::FPBits<float>::FRACTION_LEN);
  // exp2_hi_mid = 2^(-hi - mid)
  float exp2_hi_mid = fputil::FPBits<float>(exp2_hi_mid_bits).get_val();
  // Degree-3 minimax polynomial generated by Sollya with the following
  // commands:
  //   > display = hexadecimal;
  //   > P = fpminimax(expm1(2*x)/x, 2, [|SG...|], [-2^-5, 2^-5]);
  //   > 1 + x * P;
  float exp_2lo =
      fputil::polyeval(lo, 0x1p+0f, 0x1p+1f, 0x1.001p+1f, 0x1.555ddep+0f);
  return fputil::cast<float16>((exp_2lo - exp2_hi_mid) /
                               (exp_2lo + exp2_hi_mid));
}

} // namespace LIBC_NAMESPACE_DECL
