/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


////////////////////////////// Stdlib includes
#include <vector>


/////////////////////// Qt includes
#include <QString>


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/Monomer.hpp"
#include "MsXpS/libXpertMassCore/IndexRangeCollection.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


typedef std::vector<MonomerSPtr>::iterator SequenceIterator;
typedef std::vector<MonomerSPtr>::const_iterator SequenceCstIterator;


class DECLSPEC Sequence
{
  friend class Polymer;
  friend class Oligomer;

  public:
  Sequence();
  Sequence(PolChemDefCstSPtr pol_chem_def_csp, const QString &sequence_text);
  Sequence(const Sequence &other);

  virtual ~Sequence();

  //////////////// POLYMER CHEMISTRY DEFINTIION /////////////////////

  void setPolChemDefCstSPtr(PolChemDefCstSPtr pol_chem_def_csp);
  PolChemDefCstSPtr getPolChemDef() const;

  //////////////// MONOMERS TEXT / INSTANCES /////////////////////

  int setSequence(const QString &sequence,
                   std::vector<std::size_t> &failing_indices);
  int appendSequence(const QString &sequence,
                     std::vector<std::size_t> &failing_indices);
  QString getSequence() const;
  QString getSequence(std::size_t start_index,
                      std::size_t stop_index,
                      bool with_modifs = false) const;
  QString getSequence(const IndexRangeCollection &index_ranges,
                      bool with_modifs       = false,
                      bool delimited_regions = false) const;

  int makeMonomers(const QString &sequence_text,
                   bool reset,
                   std::vector<std::size_t> &failing_indices);

  const std::vector<MonomerSPtr> &getMonomersCstRef() const;
  std::vector<MonomerSPtr> &getMonomersRef();

  //////////////// MONOMER ACCESSING FUNCTIONS /////////////////////

  MonomerCstRPtr getMonomerCstRPtrAt(std::size_t index) const;
  MonomerRPtr getMonomerRPtrAt(std::size_t index);
  MonomerSPtr getMonomerCstSPtrAt(std::size_t index) const;
  MonomerSPtr getMonomerSPtrAt(std::size_t index);

  std::size_t monomerIndex(MonomerSPtr monomer_sp, bool &ok) const;
  std::size_t monomerIndex(MonomerCstSPtr monomer_csp, bool &ok) const;
  std::size_t monomerIndex(MonomerCstRPtr monomer_crp, bool &ok) const;
  std::size_t nextCode(const QString &sequence,
                       QString &code,
                       std::size_t &index,
                       QString &err);

  //////////////// MONOMER HANDLING FUNCTIONS /////////////////////

  QString insertMonomerAt(const Monomer &monomer, std::size_t index);
  bool removeMonomerAt(std::size_t index);
  bool
  modifyMonomer(std::size_t index, const QString modif_name, bool override);
  bool hasModifiedMonomer(std::size_t left_index,
                          std::size_t right_index) const;
  std::vector<std::size_t>
  modifiedMonomerIndices(std::size_t left_index, std::size_t right_index) const;

  //////////////// SEQUENCE SEARCH FUNCTIONS /////////////////////

  int findForwardMotif(const Sequence &sequence_motif, std::size_t &index) const;

  //////////////// DIAGNOSTICS FUNCTIONS /////////////////////

  std::size_t size() const;
  bool isInBound(std::size_t index);

  //////////////// OPERATORS /////////////////////

  Sequence &operator=(const Sequence &other);
  bool operator==(const Sequence &other);
  bool operator!=(const Sequence &other);

  //////////////// VALIDATIONS /////////////////////

  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;


  //////////////// UTILS /////////////////////
  QString unspacifySequence(const QString &monomer_text);

  QString storeMonomer(const MonomerSPtr &monomer_sp);
  QString storeMonomer(const MonomerSPtr &monomer_sp, std::size_t index);

  bool hasMonomer(const MonomerSPtr &monomer_sp) const;
  bool hasUuid(const MonomerSPtr &monomer_sp) const;

  MonomerSPtr getMonomerForUuid(const QString &uuid) const;
  QString getUuidForMonomer(const MonomerSPtr &monomer_sp) const;

  std::vector<QString> getAllMonomerUuids() const;

  void cleanupMonomers();

  quint16 checksum(int = -1, int = -1, bool = false) const;
  void clear();

  protected:
  PolChemDefCstSPtr mcsp_polChemDef = nullptr;

  std::vector<MonomerSPtr> m_monomers;

  //  We need in code that uses this class,  to constantly keep at hand
  //  the Monomer instances that are involved in CrossLink_s or that need
  //  to be reported as containing Modif instances. For example,
  //  we need to store the Monomer pointers as strings in QListWidget items
  //  (Qt:UserRole).
  //  We thus make use of the QUuid class to craft a Uuid string that
  //  we associate to the Monomer weak pointer that in turn relates
  //  to the Monomer shared pointer.
  std::vector<UuidMonomerWPtrPair> m_uuidMonomerPairs;
  mutable bool m_isValid = false;
};


} // namespace libXpertMassCore
} // namespace MsXpS
