<!-- front-matter
id: vinyl
title: Vinyl
hide_title: true
sidebar_label: Vinyl
-->

# Vinyl

A virtual file format. When a file is read by `src()`, a Vinyl object is generated to represent the file - including the path, contents, and other metadata.

Vinyl objects can have transformations applied using [plugins][using-plugins-docs]. They may also be persisted to the file system using `dest()`.

When creating your own Vinyl objects - instead of generating with `src()` - use the external `vinyl` module, as shown in Usage below.

## Usage

```js
const Vinyl = require('vinyl');

const file = new Vinyl({
  cwd: '/',
  base: '/test/',
  path: '/test/file.js',
  contents: new Buffer('var x = 123')
});

file.relative === 'file.js';

file.dirname === '/test';
file.dirname = '/specs';
file.path === '/specs/file.js';

file.basename === 'file.js';
file.basename = 'file.txt';
file.path === '/specs/file.txt';

file.stem === 'file';
file.stem = 'foo';
file.path === '/specs/foo.txt';
file.extname === '.txt';
file.extname = '.js';
file.path === '/specs/file.js';
```

## Signature

```js
new Vinyl([options])
```

### Parameters

| parameter | type | note |
|:--------------:|:------:|-------|
| options | object | Detailed in [Options][options-section] below. |

### Returns

An instance of the Vinyl class representing a single virtual file, detailed in [Vinyl instance][vinyl-instance-section] below.

### Errors

When any passed options don't conform to the [instance property definitions][instance-properties-section] (like if `path` is set to a number) throws as defined in the table.

### Options

| name | type | default | note |
|:-------:|:------:|-----------|--------|
| cwd | string | `process.cwd()` | The directory from which relative paths will be derived. Will be [normalized][normalization-and-concatenation-section] and have trailing separators removed. |
| base | string | | Used to calculate the `relative` instance property. Falls back to the value of `cwd` if not set. Typically set to the [glob base][glob-base-concepts]. Will be [normalized][normalization-and-concatenation-section] and have trailing separators removed.|
| path | string | | The full, absolute file path. Will be [normalized][normalization-and-concatenation-section] and have trailing separators removed. |
| history | array | `[ ]` | An array of paths to pre-populate the `history` of a Vinyl instance. Usually comes from deriving a new Vinyl object from a previous Vinyl object. If `path` and `history` are both passed, `path` is appended to `history`. Each item will be [normalized][normalization-and-concatenation-section] and have trailing separators removed. |
| stat | object | | An instance of `fs.Stats`, usually the result of calling `fs.stat()` on a file. Used to determine if a Vinyl object represents a directory or symbolic link. |
| contents | ReadableStream<br>Buffer<br>`null` | `null` | The contents of the file. If `contents` is a ReadableStream, it is wrapped in a [cloneable-readable][cloneable-readable-external] stream. |

Any other properties on `options` will be directly assigned to the Vinyl instance.

```js
const Vinyl = require('vinyl');

const file = new Vinyl({ foo: 'bar' });
file.foo === 'bar';
```

## Vinyl instance

Each instance of a Vinyl object will have properties and methods to access and/or modify information about the virtual file.

### Instance properties

All internally managed paths - any instance property except `contents` and `stat` - are normalized and have trailing separators removed. See [Normalization and concatenation][normalization-and-concatenation-section] for more information.

| property | type | description | throws |
|:-----------:|:------:|----------------|----------|
| contents | ReadableStream<br>Buffer<br>`null` | Gets and sets the contents of the virtual file. If set to a ReadableStream, it is wrapped in a [cloneable-readable][cloneable-readable-external] stream. | If set to any value other than a ReadableStream, a Buffer, or `null`. |
| stat | object | Gets and sets an instance of [`fs.Stats`][fs-stats-concepts]. Used when determining if a Vinyl object represents a directory or symbolic link. | |
| cwd | string | Gets and sets the current working directory. Used for deriving relative paths. | If set to an empty string or any non-string value. |
| base | string | Gets and sets the base directory. Used to calculate the `relative` instance property. On a Vinyl object generated by `src()` will be set to the [glob base][glob-base-concepts]. If set to `null` or `undefined`, falls back to the value of the `cwd` instance property. | If set to an empty string or any non-string value (except `null` or `undefined`). |
| path | string | Gets and sets the full, absolute file path. Setting to a value different from the current `path` appends the new path to the `history` instance property. | If set to any non-string value. |
| history | array | Array of all `path` values the Vinyl object has been assigned. The first element is the original path and the last element is the current path. This property and its elements should be treated as read-only and only altered indirectly by setting the `path` instance property. | |
| relative | string | Gets the relative path segment between the `base` and the `path` instance properties. | If set to any value. If accessed when `path` is not available. |
| dirname | string | Gets and sets the directory of the `path` instance property. | If accessed when `path` is not available. |
| stem | string | Gets and sets the stem (filename without extension) of the `path` instance property. | If accessed when `path` is not available. |
| extname | string | Gets and sets the extension of the `path` instance property. | If accessed when `path` is not available. |
| basename | string | Gets and sets the filename (`stem + extname`) of the `path` instance property. | If accessed when `path` is not available. |
| symlink | string | Gets and sets the reference path of a symbolic link. | If set to any non-string value. |

### Instance methods

| method | return type | returns |
|:----------:|:--------------:|--------|
| `isBuffer()` | boolean | If the `contents` instance property is a Buffer, returns true. |
| `isStream()` | boolean | If the `contents` instance property is a Stream, returns true. |
| `isNull()` | boolean | If the `contents` instance property is `null`, returns true. |
| `isDirectory()` | boolean | If the instance represents a directory, returns true. An instance is considered a directory when `isNull()` returns true, the `stat` instance property is an object, and `stat.isDirectory()` returns true. This assumes a Vinyl object was constructed with a valid (or properly mocked) `fs.Stats` object. |
| `isSymbolic()` | boolean | If the instance represents a symbolic link, returns true. An instance is considered symbolic when `isNull()` returns true, the `stat` instance property is an object, and `stat.isSymbolicLink()` returns true. This assumes a Vinyl object was constructed with a valid (or properly mocked) `fs.Stats` object. |
| `clone([options])` | object | A new Vinyl object with all properties cloned. By default custom properties are deep cloned. If the `deep` option is false, custom attributes will be shallow cloned. If the `contents` option is false and the `contents` instance property is a Buffer, the Buffer will be reused instead of cloned. |
| `inspect()` | string | Returns a formatted interpretation of the Vinyl object. Automatically called by Node's console.log. |

## Normalization and concatenation

All path properties are normalized by their setters. Concatenate paths with `/`, instead of using `path.join()`, and normalization will occur properly on all platforms. Never concatenate with `\` - it  is a valid filename character on POSIX system.

```js
const file = new File();
file.path = '/' + 'test' + '/' + 'foo.bar';

console.log(file.path);
// posix => /test/foo.bar
// win32 => \\test\\foo.bar
```

[options-section]: #options
[vinyl-instance-section]: #vinyl-instance
[instance-properties-section]: #instance-properties
[normalization-and-concatenation-section]: #normalization-and-concatenation
[glob-base-concepts]: ../api/concepts.md#glob-base
[fs-stats-concepts]: ../api/concepts.md#file-system-stats
[using-plugins-docs]: ../getting-started/7-using-plugins.md
[cloneable-readable-external]: https://github.com/mcollina/cloneable-readable
