#include <glib-object.h>
#include "streammodel.h"
#include <QDebug>
#include <QProcessEnvironment>

class StreamModelPrivate: public DeeListModel
{
    Q_DECLARE_PUBLIC(StreamModel)

public:
    StreamModelPrivate(StreamModel *model);
    ~StreamModelPrivate();
    static void onModelReady(GObject* parent, GParamSpec *pspec, StreamModelPrivate *q);
    void updateResults(StreamModelPrivate *d);
    void createRoles();
    QHash<int, QByteArray> m_roleNames;

private:
    mutable StreamModel *q_ptr;
    QString streamId = NULL;
    QString serviceId = NULL;
    uint accountId = 0;
    DeeModel* m_resultsModel;
    DeeModel* m_sharedModel;
    DeeModel* m_sortedModel;
};

StreamModelPrivate::StreamModelPrivate(StreamModel *model):
    DeeListModel(model),
    q_ptr(model)
{
    QString modelName = "com.canonical.Friends.Streams";
    /* read environment variables */
    QProcessEnvironment environment = QProcessEnvironment::systemEnvironment();
    if (environment.contains(QLatin1String("FRIENDS_STREAMS_MODEL")))
    {
        modelName = environment.value(QLatin1String("FRIENDS_STREAMS_MODEL"));
    }

    qDebug() << "MODEL: " << modelName;

    m_sharedModel = dee_shared_model_new (modelName.toUtf8());
    m_resultsModel = m_sharedModel;
    m_sortedModel = dee_sequence_model_new ();
    g_signal_connect(m_sharedModel, "notify::synchronized", G_CALLBACK(onModelReady), this);

}

void
StreamModelPrivate::createRoles()
{
    if (m_resultsModel == NULL) {
        return;
    }

    QHash<int, QByteArray> roles;

    roles[0] = QString("service").toLocal8Bit();
    roles[1] = QString("accountId").toLocal8Bit();
    roles[2] = QString("messageId").toLocal8Bit();
    roles[3] = QString("stream").toLocal8Bit();
    roles[4] = QString("sender").toLocal8Bit();
    roles[5] = QString("senderId").toLocal8Bit();
    roles[6] = QString("senderNick").toLocal8Bit();
    roles[7] = QString("fromMe").toLocal8Bit();
    roles[8] = QString("timestamp").toLocal8Bit();
    roles[9] = QString("message").toLocal8Bit();
    roles[10] = QString("avatar").toLocal8Bit();
    roles[11] = QString("url").toLocal8Bit();
    roles[12] = QString("likes").toLocal8Bit();
    roles[13] = QString("liked").toLocal8Bit();
    roles[14] = QString("linkPicture").toLocal8Bit();
    roles[15] = QString("linkName").toLocal8Bit();
    roles[16] = QString("linkUrl").toLocal8Bit();
    roles[17] = QString("linkDescription").toLocal8Bit();
    roles[18] = QString("linkCaption").toLocal8Bit();
    roles[19] = QString("linkIcon").toLocal8Bit();
    roles[20] = QString("location").toLocal8Bit();
    roles[21] = QString("latitude").toLocal8Bit();
    roles[22] = QString("longitude").toLocal8Bit();

    m_roleNames = roles;
    Q_EMIT q_ptr->roleNamesChanged(roles);
}

void
StreamModelPrivate::onModelReady(GObject* parent __attribute__ ((unused)), GParamSpec *pspec, StreamModelPrivate *d)
{
    DeeModel *model = (DeeModel*)parent;
    if (!dee_shared_model_is_synchronized ((DeeSharedModel*)model))
        return;
    d->createRoles();

    DeeFilter _sort_filter;
    dee_filter_new_collator_desc (8, &_sort_filter);
    d->m_sortedModel = dee_filter_model_new (model, &_sort_filter);
    qDebug() << Q_FUNC_INFO << " " << dee_model_get_n_rows (d->m_sortedModel);
    d->updateResults(d);
}

void StreamModelPrivate::updateResults(StreamModelPrivate *d)
{
    if (d->streamId != NULL)
    {
        DeeFilter _key_filter;
        dee_filter_new_for_key_column (3, d->streamId.toUtf8().data(), &_key_filter);
        d->m_resultsModel = dee_filter_model_new (d->m_sortedModel, &_key_filter);
    } else
    {
        GRegex *regex;
        regex = g_regex_new ("^((?!reply_to).)*$", G_REGEX_FIRSTLINE, G_REGEX_MATCH_PARTIAL, NULL);
        DeeFilter _regex_filter;
        dee_filter_new_regex (3, regex, &_regex_filter);
        g_regex_unref (regex);
        d->m_resultsModel = dee_filter_model_new (d->m_sortedModel, &_regex_filter);
    }
    qDebug () << "STREAM: " << d->streamId << " ROWS: " << dee_model_get_n_rows (d->m_resultsModel);

    if (d->serviceId != NULL)
    {
        DeeFilter _service_filter;
        dee_filter_new_for_key_column (0, d->serviceId.toUtf8().data(), &_service_filter);
        d->m_resultsModel = dee_filter_model_new (d->m_resultsModel, &_service_filter);
    }
    qDebug () << "SERVICE: " << d->serviceId << " ROWS: " << dee_model_get_n_rows (d->m_resultsModel);

    if (d->accountId != 0)
    {
        DeeFilter _account_filter;
        dee_filter_new_for_any_column (1, g_variant_new_uint64(d->accountId), &_account_filter);
        d->m_resultsModel = dee_filter_model_new (d->m_resultsModel, &_account_filter);
        qDebug () << "ACCOUNT: " << d->accountId << " ROWS: " << dee_model_get_n_rows (d->m_resultsModel);
    }

    q_ptr->setModel(m_resultsModel);
    Q_EMIT q_ptr->streamChanged();
}

StreamModelPrivate::~StreamModelPrivate()
{
}

/*!
 * \qmltype StreamModel
 * \inqmlmodule Friends 0.1
 * \ingroup Friends
 *
 * \brief A model of the user's Friends feeds.
 *
 * The SteamsModel is a model based on the model provided by the Friends
 * service, conveniently sorted with pre-defined filters.  By default,
 * the model contains everything except comments.  You can filter
 * based on stream, account or service.
 *
 * Examples of use:
 *
 * 1. Model of the complete Friends feed:
 * \qml
 *
 * Item {
 *     StreamModel {
 *         id: StreamModel
 *     }
 *
 *     ListView {
 *         model: StreamModel
 *         delegate: Text { text: "message: " + message }
 *     }
 * }
 *
 * \endqml
 *
 * 2. Model of the Friends feed, limited to facebook:
 * \qml
 *
 * Item {
 *     StreamModel {
 *         id: StreamModel
 *         service: "facebook"
 *     }
 *
 *     ListView {
 *         model: StreamModel
 *         delegate: Text { text: "message: " + message }
 *     }
 * }
 * \endqml
 *
 * 3. Model of the Friends feed, limited to a single account:
 * \qml
 *
 * Item {
 *     StreamModel {
 *         id: StreamModel
 *         account: 1 // The account ID provided but Ubuntu.OnlineAccounts
 *     }
 *
 *     ListView {
 *         model: StreamModel
 *         delegate: Text { text: "message: " + message }
 *     }
 * }
 * \endqml
 *
 * 4. Model of the Friends feed, limited to a thread of comments:
 * \qml
 *
 * Item {
 *     StreamModel {
 *         id: StreamModel
 *         stream: "reply_to/"+messageId
 *     }
 *
 *     ListView {
 *         model: StreamModel
 *         delegate: Text { text: "message: " + message }
 *     }
 * }
 * \endqml
 *
 * 5. Model of the Friends feed, limited to a specific stream, like private messages:
 * \qml
 *
 * Item {
 *     StreamModel {
 *         id: StreamModel
 *         stream: "private"
 *     }
 *
 *     ListView {
 *         model: StreamModel
 *         delegate: Text { text: "message: " + message }
 *     }
 * }
 * \endqml
 *
 */
StreamModel::StreamModel(DeeListModel *parent) :
    DeeListModel(parent),
    d_ptr(new StreamModelPrivate(this))
{
}

StreamModel::~StreamModel()
{
}


void StreamModel::classBegin()
{
}

void StreamModel::componentComplete()
{
}

void StreamModel::setStream(const QString& streamId)
{
    Q_D(StreamModel);
    if (streamId == d->streamId) return;
    d->streamId = streamId;
    d->updateResults(d);
}

/*!
 * \qmlproperty string StreamModel::stream
 * If set, the model will include only this stream
 */
QString StreamModel::stream() const
{
    Q_D(const StreamModel);
    return d->streamId;
}

void StreamModel::setService(const QString& serviceId)
{
    Q_D(StreamModel);
    if (serviceId == d->serviceId) return;
    d->serviceId = serviceId;
    d->updateResults(d);
}

/*!
 * \qmlproperty string StreamModel::service
 * If set, the model will include only this service
 */
QString StreamModel::service() const
{
    Q_D(const StreamModel);
    return d->serviceId;
}

void StreamModel::setAccount(uint account)
{
    Q_D(StreamModel);
    if (account == d->accountId) return;
    d->accountId = account;
    d->updateResults(d);
    qDebug() << "Account: " << account;
}

/*!
 * \qmlproperty uint StreamModel::account
 * If set, the model will include only this account
 */
uint StreamModel::account() const
{
    Q_D(const StreamModel);
    return d->accountId;
}

QHash<int, QByteArray>
StreamModel::roleNames() const
{
    Q_D(const StreamModel);
    return d->m_roleNames;
}
