/*
  operator.cc, copyright (c) 2006 by Vincent Fourmond: 
  Implementation of a few different stuffs.
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

#include <calc_internals.hh>

#include <math.h>

namespace SCalc {
  
  void Operator::dump(std::ostream & stream)
  {
    stream << "operator " << name() << ": [ ";
    left->dump(stream);
    stream << name();
    right->dump(stream);
    stream << "] ";  
  }

  std::set<int> Operator::used_variables()
  {
    std::set<int> l,r;
    l = left->used_variables();
    r = right->used_variables();
    // It isn't beautiful, but it works.
    for(std::set<int>::iterator i = r.begin(); i != r.end(); i++)
      l.insert(*i);
    return l;
  }

  std::string Operator::pretty_print()
  {
    // We wrap that up in parentheses
    /// \todo In real, I should write up some decent code that could
    /// convert the expression back into well-formed text with the
    /// same meaning -- or possibly C code ?
    std::string res = std::string("(") + 
      left->pretty_print() + " " + name() + " " 
      + right->pretty_print() + ")";
    return res;
  }

  Expression * Operator::simplify()
  {
    // first, acquire a simplified version of the children:
    Expression * l = left->simplify();
    Expression * r = right->simplify();
    
    // then, call the specific rules
    Expression * ret = specific_rules(l,r);
    if(ret) 
      {
	// l and r have been dealt with
	return ret;
      }

    // operator-wide simplifications:
    if(l->is_const() && r->is_const())
      {
	// we can evaluate the operation
	Operator * o = op(l,r);
	double res = o->evaluate(NULL,NULL);
	delete o; // deletes l and r as well.
	return new Const(session(), res);
      }
    // return the operator left untouched (well, nearly so...)
    else return op(l,r);
  }

  /*********************************************************/
  /* methods for addition */
  
  Expression * Plus::derive(int id)
  {
    return new Plus(session(), left->derive(id),
		   right->derive(id));
  }

  Expression * Plus::specific_rules(Expression * l, Expression * r)
  {
    if(l->is_null())
      {
	delete l;
	return r;
      }
    if(r->is_null())
      {
	delete r;
	return l;
      }
    return NULL; /* nothing done here */
  }

  Expression * Minus::derive(int id)
  {
    return new Minus(session(), left->derive(id),
		     right->derive(id));
  }

  Expression * Minus::specific_rules(Expression * l, Expression * r)
  {
    if(r->is_null())
      {
	delete r;
	return l;
      }
    return NULL; /* nothing done here */
  }

  Expression * Times::derive(int id)
  {
    return new Plus(session(), 
		    new Times(session(), left->derive(id), right->copy()),
		    new Times(session(), left->copy(), right->derive(id)));
  }

  Expression * Times::specific_rules(Expression * l, Expression * r)
  {
    if(l->is_null() || r->is_null())
      {
	delete r;
	delete l;
	return new Null(session()); // simpler...
      }
    if(r->is_id())
      {
	delete r;
	return l;
      }
    if(l->is_id())
      {
	delete l;
	return r;
      }
    return NULL; /* nothing done here */
  }


  Expression * Divides::derive(int id)
  {
    return new Divides(session(),
		       new Minus(session(), // u' * v - u * v'  
				 new Times(session(), 
					   left->derive(id), right->copy()),
				 new Times(session(),
					   left->copy(), right->derive(id))),
		       new Times(session(), // v ** 2
				 right->copy(),
				 right->copy())
		       );
  }

  Expression * Divides::specific_rules(Expression * l, Expression * r)
  {
    if(l->is_null())
      {
	delete l;
	delete r;
	return new Null(session());
      }
    return NULL; /* nothing done here */
  }


  double Power::evaluate(const double * vals,const double * vars)
  { 
    return pow(left->evaluate(vals,vars),
	       right->evaluate(vals,vars));
  };

  // For now, we only implement it when the second argument is
  // a constant...
  Expression * Power::derive(int id)
  {
    if(right->is_const())
      {
	return new Times(session(),
			 new Times(session(),
				   left->derive(id),
				   new Const(session(), 
					     right->evaluate(NULL,NULL))),
			 new Power(session(),
				   left->copy(),
				   new Const(session(), 
					     right->evaluate(NULL,NULL) - 1.0)));
      }
    else
      return new Null(session()); // a little radical, I agree...
  }

  Expression * Power::specific_rules(Expression * l, Expression * r)
  {
    if(l->is_null())
      {
	delete l;
	delete r;
	return new Null(session());
      }
    if(l->is_id())
      {
	delete l;
	delete r;
	return new Const(session(), 1);
      }

    if(r->is_null())
      {
	delete l;
	delete r;
	return new Const(session(), 1);
      }
    if(r->is_id())
      {
	delete r;
	return l;
      }
    return NULL; /* nothing done here */
  }

  
};
