/*
 * Copyright 2008 Klaus Triendl
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free 
 * Software Foundation, 51 Franklin Street, Fifth Floor, 
 * Boston, MA 02110-1301, USA.
*/

#include <iostream>
#include <sigc++/functors/mem_fun.h>
#include <sigc++/adaptors/compose.h>
#include <sigc++/signal.h>
#include <glibmm/main.h>
#include <glibmm/thread.h>
#include <sigx/glib_auto_dispatchable.h>
#include <sigx/glib_threadable.h>
#include <sigx/signal_f.h>

using namespace std;


// fwd decl
class MainThread;

class MyThread: public sigx::glib_threadable
{
public:
	MyThread(MainThread* mainthread);


protected:
	// virtuals from sigx::threadable
	virtual void on_startup();

private:
	void on_ready();
	void on_test();


private:
	MainThread* m_mainthread;


// signal api
public:
	// expose the idle signal
	sigx::signal_f<Glib::SignalIdle> signal_idle;
};


class MainThread: public sigx::glib_auto_dispatchable
{
public:
	MainThread();
	void run();

protected:
	void on_self_ready();
	void on_mythread_idle();
	void on_end_mainthread();


protected:
	Glib::RefPtr<Glib::MainLoop> m_mainloop;
	sigc::signal<void> m_test_signal;
	MyThread m_mythread;

// threadsafe signal api
public:
	// expose a test signal
	sigx::signal_f<sigc::signal<void> > signal_test;
};






using namespace std;


MyThread::MyThread(MainThread* mainthread): 
	sigx::glib_threadable(), 
	m_mainthread(mainthread), 
	signal_idle(make_idle_signal_f())
{}

//virtual 
void MyThread::on_startup()
{
	maincontext()->signal_idle().connect(
		sigc::bind_return(sigc::mem_fun(this, &MyThread::on_ready), false)
	);
}

void MyThread::on_ready()
{
	m_mainthread->signal_test().connect(
		sigc::mem_fun(this, &MyThread::on_test)
	);
}

void MyThread::on_test()
{
	cout << "MyThread received test signal." << endl;
}



MainThread::MainThread(): 
	sigx::glib_auto_dispatchable(), 
	m_mainloop(), 
	m_test_signal(), 
	m_mythread(this), 
	// thread safe signal api
	signal_test(*this, &MainThread::m_test_signal)
{}

void MainThread::run()
{
	m_mainloop = Glib::MainLoop::create();

	// one-shot idle handler
	Glib::signal_idle().connect(
		sigc::bind_return(
			sigc::mem_fun(this, &MainThread::on_self_ready), 
			false
		)
	);
	
	m_mainloop->run();
}

void MainThread::on_self_ready()
{
	m_mythread.run();
	// connect to the thread's idle signal
	m_mythread.signal_idle().connect(
		sigc::bind_return(
			sigc::mem_fun(this, &MainThread::on_mythread_idle), 
			bool()
		)
	);
}

void MainThread::on_mythread_idle()
{
	cout << "slots connected to the test signal: " << m_test_signal.size() << endl;
	m_test_signal.emit();

	// end the thread, should disconnect its functors from the test signal
	m_mythread.finish();
	// by this point mythread's functor is still connected but the "disconnect" 
	// message is already queued in MainThread's dispatcher
	cout << "slots connected to the test signal after ending MyThread is still: " << m_test_signal.size() << endl;
	// there will be 2 messages many times: MyThread's emitted idle signal and MyThread's emitted disconnect message
	// to disconnect from MainThread's test signal
	cout << "messages in MainThread's dispatcher queue: " << dispatcher()->queued_contexts() << endl;
	// mythread's dispatcher died and mythread's functor is still connected but 
	// that won't do any harm, because the bad_dispatcher exception is caught
	// when calling the tunnel on an invalid dispatcher when emitting a signal;
	// test the signal
	m_test_signal.emit();

	// now, connect to the idle signal;
	// this gives MainThread's dispatcher the chance to execute the 
	// disconnection of MyThread's functor before on_end_mainthread() is called
	Glib::signal_idle().connect(
		sigc::bind_return(sigc::mem_fun(this, &MainThread::on_end_mainthread), false)
	);
}

void MainThread::on_end_mainthread()
{
	// m_mythread should have disconnected its functors 
	// at the time it finished
	cout << "slots connected to the test signal after ending MyThread: " << m_test_signal.size() << endl;
	// test the signal
	m_test_signal.emit();
	m_mainloop->quit();
}



int main()
{
	Glib::thread_init();

	MainThread app;
	app.run();

	cout << "press <enter>.";
	cin.get();
	return 0;
}
