#include "stdafx.h"
#include "SQL.h"
#include "Exception.h"

namespace sql {

	/**
	 * Bind helpers.
	 */

	void bind(Statement *to, Nat pos, MAYBE(Str *) str) {
		if (str)
			to->bind(pos, str);
		else
			to->bindNull(pos);
	}

	void bind(Statement *to, Nat pos, Maybe<Bool> v) {
		if (v.any())
			to->bind(pos, v.value());
		else
			to->bindNull(pos);
	}

	void bind(Statement *to, Nat pos, Maybe<Int> v) {
		if (v.any())
			to->bind(pos, v.value());
		else
			to->bindNull(pos);
	}

	void bind(Statement *to, Nat pos, Maybe<Long> v) {
		if (v.any())
			to->bind(pos, v.value());
		else
			to->bindNull(pos);
	}

	void bind(Statement *to, Nat pos, Maybe<Float> v) {
		if (v.any())
			to->bind(pos, v.value());
		else
			to->bindNull(pos);
	}

	void bind(Statement *to, Nat pos, Maybe<Double> v) {
		if (v.any())
			to->bind(pos, v.value());
		else
			to->bindNull(pos);
	}


	/**
	 * Statement.
	 */

	Statement::Statement(DBConnection *connection) : resultAliveCount(0), resultSequence(0) {
		connection->statements->putRaw(reinterpret_cast<TObject *>(this));
	}

	void Statement::invalidateResult() {
		atomicIncrement(resultSequence);
		atomicWrite(resultAliveCount, 0);
	}

	void Statement::deregister(DBConnection *c) {
		c->statements->removeRaw(reinterpret_cast<TObject *>(this));
	}

	Statement::Statement(const Statement &o) {
		throw new (this) SQLError(new (this) Str(S("Can not copy a prepared statement.")));
	}

	void Statement::deepCopy(CloneEnv *env) {
		throw new (this) SQLError(new (this) Str(S("Can not move a prepared statement across thread boundaries.")));
	}


	/**
	 * Statement result.
	 *
	 * Note: We use atomics to update the shared data in Statement. This is mostly to be able to be
	 * somewhat robust against the case where the API is misused from multiple threads. The
	 * implementation is not thread-safe (there are cases where we will fail), but as this is mostly
	 * a safeguard to be able to detect such cases, it is fine.
	 */

	Statement::Result::Result(Statement *stmt) : owner(stmt), sequence(atomicRead(stmt->resultSequence)) {
		atomicIncrement(owner->resultAliveCount);
	}

	Statement::Result::~Result() {
		if (atomicRead(owner->resultSequence) == sequence) {
			if (atomicDecrement(owner->resultAliveCount) == 0) {
				owner->disposeResult();
			}
		}
	}

	Statement::Result::Result(const Result &other) : owner(other.owner), sequence(other.sequence) {
		if (atomicRead(owner->resultSequence) == sequence) {
			atomicIncrement(owner->resultAliveCount);
		}
	}

	Statement::Result &Statement::Result::operator =(const Result &other) {
		if (this == &other)
			return *this;

		Result tmp(other);
		std::swap(owner, tmp.owner);
		std::swap(sequence, tmp.sequence);
		return *this;
	}

	Maybe<Row> Statement::Result::next() {
		if (atomicRead(owner->resultSequence) != sequence)
			return Maybe<Row>();

		return owner->nextRow();
	}

	Int Statement::Result::lastRowId() const {
		if (atomicRead(owner->resultSequence) != sequence)
			return -1;

		return owner->lastRowId();
	}

	Nat Statement::Result::changes() const {
		if (atomicRead(owner->resultSequence) != sequence)
			return 0;

		return owner->changes();
	}

	void Statement::Result::finalize() {
		if (atomicRead(owner->resultSequence) == sequence)
			owner->disposeResult();
	}

	void Statement::Result::deepCopy(CloneEnv *env) {
		throw new (env) SQLError(new (env) Str(S("Deep copies of database iterators are not supported.")));
	}


	/**
	 * Connection.
	 */

	DBConnection::DBConnection() {
		statements = new (this) WeakSetBase();
		cachedStatements = new (this) RefMap<CachedQuery *, Statement *>();
	}

	DBConnection::DBConnection(const DBConnection &o) {
		throw new (this) SQLError(new (this) Str(S("Can not copy a database connection.")));
	}

	void DBConnection::deepCopy(CloneEnv *env) {
		throw new (this) SQLError(new (this) Str(S("Can not move a database connection across thread boundaries.")));
	}

	Statement *DBConnection::prepare(QueryStr *query) {
		return prepare(query->generate(visitor()));
	}

	Statement *DBConnection::prepare(CachedQuery *cached) {
		Statement *out = cachedStatements->get(cached, null);
		if (!out) {
			out = cached->query(this);
			cachedStatements->put(cached, out);
		}
		return out;
	}

	void DBConnection::close() {
		// Note: We could close all 'cachedStatements' also, but those should all be in 'statements'
		// anyway, so we can just clear it here.
		cachedStatements->clear();

		WeakSetBase::Iter iter(statements);
		// Note: We know that everything that comes to us from the set will be alive and not finalized.
		while (RootObject *stmt = iter.nextRaw()) {
			reinterpret_cast<Statement *>(stmt)->finalize();
		}
	}

}
