/*
   uni2beta - Convert Unicode to Beta Code.

   Author: Paul Hardy, unifoundry <at> unifoundry.com

   Copyright (C) 2018, 2019, 2020 Paul Hardy

   LICENSE:

      This program is free software: you can redistribute it and/or modify
      it under the terms of the GNU General Public License as published by
      the Free Software Foundation, either version 2 of the License, or
      (at your option) any later version.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

%{
#include <config.h>   /* Generated by GNU Autotools */

#include <locale.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>

/*
   Definitions for Beta Code encoding.
*/
#define LANG_GREEK  0x00000
#define LANG_LATIN  0x01000
#define LANG_COPTIC 0x02000
#define LANG_ARABIC 0x04000  /* Defined by Thesaurus Linguae Graecae but not in their corpus */
#define LANG_HEBREW 0x08000
#define LANG_UTF    0x10000  /* Well-formed UTF-8 character; non-well-formed UTF-8 is skipped */
#define LANG_ASCII  0x20000  /* Miscellaneous character in ASCII range */

/*
   Use LATIN_START of "&" and LATIN_END of "$" for
   traditional Beta Code Latin and Greek combination.

   Use LATIN_START of "{" and LATIN_END of "}" for
   unibetacode ASCII/Unicode escape sequence.
*/
#define LATIN_START "&"
#define LATIN_END   "$"
// #define LATIN_START "{"
// #define LATIN_END   "}"

/*
   IS_HEXADECIMAL returns 1 if character is a valid
   hexadecimal digit, and returns 0 otherwise.
*/
#define IS_HEXADECIMAL(A)	( ( (A >= '0' && A <= '9') || \
				    (A >= 'A' && A <= 'F') || \
				    (A >= 'a' && A <= 'f') ) ? 1 : 0 )

#define YY_NO_INPUT
#define YY_NO_UNPUT

#define MAXSTRING 4096

int uppercase=0;          /* =1 to output uppercase Latin Beta Code; default is lowercase   */
/* lang_mode selects Greek, Latin, Coptic or Hebrew output; default is Greek */
int lang_mode=LANG_GREEK;

int  was_escaped=0;           /* =1 if last string was an escape sequence "{...}"         */
int  was_codept=0;            /* =1 if last string was a Unicode code point "\u..."       */
void escape_out (char *);     /* print escape sequence inside '{'...'}'                   */

/*
   build_output builds output one word at a time, for
   context-dependent interpretation of small middle
   and final Greek sigma.
*/
void build_output  (int, char *); /* Construct an output word; buffered for sigma */
void end_escape    ();            /* End escape sequence if printing one out      */

%}

%option noyywrap

ASCII		[\042-\051\053\055\057\072\074-\176]
ASCII_STRING	[\042-\051\053\055\057\072\074-\176]+([ \011]+[\041-\176]+)*
PUNCTUATION	[\041\054\056\073]+
GREEK		((\315\264)|(\315\276)|(\316[\220-\277])|(\317[\200-\216])|(\341[\274-\277][\200-\277]))
COPTIC		((\317[\242-\257])|(\342\262[\200-\261]))
HEBREW		\327[\220-\252]
NEWLINE		[\n]

%%
\302\253	build_output (LANG_GREEK, "\"");  /* U+00AB LEFT-POINTING DOUBLE ANGLE QUOTATION MARK  */
\302\273	build_output (LANG_GREEK, "\"");  /* U+00BB RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK */
\312\273	build_output (LANG_GREEK, "`");   /* U+02BB MODIFIER LETTER TURNED COMMA */
\312\274	build_output (LANG_GREEK, "'");   /* U+02BC MODIFIER LETTER APOSTROPHE   */
\342\200\230	build_output (LANG_GREEK, "`");   /* U+2018 LEFT SINGLE QUOTATION MARK   */
\342\200\231	build_output (LANG_GREEK, "'");   /* U+2019 RIGHT SINGLE QUOTATION MARK  */
\342\200\232	build_output (LANG_GREEK, "`");   /* U+201A SINGLE LOW-9 QUOTATION MARK           */
\342\200\233	build_output (LANG_GREEK, "'");   /* U+201B SINGLE HIGH-REVERSED-9 QUOTATION MARK */
\342\200\234	build_output (LANG_GREEK, "\"");  /* U+201C LEFT DOUBLE QUOTATION MARK  */
\342\200\235	build_output (LANG_GREEK, "\"");  /* U+201D RIGHT DOUBLE QUOTATION MARK */
\342\200\236	build_output (LANG_GREEK, "\"");  /* U+201E DOUBLE LOW-9 QUOTATION MARK */
\342\200\271	build_output (LANG_GREEK, "`");   /* U+2039 SINGLE LEFT-POINTING ANGLE QUOTATION MARK  */
\342\200\272	build_output (LANG_GREEK, "'");   /* U+203A SINGLE RIGHT-POINTING ANGLE QUOTATION MARK */
\314\200	{ /*
		      A combining grave accent is supported in Greek or Coptic.
		      If font mode is not Coptic, call as LANG_GREEK
		   */
		   if (lang_mode == LANG_COPTIC) {
		      build_output (LANG_COPTIC, "\\");	/* U+0300 COMBINING GRAVE ACCENT */
		   }
		   else {
		      build_output (LANG_GREEK, "\\");	/* U+0300 COMBINING GRAVE ACCENT */
		   }
		}
\314\201  build_output (LANG_GREEK, "/");	/* U+0301 COMBINING ACUTE ACCENT                    */
\314\210  build_output (LANG_GREEK, "+");	/* U+0308 COMBINING DIAERESIS                       */
\314\223  build_output (LANG_GREEK, ")");	/* U+0313 COMBINING COMMA ABOVE                     */
\314\224  build_output (LANG_GREEK, "(");  /* U+0314 COMBINING REVERSED COMMA ABOVE            */
\314\243  build_output (LANG_GREEK, "?");  /* U+0323 COMBINING DOT BELOW                       */
\315\202  build_output (LANG_GREEK, "=");  /* U+0342 COMBINING GREEK PERISPOMENI               */
\315\205  build_output (LANG_GREEK, "|");  /* U+0345 COMBINING GREEK YPOGEGRAMMENI             */
\315\264  build_output (LANG_GREEK, "#");	/* U+0374  GREEK NUMERAL SIGN                       */
\315\272  build_output (LANG_GREEK, "|");	/* U+037A  GREEK YPOGEGRAMMENI                      */
\302\267  build_output (LANG_GREEK, ":");	/* U+00B7  MIDDLE DOT (Greek ano teleia)            */
\315\276  build_output (LANG_GREEK, ";");	/* U+037E  GREEK QUESTION MARK                      */
\316\204  build_output (LANG_GREEK, "/");	/* U+0384  GREEK TONOS                              */
\316\205  build_output (LANG_GREEK, "+/");	/* U+0385  GREEK DIALYTIKA TONOS                    */
\316\206  build_output (LANG_GREEK, "*/a");/* U+0386  GREEK CAPITAL LETTER ALPHA WITH TONOS    */
\316\207  build_output (LANG_GREEK, ":");	/* U+0387  GREEK ANO TELEIA                         */
\316\210  build_output (LANG_GREEK, "*/e");/* U+0388  GREEK CAPITAL LETTER EPSILON WITH TONOS  */
\316\211  build_output (LANG_GREEK, "*/h");/* U+0389  GREEK CAPITAL LETTER ETA WITH TONOS      */
\316\212  build_output (LANG_GREEK, "*/i");/* U+038A  GREEK CAPITAL LETTER IOTA WITH TONOS     */
\316\214  build_output (LANG_GREEK, "*/o");/* U+038C  GREEK CAPITAL LETTER OMICRON WITH TONOS  */
\316\216  build_output (LANG_GREEK, "*/u");/* U+038E  GREEK CAPITAL LETTER UPSILON WITH TONOS  */
\316\217  build_output (LANG_GREEK, "*/w");/* U+038F  GREEK CAPITAL LETTER OMEGA WITH TONOS    */
\316\220  build_output (LANG_GREEK, "i+/");/* U+0390  GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS */
\316\221  build_output (LANG_GREEK, "*a");	/* U+0391  GREEK CAPITAL LETTER ALPHA               */
\316\222  build_output (LANG_GREEK, "*b");	/* U+0392  GREEK CAPITAL LETTER BETA                */
\316\223  build_output (LANG_GREEK, "*g");	/* U+0393  GREEK CAPITAL LETTER GAMMA               */
\316\224  build_output (LANG_GREEK, "*d");	/* U+0394  GREEK CAPITAL LETTER DELTA               */
\316\225  build_output (LANG_GREEK, "*e");	/* U+0395  GREEK CAPITAL LETTER EPSILON             */
\316\226  build_output (LANG_GREEK, "*z");	/* U+0396  GREEK CAPITAL LETTER ZETA                */
\316\227  build_output (LANG_GREEK, "*h");	/* U+0397  GREEK CAPITAL LETTER ETA                 */
\316\230  build_output (LANG_GREEK, "*q");	/* U+0398  GREEK CAPITAL LETTER THETA               */
\316\231  build_output (LANG_GREEK, "*i");	/* U+0399  GREEK CAPITAL LETTER IOTA                */
\316\232  build_output (LANG_GREEK, "*k");	/* U+039A  GREEK CAPITAL LETTER KAPPA               */
\316\233  build_output (LANG_GREEK, "*l");	/* U+039B  GREEK CAPITAL LETTER LAMDA               */
\316\234  build_output (LANG_GREEK, "*m");	/* U+039C  GREEK CAPITAL LETTER MU                  */
\316\235  build_output (LANG_GREEK, "*n");	/* U+039D  GREEK CAPITAL LETTER NU                  */
\316\236  build_output (LANG_GREEK, "*c");	/* U+039E  GREEK CAPITAL LETTER XI                  */
\316\237  build_output (LANG_GREEK, "*o");	/* U+039F  GREEK CAPITAL LETTER OMICRON             */
\316\240  build_output (LANG_GREEK, "*p");	/* U+03A0  GREEK CAPITAL LETTER PI                  */
\316\241  build_output (LANG_GREEK, "*r");	/* U+03A1  GREEK CAPITAL LETTER RHO                 */
\316\243  build_output (LANG_GREEK, "*s");	/* U+03A3  GREEK CAPITAL LETTER SIGMA               */
\316\244  build_output (LANG_GREEK, "*t");	/* U+03A4  GREEK CAPITAL LETTER TAU                 */
\316\245  build_output (LANG_GREEK, "*u");	/* U+03A5  GREEK CAPITAL LETTER UPSILON             */
\316\246  build_output (LANG_GREEK, "*f");	/* U+03A6  GREEK CAPITAL LETTER PHI                 */
\316\247  build_output (LANG_GREEK, "*x");	/* U+03A7  GREEK CAPITAL LETTER CHI                 */
\316\250  build_output (LANG_GREEK, "*y");	/* U+03A8  GREEK CAPITAL LETTER PSI                 */
\316\251  build_output (LANG_GREEK, "*w");	/* U+03A9  GREEK CAPITAL LETTER OMEGA               */
\316\252  build_output (LANG_GREEK, "*+i");/* U+03AA  GREEK CAPITAL LETTER IOTA WITH DIALYTIKA */
\316\253  build_output (LANG_GREEK, "*+u");/* U+03AB  GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA */
\316\254  build_output (LANG_GREEK, "a/");	/* U+03AC  GREEK SMALL LETTER ALPHA WITH TONOS      */
\316\255  build_output (LANG_GREEK, "e/");	/* U+03AD  GREEK SMALL LETTER EPSILON WITH TONOS    */
\316\256  build_output (LANG_GREEK, "h/");	/* U+03AE  GREEK SMALL LETTER ETA WITH TONOS        */
\316\257  build_output (LANG_GREEK, "i/");	/* U+03AF  GREEK SMALL LETTER IOTA WITH TONOS       */
\316\260  build_output (LANG_GREEK, "u+/");/* U+03B0  GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS */
\316\261  build_output (LANG_GREEK, "a");	/* U+03B1  GREEK SMALL LETTER ALPHA                 */
\316\262  build_output (LANG_GREEK, "b");	/* U+03B2  GREEK SMALL LETTER BETA                  */
\316\263  build_output (LANG_GREEK, "g");	/* U+03B3  GREEK SMALL LETTER GAMMA                 */
\316\264  build_output (LANG_GREEK, "d");	/* U+03B4  GREEK SMALL LETTER DELTA                 */
\316\265  build_output (LANG_GREEK, "e");	/* U+03B5  GREEK SMALL LETTER EPSILON               */
\316\266  build_output (LANG_GREEK, "z");	/* U+03B6  GREEK SMALL LETTER ZETA                  */
\316\267  build_output (LANG_GREEK, "h");	/* U+03B7  GREEK SMALL LETTER ETA                   */
\316\270  build_output (LANG_GREEK, "q");	/* U+03B8  GREEK SMALL LETTER THETA                 */
\316\271  build_output (LANG_GREEK, "i");	/* U+03B9  GREEK SMALL LETTER IOTA                  */
\316\272  build_output (LANG_GREEK, "k");	/* U+03BA  GREEK SMALL LETTER KAPPA                 */
\316\273  build_output (LANG_GREEK, "l");	/* U+03BB  GREEK SMALL LETTER LAMDA                 */
\316\274  build_output (LANG_GREEK, "m");	/* U+03BC  GREEK SMALL LETTER MU                    */
\316\275  build_output (LANG_GREEK, "n");	/* U+03BD  GREEK SMALL LETTER NU                    */
\316\276  build_output (LANG_GREEK, "c");	/* U+03BE  GREEK SMALL LETTER XI                    */
\316\277  build_output (LANG_GREEK, "o");	/* U+03BF  GREEK SMALL LETTER OMICRON               */
\317\200  build_output (LANG_GREEK, "p");	/* U+03C0  GREEK SMALL LETTER PI                    */
\317\201  build_output (LANG_GREEK, "r");	/* U+03C1  GREEK SMALL LETTER RHO                   */
\317\202/315[\220-\277]	build_output (LANG_GREEK, "s2");	/* U+03C2  GREEK SMALL LETTER FINAL SIGMA before Greek letter  */
\317\202/[\316-\317]  	build_output (LANG_GREEK, "s2");	/* U+03C2  GREEK SMALL LETTER FINAL SIGMA before Greek letter  */
\317\202/\341[\274-\277]  	build_output (LANG_GREEK, "s2");	/* U+03C2  GREEK SMALL LETTER FINAL SIGMA before Greek letter  */
\317\202  build_output (LANG_GREEK, "s");			/* U+03C2  GREEK SMALL LETTER FINAL SIGMA at end of Greek word */
\317\203/[^\315\316\317[\341[\274-\277]]] build_output (LANG_GREEK, "s1");	/* U+03C3  GREEK SMALL LETTER SIGMA not before Greek letter    */
\317\203  build_output (LANG_GREEK, "s");			/* U+03C3  GREEK SMALL LETTER SIGMA in middle of Greek word    */
\317\204  build_output (LANG_GREEK, "t");	/* U+03C4  GREEK SMALL LETTER TAU                   */
\317\205  build_output (LANG_GREEK, "u");	/* U+03C5  GREEK SMALL LETTER UPSILON               */
\317\206  build_output (LANG_GREEK, "f");	/* U+03C6  GREEK SMALL LETTER PHI                   */
\317\207  build_output (LANG_GREEK, "x");	/* U+03C7  GREEK SMALL LETTER CHI                   */
\317\210  build_output (LANG_GREEK, "y");	/* U+03C8  GREEK SMALL LETTER PSI                   */
\317\211  build_output (LANG_GREEK, "w");	/* U+03C9  GREEK SMALL LETTER OMEGA                 */
\317\212  build_output (LANG_GREEK, "i+"); /* U+03CA  GREEK SMALL LETTER IOTA WITH DIALYTIKA   */
\317\213  build_output (LANG_GREEK, "u+"); /* U+03CB  GREEK SMALL LETTER UPSILON WITH DIALYTIKA */
\317\214  build_output (LANG_GREEK, "o/"); /* U+03CC  GREEK SMALL LETTER OMICRON WITH TONOS    */
\317\215  build_output (LANG_GREEK, "u/"); /* U+03CD  GREEK SMALL LETTER UPSILON WITH TONOS    */
\317\216  build_output (LANG_GREEK, "w/"); /* U+03CE  GREEK SMALL LETTER OMEGA WITH TONOS      */
\317\234  build_output (LANG_GREEK, "*v"); /* U+03DC  GREEK LETTER DIGAMMA                     */
\317\235  build_output (LANG_GREEK, "v");	/* U+03DD  GREEK SMALL LETTER DIGAMMA               */
\317\242  build_output (LANG_COPTIC, "*s");	/* U+03E2 COPTIC CAPITAL LETTER SHEI   */
\317\243  build_output (LANG_COPTIC, "s");	/* U+03E3 COPTIC SMALL LETTER SHEI     */
\317\244  build_output (LANG_COPTIC, "*f");	/* U+03E4 COPTIC CAPITAL LETTER FEI    */
\317\245  build_output (LANG_COPTIC, "f");	/* U+03E5 COPTIC SMALL LETTER FEI      */
\317\246  build_output (LANG_COPTIC, "*k");	/* U+03E6 COPTIC CAPITAL LETTER KHEI   */
\317\247  build_output (LANG_COPTIC, "k");	/* U+03E7 COPTIC SMALL LETTER KHEI     */
\317\250  build_output (LANG_COPTIC, "*h");	/* U+03E8 COPTIC CAPITAL LETTER HORI   */
\317\251  build_output (LANG_COPTIC, "h");	/* U+03E9 COPTIC SMALL LETTER HORI     */
\317\252  build_output (LANG_COPTIC, "*j");	/* U+03EA COPTIC CAPITAL LETTER GANGIA */
\317\253  build_output (LANG_COPTIC, "j");	/* U+03EB COPTIC SMALL LETTER GANGIA   */
\317\254  build_output (LANG_COPTIC, "*g");	/* U+03EC COPTIC CAPITAL LETTER SHIMA  */
\317\255  build_output (LANG_COPTIC, "g");	/* U+03ED COPTIC SMALL LETTER SHIMA    */
\317\256  build_output (LANG_COPTIC, "*t");	/* U+03EE COPTIC CAPITAL LETTER DEI    */
\317\257  build_output (LANG_COPTIC, "t");	/* U+03EF COPTIC SMALL LETTER DEI      */
\317\262  build_output (LANG_GREEK, "s3");	/* U+03F2  GREEK LUNATE SIGMA SYMBOL                */
\317\271  build_output (LANG_GREEK, "*s3");	/* U+03F9  GREEK CAPITAL LUNATE SIGMA SYMBOL        */
\327\220  build_output (LANG_HEBREW, "A");	/* U+05D0 HEBREW LETTER ALEF           */
\327\221  build_output (LANG_HEBREW, "b");	/* U+05D1 HEBREW LETTER BET            */
\327\222  build_output (LANG_HEBREW, "g");	/* U+05D2 HEBREW LETTER GIMEL          */
\327\223  build_output (LANG_HEBREW, "d");	/* U+05D3 HEBREW LETTER DALET          */
\327\224  build_output (LANG_HEBREW, "h");	/* U+05D4 HEBREW LETTER HE             */
\327\225  build_output (LANG_HEBREW, "v");	/* U+05D5 HEBREW LETTER VAV            */
\327\226  build_output (LANG_HEBREW, "z");	/* U+05D6 HEBREW LETTER ZAYIN          */
\327\227  build_output (LANG_HEBREW, "H");	/* U+05D7 HEBREW LETTER HET            */
\327\230  build_output (LANG_HEBREW, "Q");	/* U+05D8 HEBREW LETTER TET            */
\327\231  build_output (LANG_HEBREW, "y");	/* U+05D9 HEBREW LETTER YOD            */
\327\232  build_output (LANG_HEBREW, "k2");	/* U+05DA HEBREW LETTER FINAL KAF      */
\327\233  build_output (LANG_HEBREW, "k1");	/* U+05DB HEBREW LETTER KAF            */
\327\234  build_output (LANG_HEBREW, "l");	/* U+05DC HEBREW LETTER LAMED          */
\327\235  build_output (LANG_HEBREW, "m2");	/* U+05DD HEBREW LETTER FINAL MEM      */
\327\236  build_output (LANG_HEBREW, "m1");	/* U+05DE HEBREW LETTER MEM            */
\327\237  build_output (LANG_HEBREW, "n2");	/* U+05DF HEBREW LETTER FINAL NUN      */
\327\240  build_output (LANG_HEBREW, "n1");	/* U+05E0 HEBREW LETTER NUN            */
\327\241  build_output (LANG_HEBREW, "S");	/* U+05E1 HEBREW LETTER SAMEKH         */
\327\242  build_output (LANG_HEBREW, "a");	/* U+05E2 HEBREW LETTER AYIN           */
\327\243  build_output (LANG_HEBREW, "p2");	/* U+05E3 HEBREW LETTER FINAL PE       */
\327\244  build_output (LANG_HEBREW, "p1");	/* U+05E4 HEBREW LETTER PE             */
\327\245  build_output (LANG_HEBREW, "T2");	/* U+05E5 HEBREW LETTER FINAL TSADI    */
\327\246  build_output (LANG_HEBREW, "T1");	/* U+05E6 HEBREW LETTER TSADI          */
\327\247  build_output (LANG_HEBREW, "q");	/* U+05E7 HEBREW LETTER QOF            */
\327\250  build_output (LANG_HEBREW, "r");	/* U+05E8 HEBREW LETTER RESH           */
\327\251  build_output (LANG_HEBREW, "s");	/* U+05E9 HEBREW LETTER SHIN           */
\327\252  build_output (LANG_HEBREW, "t");	/* U+05EA HEBREW LETTER TAV            */
\341\274\200  build_output (LANG_GREEK, "a)");	/* U+1F00  GREEK SMALL LETTER ALPHA WITH PSILI      */
\341\274\201  build_output (LANG_GREEK, "a(");	/* U+1F01  GREEK SMALL LETTER ALPHA WITH DASIA      */
\341\274\202  build_output (LANG_GREEK, "a)\\");	/* U+1F02  GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA */
\341\274\203  build_output (LANG_GREEK, "a(\\");	/* U+1F03  GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA */
\341\274\204  build_output (LANG_GREEK, "a)/");	/* U+1F04  GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA */
\341\274\205  build_output (LANG_GREEK, "a(/");	/* U+1F05  GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA */
\341\274\206  build_output (LANG_GREEK, "a)=");	/* U+1F06  GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI */
\341\274\207  build_output (LANG_GREEK, "a(=");	/* U+1F07  GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI */
\341\274\210  build_output (LANG_GREEK, "*)a");	/* U+1F08  GREEK CAPITAL LETTER ALPHA WITH PSILI    */
\341\274\211  build_output (LANG_GREEK, "*(a");	/* U+1F09  GREEK CAPITAL LETTER ALPHA WITH DASIA    */
\341\274\212  build_output (LANG_GREEK, "*)\\a");	/* U+1F0A  GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA */
\341\274\213  build_output (LANG_GREEK, "*(\\a");	/* U+1F0B  GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA */
\341\274\214  build_output (LANG_GREEK, "*)/a");	/* U+1F0C  GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA */
\341\274\215  build_output (LANG_GREEK, "*(/a");	/* U+1F0D  GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA */
\341\274\216  build_output (LANG_GREEK, "*)=a");	/* U+1F0E  GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI */
\341\274\217  build_output (LANG_GREEK, "*(=a");	/* U+1F0F  GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI */
\341\274\220  build_output (LANG_GREEK, "e)");	/* U+1F10  GREEK SMALL LETTER EPSILON WITH PSILI    */
\341\274\221  build_output (LANG_GREEK, "e(");	/* U+1F11  GREEK SMALL LETTER EPSILON WITH DASIA    */
\341\274\222  build_output (LANG_GREEK, "e)\\");	/* U+1F12  GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA */
\341\274\223  build_output (LANG_GREEK, "e(\\");	/* U+1F13  GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA */
\341\274\224  build_output (LANG_GREEK, "e)/");	/* U+1F14  GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA */
\341\274\225  build_output (LANG_GREEK, "e(/");	/* U+1F15  GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA */
\341\274\230  build_output (LANG_GREEK, "*)e");	/* U+1F18  GREEK CAPITAL LETTER EPSILON WITH PSILI  */
\341\274\231  build_output (LANG_GREEK, "*(e");	/* U+1F19  GREEK CAPITAL LETTER EPSILON WITH DASIA  */
\341\274\232  build_output (LANG_GREEK, "*)\\e");	/* U+1F1A  GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA */
\341\274\233  build_output (LANG_GREEK, "*(\\e");	/* U+1F1B  GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA */
\341\274\234  build_output (LANG_GREEK, "*)/e");	/* U+1F1C  GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA */
\341\274\235  build_output (LANG_GREEK, "*(/e");	/* U+1F1D  GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA */
\341\274\240  build_output (LANG_GREEK, "h)");	/* U+1F20  GREEK SMALL LETTER ETA WITH PSILI        */
\341\274\241  build_output (LANG_GREEK, "h(");	/* U+1F21  GREEK SMALL LETTER ETA WITH DASIA        */
\341\274\242  build_output (LANG_GREEK, "h)\\");	/* U+1F22  GREEK SMALL LETTER ETA WITH PSILI AND VARIA */
\341\274\243  build_output (LANG_GREEK, "h(\\");	/* U+1F23  GREEK SMALL LETTER ETA WITH DASIA AND VARIA */
\341\274\244  build_output (LANG_GREEK, "h)/");	/* U+1F24  GREEK SMALL LETTER ETA WITH PSILI AND OXIA */
\341\274\245  build_output (LANG_GREEK, "h(/");	/* U+1F25  GREEK SMALL LETTER ETA WITH DASIA AND OXIA */
\341\274\246  build_output (LANG_GREEK, "h)=");	/* U+1F26  GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI */
\341\274\247  build_output (LANG_GREEK, "h(=");	/* U+1F27  GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI */
\341\274\250  build_output (LANG_GREEK, "*)h");  	/* U+1F28  GREEK CAPITAL LETTER ETA WITH PSILI      */
\341\274\251  build_output (LANG_GREEK, "*)h");  	/* U+1F29  GREEK CAPITAL LETTER ETA WITH DASIA      */
\341\274\252  build_output (LANG_GREEK, "*)\\h");  /* U+1F2A  GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA */
\341\274\253  build_output (LANG_GREEK, "*)\\h");  /* U+1F2B  GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA */
\341\274\254  build_output (LANG_GREEK, "*)/h");  	/* U+1F2C  GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA */
\341\274\255  build_output (LANG_GREEK, "*)/h");  	/* U+1F2D  GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA */
\341\274\256  build_output (LANG_GREEK, "*)=h");  	/* U+1F2E  GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI */
\341\274\257  build_output (LANG_GREEK, "*)=h");  	/* U+1F2F  GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI */
\341\274\260  build_output (LANG_GREEK, "i)");	/* U+1F30  GREEK SMALL LETTER IOTA WITH PSILI       */
\341\274\261  build_output (LANG_GREEK, "i(");	/* U+1F31  GREEK SMALL LETTER IOTA WITH DASIA       */
\341\274\262  build_output (LANG_GREEK, "i)\\");	/* U+1F32  GREEK SMALL LETTER IOTA WITH PSILI AND VARIA */
\341\274\263  build_output (LANG_GREEK, "i(\\");	/* U+1F33  GREEK SMALL LETTER IOTA WITH DASIA AND VARIA */
\341\274\264  build_output (LANG_GREEK, "i)/");	/* U+1F34  GREEK SMALL LETTER IOTA WITH PSILI AND OXIA */
\341\274\265  build_output (LANG_GREEK, "i(/");	/* U+1F35  GREEK SMALL LETTER IOTA WITH DASIA AND OXIA */
\341\274\266  build_output (LANG_GREEK, "i)=");	/* U+1F36  GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI */
\341\274\267  build_output (LANG_GREEK, "i(=");	/* U+1F37  GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI */
\341\274\270  build_output (LANG_GREEK, "*)i");	/* U+1F38  GREEK CAPITAL LETTER IOTA WITH PSILI     */
\341\274\271  build_output (LANG_GREEK, "*(i");	/* U+1F39  GREEK CAPITAL LETTER IOTA WITH DASIA     */
\341\274\272  build_output (LANG_GREEK, "*)\\i");	/* U+1F3A  GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA */
\341\274\273  build_output (LANG_GREEK, "*)\\i");	/* U+1F3B  GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA */
\341\274\274  build_output (LANG_GREEK, "*)/i");	/* U+1F3C  GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA */
\341\274\275  build_output (LANG_GREEK, "*)/i");	/* U+1F3D  GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA */
\341\274\276  build_output (LANG_GREEK, "*)=i");	/* U+1F3E  GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI */
\341\274\277  build_output (LANG_GREEK, "*)=i");	/* U+1F3F  GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI */
\341\275\200  build_output (LANG_GREEK, "o)");	/* U+1F40  GREEK SMALL LETTER OMICRON WITH PSILI    */
\341\275\201  build_output (LANG_GREEK, "o(");	/* U+1F41  GREEK SMALL LETTER OMICRON WITH DASIA    */
\341\275\202  build_output (LANG_GREEK, "o)\\");	/* U+1F42  GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA */
\341\275\203  build_output (LANG_GREEK, "o(\\");	/* U+1F43  GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA */
\341\275\204  build_output (LANG_GREEK, "o)/");	/* U+1F44  GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA */
\341\275\205  build_output (LANG_GREEK, "o(/");	/* U+1F45  GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA */
\341\275\210  build_output (LANG_GREEK, "*)o");	/* U+1F48  GREEK CAPITAL LETTER OMICRON WITH PSILI  */
\341\275\211  build_output (LANG_GREEK, "*(o");	/* U+1F49  GREEK CAPITAL LETTER OMICRON WITH DASIA  */
\341\275\212  build_output (LANG_GREEK, "*)\\o");	/* U+1F4A  GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA */
\341\275\213  build_output (LANG_GREEK, "*(\\o");	/* U+1F4B  GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA */
\341\275\214  build_output (LANG_GREEK, "*)/o");	/* U+1F4C  GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA */
\341\275\215  build_output (LANG_GREEK, "*(/o");	/* U+1F4D  GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA */
\341\275\220  build_output (LANG_GREEK, "u)");	/* U+1F50  GREEK SMALL LETTER UPSILON WITH PSILI    */
\341\275\221  build_output (LANG_GREEK, "u(");	/* U+1F51  GREEK SMALL LETTER UPSILON WITH DASIA    */
\341\275\222  build_output (LANG_GREEK, "u)\\");	/* U+1F52  GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA */
\341\275\223  build_output (LANG_GREEK, "u(\\");	/* U+1F53  GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA */
\341\275\224  build_output (LANG_GREEK, "u)/");	/* U+1F54  GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA */
\341\275\225  build_output (LANG_GREEK, "u(/");	/* U+1F55  GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA */
\341\275\226  build_output (LANG_GREEK, "u)=");	/* U+1F56  GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI */
\341\275\227  build_output (LANG_GREEK, "u(=");	/* U+1F57  GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI */
\341\275\231  build_output (LANG_GREEK, "*(u");	/* U+1F59  GREEK CAPITAL LETTER UPSILON WITH DASIA  */
\341\275\233  build_output (LANG_GREEK, "*(\\u");	/* U+1F5B  GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA */
\341\275\235  build_output (LANG_GREEK, "*(/u");	/* U+1F5D  GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA */
\341\275\237  build_output (LANG_GREEK, "*(=u");	/* U+1F5F  GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI */
\341\275\240  build_output (LANG_GREEK, "w)");	/* U+1F60  GREEK SMALL LETTER OMEGA WITH PSILI      */
\341\275\241  build_output (LANG_GREEK, "w(");	/* U+1F61  GREEK SMALL LETTER OMEGA WITH DASIA      */
\341\275\242  build_output (LANG_GREEK, "w)\\");	/* U+1F62  GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA */
\341\275\243  build_output (LANG_GREEK, "w(\\");	/* U+1F63  GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA */
\341\275\244  build_output (LANG_GREEK, "w)/");	/* U+1F64  GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA */
\341\275\245  build_output (LANG_GREEK, "w(/");	/* U+1F65  GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA */
\341\275\246  build_output (LANG_GREEK, "w)=");	/* U+1F66  GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI */
\341\275\247  build_output (LANG_GREEK, "w(=");	/* U+1F67  GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI */
\341\275\250  build_output (LANG_GREEK, "*)w");	/* U+1F68  GREEK CAPITAL LETTER OMEGA WITH PSILI    */
\341\275\251  build_output (LANG_GREEK, "*(w");	/* U+1F69  GREEK CAPITAL LETTER OMEGA WITH DASIA    */
\341\275\252  build_output (LANG_GREEK, "*)\\w");	/* U+1F6A  GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA */
\341\275\253  build_output (LANG_GREEK, "*(\\w");	/* U+1F6B  GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA */
\341\275\254  build_output (LANG_GREEK, "*)/w");	/* U+1F6C  GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA */
\341\275\255  build_output (LANG_GREEK, "*(/w");	/* U+1F6D  GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA */
\341\275\256  build_output (LANG_GREEK, "*)=w");	/* U+1F6E  GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI */
\341\275\257  build_output (LANG_GREEK, "*(=w");	/* U+1F6F  GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI */
\341\275\260  build_output (LANG_GREEK, "a\\");	/* U+1F70  GREEK SMALL LETTER ALPHA WITH VARIA      */
\341\275\261  build_output (LANG_GREEK, "a/");	/* U+1F71  GREEK SMALL LETTER ALPHA WITH OXIA       */
\341\275\262  build_output (LANG_GREEK, "e\\");	/* U+1F72  GREEK SMALL LETTER EPSILON WITH VARIA    */
\341\275\263  build_output (LANG_GREEK, "e/");	/* U+1F73  GREEK SMALL LETTER EPSILON WITH OXIA     */
\341\275\264  build_output (LANG_GREEK, "h\\");	/* U+1F74  GREEK SMALL LETTER ETA WITH VARIA        */
\341\275\265  build_output (LANG_GREEK, "h/");	/* U+1F75  GREEK SMALL LETTER ETA WITH OXIA         */
\341\275\266  build_output (LANG_GREEK, "i\\");	/* U+1F76  GREEK SMALL LETTER IOTA WITH VARIA       */
\341\275\267  build_output (LANG_GREEK, "i/");	/* U+1F77  GREEK SMALL LETTER IOTA WITH OXIA        */
\341\275\270  build_output (LANG_GREEK, "o\\");	/* U+1F78  GREEK SMALL LETTER OMICRON WITH VARIA    */
\341\275\271  build_output (LANG_GREEK, "o/");	/* U+1F79  GREEK SMALL LETTER OMICRON WITH OXIA     */
\341\275\272  build_output (LANG_GREEK, "u\\");	/* U+1F7A  GREEK SMALL LETTER UPSILON WITH VARIA    */
\341\275\273  build_output (LANG_GREEK, "u/");	/* U+1F7B  GREEK SMALL LETTER UPSILON WITH OXIA     */
\341\275\274  build_output (LANG_GREEK, "w\\");	/* U+1F7C  GREEK SMALL LETTER OMEGA WITH VARIA      */
\341\275\275  build_output (LANG_GREEK, "w/");	/* U+1F7D  GREEK SMALL LETTER OMEGA WITH OXIA       */
\341\276\200  build_output (LANG_GREEK, "a)|");	/* U+1F80  GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI */
\341\276\201  build_output (LANG_GREEK, "a(|");	/* U+1F81  GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI */
\341\276\202  build_output (LANG_GREEK, "a)\\|");	/* U+1F82  GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI */
\341\276\203  build_output (LANG_GREEK, "a(\\|");	/* U+1F83  GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI */
\341\276\204  build_output (LANG_GREEK, "a)/|");	/* U+1F84  GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI */
\341\276\205  build_output (LANG_GREEK, "a(/|");	/* U+1F85  GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI */
\341\276\206  build_output (LANG_GREEK, "a)=|");	/* U+1F86  GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI */
\341\276\207  build_output (LANG_GREEK, "a(=|");	/* U+1F87  GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI */
\341\276\210  build_output (LANG_GREEK, "*)a|");	/* U+1F88  GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI */
\341\276\211  build_output (LANG_GREEK, "*(a|");	/* U+1F89  GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI */
\341\276\212  build_output (LANG_GREEK, "*)\\a|");	/* U+1F8A  GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI */
\341\276\213  build_output (LANG_GREEK, "*(\\a|");	/* U+1F8B  GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI */
\341\276\214  build_output (LANG_GREEK, "*)/a|");	/* U+1F8C  GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI */
\341\276\215  build_output (LANG_GREEK, "*(/a|");	/* U+1F8D  GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI */
\341\276\216  build_output (LANG_GREEK, "*)=a|");	/* U+1F8E  GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI */
\341\276\217  build_output (LANG_GREEK, "*(=a|");	/* U+1F8F  GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI */
\341\276\220  build_output (LANG_GREEK, "h)|");	/* U+1F90  GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI */
\341\276\221  build_output (LANG_GREEK, "h(|");	/* U+1F91  GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI */
\341\276\222  build_output (LANG_GREEK, "h)\\|");	/* U+1F92  GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI */
\341\276\223  build_output (LANG_GREEK, "h(\\|");	/* U+1F93  GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI */
\341\276\224  build_output (LANG_GREEK, "h)/|");	/* U+1F94  GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI */
\341\276\225  build_output (LANG_GREEK, "h(/|");	/* U+1F95  GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI */
\341\276\226  build_output (LANG_GREEK, "h)=|");	/* U+1F96  GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI */
\341\276\227  build_output (LANG_GREEK, "h(=|");	/* U+1F97  GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI */
\341\276\230  build_output (LANG_GREEK, "*)e|");	/* U+1F98  GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI */
\341\276\231  build_output (LANG_GREEK, "*(e|");	/* U+1F99  GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI */
\341\276\232  build_output (LANG_GREEK, "*)\\e|");	/* U+1F9A  GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI */
\341\276\233  build_output (LANG_GREEK, "*(\\e|");	/* U+1F9B  GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI */
\341\276\234  build_output (LANG_GREEK, "*)/e|");	/* U+1F9C  GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI */
\341\276\235  build_output (LANG_GREEK, "*(/e|");	/* U+1F9D  GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI */
\341\276\236  build_output (LANG_GREEK, "*)=e|");	/* U+1F9E  GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI */
\341\276\237  build_output (LANG_GREEK, "*(=e|");	/* U+1F9F  GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI */
\341\276\240  build_output (LANG_GREEK, "w)|");	/* U+1FA0  GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI */
\341\276\241  build_output (LANG_GREEK, "w(|");	/* U+1FA1  GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI */
\341\276\242  build_output (LANG_GREEK, "w)\\|");	/* U+1FA2  GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI */
\341\276\243  build_output (LANG_GREEK, "w(\\|");	/* U+1FA3  GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI */
\341\276\244  build_output (LANG_GREEK, "w)/|");	/* U+1FA4  GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI */
\341\276\245  build_output (LANG_GREEK, "w(/|");	/* U+1FA5  GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI */
\341\276\246  build_output (LANG_GREEK, "w)=|");	/* U+1FA6  GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI */
\341\276\247  build_output (LANG_GREEK, "w(=|");	/* U+1FA7  GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI */
\341\276\250  build_output (LANG_GREEK, "*)w|");	/* U+1FA8  GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI */
\341\276\251  build_output (LANG_GREEK, "*(w|");	/* U+1FA9  GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI */
\341\276\252  build_output (LANG_GREEK, "*)\\w|");	/* U+1FAA  GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI */
\341\276\253  build_output (LANG_GREEK, "*(\\w|");	/* U+1FAB  GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI */
\341\276\254  build_output (LANG_GREEK, "*)/w|");	/* U+1FAC  GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI */
\341\276\255  build_output (LANG_GREEK, "*(/w|");	/* U+1FAD  GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI */
\341\276\256  build_output (LANG_GREEK, "*)=w|");	/* U+1FAE  GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI */
\341\276\257  build_output (LANG_GREEK, "*(=w|");	/* U+1FAF  GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI */
\341\276\260  build_output (LANG_GREEK, "a'");	/* U+1FB0  GREEK SMALL LETTER ALPHA WITH VRACHY     */
\341\276\261  build_output (LANG_GREEK, "a&");	/* U+1FB1  GREEK SMALL LETTER ALPHA WITH MACRON     */
\341\276\262  build_output (LANG_GREEK, "a\\|");	/* U+1FB2  GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI */
\341\276\263  build_output (LANG_GREEK, "a|");	/* U+1FB3  GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI */
\341\276\264  build_output (LANG_GREEK, "a/|");	/* U+1FB4  GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI */
\341\276\266  build_output (LANG_GREEK, "a=");	/* U+1FB6  GREEK SMALL LETTER ALPHA WITH PERISPOMENI */
\341\276\267  build_output (LANG_GREEK, "a=|");	/* U+1FB7  GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI */
\341\276\270  build_output (LANG_GREEK, "*'a");	/* U+1FB8  GREEK CAPITAL LETTER ALPHA WITH VRACHY   */
\341\276\271  build_output (LANG_GREEK, "*&a");	/* U+1FB9  GREEK CAPITAL LETTER ALPHA WITH MACRON   */
\341\276\272  build_output (LANG_GREEK, "*\\a");	/* U+1FBA  GREEK CAPITAL LETTER ALPHA WITH VARIA    */
\341\276\273  build_output (LANG_GREEK, "*/a");	/* U+1FBB  GREEK CAPITAL LETTER ALPHA WITH OXIA     */
\341\276\274  build_output (LANG_GREEK, "*a|");	/* U+1FBC  GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI */
\341\276\275  build_output (LANG_GREEK, "'");	/* U+1FBD  GREEK KORONIS                            */
\341\276\276  build_output (LANG_GREEK, "|");	/* U+1FBE  GREEK PROSGEGRAMMENI                     */
\341\276\277  build_output (LANG_GREEK, ")");	/* U+1FBF  GREEK PSILI                              */
\341\277\200  build_output (LANG_GREEK, "=");	/* U+1FC0  GREEK PERISPOMENI                        */
\341\277\201  build_output (LANG_GREEK, "+=");	/* U+1FC1  GREEK DIALYTIKA AND PERISPOMENI          */
\341\277\202  build_output (LANG_GREEK, "h\\|");	/* U+1FC2  GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI */
\341\277\203  build_output (LANG_GREEK, "h|");	/* U+1FC3  GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI */
\341\277\204  build_output (LANG_GREEK, "h/|");	/* U+1FC4  GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI */
\341\277\206  build_output (LANG_GREEK, "h=");	/* U+1FC6  GREEK SMALL LETTER ETA WITH PERISPOMENI  */
\341\277\207  build_output (LANG_GREEK, "h=|");	/* U+1FC7  GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI */
\341\277\210  build_output (LANG_GREEK, "*\\e");	/* U+1FC8  GREEK CAPITAL LETTER EPSILON WITH VARIA  */
\341\277\211  build_output (LANG_GREEK, "*/e");	/* U+1FC9  GREEK CAPITAL LETTER EPSILON WITH OXIA   */
\341\277\212  build_output (LANG_GREEK, "*\\h");	/* U+1FCA  GREEK CAPITAL LETTER ETA WITH VARIA      */
\341\277\213  build_output (LANG_GREEK, "*/h");	/* U+1FCB  GREEK CAPITAL LETTER ETA WITH OXIA       */
\341\277\214  build_output (LANG_GREEK, "*h|");	/* U+1FCC  GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI */
\341\277\215  build_output (LANG_GREEK, ")\\");	/* U+1FCD  GREEK PSILI AND VARIA                    */
\341\277\216  build_output (LANG_GREEK, ")/");	/* U+1FCE  GREEK PSILI AND OXIA                     */
\341\277\217  build_output (LANG_GREEK, ")=");	/* U+1FCF  GREEK PSILI AND PERISPOMENI              */
\341\277\220  build_output (LANG_GREEK, "i'");	/* U+1FD0  GREEK SMALL LETTER IOTA WITH VRACHY      */
\341\277\221  build_output (LANG_GREEK, "i&");	/* U+1FD1  GREEK SMALL LETTER IOTA WITH MACRON      */
\341\277\222  build_output (LANG_GREEK, "i+\\");	/* U+1FD2  GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA */
\341\277\223  build_output (LANG_GREEK, "i+/");	/* U+1FD3  GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA */
\341\277\226  build_output (LANG_GREEK, "i=");	/* U+1FD6  GREEK SMALL LETTER IOTA WITH PERISPOMENI */
\341\277\227  build_output (LANG_GREEK, "i+=");	/* U+1FD7  GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI */
\341\277\230  build_output (LANG_GREEK, "*'i");	/* U+1FD8  GREEK CAPITAL LETTER IOTA WITH VRACHY    */
\341\277\231  build_output (LANG_GREEK, "*&i");	/* U+1FD9  GREEK CAPITAL LETTER IOTA WITH MACRON    */
\341\277\232  build_output (LANG_GREEK, "*\\i");	/* U+1FDA  GREEK CAPITAL LETTER IOTA WITH VARIA     */
\341\277\233  build_output (LANG_GREEK, "*/i");	/* U+1FDB  GREEK CAPITAL LETTER IOTA WITH OXIA      */
\341\277\235  build_output (LANG_GREEK, "(\\");	/* U+1FDD  GREEK DASIA AND VARIA                    */
\341\277\236  build_output (LANG_GREEK, "(/");	/* U+1FDE  GREEK DASIA AND OXIA                     */
\341\277\237  build_output (LANG_GREEK, "(=");	/* U+1FDF  GREEK DASIA AND PERISPOMENI              */
\341\277\240  build_output (LANG_GREEK, "u'");	/* U+1FE0  GREEK SMALL LETTER UPSILON WITH VRACHY   */
\341\277\241  build_output (LANG_GREEK, "u&");	/* U+1FE1  GREEK SMALL LETTER UPSILON WITH MACRON   */
\341\277\242  build_output (LANG_GREEK, "u+\\");	/* U+1FE2  GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA */
\341\277\243  build_output (LANG_GREEK, "u+/");	/* U+1FE3  GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA */
\341\277\244  build_output (LANG_GREEK, "r)");	/* U+1FE4  GREEK SMALL LETTER RHO WITH PSILI        */
\341\277\245  build_output (LANG_GREEK, "r(");	/* U+1FE5  GREEK SMALL LETTER RHO WITH DASIA        */
\341\277\246  build_output (LANG_GREEK, "u=");	/* U+1FE6  GREEK SMALL LETTER UPSILON WITH PERISPOMENI */
\341\277\247  build_output (LANG_GREEK, "u+=");	/* U+1FE7  GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI */
\341\277\250  build_output (LANG_GREEK, "*'u");	/* U+1FE8  GREEK CAPITAL LETTER UPSILON WITH VRACHY */
\341\277\251  build_output (LANG_GREEK, "*&u");	/* U+1FE9  GREEK CAPITAL LETTER UPSILON WITH MACRON */
\341\277\252  build_output (LANG_GREEK, "*\\u");	/* U+1FEA  GREEK CAPITAL LETTER UPSILON WITH VARIA  */
\341\277\253  build_output (LANG_GREEK, "*/u");	/* U+1FEB  GREEK CAPITAL LETTER UPSILON WITH OXIA   */
\341\277\254  build_output (LANG_GREEK, "*(r");	/* U+1FEC  GREEK CAPITAL LETTER RHO WITH DASIA      */
\341\277\255  build_output (LANG_GREEK, "+\\");	/* U+1FED  GREEK DIALYTIKA AND VARIA                */
\341\277\256  build_output (LANG_GREEK, "+/");	/* U+1FEE  GREEK DIALYTIKA AND OXIA                 */
\341\277\257  build_output (LANG_GREEK, "\\");	/* U+1FEF  GREEK VARIA                              */
\341\277\262  build_output (LANG_GREEK, "w\\|");	/* U+1FF2  GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI */
\341\277\263  build_output (LANG_GREEK, "w|");	/* U+1FF3  GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI */
\341\277\264  build_output (LANG_GREEK, "w/|");	/* U+1FF4  GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI */
\341\277\266  build_output (LANG_GREEK, "w=");	/* U+1FF6  GREEK SMALL LETTER OMEGA WITH PERISPOMENI */
\341\277\267  build_output (LANG_GREEK, "w=|");	/* U+1FF7  GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI */
\341\277\270  build_output (LANG_GREEK, "*\\o");	/* U+1FF8  GREEK CAPITAL LETTER OMICRON WITH VARIA  */
\341\277\271  build_output (LANG_GREEK, "*/o");	/* U+1FF9  GREEK CAPITAL LETTER OMICRON WITH OXIA   */
\341\277\272  build_output (LANG_GREEK, "*\\w");	/* U+1FFA  GREEK CAPITAL LETTER OMEGA WITH VARIA    */
\341\277\273  build_output (LANG_GREEK, "*/w");	/* U+1FFB  GREEK CAPITAL LETTER OMEGA WITH OXIA     */
\341\277\274  build_output (LANG_GREEK, "*w|");	/* U+1FFC  GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI */
\341\277\275  build_output (LANG_GREEK, "/");	/* U+1FFD  GREEK OXIA                               */
\341\277\276  build_output (LANG_GREEK, "(");	/* U+1FFE  GREEK DASIA                              */
\316\221\314\201           build_output (LANG_GREEK, "*/a");	/* U+0386 GREEK CAPITAL LETTER ALPHA WITH TONOS                   */
\316\225\314\201           build_output (LANG_GREEK, "*/e");	/* U+0388 GREEK CAPITAL LETTER EPSILON WITH TONOS                 */
\316\227\314\201           build_output (LANG_GREEK, "*/h");	/* U+0389 GREEK CAPITAL LETTER ETA WITH TONOS                     */
\316\231\314\201           build_output (LANG_GREEK, "*/i");	/* U+038A GREEK CAPITAL LETTER IOTA WITH TONOS                    */
\316\237\314\201           build_output (LANG_GREEK, "*/o");	/* U+038C GREEK CAPITAL LETTER OMICRON WITH TONOS                 */
\316\245\314\201           build_output (LANG_GREEK, "*/u");	/* U+038E GREEK CAPITAL LETTER UPSILON WITH TONOS                 */
\316\251\314\201           build_output (LANG_GREEK, "*/w");	/* U+038F GREEK CAPITAL LETTER OMEGA WITH TONOS                   */
\316\271\314\201\314\210   build_output (LANG_GREEK, "i+/");	/* U+0390 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS        */
\316\271\314\210\314\201   build_output (LANG_GREEK, "i+/");	/* U+0390 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS        */
\316\231\314\210           build_output (LANG_GREEK, "*+i");	/* U+03AA GREEK CAPITAL LETTER IOTA WITH DIALYTIKA                */
\316\245\314\210           build_output (LANG_GREEK, "*+u");	/* U+03AB GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA             */
\316\261\314\201           build_output (LANG_GREEK, "a/");	/* U+03AC GREEK SMALL LETTER ALPHA WITH TONOS                     */
\316\265\314\201           build_output (LANG_GREEK, "e/");	/* U+03AD GREEK SMALL LETTER EPSILON WITH TONOS                   */
\316\267\314\201           build_output (LANG_GREEK, "h/");	/* U+03AE GREEK SMALL LETTER ETA WITH TONOS                       */
\316\271\314\201           build_output (LANG_GREEK, "i/");	/* U+03AF GREEK SMALL LETTER IOTA WITH TONOS                      */
\317\205\314\201\314\210   build_output (LANG_GREEK, "u+/");	/* U+03B0 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS     */
\317\205\314\210\314\201   build_output (LANG_GREEK, "u+/");	/* U+03B0 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS     */
\316\271\314\210           build_output (LANG_GREEK, "i+");	/* U+03CA GREEK SMALL LETTER IOTA WITH DIALYTIKA                  */
\317\205\314\210           build_output (LANG_GREEK, "u+");	/* U+03CB GREEK SMALL LETTER UPSILON WITH DIALYTIKA               */
\316\277\314\201           build_output (LANG_GREEK, "o/");	/* U+03CC GREEK SMALL LETTER OMICRON WITH TONOS                   */
\317\205\314\201           build_output (LANG_GREEK, "u/");	/* U+03CD GREEK SMALL LETTER UPSILON WITH TONOS                   */
\317\211\314\201           build_output (LANG_GREEK, "w/");	/* U+03CE GREEK SMALL LETTER OMEGA WITH TONOS                     */
\316\261\314\223           build_output (LANG_GREEK, "a)");	/* U+1F00 GREEK SMALL LETTER ALPHA WITH PSILI                     */
\316\261\314\224           build_output (LANG_GREEK, "a(");	/* U+1F01 GREEK SMALL LETTER ALPHA WITH DASIA                     */
\316\261\314\223\314\200   build_output (LANG_GREEK, "a)\\");	/* U+1F02 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA           */
\316\261\314\224\314\200   build_output (LANG_GREEK, "a(\\");	/* U+1F03 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA           */
\316\261\314\223\314\201   build_output (LANG_GREEK, "a)/");	/* U+1F04 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA            */
\316\261\314\224\314\201   build_output (LANG_GREEK, "a(/");	/* U+1F05 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA            */
\316\261\314\223\315\202   build_output (LANG_GREEK, "a)=");	/* U+1F06 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI     */
\316\261\314\224\315\202   build_output (LANG_GREEK, "a(=");	/* U+1F07 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI     */
\316\221\314\223           build_output (LANG_GREEK, "*)a");	/* U+1F08 GREEK CAPITAL LETTER ALPHA WITH PSILI                   */
\316\221\314\224           build_output (LANG_GREEK, "*(a");	/* U+1F09 GREEK CAPITAL LETTER ALPHA WITH DASIA                   */
\316\221\314\223\314\200   build_output (LANG_GREEK, "*)\\a");	/* U+1F0A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA         */
\316\221\314\224\314\200   build_output (LANG_GREEK, "*(\\a");	/* U+1F0B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA         */
\316\221\314\223\314\201   build_output (LANG_GREEK, "*)/a");	/* U+1F0C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA          */
\316\221\314\224\314\201   build_output (LANG_GREEK, "*(/a");	/* U+1F0D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA          */
\316\221\314\223\315\202   build_output (LANG_GREEK, "*)=a");	/* U+1F0E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI   */
\316\221\314\224\315\202   build_output (LANG_GREEK, "*(=a");	/* U+1F0F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI   */
\316\265\314\223           build_output (LANG_GREEK, "e)");	/* U+1F10 GREEK SMALL LETTER EPSILON WITH PSILI                   */
\316\265\314\224           build_output (LANG_GREEK, "e(");	/* U+1F11 GREEK SMALL LETTER EPSILON WITH DASIA                   */
\316\265\314\223\314\200   build_output (LANG_GREEK, "e)\\");	/* U+1F12 GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA         */
\316\265\314\224\314\200   build_output (LANG_GREEK, "e(\\");	/* U+1F13 GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA         */
\316\265\314\223\314\201   build_output (LANG_GREEK, "e)/");	/* U+1F14 GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA          */
\316\265\314\224\314\201   build_output (LANG_GREEK, "e(/");	/* U+1F15 GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA          */
\316\225\314\223           build_output (LANG_GREEK, "*)e");	/* U+1F18 GREEK CAPITAL LETTER EPSILON WITH PSILI                 */
\316\225\314\224           build_output (LANG_GREEK, "*(e");	/* U+1F19 GREEK CAPITAL LETTER EPSILON WITH DASIA                 */
\316\225\314\223\314\200   build_output (LANG_GREEK, "*)\\e");	/* U+1F1A GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA       */
\316\225\314\224\314\200   build_output (LANG_GREEK, "*(\\e");	/* U+1F1B GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA       */
\316\225\314\223\314\201   build_output (LANG_GREEK, "*)/e");	/* U+1F1C GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA        */
\316\225\314\224\314\201   build_output (LANG_GREEK, "*(/e");	/* U+1F1D GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA        */
\316\267\314\223           build_output (LANG_GREEK, "h)");	/* U+1F20 GREEK SMALL LETTER ETA WITH PSILI                       */
\316\267\314\224           build_output (LANG_GREEK, "h(");	/* U+1F21 GREEK SMALL LETTER ETA WITH DASIA                       */
\316\267\314\223\314\200   build_output (LANG_GREEK, "h)\\");	/* U+1F22 GREEK SMALL LETTER ETA WITH PSILI AND VARIA             */
\316\267\314\224\314\200   build_output (LANG_GREEK, "h(\\");	/* U+1F23 GREEK SMALL LETTER ETA WITH DASIA AND VARIA             */
\316\267\314\223\314\201   build_output (LANG_GREEK, "h)/");	/* U+1F24 GREEK SMALL LETTER ETA WITH PSILI AND OXIA              */
\316\267\314\224\314\201   build_output (LANG_GREEK, "h(/");	/* U+1F25 GREEK SMALL LETTER ETA WITH DASIA AND OXIA              */
\316\267\314\223\315\202   build_output (LANG_GREEK, "h)=");	/* U+1F26 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI       */
\316\267\314\224\315\202   build_output (LANG_GREEK, "h(=");	/* U+1F27 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI       */
\316\227\314\223           build_output (LANG_GREEK, "*)h");	/* U+1F28 GREEK CAPITAL LETTER ETA WITH PSILI                     */
\316\227\314\224           build_output (LANG_GREEK, "*(h");	/* U+1F29 GREEK CAPITAL LETTER ETA WITH DASIA                     */
\316\227\314\223\314\200   build_output (LANG_GREEK, "*)\\h");	/* U+1F2A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA           */
\316\227\314\224\314\200   build_output (LANG_GREEK, "*(\\h");	/* U+1F2B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA           */
\316\227\314\223\314\201   build_output (LANG_GREEK, "*)/h");	/* U+1F2C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA            */
\316\227\314\224\314\201   build_output (LANG_GREEK, "*(/h");	/* U+1F2D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA            */
\316\227\314\223\315\202   build_output (LANG_GREEK, "*)=h");	/* U+1F2E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI     */
\316\227\314\224\315\202   build_output (LANG_GREEK, "*(=h");	/* U+1F2F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI     */
\316\271\314\223           build_output (LANG_GREEK, "i)");	/* U+1F30 GREEK SMALL LETTER IOTA WITH PSILI                      */
\316\271\314\224           build_output (LANG_GREEK, "i(");	/* U+1F31 GREEK SMALL LETTER IOTA WITH DASIA                      */
\316\271\314\223\314\200   build_output (LANG_GREEK, "i)\\");	/* U+1F32 GREEK SMALL LETTER IOTA WITH PSILI AND VARIA            */
\316\271\314\224\314\200   build_output (LANG_GREEK, "i(\\");	/* U+1F33 GREEK SMALL LETTER IOTA WITH DASIA AND VARIA            */
\316\271\314\223\314\201   build_output (LANG_GREEK, "i)/");	/* U+1F34 GREEK SMALL LETTER IOTA WITH PSILI AND OXIA             */
\316\271\314\224\314\201   build_output (LANG_GREEK, "i(/");	/* U+1F35 GREEK SMALL LETTER IOTA WITH DASIA AND OXIA             */
\316\271\314\223\315\202   build_output (LANG_GREEK, "i)=");	/* U+1F36 GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI      */
\316\271\314\224\315\202   build_output (LANG_GREEK, "i(=");	/* U+1F37 GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI      */
\316\231\314\223           build_output (LANG_GREEK, "*)i");	/* U+1F38 GREEK CAPITAL LETTER IOTA WITH PSILI                    */
\316\231\314\224           build_output (LANG_GREEK, "*(i");	/* U+1F39 GREEK CAPITAL LETTER IOTA WITH DASIA                    */
\316\231\314\223\314\200   build_output (LANG_GREEK, "*)\\i");	/* U+1F3A GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA          */
\316\231\314\224\314\200   build_output (LANG_GREEK, "*(\\i");	/* U+1F3B GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA          */
\316\231\314\223\314\201   build_output (LANG_GREEK, "*)/i");	/* U+1F3C GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA           */
\316\231\314\224\314\201   build_output (LANG_GREEK, "*(/i");	/* U+1F3D GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA           */
\316\231\314\223\315\202   build_output (LANG_GREEK, "*)=i");	/* U+1F3E GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI    */
\316\231\314\224\315\202   build_output (LANG_GREEK, "*(=i");	/* U+1F3F GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI    */
\316\277\314\223           build_output (LANG_GREEK, "o)");	/* U+1F40 GREEK SMALL LETTER OMICRON WITH PSILI                   */
\316\277\314\224           build_output (LANG_GREEK, "o(");	/* U+1F41 GREEK SMALL LETTER OMICRON WITH DASIA                   */
\316\277\314\223\314\200   build_output (LANG_GREEK, "o)\\");	/* U+1F42 GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA         */
\316\277\314\224\314\200   build_output (LANG_GREEK, "o(\\");	/* U+1F43 GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA         */
\316\277\314\223\314\201   build_output (LANG_GREEK, "o)/");	/* U+1F44 GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA          */
\316\277\314\224\314\201   build_output (LANG_GREEK, "o(/");	/* U+1F45 GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA          */
\316\237\314\223           build_output (LANG_GREEK, "*)o");	/* U+1F48 GREEK CAPITAL LETTER OMICRON WITH PSILI                 */
\316\237\314\224           build_output (LANG_GREEK, "*(o");	/* U+1F49 GREEK CAPITAL LETTER OMICRON WITH DASIA                 */
\316\237\314\223\314\200   build_output (LANG_GREEK, "*)\\o");	/* U+1F4A GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA       */
\316\237\314\224\314\200   build_output (LANG_GREEK, "*(\\o");	/* U+1F4B GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA       */
\316\237\314\223\314\201   build_output (LANG_GREEK, "*)/o");	/* U+1F4C GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA        */
\316\237\314\224\314\201   build_output (LANG_GREEK, "*(/o");	/* U+1F4D GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA        */
\317\205\314\223           build_output (LANG_GREEK, "u)");	/* U+1F50 GREEK SMALL LETTER UPSILON WITH PSILI                   */
\317\205\314\224           build_output (LANG_GREEK, "u(");	/* U+1F51 GREEK SMALL LETTER UPSILON WITH DASIA                   */
\317\205\314\223\314\200   build_output (LANG_GREEK, "u)\\");	/* U+1F52 GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA         */
\317\205\314\224\314\200   build_output (LANG_GREEK, "u(\\");	/* U+1F53 GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA         */
\317\205\314\223\314\201   build_output (LANG_GREEK, "u)/");	/* U+1F54 GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA          */
\317\205\314\224\314\201   build_output (LANG_GREEK, "u(/");	/* U+1F55 GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA          */
\317\205\314\223\315\202   build_output (LANG_GREEK, "u)=");	/* U+1F56 GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI   */
\317\205\314\224\315\202   build_output (LANG_GREEK, "u(=");	/* U+1F57 GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI   */
\316\245\314\224           build_output (LANG_GREEK, "*(u");	/* U+1F59 GREEK CAPITAL LETTER UPSILON WITH DASIA                 */
\316\245\314\224\314\200   build_output (LANG_GREEK, "*(\\u");	/* U+1F5B GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA       */
\316\245\314\224\314\201   build_output (LANG_GREEK, "*(/u");	/* U+1F5D GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA        */
\316\245\314\224\315\202   build_output (LANG_GREEK, "*(=u");	/* U+1F5F GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI */
\317\211\314\223           build_output (LANG_GREEK, "w)");	/* U+1F60 GREEK SMALL LETTER OMEGA WITH PSILI                     */
\317\211\314\224           build_output (LANG_GREEK, "w(");	/* U+1F61 GREEK SMALL LETTER OMEGA WITH DASIA                     */
\317\211\314\223\314\200   build_output (LANG_GREEK, "w)\\");	/* U+1F62 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA           */
\317\211\314\224\314\200   build_output (LANG_GREEK, "w(\\");	/* U+1F63 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA           */
\317\211\314\223\314\201   build_output (LANG_GREEK, "w)/");	/* U+1F64 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA            */
\317\211\314\224\314\201   build_output (LANG_GREEK, "w(/");	/* U+1F65 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA            */
\317\211\314\223\315\202   build_output (LANG_GREEK, "w)=");	/* U+1F66 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI     */
\317\211\314\224\315\202   build_output (LANG_GREEK, "w(=");	/* U+1F67 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI     */
\316\251\314\223           build_output (LANG_GREEK, "*)w");	/* U+1F68 GREEK CAPITAL LETTER OMEGA WITH PSILI                   */
\316\251\314\224           build_output (LANG_GREEK, "*(w");	/* U+1F69 GREEK CAPITAL LETTER OMEGA WITH DASIA                   */
\316\251\314\223\314\200   build_output (LANG_GREEK, "*)\\w");	/* U+1F6A GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA         */
\316\251\314\224\314\200   build_output (LANG_GREEK, "*(\\w");	/* U+1F6B GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA         */
\316\251\314\223\314\201   build_output (LANG_GREEK, "*)/w");	/* U+1F6C GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA          */
\316\251\314\224\314\201   build_output (LANG_GREEK, "*(/w");	/* U+1F6D GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA          */
\316\251\314\223\315\202   build_output (LANG_GREEK, "*)=w");	/* U+1F6E GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI   */
\316\251\314\224\315\202   build_output (LANG_GREEK, "*(=w");	/* U+1F6F GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI   */
\316\261\314\200   build_output (LANG_GREEK, "a\\");	/* U+1F70 GREEK SMALL LETTER ALPHA WITH VARIA    */
\316\265\314\200   build_output (LANG_GREEK, "e\\");	/* U+1F72 GREEK SMALL LETTER EPSILON WITH VARIA  */
\316\267\314\200   build_output (LANG_GREEK, "h\\");	/* U+1F74 GREEK SMALL LETTER ETA WITH VARIA      */
\316\271\314\200   build_output (LANG_GREEK, "i\\");	/* U+1F76 GREEK SMALL LETTER IOTA WITH VARIA     */
\316\277\314\200   build_output (LANG_GREEK, "o\\");	/* U+1F78 GREEK SMALL LETTER OMICRON WITH VARIA  */
\317\205\314\200   build_output (LANG_GREEK, "u\\");	/* U+1F7A GREEK SMALL LETTER UPSILON WITH VARIA  */
\317\211\314\200   build_output (LANG_GREEK, "w\\");	/* U+1F7C GREEK SMALL LETTER OMEGA WITH VARIA    */
\316\261\314\223\315\205           build_output (LANG_GREEK, "a)|");	/* U+1F80 GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI              */
\316\261\314\224\315\205           build_output (LANG_GREEK, "a(|");	/* U+1F81 GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI              */
\316\261\314\223\314\200\315\205   build_output (LANG_GREEK, "a)\\|");	/* U+1F82 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI    */
\316\261\314\224\314\200\315\205   build_output (LANG_GREEK, "a(\\|");	/* U+1F83 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI    */
\316\261\314\223\314\201\315\205   build_output (LANG_GREEK, "a)/|");	/* U+1F84 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI     */
\316\261\314\224\314\201\315\205   build_output (LANG_GREEK, "a(/|");	/* U+1F85 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI     */
\316\261\314\223\315\202\315\205   build_output (LANG_GREEK, "a)=|");	/* U+1F86 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI */
\316\261\314\224\315\202\315\205   build_output (LANG_GREEK, "a(=|");	/* U+1F87 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI */
\316\221\314\223\315\205           build_output (LANG_GREEK, "*)a|");	/* U+1F88 GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI           */
\316\221\314\224\315\205           build_output (LANG_GREEK, "*(a|");	/* U+1F89 GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI           */
\316\221\314\223\314\200\315\205   build_output (LANG_GREEK, "*)\\a|");	/* U+1F8A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI */
\316\221\314\224\314\200\315\205   build_output (LANG_GREEK, "*(\\a|");	/* U+1F8B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI */
\316\221\314\223\314\201\315\205   build_output (LANG_GREEK, "*)/a|");	/* U+1F8C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI  */
\316\221\314\224\314\201\315\205   build_output (LANG_GREEK, "*(/a|");	/* U+1F8D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI  */
\316\221\314\223\315\202\315\205   build_output (LANG_GREEK, "*)=a|");	/* U+1F8E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI */
\316\221\314\224\315\202\315\205   build_output (LANG_GREEK, "*(=a|");	/* U+1F8F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI */
\316\267\314\223\315\205           build_output (LANG_GREEK, "h)|");	/* U+1F90 GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI                */
\316\267\314\224\315\205           build_output (LANG_GREEK, "h(|");	/* U+1F91 GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI                */
\316\267\314\223\314\200\315\205   build_output (LANG_GREEK, "h)\\|");	/* U+1F92 GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI      */
\316\267\314\224\314\200\315\205   build_output (LANG_GREEK, "h(\\|");	/* U+1F93 GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI      */
\316\267\314\223\314\201\315\205   build_output (LANG_GREEK, "h)/|");	/* U+1F94 GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI       */
\316\267\314\224\314\201\315\205   build_output (LANG_GREEK, "h(/|");	/* U+1F95 GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI       */
\316\267\314\223\315\202\315\205   build_output (LANG_GREEK, "h)=|");	/* U+1F96 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI      */
\316\267\314\224\315\202\315\205   build_output (LANG_GREEK, "h(=|");	/* U+1F97 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI      */
\316\227\314\223\315\205           build_output (LANG_GREEK, "*)h|");	/* U+1F98 GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI             */
\316\227\314\224\315\205           build_output (LANG_GREEK, "*(h|");	/* U+1F99 GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI             */
\316\227\314\223\314\200\315\205   build_output (LANG_GREEK, "*)\\h|");	/* U+1F9A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI   */
\316\227\314\224\314\200\315\205   build_output (LANG_GREEK, "*(\\h|");	/* U+1F9B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI   */
\316\227\314\223\314\201\315\205   build_output (LANG_GREEK, "*)/h|");	/* U+1F9C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI    */
\316\227\314\224\314\201\315\205   build_output (LANG_GREEK, "*(/h|");	/* U+1F9D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI    */
\316\227\314\223\315\202\315\205   build_output (LANG_GREEK, "*)=h|");	/* U+1F9E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI   */
\316\227\314\224\315\202\315\205   build_output (LANG_GREEK, "*(=h|");	/* U+1F9F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI   */
\317\211\314\223\315\205           build_output (LANG_GREEK, "w)|");	/* U+1FA0 GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI              */
\317\211\314\224\315\205           build_output (LANG_GREEK, "w(|");	/* U+1FA1 GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI              */
\317\211\314\223\314\200\315\205   build_output (LANG_GREEK, "w)\\|");	/* U+1FA2 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI    */
\317\211\314\224\314\200\315\205   build_output (LANG_GREEK, "w(\\|");	/* U+1FA3 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI    */
\317\211\314\223\314\201\315\205   build_output (LANG_GREEK, "w)/|");	/* U+1FA4 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI     */
\317\211\314\224\314\201\315\205   build_output (LANG_GREEK, "w(/|");	/* U+1FA5 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI     */
\317\211\314\223\315\202\315\205   build_output (LANG_GREEK, "w)=|");	/* U+1FA6 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI    */
\317\211\314\224\315\202\315\205   build_output (LANG_GREEK, "w(=|");	/* U+1FA7 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI    */
\316\251\314\223\315\205           build_output (LANG_GREEK, "*)w|");	/* U+1FA8 GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI           */
\316\251\314\224\315\205           build_output (LANG_GREEK, "*(w|");	/* U+1FA9 GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI           */
\316\251\314\223\314\200\315\205   build_output (LANG_GREEK, "*)\\w|");	/* U+1FAA GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI       */
\316\251\314\224\314\200\315\205   build_output (LANG_GREEK, "*(\\w|");	/* U+1FAB GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI       */
\316\251\314\223\314\201\315\205   build_output (LANG_GREEK, "*)/w|");	/* U+1FAC GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI        */
\316\251\314\224\314\201\315\205   build_output (LANG_GREEK, "*(/w|");	/* U+1FAD GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI        */
\316\251\314\223\315\202\315\205   build_output (LANG_GREEK, "*)=w|");	/* U+1FAE GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI */
\316\251\314\224\315\202\315\205   build_output (LANG_GREEK, "*(=w|");	/* U+1FAF GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI */
\316\261\314\206           build_output (LANG_GREEK, "a'");	/* U+1FB0 GREEK SMALL LETTER ALPHA WITH VRACHY                        */
\316\261\314\204           build_output (LANG_GREEK, "a&");	/* U+1FB1 GREEK SMALL LETTER ALPHA WITH MACRON                        */
\316\261\314\200\315\205   build_output (LANG_GREEK, "a\\|");	/* U+1FB2 GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI       */
\316\261\315\205           build_output (LANG_GREEK, "a|");	/* U+1FB3 GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI                 */
\316\261\314\201\315\205   build_output (LANG_GREEK, "a/|");	/* U+1FB4 GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI        */
\316\261\315\202           build_output (LANG_GREEK, "a=");	/* U+1FB6 GREEK SMALL LETTER ALPHA WITH PERISPOMENI                   */
\316\261\315\202\315\205   build_output (LANG_GREEK, "a=|");	/* U+1FB7 GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI */
\316\221\314\206           build_output (LANG_GREEK, "*'a");	/* U+1FB8 GREEK CAPITAL LETTER ALPHA WITH VRACHY                      */
\316\221\314\204           build_output (LANG_GREEK, "*&a");	/* U+1FB9 GREEK CAPITAL LETTER ALPHA WITH MACRON                      */
\316\221\314\200           build_output (LANG_GREEK, "*\\a");	/* U+1FBA GREEK CAPITAL LETTER ALPHA WITH VARIA                       */
\316\221\315\205           build_output (LANG_GREEK, "*a|");	/* U+1FBC GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI              */
\316\267\314\200\315\205   build_output (LANG_GREEK, "h\\|");	/* U+1FC2 GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI         */
\316\267\315\205           build_output (LANG_GREEK, "h|");	/* U+1FC3 GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI                   */
\316\267\314\201\315\205   build_output (LANG_GREEK, "h/|");	/* U+1FC4 GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI          */
\316\267\315\202           build_output (LANG_GREEK, "h=");	/* U+1FC6 GREEK SMALL LETTER ETA WITH PERISPOMENI                     */
\316\267\315\202\315\205   build_output (LANG_GREEK, "h=|");	/* U+1FC7 GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI   */
\316\225\314\200           build_output (LANG_GREEK, "*\\e");	/* U+1FC8 GREEK CAPITAL LETTER EPSILON WITH VARIA                     */
\316\227\314\200           build_output (LANG_GREEK, "*\\h");	/* U+1FCA GREEK CAPITAL LETTER ETA WITH VARIA                         */
\316\227\315\205           build_output (LANG_GREEK, "*h|");	/* U+1FCC GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI                */
\316\271\314\206           build_output (LANG_GREEK, "i'");	/* U+1FD0 GREEK SMALL LETTER IOTA WITH VRACHY                         */
\316\271\314\204           build_output (LANG_GREEK, "&");	/* U+1FD1 GREEK SMALL LETTER IOTA WITH MACRON                         */
\316\271\314\200\314\210   build_output (LANG_GREEK, "i+\\");	/* U+1FD2 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA            */
\316\271\314\210\314\200   build_output (LANG_GREEK, "i+\\");	/* U+1FD2 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA            */
\316\271\315\202           build_output (LANG_GREEK, "i=");	/* U+1FD6 GREEK SMALL LETTER IOTA WITH PERISPOMENI                    */
\316\271\314\202\315\210   build_output (LANG_GREEK, "i+=");	/* U+1FD7 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI      */
\316\271\314\210\315\202   build_output (LANG_GREEK, "i+=");	/* U+1FD7 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI      */
\316\231\314\206           build_output (LANG_GREEK, "*'i");	/* U+1FD8 GREEK CAPITAL LETTER IOTA WITH VRACHY                       */
\316\231\314\204           build_output (LANG_GREEK, "*&i");	/* U+1FD9 GREEK CAPITAL LETTER IOTA WITH MACRON                       */
\316\231\314\200           build_output (LANG_GREEK, "*\\i");	/* U+1FDA GREEK CAPITAL LETTER IOTA WITH VARIA                        */
\317\205\314\206           build_output (LANG_GREEK, "u'");	/* U+1FE0 GREEK SMALL LETTER UPSILON WITH VRACHY                      */
\317\205\314\204           build_output (LANG_GREEK, "u&");	/* U+1FE1 GREEK SMALL LETTER UPSILON WITH MACRON                      */
\317\205\314\200\314\210   build_output (LANG_GREEK, "u+\\");	/* U+1FE2 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA         */
\317\205\314\210\314\200   build_output (LANG_GREEK, "u+\\");	/* U+1FE2 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA         */
\317\201\314\223           build_output (LANG_GREEK, "r)");	/* U+1FE4 GREEK SMALL LETTER RHO WITH PSILI                           */
\317\201\314\224           build_output (LANG_GREEK, "r(");	/* U+1FE5 GREEK SMALL LETTER RHO WITH DASIA                           */
\317\205\315\202           build_output (LANG_GREEK, "u=");	/* U+1FE6 GREEK SMALL LETTER UPSILON WITH PERISPOMENI                 */
\317\205\314\202\315\210   build_output (LANG_GREEK, "u+=");	/* U+1FE7 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI   */
\317\205\314\210\315\202   build_output (LANG_GREEK, "u+=");	/* U+1FE7 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI   */
\316\245\314\206           build_output (LANG_GREEK, "*'u");	/* U+1FE8 GREEK CAPITAL LETTER UPSILON WITH VRACHY                    */
\316\245\314\204           build_output (LANG_GREEK, "*&u");	/* U+1FE9 GREEK CAPITAL LETTER UPSILON WITH MACRON                    */
\316\245\314\200           build_output (LANG_GREEK, "*\\u");	/* U+1FEA GREEK CAPITAL LETTER UPSILON WITH VARIA                     */
\316\241\314\224           build_output (LANG_GREEK, "*(r");	/* U+1FEC GREEK CAPITAL LETTER RHO WITH DASIA                         */
\317\211\314\200\315\205   build_output (LANG_GREEK, "w\\|");	/* U+1FF2 GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI       */
\317\211\315\205           build_output (LANG_GREEK, "w|");	/* U+1FF3 GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI                 */
\317\211\314\201\315\205   build_output (LANG_GREEK, "w/|");	/* U+1FF4 GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI        */
\317\211\315\202           build_output (LANG_GREEK, "w=");	/* U+1FF6 GREEK SMALL LETTER OMEGA WITH PERISPOMENI                   */
\317\211\315\202\315\205   build_output (LANG_GREEK, "w=|");	/* U+1FF7 GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI */
\316\237\314\200           build_output (LANG_GREEK, "*\\o");	/* U+1FF8 GREEK CAPITAL LETTER OMICRON WITH VARIA                     */
\316\251\314\200           build_output (LANG_GREEK, "*\\w");	/* U+1FFA GREEK CAPITAL LETTER OMEGA WITH VARIA                       */
\316\251\315\205           build_output (LANG_GREEK, "*=w");	/* U+1FFC GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI              */
\342\262\200  build_output (LANG_COPTIC, "*A");	/* U+2C80 COPTIC CAPITAL LETTER ALFA   */
\342\262\201  build_output (LANG_COPTIC, "A");	/* U+2C81 COPTIC SMALL LETTER ALFA     */
\342\262\202  build_output (LANG_COPTIC, "*B");	/* U+2C82 COPTIC CAPITAL LETTER VIDA   */
\342\262\203  build_output (LANG_COPTIC, "B");	/* U+2C83 COPTIC SMALL LETTER VIDA     */
\342\262\204  build_output (LANG_COPTIC, "*G");	/* U+2C84 COPTIC CAPITAL LETTER GAMMA  */
\342\262\205  build_output (LANG_COPTIC, "G");	/* U+2C85 COPTIC SMALL LETTER GAMMA    */
\342\262\206  build_output (LANG_COPTIC, "*D");	/* U+2C86 COPTIC CAPITAL LETTER DALDA  */
\342\262\207  build_output (LANG_COPTIC, "D");	/* U+2C87 COPTIC SMALL LETTER DALDA    */
\342\262\210  build_output (LANG_COPTIC, "*E");	/* U+2C88 COPTIC CAPITAL LETTER EIE    */
\342\262\211  build_output (LANG_COPTIC, "E");	/* U+2C89 COPTIC SMALL LETTER EIE      */
\342\262\212  build_output (LANG_COPTIC, "*V");	/* U+2C8A COPTIC CAPITAL LETTER SOU    */
\342\262\213  build_output (LANG_COPTIC, "V");	/* U+2C8B COPTIC SMALL LETTER SOU      */
\342\262\214  build_output (LANG_COPTIC, "*Z");	/* U+2C8C COPTIC CAPITAL LETTER ZATA   */
\342\262\215  build_output (LANG_COPTIC, "Z");	/* U+2C8D COPTIC SMALL LETTER ZATA     */
\342\262\216  build_output (LANG_COPTIC, "*H");	/* U+2C8E COPTIC CAPITAL LETTER HATE   */
\342\262\217  build_output (LANG_COPTIC, "H");	/* U+2C8F COPTIC SMALL LETTER HATE     */
\342\262\220  build_output (LANG_COPTIC, "*Q");	/* U+2C90 COPTIC CAPITAL LETTER THETHE */
\342\262\221  build_output (LANG_COPTIC, "Q");	/* U+2C91 COPTIC SMALL LETTER THETHE   */
\342\262\222  build_output (LANG_COPTIC, "*I");	/* U+2C92 COPTIC CAPITAL LETTER IAUDA  */
\342\262\223  build_output (LANG_COPTIC, "I");	/* U+2C93 COPTIC SMALL LETTER IAUDA    */
\342\262\224  build_output (LANG_COPTIC, "*K");	/* U+2C94 COPTIC CAPITAL LETTER KAPA   */
\342\262\225  build_output (LANG_COPTIC, "K");	/* U+2C95 COPTIC SMALL LETTER KAPA     */
\342\262\226  build_output (LANG_COPTIC, "*L");	/* U+2C96 COPTIC CAPITAL LETTER LAULA  */
\342\262\227  build_output (LANG_COPTIC, "L");	/* U+2C97 COPTIC SMALL LETTER LAULA    */
\342\262\230  build_output (LANG_COPTIC, "*M");	/* U+2C98 COPTIC CAPITAL LETTER MI     */
\342\262\231  build_output (LANG_COPTIC, "M");	/* U+2C99 COPTIC SMALL LETTER MI       */
\342\262\232  build_output (LANG_COPTIC, "*N");	/* U+2C9A COPTIC CAPITAL LETTER NI     */
\342\262\233  build_output (LANG_COPTIC, "N");	/* U+2C9B COPTIC SMALL LETTER NI       */
\342\262\234  build_output (LANG_COPTIC, "*C");	/* U+2C9C COPTIC CAPITAL LETTER KSI    */
\342\262\235  build_output (LANG_COPTIC, "C");	/* U+2C9D COPTIC SMALL LETTER KSI      */
\342\262\236  build_output (LANG_COPTIC, "*O");	/* U+2C9E COPTIC CAPITAL LETTER O      */
\342\262\237  build_output (LANG_COPTIC, "O");	/* U+2C9F COPTIC SMALL LETTER O        */
\342\262\240  build_output (LANG_COPTIC, "*P");	/* U+2CA0 COPTIC CAPITAL LETTER PI     */
\342\262\241  build_output (LANG_COPTIC, "P");	/* U+2CA1 COPTIC SMALL LETTER PI       */
\342\262\242  build_output (LANG_COPTIC, "*R");	/* U+2CA2 COPTIC CAPITAL LETTER RO     */
\342\262\243  build_output (LANG_COPTIC, "R");	/* U+2CA3 COPTIC SMALL LETTER RO       */
\342\262\244  build_output (LANG_COPTIC, "*S");	/* U+2CA4 COPTIC CAPITAL LETTER SIMA   */
\342\262\245  build_output (LANG_COPTIC, "S");	/* U+2CA5 COPTIC SMALL LETTER SIMA     */
\342\262\246  build_output (LANG_COPTIC, "*T");	/* U+2CA6 COPTIC CAPITAL LETTER TAU    */
\342\262\247  build_output (LANG_COPTIC, "T");	/* U+2CA7 COPTIC SMALL LETTER TAU      */
\342\262\250  build_output (LANG_COPTIC, "*U");	/* U+2CA8 COPTIC CAPITAL LETTER UA     */
\342\262\251  build_output (LANG_COPTIC, "U");	/* U+2CA9 COPTIC SMALL LETTER UA       */
\342\262\252  build_output (LANG_COPTIC, "*F");	/* U+2CAA COPTIC CAPITAL LETTER FI     */
\342\262\253  build_output (LANG_COPTIC, "F");	/* U+2CAB COPTIC SMALL LETTER FI       */
\342\262\254  build_output (LANG_COPTIC, "*X");	/* U+2CAC COPTIC CAPITAL LETTER KHI    */
\342\262\255  build_output (LANG_COPTIC, "X");	/* U+2CAD COPTIC SMALL LETTER KHI      */
\342\262\256  build_output (LANG_COPTIC, "*Y");	/* U+2CAE COPTIC CAPITAL LETTER PSI    */
\342\262\257  build_output (LANG_COPTIC, "Y");	/* U+2CAF COPTIC SMALL LETTER PSI      */
\342\262\260  build_output (LANG_COPTIC, "*W");	/* U+2CB0 COPTIC CAPITAL LETTER OOU    */
\342\262\261  build_output (LANG_COPTIC, "W");	/* U+2CB1 COPTIC SMALL LETTER OOU      */
[\302-\337][\200-\277]		build_output (LANG_UTF, yytext);  /* well-formed 2 byte UTF-8 character   U+0080 ..   U+07FF */
[\340-\357][\200-\277]{2}	build_output (LANG_UTF, yytext);  /* well-formed 3 byte UTF-8 character   U+0800 ..   U+FFFF */
[\360-\363][\200-\277]{3}	build_output (LANG_UTF, yytext);  /* well-formed 4 byte UTF-8 character  U+10000 ..  U+FFFFF */
\364[\200-\217][\200-\277]{2}	build_output (LANG_UTF, yytext);  /* well-formed 4 byte UTF-8 character U+100000 .. U+10FFFF */
[ ]/{GREEK}	{  /* Switch to Greek mode before a preceding space, so Greek text is easier to read */
		   if (was_escaped == 1)
		      end_escape ();
		   if (lang_mode != LANG_GREEK) {
		      fputc ('$', yyout);
		      lang_mode = LANG_GREEK;
		   }
		   fputc (' ', yyout);
		}
[ ]/{COPTIC}	{  /* Switch to Coptic mode before a preceding space, so Coptic text is easier to read */
		   if (was_escaped == 1)
		      end_escape ();
		   if (lang_mode != LANG_COPTIC) {
		      fprintf (yyout, "&100");
		      lang_mode = LANG_COPTIC;
		   }
		   fputc (' ', yyout);
		}
[ ]/{HEBREW}	{  /* Switch to Hebrew mode before a preceding space, so Hebrew text is easier to read */
		   if (was_escaped == 1)
		      end_escape ();
		   if (lang_mode != LANG_HEBREW) {
		      fprintf (yyout, "&300");
		      lang_mode = LANG_HEBREW;
		   }
		   fputc (' ', yyout);
		}
\n	{  /*
	      Flush any pending output and set the font to LANG_GREEK at end of line.
	   */
           end_escape ();
	   if (lang_mode != LANG_GREEK) {
	      fputc ('$', yyout);  /* Return to Greek mode */
	      lang_mode = LANG_GREEK;
	   }
	   fputc ('\n', yyout);
	}
.       build_output (LANG_ASCII, yytext);  /* Miscellaneous character in the ASCII range */
%%

int
main (int argc, char *argv[])
{
   int i;            /* loop variable       */
   int exit_status;  /* program exit status */

   void print_help (char *);

   exit_status = EXIT_SUCCESS;
   yyin  = stdin;
   yyout = stdout;


   for (i = 1; i < argc; i++) {
      /*
         Parse options.  If an invalid command line argument
         was given, print a help menu and exit with error status.
      */
      if (argv[i][0] == '-') {
         switch (argv[i][1]) {
                      /* Check for "--version" */
            case '-': if (strncmp (argv[i], "--version", 9) == 0) {
                         printf ("uni2beta Version %s\n", VERSION);
                         exit (EXIT_SUCCESS);
                      }
            case 'l': uppercase = 0;  /* Output lowercase Latin (the default) */
                      break;
            case 'u': uppercase = 1;  /* Output uppercase Latin */
                      break;
                      /*
                         input file format; file name follows
                         in next parameter, so increment i
                      */
            case 'i': yyin  = fopen (argv[++i], "r");
                      break;
                      /*
                         output file format; file name follows
                         in next parameter, so increment i
                      */
            case 'o': yyout = fopen (argv[++i], "w");
                      break;
                      /* Check for "-v" */
            case 'v': printf ("uni2beta Version %s\n", VERSION);
                      exit (EXIT_SUCCESS);
            default:  print_help (argv[0]);
                      exit_status = EXIT_FAILURE;
                      break;
         }
      }
      else {
         print_help (argv[0]);
         exit_status = EXIT_FAILURE;
      }
   }

   if (exit_status == EXIT_SUCCESS) {
      yylex ();

      /* Write final Beta Code output if there is any. */
      end_escape ();
   }

   exit (exit_status);
}


/*
   Print a help message.  The parameter is the program name,
   taken from argv[0].
*/
void
print_help (char * progname)
{

   fprintf (stderr, "\nUnknown command line parameter.\n\n");
   fprintf (stderr, "Syntax: %s [-l|-u] [-i input_file] [-o output_file]\n\n",
            progname);
   fprintf (stderr, "    -l: output Greek in lowercase Latin letters (the default)\n\n");
   fprintf (stderr, "    -u: output Greek in uppercase Latin letters\n\n");
   fprintf (stderr, "    -i: specify input file name\n\n");
   fprintf (stderr, "    -o: specify output file name\n\n");

   return;
}


/*
   Print a Unicode sequence or other sequence that might be
   interpreted as another character if it does not appear within
   curly brackets ("{...}").
*/
void
escape_out (char *new_text)
{
   int      i;
   uint32_t this_char;        /* Current character in new_text         */
   unsigned mask;             /* To determine UTF-8 character length   */
   int      utf8_len;         /* # of bytes in current UTF-8 character */
   uint32_t codept;           /* An input Unicode code point to print  */

   i = 0;
   this_char = new_text[i] & 0xFF;

   if ((this_char & 0x80) == 0x80) {  /* Found start of a UTF-8 sequence */
      this_char &= 0xFF;
      /* Find out how many bytes long this UTF-8 character is */
      mask = 0x80;  /* examine input character bits from MSB to LSB */
      for (utf8_len = 0; (this_char & mask) != 0; utf8_len++) {
         this_char ^= mask;  /* toggle next UTF-8 header bit off */
         mask      >>= 1;
      }
      /*
         Now this_char contains the high-order siginificant
         bytes of the code point, with the leading UTF-8
         length indicator bits set to zero.

         utf8_len contains byte length of this UTF-8 character.
      */
      codept = this_char;
      while (--utf8_len > 0) {
         codept  <<= 6;  /* 6 code point bits per UTF-8 byte after the first byte */
         this_char = new_text [++i];
         this_char &= 0x3F;  /* mask off the two high-order bits */
         codept   |= this_char;
      }

      /*
         If last output wasn't an escape sequence, start with '{'
      */
      if (was_escaped != 1) {
         fputc ('{', yyout);
         was_escaped = 1;
      }
      fprintf (yyout, "\\u%04X", codept);
      was_codept = 1;  /* last output was a Unicode codepoint "\u..." */
   }  /* if high bit set, print Uniocde code point */
   else {  /* print ASCII character */
      /*
         If last thing printed was a Unicode code point of the form
         "\u..." and we are about to print what could be misinterpreted
         as a hexadecimal sequence, end the last escape string and
         begin a new one.
      */
      if (was_codept == 1 &&
          ((this_char >= '0' && this_char <= '9') ||
           (this_char >= 'A' && this_char <= 'F') ||
           (this_char >= 'a' && this_char <= 'f'))) {
         fprintf (yyout, "}{");
      }
      fprintf (yyout, "%s", yytext);
   }

   return;
}


/*
   Build the output, one Beta Code at a time.  If the current
   Beta Code contains a small sigma, preserve it until we see
   the Beta Code that follows.  This allows simplifying s1 to s
   for a middle sigma, and s2 to s for a final sigma, when they
   are in their proper positions in a word.
*/
void
build_output (int this_lang, char *new_text)
{
   int      i;          /* loop variable                 */
   uint32_t this_char;  /* Current character in new_text */

   void end_escape ();  /* end a running escape sequence "{...}" */

   this_char = new_text [0] & 0xFF;

   /*
      See if we need to change the font.  If this_lang < 0,
      it indicates a numeric digit, ASCII '0' through '9';
      never change the font for digits.  They are output
      the same for all fonts.
   */
   if (this_lang == LANG_ASCII) {
      if (was_escaped == 1) {  /* End a UTF-8 "\ux...x" series */
         end_escape ();
      }
      /*
         For a character common to all font modes, just print the character.
      */
      if (this_char == ' ' || this_char == '!' ||
          this_char == ',' || this_char == '.' ||
          this_char == ';') {
         fprintf (yyout, "%s", new_text);
      }
      else {
         if (lang_mode != LANG_LATIN) {
            fputc ('&', yyout);
            lang_mode = LANG_LATIN;
         }
         /*
            If a Beta Code language mode character, '$' or '&',
            print as Unicode to avoid misinterpretation.
         */
         if (this_char == '$') {
            escape_out ("\\u0024");
         }
         else if (this_char == '&') {
            escape_out ("\\u0026");
         }
         else {
            fprintf (yyout, "%s", new_text);
         }
      }
   }  /* this_lang == LANG_ASCII */
   else if (this_lang == LANG_UTF) {  /* yytext holds a UTF-8 byte sequence */
      escape_out (new_text);
   }  /* this_lang == LANG_UTF */
   /*
      Switch font mode if necessary.
   */
   else if (was_escaped == 1 || this_lang != lang_mode) {
      /*
         If changing to non-Latin language mode,
         make sure a running escape sequence ends
      */
      if (lang_mode != LANG_LATIN)
         end_escape ();

      if (this_lang != lang_mode) {
         switch (this_lang) {
            case LANG_GREEK:
                 fprintf (yyout, "$");
                 break;
            case LANG_LATIN:
                 fprintf (yyout, "&");
                 break;
            case LANG_COPTIC:
                 fprintf (yyout, "&100");
                 break;
            case LANG_HEBREW:
                 fprintf (yyout, "&300");
                 break;
            default:
                 fprintf (yyout, "$");  /* Greek mode is the default */
                 break;
         }  /* switch (this_lang) */
      }  /* this_lang != lang_mode */

   lang_mode = this_lang;

   }  /* was_escaped == 1 || this_lang != lang_mode */

   /*
      If this wasn't already processed for printing above,
      output the string that was passed to build_output.
   */
   if (this_lang != LANG_ASCII && this_lang != LANG_UTF) {
      /* Convert case of Greek Beta Code letters if desired */
      if ((lang_mode == LANG_GREEK) && (uppercase == 1)) {
         /* Don't modify new_text, because it could be a const char * */
         for (i = 0; new_text [i] != '\0'; i++) {
            fputc (toupper (new_text [i]), yyout);
         }
      }
      else {
         fprintf (yyout, "%s", new_text);
      }
   }

   return;
}


/*
   End an escape sequence if we have been printing one.
*/
void
end_escape ()
{
   /*
      Flush any pending output of a preceding escape sequence.
   */
   if (was_escaped == 1) {
      fputc ('}', yyout);
      was_escaped = 0;
      was_codept  = 0;
   }

   return;
}

